/*
 * Copyright (C) 2019-2024 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * A list of label buttons, optionally with a ... to make all choices available.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "xitkintl.h"
#include "button_list.h"
#include "labelbutton.h"
#include "skin.h"

#define MAX_BUTTONS 64
/* mrlbrowser calls "show" on the entire widget list.
 * we should not add our hidden ones there, and keep "..." separate.
 * big example: num = 7, max_buttons = 4, visible = 3, (hidden), [shown]
 * 0                 first                   num         last
 * (101) (102) (103) [104] [105] [106] (107) (   ) (   )        [...]
 * medium example: num = 4, max_buttons = 4, visible = 4
 * 0 = first               num = last
 * [101] [102] [103] [104]            (...)
 * small exammple: num = 2, max_buttons = 4, visible = 2
 * 0 = first   num = last
 * [101] [102]            (...) */

typedef struct {
  xitk_widget_t       w;
  xitk_skin_config_t *skin_config;
  xitk_state_callback_t callback;
  int                *still_here, select_limit, last_i;
  uint8_t             helipad[MAX_BUTTONS];
  int                 flags, num, visible, first, last;
  int                 x, y, dx, dy;
  int                 bw, bh;
  uint32_t            widget_type_flags;
  xitk_widget_t      *swap, *widgets[MAX_BUTTONS];
} xitk_button_list_t;

/* optimization: instead of hide/disable, remove hidden widgets from group temporarily. */
static void xitk_button_list_remove (xitk_button_list_t *bl) {
  int i = bl->first, e = i + bl->visible;

  if (e > bl->last)
    e = bl->last;
  /* [101] and [   ] */
  /* xitk_widgets_state (bl->widgets + i, e - i, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, 0); */
  while (i < e) {
    xitk_widget_set_parent (bl->widgets[i], NULL);
    i++;
  }
}

/* optimization: instead of show/enable, add to group again, and paint manually. */
static void xitk_button_list_add (xitk_button_list_t *bl) {
  int b, e, i, x, y;
  widget_event_t event;

  event.type = (bl->w.wl->flags & XITK_WL_EXPOSED) ? WIDGET_EVENT_PAINT : WIDGET_EVENT_NONE;
  event.width = bl->bw;
  event.height = bl->bh;
  
  b = bl->first;
  e = b + bl->visible;
  if (e > bl->last)
    e = bl->last;

  x = bl->x;
  y = bl->y;
  /* [101] [   } */
  for (i = b; i < e; i++) {
    xitk_widget_set_parent (bl->widgets[i], &bl->w);
    xitk_set_widget_pos (bl->widgets[i], x, y);
    event.x = x;
    event.y = y;
    bl->widgets[i]->event (bl->widgets[i], &event);
    x += bl->dx;
    y += bl->dy;
  }
  /* [...] */
  if ((bl->num > bl->visible) && bl->swap) {
    xitk_widget_set_parent (bl->swap, &bl->w);
    xitk_set_widget_pos (bl->swap, x, y);
    /* xitk_widgets_state (&bl->swap, 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u); */
    event.x = x;
    event.y = y;
    bl->swap->event (bl->swap, &event);
  }
}

static void xitk_button_list_click (xitk_widget_t *w, void *ip, int state) {
  const uint8_t *land = (const uint8_t *)ip;
  int i = *land;
  xitk_button_list_t *bl;
  xitk_widget_t *last_w;
  int last_i;

  (void)w;
  xitk_container (bl, land, helipad[i]);
  last_i = (state && (bl->select_limit > 0)) ? bl->last_i : i;
  last_w = (last_i >= 0) ? bl->widgets[last_i] : NULL;
  bl->last_i = i;
  if (last_i != i)
    xitk_widgets_state (&last_w, 1, XITK_WIDGET_STATE_ON, 0);
  if (bl->callback) {
    int still_here = 1;

    bl->still_here = &still_here;
    bl->callback (bl->widgets[i], bl->w.userdata, state);
    if (!still_here)
      return;
    bl->still_here = NULL;
    if (last_i != i)
      bl->callback (last_w, bl->w.userdata, 0);
  }
}

static void xitk_button_list_swap (xitk_widget_t *w, void *ip, int state) {
  xitk_button_list_t *bl = (xitk_button_list_t *)ip;

  (void)w;
  (void)state;

  xitk_button_list_remove (bl);

  bl->first += bl->visible;
  if (bl->first >= bl->num)
    bl->first = 0;

  xitk_button_list_add (bl);
}

static void xitk_button_list_hull (xitk_button_list_t *bl, const xitk_skin_element_info_t *info) {
  XITK_HV_INIT;
  int d, n = bl->num > bl->visible ? bl->visible + 1 : bl->num, h, v;

  if (info) {
    bl->x = info->x;
    bl->y = info->y;
    d = info->direction;
  } else {
    bl->x = bl->y = 0;
    d = DIRECTION_LEFT;
  }

  /* bl exists, so bl->widgets[0] exists, too. */
  bl->bw = h = XITK_HV_H (bl->widgets[0]->size);
  bl->bh = v = XITK_HV_V (bl->widgets[0]->size);

  switch (d) {
    case DIRECTION_UP:
      bl->dx = 0;
      bl->dy = -v - 1;
      XITK_HV_H (bl->w.pos) = bl->x;
      XITK_HV_V (bl->w.pos) = bl->y - (n - 1) * (v + 1);
      XITK_HV_H (bl->w.size) = h;
      XITK_HV_V (bl->w.size) = n * (v + 1) - 1;
      break;
    case DIRECTION_DOWN:
      bl->dx = 0;
      bl->dy = v + 1;
      XITK_HV_H (bl->w.pos) = bl->x;
      XITK_HV_V (bl->w.pos) = bl->y;
      XITK_HV_H (bl->w.size) = h;
      XITK_HV_V (bl->w.size) = n * (v + 1) - 1;
      break;
    case DIRECTION_LEFT:
    default:
      bl->dx = -h - 1;
      bl->dy = 0;
      XITK_HV_H (bl->w.pos) = bl->x - (n - 1) * (h + 1);
      XITK_HV_V (bl->w.pos) = bl->y;
      XITK_HV_H (bl->w.size) = n * (h + 1) - 1;
      XITK_HV_V (bl->w.size) = v;
      break;
    case DIRECTION_RIGHT:
      bl->dx = h + 1;
      bl->dy = 0;
      XITK_HV_H (bl->w.pos) = bl->x;
      XITK_HV_V (bl->w.pos) = bl->y;
      XITK_HV_H (bl->w.size) = n * (h + 1) - 1;
      XITK_HV_V (bl->w.size) = v;
      break;
  }
}

static void xitk_button_list_new_skin (xitk_button_list_t *bl, const widget_event_t *event) {
  const xitk_skin_element_info_t *info;
  int i, max;

  xitk_button_list_remove (bl);
  /* relay new skin to hidden ones */
  for (i = 0; i < bl->first; i++) {
    bl->widgets[i]->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    bl->widgets[i]->event (bl->widgets[i], event);
    bl->widgets[i]->state |= XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE;
  }
  for (i = bl->first + bl->visible; i < bl->num; i++) {
    bl->widgets[i]->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    bl->widgets[i]->event (bl->widgets[i], event);
    bl->widgets[i]->state |= XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE;
  }
  for (i = bl->num; i < bl->last; i++) {
    bl->widgets[i]->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    bl->widgets[i]->event (bl->widgets[i], event);
    bl->widgets[i]->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    bl->widgets[i]->state |= XITK_WIDGET_STATE_VISIBLE;
  }
  if ((bl->num <= bl->visible) && bl->swap) {
    bl->swap->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
    bl->swap->event (bl->swap, event);
    bl->swap->state |= XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ENABLE;
  }

  bl->skin_config = event->skonfig;
  info = xitk_skin_get_info (bl->skin_config, bl->w.skin_element_name);
  max = info ? info->max_buttons : 0;
  if (max <= 0)
    max = 10000;
  if (bl->num > max) {
    max -= 1;
  } else {
    max = bl->num;
  }
  bl->visible = max;

  max = (bl->num + bl->visible - 1) / bl->visible * bl->visible;
  if (max > MAX_BUTTONS)
    max = MAX_BUTTONS;
  if (max > bl->last) {
    /* more "   " */
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = bl->w.wl,
        .group = &bl->w,
        .skin_element_name = bl->w.skin_element_name,
        .add_state = XITK_WIDGET_STATE_VISIBLE,
      },
      .button_type    = CLICK_BUTTON,
      .align          = ALIGN_DEFAULT,
      .label          = "",
    };

    for (i = bl->last; i < max; i++) {
      bl->widgets[i] = xitk_labelbutton_create (&lb, bl->skin_config);
      if (!bl->widgets[i])
        break;
      bl->widgets[i]->type |= bl->widget_type_flags;
      xitk_widget_set_parent (bl->widgets[i], NULL);
    }
    bl->last = i;
  } else if (max < bl->last) {
    /* less "   " */
    xitk_widgets_delete (bl->widgets + max, bl->last - max);
    bl->last = max;
  }

  xitk_button_list_hull (bl, info);

  bl->first = 0;
  xitk_button_list_add (bl);

  bl->flags |= 1;
}

static int xitk_button_list_event (xitk_widget_t *w, const widget_event_t *event) {
  xitk_button_list_t *bl = (xitk_button_list_t *)w;

  if (event && bl && ((bl->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_BUTTON_LIST)) {
    switch (event->type) {
      case WIDGET_EVENT_CHANGE_SKIN:
        xitk_button_list_new_skin (bl, event);
        return 0;
      case WIDGET_EVENT_DESTROY:
        if (bl->still_here)
          *bl->still_here = 0;
        xitk_widgets_delete (bl->widgets, bl->last);
        xitk_widgets_delete (&bl->swap, 1);
        return 0;
      case WIDGET_EVENT_ENABLE:
        {
          int e, m;
          m = (bl->w.state & XITK_WIDGET_STATE_ENABLE) ? 1 : 0;
          if ((bl->flags & 1) == m)
            break;
          bl->flags ^= 1;
          m = m ? ~0u : 0;
          e = bl->first + bl->visible;
          if (e > bl->num)
            e = bl->num;
          xitk_widgets_state (bl->widgets + bl->first, e - bl->first, XITK_WIDGET_STATE_ENABLE, m);
          if (bl->num > bl->visible)
            xitk_widgets_state (&bl->swap, 1, XITK_WIDGET_STATE_ENABLE, m);
        }
        break;
      case WIDGET_EVENT_TIPS_TIMEOUT:
        xitk_set_widget_tips_and_timeout (bl->swap, bl->w.tips_string, event->tips_timeout);
        break;
      default: ;
    }
  }
  return 0;
}

xitk_widget_t *xitk_button_list_new (const xitk_button_list_widget_t *nbl, xitk_skin_config_t *skin_config) {
  xitk_button_list_t *bl;
  const xitk_skin_element_info_t *info;
  int i, max;

  if (!nbl)
    return NULL;
  if (!nbl->names)
    return NULL;
  for (i = 0; nbl->names[i]; i++) ;
  if (i == 0)
    return NULL;
  if (i > MAX_BUTTONS)
    i = MAX_BUTTONS;

  bl = (xitk_button_list_t *)xitk_widget_new (&nbl->nw, sizeof (*bl));
  if (!bl)
    return NULL;
  bl->skin_config = skin_config;
  bl->flags = 1;

  bl->widget_type_flags = nbl->widget_type_flags | WIDGET_GROUP_BUTTON_LIST;
  bl->callback = nbl->callback;
  bl->still_here = NULL;
  bl->select_limit = nbl->select_limit;
  bl->last_i = -1;

  info = xitk_skin_get_info (skin_config, bl->w.skin_element_name);
  max = info ? info->max_buttons : 0;
  if (max <= 0)
    max = 10000;
  if (max == 1) {
    i = 1;
  } else if (i > max) {
    max -= 1;
  } else {
    max = i;
  }
  bl->num = i;
  bl->visible = max;

  bl->last = (bl->num + bl->visible - 1) / bl->visible * bl->visible;
  if (bl->last > MAX_BUTTONS)
    bl->last = MAX_BUTTONS;

  bl->first = 0;
  bl->w.type = WIDGET_GROUP | WIDGET_TYPE_BUTTON_LIST;
  xitk_widget_state_from_info (&bl->w, info);
  bl->w.event = xitk_button_list_event;

  {
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = bl->w.wl,
        .group = &bl->w,
        .skin_element_name = bl->w.skin_element_name,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE
      },
      .button_type = (bl->select_limit > 0) ? RADIO_BUTTON : CLICK_BUTTON,
      .align       = ALIGN_DEFAULT,
      .callback    = xitk_button_list_click
    };
    int n = bl->num > 0 ? bl->num : 1;

    for (i = 0; i < MAX_BUTTONS; i++)
      bl->helipad[i] = i;

    /* "101" */
    for (i = 0; i < n; i++) {
      lb.label       = nbl->names[i];
      lb.nw.tips     = nbl->tips[i];
      lb.nw.userdata = bl->helipad + i;
      bl->widgets[i] = xitk_labelbutton_create (&lb, skin_config);
      if (!bl->widgets[i])
        break;
      bl->widgets[i]->type |= bl->widget_type_flags;
      xitk_widget_set_parent (bl->widgets[i], NULL);
    }
    if (!i) {
      free (bl);
      return NULL;
    }
    bl->num = i;

    /* "   " */
    lb.nw.add_state = XITK_WIDGET_STATE_VISIBLE;
    lb.nw.userdata = NULL;
    lb.button_type = CLICK_BUTTON;
    lb.label = "";
    lb.nw.tips = NULL;
    lb.callback = NULL;
    for (; i < bl->last; i++) {
      bl->widgets[i] = xitk_labelbutton_create (&lb, skin_config);
      if (!bl->widgets[i])
        break;
      bl->widgets[i]->type |= bl->widget_type_flags;
      xitk_widget_set_parent (bl->widgets[i], NULL);
    }
    bl->last = i;

    /* "..." */
    lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
    lb.nw.userdata = bl;
    lb.callback = xitk_button_list_swap;
    lb.label    = _("...");
    lb.nw.tips  = nbl->nw.tips;
    bl->swap    = xitk_labelbutton_create (&lb, skin_config);
    if (bl->swap) {
      bl->swap->type |= bl->widget_type_flags;
      xitk_widget_set_parent (bl->swap, NULL);
    }
  }

  bl->first = 0;
  xitk_button_list_hull (bl, info);
  xitk_button_list_add (bl);

  _xitk_new_widget_apply (&nbl->nw, &bl->w);

  return &bl->w;
}

xitk_widget_t *xitk_button_list_find (xitk_widget_t *w, const char *name) {
  xitk_button_list_t *bl = (xitk_button_list_t *)w;
  int i;
  if (!bl || !name)
    return NULL;
  if ((bl->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_BUTTON_LIST)
    return NULL;
  for (i = 0; i < bl->num; i++) {
    const char *p = xitk_labelbutton_get_label (bl->widgets[i]);
    if (p && !strcasecmp (p, name)) {
      bl->last_i = i;
      if (bl->select_limit > 0)
        xitk_widgets_state (bl->widgets + i, 1, XITK_WIDGET_STATE_ON, ~0u);
      return bl->widgets[i];
    }
  }
  return NULL;
}
