/*
 *----------------------------------------------------------------------
 *
 * PROGRAM	 : gksmc
 *
 * FILE		 : annexe_10.c
 *
 * CONTENTS	 : Routines to read items from an Annex E metafile which
 *	           are groups E.10 and E.11 of the Annex - items for
 *		   segment manipulation and items for segment attributes
 *		   respectively.
 *
 * ANNEX E ITEMS : Close Segment
 *		   Rename Segment
 *		   Set Segment Transformation Matrix
 *		   Set Segment Priority
 *
 * GLOBALS USED  : MF_infile, MF_outfile, Encoding
 *
 * DATE		 : 26th April 1988
 *
 *----------------------------------------------------------------------
 */

#include <stdio.h>
#include "annexe.h"
#include "cgm.h"
#include "defns.h"
#include "tables.h"

extern FILE *MF_infile,
       *MF_outfile;


/*
 *-------------------------------------------------------------------------
 * AEset_seg_enum_param:
 *	Used for all segment items which set a segment attribute. The first
 * parameter is a segment identifier, the second is the attribute setting
 * (enumerated). The procedure's parameter gives the row of the enumeration
 * table to be used, which is referenced by the second parameter from the
 * Annex E item.
 * Annex E items : Segment Visibility, Segment Highlighting,
 *		   Segment Detectability
 *-------------------------------------------------------------------------
 */
AEset_seg_attribute(enum_no)
	int     enum_no;
{
	int     seg_name,
	        param;

	read_string_int(MF_infile, &seg_name, DEFAULT);
	read_string_int(MF_infile, &param, DEFAULT);
	write_item_name(MF_outfile, OP_TABLE, DEFAULT);
	write_separator(MF_outfile, SOFTSEP);
	write_int(MF_outfile, seg_name);
	write_separator(MF_outfile, SEP);
	write_enum_value(MF_outfile, enum_no, param);
}


/*
 *-------------------------------------------------------------------------
 * AEclose_seg:
 *	The Annex E item Close Segment translates to the CGM item
 * End Segment.
 *-------------------------------------------------------------------------
 */
AEclose_seg()
{
	write_item_name(MF_outfile, OP_TABLE, DEFAULT);
}


/*
 *-------------------------------------------------------------------------
 * AErename_seg:
 *	Rename segment is the same in Annex E and CGM.
 *-------------------------------------------------------------------------
 */
AErename_seg()
{
	int     old_seg_name,
	        new_seg_name;

	read_string_int(MF_infile, &old_seg_name, DEFAULT);
	read_string_int(MF_infile, &new_seg_name, DEFAULT);
	write_item_name(MF_outfile, OP_TABLE, DEFAULT);
	write_separator(MF_outfile, SOFTSEP);
	write_int(MF_outfile, old_seg_name);
	write_separator(MF_outfile, SEP);
	write_int(MF_outfile, new_seg_name);
}


/*
 *-------------------------------------------------------------------------
 * AEset_seg_trans:
 *	The Set Segment Transformation item translates to the Segment
 * Transform item in CGM, the only difference being the order in which the
 * matrix values are given. The Annex E matrix is given :-
 *	T[1,1],  T[1,2],  T[1,3],  T[2,1],  T[2,2],  T[2,3]
 * The Clear Text version of the CGM item uses this order, but the
 * character encoding uses the order :-
 *	T[1,1],  T[2,1],  T[1,2],  T[2,2],  T[1,3],  T[2,3]
 *-------------------------------------------------------------------------
 */
AEset_seg_trans()
{
	int     seg_name,
	        i;
	double  trans_matrix[6];	/* 2*3 matrix, in row-major order */

	read_string_int(MF_infile, &seg_name, DEFAULT);
	read_real_list(MF_infile, trans_matrix, 6);
	write_item_name(MF_outfile, OP_TABLE, DEFAULT);
	if (Encoding == CLEAR_TEXT)
	{
		write_separator(MF_outfile, SOFTSEP);
		write_int(MF_outfile, seg_name);
		for (i = 0; i < 2; i++)
		{
			write_separator(MF_outfile, SEP);
			write_real(MF_outfile, trans_matrix[i * 3]);
			write_separator(MF_outfile, SEP);
			write_real(MF_outfile, trans_matrix[i * 3 + 1]);
			write_separator(MF_outfile, SEP);
			write_vdc(MF_outfile, trans_matrix[i * 3 + 2]);
		}
	}
	else
	{
		write_int(MF_outfile, seg_name);
		write_real(MF_outfile, trans_matrix[0]);
		write_real(MF_outfile, trans_matrix[3]);
		write_real(MF_outfile, trans_matrix[1]);
		write_real(MF_outfile, trans_matrix[4]);
		write_vdc(MF_outfile, trans_matrix[2]);
		write_vdc(MF_outfile, trans_matrix[5]);
	}
}


/*
 *-------------------------------------------------------------------------
 * AEset_seg_pri:
 *	The Set Segment Priority item translates to the two CGM items
 * Segment Display Priority and Segment Pick Priority. The Annex E item
 * has a real value in the range 0.0 - 1.0 for the priority, whilst the
 * CGM items expect an integer value. The real value is mapped on to the
 * current integer precision range and used for both CGM items.
 *-------------------------------------------------------------------------
 */
AEset_seg_pri()
{
	int     seg_name,
	        int_seg_pri;
	double  real_seg_pri;

	read_string_int(MF_infile, &seg_name, DEFAULT);
	read_string_real(MF_infile, &real_seg_pri, DEFAULT);
	if (real_seg_pri > 1.0)
		write_error(19);
	int_seg_pri = real_seg_pri * MaxInt;
	write_item_name(MF_outfile, OP_TABLE, DEFAULT);	/* display priority */
	write_separator(MF_outfile, SOFTSEP);
	write_int(MF_outfile, seg_name);
	write_separator(MF_outfile, SEP);
	write_int(MF_outfile, int_seg_pri);
	write_separator(MF_outfile, TERMINATOR);
	write_item_name(MF_outfile, CGM_TABLE, SEGPICKPRI);
	write_separator(MF_outfile, SOFTSEP);
	write_int(MF_outfile, seg_name);
	write_separator(MF_outfile, SEP);
	write_int(MF_outfile, int_seg_pri);
}
