/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmInfo.c

  (c) Simon Marlow 1990-92
  (c) Albert Graef 1994
  modified 2004,2005,2006,2007 by Bernhard R. Link (see Changelog)

  Routines for creating and initialising the info window.
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <xfmconfig.h>

#include <pwd.h>
#include <time.h>
#include <grp.h>
#include <assert.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw3d/Label.h>
#include <X11/Xaw3d/Box.h>
#include <X11/Xaw3d/Form.h>

#include "global.h"

#include "Am.h"
#include "Fm.h"

#define LABEL_WIDTH 200
#define PADDING 30

#define I_NAME 0
#define I_LENGTH 1
#define I_OWNER 2
#define I_GROUP 3
#define I_ACCESS 4
#define I_TYPE 5
#define I_CONTENT 6
#define I_LINK 7
#define I_MODIFY_T 8
#define I_STATUS_T 9
#define N_ITEMS 10

/*---------------------------------------------------------------------------
  DATA
---------------------------------------------------------------------------*/

typedef struct InfoData {
  Widget shell;
  Widget items[N_ITEMS];
} InfoData;

/*---------------------------------------------------------------------------
  Strings to go in labels
---------------------------------------------------------------------------*/

static const char *labels[] = {
  "Name",
  "Length",
  "Owner",
  "Group",
  "Access Permissions",
  "Type",
  "Content-Type",
  "Symbolic Link To",
  "Last Modification",
  "Last Status Change"
  };

/*---------------------------------------------------------------------------
  Widget argument lists
---------------------------------------------------------------------------*/

static Arg *form_args = NULL;

static Arg left_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNlabel, (XtArgVal) NULL },
  { XtNwidth, (XtArgVal) 0 },
  { XtNjustify, XtJustifyRight },
  { XtNfont, (XtArgVal) NULL },
  { XtNresize, False },
  { XtNinternalWidth, (XtArgVal) 0 },
  { XtNhorizDistance, (XtArgVal) PADDING },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainLeft }
};

static Arg right_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNlabel, (XtArgVal) NULL },
  { XtNwidth, (XtArgVal) LABEL_WIDTH },
  { XtNjustify, XtJustifyLeft },
  { XtNfont, (XtArgVal) NULL },
  { XtNresize, False },
  { XtNinternalWidth, (XtArgVal) 0 },
  { XtNhorizDistance, (XtArgVal) PADDING },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainRight }
};

static Arg button_box_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainLeft }
};

static Arg shell_args[] = {
  { XtNtitle, (XtArgVal) "File Information" },
  { XtNtransientFor, (XtArgVal) NULL },
};

/*---------------------------------------------------------------------------
  PRIVATE FUNCTIONS
---------------------------------------------------------------------------*/

static void infoOkCb(UNUSED(Widget w), void *client_data,
              UNUSED(void *call_data))
{
  freeze = False;
  XtPopdown((Widget)client_data);
}

static void infoDestroyedCb(UNUSED(Widget w), void *client_data,
              UNUSED(void *call_data))
{
  XtFree(client_data);
}

/*---------------------------------------------------------------------------
  Button data
---------------------------------------------------------------------------*/

static const ButtonRec info_buttons[] = {
  { "cancel", "Cancel", infoOkCb }
};

/*---------------------------------------------------------------------------
  PUBLIC FUNCTIONS
---------------------------------------------------------------------------*/

static void createInfoPopup(InfoData **out, FileWindowRec *fw)
{
  int i;
  Widget form, w;
  Dimension left_width = 0;
  InfoData *info;

  info = (InfoData*)XtMalloc(sizeof(InfoData));

  /* create shell */
  shell_args[1].value = (XtArgVal) fw->shell;
  info->shell = XtCreatePopupShell("info", transientShellWidgetClass,
				   aw.shell, shell_args, XtNumber(shell_args));

  /* create outer form */
  form = XtCreateManagedWidget("form", formWidgetClass, info->shell,
			       form_args, XtNumber(form_args) );

  left_args[5].value = (XtArgVal) resources.bold_font;
  right_args[5].value = (XtArgVal) resources.label_font;

  for (i=0; i<N_ITEMS; i++) {
    Dimension l;
    l = XTextWidth(resources.bold_font, labels[i], strlen(labels[i]));
    if (l > left_width)
      left_width = l;
  }

  left_args[3].value = (XtArgVal) left_width;

  /* create all the smaller labels */
  w = NULL;
  for (i=0; i<N_ITEMS; i++) {
    left_args[1].value = (XtArgVal) w;
    right_args[1].value = (XtArgVal) w;
    left_args[2].value = (XtArgVal) labels[i];
    w = XtCreateManagedWidget("leftlabel", labelWidgetClass, form,
			      left_args, XtNumber(left_args));

    right_args[0].value = (XtArgVal) w;
    info->items[i] = XtCreateManagedWidget("rightlabel", labelWidgetClass, form,
					  right_args, XtNumber(right_args));
  }

  button_box_args[1].value = (XtArgVal) w;
  w = XtCreateManagedWidget("button box", boxWidgetClass, form,
			    button_box_args, XtNumber(button_box_args));
  createButtons(info_buttons, XtNumber(info_buttons), w, info->shell);

#ifdef ENHANCE_POP_ACCEL
  XtInstallAllAccelerators(form, form);
  XtInstallAllAccelerators(info->shell, form);
#endif

  XtRealizeWidget(info->shell);
  setWMProps(info->shell);
  XtAddCallback(info->shell, XtNdestroyCallback, infoDestroyedCb, info);

  *out = info;
}

/*---------------------------------------------------------------------------*/

void infoPopup(UNUSED(Widget w), FileWindowRec *fw,
               UNUSED(void *call_data))
{
  char s[11], link[MAXPATHLEN];
  const char *type;
  struct passwd *pw;
  struct group *gp;
  const FileRec *file = NULL;
  int i;

  assert( fw != NULL );

  if (!fw->n_selections) return;

  if (fw->info == NULL)
    createInfoPopup(&fw->info, fw);

  for (i=0;; i++)
    if (fw->files[i]->selected) {
      file = fw->files[i];
      break;
    }

  assert( file != NULL );

  XtVaSetValues(fw->info->items[I_NAME], XtNlabel, (XtArgVal) file->name,
		NULL);

  sprintf(s, "%lu", (unsigned long) file->stats.st_size);
  XtVaSetValues(fw->info->items[I_LENGTH], XtNlabel, (XtArgVal) s, NULL);

  if (!(pw = getpwuid(file->stats.st_uid)))
    sprintf(s, "%lu", (unsigned long) file->stats.st_uid);
  XtVaSetValues(fw->info->items[I_OWNER], XtNlabel, (XtArgVal) (pw?pw->pw_name:s),
		NULL);

  if (!(gp = getgrgid(file->stats.st_gid)))
    sprintf(s, "%lu", (unsigned long) file->stats.st_gid);
  XtVaSetValues(fw->info->items[I_GROUP], XtNlabel, (XtArgVal) (gp?gp->gr_name:s),
		NULL);

  makePermissionsString(s, file->stats.st_mode);
  XtVaSetValues(fw->info->items[I_ACCESS], XtNlabel, (XtArgVal) s, NULL);

  if(file->info.symlink) {
    type = "Symbolic link";
    if (chdir(fw->directory)) {
      sysError("System error:");
      return;
    }

    i = readlink(file->name, link, MAXPATHLEN);
    if (i == -1) {
      sysError("Error reading link:");
      return;
    }
    link[i] = '\0';

    XtVaSetValues(fw->info->items[I_LINK], XtNlabel, (XtArgVal) link, NULL);
  } else {
    mode_t mode = file->stats.st_mode;

    /* TODO: how often is this implemented? Look if this can be removed here*/
    if (S_ISDIR(mode))
      type = "Directory";
    else if (S_ISCHR(mode))
      type = "Character special file";
    else if(S_ISBLK(mode))
      type = "Block special file";
    else if(S_ISSOCK(mode))
      type = "Socket";
    else if(S_ISFIFO(mode))
      type = "Pipe or FIFO special file";
    else
      type = "Ordinary file";

    XtVaSetValues(fw->info->items[I_LINK], XtNlabel, (XtArgVal) "-", NULL);
  }

  XtVaSetValues(fw->info->items[I_TYPE], XtNlabel, (XtArgVal) type, NULL);
  type = file->magic_type;
  XtVaSetValues(fw->info->items[I_CONTENT], XtNlabel, (XtArgVal) type, NULL);

  XtVaSetValues(fw->info->items[I_MODIFY_T], XtNlabel,
		(XtArgVal) ctime(&file->stats.st_mtime), NULL);

  XtVaSetValues(fw->info->items[I_STATUS_T], XtNlabel,
		(XtArgVal) ctime(&file->stats.st_ctime), NULL);

  freeze = True;
  popupByCursor(fw->info->shell, XtGrabExclusive);
}

