#!/usr/bin/perl
#
# Script to convert "x11-fields" file, listing fields for
# X11 dissector, into header files declaring field-index
# values and field definitions for those fields.
#
# Instructions for using this script are in epan/dissectors/README.X11
#
# Copyright 2000, Christophe Tronche <ch.tronche[AT]computer.org>
#
# Wireshark - Network traffic analyzer
# By Gerald Combs <gerald@wireshark.org>
# Copyright 1998 Gerald Combs
#
# SPDX-License-Identifier: GPL-2.0-or-later
#

use File::Spec;

my $srcdir = shift;
die "'$srcdir' is not a directory" unless -d $srcdir;

open(DECL, "> $srcdir/x11-declarations.h") || die;
open(REG, "> $srcdir/x11-register-info.h") || die;

my $script_name = File::Spec->abs2rel ($0,  $srcdir);

sub add_generated_header {
    my ($out) = @_;

    print $out <<eot
/* Do not modify this file. */
/* It was automatically generated by $script_name. */
eot
    ;

    # Add license text
    print $out <<eot
/*
 * Copyright 2000, Christophe Tronche <ch.tronche[AT]computer.org>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald[AT]wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

eot
    ;
}

add_generated_header(DECL);
add_generated_header(REG);

$prefix = '';
$subfieldStringLength = 0;

while(<>) {
    s/#.*$//go;
    next if /^\s*$/o;
    s/^(\s*)//o;
    $subfield = $1;

    if (length $subfield != $subfieldStringLength) {
        if (!length $subfield) {
            $prefix = '';
        } elsif (length $subfield > $subfieldStringLength) {
            $prefix .= "$lastAbbrev.";
        } else {
            $prefix =~ s/^(.*)\.[^\.]+\.$/$1./o;
        }
        $subfieldStringLength = length $subfield;
    }

    @fields = split /\s+/o ;
    if ($fields[0] eq '#') {
        #
        # If the line begins with "#", treat it as a comment, by
        # ignoring it.
        #
        # (We don't support comments at the end of a line; that would
        # require some more pain in our simple parser.)
        #
        next;
    }
    $abbrev = shift @fields;
    $type = shift @fields;
    $lastAbbrev = $abbrev;

    $field = $prefix.$abbrev;

    if ($fields[0] =~ /^\d+$/o) {
        #
        # This is presumably a Boolean bitfield, and this is the number
        # of bits in the parent field.
        #
        $fieldDisplay = shift @fields;
    } else {
        #
        # The next token is the base for the field.
        #
        $fieldDisplay = "BASE_".shift @fields;
    }

    if ($fields[0] eq 'VALS') {
        #
        # It's an enumerated field, with the value_string table having a
        # name based on the field's name.
        #
        shift @fields;
        $fieldStrings = "VALS(${abbrev}_vals)";
        $fieldStrings =~ s/-/_/go;
    } elsif ($fields[0] =~ /^VALS\(/o) {
        #
        # It's an enumerated field, with a specified name for the
        # value_string table.
        #
        $fieldStrings = shift @fields;
        $fieldStrings =~ s/\)/_vals\)/o;
    } else {
        #
        # It's not an enumerated field.
        #
        $fieldStrings = 'NULL';
    }

    if ($fields[0] =~ /^0x/) {
        #
        # The next token looks like a bitmask for a bitfield.
        #
        $mask = shift @fields;
    } else {
        $mask = 0;
    }

    $rest = join(' ', @fields);
    $longName = uc $name;
    $longName = $rest if ($rest);
    # Don't allow empty blurbs
    $longName = $longName eq "" ? "NULL" : "\"$longName\"";

    $variable = $field;
    $variable =~ s/-/_/go;
    $variable =~ s/\./_/go;

    print DECL "static int hf_x11_$variable = -1;\n";

    print REG <<END;
{ &hf_x11_$variable, { "$abbrev", "x11.$field", FT_$type, $fieldDisplay, $fieldStrings, $mask, $longName, HFILL }},
END
}

#
#  Editor modelines
#
#  Local Variables:
#  c-basic-offset: 4
#  tab-width: 8
#  indent-tabs-mode: nil
#  End:
#
#  ex: set shiftwidth=4 tabstop=8 expandtab:
#  :indentSize=4:tabSize=8:noTabs=true:
#
