/**
 * GraphPanel.java
 *
 * Created on Jul 3, 2006
 */
package components;

import events.MouseList;
import graphe.GraphImg;
import graphe.GraphInfo;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Shape;
import java.awt.event.InputEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;

import javax.swing.JPanel;


import maths.CoordSyst;
import java.awt.Color;

/**
 * Cette classe etend <CODE>JPanel</CODE> pour dessiner le graphe et eventuellement
 * au plus deux {@link graphe.GraphInfo informations graphiques}.
 * 
 * @author Nicolae
 */
public class GraphPanel extends JPanel {

	/**
	 * Rien.
	 */
	private static final long serialVersionUID = 1L;
	
	private GraphImg img;  //  @jve:decl-index=0:
	private GraphInfo gr1;  //  @jve:decl-index=0:
	private GraphInfo gr2;
	private BufferedImage image;
	private MouseList mouse;
	private int oldW = 0;
	private int oldH = 0;

	/**
	 * This is the default constructor
	 */
	public GraphPanel() {
		super();
		initialize();
	}
	
	/**
	 * Initialisation - les events.
	 */
	private void initialize() {        
		this.addMouseListener(new MouseAdapter() {   
			public void mouseExited(java.awt.event.MouseEvent e) {    
				if(mouse != null)
					mouse.exit();
			}
			
        	public void mouseClicked(MouseEvent e) {
        		if(mouse != null && img != null) {
        			int x = e.getX();
        			int y = e.getY();
        			CoordSyst cs = img.getCoords();
        			if(x >= 0 && x < cs.getWidth() && y >= 0 && y < cs.getHeigth()) {
        				if((e.getModifiers() & MouseEvent.BUTTON1_MASK) != 0)
        					if((e.getModifiers() & InputEvent.CTRL_MASK) != 0)
            					mouse.rclick(cs.getRe(x), cs.getIm(y));
        					else
        						mouse.click(cs.getRe(x), cs.getIm(y));
        				else if((e.getModifiers() & MouseEvent.BUTTON3_MASK) != 0)
        					mouse.rclick(cs.getRe(x), cs.getIm(y));
        			}
        		}
        	}
        });
		
		this.addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {
			public void mouseMoved(java.awt.event.MouseEvent e) {
        		if(mouse != null && img != null) {
        			int x = e.getX();
        			int y = e.getY();
        			CoordSyst cs = img.getCoords();
        			if(x >= 0 && x < cs.getWidth() && y >= 0 && y < cs.getHeigth())
        				mouse.move(cs.getRe(x), cs.getIm(y));  
        			else
        				mouse.exit();
        		}
			}
		});
		this.addComponentListener(new java.awt.event.ComponentAdapter() {   
			public void componentShown(java.awt.event.ComponentEvent e) {    
				create(false);
			}
			public void componentResized(java.awt.event.ComponentEvent e) {
				create(false);
			}
		});
		this.setSize(new java.awt.Dimension(74,58));  // Generated
		this.setBackground(Color.white);
	}
	
	private void create(boolean force) {
		int w = getWidth();
		int h = getHeight();
		if(w > 0 && h > 0 && img != null && img.getCoords() != null 
				 && (w != oldW || h != oldH || force)) {
			oldW = w;
			oldH = h;
			
			CoordSyst cs = img.getCoords();
			double rat = cs.getRatio();
			if(rat <= 0)  // on utilise toute la surface
				cs.setPix(w, h);
			else
				if(w < (int)(rat * h)) // on utilise la largeur
					cs.setPix(w, (int) (w / rat));
				else  // on utilise la hauteur
					cs.setPix((int) (h * rat), h);
			
			img.setCoords(cs); // pour calculer le graphe 
			
			repaint();
		}
	}

	/**
	 * Etablit le graphe a representer.
	 *
	 * @param grIm le graphe.
	 */
	public void setGraph(GraphImg grIm) {
		img = grIm;
		create(true);
	}	
	
	/**
	 * Etablit les deux objet graphiques a dessiner sur le graphe.
	 *
	 * @param g1 info graphique.
	 * @param g2 info graphique.
	 */
	public void setGrInfo(GraphInfo g1, GraphInfo g2) {
		gr1 = g1;
		gr2 = g2;
		repaint();
	}
	
	
	/**
	 * Etablit l'ecouteur d'evenement de la souris.
	 *
	 * @param ml l'ecouteur.
	 */
	public void setMouseList(MouseList ml) {
		mouse = ml;
	}	
	
	protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		
		
		if(img != null && img.getImage() != null) {
			Shape clip = g.getClip();
			if(img.getCoords() != null && img.getCoords().isValid())
				g.setClip(0, 0, img.getCoords().getWidth(), img.getCoords().getHeigth());

			Graphics2D g2 = (Graphics2D) g;
			
			if(image != null)
				g2.drawImage(image, null, 0, 0);
				
			if(gr1 != null && gr1.isUnder())
				gr1.paint(g);
			
			if(gr2 != null && gr2.isUnder())
				gr2.paint(g);
			
			g2.drawImage(img.getImage(), null, 0, 0);
			
			if(gr1 != null && ! gr1.isUnder())
				gr1.paint(g);
			
			if(gr2 != null && ! gr2.isUnder())
				gr2.paint(g);
			
			g.setClip(clip);
		}
	}

	public String getToolTipText(MouseEvent event) {
		return getCoords(event.getPoint());
	}

	private String getCoords(Point point) {
		double x = point.x;
		double y = point.y;
		return "(" + x + ", " + y + ")";
	}

	/**
	 * Renvoie la valeur de la propriete <code>image</code>.
	 * Cette propiete represente l'image externe a dessiner en dessous du graphe.
	 *
	 * @return la valeur de la propriete <code>image</code>
	 */
	public BufferedImage getImage() {
		return image;
	}

	/**
	 * Etablit la valeur de la propriete <code>image</code>.
	 * Cette propritete represente l'image externe a dessiner en dessous du graphe.
	 *
	 * @param image la nouvelle valeur de la propriete <code>image</code>
	 */
	public void setImage(BufferedImage image) {
		this.image = image;
	}
}  //  @jve:decl-index=0:visual-constraint="10,2"
