# -*- coding: utf-8 -*-

# Copyright(C) 2011      Gabriel Kerneis
# Copyright(C) 2010-2011 Jocelyn Jaubert
#
# This file is part of weboob.
#
# weboob is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# weboob is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with weboob. If not, see <http://www.gnu.org/licenses/>.


import re
from decimal import Decimal

from weboob.capabilities.bank import Account
from weboob.deprecated.browser import Page
from weboob.tools.capabilities.bank.transactions import FrenchTransaction


class AccountsList(Page):
    ACCOUNT_TYPES = {u'banque':        Account.TYPE_CHECKING,
                     u'épargne':       Account.TYPE_SAVINGS,
                     u'crédit':        Account.TYPE_LOAN,
                     u'assurance vie': Account.TYPE_LIFE_INSURANCE,
                     u'bourse':        Account.TYPE_MARKET,
                    }

    def get_list(self):
        blocks = self.document.xpath('//div[@id="synthese-list"]//div[@class="block"]')
        for div in blocks:
            block_title = ''.join(div.xpath('.//span[@class="title"]/a/text()')).lower().strip()
            if block_title == 'assurance':
                # Only promotional fake accounts...
                continue

            for tr in div.getiterator('tr'):
                account = Account()
                account.id = None
                account._link_id = None
                account.type = self.ACCOUNT_TYPES.get(block_title, Account.TYPE_UNKNOWN)
                for td in tr.getiterator('td'):
                    if td.get('class', '') == 'account-cb':
                        try:
                            a = td.xpath('./*/a[@class="gras"]')[0]
                        except IndexError:
                            # ignore account
                            break
                        account.type = Account.TYPE_CARD
                        account.label, account.id = [s.strip() for s in self.parser.tocleanstring(td).rsplit('-', 1)]
                        # Those cards immediately debits transactions on the linked checking account, so we ignore those accounts to avoid doubled transactions.
                        if 'DEBIT IMMEDIAT' in account.label:
                            break

                        # Sometimes there is text after the card number:
                        #   <a class="gras" href="/comptes/banque/cartes/index.phtml?CompteCourant=ulietuliedtlueditluedt&amp;currentCB=ruisecruicertuci">
                        #   CARTE PREMIER                            </a>
                        #   <br>MACHIN BIDULE TRUC - 1111********1111
                        #
                        #   <br>
                        #   <strong><a href="/aide/faq/index.phtml?document_id=472">Son échéance est le <span style="color:#ff8400; font-weight:bold;">31/03/2015</span>.<br>En savoir plus</a></strong>
                        # So we have to remove all the shit after it.
                        account.id = account.id.split(' ')[0]

                        try:
                            account._link_id = td.xpath('.//a')[0].get('href')
                            # Try to use account._link_id for account.id to prevent duplicate accounts
                            currentCB = re.search('currentCB=(.*)', account._link_id)
                            if currentCB:
                                account.id = currentCB.group(1)
                        except KeyError:
                            pass

                    elif td.get('class', '') == 'account-name':
                        try:
                            span = td.xpath('./span[@class="label"]')[0]
                        except IndexError:
                            # ignore account
                            break
                        account.label = self.parser.tocleanstring(span)
                        account.id = self.parser.tocleanstring(td).rsplit('-', 1)[-1].strip()
                        try:
                            account._link_id = td.xpath('.//a[not(./small[contains(@class, "icon")])]')[0].get('href')
                            if 'souscription' in account._link_id:
                                account._link_id = td.xpath('.//a')[1].get('href')
                            account._detail_url = account._link_id
                        except KeyError:
                            pass

                    elif td.get('class', '') == 'account-more-actions':
                        for a in td.getiterator('a'):
                            # For normal account, two "account-more-actions"
                            # One for the account, one for the credit card. Take the good one
                            if 'href' in a.attrib and "mouvements.phtml" in a.get('href') and "/cartes/" not in a.get('href'):
                                account._link_id = a.get('href')

                    elif td.get('class', '') == 'account-number':
                        id = td.text
                        id = id.strip(u' \n\t')
                        account.id = id

                    elif td.get('class', '') == 'account-total':
                        span = td.find('span')
                        if span is None:
                            balance = td.text
                        else:
                            balance = span.text
                        account.currency = account.get_currency(balance)
                        balance = FrenchTransaction.clean_amount(balance)
                        if balance != "":
                            account.balance = Decimal(balance)
                        else:
                            account.balance = Decimal(0)
                else:
                    # because of some weird useless <tr>
                    if account.id is not None and (not account._link_id or 'moneycenter' not in account._link_id):
                        yield account
