/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "UNSConfig.h"
#include "ConfigFile.h"

#ifdef _WIN32
	#include "ConfigWinInfo.h"
#else
	#include "ConfigLinuxInfo.h"
#endif

#define SECOND 1000
#define MINUTE (60 * SECOND)

#include <fstream>
using namespace std;

/*Constants*/
//Constants for configuration parameters names
const string UNSConfig::CHECK_AMT_STATUS_TIMEOUT = "CheckAMTStatusTimeout";
const string UNSConfig::NUM_POLLS_FOR_DISABLED = "NumPollsForDisabled";
const string UNSConfig::GET_POSTURE_INTERVAL = "GetPostureInterval";
const string UNSConfig::TRACE_LOG_ENABLED = "Trace";


#ifdef _DEBUG
const bool UNSConfig::DEFAULT_DEBUG_MODE = false;
const bool UNSConfig::DEFAULT_DEBUG_MODE_AMT_ENABLED = false;
#endif

//Default values for configuration parameters
const unsigned long UNSConfig::DEFAULT_CHECK_AMT_STATUS_TIMEOUT = 10 * SECOND;
const unsigned long UNSConfig::DEFAULT_NUM_POLLS_FOR_DISABLED = 7;
const unsigned long UNSConfig::DEFAULT_GET_POSTURE_INTERVAL = 5 * MINUTE;
const bool UNSConfig::DEFAULT_TRACE_LOG_ENABLED = false;

//Constants for configuration file names
const string UNSConfig::MACHINE_CFG_FILENAME = "IntelAMTUNS.config";

#ifdef _DEBUG
const string UNSConfig::DEBUG_MODE_FILENAME = "IntelAMTUNS.debug.config";
const string UNSConfig::DEBUG_MODE = "DebugMode";
const string UNSConfig::DEBUG_MODE_AMT_ENABLED = "DebugModeAMTEnabled";
#endif


//default ctor
UNSConfig::UNSConfig()
{
	LoadDefaults();
}

void UNSConfig::LoadDefaults()
{
	m_nCheckAMTStatusTimeout = DEFAULT_CHECK_AMT_STATUS_TIMEOUT;
	m_nNumPollsForDisabled = DEFAULT_NUM_POLLS_FOR_DISABLED;
	m_nGetPostureInterval = DEFAULT_GET_POSTURE_INTERVAL;
	m_bTraceLogEnabled = DEFAULT_TRACE_LOG_ENABLED;
}

//Init(): Load the configuration from configuration files.
//Returns true on success, false if either of the files could not be opened (in which
//case the defaults will be used).
bool UNSConfig::Init()
{
	LoadDefaults();
#ifdef _WIN32
	ConfigWinInfo OSInfo;
#else
	ConfigLinuxInfo OSInfo;
#endif

	m_MachineConfigFileName = OSInfo.MachineConfigPrefix() + MACHINE_CFG_FILENAME;
#ifdef _DEBUG
	m_DebugModeConfigFileName = OSInfo.MachineConfigPrefix() + DEBUG_MODE_FILENAME;
#endif

	//If file couldn't be opened, we use the defaults and return false
	return LoadMachineConfig();
}

bool UNSConfig::LoadMachineConfig()
{
	try {
		ConfigFile MachineConfigFile(m_MachineConfigFileName.c_str());

		//For every parameter that is read, if it's not found, its default value is used.
		m_nCheckAMTStatusTimeout = MachineConfigFile.read(CHECK_AMT_STATUS_TIMEOUT, DEFAULT_CHECK_AMT_STATUS_TIMEOUT);
		m_nNumPollsForDisabled = MachineConfigFile.read(NUM_POLLS_FOR_DISABLED, DEFAULT_NUM_POLLS_FOR_DISABLED);
		m_nGetPostureInterval = MachineConfigFile.read(GET_POSTURE_INTERVAL, DEFAULT_GET_POSTURE_INTERVAL);
		m_bTraceLogEnabled = MachineConfigFile.read(TRACE_LOG_ENABLED, DEFAULT_TRACE_LOG_ENABLED);
		return true;
	}
	catch (ConfigFile::file_not_found &)
	{
		return false;
	}
}

unsigned long UNSConfig::CheckAMTStatusTimeout() const
{
	return m_nCheckAMTStatusTimeout;
}

unsigned long UNSConfig::NumPollsForDisabled() const
{
	return m_nNumPollsForDisabled;
}

unsigned long UNSConfig::GetPostureInterval() const
{
	return m_nGetPostureInterval;
}

bool UNSConfig::TraceLogEnabled() const
{
	return m_bTraceLogEnabled;
}

#ifdef _DEBUG
bool UNSConfig::debugMode() const
{
	try {
		ConfigFile DebugModeFile(m_DebugModeConfigFileName);
		return DebugModeFile.read(DEBUG_MODE, DEFAULT_DEBUG_MODE);
	}
	catch (ConfigFile::file_not_found &)
	{
		return DEFAULT_DEBUG_MODE; //File doesn't exist - return default
	}
}

bool UNSConfig::debugModeAMTEnabled() const
{
	try {
		ConfigFile DebugModeFile(m_DebugModeConfigFileName);
		return DebugModeFile.read(DEBUG_MODE_AMT_ENABLED, DEFAULT_DEBUG_MODE_AMT_ENABLED);
	}
	catch (ConfigFile::file_not_found &)
	{
		return DEFAULT_DEBUG_MODE_AMT_ENABLED; //File doesn't exist - return default
	}
}
#endif
