# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Unity Autopilot Test Suite
# Copyright (C) 2012, 2013, 2014 Canonical
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import logging

from unity8.shell import emulators

from autopilot import logging as autopilot_logging
from autopilot.introspection import dbus
from testtools.matchers import MatchesAny, Equals
from ubuntuuitoolkit import emulators as toolkit_emulators


logger = logging.getLogger(__name__)


class Dash(emulators.UnityEmulatorBase):
    """An emulator that understands the Dash."""

    def __init__(self, *args):
        super(Dash, self).__init__(*args)
        self.dash_content_list = self.wait_select_single(
            'QQuickListView', objectName='dashContentList')

    def get_applications_grid(self):
        get_grid = self.get_scope('clickscope').wait_select_single(
            'CardFilterGrid', objectName='local')
        return get_grid

    def get_application_icon(self, text):
        """Returns a 'Tile' icon that has the text 'text' from the application
        grid.

        :param text: String containing the text of the icon to search for.

        """
        app_grid = self.get_applications_grid()
        resp_grid = app_grid.wait_select_single('ResponsiveGridView')
        return resp_grid.select_single('Tile', text=text)

    def get_scope(self, scope_name='clickscope'):
        return self.dash_content_list.select_single(
            'QQuickLoader', scopeId=scope_name)

    @autopilot_logging.log_action(logger.info)
    def open_scope(self, scope_id):
        """Open a dash scope.

        :parameter scope_id: The id of the scope.
        :return: The scope.

        """
        scope_loader = self._get_scope_loader(scope_id)
        if scope_loader.isCurrent:
            logger.info('The scope is already open.')
            return self._get_scope_from_loader(scope_loader)
        else:
            return self._open_scope_scrolling(scope_loader)

    def _get_scope_loader(self, scope_id):
        try:
            return self.dash_content_list.select_single(
                'QQuickLoader', scopeId=scope_id)
        except dbus.StateNotFoundError:
            raise emulators.UnityEmulatorException(
                'No scope found with id {0}'.format(scope_id))

    def _get_scope_from_loader(self, loader):
        return loader.get_children()[0]

    def _open_scope_scrolling(self, scope_loader):
        scroll = self._get_scroll_direction(scope_loader)

        while not scope_loader.isCurrent:
            scroll()
            self.dash_content_list.moving.wait_for(False)

        scope = self._get_scope_from_loader(scope_loader)
        scope.isCurrent.wait_for(True)
        return scope

    def _get_scroll_direction(self, scope_loader):
        current_scope_loader = self.dash_content_list.select_single(
            'QQuickLoader', isCurrent=True)
        if scope_loader.globalRect.x < current_scope_loader.globalRect.x:
            return self._scroll_to_left_scope
        elif scope_loader.globalRect.x > current_scope_loader.globalRect.x:
            return self._scroll_to_right_scope
        else:
            raise emulators.UnityEmulatorException('The scope is already open')

    @autopilot_logging.log_action(logger.info)
    def _scroll_to_left_scope(self):
        original_index = self.dash_content_list.currentIndex
        # Scroll on the border of the page header, because some scopes have
        # contents that can be scrolled horizontally.
        page_header = self._get_page_header()
        border = page_header.select_single('QQuickBorderImage')
        start_x = border.width / 3
        stop_x = border.width / 3 * 2
        start_y = stop_y = border.globalRect.y + border.height / 2
        self.pointing_device.drag(start_x, start_y, stop_x, stop_y)
        self.dash_content_list.currentIndex.wait_for(original_index - 1)

    def _get_page_header(self):
        return self.select_single('PageHeader', objectName='pageHeader')

    @autopilot_logging.log_action(logger.info)
    def _scroll_to_right_scope(self):
        original_index = self.dash_content_list.currentIndex
        # Scroll on the border of the page header, because some scopes have
        # contents that can be scrolled horizontally.
        page_header = self._get_page_header()
        border = page_header.select_single('QQuickBorderImage')
        start_x = border.width / 3 * 2
        stop_x = border.width / 3
        start_y = stop_y = border.globalRect.y + border.height / 2
        self.pointing_device.drag(start_x, start_y, stop_x, stop_y)
        self.dash_content_list.currentIndex.wait_for(original_index + 1)

    def enter_search_query(self, query):
        search_text_field = self._get_search_text_field()
        search_text_field.write(query)
        search_text_field.state.wait_for('idle')

    def _get_search_text_field(self):
        page_header = self._get_page_header()
        search_container = page_header.select_single(
            'QQuickItem', objectName='searchContainer')
        search_container.state.wait_for(
            MatchesAny(Equals('narrowActive'), Equals('active')))
        return search_container.select_single(toolkit_emulators.TextField)


class GenericScopeView(emulators.UnityEmulatorBase):
    """Autopilot emulator for generic scopes."""

    @autopilot_logging.log_action(logger.info)
    def open_preview(self, category, app_name):
        """Open the preview of an application.

        :parameter category: The name of the category where the application is.
        :app_name: The name of the application.
        :return: The opened preview.

        """
        category_element = self._get_category_element(category)
        icon = category_element.select_single('Card', title=app_name)
        # FIXME some categories need a long press in order to see the preview.
        # Some categories do not show previews, like recent apps.
        # --elopio - 2014-1-14
        self.pointing_device.click_object(icon)
        preview_list = self.get_root_instance().wait_select_single(
            'PreviewListView', objectName='dashContentPreviewList')
        preview_list.x.wait_for(0)
        return preview_list.select_single(
            Preview, objectName='preview{}'.format(preview_list.currentIndex))

    def _get_category_element(self, category):
        try:
            return self.wait_select_single(
                'Base', objectName='dashCategory{}'.format(category))
        except dbus.StateNotFoundError:
            raise emulators.UnityEmulatorException(
                'No category found with name {}'.format(category))


class DashApps(GenericScopeView):
    """Autopilot emulator for the applications scope."""

    def get_applications(self, category):
        """Return the list of applications on a category.

        :parameter category: The name of the category.

        """
        category_element = self._get_category_element(category)
        application_cards = category_element.select_many('Card')

        # sort by y, x
        application_cards = sorted(
            application_cards,
            key=lambda card: (card.globalRect.y, card.globalRect.x))

        result = []
        for card in application_cards:
            if card.objectName != 'cardToolCard':
                card_header = card.select_single('CardHeader')
                result.append(card_header.title)
        return result


class Preview(emulators.UnityEmulatorBase):
    """Autopilot custom proxy object for generic previews."""
