/* Copyright (c) 2003-2004 Ecole centrale de Lyon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "fdtd3d.h"
#include "init.h"
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#ifdef HAVE_MPI
#include <mpi.h>
#endif


/* Stop immediately, with an error message. */

void panic(char *message)
{
  fprintf(stderr,"Fatal : %s\n",message);
#ifdef HAVE_MPI
  MPI_Abort(MPI_COMM_WORLD,1);
#endif
  exit(-1);
}

/* Allocate memory with error checking. */

void *myalloc(size_t size)
{
  void *buf;
  buf=malloc(size);
  if(buf==NULL)
    panic("not enough memory for allocation");
  memset(buf,0,size);
  return buf;
}

/* Same for reallocation */

void *myrealloc(void *ptr, size_t size)
{
  void *buf;
  buf=realloc(ptr,size);
  if(buf==NULL)
    panic("not enough memory for reallocation");
  return buf;
}

/* Same for strdup */

char *mystrdup(const char *s)
{
  char *buf;
  buf=strdup(s);
  if(buf==NULL)
    panic("not enough memory for allocation");
  return buf;
}

/* Split the name of a file and the name of the data slice.
   Modifies path to contain only the file name, and return
   the name of the slice. */

char *split_dataname(char *path)
{
  char *foo;
  foo=rindex(path,':');
  if(foo==NULL)
    return NULL;
  foo[0]=0;
  return foo+1;
}

float get_time(void)
{
  struct rusage usage;
  getrusage(RUSAGE_SELF,&usage);
  return usage.ru_utime.tv_sec+usage.ru_utime.tv_usec/1000000.;
}

#ifdef HAVE_MPI
SimulDesc *broadcast_simuldesc(SimulDesc *simul)
{
  int tmp=0;
  int i;

  if(simul)
    tmp=1;
  
  MPI_Bcast(&tmp, 1, MPI_INT, 0, MPI_COMM_WORLD);

  if(tmp==0)
    {
      if(simul)
	free_simuldesc(simul);
      return NULL;
    }
  
  MPI_Bcast(&(simul->space->nx), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->ny), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->nz), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->block_size), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->num_pml), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->delta_x), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->min_index), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->pml_refl), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->space->pml_decay), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  if(simul->space->index==NULL)
    simul->space->index=myalloc(simul->space->nx*simul->space->ny*simul->space->nz*sizeof(double));
  MPI_Bcast(simul->space->index, simul->space->nx*simul->space->ny*simul->space->nz, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  if(simul->space->absor)
    tmp=1;
  MPI_Bcast(&tmp, 1, MPI_INT, 0, MPI_COMM_WORLD);
  if(tmp)
    {
      if(simul->space->absor==NULL)
	simul->space->absor=myalloc(simul->space->nx*simul->space->ny*simul->space->nz*sizeof(double));
      MPI_Bcast(simul->space->absor, simul->space->nx*simul->space->ny*simul->space->nz, MPI_DOUBLE, 0, MPI_COMM_WORLD);
    }
  
  MPI_Bcast(&(simul->simul->delta_t), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->simul->it_max), 1, MPI_INT, 0, MPI_COMM_WORLD);

  MPI_Bcast(&(simul->inj->x), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->inj->y), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->inj->z), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->inj->dir_x), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->inj->dir_y), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->inj->dir_z), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->inj->basefreq), 1, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  if(simul->inj->injection==NULL)
    simul->inj->injection=myalloc(simul->simul->it_max*sizeof(double));
  MPI_Bcast(simul->inj->injection, simul->simul->it_max, MPI_DOUBLE, 0, MPI_COMM_WORLD);
  
  MPI_Bcast(&(simul->output->nb_cartos), 1, MPI_INT, 0, MPI_COMM_WORLD);
  if(simul->output->nb_cartos && simul->output->cartos==NULL)
    simul->output->cartos=myalloc(simul->output->nb_cartos*sizeof(varOutputCarto));
  MPI_Bcast(&(simul->output->nb_poncts), 1, MPI_INT, 0, MPI_COMM_WORLD);
  if(simul->output->nb_poncts && simul->output->poncts==NULL)
    simul->output->poncts=myalloc(simul->output->nb_poncts*sizeof(varOutputPonct));
  MPI_Bcast(&(simul->output->nb_poynts), 1, MPI_INT, 0, MPI_COMM_WORLD);
  MPI_Bcast(&(simul->output->total_loss), 1, MPI_INT, 0, MPI_COMM_WORLD);
  if(simul->output->nb_poynts && simul->output->poynts==NULL)
    simul->output->poynts=myalloc(simul->output->nb_poynts*sizeof(varOutputPoynt));
  
  for(i=0;i<simul->output->nb_cartos;i++)
    {
      MPI_Bcast(&(simul->output->cartos[i].field), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->cartos[i].begin), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->cartos[i].interv), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->cartos[i].end), 1, MPI_INT, 0, MPI_COMM_WORLD);
    }
  for(i=0;i<simul->output->nb_poncts;i++)
    {
      MPI_Bcast(&(simul->output->poncts[i].field), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poncts[i].x), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poncts[i].y), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poncts[i].z), 1, MPI_INT, 0, MPI_COMM_WORLD);
      if(simul->output->poncts[i].valx==NULL)
	simul->output->poncts[i].valx=myalloc(simul->simul->it_max*sizeof(double));
      if(simul->output->poncts[i].valy==NULL)
	simul->output->poncts[i].valy=myalloc(simul->simul->it_max*sizeof(double));
      if(simul->output->poncts[i].valz==NULL)
	simul->output->poncts[i].valz=myalloc(simul->simul->it_max*sizeof(double));
    }
  for(i=0;i<simul->output->nb_poynts;i++)
    {
      MPI_Bcast(&(simul->output->poynts[i].dir), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poynts[i].x1), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poynts[i].x2), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poynts[i].y1), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poynts[i].y2), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poynts[i].z1), 1, MPI_INT, 0, MPI_COMM_WORLD);
      MPI_Bcast(&(simul->output->poynts[i].z2), 1, MPI_INT, 0, MPI_COMM_WORLD);
      if(simul->output->poynts[i].val==NULL)
	simul->output->poynts[i].val=myalloc(simul->simul->it_max*sizeof(double));
    }
  return simul;
}
#endif // HAVE_MPI
