;;; s7 test suite
;;;
;;; sources include 
;;;   clisp test suite
;;;   Paul Dietz's CL test suite
;;;   R Kelsey, W Clinger, and J Rees r5rs.html (and r6rs.html)
;;;   A Jaffer's r4rstest.scm (the inspiration for this...)
;;;   guile test suite
;;;   gauche test suite
;;;   gambit test suite
;;;   sacla test suite
;;;   Kent Dybvig's "The Scheme Programming Language"
;;;   Brad Lucier (who also pointed out many bugs)
;;;   GSL tests
;;;   Abramowitz and Stegun, "Handbook of Mathematical Functions"
;;;   Weisstein, "Encyclopedia of Mathematics"
;;;   the arprec package of David Bailey et al
;;;   Maxima, William Schelter et al
;;;   H Cohen, "A Course in Computational Algebraic Number Theory"
;;;   N Higham, "Accuracy and Stability of Numerical Algorithms"
;;;   various mailing lists and websites (see individual cases below)


(define with-bignums (provided? 'gmp))                         ; scheme integer has any number of bits
(define with-bigfloats (provided? 'gmp))                       ; scheme real has any number of bits
(define with-bignum-function (defined? 'bignum))               ;   this is a function that turns its string arg into a bignum
(define with-the-bug-finding-machine #f)                       ; run the machine (this variable can be set to the number of tries)
					                       ;   the default number of tries is 10000
(define with-test-at-random #f)

(define our-pi 3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117067982148086513282306647093844609550582231725359408128481117450284102701938521105559644622948954930382)


;;; --------------------------------------------------------------------------------

(if (and (defined? 'current-time) ; in Snd
	 (defined? 'mus-rand-seed))
    (set! (mus-rand-seed) (current-time)))


(define (ok? tst result expected)
  (if (not (equal? result expected))
      (format #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) tst result expected)))

(defmacro test (tst expected) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (ok? ',tst result ,expected)))

(defmacro test-t (tst) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (if (or (not result)
	     (eq? result 'error))
	 (format #t "~A: ~A got ~S~%~%" (port-line-number) ',tst result))))

(defmacro test-e (tst op arg) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (format #t "~A: (~A ~S) got ~S but expected 'error~%~%" (port-line-number) ,op ,arg result))))


(define (op-error op result expected)
  
  (define (conjugate n) 
    (make-rectangular (real-part n) (- (imag-part n))))
  
  (if (and (real? result)
	   (real? expected))
      (/ (abs (- result expected)) (max 1.0 (abs expected)))
      (case op
	((acosh)
	 (/ (magnitude (- (cosh result) (cosh expected)))
	    (max 0.001 (magnitude (cosh expected)))))
	((asin)
	 (/ (min (magnitude (- (sin result) (sin expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (* 10 (magnitude (sin expected))))))
	((acos)
	 (/ (min (magnitude (- (cos result) (cos expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (cos expected)))))
	((asinh)
	 (/ (magnitude (- (sinh result) (sinh expected)))
	    (max 0.001 (magnitude (sinh expected)))))
	((atanh)
	 (/ (min (magnitude (- (tanh result) (tanh expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tanh expected)))))
	((atan)
	 (/ (min (magnitude (- (tan result) (tan expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tan expected)))))
	((cosh)
	 (/ (min (magnitude (- result expected))
		 (magnitude (- result (- expected))))
	    (max 0.001 (magnitude expected))))
	(else (/ (magnitude (- result expected)) (max 0.001 (magnitude expected)))))))


;;; relative error (/ (abs (- x res) (abs x)))

(define (types-consistent? n)
  (not (or (and (integer? n) 
		(or (not (= (denominator n) 1))
		    (not (= n (numerator n)))
		    (not (= (imag-part n) 0))
		    (not (= (floor n) (ceiling n) (truncate n) (round n) n))
		    (not (= n (real-part n)))))
	   (and (rational? n)
		(not (integer? n))
		(or (not (= (imag-part n) 0))
		    (= (denominator n) 1)
		    (= (denominator n) 0)
		    (not (= n (real-part n)))
		    (not (= n (/ (numerator n) (denominator n))))))
	   (and (real? n)
		(not (rational? n))
		(or (not (= (imag-part n) 0))
		    (not (= n (real-part n)))))
	   (and (complex? n) 
		(not (real? n))
		(or (= (imag-part n) 0)
		    (not (= n (+ (real-part n) (* 0+i (imag-part n))))))))))


(define (our-nan? x)
  (or (and (real? x)
	   (not (= 1 (+ (if (positive? x) 1 0)
			(if (negative? x) 1 0)
			(if (zero? x) 1 0)))))
      (and (integer? x)
	   (not (= 1 (+ (if (even? x) 1 0)
			(if (odd? x) 1 0)))))
      (let ((type (+ (if (integer? x) 1 0)
		     (if (rational? x) 2 0)
		     (if (real? x) 4 0)
		     (if (complex? x) 8 0))))
	(and (not (= type 8))
	     (not (= type 12))
	     (not (= type 14))
	     (not (= type 15))))
      (nan? x)))


(define (number-ok? tst result expected)
  ;; (number? +nan.0) returns #t in Guile and Gauche
  
  (if (not (eq? result expected))
      (if (or (and (not (number? expected))
		   (not (eq? result expected)))
	      (and (number? expected)
		   (or (not (number? result))
		       (our-nan? result)))
	      (and (rational? expected)
		   (rational? result)
		   (not (= result expected)))
	      (and (or (rational? expected) 
		       (rational? result))
		   (real? expected)
		   (real? result)
		   (> (abs (- result expected)) 1.0e-12))
	      (and (pair? tst)
		   (> (op-error (car tst) result expected) 1.0e-6))
	      (and (number? result)
		   (not (types-consistent? result))))
	  (begin
	    (format #t "~A: ~A got ~A~Abut expected ~A" 
		    (port-line-number) tst result 
		    (if (and (rational? result) (not (rational? expected)))
			(format #f " (~A) " (* 1.0 result))
			" ")
		    expected)
	    
	    (if (and (not (number? expected))
		     (not (eq? result expected)))
		(format #t ", (eq? ~A ~A) -> #f" result expected)
		(if (and (number? expected)
			 (or (not (number? result))
			     (our-nan? result)))
		    (begin
		      (if (not (number? result))
			  (format #t ", (number? ~A) but not (number? ~A)" expected result)
			  (format #t ", (number? ~A) but (nan? ~A)" expected result)))
		    (if (and (rational? expected)
			     (rational? result)
			     (not (= result expected)))
			(format #t ", exact results but not (= ~A ~A): ~A" expected result (= result expected))
			(if (and (or (rational? expected) 
				     (rational? result))
				 (real? expected)
				 (real? result)
				 (> (abs (- result expected)) 1.0e-12))
			    (format #t ", rational results but diff > 1e-12: ~A" (> (abs (- result expected)) 1.0e-12))
			    (if (and (pair? tst)
				     (< (op-error (car tst) result expected) 1.0e-6))
				(let ((n result))
				  (format #t ", result not internally consistent")
				  (if (and (integer? n) 
					   (or (not (= (denominator n) 1))
					       (not (= n (numerator n)))
					       (not (= (imag-part n) 0))
					       (not (= (floor n) (ceiling n) (truncate n) (round n) n))
					       (not (= n (real-part n)))))
				      (format #t ", ~A integer but den: ~A, num: ~A, imag: ~A, real: ~A, floors: ~A ~A ~A ~A"
					      n (denominator n) (numerator n) (imag-part n) (real-part n)
					      (floor n) (ceiling n) (truncate n) (round n))
				      (if (and (rational? n)
					       (not (integer? n))
					       (or (not (= (imag-part n) 0))
						   (= (denominator n) 1)
						   (= (denominator n) 0)
						   (not (= n (real-part n)))
						   (not (= n (/ (numerator n) (denominator n))))))
					  (format #t ", ~A ratio but imag: ~A, den: ~A, real: ~A, ~A/~A=~A"
						  n (imag-part n) (denominator n) (real-part n) 
						  (numerator n) (denominator n) (* 1.0 (/ (numerator n) (denominator n))))
					  (if (and (real? n)
						   (not (rational? n))
						   (or (not (= (imag-part n) 0))
						       (not (= n (real-part n)))))
					      (format #t ", ~A real but rational: ~A, imag: ~A, real: ~A"
						      n (rational? n) (imag-part n) (real-part n))
					      (format #t ", ~A complex but real? ~A, imag: ~A, ~A+~A=~A"
						      n (real? n) (imag-part n) (real-part n) (imag-part n)
						      (+ (real-part n) (* 0+i (imag-part n)))))))))))))
	    (newline) (newline)))))

(defmacro num-test (tst expected) ;(display tst) (newline)
  `(let ((result (catch #t (lambda () ,tst) (lambda args 'error))))
     (number-ok? ',tst result ,expected)
     (if with-bignum-function
	 (letrec ((bigify (lambda (lst)
			    (if (pair? lst)
				(cons (if (number? (car lst))
					  (list 'bignum (number->string (car lst)))
					  (bigify (car lst)))
				      (bigify (cdr lst)))
				lst))))
	   
	   (let* ((big-test (bigify ',tst)))
	     (let ((big-result (catch #t (lambda () (eval big-test)) (lambda args 'error))))
	       (number-ok? big-test big-result ,expected)))
	   ))))

(define-macro (reinvert n op1 op2 arg)
  (let ((body `(,op2 (,op1 ,arg))))
    (do ((i 1 (+ i 1)))
	((= i n))
      (set! body `(,op2 (,op1 ,body))))
    body))

(define-macro (recompose n op arg)
  (define (recompose-1 n)
    (if (= n 1)
	`(,op ,arg)
	`(,op ,(recompose-1 (- n 1)))))
  (recompose-1 n))

(define-macro (with-evaluator . body)
  (if (provided? 'threads)
      `(join-thread (make-thread (lambda () ,@body) 200))
      ''error)) ; yow! otherwise we get the error procedure, not the symbol 'error!

(define-macro (test-w tst)  ;(display "test-w: ") (display tst) (newline)
  `(let* ((old-error-port (set-current-error-port (open-output-string)))
	  (result (with-evaluator (eval-string ,tst))))
     (close-output-port (current-error-port))
     (set-current-error-port old-error-port)
     (if (or (not result)
	     (not (eq? result 'error)))
	 (format #t "~A: ~A got ~A~%~%" (port-line-number) ',tst result))))



;;; --------------------------------------------------------------------------------
;;; GENERIC STUFF
;;; --------------------------------------------------------------------------------

(test (eq? 'a 3) #f)
(test (eq? #t 't) #f)
(test (eq? "abs" 'abc) #f)
(test (eq? "hi" '(hi)) #f)
(test (eq? "()" '()) #f)
(test (eq? #\a #\b) #f)
(test (eq? #t #t) #t)
(test (eq? #f #f) #t)
(test (eq? #f #t) #f)
(test (eq? (null? '()) #t) #t)
(test (eq? (null? '(a)) #f) #t)
(test (eq? (cdr '(a)) '()) #t)
(test (eq? 'a 'a) #t)
(test (eq? 'a 'b) #f)
(test (eq? 'a (string->symbol "a")) #t)
(test (eq? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eq? x x)) #t)
(test (let ((x (cons 'a 'b))) (eq? x x)) #t)
(test (eq? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eq? "abc" "cba") #f)
(test (let ((x "hi")) (eq? x x)) #t)
(test (eq? (string #\h #\i) (string #\h #\i)) #f)
(test (eq? '#(a) '#(b)) #f)
(test (let ((x (vector 'a))) (eq? x x)) #t)
(test (eq? (vector 'a) (vector 'a)) #f)
(test (eq? car car) #t)
(test (eq? car cdr) #f)
(test (let ((x (lambda () 1))) (eq? x x)) #t)
(test (eq? 'abc 'abc) #t)
(test (eq? eq? eq?) #t)
(test (eq? (if #f 1) 1) #f)
(test (eq? '() '(#||#)) #t)
(test (eq? '() '(#!@%$&!#)) #t)
(test (eq? #||# (#|%%|# append #|^|#) #|?|# (#|+|# list #|<>|#) #||#) #t)
(test (eq? '() ;a comment
	   '()) #t)
(test (eq? 3/4 3) #f)
(test (eq? '() '()) #t)
(test (eq? '()'()) #t)
(test (eq? '()(list)) #t)
(test (eq? '() (list)) #t)

(test (eq? ''2 '2) #f)
(test (eq? '2 '2) #t)
(test (eq? '2 2) #t)
(test (eq? ''2 ''2) #f)
(test (eq? ''#\a '#\a) #f)
(test (eq? '#\a #\a) #f) ; the only difference with eqv?
(test (eq? 'car car) #f)
(test (eq? '() ()) #t)
(test (eq? ''() '()) #f)
(test (eq? '#f #f) #t)
(test (eq? '#f '#f) #t)
(test (eq? '()'()) #t) ; no space

(display ";this should display #t: ")
(begin #| ; |# (display #t))
(newline)

(test (;
       eq? ';!
       (;)()#
	);((")";
       ;"#|)#""
       '#|";"|#(#!;!#); ;#
	 ;\;"#"#f 
	       )#t)

(test (+ #| this is a comment |# 2 #! and this is another !# 3) 5)
(test (eq? (if #f #t) (if #f 3)) #t)

(test (eq?) 'error) ; "this comment is missing a double-quote
(test (eq? #t) 'error)        #| "this comment is missing a double-quote |#
(test (eq? #t #t #t) 'error)
(test (eq? #f . 1) 'error)

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector) (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (eq? (vector-ref things i) (vector-ref things j))
	  (format #t "(eq? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))


;;; these are defined at user-level in s7 -- why are other schemes so coy about them?
(test (eq? (if #f #f) #<unspecified>) #t)
(test (eof-object? #<eof>) #t)
(test (eq? (symbol->value '_?__undefined__?_) #<undefined>) #t)



(test (eqv? 'a 3) #f)
(test (eqv? #t 't) #f)
(test (eqv? "abs" 'abc) #f)
(test (eqv? "hi" '(hi)) #f)
(test (eqv? "()" '()) #f)
(test (eqv? #\a #\b) #f)
(test (eqv? #\a #\a) #t)
(test (eqv? #\space #\space) #t)
(test (let ((x (string-ref "hi" 0))) (eqv? x x)) #t)
(test (eqv? #t #t) #t)
(test (eqv? #f #f) #t)
(test (eqv? #f #t) #f)
(test (eqv? (null? '()) #t) #t)
(test (eqv? (null? '(a)) #f) #t)
(test (eqv? (cdr '(a)) '()) #t)
(test (eqv? 'a 'a) #t)
(test (eqv? 'a 'b) #f)
(test (eqv? 'a (string->symbol "a")) #t)
(test (eqv? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eqv? x x)) #t)
(test (let ((x (cons 'a 'b))) (eqv? x x)) #t)
(test (eqv? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eqv? "abc" "cba") #f)
(test (let ((x "hi")) (eqv? x x)) #t)
(test (eqv? (string #\h #\i) (string #\h #\i)) #f)
(test (eqv? '#(a) '#(b)) #f)
(test (let ((x (vector 'a))) (eqv? x x)) #t)
(test (eqv? (vector 'a) (vector 'a)) #f)
(test (eqv? car car) #t)
(test (eqv? car cdr) #f)
(test (let ((x (lambda () 1))) (eqv? x x)) #t)
(test (eqv? (lambda () 1) (lambda () 1)) #f)
(test (let () (define (make-adder x) (lambda (y) (+ x y))) (eqv? (make-adder 1) (make-adder 1))) #f)
(test (eqv? 9/2 9/2) #t)

(test (eqv? most-positive-fixnum most-positive-fixnum) #t)
(test (eqv? most-positive-fixnum most-negative-fixnum) #f)
(test (eqv? 9223372036854775807 9223372036854775806) #f)
(test (eqv? 9223372036854775807 -9223372036854775808) #f)
(test (eqv? -9223372036854775808 -9223372036854775808) #t)
(test (eqv? 123456789/2 123456789/2) #t)
(test (eqv? 123456789/2 123456787/2) #f)
(test (eqv? -123456789/2 -123456789/2) #t)
(test (eqv? 2/123456789 2/123456789) #t)
(test (eqv? -2/123456789 -2/123456789) #t)
(test (eqv? 2147483647/2147483646 2147483647/2147483646) #t)
(test (eqv? 3/4 12/16) #t)
(test (eqv? 1/1 1) #t)
(test (eqv? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (eqv? x x)) #t)
(test (let ((x 1+i)) (eqv? x x)) #t)
(test (let* ((x 3.141) (y x)) (eqv? x y)) #t)
(test (let* ((x 1+i) (y x)) (eqv? x y)) #t)
(test (let* ((x 3/4) (y x)) (eqv? x y)) #t)

(test (eqv? (cons 'a 'b) (cons 'a 'c)) #f)
(test (eqv? eqv? eqv?) #t)
(test (eqv? '#(1) '#(1)) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '() '()) #t)
(test (eqv? '() (list)) #t)


(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector) (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (eqv? (vector-ref things i) (vector-ref things j))
	  (format #t "(eqv? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))

(test (eqv?) 'error)
(test (eqv? #t) 'error)
(test (eqv? #t #t #t) 'error)

(test (eqv? ''2 '2) #f)
(test (eqv? '2 '2) #t)
(test (eqv? '2 2) #t)
(test (eqv? ''2 ''2) #f)
(test (eqv? ''#\a '#\a) #f)
(test (eqv? '#\a #\a) #t)
(test (eqv? 'car car) #f)
(test (eqv? '() ()) #t)
(test (eqv? ''() '()) #f)
(test (eqv? '#f #f) #t)
(test (eqv? '#f '#f) #t)




(test (equal? 'a 3) #f)
(test (equal? #t 't) #f)
(test (equal? "abs" 'abc) #f)
(test (equal? "hi" '(hi)) #f)
(test (equal? "()" '()) #f)
(test (equal? '(()) '(() . ())) #t)
(test (equal? #\a #\b) #f)
(test (equal? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (equal? x x)) #t)
(test (equal? #t #t) #t)
(test (equal? #f #f) #t)
(test (equal? #f #t) #f)
(test (equal? (null? '()) #t) #t)
(test (equal? (null? '(a)) #f) #t)
(test (equal? (cdr '(a)) '()) #t)
(test (equal? 'a 'a) #t)
(test (equal? 'a 'b) #f)
(test (equal? 'a (string->symbol "a")) #t)
(test (equal? '(a) '(b)) #f)
(test (equal? '(a) '(a)) #t)
(test (let ((x '(a . b))) (equal? x x)) #t)
(test (let ((x (cons 'a 'b))) (equal? x x)) #t)
(test (equal? (cons 'a 'b) (cons 'a 'b)) #t)
(test (equal?(cons 'a 'b)(cons 'a 'b)) #t) ; no space
(test (equal? "abc" "cba") #f)
(test (equal? "abc" "abc") #t)
(test (let ((x "hi")) (equal? x x)) #t)
(test (equal? (string #\h #\i) (string #\h #\i)) #t)
(test (equal? '#(a) '#(b)) #f)
(test (equal? '#(a) '#(a)) #t)
(test (let ((x (vector 'a))) (equal? x x)) #t)
(test (equal? (vector 'a) (vector 'a)) #t)
(test (equal? '#(1 2) (vector 1 2)) #t)
(test (equal? '#(1.0 2/3) (vector 1.0 2/3)) #t)
(test (equal? '#(1 2) (vector 1 2.0)) #f) ; 2 not equal 2.0!
(test (equal? '(1 . 2) (cons 1 2)) #t)
(test (equal? '#(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (equal? '#((1 . 2)) (vector (cons 1 2))) #t)
(test (equal? '#(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (equal? '#(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (equal? '#(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (equal? '#(()) (vector '())) #t)
(test (equal? '#("hi" "ho") (vector "hi" '"ho")) #t)
(test (equal? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (equal? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (equal? (list 1 2) '(1 2.0)) #f)
(test (equal? '#(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (equal? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (equal? '((())) (list (list (list)))) #t)
(test (equal? car car) #t)
(test (equal? car cdr) #f)
(test (let ((x (lambda () 1))) (equal? x x)) #t)
(test (equal? (lambda () 1) (lambda () 1)) #f)
(test (equal? 9/2 9/2) #t)
(test (equal? #((())) #((()))) #t)
(test (equal? "123""123") #t);no space
(test (equal? """") #t)#|no space|#
(test (equal? #()#()) #t)
(test (equal? #()()) #f)
(test (equal? ()"") #f)

(test (equal? most-positive-fixnum most-positive-fixnum) #t)
(test (equal? most-positive-fixnum most-negative-fixnum) #f)
(test (equal? 9223372036854775807 9223372036854775806) #f)
(test (equal? 9223372036854775807 -9223372036854775808) #f)
(test (equal? -9223372036854775808 -9223372036854775808) #t)
(test (equal? 123456789/2 123456789/2) #t)
(test (equal? 123456789/2 123456787/2) #f)
(test (equal? -123456789/2 -123456789/2) #t)
(test (equal? 2/123456789 2/123456789) #t)
(test (equal? -2/123456789 -2/123456789) #t)
(test (equal? 2147483647/2147483646 2147483647/2147483646) #t)
(test (equal? 3/4 12/16) #t)
(test (equal? 1/1 1) #t)
(test (equal? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (equal? x x)) #t)
(test (let ((x 1+i)) (equal? x x)) #t)
(test (let* ((x 3.141) (y x)) (equal? x y)) #t)
(test (let* ((x 1+i) (y x)) (equal? x y)) #t)
(test (let* ((x 3/4) (y x)) (equal? x y)) #t)

(test (let ((x 3.141)) (equal? x x)) #t)
(test (equal? 3 3) #t)
(test (equal? 3 3.0) #f)
(test (equal? 3.0 3.0) #t)
(test (equal? 3-4i 3-4i) #t)
(test (equal? (string #\c) "c") #t)
(test (equal? equal? equal?) #t)
(test (equal? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (equal? '() '()) #t)
(test (equal? '() (list)) #t)
(test (equal? "\n" "\n") #t)
(test (equal? #f ((lambda () #f))) #t)
(test (equal? (+) 0) #t)
(test (equal? (recompose 32 list '(1)) (recompose 32 list (list 1))) #t)
(test (equal? (recompose 100 list '(1)) (recompose 100 list (list 1))) #t)
(test (equal? (recompose 32 vector 1) (recompose 32 vector 1)) #t)
(test (equal? (reinvert 32 list vector 1) (reinvert 32 list vector 1)) #t)
(test (equal? (recompose 32 (lambda (a) (cons 1 a)) '()) (recompose 32 (lambda (a) (cons 1 a)) '())) #t)
(test (equal? (recompose 32 (lambda (a) (list 1 a)) '()) (recompose 32 (lambda (a) (list 1 a)) '())) #t)

(test (equal? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (equal? a b)))) (test (streq "asd""asd") #t))

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) '#() (vector 1) (list 1) 'f 't #\t)))
  (do ((i 0 (+ i 1)))
      ((= i (- (vector-length things) 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j (vector-length things)))
      (if (equal? (vector-ref things i) (vector-ref things j))
	  (format #t "(equal? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j))))))

(test (equal?) 'error)
(test (equal? #t) 'error)
(test (equal? #t #t #t) 'error)



(test (boolean? #f) #t)
(test (boolean? #t) #t)
(test (boolean? 0) #f)
(test (boolean? 1) #f)
(test (boolean? "") #f)
(test (boolean? #\0) #f)
(test (boolean? '()) #f)
(test (boolean? '#()) #f)
(test (boolean? 't) #f)
(test (boolean? (list)) #f)
(test ( boolean? #t) #t)
(test (boolean? boolean?) #f)
(test (   ; a comment 
       boolean?  ;;; and another
       #t
       )
      #t)

(for-each
 (lambda (arg)
   (if (boolean? arg)
       (format #t "(boolean? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) (if #f #f) #<eof> #<undefined>))

(test (recompose 12 boolean? #f) #t)

(test (boolean?) 'error)
(test (boolean? #f #t) 'error)
(test (boolean? (lambda (x) #f)) #f)
(test (boolean? and) #f)
(test (boolean? if) #f)



(test (not #f) #t)
(test (not #t) #f)
(test (not (not #t)) #t)
(test (not 0) #f)
(test (not 1) #f)
(test (not '()) #f)
(test (not 't) #f)
(test (not (list)) #f)
(test (not (list 3)) #f)
(test (not 'nil) #f)
(test (not not) #f)

(for-each
 (lambda (arg)
   (if (not arg)
       (format #t "(not ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) (if #f #f)))

(test (recompose 12 not #f) #f)

(test (not) 'error)
(test (not #f #t) 'error)
(test (not and) #f)
(test (not case) #f)




(test (symbol? 't) #t)
(test (symbol? "t") #f)
(test (symbol? '(t)) #f)
(test (symbol? #t) #f)
(test (symbol? 4) #f)
(test (symbol? 'foo) #t)
(test (symbol? (car '(a b))) #t)
(test (symbol? 'nil) #t)
(test (symbol? '()) #f)
(test (symbol? #()) #f)
(test (symbol? #f) #f)
(test (symbol? 'car) #t)
(test (symbol? car) #f)
(test (symbol? '#f) #f)
(test (symbol? #()) #f)
(test (symbol? :hi) #t)
(test (symbol? hi:) #t)
(test (symbol? :hi:) #t)
(test (symbol? #b1) #f)
(test (symbol? 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789) #t) ;M Gran
(test (symbol? (vector-ref '#(1 a 34) 1)) #t)
(test (if (symbol? '1+) (symbol? '0e) #t) #t)

(for-each
 (lambda (arg)
   (if (symbol? arg)
       (format #t "(symbol? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) #<eof> #<undefined>))

(test (symbol?) 'error)
(test (symbol? 'hi 'ho) 'error)

;;; "Returns #t if obj is a symbol, otherwise returns #f" (r5|6rs.html)
(test (symbol? begin) #f) ; ?? this is an error in Guile, it was #t in s7
(test (symbol? expt) #f)
(test (symbol? if) #f)
(test (symbol? and) #f)
(test (symbol? lambda) #f)
(test (symbol? call/cc) #f)



(test (procedure? car) #t)
(test (procedure? procedure?) #t)
(test (procedure? 'car) #f)
(test (procedure? (lambda (x) x)) #t)
(test (procedure? '(lambda (x) x)) #f)
(test (call/cc procedure?) #t) ; ??
(test (let ((a (lambda (x) x)))	(procedure? a)) #t)
(test (letrec ((a (lambda () (procedure? a)))) (a)) #t)
(test (let ((a 1)) (let ((a (lambda () (procedure? a)))) (a))) #f)
(test (let () (define (hi) 1) (procedure? hi)) #t)
(test (let () (define-macro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
(test (procedure? begin) #f)
(test (procedure? (lambda* ((a 1)) a)) #t)
(test (procedure? and) #f)

(for-each
 (lambda (arg)
   (if (procedure? arg)
       (format #t "(procedure? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #() (if #f #f)))

(test (procedure?) 'error)
(test (procedure? abs car) 'error)

;; these are questionable -- an applicable object is a procedure
(test (procedure? "hi") #f)
(test (procedure? '(1 2)) #f)
(test (procedure? #(1 2)) #f)





;;; --------------------------------------------------------------------------------
;;; CHARACTERS
;;; --------------------------------------------------------------------------------

(test (eqv? '#\  #\space) #t)
(test (eqv? #\newline '#\newline) #t)

(test (char? #\a) #t)
(test (char? #\() #t)
(test (char? #\space) #t)
(test (char? '#\newline) #t)
(test (char? #\1) #t)
(test (char? #\$) #t)
(test (char? #\.) #t)
(test (char? #\\) #t)
(test (char? #\)) #t)
(test (char? #\%) #t)
(test (char? '#\space) #t)
(test (char? '#\ ) #t)
(test (char? '#\newline) #t)
(test (char? '#\a) #t)
(test (char? '#\8) #t)
(test (char? #\-) #t)
(test (char? #\n) #t)
(test (char? #\() #t)
(test (char? #e1) #f)
(test (char? #\#) #t)
(test (char? #\x) #t)
(test (char? #\o) #t)
(test (char? #\b) #t)
(test (char? #b101) #f)
(test (char? #o73) #f)
(test (char? #x73) #f)
(test (char? 'a) #f)
(test (char? 97) #f)
(test (char? "a") #f)
(test (char? (string-ref "hi" 0)) #t)
(test (char? (string-ref (make-string 1) 0)) #t)
(test (char? #\") #t)
(test (char? #\') #t)
(test (char? #\`) #t)
(test (char? #\@) #t)
(test (char? #<eof>) #f)

(for-each
 (lambda (arg)
   (if (char? arg)
       (format #t "(char? ~A) -> #t?~%" arg)))
 (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1))))

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (not (char? (integer->char i)))
      (format #t "(char? (integer->char ~A)) -> #f?~%" i)))

(test (char?) 'error)
(test (char? #\a #\b) 'error)
(test (char? #\x65) #t)
(test (char? #\x000000000065) #t)
(test (char=? #\x+65 #\x0000000000065) #t)
(test (char? #\x0) #t)
(test (char? #\xff) #t)
;; any larger number is a reader error

(test-w "(char? #\\100)")
(test-w "(char? #\\x-65)")
(test-w "(char? #\\x6.5)")
(test-w "(char? #\\x6/5)")
(test-w "(char? #\\x6+i)")
(test-w "(char? #\\x6asd)")
(test-w "(char? #\\x6#)")

(test (char=? #\x6a #\j) #t)

(test (char? #\return) #t)
(test (char? #\null) #t)
(test (char? #\linefeed) #t)
(test (char? #\tab) #t)
(test (char? #\space) #t)


(num-test (let ((str (make-string 258 #\space)))
	    (do ((i 1 (+ i 1)))
		((= i 256))
	      (string-set! str i (integer->char i)))
	    (string-set! str 257 (integer->char 0))
	    (string-length str))
	  258)


(let ((a-to-z (list #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\x #\y #\z))
      (cap-a-to-z (list #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\X #\Y #\Z))
      (mixed-a-to-z (list #\a #\B #\c #\D #\e #\F #\g #\H #\I #\j #\K #\L #\m #\n #\O #\p #\Q #\R #\s #\t #\U #\v #\X #\y #\Z))
      (digits (list #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
  
  (test (char-upper-case? #\a) #f)
  (test (char-upper-case? #\A) #t)
  
  (for-each
   (lambda (arg)
     (if (not (char-upper-case? arg))
	 (format #t "(char-upper-case? ~A) -> #f?~%" arg)))
   cap-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-upper-case? arg)
	 (format #t "(char-upper-case? ~A) -> #t?~%" arg)))
   a-to-z)
  
  ;; non-alpha chars are "unspecified" here
  
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? #\a #\b) 'error)

  
  (test (char-lower-case? #\A) #f)
  (test (char-lower-case? #\a) #t)
  
  (for-each
   (lambda (arg)
     (if (not (char-lower-case? arg))
	 (format #t "(char-lower-case? ~A) -> #f?~%" arg)))
   a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-lower-case? arg)
	 (format #t "(char-lower-case? ~A) -> #t?~%" arg)))
   cap-a-to-z)
  
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? #\a #\b) 'error)

  
  (test (char-upcase #\A) #\A)
  (test (char-upcase #\a) #\A)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\$) #\$)
  (test (char-upcase #\.) #\.)
  (test (char-upcase #\\) #\\)
  (test (char-upcase #\5) #\5)
  (test (char-upcase #\)) #\))
  (test (char-upcase #\%) #\%)
  (test (char-upcase #\0) #\0)
  (test (char-upcase #\_) #\_)
  (test (char-upcase #\space) #\space)
  (test (char-upcase #\newline) #\newline)
  (test (char-upcase #\null) #\null)
  
  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-upcase arg1) arg2))
	 (format #t "(char-upcase ~A) != ~A?~%" arg1 arg2)))
   a-to-z
   cap-a-to-z)
  
  (do ((i 1 (+ i 1)))
      ((= i 128))
    (if (and (not (char=? (integer->char i) (char-upcase (integer->char i))))
	     (not (char-alphabetic? (integer->char i))))
	(format #t "(char-upcase ~A) -> ~A but not alphabetic?~%" (integer->char i) (char-upcase (integer->char i)))))

  (test (recompose 12 char-upcase #\a) #\A)
  (test (reinvert 12 char-upcase char-downcase #\a) #\a)

  (test (char-upcase) 'error)
  (test (char-upcase #\a #\b) 'error)
  (test (char-upcase #<eof>) 'error)


  
  (test (char-downcase #\A) #\a)
  (test (char-downcase #\a) #\a)
  (test (char-downcase #\?) #\?)
  (test (char-downcase #\$) #\$)
  (test (char-downcase #\.) #\.)
  (test (char-downcase #\_) #\_)
  (test (char-downcase #\\) #\\)
  (test (char-downcase #\5) #\5)
  (test (char-downcase #\)) #\))
  (test (char-downcase #\%) #\%)
  (test (char-downcase #\0) #\0)
  (test (char-downcase #\space) #\space)
  
  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-downcase arg1) arg2))
	 (format #t "(char-downcase ~A) != ~A?~%" arg1 arg2)))
   cap-a-to-z
   a-to-z)

  (test (recompose 12 char-downcase #\A) #\a)

  (test (char-downcase) 'error)
  (test (char-downcase #\a #\b) 'error)  

  
  (test (char-numeric? #\a) #f)
  (test (char-numeric? #\5) #t)
  (test (char-numeric? #\A) #f)
  (test (char-numeric? #\z) #f)
  (test (char-numeric? #\Z) #f)
  (test (char-numeric? #\0) #t)
  (test (char-numeric? #\9) #t)
  (test (char-numeric? #\space) #f)
  (test (char-numeric? #\;) #f)
  (test (char-numeric? #\.) #f)
  (test (char-numeric? #\-) #f)
  
  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format #t "(char-numeric? ~A) -> #t?~%" arg)))
   cap-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format #t "(char-numeric? ~A) -> #t?~%" arg)))
   a-to-z)

  (test (char-numeric?) 'error)
  (test (char-numeric? #\a #\b) 'error)  

  
  (test (char-whitespace? #\a) #f)
  (test (char-whitespace? #\A) #f)
  (test (char-whitespace? #\z) #f)
  (test (char-whitespace? #\Z) #f)
  (test (char-whitespace? #\0) #f)
  (test (char-whitespace? #\9) #f)
  (test (char-whitespace? #\space) #t)
  (test (char-whitespace? #\newline) #t)
  (test (char-whitespace? #\;) #f)
  
  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format #t "(char-whitespace? ~A) -> #t?~%" arg)))
   mixed-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format #t "(char-whitespace? ~A) -> #t?~%" arg)))
   digits)

  (test (char-whitespace?) 'error)
  (test (char-whitespace? #\a #\b) 'error)   
 
  
  (test (char-alphabetic? #\a) #t)
  (test (char-alphabetic? #\$) #f)
  (test (char-alphabetic? #\A) #t)
  (test (char-alphabetic? #\z) #t)
  (test (char-alphabetic? #\Z) #t)
  (test (char-alphabetic? #\0) #f)
  (test (char-alphabetic? #\9) #f)
  (test (char-alphabetic? #\space) #f)
  (test (char-alphabetic? #\;) #f)
  (test (char-alphabetic? #\.) #f)
  (test (char-alphabetic? #\-) #f)
  (test (char-alphabetic? #\_) #f)
  (test (char-alphabetic? #\^) #f)
  (test (char-alphabetic? #\[) #f)
  
  (for-each
   (lambda (arg)
     (if (char-alphabetic? arg)
	 (format #t "(char-alphabetic? ~A) -> #t?~%" arg)))
   digits)
  
  (for-each
   (lambda (arg)
     (if (not (char-alphabetic? arg))
	 (format #t "(char-alphabetic? ~A) -> #f?~%" arg)))
   mixed-a-to-z)

  (test (char-alphabetic?) 'error)
  (test (char-alphabetic? #\a #\b) 'error)  

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg) 'error))
      (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1)))))
   (list char-upper-case? char-lower-case? char-upcase char-downcase char-numeric? char-whitespace? char-alphabetic?))


  
  (test 
   (let ((unhappy '()))
     (do ((i 0 (+ i 1))) 
	 ((= i 256)) 
       (let* ((ch (integer->char i))
	      (chu (char-upcase ch))
	      (chd (char-downcase ch)))
	 (if (or (and (not (char=? ch chu))
		      (not (char=? ch (char-downcase chu))))
		 (and (not (char=? ch chd))
		      (not (char=? ch (char-upcase chd))))
		 (and (not (char=? ch chd))
		      (not (char=? ch chu)))
		 (not (char-ci=? chu chd))
		 (not (char-ci=? ch chu))
		 (and (char-alphabetic? ch)
		      (or (not (char-alphabetic? chd))
			  (not (char-alphabetic? chu))))
		 (and (char-numeric? ch)
		      (or (not (char-numeric? chd))
			  (not (char-numeric? chu))))
		 (and (char-whitespace? ch)
		      (or (not (char-whitespace? chd))
			  (not (char-whitespace? chu))))
		 (and (char-alphabetic? ch)
		      (char-whitespace? ch))
		 (and (char-numeric? ch)
		      (char-whitespace? ch))
		 (and (char-alphabetic? ch)
		      (char-numeric? ch)))
	     ;; there are characters that are alphabetic but the result of char-upcase is not an upper-case character
	     ;; 223 for example, or 186 for lower case
	     (set! unhappy (cons (format #f "~C: ~C ~C (~D)~%" ch chu chd i) unhappy)))))
     unhappy)
   '())
  

  
  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op #\a arg) 'error))
      (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg #\a) 'error))
      (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))


  (test (char=? #\d #\d) #t)
  (test (char=? #\A #\a) #f)
  (test (char=? #\d #\x) #f)
  (test (char=? #\d #\D) #f)
  (test (char=? #\a #\a) #t)
  (test (char=? #\A #\B) #f)
  (test (char=? #\a #\b) #f)
  (test (char=? #\9 #\0) #f)
  (test (char=? #\A #\A) #t)
  (test (char=? #\  #\space) #t)
  (let ((i (char->integer #\space)))
    (test (char=? (integer->char i) #\space) #t))
  (test (char=? (integer->char (char->integer #\")) #\") #t)
  (test (char=? #\x65 #\e) #t)
  
  (test (char=? #\d #\d #\d #\d) #t)
  (test (char=? #\d #\d #\x #\d) #f)
  (test (char=? #\d #\y #\x #\c) #f)
  (test (apply char=? cap-a-to-z) #f)
  (test (apply char=? mixed-a-to-z) #f)
  (test (apply char=? digits) #f)
  (test (char=? #\d #\c #\d) #f)

  (test (char=? #\a) 'error)
  (test (char=?) 'error)
  (test (char=? #\a 0) 'error)
  
  
  (test (char<? #\z #\0) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\d #\d) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\A #\B) #t)
  (test (char<? #\a #\b) #t)
  (test (char<? #\9 #\0) #f)
  (test (char<? #\A #\A) #f)
  (test (char<? #\space #\space) #f)
  
  (test (char<? #\a #\e #\y #\z) #t)
  (test (char<? #\a #\e #\e #\y) #f)
  (test (apply char<? a-to-z) #t)
  (test (apply char<? cap-a-to-z) #t)
  (test (apply char<? mixed-a-to-z) #f)
  (test (apply char<? digits) #t)
  (test (apply char<? (reverse a-to-z)) #f)
  (test (apply char<? (reverse cap-a-to-z)) #f)
  (test (apply char<? (reverse mixed-a-to-z)) #f)
  (test (apply char<? (reverse digits)) #f)
  (test (char<? #\b #\c #\a) #f)
  (test (char<? #\B #\B #\A) #f)
  (test (char<? #\b #\c #\e) #t)

  (test (char<?) 'error)
  (test (char<? #\b #\a "hi") 'error)
  (test (char<? #\b #\a 0) 'error)
  
  

  (test (char<=? #\d #\x) #t)
  (test (char<=? #\d #\d) #t)
  
  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\A #\B) #t)
  (test (char<=? #\a #\b) #t)
  (test (char<=? #\9 #\0) #f)
  (test (char<=? #\A #\A) #t)
  (test (char<=? #\space #\space) #t)
  
  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\e #\e #\d #\y) #f)
  (test (apply char<=? a-to-z) #t)
  (test (apply char<=? cap-a-to-z) #t)
  (test (apply char<=? mixed-a-to-z) #f)
  (test (apply char<=? digits) #t)
  (test (apply char<=? (reverse a-to-z)) #f)
  (test (apply char<=? (reverse cap-a-to-z)) #f)
  (test (apply char<=? (reverse mixed-a-to-z)) #f)
  (test (apply char<=? (reverse digits)) #f)
  (test (char<=? #\b #\c #\a) #f)
  (test (char<=? #\B #\B #\A) #f)
  (test (char<=? #\b #\c #\e) #t)
  
  (test (char<=? #\b #\a "hi") 'error)
  (test (char<=? #\b #\a 0) 'error)
  (test (char<=?) 'error)


  
  (test (char>? #\e #\d) #t)
  (test (char>? #\z #\a) #t)
  (test (char>? #\A #\B) #f)
  (test (char>? #\a #\b) #f)
  (test (char>? #\9 #\0) #t)
  (test (char>? #\A #\A) #f)
  (test (char>? #\space #\space) #f)
  
  (test (char>? #\d #\c #\b #\a) #t)
  (test (char>? #\d #\d #\c #\a) #f)
  (test (char>? #\e #\d #\b #\c #\a) #f)
  (test (apply char>? a-to-z) #f)
  (test (apply char>? cap-a-to-z) #f)
  (test (apply char>? mixed-a-to-z) #f)
  (test (apply char>? digits) #f)
  (test (apply char>? (reverse a-to-z)) #t)
  (test (apply char>? (reverse cap-a-to-z)) #t)
  (test (apply char>? (reverse mixed-a-to-z)) #f)
  (test (apply char>? (reverse digits)) #t)
  (test (char>? #\d #\c #\a) #t)
  (test (char>? #\d #\c #\c) #f)
  (test (char>? #\B #\B #\C) #f)
  (test (char>? #\b #\c #\e) #f)

  (test (char>? #\a #\b "hi") 'error)
  (test (char>? #\a #\b 0) 'error)
  (test (char>?) 'error)

  
  
  (test (char>=? #\e #\d) #t)
  (test (char>=? #\A #\B) #f)
  (test (char>=? #\a #\b) #f)
  (test (char>=? #\9 #\0) #t)
  (test (char>=? #\A #\A) #t)
  (test (char>=? #\space #\space) #t)
  
  (test (char>=? #\d #\c #\b #\a) #t)
  (test (char>=? #\d #\d #\c #\a) #t)
  (test (char>=? #\e #\d #\b #\c #\a) #f)
  (test (apply char>=? a-to-z) #f)
  (test (apply char>=? cap-a-to-z) #f)
  (test (apply char>=? mixed-a-to-z) #f)
  (test (apply char>=? digits) #f)
  (test (apply char>=? (reverse a-to-z)) #t)
  (test (apply char>=? (reverse cap-a-to-z)) #t)
  (test (apply char>=? (reverse mixed-a-to-z)) #f)
  (test (apply char>=? (reverse digits)) #t)
  (test (char>=? #\d #\c #\a) #t)
  (test (char>=? #\d #\c #\c) #t)
  (test (char>=? #\B #\B #\C) #f)
  (test (char>=? #\b #\c #\e) #f)

  (test (char>=? #\a #\b "hi") 'error)
  (test (char>=? #\a #\b 0) 'error)
  (test (char>=?) 'error)

  
  
  (test (char-ci=? #\A #\B) #f)
  (test (char-ci=? #\a #\B) #f)
  (test (char-ci=? #\A #\b) #f)
  (test (char-ci=? #\a #\b) #f)
  (test (char-ci=? #\9 #\0) #f)
  (test (char-ci=? #\A #\A) #t)
  (test (char-ci=? #\A #\a) #t)
  (test (char-ci=? #\a #\A) #t)
  (test (char-ci=? #\space #\space) #t)
  
  (test (char-ci=? #\d #\D #\d #\d) #t)
  (test (char-ci=? #\d #\d #\X #\d) #f)
  (test (char-ci=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci=? cap-a-to-z) #f)
  (test (apply char-ci=? mixed-a-to-z) #f)
  (test (apply char-ci=? digits) #f)
  (test (char-ci=? #\d #\c #\d) #f)

  (test (char-ci=?) 'error)
  

  
  (test (char-ci<? #\A #\B) #t)
  (test (char-ci<? #\a #\B) #t)
  (test (char-ci<? #\A #\b) #t)
  (test (char-ci<? #\a #\b) #t)
  (test (char-ci<? #\9 #\0) #f)
  (test (char-ci<? #\0 #\9) #t)
  (test (char-ci<? #\A #\A) #f)
  (test (char-ci<? #\A #\a) #f)
  (test (char-ci<? #\Y #\_) #t)
  (test (char-ci<? #\\ #\J) #f)
  (test (char-ci<? #\_ #\e) #f)
  (test (char-ci<? #\t #\_) #t)
  (test (char-ci<? #\a #\]) #t)
  (test (char-ci<? #\z #\^) #t)
  
  (test (char-ci<? #\b #\a "hi") 'error)
  (test (char-ci<? #\b #\a 0) 'error)
  
;;; this tries them all:
					;(do ((i 0 (+ i 1)))
					;    ((= i 128))
					;  (do ((k 0 (+ k 1)))
					;      ((= k 128))
					;    (let ((c1 (integer->char i))
					;	  (c2 (integer->char k)))
					;      (for-each
					;       (lambda (op1 op2)
					;	 (if (not (eq? (op1 c1 c2) (op2 (string c1) (string c2))))
					;	     (format #t "(~A|~A ~A ~A) -> ~A|~A~%" op1 op2 c1 c2 (op1 c1 c2) (op2 (string c1) (string c2)))))
					;       (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?)
					;       (list string=? string<? string<=? string>? string>=? string-ci=? string-ci<? string-ci<=? string-ci>? string-ci>=?)))))
  
  
  (test (char-ci<? #\d #\D #\d #\d) #f)
  (test (char-ci<? #\d #\d #\X #\d) #f)
  (test (char-ci<? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<? cap-a-to-z) #t)
  (test (apply char-ci<? mixed-a-to-z) #t)
  (test (apply char-ci<? digits) #t)
  (test (char-ci<? #\d #\c #\d) #f)
  (test (char-ci<? #\b #\c #\a) #f)
  (test (char-ci<? #\b #\C #\e) #t)
  (test (char-ci<? #\3 #\? #\Z #\[) #t)
  
  (test (char-ci>? #\a #\b "hi") 'error)
  (test (char-ci>? #\a #\b 0) 'error)


  
  (test (char-ci>? #\A #\B) #f)
  (test (char-ci>? #\a #\B) #f)
  (test (char-ci>? #\A #\b) #f)
  (test (char-ci>? #\a #\b) #f)
  (test (char-ci>? #\9 #\0) #t)
  (test (char-ci>? #\A #\A) #f)
  (test (char-ci>? #\A #\a) #f)
  (test (char-ci>? #\^ #\a) #t)
  (test (char-ci>? #\_ #\e) #t)
  (test (char-ci>? #\[ #\S) #t)
  (test (char-ci>? #\\ #\l) #t)
  (test (char-ci>? #\t #\_) #f)
  (test (char-ci>? #\a #\]) #f)
  (test (char-ci>? #\z #\^) #f)
  (test (char-ci>? #\] #\X) #t)
  
  (test (char-ci>? #\d #\D #\d #\d) #f)
  (test (char-ci>? #\d #\d #\X #\d) #f)
  (test (char-ci>? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>? cap-a-to-z) #f)
  (test (apply char-ci>? mixed-a-to-z) #f)
  (test (apply char-ci>? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>? digits) #f)
  (test (char-ci>? #\d #\c #\d) #f)
  (test (char-ci>? #\b #\c #\a) #f)
  (test (char-ci>? #\d #\C #\a) #t)
  
  
  (test (char-ci<=? #\A #\B) #t)
  (test (char-ci<=? #\a #\B) #t)
  (test (char-ci<=? #\A #\b) #t)
  (test (char-ci<=? #\a #\b) #t)
  (test (char-ci<=? #\9 #\0) #f)
  (test (char-ci<=? #\A #\A) #t)
  (test (char-ci<=? #\A #\a) #t)
  (test (char-ci<=? #\` #\H) #f)
  (test (char-ci<=? #\[ #\m) #f)
  (test (char-ci<=? #\j #\`) #t)
  (test (char-ci<=? #\\ #\E) #f)
  (test (char-ci<=? #\t #\_) #t)
  (test (char-ci<=? #\a #\]) #t)
  (test (char-ci<=? #\z #\^) #t)
  
  (test (char-ci<=? #\d #\D #\d #\d) #t)
  (test (char-ci<=? #\d #\d #\X #\d) #f)
  (test (char-ci<=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<=? cap-a-to-z) #t)
  (test (apply char-ci<=? mixed-a-to-z) #t)
  (test (apply char-ci<=? digits) #t)
  (test (char-ci<=? #\d #\c #\d) #f)
  (test (char-ci<=? #\b #\c #\a) #f)
  (test (char-ci<=? #\b #\c #\C) #t)
  (test (char-ci<=? #\b #\C #\e) #t)
  
  (test (char-ci<=? #\b #\a "hi") 'error)
  (test (char-ci<=? #\b #\a 0) 'error)


  
  (test (char-ci>=? #\A #\B) #f)
  (test (char-ci>=? #\a #\B) #f)
  (test (char-ci>=? #\A #\b) #f)
  (test (char-ci>=? #\a #\b) #f)
  (test (char-ci>=? #\9 #\0) #t)
  (test (char-ci>=? #\A #\A) #t)
  (test (char-ci>=? #\A #\a) #t)
  (test (char-ci>=? #\Y #\_) #f)
  (test (char-ci>=? #\` #\S) #t)
  (test (char-ci>=? #\[ #\Y) #t)
  (test (char-ci>=? #\t #\_) #f)
  (test (char-ci>=? #\a #\]) #f)
  (test (char-ci>=? #\z #\^) #f)
  
  (test (char-ci>=? #\d #\D #\d #\d) #t)
  (test (char-ci>=? #\d #\d #\X #\d) #f)
  (test (char-ci>=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>=? cap-a-to-z) #f)
  (test (apply char-ci>=? mixed-a-to-z) #f)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? digits) #f)
  (test (char-ci>=? #\d #\c #\d) #f)
  (test (char-ci>=? #\b #\c #\a) #f)
  (test (char-ci>=? #\d #\D #\a) #t)
  (test (char-ci>=? #\\ #\J #\+) #t)

  (test (char-ci>=? #\a #\b "hi") 'error)
  (test (char-ci>=? #\a #\b 0) 'error)

  ) ; end let with a-to-z



(test (integer->char (char->integer #\.)) #\.)
(test (integer->char (char->integer #\A)) #\A)
(test (integer->char (char->integer #\a)) #\a)
(test (integer->char (char->integer #\space)) #\space)

(test (reinvert 12 integer->char char->integer 60) 60)

(test (char->integer 33) 'error)
(test (char->integer) 'error)
(test (integer->char) 'error)
(test (integer->char (expt 2 31)) 'error)
(test (integer->char (expt 2 32)) 'error)
(test (integer->char 12 14) 'error)
(test (char->integer #\a #\b) 'error)

(for-each
 (lambda (arg)
   (test (char->integer arg) 'error))
 (list -1 1 0 123456789 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (integer->char arg) 'error))
 (list -1 123456789 -123456789 #\a "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (#\a) 'error)
(test (#\newline 1) 'error)




;;; --------------------------------------------------------------------------------
;;; STRINGS
;;; --------------------------------------------------------------------------------

(test (string? "abc") #t)
(test (string? ':+*/-) #f)
(test (string? "das ist einer der teststrings") #t)
(test (string? '(das ist natuerlich falsch)) #f)
(test (string? "aaaaaa") #t)
(test (string? #\a) #f)
(test (string? "\"\\\"") #t)

(for-each
 (lambda (arg)
   (test (string? arg) #f))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (string?) 'error)
(test (string? "hi" "ho") 'error)
(test (string? #\null) #f)



(test (string=? "foo" "foo") #t)
(test (string=? "foo" "FOO") #f)
(test (string=? "foo" "bar") #f)
(test (string=? "FOO" "FOO") #t)
(test (string=? "A" "B") #f)
(test (string=? "a" "b") #f)
(test (string=? "9" "0") #f)
(test (string=? "A" "A") #t)
(test (string=? "" "") #t)
(test (string=? (string #\newline) (string #\newline)) #t)

(test (string=? "A" "B" "a") #f)
(test (string=? "A" "A" "a") #f)
(test (string=? "A" "A" "A") #t)
(test (string=? "foo" "foo" "foo") #t)
(test (string=? "foo" "foo" "") #f)
(test (string=? "foo" "foo" "fOo") #f)

(test (string=? "foo" "FOO" 1.0) 'error)

(test (let ((str (string #\" #\1 #\\ #\2 #\")))	(string=? str "\"1\\2\"")) #t)
(test (let ((str (string #\\ #\\ #\\)))	(string=? str "\\\\\\")) #t)
(test (let ((str (string #\")))	(string=? str "\"")) #t)
(test (let ((str (string #\\ #\"))) (string=? str "\\\"")) #t)
(test (let ((str (string #\space #\? #\)))) (string=? str " ?)")) #t)
(test (let ((str (string #\# #\\ #\t))) (string=? str "#\\t")) #t)

(test (string=? (string) "") #t)
(test (string=? (string) (make-string 0)) #t)
(test (string=? (string-copy (string)) (make-string 0)) #t)
(test (string=? "" (make-string 0)) #t)
(test (string=? "" (string-append)) #t)
(test (string=? (string #\space #\newline) " \n") #t)

(test (string=? "......" "...\ ...") #t)
(test (string=? "......" "...\
...") #t)
(test (string=? "" "\ \ \ \ \ \ \ ") #t)
(test (string=? "\n" (string #\newline)) #t)
(test (string=? "\
\
\
\
" "") #t)
(test (string=? "" (string #\null)) #f)
(test (string=? (string #\null #\null) (string #\null)) #f)
(test (string=? "" "asd") #f)
(test (string=? "asd" "") #f)
(test (string=? "xx" (make-string 2 #\x) (string #\x #\x) (list->string (list #\x #\x)) (substring "axxb" 1 3) (string-append "x" "x")) #t)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)


(test (string<? "aaaa" "aaab") #t)
(test (string<? "aaaa" "aaaaa") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "a" "abcdefgh") #t)
(test (string<? "abc" "abcdefgh") #t)
(test (string<? "cabc" "abcdefgh") #f)
(test (string<? "abcdefgh" "abcdefgh") #f)
(test (string<? "xyzabc" "abcdefgh") #f)
(test (string<? "abc" "xyzabcdefgh") #t)
(test (string<? "abcdefgh" "") #f)
(test (string<? "abcdefgh" "a") #f)
(test (string<? "abcdefgh" "abc") #f)
(test (string<? "abcdefgh" "cabc") #t)
(test (string<? "abcdefgh" "xyzabc") #t)
(test (string<? "xyzabcdefgh" "abc") #f)
(test (string<? "abcdef" "bcdefgh") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "" "") #f)
(test (string<? "A" "B") #t)
(test (string<? "a" "b") #t)
(test (string<? "9" "0") #f)
(test (string<? "A" "A") #f)

(test (string<? "A" "B" "A") #f)
(test (string<? "A" "A" "B") #f)
(test (string<? "A" "A" "A") #f)
(test (string<? "B" "B" "C") #f)
(test (string<? "foo" "foo" "foo") #f)
(test (string<? "foo" "foo" "") #f)
(test (string<? "foo" "foo" "fOo") #f)

(test (string<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)




(test (string>? "aaab" "aaaa") #t)
(test (string>? "aaaaa" "aaaa") #t)
(test (string>? "" "abcdefgh") #f)
(test (string>? "a" "abcdefgh") #f)
(test (string>? "abc" "abcdefgh") #f)
(test (string>? "cabc" "abcdefgh") #t)
(test (string>? "abcdefgh" "abcdefgh") #f)
(test (string>? "xyzabc" "abcdefgh") #t)
(test (string>? "abc" "xyzabcdefgh") #f)
(test (string>? "abcdefgh" "") #t)
(test (string>? "abcdefgh" "a") #t)
(test (string>? "abcdefgh" "abc") #t)
(test (string>? "abcdefgh" "cabc") #f)
(test (string>? "abcdefgh" "xyzabc") #f)
(test (string>? "xyzabcdefgh" "abc") #t)
(test (string>? "abcde" "bc") #f)
(test (string>? "bcdef" "abcde") #t)
(test (string>? "bcdef" "abcdef") #t)
(test (string>? "" "") #f)
(test (string>? "A" "B") #f)
(test (string>? "a" "b") #f)
(test (string>? "9" "0") #t)
(test (string>? "A" "A") #f)

(test (string>? "A" "B" "a") #f)
(test (string>? "C" "B" "A") #t)
(test (string>? "A" "A" "A") #f)
(test (string>? "B" "B" "A") #f)
(test (string>? "foo" "foo" "foo") #f)
(test (string>? "foo" "foo" "") #f)
(test (string>? "foo" "foo" "fOo") #f)

(test (string>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)



(test (string<=? "aaa" "aaaa") #t)
(test (string<=? "aaaaa" "aaaa") #f)
(test (string<=? "a" "abcdefgh") #t)
(test (string<=? "abc" "abcdefgh") #t)
(test (string<=? "aaabce" "aaabcdefgh") #f)
(test (string<=? "cabc" "abcdefgh") #f)
(test (string<=? "abcdefgh" "abcdefgh") #t)
(test (string<=? "xyzabc" "abcdefgh") #f)
(test (string<=? "abc" "xyzabcdefgh") #t)
(test (string<=? "abcdefgh" "") #f)
(test (string<=? "abcdefgh" "a") #f)
(test (string<=? "abcdefgh" "abc") #f)
(test (string<=? "abcdefgh" "cabc") #t)
(test (string<=? "abcdefgh" "xyzabc") #t)
(test (string<=? "xyzabcdefgh" "abc") #f)
(test (string<=? "abcdef" "bcdefgh") #t)
(test (string<=? "" "") #t)
(test (string<=? "A" "B") #t)
(test (string<=? "a" "b") #t)
(test (string<=? "9" "0") #f)
(test (string<=? "A" "A") #t)

(test (string<=? "A" "B" "C") #t)
(test (string<=? "C" "B" "A") #f)
(test (string<=? "A" "B" "B") #t)
(test (string<=? "A" "A" "A") #t)
(test (string<=? "B" "B" "A") #f)
(test (string<=? "foo" "foo" "foo") #t)
(test (string<=? "foo" "foo" "") #f)
(test (string<=? "foo" "foo" "fooo") #t)

(test (string<=? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)



(test (string>=? "aaaaa" "aaaa") #t)
(test (string>=? "aaaa" "aaaa") #t)
(test (string>=? "aaa" "aaaa") #f)
(test (string>=? "" "abcdefgh") #f)
(test (string>=? "a" "abcdefgh") #f)
(test (string>=? "abc" "abcdefgh") #f)
(test (string>=? "cabc" "abcdefgh") #t)
(test (string>=? "abcdefgh" "abcdefgh") #t)
(test (string>=? "xyzabc" "abcdefgh") #t)
(test (string>=? "abc" "xyzabcdefgh") #f)
(test (string>=? "abcdefgh" "") #t)
(test (string>=? "abcdefgh" "a") #t)
(test (string>=? "abcdefgh" "abc") #t)
(test (string>=? "abcdefgh" "cabc") #f)
(test (string>=? "abcdefgh" "xyzabc") #f)
(test (string>=? "xyzabcdefgh" "abc") #t)
(test (string>=? "bcdef" "abcdef") #t)
(test (string>=? "A" "B") #f)
(test (string>=? "a" "b") #f)
(test (string>=? "9" "0") #t)
(test (string>=? "A" "A") #t)
(test (string>=? "" "") #t)

(test (string>=? "A" "B" "C") #f)
(test (string>=? "C" "B" "A") #t)
(test (string>=? "C" "B" "B") #t)
(test (string>=? "A" "B" "B") #f)
(test (string>=? "A" "A" "A") #t)
(test (string>=? "B" "B" "A") #t)
(test (string>=? "B" "B" "C") #f)
(test (string>=? "foo" "foo" "foo") #t)
(test (string>=? "foo" "foo" "") #t)
(test (string>=? "foo" "foo" "fo") #t)

(test (string>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)



(test (string-ci=? "A" "B") #f)
(test (string-ci=? "a" "B") #f)
(test (string-ci=? "A" "b") #f)
(test (string-ci=? "a" "b") #f)
(test (string-ci=? "9" "0") #f)
(test (string-ci=? "A" "A") #t)
(test (string-ci=? "A" "a") #t)
(test (string-ci=? "" "") #t)
(test (string-ci=? "aaaa" "AAAA") #t)
(test (string-ci=? "aaaa" "Aaaa") #t)

(test (string-ci=? "A" "B" "a") #f)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "foo" "foo" "foo") #t)
(test (string-ci=? "foo" "foo" "") #f)
(test (string-ci=? "foo" "Foo" "fOo") #t)

(test (string-ci=? "foo" "GOO" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)




(test (string-ci<? "a" "Aa") #t)
(test (string-ci<? "A" "B") #t)
(test (string-ci<? "a" "B") #t)
(test (string-ci<? "A" "b") #t)
(test (string-ci<? "a" "b") #t)
(test (string-ci<? "9" "0") #f)
(test (string-ci<? "0" "9") #t)
(test (string-ci<? "A" "A") #f)
(test (string-ci<? "A" "a") #f)
(test (string-ci<? "" "") #f)

(test (string-ci<? "t" "_") #t)
(test (string-ci<? "a" "]") #t)
(test (string-ci<? "z" "^") #t)
(test (string-ci<? "]4.jVKo\\\\^:\\A9Z4" "MImKA[mNv1`") #f)

(test (string-ci<? "A" "B" "A") #f)
(test (string-ci<? "A" "A" "B") #f)
(test (string-ci<? "A" "A" "A") #f)
(test (string-ci<? "B" "B" "C") #f)
(test (string-ci<? "B" "b" "C") #f)
(test (string-ci<? "foo" "foo" "foo") #f)
(test (string-ci<? "foo" "foo" "") #f)
(test (string-ci<? "foo" "foo" "fOo") #f)
(test (string-ci<? "34ZsfQD<obff33FBPFl" "7o" "9l7OM" "FC?M63=" "rLM5*J") #t)
(test (string-ci<? "NX7" "-;h>P" "DMhk3Bg") #f)
(test (string-ci<? "+\\mZl" "bE7\\e(HaW5CDXbPi@U_" "B_") #t)

(test (string-ci<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "123") (s2 "12")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)



(test (string-ci>? "Aaa" "AA") #t)
(test (string-ci>? "A" "B") #f)
(test (string-ci>? "a" "B") #f)
(test (string-ci>? "A" "b") #f)
(test (string-ci>? "a" "b") #f)
(test (string-ci>? "9" "0") #t)
(test (string-ci>? "A" "A") #f)
(test (string-ci>? "A" "a") #f)
(test (string-ci>? "" "") #f)
(test (string-ci>? "Z" "DjNTl0") #t)
(test (string-ci>? "2399dt7BVN[,A" "^KHboHV") #f)

(test (string-ci>? "t" "_") #f)
(test (string-ci>? "a" "]") #f)
(test (string-ci>? "z" "^") #f)
(test (string-ci>? "R*95oG.k;?" "`2?J6LBbLG^alB[fMD") #f)
(test (string-ci>? "]" "X") #t)

(test (string-ci>? "A" "B" "a") #f)
(test (string-ci>? "C" "b" "A") #t)
(test (string-ci>? "a" "A" "A") #f)
(test (string-ci>? "B" "B" "A") #f)
(test (string-ci>? "foo" "foo" "foo") #f)
(test (string-ci>? "foo" "foo" "") #f)
(test (string-ci>? "foo" "foo" "fOo") #f)
(test (string-ci>? "ZNiuEa@/V" "KGbKliYMY" "9=69q3ica" ":]") #f)
(test (string-ci>? "^" "aN@di;iEO" "7*9q6uPmX9)PaY,6J" "15vH") #t)

(test (string-ci>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)




(test (string-ci<=? "A" "B") #t)
(test (string-ci<=? "a" "B") #t)
(test (string-ci<=? "A" "b") #t)
(test (string-ci<=? "a" "b") #t)
(test (string-ci<=? "9" "0") #f)
(test (string-ci<=? "A" "A") #t)
(test (string-ci<=? "A" "a") #t)
(test (string-ci<=? "" "") #t)
(test (string-ci<=? ":LPC`" ",O0>affA?(") #f)

(test (string-ci<=? "t" "_") #t)
(test (string-ci<=? "a" "]") #t)
(test (string-ci<=? "z" "^") #t)
(test (string-ci<=? "G888E>beF)*mwCNnagP" "`2uTd?h") #t)

(test (string-ci<=? "A" "b" "C") #t)
(test (string-ci<=? "c" "B" "A") #f)
(test (string-ci<=? "A" "B" "B") #t)
(test (string-ci<=? "a" "A" "A") #t)
(test (string-ci<=? "B" "b" "A") #f)
(test (string-ci<=? "foo" "foo" "foo") #t)
(test (string-ci<=? "foo" "foo" "") #f)
(test (string-ci<=? "FOO" "fOo" "fooo") #t)
(test (string-ci<=? "78mdL82*" "EFaCrIdm@_D+" "eMu\\@dSSY") #t)
(test (string-ci<=? "`5pNuFc3PM<rNs" "e\\Su_raVNk6HD" "vXnuN7?S0?S(w+M?p") #f)

(test (string-ci<=? "fOo" "fo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)



(test (string-ci>=? "A" "B") #f)
(test (string-ci>=? "a" "B") #f)
(test (string-ci>=? "A" "b") #f)
(test (string-ci>=? "a" "b") #f)
(test (string-ci>=? "9" "0") #t)
(test (string-ci>=? "A" "A") #t)
(test (string-ci>=? "A" "a") #t)
(test (string-ci>=? "" "") #t)
(test (string-ci>=? "5d7?[o[:hop=ktv;9)" "p^r9;TAXO=^") #f)

(test (string-ci>=? "t" "_") #f)
(test (string-ci>=? "a" "]") #f)
(test (string-ci>=? "z" "^") #f)
(test (string-ci>=? "jBS" "`<+s[[:`l") #f)

(test (string-ci>=? "A" "b" "C") #f)
(test (string-ci>=? "C" "B" "A") #t)
(test (string-ci>=? "C" "B" "b") #t)
(test (string-ci>=? "a" "B" "B") #f)
(test (string-ci>=? "A" "A" "A") #t)
(test (string-ci>=? "B" "B" "A") #t)
(test (string-ci>=? "B" "b" "C") #f)
(test (string-ci>=? "foo" "foo" "foo") #t)
(test (string-ci>=? "foo" "foo" "") #t)
(test (string-ci>=? "foo" "foo" "fo") #t)
(test (string-ci>=? "tF?8`Sa" "NIkMd7" "f`" "1td-Z?teE" "-ik1SK)hh)Nq].>") #t)
(test (string-ci>=? "Z6a8P" "^/VpmWwt):?o[a9\\_N" "8[^h)<KX?[utsc") #f)

(test (string-ci>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)



(for-each
 (lambda (arg)
   (test (string=? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string<? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string>? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string<=? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string>=? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ci=? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


(for-each
 (lambda (arg)
   (test (string-ci<? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ci>? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ci<=? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ci>=? "hi" arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




(test (string-length "abc") 3)
(test (string-length "") 0)
(test (string-length (string)) 0)
(test (string-length "\"\\\"") 3)
(test (string-length (string #\newline)) 1)
(test (string-length "hi there") 8)
(test (string-length "\"") 1)
(test (string-length "\\") 1)
(test (string-length "\n") 1)
(test (string-length (make-string 100 #\a)) 100)
(test (string-length "1\\2") 3)
(test (string-length "1\\") 2)
(test (string-length "hi\\") 3)
(test (string-length "\\\\\\\"") 4)
(test (string-length "A ; comment") 11)
(test (string-length "#| comment |#") 13)
(test (let ((str (string #\# #\\ #\t))) (string-length str)) 3)

(test (string-length "#\\(") 3)
(test (string-length ")()") 3)
(test (string-length "(()") 3)
(test (string-length "(string #\\( #\\+ #\\space #\\1 #\\space #\\3 #\\))") 44)
(test (string-length) 'error)
(test (string-length "hi" "ho") 'error)
(test (string-length "..\ ..") 4)
(test (string-length (string #\null)) 1) ; ??
(test (string-length (string #\null #\null)) 2) ; ??
(test (string-length (string #\null #\newline)) 2) ; ??

(for-each
 (lambda (arg)
   (test (string-length arg) 'error))
 (list #\a '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


(for-each
 (lambda (arg)
   (test (string #\a arg) 'error))
 (list '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (string) "")
(test (string #\a #\b #\c) "abc")
(test (string #\a) "a")
(test (map string '(#\a #\b)) '("a" "b"))
(test (map string '(#\a #\b) '(#\c #\d)) '("ac" "bd"))
(test (map string '(#\a #\b #\c) '(#\d #\e #\f) '(#\g #\h #\i)) '("adg" "beh" "cfi"))
(test (map string "abc" "def" "ghi") '("adg" "beh" "cfi"))
(test (string #\" #\# #\") "\"#\"")
(test (string #\\ #\\ #\# #\\ #\# #\#) "\\\\#\\##")
(test (string #\' #\' #\` #\") '"''`\"")
;;; some schemes accept \' and other such sequences in a string, but the spec only mentions \\ and \"
(test (string '()) 'error)


(test (make-string 0) "")
(test (make-string 3 #\a) "aaa")
(test (make-string 0 #\a) "")
(test (make-string 3 #\space) "   ")
(test (let ((hi (make-string 3 #\newline))) (string-length hi)) 3)

(test (make-string -1) 'error)
(test (make-string 2 #\a #\b) 'error)
(test (make-string) 'error)

(for-each
 (lambda (arg)
   (test (make-string 3 arg) 'error))
 (list "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg #\a) 'error))
 (list #\a "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg) 'error))
 (list #\a "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



(test (string-ref "abcdef-dg1ndh" 0) #\a)
(test (string-ref "abcdef-dg1ndh" 1) #\b)
(test (string-ref "abcdef-dg1ndh" 6) #\-)
(test (string-ref "\"\\\"" 1) #\\)
(test (string-ref "\"\\\"" 2) #\")
(test (string-ref "12\ 34" 2) #\3)

(test (let ((str (make-string 3 #\x))) (set! (string-ref str 1) #\a) str) "xax")

(test (string-ref "abcdef-dg1ndh" 20) 'error)
(test (string-ref "abcdef-dg1ndh") 'error)
(test (string-ref "abcdef-dg1ndh" -3) 'error)
(test (string-ref) 'error)
(test (string-ref 2) 'error)
(test (string-ref "\"\\\"" 3) 'error)
(test (string-ref "" 0) 'error)  
(test (string-ref "" 1) 'error)
(test (string-ref "hiho" (expt 2 32)) 'error)
(test (char=? (string-ref (string #\null) 0) #\null) #t)
(test (char=? (string-ref (string #\newline) 0) #\newline) #t)
(test (char=? (string-ref (string #\space) 0) #\space) #t)

(for-each
 (lambda (arg)
   (test (string-ref arg 0) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ref "hiho" arg) 'error))
 (list #\a -1 123 4 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test ("hi" 1) #\i)
(test (("hi" 1) 0) 'error)
(test ("hi" 1 2) 'error)
(test ("" 0) 'error)
(test (set! ("" 0) #\a) 'error)
(test (set! ("hi" 1 2) #\a) 'error)
(test (set! ("hi" 1) #\a #\b) 'error)
(test ("hi") 'error)
(test ("") 'error)
(test ((let () "hi")) 'error)
(test ((let () "hi") 0) #\h)


(test (let ((hi (string-copy "hi"))) (string-set! hi 0 #\H) hi) "Hi")
(test (let ((hi (string-copy "hi"))) (string-set! hi 1 #\H) hi) "hH")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 0 #\a) hi) "a\\\"")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 1 #\a) hi) "\"a\"")
(test (let ((hi (string #\a #\newline #\b))) (string-set! hi 1 #\c) hi) "acb")
(test (string-copy "ab") "ab")
(test (string-copy "") "")
(test (string-copy "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-copy hi))) #f)
(test (let ((hi (string-copy (make-string 8 (integer->char 0))))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (string-copy (string-copy (string-copy "a"))) "a")
(test (string-copy (string-copy (string-copy ""))) "")

(test (string-copy) 'error)
(test (string-copy "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (test (string-copy arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (length (string-copy (string #\null))) 1)



(let ((str (make-string 10 #\x)))
  (string-set! str 3 (integer->char 0))
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\x) #t)
  (string-set! str 4 #\a)
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\a) #t)
  (string-set! str 3 #\x)
  (test (string=? str "xxxxaxxxxx") #t))

(test (string-set! "hiho" 1 #\c) #\c)
(test (set! ("hi" 1 2) #\i) 'error)
(test (set! ("hi" 1) "ho") 'error)
(test (set! ("hi") #\i) 'error)

(test (let ((hi (make-string 3 #\a)))
	(string-set! hi 1 (let ((ho (make-string 4 #\x)))
			    (string-set! ho 1 #\b)
			    (string-ref ho 0)))
	hi)
      "axa")

(test (string-set! "hiho" (expt 2 32) #\a) 'error)

(test (let ((hi (string-copy "hi"))) (string-set! hi 2 #\H) hi) 'error)
(test (let ((hi (string-copy "hi"))) (string-set! hi -1 #\H) hi) 'error)
(test (let ((g (lambda () "***"))) (string-set! (g) 0 #\?)) #\?)
(test (string-set! "" 0 #\a) 'error)
(test (string-set! "" 1 #\a) 'error)
(test (string-set! (string) 0 #\a) 'error)
(test (string-set! (symbol->string 'lambda) 0 #\a) #\a)
(test (let ((ho (make-string 0 #\x))) (string-set! ho 0 #\a) ho) 'error)
(test (let ((str "hi")) (string-set! (let () str) 1 #\a) str) "ha") ; (also in Guile)
(test (let ((x 2) (str "hi")) (string-set! (let () (set! x 3) str) 1 #\a) (list x str)) '(3 "ha"))
(test (let ((str "hi")) (set! ((let () str) 1) #\a) str) "ha")
(test (let ((str "hi")) (string-set! (let () (string-set! (let () str) 0 #\x) str) 1 #\x) str) "xx")
(test (let ((str "hi")) (string-set! (let () (set! str "hiho") str) 3 #\x) str) "hihx") ; ! (this works in Guile also)

(for-each
 (lambda (arg)
   (test (string-set! arg 0 #\a) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" arg #\a) 'error))
 (list #\a -1 123 4 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" 0 arg) 'error))
 (list 1 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (equal? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (string=? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)



(test (string-fill! "hiho" #\c) #\c)
(test (string-fill! "" #\a) #\a)
(test (string-fill! "hiho" #\a) #\a)
(test (let ((g (lambda () "***"))) (string-fill! (g) #\?)) #\?)
(test (string-fill!) 'error)
(test (string-fill! "hiho" #\a #\b) 'error)

(test (let ((hi (string-copy "hi"))) (string-fill! hi #\s) hi) "ss")
(test (let ((hi (string-copy ""))) (string-fill! hi #\x) hi) "")
(test (let ((str (make-string 0))) (string-fill! str #\a) str) "")
(test (let ((hi (make-string 8 (integer->char 0)))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (recompose 12 string-copy "xax") "xax")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! a #\a) a) hi)) "aaa")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! hi a)) #\a) hi) "aaa")
(test (let ((str (string #\null #\null))) (fill! str #\x) str) "xx")

(for-each
 (lambda (arg)
   (test (let ((hiho "hiho")) (string-fill! hiho arg) hiho) 'error))
 (list 1 "hi" '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-fill! arg #\a) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))




(test (substring "ab" 0 0) "")
(test (substring "ab" 1 1) "")
(test (substring "ab" 2 2) "")
(test (substring "ab" 0 1) "a")
(test (substring "ab" 1 2) "b")
(test (substring "ab" 0 2) "ab")
(test (substring "hi there" 3 6) "the")
(test (substring "hi there" 0 (string-length "hi there")) "hi there")
(test (substring "" 0 0) "")
(test (let ((str "012345"))
	(let ((str1 (substring str 2 4)))
	  (string-set! str1 1 #\x))
	(string=? str "012345"))
      #t)
(test (substring (substring "hiho" 0 2) 1) "i")
(test (substring (substring "hiho" 0 2) 2) "")
(test (substring (substring "hiho" 0 2) 0 1) "h")
(test (substring "hi\nho" 3 5) "ho")
(test (substring (substring "hi\nho" 1 4) 2) "h")
(test (substring (substring "hi\nho" 3 5) 1 2) "o")
(test (substring "hi\"ho" 3 5) "ho")
(test (substring (substring "hi\"ho" 1 4) 2) "h")
(test (substring (substring "hi\"ho" 3 5) 1 2) "o")
(test (substring "01\ \ 34" 2) "34")


(test (recompose 12 (lambda (a) (substring a 0 3)) "12345") "123")
(test (reinvert 12 (lambda (a) (substring a 0 3)) (lambda (a) (string-append a "45")) "12345") "12345")

(test (substring "ab" 0 3) 'error)
(test (substring "ab" 3 3) 'error)
(test (substring "ab" 2 3) 'error)
(test (substring "" 0 1) 'error)
(test (substring "" -1 0) 'error)
(test (substring "abc" -1 0) 'error)
(test (substring "hiho" (expt 2 32) (+ 2 (expt 2 32))) 'error)
(test (substring) 'error)
(test (substring "hiho" 0 1 2) 'error)
(test (substring "1234" -1 -1) 'error)
(test (substring "1234" 1 0) 'error)
(test (substring "" most-positive-fixnum 1) 'error)

(test (let ((str "0123456789"))
	(string-set! str 5 #\null)
	(substring str 6))
      "6789")

(for-each
 (lambda (arg)
   (test (substring "hiho" arg 0) 'error))
 (list "hi" #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring "hiho" 1 arg) 'error))
 (list "hi" #\a 0 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring arg 1 2) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



(test (string-append "hi" "ho") "hiho")
(test (string-append "hi") "hi")
(test (string-append "hi" "") "hi")
(test (string-append "hi" "" "ho") "hiho")
(test (string-append "" "hi") "hi")
(test (string-append) "")
(test (string-append "a" (string-append (string-append "b" "c") "d") "e") "abcde")
(test (string-append "a" "b" "c" "d" "e") "abcde")
(test (string-append (string-append) (string-append (string-append))) "")
(test (let ((hi "hi")) (let ((ho (string-append hi))) (eq? hi ho))) #f)
(test (let ((hi "hi")) (let ((ho (string-append hi))) (string-set! ho 0 #\a) hi)) "hi")
(test (let ((hi "hi")) (set! hi (string-append hi hi hi hi)) hi) "hihihihi")
(test (string-append '()) 'error)
(test (string=? (string-append "012" (string #\null) "456") 
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #t)
(test (string=? (string-append "012" (string #\null) "356") 
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #f)

(num-test (letrec ((hi (lambda (str n)
			 (if (= n 0)
			     str
			     (hi (string-append str "a") (- n 1))))))
	    (string-length (hi "" 100)))
	  100)

(test (let* ((str "hiho")
	     (str1 "ha")
	     (str2 (string-append str1 str)))
	(string-set! str2 1 #\x)
	(string-set! str2 4 #\x)
	(and (string=? str "hiho")
	     (string=? str1 "ha")
	     (string=? str2 "hxhixo")))
      #t)
(test (let* ((str (string-copy "hiho"))
	     (str1 (string-copy "ha"))
	     (str2 (string-append str1 str)))
	(string-set! str1 1 #\x)
	(string-set! str 2 #\x)
	(and (string=? str "hixo")
	     (string=? str1 "hx")
	     (string=? str2 "hahiho")))
      #t)

(test (recompose 12 string-append "x") "x")
(test (recompose 12 (lambda (a) (string-append a "x")) "a") "axxxxxxxxxxxx")
(test (recompose 12 (lambda (a) (string-append "x" a)) "a") "xxxxxxxxxxxxa")

(test (string-append "hi" 1) 'error)
(for-each
 (lambda (arg)
   (test (string-append "hiho" arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


(test (let ((str (make-string 4 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c)
	   (string-set! str ctr c)
	   (set! ctr (+ ctr 1)))
	 "1234")
	str)
      "1234")

(test (let ((str (make-string 8 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c1 c2)
	   (string-set! str ctr c1)
	   (string-set! str (+ ctr 1) c2)
	   (set! ctr (+ ctr 2)))
	 "1234"
	 "hiho")
	str)
      "1h2i3h4o")


(test (string->list "abc") (list #\a #\b #\c))
(test (string->list "") '())
(test (string->list (make-string 0)) '())
(test (string->list (string #\null)) '()) ; should this be '(#\null) ? -- this is what Guile returns
(test (string->list (string)) '())
(test (string->list (substring "hi" 0 0)) '())
(test (string->list (list->string (list #\a #\b #\c))) (list #\a #\b #\c))
(test (string->list (list->string '())) '())
(test (list->string (string->list "abc")) "abc")
(test (list->string (string->list "hi there")) "hi there")
(test (list->string (string->list "&*#%^@%$)~@")) "&*#%^@%$)~@")
(test (list->string (string->list "")) "")
(test (let* ((str "abc")
	     (lst (string->list str)))
	(and (string=? str "abc")
	     (equal? lst (list #\a #\b #\c))))
      #t)
(test (list->string '()) "")

(test (list->string (list #\a #\b #\c)) "abc")
(test (list->string (list)) "")

(test (list->string (list #\" #\# #\")) "\"#\"")
(test (list->string (list #\\ #\\ #\# #\\ #\# #\#)) "\\\\#\\##")
(test (list->string (list #\' #\' #\` #\")) '"''`\"")

(test (reinvert 12 string->list list->string "12345") "12345")

(test (string->list) 'error)
(test (list->string) 'error)
(test (string->list "hi" "ho") 'error)
(test (list->string '() '(1 2)) 'error)

(for-each
 (lambda (arg)
   (test (string->list arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->string x)) 'error)

(for-each
 (lambda (arg)
   (test (list->string arg) 'error))
 (list "hi" #\a 1 ''foo '(1 . 2) (cons #\a #\b) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



#|
(define (all-strs len file)
  (let* ((funny-chars (list #\` #\# #\, #\@ #\' #\" #\. #\( #\) #\\))
	 (num-chars (length funny-chars)))
    (let ((ctrs (make-vector len 0)))

      (do ((i 0 (+ i 1)))
	  ((= i (expt num-chars len)))
	(let ((carry #t))
	  (do ((k 0 (+ k 1)))
	      ((or (= k len)
		   (not carry)))
	    (vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
	    (if (= (vector-ref ctrs k) num-chars)
		(vector-set! ctrs k 0)
		(set! carry #f)))

	  (let ((strlst '()))
	    (do ((k 0 (+ k 1)))
		((= k len))
	      (let ((c (list-ref funny-chars (vector-ref ctrs k))))
		(set! strlst (cons c strlst))))

	    (let ((str (list->string strlst)))
	      (format file "(test (and (string=? ~S (string ~{#\\~C~^ ~})) (equal? '~A (string->list ~S))) #t)~%" str strlst strlst str))))))))

(call-with-output-file "strtst.scm"
  (lambda (p)
    (do ((len 3 (+ len 1)))
	((= len 5))
      (all-strs len p))))

(load "strtst.scm")
|#

(test (and (string=? "\"" (string #\")) (equal? '(#\") (string->list "\""))) #t)
(test (and (string=? "#\\" (string #\# #\\)) (equal? '(#\# #\\) (string->list "#\\"))) #t)
(test (and (string=? "#(" (string #\# #\()) (equal? '(#\# #\() (string->list "#("))) #t)
(test (and (string=? "\"@" (string #\" #\@)) (equal? '(#\" #\@) (string->list "\"@"))) #t)
(test (and (string=? "\";" (string #\" #\;)) (equal? '(#\" #\;) (string->list "\";"))) #t)
(test (and (string=? ")(" (string #\) #\()) (equal? '(#\) #\() (string->list ")("))) #t)
(test (and (string=? "`)#" (string #\` #\) #\#)) (equal? '(#\` #\) #\#) (string->list "`)#"))) #t)
(test (and (string=? "##\\" (string #\# #\# #\\)) (equal? '(#\# #\# #\\) (string->list "##\\"))) #t)
(test (and (string=? "#\"(" (string #\# #\" #\()) (equal? '(#\# #\" #\() (string->list "#\"("))) #t)
(test (and (string=? "#.@" (string #\# #\. #\@)) (equal? '(#\# #\. #\@) (string->list "#.@"))) #t)
(test (and (string=? ",`@" (string #\, #\` #\@)) (equal? '(#\, #\` #\@) (string->list ",`@"))) #t)
(test (and (string=? "',@" (string #\' #\, #\@)) (equal? '(#\' #\, #\@) (string->list "',@"))) #t)
(test (and (string=? "\"#@" (string #\" #\# #\@)) (equal? '(#\" #\# #\@) (string->list "\"#@"))) #t)
(test (and (string=? "\")\"" (string #\" #\) #\")) (equal? '(#\" #\) #\") (string->list "\")\""))) #t)
(test (and (string=? ")#(" (string #\) #\# #\()) (equal? '(#\) #\# #\() (string->list ")#("))) #t)
(test (and (string=? "`(,@" (string #\` #\( #\, #\@)) (equal? '(#\` #\( #\, #\@) (string->list "`(,@"))) #t)
(test (and (string=? "`)#\"" (string #\` #\) #\# #\")) (equal? '(#\` #\) #\# #\") (string->list "`)#\""))) #t)
(test (and (string=? "#\"'#" (string #\# #\" #\' #\#)) (equal? '(#\# #\" #\' #\#) (string->list "#\"'#"))) #t)
(test (and (string=? "#(@\\" (string #\# #\( #\@ #\\)) (equal? '(#\# #\( #\@ #\\) (string->list "#(@\\"))) #t)
(test (and (string=? "#(\\\\" (string #\# #\( #\\ #\\)) (equal? '(#\# #\( #\\ #\\) (string->list "#(\\\\"))) #t)
(test (and (string=? ",,.@" (string #\, #\, #\. #\@)) (equal? '(#\, #\, #\. #\@) (string->list ",,.@"))) #t)
(test (and (string=? ",@`\"" (string #\, #\@ #\` #\")) (equal? '(#\, #\@ #\` #\") (string->list ",@`\""))) #t)
(test (and (string=? "\"'\")" (string #\" #\' #\" #\))) (equal? '(#\" #\' #\" #\)) (string->list "\"'\")"))) #t)
(test (and (string=? "\")#\"" (string #\" #\) #\# #\")) (equal? '(#\" #\) #\# #\") (string->list "\")#\""))) #t)
(test (and (string=? "(\\`)" (string #\( #\\ #\` #\))) (equal? '(#\( #\\ #\` #\)) (string->list "(\\`)"))) #t)
(test (and (string=? "))\"'" (string #\) #\) #\" #\')) (equal? '(#\) #\) #\" #\') (string->list "))\"'"))) #t)
(test (and (string=? "\\,\\\"" (string #\\ #\, #\\ #\")) (equal? '(#\\ #\, #\\ #\") (string->list "\\,\\\""))) #t)
(test (and (string=? "\\\"`\"" (string #\\ #\" #\` #\")) (equal? '(#\\ #\" #\` #\") (string->list "\\\"`\""))) #t)
(test (and (string=? "\\\\#\"" (string #\\ #\\ #\# #\")) (equal? '(#\\ #\\ #\# #\") (string->list "\\\\#\""))) #t)



(test (symbol->string 'hi) "hi")
(test (string->symbol (symbol->string 'hi)) 'hi)
(test (eq? (string->symbol "hi") 'hi) #t)
(test (eq? (string->symbol "hi") (string->symbol "hi")) #t)

(test (string->symbol "hi") 'hi)

(test (let ((str (symbol->string 'hi)))
	(catch #t (lambda () (string-set! str 1 #\x)) (lambda args 'error)) ; can be disallowed
	(symbol->string 'hi))
      "hi")

(test (symbol->string 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
(test (string->symbol "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
      'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 32))
	(+ sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
      33)

; all of these are errors now
(test (symbol->string (string->symbol "hi there")) 'error)
;(test (symbol->string (string->symbol "Hi There")) "Hi There")
;(test (symbol->string (string->symbol "HI THERE")) "HI THERE")
;(test (symbol->string (string->symbol "")) "")
;(test (symbol? (string->symbol "(weird name for a symbol!)")) #t)      
;(test (symbol->string (string->symbol "()")) "()")

(test (string->symbol "0") 'error) ; s7 specific
(test (string->symbol "0e") '0e)
(test (string->symbol "1+") '1+)
(test (string->symbol "1+i") 'error)
(test (string->symbol ":0") ':0)
;(test (symbol->string (string->symbol "")) "")
(test (string->symbol (string)) 'error)
(test (string->symbol "") 'error)

(test (reinvert 12 string->symbol symbol->string "hiho") "hiho")

(test (symbol->string) 'error)
(test (string->symbol) 'error)
(test (symbol->string 'hi 'ho) 'error)
(test (string->symbol "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (test (symbol->string arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string->symbol arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



(let ((sym 0))
  (test (symbol->value 'sym) 0)
  (for-each
   (lambda (arg)
     (set! sym arg)
     (test (symbol->value 'sym) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (symbol->value arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))
  
(test (symbol->value) 'error)
(test (symbol->value 'hi 'ho) 'error)

(test (let ((name "hiho"))
	(string-set! name 2 #\null)
	(string->symbol name))
      'error)




;;; --------------------------------------------------------------------------------
;;; LISTS
;;; --------------------------------------------------------------------------------

(test (cons 'a '()) '(a))
(test (cons '(a) '(b c d)) '((a) b c d))
(test (cons "a" '(b c)) '("a" b c))
(test (cons 'a 3) '(a . 3))
(test (cons '(a b) 'c) '((a b) . c))
(test (cons '() '()) '(()))
(test (cons '() 1) '(() . 1))
(test (cons 1 2) '(1 . 2))
(test (cons 1 '()) '(1))
(test (cons '() 2) '(() . 2))
(test (cons 1 (cons 2 (cons 3 (cons 4 '())))) '(1 2 3 4))
(test (cons 'a 'b) '(a . b))
(test (cons 'a (cons 'b (cons 'c '()))) '(a b c))
(test (cons 'a (list 'b 'c 'd)) '(a b c d))
(test (cons 'a (cons 'b (cons 'c 'd))) '(a b c . d))
(test '(a b c d e) '(a . (b . (c . (d . (e . ()))))))
(test (cons (cons 1 2) (cons 3 4)) '((1 . 2) 3 . 4))
(test (list (cons 1 2) (cons 3 4)) '((1 . 2) (3 . 4)))
(test (cons (cons 1 (cons 2 3)) 4) '((1 . (2 . 3)) . 4))
(test (cons (cons 1 (cons 2 '())) (cons 1 2)) '((1 2) . (1 . 2)))
(test (let ((lst (list 1 2))) (list (apply cons lst) lst)) '((1 . 2) (1 2)))
(test (let ((lst (list 1 2))) (list lst (apply cons lst))) '((1 2) (1 . 2)))
(test (cdadr (let ((lst (list 1 2))) (list (apply cons lst) lst))) '(2))
(test (cons 1 '()) '(
                      1
		       ))

(test (car (list 1 2 3)) 1)
(test (car (cons 1 2)) 1)
(test (car (list 1)) 1)
(test (car '(1 2 3)) 1)
(test (car '(1)) 1)
(test (car '(1 . 2)) 1)
(test (car '((1 2) 3)) '(1 2))
(test (car '(((1 . 2) . 3) 4)) '((1 . 2) . 3))
(test (car (list (list) (list 1 2))) '())
(test (car '(a b c)) 'a)
(test (car '((a) b c d)) '(a))
(test (car (reverse (list 1 2 3 4))) 4)
(test (car (list 'a 'b 'c 'd 'e 'f 'g)) 'a)
(test (car '(a b c d e f g)) 'a)
(test (car '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((1 2 3) 4) 5) (6 7)))
(test (car '(a)) 'a)
(test (car '(1 ^ 2)) 1)
(test (car '(1 .. 2)) 1)
(test (car ''foo) 'quote)
(test (car '(1 2 . 3)) 1)
(test (car (cons 1 '())) 1)

(for-each
 (lambda (arg)
   (if (not (equal? (car (cons arg '())) arg))
       (format #t "(car '(~A)) returned ~A?~%" arg (car (cons arg '())))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (reinvert 12 car (lambda (a) (cons a '())) '(1)) '(1))


(test (cdr (list 1 2 3)) '(2 3))
(test (cdr (cons 1 2)) 2)
(test (cdr (list 1)) '())
(test (cdr '(1 2 3)) '(2 3))
(test (cdr '(1)) '())
(test (cdr '(1 . 2)) 2)
(test (cdr '((1 2) 3)) '(3))
(test (cdr '(((1 . 2) . 3) 4)) '(4))
(test (cdr (list (list) (list 1 2))) '((1 2)))
(test (cdr '(a b c)) '(b c))
(test (cdr '((a) b c d)) '(b c d))
(test (equal? (cdr (reverse (list 1 2 3 4))) 4) #f)
(test (equal? (cdr (list 'a 'b 'c 'd 'e 'f 'g)) 'a) #f)
(test (cdr '((((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f) g)) '(g))
(test (cdr '(a)) '())
(test (cdr '(a b c d e f g)) '(b c d e f g))
(test (cdr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((u v w) x) y) ((q w e) r) (a b c) e f g))
(test (cdr ''foo) '(foo))
(test (cdr (cons (cons 1 2) (cons 3 4))) '(3 . 4))
(test (cdr '(1 2 . 3)) '(2 . 3))

(for-each
 (lambda (arg)
   (if (not (equal? (cdr (cons '() arg)) arg))
       (format #t "(cdr '(() ~A) -> ~A?~%" arg (cdr (cons '() arg)))))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(define (cons-r a b n) (if (= 0 n) (cons a b) (cons (cons-r (+ a 1) (+ b 1) (- n 1)) (cons-r (- a 1) (- b 1) (- n 1)))))
(define (list-r a b n) (if (= 0 n) (list a b) (list (list-r (+ a 1) (+ b 1) (- n 1)) (list-r (- a 1) (- b 1) (- n 1)))))

(define lists (list (list 1 2 3)
		    (cons 1 2)
		    (list 1)
		    (list)
		    (list (list 1 2) (list 3 4))
		    (list (list 1 2) 3)
		    '(1 . 2)
		    '(a b c)
		    '((a) b (c))
		    '((1 2) (3 4))
		    '((1 2 3) (4 5 6) (7 8 9))
		    '(((1) (2) (3)) ((4) (5) (6)) ((7) (8) (9)))
		    '((((1 123) (2 124) (3 125) (4 126)) ((5) (6) (7) (8)) ((9) (10) (11) (12)) ((13) (14) (15) (16)))
		      (((21 127) (22 128) (23 129) (24 130)) ((25) (26) (27) (28)) ((29) (30) (31) (32)) ((33) (34) (35) (36)))
		      (((41 131) (42 132) (43 133) (44 134)) ((45) (46) (47) (48)) ((49) (50) (51) (52)) ((53) (54) (55) (56)))
		      (((61 135) (62 136) (63 137) (64 138)) ((65) (66) (67) (68)) ((69) (70) (71) (72)) ((73) (74) (75) (76)))
		      321)
		    (cons 1 (cons 2 (cons 3 4)))
		    (cons (cons 2 (cons 3 4)) 5)
		    (cons '() 1)
		    (cons 1 '())
		    (cons '() '())
		    (list 1 2 (cons 3 4) 5 (list (list 6) 7))
		    (cons-r 0 0 4)
		    (cons-r 0 0 5)
		    (cons-r 0 0 10)
		    (list-r 0 0 3)
		    (list-r 0 0 7)
		    (list-r 0 0 11)
		    ''a
		    ))

(define (caar-1 x) (car (car x)))
(define (cadr-1 x) (car (cdr x)))
(define (cdar-1 x) (cdr (car x)))
(define (cddr-1 x) (cdr (cdr x)))
(define (caaar-1 x) (car (car (car x))))
(define (caadr-1 x) (car (car (cdr x))))
(define (cadar-1 x) (car (cdr (car x))))
(define (caddr-1 x) (car (cdr (cdr x))))
(define (cdaar-1 x) (cdr (car (car x))))
(define (cdadr-1 x) (cdr (car (cdr x))))
(define (cddar-1 x) (cdr (cdr (car x))))
(define (cdddr-1 x) (cdr (cdr (cdr x))))
(define (caaaar-1 x) (car (car (car (car x)))))
(define (caaadr-1 x) (car (car (car (cdr x)))))
(define (caadar-1 x) (car (car (cdr (car x)))))
(define (caaddr-1 x) (car (car (cdr (cdr x)))))
(define (cadaar-1 x) (car (cdr (car (car x)))))
(define (cadadr-1 x) (car (cdr (car (cdr x)))))
(define (caddar-1 x) (car (cdr (cdr (car x)))))
(define (cadddr-1 x) (car (cdr (cdr (cdr x)))))
(define (cdaaar-1 x) (cdr (car (car (car x)))))
(define (cdaadr-1 x) (cdr (car (car (cdr x)))))
(define (cdadar-1 x) (cdr (car (cdr (car x)))))
(define (cdaddr-1 x) (cdr (car (cdr (cdr x)))))
(define (cddaar-1 x) (cdr (cdr (car (car x)))))
(define (cddadr-1 x) (cdr (cdr (car (cdr x)))))
(define (cdddar-1 x) (cdr (cdr (cdr (car x)))))
(define (cddddr-1 x) (cdr (cdr (cdr (cdr x)))))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t "(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'caar 'cadr 'cdar 'cddr 'caaar 'caadr 'cadar 'cdaar 'caddr 'cdddr 'cdadr 'cddar 
       'caaaar 'caaadr 'caadar 'cadaar 'caaddr 'cadddr 'cadadr 'caddar 'cdaaar 
       'cdaadr 'cdadar 'cddaar 'cdaddr 'cddddr 'cddadr 'cdddar)
 
 (list caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar 
       cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar)
 
 (list caar-1 cadr-1 cdar-1 cddr-1 caaar-1 caadr-1 cadar-1 cdaar-1 caddr-1 cdddr-1 cdadr-1 cddar-1 
       caaaar-1 caaadr-1 caadar-1 cadaar-1 caaddr-1 cadddr-1 cadadr-1 caddar-1 cdaaar-1 
       cdaadr-1 cdadar-1 cddaar-1 cdaddr-1 cddddr-1 cddadr-1 cdddar-1))



(test (equal? (cadr (list 'a 'b 'c 'd 'e 'f 'g)) 'b) #t)
(test (equal? (cddr (list 'a 'b 'c 'd 'e 'f 'g)) '(c d e f g)) #t)
(test (equal? (caddr (list 'a 'b 'c 'd 'e 'f 'g)) 'c) #t)
(test (equal? (cdddr (list 'a 'b 'c 'd 'e 'f 'g)) '(d e f g)) #t)
(test (equal? (cadddr (list 'a 'b 'c 'd 'e 'f 'g)) 'd) #t)
(test (equal? (cddddr (list 'a 'b 'c 'd 'e 'f 'g)) '(e f g)) #t)
(test (equal? (caadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '((u v w) x)) #t)
(test (equal? (cadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(6 7)) #t)
(test (equal? (cdaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(5)) #t)
(test (equal? (cdadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(y)) #t)
(test (equal? (cddar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '()) #t)
(test (equal? (caaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(1 2 3)) #t)
(test (equal? (caadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (caaddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(q w e)) #t)
(test (equal? (cadaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 5) #t)
(test (equal? (cadadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 'y) #t)
(test (equal? (caddar (list (list (list (list (list 1 2 3) 4) 5) 1 6 (list 5 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (cadddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(a b c)) #t)
(test (equal? (cdaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(4)) #t)
(test (equal? (cdaadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(x)) #t)
(test (equal? (cdadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(7)) #t)

(test (caar '((a) b c d e f g)) 'a)
(test (cadr '(a b c d e f g)) 'b)
(test (cdar '((a b) c d e f g)) '(b))
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caaar '(((a)) b c d e f g)) 'a)
(test (caadr '(a (b) c d e f g)) 'b)
(test (cadar '((a b) c d e f g)) 'b)
(test (caddr '(a b c d e f g)) 'c)
(test (cdaar '(((a b)) c d e f g)) '(b))
(test (cdadr '(a (b c) d e f g)) '(c))
(test (cddar '((a b c) d e f g)) '(c))
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (caaaar '((((a))) b c d e f g)) 'a)
(test (caaadr '(a ((b)) c d e f g)) 'b)
(test (caadar '((a (b)) c d e f g)) 'b)
(test (caaddr '(a b (c) d e f g)) 'c)
(test (cadaar '(((a b)) c d e f g)) 'b)
(test (cadadr '(a (b c) d e f g)) 'c)
(test (caddar '((a b c) d e f g)) 'c)
(test (cadddr '(a b c d e f g)) 'd)
(test (cdaaar '((((a b))) c d e f g)) '(b))
(test (cdaadr '(a ((b c)) d e f g)) '(c))
(test (cdadar '((a (b c)) d e f g)) '(c))
(test (cdaddr '(a b (c d) e f g)) '(d))
(test (cddaar '(((a b c)) d e f g)) '(c))
(test (cddadr '(a (b c d) e f g)) '(d))
(test (cdddar '((a b c d) e f g)) '(d))
(test (cddddr '(a b c d e f g)) '(e f g))
(test (cadr '(1 2 . 3)) 2)
(test (cddr '(1 2 . 3)) 3)

;; sacla
(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (caar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((1 2 3) 4) 5))
(test (cadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((u v w) x) y))
(test (cdar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((6 7)))
(test (cddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((q w e) r) (a b c) e f g))
(test (caaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((1 2 3) 4))
(test (caadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((u v w) x))
(test (cadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(6 7))
(test (caddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((q w e) r))
(test (cdaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(5))
(test (cdadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(y))
(test (cddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cdddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((a b c) e f g))
(test (caaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(1 2 3))
(test (caaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(u v w))
(test (caadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 6)
(test (caaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(q w e))
(test (cadaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 5)
(test (cadadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'y)
(test (cadddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(a b c))
(test (cdaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(4))
(test (cdaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(x))
(test (cdadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(7))
(test (cdaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(r))
(test (cddaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cddadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '())
(test (cddddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(e f g))

(test (cadr '(a b c d e f g)) 'b)
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caddr '(a b c d e f g)) 'c)
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (cadddr '(a b c d e f g)) 'd)
(test (cddddr '(a b c d e f g)) '(e f g))

(test (caar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((a . b) c . d))
(test (caar '(((a . b) c . d) (e . f) g . h)) '(a . b))
(test (caar '((a . b) c . d)) 'a)
(test (cadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((i . j) k . l))
(test (cadr '(((a . b) c . d) (e . f) g . h)) '(e . f))
(test (cadr '((a . b) c . d)) 'c)
(test (cdar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((e . f) g . h))
(test (cdar '(((a . b) c . d) (e . f) g . h)) '(c . d))
(test (cdar '((a . b) c . d)) 'b)
(test (cddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((m . n) o . p))
(test (cddr '(((a . b) c . d) (e . f) g . h)) '(g . h))
(test (cddr '((a . b) c . d)) 'd)
(test (caaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(a . b))
(test (caaar '(((a . b) c . d) (e . f) g . h)) 'a)
(test (caadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(i . j))
(test (caadr '(((a . b) c . d) (e . f) g . h)) 'e)
(test (cddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(g . h))
(test (cddar '(((a . b) c . d) (e . f) g . h)) 'd)
(test (cdddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(o . p))
(test (cdddr '(((a . b) c . d) (e . f) g . h)) 'h)
(test (caaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'a)
(test (caaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'i)
(test (caddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'g)
(test (cadddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'o)
(test (cdaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'b)
(test (cdaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'j)
(test (cdddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'h)
(test (cddddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'p)

(test (cadr ''foo) 'foo)

(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (recompose 10 cdr '(1 2 3 4 5 6 7 8 9 10 11 12)) '(11 12))
(test (recompose 10 car '(((((((((((1 2 3)))))))))))) '(1 2 3))

(test (cons 1 . 2) 'error)
(test-w "(1 . 2 . 3)")
(test (car (list)) 'error)
(test (car '()) 'error)
(test (cdr (list)) 'error)
(test (cdr '()) 'error)
(test (caddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (cdddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (caar '(a b c d e f g)) 'error)
(test (cdar '(a b c d e f g)) 'error)
(test (caaar '(a b c d e f g)) 'error)
(test (caadr '(a b c d e f g)) 'error)
(test (cadar '(a b c d e f g)) 'error)
(test (cdaar '(a b c d e f g)) 'error)
(test (cdadr '(a b c d e f g)) 'error)
(test (cddar '(a b c d e f g)) 'error)
(test (caaaar '(a b c d e f g)) 'error)
(test (caaadr '(a b c d e f g)) 'error)
(test (caadar '(a b c d e f g)) 'error)
(test (caaddr '(a b c d e f g)) 'error)
(test (cadaar '(a b c d e f g)) 'error)
(test (cadadr '(a b c d e f g)) 'error)
(test (caddar '(a b c d e f g)) 'error)
(test (cdaaar '(a b c d e f g)) 'error)
(test (cdaadr '(a b c d e f g)) 'error)
(test (cdadar '(a b c d e f g)) 'error)
(test (cdaddr '(a b c d e f g)) 'error)
(test (cddaar '(a b c d e f g)) 'error)
(test (cddadr '(a b c d e f g)) 'error)
(test (cdddar '(a b c d e f g)) 'error)
(test (caar 'a) 'error)
(test (caar '(a)) 'error)
(test (cadr 'a) 'error)
(test (cadr '(a . b)) 'error)
(test (cdar 'a) 'error)
(test (cdar '(a . b)) 'error)
(test (cddr 'a) 'error)
(test (cddr '(a . b)) 'error)
(test (caaar 'a) 'error)
(test (caaar '(a)) 'error)
(test (caaar '((a))) 'error)
(test (caadr 'a) 'error)
(test (caadr '(a . b)) 'error)
(test (caadr '(a b)) 'error)
(test (cadar 'a) 'error)
(test (cadar '(a . b)) 'error)
(test (cadar '((a . c) . b)) 'error)
(test (caddr 'a) 'error)
(test (caddr '(a . b)) 'error)
(test (caddr '(a c . b)) 'error)
(test (cdaar 'a) 'error)
(test (cdaar '(a)) 'error)
(test (cdaar '((a . b))) 'error)
(test (cdadr 'a) 'error)
(test (cdadr '(a . b)) 'error)
(test (cdadr '(a b . c)) 'error)
(test (cddar 'a) 'error)
(test (cddar '(a . b)) 'error)
(test (cddar '((a . b) . b)) 'error)
(test (cdddr 'a) 'error)
(test (cdddr '(a . b)) 'error)
(test (cdddr '(a c . b)) 'error)
(test (caaaar 'a) 'error)
(test (caaaar '(a)) 'error)
(test (caaaar '((a))) 'error)
(test (caaaar '(((a)))) 'error)
(test (caaadr 'a) 'error)
(test (caaadr '(a . b)) 'error)
(test (caaadr '(a b)) 'error)
(test (caaadr '(a (b))) 'error)
(test (caadar 'a) 'error)
(test (caadar '(a . b)) 'error)
(test (caadar '((a . c) . b)) 'error)
(test (caadar '((a c) . b)) 'error)
(test (caaddr 'a) 'error)
(test (caaddr '(a . b)) 'error)
(test (caaddr '(a c . b)) 'error)
(test (caaddr '(a c b)) 'error)
(test (cadaar 'a) 'error)
(test (cadaar '(a)) 'error)
(test (cadaar '((a . b))) 'error)
(test (cadaar '((a b))) 'error)
(test (cadadr 'a) 'error)
(test (cadadr '(a . b)) 'error)
(test (cadadr '(a b . c)) 'error)
(test (cadadr '(a (b . e) . c)) 'error)
(test (caddar 'a) 'error)
(test (caddar '(a . b)) 'error)
(test (caddar '((a . b) . b)) 'error)
(test (caddar '((a b . c) . b)) 'error)
(test (cadddr 'a) 'error)
(test (cadddr '(a . b)) 'error)
(test (cadddr '(a c . b)) 'error)
(test (cadddr '(a c e . b)) 'error)
(test (cdaaar 'a) 'error)
(test (cdaaar '(a)) 'error)
(test (cdaaar '((a))) 'error)
(test (cdaaar '(((a . b)))) 'error)
(test (cdaadr 'a) 'error)
(test (cdaadr '(a . b)) 'error)
(test (cdaadr '(a b)) 'error)
(test (cdaadr '(a (b . c))) 'error)
(test (cdadar 'a) 'error)
(test (cdadar '(a . b)) 'error)
(test (cdadar '((a . c) . b)) 'error)
(test (cdadar '((a c . d) . b)) 'error)
(test (cdaddr 'a) 'error)
(test (cdaddr '(a . b)) 'error)
(test (cdaddr '(a c . b)) 'error)
(test (cdaddr '(a c b . d)) 'error)
(test (cddaar 'a) 'error)
(test (cddaar '(a)) 'error)
(test (cddaar '((a . b))) 'error)
(test (cddaar '((a b))) 'error)
(test (cddadr 'a) 'error)
(test (cddadr '(a . b)) 'error)
(test (cddadr '(a b . c)) 'error)
(test (cddadr '(a (b . e) . c)) 'error)
(test (cdddar 'a) 'error)
(test (cdddar '(a . b)) 'error)
(test (cdddar '((a . b) . b)) 'error)
(test (cdddar '((a b . c) . b)) 'error)
(test (cddddr 'a) 'error)
(test (cddddr '(a . b)) 'error)
(test (cddddr '(a c . b)) 'error)
(test (cddddr '(a c e . b)) 'error)




(test (length (list 'a 'b 'c 'd 'e 'f)) 6)
(test (length (list 'a 'b 'c 'd)) 4)
(test (length (list 'a (list 'b 'c) 'd)) 3)
(test (length '()) 0)
(test (length '(this-that)) 1)
(test (length '(this - that)) 3)
(test (length '(a b)) 2)
(test (length '(a b c)) 3)
(test (length '(a (b) (c d e))) 3)
(test (length (list 1 (cons 1 2))) 2)
(test (length (list 1 (cons 1 '()))) 2)

(for-each
 (lambda (arg)
   (test (length arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (length 'x) 'error)
(test (length (cons 1 2)) -1)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (infinite? (length x)) #t))
(test (length '(1 2 . 3)) -2)
(test (length) 'error)
(test (length '(1 2 3) #(1 2 3)) 'error)



(test (reverse '(a b c d)) '(d c b a))
(test (reverse '(a b c))  '(c b a))
(test (reverse '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse '()) '())
(test (reverse (list 1 2 3)) '(3 2 1))
(test (reverse (list 1)) '(1))
(test (reverse (list)) (list))
(test (reverse '(1 2 3)) (list 3 2 1))
(test (reverse '(1)) '(1))
(test (reverse '((1 2) 3)) '(3 (1 2)))
(test (reverse '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse (list (list) (list 1 2))) '((1 2) ()))
(test (reverse '((a) b c d)) '(d c b (a)))
(test (reverse (reverse (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse ''foo) '(foo quote))
(test (let ((x (list 1 2 3 4)))
	(let ((y (reverse x)))
	  (and (equal? x (list 1 2 3 4))
	       (equal? y (list 4 3 2 1)))))
      #t)
(test (letrec ((hi (lambda (lst n)
		     (if (= n 0)
			 lst
			 (hi (reverse lst) (- n 1))))))
	(hi (list 1 2 3) 100))
      (list 1 2 3))
(test (let ((var (list 1 2 3))) (reverse (cdr var)) var) (list 1 2 3))
(test (let ((var '(1 2 3))) (reverse (cdr var)) var) '(1 2 3))
(test (let ((var (list 1 (list 2 3)))) (reverse (cdr var)) var) (list 1 (list 2 3)))
(test (let ((var '(1 (2 3)))) (reverse (cdr var)) var) '(1 (2 3)))
(test (let ((var (list (list 1 2) (list 3 4 5)))) (reverse (car var)) var) '((1 2) (3 4 5)))
(test (let ((x '(1 2 3))) (list (reverse x) x)) '((3 2 1) (1 2 3)))
(test (reverse '(1 2)) '(2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '(1 2 3 4)) '(4 3 2 1))

(for-each
 (lambda (lst)
   (if (list? lst)
       (if (not (equal? lst (reverse (reverse lst))))
	   (format #t "(reverse (reverse ~A)) -> ~A?~%" (reverse (reverse lst))))))
 lists)

(for-each
 (lambda (lst)
   (if (list? lst)
       (if (not (equal? lst (reverse (reverse (reverse (reverse lst))))))
	   (format #t "(reverse...(4x) ~A) -> ~A?~%" lst (reverse (reverse (reverse (reverse lst))))))))
 lists)

(test (let ((x (list 1 2 3))) (list (recompose 32 reverse x) x)) '((1 2 3) (1 2 3)))
(test (let ((x (list 1 2 3))) (list (recompose 31 reverse x) x)) '((3 2 1) (1 2 3)))

(test (reverse (cons 1 2)) '(2 . 1))
(test (reverse '(1 . 2)) '(2 . 1))
(test (reverse '(1 2 . 3)) '(3 2 1))
(test (reverse) 'error)
(test (reverse '(1 2 3) '(3 2 1)) 'error)

(for-each
 (lambda (arg)
   (test (reverse arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



(test (reverse! '(1 . 2)) 'error)
(test (reverse! (cons 1 2)) 'error)
(test (reverse! (cons 1 (cons 2 3))) 'error)
(test (reverse!) 'error)
(test (reverse! '(1 2 3) '(3 2 1)) 'error)

(test (reverse! '(a b c d)) '(d c b a))
(test (reverse! '(a b c))  '(c b a))
(test (reverse! '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse! '()) '())
(test (reverse! (list 1 2 3)) '(3 2 1))
(test (reverse! (list 1)) '(1))
(test (reverse! (list)) (list))
(test (reverse! '(1 2 3)) (list 3 2 1))
(test (reverse! '(1)) '(1))
(test (reverse! '((1 2) 3)) '(3 (1 2)))
(test (reverse! '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse! (list (list) (list 1 2))) '((1 2) ()))
(test (reverse! '((a) b c d)) '(d c b (a)))
(test (reverse! (reverse! (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse! ''foo) '(foo quote))
(test (reverse (reverse! (list 1 2 3))) (list 1 2 3))
(test (reverse (reverse! (reverse! (reverse (list 1 2 3))))) (list 1 2 3))

(test (let ((x (list 1 2 3))) (recompose 31 reverse! x)) '(3 2 1))

(for-each
 (lambda (arg)
   (test (reverse! arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) #(1 2 3) "hiho" (lambda (a) (+ a 1))))



(test (pair? 'a) #f)
(test (pair? '()) #f)
(test (pair? '(a b c)) #t)
(test (pair? (cons 1 2)) #t)
(test (pair? ''()) #t)
(test (pair? #f) #f)
(test (pair? (make-vector 6)) #f)
(test (pair? #t) #f)
(test (pair? '(a . b)) #t)
(test (pair? '#(a b))  #f)
(test (pair? (list 1 2)) #t)
(test (pair? (list)) #f)
(test (pair? ''foo) #t)
(test (pair? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (pair? '(this-that)) #t)
(test (pair? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (pair? x) #t))
(test (pair? (list 1 (cons 1 2))) #t)
(test (pair? (list 1 (cons 1 '()))) #t)
(test (pair? (cons 1 '())) #t)
(test (pair? (cons '() '())) #t)
(test (pair? (cons '() 1)) #t)
(test (pair? (list (list))) #t)
(test (pair? '(())) #t)
(test (pair? (cons 1 (cons 2 3))) #t)
(test (pair?) 'error)

(for-each
 (lambda (arg)
   (if (pair? arg)
       (format #t "(pair? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



(test (list? 'a) #f)
(test (list? '()) #t)
(test (list? '(a b c)) #t)
(test (list? (cons 1 2)) #f)
(test (list? ''()) #t)
(test (list? #f) #f)
(test (list? (make-vector 6)) #f)
(test (list? #t) #f)
(test (list? '(a . b)) #f)
(test (list? '#(a b))  #f)
(test (list? (list 1 2)) #t)
(test (list? (list)) #t)
(test (list? ''foo) #t)
(test (list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (list? '(this-that)) #t)
(test (list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (list? x) #f))
(test (list? (list 1 (cons 1 2))) #t)
(test (list? (list 1 (cons 1 '()))) #t)
(test (list? (cons 1 '())) #t)
(test (list? (cons '() '())) #t)
(test (list? (cons '() 1)) #f)
(test (list? (list (list))) #t)
(test (list? '(())) #t)
(test (list? '(1 2 . 3)) #f)
(test (list? (cons 1 (cons 2 3))) #f)
(test (list? '(1 . ())) #t)

(test (list? '(1 2) '()) 'error)
(test (list?) 'error)
(for-each
 (lambda (arg)
   (if (list? arg)
       (format #t "(list? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



(test (null? 'a) '#f)
(test (null? '()) #t)
(test (null? '(a b c)) #f)
(test (null? (cons 1 2)) #f)
(test (null? ''()) #f)
(test (null? #f) #f)
(test (null? (make-vector 6)) #f)
(test (null? #t) #f)
(test (null? '(a . b)) #f)
(test (null? '#(a b))  #f)
(test (null? (list 1 2)) #f)
(test (null? (list)) #t)
(test (null? ''foo) #f)
(test (null? (list 'a 'b 'c 'd 'e 'f)) #f)
(test (null? '(this-that)) #f)
(test (null? '(this - that)) #f)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (null? x) #f))
(test (null? (list 1 (cons 1 2))) #f)
(test (null? (list 1 (cons 1 '()))) #f)
(test (null? (cons 1 '())) #f)
(test (null? (cons '() '())) #f)
(test (null? (cons '() 1)) #f)
(test (null? (list (list))) #f)
(test (null? '(())) #f)
(test (null? '#()) #f)
(test (null? "") #f)

(test (null? () '()) 'error)
(test (null?) 'error)

(for-each
 (lambda (arg)
   (if (null? arg)
       (format #t "(null? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) #<eof> #<undefined> (values) (lambda (a) (+ a 1))))


(test (let ((x (cons 1 2))) (set-car! x 3) x) (cons 3 2))
(test (let ((x (list 1 2))) (set-car! x 3) x) (list 3 2))
(test (let ((x (list (list 1 2) 3))) (set-car! x 22) x) (list 22 3))
(test (let ((x (cons 1 2))) (set-car! x '()) x) (cons '() 2))
(test (let ((x (list 1 (list 2 3 4)))) (set-car! x (list 5 (list 6))) x) (list (list 5 (list 6)) (list 2 3 4)))
(test (let ((x '(((1) 2) (3)))) (set-car! x '((2) 1)) x) '(((2) 1) (3)))
(test (let ((x ''foo)) (set-car! x "hi") x) (list "hi" 'foo))
(test (let ((x '((1 . 2) . 3))) (set-car! x 4) x) '(4 . 3))
(test (let ((x '(1 . 2))) (set-car! x (cdr x)) x) '(2 . 2))
(test (let ((x '(1 . 2))) (set-car! x x) (list? x)) #f)
(test (let ((x (list 1))) (set-car! x '()) x) '(()))
(test (let ((x '(((1 2) . 3) 4))) (set-car! x 1) x) '(1 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! (cdr lst) 4) lst) (cons 1 (cons 4 3)))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! lst 4) lst) (cons 4 (cons 2 3)))

					;(set-car! '(1 . 2) 3)  ??

(test (let ((x (list 1 2))) (set! (car x) 0) x) (list 0 2))
(test (let ((x (cons 1 2))) (set! (cdr x) 0) x) (cons 1 0))


(test (let ((x (cons 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list (list 1 2) 3))) (set-cdr! x 22) x) '((1 2) . 22))
(test (let ((x (cons 1 2))) (set-cdr! x '()) x) (list 1))
(test (let ((x (list 1 (list 2 3 4)))) (set-cdr! x (list 5 (list 6))) x) '(1 5 (6)))
(test (let ((x '(((1) 2) (3)))) (set-cdr! x '((2) 1)) x) '(((1) 2) (2) 1))
(test (let ((x ''foo)) (set-cdr! x "hi") x) (cons 'quote "hi"))
(test (let ((x '((1 . 2) . 3))) (set-cdr! x 4) x) '((1 . 2) . 4))
(test (let ((x '(1 . 2))) (set-cdr! x (cdr x)) x) '(1 . 2))
(test (let ((x '(1 . 2))) (set-cdr! x x) (list? x)) #f)
(test (let ((x (list 1))) (set-cdr! x '()) x) (list 1))
(test (let ((x '(1 . (2 . (3 (4 5)))))) (set-cdr! x 4) x) '(1 . 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-cdr! (cdr lst) 4) lst) (cons 1 (cons 2 4)))

(test (set-car! '() 32) 'error)
(test (set-car! () 32) 'error)
(test (set-car! (list) 32) 'error)
(test (set-car! 'x 32) 'error)
(test (set-car! #f 32) 'error)
(test (set-cdr! '() 32) 'error)
(test (set-cdr! () 32) 'error)
(test (set-cdr! (list) 32) 'error)
(test (set-cdr! 'x 32) 'error)
(test (set-cdr! #f 32) 'error)
(test (set-car!) 'error)
(test (set-cdr!) 'error)
(test (set-car! '(1 2) 1 2) 'error)
(test (set-cdr! '(1 2) 1 2) 'error)


(test (list-ref (list 1 2) 1) 2)
(test (list-ref '(a b c d) 2) 'c)
(test (list-ref (cons 1 2) 0) 1) ; !!
(test (list-ref ''foo 0) 'quote)
(test (list-ref '((1 2) (3 4)) 1) '(3 4))
(test (list-ref (list-ref (list (list 1 2) (list 3 4)) 1) 1) 4)
(test (let ((x (list 1 2 3))) (list-ref x (list-ref x 1))) 3)
					; (list-ref '(1 2 . 3) 0) -- why is this acceptable?

(test (let ((lst (list 1 2))) (set! (list-ref lst 1) 0) lst) (list 1 0))
(test (((lambda () list)) 'a 'b 'c) '(a b c))
(test (apply ((lambda () list)) (list 'a 'b 'c) (list 'c 'd 'e)) '((a b c) c d e))
(test (((lambda () (values list))) 1 2 3) '(1 2 3))
(test (apply list 'a 'b '(c)) '(a b c))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t "(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-ref:0 'list-ref:1 'list-ref:2 'list-ref:3)
 (list car cadr caddr cadddr)
 (list (lambda (l) (list-ref l 0)) (lambda (l) (list-ref l 1)) (lambda (l) (list-ref l 2)) (lambda (l) (list-ref l 3))))

(for-each
 (lambda (arg)
   (test (list-ref (list 1 arg) 1) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 0)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 1)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 100)) 1)

(test (list-ref '((1 2 3) (4 5 6)) 1) '(4 5 6))
(test (list-ref '((1 2 3) (4 5 6)) 1 2) 6)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test ('((1 2 3) (4 5 6)) 1) '(4 5 6))
(test ('((1 2 3) (4 5 6)) 1 2) 6)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (L 1)) '(4 5 6))
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1)) '((7 8 9) (10 11 12)))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) ((L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (((L 1) 0) 2) 3)) 'error)

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3))
  (test (list-ref '((1 2 3) (4 5 6)) one) '(4 5 6))
  (test (list-ref '((1 2 3) (4 5 6)) 1 two) 6)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)
  
  (test ('((1 2 3) (4 5 6)) one) '(4 5 6))
  (test ('((1 2 3) (4 5 6)) 1 two) 6)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)
  
  (test (let ((L '((1 2 3) (4 5 6)))) (L one)) '(4 5 6))
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one)) '((7 8 9) (10 11 12)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero two)) 9)
  
  (test (let ((L '((1 2 3) (4 5 6)))) ((L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) 0 two)) 9)
  
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L one) zero) two)) 9))


(test (list-ref '() 0) 'error)
(test (list-ref (list 1 2) 2) 'error)
(test (list-ref (list 1 2) -1) 'error)
(test (list-ref (list 1 2) 1.3) 'error)
(test (list-ref (list 1 2) 1/3) 'error)
(test (list-ref (list 1 2) 1+2.0i) 'error)
(test (list-ref (cons 1 2) 1) 'error)
(test (list-ref (cons 1 2) 2) 'error)
(test (list-ref (list 1 2 3) (expt 2 32)) 'error)
(test (list-ref '(1 2 3) 1 2) 'error)
(test (list-ref) 'error)
(test (list-ref '(1 2)) 'error)
(test ('(0)) 'error)
(test ((0)) 'error)
(test ('(1 2 3) 1) 2)
(test ((list 1 2 3) 2) 3)
(test ((list)) 'error)
(test ((list 1) 0 0) 'error)
(test ((list 1 (list 2 3)) 1 1) 3)
(test ((append '(3) '() '(1 2)) 0) 3)
(test ((append '(3) '() 1) 0) 3)
(test ((append '(3) '() 1) 1) 'error)
;; this works with 0 because:
(test ((cons 1 2) 0) 1)
(test (list-ref (cons 1 2) 0) 1)
(test (((list (list 1 2 3)) 0) 0) 1)
(test (((list (list 1 2 3)) 0) 1) 2)
(test (((list (list 1 2 3)) 0 1)) 'error) ; see below
(test (let ((lst (list (list 1 2 3)))) (lst 0 1)) 2) 
(test ((list (list 1 2 3)) 0 1) 2)

(let ((lst (list 1 2)))
  (for-each
   (lambda (arg)
     (test (list-ref (list 1 2) arg) 'error)
     (test ((list 1 2) arg) 'error)
     (test (lst arg) 'error))
   (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))



(test (let ((x (list 1))) (list-set! x 0 2) x) (list 2))
(test (let ((x (cons 1 2))) (list-set! x 0 3) x) '(3 . 2))
(test (let ((x '((1) 2))) (list-set! x 0 1) x) '(1 2))
(test (let ((x '(1 2))) (list-set! x 1 (list 3 4)) x) '(1 (3 4)))
(test (let ((x ''foo)) (list-set! x 0 "hi") x ) '("hi" foo))
(test (let ((x (list 1 2))) (list-set! x 0 x) (list? x)) #t)
(test (let ((x (list 1 2))) (list-set! x 1 x) (list? x)) #t)
(test (let ((x 2) (lst '(1 2))) (list-set! (let () (set! x 3) lst) 1 23) (list x lst)) '(3 (1 23)))

(test (list-set! '(1 2 3) 1 4) 4)
(test (set-car! '(1 2) 4) 4)
(test (set-cdr! '(1 2) 4) 4)
(test (fill! (list 1 2) 4) 4)

(for-each
 (lambda (arg)
   (test (let ((x (list 1 2))) (list-set! x 0 arg) (list-ref x 0)) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 2 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 3 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 1 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 4 2 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1) 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1) 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1) 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L '(((1 2 3))))) (set! ((L 0) 0 1) 32) L) '(((1 32 3))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L 0 0 1) 32) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0) 0 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0) 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L 0) 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) 1) 32) L)) '(1 32 3))

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one zero thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero two thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one) thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one) thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  
  (test (let ((L '((1 2 3) (4 5 6)))) (set! ((L one) zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one) zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L one) zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '(((1 2 3))))) (set! ((L zero) zero one) thirty-two) L) '(((1 32 3))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L zero zero one) thirty-two) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero) zero one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero 0 one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero) zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L zero) zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) one) thirty-two) L)) '(1 32 3)))
  
(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) (list x y)) '((32) (2)))
  
(test (list-set! '() 0 1) 'error)
(test (list-set! '() -1 1) 'error)
(test (list-set! '(1) 1 2) 'error)
(test (list-set! '(1 2 3) -1 2) 'error)
(test (list-set! '(1) 1.5 2) 'error)
(test (list-set! '(1) 3/2 2) 'error)
(test (list-set! '(1) 1+3i 2) 'error)
(test (let ((x (cons 1 2))) (list-set! x 1 3) x) 'error)
(test (list-set! '(1 2 3) 1 2 3) 'error)
(test (list-set! (list 1 2 3) (expt 2 32)  0) 'error)

(for-each
 (lambda (arg)
   (test (list-set! (list 1 2) arg arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))




(test (let ((tree1 (list 1 (list 1 2) (list (list 1 2 3)) (list (list (list 1 2 3 4)))))) tree1) '(1 (1 2) ((1 2 3)) (((1 2 3 4)))))
(test (let ((tree2 (list "one" (list "one" "two") (list (list "one" "two" "three"))))) tree2) '("one" ("one" "two") (("one" "two" "three"))))
(test (let ((tree1 (list 1 (list 1 2) (list 1 2 3) (list 1 2 3 4)))) tree1) '(1 (1 2) (1 2 3) (1 2 3 4)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) tree2) '(1 (1 2)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) (eqv? tree1 tree2)) #f)
(test (let ((tree1 (list ''a (list ''b ''c))) (tree2 (list ''a (list ''b ''c)))) tree2) '('a ('b 'c)))
(test (let ((lst (list 1 (list 2 3)))) lst) '(1 (2 3)))
(test (let* ((lst (list 1 (list 2 3))) (slst lst)) slst) '(1 (2 3)))
(test (list 1) '(1))
(test (let ((a 1)) (list a 2)) '(1 2))
(test (let ((a 1)) (list 'a '2)) '(a 2))
(test (let ((a 1)) (list 'a 2)) '(a 2))
(test (list) '())
(test (let ((a (list 1 2))) a) '(1 2))
(test (let ((a (list 1 2))) (list 3 4 'a (car (cons 'b 'c)) (+ 6 -2))) '(3 4 a b 4))
(test (list) '())
(test (length (list quote do map call/cc lambda define if begin set! let let* cond and or for-each)) 15)

(test (list 1 2 . 3) 'error)
(test (list 1 2 , 3) 'error)
(test (list 1 2 ,@ 3) 'error)



(test (list-tail '(1 2 3) 0) '(1 2 3))
(test (list-tail '(1 2 3) 2) '(3))
(test (list-tail '(1 2 3) 3) '())
(test (list-tail '(1 2 3 . 4) 2) '(3 . 4))
(test (list-tail '(1 2 3 . 4) 3) 4)
(test (let ((x (list 1 2 3))) (eq? (list-tail x 2) (cddr x))) #t)
(test (list-tail '() 0) '())
(test (list-tail '() 1) 'error)
(test (list-tail '() -1) 'error)
(test (list-tail (list 1 2) 2) '())
(test (list-tail (cons 1 2) 0) '(1 . 2))
(test (list-tail (cons 1 2) 1) 2)
(test (list-tail ''foo 1) '(foo))
(test (list-tail '((1 2) (3 4)) 1) '((3 4)))
(test (list-tail (list-tail (list-tail '(1 2 3 4) 1) 1) 1) '(4))

(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 0) x))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 1) (cdr x)))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 100) x))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format #t "(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-tail:0 'list-tail:1 'list-tail:2 'list-tail:3 'list-tail:4)
 (list (lambda (l) l) cdr cddr cdddr cddddr)
 (list (lambda (l) (list-tail l 0)) (lambda (l) (list-tail l 1)) (lambda (l) (list-tail l 2)) (lambda (l) (list-tail l 3)) (lambda (l) (list-tail l 4))))

(test (list-tail (list 1 2) 3) 'error)
(test (list-tail (list 1 2) -1) 'error)
(test (list-tail (list 1 2) 1.3) 'error)
(test (list-tail (list 1 2) 1/3) 'error)
(test (list-tail (list 1 2) 1+2.0i) 'error)
(test (list-tail (cons 1 2) 2) 'error)
(test (list-tail '(1 2 . 3)) 'error)
(test (list-tail '(1 2 . 3) 1) '(2 . 3))
(test (list-tail '(1 2 . 3) 0) '(1 2 . 3))
(test (list-tail (list 1 2 3) (+ 1 (expt 2 32))) 'error)
(test (list-tail) 'error)
(test (list-tail '(1)) 'error)

(for-each
 (lambda (arg)
   (test (list-tail (list 1 2) arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))



(let ((e '((a 1) (b 2) (c 3))))
  (test (assq 'a e) '(a 1))
  (test (assq 'b e) '(b 2))
  (test (assq 'd e) #f))
(test (assq (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assq #t e) (list #t 1))
    (test (assq #f e) (list #f 2))
    (test (assq 'a e) (list 'a 3))
    (test (assq xcons e) (list xcons 4))
    (test (assq xvect e) (list xvect 5))
    (test (assq xlambda e) (list xlambda 6))
    (test (assq xstr e) (list xstr 7))
    (test (assq car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3))))
  (test (assq 1+i e) #f)
  (test (assq 3.0 e) #f)
  (test (assq 5/3 e) #f))

(test (assq 'x (cdr (assq 'a '((b . 32) (a . ((a . 12) (b . 32) (x . 1))) (c . 1))))) '(x . 1))

(test (assq #f '(#f 2 . 3)) #f)
(test (assq #f '((#f 2) . 3)) '(#f 2))
(test (assq '() '((() 1) (#f 2))) '(() 1))
(test (assq '() '((1) (#f 2))) #f)
(test (assq #() '((#f 1) (() 2) (#() 3))) #f)  ; (eq? #() #()) -> #f

(test (assq 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assq 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assq 'b (list '(a . 1) '(b . 2) '() '(c . 3) #f)) '(b . 2))
(test (assq 'asdf (list '(a . 1) '(b . 2) '() '(c . 3) #f)) #f)
(test (assq "" (list '("a" . 1) '("" . 2) '(#() . 3))) #f) ; since (eq? "" "") is #f


(test (assv 1 '(1 2 . 3)) #f)
(test (assv 1 '((1 2) . 3)) '(1 2))

(let ((e '((a 1) (b 2) (c 3))))
  (test (assv 'a e) '(a 1))
  (test (assv 'b e) '(b 2))
  (test (assv 'd e) #f))
(test (assv (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assv #t e) (list #t 1))
    (test (assv #f e) (list #f 2))
    (test (assv 'a e) (list 'a 3))
    (test (assv xcons e) (list xcons 4))
    (test (assv xvect e) (list xvect 5))
    (test (assv xlambda e) (list xlambda 6))
    (test (assv xstr e) (list xstr 7))
    (test (assv car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assv 1+i e) '(1+i 1))
  (test (assv 3.0 e) '(3.0 2))
  (test (assv 5/3 e) '(5/3 3))
  (test (assv #\a e) '(#\a 4))
  (test (assv "hiho" e) #f))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assv '(a) e) #f)
  (test (assv '#(a) e) #f)
  (test (assv (string #\c) e) #f))

(let ((lst '((2 . a) (3 . b))))
  (set-cdr! (assv 3 lst) 'c)
  (test lst '((2 . a) (3 . c))))

(test (assv '() '((() 1) (#f 2))) '(() 1))
(test (assv '() '((1) (#f 2))) #f)
(test (assv #() '((#f 1) (() 2) (#() 3))) #f)  ; (eqv? #() #()) -> #f ??

(test (assv 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assv 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assv 'asdf '((a . 1) (b . 2) () (c . 3) . 4)) #f)
(test (assv 'd '((a . 1) (b . 2) () (c . 3) (d . 5))) '(d . 5))


(let ((e '((a 1) (b 2) (c 3))))
  (test (assoc 'a e) '(a 1))
  (test (assoc 'b e) '(b 2))
  (test (assoc 'd e) #f))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assoc #t e) (list #t 1))
    (test (assoc #f e) (list #f 2))
    (test (assoc 'a e) (list 'a 3))
    (test (assoc xcons e) (list xcons 4))
    (test (assoc xvect e) (list xvect 5))
    (test (assoc xlambda e) (list xlambda 6))
    (test (assoc xstr e) (list xstr 7))
    (test (assoc car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assoc 1+i e) '(1+i 1))
  (test (assoc 3.0 e) '(3.0 2))
  (test (assoc 5/3 e) '(5/3 3))
  (test (assoc #\a e) '(#\a 4))
  (test (assoc "hiho" e) '("hiho" 5)))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assoc '(a) e) '((a) 1))
  (test (assoc '#(a) e) '(#(a) 2))
  (test (assoc (string #\c) e) '("c" 3)))

(test (assoc 'a '((b c) (a u) (a i))) '(a u))
(test (assoc 'a '((b c) ((a) u) (a i))) '(a i))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))
(test (assoc 5 '((2 3) (5 7) (11 13))) '(5 7))
(test (assoc 'key '()) #f)
(test (assoc 'key '(() ())) #f)
(test (assoc '() '()) #f)

(test (assoc '() 1) 'error)
(test (assoc (cons 1 2) 1) 'error)
(test (assoc (let ((x (cons 1 2))) (set-cdr! x x)) 1) 'error)
(test (assoc '((1 2) .3) 1) 'error)
(test (assoc ''foo quote) 'error)
(test (assoc 1 '(1 2 . 3)) #f)
(test (assoc 1 '((1 2) . 3)) '(1 2))

(test (assoc '() '((() 1) (#f 2))) '(() 1))
(test (assoc '() '((1) (#f 2))) #f)
(test (assoc #() '((#f 1) (() 2) (#() 3))) '(#() 3))

(for-each
 (lambda (arg)
   (test (assoc arg (list (list 1 2) (list arg 3))) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol #() abs 3/4 #\f #t (if #f #f)))

(test (assoc 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assoc 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assoc 'c '(() (a . 1) (b . 2) () (c . 3) (c . 4) . 4)) '(c . 3))
(test (assoc 'asdf '(() (a . 1) (b . 2) () (c . 3) (c . 4) . 4)) #f)
(test (assoc "" (list '("a" . 1) '("" . 2) '(#() . 3))) '("" . 2))



(test (memq 'a '(a b c)) '(a b c))
(test (memq 'b '(a b c)) '(b c))
(test (memq 'a '(b c d)) #f)
(test (memq (list 'a) '(b (a) c))  #f)
(test (memq 'a '(b a c a d a)) '(a c a d a))
(let ((v (vector 'a))) (test (memq v (list 'a 1.2 v "hi")) (list v "hi")))
(test (memq #f '(1 a #t "hi" #f 2)) '(#f 2))
(test (memq eq? (list 2 eqv? 1 eq?)) (list eq?))
(test (memq eq? (list 2 eqv? 2)) #f)
(test (memq 6 (memq 5 (memq 4 (memq 3 (memq 2 (memq 1 '(1 2 3 4 5 6))))))) '(6))
(test (memq 'a (cons 'a 'b)) '(a . b))
(test (memq 'a (list a b . c)) 'error)
(test (memq) 'error)
(test (memq 'a) 'error)
(test (memq 'a 'b) 'error)
(test (memq 'a '(a b . c)) '(a b . c))
(test (memq 'b '(a b . c)) '(b . c))
(test (memq 'c '(a b . c)) #f) ; or should it be 'c?
(test (memq '() '(1 () 3)) '(() 3))
(test (memq '() '(1 2)) #f)
(test (memq 'a '(c d a b c)) '(a b c))
(test (memq 'a '(c d f b c)) #f)
(test (memq 'a '()) #f)
(test (memq 'a '(c d a b . c)) '(a b . c))
(test (memq 'a '(c d f b . c)) #f)



(test (memv 101 '(100 101 102)) '(101 102))
(test (memv 3.4 '(1.2 2.3 3.4 4.5)) '(3.4 4.5))
(test (memv 3.4 '(1.3 2.5 3.7 4.9)) #f)
(let ((ls (list 'a 'b 'c)))
  (set-car! (memv 'b ls) 'z)
  (test ls '(a z c)))
(test (memv 1 (cons 1 2)) '(1 . 2))
(test (memv 'a (list a b . c)) 'error)
(test (memv 'a '(a b . c)) '(a b . c))
(test (memv 'asdf '(a b . c)) #f)
(test (memv) 'error)
(test (memv 'a) 'error)
(test (memv 'a 'b) 'error)
(test (memv 'c '(a b c)) '(c))
(test (memv 'c '(a b . c)) #f)



(test (member (list 'a) '(b (a) c)) '((a) c))
(test (member "b" '("a" "c" "b")) '("b"))
(test (member 1 '(3 2 1 4)) '(1 4))
(test (member car (list abs car modulo)) (list car modulo))
(test (member do (list quote map do)) (list do))
(test (member 5/2 (list 1/3 2/4 5/2)) '(5/2))
(test (member 'a '(a b c d)) '(a b c d))
(test (member 'b '(a b c d)) '(b c d))
(test (member 'c '(a b c d)) '(c d))
(test (member 'd '(a b c d)) '(d))
(test (member 'e '(a b c d)) #f)
(test (member 1 (cons 1 2)) '(1 . 2))
(test (member 'a (list a b . c)) 'error)
(test (member 1 '(1 2 . 3)) '(1 2 . 3))
(test (member 4 '(1 2 . 3)) #f)
(test (member) 'error)
(test (member 'a) 'error)
(test (member 'a 'b) 'error)
(test (member '() '(1 2 3)) #f)
(test (member '() '(1 2 ())) '(()))
(test (member #() '(1 () 2 #() 3)) '(#() 3))
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (1 2)))) #f)
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (3 4)))) '(#2d((1 2) (3 4))))

(for-each
 (lambda (arg)
   (test (member arg (list 1 2 arg 3)) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol abs 3/4 #\f #t (if #f #f) '(1 2 (3 (4))) most-positive-fixnum))


(for-each
 (lambda (op)
   (test (op) 'error)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format #t "(~A ~A) returned ~A?~%" op arg result))
	(test (op arg '() arg) 'error)))
    (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list cons car cdr set-car! set-cdr! caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       assq assv assoc memq memv member list-ref list-tail))

(for-each
 (lambda (op)
   (test (op '(1) '(2)) 'error))
 (list reverse car cdr caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       list-ref list-tail list-set!))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op #f arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format #t "(~A #f ~A) returned ~A?~%" op arg result))))
    (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list assq assv assoc memq memv member))



(test (append '(a b c) '()) '(a b c))
(test (append '() '(a b c)) '(a b c))
(test (append '(a b) '(c d)) '(a b c d))
(test (append '(a b) 'c) '(a b . c))
(test (equal? (append (list 'a 'b 'c) (list 'd 'e 'f) '() '(g)) '(a b c d e f g)) #t)
(test (append (list 'a 'b 'c) (list 'd 'e 'f) '() (list 'g)) '(a b c d e f g))
(test (append (list 'a 'b 'c) 'd) '(a b c . d))
(test (append '() '()) '())
(test (append '() (list 'a 'b 'c)) '(a b c))
(test (append) '())
(test (append '() 1) 1)
(test (append 'a) 'a)
(test (append '(x) '(y))  '(x y))
(test (append '(a) '(b c d)) '(a b c d))
(test (append '(a (b)) '((c)))  '(a (b) (c)))
(test (append '(a b) '(c . d))  '(a b c . d))
(test (append '() 'a)  'a)
(test (append '(a b) (append (append '(c)) '(e) 'f)) '(a b c e . f))
(test (append ''foo 'foo) '(quote foo . foo))
(test (append '() (cons 1 2)) '(1 . 2))
(test (append '() '() '()) '())
(test (append (cons 1 2)) '(1 . 2))

(test (append #f) #f)
(test (append '() #f) #f)
(test (append '(1 2) #f) '(1 2 . #f))
(test (append '() '() #f) #f)
(test (append '() '(1 2) #f) '(1 2 . #f))
(test (append '(1 2) '() #f) '(1 2 . #f))
(test (append '(1 2) '(3 4) #f) '(1 2 3 4 . #f))
(test (append '() '() '() #f) #f)
(test (append '(1 2) '(3 4) '(5 6) #f) '(1 2 3 4 5 6 . #f))
(test (append () () #()) #())
(test (append () ((lambda () #f))) #f)

(test (append 0) 0) ; is this correct?
(test (append '() 0) 0)
(test (append '() '() 0) 0)
(test (let* ((x '(1 2 3)) (y (append x '()))) (eq? x y)) #f) ; check that append returns a new list
(test (let* ((x '(1 2 3)) (y (append x '()))) (equal? x y)) #t)
(test (let* ((x (list 1 2 3)) (y (append x (list)))) (eq? x y)) #f) 
(test (append '(1) 2) '(1 . 2))
(let ((x (list 1 2 3)))
  (let ((y (append x '())))
    (set-car! x 0)
    (test (= (car y) 1) #t)))
(let ((x (list 1 2 3)))
  (let ((y (append x '())))
    (set-cdr! x 0)
    (test (and (= (car y) 1)
	       (= (cadr y) 2)
	       (= (caddr y) 3))
	  #t)))

(test (let ((xx (list 1 2))) (recompose 12 (lambda (x) (append (list (car x)) (cdr x))) xx)) '(1 2))

(test (append 'a 'b) 'error)
(test (append 'a '()) 'error)
(test (append (cons 1 2) '()) 'error)
(test (append '(1) 2 '(3)) 'error)
(test (append '(1) 2 3) 'error)
(test (let ((lst (list 1 2 3))) (append lst lst)) '(1 2 3 1 2 3))

(for-each
 (lambda (arg)
   (test (append arg) arg))
 (list "hi" #\a #f 'a-symbol (make-vector 3) abs 1 3.14 3/4 1.0+1.0i #t #<unspecified> #<eof> '() #() (list 1 2) (cons 1 2) #(0) (lambda (a) (+ a 1))))

(test (let ((ht (make-hash-table))) (set! (ht 'a) 123) (map values ht)) '((a . 123)))



(test-w "(list #b)")
(test-w "(char? #\\spaces)")
(test-w "(car '( . 1))")
(test-w "(car '(. ))")
(test-w "(car '( . ))")
(test-w "(car '(. . . ))")
(test-w "'#( . 1)")
(test-w "'(1 2 . )")
(test-w "'#(1 2 . )")
(test-w "(+ 1 . . )")
(test-w "(car '(1 . ))")
(test-w "(car '(1 . . 2))")
(test-w "'#( . )")
(test-w "'#(1 . )")
(test-w "'#(. . . )")
(test-w "'#(1 . . 2)")
(test-w "'(. 1)")
(test-w "'#(. 1)")
(test-w "'(. )")
(test-w "'#(. )")
(test-w "(list 1 . 2)")
(test-w "(+ 1 . 2)")
(test-w "(car '@#`')")
(test-w "(list . )")
(test-w "'#( .)")
(test-w "(car '( .))")
(test-w "'#(1 . 2)")
;(test-w "(let ((. 3)) .)")




;;; --------------------------------------------------------------------------------
;;; VECTORS
;;; --------------------------------------------------------------------------------

(test (vector? (make-vector 6)) #t)
(test (vector? (make-vector 6 #\a)) #t)
(test (vector? (make-vector 0)) #t)
;; (test (vector? #*1011) #f)
(test (vector? '#(0 (2 2 2 2) "Anna")) #t)
(test (vector? '#()) #t)
(test (vector? '#("hi")) #t)
(test (vector? (vector 1)) #t)
(test (let ((v (vector 1 2 3))) (vector? v)) #t)

(for-each
 (lambda (arg)
   (test (vector? arg) #f))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (vector?) 'error)
(test (vector? #() #(1)) 'error)

;;; make a shared ref -- we'll check it later after enough has happened that an intervening GC is likely

(define check-shared-vector-after-gc #f)
(let ((avect (make-vector '(6 6) 32)))
  (do ((i 0 (+ i 1)))
      ((= i 6))
    (do ((j 0 (+ j 1)))
	((= j 6))
      (set! (avect i j) (cons i j))))
  (set! check-shared-vector-after-gc (avect 3)))


(test (let ((v (make-vector 3 #f))) (and (vector? v) (= (vector-length v) 3) (eq? (vector-ref v 1) #f))) #t)
(test (let ((v (make-vector 1 1))) (and (vector? v) (= (vector-length v) 1) (vector-ref v 0))) 1)
(test (let ((v (make-vector 0 1))) (and (vector? v) (= (vector-length v) 0))) #t)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) '#(0 1 2 3 4))
(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(test (make-vector 2 'hi) '#(hi hi))
(test (make-vector 0) '#())
(test (make-vector 0 'hi) '#())
(test (make-vector 3 (make-vector 1 'hi)) '#(#(hi) #(hi) #(hi)))
(test (make-vector 3 '#(hi)) '#(#(hi) #(hi) #(hi)))
(test (make-vector 3 (list)) '#(() () ()))
(test (make-vector 3 (make-vector 1 (make-vector 1 'hi))) '#(#(#(hi)) #(#(hi)) #(#(hi))))

(test (let ((v (make-vector 3 0))) (set! (vector-ref v 1) 32) v) #(0 32 0))

(for-each
 (lambda (arg)
   (test (vector-ref (make-vector 1 arg) 0) arg))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (make-vector) 'error)
(test (make-vector 1 #f #t) 'error)
(test (make-vector 1 2 3) 'error)

(for-each
 (lambda (arg)
   (test (make-vector arg) 'error))
 (list #\a '() -1 #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))



(test (vector 1 2 3) '#(1 2 3))
(test (vector 1 '(2) 3) '#(1 (2) 3))
(test (vector) '#())
(test (vector (vector (vector))) '#(#(#())))
(test (vector (vector) (vector) (vector)) '#(#() #() #()))
(test (vector (list)) '#(()))
(test '#(1 #\a "hi" hi) (vector 1 #\a "hi" 'hi))
(test (let ((v (make-vector 4 "hi")))
	(vector-set! v 0 1)
	(vector-set! v 1 #\a)
	(vector-set! v 3 'hi)
	v)
      '#(1 #\a "hi" hi))
(let ((x 34))
  (test (vector x 'x) '#(34 x)))

(for-each
 (lambda (arg)
   (test (vector-ref (vector arg) 0) arg))
 (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))



(test (vector->list '#(0)) (list 0))
(test (vector->list (vector)) '())
(test (vector->list '#(a b c)) '(a b c))
(test (vector->list '#(#(0) #(1))) '(#(0) #(1)))
(test (vector? (list-ref (let ((v (vector 1 2))) (vector-set! v 1 v) (vector->list v)) 1)) #t)

(test (list->vector '()) '#())
(test (list->vector '(a b c)) '#(a b c))
(test (list->vector (list (list 1 2) (list 3 4))) '#((1 2) (3 4)))
(test (list->vector ''foo) '#(quote foo))
(test (list->vector (list)) '#())
(test (list->vector (list 1)) '#(1))
(test (list->vector (list (list))) '#(()))
(test (list->vector (list 1 #\a "hi" 'hi)) '#(1 #\a "hi" hi))

(for-each
 (lambda (arg)
   (if (list? arg)
       (test (vector->list (list->vector arg)) arg)))
 lists)
(set! lists '())

(test (list->vector (vector->list (vector))) '#())
(test (list->vector (vector->list (vector 1))) '#(1))
(test (vector->list (list->vector (list))) '())
(test (vector->list (list->vector (list 1))) '(1))

(test (reinvert 12 vector->list list->vector #(1 2 3)) #(1 2 3))

(test (vector->list) 'error)
(test (list->vector) 'error)
(test (vector->list #(1) #(2)) 'error)
(test (list->vector '(1) '(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector->list arg) 'error))
 (list #\a 1 '() (list 1) '(1 . 2) #f 'a-symbol "hi" abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->vector x)) 'error)
(test (list->vector (cons 1 2)) 'error)
(test (list->vector '(1 2 . 3)) 'error)

(for-each
 (lambda (arg)
   (test (list->vector arg) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



(test (vector-length (vector)) 0)
(test (vector-length (vector 1)) 1)
(test (vector-length (make-vector 128)) 128)
(test (vector-length '#(a b c d e f)) 6)
(test (vector-length '#()) 0)
(test (vector-length (vector #\a (list 1 2) (vector 1 2))) 3)
(test (vector-length '#(#(#(hi)) #(#(hi)) #(#(hi)))) 3)
(test (vector-length (vector 1 2 3 4)) 4)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) v)) 2)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) (vector-ref v 1))) 2)

(test (vector-length) 'error)
(test (vector-length #(1) #(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector-length arg) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))



(test (vector-ref '#(1 1 2 3 5 8 13 21) 5) 8)
(test (vector-ref '#(1 1 2 3 5 8 13 21) (let ((i (round (* 2 (acos -1))))) (if (inexact? i) (inexact->exact i)  i))) 13)
(test (let ((v (make-vector 1 0))) (vector-ref v 0)) 0)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 1)) (list 2))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 2)) '#(#\a #\a #\a))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 1)) #\a)
(test (vector-ref '#(a b c) 1) 'b)
(test (vector-ref '#(()) 0) '())
(test (vector-ref '#(#()) 0) '#())
(test (vector-ref (vector-ref (vector-ref '#(1 (2) #(3 (4) #(5))) 2) 2) 0) 5)
(test (let ((v (vector 1 2))) (vector-set! v 1 v) (eq? (vector-ref v 1) v)) #t)

(test (vector-ref) 'error)
(test (vector-ref #(1)) 'error)
(test (vector-ref #(1) 0 0) 'error)

(test (let ((v (make-vector 1 0))) (vector-ref v 1)) 'error)
(test (let ((v (make-vector 1 0))) (vector-ref v -1)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 3)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 3) 0)) 'error)
(test (vector-ref (vector) 0) 'error)
(test (vector-ref '#() 0) 'error)
(test (vector-ref '#() -1) 'error)
(test (vector-ref '#() 1) 'error)

(test (#(1 2) 1) 2)
(test (#(1 2) 1 2) 'error)
(test ((#("hi" "ho") 0) 1) #\i)
(test (((vector (list 1 2) (cons 3 4)) 0) 1) 2)
(test ((#(#(1 2) #(3 4)) 0) 1) 2)
(test ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0)) 2)
(test (#(1 2) -1) 'error)
(test (#()) 'error)
(test (#(1)) 'error)
(test (#2d((1 2) (3 4))) 'error)
(test (apply (make-vector '(1 2))) 'error)


(let ((v #(1 2 3)))
  (for-each
   (lambda (arg)
     (test (vector-ref arg 0) 'error)
     (test (v arg) 'error)
     (test (v arg 0) 'error))
   (list "hi" #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table))))


(test (vector-ref '#(#(1 2 3) #(4 5 6)) 1) '#(4 5 6))
(test (vector-ref '#(#(1 2 3) #(4 5 6)) 1 2) 6)
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) '#(#(7 8 9) #(10 11 12)))
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) '#(7 8 9))
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test ('#(#(1 2 3) #(4 5 6)) 1) '#(4 5 6))
(test ('#(#(1 2 3) #(4 5 6)) 1 2) 6)
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) '#(#(7 8 9) #(10 11 12)))
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) '#(7 8 9))
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (L 1)) '#(4 5 6))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (L 1 2)) 6)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1)) '#(#(7 8 9) #(10 11 12)))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0)) '#(7 8 9))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) ((L 1) 2)) 6)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0)) '#(7 8 9))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-ref (L 1) 2)) 6)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-ref ((L 1) 2) 3)) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L 1) 0)) '#(7 8 9))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L 1) 0) 2)) 9)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (((L 1) 0) 2) 3)) 'error)


(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (vector-ref '#(#(1 2 3) #(4 5 6)) one) '#(4 5 6))
  (test (vector-ref '#(#(1 2 3) #(4 5 6)) one two) 6)
  (test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) '#(#(7 8 9) #(10 11 12)))
  (test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) '#(7 8 9))
  (test (vector-ref '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)
  
  (test ('#(#(1 2 3) #(4 5 6)) one) '#(4 5 6))
  (test ('#(#(1 2 3) #(4 5 6)) one two) 6)
  (test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) '#(#(7 8 9) #(10 11 12)))
  (test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) '#(7 8 9))
  (test ('#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)
  
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (L one)) '#(4 5 6))
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (L one two)) 6)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one)) '#(#(7 8 9) #(10 11 12)))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero)) '#(7 8 9))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero two)) 9)
  
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) ((L one) two)) 6)
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero)) '#(7 8 9))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero two)) 9)
  
  (test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-ref (L one) two)) 6)
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L one) zero)) '#(7 8 9))
  (test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L one) zero) two)) 9))


(test (let ((vec (vector 0 '(2 2 2 2) "Anna"))) (vector-set! vec 1 '("Sue" "Sue")) vec) '#(0 ("Sue" "Sue") "Anna"))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 32) v) '#(1 32 3))
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-set! v 1 arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 0) v) '#(1 0 3))
(test (let ((v (vector #f))) (vector-set! v 0 (vector)) v) '#(#()))
(test (let ((v (vector 1 (list 2) (vector 1 2 3)))) (vector-set! (vector-ref v 2) 0 21) v) '#(1 (2) #(21 2 3)))

(test (vector-set! (vector 1 2) 0 4) 4)
(test (vector-set!) 'error)
(test (vector-set! #(1)) 'error)
(test (vector-set! #(1) 0) 'error)
(test (vector-set! #(1) 0 0 1) 'error)
(test (vector-set! #(1) 0 0 1 2 3) 'error)
(test (vector-set! #(1) #(0) 1) 'error)
(test (vector-set! '#(1 2) 0 2) 2)
(test (let ((x 2) (v (vector 1 2))) (vector-set! (let () (set! x 3) v) 1 23) (list x v)) '(3 #(1 23)))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table)))

(let ((v (vector 1 2 3)))
  (for-each
   (lambda (arg)
     (test (vector-set! v arg 0) 'error))
   (list "hi" #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (make-vector 3) (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(let ((v (vector)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-set! v 1 0) 'error)
  (test (vector-set! v -1 0) 'error))
(test (vector-set! #() 0 123) 'error)
(test (vector-set! #(1 2 3) 0 123) 123)

(test (let ((g (lambda () '#(1 2 3)))) (vector-set! (g) 0 #\?) (g)) #(#\? 2 3))
(test (let ((g (lambda () '(1 . 2)))) (set-car! (g) 123) (g)) '(123 . 2))
(test (let ((g (lambda () '(1 2)))) (list-set! (g) 0 123) (g)) '(123 2))
(test (let ((g (lambda () (symbol->string 'hi)))) (string-set! (g) 1 #\a) (symbol->string 'hi)) "hi")

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 32) L) '#(#(1 2 3) 32))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 32) L) '#(#(1 2 3) #(32 5 6)))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 2 32) L) 'error)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (vector-set! L 1 3 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 32) L) '#(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 32) L) '#(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 1 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 4 2 32) L) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1) 32) L) '#(#(1 2 3) 32))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1 0) 32) L) '#(#(1 2 3) #(32 5 6)))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1) 32) L) '#(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! ((L 1) 0) 32) L) '#(#(1 2 3) #(32 5 6)))
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L '#(#(1 2 3) #(4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1) 0) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 0) 2) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L '#(#(#(1 2 3))))) (set! ((L 0) 0 1) 32) L) '#(#(#(1 32 3))))
(test (let ((L '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1 0) 2) 32) L) '#(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))

(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (L 0 0 1) 32) 
	L) 
      '#(#(#(#(1 2 3) 32) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0) 0 1 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0 0) 1 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0 0 1) 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (((L 0) 0) 1 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (((L 0 0) 1) 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L '#(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((((L 0) 0) 1) 2) 32) 
	L) 
      '#(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))


(test (fill! (vector 1 2) 4) 4)

(test (let ((v (vector 1 2 3))) (vector-fill! v 0) v) '#(0 0 0))
(test (let ((v (vector))) (vector-fill! v #f) v) '#())
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-fill! v arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 '() (list 1) '(1 . 2) #f "hi" 'a-symbol abs 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))

(test (let ((str "hi") (v (make-vector 3))) (vector-fill! v str) (string-set! (vector-ref v 0) 1 #\a) str) "ha")
(test (let ((lst (list 1 2)) (v (make-vector 3))) (vector-fill! v lst) (list-set! (vector-ref v 0) 1 #\a) lst) '(1 #\a))

(test (let ((v (vector 1 2 3))) (vector-set! v -1 0)) 'error)
(test (let ((v (vector 1 2 3))) (vector-set! v 3 0)) 'error)
(test (vector-fill! '#(1 2) 2) 2)
(test (vector-fill! #() 0) 0)
(test (vector-fill! (vector) 0) 0)

(for-each
 (lambda (arg)
   (test (vector-fill! arg 0) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


(test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n (- m)))) (vector 1 2 3) (vector 4 5 6)) sum) -9)
(test (let () (for-each (lambda (n) (error "oops")) (vector)) #f) #f)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n (- m) (* 2 p)))) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) sum) 33)
(test (let ((sum 0)) (for-each (lambda (n) (for-each (lambda (m) (set! sum (+ sum (* m n)))) (vector 1 2 3))) (vector 4 5 6)) sum) 90)
(test (call/cc (lambda (return) (for-each (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (for-each (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)

(for-each
 (lambda (data)
   (let ((v data)
	 (c #f)
	 (y 0))
     
     (do ((i 0 (+ i 1)))
	 ((= i 10))
       (set! (v i) i))
     
     (let ((tag 
	    (call/cc
	     (lambda (exit)
	       
	       (for-each
		(lambda (x)
		  
		  (call/cc
		   (lambda (return)
		     (set! c return)))
		  
		  (if (and (even? (inexact->exact x))
			   (> x y) 
			   (< x 10)) 
		      (begin 
			(set! (v (inexact->exact y)) 100)
			(set! y x) 
			(exit x)) 
		      (set! y x)))
		v)))))
       
       (if (and (number? tag) (< tag 10))
	   (c)))
     
     (let ((correct (vector 0 100 2 100 4 100 6 100 8 9)))
       (do ((i 0 (+ i 1)))
	   ((= i 10))
	 (if (not (= (correct i) (inexact->exact (v i))))
	     (format #t ";for-each call/cc data: ~A~%" v))))))
 
 (list (make-vector 10)
;       (make-vct 10)
       (make-list 10)))


(test (map (lambda (n) (+ 1 n)) (vector 1 2 3)) '(2 3 4))
(test (map (lambda (n m) (- n m)) (vector 1 2 3) (vector 4 5 6)) '(-3 -3 -3))
(test (map (lambda (n m p) (+ n m p)) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) '(11 14 17))
(test (map (lambda (n) (map (lambda (m) (* m n)) (vector 1 2 3))) (vector 4 5 6)) '((4 8 12) (5 10 15) (6 12 18)))
(test (call/cc (lambda (return) (map (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (map (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)


(test (vector? (symbol-table)) #t)
(test (symbol? (((symbol-table) 0) 0)) #t)
(let ((old-table (symbol-table))
      (old-list ((symbol-table) 0)))
  ;; try to clobber it...
  (vector-fill! (symbol-table) #())
  (set! ((symbol-table) 0) 1)
  (test (list? ((symbol-table) 0)) #t)
  (test (symbol? (((symbol-table) 0) 0)) #t)
  (test (sort! (symbol-table) <) 'error)
  (test (equal? old-list ((symbol-table) 0)) #t)
  (test (vector? (sort! (symbol-table) (lambda (a b) (< (length a) (length b))))) #t)
  (test (equal? old-list ((symbol-table) 0)) #t))


(let ((v (make-vector 3 (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #t)
  (test (eqv? (v 0) (v 1)) #t))

(let ((v (vector (vector 1 2) (vector 1 2) (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #f)
  (test (eqv? (v 0) (v 1)) #f))

(let ((v (vector (vector (vector (vector 1 2) 3) 4) 5)))
  (test (v 0) #(#(#(1 2) 3) 4))
  (test (v 1) 5)
  (test (((v 0) 0) 1) 3)
  (test ((((v 0) 0) 0) 1) 2))

(test (make-vector 1 (make-vector 1 (make-vector 1 0))) #(#(#(0))))


(let ((v1 (make-vector 3 1)))
  (num-test (v1 1) 1)
  (set! (v1 1) 2)
  (num-test (v1 1) 2)
  (let ((i0 0)
	(i2 2))
    (num-test (v1 i0) 1)
    (num-test (vector-ref v1 i2) 1)
    (set! (v1 i0) 0)
    (num-test (v1 0) 0)
    (set! (v1 i0) i2)
    (num-test (v1 i0) i2))
  (test (vector-dimensions v1) '(3))
  (set! v1 (make-vector '(3 2)))
  (test (vector-dimensions v1) '(3 2))
  (vector-set! v1 1 1 0)
  (num-test (vector-ref v1 1 1) 0)
  (let ((i0 1)
	(i1 1)
	(i2 32))
    (set! (v1 i0 i1) i2)
    (num-test (vector-ref v1 1 1) 32)
    (num-test (v1 i0 i1) i2)
    (vector-set! v1 0 1 3)
    (num-test (v1 0 1) 3)
    (num-test (v1 1 1) 32))
  (set! v1 (make-vector '(2 4 3) 1))
  (test (vector-dimensions v1) '(2 4 3))      
  (num-test (vector-ref v1 1 1 1) 1)
  (vector-set! v1 0 0 0 32)
  (num-test (v1 0 0 0) 32)
  (set! (v1 0 1 1) 3)
  (num-test (v1 0 1 1) 3))
  
(let ((v (make-vector '(2 2))))
  (set! (v 0 0) 1)
  (set! (v 0 1) 2)
  (set! (v 1 0) 3)
  (set! (v 1 1) 4)
  (set! (v 0 1) #2d((1 2) (3 4)))
  v)

(let ((v #2d((1 2) (3 4)))) 
  (set! (v 0 1) #2d((1 2) (3 4))) 
  v)

(test (let ((v1 (make-vector '(3 2) 1))
	    (v2 (make-vector '(3 2) 2))
	    (sum 0))
	(for-each (lambda (n m) (set! sum (+ sum n m))) v1 v2)
	sum)
      18)
(test (vector->list (make-vector '(2 3) 1)) '(1 1 1 1 1 1))
(test (vector->list #2d((1 2) (3 4))) '(1 2 3 4))
(test (list->vector '((1 2) (3 4))) #((1 2) (3 4)))

(test (#2d((1 2 3) (4 5 6)) 0 0) 1)
(test (#2d((1 2 3) (4 5 6)) 0 1) 2)
(test (#2d((1 2 3) (4 5 6)) 1 1) 5)
(test (#3D(((1 2) (3 4)) ((5 6) (7 8))) 0 0 0) 1)
(test (#3D(((1 2) (3 4)) ((5 6) (7 8))) 1 1 0) 7)
(test (#4d((((1) (2)) ((3) (4)) ((5) (6)))) 0 0 0 0) 1)
(test (vector? #2d((1 2) (3 4))) #t)
(test ((#2d((1 #2d((2 3) (4 5))) (6 7)) 0 1) 1 0) 4)
(test ((((((((((#10D((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0) 0) 0) 0) 0) 0) 0) 0) 0) 0) 1)
(test (#10D((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0 0 0 0 0 0 0 0 0 0) 1)
(let ((v (make-vector (make-list 100 1) 0)))
  (test (equal? v #100D((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((0))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) #t)
  (test (apply v (make-list 100 0)) 0)
  (test (v 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0) 0))


;; these are read-time errors 
					;(test #3D(((1 2) (3 4)) ((5 6) (7))) 'error)
					;(test #3D(((1 2) (3 4)) ((5 6) (7 8 9))) 'error)
					;(test #3D(((1 2) (3 4)) (5 (7 8 9))) 'error)

(test (vector-dimensions #3D(((1 2) (3 4)) ((5 6) (7 8)))) '(2 2 2))
(test (vector-dimensions #2d((1 2 3) (4 5 6))) '(2 3))
(test (vector-dimensions #4d((((1) (2)) ((3) (4)) ((5) (6))))) '(1 3 2 1))

(test (vector-length #3D(((1 2) (3 4)) ((5 6) (7 8)))) 8)
(test (length #2d((1 2 3) (4 5 6))) 6)

(test (#2d((1 (2) 3) (4 () 6)) 0 1) '(2))
(test (#2d((1 (2) 3) (4 () 6)) 1 1) '())
(test (#2d((1 (2) 3) (4 6 ())) 1 2) '())
(test (#2d((() (2) ()) (4 5 6)) 0 2) '())

(test (equal? (make-vector 0) (make-vector '(0))) #t)
(test (equal? #() (make-vector '(0))) #t)

(test (equal? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #t)
(test (eq? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #f)
(test (eqv? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #f)
(test (make-vector (1 . 2) "hi") 'error)
(test (make-vector (cons 1 2) "hi") 'error)
(test (equal? (make-vector 0) (vector)) #t)
(test (equal? #() (vector)) #t)

(let ((v (make-vector '(2 3) 0)))
  (num-test (vector-length v) 6)
  (test (vector-dimensions v) '(2 3))
  (num-test (v 0 0) 0)
  (num-test (v 1 2) 0)
  (test (v 2 2) 'error)
  (test (v 2 -1) 'error)
  (test (v 2 0) 'error)
  (set! (v 0 1) 1)
  (num-test (v 0 1) 1)
  (num-test (v 1 0) 0)
  (set! (v 1 2) 2)
  (num-test (v 1 2) 2)
  (test (set! (v 2 2) 32) 'error)
  (test (set! (v 1 -1) 0) 'error)
  (test (set! (v 2 0) 0) 'error)
  (num-test (vector-ref v 0 1) 1)
  (num-test (vector-ref v 1 2) 2)
  (test (vector-ref v 2 2) 'error)
  (test (vector-ref v 1 -1) 'error)
  (vector-set! v 1 1 64)
  (num-test (vector-ref v 1 1) 64)
  (num-test (vector-ref v 0 0) 0)
  (test (vector-ref v 1 2 3) 'error)
  (test (vector-set! v 1 2 3 4) 'error)
  (test (v 1 1 1) 'error)
  (test (set! (v 1 1 1) 1) 'error))

(let ((v1 (make-vector '(3 2) 0))
      (v2 (make-vector '(2 3) 0))
      (v3 (make-vector '(2 3 4) 0))
      (v4 (make-vector 6 0))
      (v5 (make-vector '(2 3) 0)))
  (test (equal? v1 v2) #f)
  (test (equal? v1 v3) #f)
  (test (equal? v1 v4) #f)
  (test (equal? v2 v2) #t)
  (test (equal? v3 v2) #f)
  (test (equal? v4 v2) #f)
  (test (equal? v5 v2) #t)
  (test (equal? v4 v3) #f)
  (test (vector-dimensions v3) '(2 3 4))
  (test (vector-dimensions v4) '(6))
  (num-test (v3 1 2 3) 0)
  (set! (v3 1 2 3) 32)
  (num-test (v3 1 2 3) 32)
  (num-test (vector-length v3) 24)
  (num-test (vector-ref v3 1 2 3) 32)
  (vector-set! v3 1 2 3 -32)
  (num-test (v3 1 2 3) -32)
  (test (v3 1 2) '#(0 0 0 -32))
  (test (set! (v3 1 2) 3) 'error)
  (test (vector-ref v3 1 2) '#(0 0 0 -32))
  (test (vector-set! v3 1 2 32) 'error))

(test (let ((v #2d((1 2) (3 4)))) (vector-fill! v #t) v) #2D((#t #t) (#t #t)))

(test-w "#2d((1 2) #2d((3 4) 5 6))")
(test (string=? (object->string #2d((1 2) (3 #2d((3 4) (5 6))))) "#2D((1 2) (3 #2D((3 4) (5 6))))") #t)
(test (string=? (object->string #3d(((#2d((1 2) (3 4)) #(1)) (#3d(((1))) 6)))) "#3D(((#2D((1 2) (3 4)) #(1)) (#3D(((1))) 6)))") #t)

(test (make-vector '(2 -2)) 'error)
(test (make-vector '(2 1/2)) 'error)
(test (make-vector '(2 1.2)) 'error)
(test (make-vector '(2 2+i)) 'error)
(test (make-vector '(2 "hi")) 'error)

(let ((v (make-vector '(1 1 1) 32)))
  (test (vector? v) #t)
  (test (equal? v #()) #f)
  (test (vector->list v) '(32))
  (test (vector-ref v 0) '#2D((32)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-ref v 0 0) #(32))
  (test (vector-set! v 0 0 0) 'error)
  (test (vector-ref v 0 0 0) 32)
  (test (let () (vector-set! v 0 0 0 31) (vector-ref v 0 0 0)) 31)
  (test (vector-length v) 1)
  (test (vector-dimensions v) '(1 1 1))
  (test (object->string v) "#3D(((31)))")
  )

(test (vector? #3D(((32)))) #t)
(test (equal? #3D(((32))) #()) #f)
(test (vector->list #3D(((32)))) '(32))
(test (#3D(((32))) 0) '#2D((32)))
(test (set! (#3D(((32))) 0) 0) 'error)
(test (#3D(((32))) 0 0) '#(32))
(test (set! (#3D(((32))) 0 0) 0) 'error)
(test (#3D(((32))) 0 0 0) 32)
(test (vector-length #3D(((32)))) 1)
(test (vector-dimensions #3D(((32)))) '(1 1 1))
(test (object->string #3D(((32)))) "#3D(((32)))")


(let ((v1 (make-vector '(1 0))))
  (test (vector? v1) #t)
  (test (equal? v1 #()) #f)
  (test (vector->list v1) '())
  (test (vector-ref v1 0) 'error)
  (test (vector-set! v1 0 0) 'error)
  (test (vector-ref v1 0 0) 'error)
  (test (vector-set! v1 0 0 0) 'error)
  (test (vector-length v1) 0)
  (test (vector-dimensions v1) '(1 0))
  (test (object->string v1) "#2D()")
  )

(let ((v2 (make-vector '(10 3 0))))
  (test (vector? v2) #t)
  (test (equal? v2 #()) #f)
  (test (vector->list v2) '())
  (test (vector-ref v2) 'error)
  (test (vector-set! v2 0) 'error)
  (test (vector-ref v2 0) 'error)
  (test (vector-set! v2 0 0) 'error)
  (test (vector-ref v2 0 0) 'error)
  (test (vector-set! v2 0 0 0) 'error)
  (test (vector-ref v2 1 2 0) 'error)
  (test (vector-set! v2 1 2 0 0) 'error)
  (test (vector-length v2) 0)
  (test (vector-dimensions v2) '(10 3 0))
  (test (object->string v2) "#3D()")
  )

(let ((v3 (make-vector '(10 0 3))))
  (test (vector? v3) #t)
  (test (equal? v3 #()) #f)
  (test (vector->list v3) '())
  (test (vector-ref v3) 'error)
  (test (vector-set! v3 0) 'error)
  (test (vector-ref v3 0) 'error)
  (test (vector-set! v3 0 0) 'error)
  (test (vector-ref v3 0 0) 'error)
  (test (vector-set! v3 0 0 0) 'error)
  (test (vector-ref v3 1 0 2) 'error)
  (test (vector-set! v3 1 0 2 0) 'error)
  (test (vector-length v3) 0)
  (test (vector-dimensions v3) '(10 0 3))
  (test (object->string v3) "#3D()")
  )

(test (((#(("hi") ("ho")) 0) 0) 1) #\i)
(test (string-ref (list-ref (vector-ref #(("hi") ("ho")) 0) 0) 1) #\i)

(test (equal? #2D() (make-vector '(0 0))) #t)
(test (equal? #2D() (make-vector '(1 0))) #f)
(test (equal? (make-vector '(2 2) 2) #2D((2 2) (2 2))) #t)
(test (equal? (make-vector '(2 2) 2) #2D((2 2) (1 2))) #f)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 2 3) 0)) #t)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 3 2) 0)) #f)
(test (make-vector '1 2 3) 'error)

(test (equal? (make-vector 10 '()) (make-hash-table 10)) #f)

(test (equal? #2d((1 2) (3 4)) (copy #2d((1 2) (3 4)))) #t)
(test (equal? #3d() #3d(((())))) #f)
(test (equal? #3d() #3d()) #t)
(test (equal? #3d() #2d()) #f)
(test (equal? #3d() (copy #3d())) #t)
(test (equal? #2d((1) (2)) #2d((1) (3))) #f)
(test (equal? #2d((1) (2)) (copy #2d((1) (2)))) #t)
(let ((v1 (make-vector '(3 2 1) #f))
      (v2 (make-vector '(3 2 1) #f)))
  (test (equal? v1 v2) #t)
  (set! (v2 0 0 0) 1)
  (test (equal? v1 v2) #f))
(test (equal? (make-vector '(3 2 1) #f) (make-vector '(1 2 3) #f)) #f)

(test (map (lambda (n) n) #2d((1 2) (3 4))) '(1 2 3 4))
(test (let ((vals '())) (for-each (lambda (n) (set! vals (cons n vals))) #2d((1 2) (3 4))) vals) '(4 3 2 1))
(test (map (lambda (x y) (+ x y)) #2d((1 2) (3 4)) #1d(4 3 2 1)) '(5 5 5 5))
(test (let ((vals '())) (for-each (lambda (x y) (set! vals (cons (+ x y) vals))) #2d((1 2) (3 4)) #1d(4 3 2 1)) vals) '(5 5 5 5))

(let ((v #2D((#(1 2) #(3 4)) (#2d((5 6) (7 8)) #2D((9 10 11) (12 13 14))))))
  (test (v 0 0) #(1 2))
  (test (v 0 1) #(3 4))
  (test (v 1 0) #2d((5 6) (7 8)))
  (test (v 1 1) #2D((9 10 11) (12 13 14)))
  (test ((v 1 0) 0 1) 6)
  (test ((v 0 1) 1) 4)
  (test ((v 1 1) 1 2) 14))

(let ((v #2D((#((1) #(2)) #(#(3) (4))) (#2d(((5) #(6)) (#(7) #(8))) #2D((#2d((9 10) (11 12)) (13)) (14 15))))))
  (test (v 0 0) #((1) #(2)))
  (test (v 0 1) #(#(3) (4)))
  (test (v 1 0) #2D(((5) #(6)) (#(7) #(8))))
  (test (v 1 1) #2D((#2D((9 10) (11 12)) (13)) (14 15)))
  (test ((v 1 0) 0 1) #(6))
  (test (((v 1 0) 0 1) 0) 6)
  (test ((v 0 1) 1) '(4))
  (test (((v 1 1) 0 0) 1 0) 11))


(test (let ((V #2D((1 2 3) (4 5 6)))) (V 0)) '#(1 2 3))
(test (let ((V #2D((1 2 3) (4 5 6)))) (V 1)) '#(4 5 6))
(test (let ((V #2D((1 2 3) (4 5 6)))) (V 2)) 'error)
(test (let ((V #2D((1 2 3) (4 5 6)))) (set! (V 1) 0)) 'error)
(test (let ((V #2D((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 1) 32) V)) '#2D((1 32 3) (4 5 6)))
(test (let ((V #2D((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 3) 32) V)) 'error)

(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1)) '#2D((7 8 9) (10 11 12)))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1 1)) '#(10 11 12))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 0 1)) '#(4 5 6))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 2 1)) 'error)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((V 0) 1)) '#(4 5 6))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((V 0) 1) 1) 32) V) '#3D(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 0 1 1 32) V) '#3D(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 1 1 0 32) V) '#3D(((1 2 3) (4 5 6)) ((7 8 9) (32 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 1))) 6)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 1))) '(2 3))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 0 1))) 3)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 0 1))) '(3))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (one 1) (zero 0)) 
	(let ((V1 (V one zero))
	      (sum 0))
	  (for-each (lambda (n) (set! sum (+ sum n))) V1)
	  sum))
      24) ; 7 8 9
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (two 2) (one 1) (zero 0)) 
	(let ((V10 (V one zero))
	      (V00 (V zero zero))
	      (V01 (V zero one))
	      (V11 (V one one))
	      (sum 0))
	  (for-each (lambda (n0 n1 n2 n3) (set! sum (+ sum n0 n1 n2 n3))) V00 V01 V10 V11)
	  sum))
      78)

(let ((old-vlen *vector-print-length*))
  (set! *vector-print-length* 32)
  (test (object->string (make-vector '(8 8) 0)) "#2D((0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0)...)")
  (test (object->string (make-vector 64 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector 32 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0)")
  (test (object->string (make-vector 33 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector '(8 4) 0)) "#2D((0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0))")
  (set! *vector-print-length* old-vlen))

(let ((old-vlen *vector-print-length*))
  (set! *vector-print-length* 1024) ; check the many-() case
  (test (object->string (make-vector '(2 1 2 1 2 1 2 1 2 1 2 1 2 1) 0)) "#14D((((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))) (((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))))")

  (test (object->string (make-vector '(16 1 1 1 1 1 1 1 1 1 1 1 1 1) 0)) "#14D((((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))))")

;;; now see if our shared vector has survived...
  (test (and (vector? check-shared-vector-after-gc)
	     (= (length check-shared-vector-after-gc) 6)
	     (do ((i 0 (+ i 1))
		  (happy #t))
		 ((= i 6) happy)
	       (if (or (not (pair? (check-shared-vector-after-gc i)))
		       (not (equal? (check-shared-vector-after-gc i) (cons 3 i))))
		   (set! haappy #f))))
	#t)
  (set! check-shared-vector-after-gc #f)

  (set! *vector-print-length* old-vlen))  





;;; -------- circular structures --------

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error))

(let ((l1 (list 1)))
  (test (object->string (list l1 1 l1)) "(#1=(1) 1 #1#)"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (append '(1) lst)) "(1 . #1=(1 2 3 . #1#))"))
(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (append lst '()) 'error)) 

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (sort! lst <) 'error))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (list lst)) "(#1=(1 2 3 . #1#))"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (make-list 4 lst)) "(#1=(1 2 3 . #1#) #1# #1# #1#)"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (vector lst lst)) "#(#1=(1 2 3 . #1#) #1#)"))

(let ((lst `(+ 1 2 3)))
   (set! (cdr (cdddr lst)) (cddr lst)) 
   (test (object->string lst) "(+ 1 . #1=(2 3 . #1#))"))


(let ((x (list 1 2)))
  (test (equal? x x) #t)
  (test (equal? x (cdr x)) #f)
  (test (equal? x '()) #f))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5))))))
  (test (equal? x y) #t))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5) 6)))))
  (test (equal? x y) #f))

(test (length '()) 0)
;;; (test (length (cons 1 2)) -1)
(test (length '(1 2 3)) 3)

(test (let ((lst (list))) (fill! lst 0) lst) '())
(test (let ((lst (list 1))) (fill! lst 0) lst) '(0))
(test (let ((lst (list 1 2))) (fill! lst 0) lst) '(0 0))
(test (let ((lst (list 1 (list 2 3)))) (fill! lst 0) lst) '(0 0))
(test (let ((lst (cons 1 2))) (fill! lst 0) lst) '(0 . 0))
(test (let ((lst (cons 1 (cons 2 3)))) (fill! lst 0) lst) '(0 0 . 0))

(let ((lst1 (list 1 2))) 
  (test (length lst1) 2)
  (list-set! lst1 0 lst1)
  (test (length lst1) 2) ; its car is a circular list, but it isn't
  (test (list->string lst1) 'error)
  (let ((lst2 (list 1 2)))
    (set-car! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (test (eq? lst1 lst2) #f)
    (test (eqv? lst1 lst2) #f)
    (test (pair? lst1) #t)
    (test (null? lst1) #f)
    (test (car lst2) lst2)
    (test (car lst1) lst1)
    (test (let ()
	    (fill! lst1 32)
	    lst1)
	  '(32 32))))

(let ((lst1 (list 1))) 
  (test (length lst1) 1)
  (set-cdr! lst1 lst1)
  (test (infinite? (length lst1)) #t)
  (test (null? lst1) #f)
  (test (pair? lst1) #t)
  (let ((lst2 (cons 1 '())))
    (set-cdr! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (set-car! lst2 0)
    (test (equal? lst1 lst2) #f)
    (test (infinite? (length lst2)) #t)))

(let ((lst1 (list 1))
      (lst2 (list 1)))
  (set-car! lst1 lst2)
  (set-car! lst2 lst1)
  (test (equal? lst1 lst2) #t)
  (test (length lst1) 1)
  (let ((lst3 (list 1)))
    (test (equal? lst1 lst3) #f)
    (set-cdr! lst3 lst3)
    (test (equal? lst1 lst3) #f)))

(let ((lst1 (list 'a 'b 'c)))
  (set! (cdr (cddr lst1)) lst1)
  (test (infinite? (length lst1)) #t)
  (test (memq 'd lst1) #f)
  (test (memq 'a lst1) lst1)
  (test (memq 'b lst1) (cdr lst1)))

(let ((lst1 (list 1 2 3)))
  (list-set! lst1 1 lst1)
  (test (object->string lst1) "#1=(1 #1# 3)"))


(test (copy (list 1 2 (list 3 4))) '(1 2 (3 4)))
(test (copy (cons 1 2)) '(1 . 2))
(test (copy '(1 2 (3 4) . 5)) '(1 2 (3 4) . 5))
(test (copy '()) '())

(test (object->string (let ((l1 (list 0 1))) (set! (l1 1) l1) (copy l1))) "(0 #1=(0 #1#))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) (copy lst))) "(1 . #1=(1 . #1#))")
(test (object->string (let ((l1 (list 1 2))) (copy (list l1 4 l1)))) "(#1=(1 2) 4 #1#)")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) (copy lst))) "(1 2 3 . #1=(2 3 . #1#))")

(test (reverse '(1 2 (3 4))) '((3 4) 2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '()) '())
(test (let ((lst (list 1 2 3))) (set! (lst 2) lst) (object->string (reverse lst))) "(#1=(1 2 #1#) 2 1)")
(test (let ((l1 (cons 1 '()))) (set-cdr! l1 l1) (object->string (reverse l1))) "(#1=(1 . #1#) 1 1 1)")


(test (equal? (vector 0) (vector 0)) #t)
(test (equal? (vector 0 #\a "hi" (list 1 2 3)) (vector 0 #\a "hi" (list 1 2 3))) #t)
(test (let ((v (vector 0))) (equal? (vector v) (vector v))) #t)

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (vector? v1) #t)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (vector-length v1) 1)
    (test (equal? v1 v2) #t)
    (test (equal? (vector-ref v1 0) v1) #t)
    (test (equal? (vector->list v1) (list v1)) #t)
    (vector-fill! v1 0)
    (test (equal? v1 (vector 0)) #t)
    (let ((v3 (copy v2)))
      (test (equal? v2 v3) #t)
      (vector-set! v3 0 0)
      (test (equal? v3 (vector 0)) #t))
    ))

(let ((v1 (make-vector 1 0))
      (v2 (vector 0)))
  (set! (v1 0) v2)
  (set! (v2 0) v1)
  (test (equal? v1 v2) #t)) 

(let* ((l1 (list 1 2))
       (v1 (vector 1 2))
       (l2 (list 1 l1 2))
       (v2 (vector l1 v1 l2)))
  (vector-set! v1 0 v2)
  (list-set! l1 1 l2)
  (test (equal? v1 v2) #f))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (equal? v1 v2) #t)))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (object->string v1) "#1=#(#1#)"))

(let ((l1 (cons 0 '()))) 
  (set-cdr! l1 l1) 
  (test (list->vector l1) 'error))

(let ((lst (list "nothing" "can" "go" "wrong")))
  (let ((slst (cddr lst))
	(result '()))
    (set! (cdr (cdddr lst)) slst)
    (test (do ((i 0 (+ i 1))
	       (l lst (cdr l)))
	      ((or (null? l) (= i 12))
	       (reverse result))
	    (set! result (cons (car l) result)))
	  '("nothing" "can" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong"))))

#|
;;; here is a circular function!
(let ()
  (define (cfunc)
    (begin
      (display "cfunc! ")
      #f))

  (let ((clst (procedure-source cfunc)))
    (set! (cdr (cdr (car (cdr (cdr clst)))))
	  (cdr (car (cdr (cdr clst))))))

  (cfunc))
|#

(test (let ((l (list 1 2))) 
	(list-set! l 0 l) 
	(string=? (object->string l) "#1=(#1# 2)")) 
      #t)
(test (let ((lst (cons 1 2))) 
	(set-cdr! lst lst)
	(string=? (object->string lst) "#1=(1 . #1#)"))
      #t)
(test (let ((lst (cons 1 2))) 
	(set-car! lst lst)
	(string=? (object->string lst) "#1=(#1# . 2)"))
      #t)
(test (let ((lst (cons (cons 1 2) 3))) 
	(set-car! (car lst) lst)
	(string=? (object->string lst) "#1=((#1# . 2) . 3)"))
      #t)
(test (let ((v (vector 1 2))) 
	(vector-set! v 0 v) 
	(string=? (object->string v) "#1=#(#1# 2)")) 
      #t)
(test (let* ((l1 (list 1 2)) (l2 (list l1))) 
	(list-set! l1 0 l1) 
	(string=? (object->string l2) "(#1=(#1# 2))")) 
      #t)

(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (object->string lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (object->string lst)) "(1 . #1=(2 3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (object->string lst)) "(1 2 . #1=(3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (car lst) (cdr lst)) (object->string lst)) "(#1=(2 3) . #1#)")
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) (cdr lst)) (object->string lst)) "(1 . #1=(#1# 3))")
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) lst) (object->string lst)) "#1=(1 #1# 3)")
(test (let ((l1 (list 1))) (let ((l2 (list l1 l1))) (object->string l2))) "(#1=(1) #1#)")

(test (let* ((v1 (vector 1 2)) (v2 (vector v1))) 
	(vector-set! v1 1 v1) 
	(string=? (object->string v2) "#(#1=#(1 #1#))")) 
      #t)
(test (let ((v1 (make-vector 3 1))) 
	(vector-set! v1 0 (cons 3 v1)) 
	(string=? (object->string v1) "#1=#((3 . #1#) 1 1)")) 
      #t)
(test (let ((h1 (make-hash-table 11))
	    (old-print-length *vector-print-length*))
	(set! *vector-print-length* 32)
	(hash-table-set! h1 "hi" h1)
	(let ((result (object->string h1)))
	  (set! *vector-print-length* old-print-length)
	  (let ((val (string=? result "#1=#(() ((\"hi\" . #1#)) () () () () () () () () () () () () () ())")))
	    (if (not val)
		(format #t ";hash display:~%  ~A~%" (object->string h1)))
	    val)))
      #t)

(test (let* ((l1 (list 1 2))
	     (v1 (vector 1 2))
	     (l2 (list 1 l1 2))
	     (v2 (vector l1 v1 l2)))
	(vector-set! v1 0 v2)
	(list-set! l1 1 l2)
	(string=? (object->string v2) "#2=#(#1=(1 #3=(1 #1# 2)) #(#2# 2) #3#)"))
      #t)

(test (let ((l1 (list 1 2))
	    (l2 (list 1 2)))
	(set! (car l1) l2)
	(set! (car l2) l1)
	(object->string (list l1 l2)))
      "(#1=(#2=(#1# 2) 2) #2#)")

(test (let* ((l1 (list 1 2)) 
	     (l2 (list 3 4)) 
	     (l3 (list 5 l1 6 l2 7)))
	(set! (cdr (cdr l1)) l1) 
	(set! (cdr (cdr l2)) l2)
	(string=? (object->string l3) "(5 #1=(1 2 . #1#) 6 #2=(3 4 . #2#) 7)"))
      #t)
(test (let* ((lst1 (list 1 2))
	     (lst2 (list (list (list 1 (list (list (list 2 (list (list (list 3 (list (list (list 4 lst1 5))))))))))))))
	(set! (cdr (cdr lst1)) lst1)
	(string=? (object->string lst2) "(((1 (((2 (((3 (((4 #1=(1 2 . #1#) 5))))))))))))"))
      #t)


(test (equal? '(a) (list 'a)) #t)
(test (equal? '(a b . c) '(a b . c)) #t)
(test (equal? '(a b (c . d)) '(a b (c . d))) #t)
(test (equal? (list "hi" "hi" "hi") '("hi" "hi" "hi")) #t)
(let ((l1 (list "hi" "hi" "hi"))
      (l2 (list "hi" "hi" "hi")))
  (fill! l1 "ho")
  (test (equal? l1 l2) #f)
  (fill! l2 (car l1))
  (test (equal? l1 l2) #t))
(let ((lst (list 1 2 3 4))) 
  (fill! lst "hi") 
  (test (equal? lst '("hi" "hi" "hi" "hi")) #t))
(let ((vect (vector 1 2 3 4)))
  (fill! vect "hi")
  (test (equal? vect #("hi" "hi" "hi" "hi")) #t))
(let ((lst (list 1 2 (list 3 4) (list (list 5) 6))))
  (test (equal? lst '(1 2 (3 4) ((5) 6))) #t)
  (fill! lst #f)
  (test (equal? lst '(#f #f #f #f)) #t))
(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdddr lst)) lst)
  (test (equal? lst lst) #t)
  (test (eq? lst lst) #t)
  (test (eqv? lst lst) #t)
  (fill! lst #f)
  (test (object->string lst) "#1=(#f #f #f #f . #1#)")
  (let ((l1 (copy lst)))
    (test (equal? lst l1) #t)
    (test (eq? lst l1) #f)
    (test (eqv? lst l1) #f)))

(test (let ((lst (list "hi" "hi" "hi"))) (fill! lst "hi") (equal? lst '("hi" "hi" "hi"))) #t)
(test (let ((lst (list "hi" "hi"))) (fill! lst "hi") (equal? lst '("hi" "hi"))) #t)
(test (let ((lst (list 1 2 3 4))) (fill! lst "hi") (equal? lst '("hi" "hi" "hi" "hi"))) #t)


(let ((lst '(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((str (apply string lst)))
    (let ((lstr (list->string lst)))
      (let ((strl (string->list str)))
	(test (eq? str str) #t)
	(test (eq? str lstr) #f)
	(test (eqv? str str) #t)
	(test (eqv? str lstr) #f)
	(test (equal? str lstr) #t)
	(test (equal? str str) #t)
	(test (eq? lst strl) #f)	
	(test (eqv? lst strl) #f)	
	(test (equal? lst strl) #t)
	(let ((l2 (copy lst))
	      (s2 (copy str)))
	  (test (eq? l2 lst) #f)
	  (test (eq? s2 str) #f)
	  (test (eqv? l2 lst) #f)
	  (test (eqv? s2 str) #f)
	  (test (equal? l2 lst) #t)
	  (test (equal? s2 str) #t))))))


(let ((vect #(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((lst (vector->list vect)))
    (let ((vect1 (list->vector lst)))
	(test (eq? lst lst) #t)
	(test (eq? lst vect) #f)
	(test (eqv? lst lst) #t)
	(test (eqv? lst vect) #f)
	(test (equal? vect1 vect) #t)
	(test (equal? lst lst) #t)
	(test (eq? vect vect1) #f)	
	(test (eqv? vect vect1) #f)	
	(test (equal? vect vect1) #t)
	(let ((l2 (copy vect))
	      (s2 (copy lst)))
	  (test (eq? l2 vect) #f)
	  (test (eq? s2 lst) #f)
	  (test (eqv? l2 vect) #f)
	  (test (eqv? s2 lst) #f)
	  (test (equal? l2 vect) #t)
	  (test (equal? s2 lst) #t)))))

(let* ((vals (list "hi" #\A 1 'a #(1) abs 3.14 3/4 1.0+1.0i #\f '(1 . 2)))
       (vlen (length vals)))
  (do ((i 0 (+ i 1)))
      ((= i 100))
    (let* ((size (max 1 (random 20)))
	   (vect (make-vector size '())))
      (do ((n 0 (+ n 1)))
	  ((= n size))
	(let ((choice (random 4))
	      (len (random 4)))
	  (if (= choice 0)
	      (let ((v (make-vector len)))
		(do ((k 0 (+ k 1)))
		    ((= k len))
		  (vector-set! v k (list-ref vals (random vlen))))
		(vector-set! vect n v))
	      (if (= choice 1)
		  (let ((lst (make-list len #f)))
		    (do ((k 0 (+ k 1)))
			((= k len))
		      (list-set! lst k (list-ref vals (random vlen))))
		    (vector-set! vect n lst))
		  (vector-set! vect n (list-ref vals (random vlen)))))))
      (test (eq? vect vect) #t)
      (test (eqv? vect vect) #t)
      (test (equal? vect vect) #t)
      (let ((lst1 (vector->list vect)))
	(let ((lst2 (copy lst1)))
	  (test (eq? lst1 lst2) #f)
	  (test (eqv? lst1 lst2) #f)
	  (test (equal? lst1 lst2) #t))))))

(let* ((lst1 (list 1 2 3))
       (vec1 (vector 1 2 lst1)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 lst2)))
    (list-set! lst2 2 vec2)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (vector-set! vec1 1 vec1)
    (test (equal? lst1 lst2) #f)
    (test (equal? vec1 vec2) #f)
    ))
  
(let* ((base (list #f))
       (lst1 (list 1 2 3))
       (vec1 (vector 1 2 base)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 base)))
    (list-set! lst2 2 vec2)
    (set! (car lst1) lst1)
    (set! (car lst2) lst2)
    (set! (cdr (cddr lst1)) base)
    (set! (cdr (cddr lst2)) base)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (test (object->string lst1) "#1=(#1# 2 #(1 2 #2=(#f)) . #2#)")))

(let ((base (list 0 #f)))
  (let ((lst1 (list 1 base 2))
	(lst2 (list 1 base 2)))
    (set! (cdr (cdr base)) base)
    (test (equal? lst1 lst2) #t)))

(let ((base1 (list 0 #f))
      (base2 (list 0 #f)))
  (let ((lst1 (list 1 base1 2))
	(lst2 (list 1 base2 2)))
    (set! (cdr (cdr base1)) lst2)
    (set! (cdr (cdr base2)) lst1)
    (test (equal? lst1 lst2) #t)
    (test (object->string lst1) "#1=(1 (0 #f 1 (0 #f . #1#) 2) 2)")))

(let ()
  (define-macro (c?r path)

  (define (X-marks-the-spot accessor tree)
    (if (pair? tree)
	(or (X-marks-the-spot (cons 'car accessor) (car tree))
	    (X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	(if (eq? tree 'X) accessor #f)))

  (let ((body 'lst))
    (for-each
     (lambda (f)
       (set! body (list f body)))
     (reverse (X-marks-the-spot '() path)))

    `(make-procedure-with-setter
      (lambda (lst) 
	,body)
      (lambda (lst val)
	(set! ,body val)))))

  (define (copy-tree lis)
    (if (pair? lis)
	(cons (copy-tree (car lis))
	      (copy-tree (cdr lis)))
	lis))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (X))))))))))))
    (set! (cxr l1) 3)
    (set! (cxr l2) 4)
    (test (equal? l1 l2) #f)
    (test (equal? l1 l3) #f)
    (set! (cxr l2) 3)
    (test (cxr l2) 3)
    (test (cxr l1) 3)
    (test (cxr l3) 8)
    (test (equal? l1 l2) #t)
    (test (equal? l2 l3) #f))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (8 . X))))))))))))
    (set! (cxr l1) l1)
    (set! (cxr l2) l2)
    (test (equal? l1 l2) #t)
    (test (equal? l1 l3) #f)
    (test (object->string l2) "#1=(0 (1 (2 (3 (4 (5 (6 (7 (8 . #1#)))))))))"))

  (let* ((l1 '(0 ((((((1))))))))
	 (l2 (copy-tree l1))
	 (cxr (c?r (0 ((((((1 . X))))))))))
    (set! (cxr l1) l2)
    (set! (cxr l2) l1)
    (test (equal? l1 l2) #t))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (cxr (c?r (0 1 (2 3 . X) 4 5))))
    (set! (cxr l1) (cdr l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 . #1#) 4 5))"))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (l2 '(6 (7 8 9) 10))
	 (cxr1 (c?r (0 1 (2 3 . X) 4 5)))
	 (cxr2 (c?r (6 . X)))
	 (cxr3 (c?r (6 (7 8 9) 10 . X)))
	 (cxr4 (c?r (0 . X))))
    (set! (cxr1 l1) (cxr2 l2))
    (set! (cxr3 l2) (cxr4 l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 (7 8 9) 10 . #1#) 4 5))")
    (test (cadr l1) 1)
    (test (cadddr l1) 4)
    )

  (let ((l1 '((a . 2) (b . 3) (c . 4)))
	(cxr (c?r ((a . 2) (b . 3) (c . 4) . X))))
    (set! (cxr l1) (cdr l1))
    (test (assq 'a l1) '(a . 2))
    (test (assv 'b l1) '(b . 3))
    (test (assoc 'c l1) '(c . 4))
    (test (object->string l1) "((a . 2) . #1=((b . 3) (c . 4) . #1#))")
    (test (assq 'asdf l1) #f)
    (test (assv 'asdf l1) #f)
    (test (assoc 'asdf l1) #f)
    )

  (let ((l1 '(a b c d e))
	(cxr (c?r (a b c d e . X))))
    (set! (cxr l1) (cddr l1))
    (test (memq 'b l1) (cdr l1))
    (test (memv 'c l1) (cddr l1))
    (test (member 'd l1) (cdddr l1))
    (test (object->string l1) "(a b . #1=(c d e . #1#))")
    (test (memq 'asdf l1) #f)
    (test (memv 'asdf l1) #f)
    (test (member 'asdf l1) #f)
    (test (pair? (member 'd l1)) #t) ; #1=(d e c . #1#)
    )

  (let ((ctr 0)
	(x 0))
    (let ((lst `(call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0)))))
      (let ((acc1 (c?r (call-with-exit (lambda (return) . X))))
	    (acc2 (c?r (call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0) . X)))))
	(set! (acc2 lst) (acc1 lst))
	(test (eval lst) 11))))
  )
  
(let ((v #2d((1 2) (3 4))))
  (set! (v 1 0) v)
  (test (object->string v) "#1=#2D((1 2) (#1# 4))")
  (test (length v) 4)
  (test ((((v 1 0) 1 0) 1 0) 0 0) 1))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (lst 100) 2)
  (test ((cdddr (cdddr (cdddr lst))) 100) 2)
  (set! (lst 100) 32)
  (test (object->string lst) "#1=(1 32 3 . #1#)"))

(let* ((l1 (list 1 2))
       (l2 (list l1 l1)))
  (set! (l1 0) 32)
  (test (equal? l2 '((32 2) (32 2))) #t))

(let ((q (list 1 2 3 4)))
  (set! (cdr (cdddr q)) q) 
  (test (car q) 1)
  (set! (car q) 5)
  (set! q (cdr q))
  (test (car q) 2)
  (test (object->string q) "#1=(2 3 4 5 . #1#)"))

(let ()
  (define (make-node prev data next) (vector prev data next))
  (define prev (make-procedure-with-setter (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (make-procedure-with-setter (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (make-procedure-with-setter (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=#(#7=#(#6=#(#5=#(#4=#(#3=#(#2=#(#8=#(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
#|
    ;; in CL:
    (let* ((head (vector nil 0 nil))
	   (cur head))
      (do ((i 1 (+ i 1)))
	  ((= i 8))
	(let ((node (vector nil i nil)))
	  (setf (aref node 0) cur)
	  (setf (aref cur 2) node)
	  (setf cur node)))
      (setf (aref head 0) cur)
      (setf (aref cur 2) head)
      (format t "~A~%" head)) -> "#1=#(#2=#(#3=#(#4=#(#5=#(#6=#(#7=#(#8=#(#1# 1 #7#) 2 #6#) 3 #5#) 4 #4#) 5 #3#) 6 #2#) 7 #1#) 0 #8#)"
|#
    (let ((ahead (do ((cur head (next cur))
		      (dat '() (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat '() (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t)))))

(let ()
  (define (make-node prev data next) (list prev data next))
  (define prev (make-procedure-with-setter (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (make-procedure-with-setter (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (make-procedure-with-setter (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#7=(#6=(#5=(#4=(#3=(#2=(#8=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
    (let ((ahead (do ((cur head (next cur))
		      (dat '() (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat '() (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 32))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#31=(#30=(#29=(#28=(#27=(#26=(#25=(#24=(#23=(#22=(#21=(#20=(#19=(#18=(#17=(#16=(#15=(#14=(#13=(#12=(#11=(#10=(#9=(#8=(#7=(#6=(#5=(#4=(#3=(#2=(#32=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #8#) 8 #9#) 9 #10#) 10 #11#) 11 #12#) 12 #13#) 13 #14#) 14 #15#) 15 #16#) 16 #17#) 17 #18#) 18 #19#) 19 #20#) 20 #21#) 21 #22#) 22 #23#) 23 #24#) 24 #25#) 25 #26#) 26 #27#) 27 #28#) 28 #29#) 29 #30#) 30 #31#) 31 #1#) 0 #32#)")))

(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (append lst lst ())) 'error)
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (object->string (append (list lst) (list lst) ()))) "(#1=(1 2 3 . #1#) #1#)")

(let ((ht (make-hash-table 3)))
  (set! (ht "hi") ht)
  (test (object->string ht) "#1=#(() ((\"hi\" . #1#)) () ())")
  (test (equal? (ht "hi") ht) #t))

(let ((l1 '(0)) (l2 '(0))) 
  (set! (car l1) l1) (set! (cdr l1) l1) (set! (car l2) l2) (set! (cdr l2) l2)
  (test (object->string l1) "#1=(#1# . #1#)")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) l2)
  (test (object->string l1) "#1=(#1# . #2=(#2# . #2#))")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) '())
  (test (equal? l1 l2) #f))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (list 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)) 
	    (result '()))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! result (cons (+ a b) result)))
		  (list 4 5 6)
		  lst)
	result)
      '(9 7 5))
(test (let ((lst (list 1 2 3))
	    (ctr 0))
	(set! (cdr (cddr lst)) lst)
	(call-with-exit
	 (lambda (return)
	   (for-each (lambda (a) 
		       (if (> ctr 12)
			   (return a))
		       (set! ctr (+ ctr a)))
		     lst))))
      2)
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)))
	(set! (cdr (cddr lst)) lst)
	(map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6 7 8 9 10)
	     lst))
      '(5 7 9 8 10 12 11))
(test (map (lambda (a) a) '(0 1 2 . 3)) '(0 1 2))
(test (let ((ctr 0)) (for-each (lambda (a) (set! ctr (+ ctr a))) '(1 2 . 3)) ctr) 3)
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     '()
	     lst)
	'()))
(test (let ((lst (list 1 2 3))
	    (ctr 0))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! ctr (+ ctr (+ a b))))
		  lst '())
	ctr)
      0)

(test (let ((lst (list 1))) (set! (cdr lst) (car lst)) (object->string lst)) "(1 . 1)")
(test (let ((lst (list 1))) (set! (car lst) (cdr lst)) (object->string lst)) "(())")

(test (let ((lst (list 1 2 3))) (fill! lst lst) (object->string lst)) "#1=(#1# #1# #1#)")
(test (let ((lst (vector 1 2 3))) (fill! lst lst) (object->string lst)) "#1=#(#1# #1# #1#)")
(test (let ((lst #2d((1) (1)))) (fill! lst lst) (object->string lst)) "#1=#2D((#1#) (#1#))")

(let ((ctr 0) (lst `(let ((x 3)) (set! ctr (+ ctr 1)) (set! (cdr (cddr lst)) `((+ x ctr))) (+ x 1))))
  (test (eval lst) 4)
  (test (eval lst) 5)
  (test (eval lst) 6))
  
(let ()
  (define fact         ; Reini Urban, http://autocad.xarch.at/lisp/self-mod.lsp.txt
    (let ((old '())
	  (result '()))
      
      (define (last lst)
	(list-tail lst (- (length lst) 1)))
      
      (define (butlast lis)
	(let ((len (length lis)))
	  (if (<= len 1) '()
	      (let ((result '()))
		(do ((i 0 (+ i 1))
		     (lst lis (cdr lst)))
		    ((= i (- len 1)) (reverse result))
		  (set! result (cons (car lst) result)))))))
      
      (lambda (n)
	(cond ((zero? n) 1)
	      (#t 
	       (set! old (procedure-source fact))
	       (set! fact (apply lambda '(n)
				       `((cond 
					 ,@(butlast (cdr (car (cdr (cdr old)))))
					 ((= n ,n) ,(let ()
						      (set! result (* n (fact (- n 1))))
						      result))
					 ,@(last (cdr (car (cdr (cdr old)))))))))
	       result)))))

  (test (fact 3) 6)
  (test (fact 5) 120)
  (test (fact 2) 2))

(let* ((x (list 1 2 3)) ; from Lambda the Ultimate I think -- I lost the reference
       (y (list 4 5))	
       (z (cons (car x) (cdr y)))
       (w (append y z))
       (v (cons (cdr x) (cdr y))))
  (set-car! x 6)
  (set-car! y 7)
  (set-cdr! (cdr x) (list 8))
  (test (object->string (list x y z w v)) "((6 . #3=(2 8)) (7 . #1=(5)) #2=(1 . #1#) (4 5 . #2#) (#3# . #1#))"))
;; guile gets this result, but prints it as: ((6 2 8) (7 5) (1 5) (4 5 1 5) ((2 8) 5))





;;; --------------------------------------------------------------------------------
;;; HASH-TABLES
;;; --------------------------------------------------------------------------------

(let ((ht (make-hash-table)))
  (test (hash-table? ht) #t)
  (test (equal? ht ht) #t)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (test (let () (hash-table-set! ht 123 "hiho") (hash-table-ref ht 123)) "hiho")
  (test (let () (hash-table-set! ht 3.14 "hi") (hash-table-ref ht 3.14)) "hi")
  (test (let () (hash-table-set! ht our-pi "hiho") (hash-table-ref ht our-pi)) "hiho")
  (test (hash-table-ref ht "123") #f)
  (let ((ht1 (copy ht)))
    (test (hash-table? ht1) #t)
    (test (= (length ht) (length ht1)) #t)
    (test (equal? ht ht1) #t)
    (set! (ht 'key) 32)
    (set! (ht1 'key) 123)
    (test (and (= (ht 'key) 32) (= (ht1 'key) 123)) #t)
    (set! (ht "key") 321)
    (test (ht "key") 321)
    (test (ht 'key) 32)
    (set! (ht 123) 43)
    (set! (ht "123") 45)
    (test (ht 123) 43)
    (test (ht "123") 45))
  (test (let () (set! (hash-table-ref ht 'key) 32) (hash-table-ref ht 'key)) 32)

  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table-set! arg 'key 32) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))


(let ((ht (make-hash-table 277)))
  (test (hash-table? ht) #t)
  (test (>= (hash-table-size ht) 277) #t)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table? arg) #f))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (hash-table? (make-vector 3 '())) #f)

(let ((ht (make-hash-table)))	
  (test (hash-table-ref ht 'not-a-key) #f)
  (test (hash-table-ref ht "not-a-key") #f)
  (hash-table-set! ht 'key 3/4)
  (hash-table-set! ht "key" "hi")
  (test (hash-table-ref ht "key") "hi")
  (test (hash-table-ref ht 'key) 3/4)
  
  (hash-table-set! ht 'asd 'hiho)
  (test (hash-table-ref ht 'asd) 'hiho)
  (hash-table-set! ht 'asd 1234)
  (test (hash-table-ref ht 'asd) 1234))

(for-each
 (lambda (arg)
   (test (hash-table-ref arg 'key) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht1 (make-hash-table 653))
      (ht2 (make-hash-table 277)))
  (test (equal? ht1 ht2) #f)
  (hash-table-set! ht1 'key 'hiho)
  (hash-table-set! ht2 (hash-table-ref ht1 'key) 3.14)
  (test (>= (hash-table-size ht1) 653) #t)
  (test (hash-table-ref ht2 'hiho) 3.14)
  (test (hash-table-ref ht2 (hash-table-ref ht1 'key)) 3.14))

(let ((ht1 (make-hash-table)))
   (set! (ht1 1) 'hi)
   (let ((ht2 (make-hash-table)))
      (set! (ht2 1) ht1)
      (test ((ht2 1) 1) 'hi)))

(test (hash-table?) 'error)
(test (hash-table? 1 2) 'error)

(test (make-hash-table 10 1) 'error)

(let ((ht (make-hash-table)))
  (test (hash-table? ht ht) 'error)
  (test (hash-table-ref ht #\a #\b) 'error)
  (test (hash-table-ref ht) 'error)
  (test (hash-table-ref) 'error)
  (test (hash-table-set!) 'error)
  (test (hash-table-set! ht) 'error)
  (test (hash-table-set! ht #\a) 'error)
  (test (hash-table-set! ht #\a #\b #\c) 'error)
  (test (fill! ht 123) 'error)
  (set! (ht 'key) 32)
  (test (ht 'key) 32)
  (set! (ht :key) 123)
  (test (ht 'key) 32)
  (test (ht :key) 123)
  (fill! ht '())
  (test (ht 'key) #f))

(let ((ht (make-hash-table)))
  (test (hash-table-set! ht #\a 'key) 'key)
  (for-each
   (lambda (arg)
     (test (hash-table-set! ht arg 3.14) 3.14))
   (list #\a '#(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (for-each
   (lambda (arg)
     (test (hash-table-ref ht arg) 3.14))
   (list #\a '#(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table-size arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (make-hash-table arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht1 (make-hash-table))
      (ht2 (make-hash-table)))
  (test (equal? ht1 ht2) #t)
  (test (equal? ht1 (make-vector (hash-table-size ht1) '())) #f)
  (hash-table-set! ht1 'key 'hiho)
  (test (equal? ht1 ht2) #f)
  (hash-table-set! ht2 'key 'hiho)
  (test (equal? ht1 ht2) #t)
  )

(let ((ht (make-hash-table 1)))
  (test (>= (length ht) 1) #t)
  (set! (ht 1) 32)
  (test (>= (length ht) 1) #t))

(let ((ht (hash-table '("hi" . 32) '("ho" . 1))))
  (test (ht "hi") 32)
  (test (ht "ho") 1))

(let ((ht (hash-table)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 461) #t)
  (test (ht 1) #f))

;; no null hash-tables?

(let ((ht (make-hash-table)))
  (test (map (lambda (x) x) ht) '())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 0)
  (test (map (lambda (x y) (cons x y)) (list 1 2 3) ht) '())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) #(1 2 3) ht) ctr) 0)
  (test (map (lambda (x y) (cons x y)) ht "123") '())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht '()) ctr) 0)

  (let ((rt (reverse ht)))
    (test (map (lambda (x) x) rt) '())
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 0))

  (set! (ht 1) 32)
  ;; these need to be independent of entry order
  
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 1)
  (test (map (lambda (x y) (cons x y)) '() ht) '())
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht "") ctr) 0)
  (test (sort! (map (lambda (x y) (max (cdr x) y)) ht (list 1 2 3)) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (max (cdr x) y))) ht #(1 2 3)) ctr) 32)

  (let ((rt (reverse ht)))
    (test (equal? (rt 32) 1) #t)
    (test (equal? (rt 1) #f) #t)
    (test (ht (rt 32)) 32)
    (test (sort! (map (lambda (x) (cdr x)) rt) <) '(1))
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 1)
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 32) 123)
    (test (rt 32) 123)
    (test (ht 32) #f)
    (test (ht 1) 32))

  (set! (ht 2) 1)
  (test (ht (ht 2)) 32)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 2)
  (set! (ht 3) 123)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32 123))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 3)
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1 2)) ctr) 2)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12345" ht '(1 2 3 4 5 6)) ctr) 3)
  
  (test (sort! (map (lambda (x y) (max x (cdr y))) (list -1 -2 -3 -4) ht) <) '(1 32 123))
  (test (let ((sum 0)) (for-each (lambda (x y) (set! sum (+ sum x (cdr y)))) #(10 20 30) ht) sum) 216)
  
  (let ((rt (reverse ht)))
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht))
  
  (set! (ht (list 1 2 3)) "hi")
  (test (ht '(1 2 3)) "hi")
  (test (ht 2) 1)
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 4)
  (set! (ht "hi") 2)
  (test (ht "hi") 2)
  (test (ht (ht (ht "hi"))) 32)

  (let ((rt (reverse ht)))
    (test (rt "hi") '(1 2 3))
    (test (rt 2) "hi")
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 2) "ho")
    (test (rt 2) "ho")
    (test (ht '(1 2 3)) "hi")
    (set! (rt 123) 321)
    (test (rt 123) 321)
    (test (ht 3) 123))

  (fill! ht '())
  (set! (ht "hi") 1)
  (set! (ht "hoi") 2)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 2))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 2)
  
  (let ((rt (reverse ht)))
    (test (rt 2) "hoi")
    (set! (rt 2) "ha")
    (test (ht "hoi") 2))

  (set! (ht #\a) #\b)
  (test (ht #\a) #\b)
  (test (ht "hi") 1)

  (fill! ht '())
  (set! (ht #(1)) #(2))
  (test (ht #(1)) #(2))
  (set! (ht '(1)) '(3))
  (set! (ht "1") "4")
  (set! (ht ht) "5")
  (test (ht ht) "5")
  (test (ht '(1)) '(3))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 4)  
    
  (let ((rt (reverse ht)))
    (test (rt "5") ht)
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht))
)  

(let ((ht (make-hash-table 31)))
  (let ((ht1 (make-hash-table 31)))
    (set! (ht1 'a1) 'b1)
    (set! (ht 'a0) ht1)
    (test ((ht 'a0) 'a1) 'b1)
    (test (hash-table-ref ht 'a0 'a1) 'b1)
    (test (ht 'a0 'a1) 'b1)))

;; there's no real need for multidim hashes:

(let ((ht (make-hash-table)))
   (set! (ht (cons 'a 1)) 'b)
   (set! (ht (cons 'a 2)) 'c)
   (set! (ht (cons 'b 1)) 'd)
   (test (ht '(a . 1)) 'b)
   (test (ht '(b . 1)) 'd)
   (set! (ht '(a . 2)) 32)
   (test (ht '(a . 2)) 32))

(let ((ht (make-hash-table)))
  (test (ht) 'error)
  (test (ht 0 1) 'error))





;;; --------------------------------------------------------------------------------
;;; PORTS
;;; --------------------------------------------------------------------------------

(define start-input-port (current-input-port))
(define start-output-port (current-output-port))

(test (input-port? (current-input-port)) #t)

(for-each
 (lambda (arg)
   (if (input-port? arg)
       (format #t "(input-port? ~A) -> #t?~%" arg)))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-input-file "s7test.scm" input-port?) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "call-with-input-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (let ((this-file (open-input-file "s7test.scm"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "open-input-file clobbered current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (call-with-input-string "(+ 1 2)" input-port?) #t)
(test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)

(test (+ 100 (call-with-input-string "123" (lambda (p) (values (read p) 1)))) 224)

(test (call-with-input-string
       "1234567890"
       (lambda (p)
	 (call-with-input-string
	  "0987654321"
	  (lambda (q)
            (+ (read p) (read q))))))
      2222222211)

(test (call-with-input-string
       "12345 67890"
       (lambda (p)
	 (call-with-input-string
	  "09876 54321"
	  (lambda (q)
            (- (+ (read p) (read q)) (read p) (read q))))))
      -99990)

(call-with-output-file "empty-file" (lambda (p) #f))
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-char p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-byte p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-line p)))) #t)
(test (load "empty-file") #<unspecified>)

(let ()
  (define (io-func) (lambda (p) (eof-object? (read-line p))))
  (test (call-with-input-file (let () "empty-file") (io-func)) #t))

(call-with-output-file "empty-file" (lambda (p) (write-char #\a p)))
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (read-char p) #\a) (eof-object? (read-char p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (symbol->string (read p)) "a") (eof-object? (read p))))) #t) ; Guile also returns a symbol here
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (integer->char (read-byte p)) #\a) (eof-object? (read-byte p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (read-line p) "a") (eof-object? (read-line p))))) #t)

(call-with-output-file "empty-file" (lambda (p) (for-each (lambda (c) (write-char c p)) "#b11")))
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (char=? (read-char p) #\#) 
						(char=? (read-char p) #\b) 
						(char=? (read-char p) #\1) 
						(char=? (read-char p) #\1) 
						(eof-object? (read-char p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (= (read p) 3) 
						(eof-object? (read p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (= (read-byte p) (char->integer #\#))
						(= (read-byte p) (char->integer #\b))
						(= (read-byte p) (char->integer #\1))
						(= (read-byte p) (char->integer #\1))
						(eof-object? (read-byte p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (string=? (read-line p) "#b11") 
						(eof-object? (read-line p))))) 
      #t)
(test (load "empty-file") 3)


;; these apparently jump out of the enclosing load too
(for-each
 (lambda (arg)
   (test (load arg) 'error)
   (test (load "empty-file" arg) 'error))
 (list '() (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1))))
(test (load) 'error)
(test (load "empty-file" (current-environment) 1) 'error)
(test (load "not a file") 'error)

(test (error) 'error)


(test (output-port? (current-output-port)) #t)
(write-char #\space (current-output-port))
(write " " (current-output-port))
(newline (current-output-port))


(for-each
 (lambda (arg)
   (if (output-port? arg)
       (format #t "(output-port? ~A) -> #t?~%" arg)))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-output-file "tmp1.r5rs" output-port?) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "call-with-output-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((this-file (open-output-file "tmp1.r5rs"))) (let ((res (output-port? this-file))) (close-output-port this-file) res)) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "open-output-file clobbered current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((val #f)) (call-with-output-string (lambda (p) (set! val (output-port? p)))) val) #t)
(test (let ((res #f)) (let ((this-file (open-output-string))) (set! res (output-port? this-file)) (close-output-port this-file) res)) #t)



(for-each
 (lambda (arg)
   (if (eof-object? arg)
       (format #t "(eof-object? ~A) -> #t?~%" arg)))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> (lambda (a) (+ a 1))))

(call-with-output-file "tmp1.r5rs" (lambda (p) (display "3.14" p)))
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (read p) (let ((val (read p))) (eof-object? val)))) #t)


(test (call-with-input-file "tmp1.r5rs" (lambda (p) (read-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (peek-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (peek-char p) (read-char p))) #\3)
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (list->string (list (read-char p) (read-char p) (read-char p) (read-char p))))) "3.14")
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (list->string (list (read-char p) (peek-char p) (read-char p) (read-char p) (peek-char p) (read-char p))))) "3..144")

(for-each
 (lambda (arg)
   (call-with-output-file "tmp1.r5rs" (lambda (p) (write arg p)))
   (test (call-with-input-file "tmp1.r5rs" (lambda (p) (read p))) arg))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3 0) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) (cons 1 2)))

;;; r4rstest
(let* ((write-test-obj '(#t #f a () 9739 -3 . #((test) "te \" \" st" "" test #() b c)))
       (load-test-obj (list 'define 'foo (list 'quote write-test-obj))))
  
  (define (check-test-file name)
    (let ((val (call-with-input-file
		   name
		 (lambda (test-file)
		   (test (read test-file) load-test-obj)
		   (test (eof-object? (peek-char test-file)) #t)
		   (test (eof-object? (read-char test-file)) #t)
		   (input-port? test-file)))))
      (if (not (eq? val #t))
	  (format #t "input-port? in call-with-input-file? returned ~A from ~A~%" val name))))
  
  (test (call-with-output-file
	    "tmp1.r5rs"
	  (lambda (test-file)
	    (write-char #\; test-file)
	    (display #\; test-file)
	    (display ";" test-file)
	    (write write-test-obj test-file)
	    (newline test-file)
	    (write load-test-obj test-file)
	    (output-port? test-file))) #t)
  (check-test-file "tmp1.r5rs")
  
  (let ((test-file (open-output-file "tmp2.r5rs")))
    (write-char #\; test-file)
    (display #\; test-file)
    (display ";" test-file)
    (write write-test-obj test-file)
    (newline test-file)
    (write load-test-obj test-file)
    (test (output-port? test-file) #t)
    (close-output-port test-file)
    (check-test-file "tmp2.r5rs")))


(call-with-output-file "tmp1.r5rs" (lambda (p) (display "3.14" p)))
(test (with-input-from-file "tmp1.r5rs" (lambda () (read))) 3.14)
(if (not (eq? start-input-port (current-input-port)))
    (format #t "with-input-from-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (with-input-from-file "tmp1.r5rs" (lambda () (eq? (current-input-port) start-input-port))) #f)

(test (with-output-to-file "tmp1.r5rs" (lambda () (eq? (current-output-port) start-output-port))) #f)
(if (not (eq? start-output-port (current-output-port)))
    (format #t "with-output-to-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))


(let ((newly-found-sonnet-probably-by-shakespeare 
       "This is the story, a sad tale but true \
        Of a programmer who had far too little to do.\
        One day as he sat in his hut swilling stew, \
        He cried \"CLM takes forever, it's stuck in a slough!,\
        Its C code is slow, too slow by a few.\
        Why, with just a small effort, say one line or two,\
        It could outpace a no-op, you could scarcely say 'boo'\"!\
        So he sat in his kitchen and worked like a dog.\
        He typed and he typed 'til his mind was a fog. \
        Now 6000 lines later, what wonders we see!  \
        CLM is much faster, and faster still it will be!\
        In fact, for most cases, C beats the DSP!  \
        But bummed is our coder; he grumbles at night.  \
        That DSP code took him a year to write.  \
        He was paid many dollars, and spent them with glee,\
        But his employer might mutter, this result were he to see."))
  
  (call-with-output-file "tmp1.r5rs"
    (lambda (p)
      (write newly-found-sonnet-probably-by-shakespeare p)))
  
  (let ((sonnet (with-input-from-file "tmp1.r5rs"
		  (lambda ()
		    (read)))))
    (if (or (not (string? sonnet))
	    (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	(format #t "write/read long string returned: ~A~%" sonnet)))
  
  (let ((file (open-output-file "tmp1.r5rs")))
    (let ((len (string-length newly-found-sonnet-probably-by-shakespeare)))
      (write-char #\" file)
      (do ((i 0 (+ i 1)))
	  ((= i len))
	(let ((chr (string-ref newly-found-sonnet-probably-by-shakespeare i)))
	  (if (char=? chr #\")
	      (write-char #\\ file))
	  (write-char chr file)))
      (write-char #\" file)
      (close-output-port file)))
  
  (let ((file (open-input-file "tmp1.r5rs")))
    (let ((sonnet (read file)))
      (close-input-port file)
      (if (or (not (string? sonnet))
	      (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	  (format #t "write-char/read long string returned: ~A~%" sonnet)))))

(let ((file (open-output-file "tmp1.r5rs")))
  (for-each
   (lambda (arg)
     (write arg file)
     (write-char #\space file))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-output-port file))

(let ((file (open-input-file "tmp1.r5rs")))
  (for-each
   (lambda (arg)
     (let ((val (read file)))
       (if (not (equal? val arg))
	   (format #t "read/write ~A returned ~A~%" arg val))))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-input-port file))

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (write lists)))

(let ((val (with-input-from-file "tmp1.r5rs"
	     (lambda ()
	       (read)))))
  (if (not (equal? val lists))
      (format #t "read/write lists returned ~A~%" val)))

(if (not (string=? "" (with-output-to-string (lambda () (display "")))))
    (format #t "with-output-to-string null string?"))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string "hiho123"
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str "hiho123"))
      (format #t "with string ports: ~S?~%" str)))


(if (not (eof-object? (with-input-from-string "" (lambda () (read-char)))))
    (format #t ";input from null string not #<eof>?~%")
    (let ((EOF (with-input-from-string "" (lambda () (read-char)))))
      (if (not (eq? (with-input-from-string "" (lambda () (read-char)))
		    (with-input-from-string "" (lambda () (read-char)))))
	  (format #t "#<eof> is not eq? to itself?~%"))
      (if (char? EOF)
	  (do ((c 0 (+ c 1)))
	      ((= c 256))
	    (if (char=? EOF (integer->char c))
		(format #t "#<eof> is char=? to ~C~%" (integer->char c)))))))

(test (+ 100 (call-with-output-file "tmp.r5rs" (lambda (p) (write "1" p) (values 1 2)))) 103)
(test (+ 100 (with-output-to-file "tmp.r5rs" (lambda () (write "2") (values 1 2)))) 103)

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string "hiho123"
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((or (not (char-ready?))
			    (eof-object? c)))
		     (display c))))))))
  (if (not (string=? str "hiho123"))
      (format #t "with string ports: ~S?~%" str)))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string ""
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str ""))
      (format #t "with string ports and null string: ~S?~%" str)))

(let ((str (with-output-to-string ; this is from the guile-user mailing list, I think -- don't know who wrote it
	     (lambda ()
	       (with-input-from-string "A2B5E3426FG0ZYW3210PQ89R."
		 (lambda ()
		   (call/cc
		    (lambda (hlt)
		      (define (nextchar)
			(let ((c (read-char)))
			  (if (and (char? c) 
				   (char=? c #\space))
			      (nextchar) 
			      c)))
		      
		      (define inx
			(lambda()
			  (let in1 ()
			    (let ((c (nextchar)))
			      (if (char-numeric? c)
				  (let ((r (nextchar)))
				    (let out*n ((n (- (char->integer c) (char->integer #\0))))
				      (out r)
				      (if (not (zero? n))
					  (out*n (- n 1)))))
				  (out c))
			      (in1)))))
		      
		      (define (move-char c)
			(write-char c)
			(if (char=? c #\.)
			    (begin (hlt))))
		      
		      (define outx
			(lambda()
			  (let out1 ()
			    (let h1 ((n 16))
			      (move-char (in))
			      (move-char (in))
			      (move-char (in))
			      (if (= n 1)
				  (begin (out1))
				  (begin (write-char #\space) (h1 (- n 1))) )))))
		      
		      (define (in)
			(call/cc (lambda(return)
				   (set! outx return)
				   (inx))))
		      
		      (define (out c)
			(call/cc (lambda(return) 
				   (set! inx return)
				   (outx c))))
		      (outx)))))))))
  (if (not (string=? str "ABB BEE EEE E44 446 66F GZY W22 220 0PQ 999 999 999 R."))
      (format #t "call/cc with-input-from-string str: ~A~%" str)))

(let ((badfile "tmp1.r5rs"))
  (let ((p (open-output-file badfile)))
    (close-output-port p))
  (load badfile))


(let ((loadit "tmp1.r5rs"))
  (let ((p (open-output-file loadit)))
    (display "(define s7test-var 314) (define (s7test-func) 314) (define-macro (s7test-mac a) `(+ ,a 2))" p)
    (newline p)
    (close-output-port p)
    (load loadit)
    (test (= s7test-var 314) #t)
    (test (s7test-func) 314)
    (test (s7test-mac 1) 3)
    (set! p (open-output-file loadit)) ; hopefully this starts a new file
    (display "(define s7test-var 3) (define (s7test-func) 3) (define-macro (s7test-mac a) `(+ ,a 1))" p)
    (newline p)
    (close-output-port p)
    (load loadit)
    (test (= s7test-var 3) #t)
    (test (s7test-func) 3)
    (test (s7test-mac 1) 2)
    ))

(test (+ 100 (with-input-from-string "123" (lambda () (values (read) 1)))) 224)

(for-each
 (lambda (op)
   (for-each
    (lambda (arg) ;(format #t "(~A ~A)~%" op arg)
      (test (op arg) 'error))
    (list (integer->char 65) 1 0 -1 (list 1) (cons 1 2) #f 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list char-ready? set-current-output-port set-current-input-port set-current-error-port
       close-input-port close-output-port open-input-file open-output-file
       read-char peek-char read 
       (lambda (arg) (write-char #\a arg))
       (lambda (arg) (write "hi" arg))
       (lambda (arg) (display "hi" arg))
       call-with-input-file with-input-from-file call-with-output-file with-output-to-file))

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (display "this is a test")
    (newline)))
    
(test (call-with-input-file "tmp1.r5rs" (lambda (p) (integer->char (read-byte p)))) #\t)
(test (with-input-from-string "123" (lambda () (read-byte))) 49)
;(test (with-input-from-string "1/0" (lambda () (read))) 'error) ; this is a reader error in CL
;;; this test causes trouble when s7test is called from snd-test -- I can't see why

(let ((bytes (vector #o000 #o000 #o000 #o034 #o000 #o001 #o215 #o030 #o000 #o000 #o000 #o022 #o000 
		     #o000 #o126 #o042 #o000 #o000 #o000 #o001 #o000 #o000 #o000 #o000 #o000 #o001)))
  (with-output-to-file "tmp1.r5rs"
    (lambda ()
      (for-each
       (lambda (b)
	 (write-byte b))
       bytes)))
  
  (let ((ctr 0))
    (call-with-input-file "tmp1.r5rs"
      (lambda (p)	
	(if (not (string=? (port-filename p) "tmp1.r5rs")) (display (port-filename p)))	
	(let loop ((val (read-byte p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format #t "read-byte done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) val))
		    (format #t "read-byte bytes[~D]: ~A ~A~%" ctr (bytes ctr) val))
		(set! ctr (+ 1 ctr))
		(loop (read-byte p))))))))
  
  (let ((ctr 0))
    (call-with-input-file "tmp1.r5rs"
      (lambda (p)
	(let loop ((val (read-char p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format #t "read-char done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) (char->integer val)))
		    (format #t "read-char bytes[~D]: ~A ~A~%" ctr (bytes ctr) (char->integer val)))
		(set! ctr (+ 1 ctr))
		(loop (read-char p))))))))
  )

(with-output-to-file "tmp1.r5rs"
  (lambda ()
    (if (not (string=? (port-filename (current-output-port)) "tmp1.r5rs")) (display (port-filename (current-output-port))))
    (display "(+ 1 2) 32")
    (newline)
    (display "#\\a  -1")))

(with-input-from-file "tmp1.r5rs"
  (lambda ()
    (if (not (string=? (port-filename (current-input-port)) "tmp1.r5rs")) (display (port-filename (current-input-port))))
    (let ((val (read)))
      (if (not (equal? val (list '+ 1 2)))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val 32))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val #\a))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val -1))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format #t "read: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format #t "read again: ~A~%" val)))))


(for-each
 (lambda (arg)
   (test (char-ready? arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3) abs 3.14 3/4 1.0+1.0i #f #t (if #f #f) (lambda (a) (+ a 1))))


;;; -------- format --------

(test (format #f "hiho") "hiho")
(test (format #f "") "")
(test (format #f "a") "a")

(test (format #f "~~") "~")
(test (format #f "~~~~") "~~")
(test (format #f "a~~") "a~")
(test (format #f "~~a") "~a")

(test (format #f "hiho~%ha") (string-append "hiho" (string #\newline) "ha"))
(test (format #f "~%") (string #\newline))
(test (format #f "~%ha") (string-append (string #\newline) "ha"))
(test (format #f "hiho~%") (string-append "hiho" (string #\newline)))

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~A" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~A\" ") (display arg) 
		(display " returned \"") (display val) 
		(display "\" but expected \"") (display res) (display "\"") 
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  '#(1 2 3)   3.14   3/4  1.5+1.5i '()  '#(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "hiho" "-1" "a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~A ho" 1) "hi 1 ho")
(test (format #f "hi ~a ho" 1) "hi 1 ho")
(test (format #f "~a~A~a" 1 2 3) "123")
(test (format #f "~a~~~a" 1 3) "1~3")
(test (format #f "~a~%~a" 1 3) (string-append "1" (string #\newline) "3"))

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~S" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~S\" ") (display arg) 
		(display " returned \"") (display val) 
		(display "\" but expected \"") (display res) (display "\"") 
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  '#(1 2 3)   3.14   3/4  1.5+1.5i '()  '#(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "\"hiho\"" "-1" "#\\a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~S ho" 1) "hi 1 ho")
(test (format #f "hi ~S ho" "abc") "hi \"abc\" ho")
(test (format #f "~s~a" #\a #\b) "#\\ab")
(test (format #f "~C~c~C" #\a #\b #\c) "abc")

(test (format #f "~{~A~}" '(1 2 3)) "123")
(test (format #f "asb~{~A ~}asb" '(1 2 3 4)) "asb1 2 3 4 asb")
(test (format #f "asb~{~A ~A.~}asb" '(1 2 3 4)) "asb1 2.3 4.asb")
(test (format #f ".~{~A~}." '()) "..")

(test (format #f "~{~A ~A ~}" '(1 "hi" 2 "ho")) "1 hi 2 ho ")
(test (format #f "~{.~{+~A+~}.~}" (list (list 1 2 3) (list 4 5 6))) ".+1++2++3+..+4++5++6+.")
(test (format #f "~{~s ~}" '(fred jerry jill)) "fred jerry jill ")
(test (format #f "~{~s~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{~s~^~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{.~{~A~}+~{~A~}~}" '((1 2) (3 4 5) (6 7 8) (9))) ".12+345.678+9")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)))) ".+-1-2+-3-4-5")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)) ((6) (7 8 9)))) ".+-1-2+-3-4-5.+-6+-7-8-9")

(test (format #f "~A ~* ~A" 1 2 3) "1  3")
(test (format #f "~*" 1) "")
(test (format #f "~{~* ~}" '(1 2 3)) "   ")

(test (format #f "this is a ~
             sentence") "this is a sentence")

;; ~nT handling is a mess -- what are the defaults?  which is column 1? do we space up to or up to and including?

(test (format #f "asdh~20Thiho") "asdh                hiho")
(test (format #f "asdh~2Thiho") "asdhhiho")
(test (format #f "a~Tb") "ab")
(test (format #f "0123456~4,8Tb") "0123456     b")
					;      (test (format #f "XXX~%0123456~4,8Tb") (string-append "XXX" (string #\newline) "0123456    b")) ; clearly wrong...
(test (format #f "0123456~0,8Tb") "0123456 b")
					;      (test (format #f "0123456~10,8Tb") "0123456           b")
(test (format #f "0123456~1,0Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~,Tb") "0123456b")
					;      (test (format #f "0123456~7,10Tb") "0123456          b") 
					;      (test (format #f "0123456~8,10tb") "0123456           b")
(test (format #f "0123456~3,12tb") "0123456        b")

					;      (test (format #f "~40TX") "                                       X")
					;      (test (format #f "X~,8TX~,8TX") "X       X       X")
(test (format #f "X~8,TX~8,TX") "X       XX")
					;      (test (format #f "X~8,10TX~8,10TX") "X                 X         X")
(test (format #f "X~8,0TX~8,0TX") "X       XX")
(test (format #f "X~0,8TX~0,8TX") "X       X       X")
					;      (test (format #f "X~1,8TX~1,8TX") "X        X       X")
					;      (test (format #f "X~,8TX~,8TX") "X       X       X")
(test (format #f "X~TX~TX") "XXX") ; clisp and sbcl say "X X X" here and similar differences elsewhere
(test (format #f "X~0,0TX~0,0TX") "XXX")
(test (format #f "X~0,TX~0,TX") "XXX")
(test (format #f "X~,0TX~,0TX") "XXX")

(test (string=? (format #f "~%~&" ) (string #\newline)) #t)
(test (string=? (format #f "~%a~&" ) (string #\newline #\a #\newline)) #t)
(test (string=? (format #f "~%~%") (string #\newline #\newline)) #t)

(test (format #f "~2,1F" 0.5) "0.5")
(test (format #f "~:2T") 'error)
(test (format #f "~2,1,3F" 0.5) 'error)
(test (format #f "~<~W~>" 'foo) 'error)
(test (format #f "~{12") 'error)
(test (format #f "~{}") 'error)
(test (format #f "~{}" '(1 2)) 'error)
(test (format #f "{~}" '(1 2)) 'error)
(test (format #f "~{~{~}}" '(1 2)) 'error)
(test (format #f "#|~|#|") 'error)
(test (format #f "~1.5F" 1.5) 'error)
(test (format #f "~1+iF" 1.5) 'error)
(test (format #f "~1,1iF" 1.5) 'error)
(test (format #f "~0" 1) 'error)
(test (format #f "~1") 'error)
(test (format #f "~^" 1) 'error)
(test (format #f "~^") "")
(test (format #f "~D~" 9) "9~")
(test (format #f "~&" 9) 'error)
(test (format #f "~D~100T~D" 1 1) "1                                                                                                   1")
(test (format #f ".~P." 1) "..")
(test (format #f ".~P." 1.0) "..")
(test (format #f ".~P." 1.2) ".s.")
(test (format #f ".~P." 2) ".s.")
(test (format #f ".~p." 1) "..")
(test (format #f ".~p." 1.0) "..")
(test (format #f ".~p." 1.2) ".s.")
(test (format #f ".~p." 2) ".s.")
(test (format #f ".~@P." 1) ".y.")
(test (format #f ".~@P." 1.0) ".y.")
(test (format #f ".~@P." 1.2) ".ies.")
(test (format #f ".~@P." 2) ".ies.")
(test (format #f ".~@p." 1) ".y.")
(test (format #f ".~@p." 1.0) ".y.")
(test (format #f ".~@p." 1.2) ".ies.")
(test (format #f ".~@p." 2) ".ies.")

(test (format #f (string #\~ #\a) 1) "1")
(test (format #f (format #f "~~a") 1) "1")
(test (format #f (format #f "~~a") (format #f "~D" 1)) "1")

(test (format #f "~f" (/ 1 3)) "1/3") ; hmmm -- should it call exact->inexact?
(test (format #f "~f" 1) "1")

(if with-bignums
    (begin
      (test (format #f "~A" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601")
      (test (format #f "~D" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601")
    ))
(test (format #f "~@F" 1.23) 'error)
(test (format #f "~{testing ~D ~C ~}" (list 0 #\( 1 #\) 2 #\* 3 #\+ 4 #\, 5 #\- 6 #\. 7 #\/ 8 #\0 9 #\1 10 #\2 11 #\3 12 #\4 13 #\5 14 #\6 15 #\7 16 #\8 17 #\9 18 #\: 19 #\; 20 #\< 21 #\= 22 #\> 23 #\? 24 #\@ 25 #\A 26 #\B 27 #\C 28 #\D 29 #\E 30 #\F 31 #\G 32 #\H 33 #\I 34 #\J 35 #\K 36 #\L 37 #\M 38 #\N 39 #\O 40 #\P 41 #\Q 42 #\R 43 #\S 44 #\T 45 #\U 46 #\V 47 #\W 48 #\X 49 #\Y 50 #\( 51 #\) 52 #\* 53 #\+ 54 #\, 55 #\- 56 #\. 57 #\/ 58 #\0 59 #\1 60 #\2 61 #\3 62 #\4 63 #\5 64 #\6 65 #\7 66 #\8 67 #\9 68 #\: 69 #\; 70 #\< 71 #\= 72 #\> 73 #\? 74 #\@ 75 #\A 76 #\B 77 #\C 78 #\D 79 #\E 80 #\F 81 #\G 82 #\H 83 #\I 84 #\J 85 #\K 86 #\L 87 #\M 88 #\N 89 #\O 90 #\P 91 #\Q 92 #\R 93 #\S 94 #\T 95 #\U 96 #\V 97 #\W 98 #\X 99 #\Y))
      "testing 0 ( testing 1 ) testing 2 * testing 3 + testing 4 , testing 5 - testing 6 . testing 7 / testing 8 0 testing 9 1 testing 10 2 testing 11 3 testing 12 4 testing 13 5 testing 14 6 testing 15 7 testing 16 8 testing 17 9 testing 18 : testing 19 ; testing 20 < testing 21 = testing 22 > testing 23 ? testing 24 @ testing 25 A testing 26 B testing 27 C testing 28 D testing 29 E testing 30 F testing 31 G testing 32 H testing 33 I testing 34 J testing 35 K testing 36 L testing 37 M testing 38 N testing 39 O testing 40 P testing 41 Q testing 42 R testing 43 S testing 44 T testing 45 U testing 46 V testing 47 W testing 48 X testing 49 Y testing 50 ( testing 51 ) testing 52 * testing 53 + testing 54 , testing 55 - testing 56 . testing 57 / testing 58 0 testing 59 1 testing 60 2 testing 61 3 testing 62 4 testing 63 5 testing 64 6 testing 65 7 testing 66 8 testing 67 9 testing 68 : testing 69 ; testing 70 < testing 71 = testing 72 > testing 73 ? testing 74 @ testing 75 A testing 76 B testing 77 C testing 78 D testing 79 E testing 80 F testing 81 G testing 82 H testing 83 I testing 84 J testing 85 K testing 86 L testing 87 M testing 88 N testing 89 O testing 90 P testing 91 Q testing 92 R testing 93 S testing 94 T testing 95 U testing 96 V testing 97 W testing 98 X testing 99 Y ")


(test (format #f "~D" 123) "123")
(test (format #f "~X" 123) "7b")
(test (format #f "~B" 123) "1111011")
(test (format #f "~O" 123) "173")

(test (format #f "~10D" 123) "       123")
(test (format #f "~10X" 123) "        7b")
(test (format #f "~10B" 123) "   1111011")
(test (format #f "~10O" 123) "       173")

(test (format #f "~D" -123) "-123")
(test (format #f "~X" -123) "-7b")
(test (format #f "~B" -123) "-1111011")
(test (format #f "~O" -123) "-173")

(test (format #f "~10D" -123) "      -123")
(test (format #f "~10X" -123) "       -7b")
(test (format #f "~10B" -123) "  -1111011")
(test (format #f "~10O" -123) "      -173")

(test (format #f "~d" 123) "123")
(test (format #f "~x" 123) "7b")
(test (format #f "~b" 123) "1111011")
(test (format #f "~o" 123) "173")

(test (format #f "~10d" 123) "       123")
(test (format #f "~10x" 123) "        7b")
(test (format #f "~10b" 123) "   1111011")
(test (format #f "~10o" 123) "       173")

(test (format #f "~d" -123) "-123")
(test (format #f "~x" -123) "-7b")
(test (format #f "~b" -123) "-1111011")
(test (format #f "~o" -123) "-173")

(test (format #f "~10d" -123) "      -123")
(test (format #f "~10x" -123) "       -7b")
(test (format #f "~10b" -123) "  -1111011")
(test (format #f "~10o" -123) "      -173")

(test (format #f "~D" most-positive-fixnum) "9223372036854775807")
(test (format #f "~D" (+ 1 most-negative-fixnum)) "-9223372036854775807")
      
(test (format #f "~X" most-positive-fixnum) "7fffffffffffffff")
(test (format #f "~X" (+ 1 most-negative-fixnum)) "-7fffffffffffffff")
      
(test (format #f "~O" most-positive-fixnum) "777777777777777777777")
(test (format #f "~O" (+ 1 most-negative-fixnum)) "-777777777777777777777")
      
(test (format #f "~B" most-positive-fixnum) "111111111111111111111111111111111111111111111111111111111111111")
(test (format #f "~B" (+ 1 most-negative-fixnum)) "-111111111111111111111111111111111111111111111111111111111111111")
      
(num-test (inexact->exact most-positive-fixnum) most-positive-fixnum)

(test (format #f "~0D" 123) "123")
(test (format #f "~0X" 123) "7b")
(test (format #f "~0B" 123) "1111011")
(test (format #f "~0O" 123) "173")

(test (format #f "" 1) 'error)
(test (format #f "hiho" 1) 'error)
(test (format #f "a~%" 1) 'error) ; some just ignore extra args

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format arg "hiho")) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format ") (display arg) (display " \"hiho\")")
		(display " returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i 'hi abs (lambda () 1) '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f ") (display arg) (display ")")
		(display " returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 #f #t '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs (lambda () 1) '#(()) (list 1 2 3) '(1 . 2)))

(test (format #f "hi ~A ho" 1 2) 'error)
(test (format #f "hi ~A ho") 'error)
(test (format #f "hi ~S ho") 'error)
(test (format #f "hi ~S ho" 1 2) 'error)
(test (format #f "~C" 1) 'error)
(test (format #f "123 ~R 321" 1) 'error)
(test (format #f "123 ~,3R 321" 1) 'error)
(test (format #f "~,2,3,4D" 123) 'error)

(test (format #f "hi ~Z ho") 'error)
(test (format #f "hi ~+ ho") 'error)
(test (format #f "hi ~# ho") 'error)

(test (format #f "hi ~} ho") 'error)
(test (format #f "hi {ho~}") 'error)

(test (format #f "asb~{~A asd" '(1 2 3)) 'error)
(test (format #f "~{~A~}" 1 2 3) 'error)
(test (format #f "asb~{~}asd" '(1 2 3)) 'error) ; this apparently makes the format.scm in Guile hang? [fixed]
(test (format #f "asb~{ ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ hiho~~~}asd" '(1 2 3)) 'error)

(test (format #f "~{~A ~A ~}" '(1 "hi" 2)) 'error)
(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f "~F" arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f \"~F\" ") (display arg)
		(display ") returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list #\a '#(1 2 3) "hi" '() 'hi abs (lambda () 1) '#(()) (list 1 2 3) '(1 . 2)))

(test (format #f "~D") 'error)
					;	    (test (format () "hi") "hi") ; not sure this is a good idea
(test (format #f "~F" "hi") 'error)
(test (format #f "~D" #\x) 'error)
(test (format #f "~C" (list 1 2 3)) 'error)
(test (format #f "~1/4F" 1.4) 'error)
(test (format #f "~1.4F" 1.4) 'error)
(test (format #f "~F" (real-part (log 0.0))) "-inf.0")
(test (format #f "~F" (/ (real-part (log 0.0)) (real-part (log 0.0)))) "nan.0")
(test (format #f "~1/4T~A" 1) 'error)
(test (format #f "~T") "")


(call-with-output-file "tmp1.r5rs" (lambda (p) (format p "this ~A ~C test ~D" "is" #\a 3)))
(let ((res (call-with-input-file "tmp1.r5rs" (lambda (p) (read-line p)))))
  (if (not (string=? res "this is a test 3"))
      (begin 
	(display "call-with-input-file + format to \"tmp1.r5rs\" ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(let ((val (format #f "line 1~%line 2~%line 3")))
  (with-input-from-string val
    (lambda ()
      (let ((line1 (read-line)))
	(test (string=? line1 "line 1") #t))
      (let ((line2 (read-line)))
	(test (string=? line2 "line 2") #t))
      (let ((line3 (read-line)))
	(test (string=? line3 "line 3") #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t)))))


(let ((val (format #f "line 1~%line 2~%line 3")))
  (call-with-input-string val
			  (lambda (p)
			    (let ((line1 (read-line p #t)))
			      (test (string=? line1 (string-append "line 1" (string #\newline))) #t))
			    (let ((line2 (read-line p #t)))
			      (test (string=? line2 (string-append "line 2" (string #\newline))) #t))
			    (let ((line3 (read-line p #t)))
			      (test (string=? line3 "line 3") #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t)))))

(let ((res #f)) 
  (let ((this-file (open-output-string))) 
    (format this-file "this ~A ~C test ~D" "is" #\a 3)
    (set! res (get-output-string this-file))
    (close-output-port this-file))
  (if (not (string=? res "this is a test 3"))
      (begin 
	(display "open-output-string + format ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(test (call/cc (lambda (return) (let ((val (format #f "line 1~%line 2~%line 3")))
				  (call-with-input-string val
							  (lambda (p) (return "oops"))))))
      "oops")

(format #t "format #t: ~D" 1)
(format (current-output-port) " output-port: ~D! (this is testing output ports)~%" 2)
;; for float formats, assume s7 for now -- use our-pi and most-positive-fixnum
;; (format with 18 digits is enough to tell what s7_Double is via built-in pi)

;; from slib/formatst.scm
(test (string=? (format #f "abc") "abc") #t)
(test (string=? (format #f "~a" 10) "10") #t)
(test (string=? (format #f "~a" -1.2) "-1.2") #t)
(test (string=? (format #f "~a" 'a) "a") #t)
(test (string=? (format #f "~a" #t) "#t") #t)
(test (string=? (format #f "~a" #f) "#f") #t)
(test (string=? (format #f "~a" "abc") "abc") #t)
(test (string=? (format #f "~a" '#(1 2 3)) "#(1 2 3)") #t)
(test (string=? (format #f "~a" '()) "()") #t)
(test (string=? (format #f "~a" '(a)) "(a)") #t)
(test (string=? (format #f "~a" '(a b)) "(a b)") #t)
(test (string=? (format #f "~a" '(a (b c) d)) "(a (b c) d)") #t)
(test (string=? (format #f "~a" '(a . b)) "(a . b)") #t)
(test (string=? (format #f "~a ~a" 10 20) "10 20") #t)
(test (string=? (format #f "~a abc ~a def" 10 20) "10 abc 20 def") #t)
(test (string=? (format #f "~d" 100) "100") #t)
(test (string=? (format #f "~x" 100) "64") #t)
(test (string=? (format #f "~o" 100) "144") #t)
(test (string=? (format #f "~b" 100) "1100100") #t)
(test (string=? (format #f "~10d" 100) "       100") #t)
(test (string=? (format #f "~10,'*d" 100) "*******100") #t)
(test (string=? (format #f "~c" #\a) "a") #t)
(test (string=? (format #f "~~~~") "~~") #t)
(test (string=? (format #f "~s" "abc") "\"abc\"") #t)
(test (string=? (format #f "~s" "abc \\ abc") "\"abc \\\\ abc\"") #t)
(test (string=? (format #f "~a" "abc \\ abc") "abc \\ abc") #t)
(test (string=? (format #f "~s" "abc \" abc") "\"abc \\\" abc\"") #t)
(test (string=? (format #f "~a" "abc \" abc") "abc \" abc") #t)
(test (string=? (format #f "~s" #\space) "#\\space") #t)
(test (string=? (format #f "~s" #\newline) "#\\newline") #t)
(test (string=? (format #f "~s" #\a) "#\\a") #t)
(test (string=? (format #f "~a" '(a "b" c)) "(a \"b\" c)") #t)
(test (string=? (format #f "abc~
         123") "abc123") #t)
(test (string=? (format #f "abc~
123") "abc123") #t)
(test (string=? (format #f "abc~
") "abc") #t)
(test (string=? (format #f "~{ ~a ~}" '(a b c)) " a  b  c ") #t)
(test (string=? (format #f "~{ ~a ~}" '()) "") #t)
(test (string=? (format #f "~{ ~a,~a ~}" '(a 1 b 2 c 3)) " a,1  b,2  c,3 ") #t)
(test (string=? (format #f "abc ~^ xyz") "abc ") #t)
(test (format (values #f "~A ~D" 1 2)) "1 2")

(test (string=? (format #f "~B" 123) "1111011") #t)
(test (string=? (format #f "~B" 123/25) "1111011/11001") #t)
(test (string=? (format #f "~B" 123.25) "1111011.01") #t)
(test (string=? (format #f "~B" 123+i) "1111011.0+1.0i") #t)

(test (string=? (format #f "~D" 123) "123") #t)
(test (string=? (format #f "~D" 123/25) "123/25") #t)

(test (string=? (format #f "~O" 123) "173") #t)
(test (string=? (format #f "~O" 123/25) "173/31") #t)
(test (string=? (format #f "~O" 123.25) "173.2") #t)
(test (string=? (format #f "~O" 123+i) "173.0+1.0i") #t)

(test (string=? (format #f "~X" 123) "7b") #t)
(test (string=? (format #f "~X" 123/25) "7b/19") #t)
(test (string=? (format #f "~X" 123.25) "7b.4") #t)
(test (string=? (format #f "~X" 123+i) "7b.0+1.0i") #t)

(for-each
 (lambda (arg)
   (test (format #f "~F" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~D" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~X" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~C" arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f arg 123) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t (if #f #f) (lambda (a) (+ a 1))))

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (display 1 p)
    (write 2 p)
    (write-char #\3 p)
    (format p "~D" 4)
    (write-byte (char->integer #\5) p)
    (call-with-output-file "tmp2.r5rs"
      (lambda (p)
	(display 6 p)
	(write 7 p)
	(write-char #\8 p)
	(format p "~D" 9)
	(write-byte (char->integer #\0) p)
	(newline p)))
    (call-with-input-file "tmp2.r5rs"
      (lambda (pin)
	(display (read-line pin) p)))
    (newline p)))

(test (call-with-input-file "tmp1.r5rs"
	(lambda (p)
	  (read-line p)))
      "1234567890")

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (format p "12345~%")
    (format p "67890~%")))

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (test (read-char p) #\1)
    (test (read-byte p) (char->integer #\2))
    (test (peek-char p) #\3)
    (test (char-ready? p) #t)
    (test (read-line p) "345")
    (test (read-line p) "67890")))

(let ((op1 (set-current-output-port (open-output-file "tmp1.r5rs"))))
  (display 1)
  (write 2)
  (write-char #\3)
  (format #t "~D" 4) ; #t -> output port
  (write-byte (char->integer #\5))
  (let ((op2 (set-current-output-port (open-output-file "tmp2.r5rs"))))
    (display 6)
    (write 7)
    (write-char #\8)
    (format #t "~D" 9)
    (write-byte (char->integer #\0))
    (newline)
    (close-output-port (current-output-port))
    (set-current-output-port op2)
    (let ((ip1 (set-current-input-port (open-input-file "tmp2.r5rs"))))
      (display (read-line))
      (close-input-port (current-input-port))
      (set-current-input-port ip1))
    (newline)
    (close-output-port (current-output-port))
    (set-current-output-port op1)))

(test (call-with-input-file "tmp1.r5rs"
	(lambda (p)
	  (read-line p)))
      "1234567890")

(for-each 
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg display) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file))

(for-each 
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list open-output-file open-input-file 
       open-input-string))

(for-each
 (lambda (op)
   (for-each 
    (lambda (arg)
      (test (op "hi" arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t (if #f #f) (lambda (a) (+ a 1)))))
 (list write display write-byte newline write-char 
       read read-char read-byte peek-char char-ready? read-line))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs (if #f #f) (lambda (a) (+ a 1)))))
 (list set-current-input-port set-current-error-port set-current-output-port close-input-port close-output-port))

(let ((hi (open-output-string)))
  (close-output-port hi)
  (test (get-output-string hi) 'error))

;; since read of closed port will generate garbage, it needs to be an error,
;;   so I guess write of closed port should also be an error

(let ((hi (open-output-string)))
  (close-output-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 set-current-output-port
	 set-current-input-port
	 set-current-error-port
	 newline)))

(let ((hi (open-input-string "hiho")))
  (close-input-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list read read-char read-byte peek-char char-ready? read-line 
	 port-filename port-line-number
	 set-current-output-port
	 set-current-input-port
	 set-current-error-port
	 )))
  
(test (close-output-port (open-input-string "hiho")) 'error)
(test (close-input-port (open-output-string)) 'error)

(let* ((new-error-port (open-output-string))
       (old-error-port (set-current-error-port new-error-port)))
  (catch #t
	 (lambda ()
	   (format #f "~R" 123))
	 (lambda args
	   (format (current-error-port) "oops")))
  (let ((str (get-output-string new-error-port)))
    (set-current-error-port old-error-port)
    (test str "oops")))


(let ((hi (open-input-string "hiho")))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'input-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 newline))
  (close-input-port hi))

(let ((hi (open-output-string)))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'output-port))
   (list read read-char read-byte peek-char char-ready? read-line))
  (close-output-port hi))

(test (output-port? (current-error-port)) #t)
(test (and (not (null? (current-error-port))) (input-port? (current-error-port))) #f)

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-byte i p))))

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (let ((b (read-byte p)))
	(if (not (= b i))
	    (format #t "read-byte got ~A, expected ~A~%" b i))))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format #t "read-byte at end: ~A~%" eof)))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format #t "read-byte at end: ~A~%" eof)))))

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-char (integer->char i) p))))

(define our-eof #f)

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (let ((b (read-char p)))
	(if (or (not (char? b))
		(not (char=? b (integer->char i))))
	    (format #t "read-char got ~A, expected ~A (~D: char? ~A)~%" b (integer->char i) i (char? (integer->char i))))))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format #t "read-char at end: ~A~%" eof))
      (set! our-eof eof))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format #t "read-char again at end: ~A~%" eof)))))

(test (eof-object? (integer->char 255)) #f)
(test (eof-object? our-eof) #t)
(test (char->integer our-eof) 'error)
(test (char? our-eof) #f)
(test (eof-object? ((lambda () our-eof))) #t)

(test (open-input-file "[*not-a-file!*]-") 'error)
(test (call-with-input-file "[*not-a-file!*]-" (lambda (p) p)) 'error)
(test (with-input-from-file "[*not-a-file!*]-" (lambda () #f)) 'error)

(test (open-input-file "") 'error)
(test (call-with-input-file "" (lambda (p) p)) 'error)
(test (with-input-from-file "" (lambda () #f)) 'error)

;(test (open-output-file "/bad-dir/badness/[*not-a-file!*]-") 'error)
;(test (call-with-output-file "/bad-dir/badness/[*not-a-file!*]-" (lambda (p) p)) 'error)
;(test (with-output-to-file "/bad-dir/badness/[*not-a-file!*]-" (lambda () #f)) 'error)

(with-output-to-file "tmp.r5rs"
  (lambda ()
    (write-char #\a)
    (with-output-to-file "tmp1.r5rs"
      (lambda ()
	(format #t "~C" #\b)
	(with-output-to-file "tmp2.r5rs"
	  (lambda ()
	    (display #\c)))
	(display (with-input-from-file "tmp2.r5rs"
		   (lambda ()
		     (read-char))))))
    (with-input-from-file "tmp1.r5rs"
      (lambda ()
	(write-byte (read-byte))
	(write-char (read-char))))))

(with-input-from-file "tmp.r5rs"
  (lambda ()
    (test (read-line) "abc")))

(with-input-from-file "tmp.r5rs" ; this assumes tmp.r5rs has "abc" as above
  (lambda ()
    (test (read-char) #\a)
    (test (eval-string "(+ 1 2)") 3)
    (test (read-char) #\b)
    (with-input-from-string "(+ 3 4)"
      (lambda ()
	(test (read) '(+ 3 4))))
    (test (read-char) #\c)))

(test (eval-string (object->string (with-input-from-string "(+ 1 2)" (lambda () (read))))) 3)
(test (eval (eval-string "(with-input-from-string \"(+ 1 2)\" (lambda () (read)))")) 3)
(test (eval-string "(eval (with-input-from-string \"(+ 1 2)\" (lambda () (read))))") 3)
(test (eval-string (object->string (eval-string (format #f "(+ 1 2)")))) 3)

;; (eval-string "(eval-string ...)") is not what it appears to be -- the outer call
;;    still sees the full string when it evaluates, not the string that results from
;;    the inner call.

(test (let ((name '+))
	(let ((+ *))	
	  (eval (list name 2 3))))
      6)
(test (let ((name +))
	(let ((+ *))	
	  (eval (list name 2 3))))
      5)
;; why is this considered confusing?  It has nothing to do with eval!

(test (let ((call/cc (lambda (x)
		       (let ((c (call/cc x))) c))))
	(call/cc (lambda (r) (r 1))))
      1)

(for-each
 (lambda (arg)
   (test
    (with-input-from-string (format #f "~A" arg)
      (lambda ()
	(read)))
    arg))
 (list 1 3/4 '(1 2) #(1 2) :hi #f #t))

(num-test (with-input-from-string "3.14" (lambda () (read))) 3.14)
(num-test (with-input-from-string "3.14+2i" (lambda () (read))) 3.14+2i)
(num-test (with-input-from-string "#x2.1" (lambda () (read))) 2.0625)
(test (with-input-from-string "'hi" (lambda () (read))) ''hi)
(test (with-input-from-string "'(1 . 2)" (lambda () (read))) ''(1 . 2))


(test
 (let ((cin #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-input-from-string "123"
	      (lambda ()
		(set! cin (current-input-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cin cerr))
 "<port string input (closed)> #t")

(test
 (let ((cout #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-output-to-string
	      (lambda ()
		(set! cout (current-output-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cout cerr))
 "<port string output (closed)> #t")

(call-with-output-file "tmp1.r5rs"
  (lambda (p)
    (display "1" p)
    (newline p)
    (newline p)
    (display "2345" p)
    (newline p)))

(call-with-input-file "tmp1.r5rs"
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (eof-object? (read-line p)) #t)))

(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (port-line-number arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '() (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format #t "(~A) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port 
       close-input-port close-output-port
       write display write-byte write-char format                     ; newline
       ;read read-char read-byte peek-char char-ready? read-line      ; these can default to current input
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file 
       open-input-string))

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op 1 2 3 4 5)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format #t "(~A 1 2 3 4 5) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port 
       close-input-port close-output-port
       write display write-byte write-char format newline
       read read-char read-byte peek-char char-ready? read-line
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file 
       open-input-string))


(test (string=? (object->string 32) "32") #t)
(test (string=? (object->string 32.5) "32.5") #t)
(test (string=? (object->string 32/5) "32/5") #t)
(test (string=? (object->string "hiho") "\"hiho\"") #t)
(test (string=? (object->string 'symb) "symb") #t)
(test (string=? (object->string (list 1 2 3)) "(1 2 3)") #t)
(test (string=? (object->string (cons 1 2)) "(1 . 2)") #t)
(test (string=? (object->string '#(1 2 3)) "#(1 2 3)") #t)
(test (string=? (object->string +) "+") #t)
(test (object->string (object->string (object->string "123"))) "\"\\\"\\\\\\\"123\\\\\\\"\\\"\"")
(test (object->string #<eof>) "#<eof>")
(test (object->string (if #f #f)) "#<unspecified>")
(test (object->string #f) "#f")
(test (object->string #t) "#t")
(test (object->string '()) "()")
(test (object->string #()) "#()")
(test (object->string "") "\"\"")

(test (object->string) 'error)
(test (object->string 1 2) 'error)
(test (object->string abs) "abs")


;;; (string-set! (with-input-from-string "\"1234\"" (lambda () (read))) 1 #\a)





;;; --------------------------------------------------------------------------------
;;; CONTROL OPS
;;; --------------------------------------------------------------------------------

(define control-ops (list lambda define quote if begin set! let let* letrec cond case and or do
			  call/cc eval apply for-each map values call-with-values dynamic-wind))
(for-each
 (lambda (op)
   (if (not (eq? op op))
       (format #t "~A not eq? to itself?~%" op)))
 control-ops)

(for-each
 (lambda (op)
   (if (not (eqv? op op))
       (format #t "~A not eqv? to itself?~%" op)))
 control-ops)

(for-each
 (lambda (op)
   (if (not (equal? op op))
       (format #t "~A not equal? to itself?~%" op)))
 control-ops)

(define question-ops (list boolean? eof-object? string?
		           number? integer? real? rational? complex? char?
			   list? vector? pair? null?))

(for-each
 (lambda (ques)
   (for-each
    (lambda (op)
      (if (ques op)
	  (format #t "(~A ~A) returned #t?~%" ques op)))
    control-ops))
 question-ops)

(let ((unspecified (if #f #f)))
  (for-each
   (lambda (op)
     (if (op unspecified)
	 (format #t "(~A #<unspecified>) returned #t?~%" op)))
   question-ops))

(for-each 
 (lambda (s)
   (if (not (symbol? s))
       (format #t "(symbol? ~A returned #f?~%" s)))
 '(+ - ... !.. $.+ %.- &.! *.: /:. <-. =. >. ?. ~. _. ^.))



;;; -------- if --------

(test ((if #f + *) 3 4) 12)
(test (if (> 3 2) 'yes 'no) 'yes)
(test (if (> 2 3) 'yes 'no) 'no)
(test (if (> 3 2) (- 3 2) (+ 3 2)) 1)
(test (if (> 3 2) 1) 1)
(test (if '() 1 2) 1)
(test (if 't 1 2) 1)
(test (if #t 1 2) 1)
(test (if '#() 1 2) 1)
(test (if 1 2 3) 2)
(test (if 0 2 3) 2)
(test (if (list) 2 3) 2)
(test (if "" 2 3) 2)
(test (eq? (if #f #f) (if #f #f)) #t) ; I assume there's only one #<unspecified>!
(test (if . (1 2)) 2)
(test (if (if #f #f) #f #t) #f)

(test (let ((a #t) (b #f) (c #t) (d #f)) (if (if (if (if d d c) d b) d a) 'a 'd)) 'a)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if a (if b (if c (if d d c) c) 'b) 'a)) 'b)
					;(test (let ((a #t) (b #f) (c #t) (d #f)) (((if a if 'gad) c if 'gad) (not d) 'a 'gad)) 'a)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if b (if a (if d 'gad) 'gad) (if d 'gad 'a))) 'a)

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if a arg 'gad) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if (not a) 'gad arg) arg))
   (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(test (let ((ctr 0) (a #t)) (if a (let ((b ctr)) (set! ctr (+ ctr 1)) (list b ctr)) (let ((c ctr)) (set! ctr (+ ctr 100)) (list c ctr)))) (list 0 1))

(test (if if if if) if)
(test (((if if if) if if) if if 'gad) if)
(test (if if (if if if) if) if)
(test (let ((car if)) (car #t 0 1)) 0)
(test ((car (list if)) #t 0 1) 0)
(test (symbol->string 'if) "if")
(test (if (and if (if if if)) if 'gad) if)
(test (let ((if #t)) (or if)) #t)
;(test (let ((if +)) (if 1 2 3)) 6)
; this is another of the "syntax" as car choices
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (= ctr 1)) 0 1)) 0)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (if (= ctr 1) (> 3 2) (< 3 2))) 0 1)) 0)
(test (        if (> 3 2) 1 2) 1)
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assoc if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assv if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assq if alist)) (list if 3))
(test (let ((alist (list map car if do))) (member if alist)) (list if do))
(test (let ((alist (list map car if do))) (memv if alist)) (list if do))
(test (let ((alist (list map car if do))) (memq if alist)) (list if do))
(test ((vector-ref (vector if) 0) #t 1 2) 1)
(test ((vector-ref (make-vector 1 if) 0) #t 1 2) 1)
(test ((if #t + -) 3 4) 7)
(test (list (if 0 1 2)) (list 1))
(test ((car (list if map)) #f 1 2) 2)
(test (let ((ctr 0)) (if (= ctr 0) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 2 3)) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 4 5)))) 2)
(test (let ((x (cons 1 2))) (set-cdr! x x) (if x 1 2)) 1)
(test (let ((ctr 0)) (if (let ((ctr 123)) (set! ctr (+ ctr 1)) (= ctr 124)) (let () (set! ctr (+ ctr 100)) ctr) (let () (set! ctr (+ ctr 1000)) ctr)) ctr) 100)
(test (if (let ((if 3)) (> 2 if)) 4 5) 5)

(test (let ((ctr 0)) (call/cc (lambda (exit) (if (> 3 2) (let () (exit ctr) (set! ctr 100) ctr) #f)))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (< 3 2) #f (let () (exit ctr) (set! ctr 100) ctr))))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (let () (exit ctr) (set! ctr 100) ctr) 123 321)))) 0)
(test (let ((ctr 0)) (if (> 3 2) (call/cc (lambda (exit) (set! ctr (+ ctr 1)) (exit ctr))) #f) ctr) 1)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4)))
				      (set! ctr (- ctr 5)))
				  (set! ctr (- ctr 6)))
			      (set! ctr (- ctr 7)))
			  (set! ctr (- ctr 8)))
		      (set! ctr (- ctr 9)))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      934)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4))))))))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      969)

(test (if #f) 'error)
(test (if (< 2 3)) 'error)
(test (if #f 1 2 3) 'error)
(test (if 1 2 3 4) 'error)
(test (if #f 1 else 2) 'error)
(test (if) 'error)
(test ('+ '1 '2) 'error)
(test (if 1 . 2) 'error)
(test (if 1 2 . 3) 'error)
(test (if . 1) 'error)
(test (if _no_var_ 1) 'error)





;;; -------- quote --------

(test (quote a) 'a)
(test 'a (quote a))
(test '1 1)
(test '1/4 1/4)
(test '(+ 2 3) '(+ 2 3))
(test '"hi" "hi")
(test '#\a #\a)
(test '#f #f)
(test '#t #t)
(test '#b1 1)
(test (= 1/2 '#e#b1e-1) #t)
(test '() '())
(test (+ '1 '2) 3)
(test (+ '1 '2) '3)
(test (+ ' 1 '   2) '    3)
(test (char? '#\a) #t)
(test (string? '"hi") #t)
(test (boolean? '#t) #t)
(test (if '#f 2 3) 3)
(test (if '#t 2 3) 2)
(test (vector? '#()) #t)
(test (char? (quote #\a)) #t)
(test (string? (quote "hi")) #t)
(test (boolean? (quote #t)) #t)
(test (if (quote #f) 2 3) 3)
(test (if (quote #t) 2 3) 2)
(test (vector? (quote #())) #t)
(test (+ (quote 1) (quote 2)) (quote 3))
(test (list? (quote ())) #t)
(test (pair? (quote (1 . 2))) #t)
(test (+ '1.0 '2.0) 3.0)
(test (+ '1/2 '3/2) 2)
(test (+ '1.0+1.0i '-2.0) -1.0+1.0i)
(test (let ((hi 2)) (equal? hi 'hi)) #f)
(test ''1 (quote (quote 1)))
(test ''a (quote (quote a)))
(test (symbol? '#f) #f)
(test ''quote (quote (quote quote)))
(test (+ (cadr ''3) (cadadr '''4) (cadr (cadr (cadr ''''5)))) 12)
(test (eq? lambda 'lambda) #t)

(test (eq? '() ()) #t) ; not sure about this -- Gauche, SCM, stklos say #t; Guile says error; clisp, cmucl, and sbcl say T

(test (let ((quote 1)) (+ quote 1)) 2)
(test ((lambda (quote) (+ quote 1)) 2) 3)
(test ((lambda (quote . args) (list quote args)) 1 2 3) '(1 (2 3)))

(test (quote . -1) 'error)
(test (quote 1 1) 'error)
(test (quote . 1) 'error)
(test (quote . (1 2)) 'error)
(test (quote 1 . 2) 'error)
(test (symbol? '1'1) #t) 



;;; -------- for-each --------

(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(test (let ((ctr 0) (v (make-vector 5))) (for-each (lambda (i) (vector-set! v ctr (* i i)) (set! ctr (+ ctr 1))) '(0 1 2 3 4)) v) '#(0 1 4 9 16))
(for-each (lambda (x) (display "for-each should not have called this")) '())
(test (let ((ctr 0)) (for-each (lambda (x y) (if (= x y) (set! ctr (+ ctr 1)))) '(1 2 3 4 5 6) '(2 3 3 4 7 6)) ctr) 3)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5)) ctr) 15)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) '(5)) ctr) 9)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '() '() '()) ctr) 0)
(test (let () (for-each abs '(1 2)) 1) 1)
(test (let ((ctr 0)) (for-each (lambda (a) (for-each (lambda (b) (set! ctr (+ ctr 1))) '(0 1))) '(2 3 4)) ctr) 6)
(test (let ((sum 0)) (for-each (lambda args (set! sum (+ sum (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0)) sum) 6)
(test (let () (for-each + '(0 1 2) '(2 1 0)) 0) 0)
(test (let () () ()) '())
(test (for-each + ()) #<unspecified>)
(test (let ((d 0))
	(for-each (let ((a 0))
		    (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		    (lambda (c) (set! d (+ d c a))))
		  (list 3 4 5))
	d)
      21)
(test (let ((d 0))
	(for-each (lambda (c)
		    (let ((a 0))
		      (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		      (set! d (+ d a c))))
		  (list 3 4 5))
	d)
      21)

(test (let ((ctr 0)) 
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (> a 3) (exit a)) 
				  (set! ctr (+ ctr 1))) 
				(list 0 1 2 3 4 5)))))) 
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (for-each 
	  (lambda (x) 
	    (if (negative? x) (exit x)))
	  '(54 0 37 -3 245 19))
	 #t))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst '()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (and (not cont) (= a 2))
				      (exit a)) 
				  (if (and cont (= a 5)) 
				      (exit a))
				  (call/cc (lambda (c) (set! cont c)))
				  (set! lst (cons ctr lst))
				  (set! ctr (+ ctr 1)))
				(list 0 1 2 3 4 5)))))) 
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(test (let ((lst '())) 
	(for-each (lambda (a) (set! lst (cons a lst))) 
		  (let ((lst '())) 
		    (for-each (lambda (b) (set! lst (cons b lst))) 
			      (list 1 2 3)) 
		    lst)) 
	lst) 
      (list 1 2 3))

;;; this is an infinite loop?
					; (let ((cont #f)) (call/cc (lambda (x) (set! cont x))) (for-each cont (list 1 2 3)))
(test (call/cc (lambda (x) (for-each x (list 1 2 3)))) 1) ; map also gives 1 ... perhaps not actually legal?

(test (let ((ctr 0))
	(for-each 
	 (lambda (x)
	   (for-each
	    (lambda (x y)
	      (for-each 
	       (lambda (x y z)
		 (set! ctr (+ x y z)))
	       (list x (+ x 1))
	       (list y (+ y 2))
	       (list (+ x y) (- x y))))
	    (list (+ x 3) (+ x 4) (+ x 5))
	    (list (- x 3) (- x 4) (- x 5))))
	 (list 1 2 3 4 5))
	ctr)
      23)

(for-each
 (lambda (a)
   (if (not (string=? a "hi"))
       (format #t "yow: ~S" a)))
 (list "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi"))


;; now some mixed cases
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n m))) (list 1 2) (vector 3 4)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n m))) (vector 1 2) (list 3 4)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m))) (vector 1 2) (list 3 4) (vector 5 6)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m p) (if (char=? p #\x) (set! sum (+ sum n m)))) (vector 1 2 3) (list 3 4 5) "xax") sum) 12)

(test (let* ((x (list (list 1 2 3))) (y (apply for-each abs x))) x) '((1 2 3)))

(test (for-each (lambda (x) (display "for-each should not have called this"))) 'error)
;(test (for-each (lambda () 1) '()) 'error)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) '()) ctr) 0)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5 6)) ctr) 15)
(test (for-each (lambda (a b) (+ a b)) (list 1)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list 2) (list 3)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list 1 2)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2 3)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list 1 2)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (cons 1 2)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (cons 1 2) (list 1 2)) #<unspecified>)
(test (for-each (lambda (a) (+ a 1)) (list 1) (list 2)) 'error)
(test (for-each (lambda (a) (+ a 1)) #\a) 'error)
(test (for-each (lambda (a) (+ a 1)) (cons 1 2)) #<unspecified>)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2)) sum) 'error)
(test (for-each (lambda (a) a) '(1 2 . 3)) #<unspecified>)
(for-each
 (lambda (arg)
   (test (for-each arg (list 1)) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
(for-each
 (lambda (arg)
   (test (for-each (lambda (n m) n) (list 1) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(for-each
 (lambda (arg)
   (test (for-each (lambda (a) a) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(test (for-each) 'error)
(test (for-each #t) 'error)
(test (for-each map #t) 'error)

(test (for-each abs '() abs) #<unspecified>)
(test (for-each abs '(1) '#(1)) 'error)
(test (let ((vals '())) (for-each for-each (list (lambda (a) (set! vals (cons (abs a) vals)))) (list (list -1 -2))) vals) '(2 1))
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) "a") c) #\a)
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) "") c) #f)
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) (string #\null)) c) #\null)

(test (let ((L (list 1 2 3 4 5)) (sum 0)) (for-each (lambda (x) (set-cdr! (cddr L) 5) (set! sum (+ sum x))) L) sum) 6)
;;; map (below) has more tests along this line




;;; -------- map --------

(test (map cadr '((a b) (d e) (g h))) '(b e h))
(test (map (lambda (n) (expt n n)) '(1 2 3 4 5)) '(1 4 27 256 3125))
(test (map + '(1 2 3) '(4 5 6)) '(5 7 9))

(test (apply vector (map (lambda (i) (* i i)) '(0 1 2 3 4))) '#(0 1 4 9 16))
(map (lambda (x) (display "map should not have called this")) '())
(test (let ((ctr 0)) (map (lambda (x y) (if (= x y) (set! ctr (+ ctr 1))) ctr) '(1 2 3 4 5 6) '(2 3 3 4 7 6))) (list 0 0 1 2 2 3))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(0 1) '(2 3) '(4 5))) (list 6 15))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) '(5))) (list 9))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '() '() '())) '())
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2)) (list 2 4))
(test (map abs '(1 -2)) (list 1 2))
(test (map + '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) (list 24 24 24))
(test (map (lambda (a) (cons a (map (lambda (b) (+ b 1)) (list 0 1 2)))) (list 3 4 5)) '((3 1 2 3) (4 1 2 3) (5 1 2 3)))
(test (map (lambda (a) (+ a 1)) (map (lambda (b) (+ b 1)) (map (lambda (c) (+ c 1)) (list 0 1 2)))) '(3 4 5))
(test (map (lambda args (apply + args)) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5)) '(3 5 7))
(test (map + () ()) ())
(test (map + (#(#() #()) 1)) '())
(test (map + #(1) #(1) #(1)) '(3))
(test (map list '(a b c)) '((a) (b) (c)))
(test (map (lambda (a b) (- a b)) (list 1 2) (vector 3 4)) '(-2 -2))
(test (map (lambda (a b c) (if (char=? a #\a) (+ b c) (- b c))) "axa" (list 1 2 3) (vector 4 5 6)) '(5 -3 9))

(test (let* ((x (list (list 1 2 3))) (y (apply map abs x))) (list x y)) '(((1 2 3)) (1 2 3)))
(test (let* ((x (quote ((1 2) (3 4)))) (y (apply map ash x))) (list x y)) '(((1 2) (3 4)) (8 32)))
(test (let* ((x (quote ((1 2 3) (4 5 6) (7 8 9)))) (y (apply map + x))) (list x y)) '(((1 2 3) (4 5 6) (7 8 9)) (12 15 18)))
(test (map * (map + '(1 2 3) '(4 5 6)) '(1 2 3)) '(5 14 27))
(test (apply map * (apply map + '(1 2 3) '((4 5 6))) '((1 2 3))) '(5 14 27))
(test (let* ((x (lambda () '(1 2 3))) (y (apply map - (list (x))))) (x)) '(1 2 3))

(test (let ((d 0))
	(map (let ((a 0))
	       (map (lambda (b) (set! a (+ a b))) (list 1 2))
	       (lambda (c) (set! d (+ d c a)) d))
	     (list 3 4 5)))
      (list 6 13 21))
(test (let ((d 0))
	(map (lambda (c)
	       (let ((a 0))
		 (map (lambda (b) (set! a (+ a b))) (list 1 2))
		 (set! d (+ d a c))
		 d))
	     (list 3 4 5)))
      (list 6 13 21))

(test (let ((ctr 0))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (> a 3) (exit a)) 
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (map 
	  (lambda (x) 
	    (if (negative? x) (exit x))
	    x)
	  '(54 0 37 -3 245 19))))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst '()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (and (not cont) (= a 2))
				 (exit a)) 
			     (if (and cont (= a 5)) 
				 (exit a))
			     (call/cc (lambda (c) (set! cont c)))
			     (set! lst (cons ctr lst))
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(test (map (lambda (a) a) (map (lambda (b) b) (list 1 2 3))) (list 1 2 3))
(test (map cons '(a b c) '(() () ())) '((a) (b) (c)))

(test (map list "hi") '((#\h) (#\i)))
(test (map string "hi") '("h" "i"))
(test (map vector "hi") '(#(#\h) #(#\i)))
(test (map char-upcase "hi") '(#\H #\I))
(test (map append #(#() #())) '(#() #()))

(test (map abs '() abs) '())
(test (map (lambda (x) (display "map should not have called this"))) 'error)
;(test (map (lambda () 1) '()) 'error)
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) '())) '())
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5 6))) '(6 15))

(test (map (lambda (a b) (+ a b)) (list 1)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list)) '())
(test (map (lambda (a b) (+ a b)) (list 1) (list 2)) (list 3))
(test (map (lambda (a b) (+ a b)) (list 1)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list 2) (list 3)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list 1 2)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2 3)) '(2 4))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list)) '())
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list 1 2)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1 2) (cons 1 2)) '(2))

(test (map (lambda (a) (+ a 1)) (list 1) (list 2)) 'error)
(test (map (lambda (a) (+ a 1)) #\a) 'error)
(test (map (lambda (a) (+ a 1)) (cons 1 2)) '(2))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2)) 'error)
(test (map (lambda (a) a) '(1 2 . 3)) '(1 2))
(test (map) 'error)
(test (map #t) 'error)
(test (map set-cdr! '(1 2 3)) 'error)
(test (map (lambda (a b) (set-cdr! a b) b) '((1) (2) (3)) '(4 5 6)) '(4 5 6))
(test (let ((str "0123")) (set! (str 2) #\null) (map append str)) '(#\0 #\1 #\null #\3))

(for-each
 (lambda (arg)
   (test (map arg (list 1)) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
(for-each
 (lambda (arg)
   (test (map (lambda (n m) n) (list 1) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(for-each
 (lambda (arg)
   (test (map (lambda (a) a) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(test (map (lambda (a1 a2 a3 a4 a5 a6 a7 a8 a9 a10)
	     (max a1 a2 a3 a4 a5 a6 a7 a8 a9 a10))
	   (list 6 7 8 9 10)
	   (list 21 22 23 24 25)
	   (list 16 17 18 19 20)
	   (list 11 12 13 14 15)
	   (list 26 27 28 29 30)
	   (list 1 2 3 4 5)
	   (list 36 37 38 39 40)
	   (list 41 42 43 44 45)
	   (list 46 47 48 49 50)
	   (list 31 32 33 34 35))
      (list 46 47 48 49 50))
  
(test (map map (list abs) (list (list -1))) '((1)))
(test (map map (list map) (list (list abs)) (list (list (list -1)))) '(((1))))
(test (map map (list map) (list (list map)) (list (list (list abs))) (list (list (list (list -1 -3))))) '((((1 3)))))
(test (let () (define (mrec a b) (if (<= b 0) (list a) (map mrec (list a) (list (- b 1))))) (mrec (list 1 2) 5)) '(((((((1 2))))))))
(test (map append '(3/4)) '(3/4))
(test (map list '(1.5)) '((1.5)))
(test (map vector '("hi")) '(#("hi")))
(test (map object->string '(:hi (1 2) (()))) '(":hi" "(1 2)" "(())"))
(test (map map (list for-each) (list (list abs)) (list (list (list 1 2 3)))) '((#<unspecified>)))
(test (map map (list vector) '((#(1 #\a (3))))) '((#(#(1 #\a (3))))))
(test (apply map map (list cdr) '((((1 2) (3 4 5))))) '(((2) (4 5))))
(test (apply map map (list char-upcase) '(("hi"))) '((#\H #\I)))
(test (apply map map (list *) '(((1 2)) ((3 4 5)))) '((3 8))) ; (* 1 3) (* 2 4)
(test (map apply (list map) (list map) (list (list *)) '((((1 2)) ((3 4 5))))) '(((3 8))))
(test (map map (list magnitude) '((1 . 2))) '((1))) ; magnitude is called once with arg 1
(test (map magnitude '(1 . 2)) '(1))
(test (map call/cc (list (lambda (r1) 1) (lambda (r2) (r2 2 3)) (lambda (r3) (values 4 5)))) '(1 2 3 4 5))
(test (map call/cc (list number? continuation?)) '(#f #t))

;; from scheme working group 
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! (cddr L) 5) x) L)) '(1 2 3))
(test (let ((L (list 1 2))) (map (lambda (x) (set! (cdr (cdr L)) L) x) L)) '(1 2))
(test (let ((L (list 1 2))) (object->string (map (lambda (x) (set! (car (cdr L)) L) x) L))) "(1 #1=(1 #1#))")
;;;(test (let ((L (list 1 2))) (map (lambda (x) (set-cdr! L L) x) L)) '(1 2)) ;?? this depends on when we cdr? infinite loop in Guile
;;;(let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! L '()) x) L)) ; another similar case -- s7 doesn't notice what happened
;;;  does that mean a GC during this map would leave us accessing freed memory? 
;;;  I think not because the original list is held by map (eval) locals that are protected
;;;  we simply stepped on something after looking at it, similar to:
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-car! L 123) x) L)) '(1 2 3 4 5))
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! (cddr L) (list 6 7 8)) x) L)) '(1 2 3 6 7))
;;; we could do something similar with strings:
(test (let ((S "12345")) (map (lambda (x) (set! (S 2) #\null) x) S)) '(#\1 #\2 #\null #\4 #\5))
;;; (length S) is still 5 even with the embedded null
(test (let ((L (list 1 2 3))) (map (lambda (x) (set! L (list 6 7 8)) x) L)) '(1 2 3))
(test (let ((L1 (list 1 2 3)) (L2 (list 4 5 6 7))) (map (lambda (x1 x2) (set-cdr! (cdr L1) '()) (cons x1 x2)) L1 L2)) '((1 . 4) (2 . 5)))
(test (let ((L (list 1 2 3))) (map (lambda (x) (set-car! (cddr L) 32) x) L)) '(1 2 32))
;;; should these notice the increased length?:
(test (let ((L1 (list 1 2)) (L2 (list 6 7 8 9))) (map (lambda (x y) (set-cdr! (cdr L1) (list 10 11 12 13 14)) (cons x y)) L1 L2)) '((1 . 6) (2 . 7)))
(test (let ((L1 (list 1)) (L2 (list 6 7 8))) (map (lambda (x y) (set-cdr! L1 (list 10 11 12)) (cons x y)) L1 L2)) '((1 . 6)))
(test (let ((L1 (list 1 2))) (map (lambda (x) (set-cdr! (cdr L1) (list 10 11 12)) x) L1)) '(1 2))
;;; a similar case could be made from hash-tables
(test (let ((H (hash-table '(a . 3) '(b . 4)))) (map (lambda (x) (set! (H 'c) 32) (cdr x)) H)) '(3 4))
(test (let ((H (hash-table '(a . 3) '(b . 4)))) (map (lambda (x) (set! (H 'b) 32) (cdr x)) H)) '(3 32))

;; in that 1st example, the set-cdr! is not the problem (map supposedly can treat its args in any order),
;;   any set! will do:
(test (let ((x 0)) (map (lambda (y) (set! x (+ x y)) x) '(1 2 3 4))) '(1 3 6 10))

(test (map begin '(1 2 3)) 'error)
(let ((funcs (map (lambda (lst) (eval `(lambda ,@lst))) '((() #f) ((arg) (+ arg 1))))))
  (test ((car funcs)) #f)
  (test ((cadr funcs) 2) 3))




;;; --------- do --------

(test (do () (#t 1)) 1)
(for-each
 (lambda (arg)
   (test (do () (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (do ((i arg)) (#t i)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (do ((i 0 (+ i 1))) ((= i 3) #f)) #f)
(test (do ((i 0 (+ i 1))) ((= i 3) i)) 3)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) '#(0 1 2 3 4))
(test (let ((x '(1 3 5 7 9))) (do ((x x (cdr x)) (sum 0 (+ sum (car x))))  ((null? x) sum))) 25)
(test (do ((i 4 (- i 1)) (a 1 (* a i))) ((zero? i) a)) 24)
(test (do ((i 2 (+ i 1))) ((> i 0) 123)) 123)

					;(test (do () (() ()) ()) '()) ; ?? -- is '() the same as ()? -- scheme bboard sez not necessarily
(test (do () ('() '())) '())
(test (do () ('())) '())
(test (do () (())) '())

(test (let ((x 0) (y 0)) (set! y (do () (#t (set! x 32) 123))) (list x y)) (list 32 123))
(test (let ((i 32)) (do ((i 0 (+ i 1)) (j i (+ j 1))) ((> j 33) i))) 2)
(test (let ((i 0)) (do () ((> i 1)) (set! i (+ i 1))) i) 2)
(test (let ((i 0) (j 0)) (do ((k #\a)) (#t i) (set! i (char->integer k)) (set! j (+ j i)))) 0)
(test (let ((i 0) (j 0)) (do ((k #\a)) ((> i 1) j) (set! i (char->integer k)) (set! j (+ j i)))) (char->integer #\a))
(test (let ((x 0)) (do ((i 0 (+ i 2)) (j 1 (* j 2))) ((= i 4) x) (set! x (+ x i j)))) 5)
(test (let ((sum 0)) (do ((lst '(1 2 3 4) (cdr lst))) ((null? lst) sum) (set! sum (+ sum (car lst))))) 10)
(test (do ((i 0 (+ 1 i))) ((= i 4) (do ((i 0 (+ i 2))) ((= i 10) i)))) 10)
(test (let ((i 0)) (do ((i 1 (+ i 1))) ((= i 3) i))) 3)
(test (let ((j 0)) (do ((i 0 (+ i 1))) ((= i 3) (+ i j)) (do ((j 0 (+ j i 1))) ((> j 3) j)))) 3)
(test (let ((add1 (lambda (a) (+ a 1)))) (do ((i 0 (add1 i))) ((= i 10) (add1 i)))) 11)
(test (do ((i 0 (do ((j 0 (+ j 1))) ((= j i) (+ i 1))))) ((= i 3) i)) 3)
(test (do ((i 0 (do ((i 0 (+ i 1))) ((= i 3) i)))) ((= i 3) i)) 3)
(test (let ((i 123)) (do ((i 0 (+ i 1)) (j i (+ j i))) ((> j 200) i))) 13)
(test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (* i 10))) 11)
(test (do ((i 123) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 123 i) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 0 i)) (i i)) 0)
(test (do ((i 1 i)) (i i (+ i i) (+ i i i))) 3)
(test (do ((i 1)) (#t 1) 123) 1)
(test (do ((i 0 (+ i j)) (j 0 (+ j 1))) (#t 1)) 1)
(test (do ((i 0 j) (j 0 (+ j 1))) ((= j 3) i)) 2) ; uh, lessee... lexical scoping...
(test (do ((i 1 j) (j 0 k) (k 0 m) (m 0 (+ i j k))) ((> m 10) (list i j k m))) (list 4 5 8 11))
(test (do ((do 1 (+ do do))) ((> do 3) do)) 4)
(test (do ((do 1 do) (j do do)) (do do)) 1)
(test (do ((do do do)) (do do)) do)
(test (do ((do do do)) (do do do)) do) ; ok ok!
(test (let ((i 10) (j 11) (k 12)) (do ((i i j) (j j k) (k k m) (m (+ i j k) (+ i j k))) ((> m 100) (list i j k m)))) (list 33 56 78 122))
(test (do ((i 0 (let () (set! j 3) (+ i 1))) (j 0 (+ j 1))) ((= i 3) j)) 4)
(test (let ((i 0)) (do () ((= i 3) (* i 2)) (set! i (+ i 1)))) 6)
(num-test (do ((i 0 (- i 1))) ((= i -3) i)) -3)
(num-test (do ((i 1/2 (+ i 1/2))) ((> i 2) i)) 5/2)
(num-test (do ((i 0.0 (+ i 0.1))) ((>= i 0.9999) i)) 1.0)
(num-test (do ((i 0 (- i 1/2))) ((< i -2) i)) -5/2)
(num-test (do ((i 0+i (+ i 0+i))) ((> (magnitude i) 2) i)) 0+3i)
(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((> i 4) x) 
	  (set! x (+ x i))
	  (set! i (+ i 0.5))))
      4.5)
(test (do ((i 0 1)) ((> i 0) i)) 1)
(test (do ((i 1.0+i 3/4)) ((= i 3/4) i)) 3/4)
(test (do ((i 0 "hi")) ((not (number? i)) i)) "hi")
(test (do ((i "hi" 1)) ((number? i) i)) 1)
(test (do ((i #\c "hi")) ((string? i) i)) "hi")
(test (do ((i #\c +)) ((not (char? i)) i)) +)
(test (let ((j 1)) (do ((i 0 j)) ((= i j) i))) 1)
(test (let ((j 1)) (do ((i 0 j)) ((= i j) i) (set! j 2))) 2)
(test (do ((j 1 2) (i 0 j)) ((= i j) i)) 2)
(test (let ((old+ +) (j 0)) (do ((i 0 (old+ i 1))) ((or (< i -3) (> i 3))) (set! old+ -) (set! j (+ j i))) j) -6)
(test (let ((old+ +) (j 0)) (do ((i 0 (+ i 1))) ((or (< i -3) (> i 3))) (set! + -) (set! j (old+ j i))) (set! + old+) j) -6)
(test (do ((i 0 (case i ((0) 1) ((1) "hi")))) ((string? i) i)) "hi")
(test (do ((i if +)) ((equal? i +) i)) +)
(test (let ((k 0)) (do ((j 0 (+ j 1)) (i 0 ((if (= i 0) + -) i 1))) ((= j 5)) (set! k (+ k i))) k) 2)
(test (let ((j -10) (k 0)) (do ((i 0 (+ i j)) (j 2)) ((> i 4) k) (set! k (+ k i)))) 6)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((> i 4) k) (set! k (+ k i)))) -24)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((= i j) k) (set! k (+ k i)))) -30)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((= i j) j) (set! k (+ k i)))) 2)
(test (let ((equal =)) (do ((i 0 (+ i 1))) ((equal i 3) i))) 3)
(test (let ((equal =)) (do ((i 0 (+ i 1))) ((equal i 3) i) (set! equal >))) 4)
(test (do ((equal =) (i 0 (+ i 1))) ((equal i 3) i)) 3)
(test (do ((equal = >) (i 0 (+ i 1))) ((equal i 3) i)) 4)
(test (do ((j 0) (plus + -) (i 0 (plus i 1))) ((= i -1) j) (set! j (+ j 1))) 3)
(test (let ((expr `(+ i 1))) (do ((j 0) (i 0 (eval expr))) ((= i 3) j) (set! j (+ j 1)))) 3)
(test (let ((expr `(+ i 1))) (do ((j 0) (i 0 (eval expr))) ((= i -3) j) (set! j (+ j 1)) (if (= j 3) (set! expr `(- i 1))))) 7)
(test (do ((i 0 (+ i 1))) ((or (= i 12) (not (number? i)) (> (expt 2 i) 32)) (expt 2 i))) 64)
(test (let ((k 0)) (do ((i 0 (+ i 1))) ((let () (set! k (+ k 1)) (set! i (+ i 1)) (> k 3)) i))) 7)

(test (let ((lst '(1 2 3))
	    (v (vector 0 0 0)))
	(do ((l lst (map (lambda (a) (+ a 1)) (cdr l))))
	    ((null? l))
	  (set! (v (- (length l) 1)) (apply + l)))
	v)
      #(5 7 6))

(test (let ((lst '(1 2 3)))
	(map (lambda (a)
	       (let ((! 1))
		 (do ((i 0 (+ i 1))
		      (sum 0))
		     ((= i a) sum)
		   (set! sum (+ sum a)))))
	     lst))
      '(1 4 9))

(test (let ((sum 0)) (do ((i_0 0 (+ i_0 0))(i_1 1 (+ i_1 1))(i_2 2 (+ i_2 2))(i_3 3 (+ i_3 3))(i_4 4 (+ i_4 4))(i_5 5 (+ i_5 5))(i_6 6 (+ i_6 6))(i_7 7 (+ i_7 7))(i_8 8 (+ i_8 8))(i_9 9 (+ i_9 9))(i_10 10 (+ i_10 10))(i_11 11 (+ i_11 11))(i_12 12 (+ i_12 12))(i_13 13 (+ i_13 13))(i_14 14 (+ i_14 14))(i_15 15 (+ i_15 15))(i_16 16 (+ i_16 16))(i_17 17 (+ i_17 17))(i_18 18 (+ i_18 18))(i_19 19 (+ i_19 19))(i_20 20 (+ i_20 20))(i_21 21 (+ i_21 21))(i_22 22 (+ i_22 22))(i_23 23 (+ i_23 23))(i_24 24 (+ i_24 24))(i_25 25 (+ i_25 25))(i_26 26 (+ i_26 26))(i_27 27 (+ i_27 27))(i_28 28 (+ i_28 28))(i_29 29 (+ i_29 29))(i_30 30 (+ i_30 30))(i_31 31 (+ i_31 31))(i_32 32 (+ i_32 32))(i_33 33 (+ i_33 33))(i_34 34 (+ i_34 34))(i_35 35 (+ i_35 35))(i_36 36 (+ i_36 36))(i_37 37 (+ i_37 37))(i_38 38 (+ i_38 38))(i_39 39 (+ i_39 39)))
    ((= i_1 10) sum)
  (set! sum (+ sum i_0 i_1 i_2 i_3 i_4 i_5 i_6 i_7 i_8 i_9 i_10 i_11 i_12 i_13 i_14 i_15 i_16 i_17 i_18 i_19 i_20 i_21 i_22 i_23 i_24 i_25 i_26 i_27 i_28 i_29 i_30 i_31 i_32 i_33 i_34 i_35 i_36 i_37 i_38 i_39))))
      35100)

(test (call-with-exit (lambda (return) (do () () (if #t (return 123))))) 123)
(test (call-with-exit (lambda (return) (do () (#f) (if #t (return 123))))) 123)
(test (call-with-exit (lambda (return) (do ((i 0 (+ i 1))) () (if (= i 100) (return 123))))) 123)
(test (call-with-exit (lambda (return) (do () ((return 123))))) 123)
(test (call-with-exit (lambda (return) (do () (#t (return 123))))) 123)

(test (do () (/ 0)) 0)
(test (do () (+)) '())
(test (do () (+ +) *) +)

(if with-bignums
    (begin
      (num-test (do ((i 24444516448431392447461 (+ i 1))
		     (j 0 (+ j 1)))
		    ((>= i 24444516448431392447471) j))
		10)
      (num-test (do ((i 0 (+ i 24444516448431392447461))
		     (j 0 (+ j 1)))
		    ((>= i 244445164484313924474610) j))
		10)
      (num-test (do ((i 4096 (* i 2))
		     (j 0 (+ j 1)))
		    ((= i 4722366482869645213696) j))
		60)))

(test (do ((i 9223372036854775805 (+ i 1))
	   (j 0 (+ j 1)))
	  ((>= i 9223372036854775807) j))
      2)
(test (do ((i -9223372036854775805 (- i 1))
	   (j 0 (+ j 1)))
	  ((<= i -9223372036854775808) j))
      3)

(num-test (do ((x (list 1 2 3) (cdr x)) (j -1)) ((null? x) j) (set! j (car x))) 3)

(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((= i (do ((j 0 (+ j 1))) ((= j 2) (+ j 1)))))
	  (set! x (+ x i)))
	x)
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) x)
	  (set! x (+ x i))))
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) (do ((j 0 (+ j 1))) ((= j 5) x) (set! x j)))
	  (set! x (+ x i))))
      4)

(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 100) i) (if (= i 2) (exit 321))))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (if (= i 3) (exit 321) (+ i 1)))) ((= i 100) i)))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) (exit 321))))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) i) (if (= i -2) (exit 321))))) 10)
(test (do ((x 0 (+ x 1)) (y 0 (call/cc (lambda (c) c)))) ((> x 5) x) #f) 6)
(test (let ((happy #f)) (do ((i 0 (+ i 1))) (happy happy) (if (> i 3) (set! happy i)))) 4)

(test (+ (do ((i 0 (+ i 1))) ((= i 3) i)) (do ((j 0 (+ j 1))) ((= j 4) j))) 7)
(test (let ((do 1) (map 2) (for-each 3) (quote 4)) (+ do map for-each quote)) 10)

(test (let ((cont #f)
	    (j 0)
	    (k 0))
	(call/cc (lambda (exit) 
		   (do ((i 0 (+ i 1))) 
		       ((= i 100) i) 
		     (set! j i)
		     (call/cc (lambda (r) (set! cont r)))
		     (if (= j 2) (exit))
		     (set! k i))))
	(if (= j 2)
	    (begin
	      (set! j 3)
	      (cont))
	    (list j k)))
      (list 99 99))

(test (call/cc (lambda (r) (do () (#f) (r 1)))) 1)
(test (let ((hi (lambda (x) (+ x 1)))) (do ((i 0 (hi i))) ((= i 3) i))) 3)
(test (do ((i 0 (+ i 1))) (list 1) ((= i 3) #t)) 1) ; a typo originally -- Guile and Gauche are happy with it
(test (do () (1 2) 3) 2)

;; from sacla tests
(test (let ((rev (lambda (list)
		   (do ((x list (cdr x))
			(reverse '() (cons (car x) reverse)))
		       ((null? x) reverse)))))
	(and (null? (rev '()))
	     (equal? (rev '(0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (let ((nrev (lambda (list)
		    (do ((f1st (if (null? list) '() (cdr list)) (if (null? f1st) '() (cdr f1st)))
			 (s2nd list f1st)
			 (t3rd '() s2nd))
			((null? s2nd) t3rd)
		      (set-cdr! s2nd t3rd)))))
	(and (null? (nrev '()))
	     (equal? (nrev (list 0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (do ((temp-one 1 (+ temp-one 1))
	   (temp-two 0 (- temp-two 1)))
	  ((> (- temp-one temp-two) 5) temp-one))
      4)

(test (do ((temp-one 1 (+ temp-one 1))
	   (temp-two 0 (+ temp-one 1)))     
	  ((= 3 temp-two) temp-one))
      3)

(let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
  (test (do ((i 0 (+ 1 i))
	     (n #f)
	     (j 9 (- j 1)))
	    ((>= i j) vec)
	  (set! n (vector-ref vec i))
	  (vector-set! vec i (vector-ref vec j))
	  (vector-set! vec j n))
	'#(9 8 7 6 5 4 3 2 1 0)))

(test (do '() (#t 1)) 'error)
(test (do . 1) 'error)
(test (do ((i i i)) (i i)) 'error)
(test (do ((i 0 i (+ i 1))) (i i)) 'error)
(test (do ((i)) (#t i)) 'error)
(test (do ((i 0 (+ i 1))) #t) 'error)
(test (do 123 (#t 1)) 'error)
(test (do ((i 1)) (#t . 1) 1) 'error)
(test (do ((i 1) . 1) (#t 1) 1) 'error)
(test (do ((i 1) ()) (= i 1)) 'error)
(test (do ((i 0 . 1)) ((= i 1)) i) 'error)
(test (do ((i 0 (+ i 1))) ((= i 3)) (set! i "hiho")) 'error)
(test (let ((do+ +)) (do ((i 0 (do+ i 1))) ((= i 3)) (set! do+ abs))) 'error)
(test (do () . 1) 'error)
(test (do ((i)) (1 2)) 'error)
(test (do (((i))) (1 2)) 'error)
(test (do ((i 1) ((j))) (1 2)) 'error)
(test (do (((1))) (1 2)) 'error)

(test (let ((j #f))
	(do ((i 0 (let ((x 0))
		    (dynamic-wind
			(lambda ()
			  (set! x i))
			(lambda ()
			  (+ x 1))
			(lambda ()
			  (if (> x 3)
			      (set! j #t)))))))
	    (j i)))
      5)
(test (let ((j 0)) (do ((i 0 (eval-string "(+ j 1)"))) ((= i 4) j) (set! j i))) 3)
(test (do ((i (do ((i (do ((i 0 (+ i 1)))
			  ((= i 3) (+ i 1)))
		      (do ((j 0 (+ j 1)))
			  ((= j 3)) (+ j i))))
		  ((> (do ((k 0 (+ k 1)))
			  ((= k 2) (* k 4)))
		      (do ((n 0 (+ n 1)))
			  ((= n 3) n)))
		   (do ((m 0 (+ m 1)))
		       ((= m 3) (+ m i)))))
	      i))
	  ((> i 6) i))
      7)

(test (let ((L (list 1 2))) 
	(do ((sum 0 (+ sum (car lst))) 
	     (i 0 (+ i 1)) 
	     (lst L (cdr lst))) 
	    ((or (null? lst) 
		 (> i 10)) 
	     sum) 
	  (set-cdr! (cdr L) L))) 
      16)



;;; -------- set! --------

(test (let ((a 1)) (set! a 2) a) 2)
(for-each
 (lambda (arg)
   (test (let ((a 0)) (set! a arg) a) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (let ((a 1)) (call/cc (lambda (r) (set! a (let () (if (= a 1) (r 123)) 321)))) a) 1)
(test (let ((a (lambda (b) (+ b 1)))) (set! a (lambda (b) (+ b 2))) (a 3)) 5)
(test (let ((a (lambda (x) (set! x 3) x))) (a 1)) 3)

(test (let ((x (vector 1 2 3))) (set! (x 1) 32) x) #(1 32 3))
(test (let* ((x (vector 1 2 3))
	     (y (lambda () x)))
	(set! ((y) 1) 32)
	x)
      #(1 32 3))
(test (let* ((x (vector 1 2 3))
	     (y (lambda () x))
	     (z (lambda () y)))
	(set! (((z)) 1) 32)
	x)
      #(1 32 3))

(test (let ((a 1)) (set! a)) 'error)
(test (let ((a 1)) (set! a 2 3)) 'error)
(test (let ((a 1)) (set! a . 2)) 'error)
(test (let ((a 1)) (set! a 1 . 2)) 'error)
(test (set! "hi" 1) 'error)
(test (set! 'a 1) 'error)
(test (set! 1 1) 'error)
(test (set! (list 1 2) 1) 'error)
(test (set! (let () 'a) 1) 'error)
(test (set!) 'error)
(test (set! #t #f) 'error)
(test (set! '() #f) 'error)
(test (set! #(1 2 3) 1) 'error)
(test (set! (call/cc (lambda (a) a)) #f) 'error)
(test (set! 3 1) 'error)
(test (set! 3/4 1) 'error)
(test (set! 3.14 1) 'error)
(test (set! #\a 12) 'error)
(test (set! (1 2) #t) 'error)
(test (set! _not_a_var_ 1) 'error)
(test (set! (_not_a_pws_) 1) 'error)

(test (let ((a (lambda (x) (set! a 3) x))) (list (a 1) a)) 'error)
(test (let ((a (let ((b 1)) (set! a 3) b))) a) 'error)            
(test (let ((a (lambda () "hi"))) (set! (a) "ho")) 'error)
(test (let ((a (let ((b 1)) (set! a 3) b))) a) 'error) 

(test (set! . -1) 'error)
(test (set!) 'error)
(test (let ((x 1)) (set! x x x)) 'error)
(test (let ((x 1)) (set! x x) x) 1)
(test (set! set! 123) 'error)
(test (set! (cons 1 2) 3) 'error)
(test (let ((var 1) (val 2)) (set! var set!) (var val 3) val) 3)
(test (let ((var 1) (val 2)) (set! var +) (var val 3)) 5)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
	(set! sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 2)
	sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      2)

(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) x) '(32))




;;; -------- or --------

(test (or (= 2 2) (> 2 1)) #t)
(test (or (= 2 2) (< 2 1)) #t)
(test (or #f #f #f) #f)
(test (or) #f)
(test (or (memq 'b '(a b c)) (+ 3 0)) '(b c))
(test (or 3 9) 3)
(test (or #f 3 asdf) 3) ; "evaluation stops immediately"
(test (or 3 (/ 1 0) (display "or is about to exit!") (exit)) 3)

(for-each
 (lambda (arg)
   (test (or arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) #<eof> #<unspecified> '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read-char p)))
	      (or (eof-object? val)
		  (> loc 1000) ; try to avoid the read-error stuff
		  (begin
		    (set! loc (+ 1 loc))
		    (loop (read-char p)))))
	    (> loc 1000))))
      #t)

(test (or (and (or (> 3 2) (> 3 4)) (> 2 3)) 4) 4)
(test (or or) or)
(test (or (or (or))) #f)
(test (or (or (or) (and))) #t)
(test (let ((a 1)) (or (let () (set! a 2) #f) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) 3)
(test (or (let ((or #t)) or)) #t)
(test (or '#f '()) '())
(test (call/cc (lambda (r) (or #f (> 3 2) (r 123) 321))) #t)
(test (call/cc (lambda (r) (or #f (< 3 2) (r 123) 321))) 123)
(test (+ (or #f (not (null? '())) 3) (or (zero? 1) 2)) 5)
(test (or 0) 0)
(test (if (or) 1 2) 2)

(test (or . 1) 'error)
(test (or #f . 1) 'error)
(test (or . (1 2)) 1)
(test (or . ()) (or))



;;; -------- and --------

(test (and (= 2 2) (> 2 1)) #t)
(test (and (= 2 2) (< 2 1)) #f)
(test (and 1 2 'c '(f g)) '(f g))
(test (and) #t)
(test (and . ()) (and))
(test (and 3) 3)
(test (and (memq 'b '(a b c)) (+ 3 0)) 3)
(test (and 3 9) 9)
(test (and #f 3 asdf) #f) ; "evaluation stops immediately"
(test (and 3 (zero? 1) (/ 1 0) (display "and is about to exit!") (exit)) #f)
(test (if (and) 1 2) 1)
(test (if (+) 1 2) 1)
(test (if (*) 1 2) 1)
(test (and (if #f #f)) (if #f #f))

(for-each
 (lambda (arg)
   (test (and arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read-char p)))
	      (and (not (eof-object? val))
		   (< loc 1000)
		   (begin
		     (set! loc (+ 1 loc))
		     (loop (read-char p)))))
	    (>= loc 1000))))
      #t)

(test (and (or (and (> 3 2) (> 3 4)) (> 2 3)) 4) #f)
(test (and and) and)
(test (and (and (and))) #t)
(test (and (and (and (and (or))))) #f)
(test (let ((a 1)) (and (let () (set! a 2) #t) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) #f)
(test (and (let ((and #t)) and)) #t)
(test (and '#t '()) '())
(test (call/cc (lambda (r) (and #t (> 3 2) (r 123) 321))) 123)
(test (call/cc (lambda (r) (and #t (< 3 2) (r 123) 321))) #f)
(test (+ (and (null? '()) 3) (and (zero? 0) 2)) 5)

(test (and . #t) 'error)
(test (and 1 . 2) 'error)
(test (and . (1 2)) 2)



;;; -------- cond --------

(test (cond ('a)) 'a)
(test (cond (3)) 3)
(test (cond (#f 'a) ('b)) 'b)
(test (cond (#t 'a) (#t 'b)) 'a)
(test (cond ((> 3 2) 'greater) ((< 3 2) 'less)) 'greater)
(test (cond ((> 3 3) 'greater) ((< 3 3) 'less)  (else 'equal)) 'equal)
(test (cond ((assv 'b '((a 1) (b 2))) => cadr)  (else #f)) 2)
(test (cond (#f 2) (else 5)) 5)
(test (cond (1 2) (else 5)) 2)
(test (cond (1 => (lambda (x) (+ x 2))) (else 8)) 3)
(test (cond ((+ 1 2))) 3)
(test (cond ((zero? 1) 123) ((= 1 1) 321)) 321)
(test (cond ('() 1)) 1)
(test (let ((x 1)) (cond ((= 1 2) 3) (else (* x 2) (+ x 3)))) 4)
(test (let ((x 1)) (cond ((= x 1) (* x 2) (+ x 3)) (else 32))) 4)
(test (let ((x 1)) (cond ((= x 1) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 5)
(test (let ((x 1)) (cond ((= x 2) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 32)
(test (let ((x 1)) (cond ((= x 2) 3) (else (let () (set! x (* x 2))) (+ x 3)))) 5)
(test (cond ((= 1 2) 3) (else 4) (else 5)) 4) ; this should probably be an error
(test (cond (1 2 3)) 3)
(test (cond (1 2) (3 4)) 2)
(test (cond ((= 1 2) 3) ((+ 3 4))) 7)
(test (cond ((= 1 1) (abs -1) (+ 2 3) (* 10 2)) (else 123)) 20)
(test (let ((a 1)) (cond ((= a 1) (set! a 2) (+ a 3)))) 5)
(test (let ((a 1)) (cond ((= a 2) (+ a 2)) (else (set! a 3) (+ a 3)))) 6)
(test (cond ((= 1 1))) #t)
(test (cond ((= 1 2) #f) (#t)) #t)
(test (cond ((+ 1 2))) 3)
(test (cond ((cons 1 2))) '(1 . 2))
(test (cond (#f #t) ((string-append "hi" "ho"))) "hiho")
(test (cond ('() 3) (#t 4)) 3)
(test (cond ((list) 3) (#t 4)) 3)
;;; (cond (1 1) (asdf 3)) -- should this be an error?
(test (cond (+ 0)) 0)

(test (cond . ((1 2) ((3 4)))) 2)

(for-each
 (lambda (arg)
   (test (cond ((or arg) => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (cond ((+ 1 2) => (lambda (x) (+ 1 x)))) 4)
(test (cond ((cons 1 2) => car)) 1)
(test (cond ((values 1 2) => +)) 3)
(test (cond (1 2 => +)) 'error)
(test (cond ((begin 1 2) => +)) 2)
(test (cond ((values -1) => abs)) 1)
(test (cond ((= 1 2) => +) (#t => not)) #f)
(test (cond ((* 2 3) => (let () -))) -6)
(test (cond ((* 2 3) => (cond ((+ 3 4) => (lambda (a) (lambda (b) (+ b a))))))) 13)
(test (let ((x 1)) ((cond ((let () (set! x 2) #f) => boolean?) (lambda => (lambda (a) (apply a '((b) (+ b 123)))))) x)) 125)
(test (cond ((values 1 2 3) => '(1 (2 3 (4 5 6 7 8))))) 7)

(test (cond (else 1)) 1)
(test (call/cc (lambda (r) (cond ((r 4) 3) (else 1)))) 4)
(test (cond ((cond (#t 1)))) 1)

(for-each
 (lambda (arg)
   (test (cond (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (#f 1) (else arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (cond ((let () 1) => (let ((x 2)) (lambda (n) (+ n x))))) 3)
(test (cond ((let () 1) => (let ((x 2)) (cond (3 => (let ((y 4)) (lambda (n) (lambda (m) (+ n m x y))))))))) 10)

(test (let ((=> 3) (cond 4)) (+ => cond)) 7)
(test (cond (cond 'cond)) 'cond)
(test (cond (3 => (lambda args (car args)))) 3)
(test (cond (3 => (lambda (a . b) a))) 3)
(test (cond ((list 3 4) => (lambda (a . b) b))) '())
(test (cond) 'error)
					;(test (cond ((= 1 2) 3) (else 4) (4 5)) 'error)
(test (cond ((+ 1 2) => (lambda (a b) (+ a b)))) 'error)
(test (equal? (cond (else)) else) #t)
(test (cond (#t => 'ok)) 'error)
(test (cond (else =>)) 'error)
(test (cond ((values -1) => => abs)) 'error)
(test (cond ((values -1) =>)) 'error)
(test (cond (cond (#t 1))) 'error)
(test (cond 1) 'error)
(test (cond (1 . 2) (else 3)) 'error)
(test (cond (#f 2) (else . 4)) 'error)
(test (cond ((values 1 2) => (lambda (x y) #t))) #t)
(test (cond #t) 'error)
(test (cond 1 2) 'error)
(test (cond 1 2 3) 'error)
(test (cond 1 2 3 4) 'error)
(test (cond (1 => (lambda (x y) #t))) 'error)
(test (cond . 1) 'error)
(test (cond ((1 2)) . 3) 'error)
(test (cond (1 => + abs)) 'error)
(test (cond (1 =>)) 'error)
(test (cond ((values 1 2) => + abs)) 'error)
(test (cond (else => not)) 'error)
(test (let ((else 3)) (cond ((= else 3) 32) (#t 1))) 32)
(test (let ((else #f)) (cond (else 32) (#t 1))) 1)

(test (let ((=> 3)) (cond (1 =>))) 3)
(test (let ((=> 3)) (cond (1 => abs))) abs)
(test (let ((=> 3) (else 4)) (cond (else => abs))) abs)

(test (let ((x 0))
	(cond ((let ((y x)) (set! x 1) (= y 1)) 0)
	      ((let ((y x)) (set! x 1) (= y 1)) 1)
	      (#t 2)))
      1)

(let ((c1 #f)
      (x 1))
  (let ((y (cond ((let ()
		    (call/cc
		     (lambda (r)
		       (set! c1 r)
		       (r x))))
		  => (lambda (n) (+ n 3)))
		 (#t 123))))
    (if (= y 4) (begin (set! x 2) (c1 321)))
    (test (list x y) '(2 324))))

(let ((c1 #f)
      (x 1))
  (let ((y (cond (x => (lambda (n) 
			 (call/cc
			  (lambda (r)
			    (set! c1 r)
			    (r (+ 3 x))))))
		 (#t 123))))
    (if (= y 4) (begin (set! x 2) (c1 321)))
    (test (list x y) '(2 321))))





;;; -------- case --------

(test (case (* 2 3) ((2 3 5 7) 'prime) ((1 4 6 8 9) 'composite))  'composite)
(test (case (car '(c d)) ((a e i o u) 'vowel) ((w y) 'semivowel) (else 'consonant)) 'consonant)
(test (case 3.1 ((1.3 2.4) 1) ((4.1 3.1 5.4) 2) (else 3)) 2)
(test (case 3/2 ((3/4 1/2) 1) ((3/2) 2) (else 3)) 2)
(test (case 3 ((1) 1 2 3) ((2) 2 3 4) ((3) 3 4 5)) 5)
(test (case 1+i ((1) 1) ((1/2) 1/2) ((1.0) 1.0) ((1+i) 1+i)) 1+i)
(test (case 'abs ((car cdr) 1) ((+ cond) 2) ((abs) 3) (else 4)) 3)
(test (case #\a ((#\b) 1) ((#\a) 2) ((#\c) 3)) 2)
(test (case (boolean? 1) ((#t) 2) ((#f) 1) (else 0)) 1)
(test (case 1 ((1 2 3) (case 2 ((1 2 3) 3)))) 3)
(test (case 1 ((1 2) 1) ((3.14 2/3) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 1)
(test (case #f ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 4)
(test (case 1 ((#t) 2) ((#f) 1) (else 0)) 0)
(test (let ((x 1)) (case x ((x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case x ((1) "hi") (else "ho"))) "hi")
(test (let ((x 1)) (case x (('x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case 'x ((x) "hi") (else "ho"))) "hi")
(test (case '() ((()) 1)) 1)
;;; but not (case #() ((#()) 1)) because (eqv? #() #()) is #f

(test (case else ((#f) 2) ((#t) 3) ((else) 4) (else 5)) 5)          ; (eqv? 'else else) is #f (Guile says "unbound variable: else")
(test (case #t ((#f) 2) ((else) 4) (else 5)) 5)                     ; else is a symbol here         

(test (let ((x 1)) (case x ((2) 3) (else (* x 2) (+ x 3)))) 4)
(test (let ((x 1)) (case x ((1) (* x 2) (+ x 3)) (else 32))) 4)
(test (let ((x 1)) (case x ((1) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 5)
(test (let ((x 1)) (case x ((2) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 32)
(test (let ((x 1)) (case x ((2) 3) (else (let () (set! x (* x 2))) (+ x 3)))) 5)
(test (let ((x 1)) (case x ((2) 3) (else 4) (else 5))) 'error)

(test (case '() ((()) 2) (else 1)) 2)    ; car: (), value: (), eqv: 1, null: 1 1
(test (case '() (('()) 2) (else 1)) 1)   ; car: (quote ()), value: (), eqv: 0, null: 0 1
(test (case () (('()) 2) (else 1)) 1)    ; car: (quote ()), value: (), eqv: 0, null: 0 1
(test (case () ((()) 2) (else 1)) 2)     ; car: (), value: (), eqv: 1, null: 1 1

;;; this is a difference between '() and () ?
;;; (eqv? '() '()) -> #t and (eqv? '() ()) is #t so it's the lack of evaluation in the search case whereas the index is evaluated
;;; equivalent to:
 
(test (case 2 (('2) 3) (else 1)) 1)      ; car: (quote 2), value: 2, eqv: 0, null: 0 0
(test (case '2 (('2) 3) (else 1)) 1)     ; car: (quote 2), value: 2, eqv: 0, null: 0 0
(test (case '2 ((2) 3) (else 1)) 3)      ; car: 2, value: 2, eqv: 1, null: 0 0
(test (case 2 ((2) 3) (else 1)) 3)       ; car: 2, value: 2, eqv: 1, null: 0 0

(test (let ((x 1)) (case (+ 1 x) ((0 "hi" #f) 3/4) ((#\a 1+3i '(1 . 2)) "3") ((-1 'hi 2 2.0) #\f))) #\f)
(test (case (case 1 ((0 2) 3) (else 2)) ((0 1) 2) ((4 2) 3) (else 45)) 3)
(test (case 3/4 ((0 1.0 5/6) 1) (("hi" 'hi 3/4) 2) (else 3)) 2)
(test (case (case (+ 1 2) (else 3)) ((3) (case (+ 2 2) ((2 3) 32) ((4) 33) ((5) 0)))) 33)
(test (let ((x 1)) (case x ((0) (set! x 12)) ((2) (set! x 32))) x) 1)

(test (case 1 (else #f)) #f)
(test (case 1 ((1 2) (let ((case 3)) (+ case 1))) ((3 4) 0)) 4)

(for-each
 (lambda (arg)
   (test (case 1 ((0) 'gad) ((1 2 3) arg) (else 'gad)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (case arg ((0) 'gad) ((1 2 3) arg) (else 'gad)) 'gad))
 (list "hi" -1 #\a 0 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call/cc (lambda (r) (case 1 ((1) (r 123) #t) (else #f)))) 123)
(test (call/cc (lambda (r) (case 1 ((0) 0) (else (r 123) #f)))) 123)

(test (case '() ((1) 1) ('() 2)) 2)
(test (case (list) ((1) 1) ('() 2)) 2)
(test (case '() ((1) 1) ((()) 2)) 2)
(test (case (list) ((1) 1) ((()) 2)) 2)
(test (case #<eof> ((#<eof>) 1)) 1)
(test (case #\newline ((#\newline) 1)) 1)

; case use eqv? -- why not case-equal?
;(test (case "" (("") 1)) 1)
;(test (case abs ((abs) 1)) 1)

(test (case 1) 'error)
(test (case 1 . "hi") 'error)
(test (case 1 ("hi")) 'error)
(test (case 1 ("a" "b")) 'error)
(test (case 1 (else #f) ((1) #t)) 'error)
(test (case "hi" (("hi" "ho") 123) ("ha" 321)) 'error)
(test (case) 'error)
(test (case . 1) 'error)
(test (case 1 . 1) 'error)
(test (case 1 (#t #f) ((1) #t)) 'error)
(test (case 1 (#t #f)) 'error)
(test (case -1 ((-1) => abs)) 'error)
(test (case #t ((1 2) (3 4)) -1) 'error)
(test (case 1 1) 'error)
(test (case 1 ((2) 1) . 1) 'error)
(test (case 1 (2 1) (1 1)) 'error)
(test (case 1 (else)) 'error)
(test (case () ((1 . 2) . 1) . 1) 'error)
(test (case 1 ((1))) 'error)
(test (case 1 ((else))) 'error)
(test (case 1 ((2) 3) ((1))) 'error)
(test (case 1 ((1)) 1 . 2) 'error)
(test (case () ((()))) 'error)
(test (case 1 (else 3) . 1) 'error)

(test (case case ((case) 1) ((cond) 3)) 1)
(test (case 101 ((0 1 2) 200) ((3 4 5 6) 600) ((7) 700) ((8) 800) ((9 10 11 12 13) 1300) ((14 15 16) 1600) ((17 18 19 20) 2000) ((21 22 23 24 25) 2500) ((26 27 28 29) 2900) ((30 31 32) 3200) ((33 34 35) 3500) ((36 37 38 39) 3900) ((40) 4000) ((41 42) 4200) ((43) 4300) ((44 45 46) 4600) ((47 48 49 50 51) 5100) ((52 53 54) 5400) ((55) 5500) ((56 57) 5700) ((58 59 60) 6000) ((61 62) 6200) ((63 64 65) 6500) ((66 67 68 69) 6900) ((70 71 72 73) 7300) ((74 75 76 77) 7700) ((78 79 80) 8000) ((81) 8100) ((82 83) 8300) ((84 85 86 87) 8700) ((88 89 90 91 92) 9200) ((93 94 95) 9500) ((96 97 98) 9800) ((99) 9900) ((100 101 102) 10200) ((103 104 105 106 107) 10700) ((108 109) 10900) ((110 111) 11100) ((112 113 114 115) 11500) ((116) 11600) ((117) 11700) ((118) 11800) ((119 120) 12000) ((121 122 123 124 125) 12500) ((126 127) 12700) ((128) 12800) ((129 130) 13000) ((131 132) 13200) ((133 134 135 136) 13600) ((137 138) 13800)) 10200)
(test (case most-positive-fixnum ((-1231234) 0) ((9223372036854775807) 1) (else 2)) 1)
(test (case most-negative-fixnum ((123123123) 0) ((-9223372036854775808) 1) (else 2)) 1)
(test (case 0 ((3/4 "hi" #t) 0) ((#f #() -1) 2) ((#\a 0 #t) 3) (else 4)) 3)
(test (case 3/4 ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 0)
(test (case 'hi ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 2)
(test (case #f ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 2)
(test (case 3 ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 4)





;;; -------- lambda --------

(test (procedure? (lambda (x) x)) #t)
(test ((lambda (x) (+ x x)) 4) 8)
(test (let ((reverse-subtract (lambda (x y) (- y x)))) (reverse-subtract 7 10)) 3)
(test (let ((add4 (let ((x 4)) (lambda (y) (+ x y))))) (add4 6)) 10)
(test ((lambda x x) 3 4 5 6) (list 3 4 5 6))
(test ((lambda (x y . z) z) 3 4 5 6) (list 5 6))
(test ((lambda (a b c d e f) (+ a b c d e f)) 1 2 3 4 5 6) 21)
(test (let ((foo (lambda () 9))) (+ (foo) 1)) 10)
(test (let ((a 1)) (let ((f (lambda (x) (set! a x) a))) (let ((c (f 123))) (list c a)))) (list 123 123))
(test (let ((a 1) (b (lambda (a) a))) (b 3)) 3)
(test (let ((ctr 0)) (letrec ((f (lambda (x) (if (> x 0) (begin (set! ctr (+ ctr 1)) (f (- x 1))) 0)))) (f 10) ctr)) 10)
(test (let ((f (lambda (x) (car x)))) (f '(4 5 6))) 4)
(test ((lambda () ((lambda (x y) ((lambda (z) (* (car z) (cdr z))) (cons x y))) 3 4))) 12)
(test (let ((ctr 0)) (define (f) (set! ctr (+ ctr 1)) ctr) (let ((x (f))) (let ((y (f))) (list x y ctr)))) (list 1 2 2))

(test (let ((x 5)) (define foo (lambda (y) (bar x y))) (define bar (lambda (a b) (+ (* a b) a))) (foo (+ x 3))) 45)
(test (let ((x 5)) (letrec ((foo (lambda (y) (bar x y))) (bar (lambda (a b) (+ (* a b) a)))) (foo (+ x 3)))) 45)
(num-test (let () (define compose (lambda (f g) (lambda args (f (apply g args))))) ((compose sqrt *) 12 75))  30.0)
(test (let ((f (lambda () (lambda (x y) (+ x y))))) ((f) 1 2)) 3)
(test ((lambda (x) (define y 4) (+ x y)) 1) 5)
(test ((lambda () (define (y x) (+ x 1)) (y 1))) 2)
(test ((lambda (x) 123 (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) "documentation" (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) (x 1)) (lambda (y) (+ y 1))) 2)
(test (let ((a 1)) (let ((b (lambda (x) (define y 1) (define z 2) (define a 3) (+ x y z a)))) (b a))) 7)
(test ((lambda (f x) (f x x)) + 11) 22)
(test ((lambda () (+ 2 3))) 5)
(test (let ((x (let () (lambda () (+ 1 2))))) (x)) 3)
(test (cond (0 => (lambda (x) x))) 0)
(test ((lambda () "hiho")) "hiho")

(test (letrec ((f (lambda (x) (g x)))
	       (g (lambda (x) x)))
	(let ((top (f 1)))
	  (set! g (lambda (x) (- x)))
	  (+ top (f 1))))
      0)

(for-each
 (lambda (arg)
   (test ((lambda (x) x) arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((list-length
       (lambda (obj)
	 (call-with-current-continuation
	  (lambda (return)
	    (letrec ((r (lambda (obj) (cond ((null? obj) 0)
					    ((pair? obj) (+ (r (cdr obj)) 1))
					    (else (return #f))))))
	      (r obj)))))))
  (test (list-length '(1 2 3 4)) 4)
  (test (list-length '(a b . c)) #f))

(test (let ((samples (vector 0 1 2 3 4 5 6 7 8 9 10)))
	(let ((make-scaler 
	       (lambda (start end)
		 (letrec ((ctr start)
			  (us (lambda (them)
				(vector-set! samples ctr (* 2 (vector-ref samples ctr)))
				(set! ctr (+ ctr 2))
				(if (<= ctr end)
				    (them us)))))
		   us))))
	  ((make-scaler 0 11)
	   (make-scaler 1 11))) 
	samples)
      (vector 0 2 4 6 8 10 12 14 16 18 20))

(test ((lambda (x . y) y) 1 2 '(3 . 4)) '(2 (3 . 4)))
(test ((lambda (x . y) y) 1) '())
(test ((lambda x x) '()) '(()))
(test ((lambda x x)) '())
(test ((lambda (x) x) '()) '())
(test (let ((lambda 4)) (+ lambda 1)) 5)
(test ((lambda (x) (+ x ((lambda (x) (+ x 1)) 2))) 3) 6)
(test ((lambda (x) (define y 1) (+ x y)) 2) 3)
(test ((lambda (a) "this is a doc string" a) 1) 1)
;;; ideally ((lambda (a) "hiho" (define x 1) x) 1) -> 1 but I'm not sure it's r5rs-ish
(test (let ((g (lambda () '3))) (= (g) 3)) #t)

(test ((lambda lambda lambda) 'x) '(x))
					;(test ((lambda (begin) (begin 1 2 3)) (lambda lambda lambda)) 3)

(test (let () ; PLP Scott p168
	(define A
	  (lambda ()
	    (let* ((x 2)
		   (C (lambda (P)
			(let ((x 4))
			  (P))))
		   (D (lambda ()
			x))
		   (B (lambda ()
			(let ((x 3))
			  (C D)))))
	      (B))))
	(A))
      2)

#|
;;; here s7 "do" uses set!
(test (let ((funcs (make-vector 3 #f)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (vector-set! funcs i (lambda () (+ i 1))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)
|#

;;; the equivalent named let version:
(test (let ((funcs (make-vector 3 #f)))
	(let loop ((i 0))
	  (if (< i 3)
	      (begin
		(vector-set! funcs i (lambda () (+ i 1)))
		(loop (+ i 1)))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((i 1))
	(let ((func1 (lambda () i)))
	  (let ((i 2))
	    (let ((func2 (lambda () i)))
	      (+ (func1) (func2))))))
      3)

(test (let ((funcs (make-vector 3 #f)))
	(map
	 (lambda (i)
	   (vector-set! funcs i (lambda () (+ i 1))))
	 (list 0 1 2))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (+ x 1))))
	(func1 1)
	(+ (func)
	   (let ()
	     (func1 2)
	     (func))))
      5)

(test (((lambda (x) (lambda () (+ x 1))) 32)) 33)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (string-append x "-"))))
	(func1 "hi")
	(string-append (func)
		       (let ()
			 (func1 "ho")
			 (func))))
      "hi-ho-")

(test (let ((func1 #f)
	    (func2 #f))
	(let ((x 1))
	  (set! func1 (lambda () x))
	  (set! func2 (lambda (y) (set! x y) y)))
	(+ (func1)
	   (let ()
	     (func2 32)
	     (func1))))
      33)

(test (let ((funcs (make-vector 3)))
	(let ((hi (lambda (a) (vector-set! funcs (- a 1) (lambda () a)))))
	  (hi 1) (hi 2) (hi 3)
	  (+ ((vector-ref funcs 0))
	     ((vector-ref funcs 1))
	     ((vector-ref funcs 2)))))
      6)

(test (let ((hi (lambda (a) (+ a 1)))
	    (ho (lambda (a) (a 32))))
	(+ (hi (hi (hi 1)))
	   (ho hi)))
      37)

(test ((if (> 3 2) + -) 3 2) 5)
(test (let ((op +)) (op 3 2)) 5)
(test (((lambda () +)) 3 2) 5)
(test ((car (cons + -)) 3 2) 5)
(test ((do ((i 0 (+ i 1))) ((= i 3) +) ) 3 2) 5)
(test (((lambda (x) x) (lambda (x) x)) 3) 3)
(test ((((lambda (x) x) (lambda (x) x)) (lambda (x) x)) 3) 3)
(test (((lambda (x) (lambda (y) x)) 3) 4) 3)
(test (((lambda (x) (lambda (x) x)) 3) 4) 4)
(test (let ((x 32)) (((lambda (x) (lambda (y) x)) 3) x)) 3)
(test ((call/cc (lambda (return) (return +))) 3 2) 5)
(test ((call-with-values (lambda () (values +)) (lambda (x) x)) 3 2) 5)
(test ((case '+ ((+) +)) 3 2) 5)
(test ((case '+ ((-) -) (else +)) 3 2) 5)
(test ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 5)
(test (+ 1 ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 2) 8)
(test (let ((lst (list + -))) ((car lst) 1 2 3)) 6)
(test (let ((a +)) ((let ((b -)) (if (eq? a b) a *)) 2 3)) 6)
(test ((list-ref (list + - * /) 0) 2 3) 5)
(test (((if #t list-ref oops) (list + - * /) 0) 2 3) 5)
(test ((((car (list car cdr)) (list car cdr)) (list + -)) 2 3) 5)
(test (let ()
	(define function lambda)
	(define hiho (function (a) (+ a 1)))
	(hiho 2))
      3)
(test ((lambda (let) (let* ((letrec 1)) (+ letrec let))) 123) 124)
(test ((lambda (let*) (let ((letrec 1)) (+ letrec let*))) 123) 124)
(test ((lambda (a b c d e f g h i j k l m n o p q r s t u v x y z)
	 (+ a b c d e f g h i j k l m n o p q r s t u v x y z))
       1 2 3 4 5 6 7 8 9 11 12 13 14 15 16 17 18 19 21 22 23 24 25 26 27)
      348)
(test ((lambda (x) "a useless string" x) 32) 32)
(test ((lambda (>< =0=? .arg.) (+ >< =0=? .arg.)) 1 2 3) 6)

(test
 (let ()
   (begin
     (define f1 #f)
     (define f2 #f)
     (let ((lv 32))
       (set! f1 (lambda (a) (+ a lv)))
       (set! f2 (lambda (a) (- a lv)))))
   (+ (f1 1) (f2 1)))
 2)

(test ((lambda () => abs)) 'error)
(test ((lambda () => => 3)) 'error)
;; actually, both Guile and Gauche accept
;; ((lambda () + 3)) and (begin + 3)
;; but surely => is an undefined variable in this context?

(test (lambda) 'error)
(test (lambda (a) ) 'error)
;; should this be an error: (lambda (a) (define x 1)) ?
(test (lambda . 1) 'error)
(test (lambda 1) 'error)
(test (lambda (x 1) x) 'error)
(test (lambda "hi" 1) 'error)
(test (lambda (x x) x) 'error)
(test ((lambda (x x) x) 1 2) 'error) 
(test (lambda (x "a")) 'error)
(test ((lambda (x y) (+ x y a)) 1 2) 'error)
(test ((lambda ())) 'error)
(test (lambda (x (y)) x) 'error)
(test ((lambda (x) x . 5) 2) 'error)
(test (lambda (1) #f) 'error)
;(test (lambda (x . y z) x) 'error) 
(test ((lambda () 1) 1) 'error)
(test ((lambda (()) 1) 1) 'error)
(test ((lambda (x) x) 1 2) 'error)
(test ((lambda (x) x)) 'error)
(test ((lambda ("x") x)) 'error)
(test ((lambda "x" x)) 'error)
(test ((lambda (x . "hi") x)) 'error)
(test (let ((hi (lambda (a 0.0) (b 0.0) (+ a b)))) (hi)) 'error)
(test (object->string
       ((lambda (arg)
	  (list arg
		(list (quote quote)
		      arg)))
	(quote (lambda (arg)
		 (list arg
		       (list (quote quote)
			     arg))))))
      "(#1=(lambda (arg) (list arg (list 'quote arg))) '#1#)")
      
(test ((apply lambda '((a) (+ a 1))) 2) 3)
(test ((apply lambda '(() #f))) #f)
(test ((apply lambda '(arg arg)) 3) '(3))
(test ((apply lambda* '((a (b 1)) (+ a b))) 3 4) 7)
(test ((apply lambda* '((a (b 1)) (+ a b))) 3) 4)





;;; -------- begin --------

(test (let () (begin) #f) #f)
(test (let () (begin (begin (begin (begin)))) #f) #f)
(test (let () (begin (define x 2) (define y 1)) (+ x y)) 3)
(test (let () (begin (define x 0)) (begin (set! x 5) (+ x 1)))  6)
(test (let () (begin (define first car)) (first '(1 2))) 1)
(test (let () (begin (define x 3)) (begin (set! x 4) (+ x x))) 8)
(test (let () (begin (define x 0) (define y x) (set! x 3) y)) 0)         ; the let's block confusing global defines
(test (let () (begin (define x 0) (define y x) (begin (define x 3) y))) 0)
(test (let () (begin (define y x) (define x 3) y)) 'error)               ; guile says 3
(test (let ((x 12)) (begin (define y x) (define x 3) y)) 12)             ; guile says 3 which is letrec-style?
;; (let ((x 12)) (begin (define y x) y)) is 12
(test (let ((x 3)) (begin x)) 3)
(test (begin 3) 3)
(test (begin . (1 2)) 2)
(test (begin . ()) (begin))
(test (begin . 1) 'error)
(test (begin 1 . 2) 'error)

(if (equal? (begin 1) 1)
    (begin
      (test (let () (begin (define x 0)) (set! x (begin (begin 5))) (begin ((begin +) (begin x) (begin (begin 1))))) 6)      
      
      (test (let ((x 5))
	      (begin (begin (begin)
			    (begin (begin (begin) (define foo (lambda (y) (bar x y)))
					  (begin)))
			    (begin))
		     (begin)
		     (begin)
		     (begin (define bar (lambda (a b) (+ (* a b) a))))
		     (begin))
	      (begin)
	      (begin (foo (+ x 3))))
	    45)
      
      (for-each
       (lambda (arg)
	 (test (begin arg) arg))
       (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
      
      (test (if (= 1 1) (begin 2) (begin 3)) 2)
      ))

(test (let ((begin 3)) (+ begin 1)) 4)
(test ((lambda (x) (begin (set! x 1) (let ((a x)) (+ a 1)))) 2) 2)
;;; apparently these can be considered errors or not (guile says error, stklos and gauche do not)
(test (begin (define x 0) (+ x 1)) 1)
(test ((lambda () (begin (define x 0) (+ x 1)))) 1)
(test (let ((f (lambda () (begin (define x 0) (+ x 1))))) (f)) 1)

(test ((lambda () (begin (define x 0)) (+ x 1))) 1)
(test (let ((f (lambda () (begin (define x 0)) (+ x 1)))) (f)) 1)
(test (let ((x 32)) (begin (define x 3)) x) 3)
(test ((lambda (x) (begin (define x 3)) x) 32) 3)
(test (let* ((x 32) (y x)) (define x 3) y) 32)

(test (let ((z 0)) (begin (define x 32)) (begin (define y x)) (set! z y) z) 32) ; so begin is like let*? -- guile uses letrec here = error
(test (let ((z 0)) (begin (define x 32) (define y x)) (set! z y) z) 32)         ; similarly here
;;; I can't find anything in r5rs.html that mandates letrec here, or that says it's in error
;;; Guile is now (1.9.5) happy with these

(test (let () (begin (define b 1) (begin (define a b) (define b 3)) a)) 1)
(test (let () (begin (begin (define a1 1) (begin (define a1 b1) (define b1 3))) a1)) 'error)
(test (let () (begin (begin (define (a3) 1)) (begin (define (a3) b3) (define b3 3)) (a3))) 3) ; yow
(test (let () (begin (begin (define (a) 1)) (a))) 1)
(test (let ((a 1)) (begin (define a 2)) a) 2)
(test (+ 1 (begin (values 2 3)) 4) 10)
(test (+ 1 (begin (values 5 6) (values 2 3)) 4) 10)





;;; -------- apply --------

(test (apply (lambda (a b) (+ a b)) (list 3 4)) 7)
(test (apply + 10 (list 3 4)) 17)
(test (apply list '()) '())
(test (apply + '(1 2)) 3)
(test (apply - '(1 2)) -1)
(test (apply max 3 5 '(2 7 3)) 7)
(test (apply cons '((+ 2 3) 4)) '((+ 2 3) . 4))
(test (apply + '()) 0)
(test (apply + (list 3 4)) 7)
(test (apply + '()) 0)
(test (apply + 2 '(3)) 5)
(test (apply + 2 3 '()) 5)
(test (apply + '(2 3)) 5)
(test (apply list 1 '(2 3)) (list 1 2 3))
(test (apply apply (list list 1 2 '(3))) (list 1 2 3))
(test (vector? (apply make-vector '(1))) #t)
(test (apply make-vector '(1 1)) '#(1))
(test (let ((f +)) (apply f '(1 2))) 3)
					;(test (let* ((x '(1 2 3)) (y (apply list x))) (not (eq? x y))) #f) ; is this standard?
(test (apply min '(1 2 3 5 4 0 9)) 0)
(test (apply min 1 2 4 3 '(4 0 9)) 0)
(test (apply vector 1 2 '(3)) '#(1 2 3))
(test (apply (lambda (x . y) x) (list 1 2 3)) 1)
(test (apply * (list 2 (apply + 1 2 '(3)))) 12)
(test (apply (if (> 3 2) + -) '(3 2)) 5)

(for-each
 (lambda (arg)
   (test (apply (lambda (x) x) (list arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (apply cadadr (list '''4)) 4)
(test (apply string-ref "hi" '(0)) #\h)
(test (let ((x (string-copy "hi"))) (apply string-set! x 0 '(#\c)) x) "ci")
(test (apply apply (list + '(3  2))) 5)
(test (apply apply apply apply (list (list (list + '(3  2))))) 5)
(test (apply + 1 2 (list 3 4)) 10)

(test (apply + #f) 'error)
(test (apply #f '(2 3)) 'error)
(test (apply make-vector '(1 2 3)) 'error)
(test (apply + 1) 'error)
(test (apply) 'error)
(test (apply 1) 'error)
(test (apply . 1) 'error)
(test (apply car ''foo) 'error)
(test (apply + '(1 . 2)) 'error)
(test (apply + '(1 2 . 3)) 'error)
(test (apply '() '()) 'error)

(for-each
 (lambda (arg)
   (test (apply arg '(1)) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t)) 

(test (apply "hi" '(1)) #\i)
(test (apply '(1 2 3) '(1)) 2)
(test (apply #(1 2 3) '(2)) 3)
(test (let ((ht (make-hash-table))) (set! (ht "hi") 32) (apply ht '("hi"))) 32)

(test (let ((x (list 1 2))) (set-cdr! x x) (apply + x)) 'error)
(test (apply + '(1 2 . 3)) 'error)
(test (apply + '(1 2) (list 3 4)) 'error)
(test (let () (define (mrec a b) (if (<= b 0) (list a) (apply mrec (list a) (list (- b 1))))) (mrec (list 1 2) 5)) '(((((((1 2))))))))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error))

(test (apply dynamic-wind (list (lambda () #f) (lambda () 1) (lambda () #f))) 1)
(test (apply call-with-exit (list (lambda (exit) 1))) 1)
(test (apply call-with-exit (list (lambda (exit) (exit 1) 32))) 1)
(test (apply catch (list #t (lambda () 1) (lambda args 'error))) 1)
(test (apply eval '((+ 1 2))) 3)
(test (apply eval '()) 'error) ; (eval) is an error -- should it be?
(test (apply eval-string '("(+ 1 2)")) 3)
(test (let () (apply begin '((define x 1) (define y x) (+ x y)))) 2)
(test (apply begin '()) (begin))
(test (apply if '(#f 1 2)) 2)
(test (let ((x 1)) (apply set! '(x 3)) x) 3)
(test (let ((x 1)) (apply cond '(((= x 2) 3) ((= x 1) 32)))) 32)
(test (apply and '((= 1 1) (> 2 3))) #f)
(test (apply and '()) (and))
(test (apply or '((= 1 1) (> 2 3))) #t)
(test (apply or '()) (or))
(test (let () (apply define '(x 32)) x) 32)
(test (let () (apply define* '((hi (a 1) (b 2)) (+ a b))) (hi 32)) 34)
(test ((apply lambda '((n) (+ n 1))) 2) 3)
(test ((apply lambda* '(((n 1)) (+ n 1)))) 2)
(test (apply let '(((x 1)) (+ x 2))) 3)
(test (apply let* '(((x 1) (y (* 2 x))) (+ x y))) 3)
(test (let () (apply define-macro `((hiho a) `(+ ,a 1))) (hiho 2)) 3)
(test (let () (apply defmacro `(hiho (a) `(+ ,a 1))) (hiho 2)) 3)
(test (let () (apply defmacro* `(hiho ((a 2)) `(+ ,a 1))) (hiho)) 3)
(test (let () (apply define-macro* `((hiho (a 2)) `(+ ,a 1))) (hiho)) 3)
(test (apply do '(((i 0 (+ i 1))) ((= i 3) i))) 3)
(test (apply case '(1 ((2 3) 4) ((1 5) 32))) 32)
(test (+ (apply values '(1 2 3))) 6)
(test (apply quote '(1)) 1)
(test (apply quote '()) 'error) ; (quote) is an error
(test (let () (apply letrec '(() (define x 9) x))) 9)
(test ((lambda (n) (apply n '(((x 1)) (+ x 2)))) let) 3)




;;; -------- define --------
;;;
;;; trying to avoid top-level definitions here

(let ()
  (define x 2)
  (test (+ x 1) 3)
  (set! x 4)
  (test (+ x 1) 5)
  (let ()
    (define (tprint x) #t)
    (test (tprint 56) #t)
    (let ()
      (define first car)
      (test (first '(1 2)) 1)
      (let ()
	(define foo (lambda () (define x 5) x))
	(test (foo) 5)
	(let ()
	  (define (foo x) ((lambda () (define x 5) x)) x)
	  (test (foo 88) 88))))))


(test (letrec ((foo (lambda (arg) (or arg (and (procedure? foo) (foo 99)))))) (define bar (foo #f)) (foo #f)) 99)
(test (letrec ((foo 77) (bar #f) (retfoo (lambda () foo))) (define baz (retfoo)) (retfoo)) 77)

(test (let () (define .. 1) ..) 1)

(test (let () (define (hi a) (+ a 1)) (hi 2)) 3)
(test (let () (define (hi a . b) (+ a (cadr b) 1)) (hi 2 3 4)) 7)
(test (let () (define (hi) 1) (hi)) 1)
(test (let () (define (hi . a) (apply + a)) (hi 1 2 3)) 6)

(for-each
 (lambda (arg)
   (test (let () (define x arg) x) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test ((lambda (x) (define (hi a) (+ a 1)) (hi x)) 1) 2)
(test (let ((x 2)) (define f (lambda (y) (+ y x))) (f 3)) 5)
(begin (define r5rstest-plus (lambda (x y) (+ x y))) (define r5rstest-x 32))
(test (r5rstest-plus r5rstest-x 3) 35)

(test (let () (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let ((asdf 1)) (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let () (define (a1 a) (define (a2 a) (define (a3 a) (define (a4 a) (+ a 1)) (+ (a4 a) 1)) (+ (a3 a) 1)) (+ (a2 a) 1)) (a1 0)) 4)

(test (let () (define (hi1 a) (define (hi1 b) (+ b 1)) (hi1 a)) (hi1 1)) 2)
(test (let () (define (hi1 a) (begin (define (hi1 b) (+ b 1))) (hi1 a)) (hi1 1)) 2)
(test (let ((j 0) (k 0))
	(define (hi1 a)
	  (let ((hi1 (lambda (b) 
		       (set! k (+ k 1)) 
		       (hi1 (- b 1)))))
	    (if (<= a 0)
		(list j k)
		(begin
		  (set! j (+ j 1))
		  (hi1 (- a 1))))))
	(hi1 3))
      '(2 2))

(test (procedure? (let () (define (a) a) (a))) #t)

(test (define) 'error)
(test (define*) 'error)
(test (define x) 'error)
(test (define . x) 'error)
(test (define x 1 2) 'error)
(test (define (x 1)) 'error)
(test (define (x)) 'error)
(test (define 1 2) 'error)
(test (define "hi" 2) 'error)
(test (define x 1 2) 'error)
(test (define x 1 . 2) 'error)
(test (define x . 1) 'error)
(test (define x (lambda ())) 'error)
					;(test (define 'hi 1) 'error) ; this redefines quote, which maybe isn't an error
(test (let () (define . 1) 1) 'error)
(test (let () (define func (do () (#t (lambda (y) 2)))) (func 1)) 2)




;;; -------- values, call-with-values --------

(test (call-with-values (lambda () (values 1 2 3)) +) 6)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) b))  5)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) (+ a b))) 9)
(test (call-with-values * -) -1) ; yeah, right... (- (*))
(test (values 1) 1)
(test (call-with-values (lambda () (values 1 2 3 4)) list) (list 1 2 3 4))
(test (+ (values 1) (values 2)) 3)
(test (+ (values '1) (values '2)) 3)
(test (if (values #t) 1 2) 1)
(test (if (values '#t) 1 2) 1)
(test (if (values #f) 1 2) 2)
(test (if (values #f #f) 1 2) 1)
(test (equal? (values #t #t)) #t)
(test (call-with-values (lambda () 4) (lambda (x) x)) 4)
(test (let () (values 1 2 3) 4) 4)
(test (apply + (values '())) 0)
(test (+ (values 1 2 3)) 6)
(test (let ((f (lambda () (values 1 2 3)))) (+ (f))) 6)
(num-test (log (values 8 2)) 3)
(test (* (values 2 (values 3 4))) 24)
(test (* (values (+ (values 1 2)) (- (values 3 4)))) -3)
(test (list (values 1 2) (values 3) 4) '(1 2 3 4))
(test (let ((f1 (lambda (x) (values x (+ x 1)))) (f2 (lambda () (values 2)))) (+ (f1 3) (* 2 (f2)))) 11)
(test (+ (let () (values 1 2)) 3) 6)
(test (let () (values 1 2) 4) 4)
(test (let () + (values 1 2) 4) 4)
(test (string-ref (values "hiho" 2)) #\h)
(test (vector-ref (values (vector 1 2 3)) 1) 2)
(test (+ (values (+ 1 (values 2 3)) 4) 5 (values 6) (values 7 8 (+ (values 9 10) 11))) 66)
(test (+ (if (values) (values 1 2) (values 3 4)) (if (null? (values)) (values 5 6) (values 7 8))) 18) ; (values) is now #<unspecified> 
(test (+ (cond (#f (values 1 2)) (#t (values 3 4))) 5) 12)
(test (+ (cond (#t (values 1 2)) (#f (values 3 4))) 5) 8)
(test (apply + (list (values 1 2))) 3)
(test (apply + (list ((lambda (n) (values n (+ n 1))) 1))) 3)
(test (+ (do ((i 0 (+ i 1))) ((= i 3) (values i (+ i 1))))) 7)
(test (+ (with-input-from-string "(values 1 2 3)" (lambda () (eval (read)))) 2) 8)
(test (< (values 1 2 3)) #t)
(test (apply (values + 1 2) '(3)) 6)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (hi (values 1 2 3))) 7)
(test (+ 1 (eval-string "(values 2 3 4)")) 10)
(test (+ 1 (eval '(values 2 3 4))) 10)
(test (or (values #t) #f) #t)
(test (and (values #t) #f) #f)
(test (let ((x 1)) (set! x (values 32)) x) 32)
(test (let ((x #(32 33))) ((values x) 0)) 32)
(test (let ((x #(32 33))) (set! ((values x) 0) 123) x) #(123 33))
(test (list-ref '(1 (2 3)) (values 1 1)) 3)
(test (list-ref (values '(1 (2 3)) 1 1)) 3)
(test (list-ref ((lambda () (values '(1 (2 3)) 1 1)))) 3)
(test (set! (values) 1) 'error)
(test (+ (values (begin (values 1 2)) (let ((x 1)) (values x (+ x 1))))) 6)
(test (vector 1 (values 2 3) 4) #(1 2 3 4))
(test(+ 1 (values (values (values 2) 3) (values (values (values 4)) 5) 6) 7) 28)

(test (let ((x 1)) (set! x (values)) x) 'error)
(test (let ((x 1)) (set! x (values 1 2 3)) x) 'error)
(test (let ((x 1)) (set! x (values 2)) x) 2)
(test (let ((x 1)) (set! (values x) 2) x) 'error) ; (no generalized set for values, so (values x) is not the same as x
(test (let ((x #(0 1))) (set! (values x 0 32)) x) 'error)
(test (let ((var (values 1 2 3))) var) 'error)
(test (let* ((var (values 1 2 3))) var) 'error)
(test (letrec ((var (values 1 2 3))) var) 'error)
(test (let ((x ((lambda () (values 1 2))))) x) 'error)
(test (+ 1 ((lambda () ((lambda () (values 2 3)))))) 6)

(test (let ((str "hi")) (string-set! (values str 0 #\x)) str) "xi")

(test ((values '(1 (2 3)) 1 1)) 3)
(test (let ((x #(32 33))) ((values x 0))) 32)
(test (+ 1 (apply values '(2 3 4))) 10)
(test (+ 1 ((lambda args (apply values args)) 2 3 4)) 10)
(test (apply begin '(1 2 3)) 3)

(test (or (values #t #f) #f) #t)
(test (or (values #f #f) #f) #f)
(test (or (values #f #t) #f) #t)
(test (or (values #f #f) #t) #t)
(test (or #f (values 1 2)) 1)
(test (+ 1 (or (values 2 3) 4)) 3)
(test (or (values #f 2 3)) 2)
(test (or (values #f 2)) 2)
(test (and (values 1 2 3)) 3)
(test (and (values 1 #f 3)) #f)
(test (+ 1 (and 2 (values 3 4)) 5) 10)
(test (and #t (values 1 2)) 2)
(test (and #t (values #f 3)) #f)
(test (or #f (values 1 2)) 1)
(test (or #f (values #f 2)) 2)
(test (and (values) 1) 1)
(test (length (values '())) 0)
(test (length (values #(1 2 3 4))) 4)
(test (vector? (values #())) #t)
(test (map + (values '(1 2 3) #(1 2 3))) '(2 4 6))
(test (map + (values '(1 2 3)) (values #(1 2 3))) '(2 4 6))
(test (map + (values '(1 2 3) #(4 5 6)) (values '(7 8 9))) '(12 15 18))

(test (let ((x 1)) 
	(and (let () (set! x 2) #f) 
	     (let () (set! x 3) #f)) 
	x) 2)
(test (let ((x 1)) 
	(and (values (let () (set! x 2) #f) 
		     (let () (set! x 3) #f)))
	x) 3)

(test (+ (values 1 2) 3) 6)
(test (+ (values 1 (values 2))) 3)
(test (list (values 1 2)) '(1 2))
(test (+ 6 (values 1 (values 2 3) 4 ) 5) 21)
(test (+ ((lambda (x) (values (+ 1 x))) 2) 3) 6)
(test (list ((lambda (x) (values (+ 1 x))) 2)) '(3))
(test (+ (begin (values 1 2))) 3)
(test (+ 1 (let () (values 1 2))) 4)
(test (apply (values + 1 2) (list 3)) 6)
(test ((lambda* ((a 1) (b 2)) (list a b)) (values :a 3)) '(3 2))
(test (+ (values (values 1 2) (values 4 5))) 12)
(test (+ (begin 3 (values 1 2) 4)) 4)
;;; (test (equal? (values) (if #f #f)) #f)
(test (map (lambda (x) (if #f x (values))) (list 1 2)) '())
(test (map (lambda (x) (if (odd? x) (values x (* x 20)) (values))) (list 1 2 3 4)) '(1 20 3 60))
(test (* 2 (case 1 ((2) (values 3 4)) ((1) (values 5 6)))) 60)
(test (* 2 (case 1 ((2) (values 3 4)) (else (values 5 6)))) 60)
(test (* 2 (case 1 ((1) (values 3 4)) (else (values 5 6)))) 24)
(test (+ (values (* 3 2) (abs (values -1)))) 7)
(test (+ (let ((x 1)) (values x (+ x 1))) (if #f #f (values 2 3))) 8)

(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m p))) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) sum) 45)
(test (map (lambda (n m p) (+ n m p)) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) '(12 15 18))
(test (string-append (values "123" "4" "5") "6" (values "78" "90")) "1234567890")
(test (+ (dynamic-wind (lambda () #f) (lambda () (values 1 2 3)) (lambda () #f)) 4) 10)

(for-each
 (lambda (arg)
   (test (values arg) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (call-with-values (lambda () (values arg arg)) (lambda (a b) b)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-values (lambda () (values "hi" 1 3/2 'a)) (lambda (a b c d) (+ b c))) 5/2)
					;(test (call-with-values values (lambda arg arg)) '())
(test (string-ref (values "hi") 1) #\i)
(test ((lambda (a b) (+ a b)) ((lambda () (values 1 2)))) 3)

(test (list (letrec ((split (lambda (ls)
			      (if (or (null? ls) (null? (cdr ls)))
				  (values ls '())
				  (call-with-values
				      (lambda () (split (cddr ls)))
				    (lambda (odds evens)
				      (values (cons (car ls) odds)
					      (cons (cadr ls) evens))))))))
	      (split '(a b c d e f))))
      '((a c e) (b d f)))

(test (call-with-values (lambda () (call/cc (lambda (k) (k 2 3)))) (lambda (x y) (list x y))) '(2 3))
(test (+ (call/cc (lambda (return) (return (values 1 2 3)))) 4) 10)

(test (let ((values 3)) (+ 2 values)) 5)
(test (let ((a (values 1))) a) 1)

(test (call-with-values (lambda () 2) (lambda (x) x)) 2)
(test (call-with-values (lambda () -1) abs) 1)
(test (call-with-values (lambda () (values -1)) abs) 1)
(test (call-with-values (lambda () (values -1)) (lambda (a) (abs a))) 1)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (- a b)))
      -18)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (+ (* a (call-with-values (lambda () (values 1 2 3)) +))
	     (* b (call-with-values (lambda () (values 1 2 3 4)) *)))))
      612)

(test (call-with-values (lambda (x) (+ x 1)) (lambda (y) y)) 'error)
(test (+ (values . 1)) 'error)
(for-each
 (lambda (arg)
   (test (call-with-values arg arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
(test (call-with-values (lambda () (values -1 2)) abs) 'error)

(test (multiple-value-bind (a b) (values 1 2) (+ a b)) 3)
(test (multiple-value-bind (a) 1 a) 1)
(test (multiple-value-bind (a . rest) (values 1 2 3) (+ a (apply + rest))) 6)
(test (multiple-value-bind a (values 1 2 3) a) '(1 2 3))

(test (let ((a 1)
	    (b 2))
	(multiple-value-set! (a b) (values 32 64))
	(+ a b))
      96)
(test (let ((add (lambda (a b) (values (+ a 1) (+ b 1))))) (+ 1 (add 2 3))) 8)
(test (min (values 1 2) (values 3 0)) 0)
(test ((lambda* ((a 1) (b 2)) (list a b)) (values :b 231)) '(1 231))
(test (cons (values 1 2) (values 3 4)) 'error)

(test (cond ((values) 3) (#t 4)) 3)          ; an error in Guile "zero values returned"
(test (cond ((values (values)) 3) (#t 4)) 3) ; same
(test (+ (cond (#t (values 1 2)))) 3)        ; 1 in guile
(test (+ (cond ((values 3 4) => (lambda (a) a)))) 'error)
(test (+ (cond ((values 3 4) => (lambda (a b) (values a b))))) 7)
(test (+ 1 (cond ((values 2 3))) 4) 10)

(test (case (values 1) ((1) 2) (else 3)) 2)
(test (case (values 1 2) ((1) 2) (else 3)) 3)
(test (case (values 1) (((values 1)) 2) (else 3)) 3)
(test (case (values 1 2) (((values 1 2)) 2) (else 3)) 3)

(test ((values) 0) 'error)
(test ((values "hi") 1) #\i)
(test (string-ref (values "hi") 0) #\h)
(test (string-ref (values "hi" "ho") 0) 'error)
(test (let ((str "hi")) (set! ((values str) 0) #\x) str) "xi")
(test (let ((str "hi")) (string-set! (values str) 0 #\x) str) "xi")
(test (let ((str "hi")) (set! (values str 0) #\x) str) 'error)
(test (let ((str "hi")) (string-set! (values str 0) #\x) str) "xi")

(test ((values 1 2 3) 0) 'error)
(test ((values "hi" "ho") 1) 'error)
(test ((values + 1 2 3)) 6)
(test ((values + 1 2) 3) 6)
(test ((values +) 1 2 3) 6)
(test ((values "hi" 0)) #\h)
(test ((values + 1) (values 2 3) 4) 10)

(test (let ((str "hi")) (set! ((values str 0) 0) #\x) str) 'error)
(test (let ((str "hi")) (set! ((values str) 0) #\x) str) "xi")
(test (+ (let ((x 0)) (do ((i (values 0) (+ i 1))) (((values = i 10)) (values x 2 3)) (set! x (+ x i)))) 4) 54)

(test (map values (list (values 1 2) (values 3 4))) '(1 2 3 4))
(test (let () (define-macro (hi a) `(+ 1 ,a)) (hi (values 2 3 4))) 10)




;;; -------- let, let*, letrec --------

(test (let ((x 2) (y 3)) (* x y)) 6)
(test (let ((x 32)) (let ((x 3) (y x)) y)) 32)
(test (let ((x 32)) (let* ((x 3) (y x)) y)) 3)
(test (let ((x 2) (y 3)) (let ((x 7) (z (+ x y))) (* z x))) 35)
(test (let ((x 2) (y 3)) (let* ((x 7)  (z (+ x y))) (* z x))) 70)
(test (letrec ((even? (lambda (n)  (if (zero? n) #t (odd? (- n 1))))) (odd? (lambda (n)  (if (zero? n) #f (even? (- n 1)))))) (even? 88))  #t)
(test (let loop ((numbers '(3 -2 1 6 -5)) 
		 (nonneg '()) 
		 (neg '())) 
	(cond ((null? numbers) 
	       (list nonneg neg)) 
	      ((>= (car numbers) 0)  
	       (loop (cdr numbers) (cons (car numbers) nonneg)  neg)) 
	      ((< (car numbers) 0)  
	       (loop (cdr numbers) nonneg (cons (car numbers) neg))))) 
      '((6 1 3) (-5 -2)))
(test(let((i 1)(j 2))(+ i j))3)

(test (let ((x 3)) (define x 5) x) 5)
(test (let* () (define x 8) x) 8)
(test (letrec () (define x 9) x) 9)
(test (letrec ((x 3)) (define x 10) x) 10)
(test (let foo () 1) 1)
(test (let ((f -)) (let f ((n (f 1))) n)) -1)
(test (let () 1 2 3 4) 4)

(test (let ((x 1)) (let ((x 32) (y x)) y)) 1)
(test (let ((x 1)) (letrec ((y (if #f x 1)) (x 32)) 1)) 1)
(test (let ((x 1)) (letrec ((y (lambda () (+ 1 x))) (x 32)) (y))) 33) 
(test (let ((x 1)) (letrec ((y (* 0 x)) (x 32)) y)) 'error)
(test (let* ((x 1) (f (letrec ((y (lambda () (+ 1 x))) (x 32)) y))) (f)) 33)
(test (letrec ((x 1) (y (let ((x 2)) x))) (+ x y)) 3)
(test (letrec ((f (lambda () (+ x 3))) (x 2)) (f)) 5)
(test (let* ((x 1) (x 2)) x) 2)
(test (let* ((x 1) (y x)) y) 1)
(test (let ((x 1)) (let ((x 32) (y x)) (+ x y))) 33)
(test (let ((x 1)) (let* ((x 32) (y x)) (+ x y))) 64)
(test (let ((x 'a) (y '(b c))) (cons x y)) '(a b c))
(test (let ((x 0) (y 1)) (let ((x y) (y x)) (list x y))) (list 1 0))
(test (let ((x 0) (y 1)) (let* ((x y) (y x)) (list x y))) (list 1 1))
(test (letrec ((sum (lambda (x) (if (zero? x) 0 (+ x (sum (- x 1))))))) (sum 5)) 15)
(test (let ((divisors (lambda (n) (let f ((i 2)) (cond ((>= i n) '()) ((integer? (/ n i)) (cons i (f (+ i 1)))) (else (f (+ i 1)))))))) (divisors 32)) '(2 4 8 16))
(test (let ((a -1)) (let loop () (if (not (positive? a)) (begin (set! a (+ a 1)) (loop)))) a) 1)
(test (let* ((let 3) (x let)) (+ x let)) 6)
(test (let () (let () (let () '()))) '())
(test (let ((x 1)) (let ((y 0)) (begin (let ((x (* 2 x))) (set! y x))) y)) 2)
(test (let* ((x 1) (x (+ x 1)) (x (+ x 2))) x) 4)
(test (let ((.. 2) (.... 4) (..... +)) (..... .. ....)) 6)

(test (let () (begin (define x 1)) x) 1)
(test (let ((y 1)) (begin (define x 1)) (+ x y)) 2)
(test (let ((: 0)) (- :)) 0)

(test ((let ((x 2))
	 (let ((x 3))
	   (lambda (arg) (+ arg x))))
       1)
      4)

(test ((let ((x 2))
	 (define (inner arg) (+ arg x))
	 (let ((x 32))
	   (lambda (arg) (inner (+ arg x)))))
       1)
      35)

(test ((let ((inner (lambda (arg) (+ arg 1))))
	 (let ((inner (lambda (arg) (inner (+ arg 2)))))
	   inner))
       3)
      6)

(test ((let ()
	 (define (inner arg) (+ arg 1))
	 (let ((inner (lambda (arg) (inner (+ arg 2)))))
	   inner))
       3)
      6)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      28)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((x 2))
	   (lambda (arg) (inner (+ (* 2 arg) x)))))
       3)
      19)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let ((f1 (lambda (arg) (f1 (+ arg 2)))))
	  (f1 1)))
      4)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let* ((f1 (lambda (arg) (f1 (+ arg 2)))))
	  (f1 1)))
      4)

(test (let ((f1 (lambda (arg) (+ arg 1))))
	(let* ((x 32)
	       (f1 (lambda (arg) (f1 (+ x arg)))))
	  (f1 1)))
      34)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let ((x 2)
	       (inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      28)

(test ((let ((x 11))
	 (define (inner arg) (+ arg x))
	 (let* ((x 2)
		(inner (lambda (arg) (inner (+ (* 2 arg) x)))))
	   inner))
       3)
      19)

(test (let ((x 1))
	(let* ((f1 (lambda (arg) (+ x arg)))
	       (x 32))
	  (f1 1)))
      2)

(test (let ((inner (lambda (arg) (+ arg 1))))
	(let ((inner (lambda (arg) (+ (inner arg) 1))))
	  (inner 1)))
      3)
(test (let ((inner (lambda (arg) (+ arg 1))))
	(let* ((inner (lambda (arg) (+ (inner arg) 1))))
	  (inner 1)))
      3)

(test (let ((caller #f)) (let ((inner (lambda (arg) (+ arg 1)))) (set! caller inner)) (caller 1)) 2)
(test (let ((caller #f)) (let ((x 11)) (define (inner arg) (+ arg x)) (set! caller inner)) (caller 1)) 12)

(test (let ((caller #f)) 
	(let ((x 11)) 
	  (define (inner arg) 
	    (+ arg x)) 
	  (let ((y 12))
	    (let ((inner (lambda (arg) 
			   (+ (inner x) y arg)))) ; 11 + 11 + 12 + arg
	      (set! caller inner))))
	(caller 1))
      35)

(test (let ((caller #f)) 
	(let ((x 11)) 
	  (define (inner arg) 
	    (+ arg x)) 
	  (let* ((y 12) 
		 (inner (lambda (arg) 
			  (+ (inner x) y arg)))) ; 11 + 11 + 12 + arg
	    (set! caller inner))) 
	(caller 1))
      35)


(test (let* ((f1 3) (f1 4)) f1) 4)
(test (let ((f1 (lambda () 4))) (define (f1) 3) (f1)) 3)

(test (let ((j -1)
	    (k 0))
	(do ((i 0 (+ i j))
	     (j 1))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3)

(test (let ((j (lambda () -1))
	    (k 0))
	(do ((i 0 (+ i (j)))
	     (j (lambda () 1)))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3)


(test (let ((j (lambda () 0))
	    (k 0))
	(do ((i (j) (j))
	     (j (lambda () 1) (lambda () (+ i 1))))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3) ; 6 in Guile which follows the spec

(test (let ((k 0)) (do ((i 0 (+ i 1)) (j 0 (+ j i))) ((= i 3) k) (set! k (+ k j)))) 1)

#|
(test (let ((j (lambda () 0))
	    (i 2)
	    (k 0))
	(do ((i (j) (j))
	     (j (lambda () i) (lambda () (+ i 1))))
	    ((= i 3) k)
	  (set! k (+ k i))))
      3) ; or 2?

(test (let ((f #f))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (let ()
	    (define (x) i)
	    (if (= i 1) (set! f x))))
	(f))
      1)
|#

(test (let ((x 1))
	(let ()
	  (define (f) x)
	  (let ((x 0))
	    (define (g) (set! x 32) (f))
	    (g))))
      1)

(let ((x 123))
  (define (hi b) (+ b x))
  (let ((x 321))
    (test (hi 1) 124)
    (set! x 322)
    (test (hi 1) 124))
  (set! x 124)
  (test (hi 1) 125)
  (let ((x 321)
	(y (hi 1)))
    (test y 125))
  (let* ((x 321)
	 (y (hi 1)))
    (test y 125))
  (test (hi 1) 125))

(test (let ((j 0)
	    (k 0))
	(let xyz
	    ((i 0))
	  (let xyz
	      ((i 0))
	    (set! j (+ j 1))
	    (if (< i 3)
		(xyz (+ i 1))))
	  (set! k (+ k 1))
	  (if (< i 3)
	      (xyz (+ i 1))))
	(list j k))
      (list 16 4))

(test (let ((x 123)) (begin (define x 0)) x) 0) ; this strikes me as weird, since (let ((x 123) (x 0)) x) is illegal, so...
(test (let ((x 123)) (begin (define (hi a) (+ x a)) (define x 0)) (hi 1)) 1) ; is non-lexical reference?

(for-each
 (lambda (arg)
   (test (let ((x arg)) x) arg))
 (list "hi" -1 #\a "" '() '#() (current-output-port) 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t abs (list 1 2 3) '(1 . 2)))

(test (let ((x 1)) (= 1 (let ((y 2)) (set! x y) x)) (+ x 1)) 3)
(test (let ((x 1)) (let ((xx (lambda (a) (set! x a) a))) (= 1 (xx 2))) (+ x 1)) 3)
(test (let ((x 32)) (begin (define x 123) (define (hi a) (+ a 1))) (hi x)) 124)
(test (let () (begin (define x 123) (define (hi a) (+ a 1))) (hi x)) 124)


					;(let ((initial-chars "aA!$%&*/:<=>?^_~")
					;      (subsequent-chars "9aA!$%&*+-./:<=>?@^_~")
					;      (ctr 0))
					;  (format #t "(let (")
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (format #t "(~A ~D) " (string (string-ref initial-chars i)) ctr)
					;    (set! ctr (+ ctr 1)))
					;
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (do ((k 0 (+ k 1)))
					;	((= k (string-length subsequent-chars)))
					;      (format #t "(~A ~D) " (string (string-ref initial-chars i) (string-ref subsequent-chars k)) ctr)
					;      (set! ctr (+ ctr 1))))
					;
					;  (format #t ")~%  (+ ")
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (format #t "~A " (string (string-ref initial-chars i))))
					;
					;  (do ((i 0 (+ i 1)))
					;      ((= i (string-length initial-chars)))
					;    (do ((k 0 (+ k 1)))
					;	((= k (string-length subsequent-chars)))
					;      (format #t "~A " (string (string-ref initial-chars i) (string-ref subsequent-chars k)))))
					;
					;  (format #t "))~%"))

(num-test (let ((a 0) (A 1) (! 2) ($ 3) (% 4) (& 5) (* 6) (/ 7) (: 8) (< 9) (= 10) (> 11) (? 12) (^ 13) (_ 14) (~ 15) (a9 16) (aa 17) (aA 18) (a! 19) (a$ 20) (a% 21) (a& 22) (a* 23) (a+ 24) (a- 25) (a. 26) (a/ 27) (a: 28) (a< 29) (a= 30) (a> 31) (a? 32) (a@ 33) (a^ 34) (a_ 35) (a~ 36) (A9 37) (Aa 38) (AA 39) (A! 40) (A$ 41) (A% 42) (A& 43) (A* 44) (A+ 45) (A- 46) (A. 47) (A/ 48) (A: 49) (A< 50) (A= 51) (A> 52) (A? 53) (A@ 54) (A^ 55) (A_ 56) (A~ 57) (!9 58) (!a 59) (!A 60) (!! 61) (!$ 62) (!% 63) (!& 64) (!* 65) (!+ 66) (!- 67) (!. 68) (!/ 69) (!: 70) (!< 71) (!= 72) (!> 73) (!? 74) (!@ 75) (!^ 76) (!_ 77) (!~ 78) ($9 79) ($a 80) ($A 81) ($! 82) ($$ 83) ($% 84) ($& 85) ($* 86) ($+ 87) ($- 88) ($. 89) ($/ 90) ($: 91) ($< 92) ($= 93) ($> 94) ($? 95) ($@ 96) ($^ 97) ($_ 98) ($~ 99) (%9 100) (%a 101) (%A 102) (%! 103) (%$ 104) (%% 105) (%& 106) (%* 107) (%+ 108) (%- 109) (%. 110) (%/ 111) (%: 112) (%< 113) (%= 114) (%> 115) (%? 116) (%@ 117) (%^ 118) (%_ 119) (%~ 120) (&9 121) (&a 122) (&A 123) (&! 124) (&$ 125) (&% 126) (&& 127) (&* 128) (&+ 129) (&- 130) (&. 131) (&/ 132) (&: 133) (&< 134) (&= 135) (&> 136) (&? 137) (&@ 138) (&^ 139) (&_ 140) (&~ 141) (*9 142) (*a 143) (*A 144) (*! 145) (*$ 146) (*% 147) (*& 148) (** 149) (*+ 150) (*- 151) (*. 152) (*/ 153) (*: 154) (*< 155) (*= 156) (*> 157) (*? 158) (*@ 159) (*^ 160) (*_ 161) (*~ 162) (/9 163) (/a 164) (/A 165) (/! 166) (/$ 167) (/% 168) (/& 169) (/* 170) (/+ 171) (/- 172) (/. 173) (// 174) (/: 175) (/< 176) (/= 177) (/> 178) (/? 179) (/@ 180) (/^ 181) (/_ 182) (/~ 183) (:9 184) (ca 185) (CA 186) (:! 187) (:$ 188) (:% 189) (:& 190) (:* 191) (:+ 192) (:- 193) (:. 194) (:/ 195) (cc 196) (:< 197) (:= 198) (:> 199) (:? 200) (:@ 201) (:^ 202) (:_ 203) (:~ 204) (<9 205) (<a 206) (<A 207) (<! 208) (<$ 209) (<% 210) (<& 211) (<* 212) (<+ 213) (<- 214) (<. 215) (</ 216) (<: 217) (<< 218) (<= 219) (<> 220) (<? 221) (<@ 222) (<^ 223) (<_ 224) (<~ 225) (=9 226) (=a 227) (=A 228) (=! 229) (=$ 230) (=% 231) (=& 232) (=* 233) (=+ 234) (=- 235) (=. 236) (=/ 237) (=: 238) (=< 239) (== 240) (=> 241) (=? 242) (=@ 243) (=^ 244) (=_ 245) (=~ 246) (>9 247) (>a 248) (>A 249) (>! 250) (>$ 251) (>% 252) (>& 253) (>* 254) (>+ 255) (>- 256) (>. 257) (>/ 258) (>: 259) (>< 260) (>= 261) (>> 262) (>? 263) (>@ 264) (>^ 265) (>_ 266) (>~ 267) (?9 268) (?a 269) (?A 270) (?! 271) (?$ 272) (?% 273) (?& 274) (?* 275) (?+ 276) (?- 277) (?. 278) (?/ 279) (?: 280) (?< 281) (?= 282) (?> 283) (?? 284) (?@ 285) (?^ 286) (?_ 287) (?~ 288) (^9 289) (^a 290) (^A 291) (^! 292) (^$ 293) (^% 294) (^& 295) (^* 296) (^+ 297) (^- 298) (^. 299) (^/ 300) (^: 301) (^< 302) (^= 303) (^> 304) (^? 305) (^@ 306) (^^ 307) (^_ 308) (^~ 309) (_9 310) (_a 311) (_A 312) (_! 313) (_$ 314) (_% 315) (_& 316) (_* 317) (_+ 318) (_- 319) (_. 320) (_/ 321) (_: 322) (_< 323) (_= 324) (_> 325) (_? 326) (_@ 327) (_^ 328) (__ 329) (_~ 330) (~9 331) (~a 332) (~A 333) (~! 334) (~$ 335) (~% 336) (~& 337) (~* 338) (~+ 339) (~- 340) (~. 341) (~/ 342) (~: 343) (~< 344) (~= 345) (~> 346) (~? 347) (~@ 348) (~^ 349) (~_ 350) (~~ 351) )
	    (+ a A ! $ % & * / : < = > ? ^ _ ~ a9 aa aA a! a$ a% a& a* a+ a- a. a/ a: a< a= a> a? a@ a^ a_ a~ A9 Aa AA A! A$ A% A& A* A+ A- A. A/ A: A< A= A> A? A@ A^ A_ A~ !9 !a !A !! !$ !% !& !* !+ !- !. !/ !: !< != !> !? !@ !^ !_ !~ $9 $a $A $! $$ $% $& $* $+ $- $. $/ $: $< $= $> $? $@ $^ $_ $~ %9 %a %A %! %$ %% %& %* %+ %- %. %/ %: %< %= %> %? %@ %^ %_ %~ &9 &a &A &! &$ &% && &* &+ &- &. &/ &: &< &= &> &? &@ &^ &_ &~ *9 *a *A *! *$ *% *& ** *+ *- *. */ *: *< *= *> *? *@ *^ *_ *~ /9 /a /A /! /$ /% /& /* /+ /- /. // /: /< /= /> /? /@ /^ /_ /~ :9 ca CA :! :$ :% :& :* :+ :- :. :/ cc :< := :> :? :@ :^ :_ :~ <9 <a <A <! <$ <% <& <* <+ <- <. </ <: << <= <> <? <@ <^ <_ <~ =9 =a =A =! =$ =% =& =* =+ =- =. =/ =: =< == => =? =@ =^ =_ =~ >9 >a >A >! >$ >% >& >* >+ >- >. >/ >: >< >= >> >? >@ >^ >_ >~ ?9 ?a ?A ?! ?$ ?% ?& ?* ?+ ?- ?. ?/ ?: ?< ?= ?> ?? ?@ ?^ ?_ ?~ ^9 ^a ^A ^! ^$ ^% ^& ^* ^+ ^- ^. ^/ ^: ^< ^= ^> ^? ^@ ^^ ^_ ^~ _9 _a _A _! _$ _% _& _* _+ _- _. _/ _: _< _= _> _? _@ _^ __ _~ ~9 ~a ~A ~! ~$ ~% ~& ~* ~+ ~- ~. ~/ ~: ~< ~= ~> ~? ~@ ~^ ~_ ~~ ))
	  61776)

(test (let ()(+ (let ((x 0) (y 1) (z 2) )(+ x y (let ((x 3) )(+ x (let ()(+ (let ()
									      (+ (let ((x 0) (y 1) (z 2) )(+ x y z (let ((x 3) )(+ x (let ((x 4) (y 5) (z 6) )
																       (+ x y z (let ()(+ (let ((x 7) )(+ x (let ()(+ (let ((x 8) (y 9) )
																							(+ x (let ((x 10) (y 11) (z 12) )(+ x  ))))))))))))))))))))))))))
      50)
(test  (let* ((x 0) (y x) )(+ x y (let ()(+ (let ((x 2) )(+ x (let ()(+ (let ((x 4) )
									  (+ x (let ((x 5) )(+ x (let ((x 6) (y x) (z y) )(+ x (let ((x 7) (y x) )
																 (+ x (let ((x 8) (y x) )(+ x y (let ((x 9) (y x) (z y) )(+ x ))))))))))))))))))))
       48)
(test (let* ((x 0) (y x) )(+ x y (let* ()(+ (let* ((x 2) )(+ x (let* ()(+ (let* ((x 4) )
									    (+ x (let* ((x 5) )(+ x (let* ((x 6) (y x) (z y) )(+ x (let* ((x 7) (y x) )
																     (+ x (let* ((x 8) (y x) )(+ x y (let* ((x 9) (y x) (z y) )(+ x ))))))))))))))))))))
      49)

(test (let ((!@$%^&*~|}{?><.,/`_-+=:! 1)) (+ !@$%^&*~|}{?><.,/`_-+=:! 1)) 2)

(test (let func ((a 1) (b 2)) (set! b a) (if (> b 0) (func (- a 1) b)) b) 1)
(test (let func ((a 1) (b 2)) (set! b a) (if (> b 0) (func (- a 1) b) b)) 0)
(test (let loop ((numbers '(3 -2 1 6 -5))
		 (nonneg '())
		 (neg '()))
	(cond ((null? numbers) (list nonneg neg))
	      ((>= (car numbers) 0)
	       (loop (cdr numbers)
		     (cons (car numbers) nonneg)
		     neg))
	      ((< (car numbers) 0)
	       (loop (cdr numbers)
		     nonneg
		     (cons (car numbers) neg)))))   
      '((6 1 3) (-5 -2)))
(test (let ((b '(1 2 3)))
	(let* ((a b)
	       (b (cons 0 a)))
	  (let b ((a b))
	    (if (null? a)
		'done
		(b (cdr a))))))
      'done)
(test (let lp ((x 100))
	(if (positive? x)
	    (lp (- x 1))
	    x))
      0)
(test (let func ((a 1) (b 2) (c 3)) (+ a b c (if (> a 1) (func (- a 1) (- b 1) (- c 1)) 0))) 6)
(test (let func ((a 1) (b 2) (c 3)) (+ a b c (if (> a 0) (func (- a 1) (- b 1) (- c 1)) 0))) 9)
(test (let func () 1) 1)
(test (let ((a 1)) (let func () (if (> a 1) (begin (set! a (- a 1)) (func)) 0))) 0)
(test (let func1 ((a 1)) (+ (let func2 ((a 2)) a) a)) 3)
(test (let func1 ((a 1)) (+ (if (> a 0) (func1 (- a 1)) (let func2 ((a 2)) (if (> a 0) (func2 (- a 1)) 0))) a)) 1)
(test (let func ((a (let func ((a 1)) a))) a) 1)
(test (let ((i 3)) (let func () (set! i (- i 1)) (if (> i 0) (func))) i) 0)
(test (let func ((a 1)) (define (func a) 2) (func 1)) 2)
(test (let func ((a 1)) (define func (lambda (a) (func a))) (if (> a 1) (func (- a 1)) 0)) 0)
(test (let loop ((i 0)) (let loop ((i 0)) (if (< i 1) (loop (+ i 1)))) i) 0)
(test (let * ((i 0)) (if (< i 1) (* (+ i 1))) i) 0)
(test (let ((j 123)) (define (f g) (set! j 0) (g 0)) (let loop ((i 1)) (if (> i 0) (f loop))) j) 0)
(test (procedure? (let loop () loop)) #t)
(test (let loop1 ((func 0)) (let loop2 ((i 0)) (if (not (procedure? func)) (loop1 loop2)) func)) 0)
(test (let ((k 0)) (let ((x (let xyz ((i 0)) (set! k (+ k 1)) xyz))) (x 0)) k) 2)
(test (let ((hi' 3) (a'b 2)) (+ hi' a'b)) 5)
(test (let ((hi''' 3) (a'''b 2)) (+ hi''' a'''b)) 5)


(let ((enter 0)
      (exit 0)
      (inner 0))
  (define (j1) 
    (set! enter (+ enter 1))
    (let ((result 
	   (let hiho
	       ((i 0))
	     (set! inner (+ inner 1))
	     (if (< i 3) 
		 hiho
		 i))))
      (set! exit (+ exit 1))
      result))

  (let ((j2 (j1)))
    (test (and (procedure? j2) (= enter 1) (= exit 1) (= inner 1)) #t)
    (let ((result (j2 1)))
      (test (and (procedure? result) (= enter 1) (= exit 1) (= inner 2)) #t)
      (set! result (j2 3))
      (test (and (= result 3) (= enter 1) (= exit 1) (= inner 3)) #t))))


(let ()
  (define (block-comment-test a b c)
    (+ a b c))

  (let ((val (block-comment-test 
#|
	    a comment
|#
	    1 #| this is a |# 
#!
            another comment
!#
 2 #! this is b !# 3)))

    (test val 6)))


(test (letrec* ((p (lambda (x)
		     (+ 1 (q (- x 1)))))
		(q (lambda (y)
		     (if (zero? y)
			 0
			 (+ 1 (p (- y 1))))))
		(x (p 5))
		(y x))
	       y)
      5)
(test (letrec ((p (lambda (x)
		     (+ 1 (q (- x 1)))))
		(q (lambda (y)
		     (if (zero? y)
			 0
			 (+ 1 (p (- y 1))))))
		(x (p 5))
		(y x))
	       y)
      'error)
(test (let* ((p (lambda (x)
		     (+ 1 (q (- x 1)))))
		(q (lambda (y)
		     (if (zero? y)
			 0
			 (+ 1 (p (- y 1))))))
		(x (p 5))
		(y x))
	       y)
      'error)

(test (let ((x 1) ((y 2))) x) 'error)
(test (let ((x 1 2 3)) x) 'error)
(test (let ((+ 1 2)) 2) 'error)
(test (let* ((x 1 2)) x) 'error)
(test (letrec ((x 1 2)) x) 'error)
(test (letrec* ((x 1 2)) x) 'error)
(test (let ((x 1 . 2)) x) 'error)
(test (let ((x 1 , 2)) x) 'error)
(test (let ((x . 1)) x) 'error)
(test (let* ((x . 1)) x) 'error)
(test (letrec ((x . 1)) x) 'error)
(test (letrec* ((x . 1)) x) 'error)
(test (let hi ()) 'error)

(test (let) 'error)
(test (let*) 'error)
(test (letrec) 'error)
(test (let . 1) 'error)
(test (let* (x)) 'error)
(test (let (x) 1) 'error)
(test (let ((x)) 3) 'error)
(test (let ((x 1) y) x) 'error)
(test (let* x ()) 'error)
(test (let* ((1 2)) 3) 'error)
(test (let () ) 'error)
(test (let '() 3) 'error)
(test (let* ((x 1))) 'error)
(test (let ((x 1)) (letrec ((x 32) (y x)) (+ 1 y))) 'error) ; #<unspecified> seems reasonable if not the 1+ 
(test (let ((x 1)) (letrec ((y x) (x 32)) (+ 1 y))) 'error)
					;(test (let ((x 1)) (letrec ((y x) (x 32)) 1)) 'error)
(test (let ((x 1)) (letrec ((y (let () (+ x 1))) (x 32)) (+ 1 y))) 'error)
(test (let ((x 1)) (letrec ((y (let ((xx (+ x 1))) xx)) (x 32)) (+ 1 y))) 'error)
					;(test (let ((x 32)) (letrec ((y (apply list `(* ,x 2))) (x 1)) y)) 'error)
(test (letrec) 'error)
(test (letrec*) 'error)
(test (let ((x . 1)) x) 'error)
(test (letrec* ((and #2D((1 2) (3 4)) 3/4))) 'error)
(test (letrec* ((hi "" #\a))) 'error)

(test (let (((x 1)) 2) 3) 'error)
(test (let ((#f 1)) #f) 'error)
(test (let (()) #f) 'error)
(test (let (lambda () ) #f) 'error)
(test (let ((f1 3) (f1 4)) f1) 'error) ; not sure about this
;;   (let () (define (f1) 3) (define (f1) 4) (f1))
(test (let ((asdf (lambda (a) (if (> a 0) (asdf (- a 1)) 0)))) (asdf 3)) 'error)
(test (let* ((asdf (lambda (a) (if (> a 0) (asdf (- a 1)) 0)))) (asdf 3)) 'error)
(test (let (('a 3)) 1) 'error)
(test (let ((#\a 3)) #\a) 'error)
;;      (test (let ((#z1 2)) 1) 'error)
(test (let ('a 3) 1) 'error)
(test (let 'a 1) 'error)
;; what about: (let ('1 ) quote) -> 1
(test (let* func ((a 1)) a) 'error)
(test (letrec func ((a 1)) a) 'error)
(test (letrec* func ((a 1)) a) 'error)

(test (let ((1 3)) 3) 'error)
(test (let ((#t 3)) 3) 'error)
(test (let ((() 3)) 3) 'error)
(test (let ((#\c 3)) 3) 'error)
(test (let (("hi" 3)) 3) 'error)
;(test (let ((:hi 3)) 3) 'error)

(test (let 1 ((i 0)) i) 'error)
(test (let #f ((i 0)) i) 'error)
(test (let "hi" ((i 0)) i) 'error)
(test (let #\c ((i 0)) i) 'error)
;(test (let :hi ((i 0)) i) 'error)

(test (let func ((a 1) . b) a) 'error)
(test (let func ((a 1) . b) (if (> a 0) (func (- a 1) 2 3) b)) 'error)
(test (let func ((a . 1)) a) 'error)
(test (let func (a . 1) a) 'error)
(test (let ((a 1) . b) a) 'error)
(test (let* ((a 1) . b) a) 'error)
(test (let func ((a func) (i 1)) i) 'error)
(test (let func ((i 0)) (if (< i 1) (func))) 'error)
(test (let func (let ((i 0)) (if (< i 1) (begin (set! i (+ i 1)) (func))))) 'error)
(test (let ((x 0)) (set! x (+ x 1)) (begin (define y 1)) (+ x y)) 2)
(test (let loop loop) 'error)
(test (let loop (loop)) 'error)
(test (let loop ((i 0) (loop 1)) i) 'error)

(test (letrec ((cons 1 (quote ())) . #(1)) 1) 'error)
(test (letrec ((a 1) . 2) 1) 'error)
(test (let* ((a 1) (b . 2) . 1) (())) 'error)
(test (let "" 1) 'error)
(test (let "hi" 1) 'error)
(test (let #(1) 1) 'error)
(test (let __hi__ #t) 'error)
(test (let* hi () 1) 'error)
(test (letrec (1 2) #t) 'error)
(test (letrec* (1 2) #t) 'error)

;;; these ought to work, but see s7.c under EVAL: (it's a speed issue)
;(test (let let ((i 0)) (if (< i 3) (let (+ i 1)) i)) 3)
;(test (let () (define (if a) a) (if 1)) 1)
;(test (let begin ((i 0)) (if (< i 3) (begin (+ i 1)) i)) 3)


;;; from the scheme wiki
;;; http://community.schemewiki.org/?sieve-of-eratosthenes

(let ((results '(2)))
  (define (primes n) 
    (let ((pvector (make-vector (+ 1 n) #t))) ; if slot k then 2k+1 is a prime 
      (let loop ((p 3) ; Maintains invariant p = 2j + 1 
		 (q 4) ; Maintains invariant q = 2j + 2jj 
		 (j 1) 
		 (k '()) 
		 (vec pvector)) 
	(letrec ((lp (lambda (p q j k vec) 
		       (loop (+ 2 p) 
			     (+ q (- (* 2 (+ 2 p)) 2)) 
			     (+ 1 j) 
			     k 
			     vec))) 
		 (eradicate (lambda (q p vec) 
			      (if (<= q n) 
				  (begin (vector-set! vec q #f) 
					 (eradicate (+ q p) p vec)) 
				  vec)))) 
          (if (<= j n) 
	      (if (eq? #t (vector-ref vec j)) 
		  (begin (set! results (cons p results))
			 (lp p q j q (eradicate q p vec))) 
		  (lp p q j k vec)) 
	      (reverse results))))))
  (test (primes 10) '(2 3 5 7 11 13 17 19)))

(test (let ((gvar 32)) (define (hi1 a) (+ a gvar)) (let ((gvar 0)) (hi1 2))) 34)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (hi2 a) gvar)) (let ((gvar 0)) (hi1 2))) 96)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ a gvar)) (let ((gvar 0)) (hi1 (hi2 2)))) 32)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (a 2) gvar)) (let ((gvar 0)) (define (hi2 a) (* a 2)) (hi1 hi2))) 36)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (a 2) gvar)) (let ((gvar 0) (hi2 (lambda (a) (hi2 a)))) (hi1 hi2))) 96)
(test (let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (a 2) gvar)) (let* ((gvar 0) (hi2 (lambda (a) (hi2 a)))) (hi1 hi2))) 32)
(test (let () ((let ((gvar 32)) (define-macro (hi2 b) `(* gvar ,b)) (define (hi1 a) (+ (hi2 2) gvar)) hi1) 2)) 96)
(test (let ((gvar 0)) ((let ((gvar 1)) (define-macro (hi2 b) `(+ gvar ,b)) (define (hi1 a) (let ((gvar 2)) (hi2 a))) hi1) 2)) 4)
(test (let ((gvar 0)) (define-macro (hi2 b) `(+ gvar ,b)) ((let ((gvar 1)) (define (hi1 a) (let ((gvar 2)) (a 2))) hi1) hi2)) 4)
(test (let ((gvar 0)) (define-macro (hi2 b) `(+ gvar ,b)) ((let ((gvar 1)) (define (hi1 a) (a 2)) hi1) hi2)) 3)
(test (let () (define-macro (hi2 b) `(+ gvar ,b)) ((let ((gvar 1)) (define (hi1 a) (a 2)) hi1) hi2)) 3)

(test (let loop ((lst (list 1 2)) 
		 (i 0) 
		 (sum 0))
	(if (or (null? lst)
		(> i 10))
	    sum
	    (begin
	      (set-cdr! (cdr lst) lst)
	      (loop (cdr lst) (+ i 1) (+ sum (car lst))))))
      16)

;;; these are confusing:
;(letrec ((if 0.0)) ((lambda () (if #t "hi")))) -> "hi"
;(let ((let 0)) let) -> 0
;(let* ((lambda 0)) ((lambda () 1.5))) -> 1.5
;(let* ((lambda 0)) lambda) -> 0

;; from test-submodel.scm, from MIT I think
(test (letrec ((factorial
		(lambda (n)
		  (if (<= n 0) 1 (* n (factorial (- n 1)))))))
	(factorial 3))
      6)

(test (letrec ((iter-fact
		(lambda (n)
		  (letrec
		      ((helper (lambda (n p)
				 (if (<= n 0) p (helper (- n 1) (* n p))))))
		    (helper n 1)))))
	(iter-fact 3))
      6)

(test (letrec ((y-factorial
		(lambda (n)
		  (letrec ((y
			    (lambda (f)
			      ((lambda (x)
				 (f (lambda (z) ((x x) z))))
			       (lambda (x)
				 (f (lambda (z) ((x x) z)))))))
			   (fact-def
			    (lambda (fact)
			      (lambda (n)
				(if (<= n 0)
				    1
				    (* n (fact (- n 1))))))))
		    ((y fact-def) n)))))
	(y-factorial 3))
      6)

(test (letrec ((x 1) (y x)) (list x y)) '(1 #<undefined>)) ; guile says '(1 1)
(test (letrec ((y x) (x 1)) (list x y)) '(1 #<undefined>)) ; guile says '(1 1)
(test (letrec ((x 1) (y (let () (set! x 2) x))) (list x y)) '(1 2))
(test (letrec ((history (list 9))) ((lambda (n) (begin (set! history (cons history n)) history)) 8)) '((9) . 8))
(test (((call/cc (lambda (k) k)) (lambda (x) x)) 'HEY!) 'HEY!)

(let ((sequence '()))
  ((call-with-current-continuation
    (lambda (goto)
      (letrec ((start
		(lambda ()
		  (begin (set! sequence (cons 'start sequence))
			 (goto next))))
	       (froz
		(lambda ()
		  (begin (set! sequence (cons 'froz sequence))
			 (goto last))))
	       (next
		(lambda ()
		  (begin (set! sequence (cons 'next sequence))
			 (goto froz))))
	       (last
		(lambda ()
		  (begin (set! sequence (cons 'last sequence))
			 #f))))
	start))))
  (test (reverse sequence) '(start next froz last)))

(let ()
  (define thunk 'dummy-thunk)

  (define (make-fringe-thunk tree)
    (call-with-exit
     (lambda (return-to-repl)
       (cond ((pair? tree) (begin (make-fringe-thunk (car tree))
				  (make-fringe-thunk (cdr tree))))
	     ((null? tree) (begin (set! thunk (lambda () 'done)) 'null))
	     (else (call/cc
		    (lambda (cc)
		      (begin
			(set! thunk
			      (lambda ()
				(begin (display tree) (cc 'leaf))))
			(return-to-repl 'thunk-set!)))))))))

  (define tr '(() () (((1 (( (() 2 (3 4)) (((5))) )) ))) ))
  (test (make-fringe-thunk tr) 'null)
  (test (thunk) 'done))




;;; -------- call/cc --------
;;;
;;; some of these were originally from Al Petrovsky, Scott G Miller, Matthias Radestock, J H Brown, Dorai Sitaram, 
;;;   and probably others.

(let ((calls (make-vector 3 #f))
      (travels (make-vector 5 0))
      (ctr 0))
  (set! (travels 0) (+ (travels 0) 1))
  (call/cc (lambda (c0) (set! (calls 0) c0)))
  (set! (travels 1) (+ (travels 1) 1))
  (call/cc (lambda (c1) (set! (calls 1) c1)))
  (set! (travels 2) (+ (travels 2) 1))
  (call/cc (lambda (c2) (set! (calls 2) c2)))
  (set! (travels 3) (+ (travels 3) 1))
  (let ((ctr1 ctr))
    (set! ctr (+ ctr1 1))
    (if (< ctr1 3)
	((calls ctr1) ctr1)))
  (set! (travels 4) (+ (travels 4) 1))
  (test travels #(1 2 3 4 1)))

(let ((calls (make-vector 5 #f))
      (travels (make-vector 5 0))
      (ctr2 0))
  (let loop ((ctr 0))
    (if (< ctr 3)
	(begin
	  (set! (travels ctr) (+ (travels ctr) 1))
	  (call/cc (lambda (c0) (set! (calls ctr) c0)))
	  (loop (+ ctr 1)))))
  (set! (travels 3) (+ (travels 3) 1))
  (let ((ctr1 ctr2))
    (set! ctr2 (+ ctr1 1))
    (if (< ctr1 3)
	((calls ctr1) ctr1)))
  (set! (travels 4) (+ (travels 4) 1))
  (test travels #(1 2 3 4 1)))

(let ((c1 #f)
      (c2 #f)
      (c3 #f)
      (x0 0)
      (x1 0)
      (x2 0)
      (x3 0))
  (let ((x (+ 1 
	      (call/cc
	       (lambda (r1)
		 (set! c1 r1)
		 (r1 2)))
	      (call/cc
	       (lambda (r2)
		 (set! c2 r2)
		 (r2 3)))
	      (call/cc
	       (lambda (r3)
		 (set! c3 r3)
		 (r3 4)))
	      5)))
    (if (= x0 0) 
	(set! x0 x)
	(if (= x1 0)
	    (set! x1 x)
	    (if (= x2 0)
		(set! x2 x)
		(if (= x3 0)
		    (set! x3 x)))))
    (if (= x 15)
	(c1 6))
    (if (= x 19)
	(c2 7))
    (if (= x 23)
	(c3 8))
    (test (list x x0 x1 x2 x3) '(27 15 19 23 27))))

(let ((c1 #f) (c2 #f) (c3 #f) (x0 0) (x1 0) (x2 0) (x3 0) (y1 0) (z0 0) (z1 0) (z2 0) (z3 0))
  (let* ((y 101)
	 (x (+ y 
	      (call/cc
	       (lambda (r1)
		 (set! c1 r1)
		 (r1 2)))
	      (call/cc
	       (lambda (r2)
		 (set! c2 r2)
		 (r2 3)))
	      (call/cc
	       (lambda (r3)
		 (set! c3 r3)
		 (r3 4)))
	      5))
	 (z (+ x y)))
    (set! y1 y)
    (if (= x0 0) 
	(begin
	  (set! x0 x)
	  (set! z0 z))
	(if (= x1 0)
	    (begin
	      (set! x1 x)
	      (set! z1 z))
	    (if (= x2 0)
		(begin
		  (set! x2 x)
		  (set! z2 z))
		(if (= x3 0)
		    (begin
		      (set! x3 x)
		      (set! z3 z))))))
    (if (= x 115)
	(c1 6))
    (if (= x 119)
	(c2 7))
    (if (= x 123)
	(c3 8))
    (test (list x x0 x1 x2 x3 y1 z0 z1 z2 z3) '(127 115 119 123 127 101 216 220 224 228))))

(let ((c1 #f)
      (c2 #f)
      (c3 #f)
      (x0 0)
      (x1 0)
      (x2 0)
      (x3 0))
  (let ((x (+ 1 
	      (call/cc
	       (lambda (r1)
		 (set! c1 r1)
		 (r1 2)))
	      (call/cc
	       (lambda (r2)
		 (set! c2 r2)
		 (r2 3)))
	      (call/cc
	       (lambda (r3)
		 (set! c3 r3)
		 (r3 4)))
	      5)))
    (if (= x0 0) 
	(set! x0 x)
	(if (= x1 0)
	    (set! x1 x)
	    (if (= x2 0)
		(set! x2 x)
		(if (= x3 0)
		    (set! x3 x)))))
    (if (= x 15)
	(c1 6 1))
    (if (= x 20)
	(c2 7 2 3))
    (if (= x 29)
	(c3 8 3 4 5))
    (test (list x x0 x1 x2 x3) '(45 15 20 29 45))))
;; 45 = (+ 1 6 1 7 2 3 8 3 4 5 5)

(let ((x 0)
      (c1 #f)
      (results '()))
  (set! x (call/cc
	   (lambda (r1)
	     (set! c1 r1)
	     (r1 2))))
  (set! results (cons x results))
  (if (= x 2) (c1 32))
  (test results '(32 2)))

(let ((x #(0))
      (y #(0))
      (c1 #f))
  (set! ((call/cc
	   (lambda (r1)
	     (set! c1 r1)
	     (r1 x)))
	 0) 32)
  (if (= (y 0) 0) (c1 y))
  (test (and (equal? x #(32)) (equal? y #(32))) #t))



(let* ((next-leaf-generator (lambda (obj eot)
			      (letrec ((return #f)
				       (cont (lambda (x)
					       (recur obj)
					       (set! cont (lambda (x) (return eot)))
					       (cont #f)))
				       (recur (lambda (obj)
						(if (pair? obj)
						    (for-each recur obj)
						    (call-with-current-continuation
						     (lambda (c)
						       (set! cont c)
						       (return obj)))))))
				(lambda () (call-with-current-continuation
					    (lambda (ret) (set! return ret) (cont #f)))))))
       (leaf-eq? (lambda (x y)
		   (let* ((eot (list 'eot))
			  (xf (next-leaf-generator x eot))
			  (yf (next-leaf-generator y eot)))
		     (letrec ((loop (lambda (x y)
				      (cond ((not (eq? x y)) #f)
					    ((eq? eot x) #t)
					    (else (loop (xf) (yf)))))))
		       (loop (xf) (yf)))))))
  
  (test (leaf-eq? '(a (b (c))) '((a) b c)) #t)
  (test (leaf-eq? '(a (b (c))) '((a) b c d)) #f))

(test (let ((r #f)
	    (a #f)
	    (b #f)
	    (c #f)
	    (i 0))
	(let () 
	  (set! r (+ 1 (+ 2 (+ 3 (call/cc (lambda (k) (set! a k) 4))))
		     (+ 5 (+ 6 (call/cc (lambda (k) (set! b k) 7))))))
	  (if (not c) 
	      (set! c a))
	  (set! i (+ i 1))
	  (case i
	    ((1) (a 5))
	    ((2) (b 8))
	    ((3) (a 6))
	    ((4) (c 4)))
	  r))
      28)

(test (let ((r #f)
	    (a #f)
	    (b #f)
	    (c #f)
	    (i 0))
	(let () 
	  (set! r (+ 1 (+ 2 (+ 3 (call/cc (lambda (k) (set! a k) 4))))
		     (+ 5 (+ 6 (call/cc (lambda (k) (set! b k) 7))))))
	  (if (not c) 
	      (set! c a))
	  (set! i (+ i 1))
	  (case i
	    ((1) (b 8))
	    ((2) (a 5))
	    ((3) (b 7))
	    ((4) (c 4)))
	  r))
      28)

(test (let ((k1 #f)
	    (k2 #f)
	    (k3 #f)
	    (state 0))
	(define (identity x) x)
	(define (fn)
	  ((identity (if (= state 0)
			 (call/cc (lambda (k) (set! k1 k) +))
			 +))
	   (identity (if (= state 0)
			 (call/cc (lambda (k) (set! k2 k) 1))
			 1))
	   (identity (if (= state 0)
			 (call/cc (lambda (k) (set! k3 k) 2))
			 2))))
	(define (check states)
	  (set! state 0)
	  (let* ((res '())
		 (r (fn)))
	    (set! res (cons r res))
	    (if (null? states)
		res
		(begin (set! state (car states))
		       (set! states (cdr states))
		       (case state
			 ((1) (k3 4))
			 ((2) (k2 2))
			 ((3) (k1 -)))))))
	(map check '((1 2 3) (1 3 2) (2 1 3) (2 3 1) (3 1 2) (3 2 1))))
      '((-1 4 5 3) (4 -1 5 3) (-1 5 4 3) (5 -1 4 3) (4 5 -1 3) (5 4 -1 3)))

(let ((c1 #f))
  (let ((x ((call/cc (lambda (r1) (set! c1 r1) (r1 "hiho"))) 0)))
    (if (char=? x #\h)
	(c1 "asdf"))
    (test x #\a)))

(test (let ((x '())
	    (y 0))
	(call/cc 
	 (lambda (escape)
	   (let* ((yin ((lambda (foo) 
			  (set! x (cons y x))
			  (if (= y 10)
			      (escape x)
			      (begin
				(set! y 0)
				foo)))
			(call/cc (lambda (bar) bar))))
		  (yang ((lambda (foo) 
			   (set! y (+ y 1))
			   foo)
			 (call/cc (lambda (baz) baz)))))
	     (yin yang)))))
      '(10 9 8 7 6 5 4 3 2 1 0))

(test (let ((c #f))
	(let ((r '()))
	  (let ((w (let ((v 1))
		     (set! v (+ (call-with-current-continuation
				 (lambda (c0) (set! c c0) v))
				v))
		     (set! r (cons v r))
		     v)))
	    (if (<= w 1024) (c w) r))))
      '(2048 1024 512 256 128 64 32 16 8 4 2))

(test (let ((cc #f)
	    (r '()))
	(let ((s (list 1 2 3 4 (call/cc (lambda (c) (set! cc c) 5)) 6 7 8)))
	  (if (null? r)
	      (begin (set! r s) (cc -1))
	      (list r s))))
      '((1 2 3 4 5 6 7 8) (1 2 3 4 -1 6 7 8)))

(test (let ((count 0))
        (let ((first-time? #t)
              (k (call/cc values)))
          (if first-time?
              (begin
                (set! first-time? #f)
                (set! count (+ count 1))
                (k values))
              (void)))
        count)
      2)

(test (procedure? (call/cc call/cc)) #t)
(test (call/cc (lambda (c) (0 (c 1)))) 1)
(test (call/cc (lambda (k) (k "foo"))) "foo")
(test (call/cc (lambda (k) "foo")) "foo")
(test (call/cc (lambda (k) (k "foo") "oops")) "foo")
(test (call/cc (lambda (return) (catch #t (lambda () (error 'hi "")) (lambda args (return "oops"))))) "oops")
(test (call/cc (lambda (return) (catch #t (lambda () (return 1)) (lambda args (return "oops"))))) 1)
(test (catch #t (lambda () (call/cc (lambda (return) (return "oops")))) (lambda arg 1)) "oops")
(test (call/cc (if (< 2 1) (lambda (return) (return 1)) (lambda (return) (return 2) 3))) 2)
(test (call/cc (let ((a 1)) (lambda (return) (set! a (+ a 1)) (return a)))) 2)
(test (call/cc (lambda (return) (let ((hi return)) (hi 2) 3))) 2)
(test (let () (define (hi) (call/cc func)) (define (func a) (a 1)) (hi)) 1)
(test (((call/cc (call/cc call/cc)) call/cc) (lambda (a) 1)) 1)
(test (+ 1 (eval-string "(+ 2 (call-with-exit (lambda (return) (return 3))) 4)") 5) 15)
(test (+ 1 (eval '(+ 2 (call-with-exit (lambda (return) (return 3))) 4)) 5) 15)
(test (call-with-exit) 'error)
(test (call-with-exit s7-version s7-version) 'error)
(test (call/cc) 'error)
(test (call/cc s7-version s7-version) 'error)
(test (call/cc (lambda () 1)) 'error)
(test (call/cc (lambda (a b) (a 1))) 'error)

;;; guile/s7 accept: (call/cc (lambda (a . b) (a 1))) -> 1
;;; same:            (call/cc (lambda arg ((car arg) 1))) -> 1

(test (let ((listindex (lambda (e l)
			 (call/cc (lambda (not_found)
				    (letrec ((loop 
					      (lambda (l)
						(cond
						 ((null? l) (not_found #f))
						 ((equal? e (car l)) 0)
						 (else (+ 1 (loop (cdr l))))))))
				      (loop l)))))))
	(listindex 1 '(0 3 2 4 8)))
      #f)

(test (let ((product (lambda (li)
		       (call/cc (lambda (break)
				  (let loop ((l li))
				    (cond
				     ((null? l) 1)
				     ((= (car l) 0) (break 0))
				     (else (* (car l) (loop (cdr l)))))))))))
	(product '(1 2 3 0 4 5 6)))
      0)

(test (let ((lst '()))
	((call/cc
	  (lambda (goto)
	    (letrec ((start (lambda () (set! lst (cons "start" lst)) (goto next)))
		     (next  (lambda () (set! lst (cons "next" lst))  (goto last)))
		     (last  (lambda () (set! lst (cons "last" lst)) (reverse lst))))
	      start)))))
      '("start" "next" "last"))

(test (let ((cont #f))
	(letrec ((x (call-with-current-continuation (lambda (c) (set! cont c) 0)))
		 (y (call-with-current-continuation (lambda (c) (set! cont c) 0))))
	  (if cont
	      (let ((c cont))
		(set! cont #f)
		(set! x 1)
		(set! y 1)
		(c 0))
	      (+ x y))))
      0)

(test (letrec ((x (call-with-current-continuation
		   (lambda (c)
		     (list #t c)))))
	(if (car x)
	    ((cadr x) (list #f (lambda () x)))
	    (eq? x ((cadr x)))))
      #t)

(test (call/cc (lambda (c) (0 (c 1)))) 1)

(test (let ((member (lambda (x ls)
		      (call/cc
		       (lambda (break)
			 (do ((ls ls (cdr ls)))
			     ((null? ls) #f)
			   (if (equal? x (car ls))
			       (break ls))))))))
	(list (member 'd '(a b c))
	      (member 'b '(a b c))))
      '(#f (b c)))

(test (+ 2 (call/cc (lambda (k) (* 5 (k 4))))) 6)
(test (+ 2 (call/cc (lambda (k) (* 5 (k 4 5 6))))) 17)
(test (+ 2 (call/cc (lambda (k) (* 5 (k (values 4 5 6)))))) 17)
(test (+ 2 (call/cc (lambda (k) (* 5 (k 1 (values 4 5 6)))))) 18)
(test (+ 2 (call/cc (lambda (k) (* 5 (k 1 (values 4 5 6) 1))))) 19)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 4))))) 6)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 4 5 6))))) 17)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k (values 4 5 6)))))) 17)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 1 (values 4 5 6)))))) 18)
(test (+ 2 (call-with-exit (lambda (k) (* 5 (k 1 (values 4 5 6) 1))))) 19)

(test (+ 2 (values 3 (call-with-exit (lambda (k1) (k1 4))) 5)) 14)
(test (+ 2 (call-with-exit (lambda (k1) (values 3 (k1 4) 5))) 8) 14)
(test (+ 2 (call-with-exit (lambda (k1) (values 3 (k1 4 -3) 5))) 8) 11)

(test (call-with-exit (let () (lambda (k1) (k1 2)))) 2)
(test (+ 2 (call/cc (let () (call/cc (lambda (k1) (k1 (lambda (k2) (k2 3)))))))) 5)
(test (+ 2 (call/cc (call/cc (lambda (k1) (k1 (lambda (k2) (k2 3))))))) 5)
(test (call-with-exit (lambda arg ((car arg) 32))) 32)
(test (call-with-exit (lambda arg ((car arg) 32)) "oops!") 'error)
(test (call-with-exit (lambda (a b) a)) 'error)

(test (let ((x (call/cc (lambda (k) k))))
	(x (lambda (y) "hi")))
      "hi")

(test (((call/cc (lambda (k) k)) (lambda (x) x)) "hi") "hi")

(test (let ((return #f)
	    (lst '()))
	(let ((val (+ 1 (call/cc 
			 (lambda (cont) 
			   (set! return cont) 
			   1)))))
	  (set! lst (cons val lst)))
	(if (= (length lst) 1)
	    (return 10)
	    (if (= (length lst) 2)
		(return 20)))
	(reverse lst))
      '(2 11 21))

(test (let ((r1 #f)
	    (r2 #f)
	    (lst '()))
	(define (somefunc x y)
	  (+ (* 2 (expt x 2)) (* 3 y) 1))
	(let ((val (somefunc (call/cc
			      (lambda (c1)
				(set! r1 c1)
				(c1 1)))
			     (call/cc
			      (lambda (c2)
				(set! r2 c2)
				(c2 1))))))
	  (set! lst (cons val lst)))
	(if (= (length lst) 1)
	    (r1 2)
	    (if (= (length lst) 2)
		(r2 3)))
	(reverse lst))
      '(6 12 18))

(let ((tree->generator
       (lambda (tree)
	 (let ((caller '*))
	   (letrec
	       ((generate-leaves
		 (lambda ()
		   (let loop ((tree tree))
		     (cond ((null? tree) 'skip)
			   ((pair? tree)
			    (loop (car tree))
			    (loop (cdr tree)))
			   (else
			    (call/cc
			     (lambda (rest-of-tree)
			       (set! generate-leaves
				     (lambda ()
				       (rest-of-tree 'resume)))
			       (caller tree))))))
		   (caller '()))))
	     (lambda ()
	       (call/cc
		(lambda (k)
		  (set! caller k)
		  (generate-leaves)))))))))
  (let ((same-fringe? 
	 (lambda (tree1 tree2)
	   (let ((gen1 (tree->generator tree1))
		 (gen2 (tree->generator tree2)))
	     (let loop ()
	       (let ((leaf1 (gen1))
		     (leaf2 (gen2)))
		 (if (eqv? leaf1 leaf2)
		     (if (null? leaf1) #t (loop))
		     #f)))))))
    
    (test (same-fringe? '(1 (2 3)) '((1 2) 3)) #t)
    (test (same-fringe? '(1 2 3) '(1 (3 2))) #f)))


(for-each
 (lambda (arg)
   (test (let ((ctr 0))
	   (let ((val (call/cc (lambda (exit)
				 (do ((i 0 (+ i 1)))
				     ((= i 10) 'gad)
				   (set! ctr (+ ctr 1))
				   (if (= i 1)
				       (exit arg)))))))
	     (and (equal? val arg)
		  (= ctr 2))))
	 #t))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (let ((ctr 0))
	   (let ((val (call/cc (lambda (exit)
				 (do ((i 0 (+ i 1)))
				     ((= i 10) arg)
				   (set! ctr (+ ctr 1))
				   (if (= i 11)
				       (exit 'gad)))))))
	     (and (equal? val arg)
		  (= ctr 10))))
	 #t))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (let ((c #f)
	    (r (string-copy "testing-hiho")))
	(let ((v (call/cc (lambda (c0) (set! c c0) (list #\a 0)))))
	  (let ((chr (car v))
		(index (cadr v)))
	    (string-set! r index chr)
	    (set! index (+ index 1))
	    (if (<= index 8) 
		(c (list (integer->char (+ 1 (char->integer chr))) index)) 
		r))))
      "abcdefghiiho")

(test (let ((x 0)
	    (again #f))
	(call/cc (lambda (r) (set! again r)))
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let* ((x 0)
	     (again #f)
	     (func (lambda (r) (set! again r))))
	(call/cc func)
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let* ((x 0)
	     (again #f))
	(call/cc (let ()
		   (lambda (r) (set! again r))))
	(set! x (+ x 1))
	(if (< x 3) (again))
	x)
      3)

(test (let ((x 0)
	    (xx 0))
	(let ((cont #f))
	  (call/cc (lambda (c) (set! xx x) (set! cont c)))
	  (set! x (+ x 1))
	  (if (< x 3)	(cont))
	  xx))
      0)

(test (call/cc procedure?) #t)
(test (procedure? (call/cc (lambda (a) a))) #t)

(for-each
 (lambda (arg)
   (test (call/cc (lambda (a) arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((a (call/cc (lambda (a) a))))
  (test (eq? a a) #t)
  (test (eqv? a a) #t)
  (test (equal? a a) #t)
  (for-each
   (lambda (ques)
     (if (ques a)
	 (format #t "(~A ~A) returned #t?~%" ques a)))
   question-ops))

(test (let ((conts (make-vector 4 #f)))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (vector-set! conts 0 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 1 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 2 a) 0))
			     (call/cc (lambda (a) (vector-set! conts 3 a) 0)))
			  lst))
	  (let ((len (length lst)))
	    (if (< len 4)
		((vector-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      '(0 2 5 9))

(test (let ((conts '()))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
			     (* (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
				(+ (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1))
				   (* (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) 1)) 2))))
			  lst))
	  (let ((len (length lst)))
	    (if (<= len 4)
		((list-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
					; (+ 1 (* 1 (+ 1 (* 1 2)))) to start
					; (+ 1 ...          2     )
					; (+ 1 ...     3    [1]   )
					; (+ 1 ...4    [1]        )
					; (+ 5   [1]              )
      '(4 6 6 13 8))

(test (let ((conts (make-vector 4 #f)))
	(let ((lst '()))
	  (set! lst (cons (+ (call/cc (lambda (a) (if (not (vector-ref conts 0)) (vector-set! conts 0 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 1)) (vector-set! conts 1 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 2)) (vector-set! conts 2 a)) 0))
			     (call/cc (lambda (a) (if (not (vector-ref conts 3)) (vector-set! conts 3 a)) 0)))
			  lst))
	  (let ((len (length lst)))
	    (if (< len 4)
		((vector-ref conts (- len 1)) (+ len 1))
		(reverse lst)))))
      '(0 2 3 4))

(test (let ((conts '()))
	(let ((lst '()))
	  (set! lst (cons (+ (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 1 0)
			     (* (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 2 1)
				(+ (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 1 0)
				   (* (if (call/cc (lambda (a) (if (< (length conts) 4) (set! conts (cons a conts))) #f)) 2 1) 2))))
			  lst))
	  (let ((len (length lst)))
	    (if (<= len 4)
		((list-ref conts (- len 1)) #t)
		(reverse lst)))))
					; (+ 0 (* 1 (+ 0 (* 1 2)))) to start
					; (+ 0 ...          2     )
					; (+ 0 ...     1   [1]    )
					; (+ 0 ...2   [0]         )
					; (+ 1   [1]              )
      '(2 4 3 4 3))

(test (let ((call/cc 2)) (+ call/cc 1)) 3)
(test (+ 1 (call/cc (lambda (r) (r 2 3 4))) 5) 15)
(test (string-ref (call/cc (lambda (s) (s "hiho" 1)))) #\i)

(let ((r5rs-ratify (lambda (ux err)
		     (if (= ux 0.0) 
			 0
			 (let ((tt 1) 
			       (a1 0) 
			       (b2 0) 
			       (a2 1) 
			       (b1 1) 
			       (a 0)  
			       (b 0)
			       (ctr 0)
			       (x (/ 1 ux)))
			   (call-with-current-continuation
			    (lambda (return)
			      (do ()
				  (#f)
				(set! a (+ (* a1 tt) a2)) 
				(set! b (+ (* tt b1) b2))
					;(format #t "~A ~A~%" a (- b a))
				(if (or (<= (abs (- ux (/ a b))) err)
					(> ctr 1000))
				    (return (/ a b)))
				(set! ctr (+ 1 ctr))
				(if (= x tt) (return))
				(set! x (/ 1 (- x tt))) 
				(set! tt (floor x))
				(set! a2 a1) 
				(set! b2 b1) 
				(set! a1 a) 
				(set! b1 b)))))))))
  
  (test (r5rs-ratify (/ (log 2.0) (log 3.0)) 1/10000000) 665/1054)
  (if (positive? 2147483648)
      (test (r5rs-ratify (/ (log 2.0) (log 3.0)) 1/100000000000) 190537/301994)))

(for-each
 (lambda (arg)
   (test (let ((ctr 0)) 
	   (let ((val (call/cc 
		       (lambda (exit) 
			 (for-each (lambda (a) 
				     (if (equal? a arg) (exit arg))
				     (set! ctr (+ ctr 1))) 
				   (list 0 1 2 3 arg 5)))))) 
	     (list ctr val)))
	 (list 4 arg)))
 (list "hi" -1 #\a 11 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #f #t '(1 . 2)))

(test (+ 2 (call/cc (lambda (rtn) (+ 1 (let () (begin (define x (+ 1 (rtn 3)))) x))))) 5)


;;; others from stackoverflow.com Paul Hollingsworth etc:

(test (procedure? (call/cc (lambda (k) k))) #t)
(test (call/cc (lambda (k) (+ 56 (k 3)))) 3)
(test (apply
       (lambda (k i) 
	 (if (> i 5) 
	     i 
	     (k (list k (* 2 i)))))
       (call/cc (lambda (k) (list k 1))))
      8)
(test (apply
       (lambda (k i n) (if (= i 0) n (k (list k (- i 1) (* i n)))))
       (call/cc (lambda (k) (list k 6 1))))
      720)
(test (let* ((ka (call/cc (lambda (k) `(,k 1)))) (k (car ka)) (a (cadr ka)))
	(if (< a 5) (k `(,k ,(* 2 a))) a))
      8)

(test (apply (lambda (k i n) (if (eq? i 0) n (k (list k (- i 1) (* i n))))) (call/cc (lambda (k) (list k 6 1)))) 720)
(test ((call/cc (lambda (k) k)) (lambda (x) 5)) 5)

(let ()
  (define (generate-one-element-at-a-time a-list)
    (define (generator)
      (call/cc control-state)) 
    (define (control-state return)
      (for-each 
       (lambda (an-element-from-a-list)
	 (set! return
	       (call/cc
		(lambda (resume-here)
		  (set! control-state resume-here)
		  (return an-element-from-a-list)))))
       a-list)
      (return 'you-fell-off-the-end-of-the-list))
    generator)
  (let ((gen (generate-one-element-at-a-time (list 3 2 1))))
    (test (gen) 3)
    (test (gen) 2)
    (test (gen) 1)
    (test (gen) 'you-fell-off-the-end-of-the-list)))

;;; from Ferguson and Duego "call with current continuation patterns"
(test (let ()
	(define count-to-n
	  (lambda (n)
	    (let ((receiver 
		   (lambda (exit-procedure)
		     (let ((count 0))
		       (letrec ((infinite-loop
				 (lambda ()
				   (if (= count n)
				       (exit-procedure count)
				       (begin
					 (set! count (+ count 1))
					 (infinite-loop))))))
			 (infinite-loop))))))
	      (call/cc receiver))))
	(count-to-n 10))
      10)

(test (let ()
	(define product-list
	  (lambda (nums)
	    (let ((receiver
		   (lambda (exit-on-zero)
		     (letrec ((product
			       (lambda (nums)
				 (cond ((null? nums) 1)
				       ((zero? (car nums)) (exit-on-zero 0))
				       (else (* (car nums)
						(product (cdr nums))))))))
		       (product nums)))))
	      (call/cc receiver))))
	(product-list '(1 2 3 0 4 5)))
      0)

(begin
  (define fact
    ((lambda (f)
       ((lambda (u) (u (lambda (x)
			 (lambda (n) ((f (u x)) n)))))
	(call/cc (call/cc (call/cc 
			   (call/cc (call/cc (lambda (x) x))))))))
     (lambda (f) (lambda (n)
		   (if (<= n 0) 1 (* n (f (- n 1))))))))
  (test (map fact '(5 6 7)) '(120 720 5040)))

;; http://okmij.org/ftp/Scheme/callcc-calc-page.html

(test (let ()
	(define product-list
	  (lambda (nums)
	    (let ((receiver
		   (lambda (exit-on-zero)
		     (letrec ((product
			       (lambda (nums) 
				 (cond ((null? nums) 1)
				       ((number? (car nums))
					(if (zero? (car nums))
					    (exit-on-zero 0)
					    (* (car nums)
					       (product (cdr nums)))))
				       (else (* (product (car nums))
						(product (cdr nums))))))))
		       (product nums)))))
	      (call/cc receiver))))
	(product-list '(1 2 (3 4) ((5)))))
      120)

(test (call/cc (lambda () 0)) 'error)
(test (call/cc (lambda (a) 0) 123) 'error)
(test (call/cc) 'error)
(test (call/cc abs) 'error)
(for-each
 (lambda (arg)
   (test (call/cc arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call/cc . 1) 'error)
(test (call/cc abs) 'error)
(test (+ 1 (call/cc (lambda (r1) (call/cc (lambda (r2) (r1 2 3))))) 4) 10)
(test (+ 1 (call/cc (lambda (r1) (+ 5 (call/cc (lambda (r2) (r2 2 3)))))) 4) 15)


#|
;;; from bug-guile
(define k #f)
(define result #f)
(define results '())
(set! result (map (lambda (x)
                    (if x x (call/cc (lambda (c)
                                       (set! k c)
                                       1))))
                  '(#t #f)))
(set! results (cons result results))
(write results)
(newline)
(if (< (cadr result) 5)
    (k (+ 1 (cadr result))))
(newline)

the claim is that this should return 

((#t 1))
((#t 2) (#t 1))
((#t 3) (#t 2) (#t 1))
((#t 4) (#t 3) (#t 2) (#t 1))
((#t 5) (#t 4) (#t 3) (#t 2) (#t 1))

but I think that depends on how we interpret the sequence of top-level statements.
The test should be written:

(let* ((k #f)
       (results '()))
  (let ((result (map (lambda (x)
		       (if x x (call/cc (lambda (c)
					  (set! k c)
					  1))))
		     '(#t #f))))
    (set! results (cons result results))
    (write results)
    (newline)
    (if (< (cadr result) 5)
	(k (+ 1 (cadr result))))
    (newline)))

and then s7 is not following r6rs because it stops at 

((#t 1))
((1 . #1=(#t 2)) #1#)

saying cadr is not a number. I don't think this example is correct in any case --
who says the continuation has to restart the map from the top?
|#

(let ((cont #f))
  (let ((x (* (call/cc
	       (lambda (return)
		 (set! cont return)
		 (return 3 4))))))
    (if (= x 12)
	(cont 5 6 7))
    (test x 210)))

;; Guile handles this very differently


(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return))) (cont 1)) 'error)
(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return))) (apply cont)) 'error)
(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return) (cont 1))) (apply cont)) 'error)
(test (let ((cont #f)) (call-with-exit (lambda (return) (set! cont return) (cont 1))) (cont 1)) 'error)
(test (procedure? (call-with-exit append)) #t)
(test (procedure? (call-with-exit values)) #t)
(test (procedure? (car (call-with-exit list))) #t)
(test (call-with-exit (call-with-exit append)) 'error)
(test (continuation? (call/cc (call/cc append))) #t)
(test (procedure? (call-with-exit call-with-exit)) #t)
(test (call-with-exit ((lambda args procedure?))) #t)

(test (let ((c1 #f)) (call-with-exit (lambda (c2) (call-with-exit (lambda (c3) (set! c1 c3) (c2))))) (c1)) 'error)
(test (let ((c1 #f)) (call/cc (lambda (c2) (call-with-exit (lambda (c3) (set! c1 c3) (c2))))) (c1)) 'error)
(test (let ((cont #f)) (catch #t (lambda () (call-with-exit (lambda (return) (set! cont return) (error 'testing " a test")))) (lambda args 'error)) (apply cont)) 'error)
(test (let ((cont #f)) (catch #t (lambda () (call-with-exit (lambda (return) (set! cont return) (error 'testing " a test")))) (lambda args 'error)) (cont 1)) 'error)

(test (let ((cc #f)
	    (doit #t)
	    (ctr 0))
	(let ((ok (call-with-exit
		   (lambda (c3)
		     (call/cc (lambda (ret) (set! cc ret)))
		     (c3 (let ((res doit)) (set! ctr (+ ctr 1)) (set! doit #f) res))))))
	  (if ok (cc)))
	ctr)
      2)

(test (let ((val (call-with-exit (lambda (ret) (let ((ret1 ret)) (ret1 2) 3))))) val) 2)





;;; -------- dynamic-wind --------

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (dynamic-wind
			(lambda () (set! ctr1 (+ ctr1 1)))
			(lambda () (set! ctr2 (+ ctr2 1)) ctr2)
			(lambda () (set! ctr3 (+ ctr3 1))))))
	  (= ctr1 ctr2 ctr3 ctr4 1)))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (set! ctr1 (+ ctr1 1)))
				   (lambda () (exit ctr2) (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (set! ctr3 (+ ctr3 1)) 123))))))
	  (and (= ctr1 ctr3 1)
	       (= ctr2 ctr4 0))))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (exit ctr1) (set! ctr1 (+ ctr1 1)))
				   (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (set! ctr3 (+ ctr3 1))))))))
	  (= ctr1 ctr2 ctr3 ctr4 0)))
      #t)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((ctr4 (call/cc (lambda (exit)
			       (dynamic-wind
				   (lambda () (set! ctr1 (+ ctr1 1)))
				   (lambda () (set! ctr2 (+ ctr2 1)) ctr2)
				   (lambda () (exit ctr3) (set! ctr3 (+ ctr3 1))))))))
	  (and (= ctr1 ctr2 1)
	       (= ctr3 ctr4 0))))
      #t)

(test (let ((path '())  
	    (c #f)) 
	(let ((add (lambda (s)  
		     (set! path (cons s path))))) 
	  (dynamic-wind  
	      (lambda () (add 'connect))  
	      (lambda () (add (call-with-current-continuation  
			       (lambda (c0) (set! c c0) 'talk1))))  
	      (lambda () (add 'disconnect))) 
	  (if (< (length path) 4) 
	      (c 'talk2) 
	      (reverse path)))) 
      '(connect talk1 disconnect  connect talk2 disconnect))


(for-each
 (lambda (arg)
   (test (dynamic-wind (lambda () #f) (lambda () arg) (lambda () #f)) arg))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (dynamic-wind (lambda () #f) (lambda () #f) (lambda () #f)) #f)
(test (+ 1 (dynamic-wind (lambda () #f) (lambda () (values 2 3 4)) (lambda () #f)) 5) 15)

(test (let ((identity (lambda (a) a)))
        (let ((x '())
              (c #f))
          (dynamic-wind
	      (lambda () (set! x (cons 'a x)))
	      (lambda ()
		(dynamic-wind
		    (lambda () (set! x (cons 'b x)))
		    (lambda ()
		      (dynamic-wind
			  (lambda () (set! x (cons 'c x)))
			  (lambda () (set! c (call/cc identity)))
			  (lambda () (set! x (cons 'd x)))))
		    (lambda () (set! x (cons 'e x))))
		(dynamic-wind
		    (lambda () (set! x (cons 'f x)))
		    (lambda () (if c (c #f)))
		    (lambda () (set! x (cons 'g x)))))
	      (lambda () (set! x (cons 'h x))))
          (reverse x)))
      '(a b c d e f g b c d e f g h))


(test (list (dynamic-wind 
		(lambda () #f)
		(lambda () (values 'a 'b 'c))
		(lambda () #f)))
      (list 'a 'b 'c))

(test (let ((dynamic-wind 1)) (+ dynamic-wind 2)) 3)

(test (let ((ctr1 0)
	    (ctr2 0)
	    (ctr3 0))
	(let ((val (dynamic-wind
		       (lambda () #f)
		       (lambda ()
			 (set! ctr1 1)
			 (call/cc
			  (lambda (exit)
			    (exit 123)
			    (set! ctr2 2)
			    321)))
		       (lambda ()
			 (set! ctr3 3)))))
	  (and (= ctr1 1) (= ctr2 0) (= ctr3 3) (= val 123))))
      #t)

(test (let ((ctr1 0))
	(let ((val (dynamic-wind
		       (let ((a 1))
			 (lambda ()
			   (set! ctr1 a)))
		       (let ((a 10))
			 (lambda ()
			   (set! ctr1 (+ ctr1 a))
			   ctr1))
		       (let ((a 100))
			 (lambda ()
			   (set! ctr1 (+ ctr1 a)))))))
	  (and (= ctr1 111) (= val 11))))
      #t)

(test (let ((ctr1 0))
	(let ((val (+ 3 (dynamic-wind
			    (let ((a 1))
			      (lambda ()
				(set! ctr1 a)))
			    (let ((a 10))
			      (lambda ()
				(set! ctr1 (+ ctr1 a))
				ctr1))
			    (let ((a 100))
			      (lambda ()
				(set! ctr1 (+ ctr1 a)))))
		      1000)))
	  (and (= ctr1 111) (= val 1014))))
      #t)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda ()
		 (set! n (+ n 1))
		 (k))
	       (lambda ()
		 (set! n (+ n 2)))
	       (lambda ()
		 (set! n (+ n 4))))))
	n)
      1)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda () #f)
	       (lambda ()
		 (dynamic-wind
		     (lambda () #f)
		     (lambda ()
		       (set! n (+ n 1))
		       (k))
		     (lambda ()
		       (set! n (+ n 2))
					;(k)
		       )))
	       (lambda ()
		 (set! n (+ n 4))))))
	n)
      7)

(test (let ((n 0))
	(call-with-current-continuation
	 (lambda (k)
	   (dynamic-wind
	       (lambda () #f)
	       (lambda ()
		 (dynamic-wind
		     (lambda () #f)
		     (lambda ()
		       (dynamic-wind
			   (lambda () #f)
			   (lambda ()
			     (set! n (+ n 1))
			     (k))
			   (lambda ()
			     (if (= n 1)
				 (set! n (+ n 2))))))
		     (lambda ()
		       (if (= n 3)
			   (set! n (+ n 4))))))
	       (lambda ()
		 (if (= n 7)
		     (set! n (+ n 8)))))))
	n)
      15)

(test (dynamic-wind) 'error)
(test (dynamic-wind (lambda () #f)) 'error)
(test (dynamic-wind (lambda () #f) (lambda () #f)) 'error)
(test (dynamic-wind (lambda (a) #f) (lambda () #f) (lambda () #f)) 'error)
(test (dynamic-wind (lambda () #f) (lambda (a b) #f) (lambda () #f)) 'error)
(test (dynamic-wind (lambda () #f) (lambda () #f) (lambda (a) #f)) 'error)
(test (dynamic-wind (lambda () 1) #f (lambda () 2)) 'error)
(test (dynamic-wind . 1) 'error)

;;; from scheme wiki
;;; http://community.schemewiki.org/?hose-the-repl
;;; jorgen-schafer

(test (let loop ()  
	(call-with-exit
	 (lambda (k)  
	   (dynamic-wind  
	       (lambda () #t)  
	       (lambda () (let loop () (loop)))  
	       k))) 
	(loop))
      'error)
;; that example calls to mind a bunch like it:
(test (call-with-exit (lambda (k) (dynamic-wind (lambda () #t) (lambda () (let loop () (loop))) k))) 'error)
(test (call-with-exit (lambda (k) (dynamic-wind (lambda () #t) k (lambda () #t)))) 'error)
(test (call-with-exit (lambda (k) (dynamic-wind k (lambda () #f) (lambda () #t)))) 'error)

(test (call-with-exit (lambda (k) (procedure-documentation k))) "")
(test (call-with-exit (lambda (k) (procedure-arity k))) '())
(test (call-with-exit (lambda (k) (procedure-source k))) '())
(test (procedure-arity (call-with-exit (lambda (k) (make-procedure-with-setter k k)))) '())
(test (procedure-arity (make-procedure-with-setter vector-ref vector-set!)) '(2 0 #t 3 0 #t))
(test (let ((pws (make-procedure-with-setter vector-ref vector-set!))) 
	(let ((pws1 (make-procedure-with-setter pws vector-set!))) 
	  (let ((v (vector 1 2))) 
	    (set! (pws1 v 1) 32) 
	    (pws1 v 1))))
      32)
(test (call-with-exit (lambda (k) (map k '(1 2 3)))) 1)
(test (call-with-exit (lambda (k) (for-each k '(1 2 3)))) 1)
(test (call-with-exit (lambda (k) (catch #t k k))) 'error)
(test (call-with-exit (lambda (k) (catch #t (lambda () #f) k))) #f)
(test (call-with-exit (lambda (k) (catch #t (lambda () (error 'an-error)) k))) 'error)
(test (call-with-exit (lambda (k) (sort! '(1 2 3) k))) 'error)
(test (sort! '(1 2 3) (lambda () #f)) 'error)
(test (sort! '(1 2 3) (lambda (a) #f)) 'error)
(test (sort! '(1 2 3) (lambda (a b c) #f)) 'error)
(test (let () (define-macro (asdf a b) `(< ,a ,b)) (sort! '(1 2 3) asdf)) 'error)
(test (let () (let asdf () (sort! '(1 2 3) asdf))) 'error)
(test (let () (let asdf () (map asdf '(1 2 3)))) 'error)
(test (let () (let asdf () (for-each asdf '(1 2 3)))) 'error)

(test (let ((ctr 0))
	(call-with-exit
	 (lambda (exit)
	   (let asdf
	       ()
	     (set! ctr (+ ctr 1))
	     (if (> ctr 2)
		 (exit ctr))
	     (dynamic-wind
		 (lambda () #f)
		 (lambda () #f)
		 asdf)))))
      3)

(test (let ((ctr 0))
	(dynamic-wind
	    (lambda () #f)
	    (lambda ()
	      (call-with-exit
	       (lambda (exit)
		 (catch #t
			(lambda ()
			  (error 'error))
			(lambda args
			  (exit 'error)))
		 (set! ctr 1))))
	    (lambda ()
	      (set! ctr (+ ctr 2))))
	ctr)
      2)
(test (call-with-exit (lambda (r1) (call-with-exit (lambda (r2) (call-with-exit (lambda (r3) (r1 12) (r2 1))) (r1 2))) 3)) 12)
(test (call-with-exit (lambda (r1) (call-with-exit (lambda (r2) (call-with-exit (lambda (r3) (r2 12) (r2 1))) (r1 2))) 3)) 3)
(test (call-with-exit (lambda (r1) (call-with-exit (lambda (r2) (call-with-exit (lambda (r3) (r3 12) (r2 1))) (r1 2))) 3)) 2)

(let ((pws (make-procedure-with-setter < >))) (test (sort! '(2 3 1 4) pws) '(1 2 3 4)))
(test (call-with-exit (lambda (k) (call-with-input-string "123" k))) 'error)
(test (call-with-exit (lambda (k) (call-with-input-file "tmp1.r5rs" k))) 'error)
(test (call-with-exit (lambda (k) (call-with-output-file "tmp1.r5rs" k))) 'error)
(test (call-with-exit (lambda (k) (call-with-output-string k))) 'error)
(let ((pws (make-procedure-with-setter (lambda (a) (+ a 1)) (lambda (a b) b))))
  (test (procedure? pws) #t)
  (test (map pws '(1 2 3)) '(2 3 4))
  (test (apply pws '(1)) 2))
(test (let ((ctr 0)) (call-with-exit (lambda (top-exit) (set! ctr (+ ctr 1)) (call-with-exit top-exit) (set! ctr (+ ctr 16)))) ctr) 1)

(test (let () (+ 5 (call-with-exit (lambda (return) (return 1 2 3) 4)))) 11)
(test (+ 5 (call-with-exit (lambda (return) (return 1)))) 6)
(test (+ 5 (call-with-exit (lambda (return) (return)))) 'error)

(test (let ((cur '()))
	(define (step pos)
	  (dynamic-wind
	      (lambda ()
		(set! cur (cons pos cur)))
	      (lambda ()
		(set! cur (cons (+ pos 1) cur))
		(if (< pos 40)
		    (step (+ pos 10)))
		(set! cur (cons (+ pos 2) cur))
		cur)
	      (lambda ()
		(set! cur (cons (+ pos 3) cur)))))
	(reverse (step 0)))
      '(0 1 10 11 20 21 30 31 40 41 42 43 32 33 22 23 12 13 2))


(test (let ((cur '()))
	(define (step pos)
	  (dynamic-wind
	      (lambda ()
		(set! cur (cons pos cur)))
	      (lambda ()
		(set! cur (cons (+ pos 1) cur))
		(if (< pos 40)
		    (step (+ pos 10))
		    (error 'all-done))
		(set! cur (cons (+ pos 2) cur))
		cur)
	      (lambda ()
		(set! cur (cons (+ pos 3) cur)))))
	(catch 'all-done
	       (lambda ()
		 (reverse (step 0)))
	       (lambda args (reverse cur))))
      '(0 1 10 11 20 21 30 31 40 41 43 33 23 13 3))

(test (let ((cur '()))
	(define (step pos ret)
	  (dynamic-wind
	      (lambda ()
		(set! cur (cons pos cur)))
	      (lambda ()
		(set! cur (cons (+ pos 1) cur))
		(if (< pos 40)
		    (step (+ pos 10) ret)
		    (ret (reverse cur)))
		(set! cur (cons (+ pos 2) cur))
		cur)
	      (lambda ()
		(set! cur (cons (+ pos 3) cur)))))
	(list (call-with-exit
	       (lambda (ret)
		 (step 0 ret)))
	      (reverse cur)))
      '((0 1 10 11 20 21 30 31 40 41) (0 1 10 11 20 21 30 31 40 41 43 33 23 13 3)))

#|
;; these test jump out of current context (OP_BARRIER)
;; but they end up exiting the load as well 
(test (let ()
	(catch #t
	       (lambda ()
		 (eval-string "(error 'hi \"hi\")"))
	       (lambda args
		 'error)))
      'error)
(test (let ()
	(call-with-exit
	 (lambda (return)
	   (eval-string "(return 3)"))))
      3)
(test (let ()
	(call/cc
	 (lambda (return)
	   (eval-string "(return 3)"))))
      3)
(test (let ()
	(call-with-exit
	 (lambda (return)
	   (eval-string "(abs (+ 1 (if #t (return 3))))"))))
      4)
|#




;;; -------- quasiquote --------

(test `(1 2 3) '(1 2 3))
(test `() '())
(test `(list ,(+ 1 2) 4)  '(list 3 4))
(test `(1 ,@(list 1 2) 4) '(1 1 2 4))
(test `#(1 ,@(list 1 2) 4) '#(1 1 2 4))
(test `(a ,(+ 1 2) ,@(map abs '(4 -5 6)) b) '(a 3 4 5 6 b))
(if (eqv? 2 (sqrt 4))
    (test `#(10 5 ,(sqrt 4) ,@(map sqrt '(16 9)) 8) '#(10 5 2 4 3 8))) ; inexactness foolishness
(test `(a `(b ,(+ 1 2) ,(foo ,(+ 1 3) d) e) f) '(a `(b ,(+ 1 2) ,(foo 4 d) e) f))
(test (let ((name1 'x) (name2 'y)) `(a `(b ,,name1 ,',name2 d) e)) '(a `(b ,x ,'y d) e))
(test `(1 2 ,(* 9 9) 3 4) '(1 2 81 3 4))
(test `(1 ,(+ 1 1) 3) '(1 2 3))                     
(test `(,(+ 1 2)) '(3))
;(test `(,'a . ,'b) (cons 'a 'b))
(test `(,@'() . foo) 'foo)

;; from gauche
(let ((quasi0 99)
      (quasi1 101)
      (quasi2 '(a b))
      (quasi3 '(c d)))
  (test `,quasi0 99)
  (test `,quasi1 101)
  (test `(,(cons 1 2)) '((1 . 2)))
  (test `(,(cons 1 2) 3) '((1 . 2) 3))
  (test `(,quasi0 3) '(99 3))
  (test `(3 ,quasi0) '(3 99))
  (test `(,(+ quasi0 1) 3) '(100 3))
  (test `(3 ,(+ quasi0 1)) '(3 100))
  (test `(,quasi1 3) '(101 3))
  (test `(3 ,quasi1) '(3 101))
  (test `(,(+ quasi1 1) 3) '(102 3))
  (test `(3 ,(+ quasi1 1)) '(3 102))
  (test `(1 ,@(list 2 3) 4) '(1 2 3 4))
  (test `(1 2 ,@(list 3 4)) '(1 2 3 4))
  (test `(,@quasi2 ,@quasi3) '(a b c d))
  (test `(1 2 . ,(list 3 4)) '(1 2 3 4))
  (test `(,@quasi2 . ,quasi3) '(a b c d))
  (test `#(,(cons 1 2) 3) '#((1 . 2) 3))
  (test `#(,quasi0 3) '#(99 3))
  (test `#(,(+ quasi0 1) 3) '#(100 3))
  (test `#(3 ,quasi1) '#(3 101))
  (test `#(3 ,(+ quasi1 1)) '#(3 102))
  (test `#(1 ,@(list 2 3) 4) '#(1 2 3 4))
  (test `#(1 2 ,@(list 3 4)) '#(1 2 3 4))
  (test `#(,@quasi2 ,@quasi3) '#(a b c d))
  (test `#(,@quasi2 ,quasi3) '#(a b (c d)))
  (test `#(,quasi2  ,@quasi3) '#((a b) c d))
  (test `#() '#())
  (test `#(,@(list)) '#())
  (test `(,@(list 1 2) ,@(list 1 2)) '(1 2 1 2))
  (test `(,@(list 1 2) a ,@(list 1 2)) '(1 2 a 1 2))
  (test `(a ,@(list 1 2) ,@(list 1 2)) '(a 1 2 1 2))
  (test `(,@(list 1 2) ,@(list 1 2) a) '(1 2 1 2 a))
  (test `(,@(list 1 2) ,@(list 1 2) a b) '(1 2 1 2 a b))
  (test `(,@(list 1 2) ,@(list 1 2) . a) '(1 2 1 2 . a))
  ;(test `(,@(list 1 2) ,@(list 1 2) . ,(cons 1 2)) '(1 2 1 2 1 . 2))
  (test `(,@(list 1 2) ,@(list 1 2) . ,quasi2) '(1 2 1 2 a b))
  ;(test `(,@(list 1 2) ,@(list 1 2) a . ,(cons 1 2)) '(1 2 1 2 a 1 . 2))
  (test `(,@(list 1 2) ,@(list 1 2) a . ,quasi3) '(1 2 1 2 a c d))
  (test `#(,@(list 1 2) ,@(list 1 2)) '#(1 2 1 2))
  (test `#(,@(list 1 2) a ,@(list 1 2)) '#(1 2 a 1 2))
  (test `#(a ,@(list 1 2) ,@(list 1 2)) '#(a 1 2 1 2))
  (test `#(,@(list 1 2) ,@(list 1 2) a) '#(1 2 1 2 a))
  (test `#(,@(list 1 2) ,@(list 1 2) a b) '#(1 2 1 2 a b))
  (test `(1 `(1 ,2 ,,(+ 1 2)) 1) '(1 `(1 ,2 ,3) 1))
  (test `(1 `(1 ,,quasi0 ,,quasi1) 1) '(1 `(1 ,99 ,101) 1))
  (test `(1 `(1 ,@2 ,@,(list 1 2))) '(1 `(1 ,@2 ,@(1 2))))
  (test `(1 `(1 ,@,quasi2 ,@,quasi3)) '(1 `(1 ,@(a b) ,@(c d))))
  (test `(1 `(1 ,(,@quasi2 x) ,(y ,@quasi3))) '(1 `(1 ,(a b x) ,(y c d))))
  (test `#(1 `(1 ,2 ,,(+ 1 2)) 1) '#(1 `(1 ,2 ,3) 1))
  (test `#(1 `(1 ,,quasi0 ,,quasi1) 1) '#(1 `(1 ,99 ,101) 1))
  (test `#(1 `(1 ,@2 ,@,(list 1 2))) '#(1 `(1 ,@2 ,@(1 2))))
  (test `#(1 `(1 ,@,quasi2 ,@,quasi3)) '#(1 `(1 ,@(a b) ,@(c d))))
  (test `#(1 `(1 ,(,@quasi2 x) ,(y ,@quasi3))) '#(1 `(1 ,(a b x) ,(y c d))))
  (test `(1 `#(1 ,(,@quasi2 x) ,(y ,@quasi3))) '(1 `#(1 ,(a b x) ,(y c d)))))

(test (let ((hi (lambda (a) `(+ 1 ,a))))
	(hi 2))
      '(+ 1 2))

(test (let ((hi (lambda (a) `(+ 1 ,@a))))
	(hi (list 2 3)))
      '(+ 1 2 3))

(test (let ((hi (lambda (a) `(let ((b ,a)) ,(+ 1 a)))))
	(hi 3))
      '(let ((b 3)) 4))

(test (let ((x '(a b c)))
	`(x ,x ,@x foo ,(cadr x) bar ,(cdr x) baz ,@(cdr x)))
      '(x (a b c) a b c foo b bar (b c) baz b c))

(test (let ((x '(a b c)))
	`(,(car `(,x))))
      '((a b c)))

(test (let ((x '(a b c)))
	`(,@(car `(,x))))
      '(a b c))

(test (let ((x '(a b c)))
	`(,(car `(,@x))))
      '(a))

(test (let ((x '(a b c)))
	``,,x)
      '(a b c))

(test (let ((x '(a b c)))
	`,(car `,x))
      'a)

(test (let ((x '(2 3)))
	`(1 ,@x 4))
      '(1 2 3 4))

(test (let ((x '(2 3)))
	`#(9 ,@x 9))
      '#(9 2 3 9))

(test `#(1 ,(/ 12 2)) '#(1 6))
(test ((lambda () `#(1 ,(/ 12 2)))) '#(1 6))

(test (let ((x '(2 3)))
	`(1 ,@(map (lambda (a) (+ a 1)) x)))
      '(1 3 4))

;;; these are from the scheme bboard
(test (let ((x '(1 2 3))) `(0 . ,x)) '(0 1 2 3))
(test (let ((x '(1 2 3))) `(0 ,x)) '(0 (1 2 3)))
(test (let ((x '(1 2 3))) `#(0 ,x)) '#(0 (1 2 3)))
					;(test (let ((x '(1 2 3))) `#(0 . ,x)) '#(0 1 2 3))

(test (let () (define-macro (tryqv . lst) `(map abs ',lst)) (tryqv 1 2 3 -4 5)) '(1 2 3 4 5))
(test (let () (define-macro (tryqv . lst) `(map abs '(,@lst))) (tryqv 1 2 3 -4 5)) '(1 2 3 4 5))
(test (let () (define-macro (tryqv . lst) `(map abs (vector ,@lst))) (tryqv 1 2 3 -4 5)) '(1 2 3 4 5))

(test (quasiquote) 'error)
(let ((d 1))
  (test (quasiquote (a b c ,d)) '(a b c 1)))

(test (quasiquote (list (unquote (+ 1 2)) 4)) '(list 3 4))




;; -------- s7 specific stuff --------

(for-each
 (lambda (arg)
   (test (keyword? arg) #f))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(let ((kw (make-keyword "hiho")))
  (test (keyword? kw) #t)
  (test (keyword->symbol kw) 'hiho)
  (test (symbol->keyword 'hiho) kw)
  (test (keyword->symbol (symbol->keyword 'key)) 'key)
  (test (symbol->keyword (keyword->symbol (make-keyword "hi"))) :hi)
  (test (keyword? :a-key) #t)
  (test (keyword? ':a-key) #t)
  (test (keyword? ':a-key:) #t)
  (test (keyword? 'a-key:) #t)
  (test (symbol? (keyword->symbol :hi)) #t)
  (test (keyword? (keyword->symbol :hi)) #f)
  (test (symbol? (symbol->keyword 'hi)) #t)
  (test (equal? kw :hiho) #t)
  (test ((lambda (arg) (keyword? arg)) :hiho) #t)
  (test ((lambda (arg) (keyword? arg)) 'hiho) #f)
  (test ((lambda (arg) (keyword? arg)) kw) #t)
  (test ((lambda (arg) (keyword? arg)) (symbol->keyword 'hiho)) #t)
  (test (make-keyword "3") :3)
  (test (keyword? :3) #t)
  (test (keyword? ':3) #t)
  (test (keyword? '3) #f)
  (test (keyword? ':) #f)
  (test (keyword? '::) #t)
  (test (keyword? :optional) #t)
  (test (symbol->string (keyword->symbol hi:)) "hi")
  (test (symbol->string (keyword->symbol :hi)) "hi")
  (test (make-keyword ":") ::))

(let ()
  (define* (hi a b) (+ a b))
  (test (hi 1 2) 3)
  (test (hi :b 3 :a 1) 4)
  (test (hi b: 3 a: 1) 4))

(for-each
 (lambda (arg)
   (test (make-keyword arg) 'error))
 (list -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (keyword->symbol arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (symbol->keyword arg) 'error))
 (list "hi" -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (keyword?) 'error)
(test (keyword? 1 2) 'error)
(test (make-keyword) 'error)
(test (make-keyword 'hi 'ho) 'error)
(test (keyword->symbol) 'error)
(test (keyword->symbol :hi :ho) 'error)
(test (symbol->keyword) 'error)
(test (symbol->keyword 'hi 'ho) 'error)



(for-each
 (lambda (arg)
   (test (gensym arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(test (gensym "hi" "ho") 'error)

(test (symbol? (gensym)) #t)
(test (symbol? (gensym "temp")) #t)
(test (eq? (gensym) (gensym)) #f)
(test (eqv? (gensym) (gensym)) #f)
(test (equal? (gensym) (gensym)) #f)
(test (keyword? (gensym)) #f)
(test (let* ((a (gensym)) (b a)) (eq? a b)) #t)
(test (let* ((a (gensym)) (b a)) (eqv? a b)) #t)

(let ((sym (gensym)))
  (test (eval `(let ((,sym 32)) (+ ,sym 1))) 33))

(let ((sym1 (gensym))
      (sym2 (gensym)))
  (test (eval `(let ((,sym1 32) (,sym2 1)) (+ ,sym1 ,sym2))) 33))

(test (let ((hi (gensym))) (eq? hi (string->symbol (symbol->string hi)))) #t)


(test (provided?) 'error)
(test (provide) 'error)
(test (or (null? *features*) (pair? *features*)) #t)
(test (provided? 1 2 3) 'error)
(test (provide 1 2 3) 'error)
(provide 's7test)
(test (provided? 's7test) #t)
(test (provided? 'not-provided!) #f)
(test (provide lambda) 'error)

(for-each
 (lambda (arg)
   (test (provide arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (provided? arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) (list 1 2 3) '(1 . 2)))


(test (integer? *vector-print-length*) #t)
(test (or (null? *trace-hook*) (procedure? *trace-hook*)) #t)
(test (or (null? *#readers*) (pair? *#readers*)) #t)
(test (or (null? *load-hook*) (procedure? *load-hook*)) #t)
(test (or (null? *load-path*) (pair? *load-path*)) #t)
(test (or (null? *error-hook*) (procedure? *error-hook*)) #t)
(test (or (null? *unbound-variable-hook*) (procedure? *unbound-variable-hook*)) #t)
(test (vector? *error-info*) #t)



(test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) <) (list 0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) (lambda (a b) (< a b))) (list 0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! (list) <) '()) #t)
(test (equal? (sort! (list 1) <) '(1)) #t)
(test (equal? (sort! (list 1 1 1) <) '(1 1 1)) #t)
(test (equal? (sort! (list 0 1 2 3 4 5 6 7 8 9) <) '(0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! (list #\a #\l #\o #\h #\a) char<?) '(#\a #\a #\h #\l #\o)) #t)
(test (equal? (sort! (list "tic" "tac" "toe") string<?) '("tac" "tic" "toe")) #t)
(test (equal? (sort! (list 3 4 8 2 0 1 5 9 7 6) >) (reverse (list 0 1 2 3 4 5 6 7 8 9))) #t)
(test (equal? (sort! '((3 . 1) (2 . 8) (5 . 9) (4 . 7) (6 . 0)) (lambda (a b) (< (car a) (car b)))) '((2 . 8) (3 . 1) (4 . 7) (5 . 9) (6 . 0))) #t)
(test (equal? (sort! '((3 . 1) (2 . 8) (5 . 9) (4 . 7) (6 . 0)) (lambda (a b) (< (cdr a) (cdr b)))) '((6 . 0) (3 . 1) (4 . 7) (2 . 8) (5 . 9))) #t)
(test (equal? (sort! (list (list 1 2) (list 4 3 2) (list) (list 1 2 3 4)) (lambda (a b) (>= (length a) (length b)))) '((1 2 3 4) (4 3 2) (1 2) ())) #t)
(test (equal? (sort! '((1 2 3) (4 5 6) (7 8 9)) (lambda (a b) (> (car a) (car b)))) '((7 8 9) (4 5 6) (1 2 3))) #t)
(test (equal? (sort! (list #\b #\A #\B #\a #\c #\C) char<?) '(#\A #\B #\C #\a #\b #\c)) #t)
(test (equal? (sort! (list (list 'u 2) (list 'i 1) (list 'a 7) (list 'k 3) (list 'c 4) (list 'b 6))
		     (lambda (a b) (< (cadr a) (cadr b))))
	      '((i 1) (u 2) (k 3) (c 4) (b 6) (a 7)))
      #t)
(test (equal? (sort! (sort! '(1 2 3) >) <) '(1 2 3)) #t)

(test (equal? (sort! (vector 3 4 8 2 0 1 5 9 7 6) <) (vector 0 1 2 3 4 5 6 7 8 9)) #t)
(test (equal? (sort! '#() <) '#()) #t)

(test (call/cc (lambda (return) (sort! '(1 2 3) (lambda (a b) (return "oops"))))) "oops")

(test (let ((v (make-vector 1000)))
	(do ((i 0 (+ i 1)))
	    ((= i 1000))
	  (vector-set! v i (random 100.0)))
	(set! v (sort! v >))
	(call-with-exit
	 (lambda (return)
	   (do ((i 0 (+ i 1)))
	       ((= i 999) #t)
	     (if (<= (v i) (v (+ i 1)))
		 (return #f))))))
      #t)

(test (let ((v '()))
	(do ((i 0 (+ i 1)))
	    ((= i 1000))
	  (set! v (cons (random 100.0) v)))
	(set! v (sort! v >))
	(apply > v))
      #t)

(test (sort! (list 3 2 1) (lambda (m n) (let ((vals (sort! (list m n) <))) (< m n)))) '(1 2 3))

(test (let ((lst '()))
	(do ((i 0 (+ i 1)))
	    ((= i 128))
	  (set! lst (cons (random 1.0) lst)))
	(let ((vals (sort! lst (lambda (m n)
				 (let ((lst1 (list 1 2 3)))
				   (sort! lst1 <))
				 (< m n)))))
	  (apply < vals)))
      #t)

(test (sort!) 'error)
(test (sort! '(1 2 3)) 'error)
(test (sort! '(1 2 3) 1) 'error)
(test (sort! '(1 2 3) < <) 'error)

(for-each
 (lambda (arg)
   (test (sort! arg <) 'error))
 (list -1 #\a 1 0 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(for-each
 (lambda (arg)
   (test (sort! '(1 2 3) arg) 'error))
 (list -1 #\a 1 0 'a-symbol 3.14 3/4 1.0+1.0i #f #t #(1) '(1) "hi" :hi))

(test (sort! '(1 2 "hi" 3) <) 'error)
(test (sort! '(1 -2 "hi" 3) (lambda (a b) 
			     (let ((a1 (if (number? a) a (length a)))
				   (b1 (if (number? b) b (length b))))
			       (< a1 b1))))
      '(-2 1 "hi" 3))

(let ((ok #f))
  (catch #t
	 (lambda ()
	   (dynamic-wind
	       (lambda () #f)
	       (lambda () (sort! '(1 2 "hi" 3) <))
	       (lambda () (set! ok #t))))
	 (lambda args 'error))
  (if (not ok) (format #t "dynamic-wind out of sort! skipped cleanup?~%")))



;;; -------- catch --------

(define (catch-test sym)
  (let ((errs '()))
    (catch 'a1
	 (lambda ()
	   (catch 'a2
		  (lambda ()
		    (catch 'a3
			   (lambda ()
			     (catch 'a4
				    (lambda ()
				      (error sym "hit error!"))
				    (lambda args
				      (set! errs (cons 'a4 errs))
				      'a4)))
			   (lambda args
			     (set! errs (cons 'a3 errs))
			     'a3)))
		  (lambda args
		    (set! errs (cons 'a2 errs))
		    'a2)))
	 (lambda args
	   (set! errs (cons 'a1 errs))
	   'a1))
    errs))

(test (catch-test 'a1) '(a1))
(test (catch-test 'a2) '(a2))
(test (catch-test 'a3) '(a3))
(test (catch-test 'a4) '(a4))

(define (catch-test-1 sym)
  (let ((errs '()))
    (catch 'a1
	 (lambda ()
	   (catch 'a2
		  (lambda ()
		    (catch 'a3
			   (lambda ()
			     (catch 'a4
				    (lambda ()
				      (error sym "hit error!"))
				    (lambda args
				      (set! errs (cons 'a4 errs))
				      (error 'a3)
				      'a4)))
			   (lambda args
			     (set! errs (cons 'a3 errs))
			     (error 'a2)
			     'a3)))
		  (lambda args
		    (set! errs (cons 'a2 errs))
		    (error 'a1)
		    'a2)))
	 (lambda args
	   (set! errs (cons 'a1 errs))
	   'a1))
    errs))

(test (catch-test-1 'a1) '(a1))
(test (catch-test-1 'a2) '(a1 a2))
(test (catch-test-1 'a3) '(a1 a2 a3))
(test (catch-test-1 'a4) '(a1 a2 a3 a4))

(test (let ((x 0))
	(catch 'a
	     (lambda ()
	       (catch 'b
		      (lambda ()
			(catch 'a
			       (lambda ()
				 (error 'a))
			       (lambda args
				 (set! x 1))))
		      (lambda args
			(set! x 2))))
	     (lambda args
	       (set! x 3)))
	x)
      1)

(test (catch) 'error)
(test (catch s7version) 'error)
(test (catch #t s7version) 'error)
(test (catch #t s7version + +) 'error)



(define (last-pair l) ; needed also by loop below
  (if (pair? (cdr l)) 
      (last-pair (cdr l)) l))
  

(let ()
  ;; from guile-user I think
  ;; (block LABEL FORMS...)
  ;;
  ;; Execute FORMS.  Within FORMS, a lexical binding named LABEL is
  ;; visible that contains an escape function for the block.  Calling
  ;; the function in LABEL with a single argument will immediatly stop
  ;; the execution of FORMS and return the argument as the value of the
  ;; block.  If the function in LABEL is not invoked, the value of the
  ;; block is the value of the last form in FORMS.
  
  (define-macro (block label . forms)
    `(let ((body (lambda (,label) ,@forms))
	   (tag (gensym "return-")))
       (catch tag
	      (lambda () (body (lambda (val) (error tag val))))
	      (lambda (tag val) val))))
  
  ;; (with-return FORMS...)
  ;;
  ;; Equivalent to (block return FORMS...)
  
  (define-macro (with-return . forms)
    `(block return ,@forms))
  
  ;; (tagbody TAGS-AND-FORMS...)
  ;;
  ;; TAGS-AND-FORMS is a list of either tags or forms.  A TAG is a
  ;; symbol while a FORM is everything else.  Normally, the FORMS are
  ;; executed sequentially.  However, control can be transferred to the
  ;; forms following a TAG by invoking the tag as a function.  That is,
  ;; within the FORMS, there is a lexical binding for each TAG with the
  ;; symbol that is the tag as its name.  The bindings carry functions
  ;; that will execute the FORMS following the respective TAG.
  ;;
  ;; The value of a tagbody is always `#f'.
  
  (define (transform-tagbody forms)
    (let ((start-tag (gensym "start-"))
	  (block-tag (gensym "block-")))
      (let loop ((cur-tag start-tag)
		 (cur-code '())
		 (tags-and-code '())
		 (forms forms))
	(cond
	 ((null? forms)
	  `(block ,block-tag
		  (letrec ,(reverse! (cons (list cur-tag `(lambda () ,@(reverse! (cons `(,block-tag #f) cur-code)))) tags-and-code))
		    (,start-tag))))
	 ((symbol? (car forms))
	  (loop (car forms)
		'()
		(cons (list cur-tag `(lambda () ,@(reverse! (cons `(,(car forms)) cur-code)))) tags-and-code)
		(cdr forms)))
	 (else
	  (loop cur-tag
		(cons (car forms) cur-code)
		tags-and-code
		(cdr forms)))))))
  
  (define-macro (tagbody . forms)
    (transform-tagbody forms))
  
  (define (first_even l)
    (with-return
     (tagbody
      continue
      (if (not (not (null? l)))
	  (break))
      (let ((e (car l)))
	(if (not (number? e))
	    (break))
	(if (even? e)
	    (return e))
	(set! l (cdr l)))
      (continue)
      break)
     (return #f)))
  
  (let ((val (first_even '(1 3 5 6 7 8 9))))
    (if (not (equal? val (list 6)))
	(format #t "first_even (tagbody, gensym, reverse!) (6): '~A~%" val)))
  
  
  
  
  (let ((hi (lambda* (a) a)))
    (test (hi 1) 1)
    (test (hi) #f)          ; all args are optional
    (test (hi :a 32) 32)    ; all args are keywords
    (test (hi 1 2) 'error)  ; extra args
    
    (for-each
     (lambda (arg)
       (test (hi arg) arg)
       (test (hi :a arg) arg))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2)))
    
    (test (hi :b 1) 'error))
  
  (let ((hi (lambda* ((a 1)) a)))
    (test (hi 2) 2)
    (test (hi) 1)
    (test (hi :a 2) 2)
    
    (for-each
     (lambda (arg)
       (test (hi arg) arg)
       (test (hi :a arg) arg))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2))))
  
  (let ((hi (lambda* (a (b "hi")) (list a b))))
    (test (hi) (list #f "hi"))
    (test (hi 1) (list 1 "hi"))
    (test (hi 1 2) (list 1 2))
    (test (hi :b 1) (list #f 1))
    (test (hi :a 1) (list 1 "hi"))
    (test (hi 1 :b 2) (list 1 2))
    (test (hi :b 3 :a 1) (list 1 3))
    (test (hi :a 3 :b 1) (list 3 1))
    (test (hi 1 :a 3) 'error)
    (test (hi 1 2 :a 3) 'error) ; trailing (extra) args
    (test (hi :a 2 :c 1) 'error)
    (test (hi 1 :c 2) 'error)
    
    (for-each
     (lambda (arg)
       (test (hi :a 1 :b arg) (list 1 arg))
       (test (hi :a arg) (list arg "hi"))
       (test (hi :b arg) (list #f arg))
       (test (hi arg arg) (list arg arg)))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2))))
  
  (let ((hi (lambda* (a :key (b 3) :optional c) (list a b c))))
    (test (hi) (list #f 3 #f))
    (test (hi 1) (list 1 3 #f))
    (test (hi :c 32) (list #f 3 32))
    (test (hi :c 32 :b 43 :a 54) (list 54 43 32))
    (test (hi 1 2 3) (list 1 2 3))
    (test (hi :b 32) (list #f 32 #f))
    (test (hi 1 2 :c 32) (list 1 2 32)))
  
  (let ((hi (lambda* (a :rest b) (list a b))))
    (test (hi 1 2 3) (list 1 (list 2 3)))
    (test (hi) (list #f ()))
    (test (hi :a 2) (list 2 '()))
    (test (hi :b 3) (list #f 3)))
  
  (let ((hi (lambda* (a :rest b :rest c) (list a b c))))
    (test (hi 1 2 3 4 5) (list 1 (list 2 3 4 5) (list 3 4 5))))
  
  (let ((hi (lambda* ((a 3) :key (b #t) :optional (c our-pi) :rest d) (list a b c d))))
    (test (hi) (list 3 #t our-pi ()))
    (test (hi 1 2 3 4) (list 1 2 3 (list 4))))
  
  (let ((hi (lambda* ((a 'hi)) (equal? a 'hi))))
    (test (hi) #t)
    (test (hi 1) #f)
    (test (hi 'hi) #t)
    (test (hi :a 1) #f))
  
  (let* ((x 32)
	 (hi (lambda* (a (b x)) (list a b))))
    (test (hi) (list #f 32))
    (test (hi :a 1) (list 1 32)))
  
  (let ((hi (lambda* (a . b) (list a b))))
    (test (hi 1 2 3) (list 1 (list 2 3)))
    (test (hi) (list #f ()))
    (test (hi :a 2) (list 2 '()))
    (test (hi :b 3) (list #f 3)))
  
  (let ((hi (lambda* ((a 0.0) :optional (b 0.0)) (+ a b))))
    (num-test (hi 1.0) 1.0)
    (num-test (hi 1.0 2.0) 3.0)
    (num-test (hi) 0.0)
    (num-test (+ (hi) (hi 1.0) (hi 1.0 2.0)) 4.0)
    (num-test (+ (hi 1.0) (hi) (hi 1.0 2.0)) 4.0)
    (num-test (+ (hi 1.0) (hi 1.0 2.0) (hi)) 4.0)
    (num-test (+ (hi 1.0 2.0) (hi) (hi 1.0)) 4.0))
  
  (test (let ((hi (lambda*))) (hi)) 'error)
  (test (let ((hi (lambda* #f))) (hi)) 'error)
  (test (let ((hi (lambda* "hi" #f))) (hi)) 'error)
  (test (let ((hi (lambda* ("hi") #f))) (hi)) 'error)
  (test (let ((hi (lambda* (a 0.0) a))) (hi)) 'error)
  (test (let ((hi (lambda* (a . 0.0) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a . 0.0)) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a 0.0 "hi")) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a 0.0 . "hi")) a))) (hi)) 'error)
  (test (let ((hi (lambda* ((a)) a))) (hi)) 'error)
  (test (let ((hi (lambda* (a 0.0) (b 0.0) (+ a b)))) (hi)) 'error)
  
  (test (let () (define* (hi) 0) (hi)) 0)
  (test (let () (define* (hi a . b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi a . b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi a . b) b) (hi 1)) '())
  (test (let () (define* (hi a . b) b) (hi :a 1)) '())
  (test (let () (define* (hi a . b) b) (hi)) '())
  
  (test (let () (define* (hi a :rest b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi a :rest b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi a :rest b) b) (hi 1)) '())
  (test (let () (define* (hi a :rest b) b) (hi :a 1)) '())
  (test (let () (define* (hi a :rest b) b) (hi)) '())
  
  (test (let () (define* (hi :key a :rest b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi :key a :rest b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi :key a :rest b) b) (hi 1)) '())
  (test (let () (define* (hi :key a :rest b) b) (hi :a 1)) '())
  (test (let () (define* (hi :key a :rest b) b) (hi)) '())
  
  (test (let () (define* (hi :optional a :rest b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi :optional a :rest b) b) (hi :a 1 2 3)) '(2 3))
  (test (let () (define* (hi :optional a :rest b) b) (hi 1)) '())
  (test (let () (define* (hi :optional a :rest b) b) (hi :a 1)) '())
  (test (let () (define* (hi :optional a :rest b) b) (hi)) '())
  
  (test (let () (define* (hi (a 1) . b) b) (hi 1 2 3)) '(2 3))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi)) '(#f 22 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :a 1)) '(1 22 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :b 1)) '(#f 1 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :c 1)) '(#f 22 1))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :a 1 2)) '(1 2 ()))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :b 1 2 3)) 'error) ; b set twice
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :c 1 2 3)) '(#f 2 (3)))
  (test (let () (define* (hi a (b 22) . c) (list a b c)) (hi :b 1 :a 2 3)) '(2 1 (3)))

  (test (let () (define* (f (a :b)) a) (list (f) (f 1) (f :c) (f :a :c) (f :a 1) (f))) '(:b 1 :c :c 1 :b))
  (test (let () (define* (f a (b :c)) b) (f :b 1 :d)) 'error)
  
  (test (let () (define* (hi b) b) (procedure? hi)) #t)
  
  (test (let ()
	  (define (hi a) a)
	  (let ((tag (catch #t
			    (lambda () (hi 1 2 3))
			    (lambda args (car args)))))
	    (eq? tag 'wrong-number-of-args)))
	#t)
  
  (test (let ()
	  (define (hi a) a)
	  (let ((tag (catch #t
			    (lambda () (hi))
			    (lambda args (car args)))))
	    (eq? tag 'wrong-number-of-args)))
	#t)
  
  (test (let ()
	  (define* (hi a) a)
	  (let ((tag (catch #t
			    (lambda () (hi 1 2 3))
			    (lambda args (car args)))))
	    (eq? tag 'wrong-number-of-args)))
	#t)

  (test (let () (define (hi :a) :a) (hi 1)) 'error)
  (test (let () (define* (hi :a) :a) (hi 1)) 'error)
  (test (let () (define* (hi (:a 2)) a) (hi 1)) 'error)
  (test (let () (define* (hi (a 1) (:a 2)) a) (hi 1)) 'error)
  (test (let () (define* (hi (pi 1)) pi) (hi 2)) 'error)
  (test (let () (define* (hi (:b 1) (:a 2)) a) (hi)) 'error)

  (test (let () (define* (hi (a 1) (a 2)) a) (hi 2)) 'error)
  (test (let () (define (hi a a) a) (hi 1 2)) 'error)
  (test (let () (define hi (lambda (a a) a)) (hi 1 1)) 'error)
  (test (let () (define hi (lambda* ((a 1) (a 2)) a)) (hi 1 2)) 'error)
  (test (let () (define (hi (a 1)) a) (hi 1)) 'error)

  (let () 
    (define* (hi (a #2d((1 2) (3 4)))) (a 1 0))
    (test (hi) 3)
    (test (hi #2d((7 8) (9 10))) 9))

  (let () (define* (f :rest a) a) (test (f :a 1) '(:a 1)))
  (let () (define* (f :rest a :rest b) (list a b)) (test (f :a 1 :b 2) '((:a 1 :b 2) (1 :b 2))))

  (test (lambda :hi 1) 'error)
  (test (lambda (:hi) 1) 'error)
  (test (lambda (:hi . :hi) 1) 'error)
  (test (lambda (i . i) 1 . 2) 'error)
  (test (lambda (i i i i) (i)) 'error)
  (test (lambda "hi" 1) 'error)
  (test (lambda* ((i 1) i i) i) 'error)
  (test (lambda* ((a 1 2)) a) 'error)
  (test (lambda* ((a . 1)) a) 'error)
  (test (lambda* ((0.0 1)) 0.0) 'error)

  (test ((lambda* ((b 3) :rest x (c 1)) (list b c x)) 32) '(32 1 ()))
  (test ((lambda* ((b 3) :rest x (c 1)) (list b c x)) 1 2 3 4 5) '(1 3 (2 3 4 5)))
  (test ((lambda* ((a 1) :rest b :rest c) (list a b c)) 1 2 3 4 5) '(1 (2 3 4 5) (3 4 5)))

  (test (let () (define-macro (hi a a) `(+ ,a 1)) (hi 1 2)) 'error)

  (test (procedure-arity car) '(1 0 #f))
  (test (procedure-arity 'car) '(1 0 #f))
  (test (procedure-arity +) '(0 0 #t))
  (test (procedure-arity '+) '(0 0 #t))
  (test (procedure-arity log) '(1 1 #f))
  (test (procedure-arity '/) '(1 0 #t))
  (test (procedure-arity) 'error)
  (test (procedure-arity abs abs) 'error)
  (test (procedure-arity "hi") 'error)
					;(test (procedure-arity vector-set!) '(3 0 #f)) ; can be '(3 0 #t)
  (test (let ((hi (lambda () 1))) (procedure-arity hi)) '(0 0 #f))
  (test (let ((hi (lambda (a) 1))) (procedure-arity hi)) '(1 0 #f))
  (test (let ((hi (lambda (a b) 1))) (procedure-arity hi)) '(2 0 #f))
  (test (let ((hi (lambda (a . b) 1))) (procedure-arity hi)) '(1 0 #t))
  (test (let ((hi (lambda a 1))) (procedure-arity hi)) '(0 0 #t))
  
  (test (let () (define (hi) 1) (procedure-arity hi)) '(0 0 #f))
  (test (let () (define (hi a) a) (procedure-arity hi)) '(1 0 #f))
  (test (let () (define* (hi a) a) (procedure-arity hi)) '(0 1 #f))
  (test (let () (define* (hi a . b) a) (procedure-arity hi)) '(0 1 #t))
  (test (let () (define* (hi (a 1) (b 2)) a) (procedure-arity hi)) '(0 2 #f))
  (test (let ((hi (lambda* (a) 1))) (procedure-arity hi)) '(0 1 #f))
  (test (call/cc (lambda (func) (procedure-arity func))) '(0 0 #t))

  (test (procedure-arity (lambda* (a :rest b) a)) '(0 1 #t))
  (test (procedure-arity (lambda* (:optional a :rest b) a)) '(0 1 #t))
  (test (procedure-arity (lambda* (:optional a :key b :rest c) a)) '(0 2 #t))
  (test (procedure-arity (lambda* (:optional a b) a)) '(0 2 #f))
  (test (procedure-arity (lambda* (:rest args) args)) '(0 0 #t))
  (test (procedure-arity (lambda* (a :optional b . c) a)) '(0 2 #t))
  (test (procedure-arity (lambda* (:rest a . b) a)) '(0 0 #t))
  (test (procedure-arity (lambda* (:key :optional a) a)) '(0 1 #f))
  (test (procedure-arity (lambda* a a)) '(0 0 #t))
  (test (let () (define-macro (hi a) `(+ ,a 1)) (procedure-arity hi)) 'error)
  (test (procedure-arity (make-procedure-with-setter (lambda (a) a) (lambda (a b) a))) '(1 0 #f 2 0 #f))
  (test (procedure-arity (make-procedure-with-setter (lambda (a . b) a) (lambda (a b) a))) '(1 0 #t 2 0 #f))
  (test (procedure-arity (make-procedure-with-setter (lambda* (a :optional b) a) (lambda (a b) a))) '(0 2 #f 2 0 #f))

    
  (test (let ((c 1)) 
	  (define* (a #:optional (b c)) b) 
	  (set! c 2) 
	  (a))
	2)
  
  (test (let ((c 1)) 
	  (define* (a #:optional (b c)) b) 
	  (let ((c 32)) 
	    (a)))
	1)
  
  (test (let ((c 1)) 
	  (define* (a (b (+ c 1))) b) 
	  (set! c 2) 
	  (a))
	3)
  
  (test (let ((c 1))
	  (define* (a (b (+ c 1))) b)
	  (set! c 2)
	  (let ((c 123))
	    (a)))
	3)
  
  (test (let* ((cc 1)
	       (c (lambda () (set! cc (+ cc 1)) cc)))
	  (define* (a (b (c))) b)
	  (list cc (a) cc))
	(list 1 2 2))

  (for-each
   (lambda (arg)
     (test (procedure-arity arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (for-each
   (lambda (arg)
     (test (trace arg) 'error)
     (test (untrace arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (for-each
   (lambda (arg)
     (eval-string (format #f "(define (func) ~S)" arg))
     (let ((source (procedure-source func)))
       (let ((val (func)))
	 (test val arg))))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i #t #f '() '#(()) ':hi "hi"))
  
  (test (string=? (let () (define (hi) "this is a string" 1) (procedure-documentation hi)) "this is a string") #t)
  
  (for-each
   (lambda (arg)
     (test (procedure-environment arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (test (let ()
	  (define (hi a)
	    (let ((func (cdr (assoc '__func__ (car (procedure-environment hi))))))
	      (list (if (symbol? func) func (car func))
		    a)))
	  (hi 1))
	(list 'hi 1))

  (test (let ()
	  (define hi (let ((a 32)) 
		       (lambda (b) 
			 (+ a b))))
	  (define ho (with-environment 
		      (procedure-environment hi) 
		      (lambda (b) 
			(+ a b))))
	  (list (hi 1) (ho 1)))
	(list 33 33))

  (test (let ()
	  (define (hi a) (+ a 1))
	  (with-environment (procedure-environment hi) 
            ((eval (procedure-source hi)) 2)))
	3)
  
  (let ()
    (define-macro (window func beg end . body)
      `(call-with-exit
	(lambda (quit)
	  (do ((notes ',body (cdr notes)))
	      ((null? notes))
	    (let* ((note (car notes))
		   (note-beg (cadr note)))
	      (if (<= ,beg note-beg)
		  (if (> note-beg (+ ,beg ,end))
		      (quit)
		      (,func note))))))))
    
    (test 
     (let ((n 0))
       (window (lambda (a-note) (set! n (+ n 1))) 0 1 
	       (fm-violin 0 1 440 .1) 
	       (fm-violin .5 1 550 .1) 
	       (fm-violin 3 1 330 .1))
       n)
     2)
    
    (test 
     (let ((notes 0)
	   (env #f))
       (set! env (current-environment))
       (window (with-environment env (lambda (n) (set! notes (+ notes 1)))) 0 1 
	       (fm-violin 0 1 440 .1) 
	       (fm-violin .5 1 550 .1) 
	       (fm-violin 3 1 330 .1))
       notes)
     2))

  (test (let ()
	  (define-macro (window func beg end . body)
	    `(let ((e (current-environment)))
	       (call-with-exit
		(lambda (quit)
		  (do ((notes ',body (cdr notes)))
		      ((null? notes))
		    (let* ((note (car notes))
			   (note-beg (cadr note)))
		      (if (<= ,beg note-beg)
			  (if (> note-beg (+ ,beg ,end))
			      (quit)
			      ((with-environment e ,func) note)))))))))
	  
	  (let ((notes 0))
	    (window (lambda (n) (set! notes (+ notes 1))) 0 1 
		    (fm-violin 0 1 440 .1) 
		    (fm-violin .5 1 550 .1) 
		    (fm-violin 3 1 330 .1))
	    notes))
	2)


  (for-each
   (lambda (arg)
     (test (continuation? arg) #f))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2) "hi" (lambda () 1)))
  
  (test (let ((cont #f)) 
	  (and (call/cc (lambda (x) (set! cont x) (continuation? x)))
	       (continuation? cont)))
	#t)
  (test (let ((cont #f)) 
	  (or (call-with-exit (lambda (x) (set! cont x) (continuation? x)))
	       (continuation? cont)))
	#f)  ; ?? 
	
  (test (continuation?) 'error)
  (test (continuation? 1 2) 'error)

  
  (test (string? (s7-version)) #t)
  (test (eval-string "(+ 1 2)") 3)
  (test (eval '(+ 1 2)) 3)
  (test (eval `(+ 1 (eval `(* 2 3)))) 7)
  (test (eval `(+ 1 (eval-string "(* 2 3)"))) 7)
  (test (eval-string "(+ 1 (eval-string \"(* 2 3)\"))") 7)
  (test (eval `(+ 1 2 . 3)) 'error)

  (test (apply "hi" 1 ()) #\i)
  (test (eval ("hi" 1)) #\i)
  (test (apply + 1 1 (cons 1 (quote ()))) 3)
  (test (eq? (eval (quote (quote ()))) ()) #t)
  (test (apply (cons (quote cons) (cons 1 (quote ((quote ()))))) 1 ()) 1) ; essentially ((list 'cons 1 ...) 1) => 1
  (test (eval ((cons (quote cons) (cons 1 (quote ((quote ()))))) 1)) 1)
  (test (eval (eval (list '+ 1 2))) 3)


  (test (apply + (+ 1) ()) 1)
  (test (apply #(1) (+) ()) 1)
  (test (apply + (+) ()) 0)
  (test (eval #()) #())
  (test (apply (lambda () #f)) #f)
  (test (eval '(if #f #f)) (if #f #f))
  (test (let ((ho 32)) (symbol? (eval (eval (eval (eval '''''ho)))))) #t)
  (test (eval '(case 0 ((1) 2) ((0) 1))) 1)
  (test (eval '(cond ((= 1 2) 3) (#t 4))) 4)

  (test (eval-string (string-append "(list 1 2 3)" (string #\newline) (string #\newline))) (list 1 2 3))
  (eval-string (string-append "(define evalstr_1 32)" (string #\newline) "(define evalstr_2 2)"))
  (test (eval-string "(+ evalstr_1 evalstr_2)") 34)
  (eval-string (string-append "(set! evalstr_1 3)" "(set! evalstr_2 12)"))
  (test (eval-string "(+ evalstr_1 evalstr_2)") 15)
  
  (test (+ (eval `(values 1 2 3)) 4) 10)
  (test (+ (eval-string "(values 1 2 3)") 4) 10)
  (test (+ 1 (eval-string "(+ 2 3)") 4) 10)
  (test ((eval-string "(lambda (a) (+ a 1))") 2) 3)
  (test (eval ((eval-string "(lambda (a) (list '+ a 1))") 2)) 3)
  (test (eval-string "(+ 1 (eval (list '+ 1 2)))") 4)

  (for-each
   (lambda (arg)
     (test (eval-string arg) 'error))
   (list -1 0 1 512 #\a '#(1 2 3) 3.14 2/3 1.5+0.3i 1+i '() 'hi abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
  (for-each
   (lambda (arg)
     (test (eval-string "(+ 1 2)" arg) 'error))
   (list -1 0 1 512 #\a '#(1 2 3) 3.14 2/3 1.5+0.3i 1+i 'hi abs "hi" '#(()) (lambda () 1)))


  
  (test (string=? (procedure-documentation abs) "(abs x) returns the absolute value of the real number x") #t)
  (test (string=? (procedure-documentation 'abs) "(abs x) returns the absolute value of the real number x") #t)
  (test (let ((hi (lambda (x) "this is a test" (+ x 1)))) 
	  (list (hi 1) (procedure-documentation hi)))
	(list 2 "this is a test"))
  (test (procedure-documentation (lambda* (a b) "docs" a)) "docs")
  
  (for-each
   (lambda (arg)
     (test (procedure-documentation arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))
  
  (test (let ((hi (lambda (x) (+ x 1)))) (procedure-source hi)) '(lambda (x) (+ x 1)))
  
  (for-each
   (lambda (arg)
     (test (procedure-source arg) 'error))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (test (procedure-documentation) 'error)
  (test (procedure-documentation abs abs) 'error)
  (test (procedure-arity) 'error)
  (test (procedure-arity abs abs) 'error)
  (test (procedure-source) 'error)
  (test (procedure-source abs abs) 'error)

  
  (test (make-list 0) '())
  (test (make-list 0 123) '())
  (test (make-list 1) '(#f))
  (test (make-list 1 123) '(123))
  (test (make-list 1 '()) '(()))
  (test (make-list 2) '(#f #f))
  (test (make-list 2 1) '(1 1))
  (test (make-list 2 (make-list 1 1)) '((1) (1)))
  (test (make-list -1) 'error)
  
  (for-each
   (lambda (arg)
     (test (make-list arg) 'error))
   (list #\a '#(1 2 3) 3.14 3/4 1.0+1.0i '() #t 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (for-each
   (lambda (arg)
     (test ((make-list 1 arg) 0) arg))
   (list #\a '#(1 2 3) 3.14 3/4 1.0+1.0i '() #f 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))

  (test (make-list) 'error)
  (test (make-list 1 2 3) 'error)

  
  (test (let () (defmacro hiho (a) `(+ ,a 1)) (hiho 3)) 4)
  (test (let () (defmacro hiho () `(+ 3 1)) (hiho)) 4)
  (test (let () (defmacro hiho () `(+ 3 1)) (hiho 1)) 'error)
  (test (let () (defmacro hi (a) `(+ ,@a)) (hi (1 2 3))) 6)
  (test (let () (defmacro hi (a) `(+ ,a 1) #f) (hi 2)) #f)
  
  (test (let () (define-macro (hiho a) `(+ ,a 1)) (hiho 3)) 4)
  (test (let () (define-macro (hiho) `(+ 3 1)) (hiho)) 4)
  (test (let () (define-macro (hiho) `(+ 3 1)) (hiho 1)) 'error)
  (test (let () (define-macro (hi a) `(+ ,@a)) (hi (1 2 3))) 6)
  (test (let () (define-macro (hi a) `(+ ,a 1) #f) (hi 2)) #f)
  (test (let () (define-macro (mac1 a) `',a) (equal? (mac1 (+ 1 2)) '(+ 1 2))) #t)
  
  (test (let () (defmacro hi (a) `(+ , a 1)) (hi 1)) 2)
  (test (let () (defmacro hi (a) `(eval `(+ ,,a 1))) (hi 1)) 2)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(+ ,,a 1)))) (hi 1)) 2)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(+ ,a 1)))) (hi 1)) 13)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(let ((a 100)) (+ ,a 1))))) (hi 1)) 13)
  (test (let () (defmacro hi (a) `(eval (let ((a 12)) `(let ((a 100)) (+ a 1))))) (hi 1)) 101)
  
  (test (let () (defmacro hi (q) ``(,,q)) (hi (* 2 3))) '(6))
  (test (let () (defmacro hi (q) `(let ((q 32)) `(,,q))) (hi (* 2 3))) '(6))
  (test (let () (defmacro hi (q) `(let ((q 32)) `(,q))) (hi (* 2 3))) '(32))
  (test (let () (defmacro hi (q) `(let () ,@(list q))) (hi (* 2 3))) 6)

  (test (let () (define-macro (tst a) ``(+ 1 ,,a)) (tst 2)) '(+ 1 2))
  (test (let () (define-macro (tst a) ```(+ 1 ,,,a)) (eval (tst 2))) '(+ 1 2))
  (test (let () (define-macro (tst a) ``(+ 1 ,,a)) (tst (+ 2 3))) '(+ 1 5))
  (test (let () (define-macro (tst a) ``(+ 1 ,@,a)) (tst '(2 3))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ``(+ 1 ,,@a)) (tst (2 3))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ```(+ 1 ,,,@a)) (eval (tst (2 3)))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ```(+ 1 ,,@,@a)) (eval (tst ('(2 3))))) '(+ 1 2 3))
  (test (let () (define-macro (tst a) ````(+ 1 ,,,,@a)) (eval (eval (eval (tst (2 3)))))) 6)
  (test (let () (define-macro (tst a) ``(+ 1 ,@,@a)) (tst ('(2 3)))) '(+ 1 2 3))
  (test (let () (define-macro (tst a b) `(+ 1 ,a (apply * `(2 ,,@b)))) (tst 3 (4 5))) 44)
  (test (let () (define-macro (tst . a) `(+ 1 ,@a)) (tst 2 3)) 6)
  (test (let () (define-macro (tst . a) `(+ 1 ,@a (apply * `(2 ,,@a)))) (tst 2 3)) 18)
  (test (let () (define-macro (tst a) ```(+ 1 ,@,@,@a)) (eval (tst ('('(2 3)))))) '(+ 1 2 3))

  (test (let () (define-macro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
  (test (let () (define-macro (hi a) `(let ((@ 32)) (+ @ ,a))) (hi @)) 64)
  (test (let () (define-macro (hi @) `(+ 1 ,@@)) (hi (2 3))) 6) ; ,@ is ambiguous
  (test (let () (define-macro (tst a) `(+ 1 (if (> ,a 0) (tst (- ,a 1)) 0))) (tst 3)) 4)
  (test (let () (define-macro (hi a) (if (list? a) `(+ 1 ,@a) `(+ 1 ,a))) (* (hi 1) (hi (2 3)))) 12)

  (test (let () (define-bacro (hiho a) `(+ ,a 1)) (hiho 3)) 4)
  (test (let () (define-bacro (hiho) `(+ 3 1)) (hiho)) 4)
  (test (let () (define-bacro (hiho) `(+ 3 1)) (hiho 1)) 'error)
  (test (let () (define-bacro (hi a) `(+ ,@a)) (hi (1 2 3))) 6)
  (test (let () (define-bacro (hi a) `(+ ,a 1) #f) (hi 2)) #f)
  (test (let () (define-bacro (mac1 a) `',a) (equal? (mac1 (+ 1 2)) '(+ 1 2))) #t)
  (test (let () (define-bacro (tst a) ``(+ 1 ,,a)) (tst 2)) '(+ 1 2))
  (test (let () (define-bacro (tst a) ```(+ 1 ,,,a)) (eval (tst 2))) '(+ 1 2))
  (test (let () (define-bacro (tst a) ``(+ 1 ,,a)) (tst (+ 2 3))) '(+ 1 5))
  (test (let () (define-bacro (tst a) ``(+ 1 ,@,a)) (tst '(2 3))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ``(+ 1 ,,@a)) (tst (2 3))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ```(+ 1 ,,,@a)) (eval (tst (2 3)))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ```(+ 1 ,,@,@a)) (eval (tst ('(2 3))))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a) ````(+ 1 ,,,,@a)) (eval (eval (eval (tst (2 3)))))) 6)
  (test (let () (define-bacro (tst a) ``(+ 1 ,@,@a)) (tst ('(2 3)))) '(+ 1 2 3))
  (test (let () (define-bacro (tst a b) `(+ 1 ,a (apply * `(2 ,,@b)))) (tst 3 (4 5))) 44)
  (test (let () (define-bacro (tst . a) `(+ 1 ,@a)) (tst 2 3)) 6)
  (test (let () (define-bacro (tst . a) `(+ 1 ,@a (apply * `(2 ,,@a)))) (tst 2 3)) 18)
  (test (let () (define-bacro (tst a) ```(+ 1 ,@,@,@a)) (eval (tst ('('(2 3)))))) '(+ 1 2 3))
  (test (let () (define-bacro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
  (test (let () (define-bacro (hi a) `(let ((@ 32)) (+ @ ,a))) (hi @)) 64)
  (test (let () (define-bacro (hi @) `(+ 1 ,@@)) (hi (2 3))) 6) ; ,@ is ambiguous
  (test (let () (define-bacro (tst a) `(+ 1 (if (> ,a 0) (tst (- ,a 1)) 0))) (tst 3)) 4)
  (test (let () (define-bacro (hi a) (if (list? a) `(+ 1 ,@a) `(+ 1 ,a))) (* (hi 1) (hi (2 3)))) 12)

  (test (defmacro) 'error)
  (test (define-macro) 'error)
  (test (defmacro 1 2 3) 'error)
  (test (define-macro (1 2) 3) 'error)
  (test (defmacro a) 'error)
  (test (define-macro (a)) 'error)
  (test (defmacro a (1) 2) 'error)
  (test (define-macro (a 1) 2) 'error)
  (test (defmacro . a) 'error)
  (test (define-macro . a) 'error)
  (test (define :hi 1) 'error)
  (test (define hi: 1) 'error)
  (test (define-macro (:hi a) `(+ ,a 1)) 'error)
  (test (defmacro :hi (a) `(+ ,a 1)) 'error)
  (test (defmacro hi (1 . 2) 1) 'error)
  (test (defmacro hi 1 . 2) 'error)
  (test (defmacro : "" . #(1)) 'error)
  (test (defmacro : #(1) . :) 'error)
  (test (defmacro hi ()) 'error)
  (test (define-macro (mac . 1) 1) 'error)
  (test (define-macro (mac 1) 1) 'error)
  (test (define-macro (a #()) 1) 'error)
  (test (define-macro (i 1) => (j 2)) 'error)
  (test (define hi 1 . 2) 'error)

  ;(test (let () (define-macro (hi a b) `(list ,@a . ,@b)) (hi (1 2) ((2 3)))) '(1 2 2 3))
  (test (let () (define-macro (hi a b) `(list ,@a . ,b)) (hi (1 2) (2 3))) '(1 2 2 3))

  (test (let ()
	  (define-macro (hanger name-and-args)
	    `(define ,(car name-and-args)
	       (+ ,@(map (lambda (arg) arg) (cdr name-and-args)))))
	  (hanger (hi 1 2 3))
	  hi)
	6)
  (test (let ()
	  (define-macro (hanger name-and-args)
	    `(define-macro (,(car name-and-args))
	       `(+ ,@(map (lambda (arg) arg) (cdr ',name-and-args)))))
	  (hanger (hi 1 2 3))
	  (hi))
	6)

  (let ()
    ;; inspired by Doug Hoyte, "Let Over Lambda"
    (define (mcxr path lst)
      (define (cxr-1 path lst)
	(if (null? path)
	    lst
	    (if (char=? (car path) #\a)
		(cxr-1 (cdr path) (car lst))
		(cxr-1 (cdr path) (cdr lst)))))
      (let ((p (string->list (symbol->string path))))
	(if (char=? (car p) #\c)
	    (set! p (cdr p)))
	(let ((p (reverse p)))
	  (if (char=? (car p) #\r)
	      (set! p (cdr p)))
	  (cxr-1 p lst))))
    
    (test (mcxr 'cr '(1 2 3)) '(1 2 3))
    (test (mcxr 'cadddddddr '(1 2 3 4 5 6 7 8)) 8)
    (test (mcxr 'caadadadadadadadr '(1 (2 (3 (4 (5 (6 (7 (8))))))))) 8)
    
    (define-macro (cxr path lst)
      (let ((p (string->list (symbol->string path))))
	(if (char=? (car p) #\c)
	    (set! p (cdr p)))
	(let ((p (reverse p)))
	  (if (char=? (car p) #\r)
	      (set! p (cdr p)))
	  (let ((func 'arg))
	    (for-each
	     (lambda (f)
	       (set! func (list (if (char=? f #\a) 'car 'cdr) func)))
	     p)
	    `((lambda (arg) ,func) ,lst)))))
    
    (test (cxr car '(1 2 3)) 1)
    (test (cxr cadddddddr '(1 2 3 4 5 6 7 8)) 8)
    (test (cxr caadadadadadadadr '(1 (2 (3 (4 (5 (6 (7 (8))))))))) 8)
    )

  ;; this is the best of them!
  (let ()
    (define-macro (c?r path)
      ;; here "path" is a list and "X" marks the spot in it that we are trying to access
      ;; (a (b ((c X)))) -- anything after the X is ignored, other symbols are just placeholders
      ;; c?r returns a function that gets X

      ;; maybe ... for cdr? (c?r (a ...);  right now it's using dot: (c?r (a . X)) -> cdr
      
      ;; (c?r (a b X)) -> caddr, 
      ;; (c?r (a (b X))) -> cadadr
      ;; ((c?r (a a a X)) '(1 2 3 4 5 6)) -> 4
      ;; ((c?r (a (b c X))) '(1 (2 3 4))) -> 4
      ;; ((c?r (((((a (b (c (d (e X)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) -> 6
      ;; ((c?r (((((a (b (c (X (e f)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) -> 4
      ;; (procedure-source (c?r (((((a (b (c (X (e f))))))))))) -> (lambda (lst) (car (car (cdr (car (cdr (car (cdr (car (car (car (car lst))))))))))))
      
      (define (X-marks-the-spot accessor tree)
	(if (pair? tree)
	    (or (X-marks-the-spot (cons 'car accessor) (car tree))
		(X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	    (if (eq? tree 'X)
		accessor
		#f)))
      
      (let ((accessor (X-marks-the-spot '() path)))
	(if (not accessor)
	    (error "can't find the spot! ~A" path)
	    (let ((len (length accessor)))
	      (if (< len 5)                   ; it's a built-in function
		  (let ((name (make-string (+ len 2))))
		    (set! (name 0) #\c)
		    (set! (name (+ len 1)) #\r)
		    (do ((i 0 (+ i 1))
			 (a accessor (cdr a)))
			((= i len))
		      (set! (name (+ i 1)) (if (eq? (car a) 'car) #\a #\d)))
		    (string->symbol name))
		  (let ((body 'lst))          ; make a new function to find the spot
		    (for-each
		     (lambda (f)
		       (set! body (list f body)))
		     (reverse accessor))
		    `(lambda (lst) ,body)))))))
    
    (test ((c?r (a b X)) (list 1 2 3 4)) 3)
    (test ((c?r (a (b X))) '(1 (2 3) ((4)))) 3)
    (test ((c?r (a a a X)) '(1 2 3 4 5 6)) 4)
    (test ((c?r (a (b c X))) '(1 (2 3 4))) 4)
    (test ((c?r (((((a (b (c (d (e X)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) 6)
    (test ((c?r (((((a (b (c (X (e f)))))))))) '(((((1 (2 (3 (4 (5 6)))))))))) 4))

  (let ()
    (define-macro (nested-for-each args func . lsts)
      (let ((body `(,func ,@args)))
	(for-each
	 (lambda (arg lst)
	   (set! body `(for-each
			(lambda (,arg)
			  ,body)
			,lst)))
	 args lsts)
	body))
    
    ;;(nested-for-each (a b) + '(1 2) '(3 4)) ->
    ;;  (for-each (lambda (b) (for-each (lambda (a) (+ a b)) '(1 2))) '(3 4))
    
    (define-macro (nested-map args func . lsts)
      (let ((body `(,func ,@args)))
	(for-each
	 (lambda (arg lst)
	   (set! body `(map
			(lambda (,arg)
			  ,body)
			,lst)))
	 args lsts)
	body))
    
    ;;(nested-map (a b) + '(1 2) '(3 4))
    ;;   ((4 5) (5 6))
    ;;(nested-map (a b) / '(1 2) '(3 4))
    ;;   ((1/3 2/3) (1/4 1/2))

    (test (nested-map (a b) + '(1 2) '(3 4)) '((4 5) (5 6)))
    (test (nested-map (a b) / '(1 2) '(3 4)) '((1/3 2/3) (1/4 1/2)))
    )
    
  (let ()
    (define-macro (define-curried name-and-args . body)	
      `(define ,@(let ((newlst `(begin ,@body)))
		   (define (rewrap lst)
		     (if (pair? (car lst))
			 (begin
			   (set! newlst (cons 'lambda (cons (cdr lst) (list newlst))))
			   (rewrap (car lst)))
			 (list (car lst) (list 'lambda (cdr lst) newlst))))
		   (rewrap name-and-args))))

    (define-curried (((((f a) b) c) d) e) (* a b c d e))
    (test (((((f 1) 2) 3) 4) 5) 120)
    (define-curried (((((f a b) c) d e) f) g) (* a b c d e f g))
    (test (((((f 1 2) 3) 4 5) 6) 7) 5040))


  
  (define-macro (eval-case key . clauses)
    ;; case with evaluated key-lists
    `(cond ,@(map (lambda (lst)
		    (if (pair? (car lst))
			(cons `(member ,key (list ,@(car lst)))
			      (cdr lst))
			lst))
		  clauses)))

  (test (let ((a 1) (b 2)) (eval-case 1 ((a) 123) ((b) 321) (else 0))) 123)
  (test (let ((a 1) (b 2) (c 3)) (eval-case 3 ((a c) 123) ((b) 321) (else 0))) 123)
  (test (let ((a 1) (b 2)) (eval-case 3 ((a) 123) ((b) 321) (((+ a b)) -1) (else 0))) -1)
  (test (let ((a 1) (b 2)) (eval-case 6 ((a (* (+ a 2) b)) 123) ((b) 321) (((+ a b)) -1) (else 0))) 123)

  (test (let ()
	  (define (set-cadr! a b)
	    (set-car! (cdr a) b)
	    b)
	  (let ((lst (list 1 2 3)))
	    (set-cadr! lst 32)
	    lst))
	'(1 32 3))

  (test (macro? eval-case) #t)
  (test (macro? pi) #f)
  (for-each
   (lambda (arg)
     (test (macro? arg) #f))
   (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi '#(()) (list 1 2 3) '(1 . 2) "hi"))
  (test (macro?) 'error)
  
  (define-macro (fully-expand form)
    (define (expand form)
      ;; walk form looking for macros, expand any that are found
      (if (pair? form)
	  (if (macro? (car form))
	      (expand ((eval (procedure-source (car form))) form))
	      (cons (expand (car form))
		    (expand (cdr form))))
	  form))
    (expand form))

  (define fe1-called #f)
  (define-macro (fe1 a) (set! fe1-called #t) `(+ ,a 1))
  (define fe2-called #f)
  (define-macro (fe2 b) (set! fe2-called #f) `(+ (fe1 ,b) 2))
  (fully-expand (define (fe3 c) (+ (fe2 c) (fe1 (+ c 1)))))
  (set! fe1-called #f)
  (set! fe2-called #f)
  (let ((val (fe3 3)))
    (if (or (not (= val 11))
	    fe1-called
	    fe2-called)
	(format #t "fully-expand: ~A ~A ~A ~A~%" val (procedure-source fe3) fe1-called fe2-called)))

  (test (let ()
	  (define-macro (pop sym)
	    (let ((v (gensym "v")))
	      `(let ((,v (car ,sym)))
		 (set! ,sym (cdr ,sym))
		 ,v)))
	  (let ((lst (list 1 2 3)))
	    (let ((val (pop lst)))
	      (and (= val 1)
		   (equal? lst (list 2 3))))))
	#t)

  (define-macro (destructuring-bind lst expr . body)
    `(let ((ex ,expr))
       
       (define (flatten lst)
	 (cond ((null? lst) '())
	       ((pair? lst)
		(if (pair? (car lst))
		    (append (flatten (car lst)) (flatten (cdr lst)))
		    (cons (car lst) (flatten (cdr lst)))))
	       (#t lst)))
       
       (define (structures-equal? l1 l2)
	 (if (pair? l1)
	     (and (pair? l2)
		  (structures-equal? (car l1) (car l2))
		  (structures-equal? (cdr l1) (cdr l2)))
	     (not (pair? l2))))
       
       (if (not (structures-equal? ',lst ex))
	   (error "~A and ~A do not match" ',lst ex))
       
       (let ((names (flatten ',lst))
	     (vals (flatten ex)))
	 (apply (eval (list 'lambda names ',@body)) vals))))
  
  (test (destructuring-bind (a b) (list 1 2) (+ a b)) 3)
  (test (destructuring-bind ((a) b) (list (list 1) 2) (+ a b)) 3)
  (test (destructuring-bind (a (b c)) (list 1 (list 2 3)) (+ a b c)) 6)
  (test (let ((x 1)) (destructuring-bind (a b) (list x 2) (+ a b))) 3)

  (define-macro (define-clean-macro name-and-args . body)
    ;; the new backquote implementation breaks this slightly -- it's currently confused about unquoted nil in the original
    (let ((syms ()))
      
      (define (walk func lst)
	(if (and (func lst)
		 (pair? lst))
	    (begin
	      (walk func (car lst))
	      (walk func (cdr lst)))))
      
      (define (car-member sym lst)
	(if (null? lst)
	    #f
	    (if (eq? sym (caar lst))
		(cdar lst)
		(car-member sym (cdr lst)))))
      
      (define (walker val)
	(if (pair? val)
	    (if (eq? (car val) 'quote)
		(or (car-member (cadr val) syms)
		    (and (pair? (cadr val))
			 (or (and (eq? (caadr val) 'quote) ; 'sym -> (quote (quote sym))
				  val)
			     (append (list 'list) 
				     (walker (cadr val)))))
		    (cadr val))
		(cons (walker (car val))
		      (walker (cdr val))))
	    (or (car-member val syms)
		val)))
      
      (walk (lambda (val)
	      (if (and (pair? val)
		       (eq? (car val) 'quote)
		       (symbol? (cadr val))
		       (not (car-member (cadr val) syms)))
		  (set! syms (cons 
			      (cons (cadr val) 
				    (gensym (symbol->string (cadr val))))
			      syms)))
	      (or (not (pair? val))
		  (not (eq? (car val) 'quote))
		  (not (pair? (cadr val)))
		  (not (eq? (caadr val) 'quote))))
	    body)
      
      (let* ((new-body (walker body))
	     (new-syms (map (lambda (slot)
			      (list (cdr slot) `(gensym)))
			    syms))
	     (new-globals 
	      (let ((result '()))
		(for-each
		 (lambda (slot)
		   (if (defined? (car slot))
		       (set! result (cons
				     (list 'set! (cdr slot) (car slot))
				     result))))
		 syms)
		result)))
	
	`(define-macro ,name-and-args 
	   (let ,new-syms
	     ,@new-globals
	     `(begin ,,@new-body))))))


  (define-macro (define-immaculo name-and-args . body)
    (let* ((gensyms (map (lambda (g) (gensym)) (cdr name-and-args)))
	   (args (cdr (copy name-and-args)))
	   (name (car name-and-args))
	   (set-args (map (lambda (a g) `(list ',g ,a)) args gensyms))
	   (get-args (map (lambda (a g) `(quote (cons ',a ,g))) args gensyms))
	   (blocked-args (map (lambda (a) `(,a ',a)) args))
	   (new-body (list (eval `(let (,@blocked-args) ,@body)))))
      `(define-macro ,name-and-args
	 `(let ,(list ,@set-args)
	    ,(list 'with-environment 
		   (append (list 'augment-environment) 
			   (list (list 'procedure-environment ,name)) 
			   (list ,@get-args))
		   ',@new-body)))))
  
  (test (let ()
	  (define-clean-macro (hi a) `(+ ,a 1))
	  (hi 1))	  
	2)
  
  (test (let ()
	  (define-immaculo (hi a) `(+ ,a 1))
	  (hi 1))	  
	2)
  
  (test (let ()
	  (define-clean-macro (hi a) `(+ ,a 1))
	  (let ((+ *)
		(a 12))
	    (hi a)))
	13)
  
  (test (let ()
	  (define-immaculo (hi a) `(+ ,a 1))
	  (let ((+ *)
		(a 12))
	    (hi a)))
	13)
  
  (test (let ()
	  (define-clean-macro (hi a) `(let ((b 23)) (+ b ,a)))
	  (hi 2))
	25)
  
  (test (let ()
	  (define-immaculo (hi a) `(let ((b 23)) (+ b ,a)))
	  (hi 2))
	25)
  
  (test (let ()
	  (define-clean-macro (hi a) `(let ((b 23)) (+ b ,a)))
	  (let ((+ *)
		(b 12))
	    (hi b)))
	35)
  
  (test (let ()
	  (define-immaculo (hi a) `(let ((b 23)) (+ b ,a)))
	  (let ((+ *)
		(b 12))
	    (hi b)))
	35)
  
  (test (let ()
	  (define-clean-macro (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
	  (mac 2 3))
	360)
  
  (test (let ()
	  (define-immaculo (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
	  (mac 2 3))
	360)
  
  (test (let ()
	  (define-clean-macro (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
	  (let ((c 2)
		(d 3))
	    (mac c d)))
	360)
  
  (test (let ()
	  (define-immaculo (mac a b) `(let ((c (+ ,a ,b))) (let ((d 12)) (* ,a ,b c d))))
	  (let ((c 2)
		(d 3))
	    (mac c d)))
	360)
  
  (test (let ()
	  (define-clean-macro (mac a . body)
	    `(+ ,a ,@body))
	  (mac 2 3 4))
	9)

  (test (let ()
	  (define-clean-macro (mac a . body)
	    `(+ ,a ,@body))
	  (let ((a 2)
		(+ *))
	    (mac a (- 5 a) (* a 2))))
	9)

  (test (let ()
	  (define-clean-macro (mac) (let ((a 1)) `(+ ,a 1)))
	  (mac))
	2)

  (test (let ()
	  (define-immaculo (mac) (let ((a 1)) `(+ ,a 1)))
	  (mac))
	2)

  (test (let ()
	  (define-immaculo (hi a) `(list 'a ,a))
	  (hi 1))
	(list 'a 1))

  (test (let ()
	  (define-immaculo (mac c d) `(let ((a 12) (b 3)) (+ a b ,c ,d)))
	  (let ((a 21) (b 10) (+ *)) (mac a b)))
	46)

;  (test (let ((values 32)) (define-macro (hi a) `(+ 1 ,@a)) (hi (2 3))) 6)
;  (test (let ((list 32)) (define-macro (hi a) `(+ 1 ,@a)) (hi (2 3))) 6)
;  (test (let () (define-macro (hi a) `(let ((apply 32)) (+ apply ,@a))) (hi (2 3))))
  (test (let () (define-macro (hi a) `(+ 1 (if ,(= a 0) 0 (hi ,(- a 1))))) (hi 3)) 4)
  (test (let () (define-macro (hi a) `(+ 1 ,a)) ((if #t hi abs) -3)) -2)
  (test (let () (apply define-macro '((m a) `(+ 1 ,a))) (m 2)) 3)
  (test (let () (apply (eval (apply define-macro '((m a) `(+ 1 ,a)))) '(3))) 4)
  (test (let () (apply (eval (apply define '((hi a) (+ a 1)))) '(2))) 3)
  (test (let () ((eval (apply define '((hi a) (+ a 1)))) 3)) 4)
  (test (let () ((eval (apply define-macro '((m a) `(+ 1 ,a)))) 3)) 4)
  (test (let () ((symbol->value (apply define '((hi a) (+ a 1)))) 3)) 4)
  (test (let () ((symbol->value (apply define-macro '((m a) `(+ 1 ,a)))) 3)) 4)
  (test (let () 
	  (define-macro (mu args . body)
	    (let ((m (gensym)))
	      `(symbol->value (apply define-macro '((,m ,@args) ,@body)))))
	  ((mu (a) `(+ 1 ,a)) 3))
	4)
  (test (let () (define-macro (hi a) `(+ 1 ,a)) (map hi '(1 2 3))) 'error)

  (define-macro* (_mac1_) `(+ 1 2))
  (test (_mac1_) 3)
  (define-macro* (_mac2_ a) `(+ ,a 2))
  (test (_mac2_ 1) 3)
  (test (_mac2_ :a 2) 4)
  (define-macro* (_mac3_ (a 1)) `(+ ,a 2))
  (test (_mac3_) 3)
  (test (_mac3_ 3) 5)
  (test (_mac3_ :a 0) 2)
  (define-macro* (_mac4_ (a 1) (b 2)) `(+ ,a ,b))
  (test (_mac4_) 3)
  (test (_mac4_ :b 3) 4)
  (test (_mac4_ 2 :b 3) 5)
  (test (_mac4_ :b 10 :a 12) 22)
  (test (_mac4_ :a 4) 6)

  (define-bacro* (_mac21_) `(+ 1 2))
  (test (_mac21_) 3)
  (define-bacro* (_mac22_ a) `(+ ,a 2))
  (test (_mac22_ 1) 3)
  (test (_mac22_ :a 2) 4)
  (define-bacro* (_mac23_ (a 1)) `(+ ,a 2))
  (test (_mac23_) 3)
  (test (_mac23_ 3) 5)
  (test (_mac23_ :a 0) 2)
  (define-bacro* (_mac24_ (a 1) (b 2)) `(+ ,a ,b))
  (test (_mac24_) 3)
  (test (_mac24_ :b 3) 4)
  (test (_mac24_ 2 :b 3) 5)
  (test (_mac24_ :b 10 :a 12) 22)
  (test (_mac24_ :a 4) 6)  
  
  (defmacro* _mac11_ () `(+ 1 2))
  (test (_mac11_) 3)
  (defmacro* _mac12_ (a) `(+ ,a 2))
  (test (_mac12_ 1) 3)
  (test (_mac12_ :a 2) 4)
  (defmacro* _mac13_ ((a 1)) `(+ ,a 2))
  (test (_mac13_) 3)
  (test (_mac13_ 3) 5)
  (test (_mac13_ :a 0) 2)
  (defmacro* _mac14_ ((a 1) (b 2)) `(+ ,a ,b))
  (test (_mac14_) 3)
  (test (_mac14_ :b 3) 4)
  (test (_mac14_ 2 :b 3) 5)
  (test (_mac14_ :b 10 :a 12) 22)
  (test (_mac14_ :a 4) 6)

  (define-bacro (symbol-set! var val) `(set! ,(symbol->value var) ,val))
  (test (let ((x 32) (y 'x)) (symbol-set! y 123) (list x y)) '(123 x))


  (let ()
    (define-macro (hi a) `````(+ ,,,,,a 1))
    (test (eval (eval (eval (eval (hi 2))))) 3)

    (define-macro (hi a) `(+ ,@@a))
    (test (hi (1 2 3)) 'error)

    (define-macro (hi @a) `(+ ,@@a))
    (test (hi (1 2 3)) 6))


  (let ()
    (set! *#readers* (list (cons #\s (lambda (str) 123))))
    (let ((val (eval-string "(+ 1 #s1)"))) ; force this into the current reader
      (test val 124))
    (set! *#readers* '()))
  
  (begin
    (define-macro (hi a) `(+ ,a 1))
    (test (hi 2) 3)
    (let ()
      (define (ho b) (+ 1 (hi b)))
      (test (ho 1) 3))
    (let ((hi 32))
      (test (+ hi 1) 33))
    (letrec ((hi (lambda (a) (if (= a 0) 0 (+ 2 (hi (- a 1)))))))
      (test (hi 3) 6))
    (letrec* ((hi (lambda (a) (if (= a 0) 0 (+ 2 (hi (- a 1)))))))
      (test (hi 3) 6))
    (test (equal? '(hi 1) (quote (hi 1))) #t)
    (test (list? '(hi 1)) #t)
    (test (list? '(((hi 1)))) #t)
    (test (equal? (vector (hi 1)) '#(2)) #t)
    (test (symbol? (vector-ref '#(hi) 0)) #t))

  (define-macro (define-with-goto name-and-args . body)
    ;; run through the body collecting label accessors, (label name)
    ;; run through getting goto positions, (goto name)
    ;; tie all the goto's to their respective labels (via set-cdr! essentially)
    
    (define (find-accessor type)
      (let ((labels '()))
	(define (gather-labels accessor tree)
	  (if (pair? tree)
	      (if (equal? (car tree) type)
		  (begin
		    (set! labels (cons (cons (cadr tree) 
					     (let ((body 'lst))
					       (for-each
						(lambda (f)
						  (set! body (list f body)))
						(reverse (cdr accessor)))
					       (make-procedure-with-setter
						(apply lambda '(lst) (list body))
						(apply lambda '(lst val) `((set! ,body val))))))
				       labels))
		    (gather-labels (cons 'cdr accessor) (cdr tree)))
		  (begin
		    (gather-labels (cons 'car accessor) (car tree))
		    (gather-labels (cons 'cdr accessor) (cdr tree))))))
	(gather-labels '() body)
	labels))
    (let ((labels (find-accessor 'label))
	  (gotos (find-accessor 'goto)))
      (if (not (null? gotos))
	  (for-each
	   (lambda (goto)
	     (let* ((name (car goto))
		    (goto-accessor (cdr goto))
		    (label (assoc name labels))
		    (label-accessor (and label (cdr label))))
	       (if label-accessor
		   (set! (goto-accessor body) (label-accessor body))
		   (error 'bad-goto "can't find label: ~S" name))))
	   gotos))
      `(define ,name-and-args
	 (let ((label (lambda (name) #f))
	       (goto (lambda (name) #f)))
	   ,@body))))
  
  (let ()
    (define-with-goto (g1 a)
      (let ((x 1))
	(if a
	    (begin
	      (set! x 2)
	      (goto 'the-end)
	      (set! x 3))
	    (set! x 4))
	(label 'the-end)
	x))

    (define-with-goto (g2 a)
      (let ((x a))
	(label 'start)
	(if (< x 4)
	    (begin
	      (set! x (+ x 1))
	      (goto 'start)))
	x))
    
    (test (g1 #f) 4)
    (test (g1 #t) 2)
    (test (g2 1) 4)
    (test (g2 32) 32))

  
  (let ()
    (define special-value
      (let ((type (make-type)))
	((cadr type) 'special)))

    (test (eq? special-value special-value) #t)
    (test (eqv? special-value special-value) #t)
    (test (equal? special-value special-value) #t)
    (test (procedure? special-value) #f)
    (for-each
     (lambda (arg)
       (test (or (eq? arg special-value)
		 (eqv? arg special-value)
		 (equal? arg special-value))
	     #f))
       (list "hi" -1 #\a 1 'special 3.14 3/4 1.0+1.0i #f #t '(1 . 2) #<unspecified> #<undefined>))

    (begin
      (define rec? #f)
      (define make-rec #f)
      (define rec-a #f)
      (define rec-b #f)

      (let* ((rec-type (make-type))
	     (? (car rec-type))
	     (make (cadr rec-type))
	     (ref (caddr rec-type)))

	(set! make-rec (lambda* ((a 1) (b 2))
				(make (vector a b))))

	(set! rec? (lambda (obj)
		     (? obj)))
  
	(set! rec-a (make-procedure-with-setter
		     (lambda (obj)
		       (and (rec? obj)
			    (vector-ref (ref obj) 0)))
		     (lambda (obj val)
		       (if (rec? obj)
			   (vector-set! (ref obj) 0 val)))))

	(set! rec-b (make-procedure-with-setter
		     (lambda (obj)
		       (and (rec? obj)
			    (vector-ref (ref obj) 1)))
		     (lambda (obj val)
		       (if (rec? obj)
			   (vector-set! (ref obj) 1 val)))))))

    (let ((hi (make-rec 32 '(1 2))))
      (test (rec? hi) #t)
      (test (equal? hi hi) #t)
      (test (rec? 32) #f)
      (test (rec-a hi) 32)
      (test (rec-b hi) '(1 2))
      (set! (rec-b hi) 123)
      (test (rec-b hi) 123)
      (let ((ho (make-rec 32 '(1 2))))
	(test (eq? hi ho) #f)
	(test (eqv? hi ho) #f)
	(test (equal? hi ho) #f)
	(set! (rec-b ho) 123)
	(test (equal? hi ho) #t))
      (let ((ho (make-rec 123 '())))
	(test (eq? hi ho) #f)
	(test (eqv? hi ho) #f)
	(test (equal? hi ho) #f))
      (test (copy hi) 'error)
      (test (fill! hi 1.0) 'error)
      (test (object->string hi) "#<anonymous-type #(32 123)>")
      (test (length hi) 'error)
      (test (reverse hi) 'error)
      (test (for-each abs hi) 'error)
      (test (map abs hi) 'error)
      (test (hi 1) 'error)
      (test (set! (hi 1) 2) 'error)
      )

    (let ((typo (make-type :equal (lambda (a b) (equal? a b)))))
      (let ((a ((cadr typo) 123))
	    (b ((cadr typo) 321))
	    (c ((cadr typo) 123)))
	(test (equal? a b) #f)
	(test (eq? a a) #t)
	(test (eq? a b) #f)
	(test (eqv? a a) #t)
	(test (eqv? a b) #f)
	(test (equal? a c) #t)
	(test (equal? b c) #f)))

    (test (let ((typo (make-type :equal (lambda (a b) (= (abs (- a b)) 2)))))
	    (let ((a ((cadr typo) 1))
		  (b ((cadr typo) 3))
		  (c ((cadr typo) 1)))
	      (and (equal? a b)
		   (not (equal? a c))
		   (equal? b c))))
	  #t)

    (test (((cadr (make-type :getter (lambda (a b) (vector-ref a b)))) (vector 1 2 3)) 1) 2)
    (test (((cadr (make-type :getter (lambda (a b) (+ 100 (vector-ref a b))))) (vector 1 2 3)) 1) 102)
    (test (length ((cadr (make-type :length (lambda (a) (vector-length a)))) (vector 1 2 3))) 3)
    (test (length ((cadr (make-type :length (lambda (a) (+ 100 (vector-length a))))) (vector 1 2 3))) 103)
    (test (string=? (object->string ((cadr (make-type)) 1)) "#<anonymous-type 1>") #t)
    (test (string=? (object->string ((cadr (make-type :name "hiho")) 123)) "#<hiho 123>") #t)
    (test (string=? (object->string ((cadr (make-type :print (lambda (a) (format #f "#<typo: ~A>" a)))) 1)) "#<typo: 1>") #t)

    (test (let* ((type (make-type :setter (lambda (a b c) (vector-set! a b c))))
		 (t? (car type))
		 (make-t (cadr type))
		 (t-ref (caddr type))
		 (newt (make-t (vector 1 2 3))))
	    (set! (newt 1) 123)
	    (vector-ref (t-ref newt) 1))
	  123)

    (let ((rec1 (make-type))
	  (rec2 (make-type)))
      (let ((rec1? (car rec1))
	    (rec2? (car rec2))
	    (make-rec1 (cadr rec1))	
	    (make-rec2 (cadr rec2))
	    (rec1-ref (caddr rec1))
	    (rec2-ref (caddr rec2)))
	(let ((r1 (make-rec1 123))
	      (r2 (make-rec2 123)))
	  (test (and (rec1? r1)
		     (rec2? r2)
		     (not (rec1? r2))
		     (not (rec2? r1))
		     (= (rec1-ref r1) (rec2-ref r2)))
		#t))))

    (let ((rec3? (car (make-type))))
      (for-each
       (lambda (arg)
	 (test (rec3? arg) #f))
       (list "hi" -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t '(1 . 2))))

    (test ((cadr (make-type :name 123))) 'error)
    (test ((cadr (make-type :length "hiho"))) 'error)
    (test ((cadr (make-type :print (lambda (a b) (= a b)))) "hi") 'error)
    (test ((cadr (make-type :length (lambda () 1))) "hi") 'error)

    (test (let* ((vzt (make-type :name "vzt" 
				 :length (lambda (v) (vector-length v))
				 :getter (lambda (v n) (vector-ref v n))))
		 (make-vzt (cadr vzt)))
	    (let ((v (make-vzt (vector 1 2 3))))
	      (let ((sum 0)) 
		(for-each (lambda (n) (set! sum (+ sum n))) v)
		sum)))
	  6)

    (test (let* ((rec-type (make-type))
		 (? (car rec-type))
		 (make (cadr rec-type))
		 (ref (caddr rec-type)))
	    (let ((val-1 (make "hi")))
	      (let ((val-2 (make val-1)))
		(let ((val-3 (make val-2)))
		  (ref (ref (ref val-3)))))))
	  "hi")

    (test (let* ((rec1-type (make-type))
		 (?1 (car rec1-type))
		 (make1 (cadr rec1-type))
		 (ref1 (caddr rec1-type)))
	    (let* ((rec2-type (make-type))
		   (?2 (car rec2-type))
		   (make2 (cadr rec2-type))
		   (ref2 (caddr rec2-type)))
	      (let ((val-1 (make1 "hi")))
		(let ((val-2 (make2 "hi")))
		  (let ((val-3 (make1 val-2)))
		    (and (string=? (ref2 (ref1 val-3)) "hi")
			 (not (equal? val-1 val-2))
			 (?1 val-1)
			 (?2 val-2)
			 (not (?2 val-3))))))))
	  #t)

    (test (let* ((rec-type (make-type :name "rec"))
		 (make (cadr rec-type))
		 (ref (caddr rec-type)))
	    (let ((val (make "hi")))
	      (ref (ref val))))
	  'error)

    (test (let* ((rec1-type (make-type))
		 (make1 (cadr rec1-type))
		 (ref1 (caddr rec1-type)))
	    (let* ((rec2-type (make-type))
		   (make2 (cadr rec2-type)))
	      (let ((val-1 (make1 "hi")))
		(let ((val-2 (make2 val-1)))
		  (ref1 val-2)))))
	  'error)

    (let ()
      (define make-float-vector #f)
      (define float-vector? #f)
      (define float-vector #f)
      
      (let* ((fv-type (make-type 
		       :getter vector-ref :length length :copy copy :fill fill!
		       :setter (lambda (obj index value)
				 (if (not (real? value))
				     (error 'wrong-type-arg-error "float-vector element must be real: ~S" value))
				 (vector-set! obj index (* 1.0 value)))
		       :name "float-vector"))
	     (fv? (car fv-type))
	     (make-fv (cadr fv-type))
	     (fv-ref (caddr fv-type)))
	
	(set! make-float-vector 
	      (lambda* (len (initial-element 0.0))
		       (if (not (real? initial-element))
			   (error 'wrong-type-arg-error "make-float-vector initial element must be real: ~S" initial-element))
		       (make-fv (make-vector len (* 1.0 initial-element)))))
	
	(set! float-vector? fv?)
	
	(set! float-vector
	      (lambda args
		(let* ((len (length args))
		       (fv (make-float-vector len))
		       (v (fv-ref fv)))
		  (do ((lst args (cdr lst))
		       (i 0 (+ i 1)))
		      ((null? lst) fv)
		    (let ((arg (car lst)))
		      (if (not (real? arg))
			  (error 'wrong-type-arg-error "float-vector element must be real: ~S in ~S" arg args))
		      (set! (v i) (* 1.0 arg))))))))
      
      (let ((v (make-float-vector 3 0.0)))
	(test (length v) 3)
	(set! (v 1) 32.0)
	(test (v 0) 0.0)
	(test (v 1) 32.0)
	(test (eq? v v) #t)
	(test (eq? v (float-vector 0.0 32.0 0.0)) #f)
	(test (equal? v (float-vector 0.0 32.0 0.0)) #t)
	(test (map + (list 1 2 3) (float-vector 1 2 3)) '(2.0 4.0 6.0))
	(test (reverse (float-vector 1.0 2.0 3.0)) (float-vector 3.0 2.0 1.0))
	(test (copy (float-vector 1.0 2.0 3.0)) (float-vector 1.0 2.0 3.0))
	(test (let () (fill! v 1.0) v) (float-vector 1.0 1.0 1.0))
	(test (object->string v) "#<float-vector #(1.0 1.0 1.0)>")
	(test (let ((v (float-vector 1.0 2.0 3.0))) (map v (list 2 1 0))) '(3.0 2.0 1.0))
	(test (let ((sum 0.0))
		(for-each
		 (lambda (x)
		   (set! sum (+ sum x)))
		 (float-vector 1.0 2.0 3.0))
		sum)
	      6.0)
	(test (length v) 3)
	))

    (let ()
      (define-macro (blet* names bindings . body)
	`(begin
	   ,@(map (lambda (name)
		    `(define ,name #f))
		  names)
	   (let* ,bindings
	     ,@body)))
      
      (blet* (make-adjustable-vector adjustable-vector? adjust-vector)
	     
	     ((av-type (make-type :name "adjustable-vector"
				  :getter (lambda (obj index)
					    ((car obj) index))
				  :setter (lambda (obj index value)
					    (set! ((car obj) index) value))
				  :length (lambda (obj)
					    (vector-length (car obj)))
				  :print (lambda (obj)
					   (object->string (car obj)))))
	      (av? (car av-type))
	      (make-av (cadr av-type))
	      (av-ref (caddr av-type)))
	     
	     (set! make-adjustable-vector (lambda args 
					    (make-av (list (apply make-vector args)))))
	     (set! adjustable-vector? av?)
	     (set! adjust-vector (lambda* (obj new-length initial-element)
					  (let* ((new-vector (make-vector new-length initial-element))
						 (copy-len (min new-length (length obj))))
					    (do ((i 0 (+ i 1)))
						((= i copy-len))
					      (set! (new-vector i) (obj i)))
					    (set! (car (av-ref obj)) new-vector)))))
      
      (let ((v (make-adjustable-vector 3 #f)))
	(test (length v) 3)
	(test (v 0) #f)
	(set! (v 1) 32.0)
	(adjust-vector v 10 #f)
	(test (length v) 10)
	(test (v 1) 32.0))

      (blet* (rec-a rec? rec-b make-rec)
	     
	     ((rec-type (make-type :name "rec" :length length :copy copy :fill fill!))
	      (? (car rec-type))
	      (make (cadr rec-type))
	      (ref (caddr rec-type)))
	     
	     (set! make-rec (lambda* ((a 1) (b 2))
				     (make (vector a b))))
	     
	     (set! rec? ?)
	     
	     (set! rec-a (make-procedure-with-setter
			  (lambda (obj)
			    (and (rec? obj)
				 (vector-ref (ref obj) 0)))
			  (lambda (obj val)
			    (if (rec? obj)
				(vector-set! (ref obj) 0 val)))))
	     
	     (set! rec-b (make-procedure-with-setter
			  (lambda (obj)
			    (and (rec? obj)
				 (vector-ref (ref obj) 1)))
			  (lambda (obj val)
			    (if (rec? obj)
				(vector-set! (ref obj) 1 val))))))
      
      (let ((r1 (make-rec)))
	(let ((r2 (copy r1)))
	  (test (eq? r1 r2) #f)
	  (test (rec? r2) #t)
	  (test (rec-a r1) 1)
	  (test (rec-b r1) 2)
	  (test (rec-a r2) 1)
	  (test (rec-b r2) 2)
	  (set! (rec-b r2) 32)
	  (test (rec-b r2) 32)
	  (test (rec-b r1) 2)
	  (fill! r2 123)
	  (test (rec-a r1) 1)
	  (test (rec-b r1) 2)
	  (test (rec-a r2) 123)
	  (test (rec-b r2) 123)
	  )
	))


    (define (notify-if-set var notifier)
      (set! (symbol-access var) (list #f notifier #f)))
    
    (define constant-access 
      (list #f
	    (lambda (symbol new-value) 
	      (error "can't change constant ~A's value to ~A" symbol new-value))
	    (lambda (symbol new-value) 
	      (error "can't bind constant ~A to a new value, ~A" symbol new-value))))
    
    (define-macro (define-global-constant symbol value)
      `(begin
	 (define ,symbol ,value)
	 (set! (symbol-access ',symbol) constant-access)
	 ',symbol))
    
    (define-macro (let-constant vars . body)
      (let ((varlist (map car vars)))
	`(let ,vars
	   ,@(map (lambda (var)
		    `(set! (symbol-access ',var) constant-access))
		  varlist)
	   ,@body)))
    
    (define-macro (define-integer var value)
      `(begin
	 (define ,var ,value)
	 (set! (symbol-access ',var) 
	       (list #f
		     (lambda (symbol new-value)
		       (if (real? new-value)
			   (floor new-value)
			   (error "~A can only take an integer value, not ~S" symbol new-value)))
		     #f))
	 ',var))
    
    (define (trace-var var)
      (let* ((cur-access (symbol-access var))
	     (cur-set (and cur-access (cadr cur-access))))
	(set! (symbol-access var)
	      (list (and cur-access (car cur-access))
		    (lambda (symbol new-value) 
		      (format #t "~A set to ~A~%" symbol new-value) 
		      (if cur-set 
			  (cur-set symbol new-value)
			  new-value))
		    (and cur-access (caddr cur-access))
		    cur-access))))
    
    (define (untrace-var var)
      (if (and (symbol-access var)
	       (cdddr (symbol-access var)))
	  (set! (symbol-access var) (cadddr (symbol-access var)))))

    (define-integer _int_ 32)
    (test _int_ 32)
    (set! _int_ 1.5)
    (test _int_ 1)

    (for-each
     (lambda (arg)
       (test (symbol-access arg) 'error))
     (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() '#(()) (list 1 2 3) '(1 . 2) "hi"))
    
    ))


(define-expansion (_expansion_ a) `(+ ,a 1))
(test (_expansion_ 3) 4)
(test (macroexpand (_expansion_ 3)) `(+ 3 1))
(test '(_expansion_ 3) (quote (_expansion_ 3)))
(test (_expansion_ (+ (_expansion_ 1) 2)) 5)

(test (let () (define-constant __c1__ 32) __c1__) 32)
(test (let () __c1__) 'error)
(test (let ((__c1__ 3)) __c1__) 'error)
(test (let* ((__c1__ 3)) __c1__) 'error)
(test (letrec ((__c1__ 3)) __c1__) 'error)
(test (let () (define (__c1__ a) a) (__c1__ 3)) 'error)
(test (let () (set! __c1__ 3)) 'error)

(test (constant? '__c1__) #t)
(test (constant? pi) #t)
(test (constant? 'pi) #t) ; take that, Clisp!
(test (constant? 12345) #t)
(test (constant? 3.14) #t)
(test (constant? :asdf) #t) 
(test (constant? 'asdf) #f)
(test (constant? "hi") #t) 
(test (constant? #\a) #t) 
(test (constant? #f) #t) 
(test (constant? #t) #t) 
(test (constant? '()) #t) 
(test (constant? ()) #t) 
(test (constant? '(a)) #t) 
(test (constant? '*features*) #f)
(test (let ((a 3)) (constant? 'a)) #f)
(test (constant? 'abs) #f)
(test (constant? abs) #t)
(test (constant? most-positive-fixnum) #t)
(test (constant? (/ (log 0))) #t)       ; nan.0 is a constant as a number I guess
(test (constant? (log 0)) #t)
(test (constant?) 'error)
(test (constant? 1 2) 'error)
(test (constant? #<eof>) #t) ; ?

;; and some I wonder about -- in CL's terms, these always evaluate to the same thing, so they're constantp
;;   but Clisp:
;;     (constantp (cons 1 2)) ->NIL
;;     (constantp #(1 2)) -> T
;;     (constantp '(1 . 2)) -> NIL
;; etc -- what a mess!

(test (constant? (cons 1 2)) #t)
(test (constant? #(1 2)) #t)
(test (constant? (list 1 2)) #t)
(test (constant? (vector 1 2)) #t)
(test (let ((v (vector 1 2))) (constant? v)) #t) ;!!
;; it's returning #t unless the arg is a symbol that is not a keyword or a defined constant
;; (it's seeing the value of v, not v):
(test (let ((v (vector 1 2))) (constant? 'v)) #f)

;; not sure this is the right thing...
;; but CL makes no sense: 
;; [3]> (constantp (vector 1))
;; T
;; [4]> (constantp (cons 1 2))
;; NIL
;; [5]> (constantp (list 1))
;; NIL
;; [7]> (constantp "hi")
;; T
;; (setf (elt "hi" 1) #\a)
;; #\a
;; at least they finally agree that pi is a constant!

(let ()
  (define-constant __hi__ (vector 3 0))
  (set! (__hi__ 1) 231)
  (test __hi__ #(3 231)))
;; that is, hi is the constant as a vector, not the vector elements


(test (defined? 'pi) #t)
(test (defined? 'pi (global-environment)) #t)
(test (defined? 'abs (global-environment)) #t)
(test (defined? 'abs (current-environment)) #t)
(test (let ((__c2__ 32)) (defined? '__c2__)) #t)
(test (let ((__c2__ 32)) (defined? '__c2__ (current-environment))) #t)
(test (let ((__c2__ 32)) (defined? '__c3__ (current-environment))) #f)
(test (let ((__c2__ 32)) (defined? '__c2__ (global-environment))) #f)
(test (let ((__c2__ 32)) (defined? '__c3__ (global-environment))) #f)
(test (defined?) 'error)
(test (defined? 'a 'b) 'error)
(for-each
 (lambda (arg)
   (test (defined? arg) 'error))
 (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() '#(()) (list 1 2 3) '(1 . 2) "hi"))
(test (defined? 'lambda car) 'error)
(test (defined? lambda gensym) 'error)
(test (defined? 'lambda defined?) 'error)
(test (defined? 'define car) 'error)
(test (defined? 'abs '(())) #f)



(test (current-environment 1) 'error)
(test (global-environment 1) 'error)


(test (let ((a 1)) (eval '(+ a b) (augment-environment (current-environment) (cons 'b 32)))) 33)
(test (let ((a 1)) (+ (eval '(+ a b) (augment-environment (current-environment) (cons 'b 32))) a)) 34)
(test (let ((a 1)) (+ (eval '(+ a b) (augment-environment (current-environment) (cons 'b 32) (cons 'a 12))) a)) 45)

(test (augment-environment) 'error)
(for-each
 (lambda (arg)
   (test (augment-environment arg '(a . 32)) 'error))
     (list -1 #\a 1 3.14 3/4 1.0+1.0i "hi"))
(let ((e (augment-environment (current-environment)
			      (cons 'a 32)
			      (cons 'b 12))))
  (test (eval '(+ a b) e) 44)
  (test (eval '(+ a b c) (augment-environment e (cons 'c 3))) 47)
  (test (eval '(+ a b) (augment-environment e (cons 'b 3))) 35))

(test (with-environment (current-environment) (let ((x 1)) x)) 1)

(test (let ((x 12))
	(let ((e (current-environment)))
	  (let ((x 32))
	    (with-environment e (* x 2)))))
      24)

(test (let ((*features* 123))
	(let ((e (global-environment)))
	  (with-environment e (list? *features*))))
      #t)

(test (with-environment) 'error)
(test (with-environment 1) 'error)
(test (with-environment () 1) 'error)
(test (with-environment (current-environment) 1) 1)
(test (let ((a 1))
	(+ (with-environment
	    (augment-environment (current-environment) (cons 'a 10))
	    a)
	   a))
      11)



(test (call-with-exit (lambda (c) (0 (c 1)))) 1)
(test (call-with-exit (lambda (k) (k "foo"))) "foo")
(test (call-with-exit (lambda (k) "foo")) "foo")
(test (call-with-exit (lambda (k) (k "foo") "oops")) "foo")
(test (let ((memb (lambda (x ls)
		    (call-with-exit
		     (lambda (break)
		       (do ((ls ls (cdr ls)))
			   ((null? ls) #f)
			 (if (equal? x (car ls))
			     (break ls))))))))
	(list (memb 'd '(a b c))
	      (memb 'b '(a b c))))
      '(#f (b c)))

(let ((x 1))
  (define y (call-with-exit (lambda (return) (set! x (return 32)))))
  (test (and (= x 1) (= y 32)) #t)
  (set! y (call-with-exit (lambda (return) ((lambda (a b c) (set! x a)) 1 2 (return 33)))))
  (test (and (= x 1) (= y 33)) #t)
  (set! y (call-with-exit (lambda (return) ((lambda (a b) (return a) (set! x b)) 2 3))))
  (test (and (= x 1) (= y 2)) #t))

(if (and (defined? 'provided?)
	 (provided? 'threads))
    (begin
      
      (test (let ((ctr 0))
	      (let ((t1 (make-thread (lambda () (set! ctr (+ ctr 1))))))
		(join-thread t1))
	      ctr)
	    1)
      
      (test (let ((ctr 0))
	      (let ((t1 (make-thread (lambda () (set! ctr (+ ctr 1))))))
		(join-thread t1)
		(thread? t1)))
	    #t)
      
      (test (let ((ctr 0)
		  (loc (make-thread-variable)))
	      (let ((t1 (make-thread (lambda () (set! (loc) (+ ctr 1)) (set! ctr (loc))))))
		(join-thread t1)
		ctr))
	    1)
      
      (test (let ((ctr 0)
		  (loc (make-thread-variable)))
	      (let ((t1 (make-thread (lambda () (set! (loc) (+ ctr 1)) (set! ctr (thread-variable? loc))))))
		(join-thread t1)
		ctr))
	    #t)
      
      (test (let ((ctr 0)
		  (lock (make-lock)))
	      (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock)))))
		(join-thread t1))
	      ctr)
	    1)
      
      (test (let ((ctr 0)
		  (lock (make-lock)))
	      (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (lock? lock)) (release-lock lock)))))
		(join-thread t1))
	      ctr)
	    #t)
      
      (test (let ((ctr 0)
		  (lock (make-lock)))
	      (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock))))
		    (t2 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock)))))
		(join-thread t1)
		(join-thread t2))
	      ctr)
	    2)

      (test (let ((ctr1 0) (ctr2 0))
	      (let ((t1 (make-thread (lambda () (set! ctr1 (+ ctr1 1)) (* ctr1 2))))
		    (t2 (make-thread (lambda () (set! ctr2 (+ ctr2 1)) (* ctr2 3)))))
		(+ (join-thread t1) 
		   (join-thread t2))))
	    5)
      
      (test (let ((ctr 0)
		  (lock (make-lock)))
	      (let ((threads '()))
		(do ((i 0 (+ 1 i)))
		    ((= i 8))
		  (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (release-lock lock)))))
		    (set! threads (cons t1 threads))))
		(for-each
		 (lambda (tn)
		   (join-thread tn))
		 threads))
	      ctr)
	    8)
      
      (test (let ((ctr 0)
		  (ctr1 0)
		  (ctr2 0)
		  (lock (make-lock))
		  (var (make-thread-variable)))
	      (let ((t1 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (set! (var) ctr) (release-lock lock) (set! ctr1 (var)))))
		    (t2 (make-thread (lambda () (grab-lock lock) (set! ctr (+ ctr 1)) (set! (var) ctr) (release-lock lock) (set! ctr2 (var))))))
		(join-thread t1)
		(join-thread t2))
	      (and (= ctr 2)
		   (= (+ ctr1 ctr2) 3)))
	    #t)

      (let ((v1 (make-vector 4096))
	    (v2 (make-vector 4096))
	    (dsum 0.0)
	    (dlock (make-lock)))
	
	(do ((i 0 (+ i 1)))
	    ((= i 4096))
	  (set! (v1 i) (- (random 2.0) 1.0))
	  (set! (v2 i) (- (random 2.0) 1.0)))

	(let ((threads '()))
	  (let loop 
	      ((i 0))
	    (set! threads (cons (make-thread
				 (lambda ()
				   (let ((sum 0.0)
					 (end (+ i 1024)))
				     (do ((k i (+ k 1)))
					 ((= k end))
				       (set! sum (+ sum (* (v1 k) (v2 k)))))
				     (grab-lock dlock)
				     (set! dsum (+ dsum sum))
				     (release-lock dlock))))
				threads))
	    (if (< i 3072)
		(loop (+ i 1024))))

	  (for-each 
	   (lambda (thread) 
	     (join-thread thread))
	   threads))

	(let ((xsum 0.0))
	  (do ((i 0 (+ i 1)))
	      ((= i 4096))
	    (set! xsum (+ xsum (* (v1 i) (v2 i)))))

	  (test (< (abs (- xsum dsum)) .001) #t)))
      
      (for-each
       (lambda (arg)
	 (test (thread? arg) #f))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
      
      (for-each
       (lambda (arg)
	 (test (lock? arg) #f))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
      
      (for-each
       (lambda (arg)
	 (test (thread-variable? arg) #f))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" abs '#(()) (list 1 2 3) '(1 . 2) (lambda () 1)))
      
      (for-each
       (lambda (arg)
	 (test (make-thread arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) (list 1 2 3) '(1 . 2)))
      
      (for-each
       (lambda (arg)
	 (test (grab-lock arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) (list 1 2 3) '(1 . 2)))
      
      (for-each
       (lambda (arg)
	 (test (release-lock arg) 'error))
       (list -1 #\a 1 '#(1 2 3) 3.14 3/4 1.0+1.0i '() 'hi "hi" '#(()) (list 1 2 3) '(1 . 2)))))

(test (apply "hi" '(1 2)) 'error)
(test ("hi" 1 2) 'error)
(test (apply '(1 2) '(1 2)) 'error)
(test ((list 1 2 3) 1 2) 'error)

(test (apply "hi" '(1)) #\i)
(test ("hi" 1) #\i)
(test (apply '(1 2) '(1)) 2)
(test ((list 1 2 3) 1) 2)

(test (let ((pi 3)) pi) 'error)
;;   or ... (let ((:asdf 3)) :asdf) and worse (let ((:key 1)) :key) or even worse (let ((:3 1)) 1)
(test (let ((x_x_x 32)) (let () (define-constant x_x_x 3) x_x_x) (set! x_x_x 31) x_x_x) 'error)


(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(pws-test))
      123)

(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(pws-test 32))
      'error)

(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(set! (pws-test 32) 123))
      'error)

(test (call-with-exit 
       (lambda (return) 
	 (let ((local 123))
	   (define pws-test (make-procedure-with-setter
			     (lambda () (return "oops"))
			     (lambda (val) (set! local val))))
	   (pws-test))))
      "oops")
(test (call-with-exit 
       (lambda (return)
	 (let ((local 123))
	   (define pws-test (make-procedure-with-setter
			     (lambda () 123)
			     (lambda (val) (return "oops"))))
	   (set! (pws-test) 1))))
      "oops")

(test (let ((local 123))
	(define pws-test (make-procedure-with-setter
			  (lambda () local)
			  (lambda (val) (set! local val))))
	(set! (pws-test) 321)
	(pws-test))
      321)

(test (let ((v (vector 1 2 3)))
	(define vset (make-procedure-with-setter
		      (lambda (loc)
			(vector-ref v loc))
		      (lambda (loc val)
			(vector-set! v loc val))))
	(let ((lst (list vset)))
	  (let ((val (vset 1)))
	    (set! (vset 1) 32)
	    (let ((val1 (vset 1)))
	      (set! ((car lst) 1) 3)
	      (list val val1 (vset 1))))))
      (list 2 32 3))

(let ((local 123))
  (define pws-test (make-procedure-with-setter
		    (lambda () local)
		    (lambda (val) (set! local val))))
  (test (pws-test) 123)
  (set! (pws-test) 32)
  (test (pws-test) 32)
  (set! (pws-test) 0)
  (test (pws-test) 0))

(let ((local 123))
  (define pws-test (make-procedure-with-setter
		    (lambda (val) (+ local val))
		    (lambda (val new-val) (set! local new-val) (+ local val))))
  (test (pws-test 1) 124)
  (set! (pws-test 1) 32)
  (test (pws-test 2) 34)
  (set! (pws-test 3) 0)
  (test (pws-test 3) 3))


(test (make-procedure-with-setter) 'error)
(test (make-procedure-with-setter abs) 'error)
(test (make-procedure-with-setter 1 2) 'error)
(test (make-procedure-with-setter (lambda () 1) (lambda (a) a) (lambda () 2)) 'error)
(test (make-procedure-with-setter (lambda () 1) 2) 'error)

(let ((pws (make-procedure-with-setter vector-ref vector-set!)))
  (let ((v (vector 1 2 3)))
    (test (pws v 1) 2)
    (set! (pws v 1) 32)
    (test (pws v 1) 32)
    (test (procedure-arity pws) '(2 0 #t 3 0 #t))))


(define (procedure-with-setter-setter-arity proc) (cdddr (procedure-arity proc)))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (a) a)))) (procedure-with-setter-setter-arity pws)) '(1 0 #f))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (a b c) a)))) (procedure-with-setter-setter-arity pws)) '(3 0 #f))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda (a . b) a)))) (procedure-with-setter-setter-arity pws)) '(1 0 #t))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda* (a (b 1)) a)))) (procedure-with-setter-setter-arity pws)) '(0 2 #f))
(test (let ((pws (make-procedure-with-setter (lambda () 1) (lambda* (a :rest b) a)))) (procedure-with-setter-setter-arity pws)) '(0 1 #t))
(test (procedure-with-setter-setter-arity symbol-access) '(2 0 #f))
(test (let ((pws (make-procedure-with-setter (lambda args (apply + args)) (lambda args (apply * args))))) (pws 2 3 4)) 9)
(test (let ((pws (make-procedure-with-setter (lambda args (apply + args)) (lambda args (apply * args))))) (set! (pws 2 3 4) 5)) 120)



;; generic length/reverse/copy/fill!
(test (length (list 1 2)) 2)
(test (length "hiho") 4)
(test (length (vector 1 2)) 2)
(test (>= (length (make-hash-table 7)) 7) #t)
(test (length '()) 0)
(test (length (#(#() #()) 1)) 0)

(test (copy 3) 3)
(test (copy 3/4) 3/4)
(test (copy "hi") "hi")
(test (copy (list 1 2 3)) (list 1 2 3))
(test (copy (vector 0.0)) (vector 0.0))
(test (copy #\f) #\f)
(test (copy (list 1 (list 2 3))) (list 1 (list 2 3)))
(test (copy (cons 1 2)) (cons 1 2))
(test (copy '(1 2 . 3)) '(1 2 . 3))
(test (copy (+)) 0)
(test (copy +) +)
(test (copy (#(#() #()) 1)) #())
(test (copy #f) #f)
(test (copy '()) '())

(test (reverse "hi") "ih")
(test (reverse "") "")
(test (reverse "123") "321")
(test (reverse "1234") "4321")
(test (reverse #()) #())
(test (reverse #(1 2 3)) #(3 2 1))
(test (reverse #(1 2 3 4)) #(4 3 2 1))
(test (reverse #2D((1 2) (3 4))) #2D((4 3) (2 1)))

(if (not (provided? 'gmp))
    (let ((r1 (make-random-state 1234)))
      (random 1.0 r1)
      (let ((r2 (copy r1)))
	(let ((v1 (random 1.0 r1))
	      (v2 (random 1.0 r2)))
	  (test (= v1 v2) #t)
	  (let ((v3 (random 1.0 r1)))
	    (random 1.0 r1)
	    (random 1.0 r1)
	    (let ((v4 (random 1.0 r2)))
	      (test (= v3 v4) #t)))))))

(if (provided? 'gmp)
    (let ((i (copy (bignum "1")))
	  (r (copy (bignum "3/4")))
	  (f (copy (bignum "1.5")))
	  (c (copy (bignum "1.0+1.0i"))))
      (test (= i (bignum "1")) #t)
      (test (= r (bignum "3/4")) #t)
      (test (= f (bignum "1.5")) #t)
      (test (= c (bignum "1.0+1.0i")) #t)))

(let ((str (string #\1 #\2 #\3)))
  (fill! str #\x)
  (test str "xxx"))
(let ((v (vector 1 2 3)))
  (fill! v 0.0)
  (test v (vector 0.0 0.0 0.0)))
(let ((lst (list 1 2 (list (list 3) 4))))
  (fill! lst 100)
  (test lst '(100 100 100)))
(let ((cn (cons 1 2)))
  (fill! cn 100)
  (test cn (cons 100 100)))
(test (fill! 1 0) 'error)
(test (fill! 'hi 0) 'error)

(test (fill!) 'error)
(test (copy) 'error)
(test (fill! '"hi") 'error)

(for-each
 (lambda (arg)
   (test (fill! arg 1) 'error))
 (list (integer->char 65) #f 'a-symbol abs 3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (let ((str (string #\a #\b)))
     (test (fill! str arg) 'error)))
 (list "hi" '(1 2 3) #() #f 'a-symbol abs 3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))


;; generic for-each/map
(test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)      
(test (map (lambda (n) (+ n 1)) (vector 1 2 3)) '(2 3 4))
(test (map (lambda (a b) (/ a b)) (list 1 2 3) (list 4 5 6)) '(1/4 2/5 1/2))

;; try some applicable stuff
(test (let ((lst (list 1 2 3)))
	(set! (lst 1) 32)
	(list (lst 0) (lst 1)))
      (list 1 32))

(test (let ((hash (make-hash-table)))
	(set! (hash 'hi) 32)
	(hash 'hi))
      32)

(test (let ((str (string #\1 #\2 #\3)))
	(set! (str 1) #\a)
	(str 1))
      #\a)

(test (let ((v (vector 1 2 3)))
	(set! (v 1) 0)
	(v 1))
      0)

(let ()
  (define (hiho a) __func__)
  (test (or (equal? (hiho 1) 'hiho)
	    (equal? (car (hiho 1)) 'hiho))
	#t))


#|
;; these 2 tests don't work in this context because the file/line are included
(let ()
  (define (a1 a) (stacktrace) (+ a 1))
  (define (a2 b) (+ b (a1 b)))
  (define (a3 c) (+ c (a2 c)))
  (let ((str (with-output-to-string
	       (lambda ()
		 (a3 1)))))
    (if (not (string=? str "(a1 (a . 1))\n(a2 (b . 1))\n(a3 (c . 1))\n"))
	(format #t ";stacktrace: ~A~%" str))))

(let ()
  (define (a1 a) (+ a #\c))
  (define (a2 b) (+ b (a1 b)))
  (define (a3 c) (+ c (a2 c)))
  (let ((str (catch #t
		    (lambda () (a3 1))
		    (lambda args 
		      (with-output-to-string 
			(lambda ()
			  (stacktrace *error-info*)))))))
    (if (not (string=? str "(a1 (a . 1))\n(a2 (b . 1))\n(a3 (c . 1))\n"))
	(format #t ";*error-info* stacktrace: ~A" str))))
|#
(test (stacktrace #(23)) 'error)


;;; -------- miscellaneous (amusements)

(test ((number->string -1) 0) #\-)
(test ((reverse '(1 2)) 0) 2)
(test ((append begin) list) list)
(test ((begin object->string) car) "car")
(test ((and abs) -1) 1)
(test (((values begin) object->string) car) "car")
(test (((values (begin begin)) object->string) car) "car")
(test ((((values append) begin) object->string) car) "car")
(test ((((((values and) or) append) begin) object->string) car) "car")
(test (((((((values values) and) or) append) begin) object->string) car) "car")
(test (((lambda case lcm))) 1)
(test (((lambda let* *))) 1)
(test ((((eval lambda) lcm gcd))) 0)
(test (((append s7-version)) 0) #\s)
(test ((values (lambda hi #()))) #())
(test (((((lambda () (lambda () (lambda () (lambda () 1)))))))) 1)
(test (((cond (cond => cond)) (cond)) ((cond (#t #t)))) #t)

(test (+ (+) (*)) 1)
(test (modulo (lcm) (gcd)) 1)
(test (max (+) (*)) 1)
(test (min (gcd) (lcm)) 0)
(test (symbol->value (gensym) (global-environment)) #<undefined>)
(test (string-ref (s7-version) (*)) #\7)
(test (string>=? (string-append) (string)) #t)
(test (substring (string-append) (+)) "")
(test (ash (*) (+)) 1)
(test (> (*) (+)) #t)
(test ((or #f list)) ())
(test ((or #f lcm)) 1)
(test ((or begin symbol?)) ())
(test ((or begin make-polar)) ())
(test ((and #t begin)) ())
(test (begin) ())
(test ((or #f lcm) 2 3) 6)
(test ((or and) #f #t) #f)
(test ((and or) #f #t) #t)
(test (or (or) (and)) #t)
(test ((car '((1 2) (3 4))) 0) 1)
(test ((or apply) lcm) 1)
(test (- ((or *))) -1)
(test ((car (list lcm))) 1)
(test ((or (cond (lcm)))) 1)
(test ((cond (asin floor *))) 1)
(test (logior (#(1 #\a (3)) 0) (truncate 1.5)) 1)
(test (real? (*)) #t)
(test (- (lcm)) -1)
(test (* (*)) 1)
(test (+ (+) (+ (+)) (+ (+ (+)))) 0)
(test (+(*(+))(*)(+(+)(+)(*))) 2)
(test (nan? (asinh (cos (real-part (log 0.0))))) #t)
(num-test(cos(sin(log(tan(*))))) 0.90951841537482)
(num-test (asinh (- 9223372036854775807)) -44.361419555836)
(num-test (imag-part (asin -9223372036854775808)) 44.361419555836)

(test ((call-with-exit object->string) 0) #\#) ; #<goto>
(test ((begin begin) 1) 1)
(test ((values begin) 1) 1)
(test ((provide or) 3/4) 'error)
(test (string? cond) #f)
(test (list? or) #f)
(test (pair? define) #f)
(test (number? lambda*) #f)
(test ((s7-version) (rationalize 0)) #\s)
(test (cond (((values '(1 2) '(3 4)) 0 0))) 'error)
(test (cond ((apply < '(1 2)))) #t)
(test (dynamic-wind lcm gcd *) 'error)
(test ((lambda (let) (+)) 0) 0)
(test (case 0 ((< 0 1) 32)) 32)
(test (char-downcase (char-downcase #\newline)) #\newline)
(test (and (and) (and (and)) (and (and (and (or))))) #f)
(test ((values begin #\a 1)) 1)
(test ((values and 1 3)) 3)
(test ((((lambda () begin)) (values begin 1))) 1)
(test (+ (((lambda* () values)) 1 2 3)) 6)

(test (let () (define (hi cond) (+ cond 1)) (hi 2)) 3)
(test (let () (define* (hi (cond 1)) (+ cond 1)) (hi 2)) 3)
(test (let () (define* (hi (cond 1)) (+ cond 1)) (hi)) 2)
(test (let () ((lambda (cond) (+ cond 1)) 2)) 3)
(test (let () ((lambda* (cond) (+ cond 1)) 2)) 3)
(test (let () (define-macro (hi cond) `(+ 1 ,cond)) (hi 2)) 3)
(test (let () (define-macro* (hi (cond 1)) `(+ 1 ,cond)) (hi)) 2)
(test (let () (define (hi abs) (+ abs 1)) (hi 2)) 3)
(test (let () (define (hi if) (+ if 1)) (hi 2)) 3)
(test (let () (define* (hi (lambda 1)) (+ lambda 1)) (hi)) 2)
(test (((lambda #\newline gcd))) 'error)

(test (let ((1,1 3) (1'1 4) (1|1 5) (1#1 6) (1\1 7) (1?1 8)) (+ 1,1 1'1 1|1 1#1 1\1 1?1)) 33)
(test (let ((,a 3)) ,a) 'error)
(test (let ((@a 3)) @a) 3)
(test (let (("a" 3)) "a") 'error)
(test (let ((`a 3)) `a) 'error)
(test (let (('a 3)) 'a) 'error)
(test (let ((a`!@#$%^&*~.,<>?/'{}[]\|+=_-a 3)) a`!@#$%^&*~.,<>?/'{}[]\|+=_-a) 3)





;;; ------ CLisms ------------------------------------------------------------------------


(let ()

      ;; **********************************************************************
      ;; 
      ;; Copyright (C) 2002 Heinrich Taube (taube@uiuc.edu) 
      ;; 
      ;; This program is free software; you can redistribute it and/or
      ;; modify it under the terms of the GNU General Public License
      ;; as published by the Free Software Foundation; either version 2
      ;; of the License, or (at your option) any later version.
      ;; 
      ;; This program is distributed in the hope that it will be useful,
      ;; but WITHOUT ANY WARRANTY; without even the implied warranty of
      ;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
      ;; GNU General Public License for more details.
      ;; 
      ;; **********************************************************************
      
      ;; $Name:  $
      ;; $Revision: 1.6 $
      ;; $Date: 2005/11/17 13:29:37 $
      
      ;;
      ;; Implementation of the CLTL2 loop macro. The following 
      ;; non Rev 5 definitions need to be in effect before the file
      ;; is loaded:
      ;;
      ;; (define-macro (name . args) ...)
      ;; (error string)
      ;; (gensym string)
      ;;
      
      (defmacro when (test . forms)
	`(if ,test (begin ,@forms)))
	  
      (define-macro (loop . args)
	(let ()
	  
	  (define-macro (push val sym)
	    `(begin (set! ,sym (cons ,val ,sym)) ,sym))
	  
	  (define-macro (pop sym)
	    (let ((v (gensym "v")))
	      `(let ((,v (car ,sym)))
		 (set! ,sym (cdr ,sym))
		 ,v)))
	  
	  ;; this next one is a no-op but i need it as a marker for my cltl2
	  ;; translator.
	  
	  (define-macro (function sym) sym)     
	  
	  ;; getters and setters for the loop-clause "struct"
	  
	  (define (loop-operator c)          (vector-ref  c 0))
	  (define (loop-operator-set! c x)   (vector-set! c 0 x))
	  (define (loop-bindings c)          (vector-ref  c 1))
	  (define (loop-bindings-set! c x)   (vector-set! c 1 x))
	  (define (loop-collectors c)        (vector-ref  c 2))
	  (define (loop-collectors-set! c x) (vector-set! c 2 x))
	  (define (loop-initially c)         (vector-ref  c 3))
	  (define (loop-initially-set! c x)  (vector-set! c 3 x))
	  (define (loop-end-tests c)         (vector-ref  c 4))
	  (define (loop-end-tests-set! c x)  (vector-set! c 4 x))
	  (define (loop-looping c)           (vector-ref  c 5))
	  (define (loop-looping-set! c x)    (vector-set! c 5 x))
	  (define (loop-stepping c)          (vector-ref  c 6))
	  (define (loop-stepping-set! c x)   (vector-set! c 6 x))
	  (define (loop-finally c)           (vector-ref  c 7))
	  (define (loop-finally-set! c x)    (vector-set! c 7 x))
	  (define (loop-returning c)         (vector-ref  c 8))
	  (define (loop-returning-set! c x)  (vector-set! c 8 x))
	  
	  (define (make-loop-clause . args)
	    (let ((v (vector #f '() '() '() '() '() '() '() '())))
	      (if (null? args) v
		  (do ((a args (cddr a)))
		      ((null? a) v)
		    (case (car a)
		      ((operator) (loop-operator-set! v (cadr a)))
		      ((bindings) (loop-bindings-set! v (cadr a)))
		      ((collectors) (loop-collectors-set! v (cadr a)))
		      ((initially) (loop-initially-set! v (cadr a)))
		      ((end-tests) (loop-end-tests-set! v (cadr a)))
		      ((looping) (loop-looping-set! v (cadr a)))
		      ((stepping) (loop-stepping-set! v (cadr a)))
		      ((finally) (loop-finally-set! v (cadr a)))
		      ((returning) (loop-returning-set! v (cadr a))))))))
	  
	  (define (gather-clauses caller clauses)
	    ;; nconc all clausal expressions into one structure
	    (let ((gather-clause 
		   (lambda (clauses accessor)
		     ;; append data from clauses
		     (do ((l '()))
			 ((null? clauses) l)
		       (set! l (append l (accessor (car clauses))))
		       (set! clauses (cdr clauses))))))
	      (make-loop-clause 'operator caller
				'bindings
				(gather-clause clauses 
					       (function loop-bindings))
				'collectors
				(gather-clause clauses 
					       (function loop-collectors))
				'initially 
				(gather-clause clauses 
					       (function loop-initially))
				'end-tests 
				(gather-clause clauses 
					       (function loop-end-tests))
				'looping 
				(gather-clause clauses 
					       (function loop-looping))
				'stepping 
				(gather-clause clauses 
					       (function loop-stepping))
				'finally 
				(gather-clause clauses
					       (function loop-finally))
				'returning 
				(gather-clause clauses
					       (function loop-returning)))))
	  
	  (define (loop-op? x ops)
	    (assoc x ops))
	  
	  (define (loop-variable? x)
	    (symbol? x))
	  
	  (define (make-binding var val)
	    (list var val))
	  
	  (define (loop-error ops forms . args)
	    ;; all error messages include error context.
	    (let ((loop-context
		   (lambda (lst ops)
		     ;; return tail of expr up to next op in cdr of tail
		     (do ((h lst)
			  (l '()))
			 ((or (null? lst)
			      ;; ignore op if in front.
			      (and (not (eq? h lst))
				   (loop-op? (car lst) ops)))
			  (reverse l))
		       (set! l (cons (car lst) l))
		       (set! lst (cdr lst))))))
	      (let ((forms (loop-context forms ops)))
		(newline)
		(display "LOOP ERROR: ")
		(do ((tail args (cdr tail)))
		    ((null? tail) #f)
		  (display (car tail)))
		(newline)
		(display "clause context: ")
		(if (null? forms) 
		    (display "()")
		    (do ((tail forms (cdr tail)))
			((null? tail) #f)
		      (if (eq? tail forms) (display "'"))
		      (display (car tail))
		      (display (if (null? (cdr tail)) "'" " "))))
		(newline)
		(error "illegal loop syntax"))))
	  
	  (define (parse-for forms clauses ops)
	    ;; forms is (FOR ...)
	    (let ((op (loop-op? (car forms) ops)))
	      (if (null? (cdr forms))
		  (loop-error ops forms "Variable expected but source code ran out." )
		  (let ((var (cadr forms)))
		    (if (loop-variable? var)
			(if (null? (cddr forms))
			    (loop-error ops forms
					"'for' clause expected but source code ran out.")
			    ;; find the iteration path in the op
			    (let ((path (assoc (caddr forms) (cdddr op))))
			      ;; path is (<pathop> <parser>)
			      (if (not path)
				  (loop-error ops forms "'" (caddr forms) "'"
					      " is not valid with 'for'.")
				  ( (cadr path) forms clauses ops))))
			(loop-error ops forms "Found '" (cadr forms)
				    "' where a variable expected."))))))
	  
	  (define (parse-numerical-for forms clauses ops)
	    ;; forms is (FOR <var> <OP> ...)
	    ;; where <OP> is guaranteed to be one of: FROM TO BELOW ABOVE DOWNTO
	    clauses
	    (let ((var (cadr forms))
		  (tail (cddr forms))
		  (bind '())
		  (from #f)
		  (head #f)
		  (last #f)
		  (stop #f)
		  (step #f)
		  (test #f)
		  (incr #f))
	      
	      (do ((next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(if (null? tail)
		    (loop-error ops forms
				"Expected expression but source code ran out."))
		(case next
		  ((from downfrom)
		   (if head (loop-error ops forms "Found '" next "' when '"
					head "' in effect."))
		   (set! head next)
		   (set! from (pop tail)))
		  ((below)
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail))
		   (set! last next))
		  ((to)
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail) )
		   (set! last next))
		  ((above )
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail))
		   (set! last next))
		  ((downto )
		   (if last (loop-error ops forms "Found '" next "' when '"
					last "' in effect."))
		   (set! stop (pop tail))
		   (set! last next))
		  ((by)
		   (if step (loop-error ops forms "Found duplicate 'by'."))
		   (set! step (pop tail)))
		  (else
		   (loop-error ops forms 
			       "'" next "' is not valid with 'for'."))))
	      (if (not head)
		  (set! head 'from))
	      (if (or (eq? head 'downfrom)
		      (eq? last 'downto)
		      (eq? last 'above))
		  (begin
		    (set! incr '-)
		    (if (eq? last 'above)
			(set! test '<=)
			(set! test '<)))   ; allow to for downto
		  (begin
		    (set! incr '+)
		    (if (eq? last 'below)
			(set! test '>=)
			(set! test '>))))
	      
	      ;; add binding for initial value
	      (push (make-binding var (or from 0)) bind)
	      ;; add binding for non-constant stepping values.
	      (if (not step)
		  (set! step 1)
		  (if (not (number? step))
		      (let ((var (gensym "v")))
			(push (make-binding var step) bind)
			(set! step var))))
	      (set! step `(set! ,var (,incr ,var ,step)))
	      (if stop
		  (let ((end (gensym "v")))
		    (push (make-binding end stop) bind)
		    (set! stop (list test var end))))
	      (values (make-loop-clause 'operator 'for
					'bindings (reverse bind)
					'stepping (list step)
					'end-tests (if (not stop)
						       '() (list stop)))
		      tail)))
	  
	  (define (parse-repeat forms clauses ops)
	    ;; forms is (REPEAT <FORM> ...)
	    (if (null? (cdr forms))
		(loop-error ops forms 
			    "'repeat' clause expected but source code ran out." )
		(call-with-values (lambda ()
				    (parse-numerical-for 
				     (list 'for (gensym "v") 'below (cadr forms))
				     clauses ops))
		  (lambda (clause ignore)
		    ignore
		    (values clause (cddr forms))))))
	  
	  (define (parse-sequence-iteration forms clauses ops)
	    ;; tail is (FOR <var> <OP> ...)
	    ;; <OP> is guaranteed to be one of: IN ON ACROSS
	    clauses
	    (let ((head forms)
		  (var (cadr forms))
		  (seq (gensym "v"))
		  (tail (cddr forms))
		  (bind '())
		  (data #f) 
		  (init '()) 
		  (loop '()) 
		  (incr #f)
		  (stop '()) 
		  (step '()) 
		  (type #f))
	      
	      (do ((next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(when (null? tail)
		      (loop-error ops head
				  "Expression expected but source code ran out." ))
		(case next
		  ((in on across)
		   (if type (loop-error ops head 
					"Extraneous '" next "' when '"
					type "' in effect."))
		   (set! type next)
		   (set! data (pop tail)))
		  ((by )
		   (if incr 
		       (loop-error ops head "Duplicate 'by'." )
		       (if (eq? type 'across)
			   (loop-error ops head "'by' is invalid with 'across'." )
			   (set! incr (pop tail)))))
		  (else
		   (loop-error ops head "'" next "' is not valid with 'for'."))))
					; add bindings for stepping var and source
	      (push (make-binding var #f) bind)
	      (push (make-binding seq data) bind)
	      (if (eq? type 'across)
		  (let ((pos (gensym "v"))
			(max (gensym "v")))
		    (push (make-binding pos 0) bind)
		    (push (make-binding max #f) bind)
		    (push `(set! ,max (vector-length ,seq)) init)
		    (push `(set! ,pos (+ 1 ,pos)) step)
		    (push `(set! ,var (vector-ref ,seq ,pos)) loop)
		    (push `(>= ,pos ,max) stop))
		  (begin
		    (if incr
			(if (and (list? incr) (eq? (car incr) 'quote))
			    (push `(set! ,seq (,(cadr incr) ,seq)) step)
			    (push `(set! ,seq (,incr ,seq)) step))
			(push `(set! ,seq (cdr ,seq)) step))
		    (push (if (eq? type 'in)
			      `(set! ,var (car ,seq))
			      `(set! ,var ,seq))
			  loop)
		    (push `(null? ,seq) stop)))
	      
	      (values (make-loop-clause 'operator 'for
					'bindings (reverse bind)
					'end-tests stop
					'initially init
					'looping loop
					'stepping step)
		      tail)))
	  
	  (define (parse-general-iteration forms clauses ops)
	    ;; forms is (FOR <var> = ...)
	    clauses
	    (let ((head forms)
		  (var (cadr forms))
		  (tail (cddr forms))
		  (init #f)
		  (type #f)
		  (loop #f)
		  (step #f))
	      (do ((next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(if (null? tail)
		    (loop-error ops head 
				"Expression expected but source code ran out."))
		(case next
		  ((= )
		   (if type (loop-error ops head "Duplicate '='."))
		   (set! loop `(set! ,var ,(pop tail)))
		   (set! type next))
		  ((then )
		   (if init (loop-error ops head "Duplicate 'then'."))
		   (set! init loop)
		   (set! loop #f)
		   (set! step `(set! ,var ,(pop tail)))
		   (set! type next))
		  (else
		   (loop-error ops head "'" next "' is not valid with 'for'."))))
	      
	      (values (make-loop-clause 'operator 'for
					'bindings (list (make-binding var #f))
					'initially (if init (list init) '())
					'looping (if loop (list loop) '())
					'stepping (if step (list step) '()))
		      tail)))
	  
	  (define (parse-with forms clauses ops)
	    ;; forms is (WITH <var> = ...)
	    clauses
	    (let ((head forms)
		  (tail (cdr forms))
		  (var #f)
		  (expr #f)
		  (and? #f)
		  (bind '())
		  (init '()))
	      (do ((need #t) 
		   (next #f))
		  ((or (null? tail) (loop-op? (car tail) ops)))
		(set! next (pop tail))
		(cond ((and (loop-variable? next) need)
		       (if var
			   (loop-error ops head
				       "Found '" next "' where 'and' expected."))
		       (if expr
			   (loop-error ops head
				       "Found '" next "' where 'and' expected."))
		       (set! var next)
		       (set! expr #f)
		       (set! and? #f)
		       (set! need #f))
		      ((eq? next 'and)
		       (if and?
			   (loop-error ops head "Duplicate 'and'.")
			   (if var 
			       (if expr
				   (begin
				     (push (make-binding var #f) bind)
				     (push `(set! ,var ,expr) init))
				   (push (make-binding var #f) bind))
			       (loop-error ops head "Extraneous 'and'.")))
		       (set! var #f)
		       (set! expr #f)
		       (set! and? #t)
		       (set! need #t))
		      ((eq? next '=)
		       (if expr
			   (loop-error ops head 
				       "Found '=' where 'and' expected.")
			   (set! expr (pop tail))))
		      (else
		       (if need
			   (loop-error ops head
				       "Found '" next "' where variable expected.")
			   (loop-error ops head "Found '" next
				       "' where '=' or 'and' expected.")))))
	      (if and? 
		  (loop-error ops head "Extraneous 'and'.")
		  (if var 
		      (if expr
			  (begin (push (make-binding var #f) bind)
				 (push `(set! ,var ,expr) init))
			  (push (make-binding var #f) bind))))
	      
	      (values (make-loop-clause 'operator 'with
					'bindings (reverse bind)
					'initially (reverse init))
		      tail)))
	  
	  (define (parse-do forms clauses ops)
	    clauses
	    (let ((head forms)
		  (oper (pop forms))
		  (body '()))
	      (do ()
		  ((or (null? forms)
		       (loop-op? (car forms) ops))
		   (if (null? body)
		       (loop-error ops head "Missing '" oper "' expression.")
		       (set! body (reverse body))))
		(push (car forms) body)
		(set! forms (cdr forms)))
	      (values
	       (make-loop-clause 'operator oper 'looping body)
	       forms)))
	  
	  (define (parse-finally forms clauses ops)
	    clauses
	    (let ((oper (pop forms))
		  (expr #f))
	      (if (null? forms)
		  (loop-error ops forms "Missing '" oper "' expression."))
	      (set! expr (pop forms))
	      (values (make-loop-clause 'operator oper 'finally (list expr))
		      forms)))
	  
	  (define (parse-initially forms clauses ops)
	    clauses
	    (let ((oper (pop forms))
		  (expr #f))
	      (if (null? forms)
		  (loop-error ops forms "Missing '" oper "' expression."))
	      (set! expr (pop forms))
	      (values (make-loop-clause 'operator oper 'initially (list expr))
		      forms)))
	  
	  (define (lookup-collector var clauses)
	    ;; collector is list: (<var> <type> <acc> <head>)
	    ;; returns the clause where the collect variable VAR is
	    ;; actually bound or nil if var hasn't already been bound
	    ;; if var is nil only the single system allocated collecter
	    ;; is possibly returned.
	    (let ((checkthem (lambda (var lis)
			       (do ((a #f)) 
				   ((or (null? lis) a) a)
				 (if (eq? var (car (car lis))) ;collector-var
				     (set! a (car lis)))
				 (set! lis (cdr lis))))))
	      (do ((c #f))
		  ((or (null? clauses) c) c)
		(set! c (checkthem var (loop-collectors (car clauses))))
		(set! clauses (cdr clauses)))))
	  
	  (define (compatible-accumulation? typ1 typ2)
	    (let ((l1 '(collect append nconc))
		  (l2 '(never always))
		  (l3 '(minimize maximize)))
	      (or (eq? typ1 typ2)
		  (and (member typ1 l1) (member typ2 l1))
		  (and (member typ1 l2) (member typ2 l2))
		  (and (member typ1 l3) (member typ2 l3)))))
	  
	  (define (parse-accumulation forms clauses ops)
	    ;; forms is (<op> form ...)
	    ;; where <op> is collect append nconc
	    (let ((save forms)
		  (oper (pop forms))
		  (make-collector (lambda (var type acc head)
				    (list var type acc head)))
		  ;; removed because noop
		  ;;(collector-var (lambda (col) (car col)))
		  (collector-type (lambda (col) (cadr col)))
		  (collector-acc (lambda (col) (caddr col)))
		  (collector-head (lambda (col) (cadddr col)))
		  (expr #f)
		  (coll #f)
		  (new? #f)
		  (into #f)
		  (loop '())
		  (bind '())
		  (init '())
		  (tests '())
		  (return '()))
	      
	      (if (null? forms)
		  (loop-error ops forms "Missing '" oper "' expression."))
	      (set! expr (pop forms))
	      (if (not (null? forms))
		  (if (eq? (car forms) 'into)
		      (begin
			(if (null? (cdr forms))
			    (loop-error ops save "Missing 'into' variable."))
			(if (loop-variable? (cadr forms))
			    (begin (set! into (cadr forms))
				   (set! forms (cddr forms)))
			    (loop-error ops save "Found '" (car forms)
					"' where 'into' variable expected.")))))
	      
	      ;; search for a clause that already binds either the user specified
	      ;; accumulator (into) or a system allocated one if no into.
	      ;; system collectors
	      ;;   o only one  allowed, all accumuations must be compatible
	      ;;   o returns value
	      ;;   value collector: (nil <op> <#:acc>)
	      ;;   list collector:  (nil <op> <#:tail> <#:head>)
	      ;; into collectors
	      ;;   o any number allowed
	      ;;   o returns nothing.
	      ;;   value collector: (<into> <op> <into> )
	      ;;   list collector:  (<into> <op> <#:tail> <#:head>)
	      (set! coll (lookup-collector into clauses))
	      (if (not coll)
		  (set! new? #t)
		  ;; accumulator already established by earlier clause
		  ;; check to make sure clauses are compatible.
		  (if (not (compatible-accumulation? oper (collector-type coll)))
		      (loop-error ops save "'" (collector-type coll)
				  "' and '" oper "' are incompatible accumulators.")))
	      (case oper 
		((sum count)
		 (let ((acc #f))
		   (if new?
		       (begin 
			 (set! acc (or into (gensym "v")))
			 (push (make-binding acc 0) bind)
			 ;; coll= (nil <op> <#:acc>) or (<into> <op> <into>)
			 (set! coll (make-collector into oper acc #f))
			 ;; only add a return value if new collector isnt into
			 (if (not into) (push acc return)))
		       (set! acc (collector-acc coll)))
		   (if (eq? oper 'sum)
		       (push `(set! ,acc (+ ,acc ,expr)) loop)
		       (push `(if ,expr (set! ,acc (+ ,acc 1))) loop))))
		((minimize maximize)
		 (let ((var (gensym "v"))
		       (opr (if (eq? oper 'minimize) '< '>))
		       (acc #f))
		   (if new?
		       (begin
			 (set! acc (or into (gensym "v")))
			 (push (make-binding acc #f) bind)
			 ;; coll= (nil <op> <#:acc>) or (<into> <op> <into>)
			 (set! coll (make-collector into oper acc #f))
			 ;; only add a return value if new collector isnt into
			 (if (not into) (push `(or ,acc 0) return)))
		       (set! acc (collector-acc coll)))
		   (push (make-binding var #f) bind)
		   (push `(begin (set! ,var ,expr)
				 (if (or (not ,acc) 
					 (,opr ,var ,acc))
				     (set! ,acc ,var)))
			 loop)))
		((append collect nconc)
		 ;; for list accumulation a pointer to the tail of the list
		 ;; is updated and the head of the list is returned. any
		 ;; into variable is set to the head inside the loop.
		 (let ((head #f)
		       (tail #f))
		   (if (not new?)
		       (begin (set! tail (collector-acc coll))
			      (set! head (collector-head coll)))
		       (begin
			 (if into (push (make-binding into '(list)) bind))
			 (set! tail (gensym "v"))
			 ;; allocate a pointer to the head of list
			 (set! head (gensym "v"))
			 (push (make-binding head '(list #f)) bind)
			 (push (make-binding tail #f) bind)
			 ;; initialize tail to head
			 (push `(set! ,tail ,head) init)
			 (set! coll (make-collector into oper tail head))
			 ;; only add a return value if new collector isnt into
			 (if (not into)
			     (push `(cdr ,head) return))))
		   ;; add loop accumulation forms
		   (if (eq? oper 'append)
		       (begin
			 (push `(set-cdr! ,tail (append ,expr (list))) loop)
			 (push `(set! ,tail (last-pair ,tail)) loop))
		       (if (eq? oper 'collect)
			   (begin
			     (push `(set-cdr! ,tail (list ,expr)) loop)
			     (push `(set! ,tail (cdr ,tail)) loop))
			   (begin 
			     (push `(set-cdr! ,tail ,expr) loop)
			     (push `(set! ,tail (last-pair ,tail)) loop))))
		   ;; update user into variable inside the main loop
		   ;; regardless of whether its a new collector or not
		   (if into
		       (push `(set! ,into (cdr ,head)) loop)))))
	      
	      (values (make-loop-clause 'operator oper
					'bindings (reverse bind)
					'initially (reverse init)
					'looping (reverse loop)
					'returning (reverse return)
					'collectors (if new? (list coll) '())
					'end-tests (reverse tests))
		      forms)))
	  
					;(define (loop-stop expr)
					;  `(%done% ,expr))
	  
	  (define (loop-return expr)
	    `(return ,expr))
	  
	  (define (parse-while-until forms clauses ops)
	    clauses
	    (let ((head forms)
		  (oper (pop forms))
		  (test #f)
		  (stop '(go #t))) ; :done
	      (if (null? forms)
		  (loop-error ops head "Missing '" oper "' expression."))
	      
	      (case oper
		((until ) (set! test (pop forms)))
		((while ) (set! test `(not ,(pop forms)))))
	      ;; calls the DONE continuation.
	      (values (make-loop-clause 'operator oper
					'looping (list `(if ,test ,stop)))
		      forms)))
	  
	  (define (parse-thereis forms clauses ops)
	    clauses
	    (let ((oper (car forms))
		  (expr #f)
		  (bool #f)
		  (func #f))
	      (if (null? (cdr forms))
		  (loop-error ops forms "Missing '" (car forms) "' expression." ))
	      (set! expr (cadr forms))
	      ;; fourth element of operator definition must be
	      ;; a function that returns the stop expression.
	      (set! func (cadddr (loop-op? oper ops) ))
	      
	      (case oper
		((thereis ) 
		 ;; return true as soon as expr is true or false at end
		 (set! bool #f))
		((always )
		 ;; return false as soon as expr is false, or true at end
		 (set! expr `(not ,expr))
		 (set! bool #t))
		((never )
		 ;; return false as soon as expr is true, or true at end
		 (set! bool #t)))
	      (set! forms (cddr forms))
	      ;; this calls the RETURN continuation
	      (values (make-loop-clause 'operator 'thereis
					'looping 
					(list `(if ,expr ,(func (not bool))))
					'returning 
					(list bool))
		      forms)))
	  
	  (define (parse-return forms clauses ops)
	    clauses
	    (let ((oper (car forms))
		  (expr #f)
		  (func #f))
	      (if (null? (cdr forms))
		  (loop-error ops forms "Missing '" (car forms) "' expression."))
	      (set! expr (cadr forms))
	      (set! forms (cddr forms))
	      ;; fourth element of operator definition must be
	      ;; a function that returns the stop expression.
	      (set! func (cadddr (loop-op? oper ops) ))
	      ;; this calls the RETURN continuation
	      (values (make-loop-clause 'operator 'return
					'looping `(,(func expr)))
		      forms)))
	  
	  (define (legal-in-conditional? x ops)
	    ;; FIXED (member (loop-operator...))
	    (let ((op (loop-op? x ops)))
	      (if (and op 
		       (not (null? (cddr op)))
		       (eq? (caddr op) 'task)
		       (not (member (car op) '(thereis never always))))
		  op #f)))
	  
	  (define (parse-then-else-dependents forms clauses ops)
	    (let ((previous forms)
		  (stop? #f)
		  (parsed '()))
	      
	      (do ((op #f)
		   (clause #f)
		   (remains #f))
		  ((or (null? forms) stop?))
		(set! op (legal-in-conditional? (car forms) ops))
		(if (not op)
		    (loop-error ops previous "'" (car forms)
				"' is not conditional operator."))
					;(multiple-value-setq 
					; (clause remains)
					; ( (cadr op) forms (append clauses parsed) ops))
		(call-with-values
		    (lambda () ( (cadr op) forms (append clauses parsed) ops))
		  (lambda (a b) (set! clause a) (set! remains b)))
		
					;(format #t "~%after call clause=~s forms=~S" clause forms)      
		
		(set! parsed (append parsed (list clause)))
		(set! previous forms)
		(set! forms remains)
		
		(if (not (null? forms))
		    (if (eq? (car forms) 'and)
			(begin
			  (set! forms (cdr forms))
			  (if (null? forms)
			      (loop-error ops previous "Missing 'and' clause.")))
			(if (eq? (car forms) 'else)
			    (set! stop? #t)
			    (if (loop-op? (car forms) ops)
				(set! stop? #t))))))
	      (values parsed forms)))
	  
	  (define (parse-conditional forms clauses ops)
	    (let ((ops (cons '(else ) ops))
		  (save forms)
		  (oper (car forms))
		  (loop (list))  ; avoid '() because of acl bug
		  (expr (list))
		  (then (list))
		  (else (list)))
	      (if (null? (cdr forms))
		  (loop-error ops save "Missing '" oper "' expression."))
	      (set! forms (cdr forms))
	      (set! expr (pop forms))
	      (if (null? forms)
		  (loop-error ops forms "Missing conditional clause."))
	      (if (eq? oper 'unless)
		  (set! expr (list 'not expr)))
	      (call-with-values
		  (lambda () (parse-then-else-dependents forms clauses ops))
		(lambda (a b)
		  (set! then a)
		  (set! forms b)))
	      
	      ;; combine dependant clauses if more than one
	      (if (not (null? (cdr then)))
		  (set! then (gather-clauses (list) then))
		  (set! then (car then)))
	      (loop-operator-set! then 'if)
	      
	      ;; this (if ...) is hacked so that it is a newly
	      ;; allocated list. otherwise acl and clisp have a
	      ;; nasty structure sharing problem.
	      (set! loop (list 'if expr 
			       (append `(begin ,@(loop-looping then)) (list))
			       #f))
	      (if (and (not (null? forms))
		       (eq? (car forms) 'else))
		  (begin
		    (set! forms (cdr forms))
		    (when (null? forms)
			  (loop-error ops save "Missing 'else' clause."))
		    (call-with-values 
			(lambda ()
			  (parse-then-else-dependents 
			   forms (append clauses (list then))
			   ops))
		      (lambda (a b) (set! else a) (set! forms b)))
		    (if (not (null? (cdr else)))
			(set! else (gather-clauses '() else))
			(set! else (car else)))
		    (set-car! (cdddr loop) `(begin ,@(loop-looping else)))
		    ;; flush loop forms so we dont gather actions.
		    (loop-looping-set! then '())
		    (loop-looping-set! else '())
		    (set! then (gather-clauses 'if (list then else)))))
	      (loop-looping-set! then (list loop))
	      (values then forms)))
	  
	  (define (parse-clauses forms cond? ops)
	    (if (or (null? forms)
		    (not (symbol? (car forms))))
		(list (make-loop-clause 'operator 'do 'looping forms))
		(let ((op-type? (lambda (op type)
				  (and (not (null? (cddr op)))
				       (eq? (caddr op) type)))))
		  (let ((previous forms)
			(clauses '()))
		    (do ((op #f)
			 (clause #f)
			 (remains '())
			 (body '()) )
			((null? forms))
		      (if (and cond? (eq? (car forms) 'and))
			  (pop forms))
		      (set! op (loop-op? (car forms) ops))
		      (if (not op)
			  (loop-error ops previous "Found '" (car forms)
				      "' where operator expected."))
					;(multiple-value-setq (clause remains)
					;                     ((cadr op) forms clauses ops))
		      (call-with-values
			  (lambda () ( (cadr op) forms clauses ops))
			(lambda (a b)
			  (set! clause a)
			  (set! remains b)))
		      (if (op-type? op 'task)
			  (set! body op)
			  (if (op-type? op 'iter)
			      (if (not (null? body))
				  (loop-error ops previous "'" (car op)
					      "' clause cannot follow '"
					      (car body) "'."))))
		      (set! previous forms)
		      (set! forms remains)
		      (set! clauses (append clauses (list clause))))
		    clauses))))
	  
	  (define (parse-iteration caller forms ops)
	    (gather-clauses caller (parse-clauses forms '() ops)))
	  
	  ;;
	  ;; loop implementation
	  ;;
	  
	  (define *loop-operators*
	    ;; each clause is (<op> <parser> <tag> . <whatever>)
	    (list (list 'with (function parse-with) #f)
		  (list 'initially (function parse-initially) #f)
		  (list 'repeat (function parse-repeat) 'iter)
		  (list 'for (function parse-for) 'iter
			(list 'from (function parse-numerical-for))
			(list 'downfrom (function parse-numerical-for))
			(list 'below (function parse-numerical-for))
			(list 'to (function parse-numerical-for))
			(list 'above (function parse-numerical-for))
			(list 'downto (function parse-numerical-for))
			(list 'in (function parse-sequence-iteration))
			(list 'on (function parse-sequence-iteration))
			(list 'across (function parse-sequence-iteration))
			(list '= (function parse-general-iteration)))
		  (list 'as (function parse-for) 'iter)
		  (list 'do (function parse-do) 'task)
		  (list 'collect (function parse-accumulation) 'task)
		  (list 'append (function parse-accumulation) 'task)
		  (list 'nconc (function parse-accumulation) 'task)
		  (list 'sum (function parse-accumulation) 'task)
		  (list 'count (function parse-accumulation) 'task)
		  (list 'minimize (function parse-accumulation) 'task)
		  (list 'maximize (function parse-accumulation) 'task)
		  (list 'thereis (function parse-thereis) 'task
			(function loop-return))
		  (list 'always (function parse-thereis) 'task
			(function loop-return))
		  (list 'never (function parse-thereis) 'task 
			(function loop-return))
		  (list 'return (function parse-return) 'task 
			(function loop-return))
		  (list 'while (function parse-while-until) #f )
		  (list 'until (function parse-while-until) #f )
		  (list 'when (function parse-conditional) 'task)
		  (list 'unless (function parse-conditional) 'task)
		  (list 'if (function parse-conditional) 'task)
		  (list 'finally (function parse-finally) #f)))
	  
	  ;;
	  ;; loop expansions for scheme and cltl2
	  ;;
	  
	  (define (scheme-loop forms)
	    (let ((name (gensym "v"))
		  (parsed (parse-iteration 'loop forms *loop-operators*))
		  (end-test '())
		  (done '(go #t))  ; :done
		  (return #f))
					;(write (list :parsed-> parsed))
	      ;; cltl2's loop needs a way to stop iteration from with the run
	      ;; block (the done form) and/or immediately return a value
	      ;; (the return form).  scheme doesnt have a block return or a
	      ;; go/tagbody mechanism these conditions are implemented using
	      ;; continuations.  The forms that done and return expand to are
	      ;; not hardwired into the code because this utility is also used
	      ;; by CM's 'process' macro. Instead, the done and return forms
	      ;; are returned by functions assocated with the relevant operator
	      ;; data. For example, the function that returns the return form
	      ;; is stored as the fourth element in the return operator data.
	      ;; and the done function is stored in the while and until op data.
	      
	      ;; the cadddr of the RETURN operator is a function that
	      ;; provides the form for immediately returning a value
	      ;; from the iteration.
	      
	      (let ((returnfn (cadddr (assoc 'return *loop-operators*))))
		(set! return (returnfn
			      (if (null? (loop-returning parsed))
				  #f
				  (car (loop-returning parsed))))))
	      
	      ;; combine any end-tests into a single IF expression
	      ;; that calls the (done) continuation if true. multiple
	      ;; tests are OR'ed togther
	      
	      (set! end-test
		    (let ((ends (loop-end-tests parsed)))
		      (if (null? ends)
			  '()
			  (list
			   `(if ,(if (null? (cdr ends))
				     (car ends)
				     (cons 'or ends))
				;;  calls the done continuation
				,done 
				#f)))))
	      `(let (,@ (loop-bindings parsed))
		 ,@(loop-initially parsed)
		 (call-with-exit
		  (lambda (return)     ; <- (return) returns from this lambda
		    (call-with-exit
		     (lambda (go)  ; <- (go #t) returns from this lambda
		       ;; a named let provides the actual looping mechanism.
		       ;; the various tests and actions may exit via the
		       ;; (done) or (return) continuations.
		       (let ,name () 
			    ,@end-test
			    ,@(loop-looping parsed)
			    ,@(loop-stepping parsed)
			    (,name))))
		    ;; this is the lexical point for (go #t) continuation.
		    ,@(loop-finally parsed)
		    ;; invoke the RETURN continuation with loop value or #f
		    ,return)))))
	  
	  
	  (scheme-loop args)))
      
      ;;
      ;; loop tests.
      ;;
      
      (test (loop for i below 10 collect i) '(0 1 2 3 4 5 6 7 8 9))
      (test (loop for i to 10 sum i) 55)
      (test (loop for i downto -10 count (even? i)) 6)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) thereis (= x 4)) #t)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) by 'cddr collect x) '(0 2 4 6 8))
      (test (loop for x on '(0 1 2 3) by 'cddr collect x) '((0 1 2 3) (2 3)))
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) thereis (= x 4)) #t)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) never (= x 4)) #f)
      (test (loop for x in '(0 1 2 3 4 5 6 7 8 9) never (= x 40)) #t)
      (test (loop for x in '(0 2 3 4 5 6 7 8 9) always (< x 40)) #t)
      (test (loop repeat 10 with x = 0 collect x do (set! x (+ x 1))) '(0 1 2 3 4 5 6 7 8 9))
      (test (loop repeat 10 for x = #t then (not x) collect x) '(#t #f #t #f #t #f #t #f #t #f))
      (test (loop repeat 10 count #t) 10)
      (test (loop repeat 10 count #f) 0)
      (test (loop for i to 10 collect i collect (* 2 i)) '(0 0 1 2 2 4 3 6 4 8 5 10 6 12 7 14 8 16 9 18 10 20))
      (test (loop for i from -10 to 10 by 2 nconc (list i (- i))) '(-10 10 -8 8 -6 6 -4 4 -2 2 0 0 2 -2 4 -4 6 -6 8 -8 10 -10))
      (test (loop for i from -10 downto 10 by -1 collect i) '())
      (test (loop for i downfrom 10 downto -10 by 2 collect i) '(10 8 6 4 2 0 -2 -4 -6 -8 -10))
      (test (loop for i from 10 to -10 by 1 collect i) '())
      (test (loop for i to 10 for j downfrom 10 collect i collect j) '(0 10 1 9 2 8 3 7 4 6 5 5 6 4 7 3 8 2 9 1 10 0))
      (test (loop for i below 0 collect i into foo finally (return foo)) '())
      (test (loop for i below 0 sum i into foo finally (return foo)) 0)
      (test (loop for i below 0 maximize i into foo finally (return foo)) #f)
      (test (loop with a and b = 'x and c = 2 repeat 10 for x = 1 then 'fred collect (list x a b c))
	    '((1 #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2) (fred #f x 2)))
      (test (loop for i across #(0 1 2 3) append (list i (expt 2 i))) '(0 1 1 2 2 4 3 8))
      (test (loop with a = 0 and b = -1 while (< a 10) sum a into foo do (set! a (+ a 1)) finally (return (list foo b))) '(45 -1))
      (test (loop for i from 0 until (> i 9) collect i) '(0 1 2 3 4 5 6 7 8 9))
      (test (loop for i from 0 while (< i 9) when (even? i) collect i) '(0 2 4 6 8))
      (test (loop with l = (list 0) for s in spec for k = s then (+ k s) do (push k l) finally (return l)) 'error)
      (test (loop with l = (list (encode-interval 'p 1)) for s in spec for k = (interval s) then (transpose k (interval s)) do (push k l) finally (return l)) 'error)
      ;; end loop

      ;; more macros from Rick's stuff

      (defmacro dolist (spec . body)
	;; spec = (var list . return)
	(let ((v (gensym)))
	  `(do ((,v ,(cadr spec) (cdr ,v))
		(,(car spec) #f))
	       ((null? ,v) ,@(cddr spec))
	     (set! ,(car spec) (car ,v))
	     ,@body)))

      (test (let ((sum 0)) (dolist (v (list 1 2 3) sum) (set! sum (+ sum v)))) 6)
      
      (defmacro dotimes (spec . body)
	;; spec = (var end . return)
	(let ((e (gensym))
	      (n (car spec)))
	  `(do ((,e ,(cadr spec))
		(,n 0))
	       ((>= ,n ,e) ,@(cddr spec))
	     ,@body
	     (set! ,n (+ ,n 1)))))

      (test (let ((sum 0)) (dotimes (i 3 sum) (set! sum (+ sum i)))) 3)
      
      (defmacro do* (spec end . body)
	`(let* (,@(map (lambda (var) (list (car var) (cadr var))) spec))
	   (do () ,end
	     ,@body
	     ,@(map (lambda (var) (list 'set! (car var) (caddr var))) spec))))

      (test (let ((sum 0)) (do* ((i 0 (+ i 1)) (j i (+ i 1))) ((= i 3) sum) (set! sum (+ sum j)))) 5)

      (define-macro (fluid-let xexe . body)
	;; taken with changes from Teach Yourself Scheme
	(let ((xx (map car xexe))
	      (ee (map cadr xexe))
	      (old-xx (map (lambda (ig) (gensym)) xexe)))
	  `(let ,(map (lambda (old-x x) `(,old-x ,x)) 
		      old-xx xx)
	     (dynamic-wind
		 (lambda () #f)
		 (lambda ()
		   ,@(map (lambda (x e)
			    `(set! ,x ,e)) 
			  xx ee)
		   (let ()
		     ,@body))
		 (lambda ()
		   ,@(map (lambda (x old-x)
			    `(set! ,x ,old-x)) 
			  xx old-xx))))))
      
      (test (let ((x 32)
		  (y 0))
	      (define (gx) x)
	      (fluid-let ((x 12))
		(set! y (gx)))
	      (list x y))
	    '(32 12))
      
      (test (let ((x "hi")
		  (y 0)
		  (z '(1 2 3)))
	      (define (gx) (+ x z))
	      (fluid-let 
		  ((x 32) (z (+ 123 (car z))))
		(set! y (gx)))
	      (list x y z))
            '("hi" 156 (1 2 3)))
      
      (test (let ((x 32)
		  (y 0))
	      (define (gx) x)
	      (call-with-exit
	       (lambda (return)
		 (fluid-let ((x 12))
		   (set! y (gx))
		   (return))))
	      (list x y))
	    '(32 12))

      (test (let ((x 32)
		  (y 0))
	      (define (gx) x)
	      (let ((x 100))
		(fluid-let ((x 12))
		  (set! y (gx))))
	      (list x y))
	    '(32 32))
      ;; oops! fluid-let doesn't actually work!

      ;; in CL: (defvar x 32) (let ((y 0)) (defun gx () x) (let ((x 12)) (setf y (gx))) (list x y)) -> '(32 12)
      ;;                      (let ((y 0)) (defun gx () x) (let ((x 100)) (let ((x 12)) (setf y (gx)))) (list x y)) -> '(32 12)
      ;;                      (let ((y 0)) (defun gx () x) (let ((x 100)) (let ((x 12)) (setf y (gx)) (setf x 123)) (list x y))) -> '(100 12) !
      ;; (the defvar makes x dynamic)

      
      ;; define** treats args before :optional as required args
      (define-macro (define** declarations . forms)
	(let ((name (car declarations))
	      (args (cdr declarations)))
	  (define (position thing lst count)
	    (if (or (null? lst)
		    (not (pair? (cdr lst))))
		#f
		(if (eq? thing (car lst))
		    count
		    (position thing (cdr lst) (+ count 1)))))
	  (let ((required-args (position :optional args 0)))
	    (if required-args
		`(define* (,name . func-args)
		   (if (< (length func-args) ,required-args)
		       (error "~A requires ~D argument~A: ~A" 
			      ',name ,required-args (if (> ,required-args 1) "s" "") func-args)
		       (apply (lambda* ,args ,@forms) func-args)))
		`(define* ,declarations ,@forms)))))

      ;; Rick's with-optkeys

      (define-macro (with-optkeys spec . body)
	(
	 (lambda (user rawspec body)
	   
	   (define (string->keyword str) (symbol->keyword (string->symbol str)))
	   
	   (define (key-parse-clause info mode args argn user)
	     ;; return a cond clause that parses one keyword. info for each
	     ;; var is: (<got> <var> <val>)
	     (let* ((got (car info))
		    (var (cadr info))
		    (key (string->keyword (symbol->string var))))
	       `((eq? (car ,args) ,key )
		 (if ,got (error "duplicate keyword: ~S" , key))
		 (set! ,var (if (null? (cdr ,args))
				(error "missing value for keyword: ~S" 
				       , user)
				(cadr ,args)))
		 (set! ,got #t) ; mark that we have a value for this param
		 (set! ,mode #t) ; mark that we are now parsing keywords
		 (set! ,argn (+ ,argn 1))
		 (set! ,args (cddr ,args)))))
	   
	   (define (pos-parse-clause info mode args argn I)
	     ;; return a cond clause that parses one positional. info for
	     ;; each var is: (<got> <var> <val>)
	     (let ((got (car info))
		   (var (cadr info)))
	       `((= ,argn ,I)
		 (set! ,var (car ,args))
		 (set! ,got #t) ; mark that we have a value for this param
		 (set! ,argn (+ ,argn 1))
		 (set! ,args (cdr ,args)))))
	   
	   (let* ((otherkeys? (member '&allow-other-keys rawspec))
		  ;; remove &allow-other-keys from spec
		  (spec (if otherkeys? (reverse (cdr (reverse rawspec))) rawspec))
		  (data (map (lambda (v)
			       ;; for each optkey variable v return a list
			       ;; (<got> <var> <val>) where the <got>
			       ;; variable indicates that <var> has been
			       ;; set, <var> is the optkey variable itself
			       ;; and <val> is its default value
			       (if (pair? v)
				   (cons (gensym (symbol->string (car v))) v)
				   (list (gensym (symbol->string v)) v #f)))
			     spec))
		  (args (gensym "args")) ; holds arg data as its parsed
		  (argn (gensym "argn"))
		  (SIZE (length data))
		  (mode (gensym "keyp")) ; true if parsing keywords
		  ;; keyc are cond clauses that parse valid keyword
		  (keyc (map (lambda (d) (key-parse-clause d mode args argn user))
			     data))
		  (posc (let lup ((tail data) (I 0))
			  (if (null? tail) (list)
			      (cons (pos-parse-clause (car tail) mode args argn I)
				    (lup (cdr tail) (+ I 1))))))
		  (bindings (map cdr data)) ; optkey variable bindings
		  )
	     
	     (if otherkeys?
		 (set! bindings (cons '(&allow-other-keys (list)) bindings)))
	     
	     `(let* ,bindings ; bind all the optkey variables with default values
		;; bind status and parsing vars
		(let ,(append (map (lambda (i) (list (car i) #f)) data)
			      `((,args ,user)
				(,argn 0)
				(,mode #f)))
		  ;; iterate arglist and set opt/key values
		  (do ()
		      ((null? ,args) #f)
		    (cond 
		     ;; add valid keyword clauses first
		     ,@ keyc
			;; a keyword in (car args) is now either added to
			;; &allow-other-keys or an error
			, (if otherkeys?
			      `((keyword? (car ,args))
				(if (not (pair? (cdr ,args)))
				    (error "missing value for keyword ~S" (car ,args)))
				(set! &allow-other-keys (append &allow-other-keys
								(list (car ,args)
								      (cadr ,args))))
				(set! ,mode #t) ; parsing keys now...
				(set! ,args (cddr ,args)) )
			      `((keyword? (car ,args)) ;(and ,mode (keyword? (car ,args)))
				(error "invalid keyword: ~S" (car ,args)) )
			      )
			  ;; positional clauses illegal if keywords have happened
			  (,mode (error "positional after keywords: ~S" (car ,args)))
			  ;; too many value specified
			  ((not (< ,argn ,SIZE)) (error "too many args: ~S" , args))
			  ;; add the valid positional clauses
			  ,@ posc
			     ))
		  ,@ body))
	     ))
	 (car spec)
	 (cdr spec)
	 body
	 ))
      
      (test (let ((args '(1 2 3)))  (with-optkeys (args a b c) (list a b c))) '(1 2 3))
      (test (let ((args '(1 2 3 4)))  (with-optkeys (args a b c) (list a b c))) 'error)
      (test (let ((args '(1 2))) (with-optkeys (args a b (c 33)) (list a b c))) '(1 2 33))
      (test (let ((args '())) (with-optkeys (args a b (c 33)) (list a b c))) '(#f #f 33))
      (test (let ((args '(:b 22))) (with-optkeys (args a b (c 33)) (list a b c))) '(#f 22 33))
      (test (let ((args '(-1 :z 22))) (with-optkeys (args a b (c 33)) (list a b c))) 'error)
      (test (let ((args '(:b 99 :z 22))) (with-optkeys (args a b (c 33)) (list a b c))) 'error)
      (test (let ((args '(:z 22))) (with-optkeys (args a b (c 33) &allow-other-keys) (list a b c &allow-other-keys))) '(#f #f 33 (:z 22)))
      (test (let ((args '(:id "0" :inst "flute" :name "Flute"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      (test (let ((args '(:inst "flute" :id "0" :name "Flute"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      (test (let ((args '(:id "0" :name "Flute" :inst "flute"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      (test (let ((args '(:name "Flute" :inst "flute" :id "0"))) (with-optkeys (args id inst &allow-other-keys) (list id inst &allow-other-keys))) '("0" "flute" (:name "Flute")))
      
      
      (let ()
	
	;; some common lispisms
	;;   where names are the same, but functions are different (abs for example), 
	;;   I'll prepend "cl-" to the CL version; otherwise we end up redefining
	;;   map and member, for example, which can only cause confusion.
	;;
	;; also I'm omitting the test-if-not and test-not args which strike me as ridiculous.
	;; If CLtL2 says something is deprecated, it's not included.
	;; Series and generators are ignored.
	;;
	;;  ... later ... I've run out of gas.
	
					;(define-macro (progn . body) `(let () ,@body))
	(define progn begin)
	(define-macro (prog1 first . body) (let ((result (gensym))) `(let ((,result ,first)) ,@body ,result)))
	(define-macro (prog2 first second . body) `(prog1 (progn ,first ,second) ,@body))

	(defmacro the (type form) form)
	(define-macro (defvar var . args) `(define ,var (or ,(and (not (null? args)) (car args)) #f)))

	(defmacro incf (sym . val) `(let () (set! ,sym (+ ,sym ,(if (null? val) 1 (car val)))) ,sym))
	(defmacro decf (sym . val) `(let () (set! ,sym (- ,sym ,(if (null? val) 1 (car val)))) ,sym))

	(defmacro push (val sym) 
	  `(let () 
	     (setf ,sym (cons ,val ,sym)) 
	     ,sym))

	(defmacro pop (sym) 
	  (let ((v (gensym))) 
	    `(let ((,v (car ,sym))) 
	       (setf ,sym (cdr ,sym)) 
	       ,v)))

	(defmacro* pushnew (val sym (test equal?) (key identity))
	  (let ((g (gensym))
		(k (if (procedure? key) key identity))) ; can be explicit nil!
	    `(let ((,g ,val))
	       (if (null? (cl-member (,k ,g) ,sym ,test ,k))
		   (push ,g ,sym))
	       ,sym)))

	(defmacro unless (test . forms) `(if (not ,test) (begin ,@forms)))
	(define-macro (declare . args) #f)
	(defmacro set (a b) `(set! ,(symbol->value a) ,b))

	(define-macro (setf . pairs)
	  (if (not (even? (length pairs)))
	      (error "setf has odd number of args"))
	  `(let () ,@(let ((var #f)) 
		       (map (lambda (p) 
			      (if var
				  (let ((val (if (pair? var)
						 (if (member (car var) '(aref svref elt char schar))
						     (list 'set! (cdr var) p)
						     (if (eq? (car var) 'car)
							 (list 'set-car! (cadr var) p)
							 (if (eq? (car var) 'cdr)
							     (list 'set-cdr! (cadr var) p)
							     (if (eq? (car var) 'nth)
								 (list 'set! (list (caddr var) (cadr var)) p)
								 (list 'set! var p)
								 ))))
						 (list 'set! var p))))
				    (set! var #f)
				    val)
				  (begin
				    (set! var p)
				    '())))
			    pairs))))

	(define-macro (setq . pairs)
	  (if (not (even? (length pairs)))
	      (error "setq has odd number of args"))
	  `(let () ,@(let ((var #f)) 
		       (map (lambda (p) 
			      (if var
				  (let ((val (list 'set! var p)))
				    (set! var #f)
				    val)
				  (begin
				    (set! var p)
				    '())))
			    pairs))))

	(define-macro (psetq . pairs)
	  (let ((vals '())
		(vars '()))
	    (do ((var-val pairs (cddr var-val)))
		((null? var-val))
	      (let ((interval (gensym)))
		(set! vals (cons (list interval (cadr var-val)) vals))
		(set! vars (cons (list 'set! (car var-val) interval) vars))))
	    `(let ,(reverse vals)
	       ,@vars)))

	(define (mapcar func . lists)
	  ;; not scheme's map because lists can be different lengths
	  ;; and args can be any sequence type (all mixed together)
	  (define (mapcar-seqs func seqs)
	    (if (null? seqs)
		'()
		(cons (func (car seqs))
		      (mapcar-seqs func (cdr seqs)))))

	  (define (mapcar-1 index lens func seqs)
	    (if (member index lens)
		'()
		(cons (apply func (mapcar-seqs (lambda (obj) (obj index)) seqs))
		      (mapcar-1 (+ index 1) lens func seqs))))

	  (let ((lens (map length lists)))
	    (mapcar-1 0 lens func lists)))
#|
(define (mapcar func . lists)
  ;; not scheme's map because lists can be different lengths
  (if (member '() lists)
      '()
      (cons (apply func (map car lists))
	    (apply mapcar func (map cdr lists)))))
|#

	(define (maplist function . lists)
	  (if (member '() lists)
	      '()
	      (cons (apply function lists)
		    (apply maplist function (map cdr lists)))))

	(define (mapc function . lists)
	  (define (mapc-1 function . lists)
	    (if (not (member '() lists))
		(begin
		  (apply function (map car lists))
		  (apply mapc-1 function (map cdr lists)))))
	  (apply mapc-1 function lists)
	  (car lists))

	(define (mapl function . lists)
	  (define (mapl-1 function . lists)
	    (if (not (member '() lists))
		(begin
		  (apply function lists)
		  (apply mapl-1 function (map cdr lists)))))
	  (apply mapl-1 function lists)
	  (car lists))

	(define (mapcon function . lists)
	  (apply nconc (apply maplist function lists)))

	(define (mapcan function . lists)
	  (apply nconc (apply mapcar function lists)))
	  
	(define* (map-into result-sequence function . sequences)
	  (if (or (null? result-sequence)
		  (null? sequences))
	      result-sequence
	      (let* ((vals (apply mapcar function sequences))
		     (len (min (length vals) (length result-sequence))))
		(do ((i 0 (+ i 1)))
		    ((= i len))
		  (set! (result-sequence i) (vals i)))
		result-sequence)))


	(define input-stream-p input-port?)
	(define output-stream-p output-port?)


	;; -------- lists

	;; in CL (cdr '()) is nil

	(define (first l) (if (not (null? l)) (list-ref l 0) '()))
	(define (second l) (if (> (length l) 1) (list-ref l 1) '()))
	(define (third l) (if (> (length l) 2) (list-ref l 2) '()))
	(define (fourth l) (if (> (length l) 3) (list-ref l 3) '()))
	(define (fifth l) (if (> (length l) 4) (list-ref l 4) '()))
	(define (sixth l) (if (> (length l) 5) (list-ref l 5) '()))
	(define (seventh l) (if (> (length l) 6) (list-ref l 6) '()))
	(define (eighth l) (if (> (length l) 7) (list-ref l 7) '()))
	(define (ninth l) (if (> (length l) 8) (list-ref l 8) '()))
	(define (tenth l) (if (> (length l) 9) (list-ref l 9) '()))
	(define (nth n l) (if (< n (length l)) (list-ref l n) '()))
	(define (endp val) (if (null? val) #t (if (pair? val) #f (error "bad arg to endp"))))
	(define rest cdr)
	(define list-length length)
	(define* (cl-make-list size (initial-element '())) (make-list size initial-element))

	(define (copy-list lis) 
	  (if (not (pair? lis))
	      lis
	      (cons (car lis) (copy-list (cdr lis)))))

	(define (rplaca x y) (set-car! x y) x)
	(define (rplacd x y) (set-cdr! x y) x)

	(define (copy-tree lis)
	  (if (pair? lis)
	      (cons (copy-tree (car lis))
		    (copy-tree (cdr lis)))
	      lis))

	(define* (butlast lis (n 1))
	  (let ((len (length lis)))
	    (if (<= len n)
		'()
		(let ((result '()))
		  (do ((i 0 (+ i 1))
		       (lst lis (cdr lst)))
		      ((= i (- len n)) (reverse result))
		    (set! result (cons (car lst) result)))))))

	(define* (last lst (n 1))
	  (let ((len (length lst)))
	    (do ((i 0 (+ i 1))
		 (l lst (cdr l)))
		((or (null? l)
		     (>= i (- len n)))
		 l))))

	(define (nthcdr n lst) 
	  (do ((i n (- i 1)) 
	       (result lst (cdr result))) 
	      ((or (null? result) (zero? i)) result)))

	(define* (tree-equal a b (test eql)) 
	  (define (teq a b)
	    (if (not (pair? a))
		(and (not (pair? b))
		     (test a b))
		(and (pair? b)
		     (teq (car a) (car b))
		     (teq (cdr a) (cdr b)))))
	  (teq a b))

	(define (acons key datum alist) (cons (cons key datum) alist))

	(define* (subst-if new test tree (key identity))
	  (if (test (key tree))
	      new
	      (if (not (pair? tree))
		  tree
		  (cons (subst-if new test (car tree) key)
			(subst-if new test (cdr tree) key)))))

	(define* (subst-if-not new test tree (key identity))
	  (subst-if new (lambda (obj) (not (test obj))) tree key))

	(define* (subst new old tree (test eql) (key identity))
	  (subst-if new (lambda (obj) (test old obj)) tree key))

	(define (list* obj1 . objs)
	  (define (list-1 obj)
	    (if (null? (cdr obj))
		(car obj)
		(cons (car obj) (list-1 (cdr obj)))))
	  (if (null? objs)
	      obj1
	      (cons obj1 (list-1 objs))))

	(define* (assoc-if predicate alist (key car))
	  (if (null? alist)
	      '()
	      (if (and (not (null? (car alist)))
		       (predicate (key (car alist))))
		  (car alist)
		  (assoc-if predicate (cdr alist) key))))
	
	(define* (assoc-if-not predicate alist (key car))
	  (assoc-if (lambda (obj) (not (predicate obj))) alist key))

	(define* (cl-assoc item alist (test eql) (key car))
	  (assoc-if (lambda (obj) (test item obj)) alist key))
	
	(define* (rassoc-if predicate alist (key cdr))
	  (if (null? alist)
	      '()
	      (if (and (not (null? (car alist)))
		       (predicate (key (car alist))))
		  (car alist)
		  (rassoc-if predicate (cdr alist) key))))
	
	(define* (rassoc-if-not predicate alist (key cdr))
	  (rassoc-if (lambda (obj) (not (predicate obj))) alist key))

	(define* (rassoc item alist (test eql) (key cdr))
	  (rassoc-if (lambda (obj) (test item obj)) alist key))

	(define (copy-alist alist)
	  (if (null? alist)
	      '()
	      (cons (if (pair? (car alist))
			(cons (caar alist) (cdar alist))
			(car alist))
		    (copy-alist (cdr alist)))))

	(define (revappend x y) (append (reverse x) y))

	
	(define* (pairlis keys data alist)
	  (if (not (= (length keys) (length data)))
	      (error "pairlis keys and data lists should have the same length"))
	  (let ((lst (or alist '())))
	    (if (null? keys)
		lst
		(do ((key keys (cdr key))
		     (datum data (cdr datum)))
		    ((null? key) lst)
		  (set! lst (cons (cons (car key) (car datum)) lst))))))

	(define* (sublis alist tree (test eql) (key car))
	  (let ((val (cl-assoc tree alist test key)))
	    (if (not (null? val))
		(cdr val)
		(if (not (pair? tree))
		    tree
		    (cons (sublis alist (car tree) test key)
			  (sublis alist (cdr tree) test key))))))

	(define* (nsublis alist tree (test eql) (key car)) ; sacla
	  (define (sub subtree)
	    (let ((ac (cl-assoc subtree alist test key)))
	      (if (not (null? ac))
		  (cdr ac)
		  (if (not (pair? subtree))
		      subtree
		      (let ()
			(set-car! subtree (sub (car subtree)))
			(set-cdr! subtree (sub (cdr subtree)))
			subtree)))))
	  (sub tree))

	(define* (nsubst-if new predicate tree (key identity)) ; sacla
	  (define (sub subtree)
	    (if (predicate (key subtree))
		new
		(if (not (pair? subtree))
		    subtree
		    (let ()
		      (set-car! subtree (sub (car subtree)))
		      (set-cdr! subtree (sub (cdr subtree)))
		      subtree))))
	  (sub tree))

	(define* (nsubst-if-not new predicate tree (key identity))
	  (nsubst-if new (lambda (obj) (not (predicate obj))) tree key))
    
	(define* (nsubst new old tree (test eql) (key identity))
	  (nsubst-if new (lambda (obj) (test old obj)) tree key))

	(define (ldiff lst object) ; sacla
	  (if (not (eqv? lst object))
	      (let* ((result (list (car lst)))
		     (splice result))
		(call-with-exit
		 (lambda (return)
		   (do ((l (cdr lst) (cdr l)))
		       ((not (pair? l))
			(if (eql l object) 
			    (set-cdr! splice '()))
			result)
		     (if (eqv? l object)
			 (return result)
			 (set! splice (cdr (rplacd splice (list (car l))))))))))
	      '()))

	(define* (member-if predicate list (key identity))
	  (if (null? list)
	      '()
	      (if (predicate (key (car list)))
		  list
		  (member-if predicate (cdr list) key))))

	(define* (member-if-not predicate list (key identity))
	  (member-if (lambda (obj) (not (predicate obj))) list key))

	(define* (cl-member item list (test eql) (key identity))
	  (if (null? list)
	      '()
	      (if (test item (key (car list)))
		  list
		  (cl-member item (cdr list) test key))))

	(define* (adjoin item list (test eql) (key identity))
	  (if (not (null? (cl-member (key item) list test key)))
	      list
	      (cons item list)))

	(define (tailp sublist list)
	  (or (eq? sublist list)
	      (and (not (null? list))
		   (tailp sublist (cdr list)))))

	(define* (union list1 list2 (test eql) (key identity))
	  (let ((new-list (copy list1)))
	    (do ((obj list2 (cdr obj)))
		((null? obj) new-list)
	      (set! new-list (adjoin (car obj) new-list test key)))))

	(define nunion union) ; this is not required to be destructive

	(define* (intersection list1 list2 (test eql) (key identity))
	  (let ((new-list '()))
	    (do ((obj list1 (cdr obj)))
		((null? obj) new-list)
	      (if (not (null? (cl-member (key (car obj)) list2 test key)))
		  (set! new-list (adjoin (car obj) new-list test key))))))

	(define nintersection intersection)
	    
	(define* (set-difference list1 list2 (test eql) (key identity))
	  (let ((new-list '()))
	    (do ((obj list1 (cdr obj)))
		((null? obj) new-list)
	      (if (null? (cl-member (key (car obj)) list2 test key))
		  (set! new-list (adjoin (car obj) new-list test key))))))

	(define nset-difference set-difference)

	(define* (set-exclusive-or list1 list2 (test eql) (key identity))
	  (let ((new-list '()))
	    (do ((obj list1 (cdr obj)))
		((null? obj))
	      (if (null? (cl-member (key (car obj)) list2 test key))
		  (set! new-list (adjoin (car obj) new-list test key))))
	    (do ((obj list2 (cdr obj)))
		((null? obj) new-list)
	      (if (null? (cl-member (key (car obj)) list1 test key))
		  (set! new-list (adjoin (car obj) new-list test key))))))

	(define nset-exclusive-or set-exclusive-or)

	(define* (subsetp list1 list2 (test eql) (key identity))
	  (call-with-exit
	   (lambda (return)
	     (do ((obj list1 (cdr obj)))
		 ((null? obj) #t)
	      (if (null? (cl-member (key (car obj)) list2 test key))
		  (return nil))))))

	(define* (nbutlast list (n 1)) ; sacla
	  (if (null? list)
	      '()
	      (let ((length (do ((p (cdr list) (cdr p))
				 (i 1 (1+ i)))
				((not (pair? p)) i))))
		(if (> length n)
		    (do ((1st (cdr list) (cdr 1st))
			 (2nd list 1st)
			 (count (- length n 1) (- count 1)))
			((zero? count) 
			 (set-cdr! 2nd '())
			 list))
		    '()))))

	(define (nconc . lists) ; sacla sort of
	  (let ((ls (let ()
		      (define (strip-nulls lst)
			(if (null? lst)
			    '()
			    (if (null? (car lst))
				(strip-nulls (cdr lst))
				lst)))
		      (strip-nulls lists))))
	    (if (null? ls)
		'()
	      (let* ((top (car ls))
		     (splice top))
		(do ((here (cdr ls) (cdr here)))
		    ((null? here) top)
		  (set-cdr! (last splice) (car here))
		  (if (not (null? (car here)))
		      (set! splice (car here))))))))

	(define (nreconc x y) (nconc (nreverse x) y))



	;; -------- numbers

	(define (conjugate z) (make-rectangular (real-part z) (- (imag-part z))))
	(define zerop zero?)
	(define oddp odd?)
	(define evenp even?)
	(define plusp positive?)
	(define minusp negative?)
	(define realpart real-part)
	(define imagpart imag-part)
	(define* (float x ignore) (* 1.0 x))
	(define rational rationalize)
	(define mod modulo)
	(define rem remainder)

	(define (logtest i1 i2) (not (zero? (logand i1 i2))))
	(define (logbitp index integer) (logtest (expt 2 index) integer))
	(define (lognand n1 n2) (lognot (logand n1 n2)))
	(define (lognor n1 n2) (lognot (logior n1 n2)))
	(define (logandc1 n1 n2) (logand (lognot n1) n2))
	(define (logandc2 n1 n2) (logand n1 (lognot n2)))
	(define (logorc1 n1 n2) (logior (lognot n1) n2))
	(define (logorc2 n1 n2) (logior n1 (logior n2)))
	(define (logeqv . ints) (lognot (apply logxor ints)))

	;; from slib
	(define (logcount n)
	  (define bitwise-bit-count
	    (letrec ((logcnt (lambda (n tot)
			       (if (zero? n)
				   tot
				   (logcnt (quotient n 16)
					   (+ (vector-ref
					       '#(0 1 1 2 1 2 2 3 1 2 2 3 2 3 3 4)
					       (modulo n 16))
					      tot))))))
	      (lambda (n)
		(cond ((negative? n) (lognot (logcnt (lognot n) 0)))
		      ((positive? n) (logcnt n 0))
		      (else 0)))))
	  (cond ((negative? n) (bitwise-bit-count (lognot n)))
		(else (bitwise-bit-count n))))

	(define-constant boole-clr 0)
	(define-constant boole-set 1)
	(define-constant boole-1 2)
	(define-constant boole-2 3)
	(define-constant boole-c1 4)
	(define-constant boole-c2 5)
	(define-constant boole-and 6)
	(define-constant boole-ior 7)
	(define-constant boole-xor 8)
	(define-constant boole-eqv 9)
	(define-constant boole-nand 10)
	(define-constant boole-nor 11)
	(define-constant boole-andc1 12)
	(define-constant boole-andc2 13)
	(define-constant boole-orc1 14)
	(define-constant boole-orc2 15)

	(define (boole op int1 int2)
	  (cond
	    ((= op boole-clr)   0)
	    ((= op boole-set)   -1) ;; all ones -- "always 1" is misleading
	    ((= op boole-1)     int1)
	    ((= op boole-2)     int2)
	    ((= op boole-c1)    (lognot int1))
	    ((= op boole-c2)    (lognot int2))
	    ((= op boole-and)   (logand int1 int2))
	    ((= op boole-ior)   (logior int1 int2))
	    ((= op boole-xor)   (logxor int1 int2))
	    ((= op boole-eqv)   (logeqv int1 int2))
	    ((= op boole-nand)  (lognot (logand int1 int2)))
	    ((= op boole-nor)   (lognot (logior int1 int2)))
	    ((= op boole-andc1) (logand (lognot int1) int2))
	    ((= op boole-andc2) (logand int1 (lognot int2)))
	    ((= op boole-orc1)  (logior (lognot int1) int2))
	    ((= op boole-orc2)  (logior int1 (lognot int2)))))

	;; from Rick
	(define (byte siz pos)
	  ;; cache size, position and mask.
	  (list siz pos (ash (- (ash 1 siz) 1) pos)))

	(define (byte-size bytespec) (car bytespec))
	(define (byte-position bytespec) (cadr bytespec))
	(define (byte-mask bytespec) (caddr bytespec))

	(define (ldb bytespec integer)
	  (ash (logand integer (byte-mask bytespec))
	       (- (byte-position bytespec))))

	(define (dpb integer bytespec into)
	  (logior (ash (logand integer (- (ash 1 (byte-size bytespec)) 1)) (byte-position bytespec))
		  (logand into (lognot (byte-mask bytespec)))))

	(define (ldb-test byte int) (not (zero? (ldb byte int))))
	(define (mask-field byte int) (logand int (dpb -1 byte 0)))
	(define (deposit-field byte spec int) (logior (logand byte (byte-mask spec)) (logand int (lognot (byte-mask spec)))))
	(define (scale-float x k) (* x (expt 2.0 k)))
	
	;; from clisp -- can't see any point to most of these
	(define-constant double-float-epsilon 1.1102230246251568e-16)
	(define-constant double-float-negative-epsilon 5.551115123125784e-17)
	(define-constant least-negative-double-float -2.2250738585072014e-308)
	(define-constant least-negative-long-float -5.676615526003731344L-646456994)
	(define-constant least-negative-normalized-double-float -2.2250738585072014e-308)
	(define-constant least-negative-normalized-long-float -5.676615526003731344L-646456994)
	(define-constant least-negative-normalized-short-float -1.1755e-38)
	(define-constant least-negative-normalized-single-float -1.1754944e-38)
	(define-constant least-negative-short-float -1.1755e-38)
	(define-constant least-negative-single-float -1.1754944e-38)
	(define-constant least-positive-double-float 2.2250738585072014e-308)
	(define-constant least-positive-long-float 5.676615526003731344e-646456994)
	(define-constant least-positive-normalized-double-float 2.2250738585072014e-308)
	(define-constant least-positive-normalized-long-float 5.676615526003731344e-646456994)
	(define-constant least-positive-normalized-short-float 1.1755e-38)
	(define-constant least-positive-normalized-single-float 1.1754944e-38)
	(define-constant least-positive-short-float 1.1755e-38)
	(define-constant least-positive-single-float 1.1754944e-38)
	(define-constant long-float-epsilon 5.4210108624275221706e-20)
	(define-constant long-float-negative-epsilon 2.7105054312137610853e-20)
	(define-constant most-negative-double-float -1.7976931348623157e308)
	;; most-negative-fixnum 
	(define-constant most-negative-long-float -8.8080652584198167656L646456992) 
	(define-constant most-negative-short-float -3.4028e38)
	(define-constant most-negative-single-float -3.4028235e38)
	(define-constant most-positive-double-float 1.7976931348623157e308)
	;; most-positive-fixnum 
	(define-constant most-positive-long-float 8.8080652584198167656e646456992)
	(define-constant most-positive-short-float 3.4028e38)
	(define-constant most-positive-single-float 3.4028235e38)
	(define-constant short-float-epsilon 7.6295e-6)
	(define-constant short-float-negative-epsilon 3.81476e-6)
	(define-constant single-float-epsilon 5.960465e-8)
	(define-constant single-float-negative-epsilon 2.9802326e-8)

	(define (lisp-implementation-type) "s7")
	(define (lisp-implementation-version) (s7-version))
	(define (software-type) "s7")
	(define (software-version) (s7-version))

	(define (machine-version)
	  (if (file-exists? "/proc/cpuinfo")
	      (call-with-input-file "/proc/cpuinfo"
		(lambda (cpufile)
		  (do ((line (read-line cpufile) (read-line cpufile)))
		      ((or (eof-object? line)
			   (string=? (substring line 0 10) "model name"))
		       (if (string? line)
			   (string-trim " " (substring line (+ 1 (position #\: line))))
			   "unknown")))))
	      "unknown"))
	
	;; = < <= > >= are the same, also min max + - * / lcm gcd exp expt log sqrt
	;; sin cos tan acos asin atan pi sinh cosh tanh asinh acosh atanh
	;; numerator denominator logior logxor logand ash integer-length random

	;; slightly different: floor ceiling truncate round and the ff cases thereof
	;; abs of complex -> magnitude
	(define (cl-abs x) (if (not (zero? (imag-part x))) (magnitude x) (abs x)))

	;; these actually return multiple values
	(define* (cl-floor x (divisor 1)) (floor (/ x divisor)))
	(define* (cl-ceiling x (divisor 1)) (ceiling (/ x divisor)))
	(define* (cl-truncate x (divisor 1)) (truncate (/ x divisor)))
	(define* (cl-round x (divisor 1)) (round (/ x divisor)))
	(define* (ffloor x divisor) (* 1.0 (cl-floor x divisor)))
	(define* (fceling x divisor) (* 1.0 (cl-ceiling x divisor)))
	(define* (ftruncate x divisor) (* 1.0 (cl-truncate x divisor)))
	(define* (fround x divisor) (* 1.0 (cl-round x divisor)))
       
	(define (/= . args) 
	  (if (null? (cdr args))
	      #t 
	      (if (member (car args) (cdr args))
		  #f
		  (apply /= (cdr args)))))

	(define (1+ x) (+ x 1))
	(define (1- x) (- x 1))
	(define (isqrt x) (floor (sqrt x)))
	(define phase angle)
	(define* (complex rl (im 0.0)) (make-rectangular rl im))
	(define (signum x) (if (zerop x) x (/ x (abs x))))
	(define (cis x) (exp (make-rectangular 0.0 x)))


	;; -------- characters

	(define char-code-limit 256)
	(define alpha-char-p char-alphabetic?)
	(define upper-case-p char-upper-case?)
	(define lower-case-p char-lower-case?)
	(define* (digit-char-p c (radix 10)) (string->number (string c) radix))
	(define (alphanumericp c) (or (char-alphabetic? c) (char-numeric? c)))

	(define* (char= . args) (or (< (length args) 2) (apply char=? args)))
	(define* (char< . args) (or (< (length args) 2) (apply char<? args)))
	(define* (char<= . args) (or (< (length args) 2) (apply char<=? args)))
	(define* (char> . args) (or (< (length args) 2) (apply char>? args)))
	(define* (char>= . args) (or (< (length args) 2) (apply char>=? args)))
	(define* (char-equal . args) (or (< (length args) 2) (apply char-ci=? args)))
	(define* (char-lessp . args) (or (< (length args) 2) (apply char-ci<? args)))
	(define* (char-greaterp . args) (or (< (length args) 2) (apply char-ci>? args)))
	(define* (char-not-lessp . args) (or (< (length args) 2) (apply char-ci>=? args)))
	(define* (char-not-greaterp . args) (or (< (length args) 2) (apply char-ci<=? args)))

	(define (char/= . args) 
	  (if (null? (cdr args))
	      #t 
	      (if (member (car args) (cdr args))
		  #f
		  (apply char/= (cdr args)))))

	(define (char-not-equal . args) 
	  (if (null? (cdr args))
	      #t 
	      (if (or (member (char-upcase (car args)) (cdr args))
		      (member (char-downcase (car args)) (cdr args)))
		  #f
		  (apply char-not-equal (cdr args)))))

	(define char-code char->integer)
	(define code-char integer->char)

	(define (character c) 
	  (if (char? c) 
	      c 
	      (if (integer? c)
		  (integer->char c)
		  (if (string? c)
		      (c 0)
		      (if (symbol? c)
			  ((symbol->string c) 0))))))

	;; char-upcase and char-downcase are ok
	(define char-int char->integer)
	(define int-char integer->char)

	(define* (digit-char w (radix 10))
	  (let ((str (number->string w radix)))
	    (and str (= (length str) 1) (str 0))))

	(define (both-case-p c) "unimplemented")
	(define (standard-char-p c) "unimplemented")
	(define (char-name c) "unimplemented")
	(define (name-char s) "unimplemented")

	;; --------

	(define terpri newline)


	;; -------- types

	(define vectorp vector?)
	(define simple-vector-p vector?)
	(define symbolp symbol?)
	(define (atom obj) (not (pair? obj)))
	(define consp pair?)
	(define (null obj) (or (not obj) (null? obj)))
	(define (listp obj) (or (null? obj) (pair? obj)))
	(define numberp number?)
	(define integerp integer?)
	(define rationalp rational?)
	(define (floatp l) (and (number? l) (not (rational? l)) (zero? (imag-part l)))) ; clisp
	(define (complexp l) (and (complex? l) (not (real? l))))
	(define realp real?)
	(define characterp char?)
	(define stringp string?)
	(define simple-string-p string?)
	(define arrayp vector?)
	(define simple-bit-vector-p vector?)
	(define keywordp keyword?)
	(define functionp procedure?)

	(define-constant t #t)
	(define-constant nil '())

	(define eq eq?)
	(define eql eqv?)
	(define equal equal?)

	(define (equalp x y)
	  (or (equal x y)
	      (and (char? x) (char? y) (char-ci=? x y))
	      (and (number? x) (number? y) (= x y))
	      (and (string? x) (string? y) (string-ci=? x y))))

	(define symbol-value symbol->value)
	(define symbol-function symbol->value)
	(define fdefinition symbol->value)
	(define boundp defined?)
	(define fboundp defined?)
	(define (funcall fn . arguments) (apply fn arguments))
	(define-constant call-arguments-limit 65536)

	(define (identity x) x)


	;; -------- sequences

	(define* (count-if predicate sequence from-end (start 0) end (key identity))
	  (let* ((counts 0)
		 (len (length sequence))
		 (nd (or (and (number? end) end) len))) ; up to but not including end
	    (if (< nd start)
		(error "count-if :start ~A is greater than ~A ~A" start (if end ":end" "length") nd))
	    (if (not from-end)
		(do ((i start (+ i 1)))
		    ((= i nd))
		  (if (predicate (key (sequence i)))
		      (set! counts (+ counts 1))))
		(do ((i (- nd 1) (- i 1)))
		    ((< i start))
		  (if (predicate (key (sequence i)))
		      (set! counts (+ counts 1)))))
		counts))

	(define* (count-if-not predicate sequence from-end (start 0) end (key identity))
	  (count-if (lambda (obj) (not (predicate obj))) sequence from-end start end key))

	(define* (count item sequence from-end (test eql) (start 0) end (key identity))
	  (count-if (lambda (arg) (test item arg)) sequence from-end start end key))

	(define* (find-if predicate sequence from-end (start 0) end (key identity))
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))) ; up to but not including end
	    (if (< nd start)
		(error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
	    (call-with-exit
	     (lambda (return)
	       (if (not from-end)
		   (do ((i start (+ i 1)))
		       ((= i nd) #f)
		     (if (predicate (key (sequence i)))
			 (return (sequence i))))
		   (do ((i (- nd 1) (- i 1)))
		       ((< i start) #f)
		     (if (predicate (key (sequence i)))
			 (return (sequence i)))))))))

	(define* (find-if-not predicate sequence from-end (start 0) end (key identity))
	  (find-if (lambda (obj) (not (predicate obj))) sequence from-end start end key))

	(define* (find item sequence from-end (test eql) (start 0) end (key identity))
	  (find-if (lambda (arg) (test item arg)) sequence from-end start end key))
	     
	(define* (position-if predicate sequence from-end (start 0) end (key identity))
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))) ; up to but not including end
	    (if (< nd start)
		(error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
	    (call-with-exit
	     (lambda (return)
	       (if (not from-end)
		   (do ((i start (+ i 1)))
		       ((= i nd) #f)
		     (if (predicate (key (sequence i)))
			 (return i)))
		   (do ((i (- nd 1) (- i 1)))
		       ((< i start) #f)
		     (if (predicate (key (sequence i)))
			 (return i))))))))

	(define* (position-if-not predicate sequence from-end (start 0) end (key identity))
	  (position-if (lambda (obj) (not (predicate obj))) sequence from-end start end key))

	(define* (position item sequence from-end (test eql) (start 0) end (key identity))
	  (position-if (lambda (arg) (test item arg)) sequence from-end start end key))


	(define* (nsubstitute-if new-item test sequence from-end (start 0) end count (key identity))
	  (if (and (number? count)
		   (not (positive? count)))
	      sequence
	      (let* ((len (length sequence))
		     (nd (or (and (number? end) end) len))) ; up to but not including end
		(if (< nd start)
		    (error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
		(let ((cur-count 0))
		  (if (not (number? count))
		      (set! count len))
		  (if (not from-end)
		      (do ((i start (+ i 1)))
			  ((or (= cur-count count)
			       (= i nd))
			   sequence)
			(if (test (key (sequence i)))
			    (begin
			      (set! cur-count (+ cur-count 1))
			      (set! (sequence i) new-item))))
		      (do ((i (- nd 1) (- i 1)))
			  ((or (= cur-count count)
			       (< i start))
			   sequence)
			(if (test (key (sequence i)))
			    (begin
			      (set! cur-count (+ cur-count 1))
			      (set! (sequence i) new-item)))))))))

	(define* (nsubstitute-if-not new-item test sequence from-end (start 0) end count (key identity))
	  (nsubstitute-if new-item (lambda (obj) (not (test obj))) sequence from-end start end count key))

	(define* (nsubstitute new-item old-item sequence from-end (test eql) (start 0) end count (key identity))
	  (nsubstitute-if new-item (lambda (arg) (test old-item arg)) sequence from-end start end count key))

	(define* (substitute-if new-item test sequence from-end (start 0) end count (key identity))
	  (nsubstitute-if new-item test (copy sequence) from-end start end count key))

	(define* (substitute-if-not new-item test sequence from-end (start 0) end count (key identity))
	  (substitute-if new-item (lambda (obj) (not (test obj))) (copy sequence) from-end start end count key))

	(define* (substitute new-item old-item sequence from-end (test eql) (start 0) end count (key identity))
	  (nsubstitute new-item old-item (copy sequence) from-end test start end count key))

	(define* (reduce function sequence from-end (start 0) end initial-value (key identity))
	  (let* ((slen (length sequence))
		 (nd (or (and (number? end) end) slen))
		 (len (min slen (- nd start))))
	    (if (< nd start)
		(error "~A :start ~A is greater than ~A ~A" __func__ start (if end ":end" "length") nd))
	    (if (not (positive? len))
		(or initial-value
		    (function))
		(if (and (= len 1)
			 (not initial-value))
		    (sequence start)
		    (if (and (not from-end) (not (null? from-end)))
			(let* ((first-arg (or initial-value (key (sequence start))))
			       (second-arg (if initial-value (key (sequence start)) (key (sequence (+ start 1)))))
			       (val (function first-arg second-arg)))
			  (do ((i (if initial-value (+ start 1) (+ start 2)) (+ i 1)))
			      ((= i nd) val)
			    (set! val (function val (key (sequence i))))))
			(let* ((second-arg (or initial-value (key (sequence (- nd 1)))))
			       (first-arg (if initial-value (key (sequence (- nd 1))) (key (sequence (- nd 2)))))
			       (val (function first-arg second-arg)))
			  (do ((i (if initial-value (- nd 2) (- nd 3)) (- i 1)))
			      ((< i start) val)
			    (set! val (function (key (sequence i)) val)))))))))

	(define (nreverse sequence)
	  (let ((len (length sequence)))
	    (do ((i 0 (+ i 1))
		 (j (- len 1) (- j 1)))
		((>= i j) sequence)
	      (let ((tmp (sequence i)))
		(set! (sequence i) (sequence j))
		(set! (sequence j) tmp)))))

	(define (cl-reverse sequence)
	  (nreverse (copy sequence)))
	
	(define copy-seq copy)
	(define (complement fn) (lambda args (not (apply fn args))))
	(define (elt sequence index) (sequence index))
	;; length is ok

	(define* (some predicate . sequences)
	  (call-with-exit
	   (lambda (return)
		     (apply for-each 
		      (lambda args
			(let ((val (apply predicate args)))
			  (if val (return val))))
		      sequences)
		     #f)))

	(define* (notany predicate . sequences)
	  (not (apply some predicate sequences)))

	(define* (every predicate . sequences)
	  (call-with-exit
	   (lambda (return)
		     (apply for-each 
		      (lambda args
			(if (not (apply predicate args))
			    (return #f)))
		      sequences)
		     #t)))

	(define* (notevery predicate . sequences)
	  (not (apply every predicate sequences)))

	(define* (cl-fill sequence item (start 0) end) ; actuall "fill" doesn't collide, but it's confusing
	  (let ((nd (or (and (not (null? end)) end)
			(length sequence))))
	    (if (and (= start 0)
		     (= nd (length sequence)))
		(fill! sequence item)
		(do ((i start (+ i 1)))
		    ((= i nd))
		  (set! (sequence i) item)))
	    sequence))

	;; many of the sequence functions return a different length sequence, but
	;;   for user-defined sequence types, we can't use the 'type kludge (or
	;;   at least it's ugly), so we need either (make obj size initial-value)
	;;   where obj is a representative of the desired type, or another
	;;   arg to copy giving the new object's size.  For now, I'll cobble up
	;;   something explicit.
	;;
	;; perhaps the extended type could give its type symbol as well as the make function?
	;; 'vct and make-vct etc

	(define (make obj size)
	  (cond ((vector? obj)     (make-vector size))
		((list? obj)       (make-list size))
		((string? obj)     (make-string size))
		((hash-table? obj) (make-hash-table size)))) ; does this make any sense?

	(define* (make-sequence type size initial-element)
	  (case type 
	    ((vector bit-vector simple-vector) (make-vector size initial-element))
	    ((hash-table) (make-hash-table size))
	    ((string) (cl-make-string size (or initial-element #\null))) ; not #f!
	    ((list) (cl-make-list size initial-element))
            (else '())))

	(define (cl-map type func . lists)
	  (let* ((vals (apply mapcar func lists))
		 (len (length vals)))
	    (let ((obj (make-sequence (or type 'list) len)))
	      (if (> (length obj) 0)
		  (do ((i 0 (+ i 1)))
		      ((= i len))
		    (set! (obj i) (vals i))))
	      obj)))

	(define* (subseq sequence start end)
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))
		 (size (- nd start))
		 (obj (make sequence size)))
	    (do ((i start (+ i 1))
		 (j 0 (+ j 1)))
		((= i nd) obj)
	      (set! (obj j) (sequence i)))))
	
	(define (concatenate type . sequences)
	  (let* ((len (apply + (map length sequences)))
		 (new-obj (make-sequence type len))
		 (ctr 0))
	    (for-each
	     (lambda (sequence)
	       (for-each
		(lambda (obj)
		  (set! (new-obj ctr) obj)
		  (set! ctr (+ ctr 1)))
		sequence))
	     sequences)
	    new-obj))

	;; :(concatenate 'list "hiho" '#(1 2)) -> (#\h #\i #\h #\o 1 2)

	(define* (replace seq1 seq2 (start1 0) end1 (start2 0) end2)
	  (let* ((len1 (length seq1))
		 (len2 (length seq2))
		 (nd1 (or (and (number? end1) end1) len1))
		 (nd2 (or (and (number? end2) end2) len2)))
	    (if (and (eq? seq1 seq2)
		     (> start1 start2))
		(let ((offset (- start1 start2)))
		  (do ((i (- nd1 1) (- i 1)))
		      ((< i start1) seq1)
		    (set! (seq1 i) (seq1 (- i offset)))))
		(do ((i start1 (+ i 1))
		     (j start2 (+ j 1)))
		    ((or (= i nd1)
			 (= j nd2))
		     seq1)
		  (set! (seq1 i) (seq2 j))))))
	
	(define* (remove-if predicate sequence from-end (start 0) end count (key identity))
	  (let* ((len (length sequence))
		 (nd (or (and (number? end) end) len))
		 (num (if (number? count) count len))
		 (changed 0))
	    (if (not (positive? num))
		sequence
		(let ((result '()))
		  (if (null from-end)
		      (do ((i 0 (+ i 1)))
			  ((= i len))
			(if (or (< i start)
				(>= i nd)
				(>= changed num)
				(not (predicate (key (sequence i)))))
			    (set! result (cons (sequence i) result))
			    (set! changed (+ changed 1))))
		      (do ((i (- len 1) (- i 1)))
			  ((< i 0))
			(if (or (< i start)
				(>= i nd)
				(>= changed num)
				(not (predicate (key (sequence i)))))
			    (set! result (cons (sequence i) result))
			    (set! changed (+ changed 1)))))		    
		  (let* ((len (length result))
			 (obj (make sequence len))
			 (vals (if (null from-end) (reverse result) result)))
		    (do ((i 0 (+ i 1)))
			((= i len))
		      (set! (obj i) (vals i)))
		    obj)))))
	
	(define* (remove-if-not predicate sequence from-end (start 0) end count (key identity))
	  (remove-if (lambda (obj) (not (predicate obj))) sequence from-end start end count key))
	
	(define* (remove item sequence from-end (test eql) (start 0) end count (key identity))
	  (remove-if (lambda (arg) (test item arg)) sequence from-end start end count key))

	(define-macro* (delete-if predicate sequence from-end (start 0) end count (key identity))
	  `(let ((obj (remove-if ,predicate ,sequence ,from-end ,start ,end ,count ,key)))
	     (if (symbol? ',sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define-macro* (delete-if-not predicate sequence from-end (start 0) end count (key identity))
	  `(let ((obj (remove-if-not ,predicate ,sequence ,from-end ,start ,end ,count ,key)))
	     (if (symbol? ',sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define-macro* (delete item sequence from-end (test eql) (start 0) end count (key identity))
	  `(let ((obj (remove ,item ,sequence ,from-end ,test ,start ,end ,count ,key)))
	     (if (symbol? ',sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define* (remove-duplicates sequence from-end (test eql) (start 0) end (key identity))
	  (let* ((result '())
		 (start-seq (+ start 1))
		 (len (length sequence))
		 (nd (if (number? end) end len)))
	    (do ((i start (+ i 1)))
		((= i nd))
	      (let* ((orig-obj (sequence i))
		     (obj (key orig-obj)))
		(if (null from-end)
		    (begin
		      (if (not (find obj sequence :start start-seq :end nd :test test :key key))
			  (set! result (cons orig-obj result)))
		      (set! start-seq (+ start-seq 1)))
		    (if (not (find obj result :test test :key key))
			(set! result (cons orig-obj result))))))
	    (let* ((res (reverse result))
		   (new-len (+ (length result) start (- len nd)))
		   (new-seq (make sequence new-len)))
	      (let ((n 0))
		(do ((i 0 (+ i 1)))
		    ((= i len) new-seq)
		  (if (or (< i start)
			  (>= i nd))
		      (begin
			(set! (new-seq n) (sequence i))
			(set! n (+ n 1)))
		      (if (not (null? res))
			  (begin
			    (set! (new-seq n) (car res))
			    (set! res (cdr res))
			    (set! n (+ n 1))))))))))
	
	(define-macro* (delete-duplicates sequence from-end (test eql) (start 0) end (key identity))
	  `(let ((obj (remove-duplicates ,sequence ,from-end ,test ,start ,end ,key)))
	     (if (symbol? ,sequence)
		 (set! ,sequence obj))
	     obj))
	
	(define* (merge result-type seq1 seq2 predicate (key identity))
	  (let* ((len1 (length seq1))
		 (len2 (length seq2))
		 (size (+ len1 len2))
		 (obj (make-sequence result-type size))
		 (i 0)
		 (j 0))
	    (do ((n 0 (+ n 1)))
		((or (= i len1)
		     (= j len2))
		 (if (< i len1)
		     (do ((k i (+ k 1)))
			 ((= k len1) obj)
		       (set! (obj n) (seq1 k))
		       (set! n (+ n 1)))
		     (if (< j len2)
			 (do ((k j (+ k 1)))
			     ((= k len2) obj)
			   (set! (obj n) (seq2 k))
			   (set! n (+ n 1)))
			 obj)))
	      (if (null (predicate (key (seq1 i)) (key (seq2 j))))
		  (begin
		    (set! (obj n) (seq2 j))
		    (set! j (+ j 1)))
		  (begin
		    (set! (obj n) (seq1 i))
		    (set! i (+ i 1)))))))
	
	(define* (search seq1 seq2 from-end (test eql) (key identity) (start1 0) (start2 0) end1 end2)
	  (let* ((len1 (length seq1))
		 (len2 (length seq2))
		 (nd1 (or (and (number? end1) end1) len1))
		 (nd2 (or (and (number? end2) end2) len2)))
	    (set! len1 (min len1 (- nd1 start1)))
	    (set! len2 (min len2 (- nd2 start2)))
	    (if (or (= len2 0)
		    (> len1 len2))
		'()
		(call-with-exit
		 (lambda (return)
		   (if (or (not from-end) (null? from-end))
		       (do ((i start2 (+ i 1)))
			   ((> i (- nd2 len1)) '())
			 (do ((j start1 (+ j 1))
			      (k i (+ k 1)))
			     ((or (= j nd1)
				  (not (test (key (seq1 j)) (key (seq2 k)))))
			      (if (= j nd1)
				  (return i)))))
		       (do ((i (- nd2 len1) (- i 1)))
			   ((< i start2) '())
			 (do ((j start1 (+ j 1))
			      (k i (+ k 1)))
			     ((or (= j nd1)
				  (not (test (key (seq1 j)) (key (seq2 k)))))
			      (if (= j nd1)
				  (return i)))))))))))
	
	(define* (mismatch seq1 seq2 from-end (test eql) (key identity) (start1 0) (start2 0) end1 end2)
	  (let* ((nd1 (or (and (number? end1) end1) (length seq1)))
		 (nd2 (or (and (number? end2) end2) (length seq2))))
	    (if (not from-end)
		(do ((i start1 (+ i 1))
		     (j start2 (+ j 1)))
		    ((or (= i nd1)
			 (= j nd2)
			 (not (test (key (seq1 i)) (key (seq2 j)))))
		     (if (and (= i nd1) (= j nd2))
			 '()
			 i)))
		(do ((i (- nd1 1) (- i 1))
		     (j (- nd2 1) (- j 1)))
		    ((or (< i start1)
			 (< j start2)
			 (not (test (key (seq1 i)) (key (seq2 j)))))
		     (if (and (< i start1) (< j start2))
			 '()
			 (+ i 1)))))))
	
	
	;; -------- strings
	
	(define char string-ref)
	(define schar string-ref)
	(define* (cl-make-string size (initial-element #\null)) (make-string size initial-element))
	
	(define (cl-string x)
	  (if (string? x) x
	      (if (char? x)
		  (string x)
		  (if (symbol? x) (symbol->string x)
		      (error "string ~A?" x)))))
	
	(define* (string= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2))))
	    (if (and (not end1) (not end2) (= start1 0) (= start2 0))
		(string=? str1 str2)
		(string=? (subseq str1 start1 nd1)
			  (subseq str2 start2 nd2)))))
	
	(define* (string-equal str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2))))
	    (if (and (not end1) (not end2) (= start1 0) (= start2 0))
		(string-ci=? str1 str2)
		(string-ci=? (subseq str1 start1 nd1)
			     (subseq str2 start2 nd2)))))
	
	(define (string-prefixes-equal str1 str2 start1 nd1 start2 nd2)
	  (do ((i start1 (+ i 1))
	       (j start2 (+ j 1)))
	      ((or (= i nd1)
		   (= j nd2)
		   (not (char=? (str1 i) (str2 j))))
	       i)))
	
	(define (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2)
	  (do ((i start1 (+ i 1))
	       (j start2 (+ j 1)))
	      ((or (= i nd1)
		   (= j nd2)
		   (not (char-ci=? (str1 i) (str2 j))))
	       i)))
	
	(define* (string< str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string<? str1 str2)
			  (string<? (subseq str1 start1 nd1)
				    (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-lessp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci<? str1 str2)
			  (string-ci<? (subseq str1 start1 nd1)
				       (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string<= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string<=? str1 str2)
			  (string<=? (subseq str1 start1 nd1)
				     (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-not-greaterp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci<=? str1 str2)
			  (string-ci<=? (subseq str1 start1 nd1)
					(subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string> str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string>? str1 str2)
			  (string>? (subseq str1 start1 nd1)
				    (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-greaterp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci>? str1 str2)
			  (string-ci>? (subseq str1 start1 nd1)
				       (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string>= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string>=? str1 str2)
			  (string>=? (subseq str1 start1 nd1)
				     (subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))

	(define* (string-not-lessp str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (string-ci>=? str1 str2)
			  (string-ci>=? (subseq str1 start1 nd1)
					(subseq str2 start2 nd2)))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define* (string/= str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (not (string=? str1 str2))
			  (not (string=? (subseq str1 start1 nd1)
					 (subseq str2 start2 nd2))))))
	    (and val (string-prefixes-equal str1 str2 start1 nd1 start2 nd2))))
	
	(define* (string-not-equal str-1 str-2 (start1 0) end1 (start2 0) end2)
	  (let* ((str1 (cl-string str-1))
		 (str2 (cl-string str-2))
		 (nd1 (if (number? end1) end1 (length str1)))
		 (nd2 (if (number? end2) end2 (length str2)))
		 (val (if (and (not end1) (not end2) (= start1 0) (= start2 0))
			  (not (string-ci=? str1 str2))
			  (not (string-ci=? (subseq str1 start1 nd1)
					    (subseq str2 start2 nd2))))))
	    (and val (string-prefixes-equal-ci str1 str2 start1 nd1 start2 nd2))))

	(define (string-left-trim bag str-1)
	  (let ((str (cl-string str-1)))
	    (if (string? bag) (set! bag (string->list bag)))
	    (let ((len (length str)))
	      (do ((i 0 (+ i 1)))
		  ((or (= i len)
		       (not (member (str i) bag)))
		   (if (= i 0)
		       str
		       (subseq str i)))))))
		 
	(define (string-right-trim bag str-1)
	  (let ((str (cl-string str-1)))
	    (if (string? bag) (set! bag (string->list bag)))
	    (let ((len (length str)))
	      (do ((i (- len 1) (- i 1)))
		  ((or (< i 0)
		       (not (member (str i) bag)))
		   (if (= i (- len 1))
		       str
		       (subseq str 0 (+ i 1))))))))
		 
	(define (string-trim bag str)
	  (string-right-trim bag (string-left-trim bag str)))

	(define* (nstring-upcase str (start 0) end)
	  (let ((nd (if (number? end) end (length str))))
	    (do ((i start (+ i 1)))
		((= i nd) str)
	      (set! (str i) (char-upcase (str i))))))

	(define* (string-upcase str-1 (start 0) end)
	  (let ((str (cl-string str-1)))
	    (nstring-upcase (copy str) start end)))

	(define* (nstring-downcase str (start 0) end)
	  (let ((nd (if (number? end) end (length str))))
	    (do ((i start (+ i 1)))
		((= i nd) str)
	      (set! (str i) (char-downcase (str i))))))

	(define* (string-downcase str-1 (start 0) end)
	  (let ((str (cl-string str-1)))
	    (nstring-downcase (copy str) start end)))

	(define* (nstring-capitalize str-1 (start 0) end)
	  (define (alpha? c) 
	    (or (char-alphabetic? c) 
		(char-numeric? c)))
	  (let ((str (cl-string str-1)))
	    (let ((nd (if (number? end) end (length str))))
	      (do ((i start (+ i 1)))
		  ((= i nd) str)
		(if (alpha? (str i))
		    (if (or (= i 0)
			    (not (alpha? (str (- i 1)))))
			(set! (str i) (char-upcase (str i)))
			(set! (str i) (char-downcase (str i)))))))))

	(define* (string-capitalize str-1 (start 0) end)
	  (let ((str (cl-string str-1)))
	    (nstring-capitalize (copy str) start end)))


	;; -------- vectors

	;; vector is ok

	(define svref vector-ref)
	(define aref vector-ref)
        (define array-dimensions vector-dimensions) 
	(define array-total-size vector-length)
        (define (array-dimension array num) (list-ref (vector-dimensions array) num))

	(define-constant array-dimension-limit 16777215)
	(define-constant array-rank-limit 4096)
	(define-constant array-total-size-limit 16777215)

	(define* (make-array dimensions element-type initial-element initial-contents adjustable fill-pointer displaced-to displaced-index-offset)
	  (if (eq? element-type 'character)
	      (or (and initial-contents
		       (string-copy initial-contents))
		  (cl-make-string dimensions initial-element))
	      (make-vector (or dimensions 1) initial-element)))

	(define (array-in-bounds-p array . subscripts)
	  (define (in-bounds dims subs)
	    (or (null? subs)
		(null? dims)
		(and (< (car subs) (car dims))
		     (in-bounds (cdr dims) (cdr subs)))))
	  (in-bounds (vector-dimensions array) subscripts))

	(define (row-major-index array . subscripts) 
	  (apply + (maplist (lambda (x y)
			      (* (car x) (apply * (cdr y))))
			    subscripts
			    (vector-dimensions array))))


	;; -------- defstruct

	(defmacro defstruct (struct-name . fields)
	  (let* ((name (if (list? struct-name) (car struct-name) struct-name))
		 (sname (if (string? name) name (symbol->string name)))
		 
		 (fsname (if (list? struct-name)
			     (let ((cname (assoc :conc-name (cdr struct-name))))
			       (if cname 
				   (symbol->string (cadr cname))
				   sname))
			     sname))
		 
		 (make-name (if (list? struct-name)
				(let ((cname (assoc :constructor (cdr struct-name))))
				  (if cname 
				      (cadr cname)
				      (string->symbol (string-append "make-" sname))))
				(string->symbol (string-append "make-" sname))))
		 
		 (copy-name (if (list? struct-name)
				(let ((cname (assoc :copier (cdr struct-name))))
				  (if cname 
				      (cadr cname)
				      (string->symbol (string-append "copy-" sname))))
				(string->symbol (string-append "copy-" sname))))
		 
		 (field-names (map (lambda (n)
				     (symbol->string (if (list? n) (car n) n)))
				   fields))
		 
		 (field-types (map (lambda (field)
				     (if (list? field)
					 (apply (lambda* (val type read-only) type) (cdr field))
					 #f))
				   fields))
		 
		 (field-read-onlys (map (lambda (field)
					  (if (list? field)
					      (apply (lambda* (val type read-only) read-only) (cdr field))
					      #f))
					fields)))
	    `(begin
	       
	       (define ,(string->symbol (string-append sname "?"))
		 (lambda (obj)
		   (and (vector? obj)
			(eq? (obj 0) ',(string->symbol sname)))))
	       
	       (define* (,make-name
			 ,@(map (lambda (n)
				  (if (and (list? n)
					   (>= (length n) 2))
				      (list (car n) (cadr n))
				      (list n #f)))
				fields))
		 (vector ',(string->symbol sname) ,@(map string->symbol field-names)))
	       
	       (define ,copy-name copy)
	       
	       ,@(map (let ((ctr 1))
			(lambda (n type read-only)
			  (let ((val (if read-only
					 `(define ,(string->symbol (string-append fsname "-" n))
					    (lambda (arg) (arg ,ctr)))
					 `(define ,(string->symbol (string-append fsname "-" n))
					    (make-procedure-with-setter 
					     (lambda (arg) (arg ,ctr)) 
					     (lambda (arg val) (set! (arg ,ctr) val)))))))
			    (set! ctr (+ 1 ctr))
			    val)))
		      field-names field-types field-read-onlys))))
	
	;; not yet implemented: :print-function :include :named :type :initial-offset
	;;   also the explicit constructor business

	(define-macro (enum . args) ; (enum zero one two)
	  `(begin
	     ,@(let ((names '()))
		 (do ((arg args (cdr arg))
		      (i 0 (+ i 1)))
		     ((null? arg) names)
		   (set! names (cons
				`(define ,(car arg) ,i)
				names))))))

	(define-macro (let*-values vals . body)
	  (let ((args '())
		(exprs '()))
	    (for-each
	     (lambda (arg+expr)
	       (set! args (cons (car arg+expr) args))
	       (set! exprs (cons (cadr arg+expr) exprs)))
	     vals)
	    (let ((form `((lambda ,(car args) ,@body) ,(car exprs))))
	      (if (not (null? (cdr args)))
		  (for-each
		   (lambda (arg expr)
		     (set! form `((lambda ,arg ,form) ,expr)))
		   (cdr args)
		   (cdr exprs)))
	      form)))

	
	;;; ----------------
	;;; some of these tests are taken (with modifications) from sacla which has 
	;;;  the following copyright notice:
	;;;
	;; Copyright (C) 2002-2004, Yuji Minejima <ggb01164@nifty.ne.jp>
	;; ALL RIGHTS RESERVED.
	;;
	;; Redistribution and use in source and binary forms, with or without
	;; modification, are permitted provided that the following conditions
	;; are met:
	;; 
	;;  * Redistributions of source code must retain the above copyright
	;;    notice, this list of conditions and the following disclaimer.
	;;  * Redistributions in binary form must reproduce the above copyright
	;;    notice, this list of conditions and the following disclaimer in
	;;    the documentation and/or other materials provided with the
	;;    distribution.
	;; 
	;; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
	;; "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
	;; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
	;; A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
	;; OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
	;; SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
	;; LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
	;; DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
	;; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	;; (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
	;; OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

	(test-t (equal 'a 'a))
	(test-t (not (equal 'a 'b)))
	(test-t (equal 'abc 'abc))
	(test-t (equal 1 1))
	(test-t (equal 2 2))
	(test-t (equal 0.1 0.1))
	(test-t (equal 1/3 1/3))
	(test-t (not (equal 0 1)))
	(test-t (not (equal 1 1.0)))
	(test-t (not (equal 1/3 1/4)))
	(test-t (equal #\a #\a))
	(test-t (equal #\b #\b))
	(test-t (not (equal #\b #\B)))
	(test-t (not (equal #\C #\c)))
	(test-t (equal '(0) '(0)))
	(test-t (equal '(0 #\a) '(0 #\a)))
	(test-t (equal '(0 #\a x) '(0 #\a x)))
	(test-t (equal '(0 #\a x (0)) '(0 #\a x (0))))

	(test-t (eql (identity 101) 101))
	(test-t (eq (identity 'x) 'x))

	;; chars
	(test-t (char= #\d #\d))
	(test-t (not (char= #\A #\a)))
	(test-t (not (char= #\d #\x)))
	(test-t (not (char= #\d #\D)))
	(test-t (not (char/= #\d #\d)))
	(test-t (char/= #\d #\x))
	(test-t (char/= #\d #\D))
	(test-t (char= #\d #\d #\d #\d))
	(test-t (not (char/= #\d #\d #\d #\d)))
	(test-t (not (char= #\d #\d #\x #\d)))
	(test-t (not (char/= #\d #\d #\x #\d)))
	(test-t (not (char= #\d #\y #\x #\c)))
	(test-t (char/= #\d #\y #\x #\c))
	(test-t (not (char= #\d #\c #\d)))
	(test-t (not (char/= #\d #\c #\d)))
	(test-t (char< #\d #\x))
	(test-t (char<= #\d #\x))
	(test-t (not (char< #\d #\d)))
	(test-t (char<= #\d #\d))
	(test-t (char< #\a #\e #\y #\z))
	(test-t (char<= #\a #\e #\y #\z))
	(test-t (not (char< #\a #\e #\e #\y)))
	(test-t (char<= #\a #\e #\e #\y))
	(test-t (char> #\e #\d))
	(test-t (char>= #\e #\d))
	(test-t (char> #\d #\c #\b #\a))
	(test-t (char>= #\d #\c #\b #\a))
	(test-t (not (char> #\d #\d #\c #\a)))
	(test-t (char>= #\d #\d #\c #\a))
	(test-t (not (char> #\e #\d #\b #\c #\a)))
	(test-t (not (char>= #\e #\d #\b #\c #\a)))
	(test-t (char-equal #\A #\a))
	(test-t (char= #\a))
	(test-t (char= #\a #\a))
	(test-t (char= #\a #\a #\a))
	(test-t (char= #\a #\a #\a #\a))
	(test-t (char= #\a #\a #\a #\a #\a))
	(test-t (char= #\a #\a #\a #\a #\a #\a))
	(test-t (let ((c #\z))  (and (eq c c)       (char= c c))))
	(test-t (not (char= #\Z #\z)))
	(test-t (not (char= #\z #\z #\z #\a)))
	(test-t (not (char= #\a #\z #\z #\z #\a)))
	(test-t (not (char= #\z #\i #\z #\z)))
	(test-t (not (char= #\z #\z #\Z #\z)))
	(test-t (char/= #\a))
	(test-t (char/= #\a #\b))
	(test-t (char/= #\a #\b #\c))
	(test-t (char/= #\a #\b #\c #\d))
	(test-t (char/= #\a #\b #\c #\d #\e))
	(test-t (char/= #\a #\b #\c #\d #\e #\f))
	(test-t (let ((c #\z))  (and (eq c c)       (not (char/= c c)))))
	(test-t (char/= #\Z #\z))
	(test-t (not (char/= #\z #\z #\z #\a)))
	(test-t (not (char= #\a #\z #\z #\z #\a)))
	(test-t (not (char= #\z #\i #\z #\z)))
	(test-t (not (char= #\z #\z #\Z #\z)))
	(test-t (not (char/= #\a #\a #\b #\c)))
	(test-t (not (char/= #\a #\b #\a #\c)))
	(test-t (not (char/= #\a #\b #\c #\a)))
	(test-t (char< #\a))
	(test-t (char< #\a #\z))
	(test-t (char< #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m       #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (not (char< #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	    #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char< #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M       #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char< #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	    #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char< #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (not (char< #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char< #\9 #\A)    (char< #\Z #\0)))
	(test-t (or (char< #\9 #\a)    (char< #\z #\0)))
	(test-t (not (char< #\a #\a #\b #\c)))
	(test-t (not (char< #\a #\b #\a #\c)))
	(test-t (not (char< #\a #\b #\c #\a)))
	(test-t (not (char< #\9 #\0)))
	(test-t (char> #\a))
	(test-t (not (char> #\a #\z)))
	(test-t (char> #\z #\a))
	(test-t (not (char> #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m	    #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char> #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n       #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (not (char> #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M	    #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char> #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N      #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (not (char> #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char> #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (or (char> #\A #\9)    (char> #\0 #\Z)))
	(test-t (or (char> #\a #\9)    (char> #\0 #\z)))
	(test-t (not (char> #\a #\a #\b #\c)))
	(test-t (not (char> #\a #\b #\a #\c)))
	(test-t (not (char> #\a #\b #\c #\a)))
	(test-t (char> #\9 #\0))
	(test-t (char<= #\a))
	(test-t (char<= #\a #\z))
	(test-t (char<= #\a #\a))
	(test-t (char<= #\Z #\Z))
	(test-t (char<= #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m	#\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (char<= #\a #\a #\b #\b #\c #\c #\d #\d #\e #\e #\f #\f #\g #\g #\h #\h 
			#\i #\i #\j #\j #\k #\k #\l #\l #\m #\m	#\n #\n #\o #\o #\p #\p #\q #\q #\r #\r #\s #\s	
			#\t #\t #\u #\u #\v #\v #\w #\w #\x #\x #\y #\y #\z #\z))
	(test-t (not (char<= #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n     #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char<= #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M	#\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (char<= #\A #\B #\B #\C #\D #\E #\E #\F #\G #\H #\I #\I #\J #\K #\L #\M	#\N #\N #\O #\P #\Q #\R #\S #\T #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char<= #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N     #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char<= #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (char<= #\0 #\1 #\2 #\2 #\3 #\3 #\3 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\9))
	(test-t (not (char<= #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char<= #\9 #\A)    (char<= #\Z #\0)))
	(test-t (or (char<= #\9 #\a)    (char<= #\z #\0)))
	(test-t (char<= #\a #\a #\b #\c))
	(test-t (not (char<= #\a #\b #\a #\c)))
	(test-t (not (char<= #\a #\b #\c #\a)))
	(test-t (not (char<= #\9 #\0)))
	(test-t (char>= #\a))
	(test-t (not (char>= #\a #\z)))
	(test-t (char>= #\z #\a))
	(test-t (char>= #\a #\a))
	(test-t (char>= #\Z #\Z))
	(test-t (not (char>= #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m     #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char>= #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	#\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (char>= #\z #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n #\n	#\m #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a #\a))
	(test-t (not (char>= #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M     #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char>= #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (char>= #\Z #\Y #\X #\W #\V #\U #\U #\T #\T #\S #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\H #\G #\G #\F #\F #\E #\D #\C #\B #\A))
	(test-t (not (char>= #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char>= #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (char>= #\9 #\8 #\8 #\8 #\7 #\6 #\5 #\4 #\3 #\3 #\3 #\2 #\1 #\0))
	(test-t (or (char>= #\A #\9)    (char>= #\0 #\Z)))
	(test-t (or (char>= #\a #\9)    (char>= #\0 #\z)))
	(test-t (char>= #\c #\b #\a #\a))
	(test-t (not (char>= #\c #\b #\a #\a #\b #\c)))
	(test-t (not (char>= #\c #\b #\a #\c)))
	(test-t (not (char>= #\c #\b #\c #\a)))
	(test-t (char>= #\9 #\0))
	(test-t (not (char>= #\0 #\9)))
	(test-t (char-equal #\a))
	(test-t (char-equal #\a #\a))
	(test-t (char-equal #\a #\a #\a))
	(test-t (char-equal #\a #\a #\a #\a))
	(test-t (char-equal #\a #\a #\a #\a #\a))
	(test-t (char-equal #\a #\a #\a #\a #\a #\a))
	(test-t (char-equal #\a #\A))
	(test-t (char-equal #\a #\A #\a))
	(test-t (char-equal #\a #\a #\A #\a))
	(test-t (char-equal #\a #\a #\a #\A #\a))
	(test-t (char-equal #\a #\a #\a #\a #\A #\a))
	(test-t (let ((c #\z))  (and (eq c c)       (char-equal c c))))
	(test-t (char-equal #\Z #\z))
	(test-t (not (char-equal #\z #\z #\z #\a)))
	(test-t (not (char-equal #\a #\z #\z #\z #\a)))
	(test-t (not (char-equal #\z #\i #\z #\z)))
	(test-t (char-equal #\z #\z #\Z #\z))
	(test-t (char-equal #\a #\A #\a #\A #\a #\A #\a #\A #\a #\A))
	(test-t (char-not-equal #\a))
	(test-t (char-not-equal #\a #\b))
	(test-t (char-not-equal #\a #\b #\c))
	(test-t (char-not-equal #\a #\b #\c #\d))
	(test-t (char-not-equal #\a #\b #\c #\d #\e))
	(test-t (char-not-equal #\a #\b #\c #\d #\e #\f))
	(test-t (let ((c #\z))  (and (eq c c)       (not (char-not-equal c c)))))
	(test-t (not (char-not-equal #\Z #\z)))
	(test-t (not (char-not-equal #\z #\z #\z #\a)))
	(test-t (not (char= #\a #\z #\z #\z #\a)))
	(test-t (not (char= #\z #\i #\z #\z)))
	(test-t (not (char= #\z #\z #\Z #\z)))
	(test-t (not (char-not-equal #\a #\a #\b #\c)))
	(test-t (not (char-not-equal #\a #\b #\a #\c)))
	(test-t (not (char-not-equal #\a #\b #\c #\a)))
	(test-t (not (char-not-equal #\a #\A #\a #\A)))
	(test-t (char-lessp #\a))
	(test-t (char-lessp #\a #\z))
	(test-t (char-lessp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m       #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (not (char-lessp #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	    #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char-lessp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M       #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char-lessp #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	    #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char-lessp #\a #\B #\c #\D #\e #\F #\g #\H #\i #\J #\k #\L #\m       #\N #\o #\P #\q #\R #\s #\T #\u #\V #\w #\X #\y #\Z))
	(test-t (char-lessp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (not (char-lessp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char-lessp #\9 #\A)    (char-lessp #\Z #\0)))
	(test-t (or (char-lessp #\9 #\a)    (char-lessp #\z #\0)))
	(test-t (not (char-lessp #\a #\a #\b #\c)))
	(test-t (not (char-lessp #\a #\b #\a #\c)))
	(test-t (not (char-lessp #\a #\b #\c #\a)))
	(test-t (not (char-lessp #\9 #\0)))
	(test-t (and (char-lessp #\a #\Z)     (char-lessp #\A #\z)))
	(test-t (char-greaterp #\a))
	(test-t (not (char-greaterp #\a #\z)))
	(test-t (char-greaterp #\z #\a))
	(test-t (not (char-greaterp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m	    #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char-greaterp #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n       #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (not (char-greaterp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M	    #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char-greaterp #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N   #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (char-greaterp #\z #\Y #\x #\W #\v #\U #\t #\S #\r #\Q #\p #\O #\n   #\M #\l #\K #\j #\I #\h #\G #\f #\E #\d #\C #\b #\A))
	(test-t (not (char-greaterp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char-greaterp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (or (char-greaterp #\A #\9)    (char-greaterp #\0 #\Z)))
	(test-t (or (char-greaterp #\a #\9)    (char-greaterp #\0 #\z)))
	(test-t (not (char-greaterp #\a #\a #\b #\c)))
	(test-t (not (char-greaterp #\a #\b #\a #\c)))
	(test-t (not (char-greaterp #\a #\b #\c #\a)))
	(test-t (char-greaterp #\9 #\0))
	(test-t (and (char-greaterp #\z #\A)     (char-greaterp #\Z #\a)))
	(test-t (char-not-greaterp #\a))
	(test-t (char-not-greaterp #\a #\z))
	(test-t (char-not-greaterp #\a #\a))
	(test-t (char-not-greaterp #\Z #\Z))
	(test-t (char-not-greaterp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z))
	(test-t (char-not-greaterp #\a #\a #\b #\b #\c #\c #\d #\d #\e #\e #\f #\f #\g #\g #\h #\h #\i #\i 
				   #\j #\j #\k #\k #\l #\l #\m #\m #\n #\n #\o #\o #\p #\p #\q #\q #\r #\r #\s #\s #\t 
				   #\t #\u #\u #\v #\v #\w #\w #\x #\x #\y #\y #\z #\z))
	(test-t (char-not-greaterp #\a #\A #\b #\B #\c #\C #\d #\D #\e #\E #\f #\F #\g #\G #\h #\H #\i #\I #\j #\J 
				   #\k #\K #\l #\L #\m #\M #\n #\N #\o #\O #\p #\P #\q #\Q #\r #\R #\s #\S #\t #\T 
				   #\u #\U #\v #\V #\w #\W #\x #\X #\y #\Y #\z #\z))
	(test-t (not (char-not-greaterp      #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n      #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a)))
	(test-t (char-not-greaterp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (char-not-greaterp #\A #\B #\B #\C #\D #\E #\E #\F #\G #\H #\I #\I #\J #\K #\L #\M #\N #\N #\O #\P #\Q #\R #\S #\T #\T #\U #\V #\W #\X #\Y #\Z))
	(test-t (not (char-not-greaterp      #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N      #\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A)))
	(test-t (char-not-greaterp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9))
	(test-t (char-not-greaterp #\0 #\1 #\2 #\2 #\3 #\3 #\3 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\9))
	(test-t (not (char-not-greaterp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0)))
	(test-t (or (char-not-greaterp #\9 #\A)    (char-not-greaterp #\Z #\0)))
	(test-t (or (char-not-greaterp #\9 #\a)    (char-not-greaterp #\z #\0)))
	(test-t (char-not-greaterp #\a #\a #\b #\c))
	(test-t (not (char-not-greaterp #\a #\b #\a #\c)))
	(test-t (not (char-not-greaterp #\a #\b #\c #\a)))
	(test-t (not (char-not-greaterp #\9 #\0)))
	(test-t (and (char-not-greaterp #\A #\z)     (char-not-greaterp #\a #\Z)))
	(test-t (char-not-lessp #\a))
	(test-t (not (char-not-lessp #\a #\z)))
	(test-t (char-not-lessp #\z #\a))
	(test-t (char-not-lessp #\a #\a))
	(test-t (char-not-lessp #\Z #\Z))
	(test-t (not (char-not-lessp #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m  #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (char-not-lessp #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n	#\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a))
	(test-t (char-not-lessp #\z #\z #\y #\x #\w #\v #\u #\t #\s #\r #\q #\p #\o #\n #\n	#\m #\m #\l #\k #\j #\i #\h #\g #\f #\e #\d #\c #\b #\a #\a))
	(test-t (not (char-not-lessp #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\m     #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (char-not-lessp #\Z #\Y #\X #\W #\V #\U #\T #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\G #\F #\E #\D #\C #\B #\A))
	(test-t (char-not-lessp #\Z #\Y #\X #\W #\V #\U #\U #\T #\T #\S #\S #\R #\Q #\P #\O #\N	#\M #\L #\K #\J #\I #\H #\H #\G #\G #\F #\F #\E #\D #\C #\B #\A))
	(test-t (char-not-lessp #\z #\Z #\y #\x #\w #\V #\v #\u #\t #\s #\r #\q #\p #\o #\n #\n	#\m #\M #\l #\k #\K #\j #\i #\h #\g #\f #\e #\d #\c #\b #\A #\a))
	(test-t (not (char-not-lessp #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
	(test-t (char-not-lessp #\9 #\8 #\7 #\6 #\5 #\4 #\3 #\2 #\1 #\0))
	(test-t (char-not-lessp #\9 #\8 #\8 #\8 #\7 #\6 #\5 #\4 #\3 #\3 #\3 #\2 #\1 #\0))
	(test-t (or (char-not-lessp #\A #\9)    (char-not-lessp #\0 #\Z)))
	(test-t (or (char-not-lessp #\a #\9)    (char-not-lessp #\0 #\z)))
	(test-t (char-not-lessp #\c #\b #\a #\a))
	(test-t (not (char-not-lessp #\c #\b #\a #\a #\b #\c)))
	(test-t (not (char-not-lessp #\c #\b #\a #\c)))
	(test-t (not (char-not-lessp #\c #\b #\c #\a)))
	(test-t (char-not-lessp #\9 #\0))
	(test-t (not (char-not-lessp #\0 #\9)))
	(test-t (and (char-not-lessp #\z #\A)     (char-not-lessp #\Z #\a)))
	(test-t (char= (character #\a) #\a))
	(test-t (char= (character #\b) #\b))
;	(test-t (char= (character #\Space) #\Space))
	(test-t (char= (character "a") #\a))
	(test-t (char= (character "X") #\X))
	(test-t (char= (character "z") #\z))
	(test-t (char= (character 'a) #\a))
;	(test-t (char= (character '\a) #\a))
	(test-t (alpha-char-p #\a))
	(test-t (every alpha-char-p '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (every alpha-char-p '(#\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (notany alpha-char-p '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
;	(test-t (not (alpha-char-p #\Newline)))
	(test-t (alphanumericp #\Z))
	(test-t (alphanumericp #\9))
	(test-t (every alphanumericp '(#\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m  #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z)))
	(test-t (every alphanumericp '(#\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M  #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z)))
	(test-t (every alphanumericp '(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
;	(test-t (not (alphanumericp #\Newline)))
	(test-t (not (alphanumericp #\#)))
	(test-t (char= (digit-char 0) #\0))
;	(test-t (char= (digit-char 10 11) #\A))
	(test-t (null (digit-char 10 10)))
	(test-t (char= (digit-char 7) #\7))
	(test-t (null (digit-char 12)))
;	(test-t (char= (digit-char 12 16) #\C))
	(test-t (null (digit-char 6 2))) 
	(test-t (char= (digit-char 1 2) #\1))
;	(test-t (char= (digit-char 35 36) #\Z))
	(test-t (= (digit-char-p #\0) 0))
	(test-t (= (digit-char-p #\5) 5))
	(test-t (not (digit-char-p #\5 2)))
	(test-t (not (digit-char-p #\A)))
	(test-t (not (digit-char-p #\a)))
;	(test-t (= (digit-char-p #\A 11) 10))
	(test-t (= (digit-char-p #\a 11) 10))
;	(test-t (standard-char-p #\a))
;	(test-t (standard-char-p #\z))
;	(test-t (standard-char-p #\Newline))
;	(test-t (every standard-char-p " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_'abcdefghijklmnopqrstuvwxyz{|}~"))
	(test-t (char= (char-upcase #\a) #\A))
	(test-t (char= (char-upcase #\A) #\A))
	(test-t (char= (char-upcase #\-) #\-))
	(test-t (char= (char-downcase #\A) #\a))
	(test-t (char= (char-downcase #\a) #\a))
	(test-t (char= (char-downcase #\-) #\-))
	(test-t (not (upper-case-p #\a)))
	(test-t (upper-case-p #\A))
	(test-t (not (upper-case-p #\-)))
	(test-t (not (lower-case-p #\A)))
	(test-t (lower-case-p #\a))
	(test-t (not (lower-case-p #\-)))
;	(test-t (char= #\ (name-char (char-name #\ ))))
;	(test-t (char= #\Space (name-char (char-name #\Space))))
;	(test-t (char= #\Newline (name-char (char-name #\Newline))))


	(test-t (simple-string-p ""))
	(test-t (simple-string-p "abc"))
	(test-t (not (simple-string-p 'not-a-string)))
	(test-t (char= (char "abc" 0) #\a))
	(test-t (char= (char "abc" 1) #\b))
	(test-t (char= (char "abc" 2) #\c))
	(test-t (char= (schar "abc" 0) #\a))
	(test-t (char= (schar "abc" 1) #\b))
	(test-t (char= (schar "abc" 2) #\c))
	(test-t (string= (cl-string "") ""))
	(test-t (string= (cl-string "abc") "abc"))
	(test-t (string= (cl-string "a") "a"))
	(test-t (string= (cl-string 'abc) "abc"))
	(test-t (string= (cl-string 'a) "a"))
	(test-t (string= (cl-string #\a) "a"))
	(test-t (string= (string-upcase "abcde") "ABCDE"))
	(test-t (string= (string-upcase "Dr. Livingston, I presume?")	 "DR. LIVINGSTON, I PRESUME?"))
	(test-t (string= (string-upcase "Dr. Livingston, I presume?" :start 6 :end 10)	 "Dr. LiVINGston, I presume?"))
	(test-t (string= (string-upcase 'Kludgy-HASH-Search) "KLUDGY-HASH-SEARCH"))
	(test-t (string= (string-upcase "abcde" :start 2 :end nil) "abCDE"))
	(test-t (string= (string-downcase "Dr. Livingston, I presume?")	 "dr. livingston, i presume?"))
	(test-t (string= (string-downcase 'Kludgy-HASH-Search) "kludgy-hash-search"))
	(test-t (string= (string-downcase "A FOOL" :start 2 :end nil) "A fool"))
	(test-t (string= (string-capitalize "elm 13c arthur;fig don't")	 "Elm 13c Arthur;Fig Don'T"))
	(test-t (string= (string-capitalize " hello ") " Hello "))
	(test-t (string= (string-capitalize  "occlUDeD cASEmenTs FOreSTAll iNADVertent DEFenestraTION") "Occluded Casements Forestall Inadvertent Defenestration"))
	(test-t (string= (string-capitalize 'kludgy-hash-search) "Kludgy-Hash-Search"))
	(test-t (string= (string-capitalize "DON'T!") "Don'T!"))
	(test-t (string= (string-capitalize "pipe 13a, foo16c") "Pipe 13a, Foo16c"))
	(test-t (string= (string-capitalize "a fool" :start 2 :end nil) "a Fool"))
	(test-t (let ((str (copy-seq "0123ABCD890a")))  (and (string= (nstring-downcase str :start 5 :end 7) "0123AbcD890a")       (string= str "0123AbcD890a"))))
	(test-t (let* ((str0 (copy-seq "abcde"))  (str  (nstring-upcase str0)))  (and (eq str0 str)       (string= str "ABCDE"))))
	(test-t (let* ((str0 (copy-seq "Dr. Livingston, I presume?")) (str  (nstring-upcase str0))) (and (eq str0 str) (string= str "DR. LIVINGSTON, I PRESUME?"))))
	(test-t (let* ((str0 (copy-seq "Dr. Livingston, I presume?")) (str  (nstring-upcase str0 :start 6 :end 10))) (and (eq str0 str) (string= str "Dr. LiVINGston, I presume?"))))
	(test-t (let* ((str0 (copy-seq "abcde")) (str (nstring-upcase str0 :start 2 :end nil)))  (string= str "abCDE")))
	(test-t (let* ((str0 (copy-seq "Dr. Livingston, I presume?")) (str  (nstring-downcase str0))) (and (eq str0 str) (string= str "dr. livingston, i presume?"))))
	(test-t (let* ((str0 (copy-seq "ABCDE")) (str (nstring-downcase str0 :start 2 :end nil)))  (string= str "ABcde")))
	(test-t (let* ((str0 (copy-seq "elm 13c arthur;fig don't")) (str  (nstring-capitalize str0))) (and (eq str0 str) (string= str "Elm 13c Arthur;Fig Don'T"))))
	(test-t (let* ((str0 (copy-seq " hello ")) (str  (nstring-capitalize str0)))  (and (eq str0 str) (string= str " Hello "))))
	(test-t (let* ((str0 (copy-seq "occlUDeD cASEmenTs FOreSTAll iNADVertent DEFenestraTION")) (str (nstring-capitalize str0))) (and (eq str0 str) (string= str		"Occluded Casements Forestall Inadvertent Defenestration"))))
	(test-t (let* ((str0 (copy-seq "DON'T!"))  (str  (nstring-capitalize str0))) (and (eq str0 str)  (string= str "Don'T!"))))
	(test-t (let* ((str0 (copy-seq "pipe 13a, foo16c"))       (str  (nstring-capitalize str0)))  (and (eq str0 str)       (string= str "Pipe 13a, Foo16c"))))
	(test-t (let* ((str0 (copy-seq "a fool"))       (str (nstring-capitalize str0 :start 2 :end nil)))  (string= str "a Fool")))
	(test-t (string= (string-trim "abc" "abcaakaaakabcaaa") "kaaak"))
	(test-t (string= (string-trim '(#\space) " garbanzo beans        ") "garbanzo beans"))
	(test-t (string= (string-trim " (*)" " ( *three (silly) words* ) ")	 "three (silly) words"))
	(test-t (string= (string-left-trim "abc" "labcabcabc") "labcabcabc"))
	(test-t (string= (string-left-trim " (*)" " ( *three (silly) words* ) ")	 "three (silly) words* ) "))
	(test-t (string= (string-right-trim " (*)" " ( *three (silly) words* ) ") 	 " ( *three (silly) words"))
	(test-t (string= (string-trim "ABC" "abc") "abc"))
	(test-t (string= (string-trim "AABBCC" "abc") "abc"))
	(test-t (string= (string-trim "" "abc") "abc"))
	(test-t (string= (string-trim "ABC" "") ""))
	(test-t (string= (string-trim "cba" "abc") ""))
	(test-t (string= (string-trim "cba" "abccba") ""))
	(test-t (string= (string-trim "ccbbba" "abccba") ""))
	(test-t (string= (string-trim "cba" "abcxabc") "x"))
	(test-t (string= (string-trim "xyz" "xxyabcxyyz") "abc"))
	(test-t (string= (string-trim "a" #\a) ""))
	(test-t (string= (string-left-trim "ABC" "abc") "abc"))
	(test-t (string= (string-left-trim "" "abc") "abc"))
	(test-t (string= (string-left-trim "ABC" "") ""))
	(test-t (string= (string-left-trim "cba" "abc") ""))
	(test-t (string= (string-left-trim "cba" "abccba") ""))
	(test-t (string= (string-left-trim "cba" "abcxabc") "xabc"))
	(test-t (string= (string-left-trim "xyz" "xxyabcxyz") "abcxyz"))
	(test-t (string= (string-left-trim "a" #\a) ""))
	(test-t (string= (string-right-trim "ABC" "abc") "abc"))
	(test-t (string= (string-right-trim "" "abc") "abc"))
	(test-t (string= (string-right-trim "ABC" "") ""))
	(test-t (string= (string-right-trim "cba" "abc") ""))
	(test-t (string= (string-right-trim "cba" "abccba") ""))
	(test-t (string= (string-right-trim "cba" "abcxabc") "abcx"))
	(test-t (string= (string-right-trim "xyz" "xxyabcxyz") "xxyabc"))
	(test-t (string= (string-right-trim "a" #\a) ""))
	(test-t (string= (cl-string "already a string") "already a string"))
	(test-t (string=  (cl-string #\c) "c"))
	(test-t (string= "foo" "foo"))
	(test-t (not (string= "foo" "Foo")))
	(test-t (not (string= "foo" "bar")))
	(test-t (string= "together" "frog" :start1 1 :end1 3 :start2 2))
	(test-t (string-equal "foo" "Foo"))
	(test-t (string= "abcd" "01234abcd9012" :start2 5 :end2 9))
	(test-t (eql (string< "aaaa" "aaab") 3))
	(test-t (eql (string>= "aaaaa" "aaaa") 4))
	(test-t (eql (string-not-greaterp "Abcde" "abcdE") 5))
	(test-t (eql (string-lessp "012AAAA789" "01aaab6"   :start1 3 :end1 7   :start2 2 :end2 6) 6))
	(test-t (not (string-not-equal "AAAA" "aaaA")))
	(test-t (string= "" ""))
	(test-t (not (string= "abc" "")))
	(test-t (not (string= "" "abc")))
	(test-t (not (string= "A" "a")))
	(test-t (string= "abc" "xyz" :start1 3 :start2 3))
	(test-t (string= "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0))
	(test-t (string= "axyza" "xyz" :start1 1 :end1 4))
	(test-t (string= "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil))
	(test-t (string= "abxyz" "xyabz" :end1 2 :start2 2 :end2 4))
	(test-t (not (string= "love" "hate")))
	(test-t (string= 'love 'love))
	(test-t (not (string= 'love "hate")))
	(test-t (string= #\a #\a))
	(test-t (not (string/= "" "")))
	(test-t (eql (string/= "abc" "") 0))
	(test-t (eql (string/= "" "abc") 0))
	(test-t (eql (string/= "A" "a") 0))
	(test-t (not (string/= "abc" "xyz" :start1 3 :start2 3)))
	(test-t (not (string/= "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0)))
	(test-t (not (string/= "axyza" "xyz" :start1 1 :end1 4)))
	(test-t (not (string/= "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil)))
	(test-t (not (string/= "abxyz" "xyabz" :end1 2 :start2 2 :end2 4)))
	(test-t (eql (string/= "love" "hate") 0))
	(test-t (eql (string/= "love" "loVe") 2))
	(test-t (not (string/= "life" "death" :start1 3 :start2 1 :end2 2)))
	(test-t (eql (string/= "abcxyz" "ABCxyZ" :start1 3 :start2 3) 5))
	(test-t (eql (string/= "abcxyz" "ABCxyZ" :start1 3 :end1 nil :start2 3 :end2 nil) 5))
	(test-t (eql (string/= "abcxyz" "ABCxyZ" :end1 nil :start2 3 :end2 3) 0))
	(test-t (eql (string/= "abc" "abcxyz") 3))
	(test-t (eql (string/= "abcxyz" "abc") 3))
	(test-t (eql (string/= "abcxyz" "") 0))
	(test-t (eql (string/= "AbcDef" "cdef" :start1 2) 3))
	(test-t (eql (string/= "cdef" "AbcDef" :start2 2) 1))
	(test-t (= (string/= 'love "hate") 0))
	(test-t (not (string/= 'love 'love)))
	(test-t (not (string/= #\a #\a)))
	(test-t (= (string/= #\a #\b) 0))
	(test-t (not (string< "" "")))
	(test-t (not (string< "dog" "dog")))
	(test-t (not (string< " " " ")))
	(test-t (not (string< "abc" "")))
	(test-t (eql (string< "" "abc") 0))
	(test-t (eql (string< "ab" "abc") 2))
	(test-t (not (string< "abc" "ab")))
	(test-t (eql (string< "aaa" "aba") 1))
	(test-t (not (string< "aba" "aaa")))
	(test-t (not (string< "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string< "cat food 2 dollars" "dog food 3 dollars"	      :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string< "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string< "abc" "abc" :end1 1) 1))
	(test-t (eql (string< "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string< "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (not (string< "abc" "abcxyz" :end2 3)))
	(test-t (eql (string< "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (not (string< "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string< "aaaa" "z") 0))
	(test-t (eql (string< "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string< "pppTTTaTTTqqq" "pTTTxTTT"     :start1 6 :end1 7     :start2 4 :end2 5) 6))
	(test-t (not (string< 'love 'hate)))
	(test-t (= (string< 'peace 'war) 0))
	(test-t (not (string< 'love 'love)))
	(test-t (not (string< #\a #\a)))
	(test-t (= (string< #\a #\b) 0))
	(test-t (not (string> "" "")))
	(test-t (not (string> "dog" "dog")))
	(test-t (not (string> " " " ")))
	(test-t (eql (string> "abc" "") 0))
	(test-t (not (string> "" "abc")))
	(test-t (not (string> "ab" "abc")))
	(test-t (eql (string> "abc" "ab") 2))
	(test-t (eql (string> "aba" "aaa") 1))
	(test-t (not (string> "aaa" "aba")))
	(test-t (not (string> "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string> "cat food 2 dollars" "dog food 3 dollars"	     :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string> "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string> "abc" "abc" :end1 1)))
	(test-t (eql (string> "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string> "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string> "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string> "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (not (string> "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string> "z" "aaaa") 0))
	(test-t (eql (string> "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string> "pppTTTxTTTqqq" "pTTTaTTT"     :start1 6 :end1 7      :start2 4 :end2 5) 6))
	(test-t (= (string> 'love 'hate) 0))
	(test-t (not (string> 'peace 'war)))
	(test-t (not (string> 'love 'love)))
	(test-t (not (string> #\a #\a)))
	(test-t (not (string> #\a #\b)))
	(test-t (= (string> #\z #\a) 0))
	(test-t (eql (string<= "" "") 0))
	(test-t (eql (string<= "dog" "dog") 3))
	(test-t (eql (string<= " " " ") 1))
	(test-t (not (string<= "abc" "")))
	(test-t (eql (string<= "ab" "abc") 2))
	(test-t (eql (string<= "aaa" "aba") 1))
	(test-t (not (string<= "aba" "aaa")))
	(test-t (eql (string<= "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string<= "cat food 2 dollars" "dog food 3 dollars"      :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string<= "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string<= "abc" "abc" :end1 1) 1))
	(test-t (eql (string<= "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string<= "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (eql (string<= "abc" "abcxyz" :end2 3) 3))
	(test-t (eql (string<= "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (eql (string<= "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string<= "aaaa" "z") 0))
	(test-t (eql (string<= "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string<= "pppTTTaTTTqqq" "pTTTxTTT"      :start1 6 :end1 7     :start2 4 :end2 5) 6))
	(test-t (not (string<= 'love 'hate)))
	(test-t (= (string<= 'peace 'war) 0))
	(test-t (= (string<= 'love 'love) 4))
	(test-t (= (string<= #\a #\a) 1))
	(test-t (= (string<= #\a #\b) 0))
	(test-t (not (string<= #\z #\a)))
	(test-t (eql (string>= "" "") 0))
	(test-t (eql (string>= "dog" "dog") 3))
	(test-t (eql (string>= " " " ") 1))
	(test-t (eql (string>= "abc" "") 0))
	(test-t (not (string>= "" "abc")))
	(test-t (not (string>= "ab" "abc")))
	(test-t (eql (string>= "abc" "ab") 2))
	(test-t (eql (string>= "aba" "aaa") 1))
	(test-t (not (string>= "aaa" "aba")))
	(test-t (eql (string>= "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string>= "cat food 2 dollars" "dog food 3 dollars"      :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string>= "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string>= "abc" "abc" :end1 1)))
	(test-t (eql (string>= "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string>= "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string>= "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string>= "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (eql (string>= "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string>= "z" "aaaa") 0))
	(test-t (eql (string>= "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string>= "pppTTTxTTTqqq" "pTTTaTTT"     :start1 6 :end1 7      :start2 4 :end2 5) 6))
	(test-t (= (string>= 'love 'hate) 0))
	(test-t (not (string>= 'peace 'war)))
	(test-t (= (string>= 'love 'love) 4))
	(test-t (= (string>= #\a #\a) 1))
	(test-t (not (string>= #\a #\b)))
	(test-t (= (string>= #\z #\a) 0))
	(test-t (string-equal "" ""))
	(test-t (not (string-equal "abc" "")))
	(test-t (not (string-equal "" "abc")))
	(test-t (string-equal "A" "a"))
	(test-t (string-equal "abc" "xyz" :start1 3 :start2 3))
	(test-t (string-equal "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0))
	(test-t (string-equal "axyza" "xyz" :start1 1 :end1 4))
	(test-t (string-equal "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil))
	(test-t (string-equal "abxyz" "xyabz" :end1 2 :start2 2 :end2 4))
	(test-t (not (string-equal "love" "hate")))
	(test-t (string-equal "xyz" "XYZ"))
	(test-t (not (string-equal 'love 'hate)))
	(test-t (not (string-equal 'peace 'war)))
	(test-t (string-equal 'love 'love))
	(test-t (string-equal #\a #\a))
	(test-t (not (string-equal #\a #\b)))
	(test-t (not (string-equal #\z #\a)))
	(test-t (not (string-not-equal "" "")))
	(test-t (eql (string-not-equal "abc" "") 0))
	(test-t (eql (string-not-equal "" "abc") 0))
	(test-t (not (string-not-equal "A" "a")))
	(test-t (not (string-not-equal "abc" "xyz" :start1 3 :start2 3)))
	(test-t (not (string-not-equal "abc" "xyz" :start1 1 :end1 1 :start2 0 :end2 0)))
	(test-t (not (string-not-equal "axyza" "xyz" :start1 1 :end1 4)))
	(test-t (not (string-not-equal "axyza" "xyz" :start1 1 :end1 4 :start2 0 :end2 nil)))
	(test-t (not (string-not-equal "abxyz" "xyabz" :end1 2 :start2 2 :end2 4)))
	(test-t (eql (string-not-equal "love" "hate") 0))
	(test-t (not (string-not-equal "love" "loVe")))
	(test-t (not (string-not-equal "life" "death" :start1 3 :start2 1 :end2 2)))
	(test-t (not (string-not-equal "abcxyz" "ABCxyZ" :start1 3 :start2 3)))
	(test-t (not (string-not-equal "abcxyz" "ABCxyZ" :start1 3 :end1 nil :start2 3 :end2 nil)))
	(test-t (eql (string-not-equal "abcxyz" "ABCxyZ" :end1 nil :start2 3 :end2 3) 0))
	(test-t (eql (string-not-equal "abc" "abcxyz") 3))
	(test-t (eql (string-not-equal "abcxyz" "abc") 3))
	(test-t (eql (string-not-equal "abcxyz" "") 0))
	(test-t (not (string-not-equal "AbcDef" "cdef" :start1 2)))
	(test-t (not (string-not-equal "cdef" "AbcDef" :start2 2)))
	(test-t (not (string-not-equal "ABC" "abc")))
	(test-t (= (string-not-equal 'love 'hate) 0))
	(test-t (= (string-not-equal 'peace 'war) 0))
	(test-t (not (string-not-equal 'love 'love)))
	(test-t (not (string-not-equal #\a #\a)))
	(test-t (= (string-not-equal #\a #\b) 0))
	(test-t (= (string-not-equal #\z #\a) 0))
	(test-t (not (string-lessp "" "")))
	(test-t (not (string-lessp "dog" "dog")))
	(test-t (not (string-lessp " " " ")))
	(test-t (not (string-lessp "abc" "")))
	(test-t (eql (string-lessp "" "abc") 0))
	(test-t (eql (string-lessp "ab" "abc") 2))
	(test-t (not (string-lessp "abc" "ab")))
	(test-t (eql (string-lessp "aaa" "aba") 1))
	(test-t (not (string-lessp "aba" "aaa")))
	(test-t (not (string-lessp "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string-lessp "cat food 2 dollars" "dog food 3 dollars"   :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string-lessp "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string-lessp "abc" "abc" :end1 1) 1))
	(test-t (eql (string-lessp "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string-lessp "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (not (string-lessp "abc" "abcxyz" :end2 3)))
	(test-t (eql (string-lessp "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (not (string-lessp "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string-lessp "aaaa" "z") 0))
	(test-t (eql (string-lessp "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string-lessp "pppTTTaTTTqqq" "pTTTxTTT"   :start1 6 :end1 7   :start2 4 :end2 5) 6))
	(test-t (and (not (string-lessp "abc" "ABC"))     (not (string-lessp "ABC" "abc"))))
	(test-t (not (string-lessp 'love 'hate)))
	(test-t (= (string-lessp 'peace 'war) 0))
	(test-t (not (string-lessp 'love 'love)))
	(test-t (not (string-lessp #\a #\a)))
	(test-t (= (string-lessp #\a #\b) 0))
	(test-t (not (string-lessp #\z #\a)))
	(test-t (not (string-greaterp "" "")))
	(test-t (not (string-greaterp "dog" "dog")))
	(test-t (not (string-greaterp " " " ")))
	(test-t (eql (string-greaterp "abc" "") 0))
	(test-t (not (string-greaterp "" "abc")))
	(test-t (not (string-greaterp "ab" "abc")))
	(test-t (eql (string-greaterp "abc" "ab") 2))
	(test-t (eql (string-greaterp "aba" "aaa") 1))
	(test-t (not (string-greaterp "aaa" "aba")))
	(test-t (not (string-greaterp "my cat food" "your dog food" :start1 6 :start2 8)))
	(test-t (not (string-greaterp "cat food 2 dollars" "dog food 3 dollars"      :start1 3 :end1 9 :start2 3 :end2 9)))
	(test-t (eql (string-greaterp "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string-greaterp "abc" "abc" :end1 1)))
	(test-t (eql (string-greaterp "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string-greaterp "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string-greaterp "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string-greaterp "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (not (string-greaterp "xyzabc" "abcdef" :start1 3 :end2 3)))
	(test-t (eql (string-greaterp "z" "aaaa") 0))
	(test-t (eql (string-greaterp "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string-greaterp "pppTTTxTTTqqq" "pTTTaTTT"     :start1 6 :end1 7	     :start2 4 :end2 5) 6))
	(test-t (and (not (string-greaterp "abc" "ABC"))     (not (string-greaterp "ABC" "abc"))))
	(test-t (= (string-greaterp 'love 'hate) 0))
	(test-t (not (string-greaterp 'peace 'war)))
	(test-t (not (string-greaterp 'love 'love)))
	(test-t (not (string-greaterp #\a #\a)))
	(test-t (not (string-greaterp #\a #\b)))
	(test-t (= (string-greaterp #\z #\a) 0))
	(test-t (eql (string-not-greaterp "" "") 0))
	(test-t (eql (string-not-greaterp "dog" "dog") 3))
	(test-t (eql (string-not-greaterp " " " ") 1))
	(test-t (not (string-not-greaterp "abc" "")))
	(test-t (eql (string-not-greaterp "ab" "abc") 2))
	(test-t (eql (string-not-greaterp "aaa" "aba") 1))
	(test-t (not (string-not-greaterp "aba" "aaa")))
	(test-t (eql (string-not-greaterp "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string-not-greaterp "cat food 2 dollars" "dog food 3 dollars"  :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string-not-greaterp "xyzabc" "abcd" :start1 3) 6))
	(test-t (eql (string-not-greaterp "abc" "abc" :end1 1) 1))
	(test-t (eql (string-not-greaterp "xyzabc" "abc" :start1 3 :end1 5) 5))
	(test-t (eql (string-not-greaterp "xyz" "abcxyzXYZ" :start2 3) 3))
	(test-t (eql (string-not-greaterp "abc" "abcxyz" :end2 3) 3))
	(test-t (eql (string-not-greaterp "xyz" "abcxyz" :end1 2 :start2 3) 2))
	(test-t (eql (string-not-greaterp "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string-not-greaterp "aaaa" "z") 0))
	(test-t (eql (string-not-greaterp "pppTTTaTTTqqq" "pTTTxTTT" :start1 3 :start2 1) 6))
	(test-t (eql (string-not-greaterp "pppTTTaTTTqqq" "pTTTxTTT"  :start1 6 :end1 7	  :start2 4 :end2 5) 6))
	(test-t (and (eql (string-not-greaterp "abc" "ABC") 3)    (eql (string-not-greaterp "ABC" "abc") 3)))
	(test-t (not (string-not-greaterp 'love 'hate)))
	(test-t (= (string-not-greaterp 'peace 'war) 0))
	(test-t (= (string-not-greaterp 'love 'love) 4))
	(test-t (= (string-not-greaterp #\a #\a) 1))
	(test-t (= (string-not-greaterp #\a #\b) 0))
	(test-t (not (string-not-greaterp #\z #\a)))
	(test-t (eql (string-not-lessp "" "") 0))
	(test-t (eql (string-not-lessp "dog" "dog") 3))
	(test-t (eql (string-not-lessp " " " ") 1))
	(test-t (eql (string-not-lessp "abc" "") 0))
	(test-t (not (string-not-lessp "" "abc")))
	(test-t (not (string-not-lessp "ab" "abc")))
	(test-t (eql (string-not-lessp "abc" "ab") 2))
	(test-t (eql (string-not-lessp "aba" "aaa") 1))
	(test-t (not (string-not-lessp "aaa" "aba")))
	(test-t (eql (string-not-lessp "my cat food" "your dog food" :start1 6 :start2 8) 11))
	(test-t (eql (string-not-lessp "cat food 2 dollars" "dog food 3 dollars"  :start1 3 :end1 9 :start2 3 :end2 9) 9))
	(test-t (eql (string-not-lessp "xyzabcde" "abcd" :start1 3) 7))
	(test-t (not (string-not-lessp "abc" "abc" :end1 1)))
	(test-t (eql (string-not-lessp "xyzabc" "a" :start1 3 :end1 5) 4))
	(test-t (eql (string-not-lessp "xyzXYZ" "abcxyz" :start2 3) 3))
	(test-t (eql (string-not-lessp "abcxyz" "abcxyz" :end2 3) 3))
	(test-t (not (string-not-lessp "xyzXYZ" "abcxyz" :end1 2 :start2 3)))
	(test-t (eql (string-not-lessp "xyzabc" "abcdef" :start1 3 :end2 3) 6))
	(test-t (eql (string-not-lessp "z" "aaaa") 0))
	(test-t (eql (string-not-lessp "pTTTxTTTqqq" "pppTTTaTTT" :start1 1 :start2 3) 4))
	(test-t (eql (string-not-lessp "pppTTTxTTTqqq" "pTTTaTTT"       :start1 6 :end1 7       :start2 4 :end2 5) 6))
	(test-t (and (eql (string-not-lessp "abc" "ABC") 3)    (eql (string-not-lessp "ABC" "abc") 3)))
	(test-t (= (string-not-lessp 'love 'hate) 0))
	(test-t (not (string-not-lessp 'peace 'war)))
	(test-t (= (string-not-lessp 'love 'love) 4))
	(test-t (= (string-not-lessp #\a #\a) 1))
	(test-t (not (string-not-lessp #\a #\b)))
	(test-t (= (string-not-lessp #\z #\a) 0))
	(test-t (stringp "aaaaaa"))
	(test-t (not (stringp #\a)))
	(test-t (not (stringp 'a)))
	(test-t (not (stringp '(string))))
	(test-t (string= (cl-make-string 3 :initial-element #\a) "aaa"))
	(test-t (string= (cl-make-string 1 :initial-element #\space) " "))
	(test-t (string= (cl-make-string 0) ""))
	
	(test-t (null (dotimes (i 10))))
	(test-t (= (dotimes (temp-one 10 temp-one)) 10))
	(test-t (let ((temp-two 0)) (and (eq t (dotimes (temp-one 10 t) (incf temp-two)))  (eql temp-two 10))))
	(test-t (let ((count 0))  (eql (dotimes (i 5 count) (incf count)) 5)))
	(test-t (let ((count 0))  (eql (dotimes (i 1 count) (incf count)) 1)))
	(test-t (let ((count 0))  (zerop (dotimes (i 0 count) (incf count)))))
	(test-t (let ((count 0))  (zerop (dotimes (i -1 count) (incf count)))))
	(test-t (let ((count 0))  (zerop (dotimes (i -100 count) (incf count)))))
	(test-t (eql (dotimes (i 3 i)) 3))
	(test-t (eql (dotimes (i 2 i)) 2))
	(test-t (eql (dotimes (i 1 i)) 1))
	(test-t (eql (dotimes (i 0 i)) 0))
	(test-t (eql (dotimes (i -1 i)) 0))
	(test-t (eql (dotimes (i -2 i)) 0))
	(test-t (eql (dotimes (i -10 i)) 0))
	(test-t (let ((list nil))  (and (eq (dotimes (i 10 t) (push i list)) t)       (equal list '(9 8 7 6 5 4 3 2 1 0)))))
	(test-t (let ((list nil))  (equal (dotimes (i 10 (push i list)) (push i list))	 '(10 9 8 7 6 5 4 3 2 1 0))))
	(test-t (let ((list nil))  (equal (dotimes (i '10 (push i list)) (push i list))	 '(10 9 8 7 6 5 4 3 2 1 0))))
	(test-t (let ((list nil))  (equal (dotimes (i (/ 100 10) (push i list)) (push i list))	 '(10 9 8 7 6 5 4 3 2 1 0))))
	(test-t (= 3 (let ((i 3)) (dotimes (i i i) ))))
	(test-t (= 3 (let ((x 0)) (dotimes (i 3 x) (incf x)))))
	(test-t (= 3 (dotimes (i 3 i) )))
	(test-t (= 3 (let ((x 0)) (dotimes (i 3 x) (declare (fixnum i)) (incf x)))))
	(test-t (null (dolist (x '()))))
	(test-t (null (dolist (x '(a)))))
	(test-t (eq t (dolist (x nil t))))
	(test-t (= 6 (let ((sum 0))       (dolist (x '(0 1 2 3) sum)	 (incf sum x)))))
	(test-t (let ((temp-two '()))  (equal (dolist (temp-one '(1 2 3 4) temp-two) (push temp-one temp-two))	 '(4 3 2 1))))
	(test-t (let ((temp-two 0))  (and (null (dolist (temp-one '(1 2 3 4)) (incf temp-two)))       (eql temp-two 4))))
	(test-t (null (dolist (var nil var))))
	(test-t (let ((list nil))  (equal (dolist (var '(0 1 2 3) list)	   (push var list))	 '(3 2 1 0))))
	(test-t (null (dolist (var '(0 1 2 3)))))
	(test-t (eql (do ((temp-one 1 (1+ temp-one))	  (temp-two 0 (1- temp-two)))	 ((> (- temp-one temp-two) 5) temp-one))     4))
	(test-t (eql (do ((temp-one 1 (1+ temp-one))	  (temp-two 0 (1+ temp-one)))     	 ((= 3 temp-two) temp-one))     3))
	(test-t (eql (do* ((temp-one 1 (1+ temp-one))	   (temp-two 0 (1+ temp-one)))	 ((= 3 temp-two) temp-one))     2))
	
	(test-t (let ((a-vector (vector 1 nil 3 nil)))
		  (do ((i 0 (+ i 1))
		       (n (array-dimension a-vector 0)))
		      ((= i n))
		    (when (null (aref a-vector i))
			  (setf (aref a-vector i) 0)))
		  (equalp a-vector #(1 0 3 0))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (equalp (do ((i 0 (1+ i))
			       (n #f)
			       (j 9 (1- j)))
			      ((>= i j) vec)
			    (setq n (aref vec i))
			    (setf (aref vec i) (aref vec j))
			    (setf (aref vec j) n))
			  #(9 8 7 6 5 4 3 2 1 0))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (n #f)
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (n #f)
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (n #f)
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((n #f)
				  (i 0 (1+ i))
				  (j 9 (1- j)))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
		  (and (null (do ((i 0 (1+ i))
				  (j 9 (1- j))
				  (n #f))
				 ((>= i j))
			       (setq n (aref vec i))
			       (setf (aref vec i) (aref vec j))
			       (setf (aref vec j) n)))
		       (equalp vec #(9 8 7 6 5 4 3 2 1 0)))))
	
	(test-t (= (funcall (lambda (x) (+ x 3)) 4) 7))
	(test-t (= (funcall (lambda args (apply + args)) 1 2 3 4) 10))
	(test-t (functionp (lambda args (apply + args))))
	
	(test-t (consp (cons 'a 'b)))
	(test-t (consp '(1 . 2)))
	(test-t (consp (list nil)))
	(test-t (not (consp 'a)))
	(test-t (not (consp nil)))
	(test-t (not (consp 1)))
	(test-t (not (consp #\a)))
	(test-t (let ((a (cons 1 2))) (and (eql (car a) 1) (eql (cdr a) 2))))
	(test-t (equal (cons 1 nil) '(1)))
	(test-t (equal (cons nil nil) '(())))
	(test-t (equal (cons 'a (cons 'b (cons 'c '()))) '(a b c)))
	(test-t (atom 'a))
	(test-t (atom nil))
	(test-t (atom 1))
	(test-t (atom #\a))
	(test-t (not (atom (cons 1 2))))
	(test-t (not (atom '(a . b))))
	(test-t (not (atom (list nil))))
	(test-t (listp nil))
	(test-t (listp '(a b c)))
	(test-t (listp '(a . b)))
	(test-t (listp (cons 'a 'b)))
	(test-t (not (listp 1)))
	(test-t (not (listp 't)))
	(test-t (null '()))
	(test-t (null nil))
	(test-t (not (null t)))
	(test-t (null (cdr '(a))))
	(test-t (not (null (cdr '(1 . 2)))))
	(test-t (not (null 'a)))
	(test-t (endp '()))
	(test-t (not (endp '(1))))
	(test-t (not (endp '(1 2))))
	(test-t (not (endp '(1 2 3))))
	(test-t (not (endp (cons 1 2))))
	(test-t (endp (cddr '(1 2))))
	(test-t (let ((a (cons 1 2))) (and (eq (rplaca a 0) a) (equal a '(0 . 2)))))
	(test-t (let ((a (list 1 2 3))) (and (eq (rplaca a 0) a) (equal a '(0 2 3)))))
	(test-t (let ((a (cons 1 2))) (and (eq (rplacd a 0) a) (equal a '(1 . 0)))))
	(test-t (let ((a (list 1 2 3))) (and (eq (rplacd a 0) a) (equal a '(1 . 0)))))
	(test-t (eq (car '(a . b)) 'a))
	(test-t (let ((a (cons 1 2))) (eq (car (list a)) a)))
	(test-t (eq (cdr '(a . b)) 'b))
	(test-t (eq (rest '(a . b)) 'b))
	(test-t (let ((a (cons 1 2))) (eq (cdr (cons 1 a)) a)))
	(test-t (let ((a (cons 1 2))) (eq (rest (cons 1 a)) a)))
	(test-t (eq (caar '((a) b c)) 'a))
	(test-t (eq (cadr '(a b c)) 'b))
	(test-t (eq (cdar '((a . aa) b c)) 'aa))
	(test-t (eq (cddr '(a b . c)) 'c))
	(test-t (eq (caaar '(((a)) b c)) 'a))
	(test-t (eq (caadr '(a (b) c)) 'b))
	(test-t (eq (cadar '((a aa) b c)) 'aa))
	(test-t (eq (caddr '(a b c)) 'c))
	(test-t (eq (cdaar '(((a . aa)) b c)) 'aa))
	(test-t (eq (cdadr '(a (b . bb) c)) 'bb))
	(test-t (eq (cddar '((a aa . aaa) b c)) 'aaa))
	(test-t (eq (cdddr '(a b c . d)) 'd))
	(test-t (eq (caaaar '((((a))) b c)) 'a))
	(test-t (eq (caaadr '(a ((b)) c)) 'b))
	(test-t (eq (caadar '((a (aa)) b c)) 'aa))
	(test-t (eq (caaddr '(a b (c))) 'c))
	(test-t (eq (cadaar '(((a aa)) b c)) 'aa))
	(test-t (eq (cadadr '(a (b bb) c)) 'bb))
	(test-t (eq (caddar '((a aa aaa) b c)) 'aaa))
	(test-t (eq (cadddr '(a b c d)) 'd))
	(test-t (eq (cdaaar '((((a . aa))) b c)) 'aa))
	(test-t (eq (cdaadr '(a ((b . bb)) c)) 'bb))
	(test-t (eq (cdadar '((a (aa . aaa)) b c)) 'aaa))
	(test-t (eq (cdaddr '(a b (c . cc))) 'cc))
	(test-t (eq (cddaar '(((a aa . aaa)) b c)) 'aaa))
	(test-t (eq (cddadr '(a (b bb . bbb) c)) 'bbb))
	(test-t (eq (cdddar '((a aa aaa . aaaa) b c)) 'aaaa))
	(test-t (eq (cddddr '(a b c d . e)) 'e))
	(test-t (eq (copy-tree nil) nil))
	(test-t (let* ((a (list 'a))
		       (b (list 'b))
		       (c (list 'c))
		       (x3 (cons c nil))
		       (x2 (cons b x3))
		       (x (cons a x2))
		       (y (copy-tree x)))
		  (and (not (eq x y))
		       (not (eq (car x) (car y)))
		       (not (eq (cdr x) (cdr y)))
		       (not (eq (cadr x) (cadr y)))
		       (not (eq (cddr x) (cddr y)))
		       (not (eq (caddr x) (caddr y)))
		       (eq (cdddr x) (cdddr y))
		       (equal x y)
		       (eq (car x) a) (eq (car a) 'a) (eq (cdr a) nil)
		       (eq (cdr x) x2)
		       (eq (car x2) b) (eq (car b) 'b) (eq (cdr b) nil)
		       (eq (cdr x2) x3)
		       (eq (car x3) c) (eq (car c) 'c) (eq (cdr c) nil)
		       (eq (cdr x3) nil))))
	(test-t (let* ((x (list (list 'a 1) (list 'b 2) (list 'c 3)))
		       (y (copy-tree x)))
		  (and (not (eq (car x) (car y)))
		       (not (eq (cadr x) (cadr y)))
		       (not (eq (caddr x) (caddr y))))))
	(test-t (let* ((x (list (list (list 1))))
		       (y (copy-tree x)))
		  (and (not (eq x y))
		       (not (eq (car x) (car y)))
		       (not (eq (caar x) (caar y))))))
	(test-t (let ((x (list 'a 'b 'c 'd)))
		  (and (equal (sublis '((a . 1) (b . 2) (c . 3)) x)
			      '(1 2 3 d))
		       (equal x '(a b c d)))))
	(test-t (eq (sublis '() '()) '()))
	(test-t (equal (sublis '() '(1 2 3)) '(1 2 3)))
	(test-t (eq (sublis '((a . 1) (b . 2)) '()) nil))
	(test-t (equal (sublis '((a . 1) (b . 2) (c . 3)) '(((a)) (b) c)) '(((1)) (2) 3)))
	(test-t (equal (sublis '(((a) . 1) ((b) . 2) ((c) . 3)) '((((a))) ((b)) (c))) '((((a))) ((b)) (c))))
	(test-t (equal (sublis '(((a) . 1) ((b) . 2) ((c) . 3)) '((((a))) ((b)) (c)) :test equal) '(((1)) (2) 3)))
	(test-t (equal (nsublis '((a . 1) (b . 2) (c . 3)) (list 'a 'b 'c 'd)) '(1 2 3 d)))
	(test-t (let* ((x (list 'a 'b 'c 'd)) (y (nsublis '((a . 1) (b . 2) (c . 3)) x))) (and (eq x y) (equal x '(1 2 3 d)))))
	(test-t (let ((x (list 'l 'm 'n))) (and (eq (nsublis '((a . 1) (b . 2) (c . 3)) x) x) (equal x '(l m n)))))
	(test-t (let* ((n (cons 'n nil))
		       (m (cons 'm n))
		       (l (cons 'l m))
		       (x (nsublis '((a . 1) (b . 2) (c . 3)) l)))
		  (and (eq x l)
		       (eq (car l) 'l)
		       (eq (cdr l) m)
		       (eq (car m) 'm)
		       (eq (cdr m) n)
		       (eq (car n) 'n)
		       (eq (cdr n) nil))))
	(test-t (eq (nsublis '() '()) '()))
	(test-t (equal (nsublis '() '(1 2 3)) '(1 2 3)))
	(test-t (eq (nsublis '((a . 1) (b . 2)) '()) nil))
	(test-t (equal (nsublis '((a b c) (b c d) (c d e)) (list 'a 'b 'c)) '((b c) (c d) (d e))))
	(test-t (equal (nsublis '((a . 1) (b . 2) (c . 3)) (copy-tree '(((a)) (b) c))) '(((1)) (2) 3)))
	(test-t (equal (nsublis '(((a) . 1) ((b) . 2) ((c) . 3)) (copy-tree '((((a))) ((b)) (c)))) '((((a))) ((b)) (c))))
	(test-t (equal (nsublis '(((a) . 1) ((b) . 2) ((c) . 3)) (copy-tree '((((a))) ((b)) (c))) :test equal) '(((1)) (2) 3)))
	(test-t (let ((tree '(old (old) ((old))))) (equal (subst 'new 'old tree) '(new (new) ((new))))))
	(test-t (eq (subst 'new 'old 'old) 'new))
	(test-t (eq (subst 'new 'old 'not-old) 'not-old))
	(test-t (equal (subst 'new '(b) '(a ((b))) :test equal) '(a (new))))
	(test-t (equal (subst 'x 3 '(1 (1 2) (1 2 3) (1 2 3 4)) :key (lambda (y) (and (listp y) (third y)))) '(1 (1 2) x x)))
	(test-t (equal (subst 'x "D" '("a" ("a" "b") ("a" "b" "c") ("a" "b" "c" "d"))
			      :test equalp
			      :key (lambda (y) (and (listp y) (fourth y))))
		       '("a" ("a" "b") ("a" "b" "c") x)))
	(test-t (equal (subst-if 'new (lambda (x) (eq x 'old)) '(old old)) '(new new)))
	(test-t (eq (subst-if 'new (lambda (x) (eq x 'old)) 'old) 'new))
	(test-t (equal (subst-if 'x (lambda (x) (eql x 3)) '(1 (1 2) (1 2 3) (1 2 3 4)) :key (lambda (y) (and (listp y) (third y)))) '(1 (1 2) x x)))
	(test-t (let ((tree '(old (old) ((old))))) (equal (subst-if 'new (lambda (x) (eq x 'old)) tree) '(new (new) ((new))))))
	(test-t (eq (subst-if 'new (lambda (x) (eq x 'old)) 'old) 'new))
	(test-t (eq (subst-if 'new (lambda (x) (eq x 'old)) 'not-old) 'not-old))
	(test-t (equal (subst-if 'new (lambda (x) (equal x '(b))) '(a ((b)))) '(a (new))))
	(test-t (equal (subst-if 'x (lambda (x) (eql x 3)) '(1 (1 2) (1 2 3) (1 2 3 4)) :key (lambda (y) (and (listp y) (third y)))) '(1 (1 2) x x)))
	(test-t (equal (subst-if 'x
				 (lambda (x) (equalp x "D"))
				 '("a" ("a" "b") ("a" "b" "c") ("a" "b" "c" "d"))
				 :key (lambda (y) (and (listp y) (fourth y))))
		       '("a" ("a" "b") ("a" "b" "c") x)))
	(test-t (equal (subst-if-not 'new (lambda (x) (not (eq x 'old))) '(old old)) '(new new)))
	(test-t (eq (subst-if-not 'new (lambda (x) (not (eq x 'old))) 'old) 'new))
	(test-t (equal (subst-if-not 'x (lambda (x) (not (eql x 3)))
				     '(1 (1 2) (1 2 3) (1 2 3 4))
				     :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (let ((tree '(old (old) ((old)))))
		  (equal (subst-if-not 'new (lambda (x) (not (eq x 'old))) tree)
			 '(new (new) ((new))))))
	(test-t (eq (subst-if-not 'new (lambda (x) (not (eq x 'old))) 'old) 'new))
	(test-t (eq (subst-if-not 'new (lambda (x) (not (eq x 'old))) 'not-old) 'not-old))
	(test-t (equal (subst-if-not 'new (lambda (x) (not (equal x '(b)))) '(a ((b)))) '(a (new))))
	(test-t (equal (subst-if-not 'x
				     (lambda (x) (not (eql x 3)))
				     '(1 (1 2) (1 2 3) (1 2 3 4))
				     :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (equal (subst-if-not 'x
				     (lambda (x) (not (equalp x "D")))
				     '("a" ("a" "b") ("a" "b" "c") ("a" "b" "c" "d"))
				     :key (lambda (y) (and (listp y) (fourth y))))
		       '("a" ("a" "b") ("a" "b" "c") x)))
	(test-t (let ((tree '(old (old) ((old)))))
		  (equal (nsubst 'new 'old (copy-tree tree))
			 '(new (new) ((new))))))
	(test-t (let* ((tree (copy-tree '(old (old) ((old)))))
		       (new-tree (nsubst 'new 'old tree)))
		  (and (eq tree new-tree)
		       (equal tree '(new (new) ((new)))))))
	(test-t (eq (nsubst 'new 'old 'old) 'new))
	(test-t (eq (nsubst 'new 'old 'not-old) 'not-old))
	(test-t (equal (nsubst 'new '(b) (copy-tree '(a ((b)))) :test equal) '(a (new))))
	(test-t (equal (nsubst 'x 3 (copy-tree '(1 (1 2) (1 2 3) (1 2 3 4)))
			       :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (equal (nsubst 'x "D"
			       (copy-tree '("a" ("a" "b") ("a" "b" "c") ("a" "b" "c" "d")))
			       :test equalp
			       :key (lambda (y) (and (listp y) (fourth y))))
		       '("a" ("a" "b") ("a" "b" "c") x)))
	(test-t (equal (nsubst-if 'new (lambda (x) (eq x 'old)) (list 'old 'old)) '(new new)))
	(test-t (eq (nsubst-if 'new (lambda (x) (eq x 'old)) 'old) 'new))
	(test-t (let* ((x (copy-tree '(old (old) ((old)) (old) old)))
		       (y (nsubst-if 'new (lambda (x) (eq x 'old)) x)))
		  (and (eq x y)
		       (equal x '(new (new) ((new)) (new) new)))))
	(test-t (equal (nsubst-if 'x
				  (lambda (x) (eql x 3))
				  (copy-tree '(1 (1 2) (1 2 3) (1 2 3 4)))
				  :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (let ((tree '(old (old) ((old)))))
		  (equal (nsubst-if 'new (lambda (x) (eq x 'old)) (copy-tree tree))
			 '(new (new) ((new))))))
	(test-t (eq (nsubst-if 'new (lambda (x) (eq x 'old)) 'old) 'new))
	(test-t (eq (nsubst-if 'new (lambda (x) (eq x 'old)) 'not-old) 'not-old))
	(test-t (equal (nsubst-if 'new (lambda (x) (equal x '(b))) (copy-tree '(a ((b))))) '(a (new))))
	(test-t (equal (nsubst-if 'x
				  (lambda (x) (eql x 3))
				  (copy-tree '(1 (1 2) (1 2 3) (1 2 3 4)))
				  :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (equal (nsubst-if 'x
				  (lambda (x) (equalp x "D"))
				  (copy-tree '("a" ("a" "b") ("a" "b" "c") ("a" "b" "c" "d")))
				  :key (lambda (y) (and (listp y) (fourth y))))
		       '("a" ("a" "b") ("a" "b" "c") x)))
	(test-t (equal (nsubst-if-not 'new (lambda (x) (not (eq x 'old)))
				      (list 'old 'old))
		       '(new new)))
	(test-t (eq (nsubst-if-not 'new (lambda (x) (not (eq x 'old))) 'old) 'new))
	(test-t (let* ((x (copy-tree '(old (old) ((old)) (old) old)))
		       (y (nsubst-if-not 'new (lambda (x) (not (eq x 'old))) x)))
		  (and (eq x y)
		       (equal x '(new (new) ((new)) (new) new)))))
	(test-t (equal (nsubst-if-not 'x (lambda (x) (not (eql x 3)))
				      (copy-tree '(1 (1 2) (1 2 3) (1 2 3 4)))
				      :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (let ((tree '(old (old) ((old)))))
		  (equal (nsubst-if-not 'new (lambda (x) (not (eq x 'old))) (copy-tree tree))
			 '(new (new) ((new))))))
	(test-t (eq (nsubst-if-not 'new (lambda (x) (not (eq x 'old))) 'old) 'new))
	(test-t (eq (nsubst-if-not 'new (lambda (x) (not (eq x 'old))) 'not-old) 'not-old))
	(test-t (equal (nsubst-if-not 'new (lambda (x) (not (equal x '(b)))) (copy-tree '(a ((b))))) '(a (new))))
	(test-t (equal (nsubst-if-not 'x
				      (lambda (x) (not (eql x 3)))
				      (copy-tree '(1 (1 2) (1 2 3) (1 2 3 4)))
				      :key (lambda (y) (and (listp y) (third y))))
		       '(1 (1 2) x x)))
	(test-t (equal
		 (nsubst-if-not 'x
				(lambda (x) (not (equalp x "D")))
				(copy-tree '("a" ("a" "b") ("a" "b" "c") ("a" "b" "c" "d")))
				:key (lambda (y) (and (listp y) (fourth y))))
		 '("a" ("a" "b") ("a" "b" "c") x)))
	(test-t (tree-equal 'a 'a))
	(test-t (not (tree-equal 'a 'b)))
	(test-t (tree-equal '(a (b (c))) '(a (b (c)))))
	(test-t (tree-equal '(a (b (c))) '(a (b (c))) :test eq))
	(test-t (not (tree-equal '("a" ("b" ("c"))) '("a" ("b" ("c"))))))
	(test-t (tree-equal '("a" ("b" ("c"))) '("a" ("b" ("c"))) :test equal))
	(test-t (not (tree-equal '(a b) '(a (b)))))
	(test-t (eq (copy-list '()) '()))
	(test-t (equal (copy-list '(a b c)) '(a b c)))
	(test-t (equal (copy-list '(a . b)) '(a . b)))
	(test-t (let* ((x '(a b c)) (y (copy-list x))) (and (equal x y) (not (eq x y)))))
	(test-t (let* ((a (list 'a))
		       (b (list 'b))
		       (c (list 'c))
		       (x (list a b c))
		       (y (copy-list x)))
		  (and (equal x y)
		       (not (eq x y))
		       (eq (car x) (car y))
		       (eq (cadr x) (cadr y))
		       (eq (caddr x) (caddr y))
		       (eq (caar x) 'a)
		       (eq (caadr x) 'b)
		       (eq (caaddr x) 'c))))
	(test-t (null (list)))
	(test-t (equal (list 1) '(1)))
	(test-t (equal (list 1 2 3) '(1 2 3)))
	(test-t (equal (list* 1 2 '(3)) '(1 2 3)))
	(test-t (equal (list* 1 2 'x) '(1 2 . x)))
	(test-t (equal (list* 1 2 '(3 4)) '(1 2 3 4)))
	(test-t (eq (list* 'x) 'x))
	(test-t (eql (list-length '()) 0))
	(test-t (eql (list-length '(1)) 1))
	(test-t (eql (list-length '(1 2)) 2))
	(test-t (equal (cl-make-list 5) '(() () () () ())))
	(test-t (equal (cl-make-list 3 :initial-element 'rah) '(rah rah rah)))
	(test-t (equal (cl-make-list 2 :initial-element '(1 2 3)) '((1 2 3) (1 2 3))))
	(test-t (null (cl-make-list 0)))
	(test-t (null (cl-make-list 0 :initial-element 'new-element)))
	(test-t (let ((place nil)) (and (equal (push 0 place) '(0)) (equal place '(0)))))
	(test-t (let ((place (list 1 2 3))) (and (equal (push 0 place) '(0 1 2 3)) (equal place '(0 1 2 3)))))
	(test-t (let ((a (list (list 1 2 3) 9))) (and (equal (push 0 (car a)) '(0 1 2 3)) (equal a '((0 1 2 3) 9)))))
	(test-t (let ((place (list 1 2 3))) (and (eql (pop place) 1) (equal place '(2 3)))))
	(test-t (let ((a (list (list 1 2 3) 9))) (and (eql (pop (car a)) 1) (equal a '((2 3) 9)))))
	(test-t (let ((x (list 'a 'b 'c))) (and (eq (pop (cdr x)) 'b) (equal x '(a c)))))
	(test-t (eq (first '(a . b)) 'a))
	(test-t (null (first nil)))
	(test-t (let ((a (cons 1 2))) (eq (first (list a)) a)))
	(test-t (eql (first '(1 2 3)) '1))
	(test-t (eql (second '(1 2 3)) '2))
	(test-t (eql (third '(1 2 3)) '3))
	(test-t (eql (fourth '(1 2 3 4)) '4))
	(test-t (eql (fifth '(1 2 3 4 5)) '5))
	(test-t (eql (sixth '(1 2 3 4 5 6)) '6))
	(test-t (eql (seventh '(1 2 3 4 5 6 7)) '7))
	(test-t (eql (eighth '(1 2 3 4 5 6 7 8)) '8))
	(test-t (eql (ninth '(1 2 3 4 5 6 7 8 9)) '9))
	(test-t (eql (tenth '(1 2 3 4 5 6 7 8 9 10)) '10))
	(test-t (let ((x '(a b c))) (eq (nthcdr 0 x) x)))
	(test-t (let ((x '(a b c))) (eq (nthcdr 1 x) (cdr x))))
	(test-t (let ((x '(a b c))) (eq (nthcdr 2 x) (cddr x))))
	(test-t (let ((x '(a b c))) (eq (nthcdr 2 x) (cddr x))))
	(test-t (let ((x '(a b c))) (eq (nthcdr 3 x) (cdddr x))))
	(test-t (equal (nthcdr 0 '(0 1 2)) '(0 1 2)))
	(test-t (equal (nthcdr 1 '(0 1 2)) '(1 2)))
	(test-t (equal (nthcdr 2 '(0 1 2)) '(2)))
	(test-t (equal (nthcdr 3 '(0 1 2)) '()))
	(test-t (eql (nthcdr 1 '(0 . 1)) 1))
	(test-t (eql (nth 0 '(a b c)) 'a))
	(test-t (eql (nth 1 '(a b c)) 'b))
	(test-t (eql (nth 2 '(a b c)) 'c))
	(test-t (eql (nth 3 '(a b c)) '()))
	(test-t (eql (nth 4 '(a b c)) '()))
	(test-t (eql (nth 5 '(a b c)) '()))
	(test-t (eql (nth 6 '(a b c)) '()))
	(test-t (let ((x (list 'a 'b 'c))) (and (eq (setf (nth 0 x) 'z) 'z) (equal x '(z b c)))))
	(test-t (let ((x (list 'a 'b 'c))) (and (eq (setf (nth 1 x) 'z) 'z) (equal x '(a z c)))))
	(test-t (let ((x (list 'a 'b 'c))) (and (eq (setf (nth 2 x) 'z) 'z) (equal x '(a b z)))))
	(test-t (let ((0-to-3 (list 0 1 2 3))) (and (equal (setf (nth 2 0-to-3) "two") "two") (equal 0-to-3 '(0 1 "two" 3)))))
	(test-t (let* ((x (list 'a 'b 'c))) (eq (nconc x) x)))
	(test-t (let* ((x (list 'a 'b 'c))
		       (y (list 'd 'e 'f))
		       (list (nconc x y)))
		  (and (eq list x)
		       (eq (nthcdr 3 list) y)
		       (equal list '(a b c d e f)))))
	(test-t (let* ((x (list 'a))
		       (y (list 'b))
		       (z (list 'c))
		       (list (nconc x y z)))
		  (and (eq x list)
		       (eq (first list) 'a)
		       (eq y (cdr list))
		       (eq (second list) 'b)
		       (eq z (cddr list))
		       (eq (third list) 'c))))
	(test-t (equal (append '(a b) '() '(c d) '(e f)) '(a b c d e f)))
	(test-t (null (append)))
	(test-t (null (append '())))
	(test-t (null (append '() '())))
	(test-t (eq (append 'a) 'a))
	(test-t (eq (append '() 'a) 'a))
	(test-t (eq (append '() '() 'a) 'a))
	(test-t (equal (append '(a b) 'c) '(a b . c)))
	(test-t (let* ((x '(a b c))
		       (y '(d e f))
		       (z (append x y)))
		  (and (equal z '(a b c d e f))
		       (eq (nthcdr 3 z) y)
		       (not (eq x z)))))
	(test-t (equal (revappend '(a b c) '(d e f)) '(c b a d e f)))
	(test-t (let* ((x '(a b c))
		       (y '(d e f))
		       (z (revappend x y)))
		  (and (equal z '(c b a d e f))
		       (not (eq x z))
		       (eq (nthcdr 3 z) y))))
	(test-t (let ((x '(a b c))) (eq (revappend '() x) x)))
	(test-t (null (revappend '() '())))
	(test-t (eq (revappend '() 'a) 'a))
	(test-t (equal (revappend '(a) 'b) '(a . b)))
	(test-t (equal (revappend '(a) '()) '(a)))
	(test-t (equal (revappend '(1 2 3) '()) '(3 2 1)))
	(test-t (equal (nreconc (list 'a 'b 'c) '(d e f)) '(c b a d e f)))
	(test-t (let* ((x (list 'a 'b 'c))
		       (y '(d e f))
		       (z (nreconc x y)))
		  (and (equal z '(c b a d e f))
		       (eq (nthcdr 3 z) y))))
	(test-t (equal (nreconc (list 'a) 'b) '(a . b)))
	(test-t (equal (nreconc (list 'a) '()) '(a)))
	(test-t (equal (nreconc (list 1 2 3) '()) '(3 2 1)))
	(test-t (null (butlast nil)))
	(test-t (null (butlast nil 1)))
	(test-t (null (butlast nil 2)))
	(test-t (null (butlast nil 3)))
	(test-t (equal (butlast '(1 2 3 4 5)) '(1 2 3 4)))
	(test-t (equal (butlast '(1 2 3 4 5) 1) '(1 2 3 4)))
	(test-t (equal (butlast '(1 2 3 4 5) 2) '(1 2 3)))
	(test-t (equal (butlast '(1 2 3 4 5) 3) '(1 2)))
	(test-t (equal (butlast '(1 2 3 4 5) 4) '(1)))
	(test-t (equal (butlast '(1 2 3 4 5) 5) '()))
	(test-t (equal (butlast '(1 2 3 4 5) 6) '()))
	(test-t (equal (butlast '(1 2 3 4 5) 7) '()))
	(test-t (let ((a '(1 2 3 4 5))) (equal (butlast a 3) '(1 2)) (equal a '(1 2 3 4 5))))
	(test-t (null (nbutlast nil)))
	(test-t (null (nbutlast nil 1)))
	(test-t (null (nbutlast nil 2)))
	(test-t (null (nbutlast nil 3)))
	(test-t (equal (nbutlast (list 1 2 3 4 5)) '(1 2 3 4)))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 1) '(1 2 3 4)))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 2) '(1 2 3)))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 3) '(1 2)))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 4) '(1)))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 5) '()))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 6) '()))
	(test-t (equal (nbutlast (list 1 2 3 4 5) 7) '()))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6)) '(1 2 3 4)))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 1) '(1 2 3 4)))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 2) '(1 2 3)))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 3) '(1 2)))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 4) '(1)))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 5) '()))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 6) '()))
	(test-t (equal (nbutlast (list* 1 2 3 4 5 6) 7) '()))
	(test-t (let* ((a '(1 2 3 4 5)) (b (nbutlast a 3))) (and (eq a b) (equal a '(1 2)))))
	(test-t (let ((x '(0 1 2 3 4 5 6 7 8 9))) (eq (last x) (nthcdr 9 x))))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 0) nil)))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x) (nthcdr 4 x))))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 1) (nthcdr 4 x))))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 2) (cdddr x))))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 3) (cddr x))))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 4) (cdr x))))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 5) x)))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 6) x)))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 7) x)))
	(test-t (let ((x '(0 1 2 3 4))) (eq (last x 8) x)))
	(test-t (tailp '() '()))
	(test-t (tailp '() '(1)))
	(test-t (tailp '() '(1 2 3 4 5 6 7 8 9)))
	(test-t (let ((x '(1 2 3))) (and (tailp x x) (tailp (cdr x) x) (tailp (cddr x) x) (tailp (cdddr x) x))))
	(test-t (let ((x '(1 . 2))) (and (tailp x x) (tailp (cdr x) x))))
	(test-t (not (tailp 'x '(1 2 3 4 5 6))))
	(test-t (not (tailp (list 1 2 3) '(1 2 3))))
	(test-t (define (ldiff . args) #f))
	(test-t (null (ldiff '() '())))
	(test-t (equal (ldiff '(1 . 2) 2) '(1)))
	(test-t (equal (ldiff '(1 2 3 4 5 6 7 8 9) '()) '(1 2 3 4 5 6 7 8 9)))
	(test-t (let ((x '(1 2 3)))
		  (and (null (ldiff x x))
		       (equal (ldiff x (cdr x)) '(1))
		       (equal (ldiff x (cddr x)) '(1 2))
		       (equal (ldiff x (cdddr x)) '(1 2 3)))))
	(test-t (let* ((x '(1 2 3))
		       (y '(a b c))
		       (z (ldiff x y)))
		  (and (not (eq x z))
		       (equal z '(1 2 3)))))
	(test-t (equal (cl-member 'a '(a b c d)) '(a b c d)))
	(test-t (equal (cl-member 'b '(a b c d)) '(b c d)))
	(test-t (equal (cl-member 'c '(a b c d)) '(c d)))
	(test-t (equal (cl-member 'd '(a b c d)) '(d)))
	(test-t (equal (cl-member 'e '(a b c d)) '()))
	(test-t (equal (cl-member 'f '(a b c d)) '()))
	(test-t (let ((x '(a b c d)))
		  (eq (cl-member 'a x) x)
		  (eq (cl-member 'b x) (cdr x))
		  (eq (cl-member 'c x) (cddr x))
		  (eq (cl-member 'd x) (cdddr x))
		  (eq (cl-member 'e x) nil)))
	(test-t (equal (cl-member 'a '(a b c d) :test eq) '(a b c d)))
	(test-t (equal (cl-member 'b '(a b c d) :test eq) '(b c d)))
	(test-t (equal (cl-member 'c '(a b c d) :test eq) '(c d)))
	(test-t (equal (cl-member 'd '(a b c d) :test eq) '(d)))
	(test-t (equal (cl-member 'e '(a b c d) :test eq) '()))
	(test-t (equal (cl-member 'f '(a b c d) :test eq) '()))
	(test-t (null (cl-member 'a '())))
	(test-t (let* ((x '((1 . a) (2 . b) (3 . c) (4 . d) (5 . e)))
		       (y (cl-member 'd x :key cdr :test eq)))
		  (and (equal y '((4 . d) (5 . e)))
		       (eq y (nthcdr 3 x)))))
	(test-t (let* ((x '((1 . a) (2 . b) (3 . c) (4 . d) (5 . e)))
		       (y (cl-member 'd x :key cdr)))
		  (and (equal y '((4 . d) (5 . e)))
		       (eq y (nthcdr 3 x)))))
	(test-t (equal (member-if (lambda (x) (eql x 'a)) '(a b c d)) '(a b c d)))
	(test-t (equal (member-if (lambda (x) (eql x 'b)) '(a b c d)) '(b c d)))
	(test-t (equal (member-if (lambda (x) (eql x 'c)) '(a b c d)) '(c d)))
	(test-t (equal (member-if (lambda (x) (eql x 'd)) '(a b c d)) '(d)))
	(test-t (equal (member-if (lambda (x) (eql x 'e)) '(a b c d)) '()))
	(test-t (equal (member-if (lambda (x) (eql x 'f)) '(a b c d)) '()))
	(test-t (null (member-if (lambda (x) (eql x 'a)) '())))
	(test-t (let* ((x '((1 . a) (2 . b) (3 . c) (4 . d) (5 . e)))
		       (y (member-if (lambda (p) (eq p 'd)) x :key cdr)))
		  (and (equal y '((4 . d) (5 . e)))
		       (eq y (nthcdr 3 x)))))
	(test-t (null (member-if zerop '(7 8 9))))
	(test-t (equal (member-if-not (lambda (x) (not (eql x 'a))) '(a b c d)) '(a b c d)))
	(test-t (equal (member-if-not (lambda (x) (not (eql x 'b))) '(a b c d)) '(b c d)))
	(test-t (equal (member-if-not (lambda (x) (not (eql x 'c))) '(a b c d)) '(c d)))
	(test-t (equal (member-if-not (lambda (x) (not (eql x 'd))) '(a b c d)) '(d)))
	(test-t (equal (member-if-not (lambda (x) (not (eql x 'e))) '(a b c d)) '()))
	(test-t (equal (member-if-not (lambda (x) (not (eql x 'f))) '(a b c d)) '()))
	(test-t (null (member-if-not (lambda (x) (not (eql x 'a))) '())))
	
	(test-t (let* ((x '((1 . a) (2 . b) (3 . c) (4 . d) (5 . e)))
		       (y (member-if-not (lambda (p) (not (eq p 'd))) x :key cdr)))
		  (and (equal y '((4 . d) (5 . e)))
		       (eq y (nthcdr 3 x)))))
	
	(test-t (let ((dummy nil)
		      (list-1 '(1 2 3 4)))
		  (and (eq (mapc (lambda x (setq dummy (append dummy x)))
				 list-1
				 '(a b c d e)
				 '(x y z))
			   list-1)
		       (equal dummy '(1 a x 2 b y 3 c z)))))
	(test-t (let* ((x '(0 1 2 3))
		       (y nil)
		       (z (mapc (lambda (a b c) (push (list a b c) y))
				x '(1 2 3 4) '(2 3 4 5))))
		  (and (eq z x)
		       (equal y '((3 4 5) (2 3 4) (1 2 3) (0 1 2))))))
	(test-t (let* ((x '(0 1 2 3))
		       (y nil)
		       (z (mapc (lambda (a b c) (push (list a b c) y))
				nil x '(1 2 3 4) '(2 3 4 5))))
		  (and (null z)
		       (null y))))
	(test-t (let ((sum 0))
		  (mapc (lambda rest (setq sum (+ sum (apply + rest))))
			'(0 1 2)
			'(1 2 0)
			'(2 0 1))
		  (eql sum 9)))
	(test-t (let ((result 'initial-value)
		      (list-1 nil))
		  (and (eq (mapc (lambda (a b) (setq result (cons (cons a b) result))) list-1) list-1)
		       (eq result 'initial-value))))
	(test-t (let ((result 'initial-value)
		      (list-1 nil))
		  (and (eq (mapc (lambda (a b) (setq result (cons (cons a b) result)))
				 list-1
				 '(1 2 3))
			   list-1)
		       (eq result 'initial-value))))
	(test-t (let ((result 'initial-value)
		      (list-1 '(1 2 3)))
		  (and (eq (mapc (lambda (a b) (setq result (cons (cons a b) result)))
				 list-1
				 '())
			   list-1)
		       (eq result 'initial-value))))
	(test-t (equal (mapcar car '((1 2) (2 3) (3 4) (4 5))) '(1 2 3 4)))
	(test-t (null (mapcar identity '())))
	(test-t (equal (mapcar list '(0 1 2 3) '(a b c d) '(w x y z)) '((0 a w) (1 b x) (2 c y) (3 d z))))
	(test-t (null (mapcar list '() '(0 1 2 3) '(1 2 3 4) '(2 3 4 5))))
	(test-t (null (mapcar list '(0 1 2 3) '() '(1 2 3 4) '(2 3 4 5))))
	(test-t (null (mapcar list '(0 1 2 3) '(1 2 3 4) '() '(2 3 4 5))))
	(test-t (null (mapcar list '(0 1 2 3) '(1 2 3 4) '(2 3 4 5) '())))
	(test-t (equal (mapcar list '(0) '(a b) '(x y z)) '((0 a x))))
	(test-t (equal (mapcar list '(a b) '(0) '(x y z)) '((a 0 x))))
	(test-t (equal (mapcar list '(a b) '(x y z) '(0)) '((a x 0))))
	(test-t (equal (mapcar cons '(a b c) '(1 2 3)) '((a . 1) (b . 2) (c . 3))))
	(test-t (equal (mapcan cdr (copy-tree '((1 2) (2 3) (3 4) (4 5)))) '(2 3 4 5)))
	(test-t (equal (mapcan append
			       '((1 2 3) (4 5 6) (7 8 9))
			       '((a) (b c) (d e f))
			       (list (list 'x 'y 'z) (list 'y 'z) (list 'z)))
		       '(1 2 3 a x y z 4 5 6 b c y z 7 8 9 d e f z)))
	(test-t (null (mapcan append '((1 2 3) (4 5 6) (7 8 9)) '((a) (b c)) '())))
	(test-t (null (mapcan append '((1 2 3) (4 5 6) (7 8 9)) '() '((a) (b c)))))
	(test-t (null (mapcan append '() '((1 2 3) (4 5 6) (7 8 9)) '((a) (b c)))))
	(test-t (equal (mapcan list
			       (list 1 2 3 4 5)
			       (list 2 3 4 5 6)
			       (list 3 4 5 6 7)
			       (list 4 5 6 7 8))
		       '(1 2 3 4 2 3 4 5 3 4 5 6 4 5 6 7 5 6 7 8)))
	(test-t (equal (mapcan (lambda (x y) (if (null x) '() (list x y)))
			       '(() () () d e)
			       '(1 2 3 4 5 6))
		       '(d 4 e 5)))
	(test-t (equal (mapcan (lambda (x) (if (numberp x) (list x) '()))
			       '(a 1 b c 3 4 d 5))
		       '(1 3 4 5)))
	(test-t (equal (maplist identity '(a b c d)) '((a b c d) (b c d) (c d) (d))))
	(test-t (equal (maplist car '((1 2) (2 3) (3 4) (4 5))) '((1 2) (2 3) (3 4) (4 5))))
	(test-t (equal (maplist list '(a b c) '(b c d) '(c d e))
		       '(((a b c) (b c d) (c d e))
			 ((b c) (c d) (d e))
			 ((c) (d) (e)))))
	(test-t (equal (maplist append '(a b c) '(b c d) '(c d e)) '((a b c b c d c d e) (b c c d d e) (c d e))))
	(test-t (equal (maplist append '(a b c) '(b c) '(c)) '((a b c b c c))))
	(test-t (null (maplist append '() '(a b c) '(b c) '(c))))
	(test-t (null (maplist append '(a b c) '() '(b c) '(c))))
	(test-t (null (maplist append '(a b c) '(b c) '(c) '())))
	(test-t (let ((x '((1 2) (2 3) (3 4) (4 5)))
		      (y nil))
		  (and (eq (mapl (lambda (a) (push (car a) y)) x) x)
		       (equal y '((4 5) (3 4) (2 3) (1 2))))))
	(test-t (let ((x nil))
		  (and (null (mapl (lambda rest (push rest x)) '() '(0) '(0 1)))
		       (null x))))
	(test-t (let ((x nil))
		  (and (equal (mapl (lambda rest (push rest x)) '(0) '() '(0 1))
			      '(0))
		       (null x))))
	(test-t (let ((x nil))
		  (and (equal (mapl (lambda rest (push rest x)) '(0) '(0 1) '())
			      '(0))
		       (null x))))
	(test-t (equal (mapcon car (copy-tree '((1 2) (2 3) (3 4) (4 5)))) '(1 2 2 3 3 4 4 5)))
	(test-t (equal (mapcon list '(0 1 2 3) '(1 2 3 4) '(2 3 4 5) '(3 4 5 6))
		       '((0 1 2 3) (1 2 3 4) (2 3 4 5) (3 4 5 6) (1 2 3) (2 3 4) (3 4 5)
			 (4 5 6) (2 3) (3 4) (4 5) (5 6) (3) (4) (5) (6))))
	(test-t (null (mapcon list '() '(0 1 2 3) '(1 2 3 4) '(2 3 4 5) '(3 4 5 6))))
	(test-t (null (mapcon list '(0 1 2 3) '() '(1 2 3 4) '(2 3 4 5) '(3 4 5 6))))
	(test-t (null (mapcon list '(0 1 2 3) '(1 2 3 4) '() '(2 3 4 5) '(3 4 5 6))))
	(test-t (null (mapcon list '(0 1 2 3) '(1 2 3 4) '(2 3 4 5) '() '(3 4 5 6))))
	(test-t (null (mapcon list '(0 1 2 3) '(1 2 3 4) '(2 3 4 5) '(3 4 5 6) '())))
	(test-t (let* ((x '((apple . 1) (orange . 2) (grapes . 3)))
		       (y (acons 'plum 9 x)))
		  (and (equal y '((plum . 9) (apple . 1) (orange . 2) (grapes . 3)))
		       (eq x (cdr y)))))
	(test-t (equal (acons 'a '0 nil) '((a . 0))))
	(test-t (equal (acons 'apple 1 (acons 'orange 2 (acons 'grapes '3 nil))) '((apple . 1) (orange . 2) (grapes . 3))))
	(test-t (equal (acons () () ()) '((()))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (cl-assoc 'y alist) (cadr alist))))
	(test-t (null (cl-assoc 'no-such-key '((x . 100) (y . 200) (z . 50)))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (cl-assoc 'y alist :test eq) (cadr alist))))
	(test-t (null (cl-assoc 'key '())))
	(test-t (null (cl-assoc '() '(()))))
	(test-t (null (cl-assoc '() '(() ()))))
	(test-t (let ((alist '(() () () (x . 100) (y . 200) (z . 50)))) (eq (cl-assoc 'y alist) (car (cddddr alist)))))
	(test-t (let ((alist '((1 . a) () (2 . b) (())))) (eq (cl-assoc '() alist) (cadddr alist))))
	(test-t (let ((alist '((x . 100) (y . 200) (x . 100) (z . 50)))) (eq (cl-assoc 'y alist) (cadr alist))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (assoc-if (lambda (arg) (eq arg 'y)) alist) (cadr alist))))
	(test-t (null (assoc-if consp '((x . 100) (y . 200) (z . 50)))))
	(test-t (null (assoc-if (lambda (x) (eq x 'key)) '())))
	(test-t (null (assoc-if identity '(()))))
	(test-t (null (assoc-if identity '(() ()))))
	(test-t (let ((alist '(() () () (x . 100) (y . 200) (z . 50)))) (eq (assoc-if (lambda (arg) (eq arg 'y)) alist) (car (cddddr alist)))))
	(test-t (let ((alist '((1 . a) () (2 . b) (())))) (eq (assoc-if (lambda (arg) (null arg)) alist) (cadddr alist))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (assoc-if-not (lambda (arg) (not (eq arg 'y))) alist) (cadr alist))))
	(test-t (null (assoc-if-not (complement consp) '((x . 100) (y . 200) (z . 50)))))
	(test-t (null (assoc-if-not (lambda (x) (not (eq x 'key))) '())))
	(test-t (null (assoc-if-not identity '(()))))
	(test-t (null (assoc-if-not identity '(() ()))))
	(test-t (let ((alist '(() () () (x . 100) (y . 200) (z . 50))))
		  (eq (assoc-if-not (lambda (arg) (not (eq arg 'y))) alist)
		      (car (cddddr alist)))))
	(test-t (equal (copy-alist '((a . 10) (b . 100) (c . 1000)))
		       '((a . 10) (b . 100) (c . 1000))))
	(test-t (let* ((alist '((a . 10) (b . 100) (c . 1000)))
		       (copy (copy-alist alist)))
		  (and (not (eq alist copy))
		       (not (eq (cdr alist) (cdr copy)))
		       (not (eq (cddr alist) (cddr copy)))
		       (not (eq (car alist) (car copy)))
		       (not (eq (cadr alist) (cadr copy)))
		       (not (eq (caddr alist) (caddr copy))))))
	(test-t (let* ((alist '((a 10 x) (b 100 y) (c 1000 z)))
		       (copy (copy-alist alist)))
		  (and (not (eq alist copy))
		       (not (eq (cdr alist) (cdr copy)))
		       (not (eq (cddr alist) (cddr copy)))
		       (not (eq (car alist) (car copy)))
		       (not (eq (cadr alist) (cadr copy)))
		       (not (eq (caddr alist) (caddr copy)))
		       (eq (cdar alist) (cdar copy))
		       (eq (cdadr alist) (cdadr copy))
		       (eq (cdaddr alist) (cdaddr copy)))))
	(test-t (let* ((alist (pairlis '(x y z) '(xx yy zz) '((a . aa) (b . bb)))))
		  (and (equal (cl-assoc 'x alist) '(x . xx))
		       (equal (cl-assoc 'y alist) '(y . yy))
		       (equal (cl-assoc 'z alist) '(z . zz))
		       (equal (cl-assoc 'a alist) '(a . aa))
		       (equal (cl-assoc 'b alist) '(b . bb))
		       (null (cl-assoc 'key alist)))))
	(test-t (let* ((alist (pairlis '(x y z) '(xx yy zz))))
		  (and (equal (cl-assoc 'x alist) '(x . xx))
		       (equal (cl-assoc 'y alist) '(y . yy))
		       (equal (cl-assoc 'z alist) '(z . zz))
		       (null (cl-assoc 'key alist)))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (rassoc '200 alist) (cadr alist))))
	(test-t (null (rassoc 'no-such-datum '((x . 100) (y . 200) (z . 50)))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (rassoc '200 alist :test =) (cadr alist))))
	(test-t (null (rassoc 'key '())))
	(test-t (null (rassoc '() '(()))))
	(test-t (null (rassoc '() '(() ()))))
	(test-t (let ((alist '(() () () (x . 100) (y . 200) (z . 50)))) (eq (rassoc '200 alist) (car (cddddr alist)))))
	(test-t (let ((alist '((1 . a) () (2 . b) (())))) (eq (rassoc '() alist) (cadddr alist))))
	(test-t (let ((alist '((x . 100) (y . 200) (x . 100) (z . 50)))) (eq (rassoc '200 alist) (cadr alist))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (rassoc-if (lambda (arg) (= arg 200)) alist) (cadr alist))))
	(test-t (null (rassoc-if consp '((x . 100) (y . 200) (z . 50)))))
	(test-t (null (rassoc-if (lambda (x) (eq x 'key)) '())))
	(test-t (null (rassoc-if identity '(()))))
	(test-t (null (rassoc-if identity '(() ()))))
	(test-t (let ((alist '(() () () (x . 100) (y . 200) (z . 50)))) (eq (rassoc-if (lambda (arg) (= arg 200)) alist) (car (cddddr alist)))))
	(test-t (let ((alist '((1 . a) () (2 . b) (())))) (eq (rassoc-if (lambda (arg) (null arg)) alist) (cadddr alist))))
	(test-t (let ((alist '((x . 100) (y . 200) (z . 50)))) (eq (rassoc-if-not (lambda (arg) (not (= arg 200))) alist) (cadr alist))))
	(test-t (null (rassoc-if-not (complement consp) '((x . 100) (y . 200) (z . 50)))))
	(test-t (null (rassoc-if-not (lambda (x) (not (eq x 'key))) '())))
	(test-t (null (rassoc-if-not identity '(()))))
	(test-t (null (rassoc-if-not identity '(() ()))))
	(test-t (let ((alist '(() () () (x . 100) (y . 200) (z . 50))))
		  (eq (rassoc-if-not (lambda (arg) (not (= arg 200))) alist)
		      (car (cddddr alist)))))
	(let ((list1 (list 1 1 2 3 4 'a 'b 'c "A" "B" "C" "d"))
	      (list2 (list 1 4 5 'b 'c 'd "a" "B" "c" "D")))
	  (test-t (null (set-exclusive-or (intersection list1 list2) '(c b 4 1 1))))
	  (test-t (null (set-exclusive-or (intersection list1 list2 :test equal)
					  '("B" c b 4 1 1)
					  :test equal)))
	  (test-t (null (set-exclusive-or (intersection list1 list2 :test equalp)
					  '("d" "C" "B" "A" c b 4 1 1)
					  :test equalp))))
	(test-t (null (intersection '(0 1 2) '())))
	(test-t (null (intersection '() '())))
	(test-t (null (intersection '() '(0 1 2))))
	(test-t (equal (intersection '(0) '(0)) '(0)))
	(test-t (equal (intersection '(0 1 2 3) '(2)) '(2)))
	(test-t (cl-member 0 (intersection '(0 0 0 0 0) '(0 1 2 3 4 5))))
	(test-t (null (set-exclusive-or (intersection '(0 1 2 3 4) '(4 3 2 1 0)) '(4 3 2 1 0))))
	(test-t (null (set-exclusive-or (intersection '(0 1 2 3 4) '(0 1 2 3 4)) '(0 1 2 3 4))))
	(test-t (null (set-exclusive-or (intersection '(0 1 2 3 4) '(4 3 2 1 0)) '(0 1 2 3 4))))
	(test-t (let ((list1 (list "A" "B" "C" "d" "e" "F" "G" "h"))
		      (list2 (list "a" "B" "c" "D" "E" "F" "g" "h")))
		  (null (set-exclusive-or (intersection list1 list2
							:test char=
							:key (lambda (x) (char x 0)))
					  '("B" "F" "h")
					  :test char=
					  :key (lambda (x) (char x 0))))))
	(test-t (let ((list1 (list "A" "B" "C" "d" "e" "F" "G" "h"))
		      (list2 (list "a" "B" "c" "D" "E" "F" "g" "h")))
		  (null (set-exclusive-or (intersection list1 list2
							:test char-equal
							:key (lambda (x) (char x 0)))
					  '("A" "B" "C" "d" "e" "F" "G" "h")
					  :test char-equal
					  :key (lambda (x) (char x 0))))))
	(test-t (let ((list1 (list "A" "B" "C" "d"))
		      (list2 (list "D" "E" "F" "g" "h")))
		  (null (set-exclusive-or (intersection list1 list2
							:test char-equal
							:key (lambda (x) (char x 0)))
					  '("d")
					  :test char-equal
					  :key (lambda (x) (char x 0))))))
	(let ((list1 (list 1 1 2 3 4 'a 'b 'c "A" "B" "C" "d"))
	      (list2 (list 1 4 5 'b 'c 'd "a" "B" "c" "D")))
	  (test-t (null (set-exclusive-or (nintersection (copy-list list1) list2) '(c b 4 1 1))))
	  (test-t (null (set-exclusive-or (nintersection (copy-list list1) list2 :test equal)
					  '("B" c b 4 1 1)
					  :test equal)))
	  (test-t (null (set-exclusive-or (nintersection (copy-list list1) list2 :test equalp)
					  '("d" "C" "B" "A" c b 4 1 1)
					  :test equalp))))
	(test-t (null (nintersection (list 0 1 2) '())))
	(test-t (null (nintersection '() '())))
	(test-t (null (nintersection '() '(0 1 2))))
	(test-t (equal (nintersection (list 0) '(0)) '(0)))
	(test-t (equal (nintersection (list 0 1 2 3) '(2)) '(2)))
	(test-t (cl-member 0 (nintersection (list 0 0 0 0 0) '(0 1 2 3 4 5))))
	(test-t (null (set-exclusive-or (nintersection (list 0 1 2 3 4) '(4 3 2 1 0)) '(4 3 2 1 0))))
	(test-t (null (set-exclusive-or (nintersection (list 0 1 2 3 4) '(0 1 2 3 4)) '(0 1 2 3 4))))
	(test-t (null (set-exclusive-or (nintersection (list 0 1 2 3 4) '(4 3 2 1 0)) '(0 1 2 3 4))))
	(test-t (let ((list1 (list "A" "B" "C" "d" "e" "F" "G" "h"))
		      (list2 (list "a" "B" "c" "D" "E" "F" "g" "h")))
		  (null (set-exclusive-or (nintersection list1 list2
							 :test char=
							 :key (lambda (x) (char x 0)))
					  '("B" "F" "h")
					  :test char=
					  :key (lambda (x) (char x 0))))))
	(test-t (let ((list1 (list "A" "B" "C" "d" "e" "F" "G" "h"))
		      (list2 (list "a" "B" "c" "D" "E" "F" "g" "h")))
		  (null (set-exclusive-or (nintersection list1 list2
							 :test char-equal
							 :key (lambda (x) (char x 0)))
					  '("A" "B" "C" "d" "e" "F" "G" "h")
					  :test char-equal
					  :key (lambda (x) (char x 0))))))
	(test-t (let ((list1 (list "A" "B" "C" "d"))
		      (list2 (list "D" "E" "F" "g" "h")))
		  (null (set-exclusive-or (nintersection list1 list2
							 :test char-equal
							 :key (lambda (x) (char x 0)))
					  '("d")
					  :test char-equal
					  :key (lambda (x) (char x 0))))))
	(test-t (let ((set '(a b c))) (eq (adjoin 'a set) set)))
	(test-t (let* ((set '(a b c)) (new-set (adjoin 'x set))) (and (equal new-set '(x a b c)) (eq set (cdr new-set)))))
	(test-t (equal (adjoin 1 nil) '(1)))
	(test-t (let ((set '((test-item 1)))) (equal (adjoin '(test-item 1) set) '((test-item 1) (test-item 1)))))
	(test-t (let ((set '((test-item 1)))) (equal (adjoin '(test-item 1) set) '((test-item 1) (test-item 1)))))
	(test-t (let ((set '((test-item 1)))) (eq (adjoin '(test-item 1) set :test equal) set)))
	(test-t (let ((set '((test-item 1)))) (eq (adjoin '(test-item) set :key car) set)))
	(test-t (let ((set '((test-item 1)))) (eq (adjoin '(test-item) set :key car :test eq) set)))
	(test-t (let ((set '(("test-item" 1)))) (eq (adjoin '("test-item") set :key car :test equal) set)))
	(test-t (let ((place nil)) (and (equal (pushnew 'a place) '(a)) (equal place '(a)))))
	(test-t (let ((place nil)) (and (equal (pushnew 'a place) '(a)) (equal place '(a)))))
	(test-t (let ((place '("love" "peace")))
		  (equal (pushnew "war" place :test equal) '("war" "love" "peace"))))
	(test-t (let ((place '("love" "peace")))
		  (and (eq (pushnew "peace" place :test equal) place)
		       (equal place '("love" "peace")))))
	(test-t (let ((place '(("love" . l) ("peace" . p))))
		  (equal (pushnew '("war" . w) place :test equal :key car)
			 '(("war" . w) ("love" . l) ("peace" . p)))))
	(test-t (let ((place '(("love" . l) ("peace" . p))))
		  (and (eq (pushnew '("love" . l) place :test equal :key car) place)
		       (equal place '(("love" . l) ("peace" . p))))))
	(test-t (let* ((list '((1) (1 2) (1 2 3)))
		       (original list))
		  (and (equal (pushnew '(1) list :test equal) '((1) (1 2) (1 2 3)))
		       (eq list original))))
	(test-t (let* ((list '((1) (1 2) (1 2 3)))
		       (original list))
		  (and (equal (pushnew '(1) list :test equal :key nil) '((1) (1 2) (1 2 3)))
		       (eq list original))))
	(test-t (null (set-difference (set-difference '(1 2 3 4 5 6 7 8 9)
						      '(2 4 6 8))
				      '(1 3 5 7 9))))
	(test-t (null (nset-difference (set-difference (list 1 2 3 4 5 6 7 8 9)
						       '(2 4 6 8))
				       '(1 3 5 7 9))))
	(test-t (null (set-difference '() '())))
	(test-t (null (set-difference '() '() :test equal :key 'identity)))
	(test-t (null (nset-difference '() '())))
	(test-t (null (set-difference '() '(1 2 3))))
	(test-t (null (set-difference '() '(1 2 3) :test equal :key 'identity)))
	(test-t (null (nset-difference '() '(1 2 3))))
	(test-t (null (set-difference '(1 2 3 4) '(4 3 2 1))))
	(test-t (null (nset-difference (list 1 2 3 4) '(4 3 2 1))))
	(test-t (null (set-difference '(1 2 3 4) '(2 4 3 1))))
	(test-t (null (nset-difference (list 1 2 3 4) '(2 4 3 1))))
	(test-t (null (set-difference '(1 2 3 4) '(1 3 4 2))))
	(test-t (null (nset-difference (list 1 2 3 4) '(1 3 4 2))))
	(test-t (null (set-difference '(1 2 3 4) '(1 3 2 4))))
	(test-t (null (nset-difference (list 1 2 3 4) '(1 3 2 4))))
	(test-t (eq (set-difference (set-difference '(1 2 3) '()) '(1 2 3)) '()))
	(test-t (eq (nset-difference (nset-difference (list 1 2 3) '()) '(1 2 3)) '()))
	(test-t (eq (set-difference (set-difference '(1 2 3) '(1)) '(2 3)) '()))
	(test-t (eq (nset-difference (nset-difference (list 1 2 3) '(1)) '(2 3)) '()))
	(test-t (eq (set-difference (set-difference '(1 2 3) '(1 2)) '(3)) '()))
	(test-t (eq (nset-difference (nset-difference (list 1 2 3) '(1 2)) '(3)) '()))
	(test-t (null (set-exclusive-or (set-exclusive-or '(1 2 3) '(2 3 4)) '(1 4))))
	(test-t (null (nset-exclusive-or (nset-exclusive-or (list 1 2 3) '(2 3 4)) '(1 4))))
	(test-t (null (set-exclusive-or (set-exclusive-or '(1 2 3) '(1 3)) '(2))))
	(test-t (null (nset-exclusive-or (nset-exclusive-or (list 1 2 3) '(1 3)) '(2))))
	(test-t (null (set-exclusive-or '() '())))
	(test-t (null (nset-exclusive-or '() '())))
	(test-t (null (set-exclusive-or '(1 2 3) '(3 2 1))))
	(test-t (null (nset-exclusive-or (list 1 2 3) '(3 2 1))))
	(test-t (null (set-exclusive-or '(1 2 3) '(2 3 1))))
	(test-t (null (nset-exclusive-or (list 1 2 3) '(2 3 1))))
	(test-t (null (set-exclusive-or '(1 2 3) '(1 3 2))))
	(test-t (null (nset-exclusive-or (list 1 2 3) '(1 3 2))))
	(test-t (null (set-exclusive-or (set-exclusive-or '(1 2 3) '()) '(3 2 1))))
	(test-t (null (nset-exclusive-or (nset-exclusive-or (list 1 2 3) '()) '(3 2 1))))
	(test-t (null (set-exclusive-or (set-exclusive-or '() '(1 2 3)) '(2 1 3))))
	(test-t (null (nset-exclusive-or (nset-exclusive-or '() '(1 2 3)) '(2 1 3))))
	(test-t (null (set-exclusive-or '("car" "ship" "airplane" "submarine")
					'("car" "ship" "airplane" "submarine")
					:test equal)))
	(test-t (null (nset-exclusive-or (copy-list '("car" "ship" "airplane" "submarine"))
					 '("car" "ship" "airplane" "submarine")
					 :test equal)))
	(test-t (null (set-exclusive-or '("car" "ship" "airplane" "submarine")
					'("CAR" "SHIP" "AIRPLANE" "SUBMARINE")
					:test equalp)))
	(test-t (null (nset-exclusive-or (copy-list '("car" "ship" "airplane" "submarine"))
					 '("CAR" "SHIP" "AIRPLANE" "SUBMARINE")
					 :test equalp)))
	(test-t (null (set-exclusive-or '(("car") ("ship") ("airplane") ("submarine"))
					'(("car") ("ship") ("airplane") ("submarine"))
					:test string=
					:key car)))
	(test-t (null (nset-exclusive-or (copy-tree
					  '(("car") ("ship") ("airplane") ("submarine")))
					 '(("car") ("ship") ("airplane") ("submarine"))
					 :test string=
					 :key car)))
	(test-t (subsetp '(1 2 3) '(1 2 3)))
	(test-t (subsetp '(1 2 3) '(3 2 1)))
	(test-t (subsetp '(1 2 3) '(2 1 3)))
	(test-t (null (subsetp '(1 2 3 4) '(2 1 3))))
	(test-t (subsetp '(1) '(2 1 3)))
	(test-t (subsetp '(1 2) '(1 2 3 4 5 6 7 8)))
	(test-t (subsetp '(1 2 3 4 5) '(8 7 6 5 4 3 2 1)))
	(test-t (null (subsetp '("car" "ship" "airplane" "submarine") '("car" "ship" "horse" "airplane" "submarine" "camel"))))
	(test-t (subsetp '("car" "ship" "airplane" "submarine")
			 '("car" "ship" "horse" "airplane" "submarine" "camel")
			 :test equal))
	(test-t (subsetp '("CAR" "SHIP" "AIRPLANE" "SUBMARINE")
			 '("car" "ship" "horse" "airplane" "submarine" "camel")
			 :test equalp))
	(test-t (subsetp '(("car") ("ship") ("airplane") ("submarine"))
			 '(("car") ("ship") ("horse") ("airplane") ("submarine") ("camel"))
			 :test string=
			 :key car))
	(test-t (null (union '() '())))
	(test-t (null (nunion '() '())))
	(test-t (null (set-difference (union '(1 2 3) '(2 3 4)) '(1 2 3 4))))
	(test-t (null (set-difference (nunion (list 1 2 3) (list 2 3 4)) '(1 2 3 4))))
	(test-t (null (set-difference (union '(1 2 3) '(1 2 3)) '(1 2 3))))
	(test-t (null (set-difference (nunion (list 1 2 3) (list 1 2 3)) '(1 2 3))))
	(test-t (null (set-difference (union '(1) '(3 2 1)) '(1 2 3))))
	(test-t (null (set-difference (nunion (list 1) (list 3 2 1)) '(1 2 3))))
	(test-t (null (set-difference (union '(1 2 3) '()) '(1 2 3))))
	(test-t (null (set-difference (nunion (list 1 2 3) '()) '(1 2 3))))
	(test-t (null (set-difference (union '() '(1 2 3)) '(1 2 3))))
	(test-t (null (set-difference (nunion '() (list 1 2 3)) '(1 2 3))))
	(test-t (null (set-difference (union '(1 2 3) '(2)) '(1 2 3))))
	(test-t (null (set-difference (nunion (list 1 2 3) (list 2)) '(1 2 3))))
	
	(test-t (eql (length "abc") 3))
	(test-t (zerop (length "")))
	(test-t (zerop (length #())))
	(test-t (zerop (length ())))
	(test-t (eql (length '(0)) 1))
	(test-t (eql (length '(0 1)) 2))
	(test-t (eql (length '(0 1 2)) 3))
	(test-t (eql (length '(0 1 2 3)) 4))
	(test-t (eql (length '(0 1 2 3 4)) 5))
	(test-t (eql (length '(0 1 2 3 4 5)) 6))
	(test-t (eql (length '(0 1 2 3 4 5 6)) 7))
	(test-t (eql (length #(0)) 1))
	(test-t (eql (length #(0 1)) 2))
	(test-t (eql (length #(0 1 2)) 3))
	(test-t (eql (length #(0 1 2 3)) 4))
	(test-t (eql (length #(0 1 2 3 4)) 5))
	(test-t (eql (length #(0 1 2 3 4 5)) 6))
	(test-t (eql (length #(0 1 2 3 4 5 6)) 7))
	(test-t (eql (length (make-array 100)) 100))
	(test-t (eql (length (make-sequence 'list 20)) 20))
	(test-t (eql (length (make-sequence 'string 10)) 10))
	(test-t (eql (length (make-sequence 'bit-vector 3)) 3))
	(test-t (eql (length (make-sequence 'bit-vector 64)) 64))
	(test-t (eql (length (make-sequence 'simple-vector 64)) 64))
	(test-t (string= (copy-seq "love") "love"))
	(test-t (equalp (copy-seq '#(a b c d)) '#(a b c d)))
	(test-t (equal (copy-seq '(love)) '(love)))
	(test-t (equal (copy-seq '(love hate war peace)) '(love hate war peace)))
	(test-t (null (copy-seq nil)))
	(test-t (string= (copy-seq "") ""))
	(test-t (let* ((seq0 "love&peace") (seq (copy-seq seq0))) (and (not (eq seq0 seq)) (string= seq0 seq))))
	(test-t (let* ((seq0 (list 'love 'and 'peace)) (seq (copy-seq seq0))) (and (not (eq seq0 seq)) (equal seq0 seq))))
	(test-t (let* ((c0 (list 'love))
		       (c1 (list 'peace))
		       (seq (copy-seq (list c0 c1))))
		  (and (equal seq '((love) (peace)))
		       (eq (car seq) c0)
		       (eq (cadr seq) c1))))
	(test-t (let* ((seq0 '#(t nil t nil))
		       (seq (copy-seq seq0)))
		  (and (not (eq seq0 seq))
		       (equalp seq seq0))))
	(test-t (vectorp (copy-seq (vector))))
	(test-t (simple-vector-p (copy-seq (vector))))
	(test-t (simple-vector-p (copy-seq (vector 0 1))))
	(test-t (simple-string-p (copy-seq "xyz")))
	(test-t (char= (elt "0123456789" 6) #\6))
	(test-t (eq (elt #(a b c d e f g) 0) 'a))
	(test-t (eq (elt '(a b c d e f g) 4) 'e))
	(test-t (let ((str (copy-seq "0123456789"))) (and (char= (elt str 6) #\6) (setf (elt str 0) #\#) (string= str "#123456789"))))
	(test-t (let ((list (list 0 1 2 3)))
		  (and (= (elt list 2) 2)
		       (setf (elt list 1) 9)
		       (= (elt list 1) 9)
		       (equal list '(0 9 2 3)))))
	(test-t (let ((vec (vector 'a 'b 'c)))
		  (and (eq (elt vec 0) 'a)
		       (eq (elt vec 1) 'b)
		       (eq (elt vec 2) 'c))))
	(test-t (let ((list (list 0 1 2 3))) (and (eq (cl-fill list nil) list) (every null list))))
	(test-t (let ((vector (vector 'x 'y 'z))) (and (eq (cl-fill vector 'a) vector) (every (lambda (arg) (eq arg 'a)) vector))))
	(test-t (let ((list (list 0 1 2 3))) (and (eq (cl-fill list '9 :start 2) list) (equal list '(0 1 9 9)))))
	(test-t (let ((list (list 0 1 2 3))) (and (eq (cl-fill list '9 :start 1 :end 3) list) (equal list '(0 9 9 3)))))
	(test-t (let ((list (list 0 1 2 3))) (and (eq (cl-fill list '9 :start 1 :end nil) list) (equal list '(0 9 9 9)))))
	(test-t (let ((list (list 0 1 2 3))) (and (eq (cl-fill list '9 :end 1) list) (equal list '(9 1 2 3)))))
	(test-t (let ((vector (vector 0 1 2 3))) (and (eq (cl-fill vector 't :start 3) vector) (equalp vector '#(0 1 2 t)))))
	(test-t (let ((vector (vector 0 1 2 3))) (and (eq (cl-fill vector 't :start 2 :end 4) vector) (equalp vector '#(0 1 t t)))))
	(test-t (let ((vector (vector 0 1 2 3))) (and (eq (cl-fill vector 't :start 2 :end nil) vector) (equalp vector '#(0 1 t t)))))
	(test-t (let ((vector (vector 0 1 2 3))) (and (eq (cl-fill vector 't :end 3) vector) (equalp vector '#(t t t 3)))))
	(test-t (null (make-sequence 'list 0)))
	(test-t (string= (make-sequence 'string 26 :initial-element #\.) ".........................."))
	(test-t (equal (make-sequence 'list 3 :initial-element 'a) '(a a a)))
	(test-t (null (make-sequence 'null 0 :initial-element 'a)))
	(test-t (equalp (make-sequence 'vector 3 :initial-element 'z) '#(z z z)))
	(test-t (string= (make-sequence 'string 4 :initial-element '#\z) "zzzz"))
	(test-t (vectorp (make-sequence 'vector 10)))
	(test-t (string= (subseq "012345" 2) "2345"))
	(test-t (string= (subseq "012345" 3 5) "34"))
	(test-t (equal (subseq '(0 1 2 3) 0) '(0 1 2 3)))
	(test-t (equal (subseq '(0 1 2 3) 1) '(1 2 3)))
	(test-t (equal (subseq '(0 1 2 3) 2) '(2 3)))
	(test-t (equal (subseq '(0 1 2 3) 3) '(3)))
	(test-t (equal (subseq '(0 1 2 3) 4) '()))
	(test-t (equalp (subseq #(a b c d) 0) #(a b c d)))
	(test-t (equalp (subseq #(a b c d) 1) #(b c d)))
	(test-t (equalp (subseq #(a b c d) 2) #(c d)))
	(test-t (equalp (subseq #(a b c d) 3) #(d)))
	(test-t (equalp (subseq #(a b c d) 4) #()))
	(test-t (string= (subseq "0123" 0) "0123"))
	(test-t (string= (subseq "0123" 1) "123"))
	(test-t (string= (subseq "0123" 2) "23"))
	(test-t (string= (subseq "0123" 3) "3"))
	(test-t (string= (subseq "0123" 4) ""))
	(test-t (equal (subseq '(0 1 2 3) 0 4) '(0 1 2 3)))
	(test-t (equal (subseq '(0 1 2 3) 0 nil) '(0 1 2 3)))
	(test-t (let* ((list0 '(0 1 2 3)) (list (subseq list0 0 4))) (and (not (eq list0 list)) (equal list0 list))))
	(test-t (let* ((list0 '(0 1 2 3)) (list (subseq list0 0 nil))) (and (not (eq list0 list)) (equal list0 list))))
	(test-t (equal (subseq '(0 1 2 3) 1 3) '(1 2)))
	(test-t (equal (subseq '(0 1 2 3) 2 2) '()))
	(test-t (equal (subseq '(0 1 2 3) 0 0) '()))
	(test-t (equal (subseq '(0 1 2 3) 1 1) '()))
	(test-t (equal (subseq '(0 1 2 3) 2 2) '()))
	(test-t (equal (subseq '(0 1 2 3) 3 3) '()))
	(test-t (equal (subseq '(0 1 2 3) 4 4) '()))
	(test-t (equalp (subseq #(0 1 2 3) 0 4) #(0 1 2 3)))
	(test-t (equalp (subseq #(0 1 2 3) 0 nil) #(0 1 2 3)))
	(test-t (let* ((vec0 #(0 1 2 3)) (vec (subseq vec0 0 4))) (and (not (eq vec0 vec)) (equalp vec0 vec))))
	(test-t (let* ((vec0 #(0 1 2 3)) (vec (subseq vec0 0 nil))) (and (not (eq vec0 vec)) (equalp vec0 vec))))
	(test-t (equalp (subseq #(0 1 2 3) 1 3) #(1 2)))
	(test-t (equalp (subseq #(0 1 2 3) 2 2) #()))
	(test-t (equalp (subseq #(0 1 2 3) 0 0) #()))
	(test-t (equalp (subseq #(0 1 2 3) 1 1) #()))
	(test-t (equalp (subseq #(0 1 2 3) 2 2) #()))
	(test-t (equalp (subseq #(0 1 2 3) 3 3) #()))
	(test-t (equalp (subseq #(0 1 2 3) 4 4) #()))
	(test-t (string= (cl-map 'string (lambda (x y) (char "01234567890ABCDEF" (mod (+ x y) 16))) '(1 2 3 4) '(10 9 8 7)) "AAAA"))
	(test-t (equal (cl-map 'list - '(1 2 3 4)) '(-1 -2 -3 -4)))
	(test-t (string= (cl-map 'string (lambda (x) (if (oddp x) #\1 #\0)) '(1 2 3 4)) "1010"))
	(test-t (equal (cl-map 'list + '(0 1) '(1 0)) '(1 1)))
	(test-t (equal (cl-map 'list - '(0 1) '(1 0)) '(-1 1)))
	(test-t (every null (list (cl-map 'list + '())
				  (cl-map 'list + '() '())
				  (cl-map 'list + '() '() '())
				  (cl-map 'list + '() '() '() '())
				  (cl-map 'list + '() '() '() '() '()))))
	(test-t (equal (cl-map 'list + '(0 1 2)) '(0 1 2)))
	(test-t (equal (cl-map 'list + '(0 1 2) '(1 2 3)) '(1 3 5)))
	(test-t (equal (cl-map 'list + '(0 1 2) '(1 2 3) '(2 3 4)) '(3 6 9)))
	(test-t (equal (cl-map 'list + '(0 1 2) '(1 2 3) '(2 3 4) '(3 4 5)) '(6 10 14)))
	(test-t (equal (cl-map 'list + '(1 2) '(1 2 3)) '(2 4)))
	(test-t (equal (cl-map 'list + '(0 1 2) '(2 3) '(2 3 4)) '(4 7)))
	(test-t (equal (cl-map 'list + '(0 1 2) '(1 2 3) '(2) '(3 4 5)) '(6)))
	(test-t (equal (cl-map 'list + '(0 1 2) '(1 2 3) '(2 3 4) '(3 4 5) '()) '()))
	(test-t (null (cl-map 'null + '())))
	(test-t (equalp (cl-map 'vector + #()) #()))
	(test-t (equalp (cl-map 'vector + #() #()) #()))
	(test-t (equalp (cl-map 'vector + #() #() #()) #()))
	(test-t (equalp (cl-map 'vector + #() #() #() #()) #()))
	(test-t (equalp (cl-map 'vector + #() #() #() #() #()) #()))
	(test-t (equalp (cl-map 'vector + '() #()) #()))
	(test-t (equalp (cl-map 'vector + '() #() "") #()))
	(test-t (equalp (cl-map 'vector + '(0 1 2)) #(0 1 2)))
	(test-t (equalp (cl-map 'vector + '(0 1 2) #(1 2 3)) #(1 3 5)))
	(test-t (equalp (cl-map 'vector + #(0 1 2) '(1 2 3) #(2 3 4)) #(3 6 9)))
	(test-t (equalp (cl-map 'vector + '(0 1 2) #(1 2 3) '(2 3 4) #(3 4 5)) #(6 10 14)))
	(test-t (equalp (cl-map 'vector + '(1 2) '(1 2 3)) #(2 4)))
	(test-t (equalp (cl-map 'vector + '(0 1 2) '(2 3) '(2 3 4)) #(4 7)))
	(test-t (equalp (cl-map 'vector + '(0 1 2) '(1 2 3) '(2) '(3 4 5)) #(6)))
	(test-t (equalp (cl-map 'vector + '(0 1 2) '(1 2 3) '(2 3 4) '(3 4 5) '()) #()))
	(test-t (equalp (cl-map 'vector + #(1 2) #(1 2 3)) #(2 4)))
	(test-t (equalp (cl-map 'vector + #(0 1 2) #(2 3) #(2 3 4)) #(4 7)))
	(test-t (equalp (cl-map 'vector + #(0 1 2) '(1 2 3) #(2) '(3 4 5)) #(6)))
	(test-t (equalp (cl-map 'vector + '(0 1 2) #(1 2 3) '(2 3 4) '(3 4 5) '()) #()))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "" "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "" "" "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "" "" "" "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "" "" "" "" "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "" '()) ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) "" #() '()) ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) '() '() "" "") ""))
	(test-t (string= (cl-map 'string (lambda rest (char-upcase (car rest))) #() #() #() #() #()) ""))
	(test-t (string= (cl-map 'string (lambda (a b) (if (char< a b) b a)) "axbycz" "xaybzc") "xxyyzz"))
	(test-t (string= (cl-map 'string (lambda (a b) (if (char< a b) b a)) "axbycz" "xayb") "xxyy"))
	(test-t (let ((list ()))
		  (and (null (cl-map nil
				     (lambda rest
				       (setq list (cons (apply + rest) list)))
				     '(0 1 2 3)
				     '(1 2 3 4)))
		       (equal list '(7 5 3 1)))))
	(test-t (let ((list ()))
		  (and (null (cl-map nil
				     (lambda rest
				       (setq list (cons (apply + rest) list)))
				     '(0 1 2 3)
				     '(1 2 3 4)
				     '(2 3 4 5)))
		       (equal list (cl-reverse '(3 6 9 12))))))
	(test-t (let ((list ()))
		  (and (null (cl-map nil
				     (lambda rest
				       (setq list (cons (apply + rest) list)))
				     '(0 1 2 3)
				     '(1)
				     '(2 3 4 5)))
		       (equal list '(3)))))
	(test-t (string= (cl-map 'string char-upcase "abc") "ABC"))
	(test-t (let ((a (list 1 2 3 4))
		      (b (list 10 10 10 10)))
		  (and (equal (map-into a + a b) '(11 12 13 14))
		       (equal a '(11 12 13 14))
		       (equal b '(10 10 10 10)))))
	(test-t (let ((a '(11 12 13 14)) (k '(one two three))) (equal (map-into a cons k a) '((one . 11) (two . 12) (three . 13) 14))))
	(test-t (null (map-into nil identity)))
	(test-t (null (map-into nil identity)))
	(test-t (null (map-into nil identity '())))
	(test-t (null (map-into nil identity '(0 1 2) '(9 8 7))))
	(test-t (let ((list (list 0 1 2))) (and (eq (map-into list identity) list) (equal list '(0 1 2)))))
	(test-t (let ((list (list 0 1 2))) (and (eq (map-into list identity '()) list) (equal list '(0 1 2)))))
	(test-t (let ((vec (vector 0 1 2))) (and (eq (map-into vec identity) vec) (equalp vec #(0 1 2)))))
	(test-t (let ((vec (vector 0 1 2))) (and (eq (map-into vec identity #()) vec) (equalp vec #(0 1 2)))))
	(test-t (let ((vec (vector 0 1 2))) (and (eq (map-into vec + #() '() #()) vec) (equalp vec #(0 1 2)))))
	(test-t (equal (map-into (list nil nil) + '(0 1) '(1 0)) '(1 1)))
	(test-t (equal (map-into (list nil nil) - '(0 1) '(1 0)) '(-1 1)))
	(test-t (let ((list (cl-make-list 3 :initial-element nil))) (and (eq (map-into list + '(0 1 2)) list) (equal list '(0 1 2)))))
	(test-t (let ((list (cl-make-list 3 :initial-element nil))) (and (eq (map-into list + '(0 1 2) '(1 2 3)) list) (equal list '(1 3 5)))))
	(test-t (let ((list (cl-make-list 3 :initial-element nil))) (and (eq (map-into list + '(0 1 2) '(1 2 3) '(2 3 4)) list) (equal list '(3 6 9)))))
	(test-t (let ((list (cl-make-list 3 :initial-element nil))) (and (eq (map-into list + '(1 2) '(1 2 3)) list) (equal list '(2 4 ())))))
	(test-t (let ((list (cl-make-list 1 :initial-element nil))) (and (eq (map-into list + '(1 2 3) '(1 2 3)) list) (equal list '(2)))))
	(test-t (let ((list (cl-make-list 3 :initial-element nil))) (and (eq (map-into list + '(1 2 3 4) '(1 2 3) '(0)) list) (equal list '(2 () ())))))
	(test-t (let ((vec (make-sequence 'vector 3 :initial-element nil))) (and (eq (map-into vec + '(0 1 2)) vec) (equalp vec #(0 1 2)))))
	(test-t (let ((vec (make-sequence 'vector 3 :initial-element nil))) (and (eq (map-into vec + '(0 1 2) #(1 2 3)) vec) (equalp vec #(1 3 5)))))
	(test-t (let ((vec (make-sequence 'vector 3 :initial-element nil))) (and (eq (map-into vec + '(0 1 2) '(1 2 3) #(2 3 4)) vec) (equalp vec #(3 6 9)))))
	(test-t (let ((vec (make-sequence 'vector 3 :initial-element nil))) (and (eq (map-into vec + '(1 2) #(1 2 3)) vec) (equalp vec #(2 4 ())))))
	(test-t (let ((vec (make-sequence 'vector 1 :initial-element nil))) (and (eq (map-into vec + '(1 2) #(1 2 3)) vec) (equalp vec #(2)))))
	(test-t (let ((vec (make-sequence 'vector 3 :initial-element nil))) (and (eq (map-into vec + '(1 2 3 4) #(1 2 3) '(0)) vec) (equalp vec #(2 () ())))))
	(test-t (eql (reduce * '(1 2 3 4 5)) 120))
	(test-t (equal (reduce append '((1) (2)) :initial-value '(i n i t)) '(i n i t 1 2)))
	(test-t (equal (reduce append '((1) (2)) :from-end t :initial-value '(i n i t)) '(1 2 i n i t)))
	(test-t (eql (reduce - '(1 2 3 4)) -8))
	(test-t (eql (reduce - '(1 2 3 4) :from-end t) -2))
	(test-t (eql (reduce + '()) 0))
	(test-t (eql (reduce + '(3)) 3))
	(test-t (eq (reduce + '(foo)) 'foo))
	(test-t (equal (reduce list '(1 2 3 4)) '(((1 2) 3) 4)))
	(test-t (equal (reduce list '(1 2 3 4) :from-end t) '(1 (2 (3 4)))))
	(test-t (equal (reduce list '(1 2 3 4) :initial-value 'foo) '((((foo 1) 2) 3) 4)))
	(test-t (equal (reduce list '(1 2 3 4) :from-end t :initial-value 'foo) '(1 (2 (3 (4 foo))))))
	(test-t (equal (reduce list '(0 1 2 3)) '(((0 1) 2) 3)))
	(test-t (equal (reduce list '(0 1 2 3) :start 1) '((1 2) 3)))
	(test-t (equal (reduce list '(0 1 2 3) :start 1 :end nil) '((1 2) 3)))
	(test-t (equal (reduce list '(0 1 2 3) :start 2) '(2 3)))
	(test-t (eq (reduce list '(0 1 2 3) :start 0 :end 0) '()))
	(test-t (eq (reduce list '(0 1 2 3) :start 0 :end 0 :initial-value 'initial-value) 'initial-value))
	(test-t (eq (reduce list '(0 1 2 3) :start 2 :end 2) '()))
	(test-t (eq (reduce list '(0 1 2 3) :start 2 :end 2 :initial-value 'initial-value) 'initial-value))
	(test-t (eq (reduce list '(0 1 2 3) :start 4 :end 4) '()))
	(test-t (eq (reduce list '(0 1 2 3) :start 4 :end 4 :initial-value 'initial-value) 'initial-value))
	(test-t (eql (reduce list '(0 1 2 3) :start 2 :end 3) 2))
	(test-t (equal (reduce list '(0 1 2 3) :start 2 :end 3 :initial-value 'initial-value) '(initial-value 2)))
	(test-t (eql (reduce + '(0 1 2 3 4 5 6 7 8 9)) 45))
	(test-t (eql (reduce - '(0 1 2 3 4 5 6 7 8 9)) -45))
	(test-t (eql (reduce - '(0 1 2 3 4 5 6 7 8 9) :from-end t) -5))
	(test-t (equal (reduce list '(0 1 2 3) :initial-value 'initial-value) '((((initial-value 0) 1) 2) 3)))
	(test-t (equal (reduce list '(0 1 2 3) :from-end t) '(0 (1 (2 3)))))
	(test-t (equal (reduce list '((1) (2) (3) (4)) :key car) '(((1 2) 3) 4)))
					;(test-t (equal (reduce list '((1) (2) (3) (4)) :key car :from-end nil) '(((1 2) 3) 4)))
	(test-t (equal (reduce list '((1) (2) (3) (4)) :key car :initial-value 0) '((((0 1) 2) 3) 4)))
	(test-t (equal (reduce list '((1) (2) (3) (4)) :key car :from-end t) '(1 (2 (3 4)))))
	(test-t (equal (reduce list '((1) (2) (3) (4)) :key car :from-end t :initial-value 5) '(1 (2 (3 (4 5))))))
	(test-t (equal (reduce list #(0 1 2 3)) '(((0 1) 2) 3)))
	(test-t (equal (reduce list #(0 1 2 3) :start 1) '((1 2) 3)))
	(test-t (equal (reduce list #(0 1 2 3) :start 1 :end nil) '((1 2) 3)))
	(test-t (equal (reduce list #(0 1 2 3) :start 2) '(2 3)))
	(test-t (eq (reduce list #(0 1 2 3) :start 0 :end 0) '()))
	(test-t (eq (reduce list #(0 1 2 3) :start 0 :end 0 :initial-value 'initial-value) 'initial-value))
	(test-t (eq (reduce list #(0 1 2 3) :start 2 :end 2) '()))
	(test-t (eq (reduce list #(0 1 2 3) :start 2 :end 2 :initial-value 'initial-value) 'initial-value))
	(test-t (eq (reduce list #(0 1 2 3) :start 4 :end 4) '()))
	(test-t (eq (reduce list #(0 1 2 3) :start 4 :end 4 :initial-value 'initial-value) 'initial-value))
	(test-t (eql (reduce list #(0 1 2 3) :start 2 :end 3) 2))
	(test-t (equal (reduce list #(0 1 2 3) :start 2 :end 3 :initial-value 'initial-value) '(initial-value 2)))
	(test-t (eql (reduce + #(0 1 2 3 4 5 6 7 8 9)) 45))
	(test-t (eql (reduce - #(0 1 2 3 4 5 6 7 8 9)) -45))
	(test-t (eql (reduce - #(0 1 2 3 4 5 6 7 8 9) :from-end t) -5))
	(test-t (equal (reduce list #(0 1 2 3) :initial-value 'initial-value) '((((initial-value 0) 1) 2) 3)))
	(test-t (equal (reduce list #(0 1 2 3) :from-end t) '(0 (1 (2 3)))))
	(test-t (equal (reduce list #((1) (2) (3) (4)) :key car) '(((1 2) 3) 4)))
					;(test-t (equal (reduce list #((1) (2) (3) (4)) :key car :from-end nil) '(((1 2) 3) 4)))
	(test-t (equal (reduce list #((1) (2) (3) (4)) :key car :initial-value 0) '((((0 1) 2) 3) 4)))
	(test-t (equal (reduce list #((1) (2) (3) (4)) :key car :from-end t) '(1 (2 (3 4)))))
	(test-t (equal (reduce list #((1) (2) (3) (4)) :key car :from-end t :initial-value 5) '(1 (2 (3 (4 5))))))
	(test-t (eql (count #\a "how many A's are there in here?") 2))
	(test-t (eql (count-if-not oddp '((1) (2) (3) (4)) :key car) 2))
	(test-t (eql (count-if upper-case-p "The Crying of Lot 49" :start 4) 2))
	(test-t (eql (count #\a (concatenate 'list "how many A's are there in here?")) 2))
	(test-t (eql (count-if alpha-char-p "-a-b-c-0-1-2-3-4-") 3))
	(test-t (eql (count-if alphanumericp "-a-b-c-0-1-2-3-4-") 8))
	(test-t (eql (count nil (list t nil t nil t nil)) 3))
	(test-t (eql (count nil (vector t nil t nil t nil)) 3))
	(test-t (zerop (count 9 '(0 1 2 3 4))))
	(test-t (zerop (count 'a '(0 1 2 3 4))))
	(test-t (eql (count 0 '(0 0 0 0 0) :start 1) 4))
	(test-t (eql (count 0 '(0 0 0 0 0) :start 1 :end nil) 4))
	(test-t (eql (count 0 '(0 0 0 0 0) :start 2) 3))
	(test-t (zerop (count 0 '(0 0 0 0) :start 0 :end 0)))
	(test-t (zerop (count 0 '(0 0 0 0) :start 2 :end 2)))
	(test-t (zerop (count 0 '(0 0 0 0) :start 4 :end 4)))
	(test-t (eql (count 0 '(0 0 0 0) :start 2 :end 3) 1))
	(test-t (eql (count #\a "abcABC" :test equalp) 2))
	(test-t (eql (count #\a "abcABC" :test char-equal) 2))
	(test-t (eql (count '(a) '((x) (y) (z) (a) (b) (c)) :test equalp) 1))
	(test-t (eql (count 'a '((x) (y) (z) (a) (b) (c)) :key car :test eq) 1))
	(test-t (eql (count nil '((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr :test eq) 3))
	(test-t (let ((list nil))
		  (and (eql (count 'a '(a b c d)
				   :test (lambda (a b) (setq list (cons b list)) (eq a b)))
			    1)
		       (equal list '(d c b a)))))
	(test-t (let ((list nil))
		  (and (eql (count 'a '(a b c d)
				   :test (lambda (a b) (setq list (cons b list)) (eq a b))
				   :from-end t)
			    1)
		       (equal list '(a b c d)))))
	(test-t (zerop (count 9 #(0 1 2 3 4))))
	(test-t (zerop (count 'a #(0 1 2 3 4))))
	(test-t (eql (count 0 #(0 0 0 0 0) :start 1) 4))
	(test-t (eql (count 0 #(0 0 0 0 0) :start 1 :end nil) 4))
	(test-t (eql (count 0 #(0 0 0 0 0) :start 2) 3))
	(test-t (zerop (count 0 #(0 0 0 0) :start 0 :end 0)))
	(test-t (zerop (count 0 #(0 0 0 0) :start 2 :end 2)))
	(test-t (zerop (count 0 #(0 0 0 0) :start 4 :end 4)))
	(test-t (eql (count 0 #(0 0 0 0) :start 2 :end 3) 1))
	(test-t (eql (count '(a) #((x) (y) (z) (a) (b) (c)) :test equalp) 1))
	(test-t (eql (count 'a #((x) (y) (z) (a) (b) (c)) :key car :test eq) 1))
	(test-t (eql (count nil #((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr :test eq) 3))
	(test-t (let ((list nil))
		  (and (eql (count 'a #(a b c d)
				   :test (lambda (a b) (setq list (cons b list)) (eq a b)))
			    1)
		       (equal list '(d c b a)))))
	(test-t (let ((list nil))
		  (and (eql (count 'a #(a b c d)
				   :test (lambda (a b) (setq list (cons b list)) (eq a b))
				   :from-end t)
			    1)
		       (equal list '(a b c d)))))
	(test-t (eql (count-if null (list t nil t nil t nil)) 3))
	(test-t (zerop (count-if (lambda (x) (eql x 9)) #(0 1 2 3 4))))
	(test-t (zerop (count-if (lambda (a) (eq 'x a)) #(0 1 2 3 4))))
	(test-t (eql (count-if zerop '(0 0 0 0 0) :start 1) 4))
	(test-t (eql (count-if zerop '(0 0 0 0 0) :start 1 :end nil) 4))
	(test-t (eql (count-if zerop '(0 0 0 0 0) :start 2) 3))
	(test-t (zerop (count-if zerop '(0 0 0 0) :start 0 :end 0)))
	(test-t (zerop (count-if zerop '(0 0 0 0) :start 2 :end 2)))
	(test-t (zerop (count-if zerop '(0 0 0 0) :start 4 :end 4)))
	(test-t (eql (count-if zerop '(0 0 0 0) :start 2 :end 3) 1))
	(test-t (eql (count-if (lambda (x) (equalp #\a x)) "abcABC") 2))
	(test-t (eql (count-if (lambda (x) (char-equal #\a x)) "abcABC") 2))
	(test-t (eql (count-if (lambda (x) (equal x '(a))) '((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if (lambda (x) (eq x 'a)) '((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if null '((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (eql (count-if (lambda (x) (equal x '(a))) '((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if (lambda (x) (eq x 'a)) '((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if null '((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (let ((list nil))
		  (and (eql (count-if (lambda (x) (setq list (cons x list)) (eq x 'a))
				      '(a b c d))
			    1)
		       (equal list '(d c b a)))))
	(test-t (let ((list nil))
		  (and (eql (count-if (lambda (x) (setq list (cons x list)) (eq x 'a))
				      '(a b c d)
				      :from-end t)
			    1)
		       (equal list '(a b c d)))))
	(test-t (eql (count-if null (vector t nil t nil t nil)) 3))
	(test-t (eql (count-if zerop #(0 0 0 0 0) :start 1) 4))
	(test-t (eql (count-if zerop #(0 0 0 0 0) :start 1 :end nil) 4))
	(test-t (eql (count-if zerop #(0 0 0 0 0) :start 2) 3))
	(test-t (zerop (count-if zerop #(0 0 0 0) :start 0 :end 0)))
	(test-t (zerop (count-if zerop #(0 0 0 0) :start 2 :end 2)))
	(test-t (zerop (count-if zerop #(0 0 0 0) :start 4 :end 4)))
	(test-t (eql (count-if zerop #(0 0 0 0) :start 2 :end 3) 1))
	(test-t (eql (count-if (lambda (x) (equal x '(a))) #((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if (lambda (x) (eq x 'a)) #((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if null #((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (eql (count-if (lambda (x) (equal x '(a))) #((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if (lambda (x) (eq x 'a)) #((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if null #((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (let ((list nil))
		  (and (eql (count-if (lambda (x) (setq list (cons x list)) (eq x 'a))
				      #(a b c d))
			    1)
		       (equal list '(d c b a)))))
	(test-t (let ((list nil))
		  (and (eql (count-if (lambda (x) (setq list (cons x list)) (eq x 'a))
				      #(a b c d)
				      :from-end t)
			    1)
		       (equal list '(a b c d)))))
	(test-t (eql (count-if-not (complement null) (list t nil t nil t nil)) 3))
	(test-t (zerop (count-if-not (lambda (x) (not (eql x 9))) #(0 1 2 3 4))))
	(test-t (zerop (count-if-not (lambda (a) (not (eq 'x a))) #(0 1 2 3 4))))
	(test-t (eql (count-if-not (complement zerop) '(0 0 0 0 0) :start 1) 4))
	(test-t (eql (count-if-not (complement zerop) '(0 0 0 0 0) :start 1 :end nil) 4))
	(test-t (eql (count-if-not (complement zerop) '(0 0 0 0 0) :start 2) 3))
	(test-t (zerop (count-if-not (complement zerop) '(0 0 0 0) :start 0 :end 0)))
	(test-t (zerop (count-if-not (complement zerop) '(0 0 0 0) :start 2 :end 2)))
	(test-t (zerop (count-if-not (complement zerop) '(0 0 0 0) :start 4 :end 4)))
	(test-t (eql (count-if-not (complement zerop) '(0 0 0 0) :start 2 :end 3) 1))
	(test-t (eql (count-if-not (lambda (x) (not (equalp #\a x))) "abcABC") 2))
	(test-t (eql (count-if-not (lambda (x) (not (char-equal #\a x))) "abcABC") 2))
	(test-t (eql (count-if-not (lambda (x) (not (equal x '(a)))) '((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if-not (lambda (x) (not (eq x 'a))) '((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if-not (complement null) '((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (eql (count-if-not (lambda (x) (not (equal x '(a)))) '((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if-not (lambda (x) (not (eq x 'a))) '((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if-not (complement null) '((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (let ((list nil))
		  (and (eql (count-if-not (lambda (x)
					    (setq list (cons x list))
					    (not (eq x 'a)))
					  '(a b c d))
			    1)
		       (equal list '(d c b a)))))
	(test-t (let ((list nil))
		  (and (eql (count-if-not (lambda (x)
					    (setq list (cons x list))
					    (not (eq x 'a)))
					  '(a b c d)
					  :from-end t)
			    1)
		       (equal list '(a b c d)))))
	(test-t (eql (count-if-not (complement null) (vector t nil t nil t nil)) 3))
	(test-t (eql (count-if-not (complement zerop) #(0 0 0 0 0) :start 1) 4))
	(test-t (eql (count-if-not (complement zerop) #(0 0 0 0 0) :start 1 :end nil) 4))
	(test-t (eql (count-if-not (complement zerop) #(0 0 0 0 0) :start 2) 3))
	(test-t (zerop (count-if-not (complement zerop) #(0 0 0 0) :start 0 :end 0)))
	(test-t (zerop (count-if-not (complement zerop) #(0 0 0 0) :start 2 :end 2)))
	(test-t (zerop (count-if-not (complement zerop) #(0 0 0 0) :start 4 :end 4)))
	(test-t (eql (count-if-not (complement zerop) #(0 0 0 0) :start 2 :end 3) 1))
	(test-t (eql (count-if-not (lambda (x) (not (equal x '(a)))) #((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if-not (lambda (x) (not (eq x 'a))) #((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if-not (complement null) #((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (eql (count-if-not (lambda (x) (not (equal x '(a)))) #((x) (y) (z) (a) (b) (c))) 1))
	(test-t (eql (count-if-not (lambda (x) (not (eq x 'a))) #((x) (y) (z) (a) (b) (c)) :key car) 1))
	(test-t (eql (count-if-not (complement null) #((x . x) (y) (z . z) (a) (b . b) (c)) :key cdr) 3))
	(test-t (let ((list nil))
		  (and (eql (count-if-not (lambda (x)
					    (setq list (cons x list))
					    (not (eq x 'a)))
					  #(a b c d))
			    1)
		       (equal list '(d c b a)))))
	(test-t (let ((list nil))
		  (and (eql (count-if-not (lambda (x)
					    (setq list (cons x list))
					    (not (eq x 'a)))
					  #(a b c d)
					  :from-end t)
			    1)
		       (equal list '(a b c d)))))
	(test-t (null (cl-reverse nil)))
	(test-t (string= (cl-reverse "") ""))
	(test-t (equalp (cl-reverse #()) #()))
	(test-t (equal (cl-reverse '(0 1 2 3)) '(3 2 1 0)))
	(test-t (string= (cl-reverse "0123") "3210"))
	(test-t (equalp (cl-reverse #(a b c d)) #(d c b a)))
	(test-t (null (nreverse nil)))
	(test-t (string= (nreverse (copy-seq "")) ""))
	(test-t (equalp (nreverse (copy-seq #())) #()))
	(test-t (equal (nreverse (list 0 1 2 3)) '(3 2 1 0)))
	(test-t (string= (nreverse (copy-seq "0123")) "3210"))
	(test-t (equalp (cl-reverse (copy-seq #(a b c d))) #(d c b a)))
	(test-t (char= (find #\d "edcba" :test char>) #\c))
	(test-t (eql (find-if oddp '(1 2 3 4 5) :end 3 :from-end t) 3))
	(test-t (eq (find 'a '(a b c)) 'a))
	(test-t (eq (find 'b '(a b c)) 'b))
	(test-t (eq (find 'c '(a b c)) 'c))
	(test-t (null (find 'x '(a b c))))
	(test-t (null (find 'a '(a b c) :start 1)))
	(test-t (null (find 'b '(a b c) :start 2)))
	(test-t (null (find 'c '(a b c) :start 3)))
	(test-t (null (find 'a '(a b c) :start 0 :end 0)))
	(test-t (null (find 'a '(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (find 'a '(a b c) :start 1 :end 1)))
	(test-t (null (find 'a '(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (find 'a '(a b c) :start 2 :end 2)))
	(test-t (null (find 'a '(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (find 'a '(a b c) :start 3 :end 3)))
	(test-t (null (find 'a '(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eq (find 'a '(a b c) :end nil) 'a))
	(test-t (eq (find 'b '(a b c) :end nil) 'b))
	(test-t (eq (find 'c '(a b c) :end nil) 'c))
	(test-t (eq (find 'a '(a b c) :end 1) 'a))
	(test-t (eq (find 'b '(a b c) :end 2) 'b))
	(test-t (eq (find 'c '(a b c) :end 3) 'c))
	(test-t (null (find 'a '(a b c) :end 0)))
	(test-t (null (find 'b '(a b c) :end 1)))
	(test-t (null (find 'c '(a b c) :end 2)))
	(test-t (null (find 'a '((a) (b) (c)))))
	(test-t (equal (find 'a '((a) (b) (c)) :key car) '(a)))
	(test-t (equal (find 'b '((a) (b) (c)) :key car) '(b)))
	(test-t (equal (find 'c '((a) (b) (c)) :key car) '(c)))
	(test-t (null (find 'z '((a) (b) (c)) :key car)))
	(test-t (let ((list '((a) (b) (c))))
		  (and (eq (find 'a list :key car) (car list))
		       (eq (find 'b list :key car) (cadr list))
		       (eq (find 'c list :key car) (caddr list))
		       (null (find 'z list :key car)))))
	(test-t (null (find '(a) '((a) (b) (c)))))
	(test-t (equal (find '(a) '((a) (b) (c)) :test equal) '(a)))
	(test-t (null (find '("a") '(("a") ("b") ("c")))))
	(test-t (null (find '("a") '(("A") ("B") ("c")) :test equal)))
	(test-t (eql (find 3 '(0 1 2 3 4 5)) 3))
	(test-t (eql (find 3 '(0 1 2 3 4 5) :test <) 4))
	(test-t (eql (find 3 '(0 1 2 3 4 5) :test >) 0))
	(test-t (equal (find 'a '((a) (b) (c) (a a) (b b) (c c)) :key car) '(a)))
	(test-t (equal (find 'a '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(a a)))
	(test-t (equal (find 'b '((a) (b) (c) (a a) (b b) (c c)) :key car) '(b)))
	(test-t (equal (find 'b '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(b b)))
	(test-t (equal (find 'c '((a) (b) (c) (a a) (b b) (c c)) :key car) '(c)))
	(test-t (equal (find 'c '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(c c)))
	(test-t (null (find 'z '((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (find 'z '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (equal (find 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) '(a a a)))
	(test-t (equal (find 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) '(a a a)))
	(test-t (equal (find 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) '(a a)))
	(test-t (null (find 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (null (find #\c '("abc" "bcd" "cde"))))
	(test-t (string= (find #\c '("abc" "bcd" "cde") :key (lambda (arg) (char arg 0)) :test char=) "cde"))
	(test-t (eq (find 'a #(a b c)) 'a))
	(test-t (eq (find 'b #(a b c)) 'b))
	(test-t (eq (find 'c #(a b c)) 'c))
	(test-t (null (find 'x #(a b c))))
	(test-t (null (find 'a #(a b c) :start 1)))
	(test-t (null (find 'b #(a b c) :start 2)))
	(test-t (null (find 'c #(a b c) :start 3)))
	(test-t (null (find 'a #(a b c) :start 0 :end 0)))
	(test-t (null (find 'a #(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (find 'a #(a b c) :start 1 :end 1)))
	(test-t (null (find 'a #(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (find 'a #(a b c) :start 2 :end 2)))
	(test-t (null (find 'a #(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (find 'a #(a b c) :start 3 :end 3)))
	(test-t (null (find 'a #(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eq (find 'a #(a b c) :end nil) 'a))
	(test-t (eq (find 'b #(a b c) :end nil) 'b))
	(test-t (eq (find 'c #(a b c) :end nil) 'c))
	(test-t (eq (find 'a #(a b c) :end 1) 'a))
	(test-t (eq (find 'b #(a b c) :end 2) 'b))
	(test-t (eq (find 'c #(a b c) :end 3) 'c))
	(test-t (null (find 'a #(a b c) :end 0)))
	(test-t (null (find 'b #(a b c) :end 1)))
	(test-t (null (find 'c #(a b c) :end 2)))
	(test-t (null (find 'a #((a) (b) (c)))))
	(test-t (equal (find 'a #((a) (b) (c)) :key car) '(a)))
	(test-t (equal (find 'b #((a) (b) (c)) :key car) '(b)))
	(test-t (equal (find 'c #((a) (b) (c)) :key car) '(c)))
	(test-t (null (find 'z #((a) (b) (c)) :key car)))
	(test-t (let ((vector #((a) (b) (c))))
		  (and (eq (find 'a vector :key car) (aref vector 0))
		       (eq (find 'b vector :key car) (aref vector 1))
		       (eq (find 'c vector :key car) (aref vector 2))
		       (null (find 'z vector :key car)))))
	(test-t (null (find '(a) #((a) (b) (c)))))
	(test-t (equal (find '(a) #((a) (b) (c)) :test equal) '(a)))
	(test-t (null (find '("a") #(("a") ("b") ("c")))))
	(test-t (null (find '("a") #(("A") ("B") ("c")) :test equal)))
	(test-t (eql (find 3 #(0 1 2 3 4 5)) 3))
	(test-t (eql (find 3 #(0 1 2 3 4 5) :test <) 4))
	(test-t (eql (find 3 #(0 1 2 3 4 5) :test >) 0))
	(test-t (equal (find 'a #((a) (b) (c) (a a) (b b) (c c)) :key car) '(a)))
	(test-t (equal (find 'a #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(a a)))
	(test-t (equal (find 'b #((a) (b) (c) (a a) (b b) (c c)) :key car) '(b)))
	(test-t (equal (find 'b #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(b b)))
	(test-t (equal (find 'c #((a) (b) (c) (a a) (b b) (c c)) :key car) '(c)))
	(test-t (equal (find 'c #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(c c)))
	(test-t (null (find 'z #((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (find 'z #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (equal (find 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) '(a a a)))
	(test-t (equal (find 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) '(a a a)))
	(test-t (equal (find 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) '(a a)))
	(test-t (null (find 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (null (find #\c #("abc" "bcd" "cde"))))
	(test-t (null (find #\z "abcABC")))
	(test-t (eql (find #\a "abcABC") #\a))
	(test-t (eql (find #\A "abcABC") #\A))
	(test-t (eql (find #\A "abcABC" :test char-equal) #\a))
	(test-t (eql (find #\A "abcABC" :test char-equal :from-end t) #\A))
	(test-t (eql (find #\a "abcABC" :test char-equal :from-end t) #\A))
	(test-t (eql (find #\a "abcABC" :test char-equal :from-end t :end 4) #\A))
	(test-t (eql (find #\a "abcABC" :test char-equal :from-end t :end 3) #\a))
	(test-t (eq (find-if (lambda (x) (eq x 'a)) '(a b c)) 'a))
	(test-t (eq (find-if (lambda (x) (eq x 'b)) '(a b c)) 'b))
	(test-t (eq (find-if (lambda (x) (eq x 'c)) '(a b c)) 'c))
	(test-t (null (find-if (lambda (arg) (eq arg 'x)) '(a b c))))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 1)))
	(test-t (null (find-if (lambda (x) (eq x 'b)) '(a b c) :start 2)))
	(test-t (null (find-if (lambda (x) (eq x 'c)) '(a b c) :start 3)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 0 :end 0)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 1 :end 1)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 2 :end 2)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 3 :end 3)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eq (find-if (lambda (x) (eq x 'a)) '(a b c) :end nil) 'a))
	(test-t (eq (find-if (lambda (x) (eq x 'b)) '(a b c) :end nil) 'b))
	(test-t (eq (find-if (lambda (x) (eq x 'c)) '(a b c) :end nil) 'c))
	(test-t (eq (find-if (lambda (x) (eq x 'a)) '(a b c) :end 1) 'a))
	(test-t (eq (find-if (lambda (x) (eq x 'b)) '(a b c) :end 2) 'b))
	(test-t (eq (find-if (lambda (x) (eq x 'c)) '(a b c) :end 3) 'c))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '(a b c) :end 0)))
	(test-t (null (find-if (lambda (x) (eq x 'b)) '(a b c) :end 1)))
	(test-t (null (find-if (lambda (x) (eq x 'c)) '(a b c) :end 2)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '((a) (b) (c)))))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) '((a) (b) (c)) :key car) '(a)))
	(test-t (equal (find-if (lambda (x) (eq x 'b)) '((a) (b) (c)) :key car) '(b)))
	(test-t (equal (find-if (lambda (x) (eq x 'c)) '((a) (b) (c)) :key car) '(c)))
	(test-t (null (find-if (lambda (x) (eq x 'z)) '((a) (b) (c)) :key car)))
	(test-t (let ((list '((a) (b) (c))))
		  (and (eq (find-if (lambda (x) (eq x 'a)) list :key car) (car list))
		       (eq (find-if (lambda (x) (eq x 'b)) list :key car) (cadr list))
		       (eq (find-if (lambda (x) (eq x 'c)) list :key car) (caddr list))
		       (null (find-if (lambda (x) (eq x 'z)) list :key car)))))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c)) :key car) '(a)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(a a)))
	(test-t (equal (find-if (lambda (x) (eq x 'b)) '((a) (b) (c) (a a) (b b) (c c)) :key car) '(b)))
	(test-t (equal (find-if (lambda (x) (eq x 'b)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(b b)))
	(test-t (equal (find-if (lambda (x) (eq x 'c)) '((a) (b) (c) (a a) (b b) (c c)) :key car) '(c)))
	(test-t (equal (find-if (lambda (x) (eq x 'c)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(c c)))
	(test-t (null (find-if (lambda (x) (eq x 'z)) '((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (find-if (lambda (x) (eq x 'z)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) '(a a a)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) '(a a a)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) '(a a)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eq (find-if (lambda (x) (eq x 'a)) #(a b c)) 'a))
	(test-t (eq (find-if (lambda (x) (eq x 'b)) #(a b c)) 'b))
	(test-t (eq (find-if (lambda (x) (eq x 'c)) #(a b c)) 'c))
	(test-t (null (find-if (lambda (arg) (eq arg 'x)) #(a b c))))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 1)))
	(test-t (null (find-if (lambda (x) (eq x 'b)) #(a b c) :start 2)))
	(test-t (null (find-if (lambda (x) (eq x 'c)) #(a b c) :start 3)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 0 :end 0)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 1 :end 1)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 2 :end 2)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 3 :end 3)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eq (find-if (lambda (x) (eq x 'a)) #(a b c) :end nil) 'a))
	(test-t (eq (find-if (lambda (x) (eq x 'b)) #(a b c) :end nil) 'b))
	(test-t (eq (find-if (lambda (x) (eq x 'c)) #(a b c) :end nil) 'c))
	(test-t (eq (find-if (lambda (x) (eq x 'a)) #(a b c) :end 1) 'a))
	(test-t (eq (find-if (lambda (x) (eq x 'b)) #(a b c) :end 2) 'b))
	(test-t (eq (find-if (lambda (x) (eq x 'c)) #(a b c) :end 3) 'c))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #(a b c) :end 0)))
	(test-t (null (find-if (lambda (x) (eq x 'b)) #(a b c) :end 1)))
	(test-t (null (find-if (lambda (x) (eq x 'c)) #(a b c) :end 2)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #((a) (b) (c)))))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) #((a) (b) (c)) :key car) '(a)))
	(test-t (equal (find-if (lambda (x) (eq x 'b)) #((a) (b) (c)) :key car) '(b)))
	(test-t (equal (find-if (lambda (x) (eq x 'c)) #((a) (b) (c)) :key car) '(c)))
	(test-t (null (find-if (lambda (x) (eq x 'z)) #((a) (b) (c)) :key car)))
	(test-t (let ((vector #((a) (b) (c))))
		  (and (eq (find-if (lambda (x) (eq x 'a)) vector :key car) (aref vector 0))
		       (eq (find-if (lambda (x) (eq x 'b)) vector :key car) (aref vector 1))
		       (eq (find-if (lambda (x) (eq x 'c)) vector :key car) (aref vector 2))
		       (null (find-if (lambda (x) (eq x 'z)) vector :key car)))))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c)) :key car) '(a)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(a a)))
	(test-t (equal (find-if (lambda (x) (eq x 'b)) #((a) (b) (c) (a a) (b b) (c c)) :key car) '(b)))
	(test-t (equal (find-if (lambda (x) (eq x 'b)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(b b)))
	(test-t (equal (find-if (lambda (x) (eq x 'c)) #((a) (b) (c) (a a) (b b) (c c)) :key car) '(c)))
	(test-t (equal (find-if (lambda (x) (eq x 'c)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(c c)))
	(test-t (null (find-if (lambda (x) (eq x 'z)) #((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (find-if (lambda (x) (eq x 'z)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) '(a a a)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) '(a a a)))
	(test-t (equal (find-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) '(a a)))
	(test-t (null (find-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'a))) '(a b c)) 'a))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'b))) '(a b c)) 'b))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'c))) '(a b c)) 'c))
	(test-t (null (find-if-not (lambda (arg) (not (eq arg 'x))) '(a b c))))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 1)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'b))) '(a b c) :start 2)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'c))) '(a b c) :start 3)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 0 :end 0)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 1 :end 1)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 2 :end 2)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 3 :end 3)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :end nil) 'a))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'b))) '(a b c) :end nil) 'b))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'c))) '(a b c) :end nil) 'c))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :end 1) 'a))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'b))) '(a b c) :end 2) 'b))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'c))) '(a b c) :end 3) 'c))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '(a b c) :end 0)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'b))) '(a b c) :end 1)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'c))) '(a b c) :end 2)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c)))))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c)) :key car) '(a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'b))) '((a) (b) (c)) :key car) '(b)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'c))) '((a) (b) (c)) :key car) '(c)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'z))) '((a) (b) (c)) :key car)))
	(test-t (let ((list '((a) (b) (c))))
		  (and (eq (find-if-not (lambda (x) (not (eq x 'a))) list :key car)
			   (car list))
		       (eq (find-if-not (lambda (x) (not (eq x 'b))) list :key car)
			   (cadr list))
		       (eq (find-if-not (lambda (x) (not (eq x 'c))) list :key car)
			   (caddr list))
		       (null (find-if-not (lambda (x) (not (eq x 'z))) list :key car)))))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c)) :key car) '(a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(a a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'b))) '((a) (b) (c) (a a) (b b) (c c)) :key car) '(b)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'b))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(b b)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'c))) '((a) (b) (c) (a a) (b b) (c c)) :key car) '(c)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'c))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(c c)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'z))) '((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'z))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) '(a a a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) '(a a a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) '(a a)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'a))) #(a b c)) 'a))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'b))) #(a b c)) 'b))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'c))) #(a b c)) 'c))
	(test-t (null (find-if-not (lambda (arg) (not (eq arg 'x))) #(a b c))))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 1)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'b))) #(a b c) :start 2)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'c))) #(a b c) :start 3)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 0 :end 0)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 1 :end 1)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 2 :end 2)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 3 :end 3)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :end nil) 'a))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'b))) #(a b c) :end nil) 'b))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'c))) #(a b c) :end nil) 'c))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :end 1) 'a))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'b))) #(a b c) :end 2) 'b))
	(test-t (eq (find-if-not (lambda (x) (not (eq x 'c))) #(a b c) :end 3) 'c))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #(a b c) :end 0)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'b))) #(a b c) :end 1)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'c))) #(a b c) :end 2)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c)))))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c)) :key car) '(a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'b))) #((a) (b) (c)) :key car) '(b)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'c))) #((a) (b) (c)) :key car) '(c)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'z))) #((a) (b) (c)) :key car)))
	(test-t (let ((vector #((a) (b) (c))))
		  (and (eq (find-if-not (lambda (x) (not (eq x 'a))) vector :key car)
			   (aref vector 0))
		       (eq (find-if-not (lambda (x) (not (eq x 'b))) vector :key car)
			   (aref vector 1))
		       (eq (find-if-not (lambda (x) (not (eq x 'c))) vector :key car)
			   (aref vector 2))
		       (null (find-if-not (lambda (x) (not (eq x 'z))) vector :key car)))))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c)) :key car) '(a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(a a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'b))) #((a) (b) (c) (a a) (b b) (c c)) :key car) '(b)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'b))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(b b)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'c))) #((a) (b) (c) (a a) (b b) (c c)) :key car) '(c)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'c))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) '(c c)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'z))) #((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'z))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) '(a a a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) '(a a a)))
	(test-t (equal (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) '(a a)))
	(test-t (null (find-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eql (position #\a "baobab" :from-end t) 4))
	(test-t (eql (position-if oddp '((1) (2) (3) (4)) :start 1 :key car) 2))
	(test-t (null (position 595 '())))
	(test-t (eql (position-if-not integerp '(1 2 3 4 5.0)) 4))
	(test-t (eql (position 'a '(a b c)) 0))
	(test-t (eql (position 'b '(a b c)) 1))
	(test-t (eql (position 'c '(a b c)) 2))
	(test-t (null (position 'x '(a b c))))
	(test-t (null (position 'a '(a b c) :start 1)))
	(test-t (null (position 'b '(a b c) :start 2)))
	(test-t (null (position 'c '(a b c) :start 3)))
	(test-t (null (position 'a '(a b c) :start 0 :end 0)))
	(test-t (null (position 'a '(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (position 'a '(a b c) :start 1 :end 1)))
	(test-t (null (position 'a '(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (position 'a '(a b c) :start 2 :end 2)))
	(test-t (null (position 'a '(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (position 'a '(a b c) :start 3 :end 3)))
	(test-t (null (position 'a '(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eql (position 'a '(a b c) :end nil) '0))
	(test-t (eql (position 'b '(a b c) :end nil) '1))
	(test-t (eql (position 'c '(a b c) :end nil) '2))
	(test-t (eql (position 'a '(a b c) :end 1) '0))
	(test-t (eql (position 'b '(a b c) :end 2) '1))
	(test-t (eql (position 'c '(a b c) :end 3) '2))
	(test-t (null (position 'a '(a b c) :end 0)))
	(test-t (null (position 'b '(a b c) :end 1)))
	(test-t (null (position 'c '(a b c) :end 2)))
	(test-t (null (position 'a '((a) (b) (c)))))
	(test-t (eql (position 'a '((a) (b) (c)) :key car) 0))
	(test-t (eql (position 'b '((a) (b) (c)) :key car) 1))
	(test-t (eql (position 'c '((a) (b) (c)) :key car) 2))
	(test-t (null (position 'z '((a) (b) (c)) :key car)))
	(test-t (null (position '(a) '((a) (b) (c)))))
	(test-t (eql (position '(a) '((a) (b) (c)) :test equal) 0))
	(test-t (null (position '("a") '(("a") ("b") ("c")))))
	(test-t (eql (position 3 '(0 1 2 3 4 5)) 3))
	(test-t (eql (position 3 '(0 1 2 3 4 5) :test <) 4))
	(test-t (eql (position 3 '(0 1 2 3 4 5) :test >) 0))
	(test-t (eql (position 'a '((a) (b) (c) (a a) (b b) (c c)) :key car) 0))
	(test-t (eql (position 'a '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 3))
	(test-t (eql (position 'b '((a) (b) (c) (a a) (b b) (c c)) :key car) 1))
	(test-t (eql (position 'b '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 4))
	(test-t (eql (position 'c '((a) (b) (c) (a a) (b b) (c c)) :key car) 2))
	(test-t (eql (position 'c '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 5))
	(test-t (null (position 'z '((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (position 'z '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (eql (position 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) 6))
	(test-t (eql (position 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) 6))
	(test-t (eql (position 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) 3))
	(test-t (null (position 'a '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eql (position 'a #(a b c)) 0))
	(test-t (eql (position 'b #(a b c)) 1))
	(test-t (eql (position 'c #(a b c)) 2))
	(test-t (null (position 'x #(a b c))))
	(test-t (null (position 'a #(a b c) :start 1)))
	(test-t (null (position 'b #(a b c) :start 2)))
	(test-t (null (position 'c #(a b c) :start 3)))
	(test-t (null (position 'a #(a b c) :start 0 :end 0)))
	(test-t (null (position 'a #(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (position 'a #(a b c) :start 1 :end 1)))
	(test-t (null (position 'a #(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (position 'a #(a b c) :start 2 :end 2)))
	(test-t (null (position 'a #(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (position 'a #(a b c) :start 3 :end 3)))
	(test-t (null (position 'a #(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eql (position 'a #(a b c) :end nil) 0))
	(test-t (eql (position 'b #(a b c) :end nil) 1))
	(test-t (eql (position 'c #(a b c) :end nil) 2))
	(test-t (eql (position 'a #(a b c) :end 1) 0))
	(test-t (eql (position 'b #(a b c) :end 2) 1))
	(test-t (eql (position 'c #(a b c) :end 3) 2))
	(test-t (null (position 'a #(a b c) :end 0)))
	(test-t (null (position 'b #(a b c) :end 1)))
	(test-t (null (position 'c #(a b c) :end 2)))
	(test-t (null (position 'a #((a) (b) (c)))))
	(test-t (eql (position 'a #((a) (b) (c)) :key car) 0))
	(test-t (eql (position 'b #((a) (b) (c)) :key car) 1))
	(test-t (eql (position 'c #((a) (b) (c)) :key car) 2))
	(test-t (null (position 'z #((a) (b) (c)) :key car)))
	(test-t (null (position '(a) #((a) (b) (c)))))
	(test-t (eql (position '(a) #((a) (b) (c)) :test equal) 0))
	(test-t (null (position '("a") #(("a") ("b") ("c")))))
	(test-t (eql (position 3 #(0 1 2 3 4 5)) 3))
	(test-t (eql (position 3 #(0 1 2 3 4 5) :test <) 4))
	(test-t (eql (position 3 #(0 1 2 3 4 5) :test >) 0))
	(test-t (eql (position 'a #((a) (b) (c) (a a) (b b) (c c)) :key car) 0))
	(test-t (eql (position 'a #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 3))
	(test-t (eql (position 'b #((a) (b) (c) (a a) (b b) (c c)) :key car) 1))
	(test-t (eql (position 'b #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 4))
	(test-t (eql (position 'c #((a) (b) (c) (a a) (b b) (c c)) :key car) 2))
	(test-t (eql (position 'c #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 5))
	(test-t (null (position 'z #((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (position 'z #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (eql (position 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) 6))
	(test-t (eql (position 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) 6))
	(test-t (eql (position 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) 3))
	(test-t (null (position 'a #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (null (position #\z "abcABC")))
	(test-t (eql (position #\a "abcABC") 0))
	(test-t (eql (position #\A "abcABC") 3))
	(test-t (eql (position #\A "abcABC" :test char-equal) 0))
	(test-t (eql (position #\A "abcABC" :test char-equal :from-end t) 3))
	(test-t (eql (position #\a "abcABC" :test char-equal :from-end t) 3))
	(test-t (eql (position #\a "abcABC" :test char-equal :from-end t :end 4) 3))
	(test-t (eql (position #\a "abcABC" :test char-equal :from-end t :end 3) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '(a b c)) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) '(a b c)) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) '(a b c)) 2))
	(test-t (null (position-if (lambda (arg) (eq arg 'x)) '(a b c))))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 1)))
	(test-t (null (position-if (lambda (x) (eq x 'b)) '(a b c) :start 2)))
	(test-t (null (position-if (lambda (x) (eq x 'c)) '(a b c) :start 3)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 0 :end 0)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 1 :end 1)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 2 :end 2)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 3 :end 3)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '(a b c) :end nil) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) '(a b c) :end nil) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) '(a b c) :end nil) 2))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '(a b c) :end 1) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) '(a b c) :end 2) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) '(a b c) :end 3) 2))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '(a b c) :end 0)))
	(test-t (null (position-if (lambda (x) (eq x 'b)) '(a b c) :end 1)))
	(test-t (null (position-if (lambda (x) (eq x 'c)) '(a b c) :end 2)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '((a) (b) (c)))))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '((a) (b) (c)) :key car) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) '((a) (b) (c)) :key car) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) '((a) (b) (c)) :key car) 2))
	(test-t (null (position-if (lambda (x) (eq x 'z)) '((a) (b) (c)) :key car)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c)) :key car) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 3))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) '((a) (b) (c) (a a) (b b) (c c)) :key car) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 4))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) '((a) (b) (c) (a a) (b b) (c c)) :key car) 2))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 5))
	(test-t (null (position-if (lambda (x) (eq x 'z)) '((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (position-if (lambda (x) (eq x 'z)) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) 6))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) 6))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) 3))
	(test-t (null (position-if (lambda (x) (eq x 'a)) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #(a b c)) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) #(a b c)) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) #(a b c)) 2))
	(test-t (null (position-if (lambda (arg) (eq arg 'x)) #(a b c))))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 1)))
	(test-t (null (position-if (lambda (x) (eq x 'b)) #(a b c) :start 2)))
	(test-t (null (position-if (lambda (x) (eq x 'c)) #(a b c) :start 3)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 0 :end 0)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 1 :end 1)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 2 :end 2)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 3 :end 3)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #(a b c) :end nil) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) #(a b c) :end nil) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) #(a b c) :end nil) 2))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #(a b c) :end 1) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) #(a b c) :end 2) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) #(a b c) :end 3) 2))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #(a b c) :end 0)))
	(test-t (null (position-if (lambda (x) (eq x 'b)) #(a b c) :end 1)))
	(test-t (null (position-if (lambda (x) (eq x 'c)) #(a b c) :end 2)))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #((a) (b) (c)))))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #((a) (b) (c)) :key car) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) #((a) (b) (c)) :key car) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) #((a) (b) (c)) :key car) 2))
	(test-t (null (position-if (lambda (x) (eq x 'z)) #((a) (b) (c)) :key car)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c)) :key car) 0))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 3))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) #((a) (b) (c) (a a) (b b) (c c)) :key car) 1))
	(test-t (eql (position-if (lambda (x) (eq x 'b)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 4))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) #((a) (b) (c) (a a) (b b) (c c)) :key car) 2))
	(test-t (eql (position-if (lambda (x) (eq x 'c)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 5))
	(test-t (null (position-if (lambda (x) (eq x 'z)) #((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (position-if (lambda (x) (eq x 'z)) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) 6))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) 6))
	(test-t (eql (position-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) 3))
	(test-t (null (position-if (lambda (x) (eq x 'a)) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '(a b c)) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) '(a b c)) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) '(a b c)) 2))
	(test-t (null (position-if-not (lambda (arg) (not (eq arg 'x))) '(a b c))))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 1)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'b))) '(a b c) :start 2)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'c))) '(a b c) :start 3)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 0 :end 0)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 1 :end 1)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 2 :end 2)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 3 :end 3)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :end nil) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) '(a b c) :end nil) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) '(a b c) :end nil) 2))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :end 1) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) '(a b c) :end 2) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) '(a b c) :end 3) 2))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '(a b c) :end 0)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'b))) '(a b c) :end 1)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'c))) '(a b c) :end 2)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c)))))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c)) :key car) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) '((a) (b) (c)) :key car) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) '((a) (b) (c)) :key car) 2))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'z))) '((a) (b) (c)) :key car)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c)) :key car) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 3))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) '((a) (b) (c) (a a) (b b) (c c)) :key car) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 4))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) '((a) (b) (c) (a a) (b b) (c c)) :key car) 2))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 5))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'z))) '((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'z))) '((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) 6))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) 6))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) 3))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) '((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #(a b c)) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) #(a b c)) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) #(a b c)) 2))
	(test-t (null (position-if-not (lambda (arg) (not (eq arg 'x))) #(a b c))))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 1)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'b))) #(a b c) :start 2)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'c))) #(a b c) :start 3)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 0 :end 0)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 0 :end 0 :from-end t)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 1 :end 1)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 1 :end 1 :from-end t)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 2 :end 2)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 2 :end 2 :from-end t)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 3 :end 3)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :start 3 :end 3 :from-end t)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :end nil) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) #(a b c) :end nil) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) #(a b c) :end nil) 2))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :end 1) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) #(a b c) :end 2) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) #(a b c) :end 3) 2))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #(a b c) :end 0)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'b))) #(a b c) :end 1)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'c))) #(a b c) :end 2)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c)))))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c)) :key car) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) #((a) (b) (c)) :key car) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) #((a) (b) (c)) :key car) 2))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'z))) #((a) (b) (c)) :key car)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c)) :key car) 0))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 3))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) #((a) (b) (c) (a a) (b b) (c c)) :key car) 1))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'b))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 4))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) #((a) (b) (c) (a a) (b b) (c c)) :key car) 2))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'c))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t) 5))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'z))) #((a) (b) (c) (a a) (b b) (c c)) :key car)))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'z))) #((a) (b) (c) (a a) (b b) (c c)) :key car :from-end t)))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t) 6))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end nil) 6))
	(test-t (eql (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :end 6) 3))
	(test-t (null (position-if-not (lambda (x) (not (eq x 'a))) #((a) (b) (c) (a a) (b b) (c c) (a a a)) :key car :from-end t :start 1 :end 3)))
	
	(test-t (eql (search "dog" "it's a dog's life") 7))
	(test-t (eql (search '(0 1) '(2 4 6 1 3 5) :key oddp) 2))
	(test-t (null (search '(a b c) '(x y z))))
	(test-t (eql (search '() '(x y z)) 0))
	(test-t (eql (search '(a) '(a)) 0))
	(test-t (eql (search '(a b c) '(a b c x y z)) 0))
	(test-t (eql (search '(a b c) '(x a b c y z)) 1))
	(test-t (eql (search '(a b c) '(x y a b c z)) 2))
	(test-t (eql (search '(a b c) '(x y z a b c)) 3))
	(test-t (eql (search '(a b c) '(a b c a b c) :start2 1) 3))
	(test-t (eql (search '(a b c) '(a b c a b c) :start2 1 :end2 nil) 3))
	(test-t (eql (search '(a b c) '(a b c a b c) :start1 1 :start2 1 :end2 nil) 1))
	(test-t (eql (search '(a b c) '(a b c a b c) :start1 1 :end1 nil :start2 1 :end2 nil) 1))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 0 :end2 0)))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 1 :end2 1)))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 2 :end2 2)))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 3 :end2 3)))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 4 :end2 4)))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 5 :end2 5)))
	(test-t (null (search '(a b c) '(a b c a b c) :start2 6 :end2 6)))
	(test-t (eql (search '(a b c) '(a b c a b c)) 0))
	(test-t (eql (search '(a b c) '(a b c a b c) :from-end t) 3))
	(test-t (eql (search '(a b c) '(a b c a b c) :start2 3 :end2 6) 3))
	(test-t (eql (search '(a b c) '(a b c a b c) :start2 3 :end2 6 :from-end t) 3))
	(test-t (eql (search '(a b c) '(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6) 0))
	(test-t (eql (search '(a b c) '(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6 :from-end t) 3))
	(test-t (null (search '(#\a #\b #\c) '(#\A #\B #\C))))
	(test-t (eql (search '(#\a #\b #\c) '(#\A #\B #\C) :test char-equal) 0))
	(test-t (eql (search '(#\a #\b) '(#\a #\b #\x #\y #\z)) 0))
	(test-t (eql (search '(#\a #\b) '(#\a #\b #\x #\y #\z) :test char<) 1))
	(test-t (null (search '((a) (b)) '((x) (y) (z) (a) (b) (c)))))
	(test-t (eql (search '((a) (b)) '((x) (y) (z) (a) (b) (c)) :key car) 3))
	(test-t (eql (search '((a) (b)) '((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search '((a) (b)) '((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car :from-end t) 6))
	(test-t (eql (search '((a a) (b b)) '((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search '((a a) (b b)) '((a nil) (b t) (c nil) (x) (y) (z) (a 0) (b 1) (c 2)) :key car :from-end t) 6))
	(test-t (eql (search '(("a" a) ("b" b))
			     '(("a" nil) ("b" t) ("c" nil) ("x") ("y") ("z")
			       ("A" 0) ("B" 1) ("C" 2))
			     :start1 1
			     :end1 2
			     :start2 3
			     :end2 nil
			     :key car
			     :test string-equal
			     :from-end t)
		     7))
	(test-t (null (search #(a b c) '(x y z))))
	(test-t (eql (search #() '(x y z)) 0))
	(test-t (eql (search #(a) '(a)) 0))
	(test-t (eql (search #(a b c) '(a b c x y z)) 0))
	(test-t (eql (search #(a b c) '(x a b c y z)) 1))
	(test-t (eql (search #(a b c) '(x y a b c z)) 2))
	(test-t (eql (search #(a b c) '(x y z a b c)) 3))
	(test-t (eql (search #(a b c) '(a b c a b c) :start2 1) 3))
	(test-t (eql (search #(a b c) '(a b c a b c) :start2 1 :end2 nil) 3))
	(test-t (eql (search #(a b c) '(a b c a b c) :start1 1 :start2 1 :end2 nil) 1))
	(test-t (eql (search #(a b c) '(a b c a b c) :start1 1 :end1 nil :start2 1 :end2 nil) 1))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 0 :end2 0)))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 1 :end2 1)))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 2 :end2 2)))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 3 :end2 3)))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 4 :end2 4)))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 5 :end2 5)))
	(test-t (null (search #(a b c) '(a b c a b c) :start2 6 :end2 6)))
	(test-t (eql (search #(a b c) '(a b c a b c)) 0))
	(test-t (eql (search #(a b c) '(a b c a b c) :from-end t) 3))
	(test-t (eql (search #(a b c) '(a b c a b c) :start2 3 :end2 6) 3))
	(test-t (eql (search #(a b c) '(a b c a b c) :start2 3 :end2 6 :from-end t) 3))
	(test-t (eql (search #(a b c) '(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6) 0))
	(test-t (eql (search #(a b c) '(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6 :from-end t) 3))
	(test-t (null (search #(#\a #\b #\c) '(#\A #\B #\C))))
	(test-t (eql (search #(#\a #\b #\c) '(#\A #\B #\C) :test char-equal) 0))
	(test-t (eql (search #(#\a #\b) '(#\a #\b #\x #\y #\z)) 0))
	(test-t (eql (search #(#\a #\b) '(#\a #\b #\x #\y #\z) :test char<) 1))
	(test-t (null (search #((a) (b)) '((x) (y) (z) (a) (b) (c)))))
	(test-t (eql (search #((a) (b)) '((x) (y) (z) (a) (b) (c)) :key car) 3))
	(test-t (eql (search #((a) (b)) '((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search #((a) (b)) '((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car :from-end t) 6))
	(test-t (eql (search #((a a) (b b)) '((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search #((a a) (b b)) '((a nil) (b t) (c nil) (x) (y) (z) (a 0) (b 1) (c 2)) :key car :from-end t) 6))
	(test-t (eql (search #(("a" a) ("b" b))
			     '(("a" nil) ("b" t) ("c" nil) ("x") ("y") ("z")
			       ("A" 0) ("B" 1) ("C" 2))
			     :start1 1
			     :end1 2
			     :start2 3
			     :end2 nil
			     :key car
			     :test string-equal
			     :from-end t)
		     7))
	(test-t (null (search '(a b c) #(x y z))))
	(test-t (eql (search '() #(x y z)) 0))
	(test-t (eql (search '(a) #(a)) 0))
	(test-t (eql (search '(a b c) #(a b c x y z)) 0))
	(test-t (eql (search '(a b c) #(x a b c y z)) 1))
	(test-t (eql (search '(a b c) #(x y a b c z)) 2))
	(test-t (eql (search '(a b c) #(x y z a b c)) 3))
	(test-t (eql (search '(a b c) #(a b c a b c) :start2 1) 3))
	(test-t (eql (search '(a b c) #(a b c a b c) :start2 1 :end2 nil) 3))
	(test-t (eql (search '(a b c) #(a b c a b c) :start1 1 :start2 1 :end2 nil) 1))
	(test-t (eql (search '(a b c) #(a b c a b c) :start1 1 :end1 nil :start2 1 :end2 nil) 1))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 0 :end2 0)))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 1 :end2 1)))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 2 :end2 2)))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 3 :end2 3)))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 4 :end2 4)))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 5 :end2 5)))
	(test-t (null (search '(a b c) #(a b c a b c) :start2 6 :end2 6)))
	(test-t (eql (search '(a b c) #(a b c a b c)) 0))
	(test-t (eql (search '(a b c) #(a b c a b c) :from-end t) 3))
	(test-t (eql (search '(a b c) #(a b c a b c) :start2 3 :end2 6) 3))
	(test-t (eql (search '(a b c) #(a b c a b c) :start2 3 :end2 6 :from-end t) 3))
	(test-t (eql (search '(a b c) #(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6) 0))
	(test-t (eql (search '(a b c) #(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6 :from-end t) 3))
	(test-t (null (search '(#\a #\b #\c) #(#\A #\B #\C))))
	(test-t (eql (search '(#\a #\b #\c) #(#\A #\B #\C) :test char-equal) 0))
	(test-t (eql (search '(#\a #\b) #(#\a #\b #\x #\y #\z)) 0))
	(test-t (eql (search '(#\a #\b) #(#\a #\b #\x #\y #\z) :test char<) 1))
	(test-t (null (search '((a) (b)) #((x) (y) (z) (a) (b) (c)))))
	(test-t (eql (search '((a) (b)) #((x) (y) (z) (a) (b) (c)) :key car) 3))
	(test-t (eql (search '((a) (b)) #((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search '((a) (b)) #((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car :from-end t) 6))
	(test-t (eql (search '((a a) (b b)) #((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search '((a a) (b b)) #((a nil) (b t) (c nil) (x) (y) (z) (a 0) (b 1) (c 2)) :key car :from-end t) 6))
	(test-t (eql (search '(("a" a) ("b" b))
			     #(("a" nil) ("b" t) ("c" nil) ("x") ("y") ("z")
			       ("A" 0) ("B" 1) ("C" 2))
			     :start1 1
			     :end1 2
			     :start2 3
			     :end2 nil
			     :key car
			     :test string-equal
			     :from-end t)
		     7))
	(test-t (null (search #(a b c) #(x y z))))
	(test-t (eql (search #() #(x y z)) 0))
	(test-t (eql (search #(a) #(a)) 0))
	(test-t (eql (search #(a b c) #(a b c x y z)) 0))
	(test-t (eql (search #(a b c) #(x a b c y z)) 1))
	(test-t (eql (search #(a b c) #(x y a b c z)) 2))
	(test-t (eql (search #(a b c) #(x y z a b c)) 3))
	(test-t (eql (search #(a b c) #(a b c a b c) :start2 1) 3))
	(test-t (eql (search #(a b c) #(a b c a b c) :start2 1 :end2 nil) 3))
	(test-t (eql (search #(a b c) #(a b c a b c) :start1 1 :start2 1 :end2 nil) 1))
	(test-t (eql (search #(a b c) #(a b c a b c) :start1 1 :end1 nil :start2 1 :end2 nil) 1))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 0 :end2 0)))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 1 :end2 1)))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 2 :end2 2)))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 3 :end2 3)))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 4 :end2 4)))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 5 :end2 5)))
	(test-t (null (search #(a b c) #(a b c a b c) :start2 6 :end2 6)))
	(test-t (eql (search #(a b c) #(a b c a b c)) 0))
	(test-t (eql (search #(a b c) #(a b c a b c) :from-end t) 3))
	(test-t (eql (search #(a b c) #(a b c a b c) :start2 3 :end2 6) 3))
	(test-t (eql (search #(a b c) #(a b c a b c) :start2 3 :end2 6 :from-end t) 3))
	(test-t (eql (search #(a b c) #(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6) 0))
	(test-t (eql (search #(a b c) #(a b c a b c) :start1 0 :end1 2 :start2 0 :end2 6 :from-end t) 3))
	(test-t (null (search #(#\a #\b #\c) #(#\A #\B #\C))))
	(test-t (eql (search #(#\a #\b #\c) #(#\A #\B #\C) :test char-equal) 0))
	(test-t (eql (search #(#\a #\b) #(#\a #\b #\x #\y #\z)) 0))
	(test-t (eql (search #(#\a #\b) #(#\a #\b #\x #\y #\z) :test char<) 1))
	(test-t (null (search #((a) (b)) #((x) (y) (z) (a) (b) (c)))))
	(test-t (eql (search #((a) (b)) #((x) (y) (z) (a) (b) (c)) :key car) 3))
	(test-t (eql (search #((a) (b)) #((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search #((a) (b)) #((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car :from-end t) 6))
	(test-t (eql (search #((a a) (b b)) #((a) (b) (c) (x) (y) (z) (a) (b) (c)) :key car) 0))
	(test-t (eql (search #((a a) (b b)) #((a nil) (b t) (c nil) (x) (y) (z) (a 0) (b 1) (c 2)) :key car :from-end t) 6))
	(test-t (eql (search #(("a" a) ("b" b))
			     #(("a" nil) ("b" t) ("c" nil) ("x") ("y") ("z")
			       ("A" 0) ("B" 1) ("C" 2))
			     :start1 1
			     :end1 2
			     :start2 3
			     :end2 nil
			     :key car
			     :test string-equal
			     :from-end t)
		     7))
	(test-t (null (search "peace" "LOVE&PEACE")))
	(test-t (eql (search "peace" "LOVE&PEACE" :test char-equal) 5))
	(test-t (null (search "PeAcE" "LoVe&pEaCe")))
	(test-t (eql (search "PeAcE" "LoVe&pEaCe" :key char-upcase) 5))
	(test-t (eql (search "abc" "abc xyz abc" :from-end t) 8))
	(test-t (eql (search "abc" "abc xyz abc xyz abc xyz abc" :start2 8 :end2 19) 8))
	(test-t (eql (search "abc" "abc xyz abc xyz abc xyz abc" :from-end t :start2 8 :end2 19) 16))
	(test-t (eql (mismatch "abcd" "ABCDE" :test char-equal) 4))
	(test-t (eql (mismatch '(3 2 1 1 2 3) '(1 2 3) :from-end t) 3))
	(test-t (null (mismatch '(1 2 3 4 5 6) '(3 4 5 6 7) :start1 2 :end2 4)))
	(test-t (null (mismatch '() '())))
	(test-t (eql (mismatch '(a b c) '(x y z)) 0))
	(test-t (eql (mismatch '() '(x y z)) 0))
	(test-t (eql (mismatch '(x y z) '()) 0))
	(test-t (null (mismatch '(a) '(a))))
	(test-t (eql (mismatch '(a b c x y z) '(a b c)) 3))
	(test-t (null (mismatch '(a b c) '(a b c))))
	(test-t (eql (mismatch '(a b c d e f) '(a b c)) 3))
	(test-t (eql (mismatch '(a b c) '(a b c d e f)) 3))
	(test-t (eql (mismatch '(a b c) '(a b x)) 2))
	(test-t (eql (mismatch '(a b c) '(a x c)) 1))
	(test-t (eql (mismatch '(a b c) '(x b c)) 0))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :start1 3) 6))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :start1 3 :end1 nil) 6))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :start1 3 :end1 4) 4))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :start1 3 :end1 3) 3))
	(test-t (null (mismatch '(x y z) '() :start1 0 :end1 0)))
	(test-t (null (mismatch '(x y z) '() :start1 1 :end1 1)))
	(test-t (null (mismatch '(x y z) '() :start1 2 :end1 2)))
	(test-t (null (mismatch '(x y z) '() :start1 3 :end1 3)))
	(test-t (null (mismatch '(x y z) '() :start1 0 :end1 0 :start2 0 :end2 0)))
	(test-t (null (mismatch '(x y z) '() :start1 1 :end1 1 :start2 1 :end2 1)))
	(test-t (null (mismatch '(x y z) '() :start1 2 :end1 2 :start2 2 :end2 2)))
	(test-t (null (mismatch '(x y z) '() :start1 3 :end1 3 :start2 3 :end2 3)))
	(test-t (null (mismatch '(x y z) '() :start1 0 :end1 0 :start2 3 :end2 3)))
	(test-t (null (mismatch '(x y z) '() :start1 1 :end1 1 :start2 2 :end2 2)))
	(test-t (null (mismatch '(x y z) '() :start1 2 :end1 2 :start2 1 :end2 1)))
	(test-t (null (mismatch '(x y z) '() :start1 3 :end1 3 :start2 0 :end2 0)))
	(test-t (eql (mismatch '(x y z) '(a b c) :start1 0 :end1 0) 0))
	(test-t (eql (mismatch '(x y z) '(a b c) :start1 1 :end1 1) 1))
	(test-t (eql (mismatch '(x y z) '(a b c) :start1 2 :end1 2) 2))
	(test-t (eql (mismatch '(x y z) '(a b c) :start1 3 :end1 3) 3))
	(test-t (eql (mismatch '(x y z) '(x y z) :start1 0 :end1 1) 1))
	(test-t (eql (mismatch '(x y z) '(x y z) :start1 0 :end1 2) 2))
	(test-t (eql (mismatch '(x y z) '(x y z Z) :start1 0 :end1 3) 3))
	(test-t (null (mismatch '(x y z) '(x y z) :start1 0 :end1 3)))
	(test-t (eql (mismatch '(a b c x y z) '(x y z a b c)) 0))
	(test-t (eql (mismatch '(a b c x y z) '(x y z a b c) :start1 3) 6))
	(test-t (eql (mismatch '(a b c x y z a b c) '(x y z a b c x y z) :start1 3) 9))
	(test-t (eql (mismatch '(a b c x y z a b c) '(x y z a b c x y z) :start1 6) 6))
	(test-t (eql (mismatch '(a b c x y z a b c) '(x y z a b c x y z) :start1 6 :start2 3) 9))
	(test-t (eql (mismatch '(a b c x y z a b c) '(x y z a b c x y z) :start1 0 :start2 3) 6))
	(test-t (eql (mismatch '(a b c) '(a b c x y z)) 3))
	(test-t (eql (mismatch '(a b c) '(x a b c y z)) 0))
	(test-t (eql (mismatch '(a b c) '(x a b c y z) :start2 1) 3))
	(test-t (eql (mismatch '(a b c) '(x a b c y z) :start2 1 :end2 nil) 3))
	(test-t (null (mismatch '(a b c) '(x a b c y z) :start2 1 :end2 4)))
	(test-t (eql (mismatch '(a b c d e) '(c d)) 0))
	(test-t (eql (mismatch '(a b c d e) '(c d) :start1 2) 4))
	(test-t (eql (mismatch '(a b c d e) '(c d) :start1 2 :end1 3) 3))
	(test-t (eql (mismatch '(a b c d e) '(c d) :start1 2 :start2 1) 2))
	(test-t (eql (mismatch '(a b c d e) '(c d) :start1 3 :start2 1) 4))
	(test-t (eql (mismatch '(a b c d e) '(c d) :start1 2 :end2 1) 3))
	(test-t (null (mismatch '(a b c d) '(a b c d) :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch '(a b c d) '(a b c d) :start1 1 :end1 3 :start2 1 :end2 3)))
	(test-t (null (mismatch '(a b c d) '(a b c d) :start1 1 :end1 4 :start2 1 :end2 4)))
	(test-t (eql (mismatch '(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 1) 1))
	(test-t (eql (mismatch '(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 2) 2))
	(test-t (eql (mismatch '(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 3) 3))
	(test-t (null (mismatch '(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 4)))
	(test-t (eql (mismatch '(a b c d) '(a b c d) :start1 1 :end1 1 :start2 1) 1))
	(test-t (eql (mismatch '(a b c d) '(a b c d) :start1 1 :end1 2 :start2 1) 2))
	(test-t (eql (mismatch '(a b c d) '(a b c d) :start1 1 :end1 3 :start2 1) 3))
	(test-t (null (mismatch '(a b c d) '(a b c d) :start1 1 :end1 4 :start2 1)))
	(test-t (null (mismatch '(a b c) '(a b c) :from-end t)))
	(test-t (eql (mismatch '(a b c d) '(a b c) :from-end t) 4))
	(test-t (eql (mismatch '(a b c) '(c) :from-end t) 2))
	(test-t (eql (mismatch '(a b c) '(z a b c) :from-end t) 0))
	(test-t (eql (mismatch '(a b c) '(x y z a b c) :from-end t) 0))
	(test-t (eql (mismatch '(x y z a b c) '(a b c) :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c) '(a b c) :end1 3 :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c) '(a b c) :end1 5 :from-end t) 5))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :end1 6 :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :start1 2 :end1 6 :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :from-end t :start1 2 :end1 5 :start2 1 :end2 2 ) 4))
	(test-t (eql (mismatch '(x y z a b c x y z) '(a b c) :start1 2 :end1 5 :start2 1 :end2 2 ) 2))
	(test-t (eql (mismatch '((a) (b) (c)) '((a) (b) (c))) 0))
	(test-t (null (mismatch '((a) (b) (c)) '((a) (b) (c)) :key car)))
	(test-t (null (mismatch '((a) (b) (c)) '((a) (b) (c)) :test equal)))
	(test-t (eql (mismatch '(#(a) #(b) #(c)) '(#(a) #(b) #(c))) 0))
	(test-t (null (mismatch '(#(a) #(b) #(c)) '(#(a) #(b) #(c)) :test equalp)))
	(test-t (eql (mismatch '((a) (b) (c) (d)) '((a) (b) (c)) :key car) 3))
	(test-t (eql (mismatch '((a) (b) (c)) '((a) (b) (c) (d)) :key car) 3))
	(test-t (eql (mismatch '(#\a #\b #\c) '(#\A #\B #\C)) 0))
	(test-t (null (mismatch '(#\a #\b #\c) '(#\A #\B #\C) :key char-upcase)))
	(test-t (null (mismatch '(#\a #\b #\c) '(#\A #\B #\C) :key char-downcase)))
	(test-t (null (mismatch '(#\a #\b #\c) '(#\A #\B #\C) :key char-upcase :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch '(#\a #\b #\c) '(#\A #\B #\C) :key char-upcase :start1 2 :start2 2)))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f))) 0))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f)) :key cdr) 0))
	(test-t (null (mismatch '((a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f)) :key cdr :test equal)))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal) 3))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 4))
	(test-t (eql (mismatch '((a a a) (a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 1))
	(test-t (null (mismatch '((a a a) (a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1 :end1 4)))
	(test-t (eql (mismatch '((a a a) (a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1) 5))
	(test-t (eql (mismatch '((a a a) (a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :end1 3 :start2 1 :end2 2) 2))
	(test-t (null (mismatch #() '())))
	(test-t (eql (mismatch #(a b c) '(x y z)) 0))
	(test-t (eql (mismatch #() '(x y z)) 0))
	(test-t (eql (mismatch #(x y z) '()) 0))
	(test-t (null (mismatch #(a) '(a))))
	(test-t (eql (mismatch #(a b c x y z) '(a b c)) 3))
	(test-t (null (mismatch #(a b c) '(a b c))))
	(test-t (eql (mismatch #(a b c d e f) '(a b c)) 3))
	(test-t (eql (mismatch #(a b c) '(a b c d e f)) 3))
	(test-t (eql (mismatch #(a b c) '(a b x)) 2))
	(test-t (eql (mismatch #(a b c) '(a x c)) 1))
	(test-t (eql (mismatch #(a b c) '(x b c)) 0))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :start1 3) 6))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :start1 3 :end1 nil) 6))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :start1 3 :end1 4) 4))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :start1 3 :end1 3) 3))
	(test-t (null (mismatch #(x y z) '() :start1 0 :end1 0)))
	(test-t (null (mismatch #(x y z) '() :start1 1 :end1 1)))
	(test-t (null (mismatch #(x y z) '() :start1 2 :end1 2)))
	(test-t (null (mismatch #(x y z) '() :start1 3 :end1 3)))
	(test-t (eql (mismatch #(x y z) '(a b c) :start1 0 :end1 0) 0))
	(test-t (eql (mismatch #(x y z) '(a b c) :start1 1 :end1 1) 1))
	(test-t (eql (mismatch #(x y z) '(a b c) :start1 2 :end1 2) 2))
	(test-t (eql (mismatch #(x y z) '(a b c) :start1 3 :end1 3) 3))
	(test-t (eql (mismatch #(x y z) '(x y z) :start1 0 :end1 1) 1))
	(test-t (eql (mismatch #(x y z) '(x y z) :start1 0 :end1 2) 2))
	(test-t (eql (mismatch #(x y z) '(x y z Z) :start1 0 :end1 3) 3))
	(test-t (null (mismatch #(x y z) '(x y z) :start1 0 :end1 3)))
	(test-t (eql (mismatch #(a b c x y z) '(x y z a b c)) 0))
	(test-t (eql (mismatch #(a b c x y z) '(x y z a b c) :start1 3) 6))
	(test-t (eql (mismatch #(a b c x y z a b c) '(x y z a b c x y z) :start1 3) 9))
	(test-t (eql (mismatch #(a b c x y z a b c) '(x y z a b c x y z) :start1 6) 6))
	(test-t (eql (mismatch #(a b c x y z a b c) '(x y z a b c x y z) :start1 6 :start2 3) 9))
	(test-t (eql (mismatch #(a b c x y z a b c) '(x y z a b c x y z) :start1 0 :start2 3) 6))
	(test-t (eql (mismatch #(a b c) '(a b c x y z)) 3))
	(test-t (eql (mismatch #(a b c) '(x a b c y z)) 0))
	(test-t (eql (mismatch #(a b c) '(x a b c y z) :start2 1) 3))
	(test-t (eql (mismatch #(a b c) '(x a b c y z) :start2 1 :end2 nil) 3))
	(test-t (null (mismatch #(a b c) '(x a b c y z) :start2 1 :end2 4)))
	(test-t (eql (mismatch #(a b c d e) '(c d)) 0))
	(test-t (eql (mismatch #(a b c d e) '(c d) :start1 2) 4))
	(test-t (eql (mismatch #(a b c d e) '(c d) :start1 2 :end1 3) 3))
	(test-t (eql (mismatch #(a b c d e) '(c d) :start1 2 :start2 1) 2))
	(test-t (eql (mismatch #(a b c d e) '(c d) :start1 3 :start2 1) 4))
	(test-t (eql (mismatch #(a b c d e) '(c d) :start1 2 :end2 1) 3))
	(test-t (null (mismatch #(a b c d) '(a b c d) :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch #(a b c d) '(a b c d) :start1 1 :end1 3 :start2 1 :end2 3)))
	(test-t (null (mismatch #(a b c d) '(a b c d) :start1 1 :end1 4 :start2 1 :end2 4)))
	(test-t (eql (mismatch #(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 1) 1))
	(test-t (eql (mismatch #(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 2) 2))
	(test-t (eql (mismatch #(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 3) 3))
	(test-t (null (mismatch #(a b c d) '(a b c d) :start1 1 :end1 nil :start2 1 :end2 4)))
	(test-t (eql (mismatch #(a b c d) '(a b c d) :start1 1 :end1 1 :start2 1) 1))
	(test-t (eql (mismatch #(a b c d) '(a b c d) :start1 1 :end1 2 :start2 1) 2))
	(test-t (eql (mismatch #(a b c d) '(a b c d) :start1 1 :end1 3 :start2 1) 3))
	(test-t (null (mismatch #(a b c d) '(a b c d) :start1 1 :end1 4 :start2 1)))
	(test-t (null (mismatch #(a b c) '(a b c) :from-end t)))
	(test-t (eql (mismatch #(a b c d) '(a b c) :from-end t) 4))
	(test-t (eql (mismatch #(a b c) '(c) :from-end t) 2))
	(test-t (eql (mismatch #(a b c) '(z a b c) :from-end t) 0))
	(test-t (eql (mismatch #(a b c) '(x y z a b c) :from-end t) 0))
	(test-t (eql (mismatch #(x y z a b c) '(a b c) :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c) '(a b c) :end1 3 :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c) '(a b c) :end1 5 :from-end t) 5))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :end1 6 :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :start1 2 :end1 6 :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :from-end t :start1 2 :end1 5 :start2 1 :end2 2 ) 4))
	(test-t (eql (mismatch #(x y z a b c x y z) '(a b c) :start1 2 :end1 5 :start2 1 :end2 2 ) 2))
	(test-t (eql (mismatch #((a) (b) (c)) '((a) (b) (c))) 0))
	(test-t (null (mismatch #((a) (b) (c)) '((a) (b) (c)) :key car)))
	(test-t (null (mismatch #((a) (b) (c)) '((a) (b) (c)) :test equal)))
	(test-t (eql (mismatch #(#(a) #(b) #(c)) '(#(a) #(b) #(c))) 0))
	(test-t (null (mismatch #(#(a) #(b) #(c)) '(#(a) #(b) #(c)) :test equalp)))
	(test-t (eql (mismatch #((a) (b) (c) (d)) '((a) (b) (c)) :key car) 3))
	(test-t (eql (mismatch #((a) (b) (c)) '((a) (b) (c) (d)) :key car) 3))
	(test-t (eql (mismatch #(#\a #\b #\c) '(#\A #\B #\C)) 0))
	(test-t (null (mismatch #(#\a #\b #\c) '(#\A #\B #\C) :key char-upcase)))
	(test-t (null (mismatch #(#\a #\b #\c) '(#\A #\B #\C) :key char-downcase)))
	(test-t (null (mismatch #(#\a #\b #\c) '(#\A #\B #\C) :key char-upcase :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch #(#\a #\b #\c) '(#\A #\B #\C) :key char-upcase :start1 2 :start2 2)))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f))) 0))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f)) :key cdr) 0))
	(test-t (null (mismatch #((a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f)) :key cdr :test equal)))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal) 3))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 4))
	(test-t (eql (mismatch #((a a a) (a b c) (b c d) (d e f)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 1))
	(test-t (null (mismatch #((a a a) (a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1 :end1 4)))
	(test-t (eql (mismatch #((a a a) (a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1) 5))
	(test-t (eql (mismatch #((a a a) (a b c) (b c d) (d e f) (e f g)) '((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :end1 3 :start2 1 :end2 2) 2))
	(test-t (null (mismatch '() #())))
	(test-t (eql (mismatch '(a b c) #(x y z)) 0))
	(test-t (eql (mismatch '() #(x y z)) 0))
	(test-t (eql (mismatch '(x y z) #()) 0))
	(test-t (null (mismatch '(a) #(a))))
	(test-t (eql (mismatch '(a b c x y z) #(a b c)) 3))
	(test-t (null (mismatch '(a b c) #(a b c))))
	(test-t (eql (mismatch '(a b c d e f) #(a b c)) 3))
	(test-t (eql (mismatch '(a b c) #(a b c d e f)) 3))
	(test-t (eql (mismatch '(a b c) #(a b x)) 2))
	(test-t (eql (mismatch '(a b c) #(a x c)) 1))
	(test-t (eql (mismatch '(a b c) #(x b c)) 0))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :start1 3) 6))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :start1 3 :end1 nil) 6))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :start1 3 :end1 4) 4))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :start1 3 :end1 3) 3))
	(test-t (null (mismatch '(x y z) #() :start1 0 :end1 0)))
	(test-t (null (mismatch '(x y z) #() :start1 1 :end1 1)))
	(test-t (null (mismatch '(x y z) #() :start1 2 :end1 2)))
	(test-t (null (mismatch '(x y z) #() :start1 3 :end1 3)))
	(test-t (eql (mismatch '(x y z) #(a b c) :start1 0 :end1 0) 0))
	(test-t (eql (mismatch '(x y z) #(a b c) :start1 1 :end1 1) 1))
	(test-t (eql (mismatch '(x y z) #(a b c) :start1 2 :end1 2) 2))
	(test-t (eql (mismatch '(x y z) #(a b c) :start1 3 :end1 3) 3))
	(test-t (eql (mismatch '(x y z) #(x y z) :start1 0 :end1 1) 1))
	(test-t (eql (mismatch '(x y z) #(x y z) :start1 0 :end1 2) 2))
	(test-t (eql (mismatch '(x y z) #(x y z Z) :start1 0 :end1 3) 3))
	(test-t (null (mismatch '(x y z) #(x y z) :start1 0 :end1 3)))
	(test-t (eql (mismatch '(a b c x y z) #(x y z a b c)) 0))
	(test-t (eql (mismatch '(a b c x y z) #(x y z a b c) :start1 3) 6))
	(test-t (eql (mismatch '(a b c x y z a b c) #(x y z a b c x y z) :start1 3) 9))
	(test-t (eql (mismatch '(a b c x y z a b c) #(x y z a b c x y z) :start1 6) 6))
	(test-t (eql (mismatch '(a b c x y z a b c) #(x y z a b c x y z) :start1 6 :start2 3) 9))
	(test-t (eql (mismatch '(a b c x y z a b c) #(x y z a b c x y z) :start1 0 :start2 3) 6))
	(test-t (eql (mismatch '(a b c) #(a b c x y z)) 3))
	(test-t (eql (mismatch '(a b c) #(x a b c y z)) 0))
	(test-t (eql (mismatch '(a b c) #(x a b c y z) :start2 1) 3))
	(test-t (eql (mismatch '(a b c) #(x a b c y z) :start2 1 :end2 nil) 3))
	(test-t (null (mismatch '(a b c) #(x a b c y z) :start2 1 :end2 4)))
	(test-t (eql (mismatch '(a b c d e) #(c d)) 0))
	(test-t (eql (mismatch '(a b c d e) #(c d) :start1 2) 4))
	(test-t (eql (mismatch '(a b c d e) #(c d) :start1 2 :end1 3) 3))
	(test-t (eql (mismatch '(a b c d e) #(c d) :start1 2 :start2 1) 2))
	(test-t (eql (mismatch '(a b c d e) #(c d) :start1 3 :start2 1) 4))
	(test-t (eql (mismatch '(a b c d e) #(c d) :start1 2 :end2 1) 3))
	(test-t (null (mismatch '(a b c d) #(a b c d) :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch '(a b c d) #(a b c d) :start1 1 :end1 3 :start2 1 :end2 3)))
	(test-t (null (mismatch '(a b c d) #(a b c d) :start1 1 :end1 4 :start2 1 :end2 4)))
	(test-t (eql (mismatch '(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 1) 1))
	(test-t (eql (mismatch '(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 2) 2))
	(test-t (eql (mismatch '(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 3) 3))
	(test-t (null (mismatch '(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 4)))
	(test-t (eql (mismatch '(a b c d) #(a b c d) :start1 1 :end1 1 :start2 1) 1))
	(test-t (eql (mismatch '(a b c d) #(a b c d) :start1 1 :end1 2 :start2 1) 2))
	(test-t (eql (mismatch '(a b c d) #(a b c d) :start1 1 :end1 3 :start2 1) 3))
	(test-t (null (mismatch '(a b c d) #(a b c d) :start1 1 :end1 4 :start2 1)))
	(test-t (null (mismatch '(a b c) #(a b c) :from-end t)))
	(test-t (eql (mismatch '(a b c d) #(a b c) :from-end t) 4))
	(test-t (eql (mismatch '(a b c) #(c) :from-end t) 2))
	(test-t (eql (mismatch '(a b c) #(z a b c) :from-end t) 0))
	(test-t (eql (mismatch '(a b c) #(x y z a b c) :from-end t) 0))
	(test-t (eql (mismatch '(x y z a b c) #(a b c) :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c) #(a b c) :end1 3 :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c) #(a b c) :end1 5 :from-end t) 5))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :end1 6 :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :start1 2 :end1 6 :from-end t) 3))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :from-end t :start1 2 :end1 5 :start2 1 :end2 2 ) 4))
	(test-t (eql (mismatch '(x y z a b c x y z) #(a b c) :start1 2 :end1 5 :start2 1 :end2 2 ) 2))
	(test-t (eql (mismatch '((a) (b) (c)) #((a) (b) (c))) 0))
	(test-t (null (mismatch '((a) (b) (c)) #((a) (b) (c)) :key car)))
	(test-t (null (mismatch '((a) (b) (c)) #((a) (b) (c)) :test equal)))
	(test-t (eql (mismatch '(#(a) #(b) #(c)) #(#(a) #(b) #(c))) 0))
	(test-t (null (mismatch '(#(a) #(b) #(c)) #(#(a) #(b) #(c)) :test equalp)))
	(test-t (eql (mismatch '((a) (b) (c) (d)) #((a) (b) (c)) :key car) 3))
	(test-t (eql (mismatch '((a) (b) (c)) #((a) (b) (c) (d)) :key car) 3))
	(test-t (eql (mismatch '(#\a #\b #\c) #(#\A #\B #\C)) 0))
	(test-t (null (mismatch '(#\a #\b #\c) #(#\A #\B #\C) :key char-upcase)))
	(test-t (null (mismatch '(#\a #\b #\c) #(#\A #\B #\C) :key char-downcase)))
	(test-t (null (mismatch '(#\a #\b #\c) #(#\A #\B #\C) :key char-upcase :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch '(#\a #\b #\c) #(#\A #\B #\C) :key char-upcase :start1 2 :start2 2)))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f))) 0))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f)) :key cdr) 0))
	(test-t (null (mismatch '((a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f)) :key cdr :test equal)))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal) 3))
	(test-t (eql (mismatch '((a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 4))
	(test-t (eql (mismatch '((a a a) (a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 1))
	(test-t (null (mismatch '((a a a) (a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1 :end1 4)))
	(test-t (eql (mismatch '((a a a) (a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1) 5))
	(test-t (eql (mismatch '((a a a) (a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :end1 3 :start2 1 :end2 2) 2))
	(test-t (null (mismatch #() #())))
	(test-t (eql (mismatch #(a b c) #(x y z)) 0))
	(test-t (eql (mismatch #() #(x y z)) 0))
	(test-t (eql (mismatch #(x y z) #()) 0))
	(test-t (null (mismatch #(a) #(a))))
	(test-t (eql (mismatch #(a b c x y z) #(a b c)) 3))
	(test-t (null (mismatch #(a b c) #(a b c))))
	(test-t (eql (mismatch #(a b c d e f) #(a b c)) 3))
	(test-t (eql (mismatch #(a b c) #(a b c d e f)) 3))
	(test-t (eql (mismatch #(a b c) #(a b x)) 2))
	(test-t (eql (mismatch #(a b c) #(a x c)) 1))
	(test-t (eql (mismatch #(a b c) #(x b c)) 0))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :start1 3) 6))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :start1 3 :end1 nil) 6))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :start1 3 :end1 4) 4))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :start1 3 :end1 3) 3))
	(test-t (null (mismatch #(x y z) #() :start1 0 :end1 0)))
	(test-t (null (mismatch #(x y z) #() :start1 1 :end1 1)))
	(test-t (null (mismatch #(x y z) #() :start1 2 :end1 2)))
	(test-t (null (mismatch #(x y z) #() :start1 3 :end1 3)))
	(test-t (eql (mismatch #(x y z) #(a b c) :start1 0 :end1 0) 0))
	(test-t (eql (mismatch #(x y z) #(a b c) :start1 1 :end1 1) 1))
	(test-t (eql (mismatch #(x y z) #(a b c) :start1 2 :end1 2) 2))
	(test-t (eql (mismatch #(x y z) #(a b c) :start1 3 :end1 3) 3))
	(test-t (eql (mismatch #(x y z) #(x y z) :start1 0 :end1 1) 1))
	(test-t (eql (mismatch #(x y z) #(x y z) :start1 0 :end1 2) 2))
	(test-t (eql (mismatch #(x y z) #(x y z Z) :start1 0 :end1 3) 3))
	(test-t (null (mismatch #(x y z) #(x y z) :start1 0 :end1 3)))
	(test-t (eql (mismatch #(a b c x y z) #(x y z a b c)) 0))
	(test-t (eql (mismatch #(a b c x y z) #(x y z a b c) :start1 3) 6))
	(test-t (eql (mismatch #(a b c x y z a b c) #(x y z a b c x y z) :start1 3) 9))
	(test-t (eql (mismatch #(a b c x y z a b c) #(x y z a b c x y z) :start1 6) 6))
	(test-t (eql (mismatch #(a b c x y z a b c) #(x y z a b c x y z) :start1 6 :start2 3) 9))
	(test-t (eql (mismatch #(a b c x y z a b c) #(x y z a b c x y z) :start1 0 :start2 3) 6))
	(test-t (eql (mismatch #(a b c) #(a b c x y z)) 3))
	(test-t (eql (mismatch #(a b c) #(x a b c y z)) 0))
	(test-t (eql (mismatch #(a b c) #(x a b c y z) :start2 1) 3))
	(test-t (eql (mismatch #(a b c) #(x a b c y z) :start2 1 :end2 nil) 3))
	(test-t (null (mismatch #(a b c) #(x a b c y z) :start2 1 :end2 4)))
	(test-t (eql (mismatch #(a b c d e) #(c d)) 0))
	(test-t (eql (mismatch #(a b c d e) #(c d) :start1 2) 4))
	(test-t (eql (mismatch #(a b c d e) #(c d) :start1 2 :end1 3) 3))
	(test-t (eql (mismatch #(a b c d e) #(c d) :start1 2 :start2 1) 2))
	(test-t (eql (mismatch #(a b c d e) #(c d) :start1 3 :start2 1) 4))
	(test-t (eql (mismatch #(a b c d e) #(c d) :start1 2 :end2 1) 3))
	(test-t (null (mismatch #(a b c d) #(a b c d) :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch #(a b c d) #(a b c d) :start1 1 :end1 3 :start2 1 :end2 3)))
	(test-t (null (mismatch #(a b c d) #(a b c d) :start1 1 :end1 4 :start2 1 :end2 4)))
	(test-t (eql (mismatch #(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 1) 1))
	(test-t (eql (mismatch #(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 2) 2))
	(test-t (eql (mismatch #(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 3) 3))
	(test-t (null (mismatch #(a b c d) #(a b c d) :start1 1 :end1 nil :start2 1 :end2 4)))
	(test-t (eql (mismatch #(a b c d) #(a b c d) :start1 1 :end1 1 :start2 1) 1))
	(test-t (eql (mismatch #(a b c d) #(a b c d) :start1 1 :end1 2 :start2 1) 2))
	(test-t (eql (mismatch #(a b c d) #(a b c d) :start1 1 :end1 3 :start2 1) 3))
	(test-t (null (mismatch #(a b c d) #(a b c d) :start1 1 :end1 4 :start2 1)))
	(test-t (null (mismatch #(a b c) #(a b c) :from-end t)))
	(test-t (eql (mismatch #(a b c d) #(a b c) :from-end t) 4))
	(test-t (eql (mismatch #(a b c) #(c) :from-end t) 2))
	(test-t (eql (mismatch #(a b c) #(z a b c) :from-end t) 0))
	(test-t (eql (mismatch #(a b c) #(x y z a b c) :from-end t) 0))
	(test-t (eql (mismatch #(x y z a b c) #(a b c) :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c) #(a b c) :end1 3 :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c) #(a b c) :end1 5 :from-end t) 5))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :end1 6 :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :start1 2 :end1 6 :from-end t) 3))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :from-end t :start1 2 :end1 5 :start2 1 :end2 2 ) 4))
	(test-t (eql (mismatch #(x y z a b c x y z) #(a b c) :start1 2 :end1 5 :start2 1 :end2 2 ) 2))
	(test-t (eql (mismatch #((a) (b) (c)) #((a) (b) (c))) 0))
	(test-t (null (mismatch #((a) (b) (c)) #((a) (b) (c)) :key car)))
	(test-t (null (mismatch #((a) (b) (c)) #((a) (b) (c)) :test equal)))
	(test-t (eql (mismatch #(#(a) #(b) #(c)) #(#(a) #(b) #(c))) 0))
	(test-t (null (mismatch #(#(a) #(b) #(c)) #(#(a) #(b) #(c)) :test equalp)))
	(test-t (eql (mismatch #((a) (b) (c) (d)) #((a) (b) (c)) :key car) 3))
	(test-t (eql (mismatch #((a) (b) (c)) #((a) (b) (c) (d)) :key car) 3))
	(test-t (eql (mismatch #(#\a #\b #\c) #(#\A #\B #\C)) 0))
	(test-t (null (mismatch #(#\a #\b #\c) #(#\A #\B #\C) :key char-upcase)))
	(test-t (null (mismatch #(#\a #\b #\c) #(#\A #\B #\C) :key char-downcase)))
	(test-t (null (mismatch #(#\a #\b #\c) #(#\A #\B #\C) :key char-upcase :start1 1 :end1 2 :start2 1 :end2 2)))
	(test-t (null (mismatch #(#\a #\b #\c) #(#\A #\B #\C) :key char-upcase :start1 2 :start2 2)))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f))) 0))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f)) :key cdr) 0))
	(test-t (null (mismatch #((a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f)) :key cdr :test equal)))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal) 3))
	(test-t (eql (mismatch #((a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 4))
	(test-t (eql (mismatch #((a a a) (a b c) (b c d) (d e f)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t) 1))
	(test-t (null (mismatch #((a a a) (a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1 :end1 4)))
	(test-t (eql (mismatch #((a a a) (a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :start1 1) 5))
	(test-t (eql (mismatch #((a a a) (a b c) (b c d) (d e f) (e f g)) #((b b c) (c c d) (e e f)) :key cdr :test equal :from-end t :end1 3 :start2 1 :end2 2) 2))
	(test-t (eql (mismatch "abc" "xyz") 0))
	(test-t (null (mismatch "" "")))
	(test-t (null (mismatch "a" "a")))
	(test-t (null (mismatch "abc" "abc")))
	(test-t (null (mismatch "abc" "ABC" :key char-downcase)))
	(test-t (null (mismatch "abc" "ABC" :test char-equal)))
	(test-t (eql (mismatch "abcde" "abc") 3))
	(test-t (eql (mismatch "abc" "abcde") 3))
	(test-t (eql (mismatch "abc" "abxyz") 2))
	(test-t (eql (mismatch "abcde" "abx") 2))
	(test-t (null (mismatch "abc" "abc" :from-end t)))
	(test-t (eql (mismatch "abcxyz" "xyzxyz" :from-end t) 3))
	(test-t (eql (mismatch "abcxyz" "xyz" :from-end t) 3))
	(test-t (eql (mismatch "xyz" "abcxyz" :from-end t) 0))
	(test-t (eql (mismatch "ayz" "abcxyz" :from-end t) 1))
	(test-t (null (mismatch "abc" "xyz" :test char<)))
	(test-t (eql (mismatch "abc" "xyz" :test char>) 0))
	(test-t (eql (mismatch "abcxyz" "abcdefg") 3))
	(test-t (eql (mismatch "1xyz" "22xyz" :from-end t) 1))
	
	(test-t (let ((lst (copy-seq "012345678"))) (and (equal (replace lst lst :start1 2 :start2 0) "010123456") (equal lst "010123456"))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z)))) (and (eq list0 list) (equal list0 '(x y z d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1))) (and (eq list0 list) (equal list0 '(a x y z e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1 :end1 nil))) (and (eq list0 list) (equal list0 '(a x y z e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1 :start2 1))) (and (eq list0 list) (equal list0 '(a y z d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1 :start2 1 :end2 nil))) (and (eq list0 list) (equal list0 '(a y z d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1 :end1 nil :start2 1 :end2 nil))) (and (eq list0 list) (equal list0 '(a y z d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1 :end1 2 :start2 1))) (and (eq list0 list) (equal list0 '(a y c d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 1 :end1 1))) (and (eq list0 list) (equal list0 '(a b c d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 2 :end1 2))) (and (eq list0 list) (equal list0 '(a b c d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 3 :end1 3))) (and (eq list0 list) (equal list0 '(a b c d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 4 :end1 4))) (and (eq list0 list) (equal list0 '(a b c d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 '(x y z) :start1 5 :end1 5))) (and (eq list0 list) (equal list0 '(a b c d e)))))
	(test-t (null (replace nil nil)))
	(test-t (null (replace nil '(a b c))))
	(test-t (let* ((list0 (list 'a 'b 'c)) (list (replace list0 '()))) (and (eq list0 list) (equal list0 '(a b c)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 list0))) (and (eq list0 list) (equal list0 '(a b c d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 list0 :start1 3))) (and (eq list0 list) (equal list0 '(a b c a b)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 list0 :start1 1))) (and (eq list0 list) (equal list0 '(a a b c d)))))
	(test-t (let* ((list0 (list 'a 'b 'c 'd 'e)) (list (replace list0 list0 :start1 1 :end1 3))) (and (eq list0 list) (equal list0 '(a a b d e)))))
	(test-t (let* ((list0 (list 'a 'b 'c)) (list (replace list0 '(x y z)))) (and (eq list0 list) (equal list0 '(x y z)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z)))) (and (eq vector0 vector) (equalp vector0 #(x y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1))) (and (eq vector0 vector) (equalp vector0 #(a x y z e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1 :end1 nil))) (and (eq vector0 vector) (equalp vector0 #(a x y z e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1 :start2 1))) (and (eq vector0 vector) (equalp vector0 #(a y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1 :start2 1 :end2 nil))) (and (eq vector0 vector) (equalp vector0 #(a y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1 :end1 nil :start2 1 :end2 nil))) (and (eq vector0 vector) (equalp vector0 #(a y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1 :end1 2 :start2 1))) (and (eq vector0 vector) (equalp vector0 #(a y c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 1 :end1 1))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 2 :end1 2))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 3 :end1 3))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 4 :end1 4))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 '(x y z) :start1 5 :end1 5))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (null (replace nil #())))
	(test-t (null (replace nil #(a b c))))
	(test-t (let* ((vector0 (vector 'a 'b 'c)) (vector (replace vector0 '()))) (and (eq vector0 vector) (equalp vector0 #(a b c)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0 :start1 3))) (and (eq vector0 vector) (equalp vector0 #(a b c a b)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0 :start1 1))) (and (eq vector0 vector) (equalp vector0 #(a a b c d)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0 :start1 1 :end1 3))) (and (eq vector0 vector) (equalp vector0 #(a a b d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c)) (vector (replace vector0 '(x y z)))) (and (eq vector0 vector) (equalp vector0 #(x y z)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z)))) (and (eq vector0 vector) (equalp vector0 #(x y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1))) (and (eq vector0 vector) (equalp vector0 #(a x y z e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1 :end1 nil))) (and (eq vector0 vector) (equalp vector0 #(a x y z e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1 :start2 1))) (and (eq vector0 vector) (equalp vector0 #(a y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1 :start2 1 :end2 nil))) (and (eq vector0 vector) (equalp vector0 #(a y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1 :end1 nil :start2 1 :end2 nil))) (and (eq vector0 vector) (equalp vector0 #(a y z d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1 :end1 2 :start2 1))) (and (eq vector0 vector) (equalp vector0 #(a y c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 1 :end1 1))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 2 :end1 2))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 3 :end1 3))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 4 :end1 4))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 #(x y z) :start1 5 :end1 5))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (null (replace nil #())))
	(test-t (null (replace nil #(a b c))))
	(test-t (let* ((vector0 (vector 'a 'b 'c)) (vector (replace vector0 #()))) (and (eq vector0 vector) (equalp vector0 #(a b c)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0))) (and (eq vector0 vector) (equalp vector0 #(a b c d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0 :start1 3))) (and (eq vector0 vector) (equalp vector0 #(a b c a b)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0 :start1 1))) (and (eq vector0 vector) (equalp vector0 #(a a b c d)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c 'd 'e)) (vector (replace vector0 vector0 :start1 1 :end1 3))) (and (eq vector0 vector) (equalp vector0 #(a a b d e)))))
	(test-t (let* ((vector0 (vector 'a 'b 'c)) (vector (replace vector0 #(x y z)))) (and (eq vector0 vector) (equalp vector0 #(x y z)))))
	(test-t (let* ((str0 (copy-seq "abc")) (str (replace str0 "xyz"))) (and (eq str0 str) (equalp str0 "xyz"))))
	(test-t (let* ((str0 (copy-seq "")) (str (replace str0 ""))) (and (eq str0 str) (equalp str0 ""))))
	(test-t (let* ((str0 (copy-seq "")) (str (replace str0 "xyz"))) (and (eq str0 str) (equalp str0 ""))))
	(test-t (let* ((str0 (copy-seq "abc")) (str (replace str0 ""))) (and (eq str0 str) (equalp str0 "abc"))))
	(test-t (let* ((str0 (copy-seq "abcdef")) (str (replace str0 "xyz" :start1 3))) (and (eq str0 str) (equalp str0 "abcxyz"))))
	(test-t (let* ((str0 (copy-seq "abcdef")) (str (replace str0 "xyz" :start1 4 :start2 1))) (and (eq str0 str) (equalp str0 "abcdyz"))))
	(test-t (let* ((str0 (copy-seq "abcdef")) (str (replace str0 "xyz" :start1 1 :end1 2 :start2 1))) (and (eq str0 str) (equalp str0 "aycdef"))))
	(test-t (let* ((str0 (copy-seq "abcdef")) (str (replace str0 "xyz" :start1 1 :start2 1 :end2 2))) (and (eq str0 str) (equalp str0 "aycdef"))))
	(test-t (let* ((str0 (copy-seq "abcdef")) (str (replace str0 str0 :start1 1))) (and (eq str0 str) (equalp str0 "aabcde"))))
	(test-t (equal (substitute #\. #\space "0 2 4 6") "0.2.4.6"))
	(test-t (equal (substitute 9 4 '(1 2 4 1 3 4 5)) '(1 2 9 1 3 9 5)))
	(test-t (equal (substitute 9 4 '(1 2 4 1 3 4 5) :count 1) '(1 2 9 1 3 4 5)))
	(test-t (equal (substitute 9 4 '(1 2 4 1 3 4 5) :count 1 :from-end t) '(1 2 4 1 3 9 5)))
	(test-t (equal (substitute 9 3 '(1 2 4 1 3 4 5) :test >) '(9 9 4 9 3 4 5)))
	(test-t (equal (substitute-if 0 evenp '((1) (2) (3) (4)) :start 2 :key car) '((1) (2) (3) 0)))
	(test-t (equal (substitute-if 9 oddp '(1 2 4 1 3 4 5)) '(9 2 4 9 9 4 9)))
	(test-t (equal (substitute-if 9 evenp '(1 2 4 1 3 4 5) :count 1 :from-end t) '(1 2 4 1 3 9 5)))
	(test-t (let ((some-things (list 'a 'car 'b 'cdr 'c)))
		  (and (equal (nsubstitute-if "function was here" fboundp some-things
					      :count 1 :from-end t)
			      '(a car b "function was here" c))
		       (equal some-things '(a car b "function was here" c)))))
	(test-t (let ((alpha-tester (copy-seq "ab "))) (and (equal (nsubstitute-if-not #\z alpha-char-p alpha-tester) "abz") (equal alpha-tester "abz"))))
	(test-t (equal (substitute 'a 'x '(x y z)) '(a y z)))
	(test-t (equal (substitute 'b 'y '(x y z)) '(x b z)))
	(test-t (equal (substitute 'c 'z '(x y z)) '(x y c)))
	(test-t (equal (substitute 'a 'p '(x y z)) '(x y z)))
	(test-t (equal (substitute 'a 'x '()) '()))
	(test-t (equal (substitute #\x #\b '(#\a #\b #\c #\d #\e) :test char<) '(#\a #\b #\x #\x #\x)))
	(test-t (equal (substitute '(a) 'x '((x) (y) (z)) :key car) '((a) (y) (z))))
	(test-t (equal (substitute 'c 'b '(a b a b a b a b)) '(a c a c a c a c)))
	(test-t (equal (substitute 'a 'b '(b b b)) '(a a a)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f)) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count nil) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 0) '(a x b x c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count -100) '(a x b x c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 1) '(a z b x c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 2) '(a z b z c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 3) '(a z b z c z d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 4) '(a z b z c z d z e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 5) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 6) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 7) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count nil :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 0 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count -100 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 1 :from-end t) '(a x b x c x d x e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 2 :from-end t) '(a x b x c x d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 3 :from-end t) '(a x b x c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 4 :from-end t) '(a x b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 5 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 6 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :count 7 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :start 2 :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :start 2 :end nil :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :start 2 :end 6 :count 100) '(a x b z c z d x e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :start 2 :end 11 :count 100) '(a x b z c z d z e z f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :start 2 :end 8 :count 10) '(a x b z c z d z e x f)))
	(test-t (equal (substitute 'z 'x '(a x b x c x d x e x f) :start 2 :end 8 :count 2 :from-end t) '(a x b x c z d z e x f)))
	(test-t (equal (substitute #\z #\c '(#\a #\b #\c #\d #\e #\f) :test char<) '(#\a #\b #\c #\z #\z #\z)))
	(test-t (equal (substitute "peace" "war" '("war" "War" "WAr" "WAR") :test string-equal) '("peace" "peace" "peace" "peace")))
	(test-t (equal (substitute "peace" "WAR" '("war" "War" "WAr" "WAR") :test string=) '("war" "War" "WAr" "peace")))
	(test-t (equal (substitute "peace" "WAR" '("war" "War" "WAr" "WAR") :test string= :key string-upcase) '("peace" "peace" "peace" "peace")))
	(test-t (equal (substitute "peace" "WAR" '("war" "War" "WAr" "WAR") :start 1 :end 2 :test string= :key string-upcase) '("war" "peace" "WAr" "WAR")))
	(test-t (equal (substitute "peace" "WAR" '("war" "War" "WAr" "WAR") :start 1 :end nil :test string= :key string-upcase) '("war" "peace" "peace" "peace")))
	(test-t (equal (substitute "peace" "war" '("war" "War" "WAr" "WAR") :test string= :key string-upcase) '("war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute 'a 'x #(x y z)) #(a y z)))
	(test-t (equalp (substitute 'b 'y #(x y z)) #(x b z)))
	(test-t (equalp (substitute 'c 'z #(x y z)) #(x y c)))
	(test-t (equalp (substitute 'a 'p #(x y z)) #(x y z)))
	(test-t (equalp (substitute 'a 'x #()) #()))
	(test-t (equalp (substitute #\x #\b #(#\a #\b #\c #\d #\e) :test char<) #(#\a #\b #\x #\x #\x)))
	(test-t (equalp (substitute '(a) 'x #((x) (y) (z)) :key car) #((a) (y) (z))))
	(test-t (equalp (substitute 'c 'b #(a b a b a b a b)) #(a c a c a c a c)))
	(test-t (equalp (substitute 'a 'b #(b b b)) #(a a a)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f)) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count nil) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 0) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count -100) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 1) #(a z b x c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 2) #(a z b z c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 3) #(a z b z c z d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 4) #(a z b z c z d z e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 5) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 6) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 7) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count nil :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 0 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count -100 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 1 :from-end t) #(a x b x c x d x e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 2 :from-end t) #(a x b x c x d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 3 :from-end t) #(a x b x c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 4 :from-end t) #(a x b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 5 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 6 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :count 7 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :start 2 :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :start 2 :end nil :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :start 2 :end 6 :count 100) #(a x b z c z d x e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :start 2 :end 11 :count 100) #(a x b z c z d z e z f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :start 2 :end 8 :count 10) #(a x b z c z d z e x f)))
	(test-t (equalp (substitute 'z 'x #(a x b x c x d x e x f) :start 2 :end 8 :count 2 :from-end t) #(a x b x c z d z e x f)))
	(test-t (equalp (substitute #\z #\c #(#\a #\b #\c #\d #\e #\f) :test char<) #(#\a #\b #\c #\z #\z #\z)))
	(test-t (equalp (substitute "peace" "war" #("love" "hate" "war" "peace") :test equal) #("love" "hate" "peace" "peace")))
	(test-t (equalp (substitute "peace" "war" #("war" "War" "WAr" "WAR") :test string-equal) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (substitute "peace" "WAR" #("war" "War" "WAr" "WAR") :test string=) #("war" "War" "WAr" "peace")))
	(test-t (equalp (substitute "peace" "WAR" #("war" "War" "WAr" "WAR") :test string= :key string-upcase) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (substitute "peace" "WAR" #("war" "War" "WAr" "WAR") :start 1 :end 2 :test string= :key string-upcase) #("war" "peace" "WAr" "WAR")))
	(test-t (equalp (substitute "peace" "WAR" #("war" "War" "WAr" "WAR") :start 1 :end nil :test string= :key string-upcase) #("war" "peace" "peace" "peace")))
	(test-t (equalp (substitute "peace" "war" #("war" "War" "WAr" "WAR") :test string= :key string-upcase) #("war" "War" "WAr" "WAR")))
	(test-t (string= (substitute #\A #\a "abcabc") "AbcAbc"))
	(test-t (string= (substitute #\A #\a "") ""))
	(test-t (string= (substitute #\A #\a "xyz") "xyz"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :start 5 :end nil) "aaaaaAAAAA"))
	(test-t (string= (substitute #\x #\5 "0123456789" :test char<) "012345xxxx"))
	(test-t (string= (substitute #\x #\5 "0123456789" :test char>) "xxxxx56789"))
	(test-t (string= (substitute #\x #\D "abcdefg" :key char-upcase :test char>) "xxxdefg"))
	(test-t (string= (substitute #\x #\D "abcdefg" :start 1 :end 2 :key char-upcase :test char>) "axcdefg"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count 2) "AAaaaaaaaa"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count -1) "aaaaaaaaaa"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count 0) "aaaaaaaaaa"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count nil) "AAAAAAAAAA"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count 100) "AAAAAAAAAA"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count 9) "AAAAAAAAAa"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :count 9 :from-end t) "aAAAAAAAAA"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :start 2 :end 8 :count 3) "aaAAAaaaaa"))
	(test-t (string= (substitute #\A #\a "aaaaaaaaaa" :start 2 :end 8 :from-end t :count 3) "aaaaaAAAaa"))
	(test-t (string= (substitute #\x #\A "aaaaaaaaaa" :start 2 :end 8 :from-end t :count 3) "aaaaaaaaaa"))
	(test-t (string= (substitute #\X #\A "aaaaaaaaaa" :start 2 :end 8 :from-end t :key char-upcase :count 3) "aaaaaXXXaa"))
	(test-t (string= (substitute #\X #\D "abcdefghij" :start 2 :end 8 :from-end t :key char-upcase :test char< :count 3) "abcdeXXXij"))
	(test-t (equal (substitute-if 'a (lambda (arg) (eq arg 'x)) '(x y z)) '(a y z)))
	(test-t (equal (substitute-if 'b (lambda (arg) (eq arg 'y)) '(x y z)) '(x b z)))
	(test-t (equal (substitute-if 'c (lambda (arg) (eq arg 'z)) '(x y z)) '(x y c)))
	(test-t (equal (substitute-if 'a (lambda (arg) (eq arg 'p)) '(x y z)) '(x y z)))
	(test-t (equal (substitute-if 'a (lambda (arg) (eq arg 'x)) '()) '()))
	(test-t (equal (substitute-if #\x (lambda (arg) (char< #\b arg)) '(#\a #\b #\c #\d #\e)) '(#\a #\b #\x #\x #\x)))
	(test-t (equal (substitute-if '(a) (lambda (arg) (eq arg 'x)) '((x) (y) (z)) :key car) '((a) (y) (z))))
	(test-t (equal (substitute-if 'c (lambda (arg) (eq arg 'b)) '(a b a b a b a b)) '(a c a c a c a c)))
	(test-t (equal (substitute-if 'a (lambda (arg) (eq arg 'b)) '(b b b)) '(a a a)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f)) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count nil) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 0) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count -100) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 1) '(a z b x c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 2) '(a z b z c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 3) '(a z b z c z d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 4) '(a z b z c z d z e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 5) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 6) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 7) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count nil :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 0 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count -100 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 1 :from-end t) '(a x b x c x d x e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 2 :from-end t) '(a x b x c x d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 3 :from-end t) '(a x b x c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 4 :from-end t) '(a x b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 5 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 6 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :count 7 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :start 2 :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :start 2 :end nil :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :start 2 :end 6 :count 100) '(a x b z c z d x e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :start 2 :end 11 :count 100) '(a x b z c z d z e z f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :start 2 :end 8 :count 10) '(a x b z c z d z e x f)))
	(test-t (equal (substitute-if 'z (lambda (arg) (eq arg 'x)) '(a x b x c x d x e x f) :start 2 :end 8 :count 2 :from-end t) '(a x b x c z d z e x f)))
	(test-t (equal (substitute-if #\z (lambda (arg) (char< #\c arg)) '(#\a #\b #\c #\d #\e #\f)) '(#\a #\b #\c #\z #\z #\z)))
	(test-t (equal (substitute-if "peace" (lambda (arg) (equal "war" arg)) '("love" "hate" "war" "peace")) '("love" "hate" "peace" "peace")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string-equal "war" arg)) '("war" "War" "WAr" "WAR")) '("peace" "peace" "peace" "peace")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR") :key string-upcase) '("peace" "peace" "peace" "peace")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR") :start 1 :end 2 :key string-upcase) '("war" "peace" "WAr" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR") :start 1 :end nil :key string-upcase) '("war" "peace" "peace" "peace")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "war" arg)) '("war" "War" "WAr" "WAR") :key string-upcase) '("war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 1 :key string-upcase) '("war" "peace" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :key string-upcase) '("war" "peace" "peace" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 0 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count -2 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count nil :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 6 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 7 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 100 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	
	(test-t (equalp (substitute-if 'a (lambda (arg) (eq arg 'x)) #(x y z)) #(a y z)))
	(test-t (equalp (substitute-if 'b (lambda (arg) (eq arg 'y)) #(x y z)) #(x b z)))
	(test-t (equalp (substitute-if 'c (lambda (arg) (eq arg 'z)) #(x y z)) #(x y c)))
	(test-t (equalp (substitute-if 'a (lambda (arg) (eq arg 'p)) #(x y z)) #(x y z)))
	(test-t (equalp (substitute-if 'a (lambda (arg) (eq arg 'x)) #()) #()))
	(test-t (equalp (substitute-if #\x (lambda (arg) (char< #\b arg)) #(#\a #\b #\c #\d #\e)) #(#\a #\b #\x #\x #\x)))
	(test-t (equalp (substitute-if '(a) (lambda (arg) (eq arg 'x)) #((x) (y) (z)) :key car) #((a) (y) (z))))
	(test-t (equalp (substitute-if 'c (lambda (arg) (eq arg 'b)) #(a b a b a b a b)) #(a c a c a c a c)))
	(test-t (equalp (substitute-if 'a (lambda (arg) (eq arg 'b)) #(b b b)) #(a a a)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f)) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count nil) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 0) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count -100) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 1) #(a z b x c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 2) #(a z b z c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 3) #(a z b z c z d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 4) #(a z b z c z d z e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 5) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 6) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 7) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count nil :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 0 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count -100 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 1 :from-end t) #(a x b x c x d x e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 2 :from-end t) #(a x b x c x d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 3 :from-end t) #(a x b x c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 4 :from-end t) #(a x b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 5 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 6 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :count 7 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :start 2 :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :start 2 :end nil :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :start 2 :end 6 :count 100) #(a x b z c z d x e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :start 2 :end 11 :count 100) #(a x b z c z d z e z f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :start 2 :end 8 :count 10) #(a x b z c z d z e x f)))
	(test-t (equalp (substitute-if 'z (lambda (arg) (eq arg 'x)) #(a x b x c x d x e x f) :start 2 :end 8 :count 2 :from-end t) #(a x b x c z d z e x f)))
	(test-t (equalp (substitute-if #\z (lambda (arg) (char< #\c arg)) #(#\a #\b #\c #\d #\e #\f)) #(#\a #\b #\c #\z #\z #\z)))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (equal "war" arg)) #("love" "hate" "war" "peace")) #("love" "hate" "peace" "peace")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string-equal "war" arg)) #("war" "War" "WAr" "WAR")) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR") :key string-upcase) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR") :start 1 :end 2 :key string-upcase) #("war" "peace" "WAr" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR") :start 1 :end nil :key string-upcase) #("war" "peace" "peace" "peace")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "war" arg)) #("war" "War" "WAr" "WAR") :key string-upcase) #("war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 1 :key string-upcase) #("war" "peace" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :key string-upcase) #("war" "peace" "peace" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 0 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count -2 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count nil :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 6 :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 7 :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if "peace" (lambda (arg) (string= "WAR" arg)) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 100 :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "abcabc") "AbcAbc"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "") ""))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "xyz") "xyz"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :start 5 :end nil) "aaaaaAAAAA"))
	(test-t (string= (substitute-if #\x (lambda (arg) (char< #\5 arg)) "0123456789") "012345xxxx"))
	(test-t (string= (substitute-if #\x (lambda (arg) (char> #\5 arg)) "0123456789") "xxxxx56789"))
	(test-t (string= (substitute-if #\x (lambda (arg) (char> #\D arg)) "abcdefg" :key char-upcase) "xxxdefg"))
	(test-t (string= (substitute-if #\x (lambda (arg) (char> #\D arg)) "abcdefg" :start 1 :end 2 :key char-upcase) "axcdefg"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count 2) "AAaaaaaaaa"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count -1) "aaaaaaaaaa"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count 0) "aaaaaaaaaa"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count nil) "AAAAAAAAAA"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count 100) "AAAAAAAAAA"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count 9) "AAAAAAAAAa"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :count 9 :from-end t) "aAAAAAAAAA"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :start 2 :end 8 :count 3) "aaAAAaaaaa"))
	(test-t (string= (substitute-if #\A (lambda (arg) (eql #\a arg)) "aaaaaaaaaa" :start 2 :end 8 :from-end t :count 3) "aaaaaAAAaa"))
	(test-t (string= (substitute-if #\x (lambda (arg) (eql #\A arg)) "aaaaaaaaaa" :start 2 :end 8 :from-end t :count 3) "aaaaaaaaaa"))
	(test-t (string= (substitute-if #\X (lambda (arg) (eql #\A arg)) "aaaaaaaaaa" :start 2 :end 8 :from-end t :key char-upcase :count 3) "aaaaaXXXaa"))
	(test-t (string= (substitute-if #\X (lambda (arg) (char< #\D arg)) "abcdefghij" :start 2 :end 8 :from-end t :key char-upcase :count 3) "abcdeXXXij"))
	(test-t (equal (substitute-if-not 'a (lambda (arg) (not (eq arg 'x))) '(x y z)) '(a y z)))
	(test-t (equal (substitute-if-not 'b (lambda (arg) (not (eq arg 'y))) '(x y z)) '(x b z)))
	(test-t (equal (substitute-if-not 'c (lambda (arg) (not (eq arg 'z))) '(x y z)) '(x y c)))
	(test-t (equal (substitute-if-not 'a (lambda (arg) (not (eq arg 'p))) '(x y z)) '(x y z)))
	(test-t (equal (substitute-if-not 'a (lambda (arg) (not (eq arg 'x))) '()) '()))
	(test-t (equal (substitute-if-not #\x (lambda (arg) (not (char< #\b arg))) '(#\a #\b #\c #\d #\e)) '(#\a #\b #\x #\x #\x)))
	(test-t (equal (substitute-if-not '(a) (lambda (arg) (not (eq arg 'x))) '((x) (y) (z)) :key car) '((a) (y) (z))))
	(test-t (equal (substitute-if-not 'c (lambda (arg) (not (eq arg 'b))) '(a b a b a b a b)) '(a c a c a c a c)))
	(test-t (equal (substitute-if-not 'a (lambda (arg) (not (eq arg 'b))) '(b b b)) '(a a a)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f)) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count nil) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 0) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count -100) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 1) '(a z b x c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 2) '(a z b z c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 3) '(a z b z c z d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 4) '(a z b z c z d z e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 5) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 6) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 7) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count nil :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 0 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count -100 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 1 :from-end t) '(a x b x c x d x e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 2 :from-end t) '(a x b x c x d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 3 :from-end t) '(a x b x c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 4 :from-end t) '(a x b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 5 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 6 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :count 7 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :start 2 :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :start 2 :end nil :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :start 2 :end 6 :count 100) '(a x b z c z d x e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :start 2 :end 11 :count 100) '(a x b z c z d z e z f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :start 2 :end 8 :count 10) '(a x b z c z d z e x f)))
	(test-t (equal (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) '(a x b x c x d x e x f) :start 2 :end 8 :count 2 :from-end t) '(a x b x c z d z e x f)))
	(test-t (equal (substitute-if-not #\z (lambda (arg) (not (char< #\c arg))) '(#\a #\b #\c #\d #\e #\f)) '(#\a #\b #\c #\z #\z #\z)))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (equal "war" arg))) '("love" "hate" "war" "peace")) '("love" "hate" "peace" "peace")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string-equal "war" arg))) '("war" "War" "WAr" "WAR")) '("peace" "peace" "peace" "peace")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR") :key string-upcase) '("peace" "peace" "peace" "peace")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR") :start 1 :end 2 :key string-upcase) '("war" "peace" "WAr" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR") :start 1 :end nil :key string-upcase) '("war" "peace" "peace" "peace")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "war" arg))) '("war" "War" "WAr" "WAR") :key string-upcase) '("war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 1 :key string-upcase) '("war" "peace" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :key string-upcase) '("war" "peace" "peace" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 0 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count -2 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count nil :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 6 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 7 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 100 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if-not 'a (lambda (arg) (not (eq arg 'x))) #(x y z)) #(a y z)))
	(test-t (equalp (substitute-if-not 'b (lambda (arg) (not (eq arg 'y))) #(x y z)) #(x b z)))
	(test-t (equalp (substitute-if-not 'c (lambda (arg) (not (eq arg 'z))) #(x y z)) #(x y c)))
	(test-t (equalp (substitute-if-not 'a (lambda (arg) (not (eq arg 'p))) #(x y z)) #(x y z)))
	(test-t (equalp (substitute-if-not 'a (lambda (arg) (not (eq arg 'x))) #()) #()))
	(test-t (equalp (substitute-if-not #\x (lambda (arg) (not (char< #\b arg))) #(#\a #\b #\c #\d #\e)) #(#\a #\b #\x #\x #\x)))
	(test-t (equalp (substitute-if-not '(a) (lambda (arg) (not (eq arg 'x))) #((x) (y) (z)) :key car) #((a) (y) (z))))
	(test-t (equalp (substitute-if-not 'c (lambda (arg) (not (eq arg 'b))) #(a b a b a b a b)) #(a c a c a c a c)))
	(test-t (equalp (substitute-if-not 'a (lambda (arg) (not (eq arg 'b))) #(b b b)) #(a a a)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f)) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count nil) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 0) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count -100) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 1) #(a z b x c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 2) #(a z b z c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 3) #(a z b z c z d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 4) #(a z b z c z d z e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 5) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 6) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 7) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count nil :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 0 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count -100 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 1 :from-end t) #(a x b x c x d x e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 2 :from-end t) #(a x b x c x d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 3 :from-end t) #(a x b x c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 4 :from-end t) #(a x b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 5 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 6 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :count 7 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :start 2 :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :start 2 :end nil :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :start 2 :end 6 :count 100) #(a x b z c z d x e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :start 2 :end 11 :count 100) #(a x b z c z d z e z f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :start 2 :end 8 :count 10) #(a x b z c z d z e x f)))
	(test-t (equalp (substitute-if-not 'z (lambda (arg) (not (eq arg 'x))) #(a x b x c x d x e x f) :start 2 :end 8 :count 2 :from-end t) #(a x b x c z d z e x f)))
	(test-t (equalp (substitute-if-not #\z (lambda (arg) (not (char< #\c arg))) #(#\a #\b #\c #\d #\e #\f)) #(#\a #\b #\c #\z #\z #\z)))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (equal "war" arg))) #("love" "hate" "war" "peace")) #("love" "hate" "peace" "peace")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string-equal "war" arg))) #("war" "War" "WAr" "WAR")) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR") :key string-upcase) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR") :start 1 :end 2 :key string-upcase) #("war" "peace" "WAr" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR") :start 1 :end nil :key string-upcase) #("war" "peace" "peace" "peace")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "war" arg))) #("war" "War" "WAr" "WAR") :key string-upcase) #("war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 1 :key string-upcase) #("war" "peace" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :key string-upcase) #("war" "peace" "peace" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 2 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 0 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count -2 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count nil :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 6 :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 7 :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (substitute-if-not "peace" (lambda (arg) (not (string= "WAR" arg))) #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR") :start 1 :end 7 :count 100 :from-end t :key string-upcase) #("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "abcabc") "AbcAbc"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "") ""))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "xyz") "xyz"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :start 5 :end nil) "aaaaaAAAAA"))
	(test-t (string= (substitute-if-not #\x (lambda (arg) (not (char< #\5 arg))) "0123456789") "012345xxxx"))
	(test-t (string= (substitute-if-not #\x (lambda (arg) (not (char> #\5 arg))) "0123456789") "xxxxx56789"))
	(test-t (string= (substitute-if-not #\x (lambda (arg) (not (char> #\D arg))) "abcdefg" :key char-upcase) "xxxdefg"))
	(test-t (string= (substitute-if-not #\x (lambda (arg) (not (char> #\D arg))) "abcdefg" :start 1 :end 2 :key char-upcase) "axcdefg"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count 2) "AAaaaaaaaa"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count -1) "aaaaaaaaaa"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count 0) "aaaaaaaaaa"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count nil) "AAAAAAAAAA"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count 100) "AAAAAAAAAA"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count 9) "AAAAAAAAAa"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :count 9 :from-end t) "aAAAAAAAAA"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :start 2 :end 8 :count 3) "aaAAAaaaaa"))
	(test-t (string= (substitute-if-not #\A (lambda (arg) (not (eql #\a arg))) "aaaaaaaaaa" :start 2 :end 8 :from-end t :count 3) "aaaaaAAAaa"))
	(test-t (string= (substitute-if-not #\x (lambda (arg) (not (eql #\A arg))) "aaaaaaaaaa" :start 2 :end 8 :from-end t :count 3) "aaaaaaaaaa"))
	(test-t (string= (substitute-if-not #\X (lambda (arg) (not (eql #\A arg))) "aaaaaaaaaa" :start 2 :end 8 :from-end t :key char-upcase :count 3) "aaaaaXXXaa"))
	(test-t (string= (substitute-if-not #\X (lambda (arg) (not (char< #\D arg))) "abcdefghij" :start 2 :end 8 :from-end t :key char-upcase :count 3) "abcdeXXXij"))
	(test-t (equal (nsubstitute 'a 'x (copy-seq '(x y z))) '(a y z)))
	(test-t (equal (nsubstitute 'b 'y (copy-seq '(x y z))) '(x b z)))
	(test-t (equal (nsubstitute 'c 'z (copy-seq '(x y z))) '(x y c)))
	(test-t (equal (nsubstitute 'a 'p (copy-seq '(x y z))) '(x y z)))
	(test-t (equal (nsubstitute 'a 'x (copy-seq '())) '()))
	(test-t (equal (nsubstitute #\x #\b (copy-seq '(#\a #\b #\c #\d #\e)) :test char<) '(#\a #\b #\x #\x #\x)))
	(test-t (equal (nsubstitute '(a) 'x (copy-seq '((x) (y) (z))) :key car) '((a) (y) (z))))
	(test-t (equal (nsubstitute 'c 'b (copy-seq '(a b a b a b a b))) '(a c a c a c a c)))
	(test-t (equal (nsubstitute 'a 'b (copy-seq '(b b b))) '(a a a)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f))) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count nil) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 0) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count -100) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 1) '(a z b x c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 2) '(a z b z c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 3) '(a z b z c z d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 4) '(a z b z c z d z e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 5) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 6) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 7) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count nil :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 0 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count -100 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 1 :from-end t) '(a x b x c x d x e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 2 :from-end t) '(a x b x c x d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 3 :from-end t) '(a x b x c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 4 :from-end t) '(a x b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 5 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 6 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :count 7 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :start 2 :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :start 2 :end nil :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :start 2 :end 6 :count 100) '(a x b z c z d x e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :start 2 :end 11 :count 100) '(a x b z c z d z e z f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :start 2 :end 8 :count 10) '(a x b z c z d z e x f)))
	(test-t (equal (nsubstitute 'z 'x (copy-seq '(a x b x c x d x e x f)) :start 2 :end 8 :count 2 :from-end t) '(a x b x c z d z e x f)))
	(test-t (equal (nsubstitute #\z #\c (copy-seq '(#\a #\b #\c #\d #\e #\f)) :test char<) '(#\a #\b #\c #\z #\z #\z)))
	(test-t (equal (nsubstitute "peace" "war" (copy-seq '("love" "hate" "war" "peace")) :test equal) '("love" "hate" "peace" "peace")))
	(test-t (equal (nsubstitute "peace" "war" (copy-seq '("war" "War" "WAr" "WAR")) :test string-equal) '("peace" "peace" "peace" "peace")))
	(test-t (equal (nsubstitute "peace" "WAR" (copy-seq '("war" "War" "WAr" "WAR")) :test string=) '("war" "War" "WAr" "peace")))
	(test-t (equal (nsubstitute "peace" "WAR" (copy-seq '("war" "War" "WAr" "WAR")) :test string= :key string-upcase) '("peace" "peace" "peace" "peace")))
	(test-t (equal (nsubstitute "peace" "WAR" (copy-seq '("war" "War" "WAr" "WAR")) :start 1 :end 2 :test string= :key string-upcase) '("war" "peace" "WAr" "WAR")))
	(test-t (equalp (nsubstitute 'a 'x (copy-seq #(x y z))) #(a y z)))
	(test-t (equalp (nsubstitute 'b 'y (copy-seq #(x y z))) #(x b z)))
	(test-t (equalp (nsubstitute 'c 'z (copy-seq #(x y z))) #(x y c)))
	(test-t (equalp (nsubstitute 'a 'p (copy-seq #(x y z))) #(x y z)))
	(test-t (equalp (nsubstitute 'a 'x (copy-seq #())) #()))
	(test-t (equalp (nsubstitute #\x #\b (copy-seq #(#\a #\b #\c #\d #\e)) :test char<) #(#\a #\b #\x #\x #\x)))
	(test-t (equalp (nsubstitute '(a) 'x (copy-seq #((x) (y) (z))) :key car) #((a) (y) (z))))
	(test-t (equalp (nsubstitute 'c 'b (copy-seq #(a b a b a b a b))) #(a c a c a c a c)))
	(test-t (equalp (nsubstitute 'a 'b (copy-seq #(b b b))) #(a a a)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f))) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count nil) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 0) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count -100) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 1) #(a z b x c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 2) #(a z b z c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 3) #(a z b z c z d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 4) #(a z b z c z d z e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 5) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 6) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 7) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count nil :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 0 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count -100 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 1 :from-end t) #(a x b x c x d x e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 2 :from-end t) #(a x b x c x d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 3 :from-end t) #(a x b x c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 4 :from-end t) #(a x b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 5 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 6 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :count 7 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :start 2 :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :start 2 :end nil :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :start 2 :end 6 :count 100) #(a x b z c z d x e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :start 2 :end 11 :count 100) #(a x b z c z d z e z f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :start 2 :end 8 :count 10) #(a x b z c z d z e x f)))
	(test-t (equalp (nsubstitute 'z 'x (copy-seq #(a x b x c x d x e x f)) :start 2 :end 8 :count 2 :from-end t) #(a x b x c z d z e x f)))
	(test-t (equalp (nsubstitute #\z #\c (copy-seq #(#\a #\b #\c #\d #\e #\f)) :test char<) #(#\a #\b #\c #\z #\z #\z)))
	(test-t (equalp (nsubstitute "peace" "war" (copy-seq #("love" "hate" "war" "peace")) :test equal) #("love" "hate" "peace" "peace")))
	(test-t (equalp (nsubstitute "peace" "war" (copy-seq #("war" "War" "WAr" "WAR")) :test string-equal) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (nsubstitute "peace" "WAR" (copy-seq #("war" "War" "WAr" "WAR")) :test string=) #("war" "War" "WAr" "peace")))
	(test-t (equalp (nsubstitute "peace" "WAR" (copy-seq #("war" "War" "WAr" "WAR")) :test string= :key string-upcase) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (nsubstitute "peace" "WAR" (copy-seq #("war" "War" "WAr" "WAR")) :start 1 :end 2 :test string= :key string-upcase) #("war" "peace" "WAr" "WAR")))
	(test-t (equalp (nsubstitute "peace" "WAR" (copy-seq #("war" "War" "WAr" "WAR")) :start 1 :end nil :test string= :key string-upcase) #("war" "peace" "peace" "peace")))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "abcabc")) "AbcAbc"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "")) ""))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "xyz")) "xyz"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :start 5 :end nil) "aaaaaAAAAA"))
	(test-t (string= (nsubstitute #\x #\5 (copy-seq "0123456789") :test char<) "012345xxxx"))
	(test-t (string= (nsubstitute #\x #\5 (copy-seq "0123456789") :test char>) "xxxxx56789"))
	(test-t (string= (nsubstitute #\x #\D (copy-seq "abcdefg") :key char-upcase :test char>) "xxxdefg"))
	(test-t (string= (nsubstitute #\x #\D (copy-seq "abcdefg") :start 1 :end 2 :key char-upcase :test char>) "axcdefg"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count 2) "AAaaaaaaaa"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count -1) "aaaaaaaaaa"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count 0) "aaaaaaaaaa"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count nil) "AAAAAAAAAA"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count 100) "AAAAAAAAAA"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count 9) "AAAAAAAAAa"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :count 9 :from-end t) "aAAAAAAAAA"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :start 2 :end 8 :count 3) "aaAAAaaaaa"))
	(test-t (string= (nsubstitute #\A #\a (copy-seq "aaaaaaaaaa") :start 2 :end 8 :from-end t :count 3) "aaaaaAAAaa"))
	(test-t (string= (nsubstitute #\x #\A (copy-seq "aaaaaaaaaa") :start 2 :end 8 :from-end t :count 3) "aaaaaaaaaa"))
	(test-t (string= (nsubstitute #\X #\A (copy-seq "aaaaaaaaaa") :start 2 :end 8 :from-end t :key char-upcase :count 3) "aaaaaXXXaa"))
	(test-t (string= (nsubstitute #\X #\D (copy-seq "abcdefghij") :start 2 :end 8 :from-end t :key char-upcase :test char< :count 3) "abcdeXXXij"))
	(test-t (equal (nsubstitute-if 'a (lambda (arg) (eq arg 'x)) (copy-seq '(x y z))) '(a y z)))
	(test-t (equal (nsubstitute-if 'b (lambda (arg) (eq arg 'y)) (copy-seq '(x y z))) '(x b z)))
	(test-t (equal (nsubstitute-if 'c (lambda (arg) (eq arg 'z)) (copy-seq '(x y z))) '(x y c)))
	(test-t (equal (nsubstitute-if 'a (lambda (arg) (eq arg 'p)) (copy-seq '(x y z))) '(x y z)))
	(test-t (equal (nsubstitute-if 'a (lambda (arg) (eq arg 'x)) (copy-seq '())) '()))
	(test-t (equal (nsubstitute-if #\x (lambda (arg) (char< #\b arg)) (copy-seq '(#\a #\b #\c #\d #\e))) '(#\a #\b #\x #\x #\x)))
	(test-t (equal (nsubstitute-if '(a) (lambda (arg) (eq arg 'x)) (copy-seq '((x) (y) (z))) :key car) '((a) (y) (z))))
	(test-t (equal (nsubstitute-if 'c (lambda (arg) (eq arg 'b)) (copy-seq '(a b a b a b a b))) '(a c a c a c a c)))
	(test-t (equal (nsubstitute-if 'a (lambda (arg) (eq arg 'b)) (copy-seq '(b b b))) '(a a a)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f))) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count nil) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 0) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count -100) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 1) '(a z b x c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 2) '(a z b z c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 3) '(a z b z c z d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 4) '(a z b z c z d z e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 5) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 6) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 7) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count nil :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 0 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count -100 :from-end t) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 1 :from-end t) '(a x b x c x d x e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 2 :from-end t) '(a x b x c x d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 3 :from-end t) '(a x b x c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 4 :from-end t) '(a x b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 5 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 6 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :count 7 :from-end t) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :start 2 :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :start 2 :end nil :count 1) '(a x b z c x d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :start 2 :end 6 :count 100) '(a x b z c z d x e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :start 2 :end 11 :count 100) '(a x b z c z d z e z f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :start 2 :end 8 :count 10) '(a x b z c z d z e x f)))
	(test-t (equal (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq '(a x b x c x d x e x f)) :start 2 :end 8 :count 2 :from-end t) '(a x b x c z d z e x f)))
	(test-t (equal (nsubstitute-if #\z (lambda (arg) (char< #\c arg)) (copy-seq '(#\a #\b #\c #\d #\e #\f))) '(#\a #\b #\c #\z #\z #\z)))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (equal "war" arg)) (copy-seq '("love" "hate" "war" "peace"))) '("love" "hate" "peace" "peace")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string-equal "war" arg)) (copy-seq '("war" "War" "WAr" "WAR"))) '("peace" "peace" "peace" "peace")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR")) :key string-upcase) '("peace" "peace" "peace" "peace")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR")) :start 1 :end 2 :key string-upcase) '("war" "peace" "WAr" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR")) :start 1 :end nil :key string-upcase) '("war" "peace" "peace" "peace")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "war" arg)) (copy-seq '("war" "War" "WAr" "WAR")) :key string-upcase) '("war" "War" "WAr" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 1 :key string-upcase) '("war" "peace" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 2 :key string-upcase) '("war" "peace" "peace" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 2 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "peace" "peace" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 0 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count -2 :from-end t :key string-upcase) '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count nil :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 6 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 7 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equal (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq '("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 100 :from-end t :key string-upcase) '("war" "peace" "peace" "peace" "peace" "peace" "peace" "WAR")))
	(test-t (equalp (nsubstitute-if 'a (lambda (arg) (eq arg 'x)) (copy-seq #(x y z))) #(a y z)))
	(test-t (equalp (nsubstitute-if 'b (lambda (arg) (eq arg 'y)) (copy-seq #(x y z))) #(x b z)))
	(test-t (equalp (nsubstitute-if 'c (lambda (arg) (eq arg 'z)) (copy-seq #(x y z))) #(x y c)))
	(test-t (equalp (nsubstitute-if 'a (lambda (arg) (eq arg 'p)) (copy-seq #(x y z))) #(x y z)))
	(test-t (equalp (nsubstitute-if 'a (lambda (arg) (eq arg 'x)) (copy-seq #())) #()))
	(test-t (equalp (nsubstitute-if #\x (lambda (arg) (char< #\b arg)) (copy-seq #(#\a #\b #\c #\d #\e))) #(#\a #\b #\x #\x #\x)))
	(test-t (equalp (nsubstitute-if '(a) (lambda (arg) (eq arg 'x)) (copy-seq #((x) (y) (z))) :key car) #((a) (y) (z))))
	(test-t (equalp (nsubstitute-if 'c (lambda (arg) (eq arg 'b)) (copy-seq #(a b a b a b a b))) #(a c a c a c a c)))
	(test-t (equalp (nsubstitute-if 'a (lambda (arg) (eq arg 'b)) (copy-seq #(b b b))) #(a a a)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f))) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count nil) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 0) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count -100) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 1) #(a z b x c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 2) #(a z b z c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 3) #(a z b z c z d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 4) #(a z b z c z d z e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 5) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 6) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 7) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count nil :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 0 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count -100 :from-end t) #(a x b x c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 1 :from-end t) #(a x b x c x d x e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 2 :from-end t) #(a x b x c x d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 3 :from-end t) #(a x b x c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 4 :from-end t) #(a x b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 5 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 6 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :count 7 :from-end t) #(a z b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :start 2 :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :start 2 :end nil :count 1) #(a x b z c x d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :start 2 :end 6 :count 100) #(a x b z c z d x e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :start 2 :end 11 :count 100) #(a x b z c z d z e z f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :start 2 :end 8 :count 10) #(a x b z c z d z e x f)))
	(test-t (equalp (nsubstitute-if 'z (lambda (arg) (eq arg 'x)) (copy-seq #(a x b x c x d x e x f)) :start 2 :end 8 :count 2 :from-end t) #(a x b x c z d z e x f)))
	(test-t (equalp (nsubstitute-if #\z (lambda (arg) (char< #\c arg)) (copy-seq #(#\a #\b #\c #\d #\e #\f))) #(#\a #\b #\c #\z #\z #\z)))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (equal "war" arg)) (copy-seq #("love" "hate" "war" "peace"))) #("love" "hate" "peace" "peace")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string-equal "war" arg)) (copy-seq #("war" "War" "WAr" "WAR"))) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq #("war" "War" "WAr" "WAR")) :key string-upcase) #("peace" "peace" "peace" "peace")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq #("war" "War" "WAr" "WAR")) :start 1 :end 2 :key string-upcase) #("war" "peace" "WAr" "WAR")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq #("war" "War" "WAr" "WAR")) :start 1 :end nil :key string-upcase) #("war" "peace" "peace" "peace")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "war" arg)) (copy-seq #("war" "War" "WAr" "WAR")) :key string-upcase) #("war" "War" "WAr" "WAR")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 1 :key string-upcase) #("war" "peace" "WAr" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 2 :key string-upcase) #("war" "peace" "peace" "WAR" "war" "War" "WAr" "WAR")))
	(test-t (equalp (nsubstitute-if "peace" (lambda (arg) (string= "WAR" arg)) (copy-seq #("war" "War" "WAr" "WAR" "war" "War" "WAr" "WAR")) :start 1 :end 7 :count 2 :from-end t :key string-upcase) #("war" "War" "WAr" "WAR" "war" "peace" "peace" "WAR")))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "abcabc")) "AbcAbc"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "")) ""))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "xyz")) "xyz"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :start 5 :end nil) "aaaaaAAAAA"))
	(test-t (string= (nsubstitute-if #\x (lambda (arg) (char< #\5 arg)) (copy-seq "0123456789")) "012345xxxx"))
	(test-t (string= (nsubstitute-if #\x (lambda (arg) (char> #\5 arg)) (copy-seq "0123456789")) "xxxxx56789"))
	(test-t (string= (nsubstitute-if #\x (lambda (arg) (char> #\D arg)) (copy-seq "abcdefg") :key char-upcase) "xxxdefg"))
	(test-t (string= (nsubstitute-if #\x (lambda (arg) (char> #\D arg)) (copy-seq "abcdefg") :start 1 :end 2 :key char-upcase) "axcdefg"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count 2) "AAaaaaaaaa"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count -1) "aaaaaaaaaa"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count 0) "aaaaaaaaaa"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count nil) "AAAAAAAAAA"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count 100) "AAAAAAAAAA"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count 9) "AAAAAAAAAa"))
	(test-t (string= (nsubstitute-if #\A (lambda (arg) (eql #\a arg)) (copy-seq "aaaaaaaaaa") :count 9 :from-end t) "aAAAAAAAAA"))
	(test-t (equal (nsubstitute-if-not 'a (lambda (arg) (not (eq arg 'x))) (copy-seq '(x y z))) '(a y z)))
	(test-t (equal (nsubstitute-if-not 'b (lambda (arg) (not (eq arg 'y))) (copy-seq '(x y z))) '(x b z)))
	(test-t (equal (nsubstitute-if-not 'c (lambda (arg) (not (eq arg 'z))) (copy-seq '(x y z))) '(x y c)))
	(test-t (equal (nsubstitute-if-not 'a (lambda (arg) (not (eq arg 'p))) (copy-seq '(x y z))) '(x y z)))
	(test-t (equal (nsubstitute-if-not 'a (lambda (arg) (not (eq arg 'x))) (copy-seq '())) '()))
	(test-t (equal (nsubstitute-if-not #\x (lambda (arg) (not (char< #\b arg))) (copy-seq '(#\a #\b #\c #\d #\e))) '(#\a #\b #\x #\x #\x)))
	(test-t (equal (nsubstitute-if-not '(a) (lambda (arg) (not (eq arg 'x))) (copy-seq '((x) (y) (z))) :key car) '((a) (y) (z))))
	(test-t (equal (nsubstitute-if-not 'c (lambda (arg) (not (eq arg 'b))) (copy-seq '(a b a b a b a b))) '(a c a c a c a c)))
	(test-t (equal (nsubstitute-if-not 'a (lambda (arg) (not (eq arg 'b))) (copy-seq '(b b b))) '(a a a)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f))) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count nil) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 0) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count -100) '(a x b x c x d x e x f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 1) '(a z b x c x d x e x f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 2) '(a z b z c x d x e x f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 3) '(a z b z c z d x e x f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 4) '(a z b z c z d z e x f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 5) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 6) '(a z b z c z d z e z f)))
	(test-t (equal (nsubstitute-if-not 'z (lambda (arg) (not (eq arg 'x))) (copy-seq '(a x b x c x d x e x f)) :count 7) '(a z b z c z d z e z f)))
	
	(test-t (string= (concatenate 'string "all" " " "together" " " "now") "all together now"))
	(test-t (equal (concatenate 'list '() '(a b c) '(x y z)) '(a b c x y z)))
	(test-t (equal (concatenate 'list '(a) #(b) '(c) #(x y) '(z)) '(a b c x y z)))
	(test-t (null (concatenate 'list)))
	(test-t (let* ((list0 '(a b c)) (list (concatenate 'list list0))) (and (not (eq list0 list)) (equal list list0) (equal list '(a b c)))))
	(test-t (equalp (concatenate 'vector '() '(a b c) '(x y z)) #(a b c x y z)))
	(test-t (equalp (concatenate 'vector '(a) #(b) '(c) #(x y) '(z)) #(a b c x y z)))
	(test-t (equalp (concatenate 'vector) #()))
	(test-t (let* ((vector0 #(a b c)) (vector (concatenate 'vector vector0))) (and (not (eq vector0 vector)) (equalp vector vector0) (equalp vector #(a b c)))))
	(test-t (string= (concatenate 'string "abc" "def" "ghi" "jkl" "mno" "pqr") "abcdefghijklmnopqr"))
	(test-t (string= (concatenate 'string "" "abc" "" "def" "" "ghi" "" "" "jkl" "" "mno" "" "pqr" "" "") "abcdefghijklmnopqr"))
	(test-t (string= (concatenate 'string) ""))
	(test-t (string= (concatenate 'string "abc" '(#\d #\e #\f #\g) #(#\h #\i #\j #\k #\l)) "abcdefghijkl"))
	(test-t (let ((test1 (list 1 3 4 6 7)) (test2 (list 2 5 8))) (equal (merge 'list test1 test2 <) '(1 2 3 4 5 6 7 8))))
	(test-t (let ((test1 (vector '(red . 1) '(blue . 4))) (test2 (vector '(yellow . 2) '(green . 7)))) (equalp (merge 'vector test1 test2 < :key cdr) #((red . 1) (yellow . 2) (blue . 4) (green . 7)))))
	(test-t (equal (merge 'list (list 1 3 5 7 9) (list 0 2 4 6 8) <) '(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equal (merge 'list (list 0 1 2) nil <) '(0 1 2)))
	(test-t (equal (merge 'list nil (list 0 1 2) <) '(0 1 2)))
	(test-t (equal (merge 'list nil nil <) nil))
	(test-t (equal (merge 'list (list '(1 1) '(2 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equal (merge 'list (list 3 1 9 5 7) (list 8 6 0 2 4) <) '(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equal (merge 'list (vector 1 3 5 7 9) (list 0 2 4 6 8) <) '(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equal (merge 'list (vector 0 1 2) nil <) '(0 1 2)))
	(test-t (equal (merge 'list #() (list 0 1 2) <) '(0 1 2)))
	(test-t (equal (merge 'list #() #() <) nil))
	(test-t (equal (merge 'list (vector '(1 1) '(2 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equal (merge 'list (vector 3 1 9 5 7) (list 8 6 0 2 4) <) '(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equal (merge 'list (list 1 3 5 7 9) (vector 0 2 4 6 8) <) '(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equal (merge 'list (list 0 1 2) #() <) '(0 1 2)))
	(test-t (equal (merge 'list nil (vector 0 1 2) <) '(0 1 2)))
	(test-t (equal (merge 'list nil #() <) nil))
	(test-t (equal (merge 'list (list '(1 1) '(2 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equal (merge 'list (list 3 1 9 5 7) (vector 8 6 0 2 4) <) '(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equal (merge 'list (vector 1 3 5 7 9) (vector 0 2 4 6 8) <) '(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equal (merge 'list (vector 0 1 2) #() <) '(0 1 2)))
	(test-t (equal (merge 'list #() (vector 0 1 2) <) '(0 1 2)))
	(test-t (equal (merge 'list #() #() <) nil))
	(test-t (equal (merge 'list (vector '(1 1) '(2 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equal (merge 'list (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) '((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equal (merge 'list (vector 3 1 9 5 7) (vector 8 6 0 2 4) <) '(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equalp (merge 'vector (list 1 3 5 7 9) (list 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (list 1 3 5 7 9) (list 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (list 0 1 2) nil <) #(0 1 2)))
	(test-t (equalp (merge 'vector nil (list 0 1 2) <) #(0 1 2)))
	(test-t (equalp (merge 'vector nil nil <) #()))
	(test-t (equalp (merge 'vector (list '(1 1) '(2 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equalp (merge 'vector (list 3 1 9 5 7) (list 8 6 0 2 4) <) #(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equalp (merge 'vector (vector 1 3 5 7 9) (list 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (vector 1 3 5 7 9) (list 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (vector 0 1 2) nil <) #(0 1 2)))
	(test-t (equalp (merge 'vector #() (list 0 1 2) <) #(0 1 2)))
	(test-t (equalp (merge 'vector #() #() <) #()))
	(test-t (equalp (merge 'vector (vector '(1 1) '(2 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (list '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equalp (merge 'vector (vector 3 1 9 5 7) (list 8 6 0 2 4) <) #(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equalp (merge 'vector (list 1 3 5 7 9) (vector 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (list 1 3 5 7 9) (vector 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (list 0 1 2) #() <) #(0 1 2)))
	(test-t (equalp (merge 'vector nil (vector 0 1 2) <) #(0 1 2)))
	(test-t (equalp (merge 'vector nil #() <) #()))
	(test-t (equalp (merge 'vector (list '(1 1) '(2 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (list '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equalp (merge 'vector (list 3 1 9 5 7) (vector 8 6 0 2 4) <) #(3 1 8 6 0 2 4 9 5 7)))
	(test-t (equalp (merge 'vector (vector 1 3 5 7 9) (vector 0 2 4 6 8) <) #(0 1 2 3 4 5 6 7 8 9)))
	(test-t (equalp (merge 'vector (vector 0 1 2) #() <) #(0 1 2)))
	(test-t (equalp (merge 'vector #() (vector 0 1 2) <) #(0 1 2)))
	(test-t (equalp (merge 'vector #() #() <) #()))
	(test-t (equalp (merge 'vector (vector '(1 1) '(2 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2))))
	(test-t (equalp (merge 'vector (vector '(1 1) '(2 1) '(2 1 1) '(3 1)) (vector '(1 2) '(2 2) '(3 2) '(3 2 2)) <= :key car) #((1 1) (1 2) (2 1) (2 1 1) (2 2) (3 1) (3 2) (3 2 2))))
	(test-t (equalp (merge 'vector (vector 3 1 9 5 7) (vector 8 6 0 2 4) <) #(3 1 8 6 0 2 4 9 5 7)))
	(test-t (string= (merge 'string (list #\a #\c #\e) (list #\b #\d #\f) char<) "abcdef"))
	(test-t (string= (merge 'string (list #\a #\b #\c) (list #\d #\e #\f) char<) "abcdef"))
	(test-t (string= (merge 'string (list #\a #\b #\c) '() char<) "abc"))
	(test-t (string= (merge 'string '() (list #\a #\b #\c) char<) "abc"))
	(test-t (string= (merge 'string (list #\a #\b #\c) (copy-seq "") char<) "abc"))
	(test-t (string= (merge 'string (list #\a #\b #\z) #(#\c #\x #\y) char<) "abcxyz"))
	(test-t (equal (remove 4 '(1 3 4 5 9)) '(1 3 5 9)))
	(test-t (equal (remove 4 '(1 2 4 1 3 4 5)) '(1 2 1 3 5)))
	(test-t (equal (remove 4 '(1 2 4 1 3 4 5) :count 1) '(1 2 1 3 4 5)))
	(test-t (equal (remove 4 '(1 2 4 1 3 4 5) :count 1 :from-end t) '(1 2 4 1 3 5)))
	(test-t (equal (remove 3 '(1 2 4 1 3 4 5) :test >) '(4 3 4 5)))
	(test-t (let* ((lst '(list of four elements)) (lst2 (copy-seq lst)) (lst3 (delete 'four lst))) (and (equal lst3 '(list of elements)) (not (equal lst lst2)))))
	(test-t (equal (remove-if oddp '(1 2 4 1 3 4 5)) '(2 4 4)))
	(test-t (equal (remove-if evenp '(1 2 4 1 3 4 5) :count 1 :from-end t) '(1 2 4 1 3 5)))
	(test-t (equal (remove-if-not evenp '(1 2 3 4 5 6 7 8 9) :count 2 :from-end t) '(1 2 3 4 5 6 8)))
	(test-t (equal (delete 4 (list 1 2 4 1 3 4 5)) '(1 2 1 3 5)))
	(test-t (equal (delete 4 (list 1 2 4 1 3 4 5) :count 1) '(1 2 1 3 4 5)))
	(test-t (equal (delete 4 (list 1 2 4 1 3 4 5) :count 1 :from-end t) '(1 2 4 1 3 5)))
	(test-t (equal (delete 3 (list 1 2 4 1 3 4 5) :test >) '(4 3 4 5)))
	(test-t (equal (delete-if oddp (list 1 2 4 1 3 4 5)) '(2 4 4)))
	(test-t (equal (delete-if evenp (list 1 2 4 1 3 4 5) :count 1 :from-end t) '(1 2 4 1 3 5)))
	(test-t (equal (delete-if evenp (list 1 2 3 4 5 6)) '(1 3 5)))
	(test-t (let* ((list0 (list 0 1 2 3 4)) (list (remove 3 list0))) (and (not (eq list0 list)) (equal list0 '(0 1 2 3 4)) (equal list '(0 1 2 4)))))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c)) '(b c b c)))
	(test-t (equal (remove 'b (list 'a 'b 'c 'a 'b 'c)) '(a c a c)))
	(test-t (equal (remove 'c (list 'a 'b 'c 'a 'b 'c)) '(a b a b)))
	(test-t (equal (remove 'a (list 'a 'a 'a)) '()))
	(test-t (equal (remove 'z (list 'a 'b 'c)) '(a b c)))
	(test-t (equal (remove 'a '()) '()))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 0) '(a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 1) '(b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) '(a b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 2) '(b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) '(b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 3) '(b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) '(b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 4) '(b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) '(b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count -1) '(a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count -10) '(a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c) :count -100) '(a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) '(a b c b c b c b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) '(a b c b c b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) '(a b c b c b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) '(a b c b c b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) '(a b c a b c b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) '(a b c a b c a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) '(a b c a b c a b c a b c)))
	(test-t (equal (remove 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) '(a b c a b c a b c a b c)))
	(test-t (equal (remove 'a (list '(a) '(b) '(c) '(a) '(b) '(c)) :key car) '((b) (c) (b) (c))))
	(test-t (equal (remove 'a (list '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) '((a . b) (b . c) (a . b) (b . c))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal) '()))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1) '(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t) '(("Love") ("LOve") ("LOVe"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 2 :from-end t) '(("Love") ("LOve"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :start 1 :end 3) '(("Love") ("LOVe") ("LOVE"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t :start 1 :end 3) '(("Love") ("LOve") ("LOVE"))))
	(test-t (equal (remove "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 10 :from-end t :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (let* ((vector0 (vector 0 1 2 3 4)) (vector (remove 3 vector0))) (and (not (eq vector0 vector)) (equalp vector0 #(0 1 2 3 4)) (equalp vector #(0 1 2 4)))))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c)) #(b c b c)))
	(test-t (equalp (remove 'b (vector 'a 'b 'c 'a 'b 'c)) #(a c a c)))
	(test-t (equalp (remove 'c (vector 'a 'b 'c 'a 'b 'c)) #(a b a b)))
	(test-t (equalp (remove 'a (vector 'a 'a 'a)) #()))
	(test-t (equalp (remove 'z (vector 'a 'b 'c)) #(a b c)))
	(test-t (equalp (remove 'a #()) #()))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 0) #(a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 1) #(b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) #(a b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 2) #(b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) #(b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 3) #(b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) #(b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 4) #(b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) #(b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count -1) #(a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count -10) #(a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c) :count -100) #(a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) #(a b c b c b c b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) #(a b c b c b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) #(a b c a b c b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove 'a (vector '(a) '(b) '(c) '(a) '(b) '(c)) :key car) #((b) (c) (b) (c))))
	(test-t (equalp (remove 'a (vector '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) #((a . b) (b . c) (a . b) (b . c))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal) #()))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :start 1 :end 3) #(("Love") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t :start 1 :end 3) #(("Love") ("LOve") ("LOVE"))))
	(test-t (equalp (remove "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 10 :from-end t :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (string= (remove #\a (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "")) ""))
	(test-t (string= (remove #\a (copy-seq "xyz")) "xyz"))
	(test-t (string= (remove #\a (copy-seq "ABCABC")) "ABCABC"))
	(test-t (string= (remove #\a (copy-seq "ABCABC") :key char-downcase) "BCBC"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (let* ((str0 (copy-seq "abc")) (str (remove #\a str0))) (and (not (eq str0 str)) (string= str0 "abc") (string= str "bc"))))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (remove #\b (copy-seq "abcabc")) "acac"))
	(test-t (string= (remove #\c (copy-seq "abcabc")) "abab"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 2) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 2 :from-end t) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 3) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 3 :from-end t) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 4) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count 4 :from-end t) "bcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count -1) "abcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabc") :count -100) "abcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1) "abcbcbcbc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :count 1) "abcbcabcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :count 2) "abcbcbcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end nil :count 2) "abcbcbcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end 8) "abcbcbcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1) "abcbcabcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1 :from-end t) "abcabcbcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count 0 :from-end t) "abcabcabcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count -100 :from-end t) "abcabcabcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 1 :end 1) "abcabcabcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 2 :end 2) "abcabcabcabc"))
	(test-t (string= (remove #\a (copy-seq "abcabcabcabc") :start 12 :end 12) "abcabcabcabc"))
	(test-t (let* ((list0 (list 0 1 2 3 4)) (list (remove-if (lambda (arg) (eql arg 3)) list0))) (and (not (eq list0 list)) (equal list0 '(0 1 2 3 4)) (equal list '(0 1 2 4)))))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c)) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'b)) (list 'a 'b 'c 'a 'b 'c)) '(a c a c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'c)) (list 'a 'b 'c 'a 'b 'c)) '(a b a b)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'a 'a)) '()))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'z)) (list 'a 'b 'c)) '(a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) '()) '()))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 0) '(a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 1) '(b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) '(a b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 2) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 3) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 4) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) '(b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count -1) '(a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count -10) '(a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count -100) '(a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) '(a b c b c b c b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) '(a b c b c b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) '(a b c b c b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) '(a b c b c b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) '(a b c a b c b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list '(a) '(b) '(c) '(a) '(b) '(c)) :key car) '((b) (c) (b) (c))))
	(test-t (equal (remove-if (lambda (arg) (eql arg 'a)) (list '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) '((a . b) (b . c) (a . b) (b . c))))
	(test-t (equal (remove-if (lambda (arg) (eql arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if (lambda (arg) (eql arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '()))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) '(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) '(("Love") ("LOve") ("LOVe"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) '(("Love") ("LOve"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :start 1 :end 3) '(("Love") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t :start 1 :end 3) '(("Love") ("LOve") ("LOVE"))))
	(test-t (equal (remove-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 10 :from-end t :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (let* ((vector0 (vector 0 1 2 3 4)) (vector (remove-if (lambda (arg) (eql arg 3)) vector0))) (and (not (eq vector0 vector)) (equalp vector0 #(0 1 2 3 4)) (equalp vector #(0 1 2 4)))))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c)) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'b)) (vector 'a 'b 'c 'a 'b 'c)) #(a c a c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'c)) (vector 'a 'b 'c 'a 'b 'c)) #(a b a b)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'a 'a)) #()))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'z)) (vector 'a 'b 'c)) #(a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) #()) #()))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 0) #(a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 1) #(b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) #(a b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 2) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 3) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 4) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) #(b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count -1) #(a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count -10) #(a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count -100) #(a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) #(a b c b c b c b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) #(a b c b c b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) #(a b c a b c b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector '(a) '(b) '(c) '(a) '(b) '(c)) :key car) #((b) (c) (b) (c))))
	(test-t (equalp (remove-if (lambda (arg) (eql arg 'a)) (vector '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) #((a . b) (b . c) (a . b) (b . c))))
	(test-t (equalp (remove-if (lambda (arg) (eql arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (eql arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #()))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #()))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :start 1 :end 3) #(("Love") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t :start 1 :end 3) #(("Love") ("LOve") ("LOVE"))))
	(test-t (equalp (remove-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 10 :from-end t :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "")) ""))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "xyz")) "xyz"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "ABCABC") :key char-downcase) "BCBC"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (let* ((str0 (copy-seq "abc")) (str (remove-if (lambda (arg) (string-equal arg #\a)) str0))) (and (not (eq str0 str)) (string= str0 "abc") (string= str "bc"))))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\b)) (copy-seq "abcabc")) "acac"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\c)) (copy-seq "abcabc")) "abab"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 2) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 2 :from-end t) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 3) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 3 :from-end t) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 4) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 4 :from-end t) "bcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -1) "abcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -100) "abcabc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabcabcabc") :start 1) "abcbcbcbc"))
	(test-t (string= (remove-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabcabcabc") :start 1 :count 1) "abcbcabcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :count 2) "abcbcbcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end nil :count 2) "abcbcbcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8) "abcbcbcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1) "abcbcabcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1 :from-end t) "abcabcbcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 0 :from-end t) "abcabcabcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count -100 :from-end t) "abcabcabcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 1) "abcabcabcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 2 :end 2) "abcabcabcabc"))
	(test-t (string= (remove-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 12 :end 12) "abcabcabcabc"))
	(test-t (let* ((list0 (list 0 1 2 3 4)) (list (remove-if-not (lambda (arg) (not (eql arg 3))) list0))) (and (not (eq list0 list)) (equal list0 '(0 1 2 3 4)) (equal list '(0 1 2 4)))))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c)) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'b))) (list 'a 'b 'c 'a 'b 'c)) '(a c a c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'c))) (list 'a 'b 'c 'a 'b 'c)) '(a b a b)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'a 'a)) '()))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'z))) (list 'a 'b 'c)) '(a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) '()) '()))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 0) '(a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 1) '(b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) '(a b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 2) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 3) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 4) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) '(b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count -1) '(a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count -10) '(a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count -100) '(a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) '(a b c b c b c b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) '(a b c b c b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) '(a b c b c b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) '(a b c b c b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) '(a b c a b c b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) '(a b c a b c a b c a b c)))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list '(a) '(b) '(c) '(a) '(b) '(c)) :key car) '((b) (c) (b) (c))))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg 'a))) (list '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) '((a . b) (b . c) (a . b) (b . c))))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (eql arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '()))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) '(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) '(("Love") ("LOve") ("LOVe"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) '(("Love") ("LOve"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :start 1 :end 3) '(("Love") ("LOVe") ("LOVE"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t :start 1 :end 3) '(("Love") ("LOve") ("LOVE"))))
	(test-t (equal (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 10 :from-end t :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (let* ((vector0 (vector 0 1 2 3 4)) (vector (remove-if-not (lambda (arg) (not (eql arg 3))) vector0))) (and (not (eq vector0 vector)) (equalp vector0 #(0 1 2 3 4)) (equalp vector #(0 1 2 4)))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c)) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'b))) (vector 'a 'b 'c 'a 'b 'c)) #(a c a c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'c))) (vector 'a 'b 'c 'a 'b 'c)) #(a b a b)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'a 'a)) #()))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'z))) (vector 'a 'b 'c)) #(a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) #()) #()))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 0) #(a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 1) #(b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) #(a b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 2) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 3) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 4) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) #(b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count -1) #(a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count -10) #(a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c) :count -100) #(a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) #(a b c b c b c b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) #(a b c b c b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) #(a b c a b c b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) #(a b c a b c a b c a b c)))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector '(a) '(b) '(c) '(a) '(b) '(c)) :key car) #((b) (c) (b) (c))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg 'a))) (vector '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) #((a . b) (b . c) (a . b) (b . c))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (eql arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #()))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #()))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :start 1 :end 3) #(("Love") ("LOVe") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t :start 1 :end 3) #(("Love") ("LOve") ("LOVE"))))
	(test-t (equalp (remove-if-not (lambda (arg) (not (string-equal arg "love"))) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 10 :from-end t :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "")) ""))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "xyz")) "xyz"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "ABCABC") :key char-downcase) "BCBC"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (let* ((str0 (copy-seq "abc")) (str (remove-if-not (lambda (arg) (not (string-equal arg #\a))) str0))) (and (not (eq str0 str)) (string= str0 "abc") (string= str "bc"))))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\b))) (copy-seq "abcabc")) "acac"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\c))) (copy-seq "abcabc")) "abab"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 2) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 2 :from-end t) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 3) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 3 :from-end t) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 4) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count 4 :from-end t) "bcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count -1) "abcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabc") :count -100) "abcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabcabcabc") :start 1) "abcbcbcbc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (string-equal arg #\a))) (copy-seq "abcabcabcabc") :start 1 :count 1) "abcbcabcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :count 2) "abcbcbcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end nil :count 2) "abcbcbcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end 8) "abcbcbcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1) "abcbcabcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1 :from-end t) "abcabcbcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 0 :from-end t) "abcabcabcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end 8 :count -100 :from-end t) "abcabcabcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 1 :end 1) "abcabcabcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 2 :end 2) "abcabcabcabc"))
	(test-t (string= (remove-if-not (lambda (arg) (not (eql arg #\a))) (copy-seq "abcabcabcabc") :start 12 :end 12) "abcabcabcabc"))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c)) '(b c b c)))
	(test-t (equal (delete 'b (list 'a 'b 'c 'a 'b 'c)) '(a c a c)))
	(test-t (equal (delete 'c (list 'a 'b 'c 'a 'b 'c)) '(a b a b)))
	(test-t (equal (delete 'a (list 'a 'a 'a)) '()))
	(test-t (equal (delete 'z (list 'a 'b 'c)) '(a b c)))
	(test-t (equal (delete 'a '()) '()))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 0) '(a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 1) '(b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) '(a b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 2) '(b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) '(b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 3) '(b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) '(b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 4) '(b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) '(b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count -1) '(a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count -10) '(a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c) :count -100) '(a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) '(a b c b c b c b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) '(a b c b c b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) '(a b c b c b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) '(a b c b c b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) '(a b c a b c b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) '(a b c a b c a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) '(a b c a b c a b c a b c)))
	(test-t (equal (delete 'a (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) '(a b c a b c a b c a b c)))
	(test-t (equal (delete 'a (list '(a) '(b) '(c) '(a) '(b) '(c)) :key car) '((b) (c) (b) (c))))
	(test-t (equal (delete 'a (list '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) '((a . b) (b . c) (a . b) (b . c))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal) '()))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1) '(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t) '(("Love") ("LOve") ("LOVe"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 2 :from-end t) '(("Love") ("LOve"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :start 1 :end 3) '(("Love") ("LOVe") ("LOVE"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t :start 1 :end 3) '(("Love") ("LOve") ("LOVE"))))
	(test-t (equal (delete "love" (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 10 :from-end t :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c)) #(b c b c)))
	(test-t (equalp (delete 'b (vector 'a 'b 'c 'a 'b 'c)) #(a c a c)))
	(test-t (equalp (delete 'c (vector 'a 'b 'c 'a 'b 'c)) #(a b a b)))
	(test-t (equalp (delete 'a (vector 'a 'a 'a)) #()))
	(test-t (equalp (delete 'z (vector 'a 'b 'c)) #(a b c)))
	(test-t (equalp (delete 'a #()) #()))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 0) #(a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 1) #(b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) #(a b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 2) #(b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) #(b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 3) #(b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) #(b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 4) #(b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) #(b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count -1) #(a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count -10) #(a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c) :count -100) #(a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) #(a b c b c b c b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) #(a b c b c b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) #(a b c a b c b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete 'a (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete 'a (vector '(a) '(b) '(c) '(a) '(b) '(c)) :key car) #((b) (c) (b) (c))))
	(test-t (equalp (delete 'a (vector '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) #((a . b) (b . c) (a . b) (b . c))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal) #()))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :start 1 :end 3) #(("Love") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 1 :from-end t :start 1 :end 3) #(("Love") ("LOve") ("LOVE"))))
	(test-t (equalp (delete "love" (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :test string-equal :count 10 :from-end t :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (string= (delete #\a (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "")) ""))
	(test-t (string= (delete #\a (copy-seq "xyz")) "xyz"))
	(test-t (string= (delete #\a (copy-seq "ABCABC")) "ABCABC"))
	(test-t (string= (delete #\a (copy-seq "ABCABC") :key char-downcase) "BCBC"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (delete #\b (copy-seq "abcabc")) "acac"))
	(test-t (string= (delete #\c (copy-seq "abcabc")) "abab"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 2) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 2 :from-end t) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 3) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 3 :from-end t) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 4) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count 4 :from-end t) "bcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count -1) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabc") :count -100) "abcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1) "abcbcbcbc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :count 1) "abcbcabcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :count 2) "abcbcbcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end nil :count 2) "abcbcbcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end 8) "abcbcbcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1) "abcbcabcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1 :from-end t) "abcabcbcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count 0 :from-end t) "abcabcabcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end 8 :count -100 :from-end t) "abcabcabcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 1 :end 1) "abcabcabcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 2 :end 2) "abcabcabcabc"))
	(test-t (string= (delete #\a (copy-seq "abcabcabcabc") :start 12 :end 12) "abcabcabcabc"))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c)) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'b)) (list 'a 'b 'c 'a 'b 'c)) '(a c a c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'c)) (list 'a 'b 'c 'a 'b 'c)) '(a b a b)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'a 'a)) '()))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'z)) (list 'a 'b 'c)) '(a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) '()) '()))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 0) '(a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 1) '(b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) '(a b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 2) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 3) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 4) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) '(b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count -1) '(a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count -10) '(a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c) :count -100) '(a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) '(a b c b c b c b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) '(a b c b c b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) '(a b c b c b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) '(a b c b c b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) '(a b c a b c b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list '(a) '(b) '(c) '(a) '(b) '(c)) :key car) '((b) (c) (b) (c))))
	(test-t (equal (delete-if (lambda (arg) (eql arg 'a)) (list '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) '((a . b) (b . c) (a . b) (b . c))))
	(test-t (equal (delete-if (lambda (arg) (eql arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if (lambda (arg) (eql arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '()))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) '(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) '(("Love") ("LOve") ("LOVe"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) '(("Love") ("LOve"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :start 1 :end 3) '(("Love") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t :start 1 :end 3) '(("Love") ("LOve") ("LOVE"))))
	(test-t (equal (delete-if (lambda (arg) (string-equal arg "love")) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 10 :from-end t :start 1 :end 3) '(("Love") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c)) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'b)) (vector 'a 'b 'c 'a 'b 'c)) #(a c a c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'c)) (vector 'a 'b 'c 'a 'b 'c)) #(a b a b)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'a 'a)) #()))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'z)) (vector 'a 'b 'c)) #(a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) #()) #()))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 0) #(a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 1) #(b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) #(a b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 2) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 3) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 4) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) #(b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count -1) #(a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count -10) #(a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c) :count -100) #(a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) #(a b c b c b c b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) #(a b c b c b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) #(a b c b c b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) #(a b c b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) #(a b c a b c b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) #(a b c a b c a b c a b c)))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector '(a) '(b) '(c) '(a) '(b) '(c)) :key car) #((b) (c) (b) (c))))
	(test-t (equalp (delete-if (lambda (arg) (eql arg 'a)) (vector '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) #((a . b) (b . c) (a . b) (b . c))))
	(test-t (equalp (delete-if (lambda (arg) (eql arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (eql arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) #(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #()))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) #()))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) #(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t) #(("Love") ("LOve") ("LOVe"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 2 :from-end t) #(("Love") ("LOve"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :start 1 :end 3) #(("Love") ("LOVe") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1 :from-end t :start 1 :end 3) #(("Love") ("LOve") ("LOVE"))))
	(test-t (equalp (delete-if (lambda (arg) (string-equal arg "love")) (vector '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 10 :from-end t :start 1 :end 3) #(("Love") ("LOVE"))))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "")) ""))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "xyz")) "xyz"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "ABCABC") :key char-downcase) "BCBC"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc")) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\b)) (copy-seq "abcabc")) "acac"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\c)) (copy-seq "abcabc")) "abab"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 0) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1) "bcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 1 :from-end t) "abcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 2) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 2 :from-end t) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 3) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 3 :from-end t) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 4) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count 4 :from-end t) "bcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -1) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -10) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabc") :count -100) "abcabc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabcabcabc") :start 1) "abcbcbcbc"))
	(test-t (string= (delete-if (lambda (arg) (string-equal arg #\a)) (copy-seq "abcabcabcabc") :start 1 :count 1) "abcbcabcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :count 2) "abcbcbcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end nil :count 2) "abcbcbcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8) "abcbcbcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1) "abcbcabcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 1 :from-end t) "abcabcbcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count 0 :from-end t) "abcabcabcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 8 :count -100 :from-end t) "abcabcabcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 1 :end 1) "abcabcabcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 2 :end 2) "abcabcabcabc"))
	(test-t (string= (delete-if (lambda (arg) (eql arg #\a)) (copy-seq "abcabcabcabc") :start 12 :end 12) "abcabcabcabc"))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c)) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'b))) (list 'a 'b 'c 'a 'b 'c)) '(a c a c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'c))) (list 'a 'b 'c 'a 'b 'c)) '(a b a b)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'a 'a)) '()))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'z))) (list 'a 'b 'c)) '(a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) '()) '()))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 0) '(a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 1) '(b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 1 :from-end t) '(a b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 2) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 2 :from-end t) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 3) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 3 :from-end t) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 4) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count 4 :from-end t) '(b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count -1) '(a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count -10) '(a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c) :count -100) '(a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1) '(a b c b c b c b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :count 2) '(a b c b c b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end nil :count 2) '(a b c b c b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8) '(a b c b c b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1) '(a b c b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 1 :from-end t) '(a b c a b c b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count 0 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 8 :count -100 :from-end t) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 1 :end 1) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 2 :end 2) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list 'a 'b 'c 'a 'b 'c 'a 'b 'c 'a 'b 'c) :start 12 :end 12) '(a b c a b c a b c a b c)))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list '(a) '(b) '(c) '(a) '(b) '(c)) :key car) '((b) (c) (b) (c))))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg 'a))) (list '(a . b) '(b . c) '(c . a) '(a . b) '(b . c) '(c . a)) :key cdr) '((a . b) (b . c) (a . b) (b . c))))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if-not (lambda (arg) (not (eql arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count -10) '(("Love") ("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car) '()))
	(test-t (equal (delete-if-not (lambda (arg) (not (string-equal arg "love"))) (list '("Love") '("LOve") '("LOVe") '("LOVE")) :key car :count 1) '(("LOve") ("LOVe") ("LOVE"))))
	(test-t (equal (delete-if-not (lambda (arg) (not (string-equal arg "lov