/*
 * Routines to interpret node coordinates and names on a SeaStar system
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_alps.h"
#include "basil_torus.h"

/* Class 0 torus: N x 4 x 8, with N = number of cages */
static struct torus_coord node2coord_class0(const struct nodecoord *nc,
					    const struct torus_info *ti)
{
	struct torus_coord tc;
	/*
	 * X coordinate counting with 2 cabinets:
	 *
		   +-----+     +-----+
		c2 |  2  |     |  3  |
		   +-----+     +-----+
		c1 |  1  |     |  4  |
		   +-----+     +-----+
		c0 |  0  |     |  5  |
		   +-----+     +-----+
		    c0-0        c1-0
	 *
	 * X coordinate counting with 3 cabinets:
	 *
		   +-----+     +-----+     +-----+
		c2 |  2  |     |  3  |     |  5  |
		   +-----+     +-----+     +-----+
		c1 |  1  |     |  4  |     |  6  |
		   +-----+     +-----+     +-----+
		c0 |  0  |     |  8  |     |  7  |
		   +-----+     +-----+     +-----+
		    c0-0        c1-0        c2-0
	 */

	if (nc->cab == 0)
		tc.x = nc->cage;
	else if (nc->cab == 1 && ti->cabs == 3 && nc->cage == 0)
		tc.x = 8;
	else if (nc->cab == 1)
		tc.x = 5 - nc->cage;
	else if (nc->cab == 2)
		tc.x = 7 - nc->cage;
	else
		fatal("Invalid cabinet %d in class0 torus", nc->cab);
	tc.y = nc->node;
	tc.z = nc->slot;

	return tc;
}

/* Class 2 torus: N x 12 x 16, with N = number of cabinets in a row */
static struct torus_coord node2coord_class2(const struct nodecoord *nc,
					    const struct torus_info *ti)
{
	struct torus_coord tc;

	if (nc->cab % 2 == 1)
		tc.x = ti->cabs - (nc->cab + 1)/ 2;
	else
		tc.x = nc->cab / 2;

	if (nc->row == 0) {
		tc.y = 4 * nc->cage + nc->node;
		tc.z = nc->slot;
	} else {
		tc.y = 4 * (2 - nc->cage) + nc->node;
		tc.z = 15 - nc->slot;
	}
	return tc;
}

/** Resolve node coordinates in a SeaStar torus */
struct torus_coord nid2seastar_toruscoord(const struct nodecoord *nc,
					  const struct torus_info *ti)
{
	struct torus_coord (*algo[])(const struct nodecoord  *,
				     const struct torus_info *) = {
		[CABLING_CLASS0] = node2coord_class0,
		[CABLING_CLASS1] = NULL,
		[CABLING_CLASS2] = node2coord_class2,
		[CABLING_CLASS3] = NULL
	};

	if (algo[ti->cabling] == NULL)
		fatal("unsupported SeaStar Cabling Class %d", ti->cabling);

	return (*algo[ti->cabling])(nc, ti);
}
