/*
 *	Node placement: manage information about placed aprun entries
 */
#include "common.h"

/** Constructor */
static struct node_placement *np_alloc(uint32_t nid)
{
	struct node_placement *new = calloc(1, sizeof(*new));

	if (new == NULL)
		fatal("out of memory allocating entry for nid %u", nid);

	new->nid = nid;
	new->cmd_idx = -1;
	return new;
}

/**
 * np_lookup - look up node placement entry by node id
 * @head: start of node placement list
 * @nid:  node ID to look up
 * Return found entry or NULL if not found.
 */
struct node_placement *np_lookup(struct node_placement *head, uint32_t nid)
{
	struct node_placement *cur;

	for (cur = head; cur; cur = cur->next)
		if (cur->nid == nid)
			return cur;
	return NULL;
}

/**
 * np_count - count number of unique node IDs in node placement list
 * @head: start of node placement list
 * Return count of unique node IDs in @head (0 may indicate error).
 */
int np_count(struct node_placement *head)
{
	struct node_placement *cur;
	int node_count = 0;

	for (cur = head; cur; cur = cur->next)
		node_count++;
	return node_count;
}

/**
 * np_add - add node placement entry to list
 * @head: start of list (initially points to NULL list entry)
 * @nid:  node ID of entry to look up  or add
 * Returns existing entry if @nid is not new, or pointer to new entry.
 */
struct node_placement *np_add(struct node_placement **head, uint32_t nid)
{
	struct node_placement *cur, *new = NULL;

	if (*head == NULL || (*head)->nid > nid) {
		new = np_alloc(nid);
		new->next = *head;
		*head 	  = new;
	}  else {
		for (cur = *head; cur; cur = cur->next)
			if (cur->nid == nid) {
				return cur;
			} else if (cur->next == NULL || nid < cur->next->nid) {
				new = np_alloc(nid);
				new->next = cur->next;
				cur->next = new;
				break;
			}
	}
	return new;
}

/** Destructor */
void np_free(struct node_placement *np)
{
	if (np) {
		np_free(np->next);
		free(np);
	}
}

/**
 * appinfo_get_node_placements  -  get node placement information
 * @ai_buf:   start of appinfo buffer
 * Return list of node placements or NULL on error.
 */
struct node_placement *appinfo_get_node_placements(const uint8_t *ai_buf)
{
	appInfoHdr_t *ai_hdr = (appInfoHdr_t *)ai_buf;
	struct node_placement *np_head = NULL;
	size_t offset;
	int i, p;

	if (ai_buf == NULL)
		return NULL;

	for (i = 0, offset = ai_hdr->apStart; i < ai_hdr->apNum; i++) {
		appInfo_t   *ai_info = (appInfo_t *)(ai_buf + offset);
		cmdDetail_t *cmd   = (cmdDetail_t *)(ai_buf + ai_info->cmdDetail);
		placeList_t *pl    = (placeList_t *)(ai_buf + ai_info->places);

		/* walk placement list */
		for (p = 0; p < ai_info->numPlaces; p++, pl++) {
			struct node_placement *new = np_add(&np_head, pl->nid);

			assert(new != NULL);
			assert(pl->cmdIx < ai_info->numCmds);
			if (!ai_info->timePlaced) {
				new->res_width   = cmd[pl->cmdIx].width;
				new->res_depth   = cmd[pl->cmdIx].depth;
				new->res_nppn    = cmd[pl->cmdIx].fixedPerNode;
				new->res_memory  = cmd[pl->cmdIx].memory;
			} else if (new->cmd_idx < 0) {
				new->cmd_idx = pl->cmdIx;
				new->apid    = ai_info->apid;
				new->resid   = ai_info->resId;
				new->cores   = 1;
				new->width   = cmd[pl->cmdIx].width;
				new->depth   = cmd[pl->cmdIx].depth;
				new->nppn    = cmd[pl->cmdIx].fixedPerNode;
				new->memory  = cmd[pl->cmdIx].memory;
			} else {
				new->cores  += 1;
				assert(new->cmd_idx == pl->cmdIx);
				assert(new->apid    == ai_info->apid);
				assert(new->resid   == ai_info->resId);
				assert(new->width   == cmd[pl->cmdIx].width);
				assert(new->depth   == cmd[pl->cmdIx].depth);
				assert(new->nppn    == cmd[pl->cmdIx].fixedPerNode);
				assert(new->memory  == cmd[pl->cmdIx].memory);
			}
		}
		offset += sizeof(appInfo_t);
		offset += ai_info->numCmds   * sizeof(cmdDetail_t);
		offset += ai_info->numPlaces * sizeof(placeList_t);
	}
	return np_head;
}

/**
 * appinfo_get_num_nodes  -  Count number of nodes in placement list
 * @ai_buf:   start of appinfo buffer
 */
int appinfo_get_num_nodes(const uint8_t *ai_buf)
{
	struct node_placement *np_head = appinfo_get_node_placements(ai_buf);
	int node_count = np_count(np_head);

	np_free(np_head);
	return node_count;
}

/* For debugging */
#ifdef _WANT_A_MAIN_PROGRAM_TO_TEST
void np_print(struct node_placement *head)
{
	struct node_placement *cur;

	for (cur = head; cur; cur = cur->next)
		printf("%3d %u %llu -n %d/%d -d %d -N %d -m %d\n", cur->nid,
			cur->resid, (unsigned long long)cur->apid, cur->width,
			cur->cores, cur->depth, cur->nppn, cur->memory);
	printf("\n");
}

int main(int ac, char *av)
{
	struct node_placement *np_head = NULL;

	np_add(&np_head, 10); np_add(&np_head, 10);
	np_add(&np_head, 10);
	np_add(&np_head, 9);
	np_add(&np_head, 8);
	np_add(&np_head, 9);
	np_add(&np_head, 3);
	np_add(&np_head, 2);
	np_add(&np_head, 5);
	np_print(np_head);
	np_free(np_head);
	return EXIT_SUCCESS;
}
#endif /* _WANT_A_MAIN_PROGRAM_TO_TEST */
