/* -------------------------------------------------------------------------- *
 *                 Simbody(tm) Example: Amy's IK Problem                      *
 * -------------------------------------------------------------------------- *
 * This is part of the SimTK biosimulation toolkit originating from           *
 * Simbios, the NIH National Center for Physics-Based Simulation of           *
 * Biological Structures at Stanford, funded under the NIH Roadmap for        *
 * Medical Research, grant U54 GM072970. See https://simtk.org/home/simbody.  *
 *                                                                            *
 * Portions copyright (c) 2010-12 Stanford University and the Authors.        *
 * Authors: Michael Sherman                                                   *
 * Contributors:                                                              *
 *                                                                            *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may    *
 * not use this file except in compliance with the License. You may obtain a  *
 * copy of the License at http://www.apache.org/licenses/LICENSE-2.0.         *
 *                                                                            *
 * Unless required by applicable law or agreed to in writing, software        *
 * distributed under the License is distributed on an "AS IS" BASIS,          *
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   *
 * See the License for the specific language governing permissions and        *
 * limitations under the License.                                             *
 * -------------------------------------------------------------------------- */

/* This is a test of repeated Assembly analysis, also known as 
InverseKinematics (IK). The problem is from a 56 dof OpenSim model built by Amy 
Silder in Scott Delp's lab. The model was read from various OpenSim data
files and machine-translated into a standalone C++ Simbody model during the 
development of the Assembler study. There are "markers" (distinguished points)
on many of the bodies, and many frames of laboratory data giving sequentially-
observed positions of these markers. The idea is to solve for the set of q's
in each frame that pose the bodies so that their attached marker positions
are a best fit for the observed data. Sequential frames are expected to be
spatially close. 

There is some commented-out code below for playing with various ways
of handling constraints.
*/


#include "Simbody.h"

#include <cstdio>
#include <exception>

using std::cout; using std::cin; using std::endl;

using namespace SimTK;

static int getNumFrames();
static int getNumObservations();
static Real getFrameTime(int i);
static const char* const* getObservationOrder();
static ArrayViewConst_<Vec3> getFrame(int i);

int main() {
  try
  { // Create the system.
    
    MultibodySystem         system;
    SimbodyMatterSubsystem  matter(system);
    GeneralForceSubsystem   forces(system);
    Force::UniformGravity   gravity(forces, matter, Vec3(0, -9.8, 0));

// BODY ground
Markers& markers = *new Markers();
DecorativeGeometry markerDecoration = DecorativeSphere(0.1).setColor(Blue);
MobilizedBody& mobod_ground = matter.updGround();
// END BODY ground


// Kludge to get the initial view large enough.
mobod_ground.addBodyDecoration(Vec3(-1,-1,-1), markerDecoration);
mobod_ground.addBodyDecoration(Vec3(1,1,1), markerDecoration);

// BODY loadcell1
Body::Rigid body_loadcell1(MassProperties(0,Vec3(0),Inertia(0,0,0)));
// Joint name is gnd_loadcell1
MobilizedBody::Free mobod_loadcell1(mobod_ground,Vec3(0), body_loadcell1,Vec3(0));
// END BODY loadcell1

// BODY loadcell2
Body::Rigid body_loadcell2(MassProperties(0,Vec3(0),Inertia(0,0,0)));
// Joint name is gnd_loadcell2
MobilizedBody::Free mobod_loadcell2(mobod_ground,Vec3(0), body_loadcell2,Vec3(0));
// END BODY loadcell2

// BODY pelvis
Body::Rigid body_pelvis(MassProperties(8.3775,Vec3(-0.0893,-0.025796,0),Inertia(0.0595677,0.128085,0.139644,-0.0192983,0,0)));
body_pelvis.addDecoration(Vec3(-.1,0,0),DecorativeEllipsoid(Vec3(.1,.06,.15)).setColor(Cyan));
// Joint name is gnd_pelvis
MobilizedBody::Free mobod_pelvis(mobod_ground,Vec3(0), body_pelvis,Vec3(0));
markers.addMarker("S2",mobod_pelvis,Vec3(-0.2,0,0),1);
body_pelvis.addDecoration(Vec3(-0.2,0,0),markerDecoration);
markers.addMarker("R.ASIS",mobod_pelvis,Vec3(0.01,0,0.128),1);
body_pelvis.addDecoration(Vec3(0.01,0,0.128),markerDecoration);
markers.addMarker("R.Iliac",mobod_pelvis,Vec3(-0.08,0.06,0.15),1);
body_pelvis.addDecoration(Vec3(-0.08,0.06,0.15),markerDecoration);
markers.addMarker("R.PSIS",mobod_pelvis,Vec3(-0.2,0,0.055),1);
body_pelvis.addDecoration(Vec3(-0.2,0,0.055),markerDecoration);
markers.addMarker("L.ASIS",mobod_pelvis,Vec3(0.01,0,-0.128),1);
body_pelvis.addDecoration(Vec3(0.01,0,-0.128),markerDecoration);
markers.addMarker("L.Iliac",mobod_pelvis,Vec3(-0.08,0.06,-0.15),1);
body_pelvis.addDecoration(Vec3(-0.08,0.06,-0.15),markerDecoration);
markers.addMarker("L.PSIS",mobod_pelvis,Vec3(-0.2,0,-0.055),1);
body_pelvis.addDecoration(Vec3(-0.2,0,-0.055),markerDecoration);
markers.addMarker("R_HJC",mobod_pelvis,Vec3(-0.08,-0.0824,0.0785),1);
body_pelvis.addDecoration(Vec3(-0.08,-0.0824,0.0785),markerDecoration);
markers.addMarker("L_HJC",mobod_pelvis,Vec3(-0.08,-0.0824,-0.0785),1);
body_pelvis.addDecoration(Vec3(-0.08,-0.0824,-0.0785),markerDecoration);
// END BODY pelvis

// BODY torso
Body::Rigid body_torso(MassProperties(24.2175,Vec3(0.0145,0.256438,0),Inertia(2.03337,0.281964,2.19416,-0.0900492,0,0)));
body_torso.addDecoration(Vec3(0,.25,0), DecorativeBrick(Vec3(.08,.3,.165)).setColor(Green).setOpacity(.2));
// Joint name is pelvis_torso
MobilizedBody::Ball mobod_torso(mobod_pelvis,Vec3(-0.0872,0.0331,0), body_torso,Vec3(0));
markers.addMarker("R.Clavicle",mobod_torso,Vec3(0.035,0.43,0.025),1);
body_torso.addDecoration(Vec3(0.035,0.43,0.025),markerDecoration);
markers.addMarker("L.Clavicle",mobod_torso,Vec3(0.035,0.43,-0.025),1);
body_torso.addDecoration(Vec3(0.035,0.43,-0.025),markerDecoration);
markers.addMarker("L.Scapula",mobod_torso,Vec3(-0.12,-0.1,0.08),1);
body_torso.addDecoration(Vec3(-0.12,-0.1,0.08),markerDecoration);
markers.addMarker("R.Shoulder",mobod_torso,Vec3(-0.0488,0.47065,0.165),1);
body_torso.addDecoration(Vec3(-0.0488,0.47065,0.165),markerDecoration);
markers.addMarker("L.Shoulder",mobod_torso,Vec3(-0.0488,0.47065,-0.165),1);
body_torso.addDecoration(Vec3(-0.0488,0.47065,-0.165),markerDecoration);
// END BODY torso

// BODY neckhead
Body::Rigid body_neckhead(MassProperties(5.205,Vec3(0.0288,0.161501,0),Inertia(0.166231,0.0252372,0.168272,-0.0242096,0,0)));
body_neckhead.addDecoration(Vec3(0.0288,0.161501,0), DecorativeSphere(.12).setColor(Yellow));
// Joint name is torso_neckhead
MobilizedBody::Ball mobod_neckhead(mobod_torso,Vec3(-0.0253,0.44895,0), body_neckhead,Vec3(0));
markers.addMarker("C7",mobod_neckhead,Vec3(0),1);
body_neckhead.addDecoration(Vec3(0),markerDecoration);
// END BODY neckhead

// BODY humerus_r
Body::Rigid body_humerus_r(MassProperties(2.0325,Vec3(0,-0.164502,0),Inertia(0.0669473,0.004121,0.0684103)));
body_humerus_r.addDecoration(Vec3(0,-.15,0), DecorativeEllipsoid(Vec3(.05,.3,.05)).setColor(Yellow));
// Joint name is acromial_r
MobilizedBody::Ball mobod_humerus_r(mobod_torso,Vec3(-0.0281,0.41645,0.199), body_humerus_r,Vec3(0));
markers.addMarker("R.Bicep",mobod_humerus_r,Vec3(0.04,-0.15,0),1);
body_humerus_r.addDecoration(Vec3(0.04,-0.15,0),markerDecoration);
markers.addMarker("R.Elbow",mobod_humerus_r,Vec3(-0.01,-0.281,0.04),1);
body_humerus_r.addDecoration(Vec3(-0.01,-0.281,0.04),markerDecoration);
markers.addMarker("R.MElbow",mobod_humerus_r,Vec3(0,-0.281,-0.04),1);
body_humerus_r.addDecoration(Vec3(0,-0.281,-0.04),markerDecoration);
// END BODY humerus_r

// BODY ulna_r
Body::Rigid body_ulna_r(MassProperties(0.6075,Vec3(0,-0.120525,0),Inertia(0.0117867,0.000618,0.0120377)));
body_ulna_r.addDecoration(Vec3(0,-.15,0), DecorativeEllipsoid(Vec3(.03,.3,.03)).setColor(Yellow));
// Joint name is elbow_r
MobilizedBody::Pin mobod_ulna_r(mobod_humerus_r,Vec3(0.013144,-0.286273,-0.009595), body_ulna_r,Vec3(0));
markers.addMarker("R.Forearm",mobod_ulna_r,Vec3(0,-0.15,0),1);
body_ulna_r.addDecoration(Vec3(0,-0.15,0),markerDecoration);
// END BODY ulna_r

// BODY radius_r
Body::Rigid body_radius_r(MassProperties(0.6075,Vec3(0,-0.120525,0),Inertia(0.0117867,0.000618,0.0120377)));
// Joint name is radioulnar_r
MobilizedBody::Pin mobod_radius_r(mobod_ulna_r,Vec3(-0.006727,-0.013007,0.026083), body_radius_r,Vec3(0));
markers.addMarker("R.Wrist",mobod_radius_r,Vec3(-0.025,-0.245,0.006),1);
body_radius_r.addDecoration(Vec3(-0.025,-0.245,0.006),markerDecoration);
// END BODY radius_r

// BODY hand_r
Body::Rigid body_hand_r(MassProperties(0.4575,Vec3(0,-0.068095,0),Inertia(0.0030134,0.000547,0.0034614)));
// Joint name is radius_hand_r
MobilizedBody::Pin mobod_hand_r(mobod_radius_r,Vec3(-0.008797,-0.235841,0.01361), body_hand_r,Vec3(0));
// END BODY hand_r

// BODY humerus_l
Body::Rigid body_humerus_l(MassProperties(2.0325,Vec3(0,-0.164502,0),Inertia(0.0669473,0.004121,0.0684103)));
body_humerus_l.addDecoration(Vec3(0,-.15,0), DecorativeEllipsoid(Vec3(.05,.3,.05)).setColor(Yellow));
// Joint name is acromial_l
MobilizedBody::Ball mobod_humerus_l(mobod_torso,Vec3(-0.0281,0.41645,-0.199), body_humerus_l,Vec3(0));
markers.addMarker("L.Bicep",mobod_humerus_l,Vec3(0.04,-0.15,0),1);
body_humerus_l.addDecoration(Vec3(0.04,-0.15,0),markerDecoration);
markers.addMarker("L.Elbow",mobod_humerus_l,Vec3(0.01,-0.281,-0.04),1);
body_humerus_l.addDecoration(Vec3(0.01,-0.281,-0.04),markerDecoration);
markers.addMarker("L.MElbow",mobod_humerus_l,Vec3(-0.02,-0.281,0.04),1);
body_humerus_l.addDecoration(Vec3(-0.02,-0.281,0.04),markerDecoration);
// END BODY humerus_l

// BODY ulna_l
Body::Rigid body_ulna_l(MassProperties(0.6075,Vec3(0,-0.120525,0),Inertia(0.0117867,0.000618,0.0120377)));
body_ulna_l.addDecoration(Vec3(0,-.15,0), DecorativeEllipsoid(Vec3(.03,.3,.03)).setColor(Yellow));
// Joint name is elbow_l
MobilizedBody::Pin mobod_ulna_l(mobod_humerus_l,Vec3(0.013144,-0.286273,0.009595), body_ulna_l,Vec3(0));
markers.addMarker("L.Forearm",mobod_ulna_l,Vec3(0,-0.15,0),1);
body_ulna_l.addDecoration(Vec3(0,-0.15,0),markerDecoration);
// END BODY ulna_l

// BODY radius_l
Body::Rigid body_radius_l(MassProperties(0.6075,Vec3(0,-0.120525,0),Inertia(0.0117867,0.000618,0.0120377)));
// Joint name is radioulnar_l
MobilizedBody::Pin mobod_radius_l(mobod_ulna_l,Vec3(-0.006727,-0.013007,-0.026083), body_radius_l,Vec3(0));
markers.addMarker("L.Wrist",mobod_radius_l,Vec3(-0.025,-0.245,-0.006),1);
body_radius_l.addDecoration(Vec3(-0.025,-0.245,-0.006),markerDecoration);
// END BODY radius_l

// BODY hand_l
Body::Rigid body_hand_l(MassProperties(0.4575,Vec3(0,-0.068095,0),Inertia(0.0030134,0.000547,0.0034614)));
// Joint name is radius_hand_l
MobilizedBody::Pin mobod_hand_l(mobod_radius_l,Vec3(-0.008797,-0.235841,-0.01361), body_hand_l,Vec3(0));
// END BODY hand_l

// BODY femur_r
Body::Rigid body_femur_r(MassProperties(10.5225,Vec3(0,-0.162162,0),Inertia(0.455313,0.036634,0.455313)));
body_femur_r.addDecoration(Vec3(0,-.2,0), DecorativeEllipsoid(Vec3(.08,.2,.1)).setColor(Yellow));
// Joint name is hip_r
MobilizedBody::Ball mobod_femur_r(mobod_pelvis,Vec3(-0.08,-0.0824,0.0785), body_femur_r,Vec3(0));
markers.addMarker("R.Knee",mobod_femur_r,Vec3(0,-0.404,0.05),1);
body_femur_r.addDecoration(Vec3(0,-0.404,0.05),markerDecoration);
markers.addMarker("R.MKnee",mobod_femur_r,Vec3(0,-0.404,-0.05),1);
body_femur_r.addDecoration(Vec3(0,-0.404,-0.05),markerDecoration);
markers.addMarker("R.TH1",mobod_femur_r,Vec3(0.0179,-0.224,0.1147),1);
body_femur_r.addDecoration(Vec3(0.0179,-0.224,0.1147),markerDecoration);
markers.addMarker("R.TH2",mobod_femur_r,Vec3(0.0179,-0.264,0.0647),1);
body_femur_r.addDecoration(Vec3(0.0179,-0.264,0.0647),markerDecoration);
markers.addMarker("R.TH3",mobod_femur_r,Vec3(0.0179,-0.264,0.0647),1);
body_femur_r.addDecoration(Vec3(0.0179,-0.264,0.0647),markerDecoration);
markers.addMarker("R.TH4",mobod_femur_r,Vec3(0.08,-0.324,0.0047),1);
body_femur_r.addDecoration(Vec3(0.08,-0.324,0.0047),markerDecoration);
// END BODY femur_r

// BODY tibia_r
Body::Rigid body_tibia_r(MassProperties(3.2475,Vec3(0,-0.171713,0),Inertia(0.125753,0.005157,0.126985)));
body_tibia_r.addDecoration(Vec3(0,-.2,0), DecorativeEllipsoid(Vec3(.05,.2,.07)).setColor(Yellow));
// Joint name is knee_r
MobilizedBody::Pin mobod_tibia_r(mobod_femur_r,Vec3(0,-.4,0), body_tibia_r,Vec3(0));
markers.addMarker("R.Ankle",mobod_tibia_r,Vec3(-0.005,-0.41,0.053),1);
body_tibia_r.addDecoration(Vec3(-0.005,-0.41,0.053),markerDecoration);
markers.addMarker("R.MAnkle",mobod_tibia_r,Vec3(0.006,-0.3888,-0.038),1);
body_tibia_r.addDecoration(Vec3(0.006,-0.3888,-0.038),markerDecoration);
markers.addMarker("R.SH1",mobod_tibia_r,Vec3(0.0104,-0.2322,0.0748),1);
body_tibia_r.addDecoration(Vec3(0.0104,-0.2322,0.0748),markerDecoration);
markers.addMarker("R.SH2",mobod_tibia_r,Vec3(0.0125,-0.3196,0.06),1);
body_tibia_r.addDecoration(Vec3(0.0125,-0.3196,0.06),markerDecoration);
markers.addMarker("R.SH3",mobod_tibia_r,Vec3(0.0125,-0.3196,0.06),1);
body_tibia_r.addDecoration(Vec3(0.0125,-0.3196,0.06),markerDecoration);
markers.addMarker("R.SH4",mobod_tibia_r,Vec3(0.0125,-0.3196,0.06),1);
body_tibia_r.addDecoration(Vec3(0.0125,-0.3196,0.06),markerDecoration);
// END BODY tibia_r

// BODY patella_r
Body::Rigid body_patella_r(MassProperties(0.0975,Vec3(0.0018,0.0264,0),Inertia(7.09536e-005,1.53159e-005,8.32695e-005,-4.6332e-006,0,0)));
// Joint name is tib_pat_r
MobilizedBody::/*FourDOF*/Cartesian mobod_patella_r(mobod_tibia_r,Vec3(0), body_patella_r,Vec3(0));
// END BODY patella_r

// BODY talus_r
Body::Rigid body_talus_r(MassProperties(0.045,Vec3(0.0055,0.0023,0),Inertia(3.23805e-006,2.36125e-006,4.5993e-006,-5.6925e-007,0,0)));
// Joint name is ankle_r
MobilizedBody::Pin mobod_talus_r(mobod_tibia_r,Vec3(0,-0.426,0), body_talus_r,Vec3(0));
// END BODY talus_r

// BODY foot_r
Body::Rigid body_foot_r(MassProperties(0.885,Vec3(0.101865,0.0156,0),Inertia(0.00112437,0.0114972,0.0120816,-0.00140635,0,0)));
// Joint name is subtalar_r
MobilizedBody::Pin mobod_foot_r(mobod_talus_r,Vec3(-0.04877,-0.04195,0.00792), body_foot_r,Vec3(0));
markers.addMarker("R.Toe",mobod_foot_r,Vec3(0.19,0.018,0),1);
body_foot_r.addDecoration(Vec3(0.19,0.018,0),markerDecoration);
markers.addMarker("R.MT1",mobod_foot_r,Vec3(0.2,0.005,-0.045),1);
body_foot_r.addDecoration(Vec3(0.2,0.005,-0.045),markerDecoration);
markers.addMarker("R.MT5",mobod_foot_r,Vec3(0.15,0.005,0.055),1);
body_foot_r.addDecoration(Vec3(0.15,0.005,0.055),markerDecoration);
markers.addMarker("R.Heel",mobod_foot_r,Vec3(-0.03,0.018,-0.01),1);
body_foot_r.addDecoration(Vec3(-0.03,0.018,-0.01),markerDecoration);
// END BODY foot_r

// BODY toes_r
Body::Rigid body_toes_r(MassProperties(0.0975,Vec3(0.0307,-0.0026,0.0105),Inertia(0.000111408,0.000357642,0.000386552,7.78245e-006,-3.14291e-005,2.66175e-006)));
// Joint name is toes_r
MobilizedBody::Pin mobod_toes_r(mobod_foot_r,Vec3(0.1768,-0.002,0.00108), body_toes_r,Vec3(0));
// END BODY toes_r

// BODY femur_l
Body::Rigid body_femur_l(MassProperties(10.5225,Vec3(0,-0.162162,0),Inertia(0.455313,0.036634,0.455313)));
body_femur_l.addDecoration(Vec3(0,-.2,0), DecorativeEllipsoid(Vec3(.08,.2,.1)).setColor(Yellow));
// Joint name is hip_l
MobilizedBody::Ball mobod_femur_l(mobod_pelvis,Vec3(-0.08,-0.0824,-0.0785), body_femur_l,Vec3(0));
markers.addMarker("L.Knee",mobod_femur_l,Vec3(0,-0.404,-0.05),1);
body_femur_l.addDecoration(Vec3(0,-0.404,-0.05),markerDecoration);
markers.addMarker("L.MKnee",mobod_femur_l,Vec3(0,-0.404,0.05),1);
body_femur_l.addDecoration(Vec3(0,-0.404,0.05),markerDecoration);
markers.addMarker("L.TH1",mobod_femur_l,Vec3(0.0179,-0.224,-0.1147),1);
body_femur_l.addDecoration(Vec3(0.0179,-0.224,-0.1147),markerDecoration);
markers.addMarker("L.TH2",mobod_femur_l,Vec3(0.0179,-0.224,-0.1147),1);
body_femur_l.addDecoration(Vec3(0.0179,-0.224,-0.1147),markerDecoration);
markers.addMarker("L.TH3",mobod_femur_l,Vec3(0.0179,-0.264,-0.0647),1);
body_femur_l.addDecoration(Vec3(0.0179,-0.264,-0.0647),markerDecoration);
markers.addMarker("L.TH4",mobod_femur_l,Vec3(0.08,-0.324,-0.0047),1);
body_femur_l.addDecoration(Vec3(0.08,-0.324,-0.0047),markerDecoration);
// END BODY femur_l

// BODY tibia_l
Body::Rigid body_tibia_l(MassProperties(3.2475,Vec3(0,-0.171713,0),Inertia(0.125753,0.005157,0.126985)));
body_tibia_l.addDecoration(Vec3(0,-.2,0), DecorativeEllipsoid(Vec3(.05,.2,.07)).setColor(Yellow));
// Joint name is knee_l
MobilizedBody::Pin mobod_tibia_l(mobod_femur_l,Vec3(0,-.4,0), body_tibia_l,Vec3(0));
markers.addMarker("L.Ankle",mobod_tibia_l,Vec3(-0.005,-0.41,-0.053),1);
body_tibia_l.addDecoration(Vec3(-0.005,-0.41,-0.053),markerDecoration);
markers.addMarker("L.MAnkle",mobod_tibia_l,Vec3(0.006,-0.3888,0.038),1);
body_tibia_l.addDecoration(Vec3(0.006,-0.3888,0.038),markerDecoration);
markers.addMarker("L.SH1",mobod_tibia_l,Vec3(0.0104,-0.2322,-0.0748),1);
body_tibia_l.addDecoration(Vec3(0.0104,-0.2322,-0.0748),markerDecoration);
markers.addMarker("L.SH2",mobod_tibia_l,Vec3(0.0425,-0.3596,0),1);
body_tibia_l.addDecoration(Vec3(0.0425,-0.3596,0),markerDecoration);
markers.addMarker("L.SH3",mobod_tibia_l,Vec3(0.0125,-0.3196,-0.06),1);
body_tibia_l.addDecoration(Vec3(0.0125,-0.3196,-0.06),markerDecoration);
markers.addMarker("L.SH4",mobod_tibia_l,Vec3(0.0125,-0.3196,-0.06),1);
body_tibia_l.addDecoration(Vec3(0.0125,-0.3196,-0.06),markerDecoration);
// END BODY tibia_l

// BODY patella_l
Body::Rigid body_patella_l(MassProperties(0.0975,Vec3(0.0018,0.0264,0),Inertia(7.09536e-005,1.53159e-005,8.32695e-005,-4.6332e-006,0,0)));
// Joint name is tib_pat_l
MobilizedBody::/*FourDOF*/Cartesian mobod_patella_l(mobod_tibia_l,Vec3(0), body_patella_l,Vec3(0));
// END BODY patella_l

// BODY talus_l
Body::Rigid body_talus_l(MassProperties(0.045,Vec3(0.0055,0.0023,0),Inertia(3.23805e-006,2.36125e-006,4.5993e-006,-5.6925e-007,0,0)));
// Joint name is ankle_l
MobilizedBody::Pin mobod_talus_l(mobod_tibia_l,Vec3(0,-0.426,0), body_talus_l,Vec3(0));
// END BODY talus_l

// BODY foot_l
Body::Rigid body_foot_l(MassProperties(0.885,Vec3(0.101865,0.0156,0),Inertia(0.00112437,0.0114972,0.0120816,-0.00140635,0,0)));
// Joint name is subtalar_l
MobilizedBody::Pin mobod_foot_l(mobod_talus_l,Vec3(-0.04877,-0.04195,-0.00792), body_foot_l,Vec3(0));
markers.addMarker("L.Toe",mobod_foot_l,Vec3(0.19,0.018,0),1);
body_foot_l.addDecoration(Vec3(0.19,0.018,0),markerDecoration);
markers.addMarker("L.Heel",mobod_foot_l,Vec3(-0.03,0.018,0.01),1);
body_foot_l.addDecoration(Vec3(-0.03,0.018,0.01),markerDecoration);
markers.addMarker("L.MT1",mobod_foot_l,Vec3(0.2,0.005,0.045),1);
body_foot_l.addDecoration(Vec3(0.2,0.005,0.045),markerDecoration);
markers.addMarker("L.MT5",mobod_foot_l,Vec3(0.15,0.005,-0.055),1);
body_foot_l.addDecoration(Vec3(0.15,0.005,-0.055),markerDecoration);
// END BODY foot_l

// BODY toes_l
Body::Rigid body_toes_l(MassProperties(0.0975,Vec3(0.0307,-0.0026,-0.0105),Inertia(0.000111408,0.000357642,0.000386552,7.78245e-006,3.14291e-005,-2.66175e-006)));
// Joint name is toes_l
MobilizedBody::Pin mobod_toes_l(mobod_foot_l,Vec3(0.1768,-0.002,-0.00108), body_toes_l,Vec3(0));
// END BODY toes_l

// Here are some constraints you can try. If you have these enabled,
// you should experiment with treating these as constraints or as
// weighted errors merged with the IK objective. See call below for
// ik.setSystemConstraintsWeight(wt) where wt==Infinity is the default
// and handles these as constraints; wt==finite value treats constraint
// satisfaction as part of the objective.

//Constraint::Rod(mobod_tibia_l, mobod_tibia_r, 2*.25);
//Constraint::Rod(mobod_tibia_l, mobod_hand_r, .25);

    matter.setShowDefaultGeometry(false);
    Visualizer viz(system);
    system.addEventReporter(new Visualizer::Reporter(viz, 0.1));


    // Initialize the system and state.
    
    system.realizeTopology();
    State state = system.getDefaultState();

    // Show initial configuration
    viz.report(state);
    State tempState = state; 
    matter.setUseEulerAngles(tempState, true);
    system.realizeModel(tempState);
    system.realize(tempState, Stage::Position);
    cout << "INITIAL CONFIGURATION\n"; 
    cout << tempState.getNU() << " dofs, " 
         << tempState.getNQErr() << " constraints.\n";
    
    cout << "Type any character to continue:\n";
    getchar();


    Assembler ik(system);
    ik.adoptAssemblyGoal(&markers);

    //ik.adoptAssemblyGoal(new QValue(mobod_tibia_r, MobilizerQIndex(0),
    //                     Pi/2), 1);


    // See comment above near the Constraints. If you use this
    // weight the constraints will be violated somewhat but the
    // IK will run much faster.
    //ik.setSystemConstraintsWeight(2);

    // This is the default treatment of constraints.
    //ik.setSystemConstraintsWeight(Infinity);

    ik.lockMobilizer(mobod_patella_l);
    ik.lockMobilizer(mobod_patella_r);
    ik.lockMobilizer(mobod_loadcell1);
    ik.lockMobilizer(mobod_loadcell2);
    //ik.restrictQ(mobod_tibia_r, MobilizerQIndex(0),
    //    -10*Pi/180, 10*Pi/180);

    markers.defineObservationOrder(getNumObservations(), getObservationOrder());
    markers.moveAllObservations(getFrame(0)); // initial observed locations
    state.setTime(getFrameTime(0));
    //ik.setForceNumericalGradient(true);
    //ik.setForceNumericalJacobian(true);

    ik.initialize(state);
    printf("UNASSEMBLED CONFIGURATION (err=%g, cost=%g, qerr=%g)\n",
        ik.calcCurrentErrorNorm(), ik.calcCurrentGoal(),
        max(abs(ik.getInternalState().getQErr())));
    cout << "num freeQs: " << ik.getNumFreeQs() << endl;

    const Real Accuracy = 1e-3;
    ik.setAccuracy(Accuracy);
    ik.assemble(); // solve first frame.
    ik.updateFromInternalState(state);
    viz.report(state);
    printf("ASSEMBLED CONFIGURATION (acc=%g tol=%g err=%g, cost=%g, qerr=%g)\n",
        ik.getAccuracyInUse(), ik.getErrorToleranceInUse(), 
        ik.calcCurrentErrorNorm(), ik.calcCurrentGoal(),
        max(abs(ik.getInternalState().getQErr())));
    printf("# initializations=%d\n", ik.getNumInitializations());
    printf("# assembly steps: %d\n", ik.getNumAssemblySteps());
    printf(" evals: goal=%d grad=%d error=%d jac=%d\n",
        ik.getNumGoalEvals(), ik.getNumGoalGradientEvals(),
        ik.getNumErrorEvals(), ik.getNumErrorJacobianEvals());

    cout << "Type any character to continue:\n";
    getchar();

    const int NSteps = getNumFrames();
    const int NToSkip = 4; // show every nth frame
    const double startReal = realTime(), startCPU = cpuTime();
    for (int f=1; f < NSteps; ++f) {
        markers.moveAllObservations(getFrame(f));
        // update internal state to match new observed locations
        ik.track(getFrameTime(f)); 
        if ((f%NToSkip)==0) {
            ik.updateFromInternalState(state);
            viz.report(state);
        }
    }

    cout << "ASSEMBLED " << NSteps-1 << " steps in " <<
        cpuTime()-startCPU << " CPU s, " << realTime()-startReal << " REAL s\n";

    printf("FINAL CONFIGURATION (acc=%g tol=%g err=%g, cost=%g, qerr=%g)\n",
        ik.getAccuracyInUse(), ik.getErrorToleranceInUse(), 
        ik.calcCurrentErrorNorm(), ik.calcCurrentGoal(),
        max(abs(ik.getInternalState().getQErr())));

    const double oons = Real(1)/ik.getNumAssemblySteps();
    printf("# initializations=%d\n", ik.getNumInitializations());
    printf("# assembly steps: %d\n", ik.getNumAssemblySteps());
    printf("  avg evals: goal=%g grad=%g error=%g jac=%g\n",
        ik.getNumGoalEvals()*oons, ik.getNumGoalGradientEvals()*oons,
        ik.getNumErrorEvals()*oons, ik.getNumErrorJacobianEvals()*oons);


    cout << "DONE ASSEMBLING\n";
    viz.report(state);

    cout << "Type any character to continue:\n";
    getchar();
   
  } catch (const std::exception& e) {
    std::printf("EXCEPTION THROWN: %s\n", e.what());
    exit(1);

  } catch (...) {
    std::printf("UNKNOWN EXCEPTION THROWN\n");
    exit(1);
  }

    return 0;
}

/* Below here is the marker observation data at 200Hz. */

static const int NObservations = 44;
static const int NFrames       = 2105;
static Real frameTimes[NFrames] = {
/*0*/0,0.005,0.01,0.015,0.02,0.025,0.03,0.035,0.04,0.045,0.05,0.055,0.06,0.065,0.07,0.075,0.08,0.085,0.09,0.095,
/*20*/0.1,0.105,0.11,0.115,0.12,0.125,0.13,0.135,0.14,0.145,0.15,0.155,0.16,0.165,0.17,0.175,0.18,0.185,0.19,0.195,
/*40*/0.2,0.205,0.21,0.215,0.22,0.225,0.23,0.235,0.24,0.245,0.25,0.255,0.26,0.265,0.27,0.275,0.28,0.285,0.29,0.295,
/*60*/0.3,0.305,0.31,0.315,0.32,0.325,0.33,0.335,0.34,0.345,0.35,0.355,0.36,0.365,0.37,0.375,0.38,0.385,0.39,0.395,
/*80*/0.4,0.405,0.41,0.415,0.42,0.425,0.43,0.435,0.44,0.445,0.45,0.455,0.46,0.465,0.47,0.475,0.48,0.485,0.49,0.495,
/*100*/0.5,0.505,0.51,0.515,0.52,0.525,0.53,0.535,0.54,0.545,0.55,0.555,0.56,0.565,0.57,0.575,0.58,0.585,0.59,0.595,
/*120*/0.6,0.605,0.61,0.615,0.62,0.625,0.63,0.635,0.64,0.645,0.65,0.655,0.66,0.665,0.67,0.675,0.68,0.685,0.69,0.695,
/*140*/0.7,0.705,0.71,0.715,0.72,0.725,0.73,0.735,0.74,0.745,0.75,0.755,0.76,0.765,0.77,0.775,0.78,0.785,0.79,0.795,
/*160*/0.8,0.805,0.81,0.815,0.82,0.825,0.83,0.835,0.84,0.845,0.85,0.855,0.86,0.865,0.87,0.875,0.88,0.885,0.89,0.895,
/*180*/0.9,0.905,0.91,0.915,0.92,0.925,0.93,0.935,0.94,0.945,0.95,0.955,0.96,0.965,0.97,0.975,0.98,0.985,0.99,0.995,
/*200*/1,1.005,1.01,1.015,1.02,1.025,1.03,1.035,1.04,1.045,1.05,1.055,1.06,1.065,1.07,1.075,1.08,1.085,1.09,1.095,
/*220*/1.1,1.105,1.11,1.115,1.12,1.125,1.13,1.135,1.14,1.145,1.15,1.155,1.16,1.165,1.17,1.175,1.18,1.185,1.19,1.195,
/*240*/1.2,1.205,1.21,1.215,1.22,1.225,1.23,1.235,1.24,1.245,1.25,1.255,1.26,1.265,1.27,1.275,1.28,1.285,1.29,1.295,
/*260*/1.3,1.305,1.31,1.315,1.32,1.325,1.33,1.335,1.34,1.345,1.35,1.355,1.36,1.365,1.37,1.375,1.38,1.385,1.39,1.395,
/*280*/1.4,1.405,1.41,1.415,1.42,1.425,1.43,1.435,1.44,1.445,1.45,1.455,1.46,1.465,1.47,1.475,1.48,1.485,1.49,1.495,
/*300*/1.5,1.505,1.51,1.515,1.52,1.525,1.53,1.535,1.54,1.545,1.55,1.555,1.56,1.565,1.57,1.575,1.58,1.585,1.59,1.595,
/*320*/1.6,1.605,1.61,1.615,1.62,1.625,1.63,1.635,1.64,1.645,1.65,1.655,1.66,1.665,1.67,1.675,1.68,1.685,1.69,1.695,
/*340*/1.7,1.705,1.71,1.715,1.72,1.725,1.73,1.735,1.74,1.745,1.75,1.755,1.76,1.765,1.77,1.775,1.78,1.785,1.79,1.795,
/*360*/1.8,1.805,1.81,1.815,1.82,1.825,1.83,1.835,1.84,1.845,1.85,1.855,1.86,1.865,1.87,1.875,1.88,1.885,1.89,1.895,
/*380*/1.9,1.905,1.91,1.915,1.92,1.925,1.93,1.935,1.94,1.945,1.95,1.955,1.96,1.965,1.97,1.975,1.98,1.985,1.99,1.995,
/*400*/2,2.005,2.01,2.015,2.02,2.025,2.03,2.035,2.04,2.045,2.05,2.055,2.06,2.065,2.07,2.075,2.08,2.085,2.09,2.095,
/*420*/2.1,2.105,2.11,2.115,2.12,2.125,2.13,2.135,2.14,2.145,2.15,2.155,2.16,2.165,2.17,2.175,2.18,2.185,2.19,2.195,
/*440*/2.2,2.205,2.21,2.215,2.22,2.225,2.23,2.235,2.24,2.245,2.25,2.255,2.26,2.265,2.27,2.275,2.28,2.285,2.29,2.295,
/*460*/2.3,2.305,2.31,2.315,2.32,2.325,2.33,2.335,2.34,2.345,2.35,2.355,2.36,2.365,2.37,2.375,2.38,2.385,2.39,2.395,
/*480*/2.4,2.405,2.41,2.415,2.42,2.425,2.43,2.435,2.44,2.445,2.45,2.455,2.46,2.465,2.47,2.475,2.48,2.485,2.49,2.495,
/*500*/2.5,2.505,2.51,2.515,2.52,2.525,2.53,2.535,2.54,2.545,2.55,2.555,2.56,2.565,2.57,2.575,2.58,2.585,2.59,2.595,
/*520*/2.6,2.605,2.61,2.615,2.62,2.625,2.63,2.635,2.64,2.645,2.65,2.655,2.66,2.665,2.67,2.675,2.68,2.685,2.69,2.695,
/*540*/2.7,2.705,2.71,2.715,2.72,2.725,2.73,2.735,2.74,2.745,2.75,2.755,2.76,2.765,2.77,2.775,2.78,2.785,2.79,2.795,
/*560*/2.8,2.805,2.81,2.815,2.82,2.825,2.83,2.835,2.84,2.845,2.85,2.855,2.86,2.865,2.87,2.875,2.88,2.885,2.89,2.895,
/*580*/2.9,2.905,2.91,2.915,2.92,2.925,2.93,2.935,2.94,2.945,2.95,2.955,2.96,2.965,2.97,2.975,2.98,2.985,2.99,2.995,
/*600*/3,3.005,3.01,3.015,3.02,3.025,3.03,3.035,3.04,3.045,3.05,3.055,3.06,3.065,3.07,3.075,3.08,3.085,3.09,3.095,
/*620*/3.1,3.105,3.11,3.115,3.12,3.125,3.13,3.135,3.14,3.145,3.15,3.155,3.16,3.165,3.17,3.175,3.18,3.185,3.19,3.195,
/*640*/3.2,3.205,3.21,3.215,3.22,3.225,3.23,3.235,3.24,3.245,3.25,3.255,3.26,3.265,3.27,3.275,3.28,3.285,3.29,3.295,
/*660*/3.3,3.305,3.31,3.315,3.32,3.325,3.33,3.335,3.34,3.345,3.35,3.355,3.36,3.365,3.37,3.375,3.38,3.385,3.39,3.395,
/*680*/3.4,3.405,3.41,3.415,3.42,3.425,3.43,3.435,3.44,3.445,3.45,3.455,3.46,3.465,3.47,3.475,3.48,3.485,3.49,3.495,
/*700*/3.5,3.505,3.51,3.515,3.52,3.525,3.53,3.535,3.54,3.545,3.55,3.555,3.56,3.565,3.57,3.575,3.58,3.585,3.59,3.595,
/*720*/3.6,3.605,3.61,3.615,3.62,3.625,3.63,3.635,3.64,3.645,3.65,3.655,3.66,3.665,3.67,3.675,3.68,3.685,3.69,3.695,
/*740*/3.7,3.705,3.71,3.715,3.72,3.725,3.73,3.735,3.74,3.745,3.75,3.755,3.76,3.765,3.77,3.775,3.78,3.785,3.79,3.795,
/*760*/3.8,3.805,3.81,3.815,3.82,3.825,3.83,3.835,3.84,3.845,3.85,3.855,3.86,3.865,3.87,3.875,3.88,3.885,3.89,3.895,
/*780*/3.9,3.905,3.91,3.915,3.92,3.925,3.93,3.935,3.94,3.945,3.95,3.955,3.96,3.965,3.97,3.975,3.98,3.985,3.99,3.995,
/*800*/4,4.005,4.01,4.015,4.02,4.025,4.03,4.035,4.04,4.045,4.05,4.055,4.06,4.065,4.07,4.075,4.08,4.085,4.09,4.095,
/*820*/4.1,4.105,4.11,4.115,4.12,4.125,4.13,4.135,4.14,4.145,4.15,4.155,4.16,4.165,4.17,4.175,4.18,4.185,4.19,4.195,
/*840*/4.2,4.205,4.21,4.215,4.22,4.225,4.23,4.235,4.24,4.245,4.25,4.255,4.26,4.265,4.27,4.275,4.28,4.285,4.29,4.295,
/*860*/4.3,4.305,4.31,4.315,4.32,4.325,4.33,4.335,4.34,4.345,4.35,4.355,4.36,4.365,4.37,4.375,4.38,4.385,4.39,4.395,
/*880*/4.4,4.405,4.41,4.415,4.42,4.425,4.43,4.435,4.44,4.445,4.45,4.455,4.46,4.465,4.47,4.475,4.48,4.485,4.49,4.495,
/*900*/4.5,4.505,4.51,4.515,4.52,4.525,4.53,4.535,4.54,4.545,4.55,4.555,4.56,4.565,4.57,4.575,4.58,4.585,4.59,4.595,
/*920*/4.6,4.605,4.61,4.615,4.62,4.625,4.63,4.635,4.64,4.645,4.65,4.655,4.66,4.665,4.67,4.675,4.68,4.685,4.69,4.695,
/*940*/4.7,4.705,4.71,4.715,4.72,4.725,4.73,4.735,4.74,4.745,4.75,4.755,4.76,4.765,4.77,4.775,4.78,4.785,4.79,4.795,
/*960*/4.8,4.805,4.81,4.815,4.82,4.825,4.83,4.835,4.84,4.845,4.85,4.855,4.86,4.865,4.87,4.875,4.88,4.885,4.89,4.895,
/*980*/4.9,4.905,4.91,4.915,4.92,4.925,4.93,4.935,4.94,4.945,4.95,4.955,4.96,4.965,4.97,4.975,4.98,4.985,4.99,4.995,
/*1000*/5,5.005,5.01,5.015,5.02,5.025,5.03,5.035,5.04,5.045,5.05,5.055,5.06,5.065,5.07,5.075,5.08,5.085,5.09,5.095,
/*1020*/5.1,5.105,5.11,5.115,5.12,5.125,5.13,5.135,5.14,5.145,5.15,5.155,5.16,5.165,5.17,5.175,5.18,5.185,5.19,5.195,
/*1040*/5.2,5.205,5.21,5.215,5.22,5.225,5.23,5.235,5.24,5.245,5.25,5.255,5.26,5.265,5.27,5.275,5.28,5.285,5.29,5.295,
/*1060*/5.3,5.305,5.31,5.315,5.32,5.325,5.33,5.335,5.34,5.345,5.35,5.355,5.36,5.365,5.37,5.375,5.38,5.385,5.39,5.395,
/*1080*/5.4,5.405,5.41,5.415,5.42,5.425,5.43,5.435,5.44,5.445,5.45,5.455,5.46,5.465,5.47,5.475,5.48,5.485,5.49,5.495,
/*1100*/5.5,5.505,5.51,5.515,5.52,5.525,5.53,5.535,5.54,5.545,5.55,5.555,5.56,5.565,5.57,5.575,5.58,5.585,5.59,5.595,
/*1120*/5.6,5.605,5.61,5.615,5.62,5.625,5.63,5.635,5.64,5.645,5.65,5.655,5.66,5.665,5.67,5.675,5.68,5.685,5.69,5.695,
/*1140*/5.7,5.705,5.71,5.715,5.72,5.725,5.73,5.735,5.74,5.745,5.75,5.755,5.76,5.765,5.77,5.775,5.78,5.785,5.79,5.795,
/*1160*/5.8,5.805,5.81,5.815,5.82,5.825,5.83,5.835,5.84,5.845,5.85,5.855,5.86,5.865,5.87,5.875,5.88,5.885,5.89,5.895,
/*1180*/5.9,5.905,5.91,5.915,5.92,5.925,5.93,5.935,5.94,5.945,5.95,5.955,5.96,5.965,5.97,5.975,5.98,5.985,5.99,5.995,
/*1200*/6,6.005,6.01,6.015,6.02,6.025,6.03,6.035,6.04,6.045,6.05,6.055,6.06,6.065,6.07,6.075,6.08,6.085,6.09,6.095,
/*1220*/6.1,6.105,6.11,6.115,6.12,6.125,6.13,6.135,6.14,6.145,6.15,6.155,6.16,6.165,6.17,6.175,6.18,6.185,6.19,6.195,
/*1240*/6.2,6.205,6.21,6.215,6.22,6.225,6.23,6.235,6.24,6.245,6.25,6.255,6.26,6.265,6.27,6.275,6.28,6.285,6.29,6.295,
/*1260*/6.3,6.305,6.31,6.315,6.32,6.325,6.33,6.335,6.34,6.345,6.35,6.355,6.36,6.365,6.37,6.375,6.38,6.385,6.39,6.395,
/*1280*/6.4,6.405,6.41,6.415,6.42,6.425,6.43,6.435,6.44,6.445,6.45,6.455,6.46,6.465,6.47,6.475,6.48,6.485,6.49,6.495,
/*1300*/6.5,6.505,6.51,6.515,6.52,6.525,6.53,6.535,6.54,6.545,6.55,6.555,6.56,6.565,6.57,6.575,6.58,6.585,6.59,6.595,
/*1320*/6.6,6.605,6.61,6.615,6.62,6.625,6.63,6.635,6.64,6.645,6.65,6.655,6.66,6.665,6.67,6.675,6.68,6.685,6.69,6.695,
/*1340*/6.7,6.705,6.71,6.715,6.72,6.725,6.73,6.735,6.74,6.745,6.75,6.755,6.76,6.765,6.77,6.775,6.78,6.785,6.79,6.795,
/*1360*/6.8,6.805,6.81,6.815,6.82,6.825,6.83,6.835,6.84,6.845,6.85,6.855,6.86,6.865,6.87,6.875,6.88,6.885,6.89,6.895,
/*1380*/6.9,6.905,6.91,6.915,6.92,6.925,6.93,6.935,6.94,6.945,6.95,6.955,6.96,6.965,6.97,6.975,6.98,6.985,6.99,6.995,
/*1400*/7,7.005,7.01,7.015,7.02,7.025,7.03,7.035,7.04,7.045,7.05,7.055,7.06,7.065,7.07,7.075,7.08,7.085,7.09,7.095,
/*1420*/7.1,7.105,7.11,7.115,7.12,7.125,7.13,7.135,7.14,7.145,7.15,7.155,7.16,7.165,7.17,7.175,7.18,7.185,7.19,7.195,
/*1440*/7.2,7.205,7.21,7.215,7.22,7.225,7.23,7.235,7.24,7.245,7.25,7.255,7.26,7.265,7.27,7.275,7.28,7.285,7.29,7.295,
/*1460*/7.3,7.305,7.31,7.315,7.32,7.325,7.33,7.335,7.34,7.345,7.35,7.355,7.36,7.365,7.37,7.375,7.38,7.385,7.39,7.395,
/*1480*/7.4,7.405,7.41,7.415,7.42,7.425,7.43,7.435,7.44,7.445,7.45,7.455,7.46,7.465,7.47,7.475,7.48,7.485,7.49,7.495,
/*1500*/7.5,7.505,7.51,7.515,7.52,7.525,7.53,7.535,7.54,7.545,7.55,7.555,7.56,7.565,7.57,7.575,7.58,7.585,7.59,7.595,
/*1520*/7.6,7.605,7.61,7.615,7.62,7.625,7.63,7.635,7.64,7.645,7.65,7.655,7.66,7.665,7.67,7.675,7.68,7.685,7.69,7.695,
/*1540*/7.7,7.705,7.71,7.715,7.72,7.725,7.73,7.735,7.74,7.745,7.75,7.755,7.76,7.765,7.77,7.775,7.78,7.785,7.79,7.795,
/*1560*/7.8,7.805,7.81,7.815,7.82,7.825,7.83,7.835,7.84,7.845,7.85,7.855,7.86,7.865,7.87,7.875,7.88,7.885,7.89,7.895,
/*1580*/7.9,7.905,7.91,7.915,7.92,7.925,7.93,7.935,7.94,7.945,7.95,7.955,7.96,7.965,7.97,7.975,7.98,7.985,7.99,7.995,
/*1600*/8,8.005,8.01,8.015,8.02,8.025,8.03,8.035,8.04,8.045,8.05,8.055,8.06,8.065,8.07,8.075,8.08,8.085,8.09,8.095,
/*1620*/8.1,8.105,8.11,8.115,8.12,8.125,8.13,8.135,8.14,8.145,8.15,8.155,8.16,8.165,8.17,8.175,8.18,8.185,8.19,8.195,
/*1640*/8.2,8.205,8.21,8.215,8.22,8.225,8.23,8.235,8.24,8.245,8.25,8.255,8.26,8.265,8.27,8.275,8.28,8.285,8.29,8.295,
/*1660*/8.3,8.305,8.31,8.315,8.32,8.325,8.33,8.335,8.34,8.345,8.35,8.355,8.36,8.365,8.37,8.375,8.38,8.385,8.39,8.395,
/*1680*/8.4,8.405,8.41,8.415,8.42,8.425,8.43,8.435,8.44,8.445,8.45,8.455,8.46,8.465,8.47,8.475,8.48,8.485,8.49,8.495,
/*1700*/8.5,8.505,8.51,8.515,8.52,8.525,8.53,8.535,8.54,8.545,8.55,8.555,8.56,8.565,8.57,8.575,8.58,8.585,8.59,8.595,
/*1720*/8.6,8.605,8.61,8.615,8.62,8.625,8.63,8.635,8.64,8.645,8.65,8.655,8.66,8.665,8.67,8.675,8.68,8.685,8.69,8.695,
/*1740*/8.7,8.705,8.71,8.715,8.72,8.725,8.73,8.735,8.74,8.745,8.75,8.755,8.76,8.765,8.77,8.775,8.78,8.785,8.79,8.795,
/*1760*/8.8,8.805,8.81,8.815,8.82,8.825,8.83,8.835,8.84,8.845,8.85,8.855,8.86,8.865,8.87,8.875,8.88,8.885,8.89,8.895,
/*1780*/8.9,8.905,8.91,8.915,8.92,8.925,8.93,8.935,8.94,8.945,8.95,8.955,8.96,8.965,8.97,8.975,8.98,8.985,8.99,8.995,
/*1800*/9,9.005,9.01,9.015,9.02,9.025,9.03,9.035,9.04,9.045,9.05,9.055,9.06,9.065,9.07,9.075,9.08,9.085,9.09,9.095,
/*1820*/9.1,9.105,9.11,9.115,9.12,9.125,9.13,9.135,9.14,9.145,9.15,9.155,9.16,9.165,9.17,9.175,9.18,9.185,9.19,9.195,
/*1840*/9.2,9.205,9.21,9.215,9.22,9.225,9.23,9.235,9.24,9.245,9.25,9.255,9.26,9.265,9.27,9.275,9.28,9.285,9.29,9.295,
/*1860*/9.3,9.305,9.31,9.315,9.32,9.325,9.33,9.335,9.34,9.345,9.35,9.355,9.36,9.365,9.37,9.375,9.38,9.385,9.39,9.395,
/*1880*/9.4,9.405,9.41,9.415,9.42,9.425,9.43,9.435,9.44,9.445,9.45,9.455,9.46,9.465,9.47,9.475,9.48,9.485,9.49,9.495,
/*1900*/9.5,9.505,9.51,9.515,9.52,9.525,9.53,9.535,9.54,9.545,9.55,9.555,9.56,9.565,9.57,9.575,9.58,9.585,9.59,9.595,
/*1920*/9.6,9.605,9.61,9.615,9.62,9.625,9.63,9.635,9.64,9.645,9.65,9.655,9.66,9.665,9.67,9.675,9.68,9.685,9.69,9.695,
/*1940*/9.7,9.705,9.71,9.715,9.72,9.725,9.73,9.735,9.74,9.745,9.75,9.755,9.76,9.765,9.77,9.775,9.78,9.785,9.79,9.795,
/*1960*/9.8,9.805,9.81,9.815,9.82,9.825,9.83,9.835,9.84,9.845,9.85,9.855,9.86,9.865,9.87,9.875,9.88,9.885,9.89,9.895,
/*1980*/9.9,9.905,9.91,9.915,9.92,9.925,9.93,9.935,9.94,9.945,9.95,9.955,9.96,9.965,9.97,9.975,9.98,9.985,9.99,9.995,
/*2000*/10,10.005,10.01,10.015,10.02,10.025,10.03,10.035,10.04,10.045,10.05,10.055,10.06,10.065,10.07,10.075,10.08,10.085,10.09,10.095,
/*2020*/10.1,10.105,10.11,10.115,10.12,10.125,10.13,10.135,10.14,10.145,10.15,10.155,10.16,10.165,10.17,10.175,10.18,10.185,10.19,10.195,
/*2040*/10.2,10.205,10.21,10.215,10.22,10.225,10.23,10.235,10.24,10.245,10.25,10.255,10.26,10.265,10.27,10.275,10.28,10.285,10.29,10.295,
/*2060*/10.3,10.305,10.31,10.315,10.32,10.325,10.33,10.335,10.34,10.345,10.35,10.355,10.36,10.365,10.37,10.375,10.38,10.385,10.39,10.395,
/*2080*/10.4,10.405,10.41,10.415,10.42,10.425,10.43,10.435,10.44,10.445,10.45,10.455,10.46,10.465,10.47,10.475,10.48,10.485,10.49,10.495,
/*2100*/10.5,10.505,10.51,10.515,10.52
};
static const char* observationOrder[NObservations] = {
/*0*/"R.Shoulder","R.Clavicle","R.Bicep","R.Elbow","R.Forearm","R.Wrist","L.Shoulder","L.Clavicle","L.Scapula","L.Bicep",
/*10*/"L.Elbow","L.Forearm","L.Wrist","R.ASIS","R.PSIS","S2","L.PSIS","L.ASIS","R.TH1","R.TH2",
/*20*/"R.TH3","R.Knee","R.SH1","R.SH2","R.SH3","R.SH4","R.Ankle","R.Heel","R.MT1","R.MT5",
/*30*/"L.TH1","L.TH2","L.TH3","L.TH4","L.Knee","L.SH1","L.SH2","L.SH3","L.Ankle","L.Heel",
/*40*/"L.MT1","L.MT5","R_HJC","L_HJC"
};
static double observations[NFrames][3*NObservations] = {
/*0*/{0.090174,1.88432,-1.71703,0.183195,1.8414,-1.85736,-0.03272,1.75705,-1.63169,-0.124639,1.65522,-1.58515,0.018105,1.57584,-1.61167,0.095363,1.52367,-1.612,0.127943,1.92047,-2.10315,0.194393,1.84906,-1.93091,-0.009998,1.87655,-2.0326,-0.014027,1.71457,-2.22251,0.035047,1.62438,-2.26588,0.185972,1.57007,-2.14576,0.244882,1.53837,-2.14,0.108311,1.41565,-1.81786,-0.080276,1.47065,-1.89218,-0.075026,1.49623,-1.95279,-0.067967,1.46745,-2.00846,0.123187,1.40594,-2.07196,0.170297,1.12647,-1.75035,0.048508,1.07145,-1.78442,0.083181,0.995259,-1.80496,0.189256,0.912886,-1.78579,0.237219,0.726513,-1.77991,0.248158,0.652262,-1.79449,0.127715,0.709792,-1.80196,0.140083,0.639374,-1.81181,0.223738,0.423112,-1.8286,0.145578,0.35512,-1.91536,0.389378,0.414344,-1.85826,0.369371,0.347959,-1.79518,0.085408,1.08672,-2.11457,0.08914,1.00371,-2.09678,-0.054983,1.0813,-2.08294,-0.049952,1.00025,-2.06546,-0.021551,0.872819,-2.06933,-0.19154,0.812236,-2.06037,-0.193648,0.925116,-2.07387,-0.264475,0.919419,-2.06751,-0.487294,0.881914,-2.0409,-0.513099,0.983557,-1.98142,-0.618697,0.72077,-1.98067,-0.647563,0.774091,-2.03551,0.031863,1.32665,-1.82976,0.038375,1.31961,-2.04478},
/*1*/{0.091297,1.87806,-1.71564,0.182311,1.83703,-1.85764,-0.028513,1.74764,-1.62789,-0.118465,1.64442,-1.57894,0.025159,1.56849,-1.60827,0.103256,1.51788,-1.61005,0.126087,1.91504,-2.10347,0.193485,1.84336,-1.93154,-0.010839,1.87079,-2.03188,-0.020112,1.71092,-2.22225,0.028199,1.61943,-2.26571,0.17971,1.5599,-2.14963,0.237434,1.52621,-2.14531,0.107126,1.41214,-1.81599,-0.081285,1.46489,-1.8934,-0.07621,1.49056,-1.95294,-0.068882,1.46171,-2.00913,0.121952,1.401,-2.07248,0.168246,1.12131,-1.75028,0.046138,1.06706,-1.78509,0.080908,0.990501,-1.80488,0.184639,0.90639,-1.78596,0.229278,0.72023,-1.78174,0.23953,0.64616,-1.79634,0.118966,0.705653,-1.80378,0.128657,0.634755,-1.81369,0.209121,0.419185,-1.83021,0.123655,0.353672,-1.91521,0.373979,0.404006,-1.86016,0.350952,0.340743,-1.79787,0.090933,1.08275,-2.11412,0.096081,0.999049,-2.09626,-0.049201,1.07404,-2.08249,-0.042046,0.994004,-2.06538,-0.010775,0.867443,-2.06977,-0.180869,0.811455,-2.06146,-0.181582,0.924533,-2.07482,-0.251431,0.919956,-2.06733,-0.478855,0.890919,-2.04362,-0.500405,0.989962,-1.98289,-0.609456,0.727974,-1.98331,-0.642838,0.789095,-2.04123,0.031282,1.32239,-1.82912,0.03836,1.31418,-2.04408},
/*2*/{0.091335,1.87169,-1.71479,0.183083,1.82953,-1.85712,-0.023951,1.7383,-1.62404,-0.11091,1.63499,-1.57333,0.032871,1.5612,-1.60512,0.110852,1.51258,-1.60793,0.124791,1.90938,-2.10431,0.193996,1.83813,-1.93194,-0.012086,1.86576,-2.03137,-0.024299,1.70718,-2.2217,0.020478,1.61543,-2.26539,0.173088,1.54986,-2.15399,0.230455,1.51413,-2.15068,0.104805,1.40744,-1.81493,-0.083976,1.45985,-1.89355,-0.0781,1.48507,-1.9529,-0.069696,1.45639,-2.01053,0.121724,1.39661,-2.07049,0.164122,1.11498,-1.74983,0.042093,1.06248,-1.78534,0.077625,0.98602,-1.80517,0.178142,0.900235,-1.78648,0.221834,0.713953,-1.78405,0.22966,0.640235,-1.79881,0.110812,0.703157,-1.80525,0.119605,0.632032,-1.81396,0.191836,0.419832,-1.83335,0.10278,0.355621,-1.91511,0.356322,0.395949,-1.86564,0.331367,0.340436,-1.79931,0.096225,1.07895,-2.11492,0.103221,0.995321,-2.09644,-0.043438,1.0675,-2.08231,-0.034777,0.987421,-2.06562,0.000839,0.862208,-2.07096,-0.169381,0.810094,-2.062,-0.167271,0.922114,-2.0754,-0.237948,0.920033,-2.06772,-0.466303,0.896161,-2.04443,-0.488694,0.995061,-1.98287,-0.597545,0.733429,-1.98694,-0.629991,0.794214,-2.04519,0.029047,1.31749,-1.82856,0.03832,1.30941,-2.04344},
/*3*/{0.092087,1.86568,-1.71333,0.182831,1.82511,-1.85789,-0.01871,1.72901,-1.62038,-0.103618,1.62466,-1.56734,0.041183,1.55492,-1.60203,0.119592,1.50664,-1.60573,0.124153,1.90441,-2.10496,0.193549,1.83257,-1.9327,-0.01286,1.86058,-2.03072,-0.030531,1.70266,-2.22109,0.012698,1.61085,-2.26538,0.166654,1.53936,-2.15824,0.222259,1.50243,-2.15566,0.102622,1.40302,-1.81331,-0.086237,1.455,-1.89456,-0.07984,1.47883,-1.95377,-0.072053,1.45063,-2.01137,0.121167,1.39182,-2.06906,0.161441,1.11035,-1.74795,0.038014,1.05788,-1.78564,0.073335,0.982563,-1.80681,0.171976,0.894191,-1.78659,0.213772,0.710666,-1.78572,0.220346,0.637231,-1.80096,0.103499,0.703459,-1.80564,0.108837,0.631532,-1.81537,0.173975,0.420755,-1.83629,0.080526,0.355486,-1.91597,0.334785,0.393686,-1.86957,0.308578,0.341522,-1.80107,0.100994,1.0748,-2.11529,0.110248,0.99185,-2.0974,-0.038583,1.0601,-2.08139,-0.027802,0.980256,-2.06559,0.011933,0.856059,-2.07199,-0.158015,0.808797,-2.06279,-0.153527,0.920215,-2.07636,-0.223751,0.919911,-2.06808,-0.452295,0.900178,-2.04574,-0.47589,0.999844,-1.98461,-0.583499,0.737196,-1.99084,-0.616894,0.798704,-2.04833,0.02694,1.31293,-1.82776,0.038016,1.30404,-2.04252},
/*4*/{0.093353,1.85929,-1.71192,0.183228,1.81831,-1.85746,-0.012746,1.7196,-1.61667,-0.095804,1.61451,-1.56114,0.049678,1.54751,-1.59868,0.128215,1.5024,-1.60385,0.123235,1.89968,-2.10598,0.19428,1.82751,-1.93354,-0.01348,1.85534,-2.02912,-0.03415,1.69845,-2.22057,0.004914,1.60661,-2.26578,0.159918,1.53025,-2.16303,0.213638,1.49,-2.16112,0.098635,1.39922,-1.81269,-0.089307,1.45081,-1.89675,-0.082555,1.47378,-1.95558,-0.074797,1.44483,-2.01252,0.122134,1.38754,-2.06714,0.153185,1.10725,-1.74555,0.034798,1.05586,-1.78504,0.06885,0.980545,-1.80828,0.165422,0.887157,-1.78753,0.209076,0.710633,-1.7865,0.212711,0.635973,-1.80211,0.097576,0.706842,-1.80572,0.100599,0.635444,-1.81662,0.152744,0.421808,-1.83836,0.058849,0.356964,-1.91696,0.312235,0.391388,-1.87279,0.286349,0.34185,-1.80251,0.106219,1.07175,-2.11511,0.117692,0.988541,-2.09801,-0.033711,1.05294,-2.08134,-0.019602,0.973584,-2.06483,0.022909,0.85025,-2.07263,-0.145615,0.806616,-2.06285,-0.138032,0.917618,-2.07615,-0.209909,0.919131,-2.06918,-0.438381,0.903351,-2.04703,-0.462152,1.00206,-1.9863,-0.570126,0.739626,-1.99337,-0.601632,0.801662,-2.05235,0.023598,1.30901,-1.82732,0.037989,1.29931,-2.04186},
/*5*/{0.094187,1.85462,-1.71052,0.1833,1.81388,-1.85825,-0.005894,1.71058,-1.61324,-0.085588,1.60545,-1.55552,0.059256,1.54157,-1.59556,0.137858,1.49781,-1.60231,0.122382,1.89488,-2.10714,0.194885,1.8218,-1.93386,-0.01458,1.85216,-2.0276,-0.039699,1.6948,-2.22017,-0.002082,1.60262,-2.2655,0.152843,1.51995,-2.16763,0.204903,1.47882,-2.16672,0.095466,1.39565,-1.81297,-0.093194,1.44607,-1.89718,-0.086238,1.46806,-1.95603,-0.076539,1.43936,-2.01355,0.120146,1.38273,-2.0666,0.150597,1.10498,-1.74172,0.03075,1.05599,-1.78697,0.066093,0.979874,-1.80947,0.160098,0.884492,-1.78824,0.204794,0.709803,-1.78745,0.204792,0.635777,-1.80302,0.093495,0.711775,-1.80495,0.093403,0.640444,-1.81595,0.134256,0.424017,-1.84014,0.035955,0.358335,-1.91768,0.291155,0.391583,-1.87439,0.26462,0.340916,-1.80316,0.110673,1.06845,-2.11586,0.124507,0.98618,-2.09821,-0.028286,1.04648,-2.08203,-0.012102,0.966787,-2.06454,0.033904,0.84432,-2.07211,-0.132945,0.806003,-2.06418,-0.124792,0.916087,-2.07658,-0.195242,0.918672,-2.07047,-0.41867,0.902701,-2.04587,-0.449693,1.00489,-1.98884,-0.554652,0.741414,-1.99687,-0.586195,0.802536,-2.05568,0.02091,1.3049,-1.82694,0.03631,1.29427,-2.04136},
/*6*/{0.095729,1.84966,-1.70919,0.184067,1.80892,-1.85878,0.001161,1.7015,-1.61103,-0.074648,1.5961,-1.55046,0.068514,1.53624,-1.59393,0.148058,1.49411,-1.60094,0.122291,1.8912,-2.10838,0.19469,1.81667,-1.93506,-0.015233,1.84998,-2.02662,-0.042729,1.69173,-2.22001,-0.009105,1.59889,-2.26398,0.144665,1.50931,-2.17191,0.195652,1.46696,-2.17202,0.091803,1.39296,-1.81113,-0.096951,1.44181,-1.89835,-0.089504,1.46355,-1.95797,-0.080636,1.4341,-2.01495,0.118084,1.37838,-2.0653,0.148764,1.1036,-1.7394,0.029243,1.05442,-1.78868,0.063905,0.979099,-1.80961,0.160076,0.88797,-1.78603,0.201104,0.709148,-1.78875,0.194944,0.634878,-1.80384,0.087633,0.715615,-1.80386,0.083845,0.645144,-1.81314,0.11387,0.424562,-1.84051,0.014128,0.35929,-1.91853,0.270946,0.390148,-1.87131,0.242574,0.340396,-1.80197,0.114808,1.06723,-2.11572,0.13163,0.984854,-2.09807,-0.022454,1.04011,-2.08147,-0.004035,0.96123,-2.06457,0.046309,0.840123,-2.07138,-0.118786,0.802535,-2.06368,-0.109887,0.913992,-2.07787,-0.180123,0.916215,-2.07074,-0.40905,0.906165,-2.05085,-0.436363,1.00486,-1.98928,-0.539061,0.740428,-2.00003,-0.569873,0.800947,-2.05886,0.01718,1.30196,-1.82588,0.034505,1.2895,-2.04006},
/*7*/{0.096568,1.84563,-1.70836,0.184708,1.80316,-1.85888,0.009135,1.6945,-1.61045,-0.065657,1.58736,-1.54593,0.07816,1.53185,-1.5927,0.159315,1.49178,-1.59935,0.121274,1.88765,-2.10932,0.195721,1.81172,-1.93587,-0.015837,1.84837,-2.02445,-0.046414,1.69033,-2.2188,-0.015707,1.59644,-2.26315,0.137288,1.50196,-2.17557,0.186289,1.45751,-2.17685,0.088901,1.39019,-1.81098,-0.101482,1.43683,-1.90087,-0.092025,1.4587,-1.95859,-0.08381,1.42908,-2.01586,0.116761,1.37474,-2.06398,0.144709,1.10356,-1.73851,0.02777,1.0539,-1.78807,0.062803,0.978161,-1.81016,0.163906,0.892783,-1.7853,0.19275,0.708128,-1.78902,0.184225,0.633192,-1.80449,0.079781,0.718722,-1.80026,0.074107,0.648974,-1.81065,0.094102,0.425354,-1.84075,-0.007014,0.360167,-1.91856,0.249896,0.389082,-1.87299,0.221352,0.339077,-1.80146,0.120243,1.06739,-2.1153,0.139397,0.98561,-2.09812,-0.017235,1.0346,-2.08133,0.003723,0.956926,-2.06442,0.059331,0.836946,-2.07049,-0.104357,0.800104,-2.06247,-0.095018,0.912091,-2.07841,-0.165251,0.914086,-2.07169,-0.393661,0.906129,-2.05161,-0.423415,1.00467,-1.99254,-0.522383,0.738633,-2.00217,-0.552303,0.798661,-2.06183,0.014425,1.29851,-1.82566,0.03359,1.28531,-2.03964},
/*8*/{0.097382,1.84288,-1.70801,0.184091,1.80043,-1.86026,0.015423,1.69006,-1.6092,-0.054602,1.58161,-1.54312,0.088364,1.52854,-1.59238,0.169979,1.49015,-1.59909,0.121337,1.88439,-2.10937,0.195026,1.80842,-1.93745,-0.015983,1.84802,-2.02398,-0.047971,1.69093,-2.21671,-0.021898,1.59518,-2.26112,0.128561,1.49452,-2.17808,0.175944,1.44767,-2.18203,0.085652,1.38856,-1.80907,-0.104585,1.43312,-1.90214,-0.094421,1.45446,-1.95905,-0.087138,1.42308,-2.01735,0.113806,1.37061,-2.06345,0.14252,1.10342,-1.73903,0.024783,1.05098,-1.78757,0.062472,0.976386,-1.80868,0.168916,0.896616,-1.78442,0.183334,0.706387,-1.78919,0.174518,0.634081,-1.80492,0.072029,0.720934,-1.79875,0.063409,0.650494,-1.8079,0.07575,0.426351,-1.84057,-0.028344,0.361033,-1.91797,0.228097,0.387174,-1.87187,0.199481,0.33897,-1.80052,0.124636,1.06824,-2.11478,0.147351,0.986516,-2.09722,-0.011155,1.03111,-2.08167,0.012436,0.954123,-2.06475,0.074001,0.836215,-2.06862,-0.090121,0.798358,-2.06179,-0.080444,0.909894,-2.07958,-0.149841,0.912341,-2.07244,-0.378784,0.903847,-2.05347,-0.411234,1.00312,-1.99493,-0.504439,0.735094,-2.00381,-0.536547,0.794336,-2.06421,0.011663,1.29636,-1.82417,0.031795,1.28054,-2.03788},
/*9*/{0.097987,1.83984,-1.70815,0.184473,1.7964,-1.86062,0.020833,1.68652,-1.60865,-0.044085,1.57764,-1.54053,0.099375,1.52665,-1.5917,0.180819,1.48987,-1.59872,0.121746,1.88194,-2.10948,0.19461,1.80464,-1.93839,-0.01577,1.84717,-2.02358,-0.052498,1.68994,-2.21369,-0.028512,1.59583,-2.25865,0.119527,1.4878,-2.18008,0.165742,1.4401,-2.18675,0.082546,1.38708,-1.80751,-0.106813,1.43073,-1.90409,-0.096968,1.45228,-1.95945,-0.090871,1.4191,-2.01805,0.111328,1.36727,-2.06348,0.138474,1.10209,-1.7387,0.021505,1.04909,-1.78791,0.059398,0.973442,-1.80819,0.173942,0.897235,-1.78466,0.173597,0.703433,-1.78778,0.160699,0.630316,-1.80373,0.062011,0.721978,-1.79712,0.051099,0.651581,-1.80801,0.055428,0.426501,-1.84,-0.049493,0.361903,-1.91804,0.20714,0.386743,-1.87261,0.177029,0.338831,-1.80038,0.131126,1.06886,-2.11478,0.155054,0.98885,-2.0966,-0.005301,1.02814,-2.08265,0.021412,0.951649,-2.06569,0.089374,0.836308,-2.06755,-0.074404,0.79764,-2.06037,-0.064627,0.90769,-2.08072,-0.134639,0.910776,-2.07429,-0.360617,0.900002,-2.05331,-0.400694,1.00006,-1.99811,-0.485664,0.72979,-2.00585,-0.517743,0.787438,-2.06755,0.008543,1.29484,-1.82334,0.029352,1.27712,-2.03683},
/*10*/{0.099501,1.83777,-1.70854,0.183263,1.79551,-1.86103,0.026735,1.68402,-1.60851,-0.03388,1.57463,-1.53843,0.110259,1.52571,-1.59143,0.19291,1.49062,-1.59942,0.122863,1.87987,-2.10904,0.192505,1.80126,-1.93932,-0.015902,1.84631,-2.02332,-0.057763,1.69149,-2.20986,-0.035401,1.59762,-2.25601,0.110867,1.48296,-2.1823,0.154895,1.43367,-2.19119,0.078954,1.38535,-1.80555,-0.108853,1.42859,-1.90512,-0.097829,1.4527,-1.95972,-0.092736,1.41687,-2.01869,0.108127,1.36533,-2.06377,0.13348,1.09958,-1.73781,0.017358,1.04691,-1.78777,0.055343,0.970583,-1.80708,0.173574,0.894387,-1.78565,0.161705,0.700863,-1.78587,0.147193,0.627427,-1.8019,0.050382,0.721548,-1.79643,0.038416,0.651994,-1.80485,0.035774,0.426238,-1.83879,-0.070713,0.362829,-1.91694,0.185499,0.386096,-1.87198,0.155991,0.33794,-1.80004,0.137909,1.07158,-2.1144,0.164622,0.991515,-2.09681,0.002483,1.02908,-2.08572,0.031297,0.950265,-2.06708,0.105042,0.838392,-2.06729,-0.059321,0.796609,-2.05912,-0.048822,0.905536,-2.08197,-0.119293,0.908594,-2.07479,-0.349659,0.898372,-2.05632,-0.389949,0.995255,-2.00057,-0.465974,0.722773,-2.00768,-0.498272,0.779223,-2.06995,0.004972,1.29324,-1.82319,0.026732,1.27528,-2.03657},
/*11*/{0.101487,1.8358,-1.70907,0.182986,1.7929,-1.86111,0.031669,1.6824,-1.60778,-0.023637,1.57318,-1.53751,0.120925,1.5257,-1.59127,0.204399,1.49264,-1.60057,0.123425,1.87808,-2.10816,0.192394,1.79888,-1.93926,-0.015896,1.84573,-2.0232,-0.062844,1.69391,-2.20634,-0.04217,1.60022,-2.25316,0.101822,1.48029,-2.18444,0.143903,1.42839,-2.19461,0.075152,1.38256,-1.79922,-0.110244,1.42756,-1.90563,-0.099716,1.45247,-1.95973,-0.094397,1.41592,-2.01929,0.106355,1.36368,-2.06448,0.129567,1.09721,-1.73785,0.012184,1.04497,-1.78725,0.048542,0.96872,-1.80542,0.168755,0.889643,-1.78452,0.148319,0.697053,-1.78435,0.132092,0.624062,-1.80074,0.038007,0.720783,-1.79528,0.023283,0.651359,-1.80727,0.014797,0.426486,-1.83804,-0.092231,0.363468,-1.91652,0.163956,0.385215,-1.87073,0.134283,0.336997,-1.79815,0.145614,1.07435,-2.11391,0.174269,0.995642,-2.09541,0.011479,1.02795,-2.08771,0.042655,0.949288,-2.06778,0.121956,0.840869,-2.06613,-0.042545,0.79326,-2.05725,-0.034204,0.903832,-2.0827,-0.103679,0.904811,-2.075,-0.335715,0.894646,-2.05756,-0.380483,0.990471,-2.00359,-0.44557,0.715379,-2.00935,-0.478964,0.769313,-2.07265,0.000525,1.29115,-1.8225,0.024539,1.27394,-2.0357},
/*12*/{0.103362,1.83464,-1.70961,0.182467,1.79045,-1.86165,0.037151,1.6807,-1.60707,-0.014949,1.57092,-1.5353,0.131625,1.52706,-1.59167,0.215737,1.49589,-1.60199,0.123831,1.87653,-2.10792,0.192009,1.79709,-1.93875,-0.016419,1.84547,-2.02338,-0.066142,1.69742,-2.20094,-0.049691,1.60308,-2.25131,0.093253,1.4789,-2.18668,0.133403,1.42539,-2.19744,0.073299,1.38255,-1.80138,-0.112103,1.42768,-1.9061,-0.101243,1.45188,-1.96005,-0.096407,1.41598,-2.01977,0.105074,1.36325,-2.06479,0.124756,1.09389,-1.73702,0.005051,1.04312,-1.78808,0.043154,0.965414,-1.80644,0.159249,0.884342,-1.78375,0.135154,0.692392,-1.78369,0.116676,0.620474,-1.79985,0.025245,0.720562,-1.79481,0.00873,0.650498,-1.80592,-0.006521,0.425876,-1.8366,-0.113708,0.364329,-1.9155,0.143868,0.384903,-1.87002,0.113297,0.337146,-1.79752,0.153402,1.07759,-2.11387,0.185742,1.0004,-2.09496,0.021059,1.02609,-2.08812,0.053341,0.951931,-2.07088,0.137867,0.843681,-2.06548,-0.026552,0.791735,-2.05673,-0.019346,0.902296,-2.08251,-0.089002,0.903176,-2.07565,-0.321451,0.888203,-2.05964,-0.370477,0.98369,-2.00806,-0.424496,0.705794,-2.01027,-0.458898,0.757362,-2.07491,-0.001043,1.29106,-1.8232,0.022701,1.2737,-2.03642},
/*13*/{0.104943,1.83409,-1.71048,0.182701,1.78677,-1.86125,0.041993,1.68032,-1.60688,-0.005174,1.57048,-1.53451,0.141148,1.52916,-1.59185,0.226856,1.50054,-1.60342,0.123712,1.87524,-2.10741,0.191799,1.79547,-1.93847,-0.01659,1.84528,-2.0233,-0.071916,1.70148,-2.19645,-0.05665,1.60695,-2.24916,0.083929,1.47758,-2.1882,0.122506,1.42404,-2.20003,0.070529,1.38132,-1.79998,-0.112486,1.42758,-1.90506,-0.102572,1.45113,-1.96001,-0.097435,1.41649,-2.0189,0.102265,1.36294,-2.06504,0.11833,1.09041,-1.73684,-0.000861,1.04209,-1.78756,0.03493,0.964454,-1.80629,0.147724,0.879793,-1.7828,0.120096,0.688922,-1.78265,0.101497,0.617055,-1.79928,0.011243,0.71921,-1.79427,-0.007605,0.649462,-1.80607,-0.027067,0.42709,-1.83601,-0.134849,0.36502,-1.91473,0.121806,0.384729,-1.86874,0.091909,0.336956,-1.79589,0.161538,1.08292,-2.11333,0.196167,1.00577,-2.09441,0.031199,1.02518,-2.08911,0.066355,0.953402,-2.07122,0.153865,0.847517,-2.06417,-0.010347,0.791629,-2.05524,-0.004361,0.902025,-2.08253,-0.074873,0.901733,-2.07646,-0.306817,0.880973,-2.06078,-0.361325,0.97482,-2.01042,-0.400828,0.695444,-2.00972,-0.438408,0.744792,-2.07717,-0.003359,1.29006,-1.82303,0.020387,1.27341,-2.0363},
/*14*/{0.107849,1.83391,-1.71058,0.182287,1.78744,-1.86306,0.046045,1.68062,-1.60618,0.002846,1.56922,-1.53304,0.150423,1.53283,-1.59281,0.23755,1.5059,-1.60557,0.1232,1.87454,-2.10766,0.191769,1.79496,-1.93837,-0.016768,1.8453,-2.02336,-0.076225,1.70681,-2.19175,-0.064397,1.61111,-2.24767,0.075621,1.47875,-2.19004,0.112063,1.4233,-2.2022,0.069836,1.38069,-1.79928,-0.113928,1.42688,-1.90501,-0.104225,1.45003,-1.95981,-0.0991,1.41704,-2.01811,0.100798,1.36339,-2.06585,0.112596,1.08634,-1.73722,-0.006814,1.04166,-1.78802,0.024897,0.962791,-1.80639,0.134204,0.874453,-1.7818,0.104745,0.684987,-1.78193,0.082262,0.613265,-1.79874,-0.002751,0.717611,-1.79473,-0.023167,0.649784,-1.80646,-0.046528,0.426824,-1.83489,-0.1558,0.365811,-1.91382,0.100195,0.383661,-1.86756,0.070382,0.336255,-1.79509,0.169015,1.08858,-2.11345,0.206312,1.01285,-2.09377,0.040626,1.02574,-2.09027,0.078659,0.955137,-2.07274,0.169469,0.852336,-2.06415,0.005813,0.791,-2.05393,0.009078,0.902482,-2.08359,-0.061159,0.900695,-2.07799,-0.287551,0.871361,-2.05954,-0.354402,0.966172,-2.01294,-0.378429,0.683744,-2.01238,-0.416321,0.73121,-2.07892,-0.004138,1.28928,-1.82343,0.019208,1.27366,-2.03682},
/*15*/{0.109227,1.83504,-1.71077,0.182131,1.78696,-1.86361,0.04963,1.68136,-1.60603,0.010533,1.56969,-1.5326,0.158976,1.53636,-1.59377,0.246765,1.51184,-1.60834,0.121418,1.8745,-2.10768,0.192039,1.7948,-1.93858,-0.017548,1.84555,-2.02326,-0.081668,1.71189,-2.18845,-0.072145,1.61571,-2.24605,0.066569,1.47985,-2.19133,0.101843,1.42411,-2.20445,0.067032,1.37915,-1.79888,-0.114367,1.42695,-1.90534,-0.104346,1.45001,-1.9597,-0.100795,1.41745,-2.01693,0.099771,1.36446,-2.06675,0.108334,1.083,-1.73711,-0.014363,1.04098,-1.78806,0.015632,0.962128,-1.80821,0.120468,0.868831,-1.78128,0.088644,0.681003,-1.78208,0.066413,0.610049,-1.79832,-0.017672,0.715926,-1.7952,-0.039773,0.647311,-1.80889,-0.067675,0.42705,-1.83376,-0.176835,0.366527,-1.91379,0.07977,0.383223,-1.8669,0.049754,0.33612,-1.79452,0.176577,1.09441,-2.11385,0.2167,1.02089,-2.09347,0.050776,1.02823,-2.09126,0.091378,0.958342,-2.07336,0.183729,0.857205,-2.06408,0.021356,0.791473,-2.05383,0.023047,0.903409,-2.08302,-0.047126,0.899871,-2.07865,-0.27803,0.865979,-2.06276,-0.346324,0.955547,-2.01661,-0.354962,0.672556,-2.01128,-0.394009,0.71767,-2.08112,-0.00593,1.28804,-1.82439,0.017844,1.27435,-2.03787},
/*16*/{0.111101,1.83622,-1.71111,0.18191,1.78688,-1.86387,0.053408,1.68284,-1.6059,0.018321,1.56918,-1.53087,0.167365,1.54098,-1.59476,0.255318,1.5191,-1.61108,0.120246,1.87475,-2.10808,0.192333,1.79512,-1.93898,-0.01888,1.84682,-2.02271,-0.085611,1.71713,-2.18482,-0.08025,1.62045,-2.2449,0.057505,1.48218,-2.19252,0.091191,1.426,-2.20548,0.066866,1.3791,-1.79904,-0.115974,1.42676,-1.90524,-0.105933,1.45026,-1.95883,-0.102081,1.41798,-2.0168,0.097623,1.36561,-2.06801,0.102081,1.07916,-1.73739,-0.020589,1.03987,-1.78671,0.005986,0.959983,-1.80602,0.105574,0.863855,-1.77986,0.072129,0.677516,-1.7821,0.047334,0.60648,-1.79783,-0.033603,0.714524,-1.79613,-0.056453,0.646055,-1.8091,-0.087203,0.426666,-1.8338,-0.197678,0.367166,-1.91297,0.05862,0.383088,-1.86639,0.028728,0.336091,-1.79413,0.183394,1.10154,-2.11355,0.226408,1.02988,-2.09328,0.060324,1.03065,-2.09143,0.102555,0.961667,-2.07393,0.19876,0.862397,-2.06422,0.038441,0.792588,-2.05387,0.036088,0.90386,-2.08349,-0.033649,0.899186,-2.07963,-0.263817,0.856914,-2.06327,-0.33657,0.942517,-2.01826,-0.330375,0.660033,-2.01231,-0.370923,0.70283,-2.08333,-0.006318,1.28789,-1.82512,0.016363,1.27521,-2.03879},
/*17*/{0.113371,1.83728,-1.71145,0.181486,1.78747,-1.86568,0.057349,1.68487,-1.60605,0.025041,1.5706,-1.53083,0.174766,1.54595,-1.59634,0.263928,1.52666,-1.61446,0.11794,1.87545,-2.10836,0.189345,1.79464,-1.9408,-0.020337,1.84755,-2.02149,-0.091509,1.72297,-2.18171,-0.089031,1.62655,-2.24366,0.049119,1.48624,-2.1939,0.081881,1.42961,-2.20676,0.066727,1.37876,-1.79926,-0.116338,1.42702,-1.90454,-0.106372,1.45109,-1.95857,-0.102951,1.41854,-2.0155,0.09716,1.36782,-2.06844,0.094536,1.07653,-1.73791,-0.029724,1.0407,-1.78812,-0.003481,0.95968,-1.80697,0.090458,0.858841,-1.78018,0.056172,0.6746,-1.78266,0.03163,0.604679,-1.79852,-0.048897,0.712517,-1.79693,-0.07259,0.645589,-1.80792,-0.107644,0.428154,-1.83406,-0.218718,0.367867,-1.91323,0.038412,0.383247,-1.8662,0.006633,0.336217,-1.79472,0.190498,1.10973,-2.11224,0.235213,1.03852,-2.09346,0.069355,1.03424,-2.09286,0.113324,0.967101,-2.07449,0.212698,0.869325,-2.06485,0.054563,0.794786,-2.05486,0.049281,0.906053,-2.08414,-0.020798,0.898105,-2.08057,-0.249337,0.846058,-2.06973,-0.329241,0.930051,-2.01955,-0.305758,0.647213,-2.01435,-0.347502,0.687734,-2.08529,-0.006097,1.28769,-1.82582,0.016089,1.27678,-2.03963},
/*18*/{0.115178,1.83963,-1.71275,0.182032,1.78811,-1.86615,0.06146,1.6873,-1.60589,0.032973,1.57251,-1.53038,0.182003,1.55144,-1.59811,0.271077,1.53504,-1.61821,0.115404,1.8771,-2.10918,0.188895,1.79526,-1.94205,-0.020943,1.84832,-2.02036,-0.097112,1.72694,-2.17747,-0.097084,1.63263,-2.24246,0.040663,1.49031,-2.19399,0.072524,1.43364,-2.20701,0.06614,1.37863,-1.7998,-0.117299,1.42818,-1.9047,-0.106991,1.45219,-1.95804,-0.103754,1.41944,-2.01534,0.096577,1.36976,-2.06864,0.085972,1.0751,-1.73831,-0.038352,1.04121,-1.78776,-0.014058,0.959908,-1.80662,0.075494,0.854177,-1.78082,0.040971,0.672438,-1.78335,0.013272,0.602639,-1.79957,-0.064593,0.712307,-1.7975,-0.089109,0.644757,-1.80829,-0.127937,0.428985,-1.83406,-0.239893,0.369028,-1.9134,0.017402,0.383073,-1.8663,-0.013639,0.336924,-1.79349,0.196144,1.11745,-2.11276,0.242873,1.04836,-2.09364,0.077195,1.03865,-2.09329,0.124529,0.973181,-2.07498,0.226514,0.876898,-2.06519,0.070495,0.797047,-2.05534,0.060968,0.908202,-2.08417,-0.008616,0.897423,-2.08131,-0.236706,0.843215,-2.07347,-0.320432,0.915417,-2.02026,-0.279934,0.634803,-2.01634,-0.324979,0.672661,-2.08716,-0.006736,1.28791,-1.82665,0.015386,1.27836,-2.04053},
/*19*/{0.1169,1.84236,-1.7137,0.181774,1.79032,-1.86751,0.065369,1.69012,-1.60605,0.041126,1.57469,-1.52974,0.189481,1.55733,-1.60041,0.277886,1.54359,-1.62188,0.112567,1.87905,-2.10943,0.188519,1.79694,-1.94391,-0.022641,1.85062,-2.0191,-0.103006,1.73213,-2.17574,-0.104604,1.63866,-2.24189,0.031428,1.49563,-2.1933,0.063,1.43913,-2.20714,0.066133,1.37899,-1.80017,-0.117799,1.42962,-1.9049,-0.10782,1.45362,-1.95781,-0.10444,1.42084,-2.01518,0.096756,1.37186,-2.0692,0.077045,1.07304,-1.74014,-0.046025,1.04164,-1.78837,-0.024642,0.959928,-1.80688,0.059511,0.851086,-1.78222,0.02442,0.671188,-1.7844,-0.003681,0.602075,-1.80122,-0.079605,0.712807,-1.7982,-0.104862,0.646452,-1.80883,-0.149193,0.429344,-1.83369,-0.260233,0.370455,-1.91335,-0.003257,0.382342,-1.86671,-0.034438,0.336832,-1.79371,0.201234,1.12532,-2.11286,0.249923,1.05819,-2.09416,0.084976,1.04213,-2.09236,0.133031,0.978714,-2.07472,0.239471,0.885146,-2.06542,0.086511,0.799682,-2.05674,0.072648,0.909723,-2.08425,0.003289,0.896194,-2.08207,-0.22294,0.82726,-2.06671,-0.31136,0.898796,-2.0195,-0.252966,0.622054,-2.01871,-0.298385,0.657498,-2.08984,-0.006925,1.28859,-1.8275,0.015081,1.28027,-2.04144},
/*20*/{0.119208,1.84549,-1.71445,0.182676,1.7919,-1.86885,0.068922,1.69299,-1.60643,0.047808,1.57712,-1.53073,0.195985,1.56447,-1.60238,0.283817,1.55285,-1.62564,0.10974,1.88094,-2.11021,0.187921,1.79873,-1.94527,-0.024434,1.85156,-2.01741,-0.109792,1.73758,-2.17331,-0.113305,1.64461,-2.23988,0.022727,1.50107,-2.1934,0.054645,1.44513,-2.20672,0.066568,1.37916,-1.80002,-0.116796,1.431,-1.90397,-0.107321,1.45546,-1.9565,-0.104293,1.42275,-2.01527,0.094598,1.37369,-2.06943,0.071114,1.07044,-1.7413,-0.056289,1.04296,-1.78896,-0.034947,0.960224,-1.80795,0.042509,0.849163,-1.78344,0.009096,0.67054,-1.78535,-0.019433,0.601907,-1.80079,-0.0951,0.714057,-1.79821,-0.121781,0.648127,-1.80905,-0.16804,0.43041,-1.83361,-0.280626,0.372131,-1.91243,-0.023973,0.381993,-1.86588,-0.055265,0.336601,-1.79419,0.205018,1.13353,-2.11326,0.256145,1.06767,-2.09409,0.091783,1.04705,-2.09313,0.142571,0.985644,-2.07552,0.251455,0.895484,-2.06593,0.102332,0.802513,-2.05806,0.0835,0.912175,-2.08496,0.014736,0.895462,-2.08332,-0.207714,0.81618,-2.06716,-0.302147,0.88247,-2.02091,-0.226148,0.609316,-2.02109,-0.273094,0.643119,-2.09274,-0.006903,1.2892,-1.82791,0.013916,1.28197,-2.04201},
/*21*/{0.120733,1.84893,-1.7155,0.183488,1.79466,-1.87033,0.073068,1.69611,-1.60657,0.054771,1.58005,-1.53061,0.201915,1.57127,-1.60552,0.289832,1.56203,-1.62989,0.106101,1.8841,-2.1106,0.186776,1.80102,-1.94738,-0.025426,1.85398,-2.01627,-0.116322,1.74331,-2.17179,-0.121351,1.6511,-2.23852,0.014862,1.50752,-2.19296,0.045816,1.45106,-2.20618,0.066572,1.37966,-1.80009,-0.116274,1.43311,-1.90226,-0.107262,1.45689,-1.95545,-0.103751,1.42489,-2.01502,0.095398,1.37664,-2.06949,0.063834,1.06868,-1.74168,-0.064906,1.04333,-1.78878,-0.045477,0.961445,-1.80784,0.026033,0.848565,-1.78575,-0.008717,0.669715,-1.78722,-0.037504,0.60181,-1.80311,-0.111041,0.714342,-1.79937,-0.137608,0.649932,-1.8096,-0.188491,0.430746,-1.83422,-0.301352,0.373683,-1.91243,-0.044789,0.381957,-1.86597,-0.076622,0.336887,-1.79371,0.210049,1.14169,-2.11385,0.262571,1.07724,-2.09453,0.098645,1.05235,-2.09299,0.150644,0.991725,-2.07494,0.263004,0.906633,-2.0665,0.117282,0.805737,-2.05925,0.09399,0.913099,-2.08543,0.025245,0.894952,-2.08374,-0.194487,0.804728,-2.06849,-0.292387,0.863576,-2.02037,-0.197289,0.597878,-2.02525,-0.248099,0.627345,-2.09503,-0.006435,1.29,-1.82841,0.014313,1.28462,-2.04257},
/*22*/{0.122665,1.85246,-1.71673,0.183315,1.79822,-1.8723,0.077505,1.69926,-1.60738,0.061629,1.58344,-1.53138,0.206655,1.57782,-1.60817,0.294677,1.57127,-1.63377,0.102947,1.88732,-2.1108,0.18672,1.80427,-1.94887,-0.026879,1.85544,-2.01425,-0.122687,1.74883,-2.17012,-0.128363,1.65727,-2.23737,0.006893,1.5141,-2.19249,0.037231,1.45783,-2.2054,0.066578,1.37986,-1.79999,-0.115928,1.43499,-1.90191,-0.107341,1.45909,-1.95477,-0.103888,1.42687,-2.01396,0.095193,1.37965,-2.06918,0.054795,1.06977,-1.7457,-0.073261,1.04572,-1.78899,-0.056622,0.964253,-1.80881,0.010672,0.849378,-1.78739,-0.025281,0.670127,-1.78835,-0.054694,0.602831,-1.80375,-0.127511,0.716111,-1.80057,-0.156269,0.65031,-1.81099,-0.208998,0.43273,-1.83418,-0.321612,0.376889,-1.91218,-0.065828,0.381438,-1.86693,-0.097983,0.336945,-1.79455,0.213768,1.14976,-2.11407,0.267324,1.087,-2.09596,0.104731,1.05811,-2.09328,0.158761,0.999042,-2.07444,0.27357,0.917685,-2.06785,0.132117,0.809565,-2.06134,0.103992,0.914521,-2.08595,0.035838,0.893975,-2.08466,-0.179386,0.792148,-2.06834,-0.282405,0.84437,-2.01934,-0.169167,0.586214,-2.02873,-0.222107,0.613651,-2.09798,-0.006554,1.29077,-1.82901,0.014099,1.28724,-2.04322},
/*23*/{0.124548,1.85636,-1.7183,0.183689,1.80127,-1.87388,0.082199,1.70265,-1.60829,0.066992,1.58677,-1.53173,0.211006,1.58529,-1.61116,0.298607,1.58103,-1.63785,0.099084,1.89099,-2.11118,0.186432,1.80743,-1.95145,-0.028625,1.85801,-2.01276,-0.128382,1.7547,-2.16832,-0.135721,1.66357,-2.23586,-0.001159,1.52079,-2.19094,0.030007,1.46431,-2.20414,0.066867,1.38065,-1.79967,-0.11513,1.4371,-1.90096,-0.106182,1.46117,-1.95345,-0.103337,1.42994,-2.01362,0.095196,1.38309,-2.06853,0.048981,1.06849,-1.74697,-0.081871,1.05183,-1.78961,-0.067979,0.967651,-1.80991,-0.002908,0.851071,-1.79014,-0.041873,0.670686,-1.79041,-0.07335,0.602655,-1.80629,-0.143909,0.717505,-1.80148,-0.17276,0.652829,-1.81213,-0.228522,0.435075,-1.8342,-0.341545,0.380861,-1.91206,-0.086051,0.381176,-1.86696,-0.11852,0.336868,-1.79491,0.216917,1.15812,-2.115,0.27241,1.09712,-2.09599,0.110282,1.06391,-2.09289,0.165632,1.00557,-2.07416,0.283881,0.92907,-2.06801,0.146296,0.813,-2.06248,0.110134,0.919892,-2.08717,0.045297,0.892689,-2.08447,-0.163764,0.78133,-2.07179,-0.271572,0.82499,-2.01825,-0.141229,0.574572,-2.03294,-0.196039,0.59893,-2.10072,-0.006206,1.2919,-1.82949,0.014463,1.29038,-2.04373},
/*24*/{0.126595,1.86074,-1.71948,0.184374,1.80498,-1.87604,0.086812,1.7064,-1.60922,0.072869,1.59008,-1.53215,0.215538,1.59312,-1.61423,0.303044,1.59043,-1.64265,0.095239,1.89482,-2.11153,0.186324,1.81103,-1.95339,-0.029986,1.85996,-2.01082,-0.13403,1.7609,-2.16678,-0.142846,1.6698,-2.23422,-0.009007,1.52664,-2.18934,0.022878,1.47173,-2.20326,0.068358,1.38154,-1.79895,-0.114367,1.4395,-1.89997,-0.106364,1.46388,-1.95234,-0.102895,1.4334,-2.01332,0.09556,1.38659,-2.06756,0.044911,1.06981,-1.74914,-0.088263,1.05243,-1.7889,-0.078379,0.970443,-1.80932,-0.014782,0.853109,-1.79205,-0.058089,0.672117,-1.79228,-0.090139,0.605255,-1.80745,-0.160348,0.720353,-1.80216,-0.190782,0.655409,-1.81305,-0.246887,0.438177,-1.83367,-0.361263,0.385115,-1.91182,-0.105741,0.381242,-1.86725,-0.138964,0.336956,-1.79537,0.220361,1.1661,-2.11519,0.277033,1.10649,-2.09604,0.115715,1.06842,-2.09219,0.173033,1.0118,-2.0751,0.292016,0.940734,-2.06893,0.159297,0.816783,-2.06441,0.117981,0.920766,-2.08685,0.053801,0.890325,-2.08444,-0.148187,0.766875,-2.06772,-0.258728,0.80397,-2.01713,-0.114101,0.566401,-2.03884,-0.166143,0.585114,-2.10357,-0.005695,1.29343,-1.82981,0.014858,1.29378,-2.04406},
/*25*/{0.127829,1.8654,-1.72109,0.184973,1.80914,-1.87814,0.091491,1.71108,-1.60988,0.077839,1.59397,-1.53268,0.218785,1.60055,-1.61688,0.305522,1.59999,-1.6471,0.091403,1.89898,-2.1113,0.186243,1.81543,-1.95557,-0.031612,1.86331,-2.00926,-0.138517,1.76775,-2.16487,-0.149038,1.67627,-2.23225,-0.015596,1.53384,-2.18713,0.015844,1.47901,-2.20207,0.069347,1.38191,-1.79809,-0.112671,1.44218,-1.89876,-0.104672,1.4665,-1.94997,-0.102018,1.43708,-2.01264,0.09612,1.39044,-2.06639,0.03655,1.06951,-1.75127,-0.096083,1.05499,-1.78747,-0.08898,0.974661,-1.81008,-0.026885,0.855244,-1.79465,-0.075173,0.674445,-1.79396,-0.107023,0.607641,-1.80803,-0.177315,0.722316,-1.80336,-0.207357,0.657209,-1.81302,-0.266299,0.44193,-1.83432,-0.380652,0.390743,-1.9113,-0.125469,0.3812,-1.86734,-0.160659,0.338115,-1.79536,0.223287,1.17424,-2.11594,0.280914,1.1161,-2.09674,0.120134,1.07484,-2.09228,0.178579,1.01825,-2.07471,0.300279,0.952086,-2.06926,0.17218,0.822078,-2.06826,0.125003,0.922233,-2.08687,0.062736,0.888018,-2.08448,-0.12909,0.753057,-2.0674,-0.245311,0.783125,-2.01607,-0.08173,0.553596,-2.04039,-0.140382,0.572815,-2.1076,-0.005019,1.29454,-1.83014,0.0155,1.29745,-2.04437},
/*26*/{0.129495,1.8701,-1.72257,0.185783,1.81335,-1.88062,0.096285,1.7149,-1.61081,0.084261,1.59877,-1.53345,0.222392,1.60795,-1.62045,0.307543,1.60911,-1.65111,0.088589,1.90309,-2.11124,0.18568,1.8195,-1.95722,-0.033371,1.86627,-2.00795,-0.143725,1.77396,-2.16419,-0.155777,1.68287,-2.23086,-0.022099,1.54061,-2.18506,0.009074,1.48659,-2.2012,0.07091,1.38346,-1.79746,-0.112364,1.44463,-1.89814,-0.104122,1.46948,-1.9491,-0.101139,1.44109,-2.01187,0.096689,1.39443,-2.06535,0.030176,1.06979,-1.75425,-0.102832,1.06,-1.78755,-0.096541,0.977891,-1.8089,-0.03805,0.857248,-1.79681,-0.09181,0.677442,-1.79549,-0.125107,0.610053,-1.81069,-0.192908,0.726129,-1.80458,-0.224247,0.661152,-1.81452,-0.285665,0.445214,-1.83469,-0.399949,0.396924,-1.91069,-0.145761,0.379774,-1.86749,-0.181563,0.338556,-1.79519,0.225841,1.18211,-2.11596,0.284822,1.12443,-2.09593,0.124063,1.08133,-2.09308,0.183782,1.02551,-2.07446,0.30608,0.962642,-2.06999,0.184008,0.824369,-2.06805,0.13097,0.92341,-2.08721,0.071212,0.886002,-2.08427,-0.11275,0.740956,-2.06629,-0.229504,0.760519,-2.0144,-0.052063,0.544221,-2.04479,-0.111985,0.560659,-2.11078,-0.004297,1.29652,-1.83049,0.016338,1.30133,-2.04468},
/*27*/{0.130871,1.875,-1.72388,0.186187,1.81786,-1.88228,0.100782,1.71964,-1.61187,0.089821,1.60335,-1.53396,0.224795,1.61588,-1.62338,0.310266,1.6186,-1.65552,0.085636,1.90721,-2.11107,0.18536,1.82322,-1.95958,-0.03502,1.86965,-2.0065,-0.147062,1.78069,-2.16138,-0.161253,1.68965,-2.22837,-0.027768,1.54917,-2.18369,0.003117,1.49365,-2.19981,0.072155,1.38495,-1.797,-0.109963,1.44803,-1.8971,-0.103165,1.47301,-1.94655,-0.100015,1.44456,-2.01124,0.097563,1.39909,-2.06448,0.025857,1.07023,-1.75669,-0.108931,1.06665,-1.78897,-0.10737,0.984507,-1.80984,-0.049592,0.860452,-1.79936,-0.108287,0.680609,-1.79671,-0.142138,0.613504,-1.81172,-0.20897,0.730569,-1.80619,-0.240024,0.666479,-1.8144,-0.304224,0.450434,-1.83435,-0.418306,0.404525,-1.91059,-0.165136,0.377291,-1.86858,-0.201867,0.338844,-1.79557,0.22854,1.18931,-2.11546,0.288285,1.13312,-2.09556,0.12879,1.08743,-2.09321,0.187743,1.03315,-2.0747,0.311793,0.972684,-2.06978,0.196415,0.827926,-2.06908,0.137025,0.923974,-2.08709,0.079261,0.882768,-2.08355,-0.096062,0.727242,-2.06593,-0.213293,0.738161,-2.01272,-0.021217,0.534567,-2.04838,-0.083224,0.549472,-2.11456,-0.002964,1.29863,-1.83077,0.017335,1.30563,-2.04493},
/*28*/{0.132551,1.88057,-1.72544,0.187249,1.8219,-1.88464,0.105686,1.72384,-1.61364,0.095324,1.60724,-1.53399,0.227853,1.62266,-1.62596,0.312013,1.62732,-1.65929,0.082274,1.91134,-2.11105,0.184627,1.82815,-1.96161,-0.033228,1.87063,-2.00451,-0.151814,1.78724,-2.16052,-0.166819,1.69597,-2.22644,-0.034134,1.55598,-2.18155,-0.002681,1.50115,-2.19854,0.073435,1.38571,-1.79667,-0.108521,1.4514,-1.89595,-0.100752,1.47599,-1.94517,-0.099181,1.44989,-2.01091,0.098478,1.40371,-2.06363,0.021605,1.07304,-1.75896,-0.11453,1.07447,-1.78918,-0.114439,0.98983,-1.81094,-0.062667,0.863597,-1.8012,-0.124433,0.685197,-1.79738,-0.159528,0.617999,-1.81164,-0.225118,0.735408,-1.80718,-0.256256,0.671867,-1.81492,-0.321812,0.45443,-1.83246,-0.436927,0.413062,-1.90975,-0.184981,0.377791,-1.86813,-0.222367,0.339075,-1.79573,0.231136,1.19712,-2.11615,0.291211,1.14067,-2.09521,0.131955,1.0944,-2.09365,0.192119,1.04005,-2.07459,0.317617,0.983374,-2.06987,0.2074,0.831019,-2.07092,0.142439,0.923105,-2.08678,0.087197,0.879182,-2.08328,-0.082287,0.713415,-2.06608,-0.198092,0.716621,-2.0112,0.009466,0.527444,-2.05292,-0.053613,0.539036,-2.11813,-0.00232,1.30029,-1.83188,0.018115,1.3103,-2.04591},
/*29*/{0.134002,1.88444,-1.72741,0.188577,1.82701,-1.88687,0.109384,1.72881,-1.61451,0.099089,1.61261,-1.53505,0.229821,1.63024,-1.62848,0.313622,1.6366,-1.66403,0.0805,1.91493,-2.1112,0.184373,1.83286,-1.96337,-0.035159,1.87445,-2.00277,-0.155134,1.79359,-2.15838,-0.171577,1.7027,-2.22423,-0.040538,1.56291,-2.17941,-0.008403,1.50808,-2.19723,0.075621,1.38724,-1.79662,-0.105957,1.454,-1.89416,-0.099848,1.47983,-1.94297,-0.097126,1.45375,-2.00998,0.098624,1.40788,-2.06257,0.014496,1.07435,-1.76236,-0.120562,1.08151,-1.79034,-0.125285,0.996184,-1.81208,-0.076335,0.868495,-1.80172,-0.140111,0.691166,-1.79793,-0.174309,0.624593,-1.81137,-0.240449,0.741283,-1.80705,-0.272291,0.677976,-1.81559,-0.338583,0.460698,-1.83223,-0.454132,0.421848,-1.90923,-0.204663,0.377469,-1.86854,-0.243306,0.339582,-1.79637,0.233567,1.20435,-2.11607,0.293332,1.1486,-2.09511,0.135084,1.10045,-2.09449,0.195398,1.04717,-2.07421,0.320047,0.992524,-2.07041,0.217729,0.833763,-2.07221,0.147596,0.922776,-2.08609,0.094987,0.875226,-2.08226,-0.064927,0.700706,-2.06578,-0.180939,0.695498,-2.01068,0.039382,0.520767,-2.05766,-0.024605,0.530157,-2.12235,-0.00058,1.30235,-1.83209,0.018911,1.31441,-2.0461},
/*30*/{0.135311,1.88917,-1.72879,0.189713,1.83133,-1.88888,0.113256,1.73376,-1.61551,0.102509,1.61736,-1.53552,0.231416,1.63711,-1.63166,0.314317,1.6441,-1.6683,0.078604,1.91889,-2.11134,0.18415,1.83711,-1.96551,-0.035549,1.87783,-2.00247,-0.15797,1.79979,-2.15617,-0.176789,1.709,-2.22184,-0.044343,1.57054,-2.1768,-0.013134,1.51553,-2.1959,0.078212,1.38814,-1.79626,-0.103451,1.45774,-1.89312,-0.098715,1.48327,-1.94124,-0.096251,1.45916,-2.00907,0.099827,1.41268,-2.06201,0.00984,1.07599,-1.76505,-0.124539,1.08678,-1.7889,-0.131782,1.00247,-1.81069,-0.09133,0.875112,-1.80255,-0.154665,0.696709,-1.79808,-0.191225,0.629889,-1.8122,-0.255321,0.748064,-1.80845,-0.287853,0.684201,-1.81614,-0.354641,0.465612,-1.83088,-0.471302,0.431329,-1.909,-0.224222,0.376635,-1.86855,-0.264609,0.340049,-1.79699,0.235128,1.21077,-2.11587,0.295964,1.1562,-2.09457,0.138471,1.10628,-2.09593,0.198353,1.05429,-2.07485,0.321886,1.00042,-2.07038,0.22791,0.836458,-2.07361,0.152353,0.921082,-2.08622,0.10241,0.869169,-2.08112,-0.042513,0.687235,-2.06482,-0.162023,0.672575,-2.0091,0.069841,0.514752,-2.06216,0.003935,0.522506,-2.12668,0.000874,1.30441,-1.83312,0.019751,1.31935,-2.04701},
/*31*/{0.136843,1.89459,-1.73068,0.189298,1.83556,-1.89125,0.116252,1.73847,-1.61673,0.107229,1.62231,-1.53598,0.23254,1.644,-1.63371,0.314849,1.6526,-1.67168,0.077054,1.9224,-2.11141,0.183933,1.84182,-1.96744,-0.03542,1.88132,-2.00137,-0.160336,1.80564,-2.15488,-0.181269,1.71543,-2.2192,-0.050111,1.57686,-2.17511,-0.017683,1.52164,-2.19443,0.079877,1.38993,-1.79749,-0.102031,1.46124,-1.89295,-0.097035,1.48672,-1.93955,-0.094147,1.46381,-2.00863,0.100401,1.41746,-2.06212,0.005652,1.07849,-1.76663,-0.128469,1.09363,-1.78914,-0.139201,1.01004,-1.81163,-0.105181,0.881302,-1.8029,-0.170091,0.702819,-1.7979,-0.206044,0.636116,-1.81099,-0.269474,0.754835,-1.80875,-0.302548,0.69185,-1.81646,-0.372552,0.471659,-1.83005,-0.487452,0.442009,-1.90891,-0.243828,0.375897,-1.86859,-0.285039,0.340561,-1.79711,0.237106,1.21838,-2.11614,0.298769,1.1638,-2.09408,0.140944,1.11208,-2.09603,0.201143,1.05993,-2.0747,0.322711,1.00821,-2.07109,0.238224,0.83814,-2.07473,0.156583,0.918112,-2.0857,0.110895,0.863904,-2.08085,-0.024375,0.673742,-2.06456,-0.141544,0.649006,-2.00609,0.099985,0.509739,-2.06617,0.033913,0.512931,-2.12875,0.002283,1.30669,-1.83465,0.020596,1.32416,-2.04839},
/*32*/{0.138099,1.89892,-1.7321,0.189753,1.84039,-1.893,0.119535,1.74343,-1.61731,0.110027,1.62737,-1.53713,0.23349,1.65082,-1.63717,0.315779,1.66025,-1.67519,0.075893,1.926,-2.11174,0.183225,1.84578,-1.96872,-0.036477,1.88498,-2.00058,-0.162481,1.8115,-2.15287,-0.185413,1.72161,-2.21613,-0.053792,1.58318,-2.17265,-0.02182,1.52852,-2.19309,0.083063,1.39143,-1.79923,-0.099371,1.46486,-1.89131,-0.095213,1.49111,-1.93794,-0.093438,1.46847,-2.00789,0.100193,1.42191,-2.06153,0.001705,1.08094,-1.76934,-0.132219,1.10065,-1.7891,-0.146225,1.01763,-1.81177,-0.117122,0.888436,-1.80234,-0.183382,0.709963,-1.797,-0.220124,0.644082,-1.80972,-0.282949,0.762869,-1.80931,-0.315599,0.700082,-1.81552,-0.387084,0.478986,-1.82904,-0.503431,0.453127,-1.90825,-0.263824,0.375804,-1.86898,-0.306479,0.341536,-1.79801,0.238362,1.22483,-2.1158,0.300087,1.17022,-2.09381,0.143267,1.11785,-2.09627,0.203397,1.06525,-2.07492,0.322734,1.01418,-2.07161,0.247446,0.840121,-2.07616,0.161567,0.914337,-2.08468,0.118353,0.857276,-2.07997,-0.005664,0.660267,-2.06369,-0.119983,0.627848,-2.00592,0.129239,0.506081,-2.0697,0.063239,0.505917,-2.13131,0.004199,1.30917,-1.83578,0.02052,1.3288,-2.04949},
/*33*/{0.139705,1.90394,-1.73374,0.19112,1.84426,-1.89503,0.121893,1.74842,-1.6187,0.113133,1.63161,-1.53684,0.23496,1.65721,-1.63974,0.316075,1.66746,-1.67919,0.074171,1.92871,-2.11208,0.183189,1.85027,-1.97027,-0.036459,1.88842,-2.00002,-0.165394,1.81691,-2.15155,-0.188941,1.72755,-2.21362,-0.057234,1.58938,-2.17072,-0.025061,1.53446,-2.19224,0.086432,1.39282,-1.8009,-0.097068,1.46852,-1.89034,-0.094591,1.49521,-1.93548,-0.091274,1.47394,-2.007,0.100949,1.42693,-2.06171,-0.002045,1.08367,-1.77109,-0.13591,1.10798,-1.78945,-0.152095,1.025,-1.81018,-0.12878,0.895612,-1.8032,-0.196154,0.717728,-1.79617,-0.233481,0.651705,-1.80817,-0.294649,0.770798,-1.80949,-0.328243,0.707933,-1.81627,-0.402749,0.484868,-1.82753,-0.516619,0.465469,-1.90658,-0.283286,0.376275,-1.86904,-0.328047,0.34272,-1.79849,0.239642,1.2308,-2.11535,0.301494,1.17638,-2.09303,0.144081,1.12326,-2.09668,0.204017,1.07381,-2.07907,0.320599,1.01904,-2.07266,0.255781,0.840341,-2.07744,0.165623,0.908843,-2.08439,0.127014,0.850753,-2.07996,0.015682,0.646812,-2.06296,-0.096343,0.605883,-2.00313,0.159197,0.502564,-2.07246,0.092688,0.498998,-2.13299,0.006347,1.31148,-1.83756,0.021062,1.33424,-2.05108},
/*34*/{0.141664,1.90844,-1.73542,0.190851,1.8491,-1.897,0.125085,1.75301,-1.61941,0.116177,1.63646,-1.53751,0.234943,1.66348,-1.64178,0.315818,1.6749,-1.68265,0.072628,1.93201,-2.11232,0.182837,1.8538,-1.97229,-0.036473,1.89167,-1.99946,-0.166949,1.82235,-2.14923,-0.192844,1.73293,-2.21094,-0.061664,1.5946,-2.16886,-0.029072,1.54015,-2.19085,0.088385,1.39458,-1.80275,-0.095356,1.47279,-1.88852,-0.091802,1.49894,-1.93413,-0.089973,1.47834,-2.00636,0.101197,1.43178,-2.06221,-0.006109,1.08703,-1.77285,-0.137955,1.11458,-1.78779,-0.158413,1.03351,-1.81063,-0.137215,0.90285,-1.8029,-0.20787,0.725279,-1.79513,-0.244497,0.659861,-1.80651,-0.306275,0.778497,-1.80905,-0.340107,0.715908,-1.81629,-0.415574,0.493065,-1.82544,-0.530892,0.479772,-1.90624,-0.301891,0.375934,-1.86953,-0.34918,0.344489,-1.79839,0.240656,1.23613,-2.11454,0.302563,1.18102,-2.09172,0.14519,1.12926,-2.09745,0.204392,1.07878,-2.07979,0.319515,1.02362,-2.07349,0.26506,0.84115,-2.07877,0.171024,0.904136,-2.08339,0.135271,0.843245,-2.07986,0.036259,0.633337,-2.06176,-0.072516,0.583818,-2.00101,0.188632,0.500236,-2.07441,0.120796,0.49254,-2.13466,0.007984,1.31403,-1.83902,0.021166,1.33908,-2.05239},
/*35*/{0.142448,1.91274,-1.73734,0.191083,1.85239,-1.89827,0.127285,1.7574,-1.62025,0.119793,1.6408,-1.53862,0.235851,1.66895,-1.64459,0.316163,1.68127,-1.68676,0.071861,1.93528,-2.11275,0.1831,1.85799,-1.97411,-0.036461,1.89409,-1.99918,-0.168549,1.82653,-2.14715,-0.195533,1.73817,-2.20901,-0.063721,1.60007,-2.16706,-0.032048,1.54534,-2.18987,0.090938,1.39654,-1.80453,-0.093524,1.47631,-1.88867,-0.090204,1.50294,-1.93269,-0.088457,1.4838,-2.00583,0.101657,1.43562,-2.06304,-0.007464,1.09175,-1.77315,-0.140892,1.12126,-1.78785,-0.162937,1.04086,-1.81085,-0.14581,0.910055,-1.80281,-0.218433,0.732659,-1.79385,-0.255359,0.667836,-1.8039,-0.31602,0.787021,-1.80864,-0.350384,0.724075,-1.81595,-0.42788,0.499835,-1.8236,-0.542267,0.493219,-1.90475,-0.321782,0.376278,-1.86964,-0.369252,0.345887,-1.79892,0.241196,1.24032,-2.11406,0.302412,1.18584,-2.09141,0.144495,1.13428,-2.09851,0.204921,1.08372,-2.08069,0.318443,1.02674,-2.07441,0.273691,0.840718,-2.07928,0.176103,0.898832,-2.08349,0.144839,0.835527,-2.08014,0.05445,0.620332,-2.06172,-0.048199,0.56404,-2.00039,0.21663,0.498107,-2.07547,0.149501,0.487425,-2.13591,0.009378,1.31668,-1.84072,0.021194,1.34369,-2.05393},
/*36*/{0.144206,1.91697,-1.73869,0.193049,1.857,-1.90002,0.129404,1.7617,-1.62115,0.120918,1.645,-1.53954,0.236246,1.67434,-1.64692,0.316741,1.68723,-1.68966,0.071069,1.93773,-2.11344,0.182507,1.86113,-1.9752,-0.035927,1.89706,-1.99878,-0.169851,1.8312,-2.14595,-0.198561,1.74294,-2.20524,-0.066365,1.60535,-2.16533,-0.034848,1.54989,-2.18858,0.093939,1.39867,-1.8059,-0.090914,1.48003,-1.88783,-0.088918,1.50681,-1.932,-0.086468,1.48857,-2.00487,0.101669,1.43979,-2.06344,-0.011773,1.09325,-1.77403,-0.142703,1.12883,-1.78736,-0.167932,1.04893,-1.81109,-0.153345,0.91721,-1.80248,-0.228128,0.740088,-1.79244,-0.266503,0.674278,-1.80229,-0.325243,0.794103,-1.80881,-0.360403,0.731832,-1.81559,-0.438982,0.507426,-1.82151,-0.552512,0.507233,-1.9036,-0.340887,0.377397,-1.86848,-0.390072,0.34832,-1.79918,0.24202,1.24445,-2.11278,0.302705,1.18935,-2.0904,0.144223,1.13948,-2.09907,0.204094,1.08666,-2.08103,0.317265,1.02841,-2.07508,0.281772,0.839673,-2.08047,0.181575,0.892271,-2.08369,0.15275,0.828214,-2.08037,0.073161,0.606858,-2.06156,-0.025325,0.545115,-2.0004,0.24478,0.496551,-2.07549,0.178326,0.48104,-2.13527,0.011347,1.31956,-1.84187,0.021476,1.34813,-2.05496},
/*37*/{0.146434,1.92114,-1.73999,0.193185,1.86006,-1.90158,0.131025,1.76617,-1.62166,0.123645,1.64913,-1.53968,0.236695,1.67949,-1.64908,0.316523,1.69277,-1.69315,0.070103,1.94052,-2.11368,0.183238,1.86447,-1.97643,-0.03598,1.90008,-1.99933,-0.170283,1.83585,-2.14389,-0.201162,1.74788,-2.20248,-0.068075,1.60956,-2.1644,-0.037086,1.55437,-2.18712,0.096504,1.40136,-1.80789,-0.089454,1.48411,-1.88666,-0.087365,1.51024,-1.93153,-0.085098,1.49391,-2.00427,0.102692,1.44429,-2.06456,-0.014464,1.09731,-1.77406,-0.143695,1.13371,-1.7856,-0.17072,1.05588,-1.80997,-0.159984,0.923624,-1.8026,-0.235805,0.747045,-1.79134,-0.274161,0.682319,-1.80048,-0.33317,0.801814,-1.80848,-0.36777,0.740251,-1.81317,-0.449966,0.51422,-1.81865,-0.561811,0.523012,-1.90228,-0.357696,0.378289,-1.86832,-0.410559,0.351446,-1.79919,0.242145,1.24758,-2.11261,0.302531,1.19147,-2.09026,0.143477,1.14309,-2.10013,0.202664,1.08941,-2.08288,0.31578,1.02962,-2.07694,0.288828,0.838708,-2.08261,0.187008,0.886,-2.0836,0.161404,0.819922,-2.07867,0.093574,0.595389,-2.06174,-0.000287,0.526266,-1.99913,0.272571,0.494941,-2.07439,0.207112,0.476083,-2.13454,0.013264,1.32268,-1.84345,0.021925,1.35304,-2.05636},
/*38*/{0.147034,1.92448,-1.74204,0.193239,1.86395,-1.90286,0.133215,1.77005,-1.62283,0.125482,1.65322,-1.54081,0.236989,1.6835,-1.65103,0.315839,1.69779,-1.69652,0.069484,1.94292,-2.11396,0.183393,1.86806,-1.9779,-0.035355,1.90236,-1.99888,-0.170918,1.84011,-2.14257,-0.203498,1.75201,-2.19952,-0.069787,1.61345,-2.16265,-0.039521,1.55814,-2.18595,0.097827,1.40328,-1.80925,-0.087068,1.48761,-1.88657,-0.086218,1.51452,-1.93023,-0.082367,1.49786,-2.00321,0.102166,1.44787,-2.06545,-0.017905,1.10002,-1.77368,-0.146031,1.14001,-1.78561,-0.1743,1.06188,-1.81011,-0.16475,0.929697,-1.80293,-0.242539,0.754177,-1.79021,-0.280909,0.689368,-1.79923,-0.339864,0.809922,-1.80653,-0.375202,0.74726,-1.81244,-0.459529,0.522164,-1.81728,-0.568921,0.539993,-1.90129,-0.374852,0.379791,-1.86694,-0.430644,0.354931,-1.7986,0.241824,1.25009,-2.11231,0.300954,1.193,-2.08962,0.141757,1.14611,-2.10132,0.200336,1.09251,-2.08429,0.314351,1.02983,-2.07847,0.296681,0.837346,-2.08335,0.191484,0.879121,-2.08473,0.169558,0.810637,-2.07887,0.113708,0.583699,-2.06145,0.025079,0.509521,-2.00023,0.299805,0.494362,-2.0723,0.23635,0.471237,-2.13306,0.014206,1.32527,-1.84459,0.021584,1.35703,-2.05734},
/*39*/{0.149317,1.92832,-1.74267,0.195164,1.86743,-1.90378,0.134653,1.77367,-1.62369,0.127873,1.65591,-1.54121,0.237154,1.68796,-1.65334,0.315565,1.70214,-1.7,0.068826,1.94556,-2.11447,0.183994,1.87127,-1.97942,-0.03504,1.90473,-1.99699,-0.17182,1.84368,-2.14032,-0.205307,1.75567,-2.19647,-0.071426,1.61718,-2.16141,-0.040983,1.56186,-2.18514,0.099971,1.40589,-1.81008,-0.084988,1.49207,-1.88458,-0.084371,1.51787,-1.92944,-0.08087,1.50197,-2.00181,0.102153,1.45139,-2.06624,-0.019317,1.10343,-1.77309,-0.14692,1.14616,-1.78616,-0.17735,1.06769,-1.81064,-0.169003,0.934911,-1.80354,-0.248653,0.760871,-1.78925,-0.286967,0.697176,-1.79698,-0.345024,0.81711,-1.806,-0.380928,0.754514,-1.80918,-0.468457,0.530356,-1.81395,-0.575404,0.555745,-1.89881,-0.392727,0.381381,-1.86553,-0.450272,0.35923,-1.79903,0.241108,1.25192,-2.11208,0.299336,1.19321,-2.08996,0.140267,1.14944,-2.102,0.197891,1.09407,-2.08537,0.312622,1.02944,-2.07983,0.303014,0.835834,-2.08391,0.19707,0.872489,-2.08574,0.176895,0.804973,-2.07938,0.13592,0.573122,-2.06157,0.04901,0.49436,-2.00037,0.324886,0.494508,-2.06974,0.263953,0.46619,-2.13039,0.015826,1.32849,-1.84471,0.021287,1.36077,-2.05744},
/*40*/{0.150145,1.9312,-1.74375,0.195657,1.87076,-1.90568,0.135924,1.77681,-1.62468,0.128175,1.65946,-1.54129,0.237392,1.69189,-1.65516,0.315806,1.70642,-1.70246,0.06841,1.94784,-2.11518,0.184278,1.87431,-1.98038,-0.034089,1.9074,-1.99761,-0.172572,1.84654,-2.13976,-0.207321,1.75878,-2.19355,-0.072848,1.62065,-2.16021,-0.042545,1.56469,-2.1839,0.100995,1.40886,-1.811,-0.083757,1.49525,-1.8844,-0.083394,1.52171,-1.92896,-0.079922,1.50573,-2.00097,0.102896,1.45511,-2.06653,-0.020895,1.10756,-1.77212,-0.150681,1.14968,-1.78635,-0.179321,1.07262,-1.81086,-0.171626,0.939956,-1.8041,-0.254419,0.766586,-1.78808,-0.294475,0.702792,-1.79504,-0.349033,0.823563,-1.80294,-0.385783,0.761556,-1.80783,-0.474727,0.536185,-1.81012,-0.580246,0.572086,-1.89562,-0.409468,0.383815,-1.86528,-0.469675,0.365004,-1.79885,0.240206,1.25253,-2.11197,0.297506,1.19296,-2.09007,0.137651,1.15073,-2.10287,0.194756,1.09478,-2.08687,0.309574,1.02805,-2.08147,0.309419,0.83386,-2.08503,0.200554,0.865238,-2.08594,0.184877,0.799547,-2.07982,0.15671,0.562781,-2.06195,0.072428,0.479236,-2.00319,0.34911,0.49463,-2.0664,0.29136,0.462251,-2.12736,0.016843,1.33155,-1.84533,0.021737,1.3647,-2.05794},
/*41*/{0.151667,1.93391,-1.7453,0.195878,1.87397,-1.9071,0.137636,1.78055,-1.62558,0.129082,1.66201,-1.5423,0.237741,1.69592,-1.65729,0.315447,1.70966,-1.70526,0.068825,1.94966,-2.11559,0.184502,1.87676,-1.98142,-0.033539,1.90912,-1.9968,-0.172271,1.849,-2.1363,-0.208412,1.76161,-2.19114,-0.074118,1.62321,-2.15939,-0.043149,1.56723,-2.18291,0.102451,1.41179,-1.81201,-0.082531,1.49898,-1.88373,-0.082147,1.52563,-1.92853,-0.078998,1.50957,-1.99976,0.103348,1.45827,-2.06669,-0.021933,1.10973,-1.77118,-0.151428,1.15403,-1.786,-0.181104,1.07759,-1.81033,-0.17382,0.944031,-1.80536,-0.257927,0.771911,-1.78762,-0.299062,0.708305,-1.79391,-0.352121,0.830242,-1.8019,-0.389395,0.769298,-1.80544,-0.483309,0.547903,-1.80871,-0.583463,0.588901,-1.89102,-0.427992,0.386615,-1.86498,-0.48914,0.371279,-1.79959,0.238143,1.25173,-2.11169,0.294915,1.192,-2.09074,0.13449,1.15188,-2.10343,0.190991,1.09455,-2.08862,0.306325,1.0268,-2.08336,0.316547,0.831629,-2.0854,0.205778,0.858405,-2.08738,0.19314,0.788273,-2.08001,0.176031,0.553067,-2.06236,0.096536,0.465891,-2.00056,0.37274,0.495308,-2.06388,0.317474,0.458375,-2.1243,0.017804,1.3348,-1.84549,0.021677,1.36831,-2.05807},
/*42*/{0.153229,1.93662,-1.74647,0.196928,1.87705,-1.90795,0.138829,1.78359,-1.62688,0.1299,1.66438,-1.54318,0.238411,1.69763,-1.65966,0.315199,1.71231,-1.70816,0.06864,1.95177,-2.11647,0.18516,1.87885,-1.98275,-0.032918,1.91132,-1.99714,-0.172693,1.85119,-2.13527,-0.208934,1.76391,-2.18941,-0.074573,1.62525,-2.15785,-0.043895,1.56967,-2.18187,0.102209,1.41421,-1.81233,-0.081373,1.50231,-1.88224,-0.080955,1.52936,-1.92839,-0.078529,1.51254,-1.99931,0.103627,1.46093,-2.06649,-0.022716,1.11243,-1.77104,-0.152261,1.15744,-1.78645,-0.181661,1.08107,-1.81025,-0.174652,0.948042,-1.80553,-0.261421,0.777153,-1.78741,-0.30294,0.714164,-1.79316,-0.354541,0.836615,-1.8017,-0.392713,0.776185,-1.80408,-0.489983,0.555086,-1.80515,-0.585746,0.605729,-1.88473,-0.444778,0.391521,-1.86625,-0.507022,0.379256,-1.80083,0.235653,1.25093,-2.11204,0.292063,1.18992,-2.09171,0.131105,1.15209,-2.10396,0.186605,1.09382,-2.08918,0.303593,1.02425,-2.08504,0.320824,0.829615,-2.08662,0.209533,0.850364,-2.08948,0.200685,0.780004,-2.0815,0.195386,0.543884,-2.06197,0.120946,0.453798,-2.00116,0.395525,0.496283,-2.05933,0.343024,0.454485,-2.12064,0.017716,1.3376,-1.84527,0.021338,1.37126,-2.05783},
/*43*/{0.15465,1.9391,-1.74766,0.19775,1.87952,-1.9094,0.140129,1.78588,-1.62774,0.130465,1.66611,-1.5433,0.238379,1.69929,-1.66004,0.314682,1.71428,-1.71059,0.068766,1.95357,-2.11686,0.185341,1.88074,-1.98392,-0.03213,1.91337,-1.99666,-0.172489,1.85383,-2.13487,-0.209681,1.76512,-2.18718,-0.074728,1.62684,-2.15711,-0.044028,1.57114,-2.18129,0.103671,1.41763,-1.81229,-0.080373,1.50528,-1.88113,-0.081082,1.53232,-1.92731,-0.077754,1.51588,-1.99771,0.103232,1.46333,-2.06597,-0.023981,1.11496,-1.77076,-0.151626,1.16063,-1.78722,-0.181447,1.08413,-1.81104,-0.174994,0.950552,-1.80555,-0.26346,0.782054,-1.78686,-0.30599,0.719812,-1.79186,-0.355333,0.842427,-1.80125,-0.395411,0.78349,-1.80233,-0.493122,0.564684,-1.80267,-0.587839,0.622382,-1.87793,-0.461,0.398887,-1.86866,-0.522088,0.387836,-1.80188,0.234182,1.24842,-2.11217,0.289583,1.18657,-2.09244,0.126941,1.15133,-2.10476,0.18228,1.0925,-2.09022,0.300194,1.02134,-2.08734,0.324638,0.827542,-2.0877,0.212485,0.84186,-2.09062,0.20705,0.772863,-2.08308,0.215025,0.536396,-2.06225,0.14335,0.443508,-2.0014,0.41654,0.49689,-2.05489,0.367262,0.451712,-2.11677,0.018781,1.34084,-1.8442,0.02107,1.37397,-2.05686},
/*44*/{0.156111,1.9403,-1.74854,0.198782,1.88185,-1.91094,0.141469,1.788,-1.62907,0.131452,1.66881,-1.54353,0.238341,1.70092,-1.66144,0.314608,1.71514,-1.71266,0.069411,1.95478,-2.11745,0.185965,1.88253,-1.98593,-0.031427,1.91566,-1.9969,-0.172775,1.85471,-2.13329,-0.21003,1.76665,-2.18543,-0.074699,1.62797,-2.15605,-0.04444,1.5721,-2.18052,0.103873,1.41962,-1.81155,-0.079835,1.50863,-1.88098,-0.080578,1.53529,-1.92703,-0.076638,1.51825,-1.99729,0.103466,1.46485,-2.06563,-0.025843,1.11766,-1.77145,-0.151543,1.16157,-1.78866,-0.180142,1.0863,-1.81146,-0.174042,0.952519,-1.80636,-0.264533,0.784727,-1.78709,-0.308839,0.723367,-1.7911,-0.355613,0.848196,-1.80035,-0.39613,0.789527,-1.8006,-0.497603,0.572899,-1.79913,-0.590015,0.638,-1.87001,-0.476813,0.407726,-1.87216,-0.538193,0.397097,-1.80528,0.230761,1.24604,-2.11329,0.284795,1.18345,-2.09387,0.123132,1.15063,-2.106,0.176864,1.09071,-2.09139,0.296506,1.01763,-2.08898,0.328453,0.825362,-2.08907,0.215059,0.834203,-2.0929,0.212965,0.764543,-2.08638,0.233167,0.529103,-2.06236,0.163119,0.433132,-2.00078,0.435842,0.498285,-2.0517,0.39,0.448008,-2.11157,0.018415,1.34341,-1.84337,0.020909,1.37588,-2.05612},
/*45*/{0.157728,1.94247,-1.75074,0.198445,1.8835,-1.91185,0.142096,1.79,-1.63007,0.131088,1.67045,-1.54483,0.237926,1.70183,-1.66351,0.313732,1.71584,-1.71476,0.06925,1.95625,-2.11922,0.186706,1.88398,-1.98625,-0.030078,1.91667,-1.99823,-0.17317,1.85494,-2.13223,-0.209988,1.76681,-2.18377,-0.073263,1.6294,-2.15456,-0.043968,1.57305,-2.18,0.104358,1.42237,-1.81209,-0.079976,1.51077,-1.88039,-0.080107,1.53812,-1.9273,-0.077575,1.52069,-1.99643,0.103444,1.46693,-2.06506,-0.026071,1.11979,-1.77185,-0.150011,1.16229,-1.78962,-0.178329,1.08669,-1.81161,-0.170936,0.954041,-1.80609,-0.264336,0.787452,-1.78642,-0.310336,0.726963,-1.79081,-0.353853,0.853205,-1.79841,-0.396508,0.795914,-1.7996,-0.50214,0.581658,-1.79912,-0.592008,0.653819,-1.86045,-0.493776,0.416238,-1.87716,-0.55295,0.407095,-1.8077,0.227541,1.24289,-2.11427,0.281421,1.17841,-2.09495,0.1181,1.1487,-2.10699,0.171406,1.08796,-2.09186,0.293433,1.01369,-2.0905,0.331077,0.822558,-2.08975,0.217657,0.827785,-2.09594,0.21889,0.757111,-2.08816,0.24696,0.520677,-2.0616,0.182718,0.424933,-2.00012,0.454334,0.499494,-2.04853,0.411456,0.44553,-2.1072,0.018677,1.346,-1.84297,0.020605,1.37816,-2.05579},
/*46*/{0.15915,1.94364,-1.75113,0.198594,1.88629,-1.91377,0.142896,1.79136,-1.63126,0.131049,1.67126,-1.54513,0.237614,1.70228,-1.66434,0.311763,1.71612,-1.71684,0.070671,1.9571,-2.11995,0.187447,1.88525,-1.98783,-0.028733,1.91801,-1.99868,-0.17063,1.85573,-2.12966,-0.209798,1.76698,-2.18241,-0.073334,1.62949,-2.15375,-0.043155,1.57266,-2.1791,0.104636,1.42513,-1.81186,-0.079135,1.51286,-1.8805,-0.080448,1.54025,-1.92477,-0.077017,1.52234,-1.99601,0.104384,1.46853,-2.0645,-0.023914,1.1204,-1.77229,-0.148183,1.16566,-1.79033,-0.174921,1.08592,-1.81094,-0.167004,0.953808,-1.80677,-0.262893,0.789159,-1.7864,-0.311401,0.730051,-1.79174,-0.350964,0.857552,-1.79679,-0.394937,0.801462,-1.79773,-0.509184,0.590157,-1.7979,-0.597036,0.668921,-1.85206,-0.508709,0.429481,-1.88081,-0.564177,0.418968,-1.81048,0.224153,1.23899,-2.11507,0.276417,1.1743,-2.09626,0.113635,1.14627,-2.1073,0.165544,1.08555,-2.0929,0.288425,1.00863,-2.09201,0.334343,0.820983,-2.08993,0.220143,0.820402,-2.09845,0.224386,0.751022,-2.09066,0.265048,0.515904,-2.06207,0.201654,0.417607,-1.99816,0.470533,0.500063,-2.04455,0.430565,0.443486,-2.10301,0.019432,1.34827,-1.84198,0.021114,1.38,-2.05485},
/*47*/{0.160098,1.94429,-1.75252,0.198679,1.88719,-1.91437,0.143776,1.79247,-1.63239,0.130575,1.67222,-1.54558,0.237214,1.70154,-1.66623,0.312194,1.71606,-1.71911,0.07163,1.95744,-2.12098,0.188196,1.88607,-1.98859,-0.027792,1.91932,-1.99878,-0.170834,1.85578,-2.12919,-0.209178,1.76667,-2.18172,-0.072405,1.62868,-2.15269,-0.041627,1.57217,-2.17864,0.104975,1.42681,-1.81132,-0.079185,1.5142,-1.87982,-0.080741,1.54223,-1.92566,-0.077161,1.52363,-1.99559,0.104002,1.46923,-2.06378,-0.022335,1.12093,-1.77241,-0.145909,1.16227,-1.78954,-0.171506,1.08533,-1.81274,-0.162637,0.952821,-1.80682,-0.262213,0.789844,-1.78637,-0.311704,0.733055,-1.7925,-0.346499,0.861939,-1.79376,-0.392858,0.806766,-1.79486,-0.512563,0.600187,-1.79635,-0.602013,0.683543,-1.84336,-0.520758,0.441876,-1.88565,-0.574317,0.431284,-1.81375,0.21962,1.2347,-2.11658,0.271907,1.16939,-2.09774,0.108424,1.14398,-2.10817,0.15957,1.0816,-2.0935,0.283341,1.00382,-2.09297,0.33592,0.818734,-2.08957,0.223049,0.813505,-2.10088,0.229672,0.74375,-2.09222,0.279355,0.510509,-2.0623,0.218217,0.412439,-1.99769,0.484499,0.501367,-2.04171,0.44794,0.441815,-2.09925,0.019256,1.35007,-1.84105,0.020825,1.38084,-2.05407},
/*48*/{0.161582,1.94457,-1.75316,0.200124,1.88803,-1.9162,0.143548,1.7931,-1.63367,0.129764,1.6729,-1.54682,0.236341,1.7012,-1.66762,0.310909,1.71279,-1.72017,0.073075,1.95798,-2.12206,0.188133,1.88642,-1.98971,-0.02679,1.92048,-2.00038,-0.170339,1.8551,-2.12807,-0.207544,1.76531,-2.18011,-0.069419,1.62805,-2.15197,-0.039799,1.57101,-2.17771,0.104888,1.42876,-1.81133,-0.078888,1.51561,-1.87957,-0.081143,1.54368,-1.92559,-0.076674,1.52395,-1.99553,0.104889,1.4708,-2.06273,-0.01642,1.12093,-1.77347,-0.142018,1.16042,-1.78989,-0.16726,1.08297,-1.81198,-0.156655,0.950447,-1.80692,-0.259548,0.790117,-1.78724,-0.312461,0.735134,-1.79324,-0.342204,0.865818,-1.79194,-0.390884,0.812358,-1.79354,-0.518735,0.610373,-1.79674,-0.606723,0.697205,-1.83613,-0.532601,0.456653,-1.89041,-0.584143,0.443243,-1.8161,0.216844,1.22965,-2.11742,0.266557,1.16406,-2.09913,0.103199,1.14152,-2.10845,0.152621,1.07875,-2.09348,0.278011,0.998799,-2.09378,0.338712,0.816976,-2.08799,0.226251,0.806624,-2.10394,0.234627,0.73791,-2.09495,0.291757,0.505392,-2.06141,0.233177,0.40701,-1.99665,0.497573,0.503339,-2.03952,0.46324,0.441094,-2.09601,0.019777,1.35168,-1.84044,0.021823,1.38207,-2.0535},
/*49*/{0.162863,1.94485,-1.75438,0.199384,1.88821,-1.91685,0.142953,1.79364,-1.63504,0.128781,1.67267,-1.54707,0.235686,1.69957,-1.66892,0.310354,1.71002,-1.72197,0.074403,1.95786,-2.12357,0.188596,1.88632,-1.99064,-0.025549,1.9209,-2.00187,-0.169358,1.85437,-2.1266,-0.206259,1.76414,-2.18011,-0.068104,1.62619,-2.15091,-0.037832,1.56969,-2.17746,0.104232,1.43064,-1.81093,-0.079088,1.51594,-1.87945,-0.081433,1.54508,-1.926,-0.077103,1.5245,-1.99563,0.106764,1.47154,-2.0619,-0.016291,1.11903,-1.77369,-0.137332,1.1606,-1.78974,-0.162179,1.08066,-1.81296,-0.149553,0.948246,-1.80691,-0.257537,0.789576,-1.78858,-0.313172,0.737091,-1.79504,-0.336665,0.869851,-1.78961,-0.387972,0.8186,-1.79062,-0.521573,0.621894,-1.79607,-0.611452,0.711181,-1.82963,-0.542453,0.470218,-1.89432,-0.591889,0.455571,-1.81933,0.212596,1.22518,-2.11842,0.261905,1.15817,-2.1002,0.097708,1.13962,-2.10754,0.147132,1.07496,-2.09442,0.271921,0.993132,-2.09476,0.339906,0.81484,-2.08622,0.229507,0.801096,-2.10663,0.240591,0.733088,-2.09716,0.302834,0.502508,-2.06174,0.24591,0.403101,-1.99708,0.507752,0.504948,-2.038,0.477038,0.441549,-2.09307,0.019835,1.35297,-1.83979,0.023016,1.38289,-2.05291},
/*50*/{0.163621,1.94417,-1.7558,0.199706,1.88865,-1.91824,0.142407,1.79336,-1.63631,0.126211,1.67348,-1.54853,0.234177,1.6978,-1.66976,0.30831,1.70697,-1.72328,0.075669,1.9572,-2.12433,0.189339,1.88611,-1.99098,-0.024926,1.92177,-2.00237,-0.167231,1.85271,-2.12574,-0.204491,1.76209,-2.17989,-0.065519,1.62449,-2.15046,-0.036297,1.56731,-2.17727,0.104306,1.43141,-1.81051,-0.079754,1.51644,-1.88024,-0.081442,1.54525,-1.92568,-0.077342,1.52413,-1.99581,0.106586,1.4715,-2.06105,-0.01238,1.11717,-1.77575,-0.131533,1.15738,-1.78887,-0.156364,1.07718,-1.813,-0.141467,0.945332,-1.80722,-0.255092,0.789219,-1.78964,-0.313031,0.739768,-1.79608,-0.330347,0.872734,-1.78698,-0.38368,0.824175,-1.78883,-0.527219,0.632245,-1.79525,-0.615505,0.725189,-1.82463,-0.551901,0.486268,-1.89738,-0.598183,0.469628,-1.82142,0.209231,1.22006,-2.11949,0.257497,1.15226,-2.10186,0.092757,1.13617,-2.10759,0.140769,1.0706,-2.09379,0.266342,0.988805,-2.09545,0.341165,0.812834,-2.08501,0.231438,0.795898,-2.10778,0.244768,0.728432,-2.09831,0.311855,0.499024,-2.06271,0.256873,0.399105,-1.99495,0.516367,0.505951,-2.03615,0.487598,0.441753,-2.09167,0.019732,1.35359,-1.83907,0.023204,1.38269,-2.0523},
/*51*/{0.16499,1.9441,-1.75668,0.20132,1.88816,-1.91893,0.141457,1.79347,-1.63728,0.124478,1.67335,-1.55011,0.232775,1.69527,-1.67088,0.307055,1.70314,-1.72538,0.076632,1.95641,-2.12592,0.190314,1.88592,-1.99161,-0.023199,1.92138,-2.0052,-0.166097,1.85064,-2.1254,-0.202667,1.75971,-2.18023,-0.0635,1.62196,-2.15037,-0.033736,1.5646,-2.17699,0.104832,1.43242,-1.80997,-0.079782,1.51594,-1.88051,-0.081442,1.54525,-1.92568,-0.077367,1.52332,-1.99547,0.106406,1.47159,-2.05988,0.004514,1.11913,-1.77544,-0.126687,1.15426,-1.78799,-0.150193,1.07375,-1.81293,-0.13355,0.941381,-1.80687,-0.252703,0.789466,-1.79086,-0.31238,0.743009,-1.79685,-0.324318,0.876384,-1.78467,-0.377988,0.830762,-1.78657,-0.530773,0.645045,-1.79446,-0.617619,0.740227,-1.82081,-0.559494,0.501146,-1.89941,-0.605343,0.483455,-1.82223,0.205447,1.21502,-2.12029,0.25297,1.14649,-2.10272,0.087306,1.13245,-2.10734,0.134906,1.06645,-2.09397,0.260508,0.983475,-2.09538,0.341343,0.810659,-2.08302,0.231788,0.791568,-2.10787,0.247444,0.724226,-2.09856,0.318236,0.4967,-2.06246,0.266049,0.39445,-1.99648,0.523432,0.508103,-2.03643,0.495726,0.441269,-2.08999,0.020463,1.35407,-1.83811,0.02394,1.38232,-2.05145},
/*52*/{0.165338,1.9433,-1.75824,0.200526,1.88763,-1.9197,0.139536,1.79316,-1.63822,0.12195,1.67206,-1.55072,0.231629,1.69242,-1.67216,0.306009,1.69892,-1.72615,0.077757,1.95557,-2.12693,0.190326,1.88555,-1.99206,-0.02308,1.9212,-2.00634,-0.165022,1.84893,-2.12502,-0.200134,1.75684,-2.18046,-0.061235,1.61895,-2.15,-0.030995,1.56122,-2.17685,0.104735,1.43288,-1.8097,-0.079867,1.51574,-1.88097,-0.081586,1.54509,-1.92571,-0.07804,1.52157,-1.99621,0.106693,1.47141,-2.0593,0.009503,1.11771,-1.77801,-0.122008,1.14726,-1.78705,-0.143034,1.06928,-1.81214,-0.125206,0.937652,-1.80739,-0.250055,0.790261,-1.79136,-0.311763,0.74569,-1.79776,-0.317686,0.879942,-1.78387,-0.374713,0.83704,-1.78441,-0.532635,0.65962,-1.79452,-0.618348,0.755658,-1.81916,-0.566678,0.516183,-1.89889,-0.612511,0.499341,-1.82308,0.202032,1.20961,-2.12107,0.247726,1.14067,-2.10391,0.082988,1.12952,-2.10679,0.128984,1.06304,-2.09392,0.254234,0.978836,-2.09527,0.339805,0.809057,-2.08184,0.231264,0.787828,-2.10722,0.246778,0.72018,-2.0981,0.323731,0.493337,-2.06194,0.272002,0.391327,-1.99662,0.527768,0.508408,-2.03693,0.502183,0.440951,-2.09015,0.020796,1.35424,-1.8375,0.024552,1.38157,-2.05096},
/*53*/{0.16545,1.94126,-1.75878,0.200154,1.88722,-1.92003,0.13819,1.79244,-1.63948,0.119802,1.67176,-1.55239,0.230059,1.68916,-1.67348,0.304261,1.69398,-1.72731,0.078817,1.95409,-2.12789,0.190605,1.88423,-1.99346,-0.022692,1.92083,-2.00807,-0.162066,1.84601,-2.1247,-0.197925,1.75338,-2.18142,-0.057645,1.61556,-2.14975,-0.028227,1.55756,-2.17729,0.104587,1.43283,-1.80919,-0.080466,1.5144,-1.88162,-0.082098,1.54414,-1.92616,-0.078152,1.52063,-1.99663,0.106868,1.47077,-2.05854,0.016831,1.11505,-1.77889,-0.116807,1.14264,-1.78691,-0.136678,1.06465,-1.81268,-0.115957,0.932716,-1.80812,-0.246254,0.79203,-1.79184,-0.309321,0.750597,-1.79906,-0.310099,0.88401,-1.78214,-0.36956,0.843239,-1.78462,-0.535846,0.672197,-1.7926,-0.616607,0.772417,-1.82027,-0.572761,0.530855,-1.897,-0.620154,0.516192,-1.82162,0.198651,1.20496,-2.12206,0.244107,1.13507,-2.10503,0.078644,1.12661,-2.1067,0.12434,1.05944,-2.0931,0.248326,0.97441,-2.09501,0.338161,0.807361,-2.08142,0.229172,0.784622,-2.10531,0.245202,0.717435,-2.09694,0.32677,0.491003,-2.06203,0.277545,0.387882,-1.99589,0.531622,0.508197,-2.03674,0.505859,0.440822,-2.09043,0.020791,1.35375,-1.83718,0.025277,1.38071,-2.05068},
/*54*/{0.166056,1.94023,-1.76027,0.200167,1.88529,-1.92105,0.13564,1.79108,-1.64065,0.116036,1.67091,-1.55362,0.227476,1.68561,-1.67388,0.302088,1.68863,-1.72862,0.079944,1.95245,-2.12953,0.190076,1.88287,-1.99334,-0.02154,1.91956,-2.0097,-0.16019,1.84226,-2.12447,-0.194833,1.74886,-2.18221,-0.054859,1.61127,-2.15026,-0.024548,1.55357,-2.17775,0.10467,1.43278,-1.80883,-0.080885,1.51299,-1.88164,-0.082378,1.54265,-1.92693,-0.078526,1.51902,-1.99802,0.107509,1.46909,-2.05817,0.023005,1.11356,-1.77971,-0.111398,1.13889,-1.78754,-0.128897,1.05992,-1.81299,-0.106676,0.928485,-1.80952,-0.24261,0.793605,-1.79248,-0.307993,0.754513,-1.79835,-0.303383,0.887973,-1.78145,-0.362468,0.849321,-1.78286,-0.536304,0.687244,-1.79196,-0.613827,0.789442,-1.82223,-0.578307,0.544162,-1.89338,-0.626715,0.533409,-1.81844,0.195593,1.20026,-2.12275,0.240936,1.12976,-2.10634,0.075363,1.12365,-2.10561,0.119716,1.05653,-2.09236,0.242528,0.969509,-2.09458,0.334844,0.804974,-2.08135,0.225341,0.780898,-2.1034,0.243672,0.713685,-2.09416,0.328198,0.488389,-2.06189,0.277268,0.386176,-1.99803,0.533121,0.507383,-2.03792,0.508562,0.440721,-2.09145,0.021031,1.35321,-1.83654,0.026044,1.3789,-2.05017},
/*55*/{0.166007,1.93918,-1.76145,0.20058,1.88344,-1.92158,0.134475,1.78987,-1.64162,0.113068,1.66893,-1.55482,0.225718,1.68139,-1.67467,0.299729,1.68295,-1.72922,0.080897,1.9504,-2.13044,0.190496,1.88115,-1.99374,-0.021424,1.91862,-2.01226,-0.158681,1.83818,-2.12438,-0.191655,1.74452,-2.18327,-0.05142,1.60723,-2.15137,-0.020526,1.54907,-2.17842,0.104697,1.43235,-1.80864,-0.080849,1.51131,-1.88173,-0.083066,1.54085,-1.92769,-0.078607,1.51636,-1.9976,0.106326,1.46768,-2.05843,0.029403,1.11289,-1.78085,-0.106008,1.1347,-1.78624,-0.122097,1.05348,-1.81194,-0.098132,0.92467,-1.80979,-0.237637,0.79585,-1.79167,-0.303765,0.759649,-1.79943,-0.294972,0.893003,-1.78064,-0.35634,0.856729,-1.78244,-0.536487,0.701573,-1.79164,-0.609437,0.806281,-1.82565,-0.581988,0.557725,-1.88959,-0.634224,0.55187,-1.81492,0.192401,1.19568,-2.12506,0.236875,1.12501,-2.10724,0.071896,1.1208,-2.10533,0.115389,1.05271,-2.09163,0.237513,0.965814,-2.09444,0.33033,0.802721,-2.08275,0.220597,0.776828,-2.10151,0.24053,0.709995,-2.09247,0.328752,0.486323,-2.06161,0.282289,0.382798,-1.99556,0.533614,0.506965,-2.03941,0.507252,0.439633,-2.09275,0.021593,1.35228,-1.83593,0.025939,1.37683,-2.04972},
/*56*/{0.165742,1.93786,-1.762,0.200019,1.8819,-1.92192,0.131008,1.78855,-1.64245,0.109085,1.66738,-1.55625,0.222333,1.67609,-1.67542,0.29839,1.67652,-1.72985,0.082317,1.94803,-2.1312,0.190468,1.87878,-1.99404,-0.021123,1.91698,-2.0127,-0.156804,1.83416,-2.12574,-0.1881,1.73934,-2.18529,-0.048451,1.60229,-2.15222,-0.017219,1.54451,-2.17934,0.10464,1.43184,-1.8084,-0.082252,1.50935,-1.88375,-0.082194,1.53858,-1.92797,-0.079273,1.5142,-1.99913,0.10681,1.46553,-2.05891,0.035602,1.10835,-1.78116,-0.100867,1.12932,-1.78555,-0.115224,1.04842,-1.81148,-0.08882,0.921298,-1.81076,-0.23309,0.798581,-1.79103,-0.301893,0.764516,-1.79691,-0.286935,0.897313,-1.78029,-0.349338,0.863509,-1.78248,-0.536565,0.716118,-1.78952,-0.603137,0.823355,-1.8305,-0.584534,0.571973,-1.88417,-0.639429,0.570094,-1.81153,0.191135,1.19177,-2.12536,0.234339,1.12113,-2.10843,0.068924,1.1188,-2.1045,0.11207,1.05046,-2.09047,0.232431,0.961812,-2.09376,0.32499,0.799094,-2.08316,0.215201,0.77288,-2.10005,0.23599,0.705766,-2.09046,0.326853,0.483197,-2.0603,0.282058,0.378982,-1.9928,0.532626,0.504642,-2.03926,0.505266,0.437424,-2.09399,0.02167,1.35114,-1.83587,0.026693,1.37452,-2.04977},
/*57*/{0.165515,1.93491,-1.76276,0.200159,1.88017,-1.92263,0.128367,1.78681,-1.64294,0.105558,1.66569,-1.55751,0.219666,1.67154,-1.67617,0.29465,1.66988,-1.73006,0.083587,1.94509,-2.1328,0.190596,1.87627,-1.99426,-0.020751,1.9152,-2.01629,-0.154905,1.82932,-2.12609,-0.185354,1.73403,-2.18634,-0.044021,1.59763,-2.15365,-0.01404,1.53939,-2.18086,0.104317,1.43051,-1.8084,-0.082016,1.50725,-1.88383,-0.081637,1.53623,-1.92828,-0.07875,1.5119,-1.99961,0.10606,1.46319,-2.05932,0.041639,1.10614,-1.78054,-0.095091,1.12375,-1.78496,-0.107835,1.04347,-1.8111,-0.080324,0.917414,-1.81063,-0.228038,0.801919,-1.78983,-0.297046,0.770635,-1.79618,-0.277753,0.902759,-1.78064,-0.341927,0.870222,-1.78302,-0.536643,0.730238,-1.78725,-0.596344,0.839156,-1.83567,-0.586716,0.584817,-1.87856,-0.642435,0.588658,-1.80789,0.189325,1.18783,-2.12624,0.231414,1.11712,-2.10993,0.066337,1.11642,-2.10416,0.108986,1.04796,-2.08976,0.228294,0.958447,-2.09387,0.320449,0.795873,-2.08382,0.210485,0.768476,-2.09869,0.231618,0.701712,-2.08879,0.320737,0.478753,-2.05962,0.279089,0.376612,-1.98961,0.529868,0.501032,-2.03983,0.502637,0.434757,-2.09412,0.02184,1.34943,-1.83563,0.026632,1.37202,-2.04962},
/*58*/{0.16453,1.93205,-1.76334,0.200634,1.8764,-1.92287,0.125826,1.78426,-1.64352,0.10208,1.66419,-1.55907,0.216184,1.66674,-1.67671,0.292682,1.66324,-1.7304,0.084066,1.94222,-2.13383,0.190832,1.87378,-1.99461,-0.020347,1.91214,-2.01792,-0.152761,1.82357,-2.1281,-0.182221,1.72796,-2.18876,-0.040386,1.59186,-2.15579,-0.009464,1.53354,-2.18195,0.104538,1.42871,-1.80874,-0.082105,1.50489,-1.88373,-0.082742,1.53374,-1.92902,-0.079608,1.50893,-2.00001,0.106768,1.46088,-2.05979,0.044857,1.10484,-1.7817,-0.090028,1.11792,-1.78483,-0.101754,1.03772,-1.81198,-0.072121,0.91428,-1.81051,-0.222951,0.804958,-1.78822,-0.292812,0.776447,-1.79399,-0.269413,0.90728,-1.78161,-0.334748,0.877452,-1.78332,-0.533184,0.745203,-1.78508,-0.588396,0.854234,-1.84087,-0.586789,0.598937,-1.87307,-0.644713,0.606931,-1.80364,0.18718,1.18456,-2.12799,0.228755,1.11256,-2.1113,0.064079,1.11505,-2.10402,0.105422,1.0455,-2.08903,0.225968,0.95569,-2.0932,0.314302,0.792053,-2.08468,0.205192,0.764713,-2.09752,0.226736,0.698245,-2.08798,0.319478,0.475922,-2.05796,0.274564,0.373331,-1.98926,0.525421,0.497027,-2.04079,0.497646,0.42991,-2.0938,0.022271,1.34742,-1.83566,0.026935,1.36951,-2.0497},
/*59*/{0.163495,1.92923,-1.76335,0.200155,1.87364,-1.92331,0.122698,1.78242,-1.64411,0.097048,1.66085,-1.56008,0.212536,1.66108,-1.67691,0.288604,1.65516,-1.73013,0.084176,1.93892,-2.13441,0.191122,1.87067,-1.99498,-0.021332,1.90994,-2.0194,-0.150815,1.81717,-2.13015,-0.179307,1.72125,-2.19107,-0.037182,1.58606,-2.15753,-0.006131,1.52765,-2.18339,0.104627,1.42733,-1.80869,-0.082702,1.50179,-1.88458,-0.082864,1.53041,-1.92944,-0.080316,1.5057,-2.00054,0.105079,1.45786,-2.06051,0.050539,1.10193,-1.78187,-0.084585,1.11209,-1.78437,-0.094622,1.03194,-1.81226,-0.063252,0.910725,-1.81092,-0.216949,0.808723,-1.78698,-0.286394,0.782319,-1.79262,-0.260318,0.912251,-1.78272,-0.325232,0.885191,-1.78441,-0.528359,0.76051,-1.78671,-0.579834,0.868586,-1.84558,-0.585624,0.61178,-1.86827,-0.644594,0.624444,-1.79935,0.185425,1.18144,-2.12853,0.225767,1.10913,-2.11304,0.061738,1.11257,-2.10235,0.103248,1.04386,-2.08854,0.223876,0.953138,-2.09304,0.309646,0.787463,-2.0852,0.200389,0.76068,-2.09695,0.220978,0.694273,-2.08704,0.312685,0.471168,-2.0568,0.268141,0.369589,-1.98533,0.520008,0.49049,-2.03926,0.491828,0.424002,-2.09197,0.022584,1.3454,-1.83533,0.026451,1.3661,-2.04954},
/*60*/{0.162156,1.92692,-1.76392,0.200339,1.87001,-1.92354,0.120351,1.77901,-1.64403,0.092562,1.65869,-1.5612,0.208422,1.65586,-1.67731,0.284858,1.64718,-1.72912,0.084219,1.93527,-2.13636,0.191404,1.86766,-1.9955,-0.020293,1.90667,-2.02173,-0.14891,1.81048,-2.13133,-0.176212,1.71427,-2.19329,-0.032132,1.58046,-2.16061,-0.001709,1.52129,-2.1854,0.104611,1.42513,-1.80847,-0.082688,1.49922,-1.88416,-0.082462,1.5275,-1.92943,-0.080736,1.50188,-2.00069,0.104875,1.45517,-2.06162,0.058017,1.09892,-1.77993,-0.079628,1.10732,-1.78417,-0.087108,1.02586,-1.81028,-0.05422,0.907777,-1.80971,-0.210239,0.811705,-1.78544,-0.281597,0.787544,-1.78969,-0.250537,0.917009,-1.78407,-0.316513,0.891655,-1.7851,-0.521548,0.774078,-1.78699,-0.571013,0.881192,-1.84921,-0.583084,0.624272,-1.86193,-0.642717,0.641879,-1.79697,0.183433,1.17822,-2.12975,0.223571,1.10558,-2.11367,0.05953,1.11061,-2.1013,0.100399,1.04116,-2.08826,0.221917,0.950582,-2.09266,0.305757,0.782756,-2.08524,0.19547,0.757143,-2.09594,0.215471,0.690241,-2.08597,0.305021,0.465839,-2.05544,0.260208,0.364914,-1.98364,0.513203,0.484203,-2.0393,0.484265,0.416044,-2.09002,0.023157,1.34297,-1.83507,0.026389,1.36296,-2.04935},
/*61*/{0.161836,1.92375,-1.76414,0.20015,1.86703,-1.92342,0.117061,1.7765,-1.64518,0.087356,1.65596,-1.56347,0.204131,1.64942,-1.67746,0.281071,1.63876,-1.72876,0.085059,1.93106,-2.13721,0.190919,1.86414,-1.99591,-0.020668,1.90336,-2.02368,-0.148911,1.80362,-2.13465,-0.173477,1.70672,-2.19572,-0.028657,1.57334,-2.16269,0.002694,1.51461,-2.18687,0.10553,1.42315,-1.80889,-0.082639,1.49561,-1.88409,-0.083275,1.52347,-1.92944,-0.081551,1.49804,-2.00041,0.104526,1.45269,-2.06213,0.062656,1.09598,-1.77905,-0.074822,1.10363,-1.7843,-0.080229,1.0205,-1.81139,-0.045475,0.903806,-1.80855,-0.203046,0.814764,-1.7844,-0.275728,0.791602,-1.78937,-0.240821,0.921578,-1.78687,-0.307579,0.89797,-1.78588,-0.516288,0.788293,-1.78561,-0.561714,0.893786,-1.85347,-0.578933,0.636822,-1.85769,-0.639447,0.657697,-1.79307,0.181939,1.17471,-2.13067,0.221175,1.10237,-2.11466,0.057427,1.10857,-2.10116,0.097878,1.03861,-2.08799,0.219878,0.947505,-2.09255,0.300921,0.777666,-2.0857,0.190636,0.753249,-2.09547,0.209792,0.686298,-2.08542,0.297087,0.460617,-2.05431,0.250341,0.361621,-1.98147,0.504868,0.475674,-2.03817,0.475367,0.40731,-2.08752,0.024625,1.34028,-1.83503,0.026786,1.35984,-2.04936},
/*62*/{0.160805,1.9202,-1.76401,0.199964,1.86308,-1.92413,0.114056,1.77341,-1.64514,0.081731,1.65306,-1.56497,0.199566,1.64325,-1.67753,0.277426,1.62979,-1.72702,0.085599,1.92684,-2.13786,0.190735,1.86046,-1.9961,-0.020796,1.89895,-2.02608,-0.146029,1.79566,-2.13742,-0.17013,1.69866,-2.19842,-0.025075,1.56622,-2.16597,0.007281,1.50765,-2.18928,0.106088,1.42066,-1.8096,-0.082959,1.49125,-1.88389,-0.083874,1.52042,-1.92945,-0.082575,1.49456,-2.00044,0.103601,1.45002,-2.06301,0.067433,1.09375,-1.77864,-0.068786,1.09786,-1.78408,-0.074166,1.01525,-1.81135,-0.036627,0.899917,-1.8073,-0.195788,0.818249,-1.78231,-0.269314,0.796385,-1.78719,-0.229376,0.925279,-1.78831,-0.298402,0.903483,-1.78847,-0.508742,0.799285,-1.78537,-0.551243,0.90493,-1.85666,-0.574287,0.648249,-1.854,-0.634569,0.672279,-1.79005,0.179534,1.1717,-2.13108,0.21925,1.09889,-2.11554,0.055331,1.10577,-2.10104,0.095638,1.03529,-2.08831,0.217456,0.944251,-2.0923,0.295978,0.772437,-2.08573,0.186381,0.747517,-2.09078,0.20439,0.682135,-2.08507,0.288597,0.455897,-2.05332,0.239786,0.359029,-1.97976,0.496442,0.467006,-2.03587,0.464105,0.397583,-2.08486,0.025442,1.33722,-1.83553,0.026453,1.35687,-2.04986},
/*63*/{0.160152,1.91641,-1.76404,0.199532,1.85839,-1.92402,0.11073,1.77047,-1.64524,0.076997,1.64991,-1.56539,0.195309,1.63655,-1.67749,0.272779,1.62089,-1.72582,0.086134,1.9222,-2.13915,0.190806,1.85627,-1.99649,-0.02078,1.89505,-2.02793,-0.145451,1.78771,-2.14067,-0.166522,1.69046,-2.20151,-0.021162,1.55871,-2.16948,0.011513,1.50061,-2.19048,0.105812,1.41764,-1.81018,-0.082879,1.48765,-1.884,-0.082933,1.51601,-1.92935,-0.083241,1.49039,-1.99938,0.103121,1.44674,-2.06463,0.07193,1.0904,-1.77682,-0.065774,1.09121,-1.78549,-0.065832,1.00922,-1.81098,-0.028045,0.896457,-1.80581,-0.187148,0.821239,-1.78056,-0.260791,0.801615,-1.78564,-0.218209,0.928057,-1.78937,-0.287612,0.909072,-1.7902,-0.501723,0.811413,-1.78641,-0.541348,0.915448,-1.85822,-0.56846,0.659219,-1.85044,-0.627712,0.685027,-1.78641,0.178507,1.16889,-2.13187,0.217991,1.0951,-2.1165,0.053514,1.10296,-2.09983,0.093777,1.03273,-2.08764,0.215277,0.940667,-2.09239,0.290981,0.765422,-2.08556,0.179853,0.746001,-2.09614,0.197559,0.677692,-2.08518,0.277794,0.449924,-2.05257,0.22823,0.354035,-1.97765,0.485878,0.457458,-2.03269,0.454643,0.388262,-2.08103,0.026254,1.33369,-1.83582,0.026151,1.35318,-2.05017},
/*64*/{0.158518,1.91147,-1.76382,0.199699,1.85374,-1.92354,0.107329,1.76674,-1.64587,0.070396,1.64686,-1.56718,0.189893,1.63001,-1.67789,0.267921,1.61141,-1.72428,0.086558,1.91713,-2.14026,0.191527,1.85205,-1.99681,-0.021671,1.89025,-2.02983,-0.143457,1.77894,-2.14422,-0.16327,1.68141,-2.20524,-0.015362,1.55178,-2.17246,0.016848,1.49284,-2.19265,0.105634,1.41514,-1.81056,-0.083303,1.48318,-1.88363,-0.083539,1.51214,-1.9291,-0.08421,1.486,-1.99869,0.102295,1.44362,-2.06531,0.076734,1.08839,-1.77584,-0.060492,1.08588,-1.78578,-0.060597,1.00353,-1.81093,-0.018797,0.892924,-1.80388,-0.179412,0.823783,-1.77968,-0.252153,0.806221,-1.78492,-0.207347,0.931648,-1.79263,-0.27636,0.913349,-1.792,-0.492454,0.82009,-1.78436,-0.531539,0.923988,-1.85946,-0.561891,0.668485,-1.8469,-0.620421,0.697067,-1.78355,0.17643,1.16509,-2.13261,0.216349,1.09201,-2.11668,0.051936,1.09992,-2.09993,0.091882,1.02923,-2.08794,0.212656,0.935909,-2.09249,0.286188,0.760505,-2.08436,0.174016,0.741502,-2.09501,0.189989,0.673444,-2.08506,0.266188,0.443195,-2.05082,0.213176,0.350649,-1.97547,0.474524,0.447543,-2.02982,0.442769,0.378449,-2.07806,0.027009,1.33041,-1.83571,0.02591,1.34959,-2.05009},
/*65*/{0.157032,1.90776,-1.76408,0.200699,1.84958,-1.92328,0.103678,1.76218,-1.64577,0.065619,1.64423,-1.56857,0.184679,1.62252,-1.67733,0.263245,1.60176,-1.72328,0.086789,1.912,-2.14178,0.191309,1.84736,-1.99701,-0.02196,1.88544,-2.03225,-0.142824,1.77028,-2.14816,-0.160068,1.67269,-2.20827,-0.011273,1.54353,-2.17584,0.022092,1.48513,-2.19377,0.105904,1.4118,-1.81155,-0.084027,1.47867,-1.88397,-0.083694,1.50775,-1.92923,-0.084531,1.48073,-1.99806,0.101517,1.44047,-2.06693,0.081429,1.08466,-1.77319,-0.055087,1.08009,-1.78624,-0.053784,0.997766,-1.81091,-0.01003,0.889389,-1.80226,-0.169114,0.825774,-1.77701,-0.244017,0.809892,-1.78268,-0.19521,0.933724,-1.79525,-0.265366,0.917867,-1.7944,-0.48176,0.830164,-1.78555,-0.521941,0.933063,-1.86082,-0.554272,0.677428,-1.84445,-0.61176,0.707425,-1.78076,0.174896,1.16153,-2.13274,0.21471,1.08825,-2.11758,0.050281,1.09646,-2.09971,0.090624,1.02533,-2.08786,0.209704,0.930625,-2.09265,0.279703,0.75391,-2.08385,0.167936,0.737105,-2.09536,0.182299,0.668828,-2.08477,0.254869,0.438389,-2.05011,0.196985,0.349465,-1.97567,0.462559,0.43715,-2.02687,0.428313,0.367829,-2.07432,0.02811,1.3265,-1.83641,0.02585,1.34573,-2.05078},
/*66*/{0.156295,1.90374,-1.76376,0.199787,1.84401,-1.92321,0.100587,1.75827,-1.64609,0.05874,1.64038,-1.56969,0.178925,1.61518,-1.67674,0.257617,1.59185,-1.72179,0.087433,1.90624,-2.14283,0.191256,1.84256,-1.99749,-0.022687,1.88016,-2.03371,-0.142014,1.76076,-2.15297,-0.156594,1.66303,-2.2116,-0.006799,1.5344,-2.17814,0.027722,1.47738,-2.19566,0.106417,1.40855,-1.8129,-0.083522,1.47376,-1.88331,-0.084436,1.50355,-1.92945,-0.085775,1.47601,-1.99749,0.100689,1.43743,-2.0681,0.085808,1.08236,-1.77188,-0.049978,1.07634,-1.78785,-0.046611,0.993056,-1.81245,-0.001074,0.886711,-1.79989,-0.159604,0.827848,-1.77561,-0.234041,0.812928,-1.7801,-0.183339,0.935157,-1.79686,-0.253258,0.920519,-1.79665,-0.472058,0.83777,-1.78417,-0.511302,0.941138,-1.86087,-0.544678,0.685242,-1.84299,-0.601958,0.716352,-1.77895,0.173045,1.1572,-2.13325,0.212781,1.08461,-2.11759,0.049131,1.09212,-2.10004,0.089317,1.02084,-2.08804,0.207126,0.925255,-2.09288,0.273862,0.747153,-2.08338,0.161826,0.732504,-2.09524,0.174542,0.664281,-2.08525,0.24377,0.435314,-2.04908,0.177988,0.355551,-1.97587,0.44704,0.424559,-2.02484,0.410418,0.357291,-2.07203,0.029509,1.32265,-1.83715,0.02557,1.3421,-2.05147},
/*67*/{0.155222,1.8986,-1.76356,0.198668,1.83804,-1.92283,0.096896,1.75374,-1.64646,0.051537,1.6381,-1.57051,0.17339,1.6079,-1.67698,0.251675,1.58175,-1.71964,0.087656,1.90071,-2.14405,0.191632,1.83717,-1.99715,-0.023275,1.87419,-2.03558,-0.140104,1.7512,-2.15573,-0.15216,1.65357,-2.21526,-0.001167,1.52759,-2.18127,0.03337,1.46906,-2.19705,0.105979,1.40468,-1.81326,-0.083589,1.46832,-1.88273,-0.085228,1.49887,-1.92879,-0.086223,1.47058,-1.99642,0.101046,1.43457,-2.06963,0.090294,1.07833,-1.77038,-0.0449,1.07011,-1.78823,-0.03948,0.987019,-1.81177,0.007641,0.883938,-1.79794,-0.149616,0.829542,-1.77478,-0.224056,0.816705,-1.77818,-0.169928,0.936447,-1.7986,-0.241348,0.92403,-1.79851,-0.461196,0.844407,-1.78542,-0.500458,0.947913,-1.86163,-0.535068,0.692848,-1.84172,-0.591161,0.724122,-1.77632,0.171973,1.15331,-2.1334,0.211447,1.08022,-2.11829,0.047705,1.0878,-2.10062,0.088465,1.01664,-2.08786,0.204594,0.919312,-2.09297,0.267652,0.740632,-2.08281,0.154915,0.727951,-2.09501,0.16646,0.658857,-2.0849,0.227609,0.424874,-2.05029,0.160516,0.352359,-1.97504,0.432278,0.409113,-2.02282,0.392418,0.344352,-2.07032,0.030707,1.31806,-1.83782,0.02604,1.3385,-2.05203},
/*68*/{0.153779,1.89357,-1.7632,0.199478,1.83281,-1.92236,0.093438,1.74917,-1.64703,0.045532,1.6339,-1.57096,0.167301,1.60024,-1.67609,0.245261,1.57113,-1.71737,0.088266,1.89523,-2.14572,0.191961,1.83312,-1.99688,-0.023534,1.86866,-2.03827,-0.138277,1.74088,-2.15996,-0.147939,1.64361,-2.21876,0.004432,1.51869,-2.18333,0.039866,1.46088,-2.19838,0.106643,1.40044,-1.8141,-0.084272,1.46333,-1.88251,-0.086337,1.49349,-1.9301,-0.087875,1.46537,-1.99583,0.099776,1.4304,-2.07134,0.094057,1.07668,-1.76974,-0.040219,1.06306,-1.79014,-0.032745,0.981187,-1.81251,0.016976,0.881112,-1.79622,-0.139281,0.829774,-1.77366,-0.212822,0.817995,-1.77769,-0.157413,0.936863,-1.80134,-0.228872,0.925645,-1.79955,-0.449761,0.850703,-1.78316,-0.490012,0.954245,-1.86116,-0.524656,0.698099,-1.83992,-0.580048,0.730772,-1.77455,0.169757,1.14875,-2.13345,0.210259,1.07634,-2.11949,0.046037,1.08324,-2.10049,0.086821,1.01262,-2.08819,0.201272,0.913508,-2.09271,0.260681,0.733615,-2.0815,0.14773,0.723512,-2.0952,0.157271,0.653733,-2.08401,0.212878,0.425419,-2.04615,0.138053,0.355908,-1.97377,0.417881,0.395314,-2.01613,0.373416,0.338793,-2.06859,0.031389,1.31361,-1.83856,0.025211,1.33381,-2.05276},
/*69*/{0.153108,1.88843,-1.76285,0.199672,1.82785,-1.92206,0.089725,1.74462,-1.647,0.039283,1.63125,-1.57187,0.161247,1.59235,-1.67491,0.239054,1.56126,-1.71543,0.089598,1.88926,-2.14716,0.192187,1.82709,-1.99697,-0.024598,1.8636,-2.03998,-0.13507,1.73099,-2.16413,-0.143438,1.63336,-2.22263,0.009808,1.50934,-2.18521,0.04607,1.45239,-2.19923,0.106549,1.39606,-1.8154,-0.085038,1.4574,-1.88249,-0.086363,1.4883,-1.92972,-0.08921,1.45999,-1.99574,0.099522,1.42602,-2.07368,0.09896,1.07214,-1.76606,-0.035185,1.05661,-1.78964,-0.026128,0.975327,-1.81277,0.027228,0.878294,-1.79388,-0.127774,0.829829,-1.77219,-0.203616,0.81994,-1.7767,-0.143929,0.937881,-1.80224,-0.215818,0.92705,-1.80051,-0.436337,0.856612,-1.78386,-0.477826,0.959315,-1.86103,-0.512797,0.703802,-1.83777,-0.568166,0.736188,-1.77284,0.167956,1.14443,-2.13419,0.209668,1.07004,-2.1184,0.043777,1.07847,-2.10055,0.08429,1.00776,-2.08799,0.197512,0.908302,-2.09273,0.252976,0.725827,-2.07939,0.141382,0.718897,-2.0948,0.149368,0.649566,-2.08355,0.194777,0.422542,-2.03957,0.11734,0.353807,-1.97215,0.398042,0.389045,-2.00684,0.351447,0.340964,-2.06864,0.032182,1.30858,-1.83971,0.024718,1.32919,-2.05383},
/*70*/{0.152139,1.883,-1.76242,0.200546,1.82257,-1.921,0.086379,1.7398,-1.6478,0.033012,1.62756,-1.57287,0.154987,1.58453,-1.67445,0.231715,1.55109,-1.71306,0.090461,1.88322,-2.14833,0.192082,1.82177,-1.99707,-0.02547,1.85676,-2.04229,-0.133357,1.7206,-2.16933,-0.138381,1.62281,-2.22628,0.017115,1.50036,-2.1872,0.053096,1.44358,-2.20047,0.107241,1.39182,-1.81685,-0.086128,1.45254,-1.8814,-0.087653,1.48287,-1.92984,-0.091217,1.45563,-1.99528,0.09777,1.42195,-2.07558,0.104061,1.06885,-1.76496,-0.030399,1.05279,-1.79109,-0.018517,0.96942,-1.81227,0.036362,0.87558,-1.79188,-0.11634,0.830048,-1.771,-0.190789,0.820629,-1.77389,-0.130637,0.937296,-1.80348,-0.203111,0.928233,-1.80148,-0.424947,0.859247,-1.78366,-0.466409,0.963633,-1.86064,-0.500672,0.707863,-1.83663,-0.555416,0.741565,-1.77095,0.165027,1.14013,-2.13603,0.20488,1.06648,-2.12072,0.040663,1.07509,-2.10101,0.080671,1.00463,-2.08788,0.192795,0.902875,-2.09301,0.245899,0.718903,-2.075,0.134238,0.714906,-2.09372,0.140307,0.6459,-2.08163,0.178938,0.424223,-2.03385,0.093511,0.353913,-1.97144,0.374619,0.387128,-2.00663,0.328501,0.342432,-2.06724,0.032721,1.30408,-1.84069,0.023075,1.32491,-2.0547},
/*71*/{0.150978,1.87769,-1.76203,0.200207,1.81597,-1.92084,0.082681,1.73451,-1.64823,0.026408,1.62437,-1.57333,0.148376,1.57698,-1.67322,0.225133,1.54181,-1.71048,0.091722,1.87709,-2.14986,0.192738,1.81635,-1.99632,-0.026041,1.85112,-2.04423,-0.129551,1.70959,-2.17388,-0.133226,1.61234,-2.22956,0.022606,1.4913,-2.18856,0.060128,1.43475,-2.20139,0.106771,1.38677,-1.81804,-0.08692,1.44675,-1.88121,-0.089383,1.47731,-1.93005,-0.09332,1.45141,-1.99533,0.095553,1.41771,-2.0768,0.108359,1.06621,-1.76345,-0.02529,1.04712,-1.79175,-0.011605,0.963498,-1.81274,0.046994,0.87221,-1.78859,-0.103724,0.829436,-1.76957,-0.177726,0.820931,-1.77335,-0.11702,0.93598,-1.80495,-0.187882,0.92823,-1.80053,-0.410856,0.864007,-1.78397,-0.453797,0.967053,-1.85986,-0.487831,0.711976,-1.83496,-0.541503,0.744273,-1.76966,0.161238,1.1365,-2.13951,0.201143,1.06271,-2.12432,0.036812,1.07359,-2.1009,0.076906,1.00285,-2.08704,0.187967,0.897019,-2.09369,0.240704,0.713837,-2.07059,0.128789,0.713598,-2.09236,0.13105,0.644172,-2.07749,0.151529,0.423158,-2.02953,0.070417,0.354202,-1.96863,0.349949,0.387525,-2.00273,0.304924,0.341454,-2.06715,0.032244,1.2988,-1.84206,0.021268,1.32055,-2.05592},
/*72*/{0.150477,1.87256,-1.76112,0.200568,1.80969,-1.91942,0.078402,1.72961,-1.64874,0.019888,1.62146,-1.57387,0.141325,1.56944,-1.6719,0.217954,1.53189,-1.70782,0.092946,1.87174,-2.15154,0.193952,1.81057,-1.99676,-0.025457,1.8457,-2.0452,-0.127193,1.69833,-2.17919,-0.126807,1.6016,-2.23362,0.030631,1.4822,-2.19016,0.067503,1.42654,-2.20202,0.106104,1.38137,-1.81888,-0.08959,1.44135,-1.88047,-0.091473,1.47129,-1.92922,-0.097062,1.44687,-1.99515,0.09361,1.41452,-2.07799,0.112731,1.06326,-1.7618,-0.018245,1.04022,-1.79155,-0.004713,0.957852,-1.81361,0.057596,0.86949,-1.78796,-0.091053,0.828252,-1.76865,-0.166432,0.820187,-1.77206,-0.102908,0.934513,-1.80393,-0.175372,0.927757,-1.80186,-0.39834,0.866954,-1.78258,-0.44169,0.969426,-1.85963,-0.473772,0.713796,-1.8331,-0.527507,0.746884,-1.76785,0.157026,1.13401,-2.14463,0.196857,1.0594,-2.12882,0.033757,1.07299,-2.10053,0.07294,1.00229,-2.08666,0.18134,0.892684,-2.09445,0.234144,0.71174,-2.06614,0.122939,0.716429,-2.08828,0.122207,0.647234,-2.07303,0.128096,0.424406,-2.02527,0.048287,0.35612,-1.96607,0.329503,0.383732,-2.00245,0.282736,0.342814,-2.0654,0.031386,1.29335,-1.84353,0.019168,1.31685,-2.05713},
/*73*/{0.149454,1.86677,-1.76095,0.20252,1.8045,-1.91916,0.07429,1.72479,-1.6492,0.012876,1.61847,-1.5751,0.134264,1.56248,-1.6699,0.209697,1.52341,-1.70518,0.094395,1.86638,-2.15355,0.194558,1.80515,-1.99608,-0.024355,1.84362,-2.04533,-0.122707,1.68779,-2.18528,-0.120788,1.59041,-2.23715,0.037798,1.47327,-2.19161,0.075654,1.41867,-2.20276,0.10494,1.37684,-1.82074,-0.091463,1.43601,-1.88028,-0.093506,1.46561,-1.92847,-0.099905,1.44356,-1.99495,0.090313,1.41142,-2.07907,0.117398,1.05983,-1.76097,-0.013396,1.0346,-1.79181,0.004729,0.952022,-1.81363,0.069605,0.866322,-1.78575,-0.077682,0.826428,-1.76787,-0.152463,0.818892,-1.77215,-0.088604,0.93298,-1.80493,-0.159273,0.927297,-1.79972,-0.384378,0.865858,-1.78237,-0.428736,0.97094,-1.85935,-0.459247,0.714817,-1.83327,-0.513163,0.747264,-1.7666,0.153619,1.13097,-2.14843,0.192984,1.05713,-2.13139,0.031083,1.07345,-2.10046,0.070683,1.00143,-2.08628,0.17753,0.891822,-2.09474,0.226166,0.710352,-2.06395,0.115318,0.721115,-2.08493,0.110094,0.652088,-2.07042,0.113543,0.428076,-2.02425,0.027035,0.356611,-1.9674,0.308231,0.383411,-2.00028,0.259612,0.341678,-2.06452,0.030459,1.28839,-1.84536,0.016513,1.31354,-2.05866},
/*74*/{0.148476,1.86205,-1.75955,0.202141,1.79843,-1.91794,0.071031,1.72071,-1.64959,0.005666,1.61607,-1.57652,0.127636,1.55637,-1.66846,0.201318,1.51464,-1.70226,0.095677,1.86117,-2.15554,0.195917,1.79935,-1.99599,-0.023595,1.84098,-2.04559,-0.117731,1.67735,-2.19197,-0.113375,1.57952,-2.24067,0.045782,1.46565,-2.19253,0.085057,1.41115,-2.20328,0.103725,1.37194,-1.82339,-0.093663,1.43172,-1.87998,-0.096456,1.46114,-1.92745,-0.102731,1.43943,-1.99386,0.087201,1.40855,-2.08053,0.12201,1.05797,-1.76021,-0.006863,1.02833,-1.79242,0.012706,0.94677,-1.81363,0.082074,0.863916,-1.78522,-0.06491,0.824115,-1.76745,-0.139473,0.819393,-1.77234,-0.073397,0.931517,-1.80275,-0.144951,0.927182,-1.79956,-0.370197,0.867901,-1.78268,-0.415819,0.971142,-1.85905,-0.44393,0.715075,-1.83226,-0.498246,0.748097,-1.76504,0.151009,1.12972,-2.14969,0.190807,1.0561,-2.13175,0.028993,1.07123,-2.09949,0.069274,1.00052,-2.08517,0.176271,0.892344,-2.09472,0.215749,0.709084,-2.06295,0.105661,0.725881,-2.08401,0.097681,0.657043,-2.06926,0.095102,0.430313,-2.02165,0.003836,0.358283,-1.96607,0.285751,0.381957,-1.99953,0.237401,0.341806,-2.06452,0.02921,1.28358,-1.84731,0.013136,1.31045,-2.06026},
/*75*/{0.148395,1.85805,-1.75853,0.200939,1.79464,-1.9166,0.067171,1.71682,-1.65002,-0.001543,1.61425,-1.5776,0.120345,1.55028,-1.6671,0.193373,1.50709,-1.69951,0.097705,1.85602,-2.15669,0.196287,1.79492,-1.99544,-0.023209,1.8387,-2.04507,-0.109403,1.66721,-2.19921,-0.105687,1.56929,-2.24393,0.0548,1.45803,-2.19331,0.094411,1.40424,-2.20343,0.102567,1.36754,-1.82551,-0.096038,1.42687,-1.87851,-0.098758,1.45768,-1.92757,-0.105655,1.43665,-1.9922,0.085191,1.40615,-2.08226,0.126645,1.05888,-1.76018,-0.001582,1.02036,-1.79298,0.022509,0.941981,-1.81392,0.095093,0.861483,-1.78524,-0.051231,0.821559,-1.76749,-0.125509,0.817511,-1.77311,-0.059197,0.929916,-1.80084,-0.130531,0.926027,-1.79696,-0.356001,0.868062,-1.78195,-0.403381,0.970522,-1.85922,-0.428954,0.71366,-1.83073,-0.483341,0.745998,-1.76628,0.14648,1.12854,-2.14919,0.187848,1.05522,-2.13181,0.026204,1.06821,-2.09804,0.067289,0.999023,-2.08441,0.177423,0.893526,-2.09345,0.20381,0.708387,-2.06448,0.094287,0.729151,-2.08246,0.084563,0.660485,-2.06899,0.07788,0.430898,-2.01977,-0.017945,0.357355,-1.96413,0.265336,0.381474,-1.99902,0.216015,0.342406,-2.06313,0.027964,1.27914,-1.84928,0.010209,1.30814,-2.06181},
/*76*/{0.147976,1.85369,-1.7574,0.201374,1.79023,-1.91621,0.063144,1.71426,-1.65004,-0.009334,1.61307,-1.57854,0.112964,1.5453,-1.6654,0.184789,1.50017,-1.69682,0.098641,1.85144,-2.15768,0.196324,1.79067,-1.99434,-0.022383,1.83714,-2.04614,-0.102779,1.65709,-2.20575,-0.095022,1.55876,-2.24642,0.06437,1.45167,-2.19381,0.104448,1.39849,-2.20369,0.101475,1.36414,-1.82758,-0.097517,1.42263,-1.87716,-0.099458,1.45664,-1.92658,-0.108414,1.43427,-1.99023,0.082612,1.40465,-2.08377,0.132987,1.06084,-1.75961,0.00497,1.01983,-1.79208,0.031281,0.939859,-1.81333,0.109581,0.86049,-1.78458,-0.036594,0.81968,-1.76808,-0.11321,0.814949,-1.77411,-0.045277,0.928855,-1.79973,-0.116668,0.925011,-1.79699,-0.343119,0.865973,-1.7807,-0.391538,0.968455,-1.85725,-0.41195,0.710946,-1.83048,-0.467811,0.742757,-1.76539,0.143198,1.12715,-2.14878,0.18483,1.05438,-2.13243,0.022643,1.06657,-2.09772,0.064379,0.996557,-2.0839,0.178154,0.894285,-2.09172,0.193991,0.707738,-2.06739,0.082969,0.730641,-2.08213,0.071291,0.66256,-2.06908,0.056348,0.43207,-2.02103,-0.040417,0.359436,-1.96491,0.242494,0.380097,-1.99995,0.193959,0.341795,-2.06299,0.027022,1.27569,-1.85102,0.00704,1.30674,-2.06306},
/*77*/{0.147389,1.8503,-1.75652,0.20102,1.78717,-1.91561,0.057897,1.71259,-1.65096,-0.017168,1.61305,-1.58058,0.10507,1.54154,-1.66394,0.175311,1.49452,-1.69389,0.100943,1.84803,-2.15742,0.196227,1.78694,-1.99366,-0.022185,1.83457,-2.0464,-0.095746,1.64942,-2.2118,-0.084568,1.54976,-2.24836,0.074383,1.44591,-2.19392,0.115418,1.39277,-2.20323,0.100595,1.35968,-1.829,-0.099421,1.42024,-1.87529,-0.100731,1.45473,-1.92592,-0.109384,1.43202,-1.98806,0.080778,1.40352,-2.0854,0.136916,1.06441,-1.76307,0.013322,1.01772,-1.79043,0.041187,0.938367,-1.81237,0.123996,0.860344,-1.7848,-0.022791,0.818174,-1.76827,-0.097486,0.814875,-1.77289,-0.030649,0.927434,-1.79801,-0.101615,0.924181,-1.79451,-0.328567,0.864274,-1.78185,-0.380637,0.965339,-1.85625,-0.394785,0.708399,-1.83018,-0.450308,0.737238,-1.76554,0.138776,1.12492,-2.14918,0.179918,1.05181,-2.13381,0.018259,1.06433,-2.09644,0.05947,0.994342,-2.08378,0.178918,0.894858,-2.08947,0.180984,0.704209,-2.07057,0.071095,0.730739,-2.08351,0.058172,0.663448,-2.06868,0.038126,0.432171,-2.01883,-0.061588,0.360507,-1.96377,0.222238,0.379601,-1.99899,0.171619,0.341202,-2.06353,0.02568,1.27201,-1.85267,0.004221,1.30555,-2.0642},
/*78*/{0.147369,1.84712,-1.75548,0.202182,1.78438,-1.91488,0.053706,1.71134,-1.65122,-0.024736,1.6142,-1.58244,0.097369,1.53884,-1.66312,0.167512,1.49066,-1.6913,0.102874,1.84509,-2.15701,0.197007,1.78465,-1.99322,-0.021737,1.83313,-2.04704,-0.088149,1.64136,-2.21664,-0.074002,1.54202,-2.25049,0.084764,1.44178,-2.19339,0.126891,1.38904,-2.20279,0.099093,1.35634,-1.82955,-0.100485,1.41878,-1.87426,-0.102471,1.45423,-1.92605,-0.111354,1.43131,-1.98592,0.078726,1.40247,-2.08694,0.143459,1.06703,-1.7631,0.020731,1.01637,-1.79008,0.051116,0.938289,-1.81026,0.1385,0.861838,-1.78449,-0.00826,0.817,-1.76838,-0.083468,0.813235,-1.77382,-0.017719,0.92716,-1.79595,-0.088201,0.923165,-1.79264,-0.314328,0.858186,-1.78032,-0.371108,0.960955,-1.85483,-0.376547,0.703215,-1.82953,-0.433413,0.730299,-1.76477,0.133806,1.12143,-2.14986,0.174129,1.04745,-2.13438,0.013011,1.05995,-2.09482,0.053415,0.991709,-2.0827,0.174911,0.893405,-2.08936,0.168044,0.699951,-2.07272,0.058182,0.729512,-2.08477,0.04374,0.662631,-2.06972,0.014273,0.430891,-2.01983,-0.083004,0.362395,-1.96455,0.200065,0.378587,-1.9993,0.150726,0.341557,-2.06347,0.023489,1.26958,-1.85384,0.000978,1.30483,-2.06498},
/*79*/{0.146653,1.84354,-1.75472,0.200714,1.78212,-1.91444,0.048539,1.71179,-1.65166,-0.032825,1.61693,-1.58399,0.088782,1.5367,-1.66193,0.158569,1.48641,-1.68812,0.105015,1.84308,-2.15603,0.196896,1.78195,-1.99256,-0.020964,1.83116,-2.0475,-0.079994,1.63473,-2.22034,-0.062874,1.5364,-2.25162,0.095405,1.43729,-2.19287,0.138694,1.387,-2.20219,0.097878,1.354,-1.83083,-0.101636,1.41875,-1.87268,-0.104085,1.4532,-1.92524,-0.111324,1.43047,-1.9845,0.077186,1.40134,-2.08848,0.150489,1.06958,-1.76299,0.029506,1.01599,-1.78881,0.062138,0.939047,-1.80901,0.152209,0.863883,-1.78488,0.00499,0.816871,-1.76946,-0.069317,0.813579,-1.77406,-0.004262,0.927092,-1.7947,-0.07538,0.922144,-1.79151,-0.300983,0.853501,-1.78068,-0.361399,0.955681,-1.85434,-0.35788,0.697551,-1.83033,-0.414413,0.723056,-1.76405,0.129164,1.11786,-2.15046,0.167991,1.04359,-2.13522,0.008268,1.05824,-2.09452,0.047923,0.987758,-2.08105,0.167443,0.888424,-2.09064,0.155911,0.695143,-2.07305,0.047685,0.727677,-2.08653,0.029834,0.662496,-2.07004,-0.00517,0.432135,-2.02081,-0.105054,0.362059,-1.96448,0.17842,0.377541,-1.99968,0.128015,0.340764,-2.06412,0.022053,1.26787,-1.85451,-0.001664,1.30403,-2.06536},
/*80*/{0.14619,1.84213,-1.75435,0.200608,1.78022,-1.91318,0.043225,1.71231,-1.65199,-0.040416,1.61974,-1.58648,0.080787,1.53669,-1.66119,0.150165,1.48394,-1.68569,0.106982,1.8413,-2.15555,0.197186,1.77941,-1.99171,-0.020216,1.83019,-2.04816,-0.070672,1.62897,-2.22207,-0.052691,1.53161,-2.25222,0.106324,1.43525,-2.19138,0.15103,1.38609,-2.20124,0.095882,1.35197,-1.83018,-0.102208,1.41842,-1.87229,-0.104126,1.45276,-1.92559,-0.112453,1.43049,-1.98353,0.075768,1.40018,-2.08954,0.160627,1.07433,-1.76226,0.038114,1.01672,-1.78802,0.073521,0.940485,-1.80817,0.166683,0.867365,-1.7856,0.019312,0.817451,-1.77015,-0.056784,0.812526,-1.77446,0.007323,0.927619,-1.79351,-0.064128,0.92192,-1.79,-0.289825,0.853981,-1.78181,-0.353036,0.948376,-1.85218,-0.337483,0.690773,-1.83165,-0.394188,0.712883,-1.76403,0.122924,1.11375,-2.15141,0.161703,1.03877,-2.13649,0.001923,1.05601,-2.0945,0.040586,0.984378,-2.08012,0.156735,0.8827,-2.09181,0.142913,0.689477,-2.07174,0.035695,0.724948,-2.08757,0.016932,0.659269,-2.06963,-0.024507,0.433369,-2.02071,-0.126383,0.36313,-1.96617,0.156909,0.376282,-2.00103,0.105686,0.340252,-2.06488,0.019901,1.26649,-1.85473,-0.0038,1.30325,-2.06547},
/*81*/{0.145903,1.84052,-1.75437,0.199016,1.77768,-1.91201,0.038495,1.7143,-1.65273,-0.04812,1.62364,-1.58824,0.073005,1.53748,-1.66013,0.14107,1.4827,-1.68282,0.109264,1.83972,-2.15542,0.197586,1.77851,-1.99057,-0.019929,1.82939,-2.0488,-0.064078,1.62522,-2.2244,-0.043075,1.5283,-2.25297,0.117974,1.4346,-2.19028,0.162498,1.38707,-2.19973,0.094712,1.35035,-1.83038,-0.102494,1.41876,-1.87202,-0.10534,1.45192,-1.92612,-0.112521,1.43057,-1.98338,0.07491,1.39941,-2.09044,0.166042,1.07955,-1.76418,0.04684,1.01772,-1.7874,0.084962,0.942754,-1.80724,0.181045,0.871342,-1.78871,0.032823,0.817739,-1.77119,-0.041534,0.810664,-1.77615,0.019188,0.928805,-1.79165,-0.051649,0.921362,-1.79046,-0.272398,0.844303,-1.7818,-0.345315,0.94051,-1.85118,-0.317657,0.684204,-1.83169,-0.372613,0.703374,-1.76296,0.117984,1.11017,-2.15239,0.154425,1.03449,-2.13691,-0.004073,1.05573,-2.09502,0.03297,0.982402,-2.08153,0.146441,0.877958,-2.09321,0.129352,0.685323,-2.07222,0.023807,0.722133,-2.08832,0.003713,0.655204,-2.06984,-0.046756,0.43318,-2.02264,-0.147963,0.363394,-1.9643,0.13555,0.376445,-2.0008,0.084554,0.340187,-2.06535,0.018423,1.2655,-1.85526,-0.005246,1.30276,-2.06592},
/*82*/{0.145118,1.84026,-1.75387,0.198605,1.77703,-1.91171,0.032963,1.71628,-1.65323,-0.055362,1.62772,-1.59018,0.065407,1.53803,-1.65902,0.132791,1.48203,-1.68098,0.111435,1.83923,-2.15484,0.197491,1.77754,-1.99006,-0.018468,1.82816,-2.04955,-0.056249,1.62325,-2.22575,-0.034015,1.52554,-2.25357,0.129654,1.43643,-2.18901,0.174176,1.38951,-2.19771,0.09387,1.35038,-1.8311,-0.102926,1.41876,-1.87244,-0.105392,1.45171,-1.92636,-0.113116,1.43049,-1.98377,0.074236,1.39839,-2.09112,0.173312,1.08473,-1.7644,0.056362,1.0187,-1.78609,0.096134,0.946051,-1.80846,0.195094,0.877308,-1.78855,0.046809,0.819763,-1.77187,-0.02749,0.810446,-1.77857,0.030288,0.929948,-1.79021,-0.040724,0.920623,-1.78853,-0.260104,0.838225,-1.7825,-0.337455,0.931433,-1.85105,-0.296247,0.676863,-1.8329,-0.351683,0.692856,-1.76238,0.112252,1.10678,-2.15304,0.147881,1.02985,-2.1381,-0.010218,1.05432,-2.0959,0.024984,0.981188,-2.08348,0.134396,0.873065,-2.09404,0.114846,0.678287,-2.07095,0.010932,0.719645,-2.08855,-0.012109,0.655487,-2.07297,-0.066287,0.434008,-2.02375,-0.169709,0.364111,-1.96755,0.113781,0.376138,-2.00128,0.0624,0.339672,-2.06617,0.017721,1.26539,-1.8554,-0.006208,1.30207,-2.06613},
/*83*/{0.14353,1.84009,-1.75345,0.19767,1.77561,-1.91,0.028591,1.71832,-1.65363,-0.063579,1.6323,-1.59237,0.058828,1.53986,-1.65773,0.124003,1.48312,-1.67989,0.113976,1.83902,-2.15417,0.197698,1.77634,-1.98847,-0.017893,1.82782,-2.0506,-0.04978,1.62039,-2.22691,-0.025123,1.52346,-2.25421,0.138897,1.43835,-2.18716,0.185828,1.3933,-2.19523,0.092663,1.35118,-1.82977,-0.103282,1.41891,-1.87286,-0.105428,1.45168,-1.92669,-0.113024,1.43019,-1.98461,0.072717,1.39628,-2.09295,0.181233,1.0915,-1.76518,0.065446,1.02214,-1.78534,0.106333,0.948859,-1.80631,0.207299,0.882939,-1.78877,0.06038,0.82046,-1.77313,-0.014503,0.810557,-1.77833,0.041759,0.931632,-1.78871,-0.029295,0.919226,-1.7865,-0.24628,0.830897,-1.78326,-0.329237,0.920411,-1.85087,-0.273823,0.668343,-1.8338,-0.329092,0.681638,-1.76242,0.107298,1.10366,-2.15339,0.139992,1.02573,-2.13841,-0.017646,1.05447,-2.09725,0.016517,0.980063,-2.08401,0.122521,0.867547,-2.09445,0.099788,0.674487,-2.07148,-0.003276,0.716786,-2.08885,-0.027708,0.65314,-2.07393,-0.085058,0.434448,-2.02344,-0.191168,0.364237,-1.9675,0.092074,0.375727,-2.0029,0.041316,0.340078,-2.06654,0.016809,1.26591,-1.85429,-0.007334,1.30032,-2.06538},
/*84*/{0.141083,1.84097,-1.75306,0.19767,1.77561,-1.91,0.022878,1.72238,-1.65436,-0.070652,1.63797,-1.59423,0.050294,1.5421,-1.65681,0.115912,1.48434,-1.67736,0.115838,1.83945,-2.15367,0.197347,1.7762,-1.98798,-0.01675,1.82749,-2.0519,-0.042597,1.61969,-2.22816,-0.01669,1.52275,-2.25463,0.148857,1.44201,-2.18524,0.197528,1.39871,-2.19224,0.091872,1.35238,-1.82885,-0.103552,1.41949,-1.87384,-0.106434,1.45203,-1.92784,-0.112977,1.43101,-1.98558,0.076628,1.39614,-2.09274,0.18792,1.09718,-1.76532,0.073876,1.02528,-1.78479,0.116961,0.953331,-1.80648,0.21967,0.888813,-1.78948,0.073509,0.821891,-1.7745,-0.000215,0.808876,-1.78059,0.051713,0.932743,-1.78729,-0.019469,0.9186,-1.78651,-0.231254,0.82455,-1.78505,-0.321253,0.908914,-1.85137,-0.251156,0.660192,-1.83414,-0.305101,0.67053,-1.76268,0.101669,1.10067,-2.15318,0.132255,1.02102,-2.13896,-0.025982,1.05416,-2.09841,0.00751,0.978476,-2.08576,0.109125,0.861851,-2.09556,0.083973,0.671138,-2.07177,-0.019115,0.714109,-2.08931,-0.044819,0.650302,-2.07424,-0.109511,0.433945,-2.0222,-0.210201,0.364733,-1.96588,0.07144,0.375926,-2.00302,0.01876,0.339642,-2.06715,0.016451,1.26682,-1.85371,-0.005197,1.3006,-2.06517},
/*85*/{0.139014,1.84212,-1.75186,0.196433,1.77618,-1.9077,0.017346,1.7266,-1.65519,-0.07722,1.64276,-1.59589,0.042498,1.54579,-1.65615,0.1075,1.48664,-1.67535,0.118455,1.84065,-2.15309,0.197833,1.77664,-1.98622,-0.015374,1.82786,-2.053,-0.037331,1.61844,-2.22903,-0.00853,1.52256,-2.25464,0.158084,1.44695,-2.18257,0.207954,1.40583,-2.18922,0.09106,1.35375,-1.82821,-0.1046,1.41955,-1.87465,-0.106301,1.45209,-1.92856,-0.113069,1.4314,-1.98703,0.083814,1.39715,-2.09135,0.19638,1.1032,-1.76435,0.086478,1.02927,-1.78373,0.12641,0.958245,-1.80571,0.231525,0.895896,-1.7901,0.0876,0.824516,-1.77571,0.013451,0.80782,-1.78235,0.061143,0.933687,-1.78424,-0.009639,0.917174,-1.78515,-0.216109,0.817642,-1.78666,-0.3131,0.895733,-1.85217,-0.228229,0.651709,-1.83432,-0.282624,0.658916,-1.7622,0.095889,1.09745,-2.15317,0.124209,1.01764,-2.13941,-0.032278,1.05454,-2.09953,-0.001826,0.977935,-2.08678,0.094689,0.855773,-2.09601,0.066975,0.666531,-2.0695,-0.034298,0.71238,-2.08961,-0.061153,0.649402,-2.07394,-0.13419,0.43488,-2.02314,-0.234504,0.367738,-1.96703,0.049721,0.375657,-2.00296,-0.002264,0.339869,-2.06698,0.016819,1.2675,-1.85336,-0.000838,1.30173,-2.06512},
/*86*/{0.136568,1.84428,-1.75116,0.196172,1.7786,-1.90648,0.012741,1.73057,-1.65494,-0.084778,1.64832,-1.59743,0.035513,1.55002,-1.65541,0.098468,1.49069,-1.67453,0.120926,1.84227,-2.15181,0.198117,1.77863,-1.98472,-0.014679,1.82808,-2.0545,-0.034142,1.61812,-2.22968,-0.001162,1.5227,-2.25495,0.167127,1.45341,-2.17994,0.218348,1.41378,-2.18584,0.089836,1.35629,-1.82817,-0.10512,1.42011,-1.87613,-0.106681,1.45231,-1.92977,-0.112804,1.43143,-1.98843,0.075855,1.39578,-2.09297,0.201795,1.11074,-1.76425,0.092178,1.03385,-1.7831,0.138131,0.963976,-1.80541,0.24279,0.902618,-1.79053,0.100513,0.825724,-1.7773,0.027508,0.807405,-1.78344,0.070005,0.934548,-1.78431,-0.000225,0.91539,-1.78435,-0.204288,0.805887,-1.78798,-0.304053,0.881959,-1.85343,-0.20579,0.643058,-1.8349,-0.259436,0.647704,-1.76208,0.089691,1.09492,-2.15294,0.115614,1.01389,-2.13937,-0.038913,1.05483,-2.09983,-0.01199,0.977606,-2.08732,0.080784,0.851364,-2.09613,0.050518,0.663464,-2.06892,-0.050745,0.712065,-2.08925,-0.077458,0.649325,-2.07424,-0.146907,0.43712,-2.02301,-0.255022,0.3684,-1.96742,0.027921,0.375405,-2.00353,-0.024039,0.340187,-2.06717,0.015803,1.2693,-1.85275,-0.004395,1.3001,-2.06481},
/*87*/{0.134118,1.84643,-1.7508,0.196036,1.7798,-1.90481,0.00627,1.73557,-1.65544,-0.091842,1.65331,-1.59869,0.027627,1.5545,-1.65537,0.091689,1.4942,-1.67301,0.122823,1.84479,-2.15104,0.198524,1.77966,-1.98287,-0.014308,1.82826,-2.05474,-0.029752,1.61814,-2.22963,0.006187,1.52416,-2.25501,0.175534,1.46051,-2.17766,0.227947,1.42249,-2.18168,0.089572,1.35724,-1.82633,-0.104959,1.42137,-1.87755,-0.106198,1.45269,-1.93077,-0.112738,1.43211,-1.99009,0.077838,1.39488,-2.09298,0.207207,1.11783,-1.76508,0.099583,1.03781,-1.783,0.144444,0.968433,-1.80627,0.254027,0.910386,-1.79153,0.113477,0.828054,-1.77862,0.041002,0.806245,-1.78439,0.078358,0.935693,-1.78345,0.008865,0.913089,-1.78344,-0.190625,0.797282,-1.78886,-0.294782,0.866581,-1.85568,-0.182219,0.634405,-1.83493,-0.234605,0.637068,-1.76137,0.082957,1.09264,-2.15261,0.106834,1.01085,-2.13891,-0.047154,1.05639,-2.10085,-0.022487,0.978559,-2.08804,0.063845,0.848446,-2.09571,0.034534,0.663289,-2.06883,-0.06653,0.712181,-2.08912,-0.094728,0.649045,-2.07355,-0.172022,0.439051,-2.02447,-0.275614,0.369798,-1.96761,0.006667,0.375299,-2.00354,-0.046026,0.340113,-2.0672,0.015245,1.27039,-1.85171,-0.003096,1.29959,-2.06417},
/*88*/{0.130987,1.84905,-1.74974,0.195739,1.78168,-1.90283,0.001377,1.74065,-1.65626,-0.099086,1.65936,-1.60026,0.020465,1.5595,-1.65551,0.084116,1.4988,-1.67258,0.124622,1.84783,-2.14992,0.199094,1.78176,-1.98093,-0.013224,1.8298,-2.05537,-0.025277,1.61898,-2.22958,0.012901,1.52598,-2.25537,0.1835,1.46885,-2.1748,0.237508,1.43272,-2.17786,0.088685,1.35849,-1.8257,-0.105551,1.42254,-1.87911,-0.105473,1.45481,-1.93216,-0.112307,1.43367,-1.99116,0.075405,1.39355,-2.09376,0.212453,1.12504,-1.7647,0.106846,1.0424,-1.78375,0.153683,0.974954,-1.80599,0.263581,0.918709,-1.79215,0.126936,0.830809,-1.77937,0.05448,0.805858,-1.78492,0.086336,0.936212,-1.78254,0.017977,0.910685,-1.78438,-0.176016,0.787143,-1.79103,-0.284726,0.850867,-1.85773,-0.15743,0.626204,-1.83369,-0.211918,0.625911,-1.7615,0.076701,1.09072,-2.15186,0.097263,1.00841,-2.13759,-0.055823,1.05816,-2.10108,-0.032621,0.97912,-2.08898,0.04868,0.847557,-2.09465,0.017026,0.661733,-2.06815,-0.082667,0.713382,-2.08783,-0.111753,0.650358,-2.07264,-0.255607,0.422182,-2.02033,-0.29619,0.372203,-1.96753,-0.014359,0.375638,-2.00427,-0.066571,0.33988,-2.06725,0.013464,1.27181,-1.85121,-0.004783,1.29913,-2.06393},
/*89*/{0.127683,1.8521,-1.74901,0.195396,1.78474,-1.90138,-0.003525,1.7457,-1.65633,-0.105529,1.66565,-1.60227,0.013755,1.56535,-1.65569,0.076613,1.50419,-1.67138,0.126508,1.85132,-2.1489,0.198843,1.78438,-1.97926,-0.01229,1.83074,-2.05597,-0.021632,1.62024,-2.22863,0.020261,1.52815,-2.25535,0.190533,1.47628,-2.17171,0.246554,1.44255,-2.17379,0.088651,1.36103,-1.82549,-0.104591,1.42384,-1.88068,-0.104708,1.45569,-1.93341,-0.110905,1.43509,-1.99266,0.077282,1.39404,-2.09342,0.21468,1.1322,-1.76527,0.110655,1.04615,-1.78435,0.160909,0.980121,-1.80635,0.27268,0.927717,-1.79304,0.138936,0.833253,-1.7802,0.068178,0.804804,-1.78695,0.093348,0.936497,-1.78224,0.027646,0.907672,-1.78281,-0.161217,0.777365,-1.79187,-0.273688,0.834149,-1.86072,-0.134237,0.617709,-1.83383,-0.188492,0.615191,-1.76118,0.069667,1.08929,-2.15086,0.088922,1.0062,-2.13596,-0.06364,1.05979,-2.10121,-0.043028,0.980916,-2.0893,0.032429,0.847232,-2.09496,-0.001811,0.662365,-2.06824,-0.100021,0.71455,-2.08707,-0.130436,0.653738,-2.07239,-0.215616,0.441118,-2.0238,-0.317426,0.374352,-1.96812,-0.036455,0.374903,-2.00384,-0.088456,0.340392,-2.06727,0.014077,1.27377,-1.85084,-0.002763,1.29981,-2.06383},
/*90*/{0.125173,1.85548,-1.74824,0.195876,1.78782,-1.89926,-0.007868,1.75047,-1.65709,-0.111946,1.67216,-1.60396,0.006727,1.57144,-1.65691,0.069377,1.50964,-1.67098,0.128314,1.85541,-2.14788,0.19959,1.78782,-1.97696,-0.011438,1.8323,-2.05685,-0.016147,1.62215,-2.22752,0.02696,1.53066,-2.25552,0.198486,1.48569,-2.16838,0.255493,1.45294,-2.16859,0.090215,1.36378,-1.82427,-0.10492,1.42625,-1.88244,-0.10413,1.45839,-1.93483,-0.109925,1.437,-1.99472,0.079235,1.39423,-2.09322,0.219923,1.13998,-1.76573,0.117246,1.05155,-1.78494,0.169113,0.986634,-1.80737,0.282696,0.93729,-1.79338,0.150316,0.835882,-1.7809,0.081655,0.804367,-1.78782,0.100329,0.937016,-1.78201,0.035258,0.904792,-1.78366,-0.147544,0.76832,-1.795,-0.262244,0.816449,-1.86348,-0.110962,0.609113,-1.83358,-0.164471,0.604383,-1.76086,0.062822,1.08814,-2.1495,0.078822,1.00479,-2.13469,-0.07166,1.0624,-2.10176,-0.051418,0.982447,-2.09002,0.016504,0.847663,-2.09425,-0.017821,0.662385,-2.06749,-0.116696,0.716445,-2.08611,-0.148041,0.654158,-2.07192,-0.232701,0.44713,-2.02935,-0.338085,0.377247,-1.96696,-0.057748,0.374948,-2.00378,-0.110526,0.340314,-2.06743,0.014253,1.27668,-1.84989,-0.001118,1.30059,-2.06324},
/*91*/{0.121382,1.85885,-1.74788,0.194648,1.79188,-1.89747,-0.012969,1.75572,-1.65762,-0.119431,1.67815,-1.60609,0.00081,1.57776,-1.65719,0.063181,1.51589,-1.67114,0.129404,1.85886,-2.14672,0.200281,1.79134,-1.97556,-0.010373,1.83449,-2.05752,-0.012341,1.62413,-2.2265,0.03325,1.53387,-2.25595,0.205257,1.49499,-2.16504,0.263117,1.46424,-2.16431,0.089893,1.36639,-1.82491,-0.10405,1.42883,-1.88335,-0.102262,1.46034,-1.93648,-0.108363,1.43884,-1.99625,0.078093,1.39476,-2.0956,0.222053,1.14694,-1.76606,0.123067,1.05594,-1.78569,0.175575,0.993327,-1.80722,0.290389,0.946542,-1.79348,0.161608,0.837828,-1.78079,0.095521,0.802843,-1.78876,0.106707,0.936167,-1.78225,0.044306,0.901501,-1.78492,-0.133212,0.756773,-1.7961,-0.249632,0.798495,-1.86689,-0.085637,0.600673,-1.83265,-0.140697,0.593945,-1.76039,0.056915,1.08697,-2.14782,0.070251,1.00349,-2.13264,-0.078909,1.06605,-2.10198,-0.062967,0.983727,-2.09051,0.001735,0.848633,-2.09285,-0.036564,0.664225,-2.067,-0.134529,0.71846,-2.08532,-0.166042,0.656978,-2.07111,-0.249735,0.448746,-2.02406,-0.359427,0.382828,-1.97024,-0.080101,0.375011,-2.00322,-0.132334,0.340467,-2.06704,0.014388,1.27912,-1.85034,-0.001534,1.3014,-2.06383},
/*92*/{0.118627,1.86285,-1.74689,0.194065,1.79492,-1.89536,-0.017384,1.76075,-1.65847,-0.124262,1.68494,-1.60814,-0.004673,1.58489,-1.65813,0.056149,1.52205,-1.67063,0.130362,1.86319,-2.14588,0.200738,1.79518,-1.97387,-0.009739,1.83711,-2.05824,-0.008936,1.62645,-2.22537,0.039969,1.53754,-2.25588,0.211432,1.50487,-2.16105,0.269907,1.47587,-2.1597,0.089542,1.36939,-1.82494,-0.103092,1.43096,-1.88573,-0.101935,1.46215,-1.93846,-0.106654,1.44155,-1.99807,0.080646,1.395,-2.09523,0.225161,1.15432,-1.76649,0.128445,1.06089,-1.78603,0.182223,0.998997,-1.80815,0.29769,0.955852,-1.79403,0.17293,0.840409,-1.78119,0.107307,0.802681,-1.78906,0.112798,0.936009,-1.78387,0.051794,0.898128,-1.78649,-0.118238,0.745345,-1.79816,-0.236681,0.780154,-1.87161,-0.061773,0.593109,-1.83115,-0.116758,0.583234,-1.76015,0.050335,1.08733,-2.14661,0.060556,1.00281,-2.13017,-0.086977,1.07005,-2.10232,-0.073833,0.988305,-2.08874,-0.012581,0.849711,-2.09154,-0.053491,0.665128,-2.06583,-0.150354,0.721561,-2.08397,-0.183238,0.66009,-2.0704,-0.273945,0.452188,-2.02465,-0.380013,0.388598,-1.9695,-0.101221,0.374878,-2.00283,-0.154124,0.340545,-2.06754,0.01422,1.28174,-1.85006,0.000386,1.30246,-2.06385},
/*93*/{0.115203,1.8664,-1.74666,0.194269,1.79826,-1.89307,-0.021673,1.76631,-1.6588,-0.130253,1.69207,-1.61022,-0.010755,1.59132,-1.65956,0.050521,1.52881,-1.67139,0.132003,1.8679,-2.14475,0.201117,1.79906,-1.97143,-0.008394,1.83952,-2.05818,-0.006429,1.63118,-2.22499,0.046766,1.5417,-2.25617,0.216501,1.51456,-2.15705,0.276237,1.48853,-2.15508,0.091362,1.37321,-1.82512,-0.101257,1.43468,-1.88713,-0.099994,1.46548,-1.93953,-0.104534,1.44396,-1.99944,0.082549,1.39558,-2.09496,0.229029,1.16152,-1.76684,0.133607,1.06532,-1.78581,0.188214,1.00577,-1.8085,0.304805,0.963724,-1.79452,0.183362,0.84326,-1.78085,0.12046,0.801214,-1.78964,0.119521,0.933815,-1.7842,0.059552,0.89453,-1.7884,-0.102056,0.734981,-1.80041,-0.223324,0.761515,-1.87604,-0.037153,0.585512,-1.83088,-0.092521,0.573753,-1.75977,0.043285,1.08687,-2.14436,0.052433,1.0028,-2.12894,-0.093463,1.07418,-2.10152,-0.083948,0.992743,-2.08886,-0.025749,0.851695,-2.0892,-0.070523,0.668539,-2.06485,-0.167156,0.725901,-2.0834,-0.199657,0.663447,-2.06991,-0.293314,0.455355,-2.02448,-0.398587,0.394381,-1.96943,-0.122129,0.374995,-2.00262,-0.173976,0.341464,-2.06584,0.015494,1.28556,-1.84909,0.002079,1.30375,-2.06313},
/*94*/{0.112771,1.87085,-1.74613,0.193659,1.80315,-1.89234,-0.025476,1.77178,-1.65971,-0.13627,1.69809,-1.61187,-0.016291,1.59754,-1.66022,0.044412,1.53541,-1.67077,0.133763,1.87264,-2.14321,0.201815,1.80311,-1.97025,-0.008216,1.8425,-2.05848,-0.002144,1.63418,-2.22418,0.053061,1.54608,-2.25563,0.222049,1.52519,-2.15371,0.282356,1.50095,-2.15039,0.092033,1.37674,-1.82518,-0.100161,1.43828,-1.88794,-0.098674,1.46798,-1.94176,-0.101979,1.44686,-2.00109,0.083575,1.39551,-2.09596,0.231141,1.16811,-1.76598,0.137338,1.06999,-1.78788,0.192484,1.01137,-1.80896,0.310305,0.973508,-1.79503,0.193006,0.845268,-1.78061,0.132105,0.800401,-1.78934,0.12403,0.934197,-1.78584,0.068907,0.891704,-1.79075,-0.086744,0.723364,-1.80176,-0.207104,0.742403,-1.87901,-0.012234,0.578774,-1.82964,-0.068023,0.564135,-1.75984,0.037217,1.08796,-2.14236,0.043292,1.00311,-2.12593,-0.102767,1.07851,-2.10056,-0.093281,0.997028,-2.08833,-0.038591,0.854454,-2.08722,-0.087647,0.672172,-2.06387,-0.183458,0.729742,-2.08277,-0.217036,0.668439,-2.06923,-0.315918,0.459643,-2.02413,-0.419191,0.401351,-1.97004,-0.143344,0.375572,-2.00219,-0.197102,0.340961,-2.06632,0.015797,1.28907,-1.84844,0.002892,1.30465,-2.06272},
/*95*/{0.110561,1.87445,-1.74585,0.194162,1.8061,-1.88998,-0.029027,1.77725,-1.66058,-0.140668,1.7053,-1.61482,-0.021795,1.6046,-1.66168,0.039997,1.54159,-1.67127,0.134608,1.87759,-2.14207,0.202494,1.80758,-1.96819,-0.006748,1.84609,-2.05863,0.001704,1.6381,-2.224,0.058704,1.55082,-2.25623,0.22567,1.534,-2.14987,0.287677,1.5134,-2.14478,0.09326,1.3804,-1.82557,-0.098602,1.44159,-1.88994,-0.095964,1.47096,-1.94296,-0.099976,1.45026,-2.00151,0.087671,1.39562,-2.09494,0.233717,1.17472,-1.7661,0.140405,1.07507,-1.78674,0.197215,1.01788,-1.80943,0.314628,0.981569,-1.7953,0.200797,0.84773,-1.77979,0.143249,0.799702,-1.78979,0.1292,0.934543,-1.78687,0.074548,0.887172,-1.79278,-0.070222,0.711596,-1.80329,-0.191529,0.722991,-1.88224,0.012991,0.571741,-1.82979,-0.042369,0.554846,-1.75942,0.031943,1.08815,-2.14005,0.034842,1.00385,-2.12295,-0.108335,1.08383,-2.10057,-0.102398,1.00207,-2.08836,-0.051622,0.857127,-2.08423,-0.104308,0.675812,-2.06297,-0.199929,0.734368,-2.08192,-0.234255,0.674035,-2.06941,-0.333658,0.464856,-2.02469,-0.437224,0.4079,-1.969,-0.165374,0.376173,-2.00188,-0.217562,0.341802,-2.06538,0.016702,1.29257,-1.84742,0.005364,1.30609,-2.06194},
/*96*/{0.10809,1.87871,-1.74551,0.193548,1.81249,-1.88901,-0.031905,1.78244,-1.66114,-0.146185,1.71202,-1.6172,-0.026282,1.61166,-1.66343,0.034709,1.5487,-1.67186,0.135873,1.88275,-2.14112,0.202928,1.81198,-1.96651,-0.005667,1.84935,-2.05875,0.005129,1.6426,-2.22441,0.064609,1.55602,-2.25682,0.230197,1.54546,-2.14622,0.291937,1.52616,-2.14052,0.095943,1.38504,-1.82738,-0.096925,1.44541,-1.89039,-0.094039,1.47398,-1.94498,-0.097102,1.45305,-2.00329,0.089737,1.39587,-2.09486,0.23519,1.18202,-1.76679,0.143542,1.08072,-1.78843,0.202257,1.02501,-1.80859,0.319349,0.989628,-1.7947,0.21224,0.849395,-1.78082,0.155123,0.799088,-1.7899,0.134496,0.931866,-1.78977,0.082681,0.882345,-1.79447,-0.054083,0.701513,-1.80661,-0.173844,0.703191,-1.88531,0.038078,0.56506,-1.82916,-0.016825,0.54593,-1.75899,0.026902,1.08896,-2.1383,0.026984,1.00532,-2.12062,-0.113859,1.08956,-2.10112,-0.111704,1.00703,-2.08792,-0.064543,0.861868,-2.08246,-0.121339,0.680459,-2.06255,-0.216086,0.739484,-2.08084,-0.250753,0.679831,-2.06869,-0.345409,0.472149,-2.02529,-0.455772,0.416463,-1.96822,-0.186219,0.376804,-2.0011,-0.239974,0.342213,-2.06563,0.018815,1.29694,-1.84669,0.007209,1.30732,-2.06136},
/*97*/{0.106341,1.88267,-1.74506,0.193594,1.81668,-1.88713,-0.035294,1.78744,-1.66238,-0.150882,1.7188,-1.6193,-0.030666,1.61801,-1.66411,0.030927,1.55485,-1.67249,0.137745,1.88821,-2.1392,0.20385,1.81663,-1.96482,-0.004218,1.85195,-2.05874,0.009506,1.64638,-2.22415,0.070875,1.56142,-2.25689,0.233554,1.555,-2.14224,0.295814,1.53869,-2.13599,0.097865,1.3892,-1.82772,-0.094575,1.44991,-1.89122,-0.092061,1.47768,-1.94691,-0.09391,1.45636,-2.00426,0.092373,1.3965,-2.09556,0.238744,1.18835,-1.76555,0.145673,1.08617,-1.78621,0.205451,1.02986,-1.80886,0.32242,0.996303,-1.79511,0.221019,0.851798,-1.77984,0.168004,0.797539,-1.78986,0.139952,0.929707,-1.79376,0.090505,0.878469,-1.79718,-0.037555,0.689074,-1.80737,-0.155911,0.683344,-1.88844,0.06318,0.558638,-1.82805,0.00872,0.537018,-1.75867,0.021253,1.0907,-2.13585,0.019176,1.00713,-2.11758,-0.119246,1.09529,-2.10131,-0.119761,1.01371,-2.08765,-0.079214,0.86672,-2.08198,-0.136788,0.68487,-2.0627,-0.231453,0.746037,-2.08089,-0.267187,0.687063,-2.06852,-0.361641,0.478028,-2.02618,-0.474108,0.426588,-1.96868,-0.208379,0.37585,-1.99992,-0.26088,0.34233,-2.06459,0.020282,1.30129,-1.84593,0.00901,1.30906,-2.06073},
/*98*/{0.104765,1.88632,-1.74505,0.194039,1.82161,-1.88613,-0.038559,1.79286,-1.66377,-0.154195,1.72547,-1.62203,-0.035594,1.62435,-1.66562,0.025821,1.56108,-1.67257,0.138828,1.89317,-2.13838,0.205428,1.82054,-1.96279,-0.003569,1.85535,-2.05893,0.012847,1.65184,-2.22513,0.075758,1.56662,-2.25712,0.23628,1.56425,-2.13783,0.299026,1.55059,-2.13146,0.099026,1.39254,-1.82878,-0.092675,1.45431,-1.8929,-0.089245,1.48096,-1.94899,-0.09054,1.45937,-2.00541,0.094139,1.39676,-2.09544,0.241284,1.19504,-1.76569,0.150329,1.09224,-1.78366,0.208884,1.03611,-1.80898,0.324455,1.00348,-1.79521,0.228783,0.852948,-1.78021,0.178954,0.797115,-1.78949,0.14483,0.926692,-1.79564,0.098052,0.872412,-1.80038,-0.019668,0.675779,-1.80966,-0.136845,0.66402,-1.89148,0.08866,0.553263,-1.82821,0.034977,0.529329,-1.75897,0.015118,1.09231,-2.13347,0.011294,1.00856,-2.11544,-0.125275,1.10188,-2.10126,-0.128269,1.02008,-2.08768,-0.093732,0.871411,-2.08196,-0.152427,0.69068,-2.06253,-0.246247,0.752739,-2.08101,-0.281613,0.693219,-2.06888,-0.37917,0.484459,-2.02694,-0.491585,0.436875,-1.96866,-0.231393,0.376168,-2.0004,-0.284329,0.342624,-2.06499,0.02092,1.30502,-1.84548,0.010286,1.31043,-2.06039},
/*99*/{0.103264,1.88994,-1.74431,0.193867,1.82476,-1.88424,-0.042012,1.79848,-1.66505,-0.157604,1.73203,-1.62521,-0.039559,1.63095,-1.66675,0.022368,1.56783,-1.67322,0.140922,1.89861,-2.13767,0.205251,1.82459,-1.96116,-0.001953,1.85919,-2.05958,0.016099,1.65655,-2.2254,0.08221,1.57196,-2.25732,0.238674,1.57499,-2.1347,0.302388,1.56185,-2.12815,0.101484,1.39746,-1.82901,-0.090984,1.45914,-1.894,-0.086773,1.48497,-1.95062,-0.087648,1.46287,-2.00584,0.097337,1.39761,-2.0955,0.242531,1.20093,-1.76543,0.153056,1.09896,-1.78305,0.211973,1.0432,-1.80674,0.324561,1.00886,-1.79547,0.237059,0.853637,-1.78055,0.189959,0.79565,-1.79088,0.14988,0.92409,-1.79765,0.104329,0.867152,-1.80316,-0.000266,0.665513,-1.81096,-0.117698,0.644761,-1.89374,0.113384,0.547864,-1.82805,0.060925,0.520836,-1.75963,0.010368,1.09437,-2.13154,0.003083,1.01104,-2.11257,-0.130375,1.10818,-2.10072,-0.135897,1.02659,-2.08656,-0.10764,0.877277,-2.08134,-0.167883,0.697054,-2.06292,-0.261027,0.759584,-2.08052,-0.299456,0.699724,-2.06884,-0.396446,0.491116,-2.02759,-0.507831,0.447878,-1.9692,-0.252741,0.375882,-1.99943,-0.30655,0.343387,-2.06444,0.022726,1.31005,-1.84425,0.012493,1.31246,-2.05923},
/*100*/{0.102153,1.89382,-1.74439,0.192625,1.82997,-1.88283,-0.044626,1.80349,-1.66665,-0.162468,1.73816,-1.62772,-0.043367,1.63663,-1.66772,0.017865,1.57329,-1.67339,0.142759,1.90361,-2.13542,0.205877,1.82969,-1.95994,-0.000951,1.862,-2.05965,0.020177,1.66145,-2.22547,0.086989,1.57723,-2.2578,0.240573,1.58447,-2.13198,0.304777,1.57311,-2.12422,0.103087,1.40165,-1.83072,-0.088153,1.46354,-1.89428,-0.084153,1.48821,-1.95275,-0.084798,1.46665,-2.00673,0.100751,1.39759,-2.09507,0.242719,1.20586,-1.76543,0.155477,1.10458,-1.78341,0.215313,1.04806,-1.80665,0.325827,1.01468,-1.7959,0.24552,0.854449,-1.78043,0.201159,0.793329,-1.79045,0.154365,0.9195,-1.79983,0.113438,0.861132,-1.80527,0.017937,0.652022,-1.81287,-0.097549,0.626134,-1.89521,0.139032,0.542436,-1.82897,0.087838,0.513441,-1.7603,0.005888,1.09664,-2.1294,-0.005111,1.01307,-2.1107,-0.134056,1.11502,-2.1012,-0.143565,1.03367,-2.08701,-0.120768,0.883409,-2.08134,-0.1825,0.704483,-2.0638,-0.274976,0.76769,-2.08033,-0.313266,0.7072,-2.06883,-0.41128,0.49774,-2.02886,-0.523194,0.458017,-1.96908,-0.275,0.37688,-2.00011,-0.328954,0.344617,-2.06511,0.024228,1.31434,-1.84343,0.014334,1.31404,-2.05843},
/*101*/{0.101149,1.89703,-1.74405,0.193543,1.83312,-1.88171,-0.046662,1.8084,-1.66792,-0.165796,1.74449,-1.63084,-0.046728,1.64292,-1.66932,0.013983,1.57886,-1.67355,0.143376,1.90883,-2.13449,0.206011,1.83374,-1.95799,0.000491,1.86573,-2.06037,0.024543,1.66727,-2.22658,0.092293,1.58271,-2.25813,0.242397,1.5938,-2.12908,0.306648,1.58421,-2.12069,0.103653,1.4057,-1.83073,-0.085581,1.46795,-1.89442,-0.081924,1.49171,-1.95389,-0.081456,1.46997,-2.00833,0.101536,1.39852,-2.09489,0.244704,1.21132,-1.76623,0.156755,1.10878,-1.78049,0.215209,1.05286,-1.80695,0.325947,1.0191,-1.79575,0.25285,0.854443,-1.7805,0.211132,0.790947,-1.79111,0.159973,0.915067,-1.80185,0.121135,0.853786,-1.80707,0.038057,0.639773,-1.81457,-0.076311,0.607725,-1.89717,0.164542,0.53804,-1.83073,0.114735,0.505742,-1.76137,0.001494,1.09866,-2.12756,-0.012563,1.01628,-2.10844,-0.139222,1.12245,-2.1013,-0.150645,1.04065,-2.08673,-0.133571,0.890236,-2.08108,-0.196135,0.711849,-2.06447,-0.289837,0.775324,-2.08015,-0.32734,0.715936,-2.0684,-0.426659,0.503652,-2.03003,-0.539565,0.471932,-1.96969,-0.296794,0.376995,-1.99999,-0.350851,0.344203,-2.06418,0.024854,1.31856,-1.84248,0.015244,1.31593,-2.05748},
/*102*/{0.100218,1.90054,-1.74417,0.193961,1.83819,-1.88106,-0.049193,1.81374,-1.66934,-0.168846,1.75029,-1.6338,-0.049955,1.64795,-1.67073,0.011138,1.58407,-1.67344,0.144661,1.91302,-2.13293,0.206352,1.83728,-1.95705,0.002226,1.86807,-2.061,0.028005,1.67157,-2.22713,0.097831,1.58814,-2.25849,0.244237,1.60243,-2.12576,0.308627,1.59471,-2.11703,0.105964,1.40961,-1.83172,-0.083079,1.4722,-1.89519,-0.078812,1.4946,-1.95587,-0.077597,1.47292,-2.00845,0.105175,1.39907,-2.09419,0.246491,1.21654,-1.7663,0.158237,1.11205,-1.78155,0.216248,1.05758,-1.80663,0.325929,1.024,-1.79631,0.260404,0.853742,-1.78048,0.223251,0.788841,-1.79059,0.165008,0.909693,-1.80501,0.129146,0.847268,-1.80985,0.057607,0.628188,-1.81529,-0.054489,0.589385,-1.89818,0.189876,0.533518,-1.83147,0.141709,0.498717,-1.76228,-0.002215,1.10148,-2.12608,-0.019199,1.01896,-2.10605,-0.141868,1.12925,-2.10131,-0.157243,1.04886,-2.08679,-0.144959,0.897442,-2.08127,-0.209865,0.720386,-2.0659,-0.303376,0.783367,-2.08035,-0.339651,0.725337,-2.06991,-0.448594,0.509187,-2.03109,-0.551876,0.485573,-1.96839,-0.318777,0.377212,-2.00013,-0.372817,0.346524,-2.06516,0.026974,1.32266,-1.84151,0.017971,1.31754,-2.0565},
/*103*/{0.099471,1.90377,-1.74403,0.193577,1.84129,-1.87949,-0.051184,1.81853,-1.67128,-0.172341,1.75608,-1.63736,-0.053283,1.65336,-1.67154,0.007637,1.58895,-1.67349,0.146668,1.91738,-2.13148,0.207121,1.84142,-1.95592,0.003445,1.87132,-2.06128,0.032076,1.67636,-2.2283,0.102191,1.5934,-2.25905,0.246315,1.61118,-2.12347,0.309932,1.60439,-2.11384,0.107041,1.4146,-1.83252,-0.080511,1.47687,-1.89575,-0.077041,1.49855,-1.95666,-0.074945,1.47638,-2.01014,0.107494,1.40011,-2.09379,0.247952,1.22107,-1.76613,0.157491,1.11651,-1.7808,0.217951,1.06266,-1.80445,0.324463,1.02652,-1.79635,0.267895,0.853121,-1.78085,0.233167,0.785892,-1.79072,0.169784,0.903158,-1.80778,0.135503,0.839946,-1.81278,0.077237,0.615997,-1.8166,-0.032156,0.571292,-1.89942,0.214464,0.529527,-1.8332,0.16885,0.49149,-1.76453,-0.006183,1.10369,-2.12482,-0.025258,1.02152,-2.10424,-0.1447,1.13644,-2.10133,-0.162714,1.05745,-2.0868,-0.155942,0.905441,-2.08094,-0.222586,0.72793,-2.06746,-0.314768,0.791598,-2.08025,-0.352714,0.731611,-2.06892,-0.45326,0.517867,-2.03348,-0.562746,0.499288,-1.96918,-0.34104,0.377755,-2.00171,-0.395549,0.346781,-2.06419,0.028237,1.32764,-1.8405,0.019546,1.31976,-2.05541},
/*104*/{0.098631,1.90659,-1.74386,0.193601,1.8459,-1.87878,-0.052747,1.8229,-1.67326,-0.174481,1.76157,-1.64057,-0.055249,1.65788,-1.67213,0.004768,1.59316,-1.67362,0.147923,1.92164,-2.13048,0.207155,1.84512,-1.9544,0.005393,1.87472,-2.06235,0.035008,1.68129,-2.22932,0.106361,1.59853,-2.25915,0.246301,1.61813,-2.12153,0.311487,1.61358,-2.11077,0.108016,1.41817,-1.83251,-0.077185,1.48109,-1.89599,-0.07402,1.50201,-1.95847,-0.071302,1.47927,-2.01017,0.109928,1.40056,-2.0927,0.248911,1.22423,-1.76614,0.15722,1.12277,-1.78088,0.21746,1.06633,-1.80449,0.323403,1.0297,-1.79743,0.274632,0.851946,-1.78107,0.24299,0.783112,-1.7906,0.174745,0.897622,-1.80955,0.14601,0.83154,-1.81334,0.097501,0.60516,-1.81746,-0.009427,0.554385,-1.90039,0.240251,0.526144,-1.8344,0.196723,0.484744,-1.76631,-0.009925,1.10616,-2.12388,-0.033029,1.02645,-2.10333,-0.147333,1.14434,-2.10182,-0.167215,1.06465,-2.08685,-0.16544,0.913314,-2.08152,-0.233515,0.736024,-2.0687,-0.326695,0.800156,-2.08123,-0.363995,0.739251,-2.06908,-0.463125,0.524434,-2.03509,-0.573194,0.51438,-1.96892,-0.361153,0.377857,-2.0027,-0.416774,0.349014,-2.06426,0.029293,1.33163,-1.83913,0.021309,1.32134,-2.05397},
/*105*/{0.098141,1.90935,-1.74383,0.192459,1.84889,-1.87779,-0.054534,1.82745,-1.67505,-0.178656,1.76592,-1.64384,-0.058151,1.66195,-1.67345,0.002324,1.59707,-1.67294,0.149791,1.92546,-2.12842,0.207666,1.84835,-1.95302,0.007072,1.8767,-2.06249,0.039355,1.6845,-2.23037,0.110342,1.60361,-2.26007,0.247624,1.62521,-2.11892,0.312548,1.62213,-2.10775,0.1104,1.42222,-1.83305,-0.075433,1.48624,-1.89681,-0.071432,1.5054,-1.96,-0.06795,1.48199,-2.01121,0.112125,1.4014,-2.09244,0.250548,1.22751,-1.76607,0.157323,1.12495,-1.7798,0.216083,1.06894,-1.80346,0.322576,1.03076,-1.79709,0.281076,0.850097,-1.78118,0.25351,0.780464,-1.79213,0.179645,0.890522,-1.81189,0.152318,0.823101,-1.81557,0.117977,0.59314,-1.81828,0.013124,0.538461,-1.90044,0.263881,0.522436,-1.83744,0.223873,0.478371,-1.76905,-0.013569,1.10849,-2.1225,-0.039355,1.0301,-2.1026,-0.149212,1.15083,-2.10204,-0.171958,1.07191,-2.08708,-0.175177,0.921772,-2.08208,-0.242361,0.743793,-2.07043,-0.337324,0.808126,-2.08217,-0.373324,0.747385,-2.06947,-0.476929,0.532656,-2.03692,-0.581191,0.530091,-1.96903,-0.380454,0.377897,-2.0042,-0.437658,0.351776,-2.06327,0.030903,1.33628,-1.83818,0.023033,1.32316,-2.05287},
/*106*/{0.09771,1.9122,-1.74359,0.193194,1.85229,-1.87727,-0.05551,1.83126,-1.67697,-0.180304,1.77055,-1.64727,-0.059968,1.66634,-1.67434,0.000377,1.60094,-1.67287,0.151,1.92863,-2.12737,0.208261,1.85136,-1.95219,0.008198,1.87926,-2.06274,0.043953,1.68873,-2.23225,0.114726,1.60745,-2.2607,0.249484,1.63173,-2.1177,0.31378,1.62945,-2.10525,0.111513,1.42549,-1.83414,-0.07423,1.48998,-1.89762,-0.068598,1.50906,-1.96072,-0.065927,1.48504,-2.01211,0.114345,1.40152,-2.0914,0.250669,1.22944,-1.76625,0.157124,1.12832,-1.77881,0.214849,1.07066,-1.80253,0.320339,1.03133,-1.79751,0.287998,0.847949,-1.78124,0.263832,0.776765,-1.79068,0.184701,0.883599,-1.8127,0.161728,0.815004,-1.81658,0.137719,0.582421,-1.81911,0.036308,0.523347,-1.901,0.287783,0.520351,-1.83805,0.249566,0.472057,-1.77181,-0.01611,1.11132,-2.12236,-0.044077,1.03411,-2.10262,-0.150789,1.1571,-2.1023,-0.176087,1.07978,-2.08735,-0.1846,0.929132,-2.08296,-0.252542,0.753138,-2.07211,-0.34704,0.815786,-2.08158,-0.382611,0.754205,-2.06975,-0.486975,0.54008,-2.0379,-0.588639,0.546172,-1.96878,-0.397587,0.378203,-2.00482,-0.460305,0.3559,-2.06286,0.031384,1.34002,-1.83737,0.023953,1.32481,-2.05194},
/*107*/{0.097565,1.91443,-1.74363,0.194158,1.85535,-1.8755,-0.056768,1.83557,-1.67836,-0.182833,1.77504,-1.65132,-0.061936,1.66991,-1.67543,-0.001081,1.60418,-1.67257,0.152893,1.93193,-2.12623,0.209627,1.8545,-1.95073,0.010399,1.881,-2.06271,0.047102,1.69261,-2.23361,0.118709,1.611,-2.26102,0.250152,1.63792,-2.11621,0.314247,1.63676,-2.10276,0.112249,1.42895,-1.83414,-0.071538,1.49427,-1.89851,-0.066489,1.51207,-1.96244,-0.06373,1.48891,-2.01324,0.116526,1.40221,-2.09031,0.250965,1.23171,-1.76716,0.156066,1.13071,-1.778,0.213001,1.07379,-1.80233,0.318769,1.03081,-1.79809,0.294141,0.845151,-1.78137,0.273817,0.772954,-1.79113,0.189569,0.876296,-1.81473,0.170936,0.806274,-1.81685,0.15778,0.571755,-1.81962,0.059154,0.508291,-1.90149,0.310677,0.516828,-1.84046,0.27614,0.466853,-1.7734,-0.018998,1.11437,-2.12242,-0.048663,1.03692,-2.10299,-0.152843,1.1629,-2.10224,-0.17974,1.08633,-2.08661,-0.19318,0.936766,-2.08297,-0.261164,0.759892,-2.07353,-0.354878,0.823896,-2.08255,-0.391871,0.762683,-2.07085,-0.492774,0.548537,-2.04036,-0.594116,0.561706,-1.96842,-0.416054,0.379183,-2.00683,-0.481323,0.362082,-2.06223,0.031643,1.34405,-1.83656,0.02516,1.32678,-2.051},
/*108*/{0.097591,1.91649,-1.74338,0.194668,1.85745,-1.87482,-0.057975,1.83869,-1.68028,-0.183741,1.77839,-1.65405,-0.063303,1.67328,-1.67599,-0.003126,1.60733,-1.67238,0.154765,1.93427,-2.12451,0.210052,1.85708,-1.94984,0.012191,1.88344,-2.06335,0.052999,1.69606,-2.23556,0.121796,1.61476,-2.26173,0.251412,1.64277,-2.11528,0.315053,1.64222,-2.10057,0.113548,1.43203,-1.83443,-0.069334,1.49751,-1.89822,-0.063981,1.51516,-1.96311,-0.060417,1.49156,-2.01389,0.118062,1.40347,-2.09005,0.253512,1.23221,-1.76743,0.154854,1.1341,-1.77781,0.21138,1.07445,-1.79943,0.316858,1.02938,-1.79765,0.300138,0.843073,-1.78128,0.283333,0.768888,-1.79114,0.194722,0.868119,-1.81591,0.178366,0.798444,-1.81857,0.177774,0.562055,-1.82048,0.082121,0.494618,-1.90191,0.33308,0.514144,-1.84191,0.302113,0.460895,-1.77627,-0.02135,1.11705,-2.12273,-0.05323,1.04085,-2.10334,-0.154397,1.16801,-2.1013,-0.182896,1.09194,-2.08642,-0.198619,0.944174,-2.08251,-0.268155,0.767745,-2.07506,-0.362131,0.830445,-2.08177,-0.398237,0.77062,-2.07269,-0.502325,0.556957,-2.04212,-0.599215,0.57823,-1.96734,-0.432847,0.378549,-2.00878,-0.50261,0.368349,-2.0605,0.032961,1.34727,-1.83604,0.026598,1.3288,-2.05039},
/*109*/{0.097719,1.91884,-1.74312,0.195136,1.85965,-1.87419,-0.058366,1.84158,-1.68165,-0.185567,1.78113,-1.65695,-0.064454,1.67632,-1.67726,-0.003449,1.61017,-1.67202,0.156336,1.93686,-2.1236,0.211002,1.85973,-1.94845,0.014308,1.88527,-2.06351,0.057202,1.69948,-2.23733,0.125326,1.61774,-2.26233,0.252504,1.64735,-2.11445,0.315692,1.64731,-2.09851,0.112689,1.43489,-1.83632,-0.067754,1.50102,-1.89879,-0.062147,1.51878,-1.96421,-0.058201,1.4943,-2.01364,0.12009,1.40437,-2.08963,0.253309,1.23215,-1.76774,0.153313,1.1358,-1.77767,0.208688,1.07482,-1.80113,0.313703,1.02791,-1.79863,0.305431,0.840418,-1.78079,0.292175,0.76565,-1.7913,0.199102,0.859483,-1.8171,0.186379,0.789082,-1.81927,0.197674,0.552182,-1.82041,0.10491,0.481868,-1.90273,0.35542,0.512059,-1.84373,0.327316,0.456993,-1.77922,-0.023997,1.11998,-2.1236,-0.055145,1.04378,-2.1046,-0.155916,1.17347,-2.10106,-0.18456,1.09651,-2.0852,-0.202697,0.950096,-2.08203,-0.27498,0.774017,-2.07585,-0.367446,0.839843,-2.08304,-0.405111,0.779101,-2.07278,-0.508435,0.566028,-2.04311,-0.601893,0.594294,-1.96827,-0.450014,0.379462,-2.00945,-0.524833,0.375927,-2.05978,0.032817,1.35039,-1.83594,0.027111,1.33094,-2.05022},
/*110*/{0.097667,1.92007,-1.7432,0.196158,1.86181,-1.87329,-0.059182,1.84411,-1.68327,-0.185824,1.78337,-1.65951,-0.065081,1.67856,-1.67768,-0.005272,1.6121,-1.67162,0.15854,1.93912,-2.12184,0.211631,1.86156,-1.94751,0.015327,1.88775,-2.06473,0.060723,1.70288,-2.23863,0.128618,1.62002,-2.26353,0.252985,1.65143,-2.11399,0.315918,1.65149,-2.09697,0.114624,1.4384,-1.83624,-0.066257,1.50417,-1.89967,-0.060614,1.52191,-1.96464,-0.056176,1.49727,-2.01448,0.121538,1.40603,-2.08885,0.251359,1.23074,-1.76823,0.151136,1.13587,-1.77858,0.205624,1.07508,-1.80091,0.311084,1.02552,-1.79829,0.309797,0.837248,-1.7809,0.301638,0.762334,-1.79104,0.203188,0.851695,-1.81692,0.193817,0.779645,-1.81991,0.217217,0.543373,-1.82145,0.127526,0.470097,-1.9028,0.377575,0.511974,-1.84604,0.351707,0.453308,-1.78217,-0.025656,1.12294,-2.12385,-0.059018,1.04806,-2.10603,-0.156528,1.17722,-2.10055,-0.185724,1.10078,-2.08472,-0.20476,0.954272,-2.08145,-0.280109,0.78041,-2.07704,-0.371297,0.847204,-2.08251,-0.410385,0.786886,-2.07368,-0.523595,0.574607,-2.04393,-0.606279,0.611085,-1.96913,-0.469046,0.377191,-2.00886,-0.544592,0.385232,-2.05776,0.034094,1.3541,-1.83532,0.028599,1.33315,-2.04947},
/*111*/{0.097953,1.92189,-1.74287,0.197398,1.86155,-1.87205,-0.059782,1.84652,-1.68498,-0.185865,1.78527,-1.66129,-0.066352,1.67997,-1.67858,-0.005974,1.61332,-1.67081,0.158495,1.94068,-2.12153,0.213155,1.86363,-1.94673,0.01734,1.88931,-2.06474,0.065743,1.70552,-2.24076,0.131184,1.62223,-2.26439,0.253342,1.6541,-2.11338,0.316297,1.65479,-2.09573,0.115136,1.44083,-1.83615,-0.064132,1.50714,-1.89934,-0.059863,1.52489,-1.96487,-0.0545,1.49986,-2.01438,0.121883,1.40753,-2.08847,0.250994,1.22975,-1.76833,0.15056,1.13647,-1.77904,0.202125,1.07372,-1.80086,0.307652,1.0228,-1.79824,0.31485,0.832759,-1.78133,0.308738,0.759308,-1.79157,0.206871,0.842321,-1.8164,0.200864,0.770435,-1.81957,0.237182,0.535336,-1.82186,0.149494,0.458041,-1.90354,0.397212,0.509705,-1.84849,0.373939,0.450156,-1.78507,-0.026396,1.12583,-2.12456,-0.05886,1.0501,-2.10667,-0.157645,1.18097,-2.09923,-0.18741,1.10455,-2.08343,-0.204921,0.957844,-2.08144,-0.284625,0.784926,-2.07814,-0.375437,0.852888,-2.08207,-0.414596,0.79404,-2.07388,-0.526519,0.585725,-2.04674,-0.610266,0.626314,-1.97108,-0.486034,0.376287,-2.00907,-0.560465,0.395284,-2.05634,0.034541,1.35691,-1.83479,0.028792,1.33523,-2.04885},
/*112*/{0.098533,1.92303,-1.74231,0.191808,1.866,-1.87308,-0.06034,1.84844,-1.68623,-0.190168,1.78636,-1.66445,-0.066587,1.68113,-1.67879,-0.00604,1.61457,-1.67083,0.159991,1.94201,-2.1209,0.213521,1.86636,-1.94645,0.018724,1.89097,-2.0645,0.070448,1.70787,-2.24269,0.134233,1.62339,-2.26513,0.254116,1.65695,-2.11359,0.317145,1.65632,-2.09431,0.115979,1.44294,-1.83668,-0.063289,1.50916,-1.89944,-0.057967,1.52711,-1.96442,-0.052951,1.503,-2.01473,0.122836,1.40915,-2.08804,0.248384,1.22674,-1.76888,0.146721,1.13612,-1.77949,0.19817,1.07256,-1.80122,0.303772,1.01966,-1.79791,0.318592,0.82938,-1.78153,0.317107,0.754214,-1.79188,0.210216,0.83388,-1.81476,0.207874,0.761262,-1.81892,0.25554,0.52792,-1.82263,0.170163,0.447467,-1.90419,0.415917,0.510072,-1.85142,0.397292,0.448101,-1.78626,-0.026244,1.12841,-2.12495,-0.059582,1.0525,-2.10715,-0.156344,1.18337,-2.09796,-0.186578,1.1076,-2.08407,-0.204481,0.960049,-2.08139,-0.288352,0.788143,-2.07872,-0.37786,0.858733,-2.08212,-0.418191,0.801015,-2.07436,-0.53343,0.595904,-2.04793,-0.613871,0.642272,-1.97248,-0.508692,0.380071,-2.00806,-0.581519,0.406523,-2.0556,0.035163,1.35896,-1.83485,0.029532,1.33743,-2.04893},
/*113*/{0.098747,1.92466,-1.74214,0.192504,1.86722,-1.87228,-0.059992,1.84924,-1.68733,-0.189857,1.78685,-1.66676,-0.067046,1.68223,-1.67966,-0.005763,1.61517,-1.66984,0.161652,1.94317,-2.12027,0.214169,1.86712,-1.94526,0.020035,1.89337,-2.06498,0.073657,1.70979,-2.24311,0.135992,1.6248,-2.26579,0.255401,1.65839,-2.11321,0.317348,1.65797,-2.09362,0.116713,1.44606,-1.8366,-0.062933,1.51144,-1.90054,-0.05747,1.52921,-1.96498,-0.052287,1.50531,-2.01543,0.124333,1.41165,-2.08791,0.24577,1.22346,-1.76905,0.142653,1.13426,-1.78094,0.194111,1.06947,-1.80141,0.30028,1.01489,-1.7976,0.321975,0.82517,-1.78214,0.324181,0.749929,-1.79309,0.213801,0.824702,-1.81155,0.214564,0.752173,-1.8173,0.271616,0.520634,-1.8247,0.190827,0.437207,-1.90513,0.434638,0.511003,-1.85282,0.418429,0.447136,-1.7886,-0.025463,1.13001,-2.12541,-0.058598,1.05379,-2.10769,-0.155413,1.18448,-2.09709,-0.185168,1.10905,-2.0824,-0.203326,0.961104,-2.08156,-0.291267,0.7911,-2.07921,-0.377516,0.86562,-2.08251,-0.42065,0.807606,-2.07433,-0.540694,0.604933,-2.04916,-0.617757,0.657177,-1.97432,-0.527393,0.393265,-2.00589,-0.598936,0.418178,-2.05375,0.035981,1.36181,-1.83493,0.03111,1.33975,-2.04897},
/*114*/{0.099471,1.92529,-1.7419,0.19939,1.86449,-1.86973,-0.059591,1.8498,-1.688,-0.189805,1.78674,-1.66805,-0.066796,1.68281,-1.67977,-0.005539,1.61488,-1.6689,0.162882,1.94369,-2.11958,0.216201,1.86788,-1.94456,0.022113,1.89472,-2.06518,0.077291,1.71193,-2.24449,0.137609,1.62557,-2.2662,0.255576,1.66053,-2.11288,0.317565,1.65812,-2.09287,0.117007,1.44752,-1.83687,-0.062512,1.51418,-1.89978,-0.056971,1.53187,-1.96531,-0.051979,1.50726,-2.01533,0.124457,1.4138,-2.08787,0.243421,1.22051,-1.76983,0.140402,1.13472,-1.77967,0.189096,1.06694,-1.80122,0.297656,1.01064,-1.79748,0.327165,0.821048,-1.78222,0.332247,0.746459,-1.79367,0.216771,0.815024,-1.80886,0.220904,0.743636,-1.81544,0.2872,0.514479,-1.8249,0.209959,0.427876,-1.90508,0.450963,0.511761,-1.85603,0.438346,0.447865,-1.79156,-0.02395,1.1314,-2.12531,-0.056784,1.05468,-2.1072,-0.154599,1.18543,-2.09684,-0.183655,1.10924,-2.08176,-0.20123,0.961662,-2.082,-0.292948,0.792897,-2.07897,-0.377562,0.868653,-2.08255,-0.421805,0.813115,-2.07413,-0.549519,0.615039,-2.04956,-0.625048,0.672118,-1.97721,-0.544126,0.403636,-2.00312,-0.6136,0.431463,-2.05165,0.035999,1.36378,-1.83502,0.030904,1.34199,-2.04909},
/*115*/{0.100052,1.9256,-1.7417,0.200715,1.86623,-1.86925,-0.059147,1.84996,-1.68872,-0.190783,1.78589,-1.67016,-0.066796,1.68281,-1.67977,-0.004821,1.61372,-1.66846,0.1649,1.94437,-2.11947,0.216357,1.86872,-1.94426,0.023086,1.89464,-2.06415,0.079094,1.71272,-2.24524,0.138902,1.62599,-2.26716,0.255984,1.65972,-2.11245,0.318086,1.65741,-2.09227,0.117156,1.44899,-1.83742,-0.062141,1.51526,-1.90082,-0.056012,1.5343,-1.96455,-0.05103,1.50965,-2.01525,0.125477,1.41535,-2.08754,0.242234,1.21525,-1.7691,0.134036,1.13007,-1.78065,0.183634,1.06369,-1.80113,0.293821,1.00577,-1.79764,0.331224,0.816433,-1.78336,0.338958,0.742462,-1.79518,0.220325,0.806794,-1.80586,0.227958,0.734848,-1.81257,0.30443,0.509215,-1.82598,0.228769,0.419297,-1.90572,0.465841,0.513905,-1.85804,0.457411,0.449288,-1.79345,-0.020868,1.13281,-2.12566,-0.052795,1.05603,-2.10866,-0.152097,1.18508,-2.09624,-0.180662,1.10966,-2.08174,-0.198084,0.961231,-2.08193,-0.294648,0.793958,-2.07875,-0.376749,0.872457,-2.08351,-0.421849,0.8186,-2.07471,-0.554844,0.627839,-2.04934,-0.630917,0.686806,-1.97923,-0.558445,0.414826,-1.99951,-0.626327,0.445051,-2.04925,0.03607,1.36515,-1.8354,0.031483,1.34399,-2.04954},
/*116*/{0.100714,1.92592,-1.74146,0.202349,1.86564,-1.86908,-0.059068,1.84936,-1.68904,-0.190855,1.78416,-1.67184,-0.065377,1.68093,-1.68036,-0.003974,1.61267,-1.66654,0.166357,1.94469,-2.11902,0.218094,1.86958,-1.94377,0.024394,1.89719,-2.06453,0.082355,1.7134,-2.24552,0.139531,1.62608,-2.26777,0.256545,1.65823,-2.11166,0.318982,1.65624,-2.09219,0.116887,1.45067,-1.83737,-0.061406,1.51607,-1.90014,-0.05588,1.53545,-1.96444,-0.05079,1.51103,-2.01471,0.126589,1.4175,-2.08727,0.238221,1.21008,-1.7699,0.132172,1.12969,-1.77904,0.177027,1.05973,-1.80234,0.289704,1.00031,-1.79754,0.335208,0.81267,-1.78466,0.344817,0.738397,-1.79767,0.224497,0.797275,-1.80308,0.233534,0.727729,-1.81074,0.316394,0.504391,-1.82611,0.245412,0.411283,-1.90731,0.48018,0.515859,-1.8593,0.473178,0.450942,-1.7958,-0.018507,1.13322,-2.12545,-0.049086,1.05557,-2.108,-0.149753,1.18532,-2.09631,-0.176852,1.10776,-2.08115,-0.193885,0.959688,-2.08179,-0.295486,0.79476,-2.0785,-0.374266,0.87625,-2.08435,-0.421273,0.824413,-2.07635,-0.564733,0.637796,-2.0502,-0.637714,0.700968,-1.98206,-0.574388,0.427479,-1.99716,-0.637793,0.458462,-2.04842,0.036721,1.36631,-1.83536,0.032568,1.34574,-2.04957},
/*117*/{0.101365,1.92615,-1.74083,0.202533,1.86567,-1.86901,-0.058936,1.84876,-1.68951,-0.189454,1.78316,-1.67303,-0.063968,1.67907,-1.6794,-0.002103,1.61047,-1.66493,0.167181,1.94457,-2.11856,0.218514,1.86954,-1.9429,0.025731,1.89842,-2.06435,0.083168,1.71445,-2.2462,0.139529,1.6257,-2.26806,0.257174,1.65643,-2.1119,0.318796,1.65331,-2.09187,0.119213,1.45168,-1.83756,-0.06177,1.51698,-1.89948,-0.055789,1.53577,-1.9642,-0.050796,1.51123,-2.01418,0.126457,1.41924,-2.08725,0.235398,1.20504,-1.7702,0.124013,1.12337,-1.78012,0.170711,1.05586,-1.8025,0.285605,0.994223,-1.79733,0.339656,0.809412,-1.78589,0.350923,0.735438,-1.79915,0.227685,0.790354,-1.79966,0.240013,0.719769,-1.80787,0.333004,0.501049,-1.82748,0.260363,0.40467,-1.90696,0.490748,0.517035,-1.86132,0.488233,0.452331,-1.79762,-0.015188,1.13301,-2.12534,-0.044746,1.05511,-2.10804,-0.147371,1.18351,-2.09597,-0.173901,1.10566,-2.08084,-0.188729,0.957725,-2.08189,-0.295842,0.79506,-2.07779,-0.371682,0.880108,-2.08529,-0.422045,0.831024,-2.0775,-0.570726,0.650077,-2.04989,-0.644229,0.716693,-1.98478,-0.586071,0.440994,-1.99437,-0.651484,0.472436,-2.04736,0.038617,1.36733,-1.83521,0.033329,1.34667,-2.04939},
/*118*/{0.102437,1.92572,-1.73999,0.203276,1.86691,-1.86912,-0.058538,1.84698,-1.69091,-0.187785,1.78014,-1.67378,-0.063182,1.67681,-1.67952,-0.001174,1.60781,-1.66398,0.169047,1.94351,-2.11833,0.218995,1.86957,-1.94255,0.02648,1.89832,-2.06389,0.083878,1.71369,-2.24529,0.13906,1.62485,-2.2678,0.25788,1.6544,-2.11172,0.31974,1.65025,-2.09207,0.120372,1.45197,-1.83764,-0.061638,1.51655,-1.89992,-0.055856,1.53638,-1.96368,-0.050149,1.51235,-2.01362,0.127737,1.42103,-2.08708,0.230708,1.19917,-1.77032,0.123876,1.12322,-1.77917,0.163834,1.05091,-1.80374,0.280691,0.988198,-1.79708,0.34169,0.806728,-1.78778,0.355938,0.732977,-1.80155,0.231565,0.783395,-1.7977,0.245607,0.714215,-1.80524,0.340268,0.497188,-1.82762,0.273258,0.40014,-1.9067,0.50166,0.517143,-1.86228,0.500482,0.451136,-1.79955,-0.010928,1.13227,-2.12474,-0.040237,1.05357,-2.10767,-0.143165,1.18071,-2.09607,-0.169124,1.1029,-2.08091,-0.182726,0.955097,-2.08145,-0.296149,0.796024,-2.07722,-0.368244,0.883431,-2.08605,-0.420126,0.835281,-2.07866,-0.578261,0.662307,-2.05023,-0.649939,0.731098,-1.98675,-0.599743,0.453592,-1.99099,-0.660487,0.486145,-2.04636,0.039655,1.3673,-1.83582,0.034751,1.34801,-2.05013},
/*119*/{0.102843,1.92555,-1.73987,0.203536,1.86635,-1.86843,-0.057236,1.84514,-1.69045,-0.186435,1.77706,-1.67421,-0.060996,1.67378,-1.67885,0.000628,1.60506,-1.66228,0.170398,1.94267,-2.11788,0.219321,1.86956,-1.94242,0.027241,1.89912,-2.06311,0.083394,1.71367,-2.24485,0.138262,1.62415,-2.26778,0.25752,1.65155,-2.11209,0.320177,1.64604,-2.0918,0.120955,1.45212,-1.8379,-0.061712,1.51645,-1.89865,-0.055865,1.53648,-1.96358,-0.050427,1.51228,-2.01265,0.128055,1.42207,-2.0871,0.227618,1.19327,-1.77031,0.113678,1.11606,-1.7826,0.156923,1.04735,-1.80537,0.274614,0.982232,-1.79807,0.343719,0.80421,-1.79023,0.360262,0.730949,-1.80302,0.233704,0.778083,-1.79693,0.250005,0.708768,-1.80413,0.350121,0.493876,-1.82735,0.283473,0.396104,-1.9068,0.510568,0.517089,-1.8643,0.510391,0.451155,-1.8014,-0.006488,1.13107,-2.12428,-0.034591,1.05211,-2.10818,-0.139763,1.17811,-2.09609,-0.164262,1.09968,-2.08117,-0.176544,0.952221,-2.08082,-0.295157,0.797156,-2.0763,-0.362996,0.887953,-2.08732,-0.417595,0.842981,-2.0786,-0.583367,0.675589,-2.05034,-0.655199,0.746753,-1.98949,-0.609734,0.468636,-1.98777,-0.669271,0.500869,-2.04516,0.040577,1.36726,-1.83582,0.035176,1.34857,-2.05018},
/*120*/{0.104568,1.92453,-1.73899,0.20393,1.86634,-1.86807,-0.056627,1.84241,-1.69048,-0.184506,1.77356,-1.67434,-0.059209,1.67085,-1.6785,0.002456,1.60166,-1.66018,0.170886,1.94164,-2.11718,0.21994,1.86892,-1.94168,0.028257,1.89965,-2.06304,0.082098,1.71283,-2.24388,0.136387,1.62374,-2.26721,0.257233,1.64855,-2.11258,0.320531,1.64144,-2.09249,0.121922,1.45215,-1.83794,-0.061098,1.51526,-1.89813,-0.05596,1.53648,-1.96347,-0.050536,1.51245,-2.01217,0.128527,1.42302,-2.08733,0.222716,1.18683,-1.77072,0.108021,1.11291,-1.78319,0.148696,1.04192,-1.80592,0.268685,0.976684,-1.79858,0.343707,0.801987,-1.79171,0.361448,0.728521,-1.8044,0.235343,0.773078,-1.79641,0.252939,0.703651,-1.80351,0.35663,0.489967,-1.82655,0.292202,0.392698,-1.90618,0.517764,0.517034,-1.86603,0.520182,0.451324,-1.80374,-0.001513,1.1295,-2.12361,-0.028612,1.05044,-2.1069,-0.135917,1.17355,-2.09569,-0.158948,1.09588,-2.08068,-0.170157,0.949031,-2.08034,-0.296344,0.797356,-2.07494,-0.3593,0.891187,-2.08766,-0.415318,0.848456,-2.07869,-0.589135,0.6885,-2.04931,-0.658913,0.76287,-1.99159,-0.618989,0.484142,-1.98542,-0.676282,0.516989,-2.04428,0.041796,1.36686,-1.8362,0.036046,1.34903,-2.05062},
/*121*/{0.105838,1.92305,-1.73815,0.204828,1.86634,-1.86822,-0.0546,1.83933,-1.69038,-0.181978,1.76918,-1.67467,-0.057082,1.66705,-1.67638,0.004379,1.59754,-1.65833,0.172424,1.9402,-2.11713,0.220769,1.8684,-1.94171,0.028777,1.89963,-2.06263,0.080026,1.71214,-2.24296,0.134445,1.62248,-2.26668,0.258167,1.6437,-2.11294,0.320448,1.63567,-2.09284,0.122519,1.45218,-1.83832,-0.060787,1.51394,-1.89823,-0.056234,1.53565,-1.96283,-0.050723,1.51181,-2.01213,0.128735,1.42333,-2.0874,0.219669,1.18045,-1.77054,0.101756,1.10832,-1.78431,0.14321,1.03801,-1.80717,0.261869,0.971407,-1.79874,0.342464,0.800589,-1.79296,0.359506,0.727879,-1.80526,0.234122,0.768295,-1.79726,0.253589,0.699938,-1.80479,0.363086,0.488883,-1.82732,0.298612,0.389192,-1.90592,0.523005,0.516395,-1.86739,0.526976,0.449533,-1.80496,0.003679,1.12765,-2.123,-0.021689,1.04826,-2.1065,-0.131575,1.16938,-2.09577,-0.153853,1.09026,-2.08036,-0.162387,0.945131,-2.07981,-0.294216,0.799013,-2.0734,-0.352381,0.89524,-2.088,-0.410714,0.85633,-2.08025,-0.592005,0.702545,-2.04952,-0.659066,0.777846,-1.99252,-0.628564,0.49898,-1.98324,-0.682558,0.532879,-2.04346,0.042902,1.36633,-1.83642,0.036807,1.3488,-2.05086},
/*122*/{0.10691,1.92146,-1.73783,0.205115,1.8652,-1.8676,-0.053388,1.836,-1.68928,-0.179733,1.76485,-1.67393,-0.053985,1.6623,-1.67499,0.007336,1.59352,-1.65635,0.172931,1.93853,-2.11659,0.220002,1.86718,-1.94142,0.029229,1.89934,-2.06237,0.076569,1.7117,-2.24155,0.132405,1.62149,-2.26606,0.258429,1.63854,-2.11336,0.32054,1.62917,-2.09385,0.121948,1.45119,-1.8386,-0.06111,1.51254,-1.89789,-0.056346,1.53522,-1.96285,-0.050938,1.51012,-2.01176,0.129699,1.4234,-2.08748,0.214642,1.17491,-1.77042,0.098234,1.10708,-1.78376,0.135923,1.03424,-1.80838,0.255104,0.965566,-1.79924,0.338894,0.797802,-1.79334,0.359443,0.724968,-1.80459,0.231432,0.763795,-1.79988,0.252812,0.695778,-1.80571,0.366459,0.485898,-1.82794,0.303365,0.385904,-1.90606,0.526132,0.51547,-1.86791,0.53096,0.448879,-1.80618,0.008964,1.12514,-2.12201,-0.015098,1.04483,-2.10571,-0.126538,1.16463,-2.09554,-0.147341,1.08608,-2.08049,-0.155108,0.940331,-2.08246,-0.290759,0.801066,-2.07283,-0.347049,0.89925,-2.08841,-0.406891,0.861162,-2.07939,-0.593292,0.716819,-2.0495,-0.659171,0.795729,-1.99454,-0.635777,0.515527,-1.98291,-0.689607,0.551234,-2.04372,0.043078,1.36506,-1.83678,0.03752,1.34836,-2.0513},
/*123*/{0.108045,1.91935,-1.73727,0.206363,1.86285,-1.8675,-0.051689,1.83154,-1.68784,-0.176052,1.76009,-1.67266,-0.051717,1.65772,-1.67326,0.009928,1.58879,-1.65454,0.172494,1.93646,-2.11595,0.220589,1.86576,-1.94135,0.029106,1.89867,-2.06128,0.074043,1.71081,-2.24056,0.129314,1.6204,-2.26511,0.258297,1.63388,-2.11426,0.319976,1.62285,-2.09543,0.12213,1.44954,-1.83854,-0.060318,1.50986,-1.89772,-0.056029,1.53312,-1.96204,-0.050648,1.50844,-2.01152,0.129941,1.42331,-2.08755,0.211589,1.169,-1.76943,0.091705,1.10046,-1.78356,0.130656,1.02952,-1.80878,0.248261,0.960033,-1.79895,0.335093,0.794828,-1.79306,0.356729,0.724092,-1.80607,0.228361,0.759753,-1.80206,0.250723,0.692302,-1.80798,0.36817,0.483114,-1.8288,0.30599,0.382339,-1.90653,0.527204,0.513945,-1.86898,0.534554,0.448272,-1.80761,0.014787,1.12245,-2.12172,-0.007612,1.04247,-2.10539,-0.123134,1.15913,-2.09572,-0.140926,1.07997,-2.07971,-0.147052,0.935406,-2.07853,-0.287102,0.802557,-2.07146,-0.339646,0.902505,-2.0875,-0.400447,0.867199,-2.07912,-0.591278,0.732093,-2.05157,-0.657494,0.813623,-1.99652,-0.643523,0.531882,-1.98237,-0.696601,0.56846,-2.04459,0.044099,1.36279,-1.8373,0.038576,1.34741,-2.05192},
/*124*/{0.109376,1.91674,-1.7362,0.206665,1.86156,-1.86749,-0.050355,1.82775,-1.68752,-0.173943,1.75423,-1.6716,-0.048093,1.65229,-1.67164,0.013555,1.58435,-1.65241,0.173206,1.93427,-2.11575,0.220749,1.86427,-1.94103,0.029204,1.89756,-2.06016,0.070225,1.70979,-2.23944,0.126065,1.61863,-2.26418,0.258855,1.62899,-2.11582,0.319382,1.61573,-2.09668,0.122052,1.44805,-1.8388,-0.061021,1.50784,-1.89701,-0.055693,1.53129,-1.96215,-0.050213,1.50668,-2.01102,0.130425,1.42303,-2.08774,0.20912,1.16261,-1.76809,0.088964,1.09902,-1.78611,0.126061,1.02636,-1.8103,0.240621,0.954906,-1.7999,0.330785,0.791307,-1.79277,0.353743,0.718098,-1.80399,0.224521,0.754797,-1.8039,0.246992,0.687607,-1.81017,0.368063,0.479506,-1.82997,0.307503,0.378175,-1.90814,0.527602,0.511679,-1.86976,0.533932,0.446779,-1.80876,0.019973,1.11979,-2.12108,-0.000863,1.03875,-2.10466,-0.118126,1.15336,-2.09536,-0.13494,1.0737,-2.07908,-0.138543,0.930093,-2.07903,-0.284786,0.8033,-2.07132,-0.331577,0.905613,-2.08652,-0.394597,0.872172,-2.07725,-0.590584,0.746393,-2.05178,-0.652176,0.830061,-1.99593,-0.649552,0.548746,-1.98272,-0.701442,0.586994,-2.04486,0.044656,1.36085,-1.83774,0.039415,1.34648,-2.05243},
/*125*/{0.110674,1.91394,-1.73556,0.20625,1.86074,-1.86806,-0.048206,1.82318,-1.68548,-0.16963,1.74844,-1.66937,-0.044977,1.64668,-1.66896,0.017506,1.57862,-1.64969,0.173224,1.93221,-2.11554,0.2207,1.86208,-1.94093,0.028991,1.89715,-2.05955,0.066978,1.70775,-2.23795,0.122635,1.6171,-2.26309,0.256846,1.62179,-2.11697,0.318497,1.60713,-2.09859,0.122366,1.44541,-1.83904,-0.060332,1.50503,-1.89654,-0.055298,1.52848,-1.96183,-0.050702,1.50413,-2.01087,0.131148,1.42204,-2.08792,0.206765,1.15807,-1.76684,0.085254,1.09641,-1.78639,0.122158,1.02277,-1.8114,0.236895,0.95097,-1.79932,0.326524,0.786378,-1.7924,0.350369,0.715461,-1.80497,0.219762,0.74947,-1.80567,0.243935,0.68258,-1.81186,0.366783,0.475866,-1.83074,0.306644,0.374603,-1.90945,0.525616,0.509942,-1.87167,0.532829,0.443741,-1.80946,0.025831,1.11627,-2.12088,0.006565,1.03471,-2.10421,-0.113182,1.14771,-2.09491,-0.128746,1.06728,-2.07825,-0.130157,0.926624,-2.07887,-0.279722,0.804575,-2.07076,-0.322724,0.908834,-2.08553,-0.386278,0.879685,-2.07677,-0.588239,0.759973,-2.05117,-0.647057,0.846213,-1.9958,-0.654583,0.564957,-1.98359,-0.704349,0.606254,-2.04525,0.045602,1.35794,-1.83834,0.040303,1.34475,-2.05311},
/*126*/{0.11146,1.91042,-1.73437,0.20688,1.85825,-1.86842,-0.046346,1.81798,-1.68416,-0.167979,1.74148,-1.66717,-0.040945,1.64093,-1.66658,0.021394,1.57328,-1.64641,0.172928,1.92886,-2.1157,0.22116,1.86002,-1.94076,0.029442,1.89487,-2.05875,0.062707,1.70634,-2.23654,0.118918,1.61521,-2.26183,0.25569,1.61477,-2.11778,0.316833,1.59851,-2.10064,0.122071,1.44273,-1.83891,-0.06081,1.50233,-1.89693,-0.055884,1.52566,-1.96171,-0.050666,1.50157,-2.01118,0.131652,1.42107,-2.08788,0.203323,1.1543,-1.76609,0.08254,1.09328,-1.78635,0.120538,1.01987,-1.81224,0.234833,0.947296,-1.79942,0.32169,0.781622,-1.79231,0.343808,0.70994,-1.80526,0.215261,0.74467,-1.80744,0.239519,0.677383,-1.81466,0.364475,0.471642,-1.83245,0.303441,0.370806,-1.91181,0.522669,0.505231,-1.87256,0.529861,0.4392,-1.81095,0.031247,1.11299,-2.12125,0.014231,1.03008,-2.10299,-0.107831,1.1414,-2.09421,-0.122736,1.06108,-2.07797,-0.121536,0.91966,-2.07677,-0.274517,0.805945,-2.07037,-0.312613,0.911058,-2.08536,-0.378496,0.883763,-2.07691,-0.585022,0.773983,-2.05153,-0.639972,0.861193,-1.99514,-0.660026,0.580964,-1.98432,-0.707849,0.624291,-2.0459,0.045599,1.35505,-1.83902,0.041113,1.34305,-2.05388},
/*127*/{0.112959,1.90673,-1.73364,0.206099,1.85521,-1.8684,-0.044691,1.81231,-1.68237,-0.164364,1.73531,-1.66452,-0.037693,1.6347,-1.6632,0.025933,1.56798,-1.64507,0.17322,1.92637,-2.11583,0.220896,1.85714,-1.9406,0.029453,1.89328,-2.05824,0.058962,1.70469,-2.23518,0.114626,1.61294,-2.26153,0.255049,1.60802,-2.1193,0.315687,1.58965,-2.10337,0.119929,1.43855,-1.83911,-0.060711,1.49916,-1.89655,-0.055561,1.52201,-1.96171,-0.051539,1.49864,-2.01108,0.132043,1.41877,-2.08794,0.200604,1.15123,-1.76493,0.081027,1.08975,-1.78714,0.119012,1.01699,-1.81333,0.233929,0.945398,-1.79958,0.317535,0.775942,-1.79236,0.341033,0.704699,-1.80468,0.21095,0.739551,-1.80875,0.235429,0.671812,-1.81573,0.360761,0.468318,-1.8345,0.298577,0.367226,-1.91506,0.518828,0.499669,-1.87305,0.524896,0.432892,-1.81161,0.035098,1.10911,-2.12151,0.021225,1.0268,-2.10303,-0.104448,1.1347,-2.09387,-0.116352,1.05411,-2.07735,-0.112367,0.914649,-2.07681,-0.268949,0.805875,-2.06966,-0.303484,0.913538,-2.08509,-0.370181,0.887801,-2.0764,-0.580022,0.787327,-2.05205,-0.630593,0.877316,-1.99545,-0.663035,0.597041,-1.98576,-0.709521,0.643662,-2.04649,0.0444,1.35082,-1.83982,0.040844,1.34049,-2.05478},
/*128*/{0.113679,1.90276,-1.73323,0.207103,1.8509,-1.86801,-0.042685,1.80607,-1.68069,-0.160975,1.72785,-1.66183,-0.033828,1.62832,-1.66031,0.030726,1.56246,-1.64271,0.171904,1.92285,-2.11599,0.220945,1.85411,-1.94088,0.02894,1.89101,-2.05724,0.054683,1.70277,-2.23327,0.110718,1.61012,-2.26043,0.252891,1.59985,-2.12118,0.313493,1.57995,-2.10608,0.119945,1.43584,-1.83839,-0.061055,1.49621,-1.89598,-0.055474,1.51908,-1.96139,-0.051393,1.49512,-2.01127,0.132402,1.41703,-2.08779,0.200583,1.14795,-1.76448,0.078515,1.08684,-1.78837,0.115846,1.01342,-1.81398,0.233385,0.943246,-1.7999,0.313014,0.771183,-1.79258,0.335638,0.699353,-1.80525,0.206996,0.734449,-1.80972,0.230418,0.666593,-1.81744,0.351964,0.4614,-1.83613,0.292352,0.363301,-1.91884,0.513061,0.49357,-1.87386,0.517158,0.42481,-1.81149,0.040833,1.1055,-2.1217,0.027459,1.02305,-2.10274,-0.099076,1.12815,-2.09331,-0.109833,1.04768,-2.07664,-0.103813,0.909006,-2.07639,-0.263103,0.805804,-2.06914,-0.292877,0.914941,-2.0848,-0.360634,0.891185,-2.07608,-0.573506,0.799629,-2.05242,-0.621291,0.891328,-1.99437,-0.664687,0.612702,-1.98712,-0.709519,0.661268,-2.04776,0.044796,1.3479,-1.83973,0.041674,1.33805,-2.05472},
/*129*/{0.11473,1.89869,-1.73229,0.206524,1.84823,-1.8685,-0.040688,1.79973,-1.67869,-0.158662,1.71938,-1.6577,-0.029384,1.62132,-1.65667,0.035282,1.55655,-1.64003,0.171568,1.91901,-2.11656,0.220917,1.85156,-1.94116,0.027918,1.88706,-2.05549,0.050005,1.70024,-2.23205,0.105994,1.60718,-2.25978,0.250368,1.59144,-2.12299,0.311054,1.56969,-2.10879,0.120057,1.43393,-1.83702,-0.061152,1.49251,-1.89645,-0.055935,1.51537,-1.96151,-0.051275,1.49169,-2.01139,0.132901,1.41501,-2.08748,0.197605,1.14501,-1.76375,0.077071,1.08489,-1.78919,0.114244,1.01033,-1.81376,0.231412,0.940425,-1.79982,0.308453,0.765181,-1.79274,0.329037,0.693718,-1.80599,0.201801,0.730349,-1.8114,0.223901,0.66281,-1.82007,0.345089,0.456457,-1.83852,0.283562,0.358739,-1.92243,0.506591,0.486011,-1.8731,0.509273,0.41555,-1.81375,0.046029,1.10125,-2.12162,0.033903,1.01914,-2.1031,-0.093673,1.12165,-2.09274,-0.103312,1.04102,-2.07573,-0.094994,0.902823,-2.07632,-0.255983,0.805722,-2.06937,-0.282611,0.916335,-2.08446,-0.350141,0.894878,-2.076,-0.56656,0.812179,-2.05329,-0.6112,0.904978,-1.99456,-0.663926,0.626889,-1.98921,-0.704283,0.678562,-2.04892,0.045511,1.34533,-1.83929,0.043174,1.33519,-2.05428},
/*130*/{0.115624,1.89356,-1.73147,0.207099,1.84488,-1.86902,-0.038308,1.79313,-1.67625,-0.154836,1.71189,-1.65359,-0.024555,1.61524,-1.65416,0.041659,1.55077,-1.63761,0.170477,1.91545,-2.11678,0.221157,1.84813,-1.94051,0.028231,1.88371,-2.05489,0.04589,1.6974,-2.23095,0.101881,1.60389,-2.25931,0.247766,1.58374,-2.12499,0.30789,1.55884,-2.11194,0.119919,1.42963,-1.83754,-0.06153,1.48887,-1.89632,-0.057039,1.51276,-1.96185,-0.051513,1.48724,-2.01154,0.132743,1.41091,-2.08825,0.195584,1.1433,-1.76485,0.074496,1.08275,-1.79038,0.112223,1.00875,-1.81351,0.229458,0.936036,-1.80014,0.303258,0.759403,-1.79335,0.324298,0.688552,-1.80763,0.196177,0.725892,-1.81175,0.217415,0.657818,-1.82103,0.335997,0.448466,-1.83978,0.273721,0.354725,-1.92605,0.497387,0.476877,-1.87373,0.500126,0.407783,-1.81555,0.050532,1.09797,-2.12141,0.040634,1.01555,-2.10349,-0.090637,1.11503,-2.0926,-0.097255,1.0341,-2.07588,-0.085936,0.897098,-2.07595,-0.248757,0.806094,-2.07001,-0.270321,0.915716,-2.08333,-0.339011,0.898989,-2.0768,-0.558303,0.822872,-2.05402,-0.599986,0.917509,-1.99455,-0.661825,0.641021,-1.99047,-0.704728,0.69603,-2.04988,0.045036,1.34144,-1.83969,0.042394,1.33135,-2.05467},
/*131*/{0.11658,1.88895,-1.73059,0.208766,1.83941,-1.86883,-0.035432,1.78617,-1.67352,-0.150756,1.7033,-1.6496,-0.019369,1.60748,-1.65015,0.047186,1.54494,-1.63473,0.169376,1.91138,-2.11699,0.22083,1.84452,-1.94096,0.027148,1.8803,-2.05452,0.042537,1.69262,-2.229,0.096744,1.6002,-2.25857,0.244295,1.57378,-2.12678,0.304825,1.54804,-2.11508,0.119778,1.42638,-1.83708,-0.061687,1.48475,-1.89691,-0.056455,1.50849,-1.96113,-0.05195,1.48328,-2.0115,0.133688,1.4086,-2.08694,0.195107,1.13867,-1.76402,0.073108,1.07993,-1.7905,0.110357,1.00527,-1.81391,0.226925,0.931316,-1.79988,0.297494,0.753099,-1.7935,0.317121,0.680777,-1.80827,0.190329,0.721459,-1.81279,0.211659,0.653247,-1.82126,0.325149,0.441495,-1.84107,0.26179,0.350492,-1.92876,0.488018,0.467821,-1.87481,0.488217,0.396506,-1.81752,0.055681,1.09388,-2.12151,0.047583,1.01115,-2.10345,-0.085176,1.10816,-2.09181,-0.089467,1.02677,-2.07543,-0.076207,0.891305,-2.0761,-0.240937,0.805612,-2.06909,-0.259788,0.916458,-2.08336,-0.329401,0.901685,-2.07672,-0.549579,0.834034,-2.05387,-0.587446,0.928754,-1.99454,-0.657382,0.654483,-1.99201,-0.698737,0.710156,-2.05214,0.045503,1.33775,-1.83949,0.043839,1.32818,-2.05451},
/*132*/{0.117993,1.88334,-1.72992,0.208701,1.83479,-1.86866,-0.0332,1.77816,-1.67093,-0.147652,1.69382,-1.64504,-0.014721,1.60013,-1.64703,0.053232,1.53911,-1.63216,0.167209,1.90725,-2.11778,0.220916,1.84041,-1.94109,0.026536,1.87562,-2.05348,0.037379,1.68926,-2.22837,0.091903,1.59661,-2.25836,0.241507,1.56462,-2.12946,0.300936,1.53669,-2.11876,0.119476,1.42322,-1.8362,-0.061535,1.48077,-1.8968,-0.056512,1.50445,-1.96116,-0.051155,1.47786,-2.01221,0.133902,1.406,-2.08643,0.193003,1.13576,-1.76406,0.070728,1.07722,-1.79238,0.107606,1.00162,-1.81421,0.223928,0.925948,-1.79961,0.291577,0.746761,-1.79467,0.310779,0.673773,-1.8085,0.183671,0.717476,-1.81414,0.203575,0.649557,-1.82315,0.315237,0.436745,-1.84208,0.249192,0.347013,-1.93069,0.477408,0.458982,-1.87578,0.476854,0.387753,-1.81916,0.061659,1.09001,-2.12205,0.054924,1.00611,-2.10377,-0.079173,1.10071,-2.09153,-0.082737,1.0196,-2.07434,-0.067086,0.885415,-2.07653,-0.233485,0.804307,-2.06977,-0.248227,0.917035,-2.08239,-0.317101,0.903288,-2.07671,-0.54034,0.842121,-2.05393,-0.575123,0.939434,-1.99454,-0.652482,0.666937,-1.99392,-0.692544,0.723512,-2.05387,0.046115,1.33421,-1.83914,0.045256,1.32445,-2.05415},
/*133*/{0.119029,1.87781,-1.72874,0.207882,1.83188,-1.86895,-0.030042,1.77011,-1.66778,-0.141801,1.68521,-1.64024,-0.008738,1.59299,-1.6438,0.059377,1.5328,-1.63004,0.16684,1.90287,-2.11842,0.221508,1.83619,-1.94109,0.025823,1.87195,-2.05196,0.032964,1.68632,-2.22747,0.086314,1.59251,-2.25829,0.237464,1.55441,-2.13213,0.297022,1.52424,-2.12253,0.120293,1.42054,-1.83574,-0.062919,1.47654,-1.89701,-0.057239,1.50083,-1.96216,-0.052455,1.47368,-2.01234,0.133195,1.40241,-2.0875,0.191456,1.13268,-1.76491,0.070663,1.07394,-1.79245,0.106337,0.99881,-1.81401,0.21985,0.920311,-1.79983,0.284505,0.740137,-1.79569,0.304003,0.667592,-1.81019,0.177023,0.712909,-1.8146,0.194569,0.643027,-1.82442,0.300215,0.431128,-1.84347,0.230598,0.353718,-1.93061,0.465837,0.447756,-1.87772,0.460711,0.376194,-1.8209,0.06731,1.08613,-2.12231,0.063172,1.00228,-2.10481,-0.073187,1.09406,-2.09095,-0.075071,1.01312,-2.07399,-0.056925,0.880296,-2.07732,-0.224973,0.804427,-2.07012,-0.23653,0.917398,-2.08249,-0.305954,0.90545,-2.07643,-0.53194,0.851962,-2.05359,-0.562705,0.947972,-1.99381,-0.644866,0.677527,-1.99636,-0.6843,0.735446,-2.05535,0.046592,1.33124,-1.83908,0.045121,1.32056,-2.05405},
/*134*/{0.11992,1.87226,-1.72832,0.209089,1.82578,-1.86874,-0.0277,1.76187,-1.6648,-0.139556,1.67586,-1.63574,-0.003165,1.58512,-1.63944,0.066439,1.52661,-1.62707,0.165855,1.89821,-2.11894,0.221466,1.83157,-1.94188,0.024713,1.86719,-2.05154,0.029487,1.68213,-2.2266,0.08146,1.58822,-2.25807,0.233722,1.54474,-2.13465,0.291531,1.5117,-2.12677,0.119927,1.41592,-1.8337,-0.063097,1.47108,-1.89852,-0.057222,1.49642,-1.96213,-0.052676,1.46861,-2.01268,0.133502,1.39954,-2.08663,0.191526,1.12797,-1.7644,0.068428,1.07109,-1.794,0.104737,0.995039,-1.8152,0.215989,0.915038,-1.80031,0.277699,0.733288,-1.79624,0.294951,0.660676,-1.81114,0.168749,0.707798,-1.81623,0.186502,0.638281,-1.82514,0.286811,0.426597,-1.84554,0.212265,0.351004,-1.93343,0.450342,0.434848,-1.88059,0.440532,0.363802,-1.82172,0.07304,1.08178,-2.12346,0.070769,0.998321,-2.10562,-0.066852,1.08706,-2.08968,-0.067256,1.00604,-2.07345,-0.04671,0.874414,-2.07798,-0.216256,0.803586,-2.07054,-0.224841,0.91666,-2.08316,-0.293992,0.906813,-2.07662,-0.518872,0.858763,-2.05435,-0.549611,0.956287,-1.99332,-0.637154,0.687067,-1.99865,-0.675266,0.746328,-2.05661,0.046301,1.32643,-1.83916,0.046488,1.31664,-2.05417},
/*135*/{0.12065,1.86667,-1.72762,0.208836,1.82166,-1.86904,-0.023669,1.75285,-1.66146,-0.134028,1.66646,-1.6305,0.00332,1.57786,-1.63586,0.073655,1.52076,-1.62458,0.164119,1.89355,-2.11934,0.221175,1.82644,-1.94213,0.024195,1.86255,-2.05066,0.02527,1.67767,-2.2264,0.076397,1.58419,-2.25832,0.229109,1.53368,-2.13785,0.286302,1.49895,-2.13085,0.120209,1.41227,-1.8325,-0.063179,1.46624,-1.8978,-0.058018,1.49177,-1.96182,-0.052759,1.46326,-2.01304,0.134142,1.39576,-2.08667,0.189598,1.12336,-1.76393,0.066584,1.06528,-1.79387,0.103484,0.990759,-1.81505,0.211799,0.910231,-1.80057,0.269429,0.726253,-1.79746,0.285618,0.653789,-1.8124,0.160687,0.702915,-1.81711,0.17658,0.6342,-1.82661,0.273601,0.422794,-1.8479,0.193518,0.354056,-1.93457,0.436983,0.421145,-1.88308,0.42344,0.350393,-1.82374,0.079386,1.07758,-2.12391,0.077762,0.994653,-2.10683,-0.061149,1.07987,-2.08965,-0.060014,0.99908,-2.07291,-0.035688,0.86894,-2.07825,-0.20624,0.802822,-2.07147,-0.212358,0.915817,-2.08342,-0.281897,0.907269,-2.07586,-0.508024,0.866435,-2.05423,-0.538362,0.963308,-1.99321,-0.628591,0.694259,-2.00206,-0.665032,0.754808,-2.05807,0.047084,1.32237,-1.83859,0.047556,1.31227,-2.05359},
/*136*/{0.121615,1.86103,-1.72689,0.209032,1.81693,-1.86949,-0.019969,1.74437,-1.65812,-0.129343,1.65573,-1.62492,0.010189,1.5704,-1.6323,0.08155,1.5153,-1.62195,0.164134,1.88882,-2.12007,0.221194,1.82209,-1.94279,0.023714,1.85719,-2.04975,0.021042,1.67406,-2.22547,0.070481,1.57948,-2.25849,0.224831,1.52372,-2.14138,0.28098,1.48644,-2.13496,0.119808,1.40896,-1.83147,-0.064285,1.46207,-1.89898,-0.058049,1.48678,-1.96155,-0.053756,1.4584,-2.01408,0.134298,1.39216,-2.08561,0.187912,1.11861,-1.76352,0.065721,1.06316,-1.79423,0.100535,0.987022,-1.81476,0.206368,0.904505,-1.80031,0.261738,0.720655,-1.79779,0.27582,0.647165,-1.81342,0.152304,0.69922,-1.81773,0.166818,0.629272,-1.82856,0.259633,0.418322,-1.85037,0.173292,0.353999,-1.93538,0.421269,0.407565,-1.8854,0.404654,0.339736,-1.82748,0.0854,1.07345,-2.12468,0.085551,0.989732,-2.10725,-0.054879,1.07355,-2.08885,-0.051701,0.992028,-2.07295,-0.025892,0.863332,-2.0793,-0.196217,0.801849,-2.07314,-0.198569,0.914631,-2.08337,-0.270375,0.908386,-2.07616,-0.494629,0.872904,-2.05411,-0.525392,0.969537,-1.99344,-0.618056,0.701283,-2.00353,-0.653104,0.76221,-2.06104,0.046811,1.31875,-1.83804,0.048392,1.308,-2.053},
/*137*/{0.123299,1.8556,-1.72579,0.210612,1.80994,-1.86904,-0.015582,1.73537,-1.65456,-0.121942,1.6465,-1.61956,0.017164,1.56319,-1.62912,0.090286,1.50938,-1.62007,0.163051,1.88456,-2.12044,0.221279,1.81699,-1.94343,0.022232,1.8524,-2.04786,0.016712,1.66969,-2.22512,0.065103,1.57576,-2.2587,0.219411,1.51228,-2.14472,0.274594,1.47382,-2.13916,0.118659,1.40474,-1.8298,-0.065667,1.45705,-1.89953,-0.059072,1.48176,-1.96139,-0.054754,1.45237,-2.0145,0.133619,1.38685,-2.08527,0.184902,1.11399,-1.76362,0.063851,1.05841,-1.79332,0.097809,0.982189,-1.81469,0.199609,0.898497,-1.80001,0.254696,0.714536,-1.7997,0.26659,0.640619,-1.81568,0.143745,0.696773,-1.81902,0.156419,0.62627,-1.82815,0.241284,0.415,-1.85311,0.152644,0.355147,-1.9345,0.405857,0.398994,-1.89103,0.385872,0.336332,-1.82738,0.09053,1.06943,-2.12567,0.093588,0.985593,-2.10797,-0.048851,1.0662,-2.08882,-0.044167,0.985563,-2.07266,-0.015029,0.857893,-2.08019,-0.18521,0.801104,-2.07358,-0.185046,0.913039,-2.08386,-0.255502,0.908771,-2.07645,-0.483108,0.877728,-2.05453,-0.512178,0.97394,-1.99125,-0.605347,0.707876,-2.0058,-0.643184,0.769584,-2.06315,0.045731,1.31436,-1.83704,0.048067,1.30248,-2.05193},
/*138*/{0.124449,1.8498,-1.72497,0.211694,1.80475,-1.86902,-0.011917,1.72579,-1.65181,-0.116782,1.63616,-1.61407,0.024872,1.55591,-1.62555,0.099217,1.50451,-1.61686,0.162122,1.88002,-2.12111,0.222076,1.81152,-1.94376,0.021843,1.84821,-2.04671,0.01327,1.66547,-2.22483,0.059188,1.57136,-2.25892,0.213403,1.50155,-2.14836,0.268125,1.46083,-2.14309,0.117674,1.40044,-1.82803,-0.067931,1.45268,-1.90035,-0.061291,1.47704,-1.96222,-0.055455,1.44774,-2.01496,0.134402,1.383,-2.08356,0.181926,1.10931,-1.76158,0.059062,1.05525,-1.79382,0.09515,0.97893,-1.81653,0.19409,0.891919,-1.7999,0.247195,0.709704,-1.8019,0.256728,0.636207,-1.81783,0.136374,0.695927,-1.81898,0.146755,0.625747,-1.82953,0.225451,0.414456,-1.85715,0.130257,0.355136,-1.93446,0.386758,0.395553,-1.89715,0.364148,0.339665,-1.82987,0.096255,1.06589,-2.12613,0.101106,0.981902,-2.10864,-0.043126,1.05926,-2.08814,-0.036129,0.97837,-2.07178,-0.003148,0.852104,-2.08112,-0.173421,0.799152,-2.07484,-0.172248,0.913286,-2.0834,-0.242005,0.908075,-2.07668,-0.468965,0.88262,-2.05457,-0.499734,0.978127,-1.99147,-0.592988,0.710797,-2.00826,-0.627565,0.772585,-2.06533,0.044061,1.31018,-1.83641,0.048668,1.29827,-2.05127},
/*139*/{0.125715,1.84515,-1.72443,0.211395,1.80011,-1.87011,-0.006106,1.71676,-1.64802,-0.10903,1.62678,-1.60848,0.03325,1.54906,-1.62213,0.108801,1.49903,-1.61471,0.16096,1.87604,-2.12167,0.22243,1.80606,-1.94509,0.020941,1.84385,-2.04508,0.008191,1.66154,-2.22428,0.053981,1.56745,-2.25911,0.207751,1.48977,-2.15196,0.260885,1.44859,-2.14779,0.115965,1.39679,-1.82756,-0.070312,1.44845,-1.90056,-0.063042,1.47099,-1.9627,-0.057861,1.44207,-2.01594,0.133664,1.37909,-2.08381,0.177229,1.10608,-1.75843,0.054793,1.05194,-1.79424,0.091154,0.977681,-1.81671,0.186934,0.884723,-1.80065,0.242084,0.709387,-1.80357,0.248388,0.635204,-1.82063,0.130308,0.698749,-1.81962,0.138049,0.628325,-1.83032,0.205754,0.418699,-1.85906,0.109018,0.355884,-1.93625,0.363881,0.392403,-1.90166,0.341472,0.340156,-1.83129,0.102482,1.06222,-2.12648,0.109412,0.978322,-2.10917,-0.036826,1.05236,-2.08866,-0.027727,0.971833,-2.07192,0.008565,0.843667,-2.08133,-0.161214,0.797556,-2.07596,-0.157957,0.911828,-2.08531,-0.226981,0.908089,-2.07654,-0.455713,0.88578,-2.05563,-0.486977,0.981332,-1.99126,-0.580387,0.714113,-2.0106,-0.611824,0.775641,-2.06786,0.04302,1.30614,-1.83624,0.048236,1.29357,-2.05105},
/*140*/{0.12706,1.84055,-1.72327,0.212156,1.79334,-1.86941,-0.000368,1.70822,-1.64507,-0.101048,1.61597,-1.6022,0.042028,1.54264,-1.61976,0.117815,1.49527,-1.61342,0.160417,1.87218,-2.12216,0.223819,1.80074,-1.94563,0.02038,1.84118,-2.04303,0.005186,1.65859,-2.22413,0.047865,1.56359,-2.25912,0.201758,1.47969,-2.15616,0.251363,1.43785,-2.15298,0.113481,1.39354,-1.82723,-0.073907,1.44508,-1.90244,-0.065189,1.46605,-1.96419,-0.059653,1.43708,-2.01746,0.132794,1.37525,-2.08259,0.173257,1.10393,-1.75425,0.050485,1.05374,-1.79426,0.08711,0.977588,-1.81821,0.182414,0.882371,-1.80132,0.236668,0.709908,-1.80436,0.240511,0.635788,-1.82155,0.124761,0.704912,-1.81857,0.129116,0.634973,-1.83113,0.184695,0.421363,-1.86146,0.086231,0.357433,-1.93672,0.342248,0.392073,-1.90309,0.319347,0.338916,-1.8308,0.107523,1.05965,-2.12667,0.117359,0.975634,-2.10977,-0.030858,1.04532,-2.08837,-0.019048,0.965025,-2.07206,0.019786,0.841432,-2.08431,-0.148039,0.795767,-2.07658,-0.143417,0.910588,-2.08578,-0.213768,0.906671,-2.07717,-0.443265,0.891528,-2.05526,-0.4724,0.982752,-1.99014,-0.564457,0.715855,-2.0134,-0.597324,0.7771,-2.07024,0.040359,1.30287,-1.83599,0.047626,1.28921,-2.05067},
/*141*/{0.1287,1.83642,-1.72265,0.213585,1.78888,-1.86985,0.005052,1.69974,-1.64217,-0.091098,1.60722,-1.59692,0.052814,1.53748,-1.6161,0.128333,1.4915,-1.61131,0.159143,1.8689,-2.12372,0.224913,1.79626,-1.94678,0.019466,1.83877,-2.04049,0.001791,1.65565,-2.22445,0.04193,1.56056,-2.25872,0.194232,1.47043,-2.16038,0.242748,1.42676,-2.15748,0.109927,1.39069,-1.82508,-0.077468,1.44212,-1.90292,-0.068734,1.462,-1.96571,-0.062395,1.43257,-2.01871,0.13238,1.37101,-2.08002,0.171766,1.10437,-1.75107,0.049835,1.05483,-1.79676,0.085722,0.978641,-1.81931,0.183381,0.886729,-1.80021,0.232025,0.710012,-1.80622,0.23271,0.635509,-1.82238,0.120272,0.710569,-1.81814,0.121017,0.639049,-1.83032,0.164514,0.42256,-1.86192,0.064812,0.358881,-1.93787,0.320276,0.390069,-1.9032,0.296971,0.338935,-1.83191,0.112938,1.05776,-2.12668,0.124934,0.974239,-2.10976,-0.02481,1.03881,-2.08841,-0.011367,0.959006,-2.0723,0.031633,0.836623,-2.08414,-0.134903,0.795219,-2.07666,-0.12767,0.908233,-2.08593,-0.198283,0.905602,-2.07735,-0.421456,0.886908,-2.05336,-0.459672,0.983947,-1.99101,-0.548993,0.715438,-2.01527,-0.580866,0.777695,-2.07279,0.036372,1.30021,-1.83436,0.046515,1.28491,-2.04881},
/*142*/{0.129625,1.83396,-1.72191,0.213229,1.7856,-1.8706,0.011633,1.69282,-1.63993,-0.082956,1.59807,-1.59172,0.061669,1.53283,-1.61447,0.139024,1.48804,-1.60923,0.159051,1.86628,-2.12491,0.225254,1.79357,-1.94845,0.019489,1.83759,-2.03864,-0.000894,1.65349,-2.22427,0.036029,1.55808,-2.25811,0.186303,1.46168,-2.163,0.233715,1.41712,-2.16179,0.107147,1.38804,-1.82377,-0.080646,1.43865,-1.90674,-0.071186,1.45865,-1.96696,-0.065514,1.42755,-2.01992,0.130939,1.36685,-2.07847,0.167521,1.10434,-1.74981,0.049367,1.05458,-1.79732,0.084231,0.978826,-1.81838,0.186806,0.893701,-1.7989,0.226012,0.709184,-1.80671,0.219869,0.635247,-1.82484,0.113376,0.714629,-1.81645,0.110381,0.643322,-1.82836,0.144439,0.421882,-1.86128,0.043371,0.359819,-1.93829,0.30039,0.389292,-1.90246,0.2763,0.338523,-1.83024,0.118798,1.05743,-2.12634,0.133035,0.974396,-2.11013,-0.019087,1.03441,-2.08894,-0.00248,0.954778,-2.07347,0.044137,0.831393,-2.08298,-0.119601,0.792647,-2.07638,-0.112492,0.903584,-2.08665,-0.182851,0.903695,-2.07779,-0.410759,0.890133,-2.05774,-0.445985,0.984617,-1.99279,-0.531307,0.714719,-2.01839,-0.564678,0.775959,-2.07468,0.032891,1.29772,-1.83366,0.045219,1.28048,-2.04785},
/*143*/{0.13055,1.83116,-1.72191,0.213701,1.78244,-1.87236,0.018144,1.68773,-1.6385,-0.073375,1.59062,-1.58704,0.072751,1.52895,-1.61186,0.151015,1.48641,-1.60827,0.159787,1.86427,-2.1255,0.22543,1.78979,-1.95033,0.02018,1.83684,-2.03651,-0.00347,1.65319,-2.22288,0.030713,1.55675,-2.25729,0.178542,1.45555,-2.1652,0.224178,1.40779,-2.1662,0.104143,1.38653,-1.82137,-0.082742,1.43653,-1.90912,-0.073376,1.45709,-1.96746,-0.067119,1.42387,-2.0224,0.129311,1.36414,-2.07704,0.165077,1.1056,-1.74896,0.046566,1.05102,-1.79556,0.084789,0.977885,-1.81877,0.19291,0.899578,-1.79773,0.218318,0.709087,-1.80718,0.207581,0.634726,-1.82483,0.105077,0.716963,-1.81465,0.099214,0.646712,-1.82696,0.127021,0.424857,-1.86246,0.022197,0.360665,-1.93795,0.279093,0.387966,-1.90149,0.255422,0.337874,-1.82974,0.123835,1.05876,-2.12555,0.141733,0.976463,-2.10951,-0.013494,1.03085,-2.08962,0.005934,0.951072,-2.07326,0.058565,0.830117,-2.08248,-0.104898,0.790986,-2.07543,-0.097215,0.902246,-2.08678,-0.167279,0.901397,-2.07846,-0.396443,0.887874,-2.05762,-0.433509,0.982872,-1.99398,-0.513715,0.711533,-2.01984,-0.547711,0.772922,-2.07706,0.02944,1.29632,-1.8328,0.044166,1.27754,-2.04671},
/*144*/{0.131738,1.82868,-1.72208,0.214139,1.78066,-1.87364,0.024571,1.68278,-1.63713,-0.061382,1.58433,-1.58295,0.084814,1.5269,-1.61198,0.162493,1.48493,-1.60764,0.160499,1.86224,-2.12554,0.225433,1.78713,-1.95133,0.020778,1.83669,-2.03651,-0.007374,1.65386,-2.22098,0.024642,1.55731,-2.25637,0.169772,1.45023,-2.16765,0.21408,1.40111,-2.17094,0.101176,1.38704,-1.81823,-0.085112,1.43544,-1.91052,-0.074634,1.45578,-1.96817,-0.070856,1.41828,-2.02458,0.128558,1.36126,-2.07632,0.163127,1.1056,-1.74965,0.0441,1.04869,-1.79593,0.082178,0.976257,-1.81869,0.198578,0.901279,-1.7999,0.207096,0.708,-1.80561,0.199898,0.634218,-1.82233,0.094556,0.719603,-1.81338,0.088658,0.649144,-1.82438,0.107419,0.424182,-1.86217,0.001257,0.36145,-1.93775,0.258833,0.387344,-1.90168,0.233799,0.337281,-1.82951,0.129753,1.06044,-2.12535,0.149995,0.979024,-2.10896,-0.007075,1.02776,-2.09106,0.015148,0.949142,-2.07533,0.073614,0.829362,-2.08187,-0.089091,0.789138,-2.07446,-0.082124,0.900135,-2.08773,-0.151291,0.900275,-2.07892,-0.380883,0.886554,-2.05936,-0.419634,0.978968,-1.99532,-0.495159,0.70688,-2.02271,-0.529358,0.768137,-2.07942,0.026881,1.29669,-1.83061,0.043369,1.27418,-2.04403},
/*145*/{0.133216,1.82681,-1.72254,0.214008,1.77928,-1.87432,0.029442,1.68018,-1.63619,-0.05151,1.57954,-1.57908,0.095621,1.52404,-1.6099,0.17495,1.4855,-1.60636,0.160887,1.86057,-2.12544,0.225396,1.78512,-1.9517,0.021446,1.83689,-2.03703,-0.011545,1.65592,-2.21872,0.017694,1.55902,-2.2558,0.160573,1.4456,-2.17019,0.204533,1.39495,-2.17525,0.098202,1.38621,-1.81535,-0.08768,1.43499,-1.91311,-0.07645,1.45603,-1.96926,-0.073255,1.41729,-2.02613,0.127114,1.35998,-2.0752,0.159355,1.10436,-1.7486,0.039311,1.04907,-1.79595,0.078239,0.975196,-1.8176,0.198857,0.89941,-1.80047,0.195561,0.705793,-1.80425,0.184491,0.631699,-1.82142,0.084018,0.720962,-1.81275,0.075325,0.650922,-1.8233,0.086866,0.423847,-1.86173,-0.019634,0.36219,-1.93767,0.238141,0.386177,-1.90154,0.212663,0.336661,-1.82901,0.13673,1.06346,-2.12482,0.159826,0.981532,-2.10808,0.001038,1.02572,-2.09184,0.024973,0.948001,-2.07668,0.08942,0.831106,-2.08029,-0.073928,0.788042,-2.07346,-0.066639,0.89777,-2.08835,-0.135904,0.897746,-2.07931,-0.366223,0.882958,-2.05918,-0.408796,0.976481,-1.99834,-0.475119,0.701937,-2.02338,-0.511418,0.760874,-2.0812,0.022702,1.29635,-1.83017,0.041626,1.27309,-2.04331},
/*146*/{0.134979,1.82568,-1.72348,0.212845,1.77772,-1.87481,0.036344,1.67846,-1.6351,-0.042001,1.57539,-1.5761,0.107484,1.52339,-1.60977,0.188096,1.48687,-1.60692,0.162383,1.85869,-2.12532,0.224488,1.78381,-1.95223,0.022253,1.83665,-2.03845,-0.016031,1.65886,-2.21606,0.011203,1.56142,-2.25517,0.151789,1.4437,-2.17291,0.194046,1.39104,-2.17916,0.095033,1.38576,-1.81317,-0.088347,1.43368,-1.91381,-0.077183,1.45551,-1.96943,-0.073962,1.41646,-2.02706,0.124859,1.35855,-2.07398,0.152696,1.10163,-1.74835,0.035108,1.04779,-1.79541,0.072213,0.973315,-1.81675,0.193654,0.895493,-1.79985,0.183315,0.702663,-1.80228,0.171929,0.629459,-1.81897,0.071794,0.719992,-1.81269,0.06124,0.650433,-1.82369,0.065526,0.425885,-1.8625,-0.041043,0.362427,-1.93704,0.216892,0.386093,-1.90013,0.190996,0.336318,-1.82871,0.145002,1.06624,-2.12432,0.169728,0.98519,-2.10789,0.009021,1.02597,-2.09436,0.035606,0.946557,-2.07795,0.106592,0.833394,-2.0796,-0.056538,0.784085,-2.07155,-0.0504,0.895961,-2.08922,-0.120148,0.895294,-2.0808,-0.346832,0.877313,-2.05913,-0.39819,0.97168,-2.00034,-0.455658,0.694445,-2.02456,-0.490601,0.751689,-2.08375,0.019769,1.29574,-1.82933,0.040235,1.27177,-2.04224},
/*147*/{0.137003,1.82525,-1.72406,0.213646,1.77337,-1.87587,0.040796,1.67687,-1.63412,-0.031332,1.57265,-1.57342,0.119281,1.52372,-1.6089,0.199999,1.48934,-1.60745,0.162723,1.85758,-2.12553,0.224618,1.78201,-1.95191,0.022074,1.83678,-2.03918,-0.020519,1.66295,-2.2132,0.003699,1.56506,-2.25484,0.143549,1.44215,-2.17588,0.184014,1.38795,-2.18276,0.092756,1.38663,-1.81063,-0.089794,1.43348,-1.91418,-0.078774,1.4561,-1.96976,-0.075788,1.41761,-2.02825,0.123364,1.35847,-2.0733,0.150447,1.09994,-1.74756,0.029571,1.04632,-1.79613,0.066798,0.971453,-1.81834,0.185029,0.891525,-1.7982,0.169784,0.699288,-1.80187,0.156212,0.625496,-1.81859,0.058463,0.720148,-1.81206,0.045857,0.649494,-1.82638,0.045545,0.424287,-1.86007,-0.061981,0.363135,-1.93668,0.195864,0.385456,-1.90021,0.169519,0.335594,-1.82774,0.152919,1.07008,-2.12454,0.18059,0.989936,-2.10737,0.018048,1.02509,-2.09582,0.047456,0.946623,-2.07919,0.123236,0.836213,-2.07822,-0.039993,0.781791,-2.06939,-0.036027,0.894464,-2.09015,-0.104724,0.893163,-2.08248,-0.335925,0.873971,-2.06159,-0.387081,0.963967,-2.00148,-0.434632,0.686239,-2.0253,-0.470754,0.741717,-2.0858,0.017064,1.2964,-1.82872,0.038853,1.27204,-2.04146},
/*148*/{0.138325,1.82561,-1.7246,0.213637,1.77325,-1.87613,0.04667,1.67634,-1.63363,-0.020729,1.57021,-1.57067,0.129806,1.52511,-1.60868,0.212907,1.49274,-1.60884,0.163579,1.857,-2.12541,0.224322,1.78198,-1.9523,0.022217,1.83673,-2.03969,-0.025937,1.6675,-2.21045,-0.002714,1.56942,-2.25481,0.134474,1.44162,-2.1788,0.173473,1.38699,-2.1858,0.090624,1.38677,-1.809,-0.091548,1.43341,-1.9143,-0.079971,1.4559,-1.96926,-0.075438,1.42188,-2.0276,0.121951,1.35922,-2.07265,0.142093,1.09456,-1.74706,0.022161,1.0435,-1.79648,0.05989,0.96878,-1.81776,0.173134,0.887104,-1.79736,0.155186,0.694946,-1.80068,0.139165,0.622534,-1.8183,0.044969,0.71877,-1.81141,0.031241,0.64959,-1.82413,0.026363,0.425581,-1.86079,-0.083161,0.363192,-1.93613,0.175143,0.385156,-1.89949,0.148522,0.335587,-1.82767,0.161446,1.0743,-2.12423,0.191139,0.995712,-2.10638,0.028457,1.02429,-2.09628,0.060344,0.947081,-2.07967,0.139581,0.840189,-2.07746,-0.023391,0.781939,-2.06828,-0.021434,0.89355,-2.09035,-0.090327,0.891437,-2.08302,-0.316334,0.864671,-2.05956,-0.376997,0.957216,-2.00363,-0.412037,0.677165,-2.02639,-0.449236,0.731071,-2.08774,0.01455,1.29621,-1.82886,0.037614,1.27352,-2.04165},
/*149*/{0.139592,1.82575,-1.72522,0.213663,1.77289,-1.87641,0.051034,1.67672,-1.6321,-0.012056,1.56889,-1.56835,0.141183,1.52802,-1.60814,0.225771,1.49836,-1.6104,0.16394,1.85686,-2.12548,0.223421,1.78051,-1.9522,0.021873,1.83753,-2.04091,-0.029869,1.67386,-2.20826,-0.010899,1.57432,-2.25496,0.126071,1.44226,-2.18167,0.163116,1.3874,-2.18935,0.088345,1.38853,-1.8071,-0.092918,1.43312,-1.91445,-0.081012,1.45673,-1.96813,-0.075338,1.42147,-2.02654,0.120691,1.3607,-2.07251,0.13666,1.09123,-1.74758,0.015413,1.04306,-1.79665,0.048279,0.967487,-1.81763,0.160309,0.881775,-1.79636,0.139887,0.690429,-1.80107,0.122648,0.618657,-1.81788,0.030903,0.717781,-1.81203,0.014572,0.647607,-1.82424,0.006826,0.425515,-1.85879,-0.103851,0.363749,-1.93578,0.154235,0.384657,-1.8989,0.127615,0.335396,-1.82648,0.170311,1.07949,-2.12341,0.202907,1.00171,-2.10621,0.038854,1.02497,-2.09851,0.072741,0.949128,-2.08107,0.155697,0.843982,-2.07762,-0.007175,0.781655,-2.06758,-0.006724,0.89354,-2.09039,-0.076296,0.89128,-2.08351,-0.305778,0.861903,-2.06317,-0.367319,0.94918,-2.00535,-0.387796,0.667076,-2.0259,-0.427792,0.718843,-2.08994,0.013648,1.29713,-1.82787,0.037538,1.2744,-2.04057},
/*150*/{0.14035,1.82719,-1.72569,0.212421,1.77425,-1.87705,0.054903,1.67758,-1.63143,-0.00093,1.5678,-1.56571,0.151639,1.5309,-1.60886,0.237743,1.50402,-1.61244,0.163927,1.85712,-2.12573,0.222524,1.78043,-1.95279,0.0223,1.83914,-2.04123,-0.035548,1.67978,-2.20592,-0.018085,1.58012,-2.25522,0.117743,1.44485,-2.18416,0.153357,1.38984,-2.19183,0.087005,1.38931,-1.80607,-0.093647,1.43343,-1.91445,-0.081153,1.45699,-1.96784,-0.076634,1.42234,-2.02535,0.119137,1.36272,-2.07194,0.129161,1.08677,-1.7482,0.007896,1.04506,-1.79681,0.039958,0.966422,-1.81762,0.147705,0.876795,-1.79621,0.125683,0.68647,-1.80124,0.106737,0.61499,-1.81844,0.016001,0.71558,-1.81318,-0.001874,0.645966,-1.82625,-0.013976,0.425553,-1.85916,-0.124317,0.364374,-1.93577,0.133766,0.384305,-1.89813,0.106776,0.335364,-1.82682,0.178656,1.08559,-2.12339,0.214301,1.00915,-2.10564,0.049515,1.0262,-2.09874,0.086969,0.950519,-2.08063,0.171571,0.849174,-2.0782,0.009428,0.781901,-2.06771,0.007683,0.894514,-2.09112,-0.061865,0.889943,-2.08403,-0.290978,0.854679,-2.06373,-0.357728,0.938642,-2.0081,-0.36431,0.656278,-2.0265,-0.405298,0.705887,-2.09142,0.012905,1.29766,-1.82759,0.037061,1.27565,-2.04033},
/*151*/{0.141093,1.82929,-1.72691,0.211285,1.77468,-1.87791,0.059611,1.6793,-1.63039,0.008194,1.5668,-1.56336,0.162202,1.53512,-1.60923,0.24906,1.51078,-1.61446,0.16372,1.85765,-2.12611,0.221387,1.7808,-1.95364,0.021224,1.84054,-2.04159,-0.039608,1.68683,-2.20406,-0.025606,1.58565,-2.25572,0.109595,1.44996,-2.18651,0.142803,1.39291,-2.19383,0.085026,1.38934,-1.80464,-0.094714,1.43355,-1.91349,-0.081566,1.45722,-1.96696,-0.078624,1.42381,-2.02387,0.119445,1.36657,-2.0713,0.123603,1.08454,-1.74802,0.000149,1.04469,-1.7964,0.029724,0.966512,-1.81832,0.133333,0.871219,-1.79609,0.110436,0.683095,-1.80146,0.08876,0.611036,-1.81912,0.001672,0.71336,-1.814,-0.017246,0.644913,-1.82633,-0.033243,0.425971,-1.85883,-0.145727,0.365992,-1.93596,0.113252,0.384455,-1.89884,0.086747,0.335378,-1.82639,0.187353,1.09208,-2.12276,0.224856,1.01732,-2.10499,0.059352,1.02846,-2.0994,0.098001,0.957456,-2.08342,0.186312,0.853512,-2.07845,0.025827,0.782918,-2.06747,0.022028,0.895928,-2.09099,-0.048133,0.888838,-2.08436,-0.272954,0.845522,-2.06323,-0.349323,0.928517,-2.01113,-0.340052,0.645415,-2.02677,-0.382306,0.693536,-2.09321,0.012305,1.29733,-1.82791,0.037587,1.27828,-2.0408},
/*152*/{0.142377,1.83209,-1.72697,0.210519,1.77615,-1.87851,0.064068,1.68094,-1.63023,0.018201,1.56732,-1.56159,0.172286,1.54007,-1.61019,0.259896,1.5188,-1.61704,0.162238,1.85874,-2.1263,0.221298,1.78246,-1.95422,0.020974,1.84315,-2.04116,-0.044998,1.69321,-2.20264,-0.032997,1.59268,-2.25613,0.101362,1.45388,-2.18802,0.133802,1.39695,-2.19562,0.084637,1.3908,-1.80423,-0.094841,1.4349,-1.91361,-0.083149,1.45834,-1.9663,-0.079794,1.4248,-2.02323,0.11929,1.36937,-2.07115,0.116632,1.08272,-1.74815,-0.005143,1.04612,-1.79646,0.021449,0.966419,-1.81875,0.118508,0.866153,-1.7961,0.094439,0.679659,-1.80197,0.072427,0.608934,-1.82096,-0.013881,0.71163,-1.81539,-0.033356,0.64308,-1.82726,-0.053304,0.425694,-1.85902,-0.165798,0.366096,-1.93563,0.092868,0.383928,-1.89835,0.065904,0.335621,-1.8261,0.194213,1.09965,-2.12322,0.234878,1.02575,-2.10487,0.069444,1.03104,-2.09988,0.110071,0.96219,-2.084,0.200454,0.859289,-2.07885,0.043093,0.785024,-2.0685,0.035754,0.897469,-2.09131,-0.034175,0.888693,-2.08563,-0.261463,0.839118,-2.06577,-0.340436,0.916441,-2.01113,-0.314089,0.635222,-2.02896,-0.358184,0.679887,-2.09502,0.012431,1.29866,-1.8279,0.037717,1.28036,-2.04086},
/*153*/{0.144299,1.83487,-1.72801,0.211571,1.77783,-1.87959,0.069105,1.68322,-1.6299,0.026871,1.56856,-1.56007,0.182129,1.54604,-1.61158,0.27011,1.52688,-1.62105,0.160269,1.86109,-2.12727,0.220339,1.78359,-1.95571,0.020419,1.84609,-2.04072,-0.049648,1.7002,-2.20144,-0.040517,1.59945,-2.25653,0.093006,1.45985,-2.18901,0.124738,1.40241,-2.19681,0.083239,1.39031,-1.80255,-0.094952,1.43524,-1.91318,-0.083039,1.45989,-1.96595,-0.081139,1.42612,-2.02319,0.118186,1.37214,-2.07098,0.108784,1.08213,-1.74965,-0.014173,1.0474,-1.7958,0.010025,0.966621,-1.8178,0.103009,0.861867,-1.79608,0.077651,0.678121,-1.80309,0.057163,0.606944,-1.82088,-0.029175,0.711284,-1.81428,-0.049746,0.64338,-1.82842,-0.073799,0.427169,-1.85972,-0.186439,0.366849,-1.93622,0.071885,0.383947,-1.8989,0.044463,0.335458,-1.8254,0.202003,1.10729,-2.12223,0.243874,1.03503,-2.10441,0.078357,1.03571,-2.10095,0.120967,0.967104,-2.08447,0.215241,0.865731,-2.07888,0.05956,0.787622,-2.0692,0.048335,0.900429,-2.09031,-0.020681,0.888829,-2.08647,-0.247476,0.830144,-2.06635,-0.33076,0.903398,-2.01223,-0.288477,0.623421,-2.02964,-0.334413,0.666609,-2.09689,0.011314,1.29844,-1.82852,0.037254,1.28237,-2.04159},
/*154*/{0.146131,1.83809,-1.72852,0.21182,1.78083,-1.88051,0.074492,1.68596,-1.62946,0.037518,1.57087,-1.5591,0.190902,1.55187,-1.61324,0.279501,1.53583,-1.62491,0.158748,1.86328,-2.12747,0.219854,1.78572,-1.95645,0.019567,1.84748,-2.03993,-0.054906,1.70712,-2.20066,-0.046942,1.6069,-2.25664,0.08522,1.46638,-2.19016,0.116002,1.40865,-2.19785,0.082991,1.39083,-1.80203,-0.095771,1.43816,-1.91309,-0.083108,1.46149,-1.96522,-0.081691,1.42803,-2.02273,0.117834,1.37481,-2.07079,0.103768,1.08177,-1.75069,-0.022772,1.04948,-1.79633,4.1e-005,0.967098,-1.81779,0.086662,0.859366,-1.79716,0.062415,0.676845,-1.80415,0.040147,0.606466,-1.82138,-0.044698,0.711036,-1.81534,-0.064988,0.644208,-1.82826,-0.094167,0.426568,-1.85911,-0.206273,0.367673,-1.93635,0.052106,0.383115,-1.89867,0.024496,0.33552,-1.82569,0.208409,1.11612,-2.12124,0.252753,1.04483,-2.10422,0.086794,1.03972,-2.10124,0.131635,0.9733,-2.08442,0.229253,0.873797,-2.07859,0.076236,0.790208,-2.07006,0.06065,0.902374,-2.09067,-0.00792,0.887929,-2.08588,-0.232084,0.820722,-2.06769,-0.319933,0.888957,-2.01272,-0.262985,0.611931,-2.03144,-0.310945,0.652462,-2.0984,0.010624,1.29963,-1.82876,0.036671,1.28469,-2.04189},
/*155*/{0.148654,1.84175,-1.72968,0.212552,1.78335,-1.88182,0.080158,1.68914,-1.62898,0.045215,1.57229,-1.55758,0.199265,1.55817,-1.61466,0.287799,1.54491,-1.62862,0.156587,1.86602,-2.12853,0.219845,1.78893,-1.95803,0.0178,1.8506,-2.03975,-0.060751,1.71402,-2.20046,-0.05428,1.61385,-2.25747,0.077291,1.47321,-2.19151,0.107887,1.41516,-2.19843,0.082949,1.39111,-1.80189,-0.095677,1.43989,-1.91166,-0.083597,1.4633,-1.96479,-0.081173,1.42947,-2.02202,0.116993,1.37671,-2.07074,0.09733,1.07837,-1.75041,-0.030778,1.05118,-1.79604,-0.007608,0.968556,-1.81636,0.069032,0.858198,-1.79929,0.046851,0.676183,-1.80493,0.023628,0.606036,-1.82326,-0.05983,0.712217,-1.81595,-0.080785,0.644842,-1.82819,-0.114261,0.427378,-1.85938,-0.226599,0.368524,-1.93642,0.032077,0.38358,-1.89879,0.004203,0.335609,-1.82552,0.214219,1.12477,-2.12116,0.260231,1.05468,-2.10369,0.094523,1.04533,-2.10102,0.141884,0.979674,-2.08417,0.242638,0.883274,-2.07876,0.092108,0.79354,-2.07144,0.072363,0.904939,-2.09122,0.003905,0.888904,-2.08803,-0.218015,0.810062,-2.06734,-0.309299,0.873287,-2.01289,-0.236491,0.600424,-2.03379,-0.286523,0.638529,-2.10011,0.010419,1.30038,-1.82885,0.036001,1.28643,-2.0421},
/*156*/{0.151406,1.84564,-1.73021,0.213086,1.78642,-1.88322,0.086463,1.69245,-1.6284,0.054007,1.5761,-1.5574,0.207284,1.56572,-1.61698,0.295743,1.55471,-1.63248,0.153208,1.86962,-2.12884,0.218968,1.79167,-1.95903,0.016665,1.85316,-2.0386,-0.067005,1.72104,-2.20009,-0.060825,1.62103,-2.25764,0.06921,1.48037,-2.19224,0.100215,1.42254,-2.19915,0.08506,1.3916,-1.80265,-0.095068,1.44232,-1.91091,-0.083128,1.46597,-1.9627,-0.080848,1.43192,-2.02181,0.117153,1.37996,-2.07064,0.089732,1.07806,-1.75249,-0.040249,1.05229,-1.79624,-0.020537,0.970986,-1.81826,0.053168,0.85778,-1.79995,0.030666,0.675641,-1.80672,0.004671,0.60548,-1.82366,-0.075166,0.713397,-1.81691,-0.098864,0.645995,-1.82914,-0.134549,0.431214,-1.85977,-0.246748,0.369957,-1.93619,0.011153,0.383548,-1.89898,-0.01626,0.335541,-1.82579,0.219259,1.13346,-2.12113,0.267412,1.0651,-2.10394,0.10228,1.05056,-2.10106,0.150605,0.986421,-2.08372,0.255708,0.894014,-2.07875,0.107867,0.797102,-2.07247,0.082883,0.907559,-2.0921,0.016165,0.889138,-2.08826,-0.202654,0.799845,-2.06875,-0.298444,0.856993,-2.01226,-0.208699,0.589393,-2.03672,-0.262102,0.62536,-2.10205,0.011693,1.30149,-1.82958,0.036105,1.28943,-2.04309},
/*157*/{0.153802,1.85013,-1.73133,0.212155,1.79035,-1.88428,0.092132,1.69607,-1.6284,0.063112,1.57992,-1.55706,0.215001,1.57338,-1.62027,0.302877,1.5646,-1.6372,0.150768,1.87343,-2.1291,0.21888,1.79527,-1.96021,0.01471,1.85584,-2.03718,-0.072727,1.72749,-2.19953,-0.06797,1.62855,-2.25743,0.06192,1.48783,-2.1924,0.092652,1.43039,-2.20032,0.085755,1.39243,-1.80205,-0.094342,1.44501,-1.9101,-0.082371,1.46821,-1.96161,-0.080609,1.43429,-2.02112,0.117056,1.38328,-2.07073,0.081842,1.07797,-1.75245,-0.048105,1.05479,-1.79475,-0.031034,0.974143,-1.81815,0.038105,0.859065,-1.80166,0.014409,0.675762,-1.80837,-0.009862,0.60719,-1.82578,-0.091205,0.714557,-1.81735,-0.11479,0.64812,-1.82938,-0.151588,0.428551,-1.85928,-0.266506,0.372027,-1.93545,-0.008716,0.38305,-1.89869,-0.036497,0.335954,-1.82664,0.222975,1.14168,-2.1212,0.273893,1.07522,-2.10364,0.109213,1.05616,-2.101,0.15949,0.993156,-2.08423,0.267884,0.905305,-2.0791,0.123016,0.801379,-2.0743,0.093723,0.911454,-2.09211,0.027272,0.887039,-2.088,-0.187695,0.789891,-2.06979,-0.288286,0.840733,-2.01271,-0.185208,0.57789,-2.03733,-0.237498,0.611134,-2.10379,0.012245,1.30285,-1.82988,0.036214,1.29231,-2.04351},
/*158*/{0.156567,1.85476,-1.73242,0.21284,1.79403,-1.88533,0.098469,1.70032,-1.62865,0.071864,1.58344,-1.55561,0.221122,1.58116,-1.62272,0.30968,1.57451,-1.64147,0.1477,1.87747,-2.13015,0.217629,1.79876,-1.96219,0.013504,1.85829,-2.0357,-0.077507,1.73463,-2.19931,-0.074399,1.63553,-2.2577,0.054627,1.4958,-2.19314,0.085484,1.43839,-2.20096,0.086428,1.39282,-1.80186,-0.092003,1.44803,-1.90799,-0.082778,1.47045,-1.96089,-0.080603,1.43711,-2.02005,0.117396,1.38666,-2.0706,0.075438,1.0777,-1.75534,-0.055997,1.05781,-1.79565,-0.043076,0.97732,-1.81846,0.024928,0.860376,-1.80355,-0.002345,0.676627,-1.80993,-0.03035,0.606903,-1.82657,-0.107126,0.716806,-1.81823,-0.130647,0.650803,-1.82967,-0.170952,0.4317,-1.85956,-0.286926,0.375137,-1.93529,-0.028398,0.383348,-1.89891,-0.056366,0.336059,-1.82605,0.227578,1.15046,-2.121,0.279756,1.08406,-2.10332,0.115097,1.06201,-2.10038,0.167873,1.00042,-2.08376,0.279642,0.916875,-2.07903,0.138097,0.805681,-2.0756,0.102776,0.913234,-2.09227,0.037311,0.887675,-2.08828,-0.171626,0.778185,-2.07068,-0.275458,0.821658,-2.01228,-0.1529,0.56868,-2.04256,-0.210688,0.597015,-2.10624,0.012958,1.30405,-1.83022,0.036129,1.29538,-2.04403},
/*159*/{0.158703,1.85979,-1.7339,0.213959,1.79849,-1.88703,0.105632,1.70484,-1.62897,0.079176,1.58744,-1.55501,0.227645,1.58925,-1.62602,0.315666,1.58524,-1.64634,0.14444,1.88211,-2.13026,0.217171,1.8024,-1.96403,0.011898,1.86141,-2.03391,-0.081718,1.74127,-2.19978,-0.080497,1.64285,-2.25724,0.048729,1.50459,-2.19306,0.079597,1.44723,-2.20119,0.087504,1.39369,-1.80183,-0.092258,1.45074,-1.9076,-0.081807,1.47347,-1.95947,-0.079975,1.44065,-2.01943,0.117637,1.39037,-2.06997,0.068079,1.07755,-1.75866,-0.062094,1.06278,-1.79463,-0.053129,0.980565,-1.81787,0.013377,0.862471,-1.80583,-0.019029,0.678035,-1.81132,-0.045882,0.608979,-1.82834,-0.12394,0.718946,-1.81924,-0.150253,0.652677,-1.83086,-0.19122,0.432541,-1.85874,-0.306587,0.377977,-1.93475,-0.048827,0.381987,-1.89935,-0.077678,0.336239,-1.82678,0.232094,1.15903,-2.12075,0.284776,1.09473,-2.10367,0.121674,1.0678,-2.10028,0.17507,1.00698,-2.08324,0.289562,0.927933,-2.07882,0.152658,0.810052,-2.077,0.111661,0.915369,-2.0916,0.047851,0.886028,-2.08815,-0.156767,0.766158,-2.07151,-0.262042,0.802818,-2.01206,-0.125428,0.55806,-2.04533,-0.186096,0.583767,-2.10799,0.013221,1.30554,-1.83087,0.036327,1.29898,-2.04476},
/*160*/{0.161579,1.8655,-1.73552,0.214657,1.80262,-1.88864,0.112446,1.70925,-1.62926,0.087124,1.5933,-1.55544,0.232744,1.59775,-1.62926,0.320742,1.59487,-1.65198,0.140614,1.88687,-2.13056,0.217292,1.807,-1.96565,0.010397,1.86458,-2.03211,-0.086824,1.74863,-2.19984,-0.086813,1.65037,-2.25681,0.042356,1.51324,-2.19362,0.073933,1.45552,-2.20208,0.088722,1.39454,-1.80094,-0.090165,1.45329,-1.90559,-0.080821,1.4769,-1.95717,-0.078614,1.4441,-2.01783,0.117945,1.39392,-2.06932,0.064236,1.07679,-1.76076,-0.070734,1.06605,-1.79455,-0.063899,0.98386,-1.81826,0.002493,0.865032,-1.80815,-0.036338,0.679652,-1.81319,-0.063953,0.611092,-1.82944,-0.140848,0.720873,-1.82061,-0.167558,0.654483,-1.83272,-0.209558,0.43666,-1.85901,-0.325763,0.382853,-1.93379,-0.068469,0.382234,-1.89992,-0.097832,0.3369,-1.8275,0.235428,1.16777,-2.12057,0.290226,1.10496,-2.10312,0.12737,1.07294,-2.09876,0.182294,1.01434,-2.08319,0.298286,0.939044,-2.07934,0.166589,0.816183,-2.07922,0.120374,0.918236,-2.09222,0.057817,0.885322,-2.08755,-0.14069,0.75447,-2.07277,-0.24868,0.784628,-2.01187,-0.095638,0.549532,-2.04891,-0.158252,0.570438,-2.11034,0.014262,1.30698,-1.83072,0.036644,1.3025,-2.04474},
/*161*/{0.163528,1.8713,-1.73713,0.215537,1.80731,-1.89058,0.118569,1.71441,-1.62969,0.093825,1.59833,-1.55577,0.238673,1.60612,-1.63189,0.325866,1.60512,-1.65724,0.13723,1.89186,-2.13062,0.21701,1.81141,-1.96769,0.008979,1.86762,-2.03068,-0.091092,1.75572,-2.19976,-0.091753,1.6577,-2.25609,0.036819,1.52207,-2.19318,0.068576,1.46439,-2.20289,0.091091,1.3959,-1.80072,-0.088861,1.45636,-1.90452,-0.079694,1.48021,-1.95583,-0.07809,1.44832,-2.01669,0.118387,1.39808,-2.06882,0.056744,1.0779,-1.76369,-0.07717,1.07201,-1.79415,-0.073513,0.988365,-1.81782,-0.007536,0.866987,-1.80973,-0.052569,0.682337,-1.81392,-0.082136,0.613394,-1.83038,-0.157534,0.724648,-1.82127,-0.184874,0.658405,-1.83137,-0.229726,0.440561,-1.8587,-0.344665,0.389388,-1.93332,-0.086917,0.381765,-1.90005,-0.119108,0.337212,-1.82687,0.238689,1.17584,-2.11985,0.293993,1.11548,-2.1032,0.132418,1.07925,-2.09854,0.188624,1.02069,-2.08272,0.307292,0.951012,-2.07962,0.179612,0.817755,-2.08004,0.127907,0.919647,-2.09164,0.067136,0.883875,-2.08769,-0.123735,0.741668,-2.07299,-0.234638,0.763872,-2.01088,-0.065981,0.539867,-2.05125,-0.132042,0.557678,-2.11232,0.015652,1.30902,-1.83118,0.037151,1.30655,-2.04532},
/*162*/{0.166033,1.8769,-1.73833,0.216372,1.8122,-1.89249,0.124502,1.71967,-1.63039,0.101879,1.60324,-1.55452,0.243471,1.61403,-1.6346,0.329359,1.61485,-1.66249,0.133962,1.89657,-2.13053,0.21672,1.81627,-1.96938,0.007167,1.87087,-2.02878,-0.095092,1.76211,-2.19973,-0.098776,1.66514,-2.25454,0.031446,1.53103,-2.19306,0.063611,1.47342,-2.20331,0.092208,1.39662,-1.80008,-0.0872,1.45978,-1.90225,-0.078553,1.48356,-1.95378,-0.077631,1.45237,-2.01606,0.119335,1.40269,-2.06813,0.050137,1.08052,-1.76548,-0.085288,1.07559,-1.79474,-0.082516,0.993799,-1.81868,-0.019827,0.86935,-1.81266,-0.069237,0.685622,-1.81511,-0.098113,0.617501,-1.83171,-0.174206,0.728589,-1.82204,-0.200894,0.662623,-1.83236,-0.246323,0.443743,-1.85883,-0.362857,0.395785,-1.93343,-0.106667,0.38106,-1.90054,-0.138628,0.337726,-1.82782,0.241734,1.18412,-2.11941,0.298902,1.12455,-2.10228,0.137378,1.08558,-2.09878,0.194929,1.0281,-2.08265,0.314321,0.962278,-2.07901,0.191767,0.822906,-2.08147,0.134592,0.921164,-2.09154,0.075953,0.881519,-2.0869,-0.10459,0.72905,-2.07223,-0.217524,0.743915,-2.01038,-0.03673,0.530304,-2.05389,-0.104272,0.544606,-2.11463,0.016497,1.31062,-1.83165,0.037657,1.31093,-2.04584},
/*163*/{0.167831,1.88277,-1.74009,0.216721,1.81692,-1.89397,0.130099,1.72544,-1.6314,0.107499,1.60918,-1.5547,0.246568,1.6219,-1.63707,0.332207,1.62515,-1.66537,0.131242,1.90177,-2.13048,0.216979,1.82118,-1.97153,0.006465,1.87383,-2.02695,-0.099139,1.76865,-2.20038,-0.103531,1.67241,-2.25347,0.0256,1.53942,-2.19244,0.058468,1.48193,-2.20361,0.09469,1.39815,-1.80005,-0.086014,1.46322,-1.90102,-0.07703,1.48699,-1.95134,-0.075753,1.45739,-2.0148,0.119642,1.40707,-2.06726,0.044696,1.08203,-1.76824,-0.09115,1.08124,-1.79522,-0.092875,0.999129,-1.8198,-0.03348,0.873487,-1.8136,-0.085598,0.690047,-1.81599,-0.115926,0.621603,-1.83137,-0.18994,0.733409,-1.82288,-0.216211,0.668375,-1.83236,-0.264811,0.448797,-1.85824,-0.381172,0.402913,-1.93287,-0.125838,0.379311,-1.90129,-0.159588,0.337625,-1.8279,0.244558,1.19224,-2.1195,0.302647,1.13385,-2.10137,0.141902,1.09297,-2.0987,0.199835,1.03662,-2.08254,0.320115,0.973132,-2.07906,0.204604,0.827446,-2.08243,0.14152,0.922083,-2.0908,0.085466,0.879486,-2.08723,-0.086257,0.715888,-2.07239,-0.202061,0.723995,-2.01125,-0.007693,0.522538,-2.057,-0.07509,0.533129,-2.11681,0.017945,1.31278,-1.83203,0.038253,1.31543,-2.04629},
/*164*/{0.170062,1.88982,-1.74224,0.217355,1.82233,-1.89641,0.135821,1.73127,-1.63214,0.1155,1.61525,-1.55452,0.250572,1.63042,-1.63981,0.335175,1.63503,-1.67012,0.12746,1.90645,-2.13125,0.216412,1.8254,-1.97316,0.005455,1.87747,-2.02521,-0.102539,1.77559,-2.20023,-0.109239,1.67962,-2.2527,0.020206,1.54674,-2.19164,0.05435,1.48995,-2.2039,0.09552,1.39863,-1.79961,-0.08424,1.46636,-1.90005,-0.075703,1.49093,-1.94926,-0.074815,1.46245,-2.01461,0.120776,1.41189,-2.0668,0.037973,1.08259,-1.77045,-0.095368,1.08851,-1.79423,-0.09964,1.00574,-1.81843,-0.049888,0.877575,-1.8143,-0.101681,0.695332,-1.81624,-0.132962,0.626005,-1.83205,-0.205114,0.739578,-1.82359,-0.232789,0.674308,-1.83372,-0.282244,0.453574,-1.8571,-0.398615,0.411343,-1.9332,-0.144603,0.378499,-1.90106,-0.179522,0.338089,-1.82831,0.247277,1.20036,-2.11919,0.306442,1.14171,-2.10047,0.145854,1.10067,-2.09931,0.203787,1.04385,-2.08211,0.324103,0.982561,-2.0796,0.21522,0.830736,-2.08394,0.14782,0.922999,-2.09056,0.093579,0.876969,-2.08788,-0.068661,0.703693,-2.07248,-0.183801,0.704084,-2.01208,0.023592,0.515546,-2.05996,-0.045635,0.522359,-2.11954,0.018316,1.31426,-1.83336,0.03887,1.32036,-2.04752},
/*165*/{0.171457,1.89442,-1.74441,0.218503,1.8265,-1.89815,0.140613,1.73722,-1.63301,0.120441,1.62163,-1.55424,0.253678,1.63887,-1.64286,0.337989,1.6439,-1.67402,0.12528,1.91128,-2.13114,0.216698,1.83087,-1.97489,0.003962,1.88137,-2.02414,-0.106201,1.78099,-2.19975,-0.114097,1.68724,-2.25116,0.016862,1.55427,-2.19037,0.050054,1.49792,-2.20443,0.098949,1.40059,-1.80063,-0.081374,1.46998,-1.89882,-0.074664,1.49522,-1.94758,-0.073238,1.46787,-2.0136,0.121121,1.41658,-2.06659,0.03359,1.08544,-1.77299,-0.101618,1.09563,-1.79498,-0.108594,1.01171,-1.81811,-0.065562,0.883032,-1.8145,-0.116899,0.70055,-1.8163,-0.147954,0.632764,-1.83192,-0.220787,0.746049,-1.8244,-0.249358,0.679854,-1.8344,-0.30005,0.458887,-1.85861,-0.415387,0.420438,-1.93357,-0.163396,0.377205,-1.90145,-0.199179,0.339136,-1.82847,0.24972,1.20843,-2.11905,0.309117,1.15043,-2.0999,0.14923,1.10749,-2.09987,0.207587,1.05158,-2.08143,0.327869,0.992547,-2.07916,0.227034,0.834978,-2.08547,0.153458,0.92132,-2.08919,0.104087,0.872698,-2.08598,-0.052516,0.691263,-2.07458,-0.165914,0.684115,-2.01294,0.05388,0.510059,-2.06276,-0.015548,0.51313,-2.12215,0.020582,1.31697,-1.83433,0.039401,1.32528,-2.04858},
/*166*/{0.173414,1.90016,-1.74578,0.218423,1.83196,-1.90005,0.14444,1.74356,-1.63363,0.124315,1.62756,-1.55449,0.255593,1.64682,-1.64576,0.339916,1.65359,-1.67967,0.122796,1.91577,-2.13084,0.216056,1.83551,-1.97615,0.002719,1.88477,-2.02276,-0.107492,1.78675,-2.19908,-0.118821,1.69334,-2.24924,0.011181,1.56232,-2.18973,0.045992,1.50629,-2.20424,0.101937,1.40189,-1.80159,-0.079429,1.47401,-1.89746,-0.072389,1.49868,-1.94527,-0.071859,1.47253,-2.01235,0.122051,1.42169,-2.0658,0.028864,1.08757,-1.77523,-0.107042,1.1027,-1.79413,-0.116261,1.01871,-1.81859,-0.080617,0.889873,-1.81378,-0.132583,0.706701,-1.81546,-0.163289,0.639117,-1.83019,-0.235027,0.752692,-1.82503,-0.263858,0.687931,-1.83404,-0.31796,0.465523,-1.85849,-0.431882,0.430157,-1.93326,-0.182555,0.3772,-1.90175,-0.220266,0.339764,-1.82913,0.252067,1.2162,-2.11867,0.312219,1.15912,-2.09929,0.152949,1.11319,-2.1004,0.210841,1.05945,-2.0806,0.330277,1.00083,-2.0792,0.237903,0.83872,-2.08656,0.159125,0.920495,-2.08882,0.111935,0.868834,-2.08601,-0.033169,0.678495,-2.07462,-0.14554,0.663209,-2.01168,0.084056,0.504421,-2.06512,0.014963,0.503955,-2.12478,0.022724,1.31917,-1.83528,0.040358,1.33026,-2.0495},
/*167*/{0.175182,1.90618,-1.74761,0.220075,1.83671,-1.9023,0.148666,1.74961,-1.63438,0.128578,1.63353,-1.55395,0.257814,1.65403,-1.64808,0.341017,1.66212,-1.68357,0.120821,1.92018,-2.13134,0.215149,1.84043,-1.97789,0.002055,1.88929,-2.02199,-0.110974,1.79172,-2.19849,-0.122959,1.70048,-2.24812,0.008281,1.57008,-2.18895,0.042722,1.51366,-2.20455,0.10358,1.40307,-1.80216,-0.077246,1.47741,-1.89627,-0.070701,1.50336,-1.94304,-0.070054,1.47767,-2.01192,0.123436,1.42719,-2.06555,0.024239,1.09059,-1.77675,-0.111659,1.10739,-1.79381,-0.123856,1.02633,-1.8181,-0.093381,0.896963,-1.81372,-0.147378,0.713318,-1.81414,-0.177656,0.645837,-1.82927,-0.248888,0.760321,-1.8252,-0.277648,0.695232,-1.83467,-0.329543,0.470463,-1.85652,-0.447715,0.440304,-1.93365,-0.201408,0.377108,-1.90177,-0.241529,0.340561,-1.82974,0.253724,1.22376,-2.11838,0.313949,1.16706,-2.09884,0.155424,1.11965,-2.10012,0.213888,1.06551,-2.07998,0.331113,1.00866,-2.07942,0.247791,0.841437,-2.08784,0.165814,0.918926,-2.08871,0.120255,0.864045,-2.0858,-0.011452,0.665651,-2.07373,-0.124942,0.641889,-2.01251,0.114815,0.500216,-2.0679,0.045877,0.495794,-2.12768,0.024045,1.32123,-1.83689,0.041418,1.33586,-2.05092},
/*168*/{0.177384,1.91146,-1.7491,0.220477,1.84118,-1.90433,0.152322,1.75512,-1.63497,0.133547,1.63969,-1.55383,0.259752,1.66161,-1.65093,0.341558,1.67005,-1.68855,0.118338,1.92444,-2.13114,0.214992,1.84527,-1.97938,0.001305,1.89256,-2.0215,-0.114609,1.7971,-2.19677,-0.127738,1.70682,-2.24681,0.004875,1.57724,-2.18852,0.039068,1.52113,-2.20482,0.106495,1.40423,-1.80356,-0.074895,1.48118,-1.8951,-0.069661,1.50782,-1.9413,-0.068599,1.4826,-2.01057,0.124447,1.43223,-2.06559,0.019626,1.09314,-1.77758,-0.1153,1.11614,-1.79326,-0.13028,1.03459,-1.81721,-0.10501,0.90408,-1.81365,-0.16105,0.720549,-1.81357,-0.192155,0.65337,-1.82778,-0.262562,0.768332,-1.82569,-0.291874,0.703095,-1.83641,-0.347163,0.477525,-1.85602,-0.462642,0.451823,-1.93425,-0.220697,0.376494,-1.90175,-0.261374,0.341037,-1.8299,0.255072,1.22998,-2.11806,0.315921,1.17397,-2.09744,0.156964,1.12593,-2.10083,0.216037,1.07386,-2.08297,0.331783,1.01632,-2.0796,0.257444,0.843175,-2.08856,0.17106,0.91602,-2.08848,0.129847,0.859454,-2.08616,0.005849,0.654932,-2.07755,-0.103549,0.622726,-2.01204,0.145192,0.496221,-2.06967,0.07713,0.488994,-2.12857,0.02593,1.32348,-1.83845,0.041944,1.34112,-2.05236},
/*169*/{0.178999,1.91665,-1.75072,0.220847,1.84639,-1.90645,0.155416,1.76118,-1.63544,0.137393,1.64612,-1.55423,0.260999,1.6684,-1.653,0.343358,1.67743,-1.69148,0.116966,1.92837,-2.13172,0.214939,1.85017,-1.98101,0.001023,1.89583,-2.02124,-0.116808,1.80299,-2.19524,-0.131423,1.71292,-2.24561,0.001359,1.58353,-2.18738,0.035927,1.52752,-2.20483,0.109392,1.40539,-1.80484,-0.072407,1.48528,-1.89287,-0.067314,1.5113,-1.93914,-0.066715,1.48864,-2.01039,0.125284,1.43731,-2.06574,0.015501,1.09616,-1.77927,-0.117754,1.12469,-1.79259,-0.136489,1.04191,-1.81737,-0.114867,0.910486,-1.81363,-0.173744,0.728104,-1.81194,-0.20618,0.660976,-1.82591,-0.274823,0.775939,-1.82697,-0.305221,0.711836,-1.83585,-0.361643,0.484434,-1.85475,-0.476378,0.464319,-1.93225,-0.239638,0.376557,-1.90202,-0.282295,0.341825,-1.83038,0.256849,1.23692,-2.11787,0.317727,1.18012,-2.09654,0.158726,1.13283,-2.10104,0.216971,1.07987,-2.08381,0.331483,1.02228,-2.0804,0.267055,0.844523,-2.08914,0.176613,0.912578,-2.08796,0.137821,0.854242,-2.08649,0.025809,0.642438,-2.077,-0.080879,0.602501,-2.01264,0.175012,0.493875,-2.0711,0.107472,0.482231,-2.1305,0.027864,1.3257,-1.84015,0.042507,1.34657,-2.05387},
/*170*/{0.181091,1.92194,-1.75242,0.222288,1.85179,-1.90835,0.158584,1.76658,-1.63598,0.140559,1.6516,-1.55368,0.263047,1.67559,-1.6557,0.343275,1.68523,-1.69602,0.115039,1.93191,-2.1316,0.215081,1.85452,-1.98288,-0.000138,1.89938,-2.02094,-0.11992,1.80804,-2.19298,-0.135236,1.71784,-2.24332,-0.001801,1.58978,-2.18695,0.033777,1.53381,-2.20487,0.112469,1.40712,-1.80622,-0.0702,1.48934,-1.89177,-0.065798,1.51543,-1.93718,-0.064609,1.49381,-2.00871,0.123789,1.44282,-2.06611,0.01171,1.09896,-1.77975,-0.122419,1.13115,-1.79283,-0.142067,1.04995,-1.81739,-0.123559,0.917734,-1.8131,-0.185597,0.735526,-1.81111,-0.219172,0.667553,-1.82456,-0.286277,0.784554,-1.82717,-0.316046,0.720325,-1.83582,-0.374375,0.490675,-1.85302,-0.488897,0.47737,-1.93257,-0.258832,0.37636,-1.90204,-0.302799,0.343351,-1.8309,0.258843,1.24293,-2.11652,0.3178,1.18606,-2.09537,0.15888,1.13874,-2.10153,0.218016,1.0858,-2.0838,0.331395,1.02755,-2.08027,0.276058,0.845246,-2.08976,0.182264,0.908217,-2.08856,0.146701,0.84714,-2.08518,0.046791,0.62865,-2.07581,-0.056626,0.583854,-2.01126,0.20439,0.491743,-2.07264,0.137302,0.476637,-2.13133,0.029831,1.3284,-1.84162,0.042137,1.35197,-2.05521},
/*171*/{0.181604,1.92598,-1.75433,0.222217,1.85566,-1.90952,0.160994,1.77198,-1.63611,0.143785,1.65649,-1.55269,0.263218,1.68173,-1.65737,0.343997,1.69166,-1.69918,0.113661,1.93589,-2.13159,0.214714,1.85874,-1.98419,0.001406,1.90112,-2.01855,-0.12145,1.81404,-2.19073,-0.139267,1.72373,-2.24198,-0.004208,1.59512,-2.18647,0.031186,1.54006,-2.20424,0.114253,1.40873,-1.80676,-0.067618,1.49343,-1.89064,-0.063185,1.51878,-1.9358,-0.062538,1.49903,-2.00764,0.125201,1.44771,-2.06642,0.00939,1.10244,-1.78017,-0.123965,1.13778,-1.79188,-0.145745,1.05754,-1.81634,-0.132391,0.924849,-1.81265,-0.196755,0.743779,-1.80995,-0.228787,0.67731,-1.82235,-0.296971,0.792283,-1.82787,-0.32649,0.728105,-1.83523,-0.386083,0.497601,-1.85181,-0.500328,0.492012,-1.93273,-0.277287,0.376651,-1.90168,-0.32349,0.344999,-1.83154,0.25966,1.24774,-2.11619,0.318572,1.19074,-2.09412,0.159307,1.14451,-2.10228,0.218027,1.09083,-2.08472,0.331415,1.03126,-2.08052,0.285111,0.845806,-2.09019,0.188203,0.904159,-2.08943,0.15624,0.840202,-2.08785,0.068021,0.617443,-2.0756,-0.033086,0.564533,-2.01171,0.233347,0.491174,-2.07287,0.168054,0.471381,-2.13252,0.031436,1.33086,-1.84293,0.043202,1.35704,-2.05624},
/*172*/{0.183938,1.93062,-1.75563,0.223645,1.85939,-1.91142,0.162843,1.77709,-1.63671,0.146501,1.66165,-1.55343,0.2643,1.68731,-1.65949,0.344324,1.69798,-1.70267,0.112608,1.93888,-2.132,0.215017,1.86298,-1.98524,0.001118,1.90435,-2.01877,-0.123785,1.81913,-2.18885,-0.142123,1.72878,-2.23987,-0.006584,1.60088,-2.18623,0.029004,1.54554,-2.20412,0.11711,1.41067,-1.80776,-0.06588,1.4971,-1.8897,-0.061761,1.52317,-1.93499,-0.060853,1.50433,-2.00674,0.125724,1.45243,-2.06701,0.005498,1.10557,-1.77925,-0.124532,1.14288,-1.78952,-0.151393,1.0648,-1.81639,-0.139805,0.932016,-1.81167,-0.205953,0.751465,-1.80838,-0.238245,0.685214,-1.82023,-0.306336,0.80079,-1.82783,-0.336154,0.737045,-1.83534,-0.399235,0.505851,-1.85026,-0.510592,0.505869,-1.93211,-0.294909,0.377671,-1.90083,-0.344002,0.347598,-1.8319,0.259829,1.25266,-2.11508,0.318696,1.19478,-2.09278,0.158905,1.14971,-2.10304,0.217559,1.09498,-2.08619,0.329768,1.03373,-2.08145,0.293478,0.845758,-2.09058,0.193278,0.898424,-2.0898,0.164702,0.833538,-2.0879,0.087468,0.605775,-2.07536,-0.009824,0.547835,-2.01314,0.262553,0.490118,-2.07358,0.198152,0.467309,-2.1328,0.033073,1.33373,-1.84438,0.043594,1.36212,-2.05747},
/*173*/{0.185171,1.93452,-1.75715,0.223996,1.86362,-1.91298,0.164771,1.78168,-1.63727,0.149952,1.66676,-1.55277,0.265129,1.6924,-1.6614,0.343778,1.70411,-1.70662,0.111579,1.94243,-2.13237,0.21503,1.86699,-1.98666,0.002184,1.90613,-2.01784,-0.123986,1.82448,-2.18699,-0.144827,1.73358,-2.23838,-0.008261,1.60551,-2.18553,0.027646,1.55065,-2.20386,0.119917,1.41279,-1.80883,-0.062437,1.50064,-1.88862,-0.060035,1.52713,-1.93278,-0.060189,1.51001,-2.00584,0.126342,1.45728,-2.06701,0.002523,1.109,-1.77853,-0.128863,1.1493,-1.79013,-0.154968,1.07144,-1.8161,-0.146642,0.938481,-1.81237,-0.2144,0.759287,-1.80659,-0.248628,0.691961,-1.81828,-0.313529,0.808518,-1.82724,-0.34492,0.744003,-1.83544,-0.409795,0.514051,-1.84972,-0.519766,0.521574,-1.93238,-0.312722,0.378818,-1.90052,-0.364219,0.350741,-1.83188,0.259554,1.25626,-2.11461,0.317938,1.198,-2.09226,0.158577,1.15391,-2.10362,0.216582,1.09853,-2.08784,0.329147,1.03601,-2.082,0.301547,0.845688,-2.09168,0.199157,0.892551,-2.09104,0.174052,0.826288,-2.08686,0.111479,0.595467,-2.07447,0.014806,0.531691,-2.01338,0.289885,0.490651,-2.07241,0.226998,0.463453,-2.13262,0.035174,1.33657,-1.84559,0.043992,1.36735,-2.05843},
/*174*/{0.18639,1.93852,-1.75829,0.224417,1.86776,-1.91422,0.16673,1.7863,-1.63762,0.151729,1.67125,-1.5525,0.265748,1.69798,-1.66333,0.343592,1.70932,-1.70954,0.110856,1.94496,-2.13208,0.215256,1.8704,-1.98727,0.002409,1.90904,-2.01798,-0.125383,1.82742,-2.18202,-0.147717,1.7382,-2.2363,-0.010284,1.61064,-2.18525,0.025712,1.55575,-2.20384,0.121176,1.41404,-1.80961,-0.060691,1.50477,-1.88667,-0.059122,1.53085,-1.93178,-0.056532,1.51362,-2.00357,0.125698,1.46123,-2.06685,0.001199,1.11375,-1.77824,-0.128325,1.15659,-1.79086,-0.157883,1.07743,-1.81641,-0.152173,0.945331,-1.81234,-0.221202,0.766475,-1.80612,-0.255001,0.699463,-1.81699,-0.320279,0.81547,-1.82596,-0.351594,0.752086,-1.83363,-0.420002,0.521427,-1.84724,-0.527848,0.538504,-1.93179,-0.329982,0.380243,-1.89923,-0.385438,0.354435,-1.83261,0.259593,1.25926,-2.1135,0.317348,1.20016,-2.09169,0.157104,1.15761,-2.10481,0.214884,1.10106,-2.08858,0.327345,1.0371,-2.08213,0.309207,0.844506,-2.09183,0.204207,0.886057,-2.09184,0.182338,0.817419,-2.08736,0.129678,0.58473,-2.07515,0.039803,0.515193,-2.01408,0.316165,0.490924,-2.07143,0.255769,0.460472,-2.13166,0.035898,1.3389,-1.84607,0.04369,1.37146,-2.05869},
/*175*/{0.188441,1.9424,-1.75981,0.225084,1.87212,-1.91512,0.1681,1.79075,-1.63809,0.15429,1.67532,-1.55206,0.265895,1.70221,-1.66479,0.342809,1.71334,-1.71269,0.110334,1.94827,-2.1321,0.215549,1.8738,-1.98836,0.0024,1.91103,-2.01758,-0.126482,1.83158,-2.17988,-0.149744,1.74263,-2.23451,-0.011566,1.61472,-2.18525,0.024272,1.56015,-2.20344,0.123861,1.41693,-1.81008,-0.059459,1.50851,-1.88588,-0.057677,1.53501,-1.93026,-0.055023,1.51809,-2.00169,0.1269,1.46494,-2.06672,-0.003071,1.11622,-1.77593,-0.130528,1.16299,-1.79422,-0.160456,1.0848,-1.81595,-0.156845,0.951416,-1.81305,-0.227239,0.774118,-1.80438,-0.261427,0.706461,-1.81568,-0.325806,0.824166,-1.82321,-0.357894,0.759904,-1.83186,-0.429414,0.528673,-1.84463,-0.533794,0.554507,-1.93096,-0.347638,0.382358,-1.8983,-0.405279,0.359378,-1.83318,0.259738,1.26117,-2.11258,0.316487,1.20122,-2.0914,0.155722,1.16057,-2.10513,0.21238,1.10362,-2.08926,0.325436,1.0375,-2.08346,0.31626,0.843773,-2.09216,0.209826,0.8796,-2.09289,0.190622,0.812305,-2.08786,0.150847,0.575073,-2.0753,0.064516,0.499853,-2.01463,0.341579,0.492477,-2.07038,0.283577,0.457846,-2.13069,0.037626,1.34221,-1.84608,0.04432,1.37571,-2.05859},
/*176*/{0.189131,1.94538,-1.76113,0.22532,1.87489,-1.91638,0.169456,1.79463,-1.63853,0.155254,1.679,-1.55158,0.265892,1.70712,-1.66687,0.343013,1.71814,-1.71528,0.109606,1.95096,-2.13217,0.216491,1.87708,-1.989,0.003087,1.91375,-2.0167,-0.127479,1.83555,-2.17809,-0.151637,1.74631,-2.23265,-0.012721,1.61873,-2.1851,0.023542,1.56478,-2.20376,0.125271,1.41938,-1.81065,-0.056785,1.51169,-1.88484,-0.056246,1.53861,-1.92823,-0.053673,1.5219,-2.00061,0.126809,1.46799,-2.06655,-0.005115,1.12012,-1.77506,-0.13187,1.16704,-1.79552,-0.162443,1.08909,-1.81636,-0.158617,0.956695,-1.81332,-0.233136,0.779471,-1.80414,-0.268767,0.712617,-1.81453,-0.329511,0.83137,-1.82218,-0.363599,0.766791,-1.83097,-0.438016,0.53862,-1.84229,-0.539739,0.571077,-1.92871,-0.366093,0.384981,-1.89873,-0.42615,0.365683,-1.834,0.257838,1.26253,-2.1123,0.314828,1.20201,-2.0913,0.153315,1.16272,-2.10501,0.209246,1.10467,-2.08972,0.322945,1.03672,-2.0843,0.323707,0.8416,-2.09207,0.21527,0.87322,-2.09405,0.19932,0.803916,-2.08919,0.171684,0.566327,-2.07791,0.087044,0.485541,-2.01375,0.367645,0.494812,-2.06869,0.310235,0.456235,-2.1287,0.038838,1.34498,-1.84601,0.044136,1.37919,-2.05845},
/*177*/{0.190537,1.94856,-1.76215,0.227149,1.87905,-1.91796,0.171429,1.79884,-1.63945,0.156457,1.68197,-1.55144,0.266263,1.70977,-1.66813,0.342368,1.72172,-1.71827,0.109559,1.95334,-2.13254,0.21655,1.88001,-1.99055,0.00335,1.91474,-2.01658,-0.128778,1.83946,-2.17475,-0.153083,1.74973,-2.23095,-0.014014,1.62277,-2.18472,0.022232,1.56853,-2.20383,0.127321,1.42232,-1.8114,-0.054428,1.5153,-1.88302,-0.055156,1.54181,-1.92738,-0.052395,1.52581,-2.00038,0.127032,1.47078,-2.06612,-0.005908,1.1237,-1.77449,-0.132068,1.17107,-1.79716,-0.163337,1.09456,-1.81804,-0.159497,0.960795,-1.81423,-0.237206,0.785624,-1.80374,-0.272758,0.720202,-1.81293,-0.333668,0.837588,-1.82251,-0.367714,0.774573,-1.82831,-0.448185,0.549119,-1.84078,-0.543755,0.588607,-1.92517,-0.381817,0.390118,-1.89845,-0.4465,0.37235,-1.83556,0.257175,1.26265,-2.11143,0.312258,1.20074,-2.09099,0.150949,1.16432,-2.10527,0.206107,1.10549,-2.0905,0.320335,1.03583,-2.08606,0.329732,0.840609,-2.09217,0.218801,0.866114,-2.09524,0.207277,0.797127,-2.09178,0.19167,0.557963,-2.0758,0.110512,0.473509,-2.01145,0.389516,0.496826,-2.06699,0.336057,0.454757,-2.12691,0.040346,1.34819,-1.84564,0.044258,1.38237,-2.05811},
/*178*/{0.191252,1.9512,-1.76281,0.227771,1.88206,-1.91859,0.17202,1.80204,-1.6395,0.158173,1.68525,-1.55119,0.265742,1.7127,-1.6699,0.341611,1.72443,-1.72068,0.108703,1.95572,-2.13263,0.217218,1.88156,-1.99095,0.003275,1.91741,-2.01578,-0.128059,1.8426,-2.17249,-0.154751,1.75301,-2.2289,-0.013472,1.626,-2.1842,0.022103,1.57226,-2.20315,0.128609,1.42532,-1.81152,-0.053822,1.51892,-1.88231,-0.053987,1.5454,-1.92599,-0.051658,1.52929,-1.99841,0.127208,1.47352,-2.06563,-0.00635,1.12741,-1.77351,-0.132325,1.17446,-1.79797,-0.163272,1.09819,-1.81965,-0.15975,0.964394,-1.81456,-0.239547,0.790339,-1.80368,-0.27844,0.724945,-1.81264,-0.335264,0.844609,-1.82039,-0.370423,0.782231,-1.82658,-0.457243,0.557559,-1.83929,-0.548177,0.605917,-1.92064,-0.399137,0.396332,-1.89935,-0.463594,0.380475,-1.83716,0.255563,1.26217,-2.11163,0.309855,1.19898,-2.09147,0.147847,1.16508,-2.10565,0.202567,1.10565,-2.09156,0.317918,1.03408,-2.08733,0.335488,0.838357,-2.09234,0.222696,0.860866,-2.09748,0.214753,0.789032,-2.09294,0.210334,0.550261,-2.07523,0.135382,0.46086,-2.01286,0.411763,0.498975,-2.06415,0.361737,0.453912,-2.12436,0.04108,1.35139,-1.84487,0.044018,1.38558,-2.05736},
/*179*/{0.192432,1.95394,-1.76457,0.228446,1.88488,-1.92004,0.172703,1.80522,-1.64009,0.158809,1.68845,-1.55086,0.265469,1.71543,-1.67045,0.341124,1.72681,-1.72322,0.108957,1.95777,-2.133,0.217859,1.88407,-1.99241,0.004156,1.91934,-2.01585,-0.130087,1.8446,-2.16917,-0.156352,1.75584,-2.22719,-0.014031,1.62932,-2.18312,0.021386,1.57502,-2.20321,0.129404,1.42801,-1.81148,-0.052843,1.52159,-1.8815,-0.053338,1.54838,-1.92485,-0.0506,1.53178,-1.99716,0.127745,1.47596,-2.0655,-0.005076,1.13033,-1.7727,-0.132177,1.17666,-1.79764,-0.162609,1.10133,-1.82024,-0.158913,0.967198,-1.81517,-0.241639,0.793715,-1.80365,-0.280217,0.729035,-1.81172,-0.335134,0.850731,-1.81919,-0.372336,0.789361,-1.82404,-0.464208,0.568343,-1.83647,-0.553809,0.62199,-1.91582,-0.416029,0.404902,-1.90125,-0.480692,0.389317,-1.83993,0.252812,1.26026,-2.11177,0.306564,1.19662,-2.09147,0.144348,1.16506,-2.10612,0.198206,1.10417,-2.09177,0.315148,1.03174,-2.08875,0.339544,0.836463,-2.09321,0.227749,0.851948,-2.0988,0.220925,0.782879,-2.0948,0.229342,0.542443,-2.07513,0.155624,0.451688,-2.01261,0.432363,0.501642,-2.06139,0.384974,0.453107,-2.12102,0.041985,1.35402,-1.84428,0.044288,1.3882,-2.05677},
/*180*/{0.193294,1.95565,-1.7648,0.229106,1.8871,-1.92057,0.173136,1.80787,-1.64078,0.158877,1.69102,-1.55156,0.265249,1.71744,-1.67188,0.340115,1.72877,-1.72544,0.108202,1.95973,-2.1327,0.219102,1.88606,-1.99315,0.0042,1.92197,-2.01548,-0.130551,1.84718,-2.16689,-0.15685,1.75819,-2.22539,-0.013989,1.63198,-2.18272,0.02143,1.57813,-2.20329,0.130413,1.43074,-1.81162,-0.051621,1.52466,-1.88002,-0.05273,1.55122,-1.92451,-0.051287,1.53524,-1.99608,0.127693,1.47846,-2.06493,-0.005503,1.13267,-1.77246,-0.131025,1.17793,-1.79994,-0.160466,1.10277,-1.82216,-0.155733,0.968675,-1.81593,-0.242098,0.796539,-1.80405,-0.284177,0.734255,-1.81251,-0.335532,0.856895,-1.819,-0.372617,0.796228,-1.82321,-0.46774,0.577065,-1.83514,-0.557757,0.637631,-1.90817,-0.431967,0.413577,-1.90455,-0.49614,0.399128,-1.84265,0.250622,1.25759,-2.11159,0.30292,1.19423,-2.09247,0.140886,1.16473,-2.10702,0.193575,1.10379,-2.09257,0.311831,1.02867,-2.08988,0.343125,0.834629,-2.09399,0.230367,0.845582,-2.10082,0.227828,0.775508,-2.09529,0.247295,0.535873,-2.07512,0.177125,0.441325,-2.01342,0.451025,0.503951,-2.05874,0.406403,0.452005,-2.11826,0.042604,1.35695,-1.84363,0.043825,1.391,-2.05616},
/*181*/{0.193918,1.95787,-1.76596,0.230011,1.88926,-1.92145,0.173594,1.81038,-1.6416,0.15907,1.6925,-1.55072,0.264055,1.71905,-1.67306,0.337532,1.73189,-1.72716,0.10909,1.96133,-2.13368,0.218954,1.8873,-1.99448,0.005624,1.92356,-2.01557,-0.130576,1.84827,-2.16536,-0.158113,1.76066,-2.22387,-0.014546,1.63471,-2.18259,0.021381,1.5806,-2.20367,0.130893,1.43318,-1.81137,-0.0513,1.52737,-1.87988,-0.052652,1.5544,-1.92407,-0.050135,1.53698,-1.99578,0.129176,1.48027,-2.06474,-0.00238,1.13404,-1.77282,-0.129585,1.17832,-1.79722,-0.158427,1.10327,-1.82213,-0.153086,0.969304,-1.81652,-0.242162,0.799237,-1.80364,-0.285564,0.737853,-1.812,-0.333248,0.862322,-1.81682,-0.373628,0.802836,-1.82098,-0.47384,0.585486,-1.832,-0.56523,0.653278,-1.9018,-0.448907,0.424095,-1.9091,-0.509423,0.410592,-1.84494,0.247181,1.25515,-2.11242,0.299621,1.19016,-2.09296,0.136671,1.16287,-2.10752,0.187914,1.10141,-2.09296,0.308989,1.0248,-2.09141,0.346796,0.832948,-2.09461,0.233614,0.839159,-2.10226,0.234559,0.769182,-2.09671,0.263713,0.529559,-2.0748,0.196437,0.4339,-2.013,0.468183,0.506096,-2.05594,0.427556,0.452231,-2.11529,0.042998,1.3595,-1.84302,0.044588,1.39309,-2.05561},
/*182*/{0.195623,1.96064,-1.76667,0.229638,1.89058,-1.92252,0.173412,1.8124,-1.64184,0.158609,1.6951,-1.55114,0.26338,1.71986,-1.67388,0.336774,1.73037,-1.72889,0.109315,1.96263,-2.13397,0.220074,1.8887,-1.99506,0.005773,1.92521,-2.01579,-0.130717,1.84986,-2.16206,-0.15804,1.76258,-2.22268,-0.014794,1.63621,-2.18161,0.021498,1.5826,-2.20391,0.132451,1.4355,-1.81097,-0.050779,1.52925,-1.87953,-0.051678,1.5564,-1.92327,-0.049715,1.53889,-1.99503,0.129301,1.48157,-2.06395,0.002109,1.13516,-1.77519,-0.125241,1.17867,-1.79644,-0.154786,1.10339,-1.82316,-0.147791,0.969329,-1.81694,-0.241389,0.800604,-1.80481,-0.28725,0.740949,-1.81282,-0.330706,0.866742,-1.81556,-0.372628,0.80951,-1.8194,-0.480734,0.595368,-1.83077,-0.57207,0.668421,-1.89296,-0.462711,0.436299,-1.91392,-0.522852,0.421469,-1.84715,0.244783,1.25191,-2.11285,0.296798,1.1856,-2.09321,0.132664,1.16088,-2.10779,0.183288,1.09916,-2.09335,0.304397,1.02032,-2.09203,0.349828,0.831303,-2.09421,0.235922,0.83243,-2.10415,0.23747,0.761623,-2.0978,0.279105,0.524484,-2.07407,0.215195,0.426977,-2.01168,0.484451,0.508584,-2.0534,0.447002,0.451733,-2.11147,0.044029,1.36171,-1.84196,0.044923,1.39455,-2.05467},
/*183*/{0.196455,1.96099,-1.76729,0.230792,1.89287,-1.92356,0.17316,1.81431,-1.64238,0.158578,1.69595,-1.55138,0.262498,1.71976,-1.67458,0.335705,1.72905,-1.73065,0.109487,1.96369,-2.13431,0.220326,1.88917,-1.99602,0.006857,1.9266,-2.01588,-0.130105,1.85143,-2.16039,-0.158192,1.76377,-2.22047,-0.013725,1.63797,-2.18141,0.021344,1.58356,-2.20415,0.13338,1.4378,-1.81081,-0.050099,1.53103,-1.87819,-0.051538,1.55919,-1.92358,-0.049556,1.54047,-1.99401,0.130185,1.48251,-2.06346,0.004754,1.13532,-1.77593,-0.121321,1.17864,-1.79985,-0.149828,1.10156,-1.82191,-0.141541,0.968491,-1.81774,-0.239795,0.801499,-1.80535,-0.287593,0.743494,-1.81417,-0.325994,0.871002,-1.81282,-0.371708,0.815309,-1.81739,-0.487285,0.606383,-1.83006,-0.579686,0.6829,-1.8848,-0.476293,0.448596,-1.9187,-0.533257,0.433833,-1.84982,0.241453,1.24758,-2.1126,0.29215,1.18083,-2.0943,0.127897,1.15921,-2.10841,0.177989,1.09554,-2.09363,0.300518,1.01608,-2.09296,0.351658,0.830082,-2.09416,0.238632,0.825986,-2.10608,0.244834,0.756358,-2.09905,0.293223,0.520283,-2.07472,0.23196,0.421487,-2.01096,0.498871,0.509919,-2.05019,0.464062,0.450808,-2.10831,0.044701,1.36399,-1.84081,0.045107,1.39587,-2.05367},
/*184*/{0.196408,1.96107,-1.76799,0.230461,1.89326,-1.92438,0.173184,1.81506,-1.64303,0.156585,1.6967,-1.55154,0.261587,1.71932,-1.67594,0.334304,1.72754,-1.73229,0.110836,1.96427,-2.13497,0.220874,1.88945,-1.99687,0.008298,1.92781,-2.0159,-0.130662,1.85251,-2.15889,-0.158224,1.76513,-2.21941,-0.012793,1.63931,-2.18109,0.022206,1.58538,-2.20463,0.133611,1.43968,-1.81034,-0.049356,1.53226,-1.87807,-0.051741,1.56056,-1.92294,-0.049101,1.54086,-1.99304,0.130749,1.48306,-2.06252,0.010907,1.1349,-1.77719,-0.117084,1.17742,-1.79947,-0.144682,1.09987,-1.82339,-0.135431,0.96637,-1.81881,-0.23828,0.801939,-1.80595,-0.289071,0.746041,-1.81605,-0.32199,0.875354,-1.81094,-0.369309,0.822145,-1.81531,-0.492781,0.616758,-1.82993,-0.586105,0.696655,-1.8775,-0.488972,0.463131,-1.92262,-0.542726,0.446093,-1.85232,0.238304,1.24331,-2.11407,0.287963,1.17598,-2.09538,0.122469,1.15657,-2.10828,0.171989,1.09289,-2.09373,0.296734,1.01183,-2.09341,0.354143,0.828373,-2.09312,0.241576,0.819717,-2.10762,0.250136,0.750553,-2.10027,0.305051,0.515528,-2.07332,0.246892,0.416597,-2.01026,0.512005,0.511737,-2.04825,0.479388,0.450982,-2.10527,0.045267,1.36552,-1.83953,0.045595,1.39645,-2.05253},
/*185*/{0.19667,1.96125,-1.76869,0.230436,1.89383,-1.92504,0.172135,1.81624,-1.64379,0.154807,1.69741,-1.55242,0.259763,1.71873,-1.67662,0.332283,1.72597,-1.73353,0.111582,1.96461,-2.13544,0.22217,1.89022,-1.99715,0.008622,1.92924,-2.0166,-0.131049,1.8527,-2.15719,-0.158272,1.76545,-2.21762,-0.012615,1.63981,-2.18105,0.022731,1.58581,-2.20472,0.133997,1.44153,-1.80959,-0.049224,1.53316,-1.87775,-0.052001,1.5613,-1.92298,-0.048969,1.54125,-1.99307,0.131802,1.4833,-2.06224,0.016683,1.13374,-1.77954,-0.11223,1.17586,-1.79793,-0.138872,1.09706,-1.82352,-0.128566,0.964006,-1.8197,-0.236212,0.802085,-1.8081,-0.288676,0.749521,-1.8178,-0.317202,0.87978,-1.80816,-0.366922,0.82831,-1.81216,-0.497606,0.627456,-1.82867,-0.592479,0.711263,-1.86934,-0.500671,0.477549,-1.92649,-0.551377,0.459275,-1.85535,0.235271,1.23881,-2.11465,0.283419,1.17066,-2.09611,0.118543,1.15345,-2.10873,0.166649,1.0894,-2.09445,0.291359,1.00698,-2.09427,0.354782,0.826842,-2.09193,0.243787,0.814882,-2.10944,0.254519,0.745958,-2.10166,0.316973,0.512833,-2.07281,0.259424,0.413932,-2.00926,0.523337,0.513529,-2.04539,0.492425,0.450035,-2.10165,0.045893,1.36695,-1.83843,0.046404,1.39668,-2.0516},
/*186*/{0.197541,1.961,-1.7692,0.231108,1.89468,-1.92592,0.171406,1.81676,-1.6442,0.15355,1.69752,-1.55215,0.258436,1.71732,-1.67747,0.331001,1.72345,-1.73505,0.111994,1.96453,-2.13604,0.222619,1.89087,-1.9973,0.009579,1.93022,-2.01716,-0.128527,1.85223,-2.15515,-0.15868,1.76579,-2.21662,-0.012242,1.64,-2.18077,0.02361,1.58605,-2.20516,0.134704,1.44276,-1.80982,-0.049591,1.53323,-1.87814,-0.051884,1.56211,-1.92321,-0.049101,1.54086,-1.99304,0.132677,1.48362,-2.0616,0.023985,1.13237,-1.78165,-0.108068,1.17137,-1.79742,-0.132697,1.09281,-1.82143,-0.120642,0.961253,-1.82092,-0.233694,0.802508,-1.80955,-0.289546,0.752128,-1.81852,-0.311602,0.88358,-1.80586,-0.363461,0.834569,-1.81062,-0.502508,0.639983,-1.82767,-0.596894,0.725988,-1.86241,-0.511244,0.492626,-1.93051,-0.560413,0.47329,-1.85705,0.232195,1.23417,-2.11484,0.278697,1.16567,-2.09727,0.113894,1.15135,-2.10715,0.161185,1.08533,-2.09395,0.285611,1.00256,-2.09487,0.356803,0.825167,-2.09018,0.24558,0.809524,-2.11067,0.258657,0.74152,-2.10236,0.326607,0.509236,-2.07276,0.270329,0.409611,-2.0088,0.532041,0.514152,-2.04374,0.503844,0.450045,-2.09927,0.046659,1.36774,-1.838,0.047316,1.39684,-2.05126},
/*187*/{0.19833,1.96044,-1.77024,0.231197,1.89453,-1.92718,0.170189,1.81686,-1.645,0.151071,1.69738,-1.55266,0.257315,1.71547,-1.67827,0.329136,1.72052,-1.73576,0.113074,1.96438,-2.13673,0.22257,1.89097,-1.99785,0.009918,1.93023,-2.0181,-0.128716,1.85283,-2.1535,-0.157766,1.76534,-2.21587,-0.011512,1.6397,-2.18085,0.024525,1.58612,-2.20555,0.13492,1.44385,-1.80903,-0.049425,1.53283,-1.87848,-0.051884,1.56211,-1.92321,-0.048994,1.54055,-1.9935,0.133681,1.48356,-2.0607,0.030824,1.13119,-1.78516,-0.102798,1.16703,-1.79585,-0.126335,1.08916,-1.82269,-0.112813,0.957452,-1.82263,-0.231932,0.803196,-1.81107,-0.288684,0.755924,-1.81967,-0.304721,0.888162,-1.80319,-0.359757,0.840981,-1.80883,-0.506481,0.652068,-1.82573,-0.599754,0.741545,-1.85569,-0.521416,0.508164,-1.93282,-0.569319,0.487561,-1.85911,0.229027,1.22956,-2.11597,0.274342,1.15975,-2.09845,0.109505,1.14861,-2.10697,0.155592,1.0822,-2.09319,0.280782,0.997878,-2.09489,0.357346,0.822959,-2.08906,0.246613,0.804998,-2.11083,0.260547,0.736559,-2.10264,0.335307,0.507191,-2.07122,0.280265,0.405138,-2.00786,0.53962,0.515679,-2.0416,0.512856,0.449902,-2.09676,0.047247,1.36826,-1.83721,0.04859,1.39658,-2.05057},
/*188*/{0.19846,1.96011,-1.77007,0.231,1.89401,-1.92709,0.168849,1.81611,-1.6457,0.148229,1.6973,-1.55365,0.254758,1.7133,-1.67943,0.327171,1.71703,-1.73624,0.113903,1.96443,-2.13708,0.222044,1.89028,-1.99831,0.011517,1.93033,-2.01857,-0.128042,1.851,-2.15173,-0.156895,1.76454,-2.21457,-0.009301,1.63933,-2.18076,0.025364,1.58538,-2.20618,0.134935,1.44406,-1.80893,-0.049274,1.53235,-1.87687,-0.05208,1.56102,-1.92313,-0.048986,1.53832,-1.99366,0.134402,1.48349,-2.06008,0.035934,1.12926,-1.78774,-0.096747,1.16231,-1.79489,-0.119336,1.08481,-1.82266,-0.10387,0.953638,-1.82475,-0.229316,0.804472,-1.8122,-0.289649,0.759053,-1.82101,-0.29855,0.891897,-1.80181,-0.354966,0.847711,-1.80691,-0.509214,0.665027,-1.82446,-0.600625,0.758062,-1.85115,-0.53004,0.524174,-1.93337,-0.578394,0.502706,-1.85878,0.226112,1.22443,-2.11686,0.270814,1.15473,-2.09957,0.105935,1.14577,-2.10699,0.150524,1.07816,-2.09318,0.275025,0.993534,-2.09525,0.356909,0.820889,-2.08759,0.246858,0.800524,-2.11036,0.263193,0.732862,-2.10251,0.341325,0.504412,-2.07094,0.289762,0.403718,-2.0061,0.546823,0.516141,-2.03903,0.5209,0.450311,-2.09494,0.048292,1.36793,-1.83638,0.049556,1.39572,-2.04981},
/*189*/{0.198788,1.95857,-1.77036,0.230762,1.89355,-1.92735,0.167284,1.81565,-1.64626,0.147652,1.69661,-1.55371,0.252829,1.71012,-1.67959,0.325558,1.71321,-1.7373,0.114782,1.96336,-2.1375,0.22196,1.8896,-1.99873,0.011823,1.93025,-2.01902,-0.127169,1.85174,-2.15084,-0.156376,1.76332,-2.21414,-0.008122,1.63796,-2.18081,0.027224,1.58453,-2.20725,0.134979,1.44417,-1.80871,-0.049716,1.53103,-1.87733,-0.052142,1.56064,-1.92404,-0.048387,1.53659,-1.99429,0.135018,1.48284,-2.05949,0.042947,1.12751,-1.79079,-0.090173,1.15966,-1.79456,-0.112382,1.07998,-1.82268,-0.095298,0.949697,-1.82621,-0.225897,0.805848,-1.8131,-0.287977,0.763565,-1.82146,-0.290885,0.896237,-1.80068,-0.349421,0.854248,-1.80563,-0.513402,0.67842,-1.82263,-0.599529,0.7752,-1.84676,-0.537994,0.540485,-1.9327,-0.586051,0.519805,-1.85867,0.223464,1.22014,-2.11762,0.267861,1.14941,-2.10062,0.102192,1.14222,-2.10688,0.146831,1.07453,-2.09284,0.269996,0.989405,-2.09472,0.355867,0.819273,-2.08674,0.246162,0.796402,-2.10947,0.263216,0.729734,-2.10183,0.344788,0.50188,-2.06991,0.293956,0.400744,-2.00578,0.551259,0.516638,-2.03801,0.526045,0.45029,-2.09364,0.048693,1.36756,-1.83606,0.050737,1.39467,-2.04957},
/*190*/{0.198874,1.95766,-1.77091,0.231297,1.89312,-1.92778,0.165071,1.81449,-1.647,0.144837,1.69573,-1.5547,0.249851,1.70732,-1.68031,0.322725,1.70887,-1.73775,0.114929,1.96236,-2.13757,0.222211,1.88923,-1.99909,0.012638,1.9298,-2.01937,-0.127815,1.85077,-2.14858,-0.155082,1.76207,-2.21344,-0.007489,1.63682,-2.1815,0.028592,1.58276,-2.20736,0.135015,1.44417,-1.80869,-0.049445,1.52959,-1.87753,-0.051466,1.55912,-1.92421,-0.047925,1.53487,-1.99429,0.13592,1.48132,-2.05927,0.050117,1.12473,-1.79251,-0.083137,1.15399,-1.79408,-0.105233,1.07424,-1.82181,-0.085516,0.945288,-1.82811,-0.221194,0.808176,-1.81302,-0.286126,0.767822,-1.8212,-0.283105,0.900461,-1.80029,-0.343533,0.860725,-1.80508,-0.51355,0.692965,-1.82071,-0.597356,0.79261,-1.84502,-0.544706,0.555863,-1.93117,-0.594648,0.536647,-1.85741,0.221245,1.21549,-2.1188,0.264434,1.14415,-2.1019,0.099143,1.13937,-2.10546,0.142929,1.07119,-2.09212,0.265465,0.984928,-2.09493,0.354356,0.816504,-2.08566,0.244715,0.792902,-2.10801,0.262371,0.724909,-2.09985,0.348441,0.499439,-2.06918,0.298648,0.397083,-2.00412,0.553237,0.517107,-2.03788,0.529131,0.449657,-2.0929,0.049499,1.36682,-1.83542,0.051768,1.39302,-2.04905},
/*191*/{0.198963,1.95563,-1.77095,0.231027,1.89124,-1.92774,0.163146,1.81343,-1.64724,0.140684,1.69425,-1.55501,0.247734,1.70319,-1.6808,0.320823,1.70354,-1.7382,0.1159,1.96077,-2.13779,0.222338,1.88787,-1.99944,0.012834,1.92936,-2.02049,-0.125513,1.84962,-2.14923,-0.153955,1.75968,-2.21286,-0.006081,1.63452,-2.18191,0.029749,1.58138,-2.20797,0.135068,1.44381,-1.80862,-0.04992,1.52843,-1.87845,-0.049734,1.55668,-1.92581,-0.04796,1.53277,-1.99484,0.136228,1.48023,-2.05905,0.058437,1.12357,-1.79468,-0.077907,1.14698,-1.79329,-0.096794,1.06903,-1.82298,-0.076879,0.941211,-1.82922,-0.217261,0.810681,-1.81366,-0.284037,0.773143,-1.82087,-0.274925,0.905881,-1.80042,-0.336973,0.868809,-1.80412,-0.51555,0.707639,-1.81903,-0.59382,0.8106,-1.84362,-0.550245,0.571383,-1.92872,-0.602417,0.554411,-1.85673,0.219174,1.21073,-2.11992,0.262499,1.13864,-2.10295,0.096206,1.13676,-2.1057,0.139339,1.06822,-2.09168,0.260511,0.980168,-2.09448,0.350764,0.813719,-2.08556,0.241801,0.788745,-2.10627,0.260744,0.721257,-2.09787,0.348548,0.496303,-2.06764,0.301195,0.394038,-2.00352,0.554958,0.51602,-2.03721,0.529694,0.448655,-2.09183,0.050141,1.36593,-1.83528,0.052982,1.39121,-2.04901},
/*192*/{0.198025,1.95363,-1.77129,0.231606,1.89072,-1.92793,0.160557,1.81166,-1.6476,0.137237,1.69257,-1.55608,0.244702,1.69972,-1.68094,0.317631,1.69815,-1.73811,0.116513,1.95939,-2.13807,0.222081,1.88657,-1.99932,0.0137,1.92802,-2.02161,-0.124351,1.84615,-2.14887,-0.152352,1.7575,-2.2131,-0.003731,1.63258,-2.18234,0.032157,1.57892,-2.20903,0.135071,1.44275,-1.80855,-0.049329,1.52539,-1.87921,-0.050528,1.55446,-1.92672,-0.047817,1.53002,-1.99529,0.137363,1.47788,-2.05932,0.06418,1.11985,-1.7954,-0.071976,1.14118,-1.79317,-0.089791,1.06322,-1.82311,-0.067795,0.936671,-1.83045,-0.212469,0.812896,-1.81375,-0.279929,0.777097,-1.82143,-0.266354,0.910131,-1.80035,-0.329322,0.875665,-1.80353,-0.515515,0.721821,-1.8169,-0.588736,0.828541,-1.84356,-0.554742,0.586683,-1.9258,-0.608375,0.573522,-1.85436,0.217917,1.20643,-2.12043,0.25957,1.13415,-2.10439,0.094292,1.13406,-2.10491,0.136946,1.06519,-2.09154,0.256313,0.975939,-2.09448,0.347573,0.810322,-2.08522,0.238813,0.784197,-2.10389,0.258634,0.718044,-2.096,0.349946,0.494637,-2.06534,0.301165,0.389451,-2.00174,0.555114,0.514102,-2.03695,0.53024,0.446946,-2.09157,0.050894,1.36415,-1.83516,0.05419,1.38867,-2.04897},
/*193*/{0.197698,1.9508,-1.77098,0.231249,1.88872,-1.92804,0.158883,1.8096,-1.64785,0.132285,1.69027,-1.55745,0.241565,1.69541,-1.68172,0.315262,1.69212,-1.73804,0.117008,1.95744,-2.13841,0.222116,1.88474,-1.99958,0.013354,1.92687,-2.02222,-0.123157,1.8439,-2.14865,-0.151246,1.75414,-2.21312,-0.001592,1.62995,-2.18301,0.033992,1.57636,-2.20943,0.135536,1.44146,-1.8088,-0.048604,1.52344,-1.87945,-0.050768,1.55182,-1.92701,-0.046918,1.52777,-1.99648,0.137597,1.47538,-2.06,0.06991,1.11738,-1.79656,-0.06659,1.13627,-1.79287,-0.082144,1.05724,-1.8234,-0.058338,0.932319,-1.83104,-0.206992,0.815826,-1.81272,-0.275881,0.783029,-1.81917,-0.257037,0.915456,-1.80059,-0.321229,0.883166,-1.8038,-0.514163,0.736245,-1.81413,-0.582746,0.846189,-1.84342,-0.557767,0.601849,-1.92167,-0.613109,0.59206,-1.851,0.216277,1.20186,-2.1227,0.257331,1.12958,-2.10596,0.092494,1.13135,-2.1045,0.133901,1.06209,-2.09117,0.253342,0.971473,-2.09454,0.343834,0.807202,-2.08465,0.235294,0.780629,-2.10287,0.254917,0.713364,-2.09372,0.346888,0.491195,-2.06282,0.300802,0.386091,-1.99875,0.552964,0.510807,-2.03704,0.527702,0.444748,-2.09186,0.051648,1.36249,-1.8351,0.054735,1.38611,-2.04902},
/*194*/{0.197575,1.94904,-1.77114,0.230759,1.88654,-1.92792,0.15635,1.80725,-1.64792,0.128878,1.68737,-1.55804,0.238593,1.69071,-1.68182,0.312463,1.68563,-1.73775,0.117109,1.9554,-2.13846,0.22212,1.88239,-1.99937,0.013414,1.92567,-2.02326,-0.12243,1.84067,-2.1484,-0.150316,1.75068,-2.2133,0.000125,1.62675,-2.18413,0.03583,1.57326,-2.21055,0.135586,1.44028,-1.80888,-0.048684,1.52119,-1.8802,-0.050129,1.54887,-1.92806,-0.047495,1.52482,-1.99715,0.137908,1.47231,-2.06101,0.074656,1.1143,-1.79697,-0.06052,1.12929,-1.79218,-0.076097,1.05164,-1.82312,-0.050084,0.928879,-1.83186,-0.201373,0.818765,-1.81225,-0.272181,0.788535,-1.81823,-0.248275,0.920623,-1.80073,-0.313536,0.890242,-1.80314,-0.511769,0.752021,-1.81267,-0.575113,0.863531,-1.84452,-0.560009,0.616606,-1.91782,-0.616295,0.610577,-1.84789,0.215658,1.19726,-2.12379,0.255641,1.12545,-2.10719,0.09096,1.12919,-2.10414,0.131333,1.05887,-2.08964,0.249622,0.967033,-2.09437,0.340312,0.802888,-2.08355,0.231529,0.77634,-2.10096,0.252457,0.709764,-2.09133,0.344264,0.48809,-2.0608,0.297255,0.383197,-1.99585,0.54968,0.506854,-2.03648,0.524102,0.441051,-2.09159,0.052152,1.3608,-1.83482,0.055053,1.38289,-2.0489},
/*195*/{0.196571,1.94583,-1.7713,0.231087,1.88419,-1.92815,0.1538,1.80423,-1.64854,0.124559,1.68471,-1.55927,0.234996,1.68568,-1.682,0.309246,1.6789,-1.73723,0.117417,1.9529,-2.1386,0.222705,1.88048,-1.99935,0.014108,1.9231,-2.02391,-0.121534,1.83664,-2.14806,-0.148907,1.74653,-2.21439,0.001854,1.62327,-2.18542,0.038723,1.56995,-2.21148,0.136181,1.43841,-1.80943,-0.049031,1.5176,-1.88051,-0.049553,1.5469,-1.92851,-0.046872,1.52149,-1.99821,0.138792,1.47016,-2.0621,0.080093,1.11311,-1.79719,-0.055825,1.12474,-1.79265,-0.068441,1.04553,-1.82384,-0.040008,0.924523,-1.83233,-0.195568,0.820931,-1.81084,-0.265399,0.794027,-1.81642,-0.238908,0.925412,-1.80257,-0.304514,0.897577,-1.80316,-0.507021,0.766467,-1.81183,-0.566292,0.879941,-1.84616,-0.560263,0.631436,-1.91374,-0.618077,0.628153,-1.84418,0.214659,1.19343,-2.12503,0.253812,1.12086,-2.10898,0.08907,1.12668,-2.10344,0.12921,1.05658,-2.08916,0.248099,0.963256,-2.0933,0.336505,0.79918,-2.08391,0.227312,0.772543,-2.09967,0.248255,0.705843,-2.08991,0.339144,0.483537,-2.05831,0.2937,0.379584,-1.99011,0.546188,0.502778,-2.03677,0.519099,0.436437,-2.09129,0.053233,1.35836,-1.83548,0.056184,1.38045,-2.04956},
/*196*/{0.196307,1.94282,-1.77082,0.230569,1.88181,-1.92818,0.151424,1.80116,-1.64847,0.120124,1.68171,-1.56027,0.231369,1.68017,-1.68196,0.306425,1.6715,-1.73621,0.11677,1.94985,-2.13899,0.222673,1.87802,-1.99868,0.013226,1.92136,-2.02538,-0.119837,1.83295,-2.149,-0.147404,1.74225,-2.21535,0.003624,1.61896,-2.18656,0.04084,1.56588,-2.21212,0.136938,1.43611,-1.81015,-0.049932,1.5142,-1.88257,-0.049848,1.54277,-1.92919,-0.04712,1.51848,-1.99871,0.138868,1.46756,-2.06299,0.086609,1.10895,-1.79657,-0.050523,1.12196,-1.79419,-0.061022,1.03998,-1.82362,-0.032158,0.920557,-1.83231,-0.188583,0.82317,-1.81052,-0.260388,0.799123,-1.8154,-0.228486,0.927802,-1.8043,-0.29575,0.904098,-1.80279,-0.503281,0.779999,-1.81005,-0.55622,0.895178,-1.84679,-0.559641,0.645973,-1.909,-0.618222,0.647577,-1.83938,0.214043,1.18988,-2.12623,0.25256,1.11618,-2.11018,0.087838,1.12397,-2.10322,0.1271,1.05357,-2.08896,0.246144,0.959416,-2.09283,0.332915,0.79497,-2.0835,0.222916,0.768652,-2.09816,0.242979,0.701442,-2.08829,0.332531,0.478893,-2.05604,0.287412,0.377358,-1.98759,0.5403,0.496249,-2.03722,0.511989,0.429835,-2.08997,0.05399,1.35549,-1.83637,0.056887,1.3775,-2.05046},
/*197*/{0.195285,1.93965,-1.77004,0.231033,1.8788,-1.92736,0.148538,1.79778,-1.64848,0.115091,1.67824,-1.56106,0.227193,1.6744,-1.68167,0.302364,1.66433,-1.73622,0.11682,1.94678,-2.13931,0.222667,1.87527,-1.99866,0.012573,1.91835,-2.02606,-0.119476,1.82852,-2.1491,-0.145625,1.73694,-2.21598,0.005453,1.61506,-2.18779,0.043555,1.56241,-2.21375,0.137242,1.43372,-1.81038,-0.049565,1.51069,-1.8825,-0.049906,1.53963,-1.93033,-0.047733,1.51449,-1.99917,0.138928,1.46477,-2.06454,0.091321,1.10661,-1.79618,-0.045983,1.11624,-1.79375,-0.054992,1.03516,-1.82359,-0.023349,0.916533,-1.83187,-0.181629,0.82615,-1.80913,-0.254107,0.804534,-1.8139,-0.217569,0.93291,-1.80446,-0.284756,0.909562,-1.80303,-0.496605,0.794349,-1.80844,-0.546284,0.91043,-1.8495,-0.558367,0.659179,-1.90429,-0.616516,0.664832,-1.83528,0.213174,1.18659,-2.12782,0.250839,1.11272,-2.11116,0.086409,1.12302,-2.10255,0.12492,1.05124,-2.089,0.24502,0.956558,-2.09268,0.328925,0.789541,-2.08292,0.21845,0.764146,-2.0964,0.237983,0.697679,-2.08662,0.325983,0.473961,-2.05357,0.279935,0.372587,-1.98219,0.533031,0.48893,-2.03667,0.504419,0.421725,-2.08829,0.054926,1.35273,-1.83674,0.057189,1.37419,-2.05089},
/*198*/{0.193917,1.93588,-1.76944,0.231552,1.8764,-1.92748,0.146065,1.794,-1.64845,0.112161,1.67493,-1.56155,0.222779,1.66818,-1.68142,0.298896,1.65553,-1.73439,0.116863,1.94355,-2.13933,0.222829,1.87248,-1.99893,0.012543,1.91538,-2.02659,-0.119605,1.82364,-2.15129,-0.144466,1.73129,-2.21765,0.008198,1.61017,-2.18916,0.046053,1.55749,-2.21534,0.137133,1.43153,-1.8111,-0.04949,1.50714,-1.88327,-0.049558,1.53642,-1.93102,-0.048031,1.51021,-2.00027,0.139103,1.46186,-2.06547,0.096174,1.10335,-1.79501,-0.041227,1.10926,-1.79404,-0.049127,1.02996,-1.82601,-0.01417,0.9126,-1.83104,-0.174713,0.828181,-1.80767,-0.247505,0.808422,-1.8125,-0.206785,0.935932,-1.80547,-0.275329,0.915962,-1.8033,-0.4897,0.807914,-1.80587,-0.535958,0.924017,-1.85107,-0.554681,0.671815,-1.8995,-0.613628,0.681461,-1.83118,0.212003,1.18297,-2.12867,0.249546,1.10886,-2.11231,0.084789,1.12002,-2.10202,0.122818,1.04814,-2.08943,0.243353,0.953166,-2.0916,0.32454,0.784176,-2.08285,0.21372,0.760356,-2.09503,0.232501,0.693164,-2.08502,0.31785,0.468629,-2.05145,0.271654,0.367559,-1.97784,0.525658,0.481227,-2.03565,0.495459,0.413652,-2.0864,0.055667,1.34996,-1.83716,0.057795,1.37077,-2.05137},
/*199*/{0.192742,1.93201,-1.7687,0.231223,1.87257,-1.92715,0.143404,1.78957,-1.64795,0.105645,1.6713,-1.56271,0.218558,1.66158,-1.68109,0.295041,1.64737,-1.73327,0.116833,1.93975,-2.13951,0.222701,1.86982,-1.99892,0.012615,1.91135,-2.02775,-0.118279,1.81737,-2.15216,-0.142767,1.72534,-2.21861,0.010087,1.60487,-2.19111,0.04951,1.55249,-2.21628,0.136923,1.42859,-1.81274,-0.049419,1.5029,-1.88366,-0.051228,1.53227,-1.93224,-0.048031,1.50554,-2.00015,0.138751,1.45896,-2.06685,0.100872,1.10092,-1.79456,-0.036286,1.10397,-1.79542,-0.042303,1.02389,-1.82559,-0.006172,0.908964,-1.82914,-0.166518,0.830801,-1.80713,-0.240452,0.813216,-1.8103,-0.19532,0.939709,-1.80589,-0.264026,0.920713,-1.80449,-0.482656,0.819823,-1.80333,-0.524801,0.937296,-1.85299,-0.551288,0.683539,-1.89468,-0.609009,0.696515,-1.82704,0.211167,1.17924,-2.12875,0.247248,1.10474,-2.11331,0.083455,1.11667,-2.10172,0.121486,1.04568,-2.08909,0.241694,0.950103,-2.09141,0.319709,0.778804,-2.08276,0.208748,0.755825,-2.09383,0.226957,0.688552,-2.08319,0.308467,0.463121,-2.04872,0.26056,0.364309,-1.97578,0.516565,0.471921,-2.03368,0.48537,0.404295,-2.08429,0.056521,1.34638,-1.83817,0.057942,1.36726,-2.05239},
/*200*/{0.192019,1.92781,-1.76759,0.231822,1.86897,-1.92727,0.139864,1.78522,-1.64814,0.100642,1.66692,-1.56422,0.214336,1.65476,-1.68133,0.290629,1.63858,-1.73199,0.117199,1.93553,-2.13954,0.222376,1.86624,-1.99847,0.012319,1.90759,-2.02834,-0.11839,1.81102,-2.15499,-0.140989,1.71906,-2.21975,0.014243,1.59987,-2.19272,0.052099,1.54751,-2.21791,0.138549,1.42604,-1.81398,-0.049866,1.49858,-1.88413,-0.050519,1.52829,-1.93207,-0.049082,1.50205,-2.00109,0.13855,1.45572,-2.06803,0.105458,1.09784,-1.79303,-0.031312,1.09849,-1.79549,-0.035113,1.01856,-1.8255,0.00221,0.905346,-1.82748,-0.158541,0.832854,-1.80464,-0.23192,0.816037,-1.81009,-0.183897,0.942714,-1.80652,-0.253753,0.926313,-1.80469,-0.474586,0.833489,-1.80306,-0.513729,0.949355,-1.85601,-0.544341,0.695601,-1.8901,-0.60278,0.710864,-1.82247,0.208915,1.17626,-2.12969,0.246478,1.10115,-2.11402,0.082025,1.11312,-2.10129,0.119955,1.04178,-2.08882,0.239947,0.946359,-2.09134,0.31496,0.772097,-2.08275,0.203074,0.751437,-2.09317,0.219428,0.683922,-2.0818,0.298909,0.456994,-2.04699,0.248211,0.363025,-1.9736,0.506461,0.462314,-2.03229,0.47406,0.394421,-2.08144,0.058054,1.34317,-1.83892,0.058183,1.36382,-2.05317},
/*201*/{0.190695,1.92349,-1.76722,0.231811,1.86474,-1.92608,0.136923,1.78035,-1.64752,0.094605,1.66195,-1.5649,0.209787,1.64751,-1.68126,0.285938,1.62899,-1.73012,0.117349,1.93176,-2.13988,0.222072,1.86217,-1.9983,0.010957,1.90259,-2.02874,-0.117637,1.80396,-2.15671,-0.139116,1.71208,-2.22191,0.016961,1.59378,-2.19351,0.055382,1.54219,-2.21919,0.139129,1.42285,-1.81455,-0.050344,1.49351,-1.88484,-0.05067,1.5235,-1.93262,-0.04938,1.49662,-2.00073,0.137506,1.45193,-2.06937,0.109839,1.09508,-1.79121,-0.026427,1.096,-1.79797,-0.029331,1.01359,-1.82693,0.010971,0.902087,-1.82587,-0.150138,0.835239,-1.80255,-0.224332,0.821482,-1.80511,-0.172638,0.944961,-1.80916,-0.243034,0.930924,-1.80672,-0.465263,0.845229,-1.80298,-0.503453,0.959931,-1.85793,-0.537322,0.704944,-1.88589,-0.595605,0.723975,-1.81917,0.20851,1.17223,-2.1306,0.245286,1.09763,-2.11514,0.081088,1.10945,-2.10143,0.119947,1.03789,-2.08858,0.237501,0.941627,-2.09147,0.30996,0.765761,-2.0823,0.196992,0.747149,-2.09169,0.213169,0.678921,-2.08085,0.288271,0.451744,-2.04269,0.232548,0.358427,-1.96549,0.49602,0.452553,-2.03104,0.461808,0.384655,-2.07832,0.059277,1.33924,-1.83926,0.058306,1.35931,-2.05356},
/*202*/{0.189155,1.91888,-1.76614,0.230064,1.86056,-1.92562,0.133782,1.77543,-1.64816,0.088927,1.65798,-1.56678,0.204479,1.6399,-1.68012,0.281919,1.62044,-1.72843,0.117586,1.92765,-2.14027,0.222218,1.85829,-1.998,0.010872,1.89864,-2.03027,-0.11861,1.79665,-2.15781,-0.13692,1.70475,-2.22387,0.019904,1.58797,-2.19635,0.05888,1.53622,-2.2208,0.13918,1.41949,-1.8161,-0.050218,1.48858,-1.88512,-0.050707,1.51817,-1.93392,-0.050127,1.49202,-2.00096,0.137146,1.44868,-2.07076,0.114497,1.09233,-1.79,-0.022746,1.08938,-1.79939,-0.02162,1.00844,-1.82742,0.019903,0.899,-1.82345,-0.140978,0.836925,-1.80119,-0.215792,0.82489,-1.80276,-0.16077,0.947009,-1.81022,-0.231645,0.935076,-1.80787,-0.455206,0.855379,-1.80242,-0.491979,0.969708,-1.86039,-0.52939,0.714862,-1.88148,-0.586938,0.736828,-1.81396,0.206758,1.1678,-2.13136,0.244082,1.09321,-2.11601,0.079851,1.10541,-2.10089,0.118791,1.03384,-2.08859,0.235258,0.935987,-2.09135,0.302789,0.758475,-2.08231,0.19095,0.742282,-2.09099,0.205221,0.67428,-2.08042,0.277279,0.44485,-2.04312,0.222034,0.353712,-1.96651,0.484746,0.44215,-2.02826,0.449563,0.373998,-2.0755,0.060458,1.33511,-1.84047,0.058664,1.3554,-2.05474},
/*203*/{0.187994,1.9144,-1.76529,0.231036,1.85597,-1.92465,0.130454,1.77034,-1.64812,0.083636,1.65413,-1.56802,0.20008,1.63274,-1.67984,0.277988,1.61061,-1.72737,0.117459,1.92244,-2.14023,0.222168,1.85452,-1.9977,0.010103,1.89372,-2.03141,-0.11846,1.78876,-2.16073,-0.134714,1.69738,-2.22563,0.022557,1.5809,-2.19784,0.062786,1.52991,-2.22211,0.139202,1.41563,-1.81751,-0.050673,1.48396,-1.88449,-0.051605,1.51335,-1.9341,-0.050443,1.48629,-2.00055,0.136853,1.44571,-2.07214,0.118746,1.0889,-1.78703,-0.017773,1.08645,-1.80118,-0.016947,1.00308,-1.8267,0.028832,0.8961,-1.82113,-0.132626,0.83909,-1.7992,-0.207484,0.828689,-1.80306,-0.148938,0.949188,-1.81146,-0.219885,0.938455,-1.80947,-0.44546,0.865685,-1.80213,-0.481085,0.978692,-1.86291,-0.520807,0.7237,-1.87668,-0.577457,0.747872,-1.80989,0.205383,1.16332,-2.13172,0.242974,1.08929,-2.11696,0.078786,1.10094,-2.1005,0.11794,1.02856,-2.08773,0.233602,0.930016,-2.09138,0.297387,0.751156,-2.08236,0.184519,0.737478,-2.09062,0.197384,0.669004,-2.07956,0.265009,0.439438,-2.04244,0.207064,0.351479,-1.96537,0.472719,0.430565,-2.02489,0.435585,0.363121,-2.07238,0.061623,1.33075,-1.84135,0.058763,1.35155,-2.05556},
/*204*/{0.18667,1.90906,-1.76433,0.229679,1.85107,-1.92347,0.126547,1.76498,-1.64774,0.077449,1.65013,-1.56972,0.194254,1.62535,-1.67942,0.272529,1.60048,-1.72543,0.117626,1.91753,-2.14056,0.222812,1.85026,-1.99765,0.008715,1.8878,-2.03229,-0.115652,1.78055,-2.16374,-0.131939,1.68967,-2.22819,0.025865,1.574,-2.19939,0.067055,1.52342,-2.22366,0.140127,1.41215,-1.81877,-0.050342,1.47797,-1.88486,-0.051473,1.50827,-1.93437,-0.052312,1.48044,-2.00011,0.136178,1.44145,-2.07363,0.12264,1.08693,-1.78409,-0.014678,1.07889,-1.80149,-0.010959,0.99755,-1.82738,0.037265,0.89304,-1.81922,-0.122293,0.840594,-1.79626,-0.197285,0.832323,-1.80015,-0.136425,0.950558,-1.81243,-0.207729,0.942321,-1.81017,-0.434035,0.874684,-1.80163,-0.469954,0.986713,-1.86517,-0.510409,0.731174,-1.8726,-0.567185,0.756816,-1.80729,0.203871,1.15897,-2.13288,0.241727,1.08493,-2.11725,0.077547,1.09578,-2.1003,0.116731,1.02446,-2.08826,0.230886,0.9239,-2.09128,0.290678,0.744038,-2.08191,0.177231,0.732587,-2.09015,0.189443,0.663551,-2.07902,0.252646,0.435038,-2.04075,0.1882,0.356265,-1.96703,0.457588,0.417512,-2.02279,0.417085,0.352308,-2.06955,0.063405,1.3264,-1.8419,0.058735,1.34667,-2.05613},
/*205*/{0.184947,1.9038,-1.76334,0.230649,1.84542,-1.92281,0.123416,1.75964,-1.64814,0.071842,1.64634,-1.57135,0.189096,1.6172,-1.67911,0.26748,1.59013,-1.7232,0.118365,1.91267,-2.1413,0.222902,1.8448,-1.99718,0.008896,1.88211,-2.03363,-0.115645,1.7728,-2.16705,-0.129013,1.68151,-2.23017,0.031366,1.56712,-2.20114,0.070701,1.51674,-2.22552,0.140001,1.40811,-1.82037,-0.050098,1.47221,-1.88525,-0.051747,1.50206,-1.93472,-0.05287,1.47512,-2.00023,0.13583,1.43731,-2.0752,0.126959,1.0832,-1.7813,-0.009286,1.07218,-1.80191,-0.00455,0.991716,-1.82822,0.046023,0.890404,-1.81604,-0.112509,0.842027,-1.79374,-0.187181,0.834406,-1.79757,-0.123724,0.951583,-1.81408,-0.195613,0.94512,-1.81101,-0.421441,0.882187,-1.80037,-0.458001,0.993468,-1.86779,-0.500311,0.73821,-1.86887,-0.556357,0.766052,-1.80304,0.20209,1.15444,-2.13358,0.24078,1.08077,-2.11896,0.076084,1.09075,-2.10112,0.115305,1.01932,-2.08839,0.228134,0.9175,-2.09107,0.283965,0.736937,-2.08192,0.171098,0.727153,-2.08958,0.182477,0.657578,-2.07766,0.23923,0.429841,-2.04015,0.167747,0.35487,-1.96699,0.443837,0.401781,-2.01942,0.399367,0.340914,-2.06716,0.064647,1.32147,-1.84305,0.058973,1.34199,-2.05723},
/*206*/{0.18391,1.89942,-1.76232,0.232053,1.83955,-1.92236,0.119076,1.75468,-1.64785,0.065441,1.64165,-1.57175,0.18427,1.60868,-1.67802,0.261404,1.58012,-1.72168,0.118653,1.90758,-2.14164,0.22327,1.84045,-1.99694,0.008304,1.87525,-2.03416,-0.113426,1.76344,-2.16964,-0.125394,1.67268,-2.23259,0.034888,1.55949,-2.2031,0.075489,1.50931,-2.22701,0.140674,1.40395,-1.82147,-0.051236,1.46638,-1.88568,-0.052342,1.49676,-1.9355,-0.053476,1.46848,-1.99978,0.13591,1.43334,-2.07678,0.128951,1.07971,-1.78017,-0.005856,1.06621,-1.80245,0.002672,0.986123,-1.82784,0.056721,0.888298,-1.81398,-0.100861,0.843375,-1.79253,-0.176263,0.837737,-1.79531,-0.109871,0.953508,-1.81508,-0.181847,0.947702,-1.81264,-0.4109,0.888952,-1.80123,-0.446329,0.999364,-1.87011,-0.488505,0.74333,-1.86472,-0.544344,0.773554,-1.7994,0.199211,1.14853,-2.13488,0.239383,1.07441,-2.11859,0.07416,1.08562,-2.10157,0.113515,1.01416,-2.08815,0.224604,0.911535,-2.0912,0.277637,0.729971,-2.08077,0.163938,0.721876,-2.08941,0.173463,0.653,-2.07847,0.225188,0.425169,-2.0373,0.147363,0.356985,-1.96683,0.429315,0.38824,-2.00909,0.380368,0.340388,-2.06652,0.066034,1.3167,-1.84388,0.059517,1.33719,-2.05804},
/*207*/{0.18279,1.89304,-1.76129,0.231553,1.83378,-1.92106,0.116236,1.74873,-1.64794,0.059739,1.63758,-1.57333,0.178948,1.60119,-1.67766,0.256093,1.56979,-1.71925,0.119192,1.90209,-2.14206,0.223597,1.83475,-1.99589,0.007562,1.86971,-2.03593,-0.112013,1.75473,-2.17257,-0.121927,1.6641,-2.23549,0.038747,1.5535,-2.20589,0.080926,1.50236,-2.22861,0.140459,1.39891,-1.82249,-0.052299,1.46081,-1.8865,-0.052954,1.49034,-1.9362,-0.055193,1.46379,-2.00025,0.135118,1.42857,-2.07861,0.132363,1.0756,-1.77658,-0.000217,1.06078,-1.80364,0.009751,0.979422,-1.82803,0.066164,0.886915,-1.81117,-0.089843,0.844116,-1.7903,-0.163904,0.840365,-1.7944,-0.097291,0.95312,-1.81558,-0.169361,0.950083,-1.81452,-0.397629,0.894203,-1.80041,-0.434369,1.00401,-1.87195,-0.475139,0.74803,-1.86049,-0.531102,0.779859,-1.79662,0.196908,1.14382,-2.13699,0.236013,1.06929,-2.12052,0.070919,1.08021,-2.10124,0.11112,1.00902,-2.08846,0.220249,0.906393,-2.09198,0.270863,0.722658,-2.0785,0.157397,0.717371,-2.08909,0.164292,0.648241,-2.07768,0.207577,0.422769,-2.03047,0.124621,0.353646,-1.96586,0.407587,0.385522,-2.0044,0.357848,0.341239,-2.06716,0.065954,1.31135,-1.84527,0.058745,1.33227,-2.05936},
/*208*/{0.181141,1.8869,-1.76031,0.231133,1.82874,-1.91946,0.111462,1.74308,-1.64813,0.052295,1.63299,-1.57425,0.17319,1.5931,-1.67605,0.249801,1.55941,-1.71642,0.120332,1.89676,-2.14262,0.224398,1.82967,-1.99613,0.006797,1.86231,-2.03695,-0.109227,1.74475,-2.17575,-0.118016,1.6551,-2.23817,0.044493,1.54563,-2.20835,0.086288,1.49531,-2.23032,0.140129,1.3946,-1.82434,-0.052985,1.45436,-1.88556,-0.054424,1.48429,-1.93666,-0.05724,1.45794,-1.9996,0.134174,1.42376,-2.08062,0.137027,1.07217,-1.77471,0.005441,1.05604,-1.80432,0.01667,0.973083,-1.82833,0.07618,0.882667,-1.80943,-0.078025,0.844716,-1.78806,-0.153586,0.842473,-1.79288,-0.082148,0.95383,-1.81756,-0.154125,0.951264,-1.8132,-0.384108,0.900043,-1.79958,-0.422456,1.00744,-1.87421,-0.461446,0.752536,-1.85777,-0.51772,0.784769,-1.79327,0.193463,1.13808,-2.13976,0.231664,1.06421,-2.12402,0.067403,1.07737,-2.10257,0.106532,1.00567,-2.08859,0.214877,0.899885,-2.09324,0.264511,0.715104,-2.07392,0.151718,0.712977,-2.08844,0.156153,0.64461,-2.07566,0.187854,0.424517,-2.02496,0.101358,0.354534,-1.96257,0.38143,0.386402,-2.00752,0.332808,0.341767,-2.06671,0.06675,1.30619,-1.84619,0.057728,1.3271,-2.06021},
/*209*/{0.179337,1.88121,-1.75905,0.232528,1.82156,-1.91891,0.107234,1.73722,-1.64866,0.045916,1.63,-1.57546,0.167522,1.5853,-1.67543,0.243572,1.55023,-1.71404,0.121382,1.89156,-2.14394,0.225544,1.82365,-1.9947,0.006098,1.85702,-2.03808,-0.106206,1.73615,-2.17918,-0.112812,1.64605,-2.24138,0.049673,1.53781,-2.21065,0.092297,1.48804,-2.23178,0.139815,1.38948,-1.82462,-0.055071,1.44939,-1.88534,-0.05648,1.47799,-1.93641,-0.059411,1.45354,-2.00042,0.13232,1.41808,-2.082,0.141371,1.06808,-1.77261,0.009993,1.04963,-1.805,0.024845,0.96669,-1.82773,0.086938,0.880597,-1.80584,-0.065656,0.845252,-1.78597,-0.140469,0.844002,-1.7898,-0.068972,0.953603,-1.81747,-0.140686,0.952001,-1.81422,-0.370586,0.904086,-1.80076,-0.409507,1.0101,-1.87572,-0.448336,0.75548,-1.85402,-0.503426,0.788478,-1.79016,0.188752,1.13416,-2.14516,0.227057,1.05922,-2.12951,0.063169,1.07518,-2.10235,0.102371,1.00368,-2.08826,0.208162,0.893397,-2.09481,0.260163,0.709138,-2.06903,0.147543,0.711409,-2.08664,0.147459,0.64296,-2.07067,0.162037,0.425602,-2.01985,0.077332,0.354611,-1.96081,0.359353,0.382905,-2.00236,0.310017,0.340978,-2.066,0.065604,1.30107,-1.8466,0.05564,1.32169,-2.06061},
/*210*/{0.17767,1.87589,-1.75833,0.233136,1.81669,-1.91764,0.102904,1.73173,-1.64909,0.039122,1.62583,-1.57748,0.161209,1.57785,-1.67489,0.23778,1.54134,-1.7117,0.12223,1.88675,-2.14534,0.225815,1.81818,-1.99466,0.006505,1.8505,-2.03919,-0.104342,1.7266,-2.18338,-0.107127,1.63668,-2.24444,0.055622,1.52967,-2.21314,0.098634,1.48077,-2.2335,0.139668,1.38434,-1.82628,-0.057453,1.44335,-1.8857,-0.05835,1.47207,-1.9363,-0.062009,1.44887,-2.00108,0.129976,1.41421,-2.08231,0.146012,1.06625,-1.76981,0.016991,1.04229,-1.80403,0.03288,0.959946,-1.82667,0.097614,0.876751,-1.80339,-0.051971,0.844803,-1.7842,-0.127415,0.844636,-1.78831,-0.055612,0.952794,-1.81784,-0.126219,0.952183,-1.81385,-0.35866,0.906515,-1.79935,-0.397027,1.01226,-1.87753,-0.433727,0.75702,-1.85067,-0.489542,0.790905,-1.78719,0.183618,1.13069,-2.15097,0.222214,1.05536,-2.13501,0.06045,1.0736,-2.10084,0.098431,1.00164,-2.08773,0.201694,0.88958,-2.09516,0.25333,0.706946,-2.0647,0.141144,0.715599,-2.08382,0.137454,0.648005,-2.06801,0.141835,0.427397,-2.01808,0.055738,0.356028,-1.9578,0.336303,0.380536,-2.00332,0.286412,0.341806,-2.06458,0.064979,1.29563,-1.84818,0.05401,1.31747,-2.06203},
/*211*/{0.177203,1.87069,-1.75687,0.233775,1.81031,-1.91628,0.099466,1.72751,-1.64966,0.033235,1.62258,-1.57901,0.155103,1.57018,-1.67229,0.229768,1.53272,-1.70899,0.123817,1.88143,-2.14624,0.228125,1.81285,-1.9935,0.007222,1.8476,-2.03971,-0.099034,1.71732,-2.18699,-0.098869,1.62709,-2.2475,0.06209,1.52237,-2.21626,0.106014,1.47427,-2.23515,0.138923,1.37887,-1.82817,-0.05958,1.43712,-1.88658,-0.060671,1.46621,-1.93534,-0.064591,1.4443,-2.0007,0.12666,1.41056,-2.08352,0.150699,1.06258,-1.76882,0.020925,1.03344,-1.80436,0.04122,0.953781,-1.8271,0.110128,0.874004,-1.79974,-0.039223,0.843804,-1.78239,-0.114428,0.843966,-1.78746,-0.040788,0.951026,-1.81721,-0.111629,0.952463,-1.81469,-0.34317,0.906687,-1.79934,-0.383622,1.01247,-1.87885,-0.41854,0.757299,-1.8473,-0.473489,0.792463,-1.7839,0.181067,1.1272,-2.15429,0.218445,1.05208,-2.13772,0.057544,1.07273,-2.10052,0.095897,1.00056,-2.087,0.198238,0.888576,-2.09612,0.243547,0.706446,-2.0636,0.132698,0.720568,-2.08258,0.125664,0.652656,-2.06742,0.127357,0.430661,-2.01509,0.034158,0.357932,-1.95783,0.315269,0.382108,-1.9991,0.263952,0.34236,-2.06347,0.064145,1.28985,-1.85026,0.05149,1.3135,-2.06382},
/*212*/{0.176099,1.86558,-1.75526,0.233657,1.80412,-1.91464,0.095137,1.72272,-1.64983,0.026268,1.6198,-1.58049,0.148475,1.56443,-1.67103,0.223004,1.5251,-1.70573,0.125437,1.87711,-2.14808,0.228424,1.80693,-1.9929,0.00828,1.84695,-2.04025,-0.092749,1.70808,-2.19288,-0.099559,1.61901,-2.25031,0.06958,1.51574,-2.21874,0.114032,1.46797,-2.23649,0.137098,1.3743,-1.83077,-0.061821,1.43302,-1.88544,-0.063115,1.46138,-1.9356,-0.068458,1.44065,-2.00121,0.124309,1.40813,-2.08496,0.15521,1.06032,-1.76731,0.026918,1.02722,-1.80428,0.049779,0.948155,-1.82642,0.123155,0.870506,-1.79649,-0.02519,0.842319,-1.78134,-0.101002,0.842717,-1.78654,-0.02574,0.949559,-1.81661,-0.098171,0.951344,-1.81397,-0.328943,0.908431,-1.79895,-0.371082,1.01151,-1.87947,-0.402986,0.757149,-1.84493,-0.458258,0.792856,-1.78158,0.176959,1.12533,-2.15551,0.216189,1.05096,-2.13788,0.055137,1.07021,-2.10064,0.094105,0.998868,-2.08625,0.19825,0.88861,-2.09668,0.233506,0.705167,-2.06413,0.122941,0.726082,-2.08115,0.112029,0.65725,-2.06763,0.103505,0.429467,-2.01383,0.01151,0.359249,-1.95683,0.293907,0.38009,-2.00069,0.242232,0.342458,-2.06223,0.062652,1.28529,-1.85246,0.048602,1.3107,-2.06573},
/*213*/{0.17501,1.86097,-1.75379,0.234402,1.80018,-1.91385,0.091145,1.71793,-1.65012,0.018059,1.61727,-1.58189,0.142655,1.55818,-1.67044,0.214838,1.51869,-1.70331,0.1275,1.87297,-2.14936,0.229627,1.80296,-1.99178,0.009047,1.84404,-2.03995,-0.087045,1.69884,-2.19805,-0.086555,1.60805,-2.25357,0.078131,1.50887,-2.22063,0.121861,1.46162,-2.23762,0.135451,1.36947,-1.83132,-0.063955,1.42873,-1.88528,-0.065327,1.4584,-1.93557,-0.070483,1.43745,-1.99959,0.122101,1.40555,-2.08603,0.161124,1.06127,-1.76551,0.03256,1.02464,-1.80456,0.059161,0.944748,-1.82575,0.134696,0.866663,-1.79387,-0.011905,0.839811,-1.78223,-0.087085,0.841194,-1.78581,-0.011797,0.948624,-1.81485,-0.083079,0.949489,-1.81223,-0.313354,0.906598,-1.79828,-0.359115,1.01004,-1.88008,-0.38662,0.755045,-1.84225,-0.441866,0.790456,-1.77921,0.173427,1.12433,-2.15465,0.21262,1.04999,-2.13843,0.05242,1.06714,-2.10003,0.091569,0.996979,-2.08631,0.200463,0.888173,-2.09612,0.22189,0.704681,-2.06687,0.111164,0.728834,-2.08095,0.099176,0.660828,-2.06799,0.087196,0.430844,-2.0132,-0.009717,0.359824,-1.95792,0.272796,0.379489,-1.99946,0.220332,0.341956,-2.06261,0.060706,1.28082,-1.85379,0.045989,1.3081,-2.06678},
/*214*/{0.174461,1.85669,-1.75244,0.235122,1.79595,-1.91244,0.086497,1.71415,-1.64981,0.010663,1.61512,-1.58268,0.135517,1.55396,-1.66918,0.207683,1.51309,-1.70067,0.129761,1.8689,-2.15009,0.230508,1.79806,-1.98994,0.009551,1.84321,-2.04069,-0.080107,1.68968,-2.20326,-0.077763,1.59892,-2.25641,0.086798,1.50324,-2.22155,0.131686,1.4569,-2.23814,0.133795,1.36557,-1.83228,-0.065386,1.42487,-1.88369,-0.066001,1.45731,-1.93444,-0.073077,1.43512,-1.99734,0.12031,1.40432,-2.08783,0.165298,1.06556,-1.76855,0.040808,1.02201,-1.80257,0.069183,0.942652,-1.82292,0.147963,0.865018,-1.79268,0.001871,0.837973,-1.78187,-0.073129,0.838185,-1.78731,0.002455,0.947337,-1.81321,-0.069028,0.948714,-1.81263,-0.304282,0.908953,-1.80099,-0.34856,1.00737,-1.8811,-0.369312,0.752074,-1.83979,-0.425462,0.78649,-1.77889,0.170171,1.1231,-2.1546,0.208488,1.04894,-2.13972,0.049004,1.06354,-2.09889,0.088803,0.994896,-2.08676,0.20247,0.888546,-2.09509,0.209269,0.703365,-2.07176,0.099651,0.72917,-2.08253,0.085675,0.661593,-2.06758,0.064589,0.43251,-2.01333,-0.031735,0.361395,-1.95424,0.251996,0.378109,-2.00031,0.198186,0.341064,-2.06247,0.059401,1.27715,-1.85531,0.043132,1.30689,-2.06786},
/*215*/{0.17434,1.8529,-1.75063,0.235397,1.79328,-1.91117,0.081824,1.71226,-1.65075,0.003604,1.61448,-1.58404,0.128142,1.5511,-1.66841,0.200147,1.50754,-1.69879,0.131525,1.86548,-2.14963,0.230926,1.79553,-1.98954,0.009607,1.84063,-2.04073,-0.075742,1.6823,-2.20904,-0.068469,1.59069,-2.25862,0.096107,1.4991,-2.22221,0.141839,1.45326,-2.23886,0.131699,1.36197,-1.83385,-0.067317,1.42218,-1.88242,-0.067377,1.4563,-1.93377,-0.074275,1.43382,-1.99552,0.118648,1.40293,-2.08943,0.173058,1.06658,-1.76566,0.049656,1.02049,-1.80022,0.07857,0.941956,-1.82115,0.162721,0.866257,-1.79375,0.016249,0.836612,-1.78097,-0.059433,0.838363,-1.78677,0.015642,0.946222,-1.81129,-0.055566,0.946894,-1.8094,-0.287288,0.903335,-1.79889,-0.337887,1.00329,-1.8806,-0.35118,0.74757,-1.83947,-0.407237,0.780471,-1.77606,0.165698,1.12106,-2.15562,0.204389,1.04577,-2.1408,0.044788,1.06222,-2.09864,0.084752,0.990281,-2.08528,0.201953,0.888753,-2.0938,0.196011,0.698613,-2.07352,0.086742,0.728643,-2.08421,0.070937,0.661961,-2.06877,0.044278,0.432434,-2.01425,-0.053236,0.362709,-1.9557,0.229482,0.377217,-1.99985,0.17682,0.341272,-2.06243,0.057367,1.2739,-1.85691,0.040142,1.30594,-2.06905},
/*216*/{0.173541,1.8492,-1.74951,0.234834,1.79096,-1.90997,0.076695,1.71164,-1.65107,-0.005079,1.61515,-1.58647,0.120526,1.54871,-1.66703,0.193149,1.50376,-1.69614,0.13353,1.86321,-2.14929,0.231153,1.79253,-1.98797,0.01042,1.83957,-2.04134,-0.069079,1.67683,-2.21393,-0.059624,1.58374,-2.26089,0.105931,1.49599,-2.22258,0.152992,1.45113,-2.23889,0.130538,1.35877,-1.83357,-0.068613,1.42098,-1.88101,-0.069148,1.45568,-1.93411,-0.075391,1.43249,-1.99382,0.117105,1.40183,-2.09111,0.180061,1.06922,-1.76593,0.055242,1.01863,-1.79996,0.090063,0.941821,-1.81862,0.176725,0.86752,-1.79363,0.029375,0.835906,-1.78114,-0.045137,0.835486,-1.78776,0.028894,0.945847,-1.80908,-0.042974,0.946456,-1.80875,-0.273314,0.899913,-1.79941,-0.329493,0.997993,-1.87969,-0.332711,0.742213,-1.83914,-0.388421,0.773424,-1.77408,0.160562,1.11788,-2.15629,0.198281,1.0424,-2.14171,0.040363,1.06134,-2.09812,0.079326,0.98844,-2.08549,0.194531,0.886684,-2.09508,0.182802,0.696123,-2.07572,0.073648,0.726939,-2.08515,0.057617,0.66083,-2.0691,0.023619,0.432756,-2.01514,-0.07438,0.362935,-1.95603,0.208763,0.376595,-2.00091,0.153992,0.340789,-2.06319,0.055533,1.27166,-1.85759,0.037633,1.30492,-2.06948},
/*217*/{0.1726,1.84627,-1.74903,0.233637,1.78783,-1.90881,0.071252,1.71188,-1.65136,-0.013871,1.61657,-1.58898,0.113494,1.54727,-1.6667,0.186519,1.50095,-1.6932,0.135572,1.86107,-2.14856,0.231555,1.79033,-1.9871,0.011611,1.83744,-2.04141,-0.062943,1.67071,-2.21655,-0.049993,1.57834,-2.26259,0.116281,1.49358,-2.22225,0.164142,1.45009,-2.23841,0.127954,1.35565,-1.83368,-0.070653,1.42067,-1.88031,-0.06952,1.45445,-1.93376,-0.076043,1.43264,-1.99212,0.115063,1.40161,-2.09247,0.186326,1.07375,-1.7668,0.064521,1.01877,-1.79726,0.10113,0.943024,-1.817,0.191355,0.870302,-1.79343,0.042255,0.835846,-1.78081,-0.032799,0.835886,-1.78809,0.04103,0.945907,-1.80691,-0.030993,0.945786,-1.80748,-0.258909,0.893384,-1.79995,-0.321575,0.991167,-1.87897,-0.313607,0.735695,-1.83831,-0.368736,0.764233,-1.77246,0.155512,1.11428,-2.15732,0.191945,1.03837,-2.143,0.033359,1.05971,-2.09829,0.072786,0.987123,-2.0856,0.184645,0.881352,-2.09622,0.169615,0.690321,-2.07599,0.061021,0.725162,-2.08714,0.043291,0.659629,-2.07025,0.00253,0.433264,-2.0162,-0.0971,0.362994,-1.95609,0.186267,0.37678,-2.00101,0.132188,0.340391,-2.06369,0.052834,1.26938,-1.85886,0.034827,1.30492,-2.07038},
/*218*/{0.172977,1.84463,-1.74822,0.232717,1.78545,-1.9069,0.066434,1.7123,-1.65201,-0.020257,1.61964,-1.59113,0.105952,1.54705,-1.66604,0.178767,1.49861,-1.69146,0.138023,1.8595,-2.14763,0.231374,1.78842,-1.986,0.012249,1.83622,-2.04225,-0.056852,1.66583,-2.21721,-0.041402,1.57422,-2.2636,0.126878,1.49228,-2.22109,0.175283,1.45001,-2.2377,0.127012,1.35311,-1.83298,-0.070772,1.42069,-1.87967,-0.071208,1.4538,-1.93372,-0.076898,1.43241,-1.99132,0.114596,1.4006,-2.09302,0.194098,1.07943,-1.76655,0.074364,1.01948,-1.79439,0.111631,0.943981,-1.81515,0.205562,0.874462,-1.79246,0.055958,0.836637,-1.78073,-0.01898,0.834443,-1.78824,0.052402,0.946111,-1.80482,-0.019887,0.945485,-1.80584,-0.245073,0.888483,-1.80078,-0.31327,0.983701,-1.87843,-0.292823,0.728641,-1.83819,-0.347883,0.754443,-1.77125,0.149704,1.11147,-2.1582,0.184694,1.03451,-2.14398,0.027045,1.06056,-2.09971,0.065109,0.985077,-2.08608,0.173542,0.876831,-2.09801,0.155485,0.685979,-2.07628,0.048967,0.723419,-2.08867,0.027658,0.656073,-2.0705,-0.018169,0.434001,-2.01836,-0.118263,0.363914,-1.95712,0.164926,0.376557,-2.00252,0.110779,0.34008,-2.06476,0.05108,1.26789,-1.85903,0.033211,1.30424,-2.07042},
/*219*/{0.171982,1.84336,-1.74754,0.232677,1.78428,-1.90648,0.061039,1.71383,-1.65237,-0.026761,1.62296,-1.59325,0.099594,1.54733,-1.66471,0.170221,1.4984,-1.68906,0.140375,1.85828,-2.14692,0.231897,1.78738,-1.98444,0.012953,1.83608,-2.04306,-0.050645,1.66219,-2.21766,-0.032905,1.57073,-2.26481,0.137463,1.49204,-2.22053,0.187112,1.45116,-2.23605,0.124744,1.35176,-1.83242,-0.070859,1.42064,-1.87962,-0.07137,1.45301,-1.93347,-0.076989,1.43245,-1.99118,0.114339,1.39969,-2.0938,0.201726,1.08418,-1.76633,0.083381,1.0209,-1.79284,0.122923,0.946417,-1.81355,0.219354,0.878373,-1.79339,0.069689,0.837174,-1.78109,-0.004171,0.832847,-1.79037,0.063624,0.947276,-1.80278,-0.008007,0.943037,-1.80288,-0.231767,0.88192,-1.80177,-0.306273,0.974779,-1.8772,-0.271325,0.720673,-1.83885,-0.326141,0.743967,-1.76942,0.14502,1.10767,-2.15937,0.178752,1.03043,-2.14438,0.020554,1.0605,-2.1004,0.055498,0.982859,-2.08678,0.161125,0.873222,-2.09973,0.140943,0.681258,-2.0763,0.035728,0.72075,-2.08886,0.012773,0.654892,-2.07144,-0.038255,0.43468,-2.01984,-0.140511,0.364953,-1.96039,0.141939,0.373859,-2.00375,0.088394,0.33907,-2.06543,0.0494,1.26683,-1.85915,0.032183,1.30367,-2.07051},
/*220*/{0.170745,1.84279,-1.74713,0.231015,1.78271,-1.90457,0.055154,1.71591,-1.6528,-0.034897,1.6275,-1.59579,0.092096,1.54843,-1.66372,0.163286,1.49852,-1.68681,0.142527,1.85803,-2.14667,0.231643,1.78635,-1.98382,0.013559,1.83563,-2.04324,-0.043172,1.66013,-2.21789,-0.024635,1.56792,-2.2658,0.148629,1.49391,-2.21916,0.198114,1.45419,-2.23425,0.124165,1.35093,-1.83066,-0.071257,1.42067,-1.88002,-0.072795,1.4534,-1.9343,-0.077233,1.43293,-1.99138,0.115034,1.3994,-2.09374,0.209274,1.0897,-1.76591,0.092954,1.02297,-1.79153,0.13469,0.949656,-1.81192,0.232833,0.884024,-1.79318,0.083247,0.836983,-1.78154,0.009169,0.832296,-1.79025,0.074709,0.948651,-1.80121,0.003654,0.942583,-1.80126,-0.218062,0.874955,-1.80188,-0.299069,0.964202,-1.87625,-0.250964,0.712026,-1.8399,-0.303947,0.732567,-1.76838,0.13848,1.10444,-2.15908,0.169853,1.02586,-2.14533,0.013495,1.05841,-2.10036,0.046415,0.982208,-2.08829,0.148275,0.869072,-2.10036,0.125236,0.676857,-2.07601,0.021329,0.719704,-2.09067,-0.002013,0.654595,-2.0728,-0.057983,0.43541,-2.0204,-0.161404,0.365679,-1.96106,0.121526,0.375823,-2.00449,0.068866,0.339164,-2.06603,0.048092,1.26662,-1.85886,0.032265,1.30361,-2.0703},
/*221*/{0.169016,1.84339,-1.74634,0.231595,1.78261,-1.90387,0.05058,1.71845,-1.65342,-0.042823,1.63145,-1.5976,0.085539,1.55057,-1.66338,0.15666,1.49927,-1.68496,0.145683,1.85746,-2.14528,0.232126,1.78612,-1.98152,0.014604,1.83582,-2.04389,-0.037781,1.65816,-2.21818,-0.016347,1.56595,-2.26674,0.157786,1.49506,-2.21688,0.209105,1.4582,-2.23181,0.121776,1.35106,-1.82869,-0.072184,1.42037,-1.88047,-0.072902,1.4535,-1.93444,-0.077563,1.4333,-1.99248,0.113183,1.39868,-2.09446,0.216158,1.0967,-1.76628,0.101294,1.02556,-1.79036,0.146425,0.954111,-1.81044,0.246151,0.889812,-1.79314,0.097124,0.838373,-1.78277,0.022249,0.830113,-1.79098,0.084227,0.949647,-1.79988,0.011797,0.941661,-1.80066,-0.204977,0.867246,-1.80439,-0.292028,0.952475,-1.87594,-0.22879,0.704093,-1.83934,-0.281211,0.721937,-1.76747,0.13253,1.10214,-2.15892,0.161219,1.02197,-2.14555,0.00557,1.05829,-2.1018,0.036612,0.981912,-2.08959,0.134802,0.864692,-2.10073,0.109758,0.671632,-2.07605,0.006846,0.716465,-2.09034,-0.018696,0.653532,-2.07382,-0.081086,0.435315,-2.02113,-0.183007,0.368205,-1.95968,0.099514,0.375168,-2.00578,0.045347,0.338422,-2.06698,0.04613,1.26668,-1.8583,0.031127,1.30298,-2.06992},
/*222*/{0.166856,1.84394,-1.74498,0.229972,1.78386,-1.90126,0.044042,1.72233,-1.65411,-0.049188,1.6358,-1.59989,0.078354,1.5529,-1.66173,0.148662,1.5014,-1.6828,0.147922,1.85781,-2.14486,0.232553,1.78673,-1.9803,0.015779,1.83589,-2.04498,-0.03288,1.65652,-2.21798,-0.00769,1.56432,-2.26709,0.168354,1.49971,-2.21514,0.220603,1.46338,-2.22843,0.121031,1.35167,-1.82752,-0.072504,1.4208,-1.88145,-0.072704,1.45341,-1.93474,-0.077018,1.4335,-1.9934,0.114044,1.39864,-2.09407,0.221803,1.10349,-1.76552,0.109389,1.02846,-1.78933,0.156533,0.958297,-1.8091,0.258947,0.895612,-1.79329,0.110313,0.840564,-1.78339,0.037508,0.828999,-1.79165,0.093547,0.950779,-1.79808,0.023035,0.939883,-1.79889,-0.191905,0.859821,-1.80529,-0.284002,0.939477,-1.87602,-0.207157,0.694775,-1.83972,-0.258817,0.710171,-1.76676,0.125902,1.09957,-2.1587,0.153642,1.01872,-2.14535,-0.001955,1.05887,-2.10279,0.026742,0.981644,-2.09113,0.120845,0.859639,-2.10087,0.093421,0.668041,-2.07585,-0.008773,0.714235,-2.09079,-0.035224,0.65124,-2.07404,-0.102198,0.437317,-2.02119,-0.203112,0.368661,-1.95938,0.078296,0.374876,-2.00558,0.024118,0.339603,-2.06787,0.045713,1.26713,-1.85778,0.032109,1.30291,-2.06959},
/*223*/{0.163955,1.84658,-1.74453,0.230868,1.78424,-1.9008,0.038906,1.72615,-1.65439,-0.056348,1.64068,-1.60165,0.07196,1.55666,-1.66113,0.141467,1.5046,-1.68113,0.150073,1.85913,-2.14442,0.232742,1.78749,-1.97864,0.016797,1.83682,-2.04636,-0.026813,1.65488,-2.21775,0.000621,1.56324,-2.2679,0.177281,1.50352,-2.21273,0.230832,1.46937,-2.2249,0.119774,1.35316,-1.82629,-0.073457,1.42131,-1.88243,-0.072955,1.45427,-1.93601,-0.077262,1.43414,-1.99418,0.114432,1.39838,-2.09371,0.228253,1.10994,-1.7661,0.120697,1.03358,-1.78757,0.166516,0.964428,-1.80863,0.270611,0.902828,-1.79391,0.124458,0.842587,-1.784,0.051016,0.828969,-1.79281,0.10291,0.952277,-1.79638,0.03139,0.938601,-1.79758,-0.178377,0.849979,-1.80571,-0.276021,0.925726,-1.8765,-0.185073,0.685383,-1.83993,-0.235952,0.698252,-1.76644,0.12002,1.09705,-2.15825,0.143904,1.01491,-2.14484,-0.010364,1.05985,-2.10375,0.016803,0.982068,-2.09147,0.105051,0.854455,-2.10047,0.076798,0.665677,-2.07379,-0.025292,0.713101,-2.09037,-0.051253,0.648983,-2.07451,-0.121496,0.438705,-2.02198,-0.225601,0.368265,-1.96155,0.056463,0.374868,-2.0058,0.002175,0.339714,-2.06801,0.044663,1.26832,-1.85692,0.03251,1.30283,-2.06902},
/*224*/{0.161306,1.84882,-1.74353,0.229428,1.78601,-1.89855,0.033642,1.73034,-1.65537,-0.062909,1.64501,-1.60309,0.065314,1.56095,-1.66117,0.134634,1.50802,-1.67984,0.152904,1.86093,-2.143,0.232549,1.78883,-1.97662,0.017652,1.83696,-2.04668,-0.021086,1.65387,-2.2171,0.008791,1.56335,-2.26843,0.186778,1.50911,-2.2101,0.240798,1.47692,-2.2212,0.118613,1.35507,-1.82566,-0.073316,1.4224,-1.88336,-0.072573,1.45493,-1.93663,-0.076668,1.4349,-1.99587,0.114432,1.39819,-2.09376,0.235203,1.11781,-1.76461,0.129454,1.03854,-1.78678,0.175711,0.969134,-1.80833,0.282121,0.910587,-1.79356,0.137153,0.844219,-1.7849,0.063296,0.827979,-1.79301,0.111246,0.953353,-1.79471,0.040536,0.936556,-1.79688,-0.164501,0.839085,-1.80564,-0.267326,0.909476,-1.87678,-0.162536,0.676671,-1.84006,-0.213847,0.686218,-1.76678,0.113056,1.09485,-2.15748,0.134562,1.01254,-2.14436,-0.01826,1.06128,-2.10397,0.006173,0.982329,-2.09161,0.090047,0.850537,-2.1004,0.060598,0.664149,-2.07255,-0.040595,0.713031,-2.08987,-0.067982,0.649998,-2.07401,-0.144109,0.439481,-2.02177,-0.246745,0.370375,-1.96308,0.035018,0.374936,-2.00671,-0.019026,0.339411,-2.06823,0.044163,1.26977,-1.85625,0.032944,1.30279,-2.06864},
/*225*/{0.15771,1.85119,-1.74316,0.228989,1.78822,-1.89637,0.02764,1.73476,-1.65526,-0.070624,1.65002,-1.60497,0.0586,1.56557,-1.66155,0.127583,1.51273,-1.67928,0.154573,1.86312,-2.14188,0.23365,1.79109,-1.975,0.018133,1.83724,-2.04726,-0.016241,1.65286,-2.21669,0.016351,1.56379,-2.26909,0.195158,1.51499,-2.20681,0.250753,1.48471,-2.21655,0.119704,1.35714,-1.82493,-0.073457,1.42238,-1.8852,-0.072469,1.45499,-1.93709,-0.076271,1.43562,-1.99752,0.116202,1.39732,-2.09322,0.241024,1.12491,-1.76478,0.136475,1.04226,-1.78665,0.182983,0.973746,-1.80852,0.293839,0.919738,-1.79425,0.15045,0.845649,-1.78598,0.077569,0.826659,-1.794,0.118759,0.954614,-1.79335,0.049309,0.93457,-1.79627,-0.15325,0.829942,-1.80874,-0.257982,0.894123,-1.87856,-0.139776,0.666856,-1.83972,-0.191348,0.674299,-1.76629,0.106853,1.09328,-2.15665,0.125637,1.01002,-2.14345,-0.025879,1.06272,-2.104,-0.003748,0.983423,-2.0918,0.074502,0.84878,-2.0993,0.042332,0.663589,-2.07203,-0.057021,0.713943,-2.08906,-0.085296,0.650525,-2.07356,-0.193864,0.432279,-2.02065,-0.266991,0.371958,-1.96386,0.013999,0.374824,-2.00642,-0.040668,0.339408,-2.06909,0.044923,1.27111,-1.85536,0.034797,1.30232,-2.06808},
/*226*/{0.155229,1.85453,-1.74221,0.229194,1.79073,-1.89508,0.023413,1.73914,-1.65644,-0.077233,1.65501,-1.60687,0.05217,1.5706,-1.66159,0.121565,1.51727,-1.67817,0.156357,1.86554,-2.14064,0.233127,1.79286,-1.97288,0.019631,1.83834,-2.04832,-0.011611,1.65276,-2.21663,0.023746,1.56516,-2.27007,0.202401,1.52101,-2.20319,0.260482,1.49327,-2.21182,0.119533,1.35989,-1.82471,-0.072927,1.42385,-1.88684,-0.072155,1.45632,-1.93903,-0.074775,1.43657,-1.99799,0.117059,1.39695,-2.09355,0.245977,1.1328,-1.76481,0.142502,1.04662,-1.78645,0.192417,0.981474,-1.80875,0.303295,0.927458,-1.79471,0.162978,0.849066,-1.78586,0.090824,0.825896,-1.79444,0.126854,0.954598,-1.79304,0.057307,0.932029,-1.79544,-0.13705,0.817259,-1.8075,-0.248081,0.877534,-1.87967,-0.117329,0.656768,-1.83957,-0.169424,0.662412,-1.76633,0.100131,1.09226,-2.15537,0.117395,1.0082,-2.14128,-0.033474,1.0656,-2.10441,-0.015096,0.98516,-2.09235,0.058274,0.849211,-2.09859,0.025008,0.663482,-2.07125,-0.074102,0.715814,-2.08785,-0.102997,0.652391,-2.0729,-0.244482,0.427486,-2.01896,-0.287783,0.376903,-1.96264,-0.008222,0.374381,-2.00669,-0.063554,0.339322,-2.06839,0.045164,1.2733,-1.85455,0.035869,1.3023,-2.06762},
/*227*/{0.151202,1.85724,-1.74174,0.22878,1.79344,-1.89262,0.017776,1.74371,-1.65723,-0.08363,1.66066,-1.60955,0.045996,1.57643,-1.66209,0.11547,1.52259,-1.6775,0.157932,1.86877,-2.13949,0.233748,1.79627,-1.9708,0.020359,1.84042,-2.04982,-0.005506,1.65313,-2.21649,0.031844,1.56644,-2.27034,0.211264,1.52927,-2.20016,0.268936,1.50269,-2.20731,0.119135,1.3629,-1.82397,-0.072364,1.42583,-1.88799,-0.070636,1.45814,-1.94038,-0.074519,1.43811,-1.9999,0.117877,1.39698,-2.09349,0.249163,1.13987,-1.76513,0.148205,1.05119,-1.7878,0.200413,0.987405,-1.80903,0.313191,0.93681,-1.79498,0.175229,0.850892,-1.78649,0.104822,0.824603,-1.79538,0.134263,0.954884,-1.79196,0.065337,0.929558,-1.7953,-0.123765,0.806191,-1.80909,-0.236989,0.859464,-1.88156,-0.09478,0.647644,-1.83903,-0.14805,0.649863,-1.76602,0.093242,1.09095,-2.15367,0.107969,1.00662,-2.13913,-0.041919,1.06881,-2.10445,-0.024681,0.988385,-2.09312,0.043075,0.850498,-2.09724,0.007137,0.664538,-2.0707,-0.09171,0.717807,-2.08712,-0.121111,0.655679,-2.07212,-0.264962,0.430419,-2.01894,-0.310036,0.378881,-1.96427,-0.029131,0.375078,-2.00668,-0.084635,0.338914,-2.06817,0.045048,1.27592,-1.85359,0.036806,1.30268,-2.06699},
/*228*/{0.148174,1.86096,-1.74115,0.228517,1.79785,-1.89053,0.014181,1.74857,-1.6577,-0.090235,1.66554,-1.61153,0.040646,1.58258,-1.66233,0.109303,1.52813,-1.67683,0.160317,1.87252,-2.13831,0.234124,1.79944,-1.96913,0.021156,1.84135,-2.04958,-0.000767,1.6541,-2.21654,0.039451,1.56888,-2.27087,0.217682,1.53615,-2.19675,0.277818,1.5129,-2.20222,0.120697,1.36687,-1.82429,-0.071876,1.42835,-1.88963,-0.069345,1.46003,-1.94095,-0.072683,1.44043,-2.00112,0.118848,1.39716,-2.09362,0.253398,1.14772,-1.76498,0.155352,1.05664,-1.78722,0.209293,0.994335,-1.80945,0.321984,0.94658,-1.79466,0.187266,0.852937,-1.78683,0.117686,0.823185,-1.79598,0.140162,0.954749,-1.79185,0.075209,0.925634,-1.79468,-0.11036,0.794924,-1.81093,-0.224928,0.84144,-1.88332,-0.071364,0.638112,-1.83894,-0.125914,0.637266,-1.76676,0.086144,1.08975,-2.15227,0.098503,1.0056,-2.1367,-0.049985,1.07138,-2.10464,-0.034435,0.989686,-2.09303,0.027571,0.851888,-2.09572,-0.010166,0.666537,-2.06989,-0.109353,0.720746,-2.08598,-0.139198,0.657709,-2.07117,-0.226067,0.449221,-2.02253,-0.330084,0.383249,-1.96352,-0.05012,0.375069,-2.0066,-0.105696,0.339741,-2.06801,0.046264,1.27923,-1.85271,0.038125,1.30349,-2.06642},
/*229*/{0.144699,1.86459,-1.74101,0.227666,1.79968,-1.88862,0.008788,1.75346,-1.65834,-0.096628,1.67139,-1.61348,0.035178,1.58845,-1.66347,0.103638,1.53417,-1.67603,0.160927,1.8765,-2.13695,0.235039,1.80329,-1.96758,0.022297,1.84369,-2.05023,0.005549,1.65479,-2.21642,0.047249,1.57149,-2.27129,0.225232,1.54565,-2.19298,0.284847,1.5228,-2.19706,0.121189,1.3702,-1.82457,-0.070931,1.43066,-1.8911,-0.068309,1.46257,-1.94255,-0.071299,1.44204,-2.00243,0.119806,1.39702,-2.09376,0.255482,1.15542,-1.76501,0.159698,1.06135,-1.78909,0.215664,1.00032,-1.81031,0.330246,0.956527,-1.79531,0.197828,0.854754,-1.78655,0.130495,0.822029,-1.7956,0.146404,0.954552,-1.79146,0.083088,0.922373,-1.79542,-0.095119,0.781606,-1.81116,-0.212249,0.823206,-1.88535,-0.047891,0.628478,-1.8383,-0.103553,0.625203,-1.76604,0.080466,1.08938,-2.14975,0.089432,1.00487,-2.13398,-0.056435,1.07548,-2.1048,-0.044544,0.992003,-2.09316,0.012702,0.853857,-2.09402,-0.027498,0.668518,-2.06824,-0.125983,0.72398,-2.08449,-0.157657,0.663723,-2.06921,-0.24994,0.451527,-2.02078,-0.350193,0.38877,-1.96574,-0.071805,0.374949,-2.00609,-0.127917,0.339835,-2.06791,0.046667,1.28221,-1.85194,0.039064,1.30397,-2.06594},
/*230*/{0.141757,1.86852,-1.74034,0.228357,1.80542,-1.88727,0.004734,1.75819,-1.65888,-0.101186,1.67716,-1.61585,0.029683,1.595,-1.6644,0.097851,1.53992,-1.67583,0.161734,1.88071,-2.1358,0.236909,1.80656,-1.9645,0.0237,1.84549,-2.05042,0.0113,1.65624,-2.21691,0.054439,1.57507,-2.27203,0.231766,1.55396,-2.18896,0.291979,1.53378,-2.19262,0.12305,1.37396,-1.82444,-0.06913,1.43343,-1.89168,-0.065996,1.46406,-1.94337,-0.069186,1.44479,-2.0037,0.121279,1.39736,-2.09383,0.260623,1.16194,-1.76475,0.164949,1.06681,-1.78872,0.221816,1.00784,-1.81092,0.338355,0.9664,-1.79494,0.20913,0.857299,-1.78631,0.143732,0.821138,-1.7956,0.153388,0.953867,-1.79147,0.089936,0.918125,-1.79649,-0.080144,0.770217,-1.81068,-0.199265,0.804158,-1.88737,-0.024427,0.619642,-1.83718,-0.080803,0.613262,-1.76661,0.074265,1.08927,-2.14702,0.080586,1.00498,-2.13114,-0.064792,1.07913,-2.10533,-0.055366,0.997709,-2.09226,-0.001185,0.855627,-2.09149,-0.044925,0.671676,-2.06686,-0.14296,0.727723,-2.08344,-0.175439,0.666899,-2.06906,-0.266718,0.457358,-2.02174,-0.37014,0.39503,-1.96458,-0.09322,0.375111,-2.00573,-0.148188,0.340037,-2.06798,0.048396,1.28548,-1.85095,0.040851,1.3049,-2.06517},
/*231*/{0.138463,1.87289,-1.73982,0.227447,1.80853,-1.88453,0.000919,1.76335,-1.6603,-0.106837,1.68323,-1.61773,0.024944,1.60109,-1.66531,0.092951,1.54655,-1.67612,0.163921,1.88527,-2.13476,0.238023,1.81081,-1.96248,0.02434,1.84847,-2.05109,0.015807,1.65879,-2.21824,0.061907,1.57835,-2.27196,0.237689,1.56287,-2.18444,0.298473,1.5445,-2.18672,0.123902,1.3779,-1.8248,-0.068492,1.43708,-1.89276,-0.064591,1.46749,-1.94552,-0.067184,1.44768,-2.00493,0.123383,1.39774,-2.09415,0.262854,1.16859,-1.76483,0.168646,1.07229,-1.79033,0.227924,1.01298,-1.81207,0.34447,0.976194,-1.79525,0.220609,0.860429,-1.78556,0.155818,0.819401,-1.79581,0.158336,0.953882,-1.79164,0.098813,0.914187,-1.79827,-0.066121,0.759063,-1.81125,-0.185891,0.785182,-1.89077,-0.000884,0.611014,-1.83711,-0.056839,0.602409,-1.76696,0.068083,1.08992,-2.1448,0.072208,1.00512,-2.12825,-0.071111,1.08357,-2.1046,-0.064758,1.00151,-2.09142,-0.014165,0.857601,-2.08825,-0.062064,0.674814,-2.06586,-0.159071,0.732194,-2.08217,-0.193262,0.671449,-2.0674,-0.284813,0.461989,-2.02174,-0.389226,0.401881,-1.96505,-0.115126,0.375241,-2.00582,-0.170488,0.34018,-2.0673,0.048759,1.28936,-1.8502,0.042113,1.30626,-2.06467},
/*232*/{0.135515,1.87676,-1.73962,0.22631,1.81372,-1.88334,-0.001953,1.76868,-1.66083,-0.111784,1.68986,-1.62053,0.019192,1.60784,-1.66609,0.088169,1.55256,-1.67539,0.165886,1.88994,-2.13304,0.237693,1.81508,-1.96104,0.025292,1.85165,-2.05109,0.020735,1.66216,-2.22012,0.069057,1.58222,-2.27238,0.242256,1.572,-2.1809,0.304094,1.55614,-2.18153,0.125969,1.38068,-1.82565,-0.066544,1.44066,-1.89432,-0.062709,1.4699,-1.94706,-0.064961,1.45044,-2.00582,0.125343,1.39729,-2.09411,0.267261,1.17604,-1.76454,0.173699,1.07548,-1.78968,0.233618,1.02056,-1.81199,0.350572,0.983829,-1.79606,0.229859,0.861409,-1.78493,0.167394,0.818173,-1.79536,0.164303,0.951652,-1.79126,0.105396,0.909859,-1.79933,-0.050668,0.746428,-1.81332,-0.171047,0.765582,-1.89243,0.02355,0.602609,-1.83712,-0.032888,0.590302,-1.7672,0.062724,1.09041,-2.14147,0.064531,1.00601,-2.12478,-0.077012,1.08863,-2.10457,-0.074044,1.00684,-2.09112,-0.027564,0.860934,-2.08604,-0.079298,0.678899,-2.0648,-0.17561,0.736909,-2.0812,-0.209287,0.67763,-2.06721,-0.304007,0.46624,-2.02207,-0.408037,0.409723,-1.96559,-0.136297,0.375753,-2.00532,-0.192868,0.341349,-2.06795,0.049725,1.29244,-1.84962,0.043276,1.30698,-2.06426},
/*233*/{0.133213,1.88069,-1.73941,0.22694,1.81931,-1.88145,-0.005614,1.7737,-1.66143,-0.116796,1.69603,-1.62283,0.01521,1.61395,-1.66719,0.083198,1.55857,-1.67602,0.167144,1.89544,-2.13162,0.237847,1.81967,-1.95857,0.026927,1.85401,-2.05204,0.025431,1.66716,-2.22353,0.076149,1.5869,-2.27294,0.247453,1.58029,-2.17578,0.308933,1.56687,-2.1769,0.127585,1.3849,-1.82614,-0.064558,1.4452,-1.89467,-0.060428,1.47328,-1.94714,-0.061816,1.45303,-2.00658,0.127166,1.39741,-2.09439,0.269163,1.18308,-1.76443,0.179401,1.0835,-1.79003,0.237781,1.02616,-1.81282,0.355564,0.993362,-1.79532,0.239852,0.862828,-1.78526,0.179863,0.816867,-1.79513,0.169231,0.950997,-1.79596,0.113334,0.905161,-1.80024,-0.035197,0.734956,-1.81504,-0.155412,0.746112,-1.8936,0.047768,0.594201,-1.83661,-0.007934,0.57945,-1.76732,0.057483,1.09165,-2.13885,0.05595,1.00733,-2.12128,-0.082887,1.09423,-2.10428,-0.082701,1.01246,-2.09077,-0.040323,0.865017,-2.08378,-0.095357,0.684045,-2.06368,-0.191004,0.742237,-2.08014,-0.225349,0.682555,-2.06644,-0.31778,0.473611,-2.02282,-0.427424,0.418482,-1.96528,-0.158514,0.375561,-2.00551,-0.213972,0.340663,-2.06707,0.051149,1.29662,-1.84817,0.044451,1.30819,-2.06299},
/*234*/{0.130985,1.88486,-1.73919,0.22618,1.82319,-1.88018,-0.009852,1.77913,-1.66297,-0.12109,1.70198,-1.6249,0.011125,1.61976,-1.66822,0.079198,1.56463,-1.67598,0.168724,1.90023,-2.1303,0.23799,1.82464,-1.95726,0.02831,1.85751,-2.05307,0.029766,1.67142,-2.22584,0.08334,1.59173,-2.27359,0.252339,1.58966,-2.17177,0.313214,1.57756,-2.17191,0.130001,1.3883,-1.82664,-0.062761,1.44937,-1.89441,-0.058354,1.47677,-1.94971,-0.058517,1.4559,-2.00726,0.129074,1.39742,-2.09429,0.271443,1.18966,-1.76451,0.182353,1.08989,-1.78989,0.242002,1.03296,-1.81303,0.359115,1.00103,-1.79562,0.249332,0.864304,-1.78499,0.191875,0.816047,-1.79525,0.173899,0.949148,-1.7957,0.120872,0.900392,-1.80166,-0.019012,0.721579,-1.81538,-0.1386,0.725366,-1.89496,0.072782,0.587404,-1.83694,0.016729,0.56922,-1.76699,0.052875,1.09295,-2.13608,0.048195,1.00869,-2.1179,-0.088588,1.09996,-2.10396,-0.090985,1.01774,-2.09008,-0.054448,0.87016,-2.08223,-0.110726,0.689396,-2.06343,-0.206124,0.748957,-2.07946,-0.241195,0.690474,-2.06634,-0.338957,0.478005,-2.02268,-0.445058,0.426418,-1.96354,-0.178734,0.375941,-2.00479,-0.23629,0.341515,-2.06583,0.052388,1.3005,-1.84712,0.045762,1.30933,-2.06207},
/*235*/{0.129085,1.88847,-1.73934,0.22586,1.82898,-1.87847,-0.012785,1.78455,-1.66392,-0.125472,1.70909,-1.62778,0.006839,1.62569,-1.66952,0.075957,1.57055,-1.67591,0.171426,1.90542,-2.12871,0.238201,1.82895,-1.95513,0.029261,1.86072,-2.0537,0.034416,1.67806,-2.22885,0.09097,1.59581,-2.27345,0.255454,1.5976,-2.16756,0.318088,1.5874,-2.16725,0.131493,1.39163,-1.82728,-0.059412,1.45337,-1.8947,-0.055535,1.48025,-1.95051,-0.055848,1.45983,-2.00775,0.131588,1.39739,-2.09422,0.273488,1.19675,-1.76425,0.184278,1.0932,-1.79053,0.246076,1.0398,-1.8126,0.362419,1.00886,-1.7955,0.258253,0.86599,-1.78456,0.204352,0.814064,-1.79517,0.179746,0.946132,-1.79821,0.128491,0.895652,-1.80368,-0.000798,0.71071,-1.81593,-0.122157,0.70545,-1.89578,0.09746,0.579851,-1.83659,0.042988,0.558908,-1.76694,0.047207,1.09425,-2.13343,0.040276,1.01029,-2.11461,-0.094089,1.10584,-2.10326,-0.098955,1.02374,-2.089,-0.068403,0.875871,-2.08145,-0.127198,0.694835,-2.06263,-0.22099,0.756153,-2.07882,-0.258116,0.694567,-2.06483,-0.361004,0.483766,-2.02262,-0.461924,0.435968,-1.96336,-0.201285,0.375775,-2.00287,-0.254188,0.341533,-2.06436,0.053503,1.3041,-1.84626,0.046913,1.3109,-2.06129},
/*236*/{0.126923,1.89273,-1.73937,0.225744,1.83347,-1.8769,-0.015903,1.78948,-1.66569,-0.128394,1.7149,-1.62993,0.0038,1.63138,-1.66973,0.07221,1.57639,-1.67657,0.172294,1.91118,-2.12718,0.238809,1.83414,-1.95353,0.0306,1.86469,-2.05453,0.039469,1.68236,-2.23073,0.09141,1.60104,-2.27567,0.260159,1.60754,-2.16351,0.321909,1.59758,-2.16188,0.133596,1.39538,-1.82853,-0.05731,1.45826,-1.89453,-0.053751,1.48402,-1.95293,-0.052404,1.46248,-2.00855,0.133601,1.39728,-2.0939,0.275341,1.20286,-1.76445,0.187345,1.09929,-1.79089,0.250251,1.04599,-1.81258,0.364032,1.01534,-1.79593,0.267064,0.866551,-1.78476,0.214778,0.811788,-1.79464,0.184305,0.942908,-1.79996,0.136813,0.890348,-1.80497,0.015447,0.69555,-1.81684,-0.10291,0.685167,-1.89776,0.122148,0.573677,-1.8375,0.069658,0.549257,-1.76736,0.041996,1.09588,-2.13045,0.032442,1.01275,-2.1115,-0.099299,1.11174,-2.10295,-0.106565,1.03024,-2.08853,-0.082209,0.88171,-2.08095,-0.142264,0.701245,-2.06248,-0.235333,0.763166,-2.0779,-0.272833,0.703139,-2.06586,-0.377259,0.489633,-2.02315,-0.479557,0.448978,-1.96381,-0.224228,0.376279,-2.00301,-0.27955,0.343111,-2.06521,0.054716,1.30843,-1.84511,0.048034,1.31207,-2.06021},
/*237*/{0.125661,1.89602,-1.73911,0.226048,1.83664,-1.87542,-0.017656,1.7946,-1.66661,-0.128579,1.72169,-1.6326,0.000844,1.63686,-1.67123,0.069437,1.58192,-1.67666,0.174095,1.91558,-2.12516,0.239511,1.83927,-1.95116,0.031779,1.86778,-2.0547,0.045479,1.68762,-2.23336,0.103537,1.60643,-2.27442,0.262775,1.61582,-2.15955,0.324488,1.6076,-2.15686,0.135677,1.40002,-1.8286,-0.054681,1.46361,-1.89528,-0.050407,1.48738,-1.95401,-0.049681,1.46686,-2.00896,0.135697,1.39754,-2.09375,0.27753,1.20875,-1.76481,0.189712,1.1062,-1.78964,0.251775,1.05157,-1.81173,0.365267,1.02069,-1.79652,0.275063,0.86708,-1.78402,0.226621,0.809839,-1.79445,0.189742,0.939114,-1.80092,0.143561,0.883391,-1.8061,0.034038,0.682906,-1.81757,-0.084449,0.665146,-1.89774,0.147893,0.567721,-1.83753,0.096355,0.540215,-1.76833,0.03789,1.09754,-2.12792,0.025672,1.01491,-2.10853,-0.103213,1.11937,-2.10218,-0.113783,1.03773,-2.08822,-0.095525,0.887528,-2.08032,-0.156495,0.708303,-2.06289,-0.249732,0.771269,-2.07749,-0.286543,0.710934,-2.06595,-0.38494,0.499175,-2.02467,-0.493082,0.459051,-1.96292,-0.245519,0.3761,-2.00306,-0.30132,0.343952,-2.065,0.056093,1.31336,-1.84363,0.049269,1.31387,-2.05875},
/*238*/{0.1245,1.89987,-1.73894,0.224982,1.84118,-1.87411,-0.020214,1.80033,-1.66835,-0.133943,1.72607,-1.63554,-0.002299,1.64215,-1.67223,0.066703,1.587,-1.67645,0.176093,1.92087,-2.12389,0.238953,1.84307,-1.95018,0.034401,1.87084,-2.05477,0.051425,1.69265,-2.23582,0.110007,1.61139,-2.2755,0.265031,1.623,-2.15504,0.327275,1.61722,-2.15274,0.137574,1.40391,-1.82913,-0.052891,1.46885,-1.89512,-0.048073,1.49204,-1.95578,-0.04638,1.47002,-2.00949,0.138639,1.39757,-2.09341,0.279989,1.2146,-1.76448,0.191914,1.11256,-1.78759,0.252867,1.0568,-1.81125,0.365666,1.02619,-1.7968,0.283252,0.867463,-1.78392,0.238041,0.807093,-1.79402,0.194983,0.933943,-1.8039,0.151302,0.876871,-1.80828,0.052875,0.670666,-1.81735,-0.064565,0.645372,-1.89802,0.173404,0.560816,-1.83804,0.12331,0.531532,-1.76897,0.033603,1.09956,-2.12573,0.017941,1.01769,-2.10609,-0.107498,1.12553,-2.10135,-0.120204,1.04529,-2.08698,-0.107159,0.894227,-2.07971,-0.170036,0.715999,-2.06327,-0.263955,0.779263,-2.07732,-0.300971,0.719472,-2.06431,-0.39799,0.505038,-2.02562,-0.509361,0.472804,-1.96391,-0.26745,0.377345,-2.00304,-0.32359,0.344754,-2.06431,0.057012,1.318,-1.84224,0.050535,1.31559,-2.05737},
/*239*/{0.124135,1.90332,-1.73867,0.225812,1.84602,-1.87267,-0.022352,1.80497,-1.67021,-0.136985,1.731,-1.6381,-0.004806,1.64664,-1.67274,0.064472,1.59163,-1.6763,0.178255,1.92518,-2.12239,0.23986,1.8475,-1.9481,0.035679,1.87379,-2.05525,0.057348,1.6976,-2.23844,0.115864,1.61668,-2.27623,0.266909,1.63061,-2.15168,0.329866,1.62611,-2.14863,0.139512,1.40761,-1.8304,-0.049845,1.47381,-1.89611,-0.045062,1.49529,-1.95722,-0.043289,1.47326,-2.00947,0.1405,1.39765,-2.09265,0.281477,1.22005,-1.76449,0.195409,1.11773,-1.78577,0.254504,1.06284,-1.80995,0.366586,1.03116,-1.79699,0.291501,0.866876,-1.78396,0.248586,0.804771,-1.79421,0.199303,0.929876,-1.80531,0.159187,0.870033,-1.80963,0.073972,0.658459,-1.81853,-0.043807,0.625839,-1.89957,0.198393,0.556375,-1.83907,0.150127,0.522868,-1.76949,0.028776,1.1017,-2.1234,0.010776,1.02062,-2.10358,-0.11062,1.13304,-2.1011,-0.126828,1.05301,-2.08682,-0.118637,0.901568,-2.07883,-0.182709,0.723985,-2.06412,-0.276741,0.787635,-2.07678,-0.312807,0.726801,-2.06439,-0.415626,0.513347,-2.0276,-0.520762,0.486032,-1.96203,-0.289785,0.377316,-2.00272,-0.346165,0.345956,-2.06437,0.058434,1.3223,-1.84112,0.051625,1.31702,-2.05619},
/*240*/{0.123004,1.90654,-1.73843,0.22534,1.84933,-1.87183,-0.023814,1.80992,-1.67152,-0.139956,1.73474,-1.64058,-0.006318,1.65193,-1.67439,0.062482,1.59587,-1.67563,0.179861,1.92931,-2.12077,0.239723,1.85151,-1.94696,0.037568,1.8776,-2.05632,0.062646,1.70226,-2.24023,0.120942,1.62202,-2.27644,0.268638,1.63788,-2.1485,0.331478,1.63402,-2.14513,0.140064,1.4112,-1.82979,-0.047326,1.4788,-1.89586,-0.042279,1.49921,-1.95819,-0.039811,1.47692,-2.01037,0.142967,1.39828,-2.09182,0.283777,1.22461,-1.76422,0.194144,1.12221,-1.78601,0.254363,1.06667,-1.8093,0.365406,1.0351,-1.79745,0.298783,0.866558,-1.78358,0.25964,0.802556,-1.79405,0.20418,0.924283,-1.80646,0.166521,0.862172,-1.81035,0.092211,0.645308,-1.81845,-0.02268,0.606643,-1.89889,0.223903,0.551383,-1.83833,0.177663,0.514101,-1.7699,0.025387,1.10419,-2.12192,0.004242,1.02384,-2.10132,-0.11342,1.13976,-2.10078,-0.132641,1.06081,-2.08548,-0.129115,0.909471,-2.07918,-0.194996,0.732246,-2.06543,-0.289215,0.794771,-2.07698,-0.325786,0.734997,-2.06492,-0.425777,0.520833,-2.02964,-0.532364,0.500869,-1.96138,-0.310753,0.378266,-2.00413,-0.367967,0.347701,-2.06414,0.058689,1.32649,-1.83973,0.052921,1.31912,-2.05477},
/*241*/{0.122473,1.90913,-1.7383,0.218492,1.8563,-1.87242,-0.025951,1.81408,-1.67332,-0.140928,1.73932,-1.64345,-0.008124,1.65544,-1.67422,0.062088,1.59948,-1.67527,0.181247,1.93321,-2.11916,0.239856,1.85619,-1.94532,0.039041,1.88083,-2.0571,0.068071,1.70706,-2.24172,0.126597,1.62689,-2.27783,0.270865,1.64442,-2.14569,0.333762,1.64089,-2.13635,0.14181,1.41549,-1.8302,-0.04433,1.4838,-1.89573,-0.038931,1.5029,-1.9591,-0.037774,1.48068,-2.01141,0.145707,1.3986,-2.09104,0.28528,1.22929,-1.76455,0.195488,1.12742,-1.78338,0.254578,1.07096,-1.8075,0.363784,1.03837,-1.79792,0.30591,0.865161,-1.78367,0.269625,0.799285,-1.7934,0.208226,0.918077,-1.80765,0.174982,0.854502,-1.81154,0.110821,0.631126,-1.81876,-0.001248,0.588194,-1.8979,0.248454,0.545821,-1.84016,0.205119,0.506622,-1.77072,0.02193,1.10637,-2.12057,-0.001954,1.02613,-2.09978,-0.115501,1.14713,-2.09956,-0.1379,1.06814,-2.0842,-0.138775,0.917533,-2.07934,-0.20494,0.741222,-2.06664,-0.30005,0.803002,-2.07662,-0.335711,0.743203,-2.06508,-0.436543,0.528422,-2.03096,-0.541742,0.515975,-1.96172,-0.33184,0.378031,-2.00489,-0.39004,0.350068,-2.06365,0.060056,1.33124,-1.83832,0.054282,1.32103,-2.05324},
/*242*/{0.121862,1.91199,-1.73837,0.21885,1.85926,-1.8712,-0.026931,1.81774,-1.67458,-0.144259,1.743,-1.64564,-0.009449,1.65903,-1.67507,0.06,1.60265,-1.67418,0.183929,1.93715,-2.11778,0.240459,1.85941,-1.94433,0.040939,1.88348,-2.05766,0.072683,1.71156,-2.24295,0.131608,1.63174,-2.27865,0.271612,1.65036,-2.1439,0.335291,1.64738,-2.13462,0.144102,1.41894,-1.83014,-0.041732,1.4885,-1.89608,-0.036292,1.50696,-1.95977,-0.033661,1.48388,-2.01143,0.147678,1.3993,-2.08979,0.285457,1.23237,-1.76492,0.196594,1.13175,-1.78203,0.254149,1.0746,-1.80553,0.361915,1.04078,-1.79803,0.312587,0.863972,-1.78349,0.279012,0.794983,-1.7934,0.21269,0.911679,-1.8089,0.181801,0.846437,-1.81383,0.131348,0.619455,-1.81894,0.021101,0.569969,-1.89738,0.272783,0.541667,-1.84147,0.231771,0.499495,-1.77272,0.018229,1.10869,-2.11938,-0.008948,1.0321,-2.09975,-0.117737,1.15345,-2.09919,-0.141674,1.0756,-2.08421,-0.147989,0.925349,-2.07964,-0.215195,0.749376,-2.06804,-0.309541,0.811544,-2.07741,-0.345698,0.750449,-2.06512,-0.444764,0.535082,-2.03161,-0.550464,0.532337,-1.96298,-0.349298,0.378595,-2.0061,-0.411109,0.353899,-2.06385,0.061381,1.33548,-1.8369,0.055741,1.32302,-2.0517},
/*243*/{0.121992,1.91406,-1.73817,0.22544,1.86042,-1.86864,-0.027482,1.82136,-1.67623,-0.144957,1.74568,-1.64793,-0.011087,1.66181,-1.67542,0.059827,1.60583,-1.67382,0.185711,1.94036,-2.11604,0.241977,1.86185,-1.94257,0.043661,1.88638,-2.05794,0.076977,1.71646,-2.24461,0.13576,1.63598,-2.27952,0.273795,1.65576,-2.14103,0.336469,1.65418,-2.13011,0.14436,1.42193,-1.83084,-0.039346,1.49266,-1.89604,-0.033992,1.51074,-1.96168,-0.03066,1.48684,-2.01159,0.14987,1.40011,-2.0884,0.287205,1.23572,-1.76515,0.195317,1.13574,-1.77951,0.252638,1.0781,-1.80359,0.360169,1.04191,-1.7979,0.318401,0.86143,-1.78353,0.289482,0.791796,-1.79273,0.217161,0.904105,-1.80941,0.188685,0.837601,-1.81139,0.150348,0.607738,-1.81882,0.043492,0.553032,-1.89776,0.296903,0.538643,-1.84109,0.259173,0.492622,-1.7729,0.015052,1.11177,-2.11889,-0.01386,1.03441,-2.09867,-0.119489,1.15997,-2.09882,-0.144715,1.08233,-2.08315,-0.156529,0.933294,-2.08026,-0.223839,0.757799,-2.06964,-0.318211,0.819897,-2.07749,-0.354172,0.759064,-2.06595,-0.456176,0.54341,-2.0343,-0.556249,0.548307,-1.96166,-0.367237,0.377911,-2.0069,-0.433442,0.358055,-2.06311,0.061555,1.33915,-1.83606,0.056832,1.32502,-2.05077},
/*244*/{0.121541,1.91635,-1.738,0.226907,1.86082,-1.86758,-0.028557,1.82383,-1.67755,-0.145952,1.74793,-1.64941,-0.011402,1.66441,-1.67541,0.058881,1.60801,-1.67265,0.187875,1.94344,-2.115,0.243436,1.86447,-1.94109,0.044642,1.88912,-2.05869,0.081286,1.72024,-2.24583,0.140179,1.63967,-2.2802,0.27445,1.66056,-2.13946,0.337093,1.65974,-2.12784,0.145026,1.42482,-1.83155,-0.037341,1.49665,-1.89715,-0.032576,1.51439,-1.96229,-0.028697,1.49015,-2.01238,0.151163,1.40101,-2.08759,0.287361,1.23814,-1.76606,0.192949,1.13937,-1.77842,0.250578,1.08027,-1.8023,0.357717,1.0418,-1.79787,0.325097,0.858825,-1.78248,0.298385,0.787388,-1.79203,0.221546,0.896703,-1.80991,0.197859,0.828984,-1.812,0.170626,0.595479,-1.8188,0.066626,0.53519,-1.89697,0.319656,0.534821,-1.84277,0.285005,0.485879,-1.77368,0.012531,1.11479,-2.11914,-0.018059,1.03739,-2.09924,-0.121246,1.16621,-2.09773,-0.148004,1.08937,-2.08258,-0.163985,0.940841,-2.08011,-0.231823,0.76488,-2.07076,-0.325926,0.827737,-2.07771,-0.361787,0.7674,-2.06672,-0.464233,0.552142,-2.03582,-0.562363,0.564701,-1.96169,-0.385881,0.379424,-2.00858,-0.454755,0.364493,-2.06176,0.061618,1.34274,-1.83563,0.057221,1.32715,-2.05025},
/*245*/{0.122332,1.91849,-1.73755,0.221145,1.86747,-1.86831,-0.028703,1.82647,-1.67818,-0.146981,1.74981,-1.65134,-0.011519,1.66639,-1.67549,0.058209,1.60947,-1.67212,0.18934,1.94623,-2.11337,0.243457,1.86843,-1.94074,0.046967,1.89112,-2.05921,0.085321,1.72536,-2.24661,0.144505,1.64321,-2.28145,0.276131,1.66548,-2.13812,0.339512,1.66336,-2.1241,0.145429,1.42826,-1.83171,-0.034834,1.50132,-1.8972,-0.031189,1.5185,-1.96369,-0.026182,1.49405,-2.01273,0.153519,1.40277,-2.08694,0.287904,1.23958,-1.76725,0.19199,1.1415,-1.77693,0.24715,1.08131,-1.80122,0.354798,1.04093,-1.79761,0.331287,0.856033,-1.78138,0.308056,0.783076,-1.79147,0.225915,0.887903,-1.81045,0.204259,0.819389,-1.81454,0.19115,0.584071,-1.81881,0.088514,0.520086,-1.89715,0.343608,0.531219,-1.84329,0.310765,0.481266,-1.77536,0.009907,1.11771,-2.11953,-0.021488,1.04181,-2.1,-0.122147,1.17123,-2.09587,-0.150871,1.09469,-2.08141,-0.169526,0.94773,-2.0798,-0.238519,0.771315,-2.07194,-0.331956,0.835399,-2.07708,-0.36886,0.775072,-2.06754,-0.472107,0.560455,-2.03711,-0.566526,0.581775,-1.96192,-0.40344,0.379028,-2.00985,-0.473951,0.371776,-2.05986,0.061865,1.34681,-1.83511,0.058317,1.33007,-2.04967},
/*246*/{0.122272,1.91992,-1.73768,0.22836,1.86508,-1.86577,-0.0288,1.82799,-1.67934,-0.148858,1.7503,-1.65318,-0.011626,1.66835,-1.67585,0.059547,1.61113,-1.6703,0.192076,1.94879,-2.11236,0.245417,1.86918,-1.93896,0.049314,1.89469,-2.05944,0.089198,1.72904,-2.2481,0.147356,1.6472,-2.28282,0.277222,1.66918,-2.13725,0.34034,1.66737,-2.12241,0.146128,1.43143,-1.83148,-0.033279,1.50387,-1.89778,-0.027904,1.52041,-1.96375,-0.024515,1.49694,-2.01376,0.154645,1.40432,-2.08645,0.287058,1.24067,-1.76808,0.188708,1.14433,-1.77687,0.245475,1.08254,-1.79988,0.3521,1.03911,-1.79729,0.33608,0.852312,-1.78099,0.317513,0.778908,-1.79083,0.22973,0.879041,-1.8123,0.211116,0.810226,-1.81552,0.210216,0.573107,-1.81804,0.112278,0.503973,-1.89676,0.365546,0.527977,-1.84351,0.334982,0.475037,-1.77629,0.007863,1.12107,-2.11994,-0.024637,1.04485,-2.1012,-0.122303,1.17572,-2.09512,-0.152597,1.09956,-2.08044,-0.172115,0.95326,-2.07918,-0.244264,0.777815,-2.07302,-0.336613,0.843785,-2.07773,-0.374384,0.783573,-2.06872,-0.483122,0.569459,-2.03784,-0.570565,0.598316,-1.96256,-0.421292,0.378892,-2.00974,-0.495193,0.380445,-2.05846,0.062782,1.34972,-1.83469,0.059692,1.33201,-2.04918},
/*247*/{0.122873,1.92159,-1.73715,0.229196,1.86633,-1.86436,-0.029213,1.82964,-1.68044,-0.148124,1.75091,-1.65499,-0.011654,1.66911,-1.67544,0.059527,1.6124,-1.66949,0.193891,1.95098,-2.11182,0.245561,1.87132,-1.93829,0.051451,1.89633,-2.05947,0.092594,1.73237,-2.2486,0.150804,1.65012,-2.28334,0.278871,1.6721,-2.13688,0.339108,1.67093,-2.12333,0.147675,1.43487,-1.83089,-0.033559,1.50788,-1.89837,-0.027206,1.52424,-1.96465,-0.022265,1.49939,-2.01407,0.156003,1.4058,-2.08608,0.286973,1.24082,-1.76876,0.194057,1.14599,-1.77534,0.242796,1.08296,-1.79995,0.34868,1.03664,-1.79722,0.34052,0.848354,-1.77993,0.32658,0.774247,-1.78983,0.233912,0.869846,-1.81203,0.21998,0.798883,-1.81488,0.229354,0.56269,-1.81774,0.135206,0.489814,-1.89737,0.386766,0.523916,-1.84323,0.359042,0.470148,-1.77588,0.006504,1.12442,-2.121,-0.026779,1.04834,-2.10255,-0.123358,1.17942,-2.0928,-0.15378,1.10328,-2.0786,-0.173834,0.956556,-2.07823,-0.248774,0.783583,-2.07434,-0.340805,0.85046,-2.07729,-0.378661,0.791237,-2.06892,-0.489143,0.579765,-2.03962,-0.574154,0.614862,-1.96402,-0.439484,0.37878,-2.0089,-0.513411,0.389848,-2.05625,0.063234,1.35372,-1.83383,0.060638,1.33427,-2.04816},
/*248*/{0.123064,1.9232,-1.73682,0.231876,1.86739,-1.86328,-0.029097,1.83075,-1.68124,-0.147911,1.75104,-1.6554,-0.010565,1.66957,-1.67514,0.061165,1.61274,-1.66746,0.195814,1.95278,-2.11021,0.247955,1.87315,-1.93671,0.053423,1.89909,-2.05963,0.096572,1.73571,-2.24946,0.153603,1.65278,-2.28492,0.279026,1.67321,-2.13603,0.339863,1.67197,-2.11932,0.15018,1.43738,-1.83104,-0.032017,1.5104,-1.89807,-0.02575,1.52769,-1.96472,-0.021398,1.50289,-2.01425,0.15675,1.40805,-2.08576,0.284195,1.24059,-1.76959,0.189299,1.14706,-1.77471,0.240298,1.08317,-1.79819,0.345787,1.03341,-1.7958,0.344754,0.844258,-1.779,0.334651,0.769578,-1.78917,0.237216,0.859999,-1.81139,0.227133,0.789046,-1.81467,0.247194,0.553221,-1.81802,0.157155,0.477067,-1.89758,0.406208,0.520867,-1.84416,0.382835,0.465914,-1.77759,0.005687,1.12721,-2.12161,-0.02745,1.05153,-2.10352,-0.124203,1.18267,-2.09148,-0.153475,1.10665,-2.07708,-0.172691,0.959719,-2.07799,-0.253541,0.788102,-2.07442,-0.342845,0.856564,-2.07722,-0.383186,0.798577,-2.06911,-0.498267,0.589934,-2.04021,-0.578469,0.631286,-1.96595,-0.458231,0.377434,-2.00656,-0.529844,0.399915,-2.05489,0.064719,1.35664,-1.83372,0.061346,1.33698,-2.04804},
/*249*/{0.123811,1.92455,-1.73628,0.231839,1.86807,-1.8627,-0.028073,1.83099,-1.68093,-0.147738,1.75085,-1.65637,-0.009596,1.66938,-1.6748,0.062011,1.61282,-1.66633,0.198105,1.9542,-2.10929,0.249567,1.87475,-1.93548,0.05567,1.90104,-2.05951,0.098695,1.73871,-2.25032,0.15568,1.65515,-2.28586,0.280024,1.6742,-2.13567,0.342014,1.67258,-2.11496,0.151012,1.44029,-1.83088,-0.030396,1.51254,-1.89919,-0.025867,1.53071,-1.96535,-0.019356,1.50597,-2.01401,0.157455,1.40967,-2.08521,0.28222,1.23896,-1.7699,0.187451,1.1467,-1.77492,0.237291,1.08206,-1.79678,0.341236,1.03027,-1.79459,0.347714,0.839884,-1.77773,0.342545,0.764786,-1.78818,0.239692,0.849662,-1.81047,0.233354,0.77809,-1.81409,0.265814,0.543435,-1.81744,0.178323,0.465238,-1.8976,0.426274,0.518208,-1.84423,0.40346,0.460216,-1.77764,0.005064,1.12966,-2.12183,-0.027434,1.05345,-2.10364,-0.123618,1.18498,-2.09001,-0.153164,1.10825,-2.07608,-0.1716,0.961134,-2.07789,-0.256214,0.791371,-2.07462,-0.344101,0.862002,-2.07736,-0.385581,0.804893,-2.06959,-0.509475,0.59787,-2.04034,-0.583807,0.64668,-1.96846,-0.477731,0.388165,-2.00352,-0.548966,0.411655,-2.05222,0.065219,1.35961,-1.83341,0.062153,1.33916,-2.04765},
/*250*/{0.124222,1.92525,-1.73576,0.234446,1.86922,-1.86121,-0.027727,1.83106,-1.6811,-0.147241,1.75015,-1.65716,-0.008522,1.66954,-1.67487,0.063506,1.61216,-1.66481,0.199779,1.95549,-2.10874,0.25038,1.87577,-1.93521,0.057915,1.90244,-2.05944,0.101408,1.7412,-2.25078,0.15725,1.65705,-2.28673,0.280925,1.67497,-2.13519,0.3411,1.67354,-2.11759,0.150153,1.44269,-1.83093,-0.031064,1.51558,-1.90009,-0.024978,1.53306,-1.96498,-0.018473,1.50796,-2.01509,0.158272,1.41246,-2.08498,0.280543,1.23586,-1.77018,0.183297,1.14473,-1.77421,0.231436,1.07925,-1.79763,0.336805,1.02577,-1.79368,0.350861,0.835206,-1.77712,0.347798,0.759949,-1.78773,0.242349,0.839075,-1.80858,0.240556,0.767462,-1.8125,0.284808,0.535053,-1.81666,0.199196,0.454105,-1.89818,0.44562,0.515568,-1.84414,0.425273,0.456117,-1.77879,0.006327,1.13206,-2.12186,-0.026057,1.05558,-2.10505,-0.122388,1.18583,-2.08913,-0.152315,1.10974,-2.07528,-0.169713,0.962105,-2.07783,-0.258652,0.793377,-2.07395,-0.344669,0.866777,-2.07748,-0.386906,0.810538,-2.0691,-0.511411,0.611364,-2.04519,-0.589424,0.662115,-1.97032,-0.496585,0.399419,-1.99972,-0.565762,0.424993,-2.05172,0.064641,1.36197,-1.83372,0.062853,1.34177,-2.048},
/*251*/{0.124772,1.92621,-1.73504,0.234707,1.87015,-1.86068,-0.027117,1.8309,-1.68082,-0.145975,1.74835,-1.65765,-0.007075,1.66851,-1.674,0.065075,1.6115,-1.66277,0.202002,1.9564,-2.10767,0.251921,1.87804,-1.93359,0.05939,1.90511,-2.05875,0.103557,1.7435,-2.25124,0.158542,1.65856,-2.28717,0.281442,1.67535,-2.13485,0.341813,1.6725,-2.11714,0.151457,1.44335,-1.83108,-0.030209,1.51672,-1.89956,-0.024128,1.53538,-1.9653,-0.018459,1.51018,-2.01421,0.15876,1.41462,-2.08491,0.279032,1.23166,-1.76983,0.179917,1.14281,-1.77377,0.226267,1.07646,-1.79641,0.33228,1.02089,-1.79292,0.353389,0.829803,-1.77648,0.356672,0.75581,-1.78783,0.244517,0.828967,-1.80655,0.246289,0.756854,-1.81083,0.300278,0.526369,-1.81634,0.219554,0.442887,-1.89871,0.461955,0.514623,-1.84387,0.444558,0.45309,-1.77802,0.00848,1.13316,-2.12237,-0.025666,1.05618,-2.10507,-0.121755,1.18629,-2.08887,-0.150034,1.10939,-2.0746,-0.166802,0.961729,-2.07761,-0.260024,0.794251,-2.07356,-0.343475,0.87123,-2.07803,-0.388617,0.816903,-2.06943,-0.519852,0.621421,-2.04313,-0.596748,0.677762,-1.97313,-0.511702,0.409843,-1.99662,-0.58071,0.438368,-2.04989,0.065371,1.36303,-1.83415,0.063171,1.34389,-2.04852},
/*252*/{0.126261,1.92676,-1.73416,0.235126,1.87312,-1.85996,-0.025555,1.82951,-1.68043,-0.144646,1.74619,-1.65775,-0.004416,1.66749,-1.67327,0.067935,1.60998,-1.66063,0.204331,1.95699,-2.10683,0.253067,1.87856,-1.93306,0.061497,1.90669,-2.05878,0.104879,1.74516,-2.25142,0.159126,1.65989,-2.28802,0.283629,1.67531,-2.13534,0.342405,1.67063,-2.11589,0.153595,1.44748,-1.8323,-0.029744,1.51874,-1.89958,-0.024409,1.53771,-1.96549,-0.017933,1.51208,-2.01511,0.15956,1.41654,-2.08471,0.275036,1.2281,-1.77022,0.169884,1.13953,-1.77614,0.220683,1.07315,-1.79674,0.32813,1.01503,-1.79188,0.358253,0.825203,-1.77602,0.362478,0.749757,-1.78752,0.247283,0.819167,-1.80282,0.251592,0.747299,-1.80826,0.317362,0.521015,-1.81672,0.238506,0.433016,-1.89847,0.479879,0.514348,-1.84392,0.46385,0.452053,-1.77977,0.010511,1.13388,-2.12252,-0.020069,1.05697,-2.10523,-0.11888,1.18651,-2.08865,-0.146675,1.10926,-2.07469,-0.162897,0.961271,-2.07748,-0.261605,0.794659,-2.07205,-0.340743,0.875691,-2.07884,-0.389147,0.822036,-2.07001,-0.526063,0.633098,-2.04274,-0.604743,0.692738,-1.97652,-0.527134,0.423178,-1.99287,-0.593395,0.451501,-2.04776,0.067563,1.36659,-1.83373,0.064443,1.34577,-2.04793},
/*253*/{0.126906,1.92723,-1.7334,0.236328,1.87274,-1.85875,-0.024645,1.82849,-1.6802,-0.141924,1.74475,-1.65705,-0.002744,1.66499,-1.67187,0.069157,1.60763,-1.65787,0.20552,1.95721,-2.10618,0.254456,1.87919,-1.93252,0.063078,1.90786,-2.05875,0.106828,1.74621,-2.25133,0.159304,1.66035,-2.28792,0.282731,1.67276,-2.13439,0.343126,1.66887,-2.1183,0.153343,1.44805,-1.83118,-0.029713,1.51957,-1.90004,-0.023856,1.53892,-1.96507,-0.017502,1.51324,-2.01437,0.16053,1.41841,-2.08461,0.270895,1.22318,-1.76989,0.166528,1.13405,-1.77757,0.213788,1.06881,-1.79703,0.322702,1.00927,-1.79184,0.360889,0.820593,-1.7761,0.368346,0.746016,-1.78813,0.250494,0.809534,-1.79909,0.257573,0.738602,-1.80541,0.330675,0.51398,-1.81619,0.255967,0.424463,-1.89825,0.493783,0.514225,-1.84348,0.480249,0.449699,-1.77931,0.01446,1.13394,-2.1224,-0.015913,1.05573,-2.10516,-0.116989,1.18446,-2.08847,-0.143939,1.10712,-2.07427,-0.158648,0.959135,-2.07705,-0.262908,0.795454,-2.07144,-0.339596,0.878687,-2.07933,-0.388489,0.828797,-2.07133,-0.537565,0.645003,-2.04253,-0.612155,0.707843,-1.9795,-0.540053,0.436103,-1.98872,-0.604127,0.465315,-2.04621,0.067426,1.36712,-1.83384,0.065288,1.34734,-2.04816},
/*254*/{0.127647,1.92744,-1.73255,0.2369,1.87423,-1.85847,-0.023275,1.82658,-1.67916,-0.140703,1.74186,-1.65656,-0.00022,1.66298,-1.67038,0.072606,1.60569,-1.65646,0.208411,1.95745,-2.1056,0.255646,1.87935,-1.93169,0.064339,1.90921,-2.05807,0.107964,1.74721,-2.25146,0.158624,1.66106,-2.28812,0.28329,1.67127,-2.13478,0.343807,1.66703,-2.1185,0.151858,1.44812,-1.832,-0.028931,1.52008,-1.89971,-0.024382,1.54037,-1.96445,-0.017796,1.5148,-2.01493,0.161067,1.42034,-2.08453,0.26805,1.21654,-1.76903,0.159679,1.1314,-1.77625,0.206499,1.0654,-1.7976,0.317749,1.00247,-1.79132,0.364292,0.816461,-1.77681,0.37468,0.741927,-1.78837,0.253342,0.801489,-1.79623,0.263067,0.729599,-1.80278,0.344212,0.506259,-1.81535,0.271501,0.415811,-1.89825,0.507008,0.514523,-1.84382,0.496222,0.449878,-1.77996,0.017697,1.13341,-2.12183,-0.011315,1.05538,-2.1051,-0.113424,1.18226,-2.08883,-0.139893,1.10556,-2.07486,-0.153315,0.956644,-2.07634,-0.263118,0.795974,-2.06947,-0.335846,0.883588,-2.08093,-0.387648,0.833971,-2.07171,-0.542331,0.656922,-2.04286,-0.619365,0.723254,-1.98271,-0.553342,0.449581,-1.98523,-0.61487,0.479463,-2.04375,0.066624,1.36712,-1.8349,0.06521,1.3492,-2.04938},
/*255*/{0.129084,1.92741,-1.73152,0.238863,1.87384,-1.85788,-0.021659,1.82475,-1.67866,-0.139432,1.7386,-1.65668,0.002192,1.66035,-1.66875,0.074785,1.60308,-1.65397,0.210489,1.95679,-2.1049,0.256143,1.87979,-1.93102,0.06555,1.91061,-2.05828,0.108216,1.74759,-2.2509,0.158138,1.66081,-2.28776,0.282919,1.6682,-2.13499,0.343816,1.66251,-2.11833,0.15071,1.44825,-1.83233,-0.028939,1.51988,-1.89993,-0.02403,1.54075,-1.96439,-0.017348,1.5155,-2.01492,0.162081,1.42117,-2.08448,0.264678,1.21022,-1.76877,0.152839,1.1272,-1.77718,0.200281,1.05971,-1.7967,0.311915,0.996219,-1.7907,0.367653,0.812468,-1.77756,0.379856,0.737864,-1.78986,0.2568,0.792897,-1.79198,0.26849,0.722544,-1.79932,0.356087,0.502255,-1.81465,0.28577,0.408328,-1.89728,0.518819,0.514799,-1.84379,0.511277,0.450742,-1.78123,0.021725,1.1325,-2.12125,-0.006365,1.05382,-2.10451,-0.111325,1.17947,-2.08932,-0.135189,1.10244,-2.07505,-0.147378,0.953709,-2.07557,-0.26374,0.796698,-2.06786,-0.330194,0.886756,-2.08275,-0.3864,0.841078,-2.07242,-0.547995,0.671454,-2.04276,-0.625025,0.739519,-1.98501,-0.565861,0.46352,-1.98113,-0.623532,0.493969,-2.04191,0.066173,1.36689,-1.83549,0.065819,1.35001,-2.05006},
/*256*/{0.130213,1.92685,-1.73074,0.239207,1.87406,-1.85733,-0.020361,1.82243,-1.67759,-0.135762,1.73534,-1.65552,0.005266,1.6569,-1.66667,0.077238,1.59929,-1.65136,0.211567,1.95595,-2.10456,0.257194,1.87969,-1.93049,0.067221,1.91144,-2.05743,0.107176,1.74792,-2.25015,0.15664,1.66031,-2.28724,0.283417,1.6657,-2.13558,0.344283,1.65801,-2.11818,0.15098,1.44862,-1.8324,-0.028932,1.5198,-1.90001,-0.02403,1.54075,-1.96439,-0.016782,1.51551,-2.01447,0.162611,1.42342,-2.0843,0.260022,1.2034,-1.76872,0.149949,1.12495,-1.77676,0.191931,1.05446,-1.79827,0.305767,0.989857,-1.79074,0.369679,0.808998,-1.7789,0.382314,0.735398,-1.79095,0.259005,0.78595,-1.78938,0.273017,0.716,-1.79664,0.365657,0.498502,-1.81385,0.297832,0.403007,-1.89604,0.526991,0.514216,-1.84599,0.522906,0.449032,-1.78248,0.026712,1.13113,-2.12136,-1.1e-005,1.0522,-2.10311,-0.106643,1.17625,-2.09045,-0.130898,1.09821,-2.0749,-0.140991,0.950781,-2.07467,-0.265094,0.796715,-2.06652,-0.326735,0.888761,-2.08085,-0.384331,0.847404,-2.07368,-0.554429,0.684451,-2.04125,-0.628794,0.755792,-1.98655,-0.576444,0.479221,-1.97867,-0.63274,0.509655,-2.04137,0.067024,1.36685,-1.83603,0.067153,1.35113,-2.05069},
/*257*/{0.131852,1.92579,-1.72945,0.240104,1.87504,-1.85734,-0.018302,1.81974,-1.6764,-0.133802,1.7313,-1.65423,0.008441,1.65324,-1.66487,0.080698,1.59629,-1.64871,0.21334,1.95501,-2.10378,0.258116,1.87887,-1.929,0.068168,1.91157,-2.05648,0.105283,1.74781,-2.24872,0.15533,1.66012,-2.28692,0.282837,1.66206,-2.13582,0.344234,1.65266,-2.11872,0.153704,1.44881,-1.83252,-0.029764,1.51918,-1.9,-0.024109,1.5405,-1.9644,-0.016914,1.51481,-2.01438,0.163966,1.42369,-2.08449,0.254456,1.19674,-1.76821,0.140099,1.11924,-1.77841,0.18466,1.04919,-1.79864,0.300144,0.983729,-1.79062,0.371119,0.805412,-1.77969,0.386244,0.732008,-1.79146,0.260829,0.779692,-1.78783,0.27618,0.709772,-1.79483,0.374009,0.494969,-1.81261,0.307328,0.398207,-1.89279,0.535396,0.514307,-1.84648,0.532836,0.448429,-1.78404,0.031821,1.12876,-2.12,0.006177,1.0492,-2.10199,-0.102493,1.17255,-2.09098,-0.125523,1.09366,-2.07499,-0.134973,0.946972,-2.07296,-0.263104,0.798765,-2.06484,-0.322348,0.893797,-2.08131,-0.380301,0.85425,-2.07352,-0.5579,0.698515,-2.04099,-0.631271,0.772222,-1.98824,-0.587419,0.494947,-1.97608,-0.642996,0.526398,-2.04082,0.069028,1.36689,-1.83602,0.068637,1.35098,-2.05066},
/*258*/{0.133104,1.92491,-1.72913,0.240744,1.87379,-1.85707,-0.016207,1.81676,-1.67491,-0.129521,1.72779,-1.65321,0.011569,1.64982,-1.6625,0.084682,1.5926,-1.64575,0.214808,1.95342,-2.10278,0.258438,1.87832,-1.92881,0.069085,1.9118,-2.0555,0.104083,1.74739,-2.24739,0.152998,1.65854,-2.28601,0.28452,1.65707,-2.13667,0.34353,1.64688,-2.12007,0.154824,1.44894,-1.83258,-0.030059,1.51723,-1.90067,-0.024238,1.53965,-1.96546,-0.017063,1.51362,-2.01425,0.164382,1.42414,-2.0845,0.251274,1.18967,-1.76709,0.137394,1.11684,-1.77738,0.177062,1.04466,-1.79927,0.292836,0.977333,-1.79048,0.369688,0.802692,-1.78069,0.387302,0.729626,-1.79157,0.260714,0.774282,-1.78744,0.279053,0.704629,-1.793,0.379744,0.491456,-1.81223,0.314861,0.393589,-1.89145,0.541395,0.514019,-1.84784,0.541742,0.446909,-1.7842,0.03717,1.12656,-2.11862,0.013083,1.04736,-2.10122,-0.099115,1.1671,-2.09144,-0.119394,1.08806,-2.07508,-0.127747,0.943682,-2.07258,-0.260328,0.800966,-2.0636,-0.316765,0.898095,-2.08099,-0.37672,0.860255,-2.07344,-0.561795,0.712501,-2.04027,-0.631932,0.790215,-1.9895,-0.596983,0.511217,-1.975,-0.650807,0.543845,-2.04035,0.070332,1.3664,-1.83635,0.070064,1.35047,-2.05099},
/*259*/{0.134855,1.92327,-1.72788,0.241701,1.87318,-1.85666,-0.01422,1.81414,-1.67436,-0.125469,1.72383,-1.65117,0.015682,1.64551,-1.66017,0.088535,1.58887,-1.64365,0.214832,1.95082,-2.10202,0.258706,1.87721,-1.92863,0.070147,1.91123,-2.05448,0.101153,1.74699,-2.24537,0.149888,1.65777,-2.28464,0.283593,1.65189,-2.13845,0.343609,1.64034,-2.12162,0.152908,1.44804,-1.8327,-0.030019,1.51537,-1.90121,-0.023852,1.53876,-1.96532,-0.016438,1.51233,-2.01414,0.164549,1.42413,-2.08446,0.245269,1.18304,-1.7674,0.130075,1.11199,-1.78065,0.17037,1.03982,-1.80074,0.285731,0.97122,-1.79048,0.368055,0.800121,-1.7812,0.38743,0.726929,-1.79181,0.259081,0.768965,-1.78802,0.279063,0.700215,-1.7937,0.386043,0.488274,-1.81115,0.321108,0.389659,-1.89042,0.547311,0.512208,-1.84816,0.548456,0.446348,-1.78537,0.042682,1.12395,-2.11767,0.019714,1.04346,-2.10005,-0.093998,1.16257,-2.09179,-0.11372,1.08328,-2.07561,-0.119983,0.93956,-2.0715,-0.25726,0.803391,-2.06289,-0.311088,0.902071,-2.08072,-0.371374,0.86639,-2.07194,-0.563805,0.727778,-2.04097,-0.631026,0.807898,-1.98965,-0.60574,0.527918,-1.97424,-0.658209,0.56222,-2.03904,0.069741,1.36486,-1.83689,0.070626,1.34996,-2.05161},
/*260*/{0.136348,1.92159,-1.72691,0.2422,1.87156,-1.85688,-0.011545,1.81007,-1.67236,-0.12127,1.7193,-1.64925,0.019331,1.64135,-1.6574,0.092076,1.58491,-1.64096,0.21628,1.94923,-2.10112,0.259343,1.87577,-1.92844,0.070196,1.91075,-2.05389,0.098324,1.74671,-2.24354,0.146771,1.65699,-2.28369,0.282839,1.64672,-2.13958,0.343472,1.6325,-2.12236,0.153795,1.44628,-1.83212,-0.029437,1.5132,-1.90028,-0.023842,1.53592,-1.9656,-0.016652,1.51019,-2.01429,0.165579,1.42351,-2.08451,0.242,1.17671,-1.76676,0.124564,1.1069,-1.78048,0.163506,1.0348,-1.80185,0.278786,0.965035,-1.79027,0.36384,0.79727,-1.78107,0.384636,0.725057,-1.79123,0.256397,0.765057,-1.78903,0.277069,0.695965,-1.79514,0.390084,0.485607,-1.81098,0.324863,0.386141,-1.88842,0.549642,0.510291,-1.84937,0.552436,0.445268,-1.78668,0.049121,1.12074,-2.11668,0.026508,1.03994,-2.09872,-0.088655,1.15719,-2.09294,-0.107393,1.07757,-2.07595,-0.11213,0.934206,-2.0709,-0.253748,0.805453,-2.06169,-0.304704,0.905562,-2.07978,-0.367694,0.876942,-2.06742,-0.556099,0.740407,-2.03953,-0.627744,0.825693,-1.98972,-0.61321,0.544547,-1.97345,-0.665162,0.581402,-2.03953,0.071036,1.36277,-1.83701,0.07207,1.34845,-2.05177},
/*261*/{0.138115,1.91947,-1.72591,0.241631,1.86985,-1.85651,-0.009328,1.80614,-1.67037,-0.117562,1.71411,-1.64729,0.023141,1.63586,-1.65453,0.09633,1.57987,-1.63744,0.216544,1.94687,-2.10089,0.259248,1.87378,-1.92819,0.070669,1.90978,-2.05329,0.09551,1.74371,-2.2412,0.143341,1.65561,-2.2826,0.282477,1.64207,-2.13932,0.342566,1.62556,-2.12511,0.154166,1.44466,-1.83168,-0.030037,1.51033,-1.90098,-0.024092,1.53475,-1.96545,-0.016994,1.5081,-2.01506,0.166498,1.42213,-2.08468,0.238899,1.16937,-1.76501,0.118585,1.1003,-1.77952,0.157889,1.0301,-1.80354,0.272208,0.95831,-1.79075,0.360542,0.794095,-1.78022,0.381384,0.721187,-1.79037,0.253262,0.760028,-1.79027,0.274168,0.692031,-1.7961,0.38922,0.481297,-1.81029,0.326166,0.382329,-1.88768,0.551269,0.508262,-1.84985,0.5545,0.442855,-1.78683,0.054454,1.11755,-2.11602,0.033376,1.03662,-2.098,-0.084549,1.15341,-2.09378,-0.100993,1.07127,-2.07491,-0.104443,0.929746,-2.07055,-0.253514,0.806219,-2.06129,-0.296299,0.909294,-2.079,-0.359652,0.877443,-2.06599,-0.560379,0.759211,-2.04249,-0.62369,0.843705,-1.98836,-0.620673,0.561735,-1.97499,-0.66931,0.601369,-2.03933,0.07127,1.36084,-1.83737,0.072958,1.34687,-2.05215},
/*262*/{0.139348,1.91676,-1.725,0.242635,1.86823,-1.85628,-0.006252,1.80212,-1.66887,-0.114349,1.70839,-1.64467,0.027622,1.63102,-1.65127,0.1001,1.57607,-1.63566,0.216411,1.94418,-2.10011,0.260071,1.87164,-1.92749,0.070826,1.90891,-2.05236,0.09205,1.74243,-2.23949,0.139606,1.65363,-2.28086,0.281243,1.63539,-2.14026,0.340706,1.61746,-2.12728,0.155454,1.44222,-1.8314,-0.029544,1.50696,-1.9009,-0.023072,1.53172,-1.96548,-0.016635,1.50493,-2.01434,0.166893,1.42132,-2.08421,0.235757,1.16286,-1.76301,0.11624,1.09645,-1.77919,0.153524,1.02565,-1.80377,0.266487,0.953179,-1.79022,0.355332,0.790373,-1.77843,0.37769,0.717428,-1.78935,0.249336,0.754847,-1.79075,0.271338,0.686309,-1.79659,0.38957,0.477095,-1.80992,0.326191,0.377788,-1.88823,0.550827,0.505727,-1.84945,0.55384,0.438806,-1.78632,0.059949,1.11336,-2.11542,0.040559,1.03189,-2.09646,-0.079037,1.14763,-2.09407,-0.094192,1.06529,-2.07486,-0.09611,0.924787,-2.06928,-0.2491,0.80763,-2.06085,-0.288752,0.913038,-2.07777,-0.352628,0.885221,-2.06884,-0.558107,0.774107,-2.04254,-0.617141,0.860372,-1.98696,-0.626005,0.578482,-1.97628,-0.674935,0.62139,-2.04044,0.072939,1.35799,-1.83751,0.074598,1.34481,-2.05234},
/*263*/{0.141152,1.91356,-1.72427,0.242657,1.86538,-1.85577,-0.003689,1.79795,-1.66694,-0.109827,1.70308,-1.64182,0.031857,1.6264,-1.64843,0.10485,1.57133,-1.63226,0.217257,1.94156,-2.0996,0.259252,1.86904,-1.92779,0.071005,1.9068,-2.05086,0.087945,1.74151,-2.23746,0.135479,1.65136,-2.27919,0.279544,1.62906,-2.14409,0.339674,1.609,-2.12956,0.155117,1.43921,-1.83117,-0.029609,1.50426,-1.90069,-0.023257,1.52835,-1.96555,-0.015888,1.50237,-2.01559,0.167987,1.41992,-2.08354,0.232744,1.15768,-1.76163,0.112249,1.09616,-1.78051,0.150132,1.0224,-1.80454,0.263683,0.949312,-1.78912,0.352325,0.786005,-1.77938,0.37462,0.712825,-1.78952,0.244804,0.749412,-1.79227,0.268463,0.681306,-1.79695,0.387519,0.473068,-1.81068,0.324146,0.373421,-1.88971,0.54762,0.501517,-1.85038,0.551311,0.434238,-1.78729,0.065562,1.11051,-2.11527,0.048428,1.02875,-2.09587,-0.074151,1.13909,-2.0925,-0.087948,1.05833,-2.07422,-0.087848,0.919459,-2.06831,-0.24412,0.80903,-2.06068,-0.279242,0.916179,-2.0775,-0.345945,0.890452,-2.0675,-0.550683,0.785989,-2.0409,-0.609535,0.876714,-1.98553,-0.630177,0.595853,-1.97733,-0.67881,0.640947,-2.04093,0.072892,1.35483,-1.83807,0.075934,1.34278,-2.05295},
/*264*/{0.142669,1.91026,-1.72334,0.243748,1.86227,-1.85581,-0.000557,1.7929,-1.66402,-0.105296,1.69786,-1.63857,0.036337,1.62094,-1.64463,0.110033,1.5669,-1.6301,0.217001,1.93827,-2.09992,0.25979,1.86624,-1.92762,0.071111,1.90443,-2.05,0.084507,1.73935,-2.23509,0.131175,1.64934,-2.27802,0.277417,1.62289,-2.14545,0.337701,1.60104,-2.13218,0.154139,1.43592,-1.83062,-0.02933,1.50092,-1.90029,-0.023185,1.52496,-1.96552,-0.016642,1.49962,-2.01553,0.169408,1.41749,-2.08358,0.230895,1.1534,-1.75981,0.110654,1.09164,-1.78133,0.148224,1.01821,-1.80637,0.262251,0.946708,-1.78946,0.347131,0.78035,-1.77889,0.368924,0.707975,-1.78955,0.241108,0.744116,-1.7936,0.263282,0.676521,-1.79933,0.38409,0.470258,-1.81327,0.320012,0.368989,-1.89074,0.543857,0.495771,-1.85129,0.547462,0.428613,-1.78852,0.070985,1.10614,-2.11489,0.055361,1.02496,-2.09526,-0.069119,1.13252,-2.09214,-0.081156,1.05234,-2.07349,-0.079325,0.913575,-2.06795,-0.23882,0.810347,-2.06071,-0.270341,0.919197,-2.07659,-0.336823,0.896246,-2.06728,-0.550409,0.802911,-2.04254,-0.599891,0.892787,-1.98332,-0.636572,0.613205,-1.98053,-0.67988,0.661495,-2.04094,0.072638,1.3513,-1.83833,0.076624,1.34021,-2.05324},
/*265*/{0.1444,1.90662,-1.72224,0.24461,1.85772,-1.85566,0.002329,1.78716,-1.66154,-0.100473,1.6914,-1.63486,0.041496,1.61552,-1.64095,0.115328,1.56226,-1.62672,0.215818,1.93467,-2.09985,0.260207,1.86342,-1.92766,0.070228,1.90144,-2.04982,0.07983,1.7372,-2.23324,0.126674,1.64598,-2.27631,0.273962,1.61486,-2.14696,0.333852,1.5926,-2.13577,0.156252,1.43262,-1.82968,-0.029389,1.49713,-1.89967,-0.023032,1.52111,-1.96535,-0.015749,1.49565,-2.01541,0.170214,1.41559,-2.08305,0.228247,1.15051,-1.75886,0.107754,1.08698,-1.78191,0.146181,1.01516,-1.80671,0.261439,0.944836,-1.78903,0.342955,0.775052,-1.77839,0.365836,0.702567,-1.78963,0.235756,0.740167,-1.79397,0.25894,0.671722,-1.80058,0.379208,0.464322,-1.8142,0.314105,0.364687,-1.89341,0.538636,0.490167,-1.85176,0.539854,0.422446,-1.78929,0.07641,1.10222,-2.1143,0.062056,1.02011,-2.09431,-0.064998,1.12631,-2.0922,-0.075507,1.04563,-2.07238,-0.070981,0.908253,-2.06737,-0.232874,0.811417,-2.06018,-0.260383,0.920985,-2.07602,-0.32857,0.899168,-2.06636,-0.543607,0.816122,-2.0423,-0.589589,0.907684,-1.9816,-0.638304,0.629499,-1.98284,-0.681609,0.680674,-2.04246,0.074425,1.34783,-1.83815,0.078485,1.33728,-2.05309},
/*266*/{0.145838,1.9027,-1.72139,0.245236,1.85459,-1.85556,0.005194,1.78208,-1.65907,-0.096819,1.68513,-1.63072,0.046098,1.61016,-1.63808,0.120196,1.55686,-1.62379,0.21488,1.93108,-2.10019,0.259896,1.85949,-1.92747,0.069938,1.89908,-2.04742,0.076737,1.73443,-2.23162,0.121885,1.64324,-2.27489,0.271729,1.60696,-2.14854,0.330615,1.58287,-2.13842,0.153304,1.42789,-1.82946,-0.030536,1.49334,-1.90009,-0.022582,1.51807,-1.96507,-0.015984,1.49118,-2.01549,0.170294,1.41244,-2.08182,0.227233,1.14652,-1.7568,0.106088,1.08191,-1.78085,0.143157,1.01215,-1.80714,0.260136,0.942333,-1.78907,0.339201,0.76912,-1.77826,0.360314,0.69619,-1.78985,0.23214,0.73458,-1.79486,0.253849,0.66709,-1.80167,0.371351,0.458054,-1.81551,0.30596,0.359756,-1.89648,0.532359,0.481354,-1.85235,0.533131,0.413552,-1.78977,0.081819,1.09854,-2.11396,0.069197,1.01611,-2.09398,-0.060087,1.11875,-2.09136,-0.069059,1.0388,-2.07264,-0.061776,0.902827,-2.06711,-0.226543,0.812381,-2.06108,-0.249507,0.923416,-2.07561,-0.317727,0.903195,-2.06601,-0.537373,0.829748,-2.04224,-0.578308,0.92199,-1.98043,-0.637695,0.646123,-1.98519,-0.680526,0.700973,-2.04435,0.072168,1.34327,-1.83838,0.078178,1.33388,-2.05333},
/*267*/{0.147584,1.89855,-1.72044,0.246551,1.85088,-1.85592,0.008333,1.77605,-1.65657,-0.09268,1.67851,-1.6269,0.050077,1.604,-1.6347,0.126064,1.55256,-1.62046,0.214798,1.92735,-2.10081,0.260293,1.85589,-1.92765,0.069681,1.89506,-2.04665,0.071915,1.73172,-2.22932,0.117043,1.63954,-2.2737,0.268017,1.59924,-2.1505,0.327816,1.57285,-2.14185,0.15406,1.42447,-1.8278,-0.030196,1.48932,-1.90033,-0.023104,1.51449,-1.96485,-0.016502,1.48724,-2.01561,0.1709,1.40949,-2.08127,0.2249,1.14361,-1.75648,0.104628,1.08382,-1.78334,0.143898,1.00915,-1.80613,0.25923,0.938842,-1.78925,0.333557,0.762607,-1.77898,0.354991,0.689815,-1.79044,0.226485,0.729859,-1.79545,0.247707,0.661008,-1.80208,0.362067,0.451397,-1.81787,0.296815,0.35551,-1.9,0.524669,0.472025,-1.85145,0.523314,0.404003,-1.79131,0.086918,1.09437,-2.11345,0.076285,1.01227,-2.09366,-0.054686,1.11221,-2.09101,-0.062518,1.03184,-2.07193,-0.053792,0.896676,-2.06659,-0.219891,0.8128,-2.06079,-0.238494,0.923296,-2.07362,-0.307745,0.907558,-2.06614,-0.528973,0.841508,-2.04215,-0.565191,0.935423,-1.97963,-0.635389,0.661926,-1.98739,-0.67676,0.718721,-2.04577,0.072665,1.33981,-1.8379,0.079146,1.33041,-2.05283},
/*268*/{0.148603,1.89407,-1.7197,0.24521,1.84832,-1.85612,0.011867,1.7701,-1.65329,-0.087219,1.67165,-1.62281,0.056292,1.59842,-1.63064,0.132038,1.54771,-1.61874,0.21299,1.92283,-2.10126,0.260299,1.85216,-1.92776,0.069059,1.89028,-2.04596,0.068544,1.72791,-2.22788,0.11218,1.63589,-2.27224,0.265666,1.59187,-2.15255,0.323946,1.56287,-2.14466,0.153759,1.42083,-1.82694,-0.030369,1.48497,-1.89995,-0.02329,1.51043,-1.9648,-0.01561,1.48194,-2.0163,0.171334,1.40627,-2.08054,0.223951,1.14098,-1.75609,0.102774,1.08092,-1.78427,0.139939,1.00632,-1.80596,0.257973,0.93446,-1.78884,0.329274,0.756,-1.7794,0.349207,0.683268,-1.79149,0.220554,0.725187,-1.79679,0.241231,0.655611,-1.80279,0.353473,0.444833,-1.81933,0.286489,0.350707,-1.90363,0.515412,0.463522,-1.8523,0.513856,0.395524,-1.79315,0.092679,1.09098,-2.1133,0.082978,1.00734,-2.09375,-0.049056,1.10527,-2.09045,-0.055591,1.02502,-2.07155,-0.044737,0.891206,-2.06642,-0.212431,0.812654,-2.05984,-0.227489,0.925211,-2.07246,-0.29709,0.910227,-2.06525,-0.519651,0.852893,-2.04256,-0.553173,0.947092,-1.97847,-0.632343,0.676523,-1.98924,-0.671351,0.734153,-2.04697,0.072945,1.33586,-1.83749,0.080361,1.3264,-2.05239},
/*269*/{0.150102,1.88894,-1.71918,0.246152,1.84273,-1.85638,0.014845,1.76341,-1.65088,-0.083295,1.66443,-1.61811,0.062327,1.59254,-1.62694,0.137639,1.54282,-1.6147,0.212248,1.91835,-2.10252,0.260897,1.84845,-1.92789,0.067987,1.88576,-2.04479,0.064202,1.72445,-2.22658,0.106484,1.63145,-2.27133,0.261137,1.58182,-2.15447,0.319579,1.55187,-2.14846,0.15314,1.41676,-1.82603,-0.030747,1.48071,-1.90024,-0.022596,1.5059,-1.9642,-0.015453,1.47738,-2.01627,0.17204,1.40313,-2.07996,0.222335,1.13653,-1.75486,0.100704,1.07617,-1.78559,0.138139,1.00315,-1.80655,0.254532,0.928598,-1.78879,0.322245,0.749177,-1.77974,0.342868,0.676112,-1.79286,0.213921,0.719427,-1.79674,0.233135,0.650659,-1.80414,0.342204,0.437954,-1.82053,0.273629,0.348286,-1.9065,0.504567,0.453452,-1.8528,0.502224,0.383147,-1.7943,0.097654,1.08638,-2.1137,0.090339,1.00337,-2.09351,-0.042849,1.09825,-2.08948,-0.047795,1.017,-2.07064,-0.035568,0.885063,-2.06634,-0.204786,0.812524,-2.05964,-0.216326,0.925033,-2.07277,-0.285984,0.912968,-2.06544,-0.505641,0.859551,-2.04058,-0.540289,0.957645,-1.97792,-0.626887,0.689873,-1.99199,-0.656945,0.740533,-2.04334,0.072927,1.33151,-1.83735,0.081396,1.32265,-2.05224},
/*270*/{0.151125,1.88362,-1.71837,0.246101,1.83916,-1.8564,0.018295,1.75628,-1.64773,-0.078578,1.65642,-1.61346,0.066386,1.58633,-1.62276,0.143911,1.5385,-1.61196,0.210251,1.91361,-2.10318,0.260645,1.84402,-1.92881,0.066631,1.88092,-2.04386,0.059385,1.72,-2.22525,0.101525,1.62715,-2.27006,0.257347,1.57398,-2.15721,0.314861,1.54081,-2.15239,0.154952,1.41315,-1.82438,-0.030812,1.47602,-1.9008,-0.023471,1.50165,-1.96358,-0.015723,1.47248,-2.01618,0.172016,1.39956,-2.08025,0.220156,1.13328,-1.75527,0.099721,1.07435,-1.78655,0.136046,0.999296,-1.80539,0.251517,0.922663,-1.78831,0.31622,0.741456,-1.7808,0.333291,0.668169,-1.79257,0.206995,0.71418,-1.79806,0.224613,0.644233,-1.80376,0.327679,0.432169,-1.82135,0.255302,0.353874,-1.9076,0.493236,0.441844,-1.85396,0.484993,0.371147,-1.79474,0.104431,1.08212,-2.11357,0.098417,0.998542,-2.09362,-0.038102,1.09072,-2.08845,-0.040614,1.01018,-2.06952,-0.026806,0.879514,-2.06639,-0.196895,0.811778,-2.06016,-0.204702,0.924084,-2.07253,-0.274837,0.915314,-2.06505,-0.50007,0.872855,-2.04297,-0.527846,0.967012,-1.97623,-0.622758,0.700228,-1.99502,-0.657477,0.762565,-2.05079,0.07421,1.32773,-1.83682,0.082254,1.31845,-2.05171},
/*271*/{0.151802,1.87843,-1.71819,0.246038,1.83402,-1.85633,0.022241,1.74912,-1.64517,-0.073151,1.64923,-1.6081,0.072551,1.58048,-1.61918,0.150677,1.53414,-1.60984,0.208972,1.90873,-2.10416,0.260478,1.83951,-1.92939,0.06576,1.87609,-2.04327,0.054935,1.71632,-2.22426,0.096527,1.62245,-2.26886,0.252657,1.56333,-2.15954,0.309512,1.5291,-2.15649,0.154973,1.40953,-1.82376,-0.030903,1.47023,-1.9007,-0.024229,1.4976,-1.96352,-0.016075,1.46714,-2.01682,0.172109,1.39525,-2.0788,0.22021,1.12849,-1.75481,0.098544,1.07139,-1.78756,0.134973,0.995327,-1.80551,0.248432,0.916301,-1.78797,0.308151,0.734404,-1.78,0.323769,0.661557,-1.79471,0.199048,0.708582,-1.79938,0.215436,0.63937,-1.80778,0.3153,0.429994,-1.82404,0.237461,0.350806,-1.91122,0.47853,0.428846,-1.85615,0.46407,0.360532,-1.79623,0.109967,1.07734,-2.11357,0.105793,0.994033,-2.09445,-0.03205,1.08368,-2.08723,-0.033787,1.00333,-2.06886,-0.016266,0.873807,-2.06672,-0.188019,0.811254,-2.06079,-0.191248,0.923518,-2.07263,-0.262916,0.916629,-2.06473,-0.488767,0.881772,-2.04297,-0.514996,0.976494,-1.97736,-0.613792,0.711497,-1.99667,-0.647773,0.774498,-2.05207,0.07442,1.32361,-1.83607,0.083042,1.31372,-2.0509},
/*272*/{0.153368,1.87291,-1.71765,0.247348,1.82836,-1.85668,0.026761,1.74184,-1.64131,-0.068646,1.64175,-1.60305,0.078347,1.57436,-1.61464,0.157081,1.52962,-1.60681,0.207862,1.90367,-2.10469,0.261649,1.8347,-1.92991,0.064925,1.86965,-2.04237,0.051781,1.71153,-2.22354,0.090509,1.6175,-2.26798,0.247414,1.55384,-2.1628,0.304065,1.51739,-2.16009,0.153792,1.40565,-1.82214,-0.031599,1.46518,-1.90088,-0.023928,1.49154,-1.96226,-0.015606,1.46178,-2.01566,0.172233,1.39196,-2.07843,0.218003,1.12377,-1.75373,0.096788,1.06571,-1.78652,0.133032,0.990078,-1.80572,0.243645,0.910316,-1.78811,0.299755,0.727058,-1.78104,0.314473,0.653204,-1.79461,0.19067,0.703275,-1.80078,0.20694,0.633872,-1.80859,0.300505,0.423006,-1.82708,0.218907,0.355937,-1.9124,0.464716,0.414024,-1.85818,0.446032,0.348232,-1.79662,0.11588,1.0733,-2.11451,0.113656,0.989909,-2.09515,-0.02541,1.07715,-2.08628,-0.025,0.997152,-2.06859,-0.00694,0.868062,-2.06773,-0.178928,0.810584,-2.06111,-0.179796,0.924097,-2.07152,-0.250899,0.918516,-2.0647,-0.477261,0.888786,-2.04326,-0.502141,0.9838,-1.97841,-0.604024,0.720583,-1.9975,-0.636678,0.783534,-2.05462,0.074728,1.31885,-1.83535,0.084251,1.30941,-2.05017},
/*273*/{0.154682,1.86766,-1.71668,0.247525,1.82476,-1.85729,0.030985,1.73442,-1.63846,-0.062539,1.6342,-1.59829,0.084784,1.56883,-1.61098,0.164019,1.52582,-1.60418,0.205774,1.89909,-2.10585,0.260801,1.83002,-1.93101,0.063706,1.86453,-2.04197,0.04608,1.70688,-2.22239,0.084855,1.61275,-2.26701,0.242273,1.54349,-2.16522,0.29789,1.50571,-2.16427,0.15321,1.40096,-1.82072,-0.031692,1.45966,-1.90093,-0.024892,1.48629,-1.96177,-0.016355,1.4558,-2.01629,0.173641,1.38734,-2.0757,0.216072,1.1184,-1.75321,0.095573,1.06289,-1.78724,0.130057,0.985619,-1.80553,0.238296,0.904212,-1.78766,0.291717,0.719618,-1.78211,0.308658,0.648193,-1.79641,0.182736,0.698655,-1.80134,0.196047,0.628779,-1.81074,0.283486,0.417042,-1.8279,0.197562,0.354352,-1.9122,0.449907,0.400528,-1.85983,0.428295,0.337197,-1.79636,0.121476,1.06905,-2.11429,0.121061,0.985641,-2.0957,-0.019659,1.07017,-2.0854,-0.017996,0.989956,-2.06808,0.003851,0.862091,-2.06817,-0.169081,0.809842,-2.06217,-0.167211,0.92393,-2.07198,-0.237077,0.919755,-2.06419,-0.464907,0.895589,-2.04296,-0.489623,0.991524,-1.97856,-0.594603,0.728497,-1.9997,-0.62491,0.791874,-2.05662,0.07433,1.31399,-1.83415,0.085639,1.30425,-2.04887},
/*274*/{0.156371,1.8625,-1.71571,0.247628,1.81748,-1.85713,0.035338,1.72665,-1.63551,-0.057618,1.6261,-1.59274,0.091545,1.56396,-1.60726,0.171216,1.5225,-1.60163,0.20411,1.89454,-2.10639,0.261621,1.82464,-1.93173,0.062149,1.85849,-2.04133,0.042685,1.70206,-2.22155,0.079736,1.60813,-2.26661,0.236516,1.53279,-2.16791,0.290724,1.49297,-2.16826,0.152677,1.39686,-1.82002,-0.033587,1.45495,-1.90088,-0.025501,1.48041,-1.9611,-0.017335,1.45019,-2.01624,0.173596,1.38282,-2.07516,0.213127,1.11272,-1.75184,0.091778,1.05807,-1.78738,0.128473,0.981175,-1.80606,0.232131,0.897816,-1.78603,0.28459,0.713371,-1.78348,0.295351,0.639243,-1.79775,0.174472,0.695931,-1.80313,0.186786,0.62631,-1.81237,0.267339,0.416724,-1.83073,0.1764,0.356813,-1.91001,0.432781,0.3936,-1.86574,0.407434,0.339044,-1.79803,0.126831,1.0653,-2.11511,0.127486,0.981375,-2.09639,-0.01432,1.06347,-2.0851,-0.010614,0.983036,-2.06804,0.014804,0.853723,-2.06841,-0.158583,0.808887,-2.06331,-0.153723,0.922884,-2.07296,-0.224294,0.919441,-2.06423,-0.451799,0.900971,-2.04357,-0.477709,0.99821,-1.97931,-0.581714,0.736527,-2.00129,-0.612878,0.798299,-2.05822,0.074095,1.30949,-1.83339,0.08581,1.29926,-2.04806},
/*275*/{0.157782,1.85742,-1.71529,0.249516,1.81245,-1.85763,0.03981,1.71929,-1.63236,-0.050042,1.61842,-1.58706,0.09806,1.55947,-1.60491,0.179223,1.51953,-1.59888,0.203106,1.88959,-2.10703,0.261809,1.81976,-1.93275,0.060456,1.85303,-2.03984,0.039427,1.69757,-2.22109,0.073989,1.60355,-2.26603,0.230779,1.52209,-2.17133,0.283326,1.48058,-2.17284,0.151252,1.39273,-1.81831,-0.035861,1.45039,-1.90158,-0.026742,1.47505,-1.96154,-0.018061,1.44508,-2.0169,0.171084,1.3781,-2.07535,0.210806,1.10863,-1.74851,0.088745,1.05446,-1.78651,0.124842,0.97771,-1.80746,0.224183,0.889898,-1.78562,0.277107,0.709763,-1.78506,0.287481,0.636021,-1.79932,0.166621,0.697257,-1.80404,0.176413,0.626012,-1.81456,0.2507,0.418964,-1.83508,0.154606,0.355811,-1.91183,0.409573,0.391824,-1.87024,0.385865,0.339812,-1.79897,0.132077,1.06102,-2.11566,0.134966,0.97712,-2.0972,-0.009052,1.05678,-2.08441,-0.003282,0.97583,-2.06714,0.025486,0.85073,-2.07142,-0.148123,0.80932,-2.06432,-0.139804,0.921588,-2.07269,-0.209561,0.919713,-2.06475,-0.434357,0.903255,-2.04274,-0.46415,1.0031,-1.9801,-0.569291,0.740802,-2.00266,-0.597115,0.804164,-2.06154,0.072379,1.30524,-1.8329,0.084567,1.29416,-2.0475},
/*276*/{0.159555,1.85278,-1.71419,0.250881,1.80677,-1.85801,0.045155,1.71156,-1.62901,-0.044154,1.61117,-1.58213,0.105994,1.5551,-1.60189,0.187257,1.51671,-1.59726,0.20072,1.88563,-2.10804,0.263074,1.81451,-1.93348,0.059324,1.84859,-2.03869,0.033948,1.6928,-2.2206,0.067521,1.59917,-2.2657,0.224315,1.51092,-2.17519,0.275514,1.46804,-2.17683,0.14863,1.39011,-1.81923,-0.038116,1.44612,-1.90186,-0.028431,1.46904,-1.96308,-0.020548,1.44,-2.01744,0.17406,1.37401,-2.07252,0.205519,1.10668,-1.74519,0.086007,1.05231,-1.78577,0.121353,0.977751,-1.80837,0.216231,0.883002,-1.78778,0.271195,0.710983,-1.78563,0.278369,0.637921,-1.8014,0.160155,0.702038,-1.80533,0.165983,0.630928,-1.81454,0.229554,0.421237,-1.83776,0.131703,0.35765,-1.91365,0.387849,0.39087,-1.87214,0.361847,0.340325,-1.80107,0.138387,1.0561,-2.11523,0.143257,0.973796,-2.09812,-0.003222,1.04986,-2.084,0.004339,0.968683,-2.06679,0.03599,0.844844,-2.07298,-0.135492,0.806067,-2.06522,-0.126159,0.919445,-2.07323,-0.196231,0.91947,-2.06462,-0.424416,0.910362,-2.04454,-0.450847,1.00708,-1.98078,-0.556754,0.743499,-2.00455,-0.582946,0.808436,-2.06398,0.071155,1.30181,-1.832,0.086068,1.28967,-2.04638},
/*277*/{0.161818,1.84867,-1.71347,0.251712,1.80324,-1.85903,0.051043,1.70432,-1.62648,-0.035946,1.60413,-1.57639,0.113885,1.55107,-1.59915,0.195149,1.51449,-1.5953,0.199641,1.88238,-2.10979,0.264234,1.80977,-1.93494,0.058238,1.84589,-2.0369,0.030498,1.68847,-2.21994,0.061718,1.59489,-2.26517,0.217874,1.50064,-2.17953,0.267692,1.4562,-2.18086,0.145448,1.38818,-1.81879,-0.041516,1.44219,-1.90342,-0.031076,1.46405,-1.96374,-0.022982,1.43478,-2.01886,0.17256,1.36966,-2.07102,0.202388,1.10524,-1.74247,0.083718,1.05349,-1.78586,0.118918,0.978096,-1.80892,0.212473,0.883193,-1.78786,0.267492,0.712207,-1.78611,0.271933,0.639145,-1.8022,0.155784,0.709639,-1.80405,0.158561,0.638398,-1.81362,0.207762,0.424234,-1.83908,0.110766,0.357507,-1.91519,0.365841,0.390058,-1.87258,0.339391,0.338609,-1.80202,0.142829,1.05396,-2.11573,0.149465,0.971202,-2.09846,0.00244,1.04271,-2.08302,0.012121,0.96196,-2.06561,0.047383,0.838392,-2.07272,-0.12113,0.802481,-2.06558,-0.111548,0.917559,-2.07275,-0.180956,0.918041,-2.06477,-0.409845,0.912265,-2.04554,-0.437188,1.00961,-1.9815,-0.540958,0.746736,-2.00828,-0.56846,0.810845,-2.06559,0.068536,1.29922,-1.83089,0.085071,1.28493,-2.04501},
/*278*/{0.16398,1.84478,-1.71221,0.252823,1.79898,-1.8598,0.057062,1.6985,-1.62418,-0.027904,1.59718,-1.5717,0.121912,1.54793,-1.5966,0.203511,1.51274,-1.59407,0.198046,1.87969,-2.11175,0.265159,1.80554,-1.93638,0.057234,1.84404,-2.03393,0.027524,1.68489,-2.21936,0.056094,1.59102,-2.2634,0.210585,1.49115,-2.18326,0.258155,1.44492,-2.18486,0.142924,1.38458,-1.81744,-0.044619,1.43878,-1.90396,-0.035076,1.45987,-1.96538,-0.025523,1.43034,-2.01895,0.171891,1.36564,-2.06895,0.20149,1.10618,-1.73896,0.083523,1.05558,-1.78874,0.117863,0.97907,-1.80877,0.216895,0.89142,-1.78549,0.262334,0.712475,-1.78665,0.26038,0.637697,-1.80254,0.149835,0.714921,-1.80132,0.149465,0.642885,-1.81268,0.18887,0.424711,-1.83922,0.08842,0.359403,-1.91477,0.344283,0.388673,-1.87219,0.317639,0.339061,-1.80087,0.147225,1.05268,-2.11558,0.156591,0.970329,-2.09875,0.007527,1.03614,-2.08267,0.019862,0.956851,-2.06577,0.058686,0.832936,-2.07137,-0.107062,0.79944,-2.06482,-0.096462,0.914973,-2.07201,-0.166633,0.916945,-2.06547,-0.394776,0.914072,-2.04653,-0.422755,1.01061,-1.98314,-0.526681,0.747732,-2.00941,-0.553178,0.811852,-2.06805,0.065334,1.29594,-1.8299,0.083795,1.28075,-2.04381},
/*279*/{0.165277,1.84156,-1.71156,0.253331,1.7955,-1.86039,0.063364,1.69367,-1.62278,-0.019857,1.59228,-1.56686,0.131163,1.54631,-1.59556,0.213094,1.51239,-1.59293,0.197397,1.87635,-2.11309,0.265226,1.80218,-1.93752,0.056423,1.84299,-2.0314,0.023902,1.68218,-2.21824,0.051542,1.58815,-2.2615,0.202791,1.48214,-2.18532,0.24885,1.43524,-2.18792,0.140163,1.38319,-1.81511,-0.048104,1.43562,-1.90714,-0.036762,1.45574,-1.96559,-0.028968,1.42416,-2.0207,0.16966,1.36167,-2.06768,0.198024,1.10693,-1.73764,0.080453,1.05461,-1.78853,0.119074,0.978462,-1.8086,0.222306,0.897821,-1.78304,0.255289,0.711721,-1.78723,0.250001,0.638135,-1.80224,0.142344,0.71919,-1.79977,0.139228,0.648282,-1.80969,0.171151,0.42681,-1.83956,0.067443,0.360279,-1.91509,0.324334,0.387966,-1.87217,0.296858,0.338156,-1.80055,0.1513,1.05315,-2.11553,0.164172,0.970274,-2.09918,0.012588,1.03163,-2.08251,0.027941,0.952501,-2.06541,0.071287,0.829074,-2.07185,-0.093096,0.796365,-2.06464,-0.081851,0.912972,-2.07456,-0.150887,0.915165,-2.06608,-0.379825,0.913976,-2.04737,-0.409606,1.01061,-1.9832,-0.511283,0.747408,-2.01103,-0.536906,0.810571,-2.07034,0.062519,1.29422,-1.82827,0.082636,1.27596,-2.04178},
/*280*/{0.166467,1.83901,-1.7113,0.253811,1.79177,-1.86175,0.069544,1.69089,-1.62182,-0.012011,1.58736,-1.56282,0.139437,1.54459,-1.59488,0.222228,1.51186,-1.59221,0.198102,1.87456,-2.11392,0.266061,1.79953,-1.93896,0.056899,1.8427,-2.03111,0.020526,1.68105,-2.21579,0.045876,1.58646,-2.25972,0.195001,1.47582,-2.18641,0.238905,1.42644,-2.19076,0.137257,1.3832,-1.81246,-0.051123,1.43416,-1.90849,-0.037932,1.45426,-1.9653,-0.031013,1.41948,-2.02224,0.168403,1.3586,-2.06621,0.196066,1.10785,-1.73802,0.078153,1.05165,-1.78731,0.119061,0.978384,-1.80868,0.229763,0.903274,-1.78241,0.246202,0.712039,-1.78692,0.239521,0.638027,-1.80263,0.134095,0.721948,-1.79729,0.128327,0.650286,-1.80701,0.151285,0.425716,-1.83933,0.046324,0.361565,-1.91521,0.303208,0.38715,-1.8719,0.274099,0.338155,-1.80014,0.156836,1.05438,-2.11541,0.172271,0.972196,-2.09926,0.018004,1.02866,-2.08295,0.035214,0.94902,-2.06528,0.085002,0.827558,-2.07098,-0.078262,0.794295,-2.06371,-0.066331,0.910353,-2.07416,-0.136668,0.914279,-2.06724,-0.366793,0.913289,-2.04792,-0.395386,1.00982,-1.98471,-0.493687,0.744441,-2.01282,-0.519927,0.807611,-2.07082,0.059722,1.29402,-1.82606,0.081617,1.27269,-2.03912},
/*281*/{0.16814,1.83812,-1.71138,0.254317,1.79035,-1.86281,0.074476,1.68858,-1.6207,-0.000893,1.58466,-1.55942,0.148559,1.54357,-1.59421,0.231731,1.51268,-1.59173,0.198112,1.87267,-2.11357,0.265724,1.7986,-1.94019,0.057029,1.84265,-2.03101,0.018029,1.68192,-2.2132,0.039983,1.58726,-2.25795,0.187097,1.47177,-2.18595,0.22901,1.41983,-2.19339,0.134322,1.38309,-1.80922,-0.051674,1.43275,-1.90988,-0.038832,1.45399,-1.96601,-0.033328,1.41651,-2.02434,0.166165,1.35647,-2.06567,0.192285,1.1074,-1.73749,0.076601,1.05146,-1.7869,0.116316,0.976705,-1.80693,0.233759,0.904327,-1.78361,0.236273,0.710289,-1.78486,0.226842,0.635439,-1.80032,0.124157,0.724201,-1.79544,0.116632,0.652674,-1.8056,0.130601,0.425936,-1.83885,0.024958,0.362766,-1.91445,0.281589,0.385922,-1.87042,0.253964,0.337574,-1.79956,0.162539,1.05542,-2.11482,0.180713,0.973956,-2.09859,0.023685,1.02591,-2.08298,0.044308,0.946706,-2.0661,0.099672,0.827321,-2.07036,-0.063463,0.79286,-2.06277,-0.051516,0.907026,-2.07546,-0.120548,0.911452,-2.06743,-0.344698,0.906795,-2.0464,-0.383218,1.00753,-1.9866,-0.476888,0.740601,-2.01458,-0.503798,0.803101,-2.07221,0.056834,1.29388,-1.82486,0.080149,1.27041,-2.03753},
/*282*/{0.170509,1.83697,-1.7123,0.253995,1.79032,-1.86329,0.079778,1.68749,-1.62036,0.007387,1.58285,-1.55658,0.157714,1.54372,-1.59381,0.241968,1.51422,-1.59177,0.199144,1.87093,-2.11388,0.265246,1.79685,-1.94052,0.057379,1.84265,-2.03131,0.012663,1.68415,-2.20902,0.034334,1.58925,-2.25547,0.177889,1.46739,-2.18551,0.218726,1.41491,-2.19534,0.131692,1.38252,-1.80707,-0.052493,1.43342,-1.91034,-0.040778,1.45472,-1.96618,-0.035408,1.41668,-2.02463,0.163844,1.35476,-2.06507,0.188264,1.10542,-1.73643,0.072154,1.04966,-1.78601,0.110392,0.974687,-1.80481,0.232277,0.900619,-1.78377,0.224846,0.707471,-1.7834,0.21394,0.633436,-1.79841,0.112942,0.723797,-1.79455,0.103688,0.653342,-1.80425,0.110565,0.427126,-1.83777,0.004262,0.363164,-1.91379,0.261238,0.385529,-1.86906,0.231954,0.336975,-1.79865,0.168796,1.05777,-2.11477,0.189664,0.976189,-2.09783,0.030726,1.02588,-2.08494,0.053351,0.945414,-2.06653,0.115392,0.828459,-2.06975,-0.048155,0.792222,-2.06245,-0.035188,0.903901,-2.07649,-0.105176,0.908941,-2.06792,-0.336146,0.908271,-2.05118,-0.371419,1.00437,-1.98889,-0.457787,0.735748,-2.01491,-0.486516,0.796754,-2.07381,0.053289,1.29405,-1.82396,0.077165,1.26952,-2.03646},
/*283*/{0.172733,1.83656,-1.71311,0.25375,1.78891,-1.86439,0.085133,1.68699,-1.61911,0.015864,1.58123,-1.55436,0.167181,1.54459,-1.59332,0.251957,1.51689,-1.59225,0.199538,1.86986,-2.1136,0.264825,1.79579,-1.9412,0.057749,1.84293,-2.03238,0.00939,1.68667,-2.205,0.028407,1.59212,-2.25346,0.16958,1.46568,-2.18575,0.208523,1.41107,-2.19613,0.130019,1.38258,-1.80433,-0.053338,1.43344,-1.91048,-0.041199,1.4547,-1.96535,-0.035694,1.417,-2.02433,0.161574,1.35373,-2.06469,0.184212,1.10316,-1.73575,0.067602,1.04863,-1.78534,0.104785,0.972715,-1.80461,0.224803,0.896292,-1.7822,0.211602,0.702958,-1.78213,0.198514,0.62968,-1.79753,0.101241,0.72303,-1.7937,0.09065,0.652201,-1.80515,0.08957,0.427433,-1.83639,-0.01756,0.364235,-1.91278,0.240222,0.385081,-1.86904,0.21116,0.336376,-1.7978,0.176065,1.06038,-2.11534,0.198709,0.979984,-2.09755,0.039279,1.02405,-2.08536,0.06502,0.944486,-2.06704,0.131527,0.829576,-2.0684,-0.033458,0.79167,-2.06136,-0.020093,0.903163,-2.07713,-0.090298,0.907271,-2.06905,-0.316036,0.900833,-2.04872,-0.360944,1.00033,-1.99177,-0.439366,0.7282,-2.01437,-0.469238,0.788296,-2.07537,0.051213,1.29417,-1.82278,0.075483,1.26892,-2.03514},
/*284*/{0.174186,1.83704,-1.71445,0.252782,1.7876,-1.8653,0.088865,1.68686,-1.6188,0.025128,1.58165,-1.55223,0.176027,1.54616,-1.59265,0.261816,1.52026,-1.59298,0.199602,1.869,-2.11418,0.263967,1.79552,-1.94229,0.058212,1.84337,-2.03337,0.004465,1.69074,-2.20099,0.02179,1.59532,-2.25208,0.160754,1.46579,-2.18688,0.198137,1.40923,-2.19832,0.128603,1.38329,-1.80248,-0.054099,1.43342,-1.91036,-0.042037,1.45513,-1.96521,-0.036981,1.41863,-2.02426,0.160247,1.35439,-2.06434,0.177617,1.0986,-1.73612,0.059535,1.0461,-1.78696,0.098863,0.970694,-1.80395,0.215518,0.89111,-1.78178,0.198656,0.699161,-1.78178,0.183129,0.626238,-1.79733,0.088131,0.721873,-1.79302,0.074929,0.651474,-1.80366,0.070077,0.428541,-1.83571,-0.038594,0.364616,-1.91175,0.218889,0.384114,-1.86826,0.188893,0.336629,-1.79634,0.185108,1.06341,-2.1145,0.2095,0.984137,-2.09718,0.047981,1.02322,-2.086,0.075586,0.943871,-2.06879,0.148013,0.832306,-2.068,-0.018285,0.791869,-2.06033,-0.005266,0.902038,-2.07766,-0.074579,0.905838,-2.06962,-0.301487,0.895076,-2.04966,-0.350824,0.993862,-1.99439,-0.4197,0.719982,-2.01448,-0.450232,0.777977,-2.07678,0.049691,1.29466,-1.82239,0.074494,1.26969,-2.03473},
/*285*/{0.175921,1.83789,-1.71524,0.254012,1.78708,-1.86591,0.093933,1.68802,-1.61742,0.034395,1.58148,-1.54965,0.185687,1.54849,-1.59287,0.271109,1.52397,-1.59453,0.199134,1.86895,-2.1145,0.264406,1.79564,-1.94245,0.057953,1.8441,-2.03392,0.000522,1.69671,-2.19716,0.014626,1.59948,-2.25079,0.152715,1.46642,-2.18701,0.188182,1.40907,-2.1984,0.126274,1.38329,-1.8007,-0.054783,1.43407,-1.9095,-0.042601,1.45625,-1.96364,-0.037894,1.42003,-2.02291,0.159296,1.35539,-2.06401,0.17429,1.09547,-1.73473,0.055007,1.04529,-1.78554,0.090033,0.969452,-1.8052,0.203682,0.885499,-1.78117,0.184238,0.695033,-1.78136,0.167749,0.622308,-1.79713,0.074354,0.720352,-1.79258,0.05926,0.650901,-1.804,0.049291,0.428511,-1.83488,-0.059892,0.364982,-1.91064,0.197345,0.384398,-1.86718,0.167937,0.33623,-1.7958,0.19322,1.0671,-2.11389,0.220286,0.988911,-2.0966,0.058013,1.02259,-2.08734,0.087692,0.944659,-2.06914,0.164446,0.836802,-2.06755,-0.001636,0.789158,-2.05837,0.009832,0.901453,-2.0779,-0.06039,0.903774,-2.07065,-0.292478,0.89277,-2.05366,-0.340602,0.986944,-1.99645,-0.397443,0.71018,-2.01284,-0.430825,0.765778,-2.07744,0.04789,1.2947,-1.82181,0.073175,1.27085,-2.03422},
/*286*/{0.177766,1.83822,-1.71552,0.253652,1.78758,-1.86671,0.097991,1.68977,-1.61665,0.041784,1.58213,-1.54763,0.194604,1.55167,-1.59305,0.28094,1.52945,-1.59607,0.198307,1.86908,-2.1151,0.264075,1.79497,-1.94341,0.057567,1.84551,-2.03431,-0.0041,1.702,-2.1938,0.007535,1.60401,-2.24923,0.14443,1.46809,-2.18746,0.178822,1.41052,-2.19926,0.124897,1.38416,-1.79996,-0.055962,1.43444,-1.90818,-0.043807,1.45651,-1.96356,-0.038554,1.42203,-2.02114,0.157655,1.35728,-2.0644,0.16931,1.09175,-1.73596,0.049862,1.04638,-1.78469,0.081599,0.968198,-1.80537,0.192068,0.881059,-1.78036,0.169517,0.690624,-1.78134,0.151274,0.618252,-1.79661,0.06089,0.718796,-1.79282,0.043922,0.649328,-1.80397,0.028754,0.428005,-1.83299,-0.080823,0.366204,-1.90974,0.176361,0.384759,-1.86659,0.146713,0.336092,-1.79497,0.200849,1.07336,-2.11481,0.231156,0.99491,-2.09691,0.068121,1.02295,-2.08788,0.099755,0.945949,-2.06976,0.180252,0.840028,-2.06706,0.014826,0.788629,-2.05789,0.024236,0.900934,-2.07738,-0.047793,0.902271,-2.07103,-0.275218,0.883714,-2.05216,-0.332281,0.979363,-1.99932,-0.375615,0.700224,-2.01308,-0.410911,0.753535,-2.0788,0.047153,1.29516,-1.82172,0.072187,1.27245,-2.03428},
/*287*/{0.179499,1.84074,-1.71633,0.253867,1.78883,-1.86799,0.101679,1.69232,-1.61627,0.048695,1.5832,-1.54632,0.203073,1.55561,-1.59284,0.290246,1.53453,-1.59855,0.197237,1.86975,-2.11552,0.26312,1.7952,-1.94406,0.057133,1.84705,-2.03445,-0.007385,1.70792,-2.19069,0.000663,1.60988,-2.24826,0.135624,1.47082,-2.18848,0.169141,1.4135,-2.19961,0.12433,1.38494,-1.79885,-0.056048,1.43466,-1.90783,-0.044706,1.45755,-1.962,-0.040133,1.42336,-2.01969,0.157685,1.35867,-2.06497,0.165206,1.0887,-1.73644,0.042413,1.04672,-1.78495,0.073446,0.967081,-1.80528,0.179049,0.875545,-1.78066,0.153959,0.686682,-1.7816,0.134614,0.61408,-1.79783,0.04597,0.716234,-1.79378,0.027632,0.648709,-1.80455,0.009215,0.428521,-1.8327,-0.101226,0.366749,-1.90965,0.156343,0.38458,-1.8657,0.126358,0.335795,-1.79299,0.209001,1.07962,-2.1148,0.242388,1.00214,-2.0968,0.07775,1.02368,-2.08838,0.113885,0.948348,-2.07037,0.195919,0.844595,-2.06664,0.03137,0.788612,-2.05709,0.038647,0.901653,-2.07781,-0.032914,0.900281,-2.07312,-0.260975,0.874991,-2.05233,-0.322093,0.969516,-2.0018,-0.353828,0.688437,-2.01334,-0.389978,0.739353,-2.07952,0.047099,1.29562,-1.82145,0.071731,1.27386,-2.03416},
/*288*/{0.182057,1.84296,-1.71689,0.254006,1.78951,-1.86932,0.105333,1.69495,-1.61566,0.057819,1.58543,-1.54513,0.210971,1.55955,-1.59316,0.298972,1.54102,-1.60108,0.196617,1.87145,-2.11616,0.262676,1.79598,-1.94502,0.056182,1.84944,-2.03399,-0.010903,1.71224,-2.1875,-0.006817,1.61581,-2.247,0.127427,1.47487,-2.18836,0.160228,1.41757,-2.19952,0.123408,1.38624,-1.79828,-0.057097,1.43584,-1.90789,-0.044984,1.45809,-1.96088,-0.042138,1.42523,-2.01759,0.157508,1.36139,-2.06572,0.158675,1.0862,-1.73662,0.033814,1.04567,-1.78505,0.062894,0.967897,-1.80525,0.16597,0.870821,-1.78061,0.138975,0.683695,-1.78204,0.117835,0.612015,-1.79772,0.030667,0.714946,-1.79403,0.012206,0.64614,-1.80468,-0.010336,0.42845,-1.83256,-0.121709,0.367471,-1.90896,0.135632,0.383567,-1.8654,0.105475,0.335898,-1.79338,0.216889,1.08643,-2.11457,0.252709,1.01016,-2.09644,0.087686,1.02728,-2.08914,0.124138,0.954675,-2.07329,0.211562,0.849824,-2.06705,0.047626,0.789509,-2.05706,0.051174,0.901683,-2.07898,-0.01908,0.900269,-2.07356,-0.250784,0.870102,-2.05453,-0.315396,0.959336,-2.00286,-0.330555,0.676865,-2.01349,-0.368057,0.725235,-2.08081,0.047067,1.29654,-1.82154,0.071399,1.27607,-2.03441},
/*289*/{0.184073,1.84598,-1.71831,0.254226,1.79146,-1.87042,0.108925,1.69813,-1.61605,0.064856,1.58732,-1.543,0.218318,1.56411,-1.59437,0.306911,1.5473,-1.60309,0.194522,1.87363,-2.11719,0.262527,1.79805,-1.94731,0.055689,1.85083,-2.03418,-0.017347,1.72052,-2.18686,-0.01414,1.62213,-2.24626,0.120111,1.47977,-2.18922,0.152229,1.42269,-2.19886,0.123951,1.38735,-1.79814,-0.056667,1.43744,-1.90656,-0.0455,1.45957,-1.96027,-0.042883,1.42639,-2.0168,0.155243,1.36449,-2.0662,0.151949,1.08441,-1.73769,0.026598,1.04698,-1.78481,0.052797,0.968822,-1.80539,0.150557,0.866244,-1.78087,0.122973,0.681418,-1.78293,0.103204,0.611494,-1.79944,0.01542,0.713841,-1.79459,-0.003529,0.646177,-1.80464,-0.031111,0.429179,-1.83287,-0.142175,0.368158,-1.90947,0.114155,0.383679,-1.86556,0.084556,0.336397,-1.79372,0.22458,1.09391,-2.11527,0.262674,1.01935,-2.09647,0.097265,1.03029,-2.08979,0.13599,0.959346,-2.07285,0.22607,0.855484,-2.06752,0.064522,0.790956,-2.05741,0.064819,0.903745,-2.07935,-0.005828,0.899989,-2.07432,-0.236706,0.861331,-2.05641,-0.30623,0.947071,-2.00484,-0.307083,0.664295,-2.01268,-0.345739,0.710451,-2.08153,0.047767,1.29776,-1.82167,0.070621,1.27821,-2.0348},
/*290*/{0.186391,1.84918,-1.71885,0.254257,1.79398,-1.8717,0.113199,1.70151,-1.61574,0.071652,1.58993,-1.54202,0.22648,1.5697,-1.59454,0.314383,1.55435,-1.60696,0.19288,1.87598,-2.11789,0.261267,1.79975,-1.9485,0.054555,1.85362,-2.03257,-0.022375,1.7254,-2.1855,-0.021294,1.62857,-2.24573,0.111959,1.48587,-2.18833,0.143316,1.42882,-2.19867,0.125189,1.38808,-1.79883,-0.056343,1.43917,-1.90499,-0.045191,1.46119,-1.95923,-0.043151,1.42837,-2.01566,0.154533,1.36698,-2.06682,0.149259,1.0836,-1.73794,0.01906,1.04867,-1.78467,0.044275,0.968466,-1.80559,0.133963,0.863854,-1.78202,0.107478,0.680472,-1.78428,0.084186,0.609222,-1.8005,-1e-006,0.714204,-1.79455,-0.020648,0.646371,-1.8049,-0.052538,0.430053,-1.83342,-0.162765,0.368586,-1.90998,0.093983,0.38369,-1.86618,0.06411,0.336623,-1.79382,0.230361,1.10215,-2.11567,0.2711,1.02843,-2.09704,0.106242,1.0349,-2.08975,0.146609,0.965,-2.07366,0.2407,0.861786,-2.0678,0.080869,0.792304,-2.05745,0.077106,0.905296,-2.07968,0.007229,0.898115,-2.07431,-0.222678,0.8517,-2.0563,-0.297215,0.934002,-2.00586,-0.28241,0.652137,-2.0135,-0.323135,0.695575,-2.0824,0.048878,1.29872,-1.82204,0.070142,1.28042,-2.03544},
/*291*/{0.188919,1.85269,-1.72005,0.255636,1.79589,-1.8732,0.117832,1.70534,-1.61573,0.078834,1.59293,-1.54113,0.233098,1.57568,-1.59668,0.321926,1.5616,-1.6098,0.189121,1.87901,-2.11835,0.261026,1.80218,-1.94992,0.052649,1.85584,-2.03172,-0.028649,1.73141,-2.18494,-0.028314,1.63534,-2.24517,0.103948,1.49217,-2.18753,0.135388,1.43502,-2.1977,0.126499,1.38908,-1.79908,-0.055718,1.44093,-1.90454,-0.045847,1.46313,-1.95851,-0.043176,1.42957,-2.0145,0.15368,1.37042,-2.06808,0.137349,1.08238,-1.7396,0.012605,1.05219,-1.78416,0.032629,0.970167,-1.80495,0.116962,0.861688,-1.78287,0.091792,0.67865,-1.78624,0.068338,0.60877,-1.80202,-0.014571,0.714535,-1.79589,-0.037262,0.647562,-1.80661,-0.071852,0.43012,-1.83383,-0.183062,0.370458,-1.90995,0.073978,0.383476,-1.86586,0.043737,0.336691,-1.79297,0.236918,1.10991,-2.11552,0.279421,1.03837,-2.09736,0.113854,1.03945,-2.08964,0.157286,0.971162,-2.07329,0.254691,0.869475,-2.0686,0.096793,0.794968,-2.05861,0.088218,0.908277,-2.07881,0.019195,0.898299,-2.07584,-0.20963,0.841275,-2.05668,-0.28826,0.919426,-2.00731,-0.258416,0.638911,-2.01439,-0.300825,0.680876,-2.08373,0.050302,1.2999,-1.82276,0.07014,1.28298,-2.03641},
/*292*/{0.191765,1.85645,-1.72112,0.254408,1.79938,-1.87443,0.122954,1.70893,-1.61556,0.085934,1.59654,-1.54004,0.238928,1.58176,-1.59886,0.327871,1.56936,-1.61347,0.187266,1.88273,-2.11909,0.260811,1.80455,-1.95191,0.051452,1.85854,-2.03043,-0.034938,1.7395,-2.18595,-0.035847,1.64225,-2.24474,0.096229,1.49942,-2.18683,0.127571,1.44189,-2.19669,0.127065,1.3903,-1.79982,-0.055346,1.44373,-1.90281,-0.045057,1.4656,-1.95699,-0.043522,1.43192,-2.01385,0.153671,1.37392,-2.06762,0.135993,1.0823,-1.74033,0.003404,1.05504,-1.78506,0.022698,0.972788,-1.80549,0.10028,0.860215,-1.78477,0.075527,0.678052,-1.78785,0.051215,0.609289,-1.80378,-0.029654,0.715885,-1.79588,-0.053562,0.648627,-1.80786,-0.089862,0.433191,-1.83336,-0.203125,0.371651,-1.90943,0.053416,0.382932,-1.86623,0.022861,0.336848,-1.79333,0.242008,1.12004,-2.11546,0.286776,1.04885,-2.09783,0.121627,1.04378,-2.08911,0.166546,0.977453,-2.07319,0.268149,0.878589,-2.06893,0.113388,0.797074,-2.05964,0.100584,0.909643,-2.08127,0.031715,0.895765,-2.07664,-0.191835,0.829521,-2.05562,-0.28004,0.904812,-2.0061,-0.232647,0.625819,-2.01509,-0.278458,0.665301,-2.08452,0.05101,1.30148,-1.82295,0.07002,1.28604,-2.03679},
/*293*/{0.19448,1.8604,-1.72225,0.257356,1.80342,-1.87706,0.127625,1.71245,-1.61561,0.093576,1.60027,-1.53922,0.245669,1.5886,-1.60051,0.333628,1.57719,-1.61697,0.18274,1.88629,-2.11976,0.260152,1.80768,-1.95372,0.049614,1.86045,-2.02843,-0.041281,1.74598,-2.18498,-0.042665,1.64856,-2.24406,0.089181,1.50667,-2.18602,0.120672,1.45011,-2.19597,0.128035,1.39114,-1.80037,-0.054589,1.44642,-1.90167,-0.044509,1.46827,-1.95501,-0.04287,1.43433,-2.01237,0.153001,1.37725,-2.06819,0.126265,1.08063,-1.74294,-0.004617,1.05768,-1.78485,0.01096,0.974905,-1.80618,0.083019,0.860163,-1.78702,0.059776,0.677636,-1.78971,0.032397,0.608911,-1.80621,-0.046041,0.716543,-1.79714,-0.070031,0.65124,-1.80776,-0.111866,0.434615,-1.83483,-0.223481,0.373118,-1.9092,0.032721,0.382913,-1.86686,0.001778,0.336746,-1.79389,0.247448,1.12793,-2.11509,0.293721,1.05878,-2.09832,0.128678,1.04885,-2.08851,0.176098,0.983369,-2.0725,0.280782,0.889547,-2.06958,0.129046,0.800391,-2.0604,0.111426,0.912146,-2.08244,0.042616,0.896039,-2.07804,-0.182903,0.818902,-2.05615,-0.270552,0.887402,-2.00567,-0.206925,0.612829,-2.01702,-0.253253,0.650059,-2.08516,0.051982,1.30277,-1.82342,0.069558,1.28907,-2.03749},
/*294*/{0.196834,1.86488,-1.72369,0.256664,1.80608,-1.87891,0.132796,1.71654,-1.61546,0.099263,1.60447,-1.53854,0.251487,1.59488,-1.60291,0.339977,1.58533,-1.62138,0.179317,1.8907,-2.12023,0.25975,1.81141,-1.95579,0.048021,1.86385,-2.02707,-0.046281,1.75252,-2.18457,-0.049434,1.65565,-2.24339,0.082571,1.51462,-2.18518,0.113566,1.45772,-2.19503,0.129931,1.39165,-1.80037,-0.05228,1.44938,-1.89995,-0.043854,1.47058,-1.95352,-0.042551,1.43694,-2.01123,0.153826,1.38128,-2.06827,0.117926,1.07951,-1.74511,-0.013992,1.05942,-1.78592,0.00102,0.978259,-1.80624,0.06866,0.861863,-1.789,0.042327,0.678935,-1.79178,0.016492,0.610129,-1.80702,-0.062176,0.718254,-1.79916,-0.087461,0.653135,-1.80713,-0.12987,0.434343,-1.83352,-0.24383,0.376705,-1.90878,0.01291,0.383037,-1.86725,-0.018281,0.337327,-1.79391,0.251518,1.13653,-2.11554,0.30043,1.06877,-2.09863,0.135219,1.05442,-2.08854,0.184297,0.990608,-2.07226,0.292469,0.899725,-2.06995,0.144501,0.803599,-2.06136,0.121527,0.91417,-2.08241,0.053166,0.896023,-2.07838,-0.168253,0.808134,-2.05703,-0.261153,0.870287,-2.00576,-0.179586,0.600138,-2.02018,-0.228322,0.634289,-2.08728,0.053798,1.30397,-1.82394,0.070249,1.29248,-2.03824},
/*295*/{0.199853,1.86958,-1.72492,0.256852,1.81038,-1.88084,0.138135,1.7211,-1.61551,0.106871,1.60858,-1.53813,0.256277,1.60162,-1.60563,0.344231,1.59346,-1.62545,0.17498,1.89449,-2.12011,0.259247,1.81492,-1.95794,0.046381,1.867,-2.02569,-0.051697,1.75953,-2.18396,-0.055494,1.6624,-2.24264,0.075628,1.5223,-2.18391,0.107005,1.46565,-2.19413,0.131227,1.39286,-1.80036,-0.052023,1.45238,-1.89792,-0.042829,1.47344,-1.95215,-0.041603,1.44094,-2.01015,0.153856,1.38481,-2.06797,0.112051,1.07956,-1.74721,-0.020803,1.06193,-1.78348,-0.010079,0.981089,-1.80661,0.055459,0.863694,-1.79146,0.025317,0.68026,-1.79356,-0.001501,0.610997,-1.80821,-0.079067,0.72074,-1.80007,-0.10468,0.654086,-1.80975,-0.149147,0.437626,-1.83389,-0.263567,0.380601,-1.9083,-0.007022,0.382816,-1.86762,-0.039045,0.337,-1.79502,0.25581,1.14471,-2.11605,0.306664,1.07861,-2.09864,0.14211,1.06002,-2.0877,0.192493,0.997241,-2.07181,0.304232,0.911316,-2.07106,0.159623,0.807102,-2.06327,0.130815,0.916493,-2.08257,0.062771,0.89627,-2.07898,-0.152829,0.796651,-2.059,-0.248276,0.85163,-2.00521,-0.151982,0.587842,-2.02262,-0.205421,0.618812,-2.08943,0.054467,1.30569,-1.82419,0.070151,1.29609,-2.03864},
/*296*/{0.202122,1.87428,-1.72646,0.258858,1.81515,-1.88352,0.144299,1.72505,-1.61555,0.113771,1.61294,-1.53696,0.261153,1.60803,-1.60799,0.349833,1.60152,-1.62925,0.171291,1.89917,-2.12052,0.25959,1.81954,-1.95992,0.045193,1.86944,-2.02297,-0.056912,1.76639,-2.18283,-0.062085,1.66862,-2.24158,0.069294,1.53053,-2.18289,0.101253,1.47419,-2.19322,0.132462,1.39432,-1.8004,-0.050267,1.45553,-1.89601,-0.042196,1.47701,-1.95002,-0.040626,1.44395,-2.00885,0.154147,1.38928,-2.06741,0.105237,1.08019,-1.74977,-0.029467,1.06997,-1.78554,-0.020138,0.984941,-1.80814,0.044701,0.866659,-1.79354,0.008412,0.681916,-1.79516,-0.019958,0.612906,-1.81032,-0.096367,0.723066,-1.80188,-0.120649,0.656968,-1.81192,-0.167149,0.440077,-1.83369,-0.283071,0.385775,-1.908,-0.026976,0.381925,-1.86725,-0.058496,0.338116,-1.79448,0.259519,1.15301,-2.1165,0.31201,1.08749,-2.09907,0.147689,1.06653,-2.08764,0.199622,1.0046,-2.07178,0.313948,0.922755,-2.07151,0.173392,0.810808,-2.06442,0.139411,0.918488,-2.08293,0.073278,0.89255,-2.07887,-0.137905,0.784553,-2.05878,-0.239141,0.833222,-2.00345,-0.124467,0.575444,-2.02531,-0.181034,0.604388,-2.09104,0.055826,1.30763,-1.82431,0.070598,1.30008,-2.0389},
/*297*/{0.204888,1.87906,-1.72797,0.25968,1.81916,-1.88518,0.150951,1.72938,-1.6164,0.12027,1.61833,-1.53706,0.266537,1.61591,-1.61083,0.353856,1.60975,-1.63352,0.167914,1.90407,-2.1206,0.259181,1.82358,-1.96204,0.04329,1.87271,-2.02153,-0.060677,1.773,-2.1821,-0.068281,1.67572,-2.23981,0.062472,1.53896,-2.18174,0.095507,1.4826,-2.19251,0.134724,1.39536,-1.80023,-0.048724,1.45838,-1.89435,-0.040992,1.47985,-1.94875,-0.040744,1.44866,-2.00766,0.154833,1.39365,-2.06712,0.103342,1.08219,-1.75113,-0.034233,1.07311,-1.78474,-0.030583,0.989728,-1.8087,0.033563,0.869864,-1.79616,-0.007694,0.684592,-1.79655,-0.036925,0.615685,-1.81116,-0.112768,0.7264,-1.80228,-0.13894,0.6606,-1.81161,-0.186762,0.444411,-1.83388,-0.302088,0.391143,-1.90796,-0.047168,0.381552,-1.8684,-0.080275,0.338063,-1.79518,0.262859,1.16143,-2.11662,0.316674,1.09799,-2.09977,0.153358,1.07109,-2.08653,0.206965,1.0112,-2.07098,0.323183,0.933557,-2.07197,0.187316,0.814252,-2.06649,0.147219,0.920501,-2.083,0.083034,0.89084,-2.0785,-0.121992,0.770963,-2.05902,-0.226013,0.81342,-2.0024,-0.095728,0.565363,-2.02918,-0.154878,0.589191,-2.09279,0.057255,1.3093,-1.82502,0.071046,1.30433,-2.03976},
/*298*/{0.207574,1.88424,-1.72954,0.259779,1.82377,-1.88712,0.156352,1.73459,-1.61611,0.125661,1.62275,-1.53714,0.270437,1.62196,-1.61298,0.35743,1.61766,-1.63771,0.165088,1.90895,-2.12026,0.258333,1.82852,-1.96432,0.041745,1.87621,-2.01901,-0.065647,1.77928,-2.18131,-0.073793,1.68222,-2.23834,0.056897,1.54717,-2.18003,0.090238,1.49099,-2.19142,0.136095,1.3962,-1.79946,-0.046874,1.46281,-1.89332,-0.039917,1.48348,-1.94615,-0.039707,1.4521,-2.00641,0.155418,1.39813,-2.06633,0.092604,1.0816,-1.75479,-0.042149,1.0768,-1.78494,-0.039954,0.995088,-1.80853,0.021813,0.871481,-1.79881,-0.024857,0.688208,-1.7977,-0.053741,0.619508,-1.81246,-0.128634,0.730839,-1.80337,-0.155162,0.664728,-1.81241,-0.204061,0.449088,-1.83456,-0.321114,0.398702,-1.90785,-0.06649,0.381072,-1.86895,-0.100562,0.338232,-1.7957,0.266496,1.16969,-2.11709,0.320782,1.10767,-2.09996,0.158397,1.07751,-2.08626,0.213437,1.01864,-2.07091,0.331378,0.944927,-2.07233,0.200791,0.81904,-2.06783,0.15458,0.92206,-2.08326,0.091707,0.88946,-2.07885,-0.104346,0.757389,-2.05858,-0.211822,0.792264,-2.00079,-0.067382,0.553736,-2.03257,-0.128477,0.575471,-2.0958,0.058023,1.31131,-1.8251,0.07134,1.30849,-2.03991},
/*299*/{0.209462,1.88945,-1.73082,0.260483,1.82844,-1.88948,0.162716,1.73917,-1.61731,0.132756,1.6281,-1.5361,0.274388,1.62953,-1.61629,0.36083,1.62563,-1.64212,0.161696,1.91355,-2.12032,0.258429,1.83316,-1.96619,0.040493,1.87968,-2.0172,-0.069735,1.78516,-2.18005,-0.079672,1.68949,-2.23719,0.051063,1.55459,-2.17832,0.085435,1.4991,-2.19078,0.138201,1.39801,-1.79929,-0.045261,1.46599,-1.89158,-0.038228,1.48731,-1.94435,-0.038263,1.45737,-2.00594,0.156399,1.40282,-2.06531,0.088197,1.08327,-1.75765,-0.048131,1.08203,-1.785,-0.048625,1.00032,-1.8093,0.009715,0.874294,-1.80005,-0.041372,0.692302,-1.79856,-0.070599,0.623382,-1.81237,-0.144721,0.735782,-1.80444,-0.171957,0.670037,-1.81338,-0.221967,0.45115,-1.8319,-0.338867,0.405626,-1.90859,-0.08601,0.38012,-1.86888,-0.121026,0.338851,-1.79644,0.269914,1.17787,-2.11731,0.325957,1.11673,-2.09954,0.16307,1.08425,-2.08614,0.219058,1.02632,-2.07137,0.338971,0.95651,-2.07292,0.214034,0.820778,-2.06875,0.161065,0.923463,-2.0834,0.101002,0.886738,-2.07858,-0.087265,0.7437,-2.05767,-0.198702,0.771611,-1.99806,-0.038587,0.542835,-2.03592,-0.10137,0.561744,-2.09815,0.059376,1.31363,-1.8256,0.072331,1.31328,-2.04044},
/*300*/{0.211821,1.89428,-1.73259,0.261272,1.8337,-1.89124,0.16784,1.74425,-1.61825,0.138105,1.63325,-1.53589,0.278014,1.63655,-1.61868,0.364233,1.63351,-1.64665,0.158382,1.91804,-2.12065,0.257657,1.83855,-1.96829,0.038943,1.88407,-2.01551,-0.073793,1.79197,-2.17955,-0.084689,1.696,-2.23583,0.047164,1.56251,-2.17604,0.080455,1.50721,-2.19044,0.139764,1.39916,-1.79923,-0.043819,1.46912,-1.89059,-0.037218,1.49084,-1.94274,-0.03746,1.46215,-2.00508,0.157091,1.40788,-2.06446,0.083123,1.08438,-1.76171,-0.053657,1.08835,-1.78485,-0.058036,1.00602,-1.8096,-0.004908,0.879081,-1.80084,-0.057133,0.696929,-1.79916,-0.087548,0.628807,-1.81345,-0.159961,0.741495,-1.80567,-0.188429,0.675948,-1.8144,-0.239243,0.457114,-1.83378,-0.356397,0.413766,-1.90798,-0.104275,0.380055,-1.86966,-0.141319,0.340252,-1.79617,0.272749,1.18592,-2.11755,0.329485,1.12568,-2.09957,0.167532,1.09089,-2.08623,0.224769,1.03365,-2.07116,0.345173,0.9671,-2.07297,0.225383,0.826107,-2.07081,0.168298,0.923963,-2.08285,0.109706,0.883882,-2.07749,-0.073563,0.731455,-2.05699,-0.183389,0.749757,-1.99646,-0.008095,0.532483,-2.03909,-0.074021,0.549499,-2.10084,0.060307,1.31549,-1.82656,0.073057,1.31813,-2.0414},
/*301*/{0.213071,1.90003,-1.73477,0.261837,1.83817,-1.89326,0.173174,1.74937,-1.61898,0.143688,1.63899,-1.53592,0.281108,1.64357,-1.6213,0.367048,1.64091,-1.6508,0.156584,1.9226,-2.12051,0.2573,1.84286,-1.97004,0.037629,1.88801,-2.01433,-0.077259,1.79812,-2.17734,-0.089481,1.70318,-2.23347,0.041314,1.57014,-2.17515,0.076099,1.51504,-2.18979,0.142143,1.40119,-1.79871,-0.041667,1.47228,-1.88923,-0.035503,1.49499,-1.94006,-0.035554,1.46772,-2.00435,0.15874,1.41307,-2.06372,0.07754,1.08733,-1.76317,-0.059496,1.0946,-1.78491,-0.064301,1.01195,-1.80697,-0.01976,0.884336,-1.80134,-0.072072,0.702006,-1.79983,-0.103704,0.633498,-1.81344,-0.175121,0.747375,-1.80621,-0.203988,0.68202,-1.81504,-0.256222,0.46268,-1.83367,-0.373222,0.422786,-1.90856,-0.122197,0.379535,-1.86943,-0.161683,0.341111,-1.79668,0.275548,1.19329,-2.11708,0.332694,1.13385,-2.099,0.171987,1.09774,-2.08711,0.229037,1.04102,-2.07092,0.350427,0.977012,-2.07279,0.237334,0.829867,-2.0729,0.17346,0.924321,-2.08275,0.116566,0.881021,-2.07836,-0.055657,0.718855,-2.05684,-0.166268,0.72886,-1.99611,0.021509,0.524071,-2.04334,-0.045008,0.536987,-2.10355,0.062157,1.31794,-1.82716,0.07461,1.32344,-2.04196},
/*302*/{0.214992,1.90515,-1.73686,0.262014,1.84346,-1.89519,0.177891,1.75521,-1.62003,0.14901,1.6443,-1.5357,0.284072,1.65028,-1.62338,0.369511,1.64832,-1.65529,0.153661,1.927,-2.1205,0.256854,1.84774,-1.97184,0.036728,1.89157,-2.01311,-0.080645,1.80359,-2.17508,-0.094492,1.7094,-2.23186,0.03764,1.57731,-2.17339,0.072122,1.52297,-2.18928,0.14405,1.40214,-1.79894,-0.039191,1.47587,-1.88827,-0.034003,1.4991,-1.93881,-0.033867,1.47278,-2.0036,0.159324,1.41775,-2.06295,0.071179,1.09006,-1.76639,-0.064957,1.1006,-1.7843,-0.072949,1.01963,-1.80915,-0.033944,0.890585,-1.80032,-0.086899,0.708253,-1.79949,-0.119562,0.639607,-1.81176,-0.189519,0.754261,-1.80642,-0.219487,0.688617,-1.81603,-0.27182,0.468708,-1.83311,-0.389881,0.433064,-1.90924,-0.141736,0.378199,-1.86992,-0.181884,0.3417,-1.79711,0.278917,1.20129,-2.11729,0.336964,1.14255,-2.09799,0.175732,1.10439,-2.08741,0.23305,1.04869,-2.07151,0.354172,0.986328,-2.07268,0.248037,0.833053,-2.07426,0.178469,0.923694,-2.08235,0.12515,0.877325,-2.07739,-0.038548,0.705018,-2.05696,-0.150325,0.708248,-1.99569,0.052236,0.516331,-2.0472,-0.015627,0.525566,-2.10722,0.06312,1.32,-1.82825,0.075194,1.32824,-2.04299},
/*303*/{0.21659,1.9107,-1.73842,0.263322,1.84813,-1.89718,0.182518,1.76044,-1.62105,0.154353,1.64983,-1.53571,0.286164,1.65711,-1.62709,0.370859,1.65488,-1.65844,0.15168,1.93103,-2.12073,0.256687,1.85266,-1.97347,0.035833,1.89556,-2.01275,-0.084655,1.80934,-2.17507,-0.098486,1.71594,-2.22912,0.03279,1.58472,-2.17228,0.068631,1.52973,-2.18898,0.146069,1.40364,-1.7997,-0.037272,1.47954,-1.88675,-0.032897,1.5033,-1.93692,-0.032247,1.47791,-2.00294,0.160348,1.42314,-2.06258,0.066972,1.09177,-1.76814,-0.066994,1.10925,-1.7828,-0.080178,1.02668,-1.80702,-0.04658,0.896908,-1.80042,-0.101806,0.714781,-1.79866,-0.133135,0.645534,-1.8115,-0.203687,0.761237,-1.80684,-0.234055,0.696163,-1.81614,-0.288849,0.475715,-1.83406,-0.405505,0.443058,-1.90992,-0.160387,0.377407,-1.87053,-0.202575,0.342176,-1.79787,0.281858,1.20894,-2.11731,0.340214,1.15046,-2.09759,0.17845,1.1113,-2.08814,0.237115,1.0566,-2.07219,0.356765,0.995206,-2.07254,0.258304,0.836651,-2.07546,0.183997,0.921473,-2.08146,0.133657,0.872272,-2.07701,-0.020809,0.693077,-2.05939,-0.131939,0.686464,-1.99398,0.08267,0.509121,-2.05154,0.012676,0.515159,-2.11013,0.064569,1.32229,-1.82956,0.076014,1.33365,-2.04419},
/*304*/{0.218954,1.91549,-1.73964,0.262555,1.85186,-1.89945,0.186263,1.76572,-1.62151,0.15885,1.65597,-1.5355,0.288723,1.66321,-1.62907,0.372988,1.66203,-1.66359,0.149612,1.93465,-2.12077,0.25612,1.85717,-1.97551,0.036143,1.89773,-2.01086,-0.087432,1.81478,-2.17273,-0.102756,1.72186,-2.22764,0.029136,1.5915,-2.17115,0.066568,1.53622,-2.18788,0.148399,1.40503,-1.80046,-0.034558,1.48256,-1.88669,-0.030392,1.50737,-1.93545,-0.030425,1.48263,-2.00287,0.16111,1.42821,-2.06243,0.062322,1.09432,-1.76941,-0.073258,1.11407,-1.78314,-0.087212,1.03401,-1.8068,-0.058962,0.903227,-1.79992,-0.115247,0.72165,-1.79748,-0.148145,0.653329,-1.81108,-0.217311,0.768292,-1.80788,-0.247127,0.703569,-1.81636,-0.306292,0.482064,-1.83394,-0.420876,0.45381,-1.91007,-0.179697,0.377143,-1.87064,-0.223392,0.34306,-1.79863,0.283822,1.21658,-2.11703,0.342967,1.15827,-2.0968,0.18223,1.11802,-2.08925,0.240407,1.06306,-2.07198,0.358588,1.00288,-2.07255,0.268026,0.839318,-2.07664,0.188929,0.919715,-2.08054,0.141767,0.867353,-2.07618,0.001425,0.678107,-2.05569,-0.111657,0.663835,-1.99084,0.112902,0.502997,-2.05619,0.042202,0.505751,-2.1136,0.066352,1.32442,-1.83124,0.077266,1.3386,-2.04574},
/*305*/{0.220353,1.92121,-1.74141,0.262797,1.85654,-1.90123,0.189618,1.77096,-1.6227,0.162407,1.66183,-1.53611,0.291065,1.6694,-1.63273,0.374949,1.66856,-1.66706,0.147663,1.93834,-2.12116,0.255947,1.86174,-1.97754,0.036551,1.90123,-2.00921,-0.089874,1.82059,-2.17173,-0.106175,1.72793,-2.22537,0.026415,1.59794,-2.16978,0.063421,1.54272,-2.18751,0.150756,1.40683,-1.80164,-0.032664,1.48594,-1.88497,-0.029134,1.51141,-1.93323,-0.028716,1.48774,-2.00219,0.161818,1.43298,-2.06236,0.062087,1.09926,-1.77029,-0.076218,1.1211,-1.78202,-0.093587,1.04173,-1.80661,-0.069527,0.910047,-1.79994,-0.12888,0.727833,-1.79654,-0.161688,0.66013,-1.80946,-0.229913,0.776379,-1.80809,-0.260055,0.712258,-1.81642,-0.317997,0.488285,-1.83207,-0.43497,0.466058,-1.91114,-0.198904,0.377038,-1.87076,-0.243958,0.344719,-1.79859,0.286502,1.22259,-2.11677,0.345269,1.1652,-2.09582,0.184835,1.12447,-2.09027,0.243242,1.06879,-2.07251,0.358795,1.0101,-2.07311,0.277614,0.841135,-2.07753,0.194099,0.916695,-2.08032,0.150152,0.861774,-2.07423,0.019502,0.664698,-2.0559,-0.092095,0.643498,-1.99127,0.143641,0.49787,-2.05961,0.071953,0.496792,-2.11649,0.06806,1.32682,-1.83242,0.077758,1.34368,-2.04678},
/*306*/{0.221245,1.92553,-1.74316,0.264725,1.862,-1.9035,0.193499,1.77616,-1.62345,0.166949,1.66688,-1.53569,0.292563,1.67506,-1.6345,0.376737,1.67391,-1.67059,0.146401,1.94187,-2.12113,0.2551,1.86566,-1.97915,0.036054,1.90359,-2.00964,-0.091802,1.82421,-2.16887,-0.110172,1.73317,-2.22301,0.02336,1.60338,-2.16833,0.059232,1.5495,-2.18817,0.153346,1.40825,-1.80301,-0.030532,1.49048,-1.88442,-0.027342,1.51527,-1.93175,-0.026612,1.49282,-2.00146,0.16258,1.43822,-2.06232,0.05524,1.1006,-1.77125,-0.07784,1.13013,-1.78149,-0.09933,1.04946,-1.80524,-0.078603,0.916795,-1.79918,-0.139802,0.735539,-1.7953,-0.17382,0.668413,-1.80695,-0.24189,0.78422,-1.80839,-0.272091,0.719503,-1.8173,-0.333369,0.49404,-1.83154,-0.448745,0.478966,-1.91149,-0.21697,0.37712,-1.87111,-0.263786,0.346232,-1.79959,0.288299,1.22969,-2.11603,0.347323,1.17216,-2.09515,0.186507,1.13098,-2.09099,0.246117,1.07429,-2.07181,0.358471,1.01615,-2.07297,0.285991,0.842407,-2.07885,0.198823,0.913324,-2.07941,0.158121,0.854993,-2.0733,0.039202,0.651049,-2.0551,-0.069219,0.620103,-1.98946,0.173245,0.493402,-2.06306,0.102409,0.488676,-2.11884,0.069685,1.32936,-1.83407,0.078484,1.349,-2.04823},
/*307*/{0.22309,1.9294,-1.74473,0.264741,1.86509,-1.90497,0.196149,1.78104,-1.62396,0.170144,1.67119,-1.5362,0.294827,1.68054,-1.63645,0.377492,1.67977,-1.67521,0.145037,1.94522,-2.12143,0.254632,1.86974,-1.98099,0.036302,1.90742,-2.00873,-0.093377,1.83013,-2.16651,-0.113438,1.73855,-2.22088,0.021502,1.60852,-2.16729,0.056504,1.55519,-2.18802,0.155389,1.41007,-1.80393,-0.027972,1.49388,-1.88388,-0.025719,1.51912,-1.92967,-0.024512,1.49773,-2.00149,0.16294,1.44242,-2.063,0.051315,1.10362,-1.77146,-0.080541,1.13768,-1.78148,-0.105022,1.05783,-1.80486,-0.08775,0.924011,-1.79918,-0.152588,0.743817,-1.79314,-0.184312,0.676702,-1.80508,-0.253222,0.791603,-1.8104,-0.283094,0.727793,-1.81677,-0.346362,0.501236,-1.82938,-0.460152,0.491797,-1.91221,-0.236432,0.377522,-1.87058,-0.285435,0.347355,-1.79944,0.290402,1.2354,-2.11545,0.34872,1.17777,-2.09373,0.188318,1.13568,-2.09371,0.245954,1.08181,-2.07536,0.358357,1.02149,-2.07278,0.295586,0.842974,-2.07959,0.203956,0.908779,-2.07891,0.166777,0.848813,-2.07339,0.059739,0.638458,-2.05597,-0.046565,0.599883,-1.98898,0.202574,0.490058,-2.06552,0.131887,0.481268,-2.12102,0.071172,1.33184,-1.83542,0.078867,1.35357,-2.04942},
/*308*/{0.224016,1.93356,-1.74631,0.264991,1.86977,-1.90741,0.198513,1.78571,-1.62494,0.17353,1.6772,-1.53668,0.296181,1.6857,-1.63856,0.378748,1.68549,-1.67907,0.143658,1.94822,-2.12196,0.254422,1.87376,-1.9816,0.035578,1.91045,-2.00764,-0.095121,1.83447,-2.16453,-0.116861,1.7434,-2.21789,0.019103,1.61378,-2.16679,0.054538,1.55999,-2.18758,0.157834,1.41191,-1.80511,-0.025895,1.49745,-1.88304,-0.023216,1.52252,-1.93007,-0.022628,1.50284,-2.0004,0.163448,1.44653,-2.06311,0.046594,1.10763,-1.77128,-0.0822,1.14335,-1.77939,-0.109205,1.06434,-1.80407,-0.09607,0.931574,-1.79849,-0.161879,0.751093,-1.79159,-0.19418,0.683853,-1.80322,-0.26238,0.800358,-1.80813,-0.293696,0.735386,-1.81704,-0.357907,0.508866,-1.82788,-0.469869,0.50588,-1.91078,-0.254674,0.37808,-1.86991,-0.305993,0.349625,-1.80018,0.292046,1.24128,-2.11437,0.349889,1.1823,-2.09241,0.189057,1.14202,-2.09431,0.246235,1.08619,-2.07714,0.357957,1.02548,-2.07366,0.304107,0.843086,-2.08032,0.2101,0.904131,-2.07863,0.174984,0.841601,-2.07315,0.079431,0.625586,-2.05652,-0.023201,0.580619,-1.9886,0.232467,0.487292,-2.06749,0.162015,0.474437,-2.12312,0.072716,1.33445,-1.83669,0.079516,1.35798,-2.05053},
/*309*/{0.226079,1.93748,-1.7478,0.26585,1.87272,-1.90861,0.201144,1.78988,-1.62552,0.177614,1.68146,-1.53673,0.298172,1.69013,-1.6406,0.379762,1.69035,-1.68194,0.143177,1.95113,-2.12202,0.254423,1.87703,-1.98321,0.035644,1.9132,-2.00684,-0.095687,1.83842,-2.16199,-0.119347,1.74806,-2.21555,0.017426,1.61838,-2.16618,0.05312,1.56514,-2.18749,0.159282,1.41387,-1.80596,-0.023526,1.50061,-1.88241,-0.021857,1.52618,-1.9288,-0.020706,1.5074,-2.00034,0.164364,1.45119,-2.06348,0.045053,1.10986,-1.77064,-0.085583,1.15029,-1.77947,-0.113283,1.07196,-1.80414,-0.104021,0.938433,-1.79809,-0.170664,0.758507,-1.79024,-0.204074,0.691524,-1.80078,-0.271,0.808315,-1.80824,-0.301965,0.743014,-1.81513,-0.368356,0.516273,-1.82612,-0.479536,0.521061,-1.91076,-0.273218,0.378701,-1.86962,-0.32652,0.351902,-1.80007,0.293004,1.24549,-2.11306,0.350908,1.18699,-2.09109,0.190312,1.14715,-2.09606,0.24683,1.09069,-2.07911,0.356967,1.02819,-2.07366,0.312837,0.842823,-2.08074,0.215548,0.898667,-2.07865,0.18411,0.834828,-2.07291,0.097179,0.612885,-2.05686,1.1e-005,0.560229,-1.98821,0.260803,0.485165,-2.06845,0.19169,0.468976,-2.12405,0.074115,1.3369,-1.83815,0.080456,1.36274,-2.05174},
/*310*/{0.227243,1.94192,-1.74926,0.266585,1.87748,-1.90971,0.202891,1.79409,-1.62623,0.179607,1.68513,-1.53621,0.299316,1.69445,-1.64264,0.38084,1.69438,-1.6852,0.141772,1.95409,-2.12197,0.254206,1.88027,-1.98458,0.035761,1.9152,-2.00691,-0.09719,1.84243,-2.15967,-0.12223,1.75198,-2.21303,0.016311,1.6229,-2.16551,0.051528,1.56919,-2.18761,0.16194,1.41595,-1.80663,-0.021404,1.50476,-1.88203,-0.020664,1.52964,-1.92836,-0.019454,1.51161,-2.00004,0.164701,1.45523,-2.06404,0.042011,1.1146,-1.7713,-0.086059,1.15564,-1.77834,-0.117329,1.07857,-1.80294,-0.110351,0.945199,-1.79799,-0.179897,0.766338,-1.78831,-0.212837,0.699088,-1.7985,-0.278832,0.815748,-1.80747,-0.309219,0.750799,-1.81389,-0.380328,0.523834,-1.82367,-0.488259,0.537519,-1.91057,-0.291092,0.380372,-1.86844,-0.347621,0.356229,-1.80173,0.294407,1.24931,-2.11164,0.350788,1.19033,-2.08988,0.190182,1.15134,-2.09693,0.246785,1.09414,-2.0807,0.356289,1.02974,-2.07414,0.321014,0.841775,-2.08128,0.22026,0.891637,-2.07911,0.192775,0.827347,-2.07311,0.117934,0.600671,-2.05725,0.023985,0.541779,-1.98944,0.28911,0.48369,-2.06836,0.221039,0.463469,-2.12442,0.075721,1.33991,-1.83905,0.080851,1.36682,-2.05253},
/*311*/{0.228504,1.94424,-1.75079,0.266691,1.8807,-1.91101,0.205075,1.79835,-1.62719,0.182738,1.68885,-1.53616,0.299837,1.69872,-1.64468,0.381328,1.69865,-1.68899,0.141288,1.95663,-2.12207,0.25453,1.88366,-1.98532,0.035953,1.91798,-2.00625,-0.097827,1.84662,-2.15813,-0.124103,1.75572,-2.21043,0.014932,1.62673,-2.16494,0.049933,1.57342,-2.18752,0.163405,1.41798,-1.80819,-0.020205,1.50826,-1.88082,-0.019185,1.53329,-1.92649,-0.017475,1.51646,-1.99932,0.165693,1.45837,-2.06451,0.036206,1.11798,-1.76892,-0.088445,1.16217,-1.77926,-0.120031,1.08444,-1.80247,-0.115575,0.951168,-1.79829,-0.186807,0.772923,-1.78685,-0.221504,0.706092,-1.79676,-0.285691,0.823307,-1.80701,-0.317177,0.759241,-1.81226,-0.39072,0.531621,-1.82186,-0.494626,0.552258,-1.90953,-0.307572,0.382423,-1.86785,-0.368383,0.360709,-1.80237,0.294887,1.25312,-2.11043,0.351,1.19242,-2.08882,0.189434,1.15467,-2.09765,0.245865,1.0968,-2.08143,0.354947,1.03085,-2.07537,0.329477,0.840826,-2.08158,0.225593,0.88515,-2.07928,0.201307,0.818446,-2.07295,0.137625,0.589553,-2.05703,0.046907,0.525593,-1.98978,0.316382,0.482879,-2.06825,0.250281,0.45914,-2.12465,0.076577,1.34242,-1.83994,0.080803,1.37097,-2.05323},
/*312*/{0.230068,1.94732,-1.75171,0.266909,1.8836,-1.91198,0.206577,1.80185,-1.62768,0.184018,1.69245,-1.53722,0.300841,1.70185,-1.64652,0.38159,1.70196,-1.69075,0.140348,1.95903,-2.12269,0.254309,1.88643,-1.98661,0.036676,1.91964,-2.00568,-0.09866,1.84951,-2.15501,-0.126151,1.75908,-2.20838,0.013761,1.63009,-2.16479,0.04881,1.57692,-2.18755,0.165016,1.42032,-1.80863,-0.017812,1.51144,-1.88044,-0.017976,1.53758,-1.92617,-0.016186,1.52048,-1.99863,0.165125,1.46136,-2.06465,0.036454,1.12027,-1.76665,-0.090249,1.16636,-1.77912,-0.122804,1.08994,-1.80362,-0.118491,0.956534,-1.7983,-0.193017,0.77937,-1.78602,-0.227958,0.713454,-1.79529,-0.291168,0.830367,-1.80613,-0.323269,0.767336,-1.81223,-0.400168,0.539203,-1.81901,-0.500611,0.568553,-1.90736,-0.327274,0.384906,-1.86726,-0.388131,0.366521,-1.80355,0.294696,1.25562,-2.10925,0.350401,1.19432,-2.08787,0.188826,1.15743,-2.09837,0.24427,1.09958,-2.08253,0.353454,1.03084,-2.07671,0.33709,0.838831,-2.08322,0.230857,0.877506,-2.07865,0.209935,0.810912,-2.07382,0.159799,0.579438,-2.05801,0.071039,0.508425,-1.99085,0.343537,0.482838,-2.06714,0.278656,0.455117,-2.12426,0.077379,1.34545,-1.8402,0.080433,1.37439,-2.05346},
/*313*/{0.23106,1.95021,-1.75337,0.268379,1.88663,-1.91324,0.208287,1.80562,-1.62844,0.186383,1.6958,-1.53704,0.30248,1.70481,-1.64836,0.381619,1.70489,-1.69406,0.14058,1.96141,-2.12331,0.255004,1.8888,-1.98724,0.036534,1.9218,-2.0059,-0.098804,1.85319,-2.15271,-0.127925,1.76259,-2.20524,0.01244,1.63335,-2.16449,0.047851,1.58049,-2.18826,0.166285,1.42296,-1.80908,-0.01647,1.51397,-1.88007,-0.017076,1.54058,-1.92528,-0.013452,1.52301,-1.99698,0.16636,1.46451,-2.06482,0.034498,1.1238,-1.76495,-0.092217,1.17035,-1.78227,-0.124245,1.09411,-1.80472,-0.120603,0.960733,-1.79928,-0.197497,0.785225,-1.78563,-0.23361,0.719333,-1.79353,-0.295651,0.837795,-1.80523,-0.328392,0.775207,-1.81025,-0.409552,0.547468,-1.8172,-0.506052,0.585306,-1.90432,-0.343829,0.389442,-1.86743,-0.408187,0.37335,-1.80469,0.294284,1.25597,-2.10842,0.349211,1.19465,-2.08731,0.187228,1.15937,-2.09859,0.242155,1.10063,-2.08329,0.351321,1.02978,-2.07773,0.343965,0.837113,-2.08389,0.236388,0.870413,-2.07846,0.218394,0.803235,-2.07458,0.178533,0.568136,-2.05845,0.094575,0.493384,-1.99241,0.369173,0.483329,-2.06525,0.306486,0.451439,-2.12312,0.079047,1.34788,-1.84029,0.081723,1.37757,-2.05345},
/*314*/{0.232809,1.95251,-1.75407,0.268557,1.88998,-1.9153,0.210012,1.80877,-1.62924,0.187834,1.69726,-1.53695,0.302553,1.70789,-1.64967,0.381539,1.70783,-1.69726,0.140034,1.96345,-2.12317,0.255789,1.89115,-1.98898,0.037365,1.92361,-2.00496,-0.099806,1.85503,-2.14963,-0.129297,1.76506,-2.20247,0.012251,1.63551,-2.164,0.047113,1.5836,-2.18839,0.167153,1.42535,-1.80896,-0.015795,1.5177,-1.88022,-0.0146,1.54349,-1.92471,-0.012873,1.52599,-1.99598,0.166877,1.46733,-2.0649,0.033508,1.12727,-1.76417,-0.093562,1.1733,-1.78271,-0.124528,1.09832,-1.80533,-0.121786,0.964537,-1.79957,-0.201495,0.790602,-1.78506,-0.238285,0.725038,-1.79319,-0.298319,0.844936,-1.80486,-0.333935,0.781355,-1.8086,-0.416999,0.55739,-1.81421,-0.511085,0.602313,-1.89932,-0.360763,0.395563,-1.86799,-0.425626,0.3814,-1.8068,0.293167,1.25639,-2.10773,0.348419,1.19321,-2.08677,0.185152,1.16012,-2.09892,0.239001,1.10095,-2.08368,0.349287,1.02866,-2.07864,0.349511,0.834783,-2.08529,0.240853,0.863824,-2.08026,0.226558,0.795288,-2.07514,0.199295,0.55906,-2.06017,0.117853,0.478642,-1.99435,0.393587,0.484601,-2.06316,0.334368,0.448312,-2.12129,0.079634,1.3507,-1.84013,0.081984,1.3805,-2.05328},
/*315*/{0.234035,1.95519,-1.75567,0.26838,1.89139,-1.9156,0.210789,1.81126,-1.6295,0.189488,1.70033,-1.53664,0.30326,1.70999,-1.65144,0.381439,1.70962,-1.69965,0.1398,1.96511,-2.12359,0.255865,1.89266,-1.98976,0.038023,1.92574,-2.00458,-0.100298,1.85729,-2.14817,-0.130665,1.76714,-2.20041,0.011462,1.63847,-2.16421,0.046729,1.58601,-2.1892,0.169008,1.42798,-1.81024,-0.01464,1.52137,-1.87913,-0.014426,1.54659,-1.92385,-0.011983,1.52937,-1.9954,0.167139,1.46918,-2.06464,0.032744,1.13049,-1.7641,-0.093796,1.17633,-1.78501,-0.124865,1.10132,-1.80746,-0.121961,0.967356,-1.80043,-0.205312,0.794386,-1.78566,-0.243797,0.729617,-1.79225,-0.300613,0.85054,-1.80363,-0.33674,0.789277,-1.80759,-0.424992,0.566975,-1.81281,-0.517177,0.61851,-1.89442,-0.376177,0.403267,-1.86971,-0.442896,0.389111,-1.8093,0.291446,1.2553,-2.10734,0.344764,1.19256,-2.08674,0.181946,1.16063,-2.09984,0.23627,1.10042,-2.08433,0.346263,1.02674,-2.07996,0.355287,0.832296,-2.08578,0.245299,0.856899,-2.08141,0.234407,0.7867,-2.07592,0.21972,0.549713,-2.06101,0.141725,0.467365,-1.99605,0.416709,0.486634,-2.06114,0.359963,0.445624,-2.11934,0.080552,1.3537,-1.83972,0.081527,1.38311,-2.05293},
/*316*/{0.234822,1.95688,-1.75654,0.269878,1.89416,-1.91715,0.211479,1.81418,-1.63039,0.190454,1.7023,-1.53718,0.303587,1.71168,-1.65367,0.381982,1.71092,-1.70177,0.139748,1.96682,-2.12357,0.256513,1.89448,-1.99067,0.038165,1.92684,-2.00471,-0.101039,1.85897,-2.14535,-0.131502,1.76891,-2.19839,0.010776,1.64059,-2.16443,0.046012,1.588,-2.18957,0.169661,1.43067,-1.81016,-0.013453,1.52387,-1.87809,-0.013706,1.54982,-1.92307,-0.011698,1.53244,-1.99416,0.167206,1.47116,-2.06445,0.032363,1.13218,-1.76392,-0.093803,1.17797,-1.78554,-0.124646,1.10291,-1.80781,-0.120946,0.969008,-1.80109,-0.206493,0.797747,-1.7848,-0.247497,0.73434,-1.79185,-0.301262,0.856665,-1.80185,-0.338884,0.796275,-1.80521,-0.431227,0.575782,-1.80827,-0.522923,0.633874,-1.887,-0.39258,0.412182,-1.87307,-0.45832,0.399296,-1.81179,0.289358,1.25391,-2.10759,0.34192,1.19006,-2.08701,0.179351,1.15981,-2.1002,0.232441,1.09947,-2.0854,0.343802,1.02454,-2.08136,0.36046,0.829777,-2.08678,0.249703,0.848796,-2.08317,0.241599,0.779315,-2.07727,0.238425,0.541293,-2.06216,0.162795,0.455147,-1.99861,0.439579,0.488368,-2.05787,0.38604,0.444378,-2.11667,0.081062,1.35643,-1.83899,0.081166,1.3856,-2.05224},
/*317*/{0.236751,1.95847,-1.75768,0.269446,1.89625,-1.91766,0.212943,1.81621,-1.63124,0.191689,1.70441,-1.53783,0.302916,1.71249,-1.65494,0.38123,1.71179,-1.70414,0.139639,1.96805,-2.12453,0.256915,1.89552,-1.99182,0.038948,1.92869,-2.00507,-0.101431,1.8594,-2.14275,-0.132315,1.77011,-2.19633,0.010911,1.64193,-2.16459,0.046473,1.5899,-2.19021,0.169955,1.4332,-1.8097,-0.01337,1.52637,-1.87812,-0.013014,1.55229,-1.92358,-0.011289,1.53452,-1.99397,0.167144,1.47319,-2.0637,0.032557,1.13506,-1.76318,-0.093626,1.17881,-1.78651,-0.123587,1.10388,-1.8092,-0.118382,0.970607,-1.80111,-0.207641,0.800358,-1.78525,-0.251104,0.73844,-1.79142,-0.299809,0.862787,-1.79935,-0.339381,0.80334,-1.80107,-0.437946,0.58588,-1.80766,-0.529783,0.649431,-1.88006,-0.409269,0.421731,-1.87717,-0.472066,0.40997,-1.81329,0.286436,1.2515,-2.10799,0.338689,1.18679,-2.08737,0.175697,1.15837,-2.10049,0.227253,1.09719,-2.08583,0.341912,1.02139,-2.08222,0.365995,0.827107,-2.08747,0.252845,0.842036,-2.08497,0.247173,0.772024,-2.07928,0.255584,0.532891,-2.06313,0.184649,0.44262,-1.99797,0.458919,0.49017,-2.05577,0.409617,0.441883,-2.11371,0.081151,1.35902,-1.83836,0.081462,1.38755,-2.0517},
/*318*/{0.237642,1.95977,-1.75841,0.270213,1.89761,-1.91908,0.213781,1.81795,-1.63179,0.192185,1.70553,-1.53826,0.303465,1.71387,-1.65699,0.38072,1.71227,-1.70672,0.140405,1.96925,-2.12511,0.25749,1.89687,-1.9928,0.040394,1.93018,-2.00509,-0.100642,1.86068,-2.14098,-0.132893,1.77079,-2.19411,0.011875,1.64405,-2.16525,0.046079,1.59107,-2.191,0.170068,1.43518,-1.80949,-0.012969,1.52805,-1.87708,-0.012649,1.55467,-1.92338,-0.010863,1.53629,-1.99319,0.168768,1.47501,-2.06318,0.034613,1.13638,-1.76385,-0.092271,1.17913,-1.78738,-0.120937,1.1039,-1.80955,-0.115041,0.970433,-1.80153,-0.207704,0.802094,-1.78563,-0.252471,0.741339,-1.79331,-0.298338,0.866748,-1.79814,-0.338396,0.809133,-1.80008,-0.445058,0.594826,-1.80561,-0.53837,0.663958,-1.87194,-0.423836,0.433334,-1.88156,-0.484633,0.421337,-1.81573,0.283751,1.24848,-2.10817,0.335585,1.18258,-2.08753,0.172169,1.15774,-2.10141,0.222563,1.09495,-2.08562,0.338951,1.018,-2.08352,0.369307,0.825045,-2.08828,0.256398,0.834986,-2.0878,0.253036,0.76446,-2.08122,0.274479,0.526586,-2.06377,0.20466,0.43403,-1.9993,0.477983,0.492315,-2.05193,0.431766,0.440938,-2.11067,0.081606,1.36088,-1.83789,0.082321,1.38959,-2.0512},
/*319*/{0.238221,1.96056,-1.75949,0.269751,1.89889,-1.91976,0.213422,1.81917,-1.63299,0.191044,1.70717,-1.53928,0.303189,1.71377,-1.65847,0.380709,1.71163,-1.70877,0.141019,1.96995,-2.12584,0.257473,1.89709,-1.99371,0.041257,1.931,-2.00545,-0.101125,1.86109,-2.13865,-0.133342,1.77144,-2.19221,0.01288,1.6447,-2.16528,0.046923,1.59199,-2.19166,0.170758,1.43753,-1.80896,-0.012394,1.52987,-1.87666,-0.013783,1.5567,-1.92322,-0.010785,1.5379,-1.99241,0.168497,1.47568,-2.06213,0.038054,1.1364,-1.76465,-0.089654,1.17913,-1.78795,-0.117487,1.10365,-1.81045,-0.109995,0.970085,-1.80272,-0.207054,0.803421,-1.78651,-0.254357,0.744754,-1.7941,-0.295396,0.871796,-1.79546,-0.338769,0.81617,-1.79653,-0.451474,0.605363,-1.80541,-0.546565,0.678073,-1.86377,-0.437849,0.445227,-1.88648,-0.495079,0.431877,-1.81798,0.281361,1.24435,-2.10824,0.331644,1.17806,-2.08812,0.167832,1.15538,-2.10147,0.217634,1.09333,-2.08639,0.335973,1.01427,-2.08434,0.373168,0.823682,-2.08818,0.258522,0.829017,-2.0917,0.260393,0.757915,-2.08549,0.291208,0.520554,-2.06432,0.223444,0.425171,-2.00053,0.495281,0.494209,-2.04853,0.452709,0.440007,-2.10698,0.081832,1.36317,-1.83645,0.082051,1.39055,-2.04994},
/*320*/{0.239811,1.96077,-1.7603,0.271421,1.8997,-1.92162,0.213875,1.8197,-1.634,0.191285,1.70774,-1.53971,0.302923,1.71257,-1.65774,0.378838,1.70935,-1.71005,0.141901,1.97061,-2.12662,0.257493,1.89749,-1.99491,0.041583,1.93153,-2.00616,-0.101203,1.86103,-2.13603,-0.13351,1.77167,-2.19075,0.013522,1.64533,-2.16574,0.047491,1.59251,-2.19253,0.170876,1.43981,-1.80911,-0.011903,1.53127,-1.87654,-0.014049,1.5587,-1.92245,-0.010404,1.53893,-1.99244,0.169307,1.47727,-2.06148,0.042511,1.1365,-1.7648,-0.085909,1.17941,-1.78812,-0.113591,1.10254,-1.81057,-0.104766,0.968327,-1.8036,-0.206063,0.803494,-1.78734,-0.257244,0.74715,-1.79506,-0.291605,0.876242,-1.79338,-0.337324,0.821572,-1.79448,-0.457439,0.61598,-1.80402,-0.554273,0.692183,-1.85526,-0.450836,0.458929,-1.89121,-0.504616,0.443905,-1.81961,0.277559,1.24011,-2.1085,0.327301,1.17426,-2.08888,0.163386,1.1527,-2.10161,0.211764,1.08971,-2.08639,0.331591,1.00937,-2.08502,0.37555,0.821943,-2.08836,0.261472,0.822533,-2.09402,0.266298,0.752413,-2.08762,0.306162,0.514836,-2.06453,0.241152,0.418904,-2.00195,0.510895,0.496504,-2.04647,0.472122,0.439746,-2.10446,0.082435,1.36502,-1.83596,0.082765,1.3921,-2.04948},
/*321*/{0.240498,1.96067,-1.76067,0.270877,1.89982,-1.92189,0.213793,1.82017,-1.63511,0.190856,1.70761,-1.54004,0.301876,1.71128,-1.659,0.377939,1.7081,-1.7123,0.142502,1.97036,-2.12759,0.258604,1.89829,-1.99544,0.042369,1.93289,-2.00639,-0.100713,1.86054,-2.13446,-0.133406,1.77154,-2.18892,0.013777,1.64532,-2.16584,0.048192,1.59261,-2.19309,0.171523,1.44135,-1.80868,-0.01203,1.53189,-1.87632,-0.013175,1.56001,-1.92246,-0.010192,1.53917,-1.9926,0.170432,1.47791,-2.06069,0.047199,1.13566,-1.76725,-0.080495,1.17862,-1.78747,-0.108927,1.09994,-1.81158,-0.09844,0.967055,-1.80392,-0.204752,0.803995,-1.78882,-0.257528,0.749358,-1.79704,-0.287248,0.880401,-1.79149,-0.335718,0.828885,-1.79361,-0.463329,0.62636,-1.80398,-0.561829,0.70594,-1.84744,-0.462505,0.473042,-1.89591,-0.513935,0.456124,-1.82267,0.274899,1.23633,-2.10911,0.32286,1.16837,-2.08955,0.15875,1.15052,-2.10197,0.207403,1.08644,-2.08659,0.328141,1.00505,-2.0856,0.378189,0.819529,-2.08717,0.265366,0.815401,-2.09656,0.271515,0.745644,-2.08955,0.319697,0.510654,-2.06514,0.258176,0.411457,-2.00166,0.525047,0.498342,-2.04343,0.489145,0.439883,-2.10076,0.083143,1.36626,-1.83519,0.083854,1.39264,-2.0488},
/*322*/{0.241234,1.96007,-1.76153,0.270413,1.89937,-1.92245,0.213221,1.82003,-1.63592,0.189432,1.70714,-1.54043,0.300821,1.70921,-1.66048,0.37688,1.70585,-1.71387,0.142972,1.97018,-2.1285,0.259275,1.89827,-1.99571,0.043878,1.933,-2.00752,-0.10099,1.86,-2.13238,-0.133383,1.7706,-2.18805,0.014912,1.64449,-2.16574,0.049331,1.59188,-2.19372,0.170632,1.44246,-1.80754,-0.011983,1.53192,-1.87636,-0.013744,1.56,-1.92303,-0.010305,1.53895,-1.99255,0.171259,1.47845,-2.05966,0.052355,1.13401,-1.76953,-0.077144,1.17594,-1.78714,-0.103345,1.09692,-1.81146,-0.091867,0.964041,-1.80522,-0.203017,0.804561,-1.78996,-0.258527,0.752914,-1.79809,-0.281862,0.88488,-1.78948,-0.333363,0.834937,-1.79137,-0.470199,0.638193,-1.80369,-0.56732,0.720292,-1.84021,-0.474666,0.48756,-1.89974,-0.52216,0.469194,-1.82516,0.271351,1.23137,-2.10974,0.319229,1.16337,-2.09042,0.153977,1.14827,-2.10093,0.201824,1.08253,-2.08638,0.324111,1.00019,-2.08605,0.381553,0.8177,-2.08613,0.268578,0.809312,-2.09957,0.278233,0.739819,-2.09163,0.332706,0.506153,-2.06547,0.271547,0.407634,-2.00361,0.538112,0.500018,-2.04029,0.504493,0.439874,-2.09833,0.083073,1.36688,-1.83439,0.084933,1.39276,-2.04806},
/*323*/{0.242297,1.95981,-1.76287,0.269789,1.89937,-1.92315,0.212565,1.81959,-1.63691,0.188357,1.7065,-1.54077,0.300205,1.7079,-1.66194,0.375826,1.70329,-1.71543,0.144109,1.96946,-2.12943,0.259355,1.89778,-1.99629,0.044629,1.93332,-2.00816,-0.099175,1.8589,-2.13047,-0.132553,1.76945,-2.18679,0.017214,1.64354,-2.16594,0.050223,1.59069,-2.19442,0.170254,1.44343,-1.80722,-0.01203,1.53189,-1.87632,-0.013744,1.56,-1.92303,-0.009764,1.53775,-1.99218,0.172033,1.4785,-2.05866,0.058135,1.13253,-1.77233,-0.072111,1.17336,-1.7875,-0.09725,1.09361,-1.81196,-0.08566,0.961167,-1.8067,-0.201741,0.804869,-1.79082,-0.259423,0.755851,-1.79715,-0.276838,0.889007,-1.78611,-0.329703,0.841012,-1.78973,-0.47448,0.649573,-1.80221,-0.571054,0.735367,-1.83348,-0.485076,0.503065,-1.90216,-0.531304,0.483323,-1.82728,0.268151,1.22662,-2.10959,0.314853,1.15791,-2.09159,0.149631,1.14524,-2.10026,0.196376,1.07831,-2.08646,0.319402,0.995712,-2.08655,0.383878,0.814969,-2.0837,0.272535,0.804053,-2.10255,0.28177,0.734528,-2.09469,0.343769,0.502068,-2.06517,0.284556,0.401317,-2.00325,0.548055,0.501118,-2.03853,0.516337,0.438819,-2.09539,0.083522,1.36728,-1.83347,0.085962,1.39241,-2.04722},
/*324*/{0.242842,1.959,-1.76335,0.270301,1.89966,-1.92375,0.211113,1.81921,-1.63731,0.186383,1.70574,-1.54206,0.29868,1.7053,-1.66314,0.374618,1.69965,-1.71676,0.144646,1.96863,-2.13019,0.259239,1.89731,-1.99661,0.045044,1.93291,-2.00853,-0.09797,1.8569,-2.12868,-0.131545,1.76772,-2.18633,0.018412,1.64205,-2.16609,0.051901,1.58914,-2.19473,0.170107,1.44371,-1.80694,-0.012101,1.53161,-1.87674,-0.013823,1.55979,-1.9236,-0.009545,1.53613,-1.99271,0.173123,1.47867,-2.05722,0.064396,1.13217,-1.77498,-0.065681,1.16932,-1.78671,-0.091483,1.08941,-1.8113,-0.077449,0.957547,-1.80815,-0.199466,0.805699,-1.79228,-0.259399,0.759493,-1.7993,-0.270855,0.89324,-1.78517,-0.326358,0.847498,-1.788,-0.478082,0.662248,-1.79977,-0.57364,0.751758,-1.82831,-0.493584,0.518352,-1.90424,-0.539928,0.497695,-1.82948,0.265449,1.22181,-2.11018,0.311862,1.15243,-2.0922,0.145598,1.14155,-2.10009,0.191851,1.0749,-2.08637,0.314561,0.990649,-2.08709,0.385755,0.812945,-2.08151,0.275467,0.798813,-2.10428,0.286938,0.729699,-2.0966,0.352236,0.498506,-2.06548,0.294206,0.397469,-2.00226,0.557002,0.501835,-2.03647,0.527563,0.438638,-2.09334,0.083879,1.36727,-1.83298,0.087454,1.39195,-2.04677},
/*325*/{0.242571,1.95758,-1.76389,0.269594,1.89869,-1.9241,0.209865,1.81811,-1.63866,0.183372,1.70467,-1.54367,0.297264,1.70277,-1.66393,0.372018,1.69582,-1.71745,0.145895,1.9675,-2.13079,0.259222,1.89649,-1.99716,0.04528,1.93191,-2.00931,-0.09683,1.8544,-2.12752,-0.130494,1.76552,-2.18594,0.019727,1.63981,-2.16653,0.053934,1.58692,-2.1952,0.170198,1.44393,-1.80554,-0.012646,1.53046,-1.87681,-0.014208,1.5582,-1.9233,-0.009864,1.53461,-1.99304,0.173951,1.47845,-2.05616,0.071618,1.12838,-1.77687,-0.061272,1.16274,-1.78619,-0.085424,1.08469,-1.81116,-0.069312,0.95325,-1.80955,-0.197261,0.807432,-1.79266,-0.258358,0.764123,-1.79896,-0.264568,0.89735,-1.784,-0.322257,0.854208,-1.78675,-0.481869,0.675378,-1.79862,-0.573929,0.767756,-1.82447,-0.501937,0.533928,-1.9045,-0.549839,0.512659,-1.83003,0.262587,1.21669,-2.11108,0.308098,1.14657,-2.09313,0.141751,1.13788,-2.09995,0.186719,1.07085,-2.08641,0.310369,0.986104,-2.08717,0.386694,0.810333,-2.07901,0.278095,0.794069,-2.10637,0.290856,0.725825,-2.0981,0.358923,0.495213,-2.06575,0.303963,0.394926,-2.00361,0.564005,0.50178,-2.03417,0.536639,0.43838,-2.09094,0.084517,1.36686,-1.83204,0.088913,1.39103,-2.04587},
/*326*/{0.2425,1.95606,-1.76488,0.26948,1.89666,-1.92416,0.208773,1.81722,-1.63967,0.182295,1.70331,-1.54457,0.29469,1.69871,-1.66491,0.370057,1.69107,-1.71843,0.145751,1.96621,-2.13199,0.259444,1.89562,-1.99753,0.046922,1.93108,-2.01058,-0.096472,1.85189,-2.12524,-0.128946,1.763,-2.18606,0.021247,1.63717,-2.16674,0.055754,1.58434,-2.19614,0.169786,1.44304,-1.80527,-0.012918,1.52862,-1.8775,-0.01367,1.55667,-1.9251,-0.009548,1.53306,-1.99448,0.175167,1.47797,-2.05537,0.077857,1.12599,-1.77939,-0.05612,1.15781,-1.78543,-0.078183,1.07944,-1.81181,-0.060902,0.948854,-1.81103,-0.193749,0.808926,-1.79351,-0.257831,0.767897,-1.79991,-0.258308,0.901406,-1.78376,-0.317007,0.861033,-1.78648,-0.485858,0.688214,-1.79637,-0.573046,0.785281,-1.82175,-0.509329,0.549611,-1.90449,-0.558289,0.529355,-1.83092,0.260415,1.21191,-2.11136,0.304623,1.14118,-2.09419,0.138457,1.13518,-2.09977,0.182784,1.06717,-2.08619,0.306268,0.981178,-2.08716,0.387146,0.808301,-2.07767,0.27892,0.789864,-2.10648,0.293642,0.721234,-2.09819,0.364878,0.492113,-2.0658,0.309169,0.390685,-2.00435,0.569318,0.500754,-2.03343,0.541401,0.437068,-2.09005,0.084624,1.36564,-1.83253,0.09074,1.38999,-2.0463},
/*327*/{0.241558,1.95452,-1.76577,0.270231,1.89551,-1.92454,0.206566,1.81571,-1.64054,0.178597,1.70158,-1.54617,0.293409,1.69531,-1.66559,0.368548,1.68607,-1.71887,0.14637,1.96419,-2.13257,0.259034,1.89392,-1.99767,0.046988,1.93015,-2.01267,-0.095572,1.85231,-2.12585,-0.127745,1.76018,-2.1863,0.024255,1.63438,-2.16716,0.057847,1.58153,-2.19677,0.169846,1.44239,-1.80487,-0.013147,1.52686,-1.87791,-0.013523,1.55471,-1.92585,-0.009364,1.5303,-1.9945,0.175068,1.47621,-2.05468,0.083596,1.12313,-1.78053,-0.050206,1.15216,-1.78487,-0.070817,1.07347,-1.81168,-0.051788,0.944355,-1.81246,-0.190678,0.811572,-1.79345,-0.255886,0.772124,-1.79957,-0.251524,0.905944,-1.7834,-0.311411,0.868107,-1.78537,-0.48664,0.702238,-1.79532,-0.570581,0.802331,-1.81986,-0.515727,0.564956,-1.90311,-0.56698,0.546409,-1.83058,0.258606,1.20713,-2.11235,0.302213,1.13522,-2.09482,0.135381,1.13197,-2.09983,0.179092,1.06373,-2.08569,0.301409,0.976759,-2.08729,0.386806,0.80622,-2.07618,0.278641,0.78586,-2.1056,0.293583,0.718279,-2.09801,0.368733,0.490286,-2.06487,0.314856,0.385803,-2.00546,0.573061,0.501204,-2.03179,0.546829,0.436896,-2.08959,0.08507,1.36451,-1.83171,0.091448,1.3877,-2.0456},
/*328*/{0.241722,1.95223,-1.76608,0.269396,1.89385,-1.92491,0.204019,1.81366,-1.64128,0.175713,1.69901,-1.54734,0.290216,1.69142,-1.66649,0.365077,1.68126,-1.7197,0.146428,1.96202,-2.13299,0.259296,1.8926,-1.99807,0.046581,1.92832,-2.0132,-0.093066,1.84629,-2.1237,-0.125676,1.75653,-2.18697,0.026682,1.63049,-2.16731,0.060234,1.57823,-2.19751,0.169896,1.44171,-1.80448,-0.013374,1.52474,-1.87853,-0.013092,1.55295,-1.92559,-0.009303,1.52786,-1.99527,0.176251,1.47496,-2.05447,0.088551,1.12164,-1.78243,-0.044721,1.14643,-1.78403,-0.06409,1.06792,-1.81193,-0.043318,0.939866,-1.81331,-0.186467,0.813589,-1.79312,-0.253815,0.777342,-1.79878,-0.243557,0.911011,-1.78343,-0.303311,0.874992,-1.78528,-0.487167,0.717494,-1.79322,-0.567005,0.819231,-1.81939,-0.52044,0.580045,-1.90121,-0.573551,0.563511,-1.82899,0.257023,1.20302,-2.1131,0.298628,1.13179,-2.0964,0.133627,1.12879,-2.09945,0.176513,1.06073,-2.08577,0.297293,0.972491,-2.08764,0.385023,0.80341,-2.07555,0.276805,0.782445,-2.10453,0.294484,0.715553,-2.09635,0.369706,0.487404,-2.0647,0.317572,0.386757,-2.00307,0.575471,0.500703,-2.03126,0.549789,0.436523,-2.08941,0.085825,1.36318,-1.83147,0.092885,1.38602,-2.04537},
/*329*/{0.240487,1.94994,-1.76654,0.26951,1.89188,-1.92498,0.202054,1.81107,-1.64187,0.172831,1.69734,-1.54898,0.287541,1.68717,-1.6671,0.362585,1.67573,-1.71957,0.147858,1.96013,-2.13339,0.258763,1.88999,-1.99814,0.047036,1.9267,-2.01415,-0.09054,1.84267,-2.12332,-0.124031,1.75287,-2.18814,0.028505,1.62618,-2.16781,0.062925,1.57438,-2.1981,0.170298,1.44057,-1.8046,-0.013329,1.52205,-1.87945,-0.012308,1.54941,-1.92713,-0.008601,1.52505,-1.99621,0.176601,1.47235,-2.05443,0.096038,1.1178,-1.78285,-0.039513,1.14141,-1.78348,-0.057464,1.06223,-1.81181,-0.035566,0.935584,-1.81462,-0.182502,0.816026,-1.79259,-0.249596,0.781879,-1.7989,-0.235013,0.915178,-1.78361,-0.29767,0.881796,-1.78392,-0.487237,0.730901,-1.78998,-0.56252,0.836746,-1.81978,-0.524294,0.594869,-1.89902,-0.579702,0.581733,-1.82724,0.255731,1.19824,-2.11449,0.296143,1.12624,-2.0981,0.131989,1.12657,-2.09971,0.173474,1.05716,-2.08552,0.293286,0.967986,-2.08757,0.38232,0.800764,-2.07597,0.273997,0.778515,-2.10311,0.291242,0.711216,-2.09464,0.370232,0.484377,-2.06352,0.319282,0.379644,-2.00224,0.575366,0.499551,-2.03179,0.549939,0.434298,-2.08929,0.086725,1.36131,-1.83131,0.09421,1.38298,-2.04532},
/*330*/{0.240104,1.94707,-1.76674,0.269982,1.88891,-1.92551,0.200119,1.80928,-1.643,0.169122,1.6951,-1.55126,0.284116,1.68269,-1.66769,0.359853,1.67002,-1.71989,0.148153,1.95749,-2.13438,0.258092,1.8872,-1.99765,0.046201,1.92524,-2.01613,-0.089328,1.83915,-2.12369,-0.121369,1.74842,-2.18893,0.030307,1.62262,-2.16881,0.065531,1.57029,-2.19898,0.170985,1.43952,-1.80459,-0.01331,1.519,-1.88016,-0.012682,1.54709,-1.92711,-0.008734,1.52225,-1.99685,0.176883,1.46994,-2.05491,0.10223,1.11584,-1.78389,-0.034607,1.13561,-1.78396,-0.050638,1.05651,-1.81275,-0.026991,0.931195,-1.8152,-0.176789,0.818869,-1.79181,-0.246726,0.787769,-1.79671,-0.226586,0.919579,-1.78417,-0.289827,0.889158,-1.78382,-0.485613,0.743998,-1.78683,-0.555856,0.853683,-1.82033,-0.527288,0.609108,-1.89596,-0.584582,0.599452,-1.82584,0.255432,1.19377,-2.11579,0.295826,1.12185,-2.09845,0.130292,1.12378,-2.09982,0.171433,1.05417,-2.08542,0.289881,0.963551,-2.08797,0.37812,0.797195,-2.07641,0.270323,0.774667,-2.10161,0.288063,0.70718,-2.09301,0.36845,0.48115,-2.06254,0.318607,0.375533,-2.0009,0.573973,0.496093,-2.03146,0.548605,0.432155,-2.08915,0.087711,1.35956,-1.83109,0.094902,1.38035,-2.0452},
/*331*/{0.239366,1.94578,-1.76683,0.268673,1.88694,-1.92558,0.196349,1.80631,-1.64325,0.164687,1.69154,-1.55159,0.280579,1.67763,-1.668,0.356278,1.66368,-1.71952,0.148988,1.9546,-2.13462,0.258461,1.88516,-1.9978,0.04769,1.92261,-2.01677,-0.087896,1.83454,-2.12413,-0.11893,1.74345,-2.19033,0.032697,1.6178,-2.16958,0.068862,1.56581,-2.19975,0.171393,1.43728,-1.80457,-0.013008,1.51617,-1.88133,-0.012824,1.5442,-1.92821,-0.008172,1.51893,-1.99739,0.17765,1.46717,-2.0559,0.107658,1.11185,-1.78371,-0.028286,1.13152,-1.78315,-0.044011,1.05054,-1.81207,-0.018471,0.926912,-1.81593,-0.171322,0.821014,-1.79107,-0.242208,0.792455,-1.79567,-0.21671,0.924251,-1.78517,-0.281586,0.896087,-1.78355,-0.483748,0.759181,-1.78923,-0.548483,0.870024,-1.82218,-0.528746,0.623364,-1.89239,-0.587571,0.61764,-1.82333,0.254925,1.18977,-2.11621,0.295405,1.11634,-2.09906,0.129447,1.12155,-2.09978,0.169406,1.05134,-2.08569,0.287666,0.959713,-2.08806,0.3743,0.793798,-2.07698,0.26608,0.770489,-2.1005,0.284328,0.702355,-2.09147,0.366755,0.477653,-2.06064,0.316639,0.373022,-1.99748,0.571618,0.4926,-2.03138,0.546069,0.428712,-2.08924,0.088361,1.35711,-1.83142,0.09582,1.37734,-2.04558},
/*332*/{0.237994,1.94262,-1.76682,0.268099,1.88409,-1.92545,0.194371,1.80319,-1.64345,0.160146,1.6886,-1.55296,0.276416,1.67271,-1.66757,0.352942,1.65681,-1.71889,0.14878,1.95157,-2.13505,0.258347,1.88233,-1.99746,0.046783,1.9196,-2.0176,-0.085733,1.8293,-2.12727,-0.116783,1.73814,-2.19197,0.036348,1.61372,-2.17101,0.071379,1.56071,-2.20044,0.171615,1.4348,-1.80451,-0.013238,1.51309,-1.88184,-0.01245,1.54092,-1.92845,-0.008453,1.51541,-1.99805,0.177142,1.46402,-2.05637,0.112369,1.10875,-1.78367,-0.024966,1.12439,-1.7848,-0.037348,1.04446,-1.81246,-0.009816,0.922747,-1.81565,-0.165316,0.823252,-1.79086,-0.23743,0.795669,-1.79479,-0.206463,0.92783,-1.7847,-0.273651,0.901907,-1.78462,-0.479709,0.771841,-1.78821,-0.539038,0.886132,-1.82394,-0.52945,0.636828,-1.889,-0.589194,0.634799,-1.82051,0.254556,1.18628,-2.11721,0.293261,1.11292,-2.10073,0.128304,1.11921,-2.09926,0.167733,1.04908,-2.08593,0.285814,0.955955,-2.08797,0.369927,0.789389,-2.07773,0.26116,0.765532,-2.09919,0.2796,0.69852,-2.09012,0.362508,0.473821,-2.05887,0.312866,0.36942,-1.99538,0.568182,0.488849,-2.03221,0.541493,0.424123,-2.08846,0.088653,1.35441,-1.83134,0.09583,1.37391,-2.04557},
/*333*/{0.236114,1.93893,-1.76654,0.268108,1.8809,-1.92546,0.191278,1.79995,-1.64397,0.15526,1.68518,-1.55481,0.272358,1.66669,-1.6682,0.348445,1.64974,-1.71828,0.149084,1.94819,-2.13543,0.258119,1.87911,-1.99766,0.046255,1.91739,-2.01965,-0.085801,1.82283,-2.12807,-0.114627,1.73236,-2.19355,0.039236,1.60828,-2.17241,0.074325,1.55552,-2.20155,0.171675,1.43226,-1.80473,-0.013128,1.50981,-1.88187,-0.0127,1.53704,-1.92885,-0.008652,1.51195,-1.99878,0.177331,1.46131,-2.0568,0.117374,1.10583,-1.78327,-0.020138,1.11913,-1.7848,-0.031465,1.03876,-1.81381,-0.002266,0.918857,-1.81538,-0.159151,0.825614,-1.78966,-0.231191,0.802039,-1.79277,-0.196242,0.932028,-1.7859,-0.263446,0.907555,-1.78519,-0.474786,0.784941,-1.78634,-0.529235,0.900344,-1.82629,-0.529334,0.649849,-1.88565,-0.589192,0.65228,-1.81727,0.253996,1.18286,-2.11836,0.291986,1.10849,-2.10218,0.127343,1.11713,-2.10014,0.166035,1.04652,-2.08603,0.285138,0.952511,-2.08738,0.365109,0.784635,-2.07901,0.256375,0.761847,-2.09809,0.274448,0.694395,-2.0884,0.355978,0.467466,-2.057,0.305962,0.364931,-1.9923,0.562897,0.482669,-2.03163,0.535402,0.417156,-2.08808,0.08927,1.35145,-1.83157,0.096388,1.37079,-2.04582},
/*334*/{0.235658,1.9357,-1.76587,0.268567,1.87732,-1.92549,0.187809,1.79596,-1.64396,0.149524,1.68141,-1.55634,0.267571,1.66117,-1.66809,0.344431,1.64234,-1.71721,0.148774,1.94448,-2.1355,0.258171,1.87559,-1.99797,0.045936,1.91329,-2.02041,-0.083997,1.81787,-2.12869,-0.111729,1.72633,-2.19569,0.041157,1.6024,-2.17392,0.07758,1.54961,-2.20262,0.171852,1.4295,-1.80511,-0.013907,1.50567,-1.88275,-0.013179,1.53328,-1.9295,-0.00856,1.50793,-1.99894,0.177306,1.45874,-2.05762,0.122553,1.10274,-1.78279,-0.014373,1.11264,-1.7843,-0.024391,1.03306,-1.81365,0.006408,0.914939,-1.81491,-0.151739,0.827427,-1.78861,-0.224955,0.80586,-1.79103,-0.186906,0.934279,-1.78788,-0.254228,0.912821,-1.78512,-0.468177,0.798232,-1.78425,-0.518842,0.914532,-1.82814,-0.527886,0.662896,-1.88148,-0.58895,0.66898,-1.81399,0.253649,1.17963,-2.1189,0.291157,1.10491,-2.10236,0.126322,1.11489,-2.09944,0.164749,1.04347,-2.08622,0.284902,0.949973,-2.08712,0.360969,0.779154,-2.07949,0.251411,0.757486,-2.09627,0.268815,0.689812,-2.08689,0.350172,0.463476,-2.05571,0.29929,0.361598,-1.98785,0.555633,0.474496,-2.03319,0.527894,0.409514,-2.08624,0.089965,1.34811,-1.83223,0.097086,1.36765,-2.04645},
/*335*/{0.233833,1.93171,-1.76574,0.267044,1.87347,-1.92451,0.183903,1.79217,-1.64414,0.144493,1.67775,-1.55796,0.262754,1.65496,-1.66834,0.340225,1.63449,-1.71624,0.149312,1.94069,-2.13623,0.258143,1.87184,-1.99829,0.045173,1.90994,-2.02207,-0.083078,1.81249,-2.12882,-0.109199,1.71939,-2.19811,0.044281,1.59676,-2.17567,0.080609,1.54334,-2.20387,0.171971,1.42649,-1.80556,-0.013119,1.50116,-1.88274,-0.011957,1.52944,-1.92999,-0.009309,1.50341,-1.99868,0.176968,1.45581,-2.05841,0.127262,1.10017,-1.78194,-0.008644,1.10867,-1.78545,-0.018029,1.0267,-1.81313,0.014485,0.910953,-1.81385,-0.144659,0.828908,-1.78849,-0.218063,0.809585,-1.79145,-0.17626,0.936805,-1.78868,-0.244728,0.917405,-1.78586,-0.461495,0.810973,-1.78232,-0.507591,0.927565,-1.8309,-0.524884,0.674898,-1.87764,-0.584706,0.684149,-1.81034,0.253232,1.17623,-2.11953,0.290175,1.10195,-2.10369,0.125121,1.11227,-2.09947,0.163129,1.04005,-2.08624,0.284481,0.946561,-2.08631,0.356886,0.773858,-2.07984,0.246091,0.753198,-2.09476,0.263174,0.685515,-2.08538,0.341257,0.457106,-2.05376,0.290488,0.358008,-1.98425,0.548289,0.467347,-2.03208,0.518939,0.400993,-2.08508,0.091163,1.34451,-1.83257,0.097515,1.36402,-2.04682},
/*336*/{0.23179,1.92785,-1.76517,0.267153,1.86952,-1.92473,0.18093,1.78777,-1.64489,0.138914,1.67396,-1.55981,0.258259,1.64879,-1.66826,0.335328,1.62606,-1.71483,0.149437,1.93646,-2.13646,0.257573,1.86786,-1.99757,0.044703,1.90533,-2.0236,-0.081753,1.80393,-2.13335,-0.107313,1.71244,-2.20051,0.046988,1.58996,-2.17742,0.083709,1.53668,-2.20512,0.173219,1.42371,-1.80641,-0.012802,1.49672,-1.88305,-0.012612,1.52536,-1.92994,-0.009966,1.49881,-1.9998,0.176351,1.4524,-2.05919,0.131137,1.09697,-1.7809,-0.006429,1.10443,-1.7874,-0.011425,1.02159,-1.81435,0.023064,0.906826,-1.8124,-0.137023,0.830371,-1.787,-0.211047,0.813163,-1.79065,-0.165298,0.939702,-1.78853,-0.23408,0.922028,-1.78664,-0.453633,0.822969,-1.78009,-0.496957,0.939366,-1.83383,-0.521859,0.686331,-1.87293,-0.580664,0.699341,-1.80612,0.252563,1.1724,-2.12014,0.288913,1.09867,-2.10394,0.124186,1.10912,-2.0999,0.162228,1.03732,-2.0868,0.282956,0.943254,-2.08599,0.352573,0.768285,-2.08033,0.241,0.749046,-2.09413,0.257006,0.681054,-2.08408,0.332613,0.452155,-2.05198,0.279651,0.354109,-1.98075,0.539604,0.457323,-2.03172,0.508043,0.390388,-2.08306,0.092601,1.34116,-1.83291,0.097766,1.36009,-2.04725},
/*337*/{0.231363,1.92391,-1.76445,0.267481,1.86504,-1.92451,0.177138,1.78329,-1.64455,0.132707,1.66936,-1.56086,0.252473,1.64243,-1.66825,0.330582,1.61772,-1.71344,0.149418,1.93177,-2.13709,0.258312,1.86316,-1.99727,0.045266,1.90077,-2.02486,-0.081999,1.79691,-2.13674,-0.104648,1.70463,-2.20319,0.050196,1.58302,-2.17911,0.086928,1.53,-2.20644,0.173563,1.42043,-1.80678,-0.012576,1.49165,-1.88249,-0.012788,1.52079,-1.92968,-0.010786,1.49418,-1.99956,0.176452,1.4491,-2.06051,0.136309,1.09201,-1.77969,-0.000749,1.09622,-1.78805,-0.005025,1.01575,-1.8134,0.031455,0.902686,-1.81155,-0.129804,0.83167,-1.78547,-0.203696,0.817298,-1.78797,-0.153466,0.941635,-1.78924,-0.223155,0.926272,-1.78766,-0.44535,0.834709,-1.78509,-0.485907,0.950352,-1.83696,-0.516254,0.695953,-1.86942,-0.57487,0.7133,-1.80201,0.250983,1.16871,-2.12089,0.288038,1.09526,-2.10503,0.123318,1.10594,-2.09996,0.161396,1.03392,-2.08668,0.280751,0.938899,-2.08561,0.346568,0.761425,-2.07957,0.236245,0.744689,-2.09269,0.250857,0.676635,-2.08299,0.322063,0.44626,-2.05085,0.268201,0.350497,-1.97882,0.531372,0.447649,-2.02924,0.497561,0.381237,-2.08087,0.093969,1.33712,-1.83328,0.098138,1.35628,-2.04762},
/*338*/{0.229465,1.91913,-1.76407,0.267357,1.86013,-1.92387,0.173642,1.77898,-1.64496,0.127019,1.66551,-1.56255,0.246788,1.63548,-1.66815,0.325433,1.60917,-1.71246,0.149466,1.92677,-2.13772,0.258076,1.85915,-1.99719,0.044181,1.89516,-2.02673,-0.079658,1.78876,-2.1399,-0.102192,1.69679,-2.20567,0.053382,1.57568,-2.18168,0.089837,1.52249,-2.20706,0.173244,1.41676,-1.80759,-0.013763,1.48749,-1.88357,-0.012557,1.51538,-1.92973,-0.011686,1.48885,-1.9993,0.175996,1.44569,-2.0616,0.140689,1.08958,-1.77851,0.003054,1.09026,-1.78921,0.00143,1.00933,-1.81417,0.039275,0.898208,-1.8101,-0.120866,0.832938,-1.78439,-0.195498,0.819897,-1.78687,-0.142411,0.942776,-1.79213,-0.21288,0.929133,-1.78858,-0.43643,0.845003,-1.78292,-0.473558,0.960311,-1.8405,-0.51035,0.706001,-1.86526,-0.569088,0.72598,-1.79868,0.251201,1.16543,-2.12188,0.287524,1.09126,-2.10533,0.122554,1.10217,-2.09962,0.160779,1.02978,-2.08646,0.278675,0.93377,-2.08602,0.342943,0.755548,-2.07993,0.23065,0.740425,-2.09266,0.244011,0.671886,-2.08256,0.308884,0.439266,-2.04857,0.255201,0.34893,-1.97653,0.519051,0.437177,-2.02805,0.484834,0.371491,-2.0785,0.094511,1.33293,-1.83389,0.098218,1.35212,-2.04824},
/*339*/{0.22749,1.91459,-1.76325,0.267548,1.85485,-1.92356,0.169652,1.77355,-1.64503,0.121528,1.66147,-1.56454,0.24172,1.62861,-1.66778,0.319493,1.60032,-1.71056,0.149827,1.92199,-2.13824,0.257573,1.85389,-1.99716,0.044179,1.88935,-2.02716,-0.079294,1.78038,-2.1432,-0.098996,1.68845,-2.20879,0.056335,1.5682,-2.18355,0.094121,1.51551,-2.2081,0.174851,1.41301,-1.80909,-0.013865,1.48187,-1.88358,-0.013003,1.51063,-1.93052,-0.012143,1.48371,-1.99974,0.175458,1.44185,-2.06292,0.144455,1.08628,-1.7768,0.007308,1.08488,-1.7902,0.00778,1.00424,-1.81639,0.047227,0.893265,-1.80759,-0.112116,0.833577,-1.78263,-0.186752,0.822382,-1.7854,-0.130896,0.943418,-1.79333,-0.201819,0.932125,-1.79159,-0.42617,0.854253,-1.78498,-0.461321,0.969453,-1.84365,-0.50285,0.714409,-1.86192,-0.56122,0.738398,-1.79462,0.249493,1.16195,-2.12337,0.286227,1.08765,-2.10643,0.121571,1.09772,-2.09961,0.159527,1.02614,-2.08709,0.276105,0.928038,-2.08678,0.337331,0.749319,-2.07955,0.22567,0.735628,-2.09221,0.236948,0.666877,-2.08215,0.300565,0.437463,-2.04739,0.237078,0.353978,-1.97703,0.507059,0.426328,-2.02555,0.469027,0.361201,-2.07635,0.09624,1.32859,-1.83489,0.098504,1.34778,-2.04925},
/*340*/{0.225944,1.90861,-1.76266,0.266463,1.84946,-1.92245,0.165587,1.7688,-1.64542,0.113927,1.65745,-1.56638,0.234971,1.62071,-1.6675,0.314357,1.59131,-1.70906,0.149809,1.91649,-2.1385,0.257599,1.84868,-1.99641,0.042993,1.88404,-2.02941,-0.077784,1.77284,-2.14652,-0.096135,1.67999,-2.21212,0.059728,1.55986,-2.18617,0.098143,1.50688,-2.20883,0.174594,1.40918,-1.81038,-0.014291,1.47634,-1.88333,-0.013027,1.50521,-1.93084,-0.012185,1.47747,-1.99857,0.175374,1.43827,-2.06454,0.146679,1.08318,-1.77542,0.012446,1.08012,-1.79157,0.012919,0.997001,-1.81624,0.056577,0.890607,-1.8057,-0.103506,0.834033,-1.78162,-0.178357,0.824913,-1.7853,-0.118647,0.943914,-1.79474,-0.191067,0.934655,-1.79331,-0.415056,0.862099,-1.78358,-0.450174,0.976811,-1.84667,-0.494474,0.721973,-1.85714,-0.550814,0.748512,-1.79141,0.247275,1.15745,-2.12367,0.286255,1.08268,-2.10654,0.120251,1.09218,-2.09948,0.159097,1.02165,-2.08707,0.27375,0.92161,-2.08685,0.332482,0.742204,-2.07878,0.219111,0.730778,-2.09191,0.229683,0.662096,-2.08181,0.287648,0.432658,-2.04632,0.218923,0.353552,-1.97693,0.492683,0.411426,-2.02337,0.450372,0.350178,-2.07349,0.097711,1.32385,-1.83556,0.098974,1.34334,-2.04991},
/*341*/{0.224509,1.90351,-1.76173,0.265967,1.84329,-1.92208,0.161285,1.76335,-1.64533,0.106359,1.65269,-1.56739,0.229334,1.61402,-1.66741,0.30729,1.58167,-1.70741,0.150626,1.91093,-2.13918,0.257436,1.84381,-1.99611,0.043181,1.87734,-2.02988,-0.07607,1.76338,-2.1513,-0.092955,1.67138,-2.21542,0.063563,1.55173,-2.18769,0.102337,1.49915,-2.20994,0.174681,1.40467,-1.81103,-0.014311,1.47052,-1.88399,-0.013915,1.4999,-1.93074,-0.013399,1.47177,-1.99806,0.174953,1.43398,-2.06625,0.149073,1.07912,-1.7736,0.016385,1.07431,-1.79277,0.018748,0.990829,-1.8166,0.06684,0.886501,-1.80364,-0.093214,0.834043,-1.77973,-0.168831,0.826906,-1.78378,-0.107247,0.944637,-1.79621,-0.177363,0.936902,-1.79355,-0.405563,0.871776,-1.78526,-0.437841,0.983924,-1.85036,-0.484538,0.729035,-1.853,-0.541365,0.757764,-1.78697,0.24612,1.1533,-2.12457,0.285119,1.07884,-2.10791,0.119412,1.08754,-2.09972,0.159125,1.01666,-2.08719,0.271553,0.916135,-2.08734,0.325865,0.735217,-2.07824,0.213413,0.726388,-2.09178,0.222521,0.656753,-2.08159,0.278118,0.429384,-2.04418,0.199888,0.356182,-1.97444,0.478321,0.395527,-2.01997,0.431834,0.339691,-2.0722,0.098556,1.31884,-1.83636,0.098757,1.33858,-2.05069},
/*342*/{0.223639,1.89789,-1.76039,0.266536,1.83927,-1.9208,0.157387,1.75763,-1.64552,0.100676,1.64868,-1.56834,0.223,1.60707,-1.66614,0.302319,1.57304,-1.70482,0.151448,1.90514,-2.13995,0.258467,1.83842,-1.99517,0.041948,1.87174,-2.03233,-0.075989,1.75448,-2.15543,-0.089384,1.66195,-2.21901,0.06869,1.54322,-2.18919,0.106745,1.49093,-2.21035,0.175561,1.4003,-1.81224,-0.014511,1.46459,-1.88399,-0.014178,1.49357,-1.93169,-0.014629,1.46633,-1.99815,0.174762,1.43001,-2.06801,0.153922,1.07529,-1.77042,0.021118,1.06763,-1.79349,0.025382,0.984547,-1.81759,0.075292,0.883218,-1.802,-0.083291,0.834824,-1.77709,-0.158802,0.829563,-1.78332,-0.093907,0.944556,-1.79769,-0.165636,0.939034,-1.79548,-0.392863,0.878615,-1.78483,-0.426046,0.989809,-1.85372,-0.473624,0.735615,-1.84966,-0.530316,0.765722,-1.78421,0.244453,1.14865,-2.12571,0.283762,1.07447,-2.1096,0.117998,1.08259,-2.10075,0.157023,1.01156,-2.0872,0.267569,0.91038,-2.08765,0.320139,0.727857,-2.07679,0.207359,0.721451,-2.09128,0.215094,0.651891,-2.08077,0.258341,0.424761,-2.03913,0.176936,0.354826,-1.97492,0.462178,0.385481,-2.00869,0.411004,0.342769,-2.07212,0.100137,1.31382,-1.83758,0.099131,1.33393,-2.05187},
/*343*/{0.221863,1.89248,-1.75979,0.266759,1.83237,-1.92013,0.152455,1.75161,-1.64593,0.093189,1.64447,-1.56984,0.216523,1.59963,-1.66541,0.294678,1.56354,-1.70317,0.152545,1.89978,-2.14073,0.258705,1.83294,-1.99511,0.041804,1.86475,-2.03249,-0.072829,1.74541,-2.15924,-0.085425,1.65319,-2.22252,0.072549,1.5344,-2.19145,0.111268,1.48181,-2.21106,0.175989,1.39546,-1.81388,-0.015507,1.45805,-1.88342,-0.014177,1.48794,-1.93128,-0.016372,1.46101,-1.99811,0.173636,1.42567,-2.07014,0.159955,1.07113,-1.76682,0.025306,1.05892,-1.79458,0.032011,0.977266,-1.81719,0.085714,0.880134,-1.79894,-0.072545,0.835098,-1.77629,-0.148018,0.829294,-1.78166,-0.081543,0.944719,-1.79884,-0.15316,0.939698,-1.79629,-0.382495,0.88424,-1.78425,-0.41427,0.994633,-1.85688,-0.463133,0.740407,-1.84628,-0.518601,0.772885,-1.78168,0.241565,1.14424,-2.12822,0.281146,1.06979,-2.11228,0.115797,1.07855,-2.10155,0.155273,1.00755,-2.08725,0.264437,0.905427,-2.08857,0.312946,0.720308,-2.07312,0.20113,0.71747,-2.09076,0.20687,0.647269,-2.07936,0.240344,0.424656,-2.03468,0.153786,0.355271,-1.97491,0.435075,0.385909,-2.01215,0.387362,0.341846,-2.07254,0.101112,1.30833,-1.83899,0.098227,1.32926,-2.05319},
/*344*/{0.220459,1.88653,-1.75894,0.266413,1.82643,-1.9191,0.148347,1.74622,-1.64624,0.085714,1.63982,-1.57076,0.209353,1.59214,-1.66554,0.287927,1.55454,-1.70055,0.153151,1.894,-2.14153,0.259239,1.82745,-1.99376,0.041783,1.85801,-2.03431,-0.071765,1.73578,-2.16307,-0.081056,1.64336,-2.22586,0.077439,1.52683,-2.19337,0.116302,1.47327,-2.21173,0.175622,1.39075,-1.81501,-0.015711,1.45209,-1.88356,-0.015894,1.48189,-1.93189,-0.018801,1.45632,-1.99783,0.17188,1.42035,-2.07193,0.164556,1.06617,-1.76544,0.029309,1.05423,-1.7957,0.039856,0.970418,-1.81866,0.094765,0.876363,-1.79704,-0.061419,0.835139,-1.7739,-0.135042,0.83105,-1.78032,-0.066981,0.944515,-1.79842,-0.139494,0.94121,-1.79853,-0.369192,0.889428,-1.78488,-0.401845,0.998334,-1.85963,-0.450784,0.744398,-1.84211,-0.506757,0.778706,-1.77986,0.237793,1.14034,-2.1328,0.277015,1.06616,-2.11675,0.112339,1.07531,-2.10151,0.152612,1.00459,-2.08695,0.259181,0.899167,-2.08967,0.308355,0.713664,-2.06838,0.197348,0.715051,-2.08994,0.197439,0.645683,-2.07679,0.215305,0.426026,-2.03034,0.129792,0.354826,-1.97168,0.409725,0.386381,-2.00814,0.36351,0.342655,-2.07209,0.100998,1.30317,-1.83989,0.096519,1.32398,-2.05407},
/*345*/{0.218918,1.88081,-1.75771,0.265831,1.82003,-1.91707,0.143874,1.74055,-1.64642,0.078636,1.63643,-1.57294,0.202307,1.58463,-1.66443,0.280109,1.54558,-1.69807,0.154956,1.88842,-2.14285,0.259301,1.82144,-1.99361,0.041146,1.8524,-2.03597,-0.068043,1.7267,-2.16832,-0.075841,1.63361,-2.2298,0.082664,1.51808,-2.19511,0.121995,1.46488,-2.21233,0.176087,1.38534,-1.8164,-0.018248,1.44686,-1.8832,-0.017789,1.47559,-1.93159,-0.021047,1.45092,-1.99766,0.16936,1.41561,-2.07354,0.167698,1.06206,-1.76538,0.034188,1.04487,-1.79606,0.047113,0.963072,-1.81875,0.104431,0.872056,-1.79551,-0.05013,0.835282,-1.77539,-0.124589,0.830384,-1.77642,-0.054764,0.943811,-1.80018,-0.126364,0.941486,-1.7984,-0.355149,0.893332,-1.78497,-0.389602,1.00105,-1.86243,-0.437082,0.74821,-1.8388,-0.493452,0.783769,-1.77689,0.23433,1.13671,-2.13885,0.273119,1.06274,-2.122,0.109131,1.07334,-2.10041,0.149549,1.00279,-2.08597,0.253671,0.893944,-2.0903,0.302942,0.710344,-2.06389,0.192519,0.717369,-2.08751,0.188848,0.648733,-2.07299,0.189291,0.427034,-2.02662,0.10714,0.357338,-1.97213,0.389591,0.382275,-2.00839,0.340384,0.342349,-2.07121,0.100723,1.29776,-1.84095,0.094295,1.31894,-2.05504},
/*346*/{0.217873,1.87509,-1.75643,0.26726,1.81413,-1.9159,0.139276,1.73509,-1.64688,0.071549,1.63314,-1.57495,0.195211,1.57792,-1.6631,0.273236,1.53837,-1.69571,0.156525,1.88279,-2.14408,0.261174,1.81582,-1.99262,0.040805,1.84827,-2.03745,-0.066066,1.71664,-2.1727,-0.070682,1.62399,-2.23327,0.088602,1.50891,-2.19731,0.128147,1.45658,-2.21261,0.17545,1.38036,-1.81829,-0.019652,1.44037,-1.88297,-0.02005,1.46986,-1.9306,-0.024504,1.4467,-1.99774,0.167085,1.41268,-2.07505,0.170923,1.05866,-1.76408,0.040417,1.03959,-1.79677,0.053582,0.956745,-1.81913,0.115676,0.86887,-1.79227,-0.037341,0.834155,-1.77442,-0.11218,0.832836,-1.77709,-0.040606,0.942467,-1.80147,-0.111464,0.942176,-1.79875,-0.343751,0.897331,-1.785,-0.376733,1.00321,-1.86446,-0.422901,0.750234,-1.83517,-0.479594,0.786346,-1.77401,0.23068,1.13327,-2.1437,0.270303,1.0592,-2.12609,0.106761,1.07269,-2.09888,0.146673,1.00151,-2.08577,0.24906,0.890712,-2.09138,0.295142,0.708861,-2.06094,0.185366,0.721664,-2.08639,0.178834,0.653342,-2.07227,0.17545,0.430433,-2.02597,0.085751,0.357835,-1.97061,0.367123,0.38252,-2.00597,0.318006,0.342514,-2.07017,0.100576,1.29222,-1.84316,0.092359,1.31558,-2.05697},
/*347*/{0.216782,1.86995,-1.75493,0.267484,1.80692,-1.91453,0.134905,1.72999,-1.64745,0.063737,1.62988,-1.57604,0.18789,1.57147,-1.66183,0.265047,1.53023,-1.69334,0.158244,1.87811,-2.14531,0.261221,1.80991,-1.99236,0.042545,1.84577,-2.0371,-0.062077,1.70739,-2.17854,-0.064405,1.61402,-2.23646,0.094766,1.50026,-2.19914,0.13448,1.44816,-2.21299,0.174397,1.37523,-1.82062,-0.02173,1.43546,-1.88255,-0.022255,1.46374,-1.92954,-0.027302,1.44229,-1.99691,0.164921,1.40982,-2.07701,0.178091,1.05722,-1.76268,0.046601,1.03275,-1.79636,0.061593,0.95021,-1.81909,0.126892,0.864387,-1.79204,-0.024068,0.832824,-1.7737,-0.098293,0.833138,-1.77531,-0.026063,0.941391,-1.80202,-0.09869,0.940946,-1.79916,-0.329123,0.899308,-1.78572,-0.364228,1.00368,-1.86586,-0.407983,0.751419,-1.83293,-0.464237,0.788075,-1.77199,0.227496,1.12992,-2.14634,0.267603,1.05651,-2.1285,0.104732,1.0713,-2.09852,0.144503,0.999929,-2.0841,0.249119,0.889987,-2.09219,0.28585,0.707603,-2.06032,0.176436,0.726445,-2.0855,0.166345,0.658141,-2.07174,0.156394,0.430924,-2.02473,0.062474,0.35852,-1.96894,0.346344,0.380976,-2.00601,0.295491,0.342417,-2.06933,0.100113,1.28685,-1.84528,0.089985,1.31224,-2.05878},
/*348*/{0.215105,1.8645,-1.75346,0.267689,1.80227,-1.91355,0.130472,1.72563,-1.6475,0.054995,1.62719,-1.57828,0.181592,1.56579,-1.66111,0.25672,1.52333,-1.69127,0.159715,1.87299,-2.14689,0.262785,1.80504,-1.99089,0.043568,1.84309,-2.03731,-0.054835,1.69885,-2.18504,-0.057779,1.60447,-2.24002,0.101156,1.49172,-2.20024,0.14228,1.44085,-2.21313,0.173603,1.3708,-1.82189,-0.024002,1.43083,-1.88108,-0.025216,1.46003,-1.93018,-0.030192,1.43881,-1.99573,0.16277,1.4064,-2.07927,0.184081,1.05557,-1.76156,0.050667,1.02643,-1.7978,0.071117,0.945057,-1.81894,0.139083,0.861889,-1.78952,-0.010603,0.831506,-1.77315,-0.086152,0.832011,-1.77554,-0.01237,0.940759,-1.80222,-0.083386,0.94029,-1.79919,-0.315091,0.90044,-1.78464,-0.351908,1.00367,-1.86705,-0.392963,0.750727,-1.83106,-0.448438,0.788165,-1.7702,0.223301,1.12804,-2.14677,0.263942,1.05422,-2.12953,0.101168,1.06826,-2.09777,0.141629,0.998064,-2.08353,0.250646,0.889602,-2.09182,0.274295,0.706749,-2.06196,0.165085,0.729529,-2.08445,0.153591,0.662155,-2.07189,0.13638,0.431776,-2.02312,0.040795,0.359766,-1.96699,0.324683,0.380287,-2.00542,0.273612,0.342334,-2.06927,0.098854,1.28254,-1.8466,0.08699,1.30889,-2.05989},
/*349*/{0.215183,1.86017,-1.75177,0.268124,1.79737,-1.91181,0.125132,1.72175,-1.64799,0.048039,1.6247,-1.57959,0.174594,1.56107,-1.6596,0.247754,1.51684,-1.68891,0.161102,1.86828,-2.14829,0.263476,1.79963,-1.98966,0.044645,1.84032,-2.03753,-0.04937,1.68926,-2.19227,-0.049881,1.59426,-2.24294,0.109429,1.48464,-2.20109,0.150456,1.43283,-2.21249,0.172829,1.36706,-1.82459,-0.025251,1.42565,-1.87997,-0.02613,1.45702,-1.92942,-0.031915,1.43498,-1.99345,0.160831,1.40477,-2.08107,0.187902,1.055,-1.76117,0.058029,1.02254,-1.79669,0.080114,0.941329,-1.81754,0.151942,0.859508,-1.78848,0.002852,0.830079,-1.77297,-0.071496,0.830693,-1.77716,0.002006,0.939161,-1.80236,-0.06901,0.939331,-1.79921,-0.30154,0.899666,-1.78619,-0.340117,1.00195,-1.86796,-0.377126,0.749145,-1.82963,-0.432825,0.786262,-1.76779,0.219186,1.12601,-2.14718,0.258973,1.05167,-2.13058,0.097377,1.06578,-2.09676,0.137722,0.99585,-2.08284,0.252444,0.889808,-2.09032,0.263723,0.70508,-2.06584,0.154392,0.731683,-2.0845,0.140224,0.663276,-2.07145,0.112437,0.430391,-2.02259,0.019635,0.361455,-1.96773,0.302557,0.379336,-2.00576,0.251389,0.341739,-2.06944,0.099154,1.27829,-1.84865,0.084981,1.30685,-2.06151},
/*350*/{0.214461,1.85584,-1.7497,0.269268,1.79248,-1.91026,0.119504,1.71894,-1.64842,0.037893,1.62345,-1.58174,0.166498,1.55682,-1.65864,0.240225,1.51217,-1.68655,0.163825,1.86422,-2.14839,0.263729,1.79547,-1.98826,0.045327,1.83916,-2.0379,-0.043329,1.68189,-2.19847,-0.040707,1.58512,-2.24585,0.118085,1.47829,-2.20054,0.158902,1.42693,-2.21199,0.171229,1.36258,-1.82552,-0.027839,1.42275,-1.87798,-0.027753,1.4549,-1.92828,-0.034072,1.43291,-1.99163,0.158361,1.40256,-2.08306,0.194808,1.05687,-1.76163,0.064014,1.01915,-1.79665,0.08974,0.938707,-1.81665,0.16505,0.85761,-1.78792,0.017207,0.828672,-1.77263,-0.058699,0.829594,-1.77781,0.01596,0.938543,-1.80094,-0.055089,0.938861,-1.7996,-0.286603,0.897204,-1.78584,-0.329144,1.00004,-1.8684,-0.360962,0.745581,-1.82853,-0.415947,0.782486,-1.76605,0.213927,1.12277,-2.14751,0.253265,1.04848,-2.13246,0.091958,1.06414,-2.09577,0.132197,0.993556,-2.08201,0.252897,0.890383,-2.08858,0.251195,0.702269,-2.06989,0.141715,0.731334,-2.08529,0.126938,0.66407,-2.07186,0.098095,0.43432,-2.0225,-0.003147,0.362327,-1.96676,0.280855,0.378577,-2.0075,0.229117,0.341472,-2.06975,0.097005,1.27443,-1.84988,0.081232,1.30495,-2.06236},
/*351*/{0.213689,1.85143,-1.74839,0.268274,1.78878,-1.90862,0.114668,1.71783,-1.64939,0.029931,1.62379,-1.58405,0.158261,1.55379,-1.65812,0.232563,1.50739,-1.68331,0.166153,1.86095,-2.14763,0.264095,1.79195,-1.98687,0.045905,1.83633,-2.0378,-0.038098,1.67538,-2.20522,-0.030745,1.57712,-2.248,0.126841,1.47331,-2.20004,0.169304,1.42277,-2.21136,0.169214,1.35819,-1.82614,-0.028632,1.42042,-1.87614,-0.027992,1.45386,-1.92789,-0.035736,1.43139,-1.98921,0.156449,1.4016,-2.08471,0.202495,1.05917,-1.76186,0.073845,1.01754,-1.79332,0.10011,0.936991,-1.81431,0.179548,0.857244,-1.78877,0.030845,0.827237,-1.77325,-0.044055,0.827829,-1.77891,0.030256,0.937183,-1.80067,-0.041604,0.938089,-1.79933,-0.272685,0.893333,-1.78666,-0.318576,0.996485,-1.8678,-0.343613,0.742136,-1.82793,-0.399769,0.776862,-1.76627,0.208645,1.119,-2.14845,0.248395,1.04439,-2.13363,0.087007,1.05977,-2.09395,0.126022,0.991328,-2.08148,0.247748,0.890205,-2.08827,0.236661,0.699458,-2.07287,0.128949,0.729819,-2.08665,0.110385,0.664235,-2.07113,0.074771,0.433204,-2.02406,-0.024595,0.362669,-1.96836,0.258936,0.377274,-2.00664,0.206951,0.341134,-2.07018,0.095089,1.27091,-1.85139,0.078187,1.30407,-2.06339},
/*352*/{0.21309,1.84848,-1.74695,0.267891,1.78613,-1.90652,0.108142,1.71724,-1.65006,0.022025,1.62494,-1.58602,0.150339,1.55212,-1.65741,0.223396,1.50372,-1.68107,0.168465,1.85817,-2.14729,0.264242,1.78866,-1.98532,0.04672,1.83542,-2.03826,-0.031238,1.66838,-2.20956,-0.020619,1.56981,-2.24898,0.137402,1.46942,-2.19894,0.17979,1.41896,-2.20977,0.168414,1.35529,-1.82781,-0.029331,1.41894,-1.87564,-0.030728,1.45277,-1.92781,-0.036606,1.43011,-1.98694,0.155005,1.40037,-2.08608,0.209404,1.06114,-1.76238,0.082341,1.0158,-1.79116,0.110474,0.936053,-1.81158,0.194221,0.858654,-1.78812,0.044298,0.826705,-1.77342,-0.029868,0.82727,-1.77905,0.043332,0.936598,-1.79907,-0.027781,0.937061,-1.7993,-0.258129,0.891603,-1.78725,-0.309674,0.992045,-1.86731,-0.324404,0.736731,-1.8277,-0.38121,0.769817,-1.76498,0.203674,1.11538,-2.14946,0.240616,1.04012,-2.13487,0.081546,1.05783,-2.0938,0.11935,0.989283,-2.08122,0.238553,0.88561,-2.08969,0.222648,0.693819,-2.07416,0.115301,0.727987,-2.08775,0.094413,0.662176,-2.07182,0.056453,0.435575,-2.02714,-0.049362,0.366425,-1.97029,0.237973,0.37742,-2.00828,0.185472,0.34097,-2.07068,0.093865,1.26856,-1.85268,0.075635,1.30315,-2.06434},
/*353*/{0.212166,1.84573,-1.74671,0.266643,1.78333,-1.90592,0.10336,1.71776,-1.65097,0.013683,1.62699,-1.58959,0.14251,1.55062,-1.65712,0.215009,1.50027,-1.67875,0.170998,1.85558,-2.14618,0.264337,1.78661,-1.9835,0.048013,1.83305,-2.0384,-0.024102,1.66187,-2.2127,-0.011699,1.56496,-2.25056,0.146693,1.46677,-2.19726,0.190725,1.41723,-2.20795,0.166548,1.35211,-1.82687,-0.02986,1.41761,-1.87446,-0.030919,1.45234,-1.92807,-0.037154,1.42965,-1.98599,0.153912,1.39966,-2.08735,0.216656,1.06307,-1.76258,0.091162,1.01489,-1.78998,0.120927,0.936333,-1.81039,0.208595,0.861164,-1.78796,0.05813,0.826249,-1.77356,-0.017171,0.825672,-1.77817,0.055881,0.936426,-1.79861,-0.015479,0.93627,-1.79867,-0.245028,0.88684,-1.7879,-0.301149,0.986158,-1.86704,-0.305964,0.730077,-1.82809,-0.361753,0.761207,-1.76346,0.196842,1.11183,-2.1517,0.234182,1.03524,-2.136,0.075111,1.05657,-2.09377,0.111263,0.98622,-2.08064,0.228001,0.880475,-2.09192,0.209275,0.687639,-2.07384,0.102601,0.72577,-2.08888,0.082054,0.658402,-2.07139,0.035799,0.435034,-2.02515,-0.068689,0.364149,-1.9692,0.215756,0.377058,-2.00816,0.163254,0.340717,-2.07151,0.091835,1.26629,-1.85345,0.07382,1.30259,-2.06483},
/*354*/{0.211479,1.84412,-1.74577,0.265776,1.78075,-1.90398,0.096827,1.71825,-1.65148,0.006827,1.6301,-1.59261,0.134246,1.55068,-1.65697,0.206212,1.4988,-1.67723,0.173511,1.85383,-2.14565,0.264866,1.78447,-1.98235,0.048841,1.83205,-2.0391,-0.017584,1.65658,-2.21501,-0.002479,1.56041,-2.25141,0.157447,1.46559,-2.19562,0.202,1.41673,-2.20596,0.165761,1.3502,-1.82745,-0.030381,1.41764,-1.87375,-0.031325,1.45116,-1.92802,-0.037154,1.42965,-1.98599,0.153593,1.39877,-2.08839,0.223669,1.06688,-1.76243,0.100949,1.01438,-1.78896,0.132475,0.936619,-1.80932,0.222924,0.862932,-1.78785,0.071987,0.826185,-1.774,-0.003379,0.824765,-1.77982,0.068016,0.936374,-1.79774,-0.00306,0.935453,-1.79728,-0.230439,0.88061,-1.79097,-0.293034,0.97934,-1.86574,-0.287021,0.72306,-1.82966,-0.340531,0.751303,-1.76265,0.191414,1.10736,-2.15249,0.226648,1.03025,-2.13714,0.068343,1.05538,-2.09385,0.10511,0.982628,-2.08101,0.215583,0.874289,-2.09319,0.193177,0.683084,-2.07516,0.089002,0.723373,-2.0897,0.066328,0.654818,-2.07248,0.012992,0.434471,-2.02731,-0.090454,0.364988,-1.97023,0.194451,0.376456,-2.00963,0.141108,0.33979,-2.07229,0.09073,1.26493,-1.85418,0.072586,1.30205,-2.06542},
/*355*/{0.210325,1.84337,-1.74441,0.265868,1.77954,-1.90297,0.092504,1.72007,-1.6522,-0.001379,1.63388,-1.59583,0.126085,1.55129,-1.65761,0.198019,1.49732,-1.67518,0.177259,1.8519,-2.14456,0.264489,1.78261,-1.98091,0.049883,1.83128,-2.04036,-0.008517,1.65303,-2.21635,0.006047,1.55653,-2.25206,0.166968,1.46331,-2.19311,0.213176,1.41747,-2.20313,0.16487,1.34867,-1.82716,-0.030662,1.418,-1.87358,-0.032098,1.45032,-1.92769,-0.037154,1.42965,-1.98599,0.152907,1.39772,-2.08939,0.229893,1.07073,-1.76277,0.108965,1.01374,-1.78676,0.144242,0.938387,-1.80803,0.235808,0.866804,-1.78809,0.085965,0.825447,-1.7735,0.010316,0.823729,-1.78113,0.080133,0.936224,-1.79696,0.008458,0.933294,-1.79593,-0.21691,0.874253,-1.78994,-0.285485,0.971288,-1.86444,-0.265969,0.715387,-1.83097,-0.320722,0.741188,-1.76139,0.185824,1.10399,-2.15313,0.219051,1.02583,-2.13827,0.061741,1.05467,-2.09476,0.094985,0.979843,-2.08108,0.202983,0.86932,-2.09477,0.178455,0.677767,-2.07504,0.07498,0.720351,-2.08995,0.050897,0.656176,-2.07483,-0.00533,0.435853,-2.0273,-0.110447,0.364992,-1.96995,0.171988,0.375652,-2.00978,0.120154,0.339776,-2.0735,0.089447,1.264,-1.85436,0.071275,1.30134,-2.06555},
/*356*/{0.20762,1.84301,-1.74451,0.264999,1.77892,-1.9017,0.086328,1.72202,-1.65293,-0.009281,1.63797,-1.59868,0.118275,1.55156,-1.65603,0.188785,1.49748,-1.67291,0.179477,1.85099,-2.14408,0.265457,1.78142,-1.9793,0.051062,1.83115,-2.04122,-0.001782,1.65036,-2.21785,0.014777,1.55311,-2.25327,0.176843,1.4644,-2.19092,0.223564,1.41895,-2.19996,0.163167,1.34891,-1.82619,-0.032025,1.41807,-1.87491,-0.031882,1.45,-1.92903,-0.037797,1.42996,-1.98628,0.152274,1.397,-2.08986,0.240117,1.07547,-1.76212,0.118955,1.01655,-1.78603,0.154894,0.940723,-1.80617,0.24972,0.870438,-1.7889,0.098823,0.826065,-1.77388,0.024301,0.822372,-1.78113,0.090729,0.936925,-1.7959,0.020283,0.932611,-1.79495,-0.203763,0.868343,-1.79199,-0.277956,0.96185,-1.86359,-0.245733,0.707483,-1.83117,-0.298628,0.729723,-1.76117,0.179201,1.10073,-2.15359,0.210994,1.02175,-2.13874,0.054535,1.05443,-2.09657,0.087036,0.978664,-2.08349,0.189444,0.86421,-2.09516,0.163736,0.671757,-2.07372,0.060493,0.717448,-2.09056,0.034491,0.652153,-2.07538,-0.028927,0.435347,-2.02884,-0.133481,0.367089,-1.97251,0.151324,0.37503,-2.01105,0.097364,0.339155,-2.07404,0.087922,1.26413,-1.85413,0.070792,1.30076,-2.06554},
/*357*/{0.206223,1.84317,-1.74283,0.264444,1.77793,-1.89967,0.080113,1.72509,-1.65374,-0.017849,1.64153,-1.60065,0.110791,1.55381,-1.65556,0.179618,1.49782,-1.67209,0.18351,1.85031,-2.14353,0.265089,1.78066,-1.97765,0.052057,1.83093,-2.04164,0.005348,1.64833,-2.21987,0.023676,1.55063,-2.25413,0.187219,1.46709,-2.18822,0.234884,1.42232,-2.19565,0.161948,1.34888,-1.82584,-0.032024,1.41799,-1.8756,-0.033304,1.44971,-1.92951,-0.03757,1.42994,-1.98748,0.152494,1.39541,-2.08972,0.246584,1.08147,-1.76178,0.127329,1.01811,-1.78536,0.165523,0.943594,-1.80593,0.261835,0.875003,-1.78902,0.112578,0.826048,-1.77551,0.038207,0.820223,-1.78261,0.102253,0.937802,-1.79425,0.029961,0.931224,-1.79509,-0.188299,0.858786,-1.79495,-0.270124,0.951162,-1.86379,-0.224879,0.697732,-1.83266,-0.276919,0.718207,-1.76027,0.173497,1.09722,-2.1534,0.20261,1.01783,-2.13908,0.046765,1.05432,-2.09807,0.076455,0.97826,-2.08546,0.175825,0.859215,-2.09623,0.147824,0.667847,-2.07318,0.045246,0.714042,-2.09092,0.019241,0.650302,-2.07613,-0.05356,0.434621,-2.02824,-0.154174,0.367894,-1.97201,0.129193,0.374626,-2.01087,0.076081,0.3396,-2.07475,0.086754,1.26403,-1.85363,0.070608,1.2997,-2.06528},
/*358*/{0.202867,1.84407,-1.74191,0.264017,1.77806,-1.8978,0.074999,1.72841,-1.65454,-0.025967,1.64582,-1.60386,0.10197,1.55632,-1.65563,0.171166,1.4999,-1.67147,0.186487,1.8506,-2.14229,0.265688,1.78092,-1.97616,0.054019,1.83067,-2.04271,0.010861,1.64667,-2.22038,0.031587,1.54891,-2.25476,0.197097,1.47016,-2.18525,0.245413,1.42701,-2.19116,0.161654,1.35019,-1.82538,-0.031989,1.41795,-1.87573,-0.033199,1.44965,-1.92968,-0.036911,1.43035,-1.98826,0.152521,1.3948,-2.08978,0.25298,1.08751,-1.76212,0.134624,1.02099,-1.78495,0.176098,0.947932,-1.80487,0.274669,0.880682,-1.7881,0.126798,0.827672,-1.77616,0.050815,0.818697,-1.7832,0.112593,0.938265,-1.79383,0.041755,0.93053,-1.79383,-0.175445,0.851781,-1.79585,-0.262682,0.938924,-1.86397,-0.202721,0.689791,-1.83278,-0.255185,0.70626,-1.76049,0.168157,1.09454,-2.15318,0.194343,1.01377,-2.13905,0.039076,1.05466,-2.09909,0.067766,0.977504,-2.08627,0.161828,0.854186,-2.09652,0.130155,0.664197,-2.07254,0.02954,0.711925,-2.09106,0.002077,0.64795,-2.07623,-0.072248,0.436275,-2.03048,-0.176048,0.367934,-1.97464,0.106786,0.375021,-2.01222,0.053516,0.339416,-2.07512,0.086901,1.26476,-1.85294,0.071032,1.29928,-2.06479},
/*359*/{0.200582,1.84601,-1.74099,0.263386,1.77861,-1.89662,0.069391,1.73153,-1.65528,-0.033066,1.65047,-1.60548,0.094204,1.55966,-1.65588,0.161934,1.50213,-1.66995,0.188541,1.85122,-2.14092,0.266183,1.78137,-1.97407,0.054689,1.83123,-2.04406,0.016561,1.645,-2.22135,0.040147,1.54853,-2.25485,0.205978,1.47422,-2.18105,0.256008,1.43319,-2.18609,0.160325,1.35046,-1.82486,-0.03261,1.41897,-1.87722,-0.032776,1.44958,-1.93114,-0.037464,1.43087,-1.98978,0.151308,1.39369,-2.08999,0.25695,1.09524,-1.76338,0.141786,1.02403,-1.78508,0.186511,0.952225,-1.80473,0.286626,0.885925,-1.78813,0.13965,0.828587,-1.77744,0.066071,0.817803,-1.78443,0.122203,0.939232,-1.79315,0.050026,0.927901,-1.79287,-0.163331,0.842594,-1.79559,-0.254932,0.926186,-1.86437,-0.180825,0.68042,-1.83372,-0.232677,0.694518,-1.76044,0.161892,1.09188,-2.15318,0.185785,1.01075,-2.13865,0.031327,1.0551,-2.10024,0.056819,0.977302,-2.08786,0.146604,0.848918,-2.0968,0.115194,0.662412,-2.07186,0.013844,0.71098,-2.09175,-0.014237,0.647372,-2.07663,-0.089339,0.439195,-2.02892,-0.196731,0.369132,-1.97387,0.085564,0.374771,-2.01238,0.033749,0.340323,-2.07477,0.085248,1.26523,-1.85271,0.070184,1.29847,-2.06483},
/*360*/{0.197072,1.84756,-1.74027,0.261894,1.77951,-1.89363,0.063479,1.73552,-1.65552,-0.040255,1.6554,-1.60824,0.086325,1.56315,-1.65552,0.154303,1.50561,-1.66926,0.190873,1.8527,-2.13942,0.266813,1.78261,-1.97191,0.055717,1.8316,-2.04444,0.022189,1.64427,-2.22217,0.048198,1.54803,-2.25499,0.214153,1.4795,-2.17741,0.265964,1.44027,-2.18132,0.159265,1.35236,-1.82392,-0.032964,1.41913,-1.87876,-0.03297,1.45075,-1.93228,-0.036852,1.43105,-1.99092,0.153281,1.39366,-2.09001,0.26321,1.10221,-1.76357,0.151166,1.02713,-1.78366,0.196618,0.956816,-1.80443,0.298176,0.893235,-1.78843,0.153525,0.830305,-1.77762,0.079049,0.816117,-1.78558,0.131175,0.940073,-1.79184,0.060426,0.926465,-1.7919,-0.149439,0.832891,-1.79706,-0.246457,0.91216,-1.8657,-0.15962,0.669953,-1.83406,-0.211751,0.682404,-1.76044,0.155448,1.08925,-2.15313,0.17759,1.0077,-2.13829,0.023004,1.05633,-2.10099,0.047647,0.977717,-2.0891,0.131421,0.845537,-2.09645,0.097674,0.660778,-2.07125,-0.002344,0.711047,-2.09151,-0.030969,0.648305,-2.07608,-0.109087,0.440449,-2.02877,-0.218989,0.370753,-1.97516,0.06438,0.375049,-2.01238,0.0106,0.339538,-2.07596,0.084994,1.26653,-1.8521,0.071824,1.29856,-2.06453},
/*361*/{0.194266,1.8498,-1.73893,0.262304,1.78125,-1.89164,0.058093,1.73991,-1.6566,-0.048352,1.66019,-1.61022,0.079054,1.56775,-1.65647,0.1456,1.50937,-1.66865,0.193629,1.85487,-2.13773,0.266952,1.78386,-1.96976,0.05674,1.83178,-2.04506,0.027964,1.64403,-2.22237,0.056114,1.54842,-2.25469,0.222175,1.48495,-2.17333,0.275318,1.44807,-2.17609,0.159959,1.35309,-1.82238,-0.032406,1.41995,-1.87945,-0.032563,1.45081,-1.93331,-0.036132,1.4321,-1.99252,0.154086,1.39282,-2.08992,0.270118,1.10763,-1.7613,0.158159,1.03139,-1.78378,0.20593,0.961853,-1.80306,0.30974,0.900146,-1.7886,0.166308,0.831598,-1.77914,0.09218,0.814745,-1.78764,0.139474,0.940786,-1.79082,0.06922,0.924035,-1.79161,-0.136655,0.82389,-1.79854,-0.237727,0.896793,-1.86691,-0.13746,0.660807,-1.8345,-0.188883,0.669634,-1.76015,0.149364,1.08813,-2.15227,0.16889,1.00526,-2.13693,0.014856,1.05806,-2.10268,0.037311,0.978142,-2.09055,0.115351,0.844216,-2.09597,0.080613,0.658759,-2.0705,-0.018415,0.711882,-2.0911,-0.048442,0.649533,-2.07644,-0.129525,0.442233,-2.02922,-0.240082,0.37339,-1.97448,0.042687,0.375112,-2.01295,-0.010592,0.339573,-2.07593,0.084955,1.26742,-1.85131,0.072784,1.29807,-2.064},
/*362*/{0.190971,1.85238,-1.73801,0.261481,1.7834,-1.88902,0.053191,1.7434,-1.65651,-0.055844,1.66508,-1.61285,0.070986,1.57258,-1.65709,0.137998,1.51377,-1.66891,0.195364,1.85742,-2.13596,0.267387,1.78555,-1.96792,0.058188,1.83282,-2.04544,0.032796,1.6446,-2.22226,0.064055,1.54937,-2.25469,0.230183,1.49171,-2.16951,0.285493,1.45678,-2.1709,0.160033,1.35549,-1.82214,-0.032748,1.42109,-1.88161,-0.031179,1.45224,-1.93445,-0.035436,1.43304,-1.9936,0.155703,1.39221,-2.09013,0.275299,1.11557,-1.76166,0.168112,1.03634,-1.7832,0.214158,0.967014,-1.8044,0.320472,0.909177,-1.78919,0.178639,0.833283,-1.77947,0.105726,0.81353,-1.78817,0.147574,0.940824,-1.79037,0.078249,0.920972,-1.7915,-0.122533,0.811836,-1.79763,-0.228314,0.880737,-1.86905,-0.114534,0.650339,-1.83449,-0.167114,0.657304,-1.76008,0.142684,1.08649,-2.15016,0.159842,1.00309,-2.13524,0.007767,1.05977,-2.10309,0.027209,0.979202,-2.09109,0.100126,0.844132,-2.09548,0.062808,0.659582,-2.0702,-0.035232,0.71336,-2.09007,-0.06683,0.650593,-2.07545,-0.15298,0.444583,-2.03053,-0.260615,0.37646,-1.97505,0.021309,0.374764,-2.01369,-0.03192,0.340119,-2.07727,0.085047,1.2693,-1.85065,0.074054,1.29801,-2.06368},
/*363*/{0.187813,1.85497,-1.73691,0.261732,1.78633,-1.88697,0.047902,1.74833,-1.65784,-0.062135,1.67163,-1.61621,0.064518,1.57745,-1.65804,0.130875,1.5181,-1.66764,0.198346,1.86065,-2.13465,0.267545,1.78823,-1.96563,0.058588,1.83368,-2.04554,0.036713,1.64512,-2.2219,0.071835,1.55115,-2.2548,0.238591,1.49945,-2.16536,0.294531,1.46613,-2.16567,0.159568,1.35712,-1.8216,-0.032207,1.42227,-1.88326,-0.030587,1.45397,-1.93548,-0.034725,1.43483,-1.99564,0.156961,1.3923,-2.0892,0.277477,1.12267,-1.76254,0.173098,1.03928,-1.7837,0.222151,0.972702,-1.80439,0.33056,0.917033,-1.78955,0.191566,0.835049,-1.7804,0.119153,0.812211,-1.78835,0.155055,0.941121,-1.78996,0.086197,0.918094,-1.79094,-0.109538,0.801561,-1.79997,-0.218,0.863385,-1.8712,-0.092518,0.640103,-1.8345,-0.145741,0.645017,-1.76056,0.13604,1.0846,-2.14894,0.150569,1.00154,-2.13364,0.000342,1.06243,-2.10387,0.016799,0.981653,-2.09167,0.084513,0.845103,-2.0945,0.045485,0.659161,-2.0693,-0.052733,0.714958,-2.08945,-0.084474,0.65408,-2.07449,-0.171559,0.447146,-2.03074,-0.280806,0.380068,-1.97626,0.000206,0.374774,-2.0132,-0.054437,0.339718,-2.07547,0.084367,1.27087,-1.85034,0.075161,1.29866,-2.06357},
/*364*/{0.184264,1.85791,-1.73631,0.26034,1.78882,-1.88514,0.042974,1.75323,-1.65836,-0.068611,1.67714,-1.61785,0.05802,1.58304,-1.65931,0.123556,1.52294,-1.66751,0.200259,1.86433,-2.13333,0.268255,1.79145,-1.96278,0.059593,1.83566,-2.04535,0.04185,1.64524,-2.22121,0.079884,1.55281,-2.25451,0.246119,1.50708,-2.16138,0.303,1.47642,-2.16006,0.160315,1.35971,-1.82164,-0.03106,1.42456,-1.88494,-0.029665,1.45568,-1.93717,-0.032673,1.43606,-1.99726,0.156032,1.39164,-2.09189,0.283258,1.12895,-1.7614,0.179368,1.04357,-1.78483,0.229465,0.977962,-1.80582,0.340303,0.925925,-1.78997,0.203101,0.836706,-1.78087,0.133204,0.811057,-1.78885,0.16267,0.941289,-1.78983,0.093862,0.915976,-1.79199,-0.09579,0.791227,-1.80232,-0.207112,0.845528,-1.87339,-0.069201,0.63057,-1.83406,-0.122962,0.632368,-1.76066,0.129189,1.0845,-2.14757,0.141853,1.00024,-2.1317,-0.007923,1.065,-2.10393,0.005344,0.984034,-2.09166,0.068844,0.846261,-2.09374,0.026366,0.661683,-2.06914,-0.069693,0.717273,-2.08827,-0.102643,0.656753,-2.07484,-0.191353,0.450906,-2.02973,-0.30175,0.384383,-1.97691,-0.022307,0.375258,-2.01307,-0.076102,0.339998,-2.07627,0.084881,1.27335,-1.85023,0.074993,1.29844,-2.06377},
/*365*/{0.181708,1.86104,-1.73542,0.259791,1.79346,-1.88243,0.038295,1.75753,-1.65919,-0.075077,1.68379,-1.62114,0.051367,1.58928,-1.66069,0.116428,1.52827,-1.66754,0.201194,1.86809,-2.13173,0.268123,1.79431,-1.96068,0.061119,1.83742,-2.04596,0.046911,1.64676,-2.22044,0.087588,1.55495,-2.25403,0.252625,1.51522,-2.1573,0.311078,1.48646,-2.15457,0.161867,1.3634,-1.82163,-0.030293,1.4268,-1.88584,-0.027925,1.45761,-1.93835,-0.031098,1.43757,-1.99903,0.155529,1.39167,-2.09291,0.28828,1.13592,-1.76137,0.187565,1.04756,-1.78542,0.237228,0.983671,-1.80608,0.349103,0.935146,-1.79045,0.214877,0.838628,-1.78068,0.147428,0.809279,-1.7889,0.168588,0.941314,-1.79003,0.099639,0.913651,-1.79444,-0.082549,0.779952,-1.80332,-0.195491,0.82702,-1.87578,-0.046415,0.620559,-1.83361,-0.100368,0.620099,-1.76133,0.123174,1.08333,-2.14599,0.132175,0.999551,-2.12984,-0.015854,1.06871,-2.10456,-0.002386,0.986082,-2.09196,0.054187,0.847218,-2.09238,0.00987,0.663034,-2.06788,-0.086514,0.720514,-2.08732,-0.119799,0.659714,-2.07409,-0.215668,0.453739,-2.02977,-0.322257,0.389218,-1.97678,-0.043839,0.375355,-2.01308,-0.097654,0.340319,-2.07517,0.086158,1.27652,-1.84946,0.075644,1.29876,-2.06328},
/*366*/{0.179302,1.86426,-1.73507,0.260045,1.79614,-1.88083,0.033437,1.76214,-1.66026,-0.081595,1.69004,-1.62332,0.044959,1.59518,-1.66163,0.109912,1.53459,-1.66753,0.202356,1.87263,-2.13001,0.268855,1.79773,-1.95848,0.06216,1.83969,-2.04615,0.051766,1.64817,-2.22031,0.094665,1.55757,-2.25355,0.260082,1.52421,-2.15263,0.319005,1.49756,-2.14911,0.16147,1.36588,-1.82166,-0.029366,1.42905,-1.88735,-0.026247,1.45973,-1.94001,-0.029357,1.44037,-1.99939,0.15942,1.39132,-2.09153,0.290794,1.14245,-1.76133,0.194373,1.052,-1.78504,0.243944,0.989763,-1.80854,0.357518,0.943764,-1.79074,0.226645,0.84092,-1.78091,0.159207,0.807255,-1.78956,0.175052,0.940543,-1.7894,0.111706,0.907885,-1.79189,-0.067773,0.767039,-1.80534,-0.183787,0.808553,-1.87939,-0.022704,0.611296,-1.8334,-0.077648,0.607419,-1.76204,0.116925,1.08305,-2.14331,0.123278,0.999093,-2.12703,-0.022793,1.07292,-2.10487,-0.012583,0.990531,-2.09236,0.040198,0.849045,-2.08995,-0.008194,0.666029,-2.06708,-0.103645,0.724352,-2.08632,-0.13767,0.664016,-2.07333,-0.230966,0.457377,-2.02968,-0.341944,0.396788,-1.97704,-0.064818,0.376101,-2.01284,-0.119689,0.340347,-2.07487,0.085843,1.27879,-1.84855,0.077755,1.29967,-2.06262},
/*367*/{0.176014,1.86823,-1.73391,0.259477,1.79994,-1.87865,0.030018,1.76772,-1.66039,-0.086763,1.69585,-1.62624,0.039987,1.60127,-1.66318,0.103385,1.53968,-1.66809,0.203474,1.8773,-2.12872,0.269567,1.80155,-1.95673,0.063302,1.84189,-2.04551,0.055257,1.65178,-2.22047,0.102733,1.56092,-2.25375,0.265966,1.53318,-2.14812,0.325912,1.50832,-2.14346,0.162843,1.36979,-1.82117,-0.027922,1.43207,-1.88901,-0.02453,1.46207,-1.94061,-0.027881,1.44192,-2.00084,0.162061,1.39115,-2.09109,0.295115,1.15015,-1.76128,0.19892,1.05709,-1.7834,0.250142,0.996132,-1.80796,0.365298,0.952595,-1.79151,0.237934,0.843035,-1.78092,0.171096,0.806033,-1.79084,0.181159,0.93962,-1.79007,0.118563,0.902644,-1.79355,-0.052836,0.754175,-1.80566,-0.170904,0.789213,-1.88127,0.001482,0.601107,-1.8331,-0.054236,0.595394,-1.762,0.110477,1.08408,-2.14141,0.114305,0.999011,-2.12397,-0.029321,1.07703,-2.10481,-0.021388,0.993471,-2.0915,0.026856,0.850977,-2.08801,-0.026183,0.669328,-2.06642,-0.120299,0.728774,-2.08536,-0.155449,0.668729,-2.07284,-0.251962,0.46315,-2.03185,-0.362505,0.402336,-1.97454,-0.086393,0.376076,-2.01264,-0.141434,0.340408,-2.07412,0.087085,1.28234,-1.84698,0.079879,1.30016,-2.06136},
/*368*/{0.173729,1.8714,-1.73341,0.259873,1.80376,-1.87662,0.025978,1.77275,-1.66146,-0.092072,1.70272,-1.62937,0.033838,1.60725,-1.66435,0.097948,1.54505,-1.66845,0.205309,1.88184,-2.12723,0.270514,1.80515,-1.95466,0.064032,1.84499,-2.04622,0.060717,1.65452,-2.22016,0.109772,1.56432,-2.25337,0.271586,1.54298,-2.14413,0.333038,1.51917,-2.13772,0.16436,1.37417,-1.82249,-0.02635,1.4359,-1.88955,-0.023045,1.46505,-1.94336,-0.024565,1.44446,-2.00204,0.165363,1.39064,-2.09097,0.29786,1.15615,-1.76094,0.202352,1.06223,-1.78594,0.256591,1.00266,-1.80869,0.372319,0.96168,-1.79122,0.248662,0.844677,-1.78105,0.183566,0.804415,-1.79122,0.187284,0.937668,-1.79086,0.126268,0.89937,-1.79438,-0.0382,0.743753,-1.80772,-0.156709,0.769703,-1.8841,0.025818,0.592811,-1.83322,-0.029724,0.584013,-1.76249,0.104599,1.08424,-2.13847,0.106342,0.999616,-2.12108,-0.037051,1.08175,-2.10524,-0.031205,0.999007,-2.09221,0.013295,0.854226,-2.08554,-0.041794,0.673318,-2.06579,-0.13632,0.733728,-2.08489,-0.172121,0.674789,-2.07293,-0.271288,0.470891,-2.03536,-0.381183,0.410257,-1.9753,-0.107478,0.376509,-2.01238,-0.162228,0.341536,-2.07468,0.088398,1.28645,-1.84584,0.082017,1.30092,-2.06049},
/*369*/{0.171812,1.87528,-1.73276,0.260168,1.80734,-1.87487,0.022272,1.77793,-1.66292,-0.096338,1.70956,-1.63215,0.029689,1.61368,-1.6657,0.092999,1.5516,-1.66851,0.206597,1.88711,-2.1262,0.272733,1.8085,-1.95167,0.065989,1.84696,-2.04565,0.064995,1.65752,-2.22027,0.117097,1.56829,-2.25263,0.277446,1.55222,-2.14024,0.338747,1.53095,-2.13246,0.16618,1.37795,-1.82289,-0.024369,1.43901,-1.89074,-0.01992,1.46789,-1.94399,-0.021799,1.44787,-2.00267,0.165867,1.39112,-2.09091,0.300071,1.16305,-1.76086,0.206653,1.06915,-1.78577,0.262141,1.0096,-1.80827,0.377865,0.96948,-1.79122,0.258236,0.845606,-1.78132,0.196139,0.801719,-1.79113,0.193015,0.936518,-1.79248,0.134444,0.893977,-1.79571,-0.020607,0.730493,-1.80852,-0.141544,0.749435,-1.88669,0.050384,0.583811,-1.83295,-0.005004,0.572697,-1.7628,0.099061,1.08458,-2.13599,0.098232,1.00044,-2.11837,-0.04217,1.08662,-2.1054,-0.04042,1.00462,-2.09163,0.00087,0.858224,-2.08363,-0.058343,0.677748,-2.06496,-0.15172,0.738946,-2.08432,-0.187602,0.679851,-2.07183,-0.289507,0.475416,-2.03337,-0.399178,0.419194,-1.97609,-0.128623,0.377279,-2.01183,-0.184361,0.341368,-2.07391,0.089754,1.28998,-1.84509,0.083045,1.30232,-2.05987},
/*370*/{0.169635,1.87843,-1.73237,0.2597,1.81199,-1.87321,0.018477,1.78352,-1.66386,-0.101174,1.71613,-1.6349,0.02465,1.61941,-1.66691,0.087522,1.55713,-1.66947,0.208426,1.89174,-2.1242,0.27324,1.8127,-1.95003,0.067161,1.85086,-2.04622,0.06996,1.66095,-2.22033,0.123938,1.5729,-2.25274,0.282451,1.56056,-2.1343,0.343686,1.54201,-2.12701,0.168164,1.3819,-1.82312,-0.02236,1.44363,-1.89142,-0.018674,1.47047,-1.94584,-0.019716,1.4501,-2.00416,0.169931,1.39106,-2.09101,0.302991,1.17072,-1.7616,0.210635,1.0741,-1.78587,0.267029,1.01503,-1.80937,0.383349,0.977631,-1.79189,0.26853,0.846488,-1.78174,0.209067,0.800728,-1.79174,0.197665,0.934338,-1.7933,0.142222,0.889122,-1.79725,-0.003906,0.717867,-1.80952,-0.126176,0.729424,-1.88859,0.075775,0.575674,-1.83352,0.020368,0.561344,-1.76291,0.094315,1.08587,-2.13397,0.090201,1.00219,-2.11553,-0.04842,1.09171,-2.10507,-0.048738,1.00935,-2.09044,-0.013191,0.863458,-2.08245,-0.074543,0.683115,-2.06444,-0.168261,0.744631,-2.0833,-0.204438,0.685816,-2.07161,-0.308185,0.480505,-2.03149,-0.415908,0.427629,-1.97499,-0.151781,0.376255,-2.01087,-0.206428,0.34226,-2.07359,0.091261,1.29415,-1.84381,0.085405,1.30327,-2.05878},
/*371*/{0.168204,1.88186,-1.73175,0.260605,1.81421,-1.87039,0.015154,1.78881,-1.6655,-0.105776,1.72263,-1.63825,0.021365,1.62503,-1.66797,0.082858,1.56265,-1.66933,0.209402,1.89711,-2.12255,0.272841,1.81662,-1.94817,0.068184,1.8537,-2.046,0.074772,1.66481,-2.22064,0.130832,1.57708,-2.25249,0.287987,1.57103,-2.13009,0.348047,1.55269,-2.12177,0.169198,1.38563,-1.82419,-0.021049,1.44762,-1.89139,-0.016301,1.47391,-1.94791,-0.016239,1.45387,-2.00461,0.171082,1.39084,-2.09111,0.306553,1.17647,-1.76092,0.209612,1.08038,-1.78652,0.270511,1.02111,-1.81073,0.387389,0.985121,-1.79267,0.277733,0.847731,-1.78189,0.221253,0.798993,-1.79196,0.20251,0.93159,-1.79479,0.149671,0.883862,-1.79847,0.010421,0.704426,-1.81191,-0.108671,0.707778,-1.89074,0.100778,0.568755,-1.83439,0.046055,0.550784,-1.76329,0.08806,1.08779,-2.13164,0.081661,1.00325,-2.11219,-0.053802,1.09792,-2.10473,-0.056351,1.01562,-2.09027,-0.027572,0.868179,-2.08168,-0.090111,0.68832,-2.06415,-0.183013,0.752031,-2.08351,-0.219578,0.693316,-2.07224,-0.320406,0.486064,-2.03169,-0.434491,0.438037,-1.97522,-0.173204,0.376051,-2.00965,-0.228214,0.342611,-2.07332,0.091702,1.29799,-1.84297,0.085907,1.30463,-2.05803},
/*372*/{0.166696,1.88535,-1.73141,0.259788,1.81998,-1.87006,0.012591,1.7939,-1.66711,-0.108031,1.729,-1.64077,0.016026,1.63057,-1.66949,0.07852,1.56764,-1.67011,0.211522,1.90155,-2.12055,0.272988,1.82043,-1.94645,0.069829,1.85619,-2.0462,0.079427,1.66876,-2.22174,0.13808,1.58173,-2.25185,0.290708,1.57977,-2.12598,0.352653,1.56321,-2.11739,0.171477,1.38939,-1.82464,-0.017748,1.45121,-1.89263,-0.013745,1.47714,-1.94896,-0.013293,1.45681,-2.00611,0.173846,1.39085,-2.09065,0.309107,1.18276,-1.76043,0.213343,1.08542,-1.78558,0.274596,1.02715,-1.80976,0.389931,0.990993,-1.79287,0.286993,0.84837,-1.78232,0.233109,0.796463,-1.79188,0.207942,0.928524,-1.79584,0.157533,0.877523,-1.80041,0.029216,0.690871,-1.81253,-0.091823,0.687446,-1.89254,0.12612,0.561359,-1.83433,0.072518,0.540944,-1.76377,0.083493,1.08909,-2.12834,0.073998,1.0053,-2.10949,-0.057853,1.10402,-2.1039,-0.064931,1.02222,-2.08969,-0.041457,0.873683,-2.08143,-0.105327,0.69493,-2.06416,-0.19729,0.758626,-2.08314,-0.234697,0.69954,-2.07172,-0.338747,0.492683,-2.0328,-0.451647,0.450461,-1.97494,-0.195398,0.376881,-2.01002,-0.250581,0.342747,-2.07252,0.093465,1.30187,-1.84202,0.088023,1.30582,-2.05715},
/*373*/{0.1652,1.88798,-1.73105,0.259267,1.82357,-1.86786,0.009698,1.79878,-1.66845,-0.112399,1.73475,-1.64414,0.012405,1.63586,-1.67024,0.074865,1.57297,-1.6704,0.212842,1.90659,-2.1191,0.273659,1.82453,-1.94473,0.0712,1.85948,-2.04638,0.084787,1.67313,-2.22181,0.14339,1.58645,-2.25175,0.295643,1.58889,-2.12215,0.356481,1.574,-2.11277,0.172644,1.39378,-1.82548,-0.015914,1.45564,-1.8927,-0.01029,1.48003,-1.94993,-0.010059,1.45967,-2.00649,0.176324,1.39034,-2.09002,0.310515,1.18885,-1.76042,0.21655,1.09087,-1.78753,0.277167,1.0335,-1.80844,0.39259,0.997817,-1.79391,0.296353,0.848906,-1.78179,0.244418,0.793943,-1.79228,0.213136,0.924809,-1.79768,0.165468,0.872074,-1.80188,0.047077,0.677215,-1.81394,-0.072812,0.66734,-1.8941,0.151997,0.554206,-1.8348,0.099617,0.530883,-1.76458,0.078074,1.08998,-2.12586,0.065674,1.00854,-2.10743,-0.063239,1.11125,-2.1035,-0.072694,1.02884,-2.08922,-0.054603,0.879364,-2.08092,-0.11973,0.700941,-2.06451,-0.211103,0.765636,-2.08327,-0.249584,0.70772,-2.07211,-0.357231,0.500021,-2.03478,-0.467258,0.461452,-1.97541,-0.218271,0.376865,-2.00939,-0.271447,0.343645,-2.07197,0.094727,1.30615,-1.84028,0.089494,1.3068,-2.05545},
/*374*/{0.163849,1.89094,-1.73081,0.25932,1.82574,-1.86624,0.007721,1.8043,-1.66989,-0.115631,1.74086,-1.64716,0.009357,1.64109,-1.67176,0.071678,1.57791,-1.6711,0.215215,1.9112,-2.11716,0.27408,1.8282,-1.94296,0.07276,1.86247,-2.04696,0.090288,1.67723,-2.2218,0.150643,1.59124,-2.25166,0.297402,1.59653,-2.11919,0.360043,1.58414,-2.10865,0.17479,1.39715,-1.82596,-0.013206,1.46076,-1.89339,-0.008863,1.4838,-1.95182,-0.006302,1.46276,-2.00672,0.178187,1.39097,-2.08929,0.313107,1.19475,-1.76063,0.219907,1.09573,-1.78492,0.279996,1.03868,-1.80871,0.393919,1.00391,-1.79402,0.305486,0.848572,-1.78228,0.256771,0.791296,-1.79322,0.218804,0.920448,-1.79857,0.172835,0.864995,-1.80389,0.066655,0.664009,-1.81539,-0.053696,0.64668,-1.89569,0.17761,0.547481,-1.83569,0.126115,0.520859,-1.76514,0.073417,1.09251,-2.12382,0.058367,1.01041,-2.10477,-0.066854,1.118,-2.10333,-0.079959,1.03699,-2.08846,-0.066791,0.885278,-2.08044,-0.133392,0.708645,-2.06537,-0.225243,0.773372,-2.08299,-0.263168,0.715796,-2.07277,-0.372578,0.505965,-2.03521,-0.481974,0.473306,-1.9739,-0.239482,0.377181,-2.00988,-0.294857,0.344643,-2.0724,0.095805,1.3104,-1.83922,0.090754,1.30854,-2.05439},
/*375*/{0.162919,1.89386,-1.73033,0.261256,1.83017,-1.86519,0.005562,1.80872,-1.67211,-0.118175,1.74588,-1.64984,0.006393,1.64627,-1.67338,0.068057,1.58235,-1.67112,0.216975,1.91529,-2.11545,0.274642,1.83231,-1.94128,0.074381,1.86535,-2.04719,0.094495,1.68156,-2.22177,0.156679,1.59597,-2.2517,0.300231,1.60475,-2.11591,0.36329,1.59398,-2.10466,0.176622,1.40148,-1.82596,-0.010612,1.46476,-1.89312,-0.006409,1.48658,-1.95324,-0.003907,1.46599,-2.00781,0.181058,1.39076,-2.08893,0.31526,1.20093,-1.76087,0.223885,1.10151,-1.78305,0.281806,1.0434,-1.80803,0.394771,1.00873,-1.79401,0.313457,0.848417,-1.78221,0.269773,0.789831,-1.79318,0.224625,0.914831,-1.7989,0.180682,0.857562,-1.80544,0.084407,0.650043,-1.81676,-0.033214,0.626132,-1.89757,0.203671,0.540899,-1.8357,0.153065,0.511323,-1.76659,0.06877,1.09437,-2.12215,0.050649,1.01286,-2.10265,-0.070911,1.12514,-2.103,-0.08691,1.04472,-2.08778,-0.077778,0.892905,-2.08084,-0.146779,0.716242,-2.06673,-0.23751,0.782048,-2.08328,-0.276543,0.721882,-2.07278,-0.385759,0.513274,-2.0366,-0.494838,0.486553,-1.97623,-0.262724,0.376676,-2.00933,-0.316937,0.345488,-2.07184,0.097387,1.31475,-1.83768,0.09255,1.30974,-2.0528},
/*376*/{0.162163,1.89658,-1.73038,0.259599,1.83323,-1.86355,0.004609,1.81321,-1.67355,-0.121156,1.7517,-1.65313,0.002802,1.65093,-1.67438,0.065277,1.58643,-1.67184,0.218485,1.91938,-2.11375,0.274674,1.83524,-1.93969,0.075862,1.86748,-2.0478,0.099934,1.68579,-2.22214,0.162161,1.5999,-2.25168,0.303106,1.61309,-2.11314,0.366259,1.6032,-2.10051,0.177296,1.40513,-1.82705,-0.0083,1.46922,-1.89325,-0.003295,1.49067,-1.95444,-0.000922,1.46882,-2.00807,0.182353,1.39067,-2.08831,0.316505,1.20572,-1.76058,0.224681,1.10573,-1.7832,0.28351,1.04856,-1.80622,0.394923,1.01343,-1.79462,0.321626,0.847857,-1.78247,0.279024,0.785568,-1.7928,0.228798,0.909891,-1.80151,0.188128,0.850621,-1.8079,0.10401,0.636815,-1.81697,-0.011981,0.606263,-1.89847,0.229797,0.535763,-1.83728,0.1815,0.503019,-1.76745,0.064572,1.09654,-2.12028,0.042196,1.01571,-2.10104,-0.074103,1.13178,-2.10179,-0.092873,1.05248,-2.08744,-0.0889,0.900479,-2.0809,-0.158845,0.724264,-2.06801,-0.250781,0.789521,-2.08353,-0.288544,0.731852,-2.07362,-0.395563,0.519317,-2.03756,-0.50669,0.500745,-1.9752,-0.283711,0.377775,-2.01021,-0.339128,0.347109,-2.07202,0.097919,1.31888,-1.83648,0.092957,1.31117,-2.05152},
/*377*/{0.161893,1.89911,-1.72965,0.259918,1.83813,-1.86281,0.003039,1.8177,-1.67485,-0.123693,1.75628,-1.6563,0.001271,1.65472,-1.67496,0.062501,1.58999,-1.67192,0.219743,1.92295,-2.11184,0.275804,1.83896,-1.93819,0.078406,1.86968,-2.04819,0.104377,1.68947,-2.22289,0.168065,1.60459,-2.25184,0.30506,1.61924,-2.11037,0.368419,1.6117,-2.09728,0.17919,1.40994,-1.82761,-0.005874,1.47374,-1.89417,-0.001821,1.49412,-1.95516,0.002864,1.47181,-2.00847,0.184685,1.39091,-2.08719,0.318848,1.20984,-1.76011,0.227544,1.11102,-1.7812,0.286589,1.05332,-1.80572,0.394392,1.01736,-1.7947,0.329201,0.846772,-1.78242,0.29192,0.782389,-1.79296,0.233951,0.903619,-1.80275,0.197779,0.842018,-1.80805,0.123644,0.625248,-1.81805,0.009575,0.587442,-1.89944,0.255475,0.530756,-1.8379,0.208508,0.493779,-1.76914,0.060867,1.09862,-2.11911,0.035605,1.019,-2.09954,-0.076508,1.13949,-2.10135,-0.098203,1.0599,-2.08607,-0.099763,0.908341,-2.08127,-0.170622,0.732363,-2.06957,-0.262553,0.797221,-2.08364,-0.301391,0.737515,-2.07242,-0.406049,0.526114,-2.03899,-0.515766,0.514579,-1.97468,-0.304712,0.378522,-2.01071,-0.361236,0.349325,-2.07168,0.099549,1.32381,-1.83477,0.094761,1.31275,-2.04967},
/*378*/{0.161172,1.90088,-1.7294,0.260794,1.84007,-1.86087,0.001941,1.82122,-1.67631,-0.124567,1.7601,-1.65961,-0.001418,1.65815,-1.67618,0.060875,1.59373,-1.67204,0.221523,1.92636,-2.11055,0.277058,1.84218,-1.93638,0.080017,1.87236,-2.04814,0.109204,1.69371,-2.2229,0.173672,1.60903,-2.25234,0.307005,1.62565,-2.1083,0.37087,1.61909,-2.09366,0.180936,1.41277,-1.82779,-0.003541,1.47738,-1.89369,0.001515,1.49779,-1.95711,0.005411,1.47494,-2.00945,0.186569,1.39088,-2.08615,0.319518,1.21356,-1.76027,0.22889,1.11465,-1.77917,0.284691,1.0572,-1.80524,0.392765,1.02015,-1.7949,0.336877,0.845811,-1.78209,0.302816,0.779741,-1.79338,0.23883,0.897599,-1.80463,0.204733,0.833432,-1.80974,0.143077,0.612205,-1.81874,0.031948,0.568747,-1.89977,0.280828,0.525681,-1.83852,0.235841,0.485299,-1.77105,0.056865,1.10114,-2.11834,0.029515,1.02223,-2.09845,-0.079193,1.14602,-2.10074,-0.102457,1.06707,-2.08613,-0.109672,0.915972,-2.08201,-0.18045,0.740981,-2.07161,-0.274277,0.80494,-2.08366,-0.311276,0.745362,-2.0724,-0.420363,0.533662,-2.04083,-0.524624,0.530365,-1.97543,-0.323933,0.378454,-2.01179,-0.383408,0.352235,-2.07166,0.100272,1.32739,-1.83376,0.09576,1.31416,-2.04854},
/*379*/{0.160925,1.90306,-1.72945,0.254119,1.84611,-1.86186,0.001458,1.82469,-1.67739,-0.127606,1.76303,-1.6617,-0.002981,1.66182,-1.67746,0.058818,1.59622,-1.67226,0.22408,1.9294,-2.1083,0.276855,1.84559,-1.93572,0.081396,1.87431,-2.04857,0.113897,1.69713,-2.22416,0.177952,1.61241,-2.25258,0.308623,1.6323,-2.10607,0.372329,1.62675,-2.09081,0.181292,1.41606,-1.82858,-0.00036,1.48127,-1.89362,0.004702,1.50083,-1.95774,0.008038,1.47785,-2.00965,0.188283,1.39106,-2.08535,0.321006,1.21693,-1.76014,0.228801,1.1187,-1.7782,0.284344,1.05993,-1.8036,0.391936,1.02229,-1.79498,0.34411,0.843423,-1.78199,0.310883,0.774965,-1.79225,0.243348,0.890968,-1.80549,0.213277,0.825042,-1.81081,0.163758,0.599953,-1.81981,0.055224,0.550351,-1.90016,0.304774,0.521005,-1.84054,0.26266,0.477774,-1.77227,0.052869,1.10348,-2.1178,0.024255,1.02554,-2.09819,-0.081092,1.15184,-2.10048,-0.108524,1.0748,-2.08615,-0.11944,0.923429,-2.08327,-0.191383,0.748615,-2.0735,-0.284697,0.812961,-2.08414,-0.321562,0.753459,-2.07308,-0.429938,0.540771,-2.04205,-0.53153,0.545943,-1.97398,-0.342141,0.378292,-2.01286,-0.404257,0.35601,-2.07089,0.101063,1.33097,-1.83269,0.096565,1.31569,-2.04734},
/*380*/{0.16102,1.90488,-1.72898,0.255279,1.84856,-1.86095,0.000741,1.82676,-1.67989,-0.12888,1.76715,-1.66474,-0.004294,1.66468,-1.67864,0.057459,1.59904,-1.67213,0.22551,1.93203,-2.10722,0.277816,1.84846,-1.93422,0.083979,1.87656,-2.04886,0.118586,1.7006,-2.22471,0.183126,1.61587,-2.25315,0.311621,1.63873,-2.10467,0.373318,1.63228,-2.08712,0.1818,1.41935,-1.82881,0.001178,1.48515,-1.89369,0.007435,1.50396,-1.95815,0.011648,1.48013,-2.00944,0.189373,1.39166,-2.08444,0.321137,1.22002,-1.76078,0.228539,1.12192,-1.77627,0.283986,1.06334,-1.80156,0.389661,1.02328,-1.79524,0.35033,0.841669,-1.7814,0.323197,0.772732,-1.79261,0.248916,0.883526,-1.8065,0.222116,0.816733,-1.80958,0.183646,0.588154,-1.81943,0.077836,0.533216,-1.90046,0.329064,0.516737,-1.8409,0.290436,0.470331,-1.7737,0.050136,1.10568,-2.11756,0.018963,1.02813,-2.09824,-0.082906,1.15863,-2.10005,-0.11155,1.0813,-2.08555,-0.128486,0.931412,-2.08396,-0.200231,0.756875,-2.07521,-0.292589,0.82164,-2.08473,-0.330377,0.761616,-2.074,-0.436658,0.54848,-2.04474,-0.537782,0.561303,-1.97502,-0.360976,0.378276,-2.01465,-0.425761,0.361657,-2.06974,0.101658,1.33454,-1.83142,0.097433,1.3173,-2.04593},
/*381*/{0.161488,1.90649,-1.72887,0.261517,1.84696,-1.85847,0.000909,1.82969,-1.68073,-0.12849,1.7694,-1.6671,-0.00525,1.66679,-1.67949,0.056633,1.60106,-1.67236,0.228052,1.93421,-2.10565,0.279294,1.84991,-1.93294,0.085315,1.8775,-2.04883,0.123344,1.70396,-2.22593,0.187332,1.61885,-2.25366,0.313112,1.64331,-2.10323,0.374965,1.63766,-2.08462,0.182122,1.42287,-1.82998,0.003765,1.48835,-1.89422,0.009449,1.50651,-1.95896,0.013255,1.48275,-2.00969,0.191142,1.39214,-2.08368,0.323257,1.22179,-1.76158,0.226004,1.12411,-1.7763,0.282326,1.06504,-1.80013,0.388835,1.02318,-1.79511,0.357243,0.838683,-1.78168,0.333387,0.767495,-1.79157,0.253265,0.875558,-1.80789,0.230273,0.805916,-1.8111,0.205345,0.576064,-1.81997,0.101012,0.516347,-1.89979,0.353271,0.513363,-1.84245,0.317364,0.464935,-1.77496,0.047787,1.10867,-2.11822,0.01452,1.0327,-2.09983,-0.084741,1.16347,-2.0988,-0.114605,1.08723,-2.08497,-0.135848,0.938974,-2.08411,-0.208141,0.764451,-2.07712,-0.299786,0.829955,-2.08517,-0.338168,0.76994,-2.07494,-0.441814,0.556272,-2.04579,-0.54156,0.57795,-1.97619,-0.377324,0.379154,-2.01559,-0.444836,0.367874,-2.0682,0.102573,1.33808,-1.83065,0.09839,1.31881,-2.04498},
/*382*/{0.162207,1.90793,-1.72832,0.262795,1.84948,-1.85687,0.001016,1.83137,-1.6815,-0.129209,1.77133,-1.66902,-0.005279,1.66848,-1.67992,0.056515,1.60233,-1.67226,0.229923,1.93632,-2.10477,0.280584,1.85258,-1.9317,0.087613,1.87959,-2.04944,0.127665,1.70653,-2.22666,0.191501,1.62137,-2.25411,0.314515,1.6468,-2.10234,0.376274,1.64256,-2.08244,0.183611,1.42569,-1.82992,0.005379,1.49185,-1.89459,0.010645,1.50971,-1.9594,0.015449,1.48645,-2.00983,0.192475,1.39354,-2.08304,0.323656,1.22296,-1.76197,0.225276,1.12537,-1.7738,0.279579,1.06625,-1.79857,0.387145,1.02212,-1.79482,0.363514,0.835456,-1.78115,0.342431,0.763013,-1.79085,0.25753,0.865714,-1.81003,0.23843,0.797849,-1.8128,0.225685,0.564399,-1.81851,0.124394,0.500095,-1.89983,0.377903,0.510607,-1.84313,0.344063,0.45877,-1.77656,0.044469,1.11177,-2.1192,0.010294,1.03631,-2.10091,-0.086174,1.16785,-2.09739,-0.11684,1.09306,-2.08402,-0.140324,0.945651,-2.08448,-0.214123,0.771099,-2.07898,-0.305736,0.838347,-2.08561,-0.343809,0.77747,-2.07624,-0.452451,0.565891,-2.0485,-0.545711,0.593868,-1.97583,-0.39355,0.377529,-2.01549,-0.466356,0.376713,-2.0683,0.103199,1.34142,-1.83014,0.099165,1.32122,-2.04439},
/*383*/{0.162235,1.90881,-1.72819,0.262975,1.85156,-1.85685,0.001374,1.83207,-1.68247,-0.129121,1.7724,-1.67101,-0.005723,1.67017,-1.68065,0.056832,1.60369,-1.67179,0.231027,1.93792,-2.10336,0.281318,1.85382,-1.93033,0.088833,1.88134,-2.04971,0.132126,1.70957,-2.22884,0.19555,1.62369,-2.25486,0.315456,1.65006,-2.10121,0.377306,1.6462,-2.08075,0.184815,1.4286,-1.82996,0.005429,1.49585,-1.89504,0.012233,1.51284,-1.95924,0.017249,1.48821,-2.00993,0.192979,1.39474,-2.08248,0.324068,1.22342,-1.76303,0.224763,1.1268,-1.77319,0.278879,1.06676,-1.7969,0.38547,1.02065,-1.7947,0.369278,0.832151,-1.77967,0.352364,0.758857,-1.79054,0.263148,0.857192,-1.80925,0.246215,0.787531,-1.81281,0.246702,0.552471,-1.8194,0.147418,0.485226,-1.899,0.400316,0.506209,-1.84337,0.370899,0.454562,-1.77649,0.042293,1.11488,-2.12002,0.007259,1.03921,-2.10213,-0.087806,1.17279,-2.09577,-0.119024,1.09735,-2.08245,-0.143384,0.95059,-2.08402,-0.219999,0.777592,-2.08067,-0.310852,0.845986,-2.08585,-0.350529,0.786358,-2.07743,-0.460464,0.575531,-2.05051,-0.548528,0.609536,-1.9769,-0.411926,0.377892,-2.01609,-0.484588,0.38556,-2.06618,0.103721,1.34492,-1.82916,0.099488,1.32303,-2.04324},
/*384*/{0.162895,1.90994,-1.72752,0.263872,1.8529,-1.85562,0.000667,1.83316,-1.6839,-0.130732,1.77278,-1.67249,-0.00579,1.67054,-1.68128,0.057017,1.60477,-1.67133,0.233136,1.93925,-2.10256,0.282703,1.8559,-1.92948,0.090982,1.88273,-2.04989,0.136512,1.71159,-2.22977,0.19903,1.62532,-2.25547,0.316388,1.65252,-2.09969,0.378417,1.64819,-2.07874,0.185336,1.43139,-1.82978,0.007279,1.49834,-1.89439,0.013147,1.51556,-1.95945,0.018302,1.4912,-2.01009,0.194181,1.39655,-2.08211,0.324006,1.22335,-1.7631,0.222692,1.12805,-1.77212,0.275054,1.06647,-1.7962,0.382066,1.01789,-1.79438,0.374649,0.828183,-1.77931,0.360433,0.754968,-1.79022,0.266737,0.848395,-1.81077,0.253789,0.777762,-1.81349,0.26549,0.543786,-1.81872,0.170503,0.470679,-1.89886,0.422298,0.504223,-1.8449,0.39417,0.448761,-1.7774,0.040357,1.11848,-2.12098,0.00535,1.04281,-2.10406,-0.089007,1.17607,-2.09391,-0.120606,1.10164,-2.08157,-0.144318,0.954321,-2.08378,-0.225408,0.782856,-2.08068,-0.31516,0.852188,-2.08611,-0.355222,0.793119,-2.07735,-0.467852,0.585413,-2.05214,-0.553016,0.625078,-1.97771,-0.430203,0.376039,-2.01509,-0.503761,0.395674,-2.06575,0.104417,1.34768,-1.82874,0.100285,1.32536,-2.04278},
/*385*/{0.163853,1.91068,-1.72677,0.264638,1.85381,-1.85469,0.000885,1.83361,-1.68419,-0.1306,1.77293,-1.6741,-0.004741,1.67121,-1.6816,0.058253,1.60477,-1.6705,0.234517,1.94016,-2.10147,0.283448,1.85682,-1.92908,0.093194,1.88415,-2.04999,0.141315,1.71351,-2.23117,0.201957,1.62664,-2.25673,0.317395,1.65471,-2.0994,0.380004,1.64993,-2.07773,0.185949,1.434,-1.83018,0.007624,1.50073,-1.89418,0.01525,1.51772,-1.95882,0.019411,1.49373,-2.00984,0.194506,1.3983,-2.08164,0.320116,1.22222,-1.76488,0.219599,1.12855,-1.77408,0.271852,1.0659,-1.79614,0.37867,1.01476,-1.79327,0.378567,0.824096,-1.77809,0.369022,0.74961,-1.78947,0.270987,0.839115,-1.8095,0.261617,0.768467,-1.81378,0.284857,0.535255,-1.81955,0.193455,0.457631,-1.89874,0.443198,0.502141,-1.84505,0.418127,0.444907,-1.7792,0.039649,1.12097,-2.12216,0.004582,1.04582,-2.10562,-0.08941,1.17942,-2.09186,-0.120926,1.10399,-2.07961,-0.143651,0.956745,-2.08359,-0.228153,0.787235,-2.08179,-0.317166,0.857578,-2.08574,-0.358207,0.800987,-2.07976,-0.476057,0.595398,-2.05356,-0.557441,0.641531,-1.98146,-0.450525,0.380295,-2.014,-0.5225,0.40638,-2.06392,0.105137,1.35016,-1.8283,0.100714,1.32744,-2.04229},
/*386*/{0.164442,1.9117,-1.72626,0.266474,1.85389,-1.85379,0.001064,1.83323,-1.68456,-0.130468,1.77224,-1.67507,-0.003111,1.67148,-1.6817,0.059366,1.60443,-1.67017,0.23662,1.94121,-2.10145,0.285018,1.85836,-1.92725,0.094927,1.88636,-2.05024,0.145561,1.71472,-2.23198,0.20422,1.62715,-2.25696,0.319468,1.65481,-2.09933,0.380541,1.65064,-2.07685,0.186737,1.43618,-1.83076,0.008815,1.50263,-1.89456,0.014922,1.52033,-1.95932,0.020423,1.49638,-2.00876,0.195113,1.40032,-2.08162,0.319358,1.21988,-1.76455,0.216867,1.12956,-1.77502,0.268422,1.06392,-1.7958,0.375675,1.01099,-1.79257,0.382533,0.820336,-1.77744,0.378007,0.745024,-1.78868,0.274646,0.829959,-1.80909,0.268863,0.758396,-1.81381,0.303456,0.525417,-1.81927,0.215952,0.445042,-1.89947,0.463867,0.500267,-1.84538,0.441275,0.441006,-1.77903,0.039383,1.12389,-2.12326,0.002685,1.04768,-2.10593,-0.088865,1.18183,-2.09093,-0.120968,1.10565,-2.07824,-0.142658,0.958186,-2.08336,-0.231574,0.789861,-2.08214,-0.318091,0.862894,-2.08642,-0.360226,0.806683,-2.07906,-0.483892,0.604593,-2.0551,-0.562429,0.656702,-1.98413,-0.46899,0.39314,-2.0114,-0.53874,0.419287,-2.06263,0.105888,1.35241,-1.82841,0.101157,1.3297,-2.04239},
/*387*/{0.164922,1.91211,-1.72592,0.267301,1.85475,-1.85301,0.002597,1.83279,-1.68441,-0.128397,1.77098,-1.67542,-0.001972,1.67026,-1.68181,0.061461,1.60359,-1.66931,0.238676,1.94201,-2.10085,0.286391,1.85926,-1.92711,0.096838,1.88784,-2.04984,0.148118,1.71613,-2.23339,0.206436,1.62817,-2.25752,0.319678,1.65495,-2.09856,0.381181,1.65057,-2.076,0.186756,1.43784,-1.83125,0.009817,1.50467,-1.8939,0.015743,1.52296,-1.95831,0.020424,1.49753,-2.00837,0.195913,1.4022,-2.08164,0.315954,1.21658,-1.76528,0.213313,1.1256,-1.7741,0.264712,1.06195,-1.7947,0.37134,1.0063,-1.79219,0.385758,0.816025,-1.77693,0.383956,0.740387,-1.78811,0.276556,0.819303,-1.80861,0.2756,0.74754,-1.81219,0.321106,0.515997,-1.81816,0.237512,0.433318,-1.89979,0.482179,0.498846,-1.84539,0.463024,0.438029,-1.77963,0.038824,1.12627,-2.12368,0.003189,1.04883,-2.10787,-0.088633,1.18286,-2.08916,-0.119876,1.10706,-2.07711,-0.14106,0.958946,-2.08338,-0.234227,0.790942,-2.08177,-0.31651,0.868417,-2.08702,-0.361856,0.812509,-2.08081,-0.491795,0.61526,-2.05606,-0.568962,0.671773,-1.98484,-0.485076,0.4041,-2.00994,-0.553207,0.432174,-2.06227,0.106503,1.35413,-1.82826,0.101381,1.33165,-2.04226},
/*388*/{0.166535,1.91235,-1.72518,0.269561,1.85526,-1.85279,0.003671,1.83185,-1.68455,-0.126185,1.76876,-1.67565,7.7e-005,1.66932,-1.68181,0.063586,1.60188,-1.66848,0.240495,1.94179,-2.10023,0.287704,1.85971,-1.92639,0.098387,1.88894,-2.04964,0.151011,1.71707,-2.23376,0.207293,1.62805,-2.25772,0.321464,1.65575,-2.09889,0.381558,1.64938,-2.07532,0.187007,1.44008,-1.83131,0.009727,1.50536,-1.89338,0.016034,1.52421,-1.95818,0.021072,1.49957,-2.00795,0.197055,1.40464,-2.08145,0.315682,1.21177,-1.7644,0.209621,1.12261,-1.77414,0.260323,1.05885,-1.79472,0.366921,1.00128,-1.79171,0.389289,0.811466,-1.77736,0.391434,0.736536,-1.78916,0.279876,0.809422,-1.80526,0.282289,0.738154,-1.81024,0.33901,0.509697,-1.81865,0.257365,0.424601,-1.90044,0.500699,0.498418,-1.84458,0.483357,0.436731,-1.78022,0.041133,1.12724,-2.12416,0.007707,1.05008,-2.1088,-0.087035,1.18318,-2.08814,-0.117947,1.1073,-2.07677,-0.13813,0.958682,-2.08318,-0.235656,0.79161,-2.0811,-0.316467,0.871622,-2.0877,-0.363294,0.818233,-2.08047,-0.500524,0.627141,-2.05635,-0.575685,0.686111,-1.98787,-0.500356,0.415645,-2.00779,-0.568191,0.445619,-2.06191,0.107417,1.3556,-1.82835,0.102764,1.33374,-2.04242},
/*389*/{0.16724,1.91241,-1.72463,0.270286,1.85573,-1.85238,0.004223,1.8303,-1.68456,-0.12492,1.76577,-1.67589,0.002708,1.66702,-1.68109,0.066139,1.60035,-1.66645,0.242319,1.94165,-2.09997,0.288594,1.85999,-1.9255,0.1,1.88981,-2.04942,0.152724,1.71701,-2.23372,0.208382,1.62759,-2.25804,0.32259,1.65328,-2.09885,0.382829,1.64684,-2.07473,0.187339,1.44152,-1.83154,0.010303,1.50743,-1.89206,0.016791,1.52539,-1.95747,0.021509,1.50104,-2.00696,0.197005,1.40619,-2.08129,0.312792,1.20692,-1.7644,0.205717,1.12086,-1.77384,0.253697,1.05455,-1.79491,0.361793,0.996109,-1.78994,0.393207,0.807251,-1.77617,0.398594,0.731617,-1.78795,0.282784,0.799432,-1.80248,0.288515,0.727828,-1.8082,0.354763,0.502782,-1.81829,0.277097,0.414202,-1.90033,0.517293,0.498409,-1.84412,0.501553,0.434835,-1.78095,0.042493,1.12807,-2.12433,0.010147,1.05104,-2.10919,-0.08536,1.18302,-2.08785,-0.115522,1.10619,-2.07622,-0.134198,0.957455,-2.08311,-0.236782,0.792488,-2.08083,-0.314227,0.874859,-2.08826,-0.363412,0.823794,-2.08259,-0.507983,0.639311,-2.05733,-0.582836,0.700896,-1.9901,-0.515427,0.428361,-2.00611,-0.581047,0.458824,-2.06159,0.108124,1.35707,-1.82787,0.102699,1.33529,-2.04193},
/*390*/{0.168301,1.91206,-1.72397,0.271837,1.85486,-1.8516,0.006001,1.82838,-1.68338,-0.123527,1.76252,-1.67562,0.005429,1.66456,-1.68056,0.068911,1.59749,-1.66563,0.243462,1.9414,-2.0998,0.289733,1.85998,-1.92518,0.10089,1.89154,-2.04925,0.154064,1.71698,-2.23373,0.208258,1.62737,-2.25813,0.323039,1.65072,-2.09804,0.383689,1.64391,-2.0748,0.187117,1.44211,-1.83179,0.010387,1.50758,-1.89203,0.016118,1.52693,-1.95718,0.021348,1.50241,-2.00647,0.197812,1.40808,-2.08125,0.307871,1.2017,-1.76464,0.199754,1.11659,-1.77476,0.24755,1.05086,-1.79597,0.357043,0.989857,-1.79047,0.397045,0.802827,-1.77731,0.404468,0.727044,-1.78856,0.286434,0.791293,-1.79946,0.29438,0.719835,-1.80497,0.368769,0.49629,-1.81757,0.294899,0.40591,-1.90086,0.53269,0.498352,-1.84374,0.520011,0.43399,-1.77991,0.045956,1.12815,-2.12466,0.014554,1.05055,-2.10943,-0.083595,1.18099,-2.08759,-0.112943,1.10418,-2.07509,-0.129558,0.955521,-2.08278,-0.237479,0.792864,-2.07958,-0.310936,0.879517,-2.09021,-0.362158,0.830439,-2.0838,-0.514607,0.650892,-2.05653,-0.589635,0.715144,-1.99331,-0.528117,0.441663,-2.00439,-0.591599,0.473206,-2.06146,0.108134,1.35745,-1.82845,0.10315,1.33699,-2.04265},
/*391*/{0.16981,1.91161,-1.72306,0.272582,1.85597,-1.85172,0.007219,1.82542,-1.68302,-0.122129,1.75897,-1.67437,0.008334,1.66169,-1.67954,0.072615,1.5953,-1.66398,0.245494,1.94058,-2.09928,0.290919,1.86021,-1.9249,0.102574,1.89214,-2.04858,0.154696,1.71692,-2.23351,0.207857,1.62694,-2.25811,0.323925,1.64849,-2.09799,0.38389,1.64014,-2.07475,0.189959,1.44382,-1.83265,0.01065,1.50806,-1.89196,0.016067,1.52786,-1.95706,0.021251,1.50296,-2.00583,0.198002,1.40949,-2.08135,0.303158,1.19573,-1.76461,0.195274,1.11563,-1.77391,0.240792,1.04651,-1.79537,0.352331,0.983843,-1.79013,0.400311,0.798625,-1.77814,0.410779,0.723893,-1.78963,0.289635,0.782197,-1.795,0.300261,0.712211,-1.80298,0.381903,0.490078,-1.81567,0.311573,0.39804,-1.9004,0.545358,0.49936,-1.84294,0.535229,0.43392,-1.78061,0.049283,1.12754,-2.12402,0.017793,1.04978,-2.10914,-0.080474,1.17957,-2.08855,-0.107914,1.1014,-2.07541,-0.124338,0.952711,-2.08212,-0.238614,0.79372,-2.07851,-0.307863,0.882737,-2.09091,-0.361393,0.836203,-2.08405,-0.522182,0.662832,-2.0577,-0.596098,0.730105,-1.99519,-0.542379,0.45486,-2.00313,-0.600976,0.487241,-2.06064,0.110651,1.35897,-1.82833,0.104098,1.33788,-2.04243},
/*392*/{0.17136,1.91063,-1.72241,0.273591,1.85608,-1.85121,0.009163,1.82339,-1.68148,-0.118708,1.75472,-1.67346,0.011697,1.65876,-1.67825,0.07601,1.59195,-1.6623,0.246784,1.9396,-2.09815,0.291574,1.86006,-1.92448,0.103407,1.89229,-2.04831,0.153886,1.71695,-2.23296,0.206516,1.6262,-2.25763,0.325072,1.64486,-2.09824,0.385073,1.63531,-2.07527,0.191373,1.44355,-1.83134,0.010528,1.50796,-1.89137,0.016417,1.52832,-1.95597,0.021149,1.50312,-2.00497,0.198818,1.41099,-2.08145,0.300432,1.18933,-1.76411,0.191999,1.11095,-1.77341,0.233999,1.04119,-1.79543,0.347214,0.97784,-1.79019,0.403077,0.795436,-1.77857,0.415667,0.720467,-1.79052,0.29249,0.775525,-1.79258,0.304534,0.705416,-1.80015,0.393574,0.484576,-1.81635,0.324757,0.391645,-1.89953,0.555933,0.498894,-1.84462,0.549047,0.433357,-1.7809,0.053666,1.126,-2.1236,0.023013,1.04824,-2.10931,-0.077434,1.17612,-2.08835,-0.103942,1.09835,-2.07562,-0.118282,0.949938,-2.08114,-0.239235,0.794779,-2.07715,-0.304929,0.886687,-2.09102,-0.360723,0.841935,-2.08418,-0.527024,0.677124,-2.05833,-0.60219,0.746958,-1.99783,-0.551898,0.469311,-2.001,-0.611487,0.502717,-2.06159,0.111659,1.35871,-1.82827,0.105072,1.3388,-2.04248},
/*393*/{0.172218,1.90985,-1.7218,0.275582,1.8563,-1.85133,0.011507,1.81975,-1.68043,-0.115192,1.75022,-1.67202,0.015284,1.6543,-1.67614,0.080122,1.58792,-1.65985,0.24774,1.93832,-2.09811,0.292309,1.85937,-1.92447,0.104564,1.89253,-2.0473,0.152508,1.7159,-2.23183,0.204664,1.62512,-2.25723,0.324774,1.6407,-2.09873,0.384957,1.63004,-2.07586,0.192091,1.44335,-1.83152,0.011906,1.50631,-1.89026,0.016317,1.52836,-1.95582,0.021814,1.50305,-2.00424,0.199486,1.41144,-2.08162,0.296901,1.1828,-1.76342,0.181742,1.10601,-1.77533,0.226969,1.03668,-1.79708,0.341044,0.971825,-1.7898,0.405127,0.792334,-1.77903,0.420562,0.717925,-1.79197,0.295343,0.767811,-1.79087,0.310011,0.698117,-1.79793,0.403731,0.482063,-1.81612,0.337477,0.386446,-1.89826,0.565811,0.499489,-1.8447,0.560847,0.4341,-1.78135,0.058475,1.12404,-2.12258,0.029255,1.04511,-2.10804,-0.073615,1.17233,-2.08907,-0.099484,1.09416,-2.07568,-0.112889,0.945766,-2.08003,-0.238892,0.796044,-2.07668,-0.300476,0.890297,-2.09111,-0.357374,0.848691,-2.08397,-0.531859,0.690807,-2.05766,-0.605423,0.762594,-1.99979,-0.563172,0.484285,-2.00068,-0.619049,0.51913,-2.06289,0.112999,1.35797,-1.82851,0.106029,1.33896,-2.04279},
/*394*/{0.174385,1.90807,-1.72082,0.276627,1.85413,-1.85073,0.013678,1.8162,-1.67947,-0.112362,1.74509,-1.67004,0.019235,1.64968,-1.67434,0.084235,1.58357,-1.6574,0.249276,1.93662,-2.0976,0.292977,1.85862,-1.92362,0.104647,1.89298,-2.04592,0.150405,1.71563,-2.23107,0.202984,1.62422,-2.25666,0.324804,1.63589,-2.09962,0.385326,1.62469,-2.07675,0.192485,1.44272,-1.83196,0.011973,1.50499,-1.88992,0.017495,1.52705,-1.95546,0.021828,1.50233,-2.00319,0.20078,1.41181,-2.08156,0.292455,1.17626,-1.76309,0.175316,1.10232,-1.77633,0.218944,1.03096,-1.79816,0.335257,0.966129,-1.78931,0.407007,0.789521,-1.78076,0.422177,0.715415,-1.79334,0.296604,0.762554,-1.78971,0.314196,0.692739,-1.79599,0.411122,0.478518,-1.81506,0.346514,0.382241,-1.89674,0.574282,0.499779,-1.8454,0.569273,0.434628,-1.78288,0.06346,1.12183,-2.12135,0.035636,1.04235,-2.10649,-0.070415,1.16807,-2.09005,-0.094039,1.08946,-2.07581,-0.106331,0.942291,-2.0794,-0.238338,0.79809,-2.07566,-0.29583,0.89486,-2.09055,-0.354701,0.855566,-2.08313,-0.534627,0.705216,-2.05774,-0.607633,0.779697,-2.00226,-0.572826,0.499707,-2.00137,-0.626731,0.536337,-2.06401,0.114186,1.35681,-1.82878,0.107329,1.33874,-2.04315},
/*395*/{0.175981,1.90609,-1.72009,0.276989,1.85265,-1.85126,0.015683,1.81298,-1.67798,-0.107864,1.73975,-1.66719,0.023882,1.6449,-1.67221,0.089545,1.57918,-1.6551,0.249922,1.93506,-2.09687,0.293475,1.85708,-1.92349,0.105511,1.89199,-2.04476,0.147864,1.71484,-2.22921,0.200139,1.62284,-2.25551,0.325264,1.63072,-2.10105,0.385218,1.61781,-2.07808,0.188976,1.43852,-1.83096,0.010811,1.50337,-1.89061,0.01686,1.52566,-1.95559,0.022149,1.50147,-2.0026,0.201138,1.4118,-2.08189,0.28723,1.16964,-1.76278,0.171268,1.09597,-1.77581,0.211849,1.02615,-1.79856,0.32821,0.959283,-1.78933,0.406532,0.786498,-1.78143,0.423489,0.713142,-1.79326,0.296991,0.757271,-1.78934,0.315137,0.687974,-1.79537,0.418056,0.476656,-1.8151,0.354089,0.377807,-1.89547,0.579304,0.498911,-1.84705,0.579621,0.434301,-1.78388,0.068286,1.11892,-2.12003,0.041899,1.03949,-2.10468,-0.066008,1.16288,-2.0905,-0.088828,1.08367,-2.07666,-0.099519,0.93833,-2.07863,-0.236758,0.799649,-2.07554,-0.290841,0.898887,-2.09016,-0.351394,0.861465,-2.08204,-0.53849,0.719596,-2.05716,-0.607556,0.796644,-2.00226,-0.581636,0.515601,-2.00144,-0.635366,0.554536,-2.06549,0.110884,1.353,-1.83041,0.106697,1.33853,-2.04512},
/*396*/{0.177216,1.90377,-1.71926,0.277858,1.8514,-1.85123,0.018123,1.80848,-1.67613,-0.103319,1.73417,-1.66453,0.028176,1.63995,-1.66919,0.094398,1.57415,-1.65315,0.249349,1.93267,-2.0961,0.293976,1.85592,-1.92351,0.105929,1.89163,-2.04415,0.144614,1.71406,-2.2284,0.197037,1.62178,-2.25441,0.325209,1.62615,-2.10217,0.38434,1.61058,-2.07988,0.191687,1.44008,-1.83212,0.012541,1.50197,-1.88945,0.01673,1.52368,-1.95532,0.022268,1.49916,-2.00277,0.201676,1.41166,-2.08202,0.28421,1.16297,-1.76134,0.16433,1.09288,-1.77636,0.205969,1.02171,-1.7992,0.321195,0.953475,-1.7894,0.404532,0.784304,-1.78126,0.4238,0.711195,-1.79277,0.295585,0.752536,-1.78932,0.315147,0.683879,-1.79515,0.422601,0.472617,-1.81411,0.359057,0.374014,-1.89381,0.5834,0.496928,-1.84717,0.583515,0.432719,-1.78442,0.0742,1.11561,-2.11891,0.048832,1.035,-2.10315,-0.060954,1.15718,-2.09035,-0.082916,1.07806,-2.07658,-0.091732,0.93362,-2.0782,-0.234533,0.802334,-2.07736,-0.283795,0.903594,-2.08933,-0.346691,0.867998,-2.08047,-0.537804,0.735006,-2.06254,-0.606077,0.814156,-2.00217,-0.589694,0.53174,-2.00334,-0.640567,0.572973,-2.0667,0.114955,1.3535,-1.82968,0.108772,1.33721,-2.04421},
/*397*/{0.17945,1.90108,-1.71821,0.27833,1.84943,-1.8514,0.020865,1.80399,-1.67356,-0.099312,1.72744,-1.66162,0.032895,1.63414,-1.66657,0.099632,1.56885,-1.65083,0.249592,1.93061,-2.09599,0.293838,1.85387,-1.92359,0.105482,1.89115,-2.04275,0.140963,1.71299,-2.22677,0.193451,1.62052,-2.2532,0.324647,1.62007,-2.10336,0.383191,1.6028,-2.08128,0.190572,1.43742,-1.83208,0.010603,1.49815,-1.89003,0.017069,1.52168,-1.95519,0.02196,1.49789,-2.00281,0.202502,1.41081,-2.08216,0.280733,1.15678,-1.76028,0.159065,1.09055,-1.77822,0.19908,1.01707,-1.80011,0.314529,0.947331,-1.78869,0.401853,0.780982,-1.78149,0.421062,0.708112,-1.79243,0.293482,0.747836,-1.79025,0.313838,0.679441,-1.79593,0.424811,0.46926,-1.81376,0.362097,0.370174,-1.89286,0.586982,0.49541,-1.84853,0.58814,0.430358,-1.78661,0.079754,1.11212,-2.11791,0.056555,1.03141,-2.10206,-0.05591,1.15104,-2.09078,-0.076147,1.07174,-2.07628,-0.084332,0.928528,-2.07738,-0.232499,0.803665,-2.07563,-0.277243,0.907281,-2.08743,-0.340628,0.875083,-2.07589,-0.538132,0.750975,-2.05917,-0.602829,0.830784,-2.00099,-0.596114,0.548316,-2.00553,-0.648584,0.591966,-2.0672,0.114035,1.35019,-1.83088,0.10939,1.33611,-2.04561},
/*398*/{0.181113,1.89828,-1.71714,0.278989,1.84659,-1.8513,0.023709,1.79894,-1.6704,-0.095904,1.72036,-1.65783,0.038107,1.62812,-1.6637,0.104689,1.56322,-1.64797,0.249809,1.9281,-2.09549,0.294245,1.8515,-1.92336,0.105831,1.88922,-2.04169,0.137,1.71158,-2.22491,0.189269,1.61903,-2.25211,0.323377,1.61331,-2.10418,0.382269,1.59466,-2.08357,0.191741,1.43616,-1.83212,0.011651,1.49551,-1.88897,0.018208,1.51881,-1.95496,0.023754,1.49586,-2.00217,0.20313,1.40897,-2.08196,0.275789,1.15119,-1.75924,0.155072,1.08668,-1.77776,0.194982,1.01358,-1.80088,0.309677,0.941447,-1.78826,0.397812,0.777336,-1.78086,0.416886,0.705364,-1.79212,0.290036,0.743195,-1.79062,0.312131,0.675145,-1.79609,0.425316,0.466554,-1.81394,0.363536,0.365908,-1.89273,0.587413,0.493013,-1.84935,0.588832,0.426382,-1.78668,0.085951,1.10813,-2.11691,0.063642,1.02726,-2.10122,-0.051806,1.1447,-2.09149,-0.069661,1.06532,-2.07648,-0.076315,0.923528,-2.0765,-0.228487,0.805636,-2.07543,-0.269309,0.911157,-2.08663,-0.33406,0.880669,-2.07872,-0.536466,0.765908,-2.05932,-0.59727,0.846627,-1.99873,-0.60237,0.565216,-2.00778,-0.651493,0.611821,-2.07033,0.115933,1.34828,-1.83041,0.110851,1.33398,-2.04511},
/*399*/{0.182629,1.89478,-1.71612,0.279482,1.84439,-1.85144,0.026409,1.79406,-1.66922,-0.091195,1.71259,-1.65358,0.043406,1.62151,-1.65991,0.111095,1.55828,-1.6454,0.249017,1.92478,-2.09508,0.293803,1.84959,-1.92314,0.105897,1.88802,-2.04086,0.133167,1.71049,-2.22349,0.185054,1.61717,-2.25096,0.321098,1.60516,-2.10473,0.380856,1.58667,-2.08631,0.191894,1.43308,-1.8323,0.011393,1.49312,-1.8893,0.018594,1.51568,-1.95473,0.023488,1.49154,-2.00238,0.204321,1.40733,-2.08137,0.272936,1.14659,-1.75833,0.15358,1.08276,-1.77761,0.192503,1.01002,-1.80222,0.306224,0.937583,-1.7876,0.393947,0.77338,-1.77957,0.414217,0.700408,-1.79125,0.28629,0.737873,-1.79079,0.308924,0.670393,-1.79677,0.425064,0.461467,-1.81356,0.361894,0.362293,-1.89249,0.586558,0.488254,-1.85013,0.589224,0.422274,-1.78735,0.091983,1.10399,-2.11682,0.071553,1.02268,-2.0997,-0.046104,1.13803,-2.09151,-0.063813,1.05824,-2.07595,-0.068478,0.918101,-2.07614,-0.224124,0.807,-2.07541,-0.261356,0.914703,-2.08588,-0.326338,0.886155,-2.07662,-0.533507,0.78028,-2.05993,-0.591776,0.863863,-1.99824,-0.60918,0.581742,-2.01183,-0.654775,0.630621,-2.07151,0.116506,1.34529,-1.83054,0.112087,1.33135,-2.04528},
/*400*/{0.184516,1.89077,-1.71511,0.278696,1.84224,-1.85155,0.029058,1.78856,-1.66692,-0.085892,1.70534,-1.64885,0.048623,1.61522,-1.65613,0.117344,1.55251,-1.64232,0.248377,1.92199,-2.09515,0.293794,1.84634,-1.92342,0.105173,1.88493,-2.03975,0.128589,1.70853,-2.22136,0.180642,1.6148,-2.25007,0.318822,1.59776,-2.10685,0.37841,1.57659,-2.08903,0.192573,1.42943,-1.83201,0.011885,1.49006,-1.88928,0.018503,1.51196,-1.95511,0.023103,1.48887,-2.00302,0.205319,1.40508,-2.08085,0.272938,1.14183,-1.75598,0.152196,1.07884,-1.77702,0.191968,1.00635,-1.80311,0.305007,0.935504,-1.78761,0.389567,0.768411,-1.77957,0.411207,0.695583,-1.79033,0.282777,0.732429,-1.79208,0.304392,0.664946,-1.79861,0.423597,0.45854,-1.81451,0.35931,0.35777,-1.89341,0.584225,0.483217,-1.85125,0.584521,0.416939,-1.78887,0.096901,1.09962,-2.1163,0.077844,1.0178,-2.09894,-0.041199,1.13168,-2.09171,-0.057458,1.05179,-2.07592,-0.060113,0.912633,-2.07569,-0.218908,0.80818,-2.07552,-0.252557,0.917392,-2.08439,-0.318548,0.891614,-2.07629,-0.529647,0.794362,-2.05976,-0.583864,0.879142,-1.99699,-0.612595,0.598252,-2.0142,-0.657641,0.649873,-2.07258,0.116799,1.34183,-1.83096,0.112989,1.32868,-2.04576},
/*401*/{0.18614,1.88666,-1.71443,0.279907,1.83918,-1.85174,0.032616,1.7825,-1.66416,-0.081703,1.69677,-1.64377,0.054873,1.60875,-1.65235,0.123283,1.54669,-1.63989,0.247306,1.91866,-2.09552,0.294186,1.84332,-1.92317,0.104152,1.88229,-2.03813,0.12404,1.70642,-2.21949,0.176079,1.61196,-2.24907,0.316862,1.59019,-2.10829,0.375833,1.56718,-2.09241,0.191709,1.42598,-1.83086,0.012423,1.48554,-1.88948,0.017781,1.5091,-1.95519,0.024339,1.48522,-2.00335,0.206726,1.40303,-2.08026,0.2703,1.1389,-1.75526,0.150773,1.07565,-1.77678,0.189071,1.00259,-1.80304,0.304343,0.933426,-1.78762,0.386108,0.763147,-1.77902,0.407565,0.690358,-1.79008,0.27896,0.727605,-1.79311,0.300607,0.659496,-1.7994,0.417388,0.453078,-1.81547,0.354424,0.352711,-1.89596,0.578661,0.476921,-1.85237,0.581319,0.410712,-1.79029,0.102124,1.09576,-2.1158,0.08507,1.01342,-2.09806,-0.037292,1.12436,-2.09107,-0.050222,1.0445,-2.07529,-0.051926,0.906692,-2.07551,-0.213371,0.80914,-2.07545,-0.242119,0.919753,-2.08283,-0.309124,0.89551,-2.07542,-0.525313,0.808411,-2.06053,-0.574523,0.894585,-1.99541,-0.613633,0.614846,-2.0168,-0.656941,0.669296,-2.07462,0.116641,1.33794,-1.83123,0.114682,1.32601,-2.04612},
/*402*/{0.187546,1.88182,-1.71314,0.279571,1.83522,-1.85226,0.035914,1.77549,-1.66064,-0.076277,1.68878,-1.63879,0.059982,1.60133,-1.64834,0.129805,1.54121,-1.63773,0.246019,1.91522,-2.09553,0.294277,1.84002,-1.92328,0.103912,1.87881,-2.03688,0.119101,1.70429,-2.21746,0.170895,1.60927,-2.24816,0.313494,1.58211,-2.1101,0.373213,1.55669,-2.09549,0.193188,1.4223,-1.83072,0.011497,1.48249,-1.88958,0.019176,1.50602,-1.95522,0.024008,1.48116,-2.00412,0.207366,1.40045,-2.0796,0.269402,1.13598,-1.7538,0.150399,1.07248,-1.77668,0.188096,1.00075,-1.80281,0.304085,0.931444,-1.78747,0.38137,0.756262,-1.77852,0.402345,0.684106,-1.79049,0.274466,0.722897,-1.79285,0.29521,0.65433,-1.80047,0.411151,0.446515,-1.8164,0.347203,0.349065,-1.89747,0.573322,0.470321,-1.85358,0.574112,0.403403,-1.79189,0.107886,1.09124,-2.11564,0.092783,1.00906,-2.0973,-0.032061,1.11793,-2.09109,-0.044147,1.03652,-2.07463,-0.043606,0.900469,-2.07482,-0.207052,0.809835,-2.07545,-0.231535,0.920082,-2.08192,-0.299933,0.901309,-2.0772,-0.519448,0.820866,-2.05939,-0.564206,0.907806,-1.99422,-0.614662,0.630698,-2.01892,-0.655538,0.687786,-2.07625,0.117272,1.33461,-1.83133,0.11557,1.32286,-2.04623},
/*403*/{0.189264,1.87756,-1.71243,0.280764,1.83112,-1.85182,0.03932,1.76854,-1.65765,-0.071134,1.68058,-1.63342,0.066558,1.59441,-1.64445,0.136927,1.53557,-1.63425,0.245128,1.91162,-2.09624,0.29415,1.83663,-1.92305,0.103054,1.87531,-2.03539,0.114519,1.70148,-2.21622,0.165835,1.60614,-2.24671,0.310442,1.57369,-2.11225,0.369392,1.54645,-2.09915,0.192275,1.41836,-1.82915,0.011892,1.47874,-1.88944,0.01927,1.5048,-1.95528,0.024379,1.47684,-2.00451,0.208412,1.39729,-2.07908,0.267433,1.13307,-1.75361,0.147802,1.06786,-1.77792,0.186205,0.996683,-1.80122,0.303012,0.927726,-1.78752,0.376802,0.750362,-1.77852,0.396334,0.678787,-1.79121,0.269375,0.717944,-1.79335,0.290036,0.649738,-1.80088,0.405615,0.440936,-1.81856,0.338045,0.345783,-1.90066,0.566621,0.461124,-1.8531,0.567293,0.394835,-1.79259,0.113467,1.08662,-2.11522,0.099662,1.00407,-2.0972,-0.025732,1.11009,-2.0906,-0.037388,1.02944,-2.07469,-0.034895,0.894445,-2.07491,-0.200613,0.808906,-2.07547,-0.22079,0.91933,-2.0808,-0.289409,0.904058,-2.077,-0.511627,0.832702,-2.05955,-0.553089,0.921522,-1.99423,-0.612507,0.645924,-2.02088,-0.652816,0.703809,-2.07768,0.116316,1.33102,-1.83105,0.11607,1.31972,-2.04599},
/*404*/{0.190842,1.8726,-1.7119,0.281692,1.8275,-1.85225,0.042652,1.76139,-1.65388,-0.066036,1.67175,-1.62814,0.071845,1.58706,-1.6402,0.144391,1.52987,-1.63185,0.242643,1.90732,-2.09622,0.294256,1.83291,-1.92349,0.10207,1.87076,-2.03353,0.109438,1.69819,-2.21461,0.159932,1.60268,-2.24658,0.306377,1.56416,-2.11492,0.366162,1.53558,-2.10332,0.193178,1.41703,-1.82862,0.011736,1.4742,-1.89007,0.018612,1.49794,-1.95457,0.024531,1.4723,-2.00477,0.20875,1.39404,-2.07873,0.266917,1.13065,-1.75333,0.146788,1.06616,-1.77763,0.184397,0.994108,-1.80167,0.30224,0.923005,-1.78643,0.372091,0.74385,-1.77855,0.391249,0.671301,-1.79169,0.263526,0.713271,-1.79447,0.284079,0.644235,-1.80132,0.3941,0.435635,-1.81923,0.322872,0.350365,-1.90147,0.557744,0.452157,-1.85366,0.553378,0.382603,-1.79291,0.119131,1.08203,-2.1149,0.107006,0.999769,-2.09671,-0.021906,1.10288,-2.09046,-0.031026,1.02239,-2.07497,-0.026117,0.888345,-2.07458,-0.193414,0.808537,-2.075,-0.209471,0.920689,-2.08123,-0.278205,0.905842,-2.07682,-0.501184,0.843479,-2.06052,-0.541888,0.932792,-1.99366,-0.610382,0.659815,-2.02325,-0.647421,0.719878,-2.0791,0.118584,1.32824,-1.83002,0.118206,1.31532,-2.04487},
/*405*/{0.192861,1.86712,-1.71049,0.281143,1.82335,-1.85213,0.046613,1.7535,-1.64991,-0.06142,1.66274,-1.62207,0.078852,1.57975,-1.63537,0.151293,1.52407,-1.62995,0.240929,1.90324,-2.0976,0.294678,1.82925,-1.92376,0.101592,1.86699,-2.03304,0.104723,1.69301,-2.21285,0.154561,1.59885,-2.24564,0.303286,1.55607,-2.11778,0.361089,1.52419,-2.10788,0.191194,1.41022,-1.82772,0.010653,1.47,-1.89061,0.018493,1.49403,-1.95445,0.024808,1.46734,-2.00509,0.210042,1.39123,-2.07671,0.26542,1.12642,-1.75252,0.144279,1.06442,-1.78043,0.183654,0.991042,-1.80069,0.299823,0.916892,-1.78621,0.366089,0.737469,-1.77921,0.384733,0.664207,-1.79224,0.258206,0.707732,-1.79458,0.276504,0.639417,-1.80362,0.382224,0.433049,-1.82124,0.306771,0.350414,-1.90474,0.545248,0.439235,-1.85518,0.535536,0.371642,-1.79319,0.125501,1.07803,-2.11471,0.114411,0.995082,-2.09763,-0.016649,1.09577,-2.09027,-0.023927,1.01512,-2.07463,-0.017265,0.882379,-2.07473,-0.185622,0.807936,-2.07505,-0.19743,0.919855,-2.08033,-0.267443,0.907771,-2.07451,-0.491577,0.851988,-2.06039,-0.529286,0.942722,-1.99317,-0.605229,0.672816,-2.02428,-0.641549,0.734377,-2.081,0.116283,1.32217,-1.83072,0.118839,1.31185,-2.04569},
/*406*/{0.194667,1.86163,-1.70995,0.28256,1.81791,-1.85229,0.050899,1.74516,-1.64602,-0.055871,1.65318,-1.6162,0.085721,1.57257,-1.63168,0.159771,1.5188,-1.62725,0.239585,1.89843,-2.09775,0.294371,1.82514,-1.92418,0.100169,1.86187,-2.03223,0.100793,1.68946,-2.21109,0.149062,1.59456,-2.24493,0.298562,1.54531,-2.12031,0.357017,1.51215,-2.11216,0.192283,1.40699,-1.82656,0.011207,1.46534,-1.89032,0.019125,1.49008,-1.954,0.02507,1.46252,-2.00562,0.210049,1.38729,-2.07721,0.264158,1.12404,-1.75274,0.144943,1.06086,-1.77954,0.182301,0.987342,-1.80083,0.29698,0.911022,-1.78557,0.36045,0.730588,-1.77982,0.377948,0.657121,-1.79297,0.251117,0.702986,-1.79569,0.269585,0.633541,-1.80318,0.368141,0.426838,-1.82304,0.289775,0.354087,-1.90771,0.53273,0.426045,-1.85691,0.518079,0.360138,-1.79418,0.131171,1.07361,-2.11506,0.121871,0.990271,-2.09718,-0.009973,1.08826,-2.08966,-0.016281,1.00715,-2.07382,-0.007764,0.875715,-2.07516,-0.177699,0.806397,-2.07494,-0.185431,0.918547,-2.08021,-0.255996,0.909394,-2.07542,-0.481585,0.862256,-2.06053,-0.517418,0.953839,-1.99424,-0.599776,0.68429,-2.02588,-0.633173,0.747435,-2.08258,0.117571,1.31856,-1.83018,0.1197,1.30753,-2.04513},
/*407*/{0.195491,1.85628,-1.70922,0.282625,1.81331,-1.85224,0.055291,1.73664,-1.64228,-0.049836,1.64349,-1.6101,0.092723,1.56564,-1.62766,0.167936,1.51355,-1.6253,0.238205,1.89398,-2.0984,0.294876,1.82059,-1.92464,0.098568,1.85757,-2.03008,0.095339,1.68509,-2.21005,0.143026,1.59042,-2.24488,0.293712,1.53509,-2.12354,0.35208,1.50002,-2.11671,0.192619,1.40201,-1.82488,0.011466,1.46023,-1.8907,0.017914,1.48574,-1.95373,0.024161,1.45826,-2.00576,0.210582,1.38361,-2.07615,0.264034,1.11918,-1.75101,0.144949,1.05886,-1.77991,0.180888,0.9842,-1.79972,0.293117,0.905358,-1.78482,0.353382,0.723491,-1.78026,0.369259,0.650015,-1.79433,0.244179,0.698941,-1.79712,0.260167,0.629059,-1.80435,0.353277,0.421322,-1.82451,0.269233,0.35456,-1.90782,0.517763,0.410831,-1.85588,0.497505,0.343541,-1.79434,0.137752,1.06925,-2.11494,0.12974,0.984746,-2.09798,-0.004227,1.08144,-2.08943,-0.008449,0.999994,-2.07291,0.002495,0.869406,-2.07548,-0.169478,0.805162,-2.07508,-0.175101,0.918892,-2.08041,-0.244218,0.909531,-2.07536,-0.471809,0.869507,-2.06058,-0.50447,0.962646,-1.99501,-0.591903,0.695264,-2.02851,-0.62402,0.757827,-2.08423,0.117279,1.31377,-1.83,0.120304,1.30346,-2.04496},
/*408*/{0.1977,1.85081,-1.70803,0.282123,1.80944,-1.85293,0.060373,1.72819,-1.63835,-0.042007,1.63434,-1.60349,0.100733,1.5589,-1.62357,0.176232,1.5084,-1.62261,0.235423,1.8892,-2.09865,0.294564,1.81608,-1.92498,0.097663,1.85168,-2.02904,0.090372,1.68121,-2.20879,0.136725,1.58625,-2.24429,0.289024,1.52559,-2.12736,0.344802,1.48892,-2.122,0.19166,1.39772,-1.82327,0.009307,1.45582,-1.89103,0.017802,1.48096,-1.95389,0.023696,1.45274,-2.00641,0.211109,1.37991,-2.07502,0.262158,1.11472,-1.75136,0.142459,1.05371,-1.78029,0.179267,0.980072,-1.79956,0.287604,0.899669,-1.78483,0.34662,0.717083,-1.78197,0.360225,0.643314,-1.79604,0.237003,0.696782,-1.79765,0.250792,0.626113,-1.80739,0.335864,0.418718,-1.82603,0.248051,0.356095,-1.90708,0.502555,0.398099,-1.85896,0.480365,0.335939,-1.79433,0.14332,1.0645,-2.11528,0.13703,0.980791,-2.09781,0.001891,1.0744,-2.08892,-0.002101,0.99302,-2.07241,0.011632,0.863038,-2.07554,-0.160604,0.803763,-2.07518,-0.162707,0.917115,-2.08018,-0.232274,0.910612,-2.07575,-0.459394,0.87748,-2.0616,-0.490545,0.969915,-1.99535,-0.584934,0.703664,-2.02948,-0.613231,0.767537,-2.0857,0.116307,1.3094,-1.82943,0.121138,1.29904,-2.04436},
/*409*/{0.198587,1.84585,-1.70789,0.282402,1.80479,-1.85334,0.065335,1.71913,-1.63412,-0.035297,1.62411,-1.59748,0.107949,1.55203,-1.62112,0.185802,1.50373,-1.62055,0.233908,1.88493,-2.0993,0.294418,1.81218,-1.92628,0.096265,1.84694,-2.02774,0.085114,1.67825,-2.20877,0.130503,1.58188,-2.2435,0.284222,1.51451,-2.13143,0.338975,1.47639,-2.12748,0.190914,1.39375,-1.82127,0.008026,1.45154,-1.8926,0.016113,1.47566,-1.95456,0.02307,1.4476,-2.0072,0.210996,1.37696,-2.07525,0.261691,1.10947,-1.74873,0.141141,1.05102,-1.7804,0.177744,0.976154,-1.80071,0.281666,0.893697,-1.78342,0.339069,0.712817,-1.78279,0.348511,0.638424,-1.79682,0.228972,0.696175,-1.79964,0.240059,0.625481,-1.80885,0.319238,0.418173,-1.82868,0.226339,0.356501,-1.90523,0.482374,0.393087,-1.86539,0.458119,0.339373,-1.79635,0.148869,1.06028,-2.11531,0.144139,0.976175,-2.09815,0.00772,1.06694,-2.08736,0.005285,0.986396,-2.07197,0.021844,0.856843,-2.07624,-0.15035,0.802146,-2.07526,-0.149856,0.915457,-2.07985,-0.220064,0.910593,-2.07599,-0.446952,0.883261,-2.06146,-0.478001,0.975765,-1.99462,-0.57361,0.7116,-2.03065,-0.602917,0.775426,-2.08712,0.115678,1.30526,-1.82955,0.121957,1.295,-2.04444},
/*410*/{0.200693,1.84116,-1.70685,0.284006,1.79779,-1.85183,0.070575,1.71015,-1.63045,-0.028329,1.61464,-1.59109,0.116833,1.54659,-1.61762,0.194325,1.49897,-1.61905,0.232701,1.88106,-2.10009,0.295102,1.80713,-1.92678,0.095255,1.84165,-2.0255,0.080905,1.67253,-2.20711,0.124056,1.57799,-2.24379,0.278423,1.50398,-2.13596,0.331323,1.46416,-2.13286,0.188403,1.39006,-1.82037,0.004986,1.44829,-1.89304,0.013891,1.47087,-1.95536,0.0216,1.44221,-2.00789,0.211142,1.37214,-2.07336,0.259116,1.10781,-1.74686,0.138903,1.0483,-1.77934,0.175887,0.974543,-1.80232,0.275082,0.887547,-1.78462,0.332314,0.711646,-1.78298,0.341471,0.637232,-1.79811,0.221083,0.699853,-1.80101,0.2296,0.629465,-1.80973,0.301097,0.419382,-1.83077,0.204102,0.35706,-1.90716,0.459968,0.392801,-1.86796,0.435466,0.340054,-1.7977,0.154907,1.05579,-2.11462,0.152116,0.972027,-2.09823,0.013219,1.06065,-2.08687,0.013464,0.979453,-2.0713,0.031357,0.846224,-2.07458,-0.140014,0.800314,-2.07625,-0.135435,0.913363,-2.0805,-0.207455,0.910706,-2.07609,-0.434815,0.888338,-2.06189,-0.46666,0.982777,-1.99588,-0.56259,0.718121,-2.03155,-0.588731,0.782075,-2.08924,0.113074,1.30175,-1.82831,0.12142,1.29008,-2.04307},
/*411*/{0.201884,1.83614,-1.70595,0.286157,1.79268,-1.85266,0.076992,1.70108,-1.62694,-0.019242,1.6047,-1.58486,0.125763,1.53925,-1.61376,0.203585,1.49487,-1.61697,0.231231,1.87702,-2.10144,0.295697,1.80182,-1.92707,0.094091,1.83804,-2.02357,0.07629,1.66911,-2.20615,0.11705,1.57395,-2.24354,0.27171,1.49272,-2.14135,0.32437,1.45256,-2.13828,0.186577,1.38678,-1.8194,0.001306,1.4446,-1.89378,0.011573,1.46522,-1.95676,0.019534,1.43785,-2.00915,0.208415,1.36776,-2.073,0.252135,1.10752,-1.74601,0.135038,1.04795,-1.78071,0.173253,0.974348,-1.80323,0.270296,0.882949,-1.78497,0.327,0.713915,-1.78401,0.331752,0.639204,-1.79814,0.214922,0.706413,-1.80044,0.220372,0.634977,-1.80941,0.27931,0.42245,-1.8336,0.181594,0.358264,-1.90777,0.438708,0.391656,-1.86952,0.412851,0.340961,-1.79864,0.160431,1.05238,-2.11428,0.15936,0.968539,-2.09803,0.018849,1.05374,-2.08605,0.020194,0.972792,-2.06997,0.042129,0.841349,-2.07688,-0.128762,0.798836,-2.07608,-0.12374,0.912671,-2.0826,-0.193476,0.910392,-2.07653,-0.417022,0.889537,-2.05975,-0.45227,0.986436,-1.99721,-0.549956,0.722858,-2.03234,-0.575802,0.787371,-2.09055,0.11065,1.29837,-1.82785,0.119705,1.28534,-2.0425},
/*412*/{0.203971,1.83215,-1.7049,0.286585,1.78876,-1.85295,0.084156,1.69214,-1.62346,-0.010568,1.59499,-1.57856,0.134305,1.53429,-1.6123,0.213798,1.49116,-1.61585,0.230554,1.87365,-2.10246,0.29713,1.79829,-1.92837,0.093868,1.83513,-2.02054,0.071914,1.66481,-2.20574,0.110648,1.57017,-2.24301,0.265443,1.48342,-2.14663,0.31628,1.4412,-2.14375,0.183772,1.38412,-1.81809,-0.002591,1.44056,-1.89613,0.008903,1.46119,-1.95793,0.017268,1.43301,-2.01021,0.209851,1.36367,-2.06945,0.253152,1.10627,-1.74085,0.132866,1.04937,-1.78256,0.173645,0.976061,-1.8037,0.270376,0.887266,-1.78413,0.321089,0.715088,-1.78382,0.317611,0.638341,-1.80235,0.2089,0.712743,-1.79922,0.211244,0.641537,-1.80868,0.258838,0.425259,-1.83465,0.160356,0.359956,-1.90926,0.415926,0.390215,-1.86964,0.389885,0.338222,-1.79702,0.165283,1.04958,-2.11403,0.166358,0.965946,-2.09738,0.0228,1.0472,-2.08557,0.026862,0.965617,-2.0695,0.053487,0.836645,-2.07659,-0.116528,0.797029,-2.07566,-0.109739,0.911224,-2.08307,-0.179087,0.908928,-2.07626,-0.410094,0.895829,-2.06116,-0.438911,0.990628,-1.99703,-0.536475,0.727031,-2.03411,-0.559802,0.790272,-2.09159,0.107501,1.29552,-1.8265,0.1203,1.28104,-2.04087},
/*413*/{0.205749,1.8276,-1.7036,0.286217,1.78411,-1.85311,0.091022,1.68441,-1.62052,-0.000471,1.58638,-1.5729,0.144462,1.52925,-1.61005,0.224405,1.48815,-1.61446,0.229445,1.87066,-2.10419,0.297822,1.79473,-1.92992,0.093274,1.83396,-2.01748,0.06762,1.66174,-2.20504,0.103852,1.56673,-2.24183,0.258166,1.47446,-2.15104,0.307536,1.43104,-2.14832,0.181165,1.38176,-1.81566,-0.005451,1.43681,-1.89772,0.006066,1.45753,-1.95878,0.013715,1.4291,-2.01209,0.209226,1.36095,-2.06674,0.248467,1.10652,-1.73773,0.131251,1.05056,-1.78502,0.172642,0.976448,-1.8046,0.273996,0.89374,-1.78195,0.314846,0.715636,-1.78378,0.309036,0.638747,-1.80099,0.202069,0.717519,-1.79779,0.200668,0.646628,-1.80811,0.240086,0.425822,-1.83456,0.137597,0.361126,-1.90912,0.394984,0.388682,-1.86916,0.368417,0.338233,-1.79575,0.169743,1.04852,-2.11349,0.174013,0.965055,-2.09759,0.0277,1.04117,-2.08478,0.035642,0.960114,-2.06847,0.065338,0.832696,-2.07616,-0.10315,0.795143,-2.07526,-0.095116,0.90908,-2.08345,-0.164575,0.908222,-2.07654,-0.393205,0.898041,-2.06404,-0.426174,0.993535,-1.99905,-0.5218,0.728451,-2.03537,-0.546901,0.792266,-2.09321,0.104431,1.29301,-1.82548,0.119955,1.27763,-2.03961},
/*414*/{0.206444,1.82486,-1.70308,0.287047,1.77986,-1.85302,0.097537,1.67761,-1.6196,0.010743,1.5784,-1.5672,0.155226,1.52549,-1.60753,0.235041,1.48595,-1.61395,0.228844,1.86838,-2.10495,0.29854,1.7914,-1.93085,0.092838,1.83388,-2.01505,0.064989,1.66018,-2.20437,0.097447,1.56453,-2.2407,0.250081,1.4668,-2.15424,0.298035,1.42256,-2.15412,0.177616,1.38074,-1.813,-0.007606,1.43376,-1.90071,0.003936,1.45577,-1.95935,0.011108,1.42393,-2.01439,0.20726,1.35813,-2.0663,0.245958,1.10834,-1.7361,0.131773,1.04887,-1.78291,0.170725,0.975672,-1.80445,0.279149,0.899943,-1.7813,0.307564,0.714998,-1.78379,0.303066,0.640909,-1.79873,0.195102,0.720906,-1.79528,0.190446,0.649413,-1.8055,0.220934,0.425665,-1.8336,0.117018,0.362062,-1.9087,0.374924,0.388718,-1.86786,0.347374,0.33829,-1.79701,0.17486,1.04929,-2.11241,0.18222,0.964933,-2.09607,0.032779,1.03575,-2.08423,0.042324,0.955926,-2.0685,0.078673,0.829845,-2.0756,-0.089241,0.792788,-2.07423,-0.080261,0.907015,-2.08415,-0.149894,0.906927,-2.07744,-0.378625,0.899261,-2.0643,-0.412198,0.994091,-1.9996,-0.506568,0.729482,-2.03593,-0.531805,0.792868,-2.09459,0.101333,1.29166,-1.82458,0.118945,1.27413,-2.03837},
/*415*/{0.206362,1.82285,-1.70252,0.286962,1.77851,-1.85441,0.104883,1.67388,-1.61741,0.019782,1.57179,-1.56271,0.165422,1.52271,-1.60626,0.246261,1.48521,-1.61297,0.229695,1.86668,-2.10531,0.298936,1.7882,-1.93139,0.093213,1.83359,-2.01453,0.061978,1.66032,-2.20158,0.091287,1.56459,-2.23961,0.24185,1.4621,-2.15576,0.288202,1.41449,-2.15901,0.17586,1.38118,-1.80901,-0.010199,1.43152,-1.90325,0.002844,1.4534,-1.96019,0.007883,1.41882,-2.01717,0.205791,1.3566,-2.06492,0.242661,1.10774,-1.73539,0.127921,1.04756,-1.78359,0.170174,0.975122,-1.80355,0.284575,0.905237,-1.78166,0.297727,0.713883,-1.7834,0.291108,0.639175,-1.79816,0.185539,0.723583,-1.79379,0.180236,0.652812,-1.803,0.202558,0.428516,-1.83435,0.09579,0.362314,-1.90849,0.353475,0.38706,-1.86727,0.325043,0.337941,-1.7953,0.179665,1.0499,-2.11177,0.189232,0.966837,-2.09684,0.038197,1.03275,-2.08435,0.050572,0.951927,-2.06756,0.09287,0.827983,-2.07472,-0.075292,0.790759,-2.07326,-0.065341,0.904491,-2.08424,-0.135103,0.90572,-2.07812,-0.363107,0.898789,-2.06442,-0.398168,0.993622,-2.00174,-0.490076,0.72793,-2.03701,-0.51586,0.791542,-2.09568,0.099483,1.2916,-1.82282,0.119271,1.27119,-2.03617},
/*416*/{0.206593,1.82165,-1.70256,0.287198,1.77745,-1.85523,0.109587,1.67108,-1.61749,0.031622,1.56727,-1.55936,0.176985,1.52093,-1.60649,0.257398,1.4847,-1.61266,0.230444,1.86463,-2.10533,0.299042,1.78617,-1.93314,0.09381,1.8338,-2.01442,0.056657,1.66184,-2.1983,0.083697,1.56615,-2.23769,0.233064,1.45838,-2.15828,0.277717,1.40847,-2.16329,0.172124,1.38027,-1.80565,-0.012258,1.43001,-1.90493,0.001441,1.45316,-1.96066,0.005962,1.41604,-2.01879,0.203572,1.35437,-2.0638,0.238339,1.10612,-1.73503,0.123991,1.04646,-1.78367,0.163993,0.973617,-1.80485,0.288694,0.906036,-1.78298,0.287194,0.711504,-1.78241,0.278638,0.637233,-1.79761,0.175649,0.72535,-1.79292,0.167724,0.654105,-1.80264,0.181321,0.427691,-1.8337,0.074699,0.362922,-1.90758,0.332611,0.38658,-1.86795,0.304711,0.337192,-1.79465,0.185219,1.05095,-2.11167,0.197355,0.96805,-2.0959,0.043584,1.03018,-2.08414,0.05958,0.949187,-2.06818,0.107167,0.826604,-2.07382,-0.060238,0.788239,-2.07248,-0.050112,0.900993,-2.08329,-0.119358,0.903183,-2.07779,-0.348863,0.897278,-2.06565,-0.384675,0.992589,-2.00319,-0.472214,0.72487,-2.03786,-0.498307,0.787541,-2.09731,0.095476,1.29085,-1.82167,0.117481,1.26897,-2.03465},
/*417*/{0.207055,1.82096,-1.70344,0.286665,1.77671,-1.85622,0.115792,1.66968,-1.61613,0.043236,1.56371,-1.55626,0.18835,1.51975,-1.60573,0.269956,1.4857,-1.61306,0.230861,1.86337,-2.10485,0.298364,1.78481,-1.93317,0.094006,1.83405,-2.01461,0.05173,1.66494,-2.19526,0.076882,1.56932,-2.23634,0.224084,1.45568,-2.16021,0.267506,1.40444,-2.16703,0.169719,1.38007,-1.80252,-0.013301,1.4295,-1.90494,0.000914,1.45299,-1.96014,0.005219,1.41586,-2.0189,0.202845,1.35371,-2.06252,0.233436,1.1036,-1.7344,0.118611,1.04537,-1.78467,0.159008,0.972176,-1.80455,0.283236,0.902126,-1.78279,0.275658,0.707826,-1.78187,0.263712,0.63364,-1.79641,0.164641,0.724852,-1.79172,0.153965,0.654306,-1.80206,0.161218,0.426899,-1.83263,0.052745,0.36356,-1.90755,0.310687,0.386269,-1.86685,0.283173,0.33664,-1.79348,0.192352,1.05238,-2.111,0.206315,0.969662,-2.09417,0.051752,1.02633,-2.08435,0.068449,0.946956,-2.06854,0.1208,0.825322,-2.07203,-0.045412,0.787139,-2.06976,-0.033566,0.898513,-2.08341,-0.10341,0.900996,-2.07776,-0.333844,0.894043,-2.06528,-0.372646,0.989719,-2.00433,-0.455859,0.720258,-2.03764,-0.481058,0.78248,-2.09811,0.093102,1.29059,-1.82043,0.116795,1.26845,-2.03321},
/*418*/{0.20868,1.82071,-1.70399,0.286967,1.77419,-1.85644,0.121876,1.66935,-1.61544,0.05415,1.5612,-1.55298,0.199681,1.51991,-1.6061,0.281291,1.48765,-1.61399,0.231689,1.8626,-2.10461,0.298202,1.78369,-1.93326,0.094324,1.83426,-2.01574,0.046953,1.66936,-2.19119,0.069754,1.57294,-2.23474,0.215366,1.45532,-2.16244,0.25653,1.40199,-2.17092,0.166639,1.38052,-1.79983,-0.014762,1.42982,-1.90649,0.00038,1.45458,-1.96027,0.003077,1.41708,-2.01935,0.201516,1.35389,-2.06155,0.229526,1.09861,-1.73384,0.113828,1.04334,-1.78477,0.153169,0.970169,-1.80591,0.274383,0.89722,-1.78141,0.263274,0.703709,-1.78128,0.249055,0.629654,-1.79628,0.152192,0.724666,-1.79165,0.139682,0.653708,-1.80115,0.140621,0.427299,-1.83192,0.031062,0.364137,-1.9064,0.288553,0.385816,-1.8656,0.261153,0.336843,-1.79324,0.199096,1.05405,-2.11057,0.215095,0.972455,-2.0948,0.058241,1.02448,-2.0849,0.078245,0.945742,-2.06944,0.137131,0.825956,-2.07056,-0.029708,0.784402,-2.06873,-0.018474,0.89668,-2.08384,-0.088742,0.899728,-2.07885,-0.318808,0.891421,-2.06551,-0.362038,0.986575,-2.00723,-0.436491,0.715309,-2.03792,-0.4629,0.775101,-2.09914,0.089621,1.29124,-1.81995,0.115163,1.26899,-2.0325},
/*419*/{0.209904,1.82123,-1.70514,0.286941,1.77339,-1.85667,0.127608,1.66908,-1.61417,0.063923,1.55986,-1.55129,0.210862,1.52151,-1.60612,0.293105,1.49128,-1.61501,0.232766,1.86229,-2.10427,0.298163,1.78279,-1.93345,0.094027,1.83474,-2.01641,0.041741,1.67494,-2.18747,0.062399,1.57744,-2.23349,0.206591,1.45566,-2.1647,0.246403,1.40111,-2.17372,0.16454,1.38101,-1.7972,-0.015753,1.43017,-1.9054,-0.001575,1.45455,-1.95913,0.003014,1.4184,-2.01882,0.200154,1.35433,-2.06103,0.222243,1.09511,-1.73303,0.107269,1.0427,-1.78589,0.145546,0.966948,-1.80626,0.263537,0.891825,-1.78068,0.249706,0.699764,-1.78072,0.233713,0.626308,-1.79638,0.139426,0.722583,-1.79086,0.125286,0.652054,-1.80199,0.119976,0.427752,-1.83153,0.00955,0.365448,-1.90605,0.267377,0.384899,-1.86513,0.238963,0.337212,-1.79331,0.206819,1.05638,-2.1104,0.226013,0.974568,-2.0941,0.067353,1.02255,-2.08548,0.089507,0.943705,-2.06877,0.153484,0.826129,-2.0695,-0.01426,0.782788,-2.06779,-0.003325,0.894733,-2.08416,-0.072748,0.895971,-2.07926,-0.303974,0.887486,-2.06581,-0.35079,0.981299,-2.0086,-0.415448,0.706866,-2.03524,-0.444021,0.766395,-2.09997,0.087772,1.2915,-1.81892,0.114068,1.26963,-2.03142},
/*420*/{0.211162,1.82199,-1.70585,0.286537,1.77301,-1.8573,0.132985,1.66927,-1.61338,0.074052,1.55963,-1.54892,0.221186,1.52371,-1.60597,0.305275,1.4956,-1.61604,0.232497,1.8624,-2.10446,0.296952,1.78234,-1.93371,0.094164,1.83579,-2.017,0.036753,1.68063,-2.1837,0.054724,1.58284,-2.23234,0.197468,1.45578,-2.16675,0.236114,1.40158,-2.17651,0.163639,1.38152,-1.79603,-0.016733,1.43113,-1.90544,-0.003212,1.45507,-1.95912,0.002091,1.41976,-2.0175,0.197725,1.35506,-2.06023,0.220443,1.09283,-1.73301,0.098238,1.04092,-1.78776,0.137139,0.965074,-1.80731,0.251844,0.885754,-1.78008,0.235594,0.694656,-1.78058,0.217517,0.621963,-1.7959,0.125781,0.719717,-1.79083,0.110405,0.650266,-1.80131,0.098079,0.427325,-1.83046,-0.011282,0.365702,-1.90534,0.246684,0.384335,-1.86486,0.217324,0.336782,-1.79208,0.21514,1.0596,-2.11017,0.236311,0.978619,-2.09332,0.076558,1.02371,-2.08736,0.100684,0.943032,-2.06995,0.168622,0.83038,-2.06903,0.001601,0.781083,-2.06619,0.011377,0.893158,-2.08416,-0.058018,0.894348,-2.07917,-0.285685,0.880351,-2.06493,-0.338713,0.975146,-2.00929,-0.395464,0.698752,-2.03599,-0.424892,0.756084,-2.10089,0.0862,1.29229,-1.81826,0.112332,1.27033,-2.03077},
/*421*/{0.212694,1.82436,-1.70633,0.285365,1.7745,-1.85864,0.137437,1.67072,-1.6119,0.083593,1.55941,-1.54693,0.23121,1.52735,-1.60654,0.316077,1.50156,-1.61869,0.231702,1.86314,-2.10455,0.296438,1.78265,-1.93423,0.093959,1.83748,-2.0177,0.03216,1.68722,-2.18063,0.046517,1.58866,-2.23167,0.189376,1.45985,-2.16913,0.225358,1.40361,-2.17913,0.160949,1.38243,-1.79421,-0.016347,1.4342,-1.905,-0.003763,1.45551,-1.9573,0.000958,1.42148,-2.0165,0.196161,1.35737,-2.05997,0.2128,1.0877,-1.73337,0.093244,1.04003,-1.78654,0.128138,0.962834,-1.80777,0.239685,0.879934,-1.77972,0.220107,0.690272,-1.7806,0.201908,0.617752,-1.79626,0.111731,0.717654,-1.79113,0.095053,0.648578,-1.80157,0.077682,0.427656,-1.83074,-0.03224,0.366285,-1.90493,0.224811,0.384368,-1.86461,0.196734,0.337124,-1.79159,0.223143,1.06363,-2.1098,0.247348,0.983061,-2.09321,0.086248,1.02267,-2.08699,0.112404,0.943813,-2.0701,0.184275,0.832832,-2.06876,0.018165,0.780419,-2.06607,0.026965,0.891968,-2.08376,-0.043496,0.892723,-2.0789,-0.276213,0.87645,-2.06567,-0.329422,0.968644,-2.01145,-0.373835,0.689313,-2.03579,-0.404735,0.745094,-2.10156,0.084529,1.29349,-1.81756,0.110961,1.27218,-2.0301},
/*422*/{0.21439,1.8267,-1.70697,0.284902,1.77465,-1.85889,0.141779,1.67267,-1.6109,0.092059,1.56,-1.5451,0.240447,1.53174,-1.60652,0.326708,1.50812,-1.62041,0.231548,1.8641,-2.10508,0.295791,1.78345,-1.93472,0.09366,1.83906,-2.01748,0.027885,1.69436,-2.17806,0.039171,1.5951,-2.23109,0.180308,1.46278,-2.17092,0.215344,1.40636,-2.1814,0.159502,1.38353,-1.7923,-0.017657,1.43269,-1.90352,-0.004727,1.45607,-1.95732,-0.000653,1.42236,-2.01538,0.195438,1.35948,-2.0595,0.206015,1.08412,-1.73485,0.085686,1.03869,-1.786,0.119064,0.961984,-1.80655,0.227251,0.873375,-1.78018,0.204691,0.685798,-1.78163,0.184353,0.613395,-1.79711,0.096396,0.714675,-1.79216,0.078207,0.646335,-1.80287,0.057984,0.427959,-1.82971,-0.053304,0.366263,-1.90459,0.204661,0.384838,-1.8641,0.175602,0.33625,-1.79015,0.231804,1.06798,-2.11007,0.258082,0.988966,-2.09333,0.096186,1.02313,-2.08778,0.124772,0.944503,-2.06997,0.200055,0.835342,-2.06756,0.034152,0.779594,-2.06472,0.040843,0.892335,-2.08428,-0.029414,0.892112,-2.07954,-0.260632,0.869662,-2.06648,-0.319834,0.960135,-2.01329,-0.35203,0.679792,-2.03523,-0.384394,0.733625,-2.10187,0.084071,1.29367,-1.81707,0.111122,1.27358,-2.02966},
/*423*/{0.2164,1.82981,-1.70803,0.286104,1.77622,-1.86068,0.145687,1.67549,-1.61058,0.100855,1.56057,-1.54329,0.249626,1.53665,-1.60749,0.335749,1.51514,-1.62285,0.230084,1.86554,-2.10532,0.29523,1.78427,-1.93572,0.09253,1.84159,-2.01787,0.022051,1.70111,-2.17541,0.031061,1.6019,-2.23097,0.171611,1.46809,-2.17318,0.20573,1.4111,-2.18335,0.158064,1.38398,-1.79142,-0.018236,1.43394,-1.90206,-0.005144,1.45673,-1.9559,2.6e-005,1.42542,-2.01325,0.194179,1.36201,-2.05929,0.202935,1.08328,-1.73527,0.079871,1.04105,-1.78488,0.110113,0.96124,-1.80685,0.213494,0.868391,-1.78038,0.189293,0.681851,-1.7825,0.167565,0.610342,-1.79753,0.081243,0.713292,-1.79104,0.061801,0.645096,-1.8029,0.037376,0.428211,-1.82996,-0.074236,0.366722,-1.90492,0.182825,0.383921,-1.8638,0.153943,0.336023,-1.79072,0.238878,1.07422,-2.11109,0.26922,0.99558,-2.093,0.105346,1.02435,-2.0883,0.137149,0.946898,-2.07123,0.215221,0.83904,-2.06855,0.050578,0.779825,-2.06426,0.054676,0.892558,-2.08368,-0.015291,0.890591,-2.07921,-0.241425,0.859407,-2.06367,-0.309914,0.950184,-2.01392,-0.329693,0.669459,-2.03579,-0.36446,0.72088,-2.10199,0.083298,1.294,-1.81699,0.110403,1.27595,-2.02975},
/*424*/{0.218852,1.83231,-1.70842,0.286406,1.77812,-1.86166,0.151136,1.67868,-1.61064,0.109481,1.56288,-1.54207,0.258497,1.54252,-1.60782,0.345458,1.52268,-1.62586,0.22808,1.86727,-2.10533,0.29462,1.78608,-1.93717,0.091558,1.84407,-2.01748,0.016674,1.70888,-2.17401,0.02266,1.60909,-2.23023,0.163068,1.4734,-2.17472,0.195757,1.41663,-2.18479,0.157898,1.38424,-1.79139,-0.018988,1.43482,-1.90184,-0.005918,1.45847,-1.95518,-0.002804,1.4268,-2.0122,0.193474,1.36378,-2.05884,0.194954,1.08002,-1.73615,0.072345,1.04242,-1.78404,0.099256,0.962194,-1.80542,0.199215,0.864395,-1.78058,0.173534,0.679979,-1.78244,0.152129,0.608451,-1.79817,0.065941,0.712699,-1.79238,0.044865,0.64499,-1.80366,0.017043,0.428644,-1.82989,-0.095345,0.367995,-1.90504,0.161968,0.384042,-1.86407,0.132698,0.336486,-1.79071,0.246557,1.08093,-2.1115,0.279172,1.00315,-2.0932,0.114596,1.02615,-2.08863,0.148891,0.949826,-2.07127,0.229541,0.842819,-2.06873,0.067609,0.780433,-2.06415,0.068684,0.893339,-2.08316,-0.002146,0.889659,-2.07983,-0.231819,0.854527,-2.06755,-0.30167,0.940169,-2.01495,-0.307455,0.658017,-2.0345,-0.341908,0.707706,-2.10244,0.082645,1.29462,-1.8171,0.109168,1.27756,-2.03001},
/*425*/{0.221911,1.83577,-1.70905,0.286569,1.77984,-1.86273,0.156255,1.68166,-1.61013,0.118788,1.56556,-1.54173,0.266272,1.54797,-1.60927,0.353408,1.53095,-1.62833,0.226266,1.86978,-2.10623,0.294063,1.78752,-1.93841,0.09054,1.84655,-2.01569,0.011213,1.71553,-2.17288,0.0148,1.61653,-2.22999,0.154248,1.47939,-2.17572,0.186647,1.42334,-2.18586,0.157804,1.38357,-1.79111,-0.018954,1.43703,-1.9009,-0.006721,1.45972,-1.954,-0.003529,1.42814,-2.01117,0.192601,1.36619,-2.05907,0.189564,1.07892,-1.73754,0.063528,1.04384,-1.78368,0.089435,0.963743,-1.80445,0.182287,0.861242,-1.78153,0.157865,0.67818,-1.78402,0.136585,0.60888,-1.79959,0.050811,0.712431,-1.79239,0.028943,0.645062,-1.80368,-0.004232,0.429185,-1.82981,-0.116301,0.368171,-1.90478,0.141695,0.383724,-1.86415,0.112254,0.336118,-1.79024,0.253606,1.08808,-2.11158,0.288498,1.01158,-2.0932,0.123324,1.02921,-2.08829,0.162906,0.956248,-2.07179,0.24463,0.847855,-2.0687,0.083609,0.781351,-2.06393,0.081441,0.894566,-2.08358,0.011664,0.888518,-2.07987,-0.217844,0.845655,-2.0669,-0.291862,0.927977,-2.01527,-0.284159,0.64631,-2.03435,-0.321539,0.694129,-2.10268,0.08226,1.2948,-1.81755,0.108123,1.27953,-2.03068},
/*426*/{0.225094,1.83932,-1.71016,0.287894,1.78306,-1.86385,0.161073,1.68501,-1.61034,0.12696,1.56837,-1.54075,0.274187,1.55437,-1.61065,0.360621,1.53914,-1.63177,0.223053,1.8726,-2.10679,0.293755,1.79005,-1.94004,0.089252,1.84894,-2.01499,0.00534,1.72257,-2.17164,0.006897,1.62392,-2.22941,0.146303,1.48667,-2.17669,0.178283,1.43007,-2.18683,0.158543,1.3843,-1.79135,-0.01915,1.4386,-1.89985,-0.006976,1.46193,-1.95341,-0.005041,1.43017,-2.01037,0.192143,1.36939,-2.05976,0.181856,1.07752,-1.73827,0.056576,1.04591,-1.7827,0.079251,0.964593,-1.80399,0.16622,0.858779,-1.78263,0.141561,0.676981,-1.785,0.117323,0.606896,-1.80045,0.035553,0.713408,-1.79279,0.012713,0.646424,-1.80348,-0.02373,0.430092,-1.83008,-0.136793,0.368968,-1.90513,0.120716,0.383039,-1.86326,0.091161,0.33643,-1.79012,0.259159,1.09599,-2.11167,0.296309,1.02033,-2.09325,0.131811,1.03343,-2.0887,0.169408,0.961593,-2.07257,0.258966,0.85417,-2.06888,0.099728,0.782805,-2.06434,0.093036,0.897414,-2.08264,0.024286,0.887271,-2.07957,-0.200159,0.834805,-2.0649,-0.282386,0.915365,-2.01395,-0.259453,0.633447,-2.03414,-0.299762,0.680055,-2.10283,0.082823,1.29581,-1.81828,0.107621,1.28225,-2.03165},
/*427*/{0.228214,1.84293,-1.71129,0.287547,1.78559,-1.86541,0.166025,1.6891,-1.61039,0.134073,1.5712,-1.54031,0.28054,1.56099,-1.61238,0.367559,1.54845,-1.63483,0.220242,1.8757,-2.10732,0.293139,1.79316,-1.94195,0.08766,1.85081,-2.01378,-0.001304,1.72884,-2.16899,-0.000109,1.63125,-2.22942,0.136955,1.49362,-2.17673,0.170032,1.4375,-2.18749,0.159544,1.38453,-1.79159,-0.018637,1.44121,-1.89816,-0.007055,1.46392,-1.95173,-0.005072,1.43184,-2.00975,0.190651,1.37098,-2.05965,0.175248,1.07714,-1.73997,0.048565,1.0472,-1.78225,0.068759,0.966153,-1.80389,0.150843,0.857846,-1.78422,0.125329,0.676224,-1.78615,0.100276,0.606746,-1.80092,0.019817,0.715347,-1.79353,-0.003029,0.646813,-1.80474,-0.044515,0.430911,-1.83018,-0.156726,0.370178,-1.90535,0.100285,0.383009,-1.86392,0.070373,0.336688,-1.79004,0.264824,1.10449,-2.11204,0.304608,1.02979,-2.09354,0.139262,1.03661,-2.08782,0.179296,0.966318,-2.07213,0.274083,0.862207,-2.06913,0.116248,0.784964,-2.06539,0.10589,0.898672,-2.08243,0.036766,0.886489,-2.07945,-0.190386,0.826324,-2.06734,-0.272616,0.900328,-2.0143,-0.235586,0.620695,-2.03451,-0.278386,0.664856,-2.10257,0.082965,1.29688,-1.81807,0.106158,1.28392,-2.03166},
/*428*/{0.231088,1.84741,-1.71256,0.287935,1.78946,-1.86684,0.171991,1.69289,-1.61071,0.14243,1.57487,-1.5399,0.287608,1.56785,-1.61415,0.3738,1.55687,-1.63836,0.217318,1.87983,-2.10779,0.292647,1.7961,-1.94365,0.086061,1.85449,-2.01196,-0.00764,1.73608,-2.16853,-0.00834,1.6385,-2.22866,0.12984,1.50157,-2.17746,0.161793,1.44565,-2.18822,0.160827,1.38547,-1.79265,-0.018645,1.44358,-1.89626,-0.006503,1.46623,-1.95054,-0.006206,1.43423,-2.00864,0.189927,1.37382,-2.06012,0.171211,1.07656,-1.74018,0.040282,1.04888,-1.78026,0.059383,0.968365,-1.80372,0.13485,0.857175,-1.78498,0.109337,0.675988,-1.78772,0.081315,0.606929,-1.80355,0.003214,0.715469,-1.79447,-0.021585,0.648564,-1.8038,-0.062963,0.433085,-1.83043,-0.177303,0.372162,-1.90493,0.07962,0.383066,-1.86403,0.049835,0.336827,-1.79101,0.269682,1.11296,-2.11192,0.311866,1.03961,-2.09382,0.146361,1.04172,-2.08797,0.188935,0.972816,-2.07193,0.287201,0.87108,-2.06891,0.132507,0.787836,-2.06567,0.11766,0.89981,-2.08243,0.049391,0.885427,-2.07963,-0.175831,0.815627,-2.06793,-0.263161,0.885272,-2.01359,-0.210182,0.607924,-2.03465,-0.256099,0.650232,-2.10253,0.083859,1.29825,-1.81847,0.105203,1.28661,-2.03233},
/*429*/{0.233639,1.85164,-1.71408,0.289607,1.7924,-1.8683,0.177597,1.69691,-1.61138,0.149728,1.57892,-1.53928,0.294202,1.57549,-1.61651,0.37988,1.56597,-1.64216,0.213472,1.88386,-2.10786,0.292137,1.7998,-1.94547,0.084464,1.85588,-2.01099,-0.01374,1.74272,-2.16742,-0.01572,1.64605,-2.228,0.123314,1.50975,-2.17778,0.154452,1.45371,-2.18905,0.161729,1.38612,-1.79364,-0.017569,1.44746,-1.89479,-0.007108,1.46976,-1.94862,-0.004926,1.43624,-2.00663,0.189353,1.37694,-2.06017,0.162313,1.07466,-1.74185,0.031651,1.04989,-1.78009,0.048227,0.971258,-1.80305,0.118875,0.857622,-1.78607,0.091738,0.676492,-1.78851,0.064321,0.607211,-1.80337,-0.012743,0.716895,-1.79462,-0.038387,0.65044,-1.80446,-0.081905,0.433955,-1.83022,-0.197818,0.374592,-1.9043,0.059137,0.382931,-1.86465,0.029181,0.33739,-1.7914,0.27456,1.12168,-2.11255,0.319363,1.04969,-2.0939,0.153285,1.04716,-2.08781,0.197668,0.978891,-2.07186,0.301248,0.881634,-2.06939,0.147989,0.790351,-2.06676,0.128429,0.901977,-2.08244,0.058631,0.884508,-2.07919,-0.163756,0.804298,-2.06715,-0.252607,0.869308,-2.01285,-0.184213,0.595749,-2.03541,-0.232869,0.634967,-2.10266,0.084416,1.29993,-1.81858,0.10426,1.28965,-2.03265},
/*430*/{0.236459,1.8563,-1.71573,0.290656,1.79612,-1.87027,0.183298,1.70075,-1.61193,0.156392,1.58289,-1.53914,0.299649,1.58286,-1.61825,0.385389,1.57553,-1.64534,0.210045,1.88757,-2.10866,0.291505,1.80311,-1.94752,0.082232,1.85962,-2.00847,-0.019681,1.74941,-2.16632,-0.022956,1.65337,-2.22727,0.114198,1.51797,-2.17802,0.147758,1.46177,-2.18967,0.162763,1.3867,-1.79349,-0.017029,1.44983,-1.89356,-0.006587,1.47228,-1.947,-0.006042,1.43992,-2.0059,0.188669,1.38049,-2.06084,0.156332,1.07484,-1.74469,0.023484,1.0537,-1.77865,0.036633,0.97415,-1.80191,0.106869,0.859218,-1.78818,0.074426,0.67663,-1.79021,0.046206,0.607766,-1.80508,-0.029737,0.718566,-1.79614,-0.05668,0.651648,-1.80596,-0.102266,0.435032,-1.82934,-0.217528,0.378616,-1.90368,0.0392,0.38265,-1.86531,0.009268,0.337926,-1.79063,0.280022,1.13002,-2.11067,0.325309,1.05986,-2.09444,0.16011,1.05196,-2.08691,0.206548,0.986075,-2.07155,0.312335,0.892784,-2.06982,0.164074,0.79368,-2.06847,0.13932,0.90294,-2.08197,0.070875,0.883744,-2.07948,-0.147128,0.792515,-2.06757,-0.243303,0.851309,-2.0134,-0.157457,0.583259,-2.03683,-0.208912,0.619326,-2.10253,0.084849,1.30109,-1.81943,0.103413,1.29309,-2.03372},
/*431*/{0.23954,1.8612,-1.71724,0.290739,1.80017,-1.87219,0.190167,1.70477,-1.61187,0.162405,1.58725,-1.53939,0.304446,1.59115,-1.62108,0.389906,1.5849,-1.64926,0.206539,1.89169,-2.10888,0.291789,1.80664,-1.94983,0.080914,1.86287,-2.00629,-0.02401,1.75651,-2.16534,-0.029917,1.66081,-2.22622,0.108626,1.52603,-2.17724,0.140618,1.4704,-2.19052,0.165197,1.38756,-1.79394,-0.016213,1.4535,-1.89101,-0.006452,1.4761,-1.94565,-0.00555,1.44261,-2.00467,0.188397,1.38433,-2.06072,0.148691,1.07591,-1.74666,0.01444,1.0569,-1.77955,0.027096,0.977174,-1.80372,0.094068,0.860805,-1.78984,0.05726,0.678339,-1.7915,0.0291,0.609018,-1.80589,-0.046662,0.720801,-1.79678,-0.073215,0.653621,-1.80597,-0.120581,0.437602,-1.82902,-0.238044,0.382976,-1.90356,0.018494,0.381603,-1.86527,-0.012944,0.337507,-1.79124,0.2845,1.13805,-2.11079,0.331435,1.06972,-2.09418,0.166326,1.05695,-2.08651,0.21414,0.992454,-2.07117,0.323283,0.904204,-2.07017,0.178742,0.797345,-2.06848,0.149795,0.903871,-2.08185,0.082853,0.882789,-2.0791,-0.128538,0.778751,-2.06431,-0.230473,0.834038,-2.0121,-0.128913,0.57136,-2.03826,-0.185696,0.603401,-2.10237,0.086151,1.30305,-1.81963,0.102959,1.29669,-2.03412},
/*432*/{0.241924,1.86655,-1.71874,0.291087,1.80402,-1.87424,0.195537,1.70935,-1.61263,0.169628,1.59155,-1.53955,0.309434,1.59812,-1.62388,0.394401,1.59368,-1.65303,0.203082,1.89651,-2.10882,0.291682,1.81115,-1.95185,0.079983,1.86582,-2.00531,-0.030011,1.76201,-2.16375,-0.036732,1.66801,-2.22522,0.1009,1.53445,-2.17819,0.133889,1.47836,-2.19128,0.16602,1.38875,-1.79411,-0.014828,1.45671,-1.88961,-0.005467,1.47847,-1.94351,-0.005252,1.44657,-2.00331,0.18763,1.38769,-2.06046,0.142657,1.07485,-1.74991,0.008493,1.06152,-1.7788,0.015784,0.980545,-1.80279,0.081804,0.863479,-1.79197,0.041075,0.680417,-1.79246,0.012282,0.61144,-1.8074,-0.063324,0.722991,-1.79763,-0.090541,0.657118,-1.80639,-0.141107,0.440258,-1.82882,-0.257336,0.388736,-1.90229,-0.000806,0.381728,-1.86455,-0.032882,0.337733,-1.7912,0.288311,1.14706,-2.11184,0.337428,1.07976,-2.09445,0.172308,1.06275,-2.08699,0.221713,0.999925,-2.07116,0.334133,0.914901,-2.07043,0.193281,0.801314,-2.07014,0.156683,0.909225,-2.0829,0.091832,0.882727,-2.07943,-0.117335,0.768341,-2.06645,-0.219926,0.814975,-2.01039,-0.101506,0.559249,-2.03861,-0.159118,0.588614,-2.10256,0.086706,1.30477,-1.8198,0.102272,1.30011,-2.03442},
/*433*/{0.243877,1.87198,-1.72013,0.291636,1.80871,-1.87615,0.20139,1.71464,-1.61449,0.175834,1.59742,-1.54017,0.313746,1.60559,-1.62613,0.398871,1.60334,-1.65693,0.199139,1.9007,-2.10898,0.291026,1.81561,-1.95388,0.07864,1.8687,-2.00275,-0.035034,1.77,-2.16272,-0.042722,1.67539,-2.22422,0.094227,1.54308,-2.1772,0.127502,1.48661,-2.19151,0.168595,1.38987,-1.79407,-0.014151,1.46005,-1.88837,-0.005106,1.48248,-1.94114,-0.004874,1.45084,-2.00196,0.187805,1.39137,-2.06037,0.137188,1.07566,-1.75245,0.003441,1.06754,-1.7779,0.005965,0.984935,-1.8028,0.070402,0.865525,-1.79433,0.024065,0.682992,-1.7939,-0.007135,0.614053,-1.80849,-0.079265,0.726716,-1.7981,-0.107533,0.661306,-1.80791,-0.158636,0.445078,-1.82867,-0.276564,0.394909,-1.90159,-0.021448,0.380937,-1.86597,-0.053442,0.338065,-1.79228,0.291145,1.15556,-2.11174,0.342394,1.08962,-2.0942,0.177512,1.06915,-2.08636,0.228252,1.00673,-2.07107,0.342969,0.926498,-2.07157,0.207154,0.805887,-2.07089,0.165692,0.91161,-2.08239,0.101264,0.881494,-2.07866,-0.100658,0.756027,-2.06684,-0.207479,0.795264,-2.00947,-0.072122,0.548848,-2.03956,-0.133093,0.574042,-2.10281,0.087736,1.30685,-1.82013,0.101874,1.30415,-2.03489},
/*434*/{0.246349,1.87742,-1.72271,0.292887,1.81358,-1.8782,0.206875,1.71997,-1.61559,0.181905,1.60215,-1.53963,0.317412,1.61368,-1.6292,0.401195,1.6121,-1.66092,0.196259,1.90547,-2.10911,0.290609,1.82006,-1.95616,0.077779,1.87224,-2.00167,-0.038402,1.77594,-2.16275,-0.048932,1.68256,-2.22232,0.088549,1.54967,-2.17614,0.12187,1.49477,-2.19206,0.169678,1.39118,-1.79366,-0.011978,1.4637,-1.88667,-0.003524,1.48639,-1.93925,-0.005345,1.45576,-2.00059,0.187554,1.39609,-2.0601,0.130574,1.07689,-1.7543,-0.005505,1.07255,-1.77959,-0.003602,0.990758,-1.80274,0.058729,0.868313,-1.7961,0.007837,0.686762,-1.79445,-0.023639,0.617406,-1.80896,-0.095662,0.731153,-1.79862,-0.124682,0.665211,-1.80737,-0.17632,0.449606,-1.8283,-0.295059,0.401643,-1.90148,-0.040373,0.381238,-1.86539,-0.073661,0.338517,-1.79309,0.294797,1.16471,-2.11226,0.347622,1.0978,-2.09406,0.182161,1.07578,-2.08672,0.235173,1.01462,-2.07136,0.351252,0.937915,-2.07159,0.220972,0.809416,-2.07219,0.173025,0.912837,-2.08182,0.111002,0.878844,-2.0783,-0.082367,0.742261,-2.06478,-0.193821,0.774799,-2.00756,-0.04407,0.53701,-2.03954,-0.106276,0.558881,-2.10255,0.088374,1.30905,-1.82063,0.101418,1.3088,-2.03547},
/*435*/{0.248557,1.88298,-1.7238,0.294388,1.81845,-1.88043,0.212842,1.7257,-1.61603,0.187987,1.60743,-1.53975,0.320797,1.62106,-1.63148,0.403986,1.62121,-1.66489,0.193334,1.90969,-2.10907,0.290349,1.82478,-1.95806,0.076186,1.87559,-1.99996,-0.042759,1.78291,-2.16229,-0.054916,1.68981,-2.22065,0.082463,1.55776,-2.17597,0.11611,1.50271,-2.19224,0.172195,1.39294,-1.79441,-0.010037,1.46661,-1.88506,-0.002946,1.48962,-1.93688,-0.004867,1.46002,-1.99939,0.187475,1.40134,-2.06017,0.124264,1.07911,-1.75693,-0.011995,1.07763,-1.77947,-0.012054,0.99613,-1.80319,0.044584,0.871028,-1.79661,-0.008303,0.690509,-1.79535,-0.03922,0.622206,-1.80826,-0.11144,0.735592,-1.79938,-0.140659,0.670299,-1.80814,-0.19458,0.454623,-1.82882,-0.313354,0.409958,-1.90128,-0.059786,0.380481,-1.86598,-0.094863,0.338789,-1.79305,0.29806,1.1734,-2.11247,0.350725,1.10926,-2.09461,0.187801,1.08235,-2.08716,0.240965,1.0225,-2.0715,0.358296,0.948743,-2.07198,0.233797,0.811649,-2.07325,0.180151,0.914068,-2.08186,0.119728,0.877173,-2.07822,-0.065976,0.728432,-2.06438,-0.179257,0.753334,-2.0063,-0.014078,0.52704,-2.04124,-0.077763,0.544478,-2.10306,0.090594,1.31114,-1.82155,0.101853,1.31358,-2.03648},
/*436*/{0.251,1.88818,-1.72601,0.295511,1.82262,-1.88317,0.217081,1.73069,-1.61736,0.193381,1.61281,-1.54004,0.323966,1.62849,-1.63376,0.406885,1.63026,-1.66879,0.190912,1.91434,-2.10909,0.290188,1.82936,-1.95981,0.0759,1.87938,-1.99917,-0.046484,1.78903,-2.16082,-0.059977,1.69657,-2.21873,0.07765,1.56526,-2.17449,0.111233,1.51055,-2.19262,0.17364,1.39449,-1.79417,-0.008277,1.46985,-1.88384,-0.001335,1.49352,-1.93475,-0.004107,1.46472,-1.99869,0.188846,1.40655,-2.05973,0.118998,1.08055,-1.75986,-0.017866,1.08388,-1.7783,-0.020968,1.00197,-1.80234,0.030065,0.876238,-1.79738,-0.024126,0.695747,-1.79468,-0.055833,0.627842,-1.80889,-0.126991,0.741496,-1.79996,-0.155959,0.677012,-1.80839,-0.212872,0.459544,-1.82856,-0.330556,0.418083,-1.90122,-0.07838,0.379635,-1.86645,-0.115619,0.339767,-1.79336,0.300921,1.18168,-2.11246,0.355005,1.11871,-2.09486,0.191629,1.08969,-2.08725,0.246505,1.02987,-2.0715,0.364581,0.959277,-2.07176,0.245221,0.817092,-2.0744,0.186657,0.91515,-2.08071,0.127931,0.874333,-2.0777,-0.049221,0.714972,-2.06366,-0.162319,0.732773,-2.00505,0.017043,0.516561,-2.04163,-0.049568,0.530802,-2.10367,0.091826,1.31328,-1.82243,0.102915,1.31863,-2.03731},
/*437*/{0.252099,1.89333,-1.72779,0.296015,1.8279,-1.88506,0.22142,1.73635,-1.61816,0.197622,1.61891,-1.54051,0.326454,1.63649,-1.63661,0.408811,1.63862,-1.67262,0.188663,1.91865,-2.10971,0.290084,1.83427,-1.96152,0.074063,1.8835,-1.9979,-0.049412,1.79492,-2.16013,-0.06475,1.70361,-2.21669,0.072245,1.57287,-2.17385,0.106654,1.51769,-2.19296,0.176086,1.3964,-1.79461,-0.007204,1.4734,-1.88258,-0.000607,1.49723,-1.93286,-0.002837,1.47011,-1.99793,0.188807,1.4122,-2.05961,0.114643,1.0828,-1.76123,-0.021351,1.09052,-1.77762,-0.029105,1.00803,-1.80284,0.014978,0.881511,-1.79786,-0.039825,0.700419,-1.79496,-0.071296,0.633103,-1.80816,-0.141842,0.747674,-1.80009,-0.171869,0.682811,-1.80915,-0.228255,0.464688,-1.82696,-0.347575,0.427509,-1.9014,-0.097462,0.378643,-1.86631,-0.136144,0.340025,-1.7936,0.303487,1.18948,-2.11224,0.357859,1.12748,-2.09468,0.195472,1.09668,-2.08723,0.250965,1.03778,-2.07159,0.3705,0.968442,-2.07142,0.256207,0.821171,-2.07511,0.192827,0.915465,-2.08077,0.137473,0.872021,-2.07791,-0.030722,0.701409,-2.06171,-0.14605,0.711217,-2.00297,0.04736,0.507779,-2.04299,-0.018685,0.518333,-2.10389,0.09337,1.31579,-1.82364,0.103403,1.32406,-2.03848},
/*438*/{0.254116,1.89882,-1.73018,0.296883,1.83243,-1.88698,0.224927,1.74182,-1.61906,0.202659,1.62493,-1.54129,0.328792,1.64294,-1.63845,0.410162,1.64688,-1.6762,0.187125,1.92267,-2.10965,0.290065,1.83888,-1.96314,0.075272,1.88613,-1.99615,-0.052086,1.79998,-2.15773,-0.069017,1.70986,-2.2146,0.068962,1.58002,-2.17246,0.101856,1.52456,-2.1933,0.177652,1.39835,-1.79518,-0.00407,1.47723,-1.88161,0.000285,1.50122,-1.93016,-0.001635,1.47496,-1.99675,0.188104,1.41743,-2.05938,0.109538,1.08537,-1.76212,-0.02701,1.09815,-1.7771,-0.037428,1.01503,-1.80171,0.000985,0.887756,-1.79727,-0.054455,0.706101,-1.79455,-0.088133,0.638241,-1.8076,-0.156289,0.753869,-1.80051,-0.186247,0.689118,-1.80835,-0.244405,0.469378,-1.82612,-0.364453,0.437546,-1.90122,-0.117131,0.377241,-1.86648,-0.157594,0.340536,-1.7942,0.30495,1.19743,-2.11248,0.361126,1.13696,-2.09446,0.199012,1.10347,-2.08827,0.255182,1.04525,-2.07181,0.374047,0.978717,-2.071,0.267908,0.82435,-2.07576,0.198467,0.914956,-2.08017,0.146112,0.86882,-2.07748,-0.012736,0.688456,-2.06116,-0.128898,0.69094,-2.00298,0.078527,0.500075,-2.04412,0.010493,0.506577,-2.10451,0.094781,1.3184,-1.82445,0.103284,1.32911,-2.03926},
/*439*/{0.255524,1.90423,-1.73152,0.297149,1.83743,-1.8889,0.22873,1.74707,-1.61976,0.206626,1.63014,-1.54091,0.330348,1.6504,-1.64186,0.411914,1.65404,-1.68022,0.185418,1.92663,-2.10981,0.289686,1.84407,-1.9645,0.07518,1.88988,-1.99576,-0.055231,1.80521,-2.1559,-0.073026,1.71633,-2.21224,0.064714,1.58618,-2.17215,0.098534,1.53127,-2.19295,0.180992,1.39984,-1.79572,-0.001979,1.48045,-1.87976,0.002203,1.50616,-1.92851,-0.000741,1.48017,-1.99603,0.188906,1.42285,-2.05961,0.102689,1.08965,-1.76665,-0.033014,1.10309,-1.7764,-0.044545,1.02253,-1.80083,-0.011889,0.894316,-1.79668,-0.068574,0.712819,-1.79361,-0.102757,0.645751,-1.80715,-0.170483,0.76152,-1.80039,-0.200651,0.696911,-1.80844,-0.26075,0.477222,-1.82584,-0.379382,0.447804,-1.90197,-0.13586,0.376703,-1.86662,-0.177831,0.34158,-1.79444,0.307743,1.20556,-2.11247,0.364082,1.14599,-2.09339,0.202046,1.11016,-2.08793,0.259267,1.05221,-2.07178,0.376379,0.987227,-2.07135,0.278577,0.827892,-2.07642,0.203749,0.911973,-2.07903,0.154938,0.861488,-2.07548,0.008235,0.674926,-2.06048,-0.110328,0.669289,-2.00139,0.110078,0.493388,-2.04549,0.040213,0.495048,-2.10515,0.096894,1.32088,-1.82584,0.103941,1.33458,-2.04052},
/*440*/{0.257721,1.90922,-1.73288,0.297751,1.84229,-1.89076,0.231377,1.75202,-1.62041,0.210093,1.63564,-1.54189,0.332415,1.65716,-1.64347,0.413104,1.66162,-1.68386,0.183663,1.9305,-2.11022,0.289643,1.84799,-1.96606,0.074495,1.89319,-1.9956,-0.056248,1.8112,-2.15403,-0.076913,1.72231,-2.21038,0.061195,1.5922,-2.1708,0.094488,1.53713,-2.1933,0.182393,1.40164,-1.79681,-0.000761,1.48379,-1.87891,0.003608,1.50955,-1.9262,0.001188,1.48476,-1.99529,0.188412,1.42856,-2.05965,0.100259,1.09301,-1.7673,-0.036244,1.11148,-1.77626,-0.052726,1.02964,-1.80147,-0.022457,0.900735,-1.79639,-0.08311,0.719255,-1.79207,-0.116136,0.65204,-1.80551,-0.183386,0.768406,-1.79926,-0.21467,0.703507,-1.80857,-0.277037,0.482396,-1.82568,-0.394453,0.459076,-1.90206,-0.155314,0.376666,-1.86694,-0.197943,0.34286,-1.79435,0.308838,1.21318,-2.11318,0.366676,1.15347,-2.09333,0.204805,1.11667,-2.08872,0.262464,1.05916,-2.07167,0.37834,0.995312,-2.07134,0.288791,0.830702,-2.07732,0.2087,0.909819,-2.07855,0.162191,0.85732,-2.07555,0.025665,0.661541,-2.05994,-0.089832,0.647473,-1.99816,0.140336,0.487625,-2.04701,0.072033,0.484976,-2.10593,0.098234,1.32308,-1.8274,0.104342,1.3399,-2.04189},
/*441*/{0.259092,1.91372,-1.7346,0.298337,1.84642,-1.89255,0.23423,1.75728,-1.62156,0.21362,1.64007,-1.54164,0.333693,1.66323,-1.64497,0.414609,1.66869,-1.68606,0.182363,1.93422,-2.11032,0.288984,1.85242,-1.96789,0.073469,1.89694,-1.99567,-0.058577,1.81678,-2.15236,-0.079897,1.72777,-2.20774,0.058473,1.59818,-2.17007,0.091394,1.54279,-2.19314,0.184773,1.40373,-1.7981,0.001283,1.48688,-1.87738,0.005109,1.51318,-1.92395,0.002356,1.48944,-1.99399,0.189096,1.43365,-2.0599,0.093353,1.0946,-1.76791,-0.040255,1.1186,-1.77559,-0.058451,1.03744,-1.80005,-0.032333,0.907253,-1.79552,-0.095808,0.726481,-1.79039,-0.128871,0.660413,-1.80277,-0.196781,0.775789,-1.80105,-0.228344,0.712109,-1.80852,-0.289906,0.48854,-1.82331,-0.409259,0.470539,-1.90165,-0.174056,0.376228,-1.86709,-0.218347,0.344044,-1.79481,0.31055,1.22042,-2.1131,0.368854,1.1608,-2.09298,0.207098,1.12351,-2.08948,0.264483,1.06596,-2.0711,0.378592,1.00296,-2.07213,0.29815,0.832455,-2.07774,0.21388,0.906833,-2.07791,0.170507,0.850947,-2.07501,0.040864,0.648664,-2.06122,-0.06846,0.627273,-1.99664,0.172061,0.482725,-2.04849,0.102847,0.475915,-2.10658,0.100426,1.32548,-1.8286,0.10497,1.34498,-2.04291},
/*442*/{0.260451,1.9181,-1.73636,0.298334,1.85036,-1.89466,0.237013,1.7623,-1.62176,0.21744,1.6458,-1.54173,0.335307,1.66936,-1.64742,0.415499,1.67515,-1.68978,0.180664,1.93714,-2.11058,0.289164,1.85687,-1.96918,0.073587,1.90006,-1.99543,-0.060282,1.8216,-2.15025,-0.082698,1.73345,-2.20582,0.055595,1.60294,-2.16905,0.088947,1.54789,-2.19305,0.187651,1.4058,-1.79894,0.003945,1.49013,-1.8764,0.006069,1.5171,-1.92229,0.003718,1.49499,-1.99275,0.189481,1.43887,-2.06008,0.090392,1.0977,-1.76823,-0.04437,1.12502,-1.77469,-0.064707,1.04581,-1.79904,-0.041989,0.913122,-1.79522,-0.107908,0.733862,-1.78898,-0.142072,0.667455,-1.80167,-0.208113,0.78458,-1.80052,-0.240114,0.719793,-1.80891,-0.305323,0.495045,-1.82272,-0.421618,0.482796,-1.90247,-0.193081,0.376369,-1.86738,-0.240156,0.345174,-1.79532,0.312691,1.22721,-2.11289,0.370188,1.16781,-2.09194,0.208638,1.12962,-2.09017,0.266163,1.0728,-2.07137,0.377968,1.00921,-2.07223,0.307533,0.833592,-2.07821,0.218904,0.902891,-2.07793,0.179075,0.844932,-2.07307,0.062878,0.634392,-2.05795,-0.044905,0.604617,-1.99363,0.20285,0.478207,-2.04862,0.134099,0.467661,-2.10688,0.102505,1.32818,-1.82995,0.105549,1.35032,-2.04402},
/*443*/{0.262478,1.92215,-1.73773,0.29958,1.85563,-1.89658,0.238915,1.76713,-1.62222,0.221452,1.65052,-1.54164,0.336684,1.67456,-1.64937,0.416483,1.68141,-1.69318,0.18025,1.9405,-2.1111,0.288737,1.86066,-1.97053,0.073706,1.90262,-1.9946,-0.061108,1.82528,-2.14705,-0.08513,1.73815,-2.20313,0.053324,1.60758,-2.16882,0.086495,1.55254,-2.19267,0.190244,1.40741,-1.80016,0.005811,1.49407,-1.87519,0.007776,1.5202,-1.92087,0.00557,1.49966,-1.9919,0.18931,1.44399,-2.06079,0.08697,1.1003,-1.76837,-0.046708,1.13204,-1.7741,-0.0704,1.05304,-1.79883,-0.051001,0.920418,-1.79343,-0.119405,0.741659,-1.78727,-0.152984,0.675166,-1.79773,-0.219776,0.792632,-1.80058,-0.250922,0.727791,-1.80841,-0.317734,0.501783,-1.82046,-0.433761,0.49684,-1.90204,-0.212588,0.376377,-1.86723,-0.260696,0.346879,-1.79603,0.313905,1.23241,-2.11207,0.371026,1.17341,-2.09118,0.209395,1.13395,-2.0912,0.267158,1.07753,-2.07205,0.377637,1.01498,-2.07228,0.316919,0.834236,-2.07854,0.224277,0.899224,-2.07767,0.187831,0.837514,-2.07205,0.083542,0.621658,-2.05677,-0.021997,0.584511,-1.99114,0.232896,0.475325,-2.04957,0.164959,0.461182,-2.1078,0.104386,1.33061,-1.8315,0.105815,1.35527,-2.04532},
/*444*/{0.263583,1.92684,-1.73868,0.300712,1.85926,-1.89828,0.241154,1.77151,-1.62302,0.223042,1.65415,-1.54154,0.337338,1.67995,-1.65104,0.416683,1.68755,-1.69535,0.179086,1.94376,-2.11162,0.288746,1.86399,-1.97202,0.074032,1.90606,-1.9941,-0.060584,1.83064,-2.14389,-0.087316,1.74301,-2.20089,0.050708,1.61193,-2.16821,0.083954,1.55705,-2.1925,0.19236,1.4087,-1.80155,0.008476,1.49737,-1.87349,0.010067,1.52425,-1.91996,0.006953,1.50433,-1.99099,0.190073,1.44865,-2.06127,0.083027,1.10286,-1.76796,-0.04722,1.13987,-1.77186,-0.074482,1.06125,-1.79785,-0.060001,0.928088,-1.79335,-0.129837,0.749832,-1.78563,-0.163545,0.682949,-1.79614,-0.229986,0.7994,-1.80089,-0.261864,0.735578,-1.80882,-0.330483,0.509654,-1.81931,-0.444189,0.511095,-1.90165,-0.230338,0.377498,-1.866,-0.281534,0.349099,-1.79647,0.315104,1.23844,-2.11204,0.37051,1.17852,-2.09086,0.209755,1.14136,-2.09234,0.266294,1.08369,-2.0747,0.376279,1.01913,-2.07281,0.325854,0.834618,-2.07911,0.230247,0.893408,-2.0776,0.196504,0.830929,-2.07182,0.103281,0.608958,-2.05503,0.001941,0.565106,-1.98772,0.263473,0.473075,-2.05049,0.195308,0.454455,-2.10765,0.105983,1.33286,-1.83311,0.106168,1.36015,-2.0466},
/*445*/{0.265301,1.92956,-1.74053,0.300892,1.86347,-1.89984,0.242594,1.77607,-1.62357,0.225965,1.65863,-1.5412,0.338094,1.68479,-1.6524,0.417364,1.69297,-1.69891,0.178098,1.94674,-2.11174,0.288713,1.86756,-1.97341,0.074329,1.909,-1.99418,-0.061987,1.83408,-2.14223,-0.089168,1.74714,-2.19832,0.049336,1.61592,-2.16765,0.082282,1.56103,-2.19191,0.194727,1.41074,-1.80256,0.010338,1.50083,-1.87311,0.011742,1.52734,-1.91837,0.007948,1.50932,-1.99027,0.189898,1.45308,-2.06156,0.080293,1.10732,-1.76748,-0.051528,1.14588,-1.77181,-0.079691,1.06861,-1.79693,-0.068611,0.935571,-1.79274,-0.139158,0.757256,-1.784,-0.173699,0.690436,-1.79446,-0.238458,0.807239,-1.80067,-0.271914,0.742813,-1.80832,-0.340874,0.517016,-1.81713,-0.453933,0.527694,-1.90067,-0.248066,0.378936,-1.86535,-0.301817,0.352338,-1.79632,0.315342,1.24253,-2.11096,0.371436,1.18267,-2.09047,0.209588,1.14569,-2.09275,0.265555,1.08775,-2.07627,0.374565,1.02216,-2.07315,0.334848,0.834232,-2.0784,0.235541,0.887884,-2.07796,0.205475,0.823234,-2.07185,0.127651,0.598122,-2.0532,0.026636,0.546777,-1.98556,0.292717,0.471822,-2.05003,0.225254,0.449081,-2.10744,0.107525,1.3355,-1.83432,0.106273,1.36475,-2.04756},
/*446*/{0.267015,1.93362,-1.74211,0.300913,1.86655,-1.90082,0.244977,1.78022,-1.62428,0.227861,1.66204,-1.54155,0.339759,1.68856,-1.65446,0.417253,1.69695,-1.70073,0.17773,1.94948,-2.11157,0.288499,1.8711,-1.97437,0.074576,1.91115,-1.99376,-0.062644,1.83911,-2.14046,-0.090743,1.75094,-2.19614,0.048083,1.61943,-2.1674,0.081062,1.56456,-2.1915,0.196248,1.41249,-1.80353,0.011656,1.50439,-1.87191,0.012635,1.53111,-1.91718,0.009074,1.51453,-1.98987,0.190008,1.45762,-2.06119,0.076292,1.10957,-1.76661,-0.054417,1.15204,-1.77133,-0.082756,1.07542,-1.79594,-0.076149,0.943032,-1.79206,-0.146954,0.764825,-1.7822,-0.182648,0.697943,-1.79249,-0.247216,0.814993,-1.79949,-0.279228,0.75134,-1.80691,-0.35283,0.524443,-1.81484,-0.460729,0.542563,-1.89969,-0.26647,0.380933,-1.86489,-0.322587,0.356293,-1.79638,0.315507,1.24672,-2.11031,0.370725,1.18572,-2.08974,0.208694,1.15032,-2.09429,0.264567,1.09181,-2.07788,0.373502,1.02418,-2.07393,0.343385,0.833848,-2.07869,0.241434,0.881848,-2.07795,0.215775,0.815435,-2.07133,0.145116,0.586436,-2.05229,0.051355,0.52955,-1.98527,0.321312,0.471522,-2.05053,0.254977,0.444387,-2.10689,0.108117,1.33813,-1.83553,0.106226,1.36967,-2.04844},
/*447*/{0.268114,1.93688,-1.74334,0.302613,1.87108,-1.90304,0.24634,1.78382,-1.62465,0.230728,1.6667,-1.54145,0.339851,1.69307,-1.65613,0.417552,1.70099,-1.70332,0.176698,1.95223,-2.11199,0.288997,1.87405,-1.97561,0.074542,1.91374,-1.9931,-0.062624,1.84207,-2.13669,-0.092626,1.7544,-2.19386,0.04771,1.62287,-2.16669,0.079895,1.56808,-2.19134,0.198878,1.41495,-1.80474,0.013992,1.50834,-1.87114,0.013714,1.53434,-1.91609,0.012501,1.51746,-1.98771,0.189972,1.46118,-2.06163,0.074364,1.11383,-1.76474,-0.056671,1.15869,-1.77148,-0.085907,1.08151,-1.79556,-0.081779,0.949448,-1.79216,-0.155569,0.771635,-1.78074,-0.18991,0.705169,-1.78912,-0.254073,0.823119,-1.79841,-0.286172,0.760539,-1.80509,-0.362815,0.531846,-1.8129,-0.467841,0.558121,-1.89768,-0.282968,0.382411,-1.86355,-0.343249,0.361739,-1.79783,0.314523,1.24935,-2.1095,0.369658,1.18842,-2.08904,0.207677,1.1539,-2.09445,0.263034,1.09498,-2.07897,0.371364,1.02524,-2.07489,0.3511,0.832908,-2.07935,0.246479,0.875543,-2.07841,0.223733,0.807537,-2.07115,0.16733,0.575749,-2.0514,0.075523,0.513405,-1.98299,0.349373,0.47197,-2.05006,0.284937,0.441524,-2.10598,0.110387,1.34099,-1.83573,0.106659,1.3732,-2.04851},
/*448*/{0.269868,1.93978,-1.74471,0.301886,1.87401,-1.9039,0.248078,1.78733,-1.62482,0.231607,1.6694,-1.54182,0.341293,1.69611,-1.65709,0.417678,1.70499,-1.70562,0.176677,1.95452,-2.11262,0.289449,1.87694,-1.97673,0.0746,1.91611,-1.99381,-0.063012,1.84584,-2.13653,-0.093992,1.75733,-2.1914,0.047205,1.62624,-2.16613,0.07896,1.57131,-2.1911,0.199623,1.41735,-1.80624,0.015531,1.51082,-1.87067,0.015134,1.53848,-1.91487,0.01364,1.52152,-1.98684,0.190717,1.46409,-2.06178,0.071412,1.11753,-1.76387,-0.055395,1.16375,-1.7703,-0.088808,1.0873,-1.79563,-0.085409,0.955099,-1.79225,-0.161304,0.779245,-1.77967,-0.196146,0.71204,-1.78758,-0.259567,0.830855,-1.79823,-0.293263,0.767959,-1.80368,-0.370406,0.539913,-1.80886,-0.473037,0.574484,-1.89511,-0.301023,0.386335,-1.86222,-0.362753,0.36763,-1.79859,0.314367,1.25172,-2.10873,0.368788,1.18913,-2.08899,0.206094,1.15622,-2.09506,0.261527,1.09652,-2.07996,0.36967,1.02532,-2.07592,0.358648,0.831737,-2.07951,0.252139,0.868434,-2.07898,0.233547,0.800097,-2.07148,0.190392,0.565621,-2.05004,0.100786,0.497406,-1.98194,0.376161,0.47271,-2.04807,0.31332,0.438131,-2.10489,0.111018,1.34353,-1.83648,0.106598,1.37695,-2.04905},
/*449*/{0.270554,1.94214,-1.74523,0.302655,1.87702,-1.90497,0.250081,1.79064,-1.62597,0.234199,1.67304,-1.54154,0.341496,1.69961,-1.65838,0.417815,1.7088,-1.70848,0.176055,1.95673,-2.11294,0.290283,1.8792,-1.97783,0.075663,1.9177,-1.99287,-0.063329,1.84832,-2.13412,-0.095306,1.75988,-2.18922,0.046571,1.62895,-2.16563,0.078314,1.57408,-2.19067,0.202187,1.41942,-1.80676,0.016912,1.51425,-1.86985,0.015608,1.54129,-1.91369,0.015564,1.52462,-1.98525,0.190857,1.46728,-2.06168,0.067846,1.12196,-1.7628,-0.056994,1.16852,-1.77074,-0.091401,1.09196,-1.79628,-0.087992,0.959738,-1.7927,-0.166426,0.785062,-1.77875,-0.203038,0.720065,-1.78629,-0.264309,0.838518,-1.79733,-0.299579,0.775341,-1.80197,-0.379722,0.550136,-1.80752,-0.47828,0.591367,-1.89173,-0.318441,0.390502,-1.86206,-0.381897,0.374642,-1.79923,0.313074,1.25191,-2.10749,0.366968,1.1899,-2.08798,0.204035,1.15775,-2.09521,0.258055,1.09825,-2.08039,0.367662,1.02469,-2.07745,0.366327,0.830146,-2.08019,0.257489,0.861401,-2.07934,0.241842,0.792428,-2.07118,0.209712,0.556197,-2.04988,0.126505,0.483189,-1.98182,0.401999,0.474302,-2.04651,0.341926,0.436026,-2.10291,0.112709,1.34612,-1.83647,0.106909,1.38023,-2.04891},
/*450*/{0.272975,1.94489,-1.74694,0.303386,1.87981,-1.90586,0.250837,1.79338,-1.62624,0.23507,1.67485,-1.54124,0.341613,1.7023,-1.66096,0.417707,1.71093,-1.71089,0.175724,1.95902,-2.11306,0.290763,1.88144,-1.97914,0.075449,1.92051,-1.99305,-0.063762,1.84983,-2.13066,-0.096191,1.76187,-2.18785,0.045945,1.63168,-2.16502,0.077515,1.57639,-2.19056,0.202594,1.42225,-1.80762,0.017931,1.51799,-1.86861,0.016628,1.54496,-1.91304,0.015047,1.52845,-1.9842,0.191241,1.46958,-2.06175,0.067679,1.12562,-1.76,-0.059794,1.17398,-1.77393,-0.091825,1.09672,-1.79725,-0.090386,0.963561,-1.79352,-0.170952,0.790806,-1.77826,-0.209496,0.724775,-1.78515,-0.268044,0.845358,-1.79604,-0.301946,0.783331,-1.79827,-0.388043,0.559336,-1.80457,-0.483441,0.607636,-1.88711,-0.335753,0.397154,-1.86312,-0.399178,0.383029,-1.80165,0.311719,1.25177,-2.10758,0.364544,1.18903,-2.08828,0.201596,1.1585,-2.09529,0.255421,1.0987,-2.08104,0.365382,1.02334,-2.07837,0.372528,0.828171,-2.0808,0.261814,0.854235,-2.07898,0.249869,0.784993,-2.07187,0.23123,0.547292,-2.04907,0.150641,0.469593,-1.97969,0.426355,0.477019,-2.04502,0.369279,0.434933,-2.10155,0.112808,1.3493,-1.83613,0.106007,1.38333,-2.04855},
/*451*/{0.273127,1.94742,-1.74737,0.303505,1.88171,-1.90683,0.25215,1.79626,-1.6273,0.236165,1.67768,-1.54104,0.34234,1.70345,-1.66071,0.418004,1.71283,-1.71242,0.1757,1.96033,-2.11337,0.29111,1.88317,-1.97992,0.076355,1.92093,-1.9926,-0.063834,1.85266,-2.13041,-0.096814,1.76336,-2.18573,0.045781,1.63338,-2.16464,0.077153,1.5784,-2.19038,0.203633,1.42485,-1.80779,0.018911,1.52144,-1.86697,0.017215,1.54749,-1.91257,0.016072,1.53106,-1.98379,0.191311,1.47162,-2.06122,0.066128,1.1291,-1.75933,-0.061103,1.17845,-1.77581,-0.09369,1.10045,-1.79831,-0.090417,0.967366,-1.7938,-0.174353,0.794918,-1.77809,-0.212798,0.731481,-1.78512,-0.269672,0.85185,-1.79356,-0.306281,0.789859,-1.79688,-0.394922,0.569853,-1.80255,-0.488617,0.622973,-1.88061,-0.3519,0.405727,-1.86388,-0.415677,0.391532,-1.80285,0.309527,1.25114,-2.10749,0.361264,1.18748,-2.08855,0.198356,1.15954,-2.09512,0.251251,1.09872,-2.0814,0.362633,1.02141,-2.08032,0.377793,0.825904,-2.08116,0.266696,0.847205,-2.07949,0.258049,0.777278,-2.0724,0.25109,0.539127,-2.04858,0.174767,0.456925,-1.97976,0.450758,0.479646,-2.04315,0.395912,0.432745,-2.09836,0.113534,1.35213,-1.83525,0.105916,1.3856,-2.04773},
/*452*/{0.273692,1.94926,-1.74838,0.304908,1.88465,-1.90849,0.253529,1.79865,-1.62783,0.236968,1.67936,-1.54095,0.342119,1.70559,-1.66074,0.417576,1.71383,-1.71456,0.175734,1.9622,-2.11365,0.291411,1.88466,-1.98099,0.076887,1.92366,-1.99257,-0.064327,1.85348,-2.12805,-0.097836,1.76454,-2.18451,0.045171,1.63501,-2.16455,0.077389,1.57988,-2.19002,0.203848,1.42696,-1.8079,0.020565,1.52377,-1.8665,0.018101,1.55083,-1.91139,0.016793,1.53371,-1.98203,0.192243,1.47412,-2.0609,0.066379,1.13073,-1.75927,-0.062267,1.1786,-1.77618,-0.092867,1.10302,-1.79879,-0.090198,0.97004,-1.79428,-0.176486,0.798168,-1.77786,-0.216592,0.736058,-1.78498,-0.270002,0.858821,-1.79199,-0.309311,0.79824,-1.79512,-0.400404,0.578988,-1.79996,-0.494618,0.63893,-1.87432,-0.367741,0.414284,-1.86704,-0.431283,0.401259,-1.80465,0.305996,1.2496,-2.10855,0.358696,1.18478,-2.08951,0.195352,1.15864,-2.09639,0.246419,1.09704,-2.0813,0.361784,1.01872,-2.08152,0.382745,0.823557,-2.08145,0.27041,0.839574,-2.0804,0.265132,0.770258,-2.07254,0.270021,0.531221,-2.04811,0.197146,0.445922,-1.97918,0.471954,0.482003,-2.04149,0.420378,0.432069,-2.09669,0.11407,1.35436,-1.83502,0.106151,1.3885,-2.04739},
/*453*/{0.275663,1.95106,-1.74951,0.304081,1.88624,-1.90912,0.254242,1.80084,-1.62876,0.236695,1.68195,-1.54199,0.342486,1.70705,-1.66292,0.416864,1.71496,-1.71629,0.17624,1.96329,-2.11444,0.291951,1.88581,-1.98178,0.077335,1.925,-1.99266,-0.065272,1.85434,-2.12653,-0.097947,1.76555,-2.18277,0.045256,1.63585,-2.16407,0.077419,1.58089,-2.19017,0.204503,1.42977,-1.80735,0.020924,1.52661,-1.86626,0.017806,1.55344,-1.91193,0.017856,1.53609,-1.98191,0.192158,1.47511,-2.06065,0.06677,1.13259,-1.75952,-0.061493,1.18066,-1.77832,-0.090731,1.10477,-1.80016,-0.087846,0.970782,-1.79484,-0.176627,0.800772,-1.77855,-0.220865,0.739298,-1.78511,-0.269513,0.864308,-1.79064,-0.309904,0.805014,-1.79342,-0.409655,0.588189,-1.79775,-0.502545,0.653357,-1.867,-0.383431,0.425096,-1.87069,-0.444414,0.412553,-1.80543,0.303718,1.24709,-2.10932,0.354524,1.18171,-2.08966,0.190297,1.15751,-2.09629,0.241863,1.09539,-2.08193,0.35845,1.01529,-2.08259,0.388019,0.821789,-2.08223,0.273751,0.831973,-2.0815,0.271893,0.76214,-2.07247,0.288275,0.523527,-2.04789,0.219712,0.436586,-1.9783,0.492582,0.484934,-2.03877,0.444192,0.431632,-2.09388,0.114159,1.35729,-1.83386,0.10605,1.3899,-2.04645},
/*454*/{0.276325,1.95229,-1.75022,0.306259,1.88836,-1.91047,0.254723,1.80256,-1.62908,0.237307,1.6838,-1.5419,0.342657,1.70802,-1.66432,0.416677,1.71532,-1.71802,0.176122,1.96426,-2.11472,0.292306,1.88723,-1.98314,0.077932,1.92637,-1.99277,-0.065584,1.85471,-2.12504,-0.098504,1.76579,-2.1816,0.045485,1.63626,-2.16337,0.077995,1.58164,-2.1902,0.205056,1.43243,-1.80742,0.02154,1.52826,-1.86514,0.017759,1.55522,-1.91244,0.017881,1.53806,-1.98067,0.192925,1.47648,-2.06022,0.069159,1.13406,-1.75982,-0.058263,1.18202,-1.77801,-0.089537,1.10472,-1.80077,-0.084454,0.971775,-1.79519,-0.177577,0.802749,-1.779,-0.222538,0.742466,-1.78622,-0.267733,0.868873,-1.78823,-0.310526,0.811403,-1.79104,-0.415992,0.599231,-1.79698,-0.510961,0.66727,-1.85979,-0.396863,0.437104,-1.87452,-0.455932,0.423625,-1.80683,0.300033,1.24383,-2.10972,0.351174,1.17778,-2.09042,0.186291,1.15647,-2.09514,0.237047,1.09311,-2.08149,0.355518,1.01166,-2.08308,0.391037,0.819824,-2.08227,0.276951,0.82414,-2.08225,0.278322,0.755145,-2.07428,0.305103,0.517313,-2.04832,0.239775,0.427779,-1.97868,0.511589,0.486955,-2.03733,0.467553,0.431205,-2.09155,0.115005,1.35949,-1.83302,0.10646,1.39146,-2.0457},
/*455*/{0.276574,1.95318,-1.75129,0.305996,1.88911,-1.91103,0.255053,1.80366,-1.62968,0.238126,1.68478,-1.54198,0.341788,1.70804,-1.66538,0.417728,1.71567,-1.71965,0.176291,1.96527,-2.11552,0.293495,1.88831,-1.98368,0.078337,1.92763,-1.99341,-0.065703,1.85507,-2.12373,-0.099089,1.76555,-2.17998,0.047873,1.63697,-2.16316,0.078395,1.58172,-2.19018,0.205355,1.43427,-1.80692,0.021447,1.5299,-1.86571,0.018418,1.55823,-1.91178,0.018092,1.53911,-1.9803,0.193445,1.47706,-2.05972,0.071676,1.13495,-1.76078,-0.056708,1.18052,-1.77942,-0.086673,1.10413,-1.80152,-0.080137,0.97087,-1.79608,-0.176926,0.804137,-1.77953,-0.225973,0.745469,-1.7869,-0.265354,0.873673,-1.78645,-0.309397,0.817788,-1.78797,-0.424731,0.609616,-1.79653,-0.520752,0.680793,-1.85168,-0.410095,0.44898,-1.87865,-0.465939,0.435141,-1.80852,0.297574,1.23996,-2.10963,0.346994,1.17325,-2.0912,0.18219,1.15441,-2.09495,0.231815,1.09051,-2.08164,0.351311,1.00659,-2.0835,0.394455,0.817792,-2.08271,0.280394,0.819045,-2.08541,0.284868,0.748604,-2.07633,0.323372,0.512243,-2.04772,0.259788,0.420529,-1.97759,0.529504,0.489831,-2.03564,0.487164,0.430446,-2.08827,0.114895,1.36147,-1.83207,0.106507,1.3925,-2.04488},
/*456*/{0.277124,1.95352,-1.75154,0.306445,1.89003,-1.91257,0.25559,1.80448,-1.63056,0.237432,1.68507,-1.54196,0.341125,1.70751,-1.66651,0.414927,1.71278,-1.72109,0.176415,1.96571,-2.11631,0.294065,1.88911,-1.98393,0.078745,1.92853,-1.9929,-0.065269,1.8538,-2.12312,-0.098725,1.76533,-2.17923,0.047949,1.63636,-2.1627,0.079255,1.58142,-2.1902,0.205264,1.43639,-1.80619,0.022024,1.53141,-1.86502,0.01862,1.5589,-1.91183,0.018185,1.54007,-1.98053,0.193867,1.47798,-2.05912,0.074674,1.1343,-1.76183,-0.053641,1.17892,-1.7794,-0.082434,1.10282,-1.80208,-0.075416,0.969299,-1.79642,-0.176805,0.80388,-1.78088,-0.228253,0.748396,-1.7879,-0.261969,0.877823,-1.78374,-0.308288,0.823415,-1.78532,-0.434789,0.619759,-1.79547,-0.529256,0.693952,-1.84427,-0.422889,0.462297,-1.88243,-0.474867,0.446537,-1.8103,0.293858,1.23573,-2.11044,0.342497,1.16851,-2.09183,0.177793,1.15216,-2.09411,0.226346,1.08724,-2.08084,0.34671,1.00291,-2.08395,0.397502,0.816598,-2.08269,0.282607,0.812582,-2.08659,0.289996,0.742151,-2.07673,0.339021,0.507829,-2.0484,0.277904,0.414853,-1.97762,0.546224,0.492794,-2.03325,0.505804,0.430735,-2.08636,0.115352,1.36318,-1.83102,0.107098,1.39322,-2.04399},
/*457*/{0.278942,1.95418,-1.75211,0.306331,1.89076,-1.91267,0.254882,1.80546,-1.63135,0.236012,1.68587,-1.54303,0.340928,1.70685,-1.667,0.414212,1.71232,-1.72211,0.177369,1.96549,-2.11703,0.293934,1.88978,-1.98539,0.079368,1.92909,-1.99374,-0.065051,1.85284,-2.12123,-0.098189,1.76445,-2.17809,0.048292,1.63604,-2.16248,0.080041,1.58071,-2.19009,0.205759,1.43792,-1.80591,0.021177,1.53237,-1.86467,0.018294,1.5601,-1.91242,0.018185,1.54007,-1.98053,0.195095,1.47824,-2.05828,0.078706,1.13304,-1.76305,-0.048109,1.17857,-1.77829,-0.077632,1.10101,-1.80245,-0.068999,0.967109,-1.79723,-0.177165,0.804159,-1.78148,-0.229385,0.751146,-1.78876,-0.258749,0.881431,-1.782,-0.305696,0.829959,-1.78388,-0.439499,0.630447,-1.79424,-0.536969,0.707433,-1.83688,-0.435044,0.476214,-1.88575,-0.484412,0.459093,-1.81167,0.290415,1.23098,-2.11072,0.338218,1.16317,-2.09292,0.173072,1.14944,-2.09407,0.220983,1.0836,-2.08062,0.341924,0.998115,-2.08438,0.398717,0.815056,-2.08132,0.28589,0.806327,-2.08704,0.295069,0.736403,-2.07739,0.353609,0.504192,-2.04684,0.295474,0.412711,-1.97074,0.560401,0.494573,-2.03141,0.523245,0.43096,-2.08372,0.115696,1.36451,-1.83008,0.107964,1.39349,-2.04321},
/*458*/{0.279448,1.95408,-1.75283,0.306236,1.89137,-1.91299,0.254572,1.80531,-1.63188,0.235955,1.68611,-1.54331,0.340177,1.70539,-1.66794,0.413506,1.70925,-1.7232,0.177362,1.96553,-2.11747,0.293929,1.88989,-1.9855,0.080806,1.92943,-1.99435,-0.065198,1.8514,-2.11983,-0.097682,1.76348,-2.17765,0.049502,1.63403,-2.16211,0.081188,1.57946,-2.19019,0.205717,1.43939,-1.80521,0.021177,1.53237,-1.86467,0.018543,1.56053,-1.91272,0.018185,1.54007,-1.98053,0.195897,1.47763,-2.0578,0.078854,1.13287,-1.76594,-0.045149,1.17681,-1.78219,-0.072682,1.09798,-1.80191,-0.063283,0.963981,-1.79799,-0.175957,0.804742,-1.78254,-0.231852,0.753704,-1.79074,-0.253707,0.885884,-1.77979,-0.304045,0.836601,-1.78177,-0.445616,0.641739,-1.79461,-0.54301,0.722505,-1.8301,-0.446001,0.490563,-1.88871,-0.492068,0.472108,-1.81369,0.287195,1.22673,-2.11158,0.333896,1.15759,-2.09331,0.168276,1.14606,-2.09302,0.215498,1.08043,-2.08036,0.336681,0.993408,-2.08403,0.40146,0.813596,-2.08046,0.28803,0.800595,-2.08814,0.301,0.732222,-2.0786,0.366625,0.500461,-2.04646,0.309906,0.40558,-1.97472,0.572943,0.496928,-2.03091,0.537994,0.431534,-2.08123,0.116006,1.36541,-1.82894,0.108632,1.39303,-2.04227},
/*459*/{0.279559,1.9537,-1.75397,0.30572,1.89196,-1.91402,0.253792,1.80532,-1.6328,0.234672,1.68661,-1.54401,0.339561,1.70326,-1.66825,0.412836,1.70687,-1.72395,0.178266,1.96539,-2.1188,0.293961,1.88965,-1.98611,0.081019,1.92966,-1.99499,-0.064704,1.84989,-2.11933,-0.096253,1.76174,-2.17638,0.051126,1.63284,-2.1618,0.082519,1.57787,-2.19014,0.206188,1.43995,-1.80464,0.02114,1.53225,-1.8647,0.018543,1.56053,-1.91272,0.018771,1.53861,-1.98023,0.197022,1.47697,-2.05653,0.090409,1.13225,-1.76798,-0.040154,1.1735,-1.77816,-0.067176,1.0935,-1.80178,-0.056661,0.961032,-1.79907,-0.174916,0.805003,-1.78385,-0.233735,0.757339,-1.79099,-0.248992,0.889878,-1.77837,-0.302574,0.842635,-1.78013,-0.450315,0.652938,-1.79322,-0.547099,0.737304,-1.82429,-0.456709,0.505447,-1.89023,-0.501733,0.486424,-1.81439,0.283419,1.22138,-2.11188,0.329835,1.15259,-2.09452,0.16371,1.14249,-2.09162,0.210148,1.07579,-2.07929,0.331754,0.988909,-2.08409,0.402952,0.8119,-2.07877,0.291051,0.795848,-2.08922,0.305454,0.727234,-2.07941,0.375948,0.497595,-2.04633,0.322561,0.400946,-1.97574,0.583004,0.499028,-2.02933,0.550521,0.43238,-2.07971,0.116681,1.36565,-1.82771,0.109829,1.39216,-2.04119},
/*460*/{0.279169,1.95288,-1.75461,0.305084,1.89116,-1.91443,0.252888,1.80502,-1.63383,0.232735,1.68582,-1.54439,0.338696,1.70165,-1.66973,0.411878,1.70354,-1.7249,0.179048,1.96477,-2.11979,0.29381,1.88969,-1.98634,0.081082,1.9295,-1.99676,-0.062631,1.84817,-2.11874,-0.095355,1.76004,-2.17607,0.053343,1.63059,-2.16138,0.084135,1.57579,-2.19053,0.205987,1.44002,-1.80424,0.021118,1.53209,-1.86516,0.018196,1.55989,-1.91279,0.018812,1.53799,-1.98111,0.197643,1.47654,-2.05612,0.096235,1.13016,-1.77032,-0.034054,1.16873,-1.7775,-0.061886,1.08977,-1.80238,-0.049812,0.958053,-1.80031,-0.173304,0.806467,-1.78494,-0.233895,0.760755,-1.792,-0.244133,0.893732,-1.77744,-0.300213,0.849427,-1.77913,-0.453838,0.667034,-1.79137,-0.549445,0.753769,-1.82021,-0.46628,0.520056,-1.89024,-0.511055,0.501003,-1.81464,0.280255,1.21636,-2.11224,0.325188,1.14715,-2.09517,0.159971,1.13999,-2.09134,0.205509,1.07245,-2.07845,0.32672,0.98477,-2.08326,0.405489,0.810546,-2.07689,0.293006,0.791217,-2.08961,0.309248,0.723839,-2.08024,0.386105,0.49501,-2.04657,0.332702,0.397196,-1.97508,0.592661,0.501225,-2.02737,0.56174,0.433178,-2.07851,0.116672,1.36556,-1.82743,0.110504,1.39159,-2.041},
/*461*/{0.279143,1.95215,-1.75498,0.305166,1.89014,-1.91467,0.251412,1.80473,-1.63444,0.23126,1.68501,-1.54485,0.337602,1.69883,-1.67011,0.410676,1.69993,-1.72549,0.179318,1.96394,-2.12049,0.293591,1.88894,-1.98641,0.081313,1.92912,-1.99727,-0.060707,1.84653,-2.11806,-0.0933,1.75761,-2.1762,0.054727,1.62837,-2.16123,0.085608,1.57304,-2.19047,0.205924,1.44021,-1.80294,0.020614,1.53083,-1.86592,0.018226,1.55885,-1.91276,0.019312,1.53565,-1.98125,0.198452,1.47544,-2.05497,0.102844,1.12745,-1.77313,-0.029884,1.16454,-1.77676,-0.055824,1.08559,-1.80143,-0.041857,0.954209,-1.80173,-0.171832,0.808635,-1.78447,-0.233955,0.764892,-1.79241,-0.238681,0.899121,-1.77487,-0.2958,0.85618,-1.77754,-0.456269,0.679338,-1.78837,-0.550019,0.770762,-1.81766,-0.475234,0.534888,-1.88926,-0.522317,0.51648,-1.81427,0.277449,1.21136,-2.11281,0.321855,1.14108,-2.09601,0.156226,1.13611,-2.09066,0.201115,1.0684,-2.07778,0.321905,0.980158,-2.08263,0.405721,0.808747,-2.07496,0.294533,0.787275,-2.08958,0.311106,0.719966,-2.08045,0.393089,0.491747,-2.04546,0.342477,0.393475,-1.9749,0.59987,0.502886,-2.02656,0.569623,0.433861,-2.07704,0.117139,1.36512,-1.82619,0.111967,1.38994,-2.03993},
/*462*/{0.27891,1.95063,-1.75642,0.304717,1.88932,-1.91528,0.25018,1.80343,-1.6356,0.227653,1.68484,-1.54704,0.335462,1.69651,-1.67104,0.409823,1.69586,-1.72616,0.179609,1.9628,-2.12108,0.293926,1.8883,-1.98711,0.081753,1.9289,-1.99838,-0.060754,1.84363,-2.11772,-0.091807,1.75481,-2.17608,0.056479,1.62538,-2.16204,0.08765,1.57038,-2.19034,0.206011,1.44032,-1.80269,0.020665,1.52969,-1.86646,0.017605,1.55796,-1.91385,0.019406,1.53403,-1.9818,0.199124,1.47395,-2.05414,0.108551,1.12493,-1.7746,-0.02516,1.16061,-1.77605,-0.049774,1.08091,-1.80276,-0.034031,0.950028,-1.80298,-0.169097,0.811341,-1.78512,-0.23344,0.770155,-1.79176,-0.2325,0.903647,-1.77499,-0.291434,0.863256,-1.77751,-0.459358,0.694244,-1.78849,-0.549424,0.788747,-1.81635,-0.483015,0.549863,-1.88769,-0.531577,0.53302,-1.81333,0.275112,1.20698,-2.11324,0.318106,1.1367,-2.09691,0.152601,1.13315,-2.08995,0.197268,1.06482,-2.0766,0.317277,0.976089,-2.08231,0.405306,0.806501,-2.07403,0.294512,0.782778,-2.08898,0.312395,0.715986,-2.07991,0.396951,0.489501,-2.04591,0.349011,0.389505,-1.97537,0.604913,0.503644,-2.02619,0.576029,0.434609,-2.07631,0.117423,1.36481,-1.82546,0.112877,1.3883,-2.03936},
/*463*/{0.278586,1.94951,-1.75729,0.304339,1.88909,-1.91542,0.24884,1.80254,-1.63631,0.225581,1.68426,-1.54772,0.33371,1.69289,-1.67108,0.407438,1.69117,-1.72595,0.18014,1.9615,-2.12168,0.293689,1.88732,-1.98774,0.082461,1.92743,-2.00063,-0.058974,1.84171,-2.11714,-0.089616,1.75199,-2.17676,0.05792,1.62243,-2.16185,0.089689,1.56731,-2.19085,0.205972,1.4392,-1.80191,0.020108,1.52713,-1.86637,0.017886,1.55602,-1.91368,0.019327,1.53232,-1.98269,0.200297,1.47231,-2.05391,0.114352,1.12265,-1.77657,-0.019982,1.15525,-1.77555,-0.042605,1.07524,-1.80284,-0.025962,0.945846,-1.80479,-0.166171,0.813952,-1.78479,-0.231572,0.774974,-1.79071,-0.226878,0.908807,-1.77482,-0.286147,0.871356,-1.77655,-0.462537,0.708685,-1.78598,-0.547372,0.80604,-1.81552,-0.490084,0.564313,-1.88531,-0.540729,0.550823,-1.81194,0.272808,1.20209,-2.11398,0.315515,1.13072,-2.0983,0.149933,1.12991,-2.08945,0.193543,1.06105,-2.07577,0.312912,0.971594,-2.08198,0.404218,0.804498,-2.07235,0.294347,0.779663,-2.08798,0.313717,0.712958,-2.07875,0.400665,0.487757,-2.04465,0.353397,0.387511,-1.97437,0.608826,0.503336,-2.02568,0.579661,0.434328,-2.07659,0.117726,1.36315,-1.82524,0.114045,1.38657,-2.03916},
/*464*/{0.277697,1.94781,-1.75804,0.303581,1.88692,-1.91569,0.246058,1.80119,-1.63702,0.22399,1.68165,-1.54839,0.332124,1.6889,-1.67174,0.405635,1.68633,-1.7265,0.180844,1.9598,-2.1227,0.292979,1.8857,-1.98769,0.082306,1.92635,-2.00085,-0.058263,1.83789,-2.11668,-0.087699,1.74839,-2.17723,0.060658,1.61888,-2.16214,0.092013,1.56376,-2.19117,0.2061,1.43825,-1.80148,0.020258,1.52553,-1.86647,0.017853,1.55382,-1.91417,0.019881,1.52952,-1.98386,0.2014,1.46968,-2.05393,0.119512,1.11938,-1.77751,-0.015358,1.14803,-1.77544,-0.036179,1.06918,-1.8037,-0.017933,0.941849,-1.80632,-0.163036,0.816694,-1.78433,-0.229462,0.780337,-1.79026,-0.219087,0.914131,-1.77525,-0.280288,0.878077,-1.77629,-0.463647,0.723188,-1.78462,-0.544137,0.824025,-1.81612,-0.495442,0.579368,-1.88188,-0.548513,0.568344,-1.80996,0.270952,1.19733,-2.1156,0.31242,1.12562,-2.09945,0.147803,1.12656,-2.08839,0.190339,1.05775,-2.07509,0.308195,0.967191,-2.08137,0.402713,0.801936,-2.07083,0.292942,0.775576,-2.08653,0.313349,0.709627,-2.07685,0.401895,0.485432,-2.04379,0.357784,0.3847,-1.97382,0.610682,0.503453,-2.0247,0.584707,0.435244,-2.07611,0.11823,1.36184,-1.8245,0.115068,1.38383,-2.03858},
/*465*/{0.276692,1.94593,-1.75803,0.304088,1.88543,-1.91633,0.244051,1.79914,-1.63751,0.21983,1.68013,-1.54995,0.32926,1.68519,-1.67114,0.404194,1.68073,-1.72648,0.18111,1.95794,-2.1234,0.293302,1.88399,-1.98834,0.082095,1.92461,-2.00245,-0.056407,1.83523,-2.11774,-0.086229,1.74455,-2.17776,0.062213,1.61513,-2.16251,0.094204,1.56068,-2.19156,0.206344,1.43709,-1.80118,0.020366,1.5231,-1.86837,0.016979,1.55092,-1.91473,0.019639,1.52601,-1.98397,0.201413,1.46705,-2.05424,0.124961,1.11643,-1.77842,-0.010257,1.14198,-1.77494,-0.029813,1.06364,-1.80372,-0.009969,0.937361,-1.80684,-0.158916,0.819762,-1.78388,-0.227548,0.786058,-1.78876,-0.210901,0.918807,-1.77637,-0.274101,0.885869,-1.7762,-0.463673,0.737102,-1.78401,-0.539814,0.84194,-1.81743,-0.499772,0.594583,-1.87847,-0.555091,0.586269,-1.80659,0.269927,1.19216,-2.11617,0.311094,1.12044,-2.09949,0.146169,1.12392,-2.08763,0.187824,1.05464,-2.07385,0.305704,0.96327,-2.08095,0.400933,0.799646,-2.07013,0.291094,0.772406,-2.08517,0.312545,0.705901,-2.07489,0.40421,0.483873,-2.04216,0.359377,0.383092,-1.97122,0.612352,0.502188,-2.02382,0.585351,0.434806,-2.07573,0.118888,1.36012,-1.82408,0.115717,1.3807,-2.0383},
/*466*/{0.275501,1.94373,-1.75861,0.303737,1.88342,-1.91658,0.241909,1.79767,-1.63804,0.215243,1.67769,-1.5505,0.32647,1.68045,-1.67167,0.401348,1.67481,-1.72623,0.180423,1.95535,-2.12323,0.29285,1.88181,-1.98782,0.082365,1.92316,-2.00347,-0.055428,1.83107,-2.11831,-0.083785,1.74022,-2.17863,0.064934,1.6106,-2.16324,0.096551,1.55616,-2.19214,0.20638,1.43543,-1.80074,0.020382,1.52057,-1.8687,0.017009,1.54798,-1.91568,0.01985,1.52358,-1.98552,0.202079,1.46424,-2.0543,0.130988,1.11465,-1.77848,-0.004589,1.13568,-1.77443,-0.022744,1.05754,-1.80386,-0.002171,0.932945,-1.80711,-0.154735,0.821873,-1.78279,-0.223303,0.791304,-1.78734,-0.20297,0.92404,-1.77744,-0.26706,0.892832,-1.77685,-0.461987,0.751765,-1.78092,-0.533379,0.858995,-1.81809,-0.5037,0.608919,-1.87491,-0.55917,0.604799,-1.8038,0.269007,1.18781,-2.11704,0.309691,1.11549,-2.10031,0.144853,1.12166,-2.08727,0.185778,1.05171,-2.07367,0.301837,0.959014,-2.08068,0.398338,0.796614,-2.06945,0.288824,0.768515,-2.08379,0.310927,0.702012,-2.07362,0.403957,0.480723,-2.04047,0.358938,0.379709,-1.96955,0.609897,0.499216,-2.02445,0.583859,0.432068,-2.07524,0.119078,1.35813,-1.82389,0.116601,1.3778,-2.03821},
/*467*/{0.274081,1.94156,-1.75896,0.303612,1.88215,-1.91762,0.238415,1.79531,-1.63827,0.212051,1.67545,-1.55182,0.324582,1.67603,-1.67165,0.399579,1.66849,-1.7255,0.181578,1.95303,-2.12402,0.292174,1.87966,-1.98844,0.081583,1.92127,-2.00453,-0.053443,1.8271,-2.11854,-0.081611,1.7352,-2.18072,0.067038,1.6062,-2.16449,0.099349,1.55184,-2.19265,0.206703,1.43403,-1.80065,0.020833,1.51765,-1.86964,0.018313,1.54495,-1.91619,0.020082,1.52055,-1.98626,0.201836,1.4606,-2.05501,0.134765,1.11021,-1.77864,0.00056,1.13072,-1.77443,-0.017166,1.05129,-1.80493,0.005547,0.929129,-1.80792,-0.149154,0.825155,-1.78182,-0.219469,0.796801,-1.78599,-0.193815,0.928958,-1.77787,-0.259694,0.900077,-1.77737,-0.459279,0.766421,-1.7781,-0.525448,0.875412,-1.81974,-0.505146,0.623697,-1.87073,-0.562497,0.622593,-1.80055,0.268315,1.18373,-2.11841,0.307543,1.11049,-2.10151,0.143148,1.11938,-2.08731,0.183304,1.04957,-2.07354,0.300303,0.954946,-2.08076,0.39519,0.792821,-2.06834,0.285715,0.764487,-2.08232,0.307505,0.698411,-2.07212,0.400187,0.476364,-2.0389,0.357341,0.375799,-1.96792,0.608963,0.495396,-2.02254,0.58082,0.428477,-2.07452,0.11978,1.35612,-1.82343,0.11692,1.37415,-2.03789},
/*468*/{0.272615,1.9384,-1.75914,0.30167,1.87941,-1.91653,0.235987,1.79283,-1.63845,0.208276,1.6729,-1.55321,0.320924,1.67122,-1.67093,0.396503,1.66203,-1.72475,0.181148,1.94996,-2.12463,0.292094,1.87708,-1.98859,0.080854,1.91886,-2.00678,-0.052515,1.82202,-2.12103,-0.079683,1.72988,-2.18177,0.068995,1.60173,-2.16549,0.102273,1.54715,-2.1929,0.20618,1.43118,-1.80001,0.019992,1.51407,-1.87041,0.017661,1.54204,-1.91682,0.019269,1.51709,-1.98698,0.202043,1.45735,-2.05563,0.139527,1.10811,-1.77838,0.005714,1.12648,-1.77529,-0.009486,1.04594,-1.8055,0.013572,0.924798,-1.80841,-0.143624,0.827507,-1.78166,-0.214305,0.801194,-1.78531,-0.184898,0.931803,-1.77945,-0.252327,0.905735,-1.77871,-0.455536,0.780865,-1.77457,-0.518066,0.890904,-1.82128,-0.506895,0.637783,-1.86604,-0.562987,0.64033,-1.79592,0.267715,1.1804,-2.11953,0.306422,1.10649,-2.10257,0.142192,1.11705,-2.08703,0.181901,1.04645,-2.07312,0.298988,0.950606,-2.08008,0.391154,0.789102,-2.06801,0.282915,0.760796,-2.08157,0.3043,0.694777,-2.07133,0.397918,0.473821,-2.03712,0.353474,0.371871,-1.96474,0.605662,0.4911,-2.0211,0.576538,0.423324,-2.07341,0.119277,1.35314,-1.82355,0.117011,1.37083,-2.03804},
/*469*/{0.271995,1.9363,-1.75906,0.301036,1.8764,-1.91656,0.233039,1.78992,-1.63864,0.2028,1.67042,-1.55479,0.317271,1.6666,-1.67085,0.39353,1.65509,-1.72306,0.181394,1.9468,-2.12527,0.29215,1.874,-1.98859,0.080133,1.91536,-2.00737,-0.050522,1.81709,-2.12143,-0.077784,1.72411,-2.18351,0.071507,1.59663,-2.16629,0.105935,1.54201,-2.19335,0.206951,1.42959,-1.80095,0.020315,1.51064,-1.87,0.017535,1.5385,-1.91779,0.019398,1.51309,-1.98686,0.202122,1.45468,-2.05615,0.143876,1.1046,-1.77736,0.008815,1.11864,-1.77579,-0.004232,1.03856,-1.80357,0.021404,0.920442,-1.80749,-0.137386,0.829659,-1.78064,-0.208678,0.805968,-1.78458,-0.175218,0.935343,-1.78103,-0.242601,0.911511,-1.77855,-0.451258,0.795025,-1.77144,-0.508614,0.90582,-1.82329,-0.506521,0.651769,-1.86193,-0.563154,0.657546,-1.79213,0.267644,1.17681,-2.11956,0.304844,1.10228,-2.10219,0.141577,1.11477,-2.08674,0.179509,1.04341,-2.07283,0.297903,0.947208,-2.07902,0.388418,0.784769,-2.06776,0.278804,0.756599,-2.08056,0.300136,0.690436,-2.06977,0.394291,0.469869,-2.0349,0.347853,0.369127,-1.96241,0.600808,0.485135,-2.02007,0.571211,0.416574,-2.07129,0.120919,1.3507,-1.82348,0.117794,1.36753,-2.03804},
/*470*/{0.269762,1.93297,-1.75902,0.301581,1.87239,-1.91711,0.229327,1.78679,-1.63862,0.198735,1.66744,-1.55574,0.31401,1.66085,-1.67088,0.390084,1.64747,-1.72236,0.181289,1.94318,-2.12602,0.291805,1.8709,-1.98867,0.079484,1.9122,-2.00895,-0.051351,1.81281,-2.12235,-0.075927,1.71742,-2.18604,0.073999,1.59084,-2.16783,0.109148,1.53642,-2.19441,0.208051,1.42684,-1.80132,0.01939,1.50685,-1.87067,0.018423,1.53471,-1.91784,0.018975,1.50899,-1.98807,0.201553,1.45164,-2.05641,0.149241,1.1023,-1.77677,0.014056,1.11463,-1.77589,0.001019,1.03325,-1.80456,0.028245,0.916347,-1.80725,-0.13112,0.831837,-1.78014,-0.202708,0.811618,-1.78293,-0.164398,0.938944,-1.78145,-0.232567,0.917551,-1.77861,-0.445638,0.80801,-1.77198,-0.498731,0.919614,-1.82514,-0.505213,0.663409,-1.85791,-0.561594,0.674229,-1.78763,0.266972,1.17334,-2.11999,0.304069,1.09864,-2.10332,0.140617,1.11246,-2.08557,0.1783,1.04052,-2.07226,0.298332,0.944106,-2.07826,0.385335,0.779717,-2.06735,0.27481,0.752404,-2.07877,0.295582,0.68591,-2.06821,0.386101,0.462468,-2.03333,0.340272,0.364718,-1.9591,0.594378,0.478505,-2.01902,0.565487,0.409135,-2.06868,0.121818,1.34755,-1.82376,0.118244,1.36393,-2.03834},
/*471*/{0.267963,1.9297,-1.75887,0.300612,1.86973,-1.91689,0.226171,1.78388,-1.63841,0.194505,1.66442,-1.55679,0.31025,1.65488,-1.67065,0.386986,1.6399,-1.72096,0.18161,1.93956,-2.12674,0.291121,1.86767,-1.98884,0.078047,1.90856,-2.01075,-0.049788,1.80626,-2.12306,-0.073446,1.711,-2.18791,0.077469,1.58538,-2.16875,0.112482,1.53128,-2.19528,0.20797,1.42379,-1.80144,0.019233,1.50284,-1.87086,0.017943,1.53042,-1.91867,0.018271,1.50458,-1.98762,0.201861,1.44868,-2.05708,0.154077,1.09769,-1.77711,0.016857,1.10719,-1.77567,0.007599,1.02767,-1.80465,0.035973,0.912044,-1.8063,-0.124365,0.833202,-1.77891,-0.198164,0.81444,-1.78218,-0.154012,0.941573,-1.78253,-0.223028,0.922767,-1.77978,-0.438503,0.820235,-1.77281,-0.488468,0.93264,-1.82713,-0.502096,0.676455,-1.85402,-0.558156,0.69045,-1.78394,0.266313,1.16951,-2.1199,0.303279,1.09504,-2.10361,0.139601,1.10897,-2.0858,0.17751,1.03724,-2.07224,0.297846,0.940849,-2.07741,0.380997,0.774147,-2.06726,0.270279,0.748169,-2.07785,0.290501,0.681433,-2.06701,0.379986,0.457728,-2.03216,0.332385,0.360874,-1.95613,0.587454,0.469612,-2.01745,0.556451,0.39982,-2.06555,0.122513,1.34402,-1.82402,0.11882,1.36035,-2.0386},
/*472*/{0.266771,1.92619,-1.75869,0.300653,1.86559,-1.91705,0.222513,1.77983,-1.63858,0.188278,1.66184,-1.55881,0.30619,1.64853,-1.67058,0.383252,1.63224,-1.71965,0.180085,1.9349,-2.12728,0.291143,1.86404,-1.98902,0.077742,1.90426,-2.01257,-0.056628,1.80471,-2.12765,-0.071162,1.70392,-2.19023,0.080985,1.57925,-2.1708,0.115838,1.526,-2.19588,0.208049,1.4208,-1.80183,0.018683,1.4983,-1.87134,0.017471,1.52602,-1.91898,0.017476,1.4997,-1.98755,0.201801,1.44523,-2.05801,0.158727,1.0937,-1.77464,0.021692,1.10161,-1.77658,0.013066,1.0211,-1.80576,0.044188,0.907859,-1.80526,-0.116866,0.83531,-1.7781,-0.190501,0.819486,-1.77935,-0.144246,0.944213,-1.78433,-0.213594,0.928287,-1.78146,-0.430257,0.832069,-1.77375,-0.477983,0.944916,-1.82994,-0.49881,0.688015,-1.84923,-0.554225,0.705509,-1.77969,0.26605,1.1661,-2.11962,0.302386,1.0909,-2.10384,0.139535,1.10536,-2.08478,0.1769,1.0339,-2.07167,0.296715,0.93766,-2.0762,0.376478,0.767956,-2.06791,0.265595,0.74356,-2.07721,0.285347,0.676744,-2.06596,0.371048,0.451442,-2.03051,0.323055,0.356098,-1.95454,0.580515,0.461245,-2.01371,0.548398,0.390977,-2.06255,0.123321,1.34033,-1.82427,0.119179,1.35633,-2.03887},
/*473*/{0.265018,1.92215,-1.75814,0.301236,1.86214,-1.91794,0.218525,1.77549,-1.63842,0.182863,1.65746,-1.55936,0.300965,1.64272,-1.66975,0.379224,1.62397,-1.71789,0.180458,1.93018,-2.12803,0.290602,1.8604,-1.98926,0.077284,1.89948,-2.01331,-0.048389,1.79164,-2.12983,-0.069316,1.69652,-2.19257,0.083906,1.57308,-2.17255,0.119241,1.51937,-2.1967,0.207863,1.41748,-1.80239,0.0184,1.49334,-1.87202,0.017103,1.5214,-1.91933,0.017117,1.49471,-1.98793,0.200984,1.44229,-2.05913,0.161388,1.09192,-1.77377,0.025753,1.09599,-1.77816,0.021189,1.01551,-1.80389,0.052109,0.9039,-1.8032,-0.109233,0.837151,-1.77634,-0.183128,0.822797,-1.77824,-0.132795,0.945983,-1.78574,-0.202758,0.93164,-1.78231,-0.421631,0.842564,-1.77449,-0.467057,0.955839,-1.83226,-0.494037,0.698438,-1.8448,-0.548798,0.719268,-1.77571,0.265679,1.16254,-2.1198,0.302392,1.08644,-2.10317,0.138242,1.10209,-2.08446,0.17634,1.03046,-2.07172,0.295246,0.934196,-2.07565,0.372255,0.760756,-2.06717,0.260681,0.738607,-2.07652,0.278989,0.671696,-2.06507,0.361659,0.445279,-2.02873,0.312198,0.351977,-1.95214,0.57073,0.451053,-2.01199,0.538293,0.380819,-2.05913,0.124134,1.33627,-1.82512,0.119523,1.35251,-2.03969},
/*474*/{0.263479,1.91812,-1.75757,0.300078,1.85732,-1.9172,0.215428,1.77126,-1.63879,0.178199,1.6538,-1.56009,0.296431,1.63566,-1.66902,0.375021,1.61533,-1.71673,0.17978,1.92503,-2.1282,0.290198,1.85631,-1.98964,0.076676,1.89426,-2.01578,-0.048321,1.78341,-2.13244,-0.066924,1.68844,-2.19495,0.086957,1.56618,-2.17384,0.123578,1.51317,-2.19721,0.208536,1.41418,-1.80311,0.018471,1.48784,-1.87181,0.016836,1.51676,-1.91929,0.016493,1.48944,-1.98732,0.201075,1.43906,-2.06001,0.166859,1.08822,-1.77123,0.029933,1.09078,-1.77897,0.026838,1.01032,-1.80664,0.060154,0.899842,-1.80167,-0.100569,0.838651,-1.77516,-0.1757,0.825009,-1.77723,-0.121438,0.947545,-1.78723,-0.191844,0.934921,-1.78352,-0.414092,0.853459,-1.77117,-0.456469,0.96509,-1.83433,-0.4875,0.708656,-1.84151,-0.54124,0.73311,-1.77159,0.265032,1.15858,-2.11995,0.302252,1.08383,-2.10317,0.138115,1.09838,-2.08488,0.175541,1.02585,-2.0716,0.29422,0.930025,-2.07546,0.367197,0.755026,-2.06715,0.255587,0.734197,-2.07543,0.272557,0.666563,-2.06431,0.351376,0.438802,-2.0279,0.298094,0.350155,-1.95228,0.560315,0.440161,-2.00909,0.527235,0.371076,-2.05643,0.125836,1.33207,-1.82556,0.120265,1.34853,-2.0401},
/*475*/{0.261323,1.9131,-1.75713,0.300103,1.85362,-1.91694,0.212359,1.76702,-1.6386,0.173691,1.65039,-1.56062,0.291414,1.62888,-1.66879,0.370765,1.60697,-1.71456,0.180821,1.92007,-2.12965,0.290602,1.85201,-1.98968,0.075396,1.88875,-2.01754,-0.04612,1.77605,-2.13531,-0.064683,1.68087,-2.19775,0.091016,1.55983,-2.17539,0.128182,1.50724,-2.19823,0.208823,1.41042,-1.80425,0.01865,1.48242,-1.87157,0.016725,1.51107,-1.91935,0.016369,1.48406,-1.98707,0.201167,1.43555,-2.06102,0.169242,1.08497,-1.77216,0.033225,1.08547,-1.78126,0.030911,1.00403,-1.80631,0.069281,0.895831,-1.79969,-0.091656,0.838826,-1.77381,-0.167394,0.827456,-1.77653,-0.109566,0.949265,-1.78851,-0.180884,0.938098,-1.78502,-0.40328,0.863109,-1.77285,-0.444809,0.974684,-1.83787,-0.480293,0.71876,-1.83774,-0.53346,0.744731,-1.76753,0.264241,1.15442,-2.11953,0.301536,1.07975,-2.10285,0.137347,1.09405,-2.08422,0.175427,1.02175,-2.07176,0.291797,0.923411,-2.07494,0.362907,0.748033,-2.0658,0.250446,0.729638,-2.07462,0.265966,0.661553,-2.06354,0.336975,0.434421,-2.0256,0.281527,0.353372,-1.94946,0.549518,0.429662,-2.00612,0.511942,0.359074,-2.05366,0.127437,1.3274,-1.8263,0.121042,1.34426,-2.04078},
/*476*/{0.259435,1.90827,-1.75688,0.30102,1.84868,-1.91657,0.208427,1.76175,-1.6387,0.166175,1.64602,-1.56138,0.286462,1.62225,-1.66831,0.365195,1.59872,-1.71257,0.180907,1.91461,-2.13031,0.290552,1.84726,-1.98944,0.074852,1.88349,-2.01948,-0.046275,1.76745,-2.13829,-0.061895,1.67236,-2.20111,0.095233,1.55248,-2.17688,0.133234,1.50113,-2.19871,0.209229,1.40717,-1.80529,0.018144,1.47657,-1.87176,0.016112,1.50568,-1.91946,0.015154,1.47824,-1.98618,0.20042,1.43128,-2.06191,0.172822,1.08193,-1.7697,0.038014,1.07981,-1.78182,0.037118,0.998755,-1.80711,0.077588,0.892023,-1.79811,-0.083231,0.840108,-1.77188,-0.158228,0.831512,-1.77437,-0.098068,0.950065,-1.79027,-0.169599,0.940956,-1.7874,-0.392567,0.872609,-1.77406,-0.433903,0.982848,-1.84112,-0.471771,0.727129,-1.8325,-0.523584,0.755496,-1.76451,0.263744,1.15039,-2.11945,0.301004,1.07562,-2.10255,0.136388,1.08912,-2.08446,0.174923,1.01726,-2.07146,0.290464,0.917765,-2.07491,0.357217,0.740272,-2.06493,0.244695,0.724062,-2.07413,0.258699,0.655753,-2.06279,0.329818,0.431947,-2.02359,0.263048,0.354312,-1.94895,0.534575,0.414342,-2.004,0.493137,0.349309,-2.05145,0.128839,1.3231,-1.82644,0.121083,1.33936,-2.04092},
/*477*/{0.25772,1.90308,-1.75573,0.300861,1.8431,-1.91649,0.204633,1.75673,-1.63924,0.159961,1.64174,-1.56206,0.281547,1.61549,-1.66754,0.360077,1.58962,-1.71046,0.181486,1.90927,-2.13141,0.290688,1.84269,-1.98906,0.074686,1.8773,-2.02038,-0.044415,1.75759,-2.14121,-0.058829,1.66381,-2.20426,0.099525,1.54539,-2.17872,0.138398,1.49429,-2.19922,0.20923,1.40306,-1.80614,0.017864,1.47087,-1.8709,0.015154,1.49987,-1.92016,0.014189,1.47246,-1.98645,0.200176,1.42749,-2.06317,0.17602,1.07812,-1.76664,0.042188,1.076,-1.78272,0.043479,0.991811,-1.80717,0.085862,0.888188,-1.79567,-0.073873,0.840681,-1.77039,-0.149157,0.83366,-1.77272,-0.08574,0.950161,-1.79105,-0.157418,0.942861,-1.78822,-0.382922,0.880446,-1.77284,-0.421776,0.989789,-1.84365,-0.462587,0.734338,-1.82994,-0.515076,0.766064,-1.7613,0.262333,1.14561,-2.11918,0.300673,1.07116,-2.10284,0.135452,1.08461,-2.08437,0.17423,1.01283,-2.07113,0.287879,0.911328,-2.07493,0.351171,0.733612,-2.06417,0.239079,0.71892,-2.07334,0.251854,0.65062,-2.06189,0.317449,0.425914,-2.02085,0.244486,0.356867,-1.94501,0.52326,0.396045,-2.0018,0.475499,0.339106,-2.04976,0.12994,1.31827,-1.82719,0.121395,1.33476,-2.04162},
/*478*/{0.256688,1.89895,-1.75566,0.29979,1.83783,-1.91537,0.200412,1.75116,-1.63917,0.15361,1.63859,-1.56272,0.275812,1.60834,-1.6658,0.355201,1.58102,-1.7081,0.181264,1.90319,-2.13287,0.290193,1.83763,-1.98882,0.07328,1.87138,-2.02324,-0.044067,1.74847,-2.14377,-0.056014,1.65504,-2.20722,0.103895,1.53982,-2.17997,0.143498,1.48771,-2.20016,0.209669,1.39839,-1.80704,0.016842,1.46484,-1.87114,0.015037,1.49472,-1.91996,0.013035,1.46672,-1.98578,0.199169,1.42349,-2.06444,0.178789,1.07404,-1.76378,0.046646,1.06738,-1.78265,0.049981,0.985072,-1.80678,0.095439,0.88427,-1.79401,-0.063191,0.841481,-1.76793,-0.140059,0.835765,-1.77103,-0.073483,0.950557,-1.79294,-0.145859,0.945175,-1.78946,-0.370588,0.888471,-1.77356,-0.410097,0.995768,-1.84656,-0.45208,0.741087,-1.8253,-0.503688,0.774847,-1.75732,0.261535,1.14169,-2.11943,0.29936,1.06696,-2.10251,0.134741,1.08014,-2.0853,0.173587,1.00786,-2.07122,0.285717,0.905726,-2.07513,0.345967,0.726286,-2.06196,0.232814,0.714365,-2.07313,0.245871,0.645524,-2.06036,0.299474,0.42153,-2.01472,0.223755,0.353568,-1.94575,0.505517,0.387281,-1.98889,0.455133,0.341983,-2.04932,0.130779,1.3131,-1.82811,0.121028,1.33018,-2.04244},
/*479*/{0.254794,1.8927,-1.75433,0.298659,1.83307,-1.91481,0.196215,1.74573,-1.63948,0.147058,1.63363,-1.56344,0.270528,1.6018,-1.66472,0.348878,1.57288,-1.70538,0.182099,1.89744,-2.134,0.29025,1.83248,-1.98923,0.072537,1.86464,-2.02466,-0.042163,1.739,-2.14686,-0.052512,1.64547,-2.21029,0.109176,1.53265,-2.18141,0.149404,1.48085,-2.20062,0.209552,1.39386,-1.80856,0.016206,1.45984,-1.87068,0.014305,1.48817,-1.92052,0.01087,1.46115,-1.98601,0.197873,1.41893,-2.06578,0.185916,1.07033,-1.76064,0.050647,1.0627,-1.784,0.056129,0.979266,-1.80624,0.104634,0.881091,-1.79088,-0.053062,0.841945,-1.76619,-0.129048,0.836853,-1.76959,-0.060844,0.949571,-1.7939,-0.132266,0.946416,-1.79026,-0.360345,0.894571,-1.77295,-0.398304,1.00054,-1.84892,-0.441158,0.746531,-1.82189,-0.492308,0.781815,-1.75432,0.258763,1.13756,-2.12013,0.297455,1.06335,-2.1043,0.132383,1.07629,-2.08526,0.171903,1.00467,-2.07129,0.281504,0.899649,-2.07548,0.339202,0.718439,-2.05736,0.227193,0.71012,-2.07193,0.236185,0.641618,-2.05919,0.283409,0.422522,-2.00898,0.199533,0.353678,-1.94618,0.480869,0.385955,-1.98993,0.43137,0.34189,-2.04904,0.131268,1.30817,-1.82893,0.120055,1.32506,-2.04321},
/*480*/{0.252851,1.88749,-1.7539,0.299719,1.82743,-1.91378,0.192292,1.74056,-1.63942,0.140226,1.63023,-1.56411,0.264244,1.59478,-1.66429,0.343699,1.5647,-1.70317,0.183021,1.89149,-2.135,0.290685,1.82689,-1.98891,0.071399,1.85897,-2.02728,-0.040528,1.72909,-2.15162,-0.048074,1.63622,-2.21349,0.113843,1.52529,-2.18258,0.155616,1.47428,-2.20086,0.209002,1.3889,-1.80908,0.014913,1.45326,-1.87097,0.012441,1.48193,-1.92024,0.008437,1.4566,-1.98561,0.196454,1.41389,-2.06702,0.191291,1.06507,-1.7593,0.055574,1.05443,-1.78361,0.062987,0.97193,-1.8065,0.114303,0.876766,-1.78947,-0.042156,0.841464,-1.76449,-0.116861,0.839376,-1.76778,-0.047271,0.949472,-1.79517,-0.119629,0.947259,-1.79199,-0.347208,0.899524,-1.77408,-0.386203,1.00479,-1.85195,-0.429298,0.751254,-1.81857,-0.480602,0.788748,-1.75196,0.255176,1.13506,-2.12314,0.294283,1.06077,-2.10748,0.130246,1.07308,-2.08559,0.169748,1.00245,-2.07096,0.276751,0.894386,-2.07609,0.334621,0.71321,-2.05272,0.222587,0.707792,-2.06984,0.227495,0.639621,-2.05556,0.2604,0.42288,-2.00391,0.177231,0.354382,-1.94092,0.456417,0.386742,-1.98601,0.408028,0.342595,-2.04797,0.130847,1.30262,-1.82975,0.118682,1.32005,-2.04394},
/*481*/{0.251403,1.88172,-1.75292,0.300324,1.82138,-1.91311,0.18773,1.73523,-1.63988,0.134,1.62677,-1.56574,0.258313,1.5883,-1.6631,0.336834,1.55713,-1.70077,0.183365,1.88616,-2.13657,0.291637,1.82166,-1.98872,0.070193,1.85195,-2.02887,-0.037843,1.72018,-2.15658,-0.04394,1.62651,-2.21678,0.120208,1.51809,-2.18394,0.162447,1.46775,-2.20149,0.208668,1.38371,-1.81074,0.012277,1.44786,-1.87,0.010966,1.47548,-1.91955,0.004497,1.45248,-1.98621,0.194019,1.40988,-2.06779,0.195719,1.0626,-1.75765,0.061227,1.04915,-1.78396,0.069795,0.965985,-1.80745,0.12389,0.873163,-1.78611,-0.031024,0.840499,-1.76387,-0.105885,0.840239,-1.76601,-0.033747,0.948724,-1.79605,-0.104562,0.947673,-1.79105,-0.333391,0.90384,-1.77471,-0.373822,1.0075,-1.85455,-0.416429,0.754477,-1.81481,-0.46711,0.793871,-1.74871,0.252218,1.13276,-2.12782,0.291202,1.05874,-2.11152,0.128451,1.0721,-2.08557,0.167449,1.00194,-2.07091,0.272097,0.890841,-2.07614,0.329552,0.710748,-2.04766,0.218826,0.711558,-2.06673,0.218547,0.642948,-2.05087,0.235793,0.424457,-1.99922,0.154398,0.355689,-1.93856,0.435899,0.383161,-1.98464,0.385073,0.34287,-2.04662,0.13004,1.29721,-1.83112,0.116283,1.31583,-2.04511},
/*482*/{0.250258,1.87671,-1.75177,0.301571,1.81539,-1.91212,0.185005,1.73012,-1.64003,0.127095,1.62284,-1.56652,0.25187,1.58187,-1.66131,0.329938,1.54955,-1.69802,0.184643,1.88079,-2.13848,0.291993,1.81614,-1.98863,0.070026,1.84843,-2.02992,-0.036047,1.70988,-2.16132,-0.038662,1.6164,-2.21976,0.126821,1.51121,-2.18537,0.169198,1.46145,-2.2018,0.207304,1.37866,-1.81303,0.010315,1.44217,-1.87025,0.008105,1.47037,-1.91926,0.001148,1.44768,-1.98593,0.191228,1.40821,-2.06822,0.198156,1.05865,-1.75609,0.06497,1.04256,-1.78521,0.076086,0.958742,-1.80695,0.134595,0.869816,-1.78414,-0.018676,0.839285,-1.7628,-0.093028,0.839642,-1.76499,-0.020212,0.946706,-1.79654,-0.091767,0.947294,-1.7912,-0.323349,0.90683,-1.77449,-0.361674,1.00912,-1.85705,-0.402869,0.757898,-1.81202,-0.453362,0.7971,-1.74684,0.249257,1.13056,-2.13281,0.288852,1.0574,-2.11497,0.125806,1.07179,-2.08503,0.164995,1.00154,-2.07105,0.269229,0.891146,-2.07606,0.322473,0.710602,-2.04326,0.211778,0.71728,-2.0639,0.208333,0.648467,-2.04786,0.219399,0.427058,-1.99782,0.131808,0.357946,-1.93734,0.414041,0.38401,-1.98248,0.363004,0.342683,-2.04517,0.129258,1.29187,-1.83358,0.114326,1.31312,-2.04725},
/*483*/{0.249747,1.87167,-1.75063,0.301404,1.81021,-1.91107,0.180959,1.72441,-1.64014,0.119928,1.61877,-1.56648,0.245487,1.57605,-1.65983,0.322935,1.54394,-1.69544,0.186116,1.87578,-2.14066,0.293179,1.81044,-1.98763,0.071047,1.84632,-2.03019,-0.032059,1.70134,-2.16748,-0.032983,1.60653,-2.2229,0.133038,1.50448,-2.1863,0.17722,1.45573,-2.20222,0.20614,1.37377,-1.81535,0.007301,1.43686,-1.87003,0.005221,1.4649,-1.91818,-0.001945,1.44372,-1.98582,0.188517,1.40619,-2.06977,0.200975,1.05495,-1.75485,0.070349,1.03546,-1.78535,0.083602,0.952582,-1.80737,0.145661,0.865712,-1.781,-0.005942,0.837865,-1.76168,-0.081879,0.838876,-1.76361,-0.007298,0.944974,-1.79605,-0.077782,0.947382,-1.79215,-0.309226,0.909782,-1.77479,-0.349097,1.00966,-1.85896,-0.388283,0.758688,-1.80922,-0.440627,0.800056,-1.74568,0.24686,1.12966,-2.13555,0.286781,1.05605,-2.11674,0.124351,1.07153,-2.08514,0.16421,1.00048,-2.07055,0.268455,0.892421,-2.07604,0.314213,0.7102,-2.04231,0.203742,0.722684,-2.06257,0.19696,0.654144,-2.04684,0.200624,0.429574,-1.995,0.109902,0.357525,-1.93709,0.391785,0.381221,-1.98049,0.339615,0.343331,-2.04406,0.128322,1.28665,-1.83613,0.111859,1.31051,-2.04941},
/*484*/{0.248308,1.86683,-1.74972,0.301037,1.8055,-1.91013,0.176749,1.72015,-1.64035,0.112016,1.61585,-1.5676,0.238772,1.57099,-1.65846,0.315712,1.53709,-1.69283,0.187078,1.87098,-2.14253,0.294133,1.80611,-1.98749,0.071159,1.84397,-2.03041,-0.02615,1.69056,-2.17435,-0.025811,1.59632,-2.22598,0.141153,1.49831,-2.18713,0.18507,1.45032,-2.20223,0.20413,1.36885,-1.81649,0.004949,1.43247,-1.8696,0.002957,1.46147,-1.91857,-0.005379,1.43994,-1.9831,0.186159,1.40412,-2.07184,0.205586,1.05432,-1.75406,0.076501,1.0289,-1.78452,0.091428,0.946714,-1.80725,0.156952,0.86268,-1.7795,0.007268,0.83656,-1.76052,-0.068353,0.837726,-1.76344,0.00723,0.943803,-1.79534,-0.06462,0.946177,-1.79188,-0.294958,0.909903,-1.77393,-0.33651,1.00963,-1.86009,-0.373148,0.759264,-1.80694,-0.425047,0.80026,-1.74435,0.24351,1.1281,-2.13607,0.283717,1.05522,-2.11743,0.122076,1.06899,-2.08403,0.162503,0.999625,-2.07007,0.271598,0.894009,-2.07545,0.303864,0.709314,-2.04333,0.192815,0.726723,-2.06177,0.184005,0.658394,-2.04741,0.180366,0.429431,-1.99264,0.088544,0.358468,-1.93415,0.370468,0.380729,-1.98123,0.318398,0.343472,-2.04251,0.126582,1.282,-1.83796,0.108774,1.30805,-2.05087},
/*485*/{0.248255,1.86266,-1.74819,0.301544,1.80082,-1.9088,0.171496,1.71644,-1.64092,0.103704,1.61409,-1.56853,0.230999,1.56677,-1.65718,0.307833,1.53146,-1.69073,0.188492,1.86655,-2.14384,0.294705,1.80215,-1.98696,0.071176,1.84252,-2.03064,-0.021291,1.68177,-2.18221,-0.018017,1.58663,-2.22869,0.149237,1.49287,-2.18767,0.193827,1.44516,-2.20222,0.202806,1.36443,-1.81858,0.002171,1.42774,-1.86788,0.001115,1.45956,-1.91675,-0.007714,1.43704,-1.9812,0.182956,1.40302,-2.07381,0.209922,1.05502,-1.75423,0.081808,1.02471,-1.78486,0.100958,0.942401,-1.80625,0.168789,0.859515,-1.77967,0.019849,0.834719,-1.76028,-0.054312,0.836006,-1.76388,0.020727,0.942804,-1.79526,-0.050013,0.944714,-1.79035,-0.281621,0.909874,-1.77612,-0.324218,1.00812,-1.86156,-0.357466,0.757594,-1.80533,-0.40944,0.799246,-1.74276,0.239943,1.12707,-2.13628,0.280011,1.05399,-2.11889,0.119096,1.06742,-2.08321,0.159837,0.997557,-2.06861,0.274374,0.893806,-2.07374,0.292863,0.708167,-2.04633,0.181259,0.729543,-2.06135,0.172219,0.660821,-2.04733,0.163964,0.43094,-1.99178,0.067198,0.360179,-1.93471,0.348834,0.379766,-1.97939,0.296508,0.343005,-2.04226,0.125288,1.27764,-1.84025,0.105218,1.3068,-2.05256},
/*486*/{0.24811,1.85824,-1.74603,0.301933,1.79764,-1.9078,0.166845,1.71474,-1.64117,0.094321,1.61367,-1.56983,0.223652,1.56263,-1.65592,0.300495,1.52638,-1.68797,0.190032,1.86351,-2.14393,0.294679,1.79924,-1.98584,0.071752,1.84044,-2.03048,-0.014182,1.67238,-2.18784,-0.009232,1.57749,-2.231,0.158117,1.48829,-2.18808,0.203702,1.44114,-2.20219,0.201011,1.36088,-1.81939,0.000371,1.42545,-1.86533,-0.000413,1.45775,-1.91615,-0.009643,1.43588,-1.97866,0.180455,1.40246,-2.07589,0.216829,1.05692,-1.75365,0.087067,1.02023,-1.78358,0.109601,0.94085,-1.80537,0.181762,0.858947,-1.7781,0.033954,0.832844,-1.76023,-0.04107,0.834612,-1.76368,0.035138,0.941576,-1.79254,-0.036429,0.943416,-1.79069,-0.267602,0.908187,-1.77576,-0.313598,1.00549,-1.86228,-0.341458,0.754846,-1.80435,-0.393369,0.795474,-1.7417,0.235014,1.12487,-2.13699,0.275593,1.051,-2.12029,0.114553,1.06396,-2.08148,0.15549,0.995414,-2.06869,0.275831,0.895304,-2.07232,0.280586,0.70649,-2.0501,0.169227,0.729572,-2.06264,0.158802,0.661398,-2.04697,0.140713,0.43131,-1.99258,0.044773,0.362319,-1.93439,0.328216,0.378448,-1.97964,0.274565,0.342486,-2.04199,0.123619,1.27456,-1.8417,0.101861,1.3062,-2.05349},
/*487*/{0.248243,1.85493,-1.74515,0.301766,1.7952,-1.90728,0.161373,1.71395,-1.64162,0.087115,1.61428,-1.57212,0.216055,1.55984,-1.65447,0.29222,1.52233,-1.68511,0.191287,1.86057,-2.1436,0.294413,1.79643,-1.98505,0.071587,1.83948,-2.03058,-0.008988,1.66575,-2.19511,0.000341,1.56937,-2.2334,0.167487,1.48459,-2.18808,0.213901,1.43814,-2.20125,0.198428,1.35747,-1.82038,-0.000929,1.42389,-1.86286,-0.002234,1.45716,-1.9162,-0.011418,1.43541,-1.97535,0.178512,1.40181,-2.07768,0.223587,1.05946,-1.7536,0.094134,1.01801,-1.78215,0.120346,0.939397,-1.80199,0.195687,0.858656,-1.77852,0.046474,0.831632,-1.76005,-0.02606,0.83335,-1.76512,0.048412,0.940435,-1.79314,-0.022466,0.943459,-1.79008,-0.253246,0.906492,-1.77737,-0.303174,1.00212,-1.8627,-0.325434,0.750671,-1.80327,-0.376678,0.790711,-1.73993,0.23065,1.12222,-2.13752,0.269961,1.04761,-2.12144,0.110218,1.06177,-2.08075,0.149559,0.993543,-2.06812,0.269663,0.893298,-2.0729,0.26682,0.702711,-2.05354,0.15704,0.728214,-2.06322,0.143016,0.660581,-2.04695,0.120618,0.430865,-1.99236,0.023656,0.362225,-1.93528,0.305972,0.37871,-1.98046,0.25294,0.342057,-2.04277,0.121172,1.27188,-1.84296,0.09821,1.30594,-2.05424},
/*488*/{0.24787,1.85224,-1.74491,0.301611,1.7922,-1.90661,0.156165,1.714,-1.64223,0.079019,1.6157,-1.57412,0.208342,1.55923,-1.65518,0.284638,1.51907,-1.68243,0.192489,1.85896,-2.14317,0.294884,1.79406,-1.98415,0.072168,1.83747,-2.03049,-0.001633,1.65837,-2.19896,0.011358,1.56237,-2.23541,0.177255,1.48166,-2.18722,0.2248,1.43622,-2.20037,0.195888,1.35435,-1.82123,-0.001774,1.42337,-1.86251,-0.004044,1.45693,-1.91619,-0.012603,1.43523,-1.97395,0.176051,1.40117,-2.07883,0.230663,1.0617,-1.75327,0.102132,1.01647,-1.78045,0.128878,0.93874,-1.80105,0.210438,0.860681,-1.77741,0.061085,0.83149,-1.76016,-0.013921,0.832163,-1.76519,0.061028,0.940122,-1.79151,-0.010987,0.941495,-1.79096,-0.240696,0.901817,-1.77693,-0.294249,0.997586,-1.8616,-0.307563,0.745017,-1.80321,-0.357814,0.783131,-1.73859,0.22412,1.11871,-2.13931,0.263371,1.04337,-2.12274,0.103783,1.06058,-2.07998,0.142282,0.99128,-2.06689,0.260904,0.88851,-2.07507,0.253789,0.697329,-2.05461,0.143941,0.72727,-2.06449,0.128548,0.660148,-2.04813,0.103186,0.432284,-1.99276,0.001531,0.36195,-1.93438,0.285565,0.37749,-1.9803,0.231471,0.341726,-2.04304,0.118283,1.26977,-1.84434,0.094819,1.30572,-2.05525},
/*489*/{0.248227,1.85064,-1.74433,0.301007,1.79075,-1.90528,0.150513,1.71449,-1.64258,0.071046,1.61846,-1.57712,0.200748,1.55851,-1.65459,0.276884,1.51706,-1.68115,0.194302,1.85742,-2.14182,0.294678,1.79244,-1.98348,0.072715,1.83666,-2.0307,0.005725,1.65188,-2.2024,0.020048,1.55758,-2.23642,0.187587,1.4803,-2.18684,0.236312,1.4357,-2.1993,0.193776,1.35221,-1.82117,-0.003734,1.42344,-1.86145,-0.005933,1.45652,-1.91556,-0.013667,1.436,-1.97305,0.174817,1.40042,-2.07955,0.237655,1.06522,-1.75301,0.110594,1.01664,-1.77941,0.140176,0.938886,-1.7987,0.223681,0.864025,-1.77595,0.07388,0.83081,-1.76015,-0.00165,0.830901,-1.76655,0.073569,0.940227,-1.79023,0.002817,0.94061,-1.78884,-0.227678,0.896771,-1.77829,-0.285641,0.99142,-1.86076,-0.289047,0.738281,-1.80495,-0.339318,0.775004,-1.73736,0.217958,1.11554,-2.14012,0.256391,1.03952,-2.12411,0.098198,1.05869,-2.0799,0.135022,0.989435,-2.06727,0.250368,0.883355,-2.07646,0.239136,0.693185,-2.05649,0.13127,0.724709,-2.06643,0.113513,0.658815,-2.04967,0.080825,0.43247,-1.99484,-0.019786,0.363203,-1.93591,0.26418,0.377438,-1.98131,0.21001,0.341417,-2.04365,0.115454,1.26842,-1.84476,0.092159,1.3057,-2.05546},
/*490*/{0.247879,1.84935,-1.7439,0.300551,1.78951,-1.90441,0.145041,1.71557,-1.64362,0.0627,1.62204,-1.57962,0.192918,1.55929,-1.65398,0.26827,1.516,-1.67847,0.196147,1.85644,-2.14143,0.295251,1.79082,-1.98191,0.073003,1.83577,-2.03188,0.013428,1.64837,-2.20407,0.029074,1.55353,-2.23703,0.197789,1.47937,-2.18548,0.247696,1.43607,-2.19736,0.192923,1.3511,-1.82126,-0.00416,1.42418,-1.86148,-0.006773,1.45656,-1.91583,-0.013744,1.43603,-1.97255,0.174785,1.4002,-2.07998,0.244729,1.06884,-1.75317,0.120066,1.01917,-1.77758,0.151027,0.940624,-1.79707,0.237135,0.866262,-1.77623,0.087267,0.830275,-1.76131,0.011186,0.829776,-1.7678,0.085193,0.94064,-1.78947,0.013618,0.939963,-1.78819,-0.212985,0.890688,-1.77755,-0.278121,0.984481,-1.85992,-0.26958,0.73157,-1.8051,-0.32007,0.765625,-1.73646,0.212786,1.11066,-2.14167,0.248465,1.03522,-2.12531,0.090772,1.05856,-2.08074,0.127128,0.987584,-2.06803,0.239171,0.879317,-2.07792,0.225421,0.688062,-2.05662,0.118335,0.721972,-2.06771,0.098984,0.656418,-2.05061,0.057956,0.432164,-1.99646,-0.040931,0.36331,-1.93723,0.242442,0.377132,-1.98301,0.187787,0.340944,-2.04563,0.114262,1.26794,-1.84506,0.091293,1.30571,-2.0557},
/*491*/{0.247118,1.84901,-1.74301,0.300718,1.78889,-1.9032,0.139001,1.71758,-1.6439,0.055335,1.62641,-1.58216,0.18517,1.55929,-1.65285,0.260271,1.51555,-1.67686,0.197934,1.85574,-2.14106,0.294938,1.78992,-1.98094,0.073989,1.8357,-2.03246,0.0209,1.6454,-2.20531,0.038368,1.55099,-2.238,0.208357,1.47941,-2.18418,0.259317,1.43835,-2.19504,0.189963,1.35025,-1.81963,-0.004912,1.42498,-1.86197,-0.007551,1.45674,-1.91669,-0.014734,1.43613,-1.97386,0.173411,1.3992,-2.08056,0.250897,1.07366,-1.75348,0.128103,1.01964,-1.77606,0.161234,0.942042,-1.79621,0.2516,0.870579,-1.77538,0.100273,0.831525,-1.76119,0.025054,0.8286,-1.76804,0.096485,0.941315,-1.78764,0.025193,0.939603,-1.78725,-0.203662,0.889099,-1.77946,-0.27118,0.976481,-1.85886,-0.249805,0.722848,-1.80655,-0.299302,0.753815,-1.73698,0.206639,1.10734,-2.14222,0.240632,1.03108,-2.12625,0.083261,1.05809,-2.08108,0.117894,0.986523,-2.06938,0.227174,0.875121,-2.0795,0.21135,0.682307,-2.056,0.105174,0.719612,-2.06848,0.086131,0.652039,-2.05058,0.040248,0.434217,-1.99737,-0.062659,0.363344,-1.93855,0.221071,0.376188,-1.98297,0.166006,0.340385,-2.04657,0.111355,1.26774,-1.84465,0.089676,1.30494,-2.05554},
/*492*/{0.245319,1.84918,-1.74247,0.298802,1.78809,-1.90229,0.133923,1.72034,-1.64448,0.046629,1.62996,-1.58428,0.177674,1.56104,-1.65198,0.25183,1.51627,-1.67432,0.200477,1.85558,-2.1404,0.295148,1.78973,-1.97951,0.074968,1.8358,-2.03288,0.026592,1.64347,-2.20595,0.047061,1.5485,-2.23869,0.218504,1.47999,-2.18206,0.27023,1.44123,-2.19262,0.188863,1.35053,-1.81849,-0.00577,1.42518,-1.86258,-0.008717,1.45711,-1.91703,-0.015424,1.437,-1.97492,0.173411,1.3992,-2.08056,0.257423,1.07805,-1.75343,0.137486,1.02163,-1.77535,0.172,0.945045,-1.79487,0.265683,0.873582,-1.77565,0.114021,0.832491,-1.76119,0.039089,0.828383,-1.76879,0.10702,0.942033,-1.78654,0.034624,0.938862,-1.78639,-0.18596,0.877874,-1.78071,-0.264131,0.966539,-1.85813,-0.22944,0.714323,-1.80756,-0.277903,0.742934,-1.73658,0.200472,1.10515,-2.1427,0.232781,1.02727,-2.12679,0.075321,1.05836,-2.08271,0.109797,0.983478,-2.06924,0.21453,0.870235,-2.08017,0.197039,0.677019,-2.05509,0.091371,0.717016,-2.0695,0.071128,0.649612,-2.0514,0.018261,0.434154,-1.99956,-0.083522,0.364424,-1.93847,0.199702,0.375692,-1.98446,0.145378,0.340665,-2.04711,0.110024,1.26814,-1.84445,0.089472,1.30514,-2.05548},
/*493*/{0.242546,1.85037,-1.74123,0.297618,1.78835,-1.90044,0.127784,1.7239,-1.64516,0.038191,1.6344,-1.58642,0.170255,1.56354,-1.65117,0.244437,1.51706,-1.67262,0.203247,1.85596,-2.13971,0.29545,1.78988,-1.97853,0.075412,1.83595,-2.03362,0.033969,1.64157,-2.20812,0.055725,1.54744,-2.2393,0.228246,1.48373,-2.18005,0.281045,1.44591,-2.18975,0.187032,1.35112,-1.81729,-0.007663,1.42602,-1.863,-0.009411,1.45717,-1.9175,-0.015927,1.43713,-1.97573,0.171072,1.39757,-2.0813,0.263758,1.08605,-1.75363,0.146112,1.02319,-1.77376,0.181917,0.948791,-1.79375,0.277279,0.87903,-1.77679,0.127097,0.833214,-1.76298,0.052337,0.827566,-1.77,0.116324,0.943387,-1.78693,0.045359,0.937981,-1.78543,-0.173871,0.868928,-1.78157,-0.256936,0.955619,-1.8575,-0.20848,0.705619,-1.80963,-0.257945,0.730735,-1.73672,0.194464,1.10281,-2.14259,0.224504,1.02343,-2.12734,0.068582,1.05894,-2.08343,0.100017,0.983648,-2.0708,0.201335,0.865514,-2.08075,0.181864,0.67249,-2.05473,0.077029,0.715017,-2.07066,0.055836,0.646904,-2.05157,-0.002798,0.434645,-2.00034,-0.104622,0.366456,-1.94007,0.177651,0.375549,-1.98499,0.123767,0.340285,-2.04777,0.107907,1.26874,-1.84336,0.087557,1.3039,-2.05472},
/*494*/{0.240161,1.8523,-1.74112,0.297203,1.78881,-1.89932,0.121801,1.72767,-1.64589,0.030768,1.63952,-1.58846,0.162514,1.56672,-1.65102,0.235906,1.51934,-1.67127,0.204663,1.85711,-2.13969,0.295343,1.7908,-1.97673,0.076883,1.83669,-2.03473,0.0393,1.64091,-2.2089,0.063448,1.54638,-2.24005,0.238764,1.48794,-2.17779,0.291557,1.4516,-2.18638,0.184893,1.35223,-1.81608,-0.008144,1.42624,-1.86418,-0.009943,1.45761,-1.91824,-0.015679,1.43778,-1.97724,0.172861,1.39748,-2.08087,0.269854,1.0918,-1.75284,0.153894,1.02502,-1.77289,0.192057,0.952661,-1.7932,0.289836,0.885174,-1.77654,0.14039,0.835152,-1.76369,0.064998,0.826553,-1.77057,0.126658,0.945031,-1.78447,0.054638,0.937191,-1.78547,-0.160351,0.859792,-1.78291,-0.250022,0.943281,-1.85761,-0.186405,0.696818,-1.81029,-0.23604,0.718664,-1.73697,0.188327,1.10053,-2.14277,0.215502,1.02018,-2.12803,0.061878,1.05913,-2.08456,0.090875,0.983511,-2.07201,0.187481,0.861145,-2.08176,0.165224,0.670529,-2.0541,0.061932,0.713284,-2.07095,0.037085,0.650659,-2.05434,-0.023696,0.436566,-2.00047,-0.125912,0.366773,-1.94347,0.156759,0.375328,-1.98578,0.101962,0.340207,-2.04823,0.106698,1.26943,-1.84287,0.088753,1.30406,-2.05454},
/*495*/{0.236811,1.85444,-1.74032,0.296422,1.79006,-1.8975,0.115215,1.7324,-1.64638,0.021665,1.64407,-1.59058,0.154983,1.56995,-1.64972,0.228447,1.52243,-1.66935,0.207384,1.85889,-2.139,0.29573,1.79159,-1.97506,0.077453,1.83708,-2.03591,0.044696,1.63947,-2.20938,0.071694,1.54659,-2.2405,0.247301,1.49337,-2.17513,0.301126,1.45826,-2.18246,0.184419,1.35425,-1.81579,-0.009214,1.42741,-1.8663,-0.010116,1.45812,-1.91905,-0.016889,1.43923,-1.9795,0.172896,1.39718,-2.08072,0.275618,1.09938,-1.75239,0.160008,1.02859,-1.77193,0.201289,0.957572,-1.79346,0.301611,0.890938,-1.77644,0.153642,0.836922,-1.76427,0.07866,0.825295,-1.77235,0.135046,0.945886,-1.78347,0.064793,0.934901,-1.78346,-0.149701,0.852301,-1.7852,-0.242516,0.929692,-1.85714,-0.164839,0.687084,-1.8112,-0.214198,0.706177,-1.73714,0.181373,1.09802,-2.14304,0.206889,1.01734,-2.12792,0.053781,1.06076,-2.08585,0.080443,0.983283,-2.07326,0.172621,0.856671,-2.0819,0.150088,0.668658,-2.05407,0.046284,0.712898,-2.07088,0.020917,0.650671,-2.05483,-0.048683,0.435132,-2.00019,-0.144699,0.367091,-1.94005,0.135361,0.375469,-1.98684,0.081365,0.340783,-2.04888,0.105836,1.2711,-1.84255,0.088873,1.30414,-2.05456},
/*496*/{0.233922,1.85711,-1.73924,0.296773,1.79221,-1.89515,0.109945,1.73675,-1.64694,0.013882,1.64993,-1.59236,0.147853,1.57508,-1.64985,0.221012,1.52672,-1.66795,0.209674,1.86132,-2.13795,0.296181,1.79404,-1.97373,0.078469,1.83748,-2.03681,0.048717,1.63943,-2.2095,0.079511,1.54714,-2.24082,0.255495,1.49903,-2.17222,0.310832,1.46658,-2.17861,0.18304,1.3554,-1.81545,-0.010048,1.42779,-1.86765,-0.011258,1.45979,-1.92092,-0.016695,1.44007,-1.98017,0.172868,1.39699,-2.08079,0.280913,1.10685,-1.75216,0.168703,1.03483,-1.77194,0.211525,0.963157,-1.79234,0.312599,0.897782,-1.77699,0.166036,0.837929,-1.76527,0.091791,0.823963,-1.77139,0.143714,0.947591,-1.78243,0.072942,0.933169,-1.78283,-0.135739,0.841353,-1.78535,-0.234327,0.915375,-1.85841,-0.143963,0.67712,-1.8123,-0.193438,0.693524,-1.73785,0.174574,1.09662,-2.14295,0.197324,1.015,-2.12752,0.045156,1.06267,-2.08617,0.070763,0.983344,-2.07312,0.156702,0.853634,-2.08185,0.133219,0.667614,-2.05294,0.030399,0.713725,-2.07134,0.004517,0.649631,-2.05497,-0.069131,0.435711,-2.00109,-0.167414,0.368022,-1.94456,0.11394,0.37606,-1.98657,0.059253,0.340517,-2.04929,0.104403,1.27216,-1.84245,0.088723,1.30437,-2.05468},
/*497*/{0.23028,1.86003,-1.73865,0.295463,1.79452,-1.89365,0.103748,1.74137,-1.64783,0.00707,1.65542,-1.59463,0.140704,1.57998,-1.64908,0.212702,1.53066,-1.66689,0.212126,1.86384,-2.13663,0.295751,1.79544,-1.97144,0.078521,1.83935,-2.0377,0.055512,1.6402,-2.20916,0.086168,1.54831,-2.24111,0.262817,1.50517,-2.16864,0.319463,1.47465,-2.17426,0.183102,1.3585,-1.8143,-0.010307,1.42955,-1.87012,-0.010082,1.46028,-1.92128,-0.016427,1.44096,-1.98198,0.173319,1.39713,-2.0807,0.285644,1.11346,-1.75198,0.174457,1.03851,-1.77224,0.220458,0.968271,-1.79201,0.323104,0.905028,-1.77689,0.178947,0.839447,-1.76582,0.106273,0.822986,-1.77276,0.151804,0.947491,-1.78165,0.081808,0.930541,-1.78249,-0.122863,0.830237,-1.78619,-0.225713,0.899293,-1.85916,-0.121473,0.666757,-1.81372,-0.172569,0.680817,-1.73906,0.167403,1.09497,-2.14175,0.188713,1.0131,-2.12645,0.036732,1.06456,-2.08718,0.059253,0.985837,-2.0745,0.140744,0.852649,-2.08177,0.115198,0.667711,-2.05277,0.013396,0.714559,-2.07095,-0.012691,0.651897,-2.05479,-0.090485,0.437657,-2.00072,-0.189129,0.371913,-1.94247,0.092887,0.375723,-1.9875,0.038441,0.340731,-2.04967,0.104782,1.27456,-1.84138,0.090039,1.30443,-2.05402},
/*498*/{0.22683,1.86303,-1.73788,0.294912,1.79774,-1.89172,0.098209,1.74616,-1.64843,-0.000565,1.661,-1.59699,0.132373,1.58533,-1.65001,0.205571,1.53505,-1.66567,0.213896,1.86747,-2.13583,0.295934,1.7986,-1.96917,0.079939,1.84026,-2.03935,0.059516,1.64083,-2.20937,0.09387,1.54985,-2.24141,0.269053,1.51221,-2.16576,0.328383,1.48369,-2.16946,0.182085,1.36023,-1.81363,-0.010642,1.4309,-1.8711,-0.009982,1.46306,-1.92271,-0.015894,1.44343,-1.98365,0.173451,1.39712,-2.08072,0.290288,1.12214,-1.75209,0.180649,1.04322,-1.77271,0.228379,0.973524,-1.79275,0.333246,0.913287,-1.77708,0.191103,0.841271,-1.76605,0.118243,0.820407,-1.77531,0.159,0.94786,-1.78126,0.089715,0.927473,-1.78235,-0.112563,0.822413,-1.78711,-0.216422,0.882801,-1.86054,-0.098884,0.656963,-1.81381,-0.151159,0.667727,-1.73936,0.160291,1.0942,-2.1413,0.17866,1.01159,-2.126,0.028217,1.06662,-2.08744,0.049043,0.987612,-2.07451,0.124793,0.853338,-2.08172,0.098203,0.666554,-2.05186,-0.003925,0.716604,-2.0699,-0.031911,0.653536,-2.05447,-0.106892,0.441658,-2.00183,-0.210252,0.373772,-1.9442,0.072112,0.375653,-1.98753,0.017261,0.340308,-2.04994,0.103205,1.27643,-1.84112,0.089835,1.30531,-2.05399},
/*499*/{0.22303,1.86655,-1.737,0.293282,1.8001,-1.88913,0.092154,1.75139,-1.64911,-0.008123,1.66622,-1.59888,0.126126,1.5901,-1.64957,0.198447,1.54009,-1.6637,0.215252,1.87033,-2.13505,0.296245,1.80146,-1.96761,0.081432,1.84152,-2.04033,0.063135,1.64221,-2.20872,0.100883,1.55233,-2.2424,0.277108,1.52122,-2.1624,0.335133,1.49431,-2.16521,0.181706,1.36321,-1.81322,-0.01068,1.43309,-1.87327,-0.008672,1.46484,-1.92393,-0.015378,1.44536,-1.98557,0.173744,1.39736,-2.08061,0.295355,1.12884,-1.74973,0.18705,1.04529,-1.77171,0.236911,0.980209,-1.79284,0.343189,0.921743,-1.77742,0.203641,0.842949,-1.76631,0.131655,0.819539,-1.77427,0.166046,0.948279,-1.78059,0.097785,0.92486,-1.78194,-0.097695,0.808283,-1.78917,-0.206442,0.865504,-1.86223,-0.077623,0.646109,-1.81472,-0.130073,0.654319,-1.7411,0.153353,1.0934,-2.13978,0.169059,1.01019,-2.12406,0.01958,1.07004,-2.08776,0.03712,0.989737,-2.07515,0.108414,0.853898,-2.08097,0.079092,0.668086,-2.05187,-0.021335,0.718229,-2.06926,-0.049409,0.655395,-2.05337,-0.127809,0.443431,-2.00219,-0.231067,0.376148,-1.94403,0.050743,0.375336,-1.98769,-0.004484,0.339973,-2.04971,0.1028,1.27906,-1.84055,0.090511,1.30596,-2.05375},
/*500*/{0.218449,1.86983,-1.73661,0.29401,1.80424,-1.88705,0.086691,1.75635,-1.64992,-0.014853,1.67262,-1.601,0.119239,1.59621,-1.64996,0.191092,1.54602,-1.66353,0.217655,1.87422,-2.13376,0.296602,1.80465,-1.96565,0.08145,1.84379,-2.04081,0.066849,1.6444,-2.20844,0.107197,1.55515,-2.24288,0.282644,1.52873,-2.15879,0.342395,1.50459,-2.16107,0.182373,1.36633,-1.81311,-0.010619,1.436,-1.87509,-0.009662,1.46738,-1.92528,-0.013787,1.44736,-1.98665,0.174122,1.39767,-2.08054,0.298725,1.13573,-1.75012,0.194818,1.05221,-1.77277,0.243747,0.986226,-1.79369,0.353195,0.931408,-1.778,0.215411,0.84406,-1.76711,0.144344,0.817781,-1.77568,0.173343,0.947599,-1.77925,0.105332,0.921024,-1.78158,-0.082648,0.795493,-1.79152,-0.19513,0.847511,-1.86482,-0.055312,0.635634,-1.81522,-0.108141,0.641114,-1.74237,0.146175,1.09249,-2.13815,0.159633,1.0096,-2.1224,0.010745,1.07322,-2.08835,0.026968,0.992655,-2.07478,0.093255,0.855676,-2.08012,0.060577,0.669525,-2.05124,-0.03983,0.721183,-2.06857,-0.068167,0.65767,-2.05261,-0.148177,0.447263,-2.00162,-0.250952,0.380405,-1.9443,0.028881,0.375018,-1.98706,-0.025921,0.340993,-2.04996,0.102673,1.28218,-1.83978,0.091005,1.30685,-2.05327},
/*501*/{0.215257,1.87384,-1.73628,0.293555,1.80737,-1.88563,0.081695,1.76167,-1.6505,-0.023182,1.67884,-1.60341,0.112338,1.6023,-1.6513,0.184469,1.55145,-1.6625,0.219475,1.87857,-2.13273,0.297762,1.80815,-1.96316,0.082736,1.84618,-2.0413,0.070578,1.64612,-2.20777,0.114648,1.55836,-2.2436,0.288611,1.53745,-2.15533,0.34923,1.51477,-2.15628,0.18182,1.37015,-1.81322,-0.010453,1.43914,-1.87619,-0.009006,1.46961,-1.92626,-0.013103,1.45046,-1.98782,0.174993,1.39884,-2.0808,0.303083,1.14351,-1.75026,0.199601,1.05695,-1.77359,0.251066,0.992129,-1.79528,0.36176,0.941274,-1.77799,0.22641,0.845879,-1.76717,0.157633,0.816097,-1.77555,0.179575,0.947562,-1.78032,0.113556,0.917562,-1.78136,-0.069399,0.784124,-1.79058,-0.183987,0.828473,-1.8681,-0.031835,0.625663,-1.81558,-0.086147,0.627884,-1.74343,0.139484,1.09231,-2.13706,0.150023,1.00909,-2.12003,0.004092,1.07703,-2.08801,0.015972,0.995766,-2.07369,0.078082,0.857073,-2.07809,0.043209,0.671911,-2.05049,-0.056256,0.724352,-2.06701,-0.085304,0.661728,-2.05227,-0.163523,0.450803,-2.00201,-0.271398,0.384439,-1.94452,0.008152,0.375584,-1.98713,-0.047349,0.34078,-2.05041,0.102457,1.28555,-1.83929,0.091555,1.3086,-2.05301},
/*502*/{0.21078,1.87795,-1.73557,0.29133,1.81196,-1.88261,0.07616,1.76691,-1.65133,-0.029207,1.68512,-1.60541,0.105991,1.60865,-1.65139,0.17786,1.55718,-1.66187,0.22086,1.88266,-2.13173,0.298267,1.81223,-1.96084,0.084248,1.8487,-2.04267,0.075943,1.64906,-2.20744,0.120904,1.56205,-2.24396,0.293997,1.54648,-2.15178,0.35451,1.52566,-2.15156,0.182328,1.37387,-1.81322,-0.010063,1.44245,-1.8774,-0.006354,1.47285,-1.92859,-0.011054,1.45284,-1.98923,0.176093,1.39936,-2.08104,0.305599,1.15037,-1.74984,0.203911,1.06258,-1.77436,0.256777,0.998004,-1.79496,0.369452,0.950493,-1.77803,0.237449,0.847789,-1.76696,0.170153,0.813317,-1.77633,0.185612,0.946307,-1.77894,0.121611,0.912357,-1.78239,-0.056602,0.771656,-1.7923,-0.171904,0.81006,-1.87052,-0.009207,0.615261,-1.81623,-0.064307,0.614824,-1.74461,0.13297,1.0928,-2.13447,0.141548,1.00916,-2.11762,-0.004643,1.08133,-2.08861,0.006118,0.999847,-2.0748,0.064003,0.859078,-2.07602,0.02635,0.67536,-2.04979,-0.072781,0.727836,-2.06602,-0.102904,0.664993,-2.05144,-0.187728,0.45438,-2.00165,-0.291036,0.392013,-1.94451,-0.013092,0.375405,-1.98712,-0.068854,0.34051,-2.04838,0.102733,1.28915,-1.83856,0.092709,1.30975,-2.05258},
/*503*/{0.207515,1.88285,-1.73503,0.29255,1.8144,-1.88078,0.071103,1.7723,-1.65234,-0.034491,1.69195,-1.60721,0.099662,1.61457,-1.65193,0.170842,1.56334,-1.66181,0.223945,1.8878,-2.13069,0.298615,1.81635,-1.95961,0.085307,1.85223,-2.04368,0.080348,1.65232,-2.20816,0.127067,1.56634,-2.24484,0.298969,1.55615,-2.14818,0.360202,1.53624,-2.14676,0.182863,1.3776,-1.81373,-0.00861,1.44619,-1.87913,-0.006491,1.47518,-1.92983,-0.009856,1.45629,-1.9912,0.17796,1.40021,-2.08087,0.307453,1.15879,-1.75012,0.207527,1.0685,-1.77517,0.261695,1.0043,-1.7965,0.376248,0.960094,-1.77887,0.248085,0.849432,-1.76709,0.182719,0.811291,-1.77624,0.189952,0.945651,-1.7801,0.12907,0.908839,-1.78289,-0.041221,0.758337,-1.79337,-0.158445,0.790335,-1.87269,0.014428,0.605731,-1.81652,-0.042114,0.601992,-1.74535,0.126702,1.09417,-2.13298,0.132323,1.00963,-2.11504,-0.010624,1.08578,-2.08849,-0.003688,1.00438,-2.07468,0.051064,0.861711,-2.07416,0.009622,0.678362,-2.04881,-0.089645,0.732004,-2.06536,-0.119586,0.669712,-2.05107,-0.201681,0.458785,-2.00149,-0.309836,0.397154,-1.94446,-0.034482,0.3756,-1.98669,-0.090224,0.340831,-2.04841,0.102961,1.2928,-1.8382,0.093868,1.31146,-2.05243},
/*504*/{0.204186,1.88624,-1.73432,0.291273,1.82088,-1.87961,0.066879,1.77813,-1.65277,-0.041705,1.69874,-1.6103,0.09318,1.62112,-1.65303,0.16512,1.56914,-1.66103,0.224997,1.89213,-2.12889,0.298933,1.82084,-1.95714,0.086443,1.85554,-2.04478,0.083609,1.65666,-2.20922,0.133921,1.57073,-2.24506,0.30281,1.56414,-2.14452,0.364475,1.54767,-2.14221,0.18499,1.38178,-1.81358,-0.006974,1.44953,-1.88016,-0.005131,1.47865,-1.93123,-0.007926,1.45983,-1.99205,0.178808,1.40079,-2.08095,0.310898,1.16514,-1.74942,0.213783,1.07379,-1.77491,0.268984,1.01178,-1.79626,0.38325,0.969983,-1.77906,0.258286,0.850901,-1.7677,0.19406,0.809592,-1.77679,0.195247,0.943921,-1.78052,0.13603,0.903862,-1.78409,-0.022742,0.745177,-1.79559,-0.143858,0.770359,-1.87473,0.038473,0.596385,-1.81681,-0.018313,0.589547,-1.74628,0.120329,1.09549,-2.12998,0.124185,1.01076,-2.11188,-0.018816,1.09069,-2.08888,-0.012956,1.0099,-2.07445,0.037207,0.864771,-2.07175,-0.007313,0.682596,-2.048,-0.105389,0.737491,-2.06478,-0.136177,0.674579,-2.05061,-0.220957,0.464793,-2.00275,-0.328052,0.406019,-1.9453,-0.055086,0.376508,-1.9864,-0.110902,0.341166,-2.04812,0.1041,1.29692,-1.83719,0.094821,1.31299,-2.05163},
/*505*/{0.201488,1.89064,-1.73384,0.291572,1.82447,-1.87707,0.062321,1.78352,-1.65422,-0.048433,1.70513,-1.6128,0.087581,1.62717,-1.65407,0.159133,1.57507,-1.66176,0.22657,1.89688,-2.12788,0.299543,1.82529,-1.95505,0.087677,1.8589,-2.04506,0.089181,1.66259,-2.21013,0.140246,1.5754,-2.24571,0.307069,1.57268,-2.14087,0.369147,1.55803,-2.13754,0.185031,1.38547,-1.81443,-0.006107,1.45436,-1.88054,-0.003512,1.48282,-1.93322,-0.005371,1.46313,-1.9929,0.180156,1.4016,-2.08175,0.313551,1.17192,-1.7492,0.217903,1.07955,-1.77498,0.272462,1.01784,-1.79763,0.387975,0.978344,-1.77878,0.26771,0.852132,-1.76762,0.206189,0.807457,-1.77739,0.200856,0.942082,-1.78057,0.141684,0.899147,-1.78642,-0.010787,0.732239,-1.79694,-0.12981,0.750568,-1.8788,0.063068,0.587546,-1.81776,0.006725,0.577133,-1.7475,0.11469,1.09631,-2.12785,0.116135,1.01254,-2.10923,-0.024785,1.09637,-2.08841,-0.0218,1.01471,-2.07448,0.024223,0.86948,-2.07025,-0.024051,0.686912,-2.04741,-0.121765,0.742821,-2.06397,-0.152757,0.681163,-2.05053,-0.242836,0.468728,-2.00177,-0.347878,0.413896,-1.94579,-0.076078,0.376333,-1.98561,-0.132102,0.341561,-2.04788,0.103878,1.30098,-1.83674,0.095078,1.31497,-2.05133},
/*506*/{0.198335,1.89477,-1.73402,0.290931,1.83037,-1.87583,0.058213,1.78946,-1.65521,-0.05329,1.71214,-1.61553,0.083326,1.63403,-1.65558,0.153392,1.58126,-1.66125,0.229361,1.90251,-2.12638,0.299409,1.82998,-1.95359,0.088975,1.86275,-2.04614,0.093015,1.66609,-2.21124,0.146525,1.58072,-2.24625,0.310011,1.58151,-2.13779,0.372325,1.5687,-2.13332,0.187226,1.38994,-1.81515,-0.004319,1.45913,-1.88094,-0.001468,1.48606,-1.93421,-0.003125,1.46686,-1.9946,0.182182,1.40219,-2.08198,0.315711,1.17935,-1.74925,0.219412,1.08219,-1.77372,0.276989,1.02434,-1.79789,0.392875,0.986952,-1.77873,0.277099,0.853485,-1.76705,0.218123,0.806227,-1.77778,0.205876,0.939699,-1.78201,0.150066,0.893133,-1.78718,0.005659,0.719562,-1.79946,-0.114199,0.730958,-1.88065,0.088713,0.579742,-1.818,0.031326,0.565277,-1.74865,0.109905,1.09759,-2.12454,0.1075,1.01376,-2.10635,-0.031034,1.10159,-2.08809,-0.030081,1.02025,-2.07378,0.009928,0.874424,-2.0689,-0.039365,0.692376,-2.04716,-0.136925,0.749309,-2.06318,-0.170623,0.686636,-2.04864,-0.264727,0.473252,-2.00166,-0.365617,0.42265,-1.94543,-0.096935,0.376932,-1.9863,-0.154191,0.342387,-2.0471,0.105321,1.30553,-1.8358,0.096341,1.31675,-2.05055},
/*507*/{0.195724,1.899,-1.73379,0.290191,1.83348,-1.87303,0.054366,1.79474,-1.6567,-0.058904,1.71895,-1.61835,0.077422,1.63971,-1.65616,0.147586,1.58664,-1.66124,0.231349,1.90765,-2.12503,0.301267,1.83441,-1.95052,0.091252,1.86556,-2.04718,0.097882,1.67245,-2.21268,0.152399,1.58587,-2.24703,0.313502,1.59194,-2.1347,0.376852,1.57783,-2.12845,0.188337,1.39408,-1.8158,-0.002479,1.46336,-1.8826,0.000516,1.4897,-1.93655,-0.000738,1.46985,-1.99485,0.183195,1.40309,-2.0822,0.318494,1.18656,-1.74924,0.224825,1.09034,-1.77265,0.280808,1.03104,-1.79767,0.395655,0.993664,-1.77877,0.286423,0.854678,-1.76764,0.231666,0.804941,-1.77821,0.210638,0.936981,-1.78345,0.158541,0.888395,-1.78855,0.022074,0.706311,-1.8006,-0.097637,0.709598,-1.88235,0.112153,0.571111,-1.81958,0.056728,0.554003,-1.7497,0.104135,1.09971,-2.12226,0.099621,1.01586,-2.10385,-0.03716,1.10791,-2.08822,-0.038762,1.02642,-2.07403,-0.00439,0.879445,-2.06843,-0.054301,0.697516,-2.0472,-0.151521,0.755573,-2.06275,-0.18505,0.695879,-2.04912,-0.277275,0.481089,-2.00482,-0.383811,0.432348,-1.94539,-0.117467,0.378576,-1.98613,-0.175458,0.343315,-2.04679,0.10614,1.30982,-1.83516,0.097249,1.31845,-2.05003},
/*508*/{0.193478,1.90357,-1.73364,0.290056,1.84009,-1.87245,0.049878,1.80013,-1.65855,-0.062882,1.72573,-1.62098,0.072419,1.64587,-1.65738,0.143307,1.59255,-1.66147,0.233492,1.91297,-2.12388,0.301224,1.83927,-1.94877,0.092197,1.86906,-2.04766,0.099827,1.67742,-2.21427,0.157796,1.59109,-2.24817,0.316149,1.59895,-2.13102,0.378635,1.5878,-2.12358,0.189302,1.39712,-1.81726,-0.000354,1.46833,-1.88274,0.002007,1.49431,-1.93887,0.002262,1.47332,-1.99587,0.185339,1.40379,-2.08247,0.319579,1.19364,-1.74983,0.229614,1.09719,-1.77369,0.28404,1.03767,-1.79714,0.397375,1.00027,-1.77919,0.295452,0.85483,-1.76809,0.239547,0.802317,-1.77722,0.215741,0.93397,-1.78371,0.164811,0.881918,-1.79015,0.038592,0.692661,-1.80206,-0.079065,0.688345,-1.88529,0.137268,0.563054,-1.81954,0.082312,0.543083,-1.75101,0.09873,1.10177,-2.11993,0.092535,1.01842,-2.10089,-0.041493,1.11409,-2.0878,-0.046678,1.03207,-2.07304,-0.018463,0.884537,-2.06769,-0.070635,0.703997,-2.04699,-0.166481,0.762531,-2.06255,-0.200193,0.703236,-2.04949,-0.290024,0.489615,-2.0041,-0.399017,0.442676,-1.94511,-0.14176,0.37632,-1.98458,-0.196426,0.343624,-2.04705,0.106355,1.31374,-1.83499,0.09784,1.32061,-2.04995},
/*509*/{0.190989,1.9079,-1.73344,0.289533,1.84391,-1.86979,0.046857,1.80642,-1.66022,-0.06783,1.73213,-1.62459,0.067405,1.65148,-1.6582,0.138342,1.5983,-1.66106,0.235707,1.91801,-2.12217,0.301518,1.8445,-1.94712,0.09387,1.8724,-2.04832,0.104133,1.68241,-2.21522,0.163632,1.59618,-2.24866,0.318581,1.60702,-2.12766,0.381177,1.59654,-2.12071,0.19106,1.40121,-1.81719,0.001584,1.47253,-1.88288,0.004819,1.49709,-1.94013,0.004605,1.47704,-1.99665,0.187419,1.40454,-2.08234,0.320687,1.20022,-1.74965,0.231418,1.10287,-1.77365,0.287177,1.04367,-1.79709,0.399168,1.00644,-1.78017,0.303159,0.855381,-1.76779,0.253765,0.799745,-1.77863,0.219947,0.929867,-1.78597,0.172968,0.876178,-1.79053,0.058897,0.680446,-1.80395,-0.0611,0.668247,-1.8862,0.163252,0.555689,-1.82016,0.108029,0.532387,-1.75217,0.093554,1.1036,-2.11749,0.084179,1.02152,-2.0985,-0.046516,1.12155,-2.08768,-0.053629,1.03972,-2.07252,-0.031192,0.890371,-2.06715,-0.085263,0.710695,-2.04747,-0.180372,0.77037,-2.06182,-0.215444,0.710719,-2.04931,-0.306522,0.496061,-2.00447,-0.416548,0.454647,-1.9445,-0.163179,0.376734,-1.98388,-0.219198,0.343699,-2.04576,0.107714,1.31789,-1.83389,0.099358,1.32242,-2.04892},
/*510*/{0.189193,1.91235,-1.73332,0.28845,1.84869,-1.86852,0.043207,1.81193,-1.66206,-0.070945,1.73872,-1.62738,0.06334,1.65719,-1.65996,0.134133,1.60347,-1.66168,0.236656,1.92254,-2.12077,0.301766,1.84892,-1.94548,0.095093,1.87656,-2.04907,0.107674,1.68757,-2.21653,0.168878,1.60164,-2.24914,0.320395,1.61518,-2.12496,0.383738,1.606,-2.11663,0.193698,1.406,-1.818,0.003214,1.47801,-1.88354,0.0064,1.50121,-1.94201,0.006402,1.48029,-1.99731,0.189595,1.40515,-2.0819,0.322662,1.20642,-1.74947,0.232037,1.1088,-1.77376,0.288901,1.04955,-1.79707,0.400647,1.01258,-1.78034,0.312026,0.855474,-1.76859,0.266323,0.798284,-1.77974,0.224124,0.925481,-1.78729,0.180807,0.868996,-1.7927,0.075639,0.665661,-1.80494,-0.041122,0.648161,-1.88867,0.188302,0.548948,-1.82163,0.135582,0.522891,-1.7531,0.089162,1.10614,-2.11537,0.076979,1.02422,-2.09593,-0.051581,1.12772,-2.08723,-0.061502,1.047,-2.0724,-0.043254,0.8967,-2.06679,-0.099556,0.717969,-2.04838,-0.194636,0.777906,-2.0618,-0.228659,0.718816,-2.04956,-0.320266,0.502257,-2.00622,-0.430954,0.466183,-1.94428,-0.185068,0.377485,-1.98377,-0.242039,0.344403,-2.04569,0.109222,1.32323,-1.83248,0.100511,1.32425,-2.04754},
/*511*/{0.187345,1.91574,-1.7329,0.28769,1.85408,-1.86797,0.04048,1.81707,-1.66369,-0.075862,1.7446,-1.63142,0.058851,1.66188,-1.66081,0.129888,1.60875,-1.66117,0.238074,1.92715,-2.11887,0.301002,1.85343,-1.94358,0.095897,1.87979,-2.04985,0.112615,1.69245,-2.21754,0.174172,1.60704,-2.24963,0.322085,1.62235,-2.1221,0.385886,1.61501,-2.11379,0.194737,1.40982,-1.81868,0.004764,1.48297,-1.88404,0.008047,1.50505,-1.94294,0.009731,1.48312,-1.99768,0.191548,1.4062,-2.08143,0.324378,1.21122,-1.74908,0.236165,1.11442,-1.77218,0.290634,1.0557,-1.79673,0.400542,1.01741,-1.78108,0.320575,0.854888,-1.76881,0.27595,0.794528,-1.77988,0.229992,0.920473,-1.78824,0.188484,0.861414,-1.79455,0.096367,0.65352,-1.80634,-0.021139,0.628283,-1.88968,0.214319,0.543072,-1.82359,0.162925,0.5127,-1.75511,0.084502,1.10876,-2.114,0.068847,1.02718,-2.09402,-0.055114,1.13608,-2.08657,-0.068586,1.05478,-2.07179,-0.05492,0.903842,-2.06695,-0.112425,0.725382,-2.04939,-0.208893,0.78522,-2.06163,-0.242702,0.724711,-2.04841,-0.335823,0.509848,-2.00707,-0.443784,0.478598,-1.94201,-0.20637,0.377662,-1.98371,-0.263386,0.345377,-2.04515,0.110001,1.32749,-1.83148,0.101667,1.32634,-2.04655},
/*512*/{0.185534,1.91964,-1.7332,0.287232,1.85759,-1.86601,0.036912,1.82269,-1.66581,-0.080752,1.75042,-1.63443,0.055665,1.66681,-1.66184,0.125975,1.61318,-1.66127,0.24051,1.932,-2.11802,0.301487,1.85801,-1.94201,0.097521,1.88293,-2.0506,0.116986,1.69718,-2.21944,0.178988,1.61165,-2.25046,0.323732,1.62953,-2.11915,0.38805,1.62292,-2.11014,0.195913,1.4137,-1.81899,0.007306,1.48666,-1.88393,0.010278,1.50942,-1.94468,0.01216,1.48706,-1.99828,0.19373,1.40709,-2.08091,0.326764,1.21599,-1.74894,0.236205,1.11897,-1.77097,0.29168,1.06036,-1.79527,0.399659,1.0218,-1.78153,0.327828,0.854191,-1.76903,0.286456,0.791372,-1.78013,0.234658,0.914255,-1.78959,0.195167,0.853956,-1.79606,0.113321,0.638686,-1.80714,-0.000344,0.608789,-1.89124,0.239782,0.537084,-1.82508,0.189082,0.502966,-1.75738,0.080092,1.11179,-2.11243,0.061245,1.03021,-2.09191,-0.059295,1.14185,-2.0861,-0.074774,1.06298,-2.07161,-0.066723,0.911254,-2.06699,-0.125208,0.733719,-2.05091,-0.221673,0.793857,-2.06186,-0.256207,0.732033,-2.04817,-0.350713,0.516384,-2.00869,-0.45552,0.492487,-1.94178,-0.227721,0.377776,-1.98461,-0.28553,0.34648,-2.04497,0.110827,1.33165,-1.83066,0.102868,1.32862,-2.04573},
/*513*/{0.18358,1.92293,-1.73361,0.286032,1.86225,-1.86533,0.034063,1.8276,-1.66776,-0.084046,1.75578,-1.63839,0.052061,1.67122,-1.66272,0.122697,1.61761,-1.6613,0.241639,1.93635,-2.11638,0.301217,1.86202,-1.94067,0.098338,1.88628,-2.05178,0.121426,1.70123,-2.22098,0.183637,1.61703,-2.25129,0.325911,1.6363,-2.11654,0.389558,1.6306,-2.10635,0.197864,1.41809,-1.81858,0.009782,1.49136,-1.88385,0.013765,1.51227,-1.94592,0.015478,1.48942,-1.9989,0.19569,1.40797,-2.0806,0.327942,1.22143,-1.74903,0.235813,1.12357,-1.77087,0.291013,1.06546,-1.79536,0.398232,1.02596,-1.7822,0.33555,0.85322,-1.7691,0.298229,0.787845,-1.78039,0.238758,0.908344,-1.79128,0.203621,0.84557,-1.79725,0.133538,0.625633,-1.80743,0.021113,0.589156,-1.89195,0.265666,0.531144,-1.82634,0.216879,0.494307,-1.75894,0.075991,1.11416,-2.11127,0.054839,1.03327,-2.09035,-0.063088,1.15014,-2.08606,-0.080766,1.07107,-2.07082,-0.076986,0.919312,-2.06747,-0.138817,0.741519,-2.05243,-0.234115,0.801789,-2.06159,-0.267648,0.740573,-2.04801,-0.36439,0.523056,-2.01109,-0.46702,0.507308,-1.94267,-0.248824,0.377971,-1.9854,-0.309175,0.348522,-2.04474,0.112828,1.33618,-1.82907,0.10485,1.33016,-2.04407},
/*514*/{0.182359,1.92673,-1.7335,0.286307,1.8663,-1.86359,0.031974,1.83255,-1.6693,-0.087507,1.76112,-1.64162,0.04899,1.67625,-1.66428,0.119785,1.62147,-1.66135,0.243023,1.94037,-2.11481,0.300465,1.86681,-1.93927,0.100023,1.88878,-2.05201,0.125756,1.70581,-2.22233,0.187596,1.62178,-2.2524,0.327142,1.64255,-2.11467,0.390925,1.63742,-2.10343,0.1995,1.42178,-1.81909,0.01124,1.49603,-1.8845,0.014871,1.5158,-1.94754,0.017452,1.49301,-1.99973,0.197514,1.40891,-2.07955,0.328187,1.22566,-1.74895,0.235259,1.12843,-1.76982,0.291636,1.06913,-1.79374,0.396369,1.02901,-1.78276,0.342623,0.851347,-1.76937,0.308646,0.783713,-1.78076,0.244082,0.901532,-1.79311,0.211541,0.837057,-1.79863,0.154089,0.613368,-1.80896,0.043605,0.571546,-1.89296,0.290168,0.526594,-1.82825,0.245114,0.485754,-1.76018,0.071927,1.11669,-2.10968,0.048836,1.03582,-2.08897,-0.064497,1.15677,-2.08558,-0.085804,1.07932,-2.07063,-0.086632,0.92779,-2.06812,-0.148444,0.750232,-2.05481,-0.245499,0.809023,-2.06177,-0.278367,0.748669,-2.04868,-0.375081,0.530229,-2.01294,-0.47646,0.522037,-1.94253,-0.268111,0.378127,-1.98697,-0.330327,0.350433,-2.04428,0.113416,1.34057,-1.82816,0.105788,1.33228,-2.0431},
/*515*/{0.181605,1.92948,-1.73324,0.285886,1.86996,-1.86243,0.029931,1.83686,-1.67131,-0.089517,1.76551,-1.64531,0.04587,1.6802,-1.6655,0.116541,1.62502,-1.66112,0.243965,1.94379,-2.11375,0.30132,1.86907,-1.93756,0.10112,1.89206,-2.0523,0.129804,1.71011,-2.22385,0.192758,1.62598,-2.25262,0.327826,1.64798,-2.11264,0.392175,1.6431,-2.10015,0.199983,1.42596,-1.81848,0.013535,1.4996,-1.88517,0.017255,1.51939,-1.94826,0.019773,1.49622,-2.0004,0.19933,1.41009,-2.07829,0.330157,1.22894,-1.74968,0.233307,1.13027,-1.7688,0.289992,1.07298,-1.79107,0.394738,1.03089,-1.78325,0.349401,0.849703,-1.76943,0.31816,0.779627,-1.78088,0.248541,0.89433,-1.7946,0.220445,0.828844,-1.79919,0.17336,0.602153,-1.81011,0.065946,0.553722,-1.89338,0.315444,0.521227,-1.8285,0.271588,0.478551,-1.76276,0.067963,1.11958,-2.10924,0.041867,1.04138,-2.08895,-0.066905,1.1643,-2.08555,-0.090014,1.08703,-2.07033,-0.097123,0.935603,-2.06916,-0.159064,0.758564,-2.05676,-0.255302,0.817598,-2.06187,-0.288542,0.756758,-2.04904,-0.386203,0.537605,-2.01464,-0.484336,0.537066,-1.9416,-0.286686,0.378799,-1.98849,-0.350015,0.354923,-2.04366,0.114032,1.34477,-1.82682,0.107282,1.33436,-2.0417},
/*516*/{0.181101,1.93216,-1.73321,0.285204,1.87137,-1.86137,0.027575,1.8412,-1.67386,-0.09389,1.76955,-1.64872,0.043084,1.68392,-1.66644,0.113335,1.62802,-1.66038,0.245769,1.94747,-2.11221,0.301323,1.87213,-1.93651,0.102344,1.89529,-2.05299,0.134546,1.71455,-2.2259,0.196372,1.63025,-2.25351,0.330298,1.65403,-2.11168,0.392858,1.64887,-2.09786,0.200244,1.42929,-1.81844,0.014742,1.50392,-1.88455,0.01956,1.52239,-1.94944,0.022031,1.49869,-2.00072,0.201366,1.41114,-2.07729,0.330189,1.23084,-1.75031,0.232565,1.13493,-1.76795,0.289907,1.07521,-1.78965,0.393044,1.03125,-1.78302,0.356641,0.847034,-1.76934,0.328307,0.776278,-1.781,0.252927,0.886442,-1.79534,0.227808,0.820017,-1.79927,0.192668,0.590388,-1.81075,0.088629,0.536598,-1.89324,0.339624,0.517422,-1.83031,0.299735,0.471182,-1.76476,0.065931,1.12207,-2.10932,0.036528,1.04459,-2.08848,-0.068174,1.17123,-2.08528,-0.094371,1.09381,-2.06988,-0.106481,0.944311,-2.0699,-0.169102,0.766096,-2.05922,-0.26475,0.825528,-2.0618,-0.297761,0.764545,-2.04984,-0.39566,0.546039,-2.01642,-0.491088,0.553581,-1.94259,-0.305433,0.378892,-1.99019,-0.371508,0.359814,-2.04273,0.114381,1.34848,-1.82562,0.108418,1.33627,-2.04043},
/*517*/{0.180615,1.93495,-1.73401,0.278764,1.87743,-1.86249,0.026336,1.84532,-1.67541,-0.095403,1.77404,-1.65224,0.040941,1.68708,-1.66758,0.110714,1.63117,-1.65991,0.247744,1.95116,-2.11123,0.300259,1.87587,-1.93581,0.103494,1.89732,-2.05366,0.138957,1.7187,-2.22771,0.200339,1.63432,-2.25422,0.331983,1.65845,-2.11042,0.394636,1.65364,-2.09549,0.201496,1.43266,-1.81872,0.01699,1.50773,-1.88516,0.021861,1.52551,-1.94957,0.024494,1.50145,-2.00097,0.203509,1.41212,-2.07664,0.330429,1.23338,-1.75164,0.231569,1.13837,-1.76661,0.286422,1.07765,-1.78918,0.390866,1.03036,-1.78395,0.362088,0.844147,-1.76992,0.338375,0.772319,-1.78091,0.258403,0.878222,-1.79631,0.235915,0.810118,-1.80185,0.215286,0.579166,-1.81122,0.111761,0.520212,-1.89384,0.363275,0.514598,-1.8319,0.326476,0.466183,-1.76628,0.063036,1.12524,-2.10952,0.032454,1.04893,-2.08962,-0.069223,1.17666,-2.08345,-0.097035,1.10038,-2.0687,-0.113607,0.952352,-2.07052,-0.176757,0.773722,-2.06076,-0.271992,0.834432,-2.06246,-0.306356,0.772829,-2.05079,-0.405153,0.554995,-2.01891,-0.497165,0.569353,-1.94267,-0.323321,0.379134,-1.99161,-0.39451,0.366263,-2.04308,0.115585,1.35213,-1.82472,0.10982,1.33823,-2.03943},
/*518*/{0.180244,1.93726,-1.73397,0.278113,1.87902,-1.86163,0.024213,1.84888,-1.67738,-0.097526,1.77756,-1.65566,0.038382,1.68959,-1.66841,0.108074,1.63387,-1.65959,0.248684,1.95383,-2.10986,0.300959,1.87886,-1.93441,0.104975,1.90039,-2.05496,0.142899,1.72303,-2.22973,0.203802,1.63795,-2.25522,0.332656,1.66165,-2.10882,0.395825,1.65741,-2.09289,0.20407,1.43562,-1.8184,0.018747,1.51035,-1.88568,0.022812,1.52862,-1.95089,0.026494,1.50457,-2.00131,0.20537,1.41377,-2.07577,0.330691,1.23456,-1.75281,0.228836,1.14213,-1.76711,0.284133,1.07931,-1.78807,0.389201,1.02974,-1.78398,0.368477,0.841245,-1.76978,0.348073,0.767712,-1.78094,0.262722,0.870734,-1.79768,0.243727,0.800645,-1.80238,0.233933,0.567487,-1.81181,0.134937,0.504148,-1.89388,0.388148,0.512319,-1.83479,0.352502,0.460267,-1.76845,0.060208,1.12877,-2.11033,0.028231,1.05162,-2.09056,-0.070543,1.18202,-2.08309,-0.099689,1.1058,-2.06872,-0.118745,0.958789,-2.07111,-0.184168,0.780583,-2.06249,-0.278154,0.842108,-2.06261,-0.313502,0.780882,-2.05319,-0.414283,0.564296,-2.02057,-0.502892,0.585956,-1.94315,-0.341413,0.378882,-1.99241,-0.415406,0.373739,-2.04118,0.117098,1.35546,-1.82421,0.111595,1.34039,-2.03885},
/*519*/{0.179836,1.93953,-1.73384,0.27782,1.88125,-1.86055,0.023374,1.85207,-1.6787,-0.100674,1.78022,-1.65902,0.037085,1.69281,-1.66955,0.10589,1.63587,-1.6596,0.249149,1.95661,-2.10844,0.301398,1.88144,-1.93384,0.106028,1.90291,-2.05545,0.147439,1.72666,-2.23086,0.207089,1.64113,-2.25581,0.333551,1.66516,-2.10828,0.396316,1.66102,-2.09133,0.20524,1.43836,-1.81815,0.019992,1.51437,-1.8858,0.024935,1.53142,-1.95052,0.028845,1.50702,-2.00159,0.205949,1.41578,-2.0752,0.331341,1.23474,-1.75345,0.229908,1.14162,-1.76693,0.283371,1.08001,-1.7866,0.386192,1.02764,-1.78352,0.374537,0.837538,-1.7693,0.358855,0.765456,-1.78049,0.267423,0.862287,-1.79943,0.252101,0.790977,-1.80349,0.254515,0.557491,-1.81216,0.157892,0.488986,-1.89325,0.409774,0.509746,-1.83556,0.379058,0.455212,-1.76934,0.057544,1.13158,-2.1113,0.025507,1.05638,-2.09302,-0.071891,1.18646,-2.08206,-0.101632,1.1102,-2.06806,-0.120852,0.962968,-2.07131,-0.189913,0.786305,-2.06401,-0.283387,0.849258,-2.06385,-0.31944,0.788944,-2.05386,-0.422121,0.574043,-2.02305,-0.508085,0.603251,-1.94526,-0.359075,0.37759,-1.99093,-0.435419,0.382951,-2.03893,0.117907,1.35868,-1.82352,0.112348,1.34271,-2.03809},
/*520*/{0.179607,1.94095,-1.73363,0.279435,1.88315,-1.86,0.022395,1.85398,-1.68096,-0.10122,1.78312,-1.66173,0.034661,1.69441,-1.67061,0.104582,1.63762,-1.65884,0.249624,1.95905,-2.10756,0.300977,1.88348,-1.93296,0.106896,1.90521,-2.05563,0.150904,1.72965,-2.23242,0.21013,1.6441,-2.25693,0.334887,1.66691,-2.10789,0.396981,1.66284,-2.08931,0.205583,1.44022,-1.81814,0.020771,1.51703,-1.8854,0.02591,1.53468,-1.95064,0.029909,1.51033,-2.0018,0.206944,1.41694,-2.07475,0.329713,1.23477,-1.75496,0.227142,1.14484,-1.76611,0.280073,1.0798,-1.78634,0.383432,1.02519,-1.78356,0.379019,0.833554,-1.76923,0.366702,0.759268,-1.77997,0.271393,0.852552,-1.79994,0.258957,0.781184,-1.80495,0.275096,0.546718,-1.81203,0.180937,0.475151,-1.89391,0.432273,0.507546,-1.83752,0.404494,0.451389,-1.77119,0.056351,1.1355,-2.11246,0.023858,1.05964,-2.0944,-0.0732,1.18989,-2.08114,-0.10318,1.11414,-2.06727,-0.121597,0.965816,-2.07109,-0.195092,0.79189,-2.06444,-0.286253,0.857641,-2.06519,-0.324362,0.796401,-2.05539,-0.429236,0.584633,-2.02503,-0.514375,0.620225,-1.94729,-0.377849,0.377225,-1.98954,-0.453171,0.392897,-2.03605,0.117589,1.36107,-1.82329,0.112334,1.34498,-2.03786},
/*521*/{0.179487,1.9426,-1.73386,0.278608,1.88486,-1.85918,0.021554,1.85618,-1.68323,-0.104123,1.78448,-1.66477,0.033288,1.69677,-1.6715,0.102266,1.63902,-1.65843,0.250148,1.9612,-2.10727,0.301596,1.88558,-1.93194,0.107903,1.90789,-2.05604,0.154629,1.73248,-2.23405,0.212881,1.64664,-2.25769,0.335105,1.66881,-2.10601,0.398017,1.66533,-2.08809,0.20722,1.44387,-1.81745,0.021881,1.51973,-1.88506,0.026824,1.53733,-1.95049,0.031304,1.51345,-2.00142,0.207852,1.41913,-2.0743,0.329119,1.23373,-1.75577,0.225069,1.14366,-1.76671,0.276226,1.07863,-1.78634,0.381156,1.02225,-1.78297,0.38387,0.829792,-1.76873,0.375613,0.755258,-1.78016,0.275517,0.842945,-1.80059,0.26696,0.771068,-1.80506,0.293068,0.537652,-1.81239,0.203669,0.462468,-1.89398,0.45311,0.505808,-1.83853,0.42829,0.448451,-1.77359,0.055556,1.13817,-2.11334,0.023467,1.0614,-2.09623,-0.073973,1.19242,-2.08017,-0.103856,1.11584,-2.06723,-0.120947,0.968157,-2.07128,-0.199114,0.794808,-2.06489,-0.289645,0.862124,-2.06672,-0.327952,0.803614,-2.05662,-0.438298,0.594798,-2.02631,-0.520304,0.63612,-1.95008,-0.401083,0.381682,-1.98791,-0.473031,0.403073,-2.0342,0.119052,1.36449,-1.82237,0.113511,1.34745,-2.03686},
/*522*/{0.17909,1.94408,-1.73357,0.278688,1.88683,-1.85878,0.021555,1.8577,-1.68395,-0.104004,1.78647,-1.66765,0.032044,1.69785,-1.67213,0.101489,1.64015,-1.65813,0.250573,1.96265,-2.10586,0.301292,1.88725,-1.93145,0.108359,1.90906,-2.05615,0.157776,1.73538,-2.23467,0.215006,1.64865,-2.25781,0.3361,1.67146,-2.10595,0.398749,1.66583,-2.08691,0.207024,1.44619,-1.8177,0.022184,1.52167,-1.88674,0.027931,1.53971,-1.95026,0.031643,1.51557,-2.00148,0.208247,1.42138,-2.0736,0.327524,1.23099,-1.75648,0.224856,1.14473,-1.76486,0.272735,1.07726,-1.78633,0.377026,1.01736,-1.7828,0.387645,0.825946,-1.76903,0.383145,0.750999,-1.78019,0.278825,0.834071,-1.80081,0.275017,0.761826,-1.80448,0.312491,0.53018,-1.81318,0.225868,0.450565,-1.89436,0.473325,0.504801,-1.83975,0.451268,0.445578,-1.77396,0.055349,1.14124,-2.11401,0.024161,1.06375,-2.09719,-0.074237,1.19462,-2.08016,-0.103418,1.11795,-2.06635,-0.119796,0.96936,-2.07095,-0.202519,0.798081,-2.06469,-0.291214,0.868312,-2.06798,-0.331622,0.810865,-2.05853,-0.447538,0.604655,-2.02867,-0.526474,0.652914,-1.95548,-0.419493,0.393816,-1.98338,-0.493152,0.415374,-2.03262,0.118985,1.36674,-1.82246,0.114062,1.34959,-2.03696},
/*523*/{0.1795,1.94474,-1.7332,0.286486,1.88432,-1.85577,0.021001,1.85914,-1.68525,-0.104647,1.7878,-1.67063,0.030271,1.69921,-1.67306,0.09987,1.64061,-1.65759,0.251186,1.96415,-2.1052,0.302972,1.88841,-1.93043,0.109218,1.9112,-2.05606,0.160642,1.73711,-2.23612,0.217043,1.65008,-2.25865,0.337638,1.67141,-2.10521,0.399332,1.66589,-2.0858,0.207856,1.44759,-1.81763,0.023076,1.52387,-1.88603,0.027648,1.54218,-1.9501,0.032512,1.51717,-2.00134,0.208695,1.42336,-2.07313,0.325462,1.22692,-1.75727,0.223819,1.14264,-1.7644,0.268635,1.07408,-1.78599,0.372435,1.01335,-1.7826,0.391226,0.822219,-1.76865,0.391045,0.746408,-1.78035,0.281856,0.824123,-1.80006,0.28104,0.752476,-1.80477,0.33038,0.520655,-1.81246,0.247161,0.439692,-1.8953,0.492351,0.504145,-1.83976,0.472563,0.44308,-1.7772,0.056733,1.14243,-2.11512,0.025158,1.06568,-2.09843,-0.073439,1.19624,-2.07973,-0.101388,1.11942,-2.06704,-0.117437,0.970139,-2.07103,-0.204528,0.799975,-2.06421,-0.290939,0.873861,-2.06942,-0.333721,0.816766,-2.0595,-0.456484,0.61466,-2.03012,-0.534988,0.668701,-1.95749,-0.440304,0.404765,-1.98056,-0.510989,0.428564,-2.02988,0.119478,1.36857,-1.82225,0.1144,1.35157,-2.03675},
/*524*/{0.18022,1.94581,-1.7323,0.286799,1.88589,-1.85561,0.020759,1.85903,-1.68643,-0.106363,1.78854,-1.67287,0.030265,1.69935,-1.67301,0.099453,1.64063,-1.6574,0.251402,1.96532,-2.1048,0.302841,1.88951,-1.92978,0.110523,1.91252,-2.05616,0.162484,1.73869,-2.2363,0.218072,1.65124,-2.25885,0.337842,1.6714,-2.10512,0.399813,1.66518,-2.08493,0.207334,1.45,-1.81774,0.024155,1.52489,-1.8856,0.028313,1.54376,-1.94957,0.032848,1.51921,-2.00093,0.2096,1.42566,-2.07291,0.323012,1.22307,-1.75785,0.214989,1.13752,-1.7659,0.263281,1.07077,-1.7867,0.36879,1.00801,-1.78211,0.394375,0.818278,-1.76886,0.397041,0.743166,-1.78051,0.284673,0.814667,-1.79772,0.288131,0.742454,-1.80364,0.347249,0.515029,-1.81365,0.266906,0.429643,-1.89659,0.510708,0.503964,-1.84,0.493652,0.441823,-1.77632,0.058424,1.1443,-2.11585,0.026548,1.06691,-2.0988,-0.070962,1.19618,-2.07998,-0.099289,1.11916,-2.06659,-0.114762,0.969776,-2.07066,-0.206132,0.80086,-2.06356,-0.290409,0.877657,-2.07091,-0.335408,0.822243,-2.06091,-0.465497,0.626794,-2.03103,-0.541482,0.683885,-1.96151,-0.457256,0.416816,-1.97715,-0.527083,0.441487,-2.02802,0.120139,1.37023,-1.82219,0.115354,1.35367,-2.03673},
/*525*/{0.180521,1.94633,-1.73159,0.28559,1.88792,-1.85532,0.020499,1.85913,-1.68683,-0.106411,1.78786,-1.6739,0.029748,1.7,-1.67356,0.099373,1.64024,-1.65564,0.25192,1.96606,-2.10445,0.303109,1.89078,-1.9295,0.110593,1.91333,-2.05578,0.163847,1.7401,-2.23655,0.219116,1.65167,-2.25913,0.338467,1.6701,-2.10469,0.40067,1.66403,-2.08447,0.208484,1.45116,-1.81812,0.024013,1.5262,-1.88506,0.027781,1.54566,-1.94929,0.033094,1.52087,-2.00088,0.210284,1.42766,-2.07243,0.321038,1.21727,-1.75811,0.211471,1.13552,-1.76609,0.256894,1.06704,-1.78788,0.365227,1.0029,-1.78153,0.397441,0.81388,-1.76958,0.403216,0.738027,-1.78067,0.287589,0.805629,-1.79635,0.293685,0.734279,-1.80204,0.364781,0.508621,-1.81343,0.287519,0.421833,-1.89791,0.526344,0.503986,-1.84029,0.512298,0.441781,-1.77649,0.06059,1.14463,-2.11594,0.029757,1.06689,-2.09972,-0.07041,1.19524,-2.07976,-0.096676,1.11831,-2.06701,-0.111417,0.968701,-2.07057,-0.209072,0.801388,-2.06253,-0.290576,0.881171,-2.07226,-0.336728,0.827783,-2.06292,-0.474856,0.638886,-2.03208,-0.552527,0.699752,-1.96584,-0.473927,0.430069,-1.97384,-0.539371,0.455555,-2.02586,0.120778,1.37154,-1.82234,0.115902,1.35556,-2.03693},
/*526*/{0.180952,1.94667,-1.73104,0.286169,1.88864,-1.85527,0.021004,1.8594,-1.68647,-0.106209,1.788,-1.67585,0.029414,1.6992,-1.67363,0.099205,1.6397,-1.65495,0.252716,1.96608,-2.10384,0.30328,1.89164,-1.92904,0.11123,1.91422,-2.05568,0.165035,1.74087,-2.23649,0.219589,1.65244,-2.25965,0.339041,1.66922,-2.10523,0.401059,1.6623,-2.0844,0.206567,1.45136,-1.81865,0.023806,1.52644,-1.88462,0.027541,1.5468,-1.94841,0.033656,1.5213,-2.00021,0.210513,1.42933,-2.07227,0.318524,1.21329,-1.75898,0.205552,1.13357,-1.76773,0.252512,1.06381,-1.78748,0.359961,0.997285,-1.78238,0.400788,0.810306,-1.7703,0.408837,0.734995,-1.78182,0.290169,0.797306,-1.79425,0.29985,0.725893,-1.79966,0.378634,0.502312,-1.81443,0.305787,0.413178,-1.89819,0.541341,0.50576,-1.84094,0.529057,0.44154,-1.77782,0.063426,1.14446,-2.11568,0.03378,1.06625,-2.09962,-0.066502,1.19388,-2.08066,-0.09291,1.11725,-2.06695,-0.106782,0.966837,-2.06949,-0.210587,0.801925,-2.0618,-0.286698,0.885644,-2.07405,-0.335825,0.834868,-2.06595,-0.485232,0.650821,-2.03173,-0.559823,0.714981,-1.96922,-0.489196,0.442928,-1.97004,-0.554693,0.469647,-2.02557,0.119998,1.37139,-1.82284,0.115903,1.35696,-2.03755},
/*527*/{0.181376,1.94649,-1.73038,0.287983,1.88853,-1.85388,0.021301,1.85828,-1.68671,-0.105893,1.7871,-1.6768,0.029877,1.69801,-1.67387,0.099378,1.63839,-1.65428,0.252854,1.96598,-2.10357,0.303691,1.8925,-1.9285,0.111539,1.91455,-2.05482,0.165618,1.74155,-2.23644,0.219335,1.6522,-2.2599,0.340814,1.66803,-2.10508,0.401649,1.65926,-2.08381,0.206228,1.45173,-1.81875,0.023605,1.52626,-1.88424,0.026837,1.54744,-1.94863,0.032623,1.5227,-1.99918,0.210887,1.43168,-2.07188,0.315117,1.20709,-1.75821,0.201414,1.12893,-1.76717,0.244539,1.05852,-1.78758,0.355679,0.991551,-1.78237,0.40426,0.806744,-1.77103,0.41366,0.732076,-1.78444,0.293715,0.789409,-1.79069,0.305672,0.719109,-1.79836,0.390839,0.498424,-1.8143,0.321617,0.406278,-1.89848,0.55546,0.508823,-1.83999,0.545072,0.442761,-1.77761,0.067415,1.14358,-2.11563,0.038782,1.06503,-2.09913,-0.064143,1.1924,-2.08142,-0.089086,1.11396,-2.06696,-0.101964,0.96492,-2.06885,-0.212268,0.802659,-2.06007,-0.282318,0.888087,-2.07564,-0.336381,0.840938,-2.06529,-0.490747,0.66378,-2.03254,-0.566787,0.731599,-1.9717,-0.503215,0.457591,-1.96677,-0.567102,0.485017,-2.02473,0.119995,1.37146,-1.82346,0.116311,1.35867,-2.03828},
/*528*/{0.182043,1.94618,-1.72983,0.287966,1.88828,-1.85326,0.021585,1.85691,-1.68714,-0.104904,1.78491,-1.67766,0.030245,1.69648,-1.67346,0.099799,1.63586,-1.65274,0.253701,1.96521,-2.10312,0.304004,1.89263,-1.9279,0.111982,1.91437,-2.05424,0.165563,1.74139,-2.23596,0.21907,1.65182,-2.26012,0.341465,1.66586,-2.10543,0.401533,1.65588,-2.08391,0.206116,1.4518,-1.81892,0.023537,1.52619,-1.88382,0.02657,1.54757,-1.94786,0.032623,1.5227,-1.99918,0.211286,1.43295,-2.07169,0.310703,1.20057,-1.75889,0.195676,1.12379,-1.76808,0.239116,1.05344,-1.78832,0.349301,0.985437,-1.78289,0.40714,0.803499,-1.77218,0.419162,0.729191,-1.78495,0.296742,0.782188,-1.78839,0.311082,0.711936,-1.79588,0.403181,0.494862,-1.81428,0.336282,0.400378,-1.89845,0.567593,0.510262,-1.84065,0.561067,0.444947,-1.77959,0.071649,1.14229,-2.11513,0.044111,1.0636,-2.09858,-0.060279,1.18964,-2.08219,-0.084951,1.11173,-2.06711,-0.096744,0.962195,-2.06786,-0.212689,0.803344,-2.05876,-0.279919,0.890038,-2.0745,-0.334912,0.847191,-2.06786,-0.500153,0.676427,-2.03218,-0.573242,0.748369,-1.97531,-0.51802,0.471907,-1.96471,-0.578354,0.501016,-2.02408,0.120321,1.37128,-1.82375,0.116853,1.35942,-2.03863},
/*529*/{0.183297,1.94527,-1.72881,0.288229,1.8882,-1.85298,0.022588,1.85543,-1.68701,-0.104687,1.78255,-1.6783,0.031592,1.69485,-1.67343,0.100624,1.63407,-1.6519,0.254691,1.96457,-2.10282,0.30457,1.89252,-1.92742,0.112359,1.91448,-2.05321,0.164181,1.74093,-2.23554,0.217538,1.65077,-2.25958,0.340958,1.66309,-2.10562,0.402353,1.65192,-2.08433,0.208565,1.45351,-1.81909,0.022985,1.52517,-1.88253,0.026145,1.54743,-1.94689,0.032294,1.52289,-1.99828,0.210812,1.43417,-2.07176,0.305854,1.19487,-1.75933,0.189825,1.12174,-1.76916,0.231158,1.04807,-1.78949,0.343474,0.97967,-1.78341,0.409829,0.80118,-1.77385,0.424553,0.726719,-1.78677,0.298995,0.776023,-1.78798,0.316097,0.706828,-1.79452,0.412297,0.490495,-1.81418,0.348883,0.39511,-1.89781,0.575608,0.510151,-1.84149,0.572367,0.444436,-1.77983,0.076523,1.14056,-2.11376,0.0498,1.06176,-2.09727,-0.056323,1.18661,-2.08188,-0.079847,1.10723,-2.0668,-0.091004,0.959594,-2.06718,-0.212753,0.805034,-2.05799,-0.276712,0.89652,-2.0769,-0.332167,0.855003,-2.06796,-0.50551,0.691775,-2.03261,-0.576875,0.765127,-1.97791,-0.530781,0.488076,-1.96307,-0.588538,0.517682,-2.02464,0.122851,1.37202,-1.82324,0.117688,1.35992,-2.03807},
/*530*/{0.183827,1.94426,-1.72812,0.288566,1.88739,-1.85272,0.023774,1.85312,-1.6864,-0.103089,1.78013,-1.67876,0.033178,1.69101,-1.67228,0.102202,1.63044,-1.65109,0.254243,1.96332,-2.10231,0.305191,1.8919,-1.92711,0.11234,1.91427,-2.05307,0.163468,1.74067,-2.2348,0.215944,1.64998,-2.25982,0.341561,1.65929,-2.10634,0.401945,1.64716,-2.0848,0.207547,1.45304,-1.81964,0.022895,1.52402,-1.88325,0.026593,1.54603,-1.94647,0.031355,1.52195,-1.99791,0.211634,1.43486,-2.07164,0.303069,1.18736,-1.75914,0.186089,1.11848,-1.76951,0.225268,1.04431,-1.79241,0.337522,0.974144,-1.78375,0.410435,0.798666,-1.77575,0.428664,0.724949,-1.78802,0.3014,0.770743,-1.78672,0.319565,0.702253,-1.79324,0.42266,0.489078,-1.81441,0.359142,0.391053,-1.89744,0.582709,0.511709,-1.84404,0.581095,0.445734,-1.78125,0.081615,1.13832,-2.11299,0.055208,1.0595,-2.0966,-0.053481,1.18132,-2.08207,-0.075324,1.10341,-2.06783,-0.085076,0.956021,-2.0666,-0.212514,0.8069,-2.05722,-0.272129,0.901552,-2.0772,-0.330627,0.86074,-2.06749,-0.509287,0.706483,-2.03291,-0.580927,0.783578,-1.97983,-0.541987,0.505308,-1.96307,-0.598395,0.535787,-2.02491,0.123047,1.37092,-1.82384,0.118538,1.35981,-2.03874},
/*531*/{0.184666,1.94302,-1.72764,0.288923,1.88646,-1.85252,0.024589,1.85009,-1.68569,-0.100954,1.77636,-1.67825,0.035504,1.68844,-1.67213,0.103916,1.62775,-1.64891,0.25443,1.96153,-2.10141,0.305148,1.89105,-1.9271,0.112344,1.91358,-2.05207,0.160843,1.73923,-2.23399,0.213873,1.64866,-2.25876,0.341418,1.65595,-2.10731,0.401731,1.64169,-2.086,0.208724,1.45294,-1.81967,0.022913,1.52198,-1.88219,0.026238,1.54556,-1.94574,0.031193,1.52039,-1.99714,0.211264,1.43493,-2.07236,0.29745,1.18158,-1.75859,0.180431,1.11154,-1.77056,0.218321,1.04014,-1.79293,0.330964,0.968665,-1.78351,0.410621,0.797112,-1.77666,0.43007,0.723946,-1.78859,0.301706,0.766464,-1.78698,0.320868,0.697555,-1.79395,0.42923,0.486666,-1.8146,0.367116,0.388237,-1.89663,0.591111,0.513593,-1.84491,0.589943,0.446231,-1.78355,0.086291,1.13625,-2.11171,0.061644,1.0561,-2.09552,-0.048756,1.17733,-2.0824,-0.06988,1.09806,-2.067,-0.077765,0.952276,-2.06605,-0.212161,0.808287,-2.05723,-0.267481,0.905538,-2.07688,-0.326702,0.868636,-2.0669,-0.512538,0.721719,-2.03303,-0.581739,0.800734,-1.98158,-0.552993,0.521494,-1.96305,-0.606372,0.554232,-2.02518,0.124739,1.37013,-1.82378,0.119023,1.35925,-2.03866},
/*532*/{0.185533,1.94154,-1.72665,0.289217,1.88534,-1.85282,0.02547,1.84734,-1.68487,-0.099838,1.77136,-1.6775,0.036588,1.68424,-1.67026,0.105914,1.62362,-1.64751,0.25499,1.95984,-2.10081,0.304413,1.88942,-1.92645,0.112118,1.91303,-2.05158,0.158713,1.73819,-2.23283,0.211779,1.6469,-2.25838,0.341087,1.65197,-2.10787,0.401204,1.63593,-2.08721,0.209234,1.4513,-1.82062,0.02254,1.51909,-1.88218,0.025121,1.54377,-1.94597,0.030978,1.51909,-1.99673,0.211181,1.43483,-2.07245,0.294502,1.17447,-1.75796,0.174641,1.1073,-1.7711,0.211699,1.03554,-1.79417,0.325623,0.963771,-1.78416,0.409479,0.795421,-1.7778,0.4301,0.722718,-1.78909,0.301493,0.762512,-1.78754,0.322225,0.694509,-1.79452,0.433893,0.484429,-1.81454,0.373347,0.385793,-1.89622,0.595729,0.512137,-1.84712,0.595544,0.445976,-1.78421,0.091181,1.13378,-2.11112,0.067943,1.05312,-2.09477,-0.044861,1.17211,-2.08253,-0.06421,1.09288,-2.06685,-0.070537,0.947399,-2.0655,-0.210223,0.810328,-2.05655,-0.261401,0.910135,-2.07608,-0.3228,0.875195,-2.0665,-0.513329,0.738435,-2.03578,-0.579843,0.819925,-1.98113,-0.562049,0.53878,-1.96373,-0.615679,0.574129,-2.02635,0.125462,1.36798,-1.82478,0.119564,1.35844,-2.03972},
/*533*/{0.186494,1.93926,-1.72573,0.289301,1.88375,-1.85254,0.027032,1.84373,-1.68346,-0.096562,1.76776,-1.67623,0.039093,1.68043,-1.6688,0.108133,1.61939,-1.64588,0.255197,1.95784,-2.1008,0.304678,1.88782,-1.92603,0.111655,1.91122,-2.04977,0.155416,1.73654,-2.23141,0.208886,1.6454,-2.25789,0.340375,1.64658,-2.10859,0.40058,1.62951,-2.08814,0.207677,1.44976,-1.82052,0.021487,1.51763,-1.88155,0.024096,1.54146,-1.94602,0.030075,1.51746,-1.99675,0.211061,1.43368,-2.07225,0.291682,1.16832,-1.75688,0.169172,1.10521,-1.77274,0.205869,1.03213,-1.79617,0.31886,0.957962,-1.78399,0.407219,0.793297,-1.77814,0.427692,0.720672,-1.78934,0.300002,0.758229,-1.7886,0.322252,0.690323,-1.79493,0.437976,0.482301,-1.81439,0.377463,0.382377,-1.89575,0.599179,0.511216,-1.84956,0.599278,0.445357,-1.78617,0.096635,1.13036,-2.11003,0.074765,1.04957,-2.09342,-0.040649,1.16646,-2.08223,-0.058774,1.08665,-2.06633,-0.062399,0.942234,-2.06513,-0.207694,0.812597,-2.05739,-0.254927,0.914095,-2.07486,-0.317396,0.882949,-2.06529,-0.514878,0.755056,-2.0367,-0.57768,0.838419,-1.98199,-0.570318,0.557219,-1.96578,-0.622968,0.594344,-2.02828,0.124506,1.36621,-1.82477,0.119214,1.35705,-2.03974},
/*534*/{0.186859,1.93684,-1.7254,0.288542,1.88219,-1.85227,0.028204,1.83976,-1.68144,-0.094575,1.76299,-1.67472,0.041984,1.67571,-1.66686,0.11077,1.61495,-1.64384,0.254928,1.95559,-2.10056,0.304294,1.8856,-1.92607,0.110947,1.90988,-2.04944,0.152842,1.7351,-2.22968,0.205547,1.6433,-2.25691,0.339934,1.64126,-2.11,0.399561,1.62326,-2.09027,0.206794,1.44729,-1.82074,0.021709,1.5142,-1.88088,0.024923,1.53853,-1.94576,0.029584,1.51431,-1.99614,0.211238,1.43226,-2.07296,0.287071,1.16361,-1.75634,0.164954,1.10246,-1.7725,0.201642,1.02837,-1.79563,0.312383,0.952546,-1.78471,0.403633,0.790662,-1.7787,0.425257,0.719844,-1.79024,0.296132,0.752818,-1.79155,0.3208,0.686378,-1.79569,0.440181,0.481368,-1.81539,0.378858,0.378993,-1.89513,0.600266,0.510482,-1.85124,0.603751,0.44475,-1.78833,0.101459,1.12671,-2.1098,0.08135,1.04581,-2.09237,-0.035939,1.15977,-2.08162,-0.052891,1.08038,-2.06587,-0.055149,0.937617,-2.06613,-0.205189,0.81438,-2.05742,-0.247507,0.918792,-2.07433,-0.310985,0.888114,-2.06367,-0.513528,0.770424,-2.03724,-0.573403,0.856348,-1.98086,-0.579125,0.575231,-1.96802,-0.627935,0.615191,-2.0291,0.125038,1.36305,-1.82527,0.119685,1.35492,-2.04029},
/*535*/{0.187559,1.93442,-1.72474,0.28887,1.87992,-1.85234,0.02934,1.83507,-1.68053,-0.09204,1.7569,-1.67251,0.044177,1.67063,-1.66468,0.113494,1.61023,-1.64185,0.254533,1.9529,-2.0999,0.304461,1.88307,-1.92587,0.11056,1.90789,-2.04871,0.14888,1.734,-2.22868,0.202454,1.64111,-2.25606,0.338947,1.63527,-2.11148,0.398292,1.61565,-2.092,0.208306,1.44498,-1.82032,0.020654,1.51155,-1.88045,0.024031,1.53556,-1.94556,0.029885,1.51108,-1.99571,0.211631,1.4312,-2.073,0.28578,1.15926,-1.75471,0.163476,1.0988,-1.77182,0.196753,1.02435,-1.79815,0.31028,0.949269,-1.78451,0.400258,0.787376,-1.77773,0.42426,0.715017,-1.78954,0.293919,0.750531,-1.79123,0.317676,0.682302,-1.79748,0.440719,0.47838,-1.8161,0.379869,0.375652,-1.89554,0.60019,0.508673,-1.85353,0.603705,0.442186,-1.79023,0.106694,1.12264,-2.10898,0.088794,1.04092,-2.09147,-0.03275,1.15325,-2.08177,-0.047691,1.07312,-2.06386,-0.046336,0.93142,-2.06431,-0.200723,0.816627,-2.05797,-0.239208,0.922511,-2.07255,-0.304232,0.895389,-2.06376,-0.51193,0.785902,-2.03786,-0.56694,0.874425,-1.98074,-0.586501,0.592301,-1.9705,-0.631644,0.636424,-2.03128,0.126313,1.36056,-1.82532,0.120827,1.35291,-2.04035},
/*536*/{0.18828,1.93117,-1.72401,0.288838,1.87737,-1.85235,0.030976,1.8303,-1.67806,-0.088408,1.75128,-1.67009,0.047569,1.66477,-1.6623,0.11753,1.60515,-1.63978,0.254332,1.94995,-2.09967,0.304072,1.87995,-1.92536,0.110524,1.90645,-2.04784,0.145123,1.73124,-2.22694,0.198582,1.63898,-2.25513,0.336737,1.62889,-2.11247,0.396581,1.60799,-2.09407,0.208403,1.44149,-1.82052,0.020722,1.50783,-1.88022,0.023234,1.53215,-1.94576,0.0289,1.50802,-1.9954,0.212191,1.42905,-2.07299,0.283014,1.15464,-1.75296,0.159576,1.09391,-1.77292,0.195703,1.02157,-1.79839,0.309309,0.947609,-1.78454,0.396531,0.783465,-1.77801,0.419922,0.713004,-1.79095,0.291272,0.746529,-1.79131,0.314859,0.678601,-1.7985,0.437441,0.474379,-1.81792,0.37728,0.372932,-1.89737,0.597128,0.506366,-1.85633,0.603124,0.439675,-1.79265,0.110696,1.11862,-2.10898,0.095025,1.03667,-2.09083,-0.027447,1.14683,-2.08087,-0.040949,1.0661,-2.06388,-0.038279,0.925111,-2.06411,-0.196548,0.818397,-2.05801,-0.230044,0.926159,-2.07205,-0.296402,0.900239,-2.06174,-0.509142,0.802295,-2.03912,-0.559062,0.89094,-1.97809,-0.590117,0.610417,-1.97393,-0.638632,0.657699,-2.03299,0.126593,1.35695,-1.82588,0.121229,1.35031,-2.04095},
/*537*/{0.189265,1.92763,-1.72332,0.288575,1.8742,-1.85228,0.032189,1.82516,-1.67704,-0.085475,1.74454,-1.66733,0.050347,1.65889,-1.66041,0.120027,1.59961,-1.63816,0.25323,1.94707,-2.09963,0.303618,1.87749,-1.92562,0.110504,1.90383,-2.0466,0.141491,1.72903,-2.22508,0.195176,1.63638,-2.25405,0.335059,1.62234,-2.11367,0.394412,1.59977,-2.09665,0.205768,1.43827,-1.82043,0.019592,1.50415,-1.88055,0.022741,1.52892,-1.94622,0.02843,1.50419,-1.99625,0.212349,1.42667,-2.07316,0.280681,1.15244,-1.75173,0.159513,1.0936,-1.77311,0.193721,1.01934,-1.79858,0.309224,0.946791,-1.78446,0.393242,0.779,-1.77841,0.415656,0.708173,-1.79152,0.287357,0.741721,-1.79305,0.310883,0.67408,-1.80034,0.435864,0.470784,-1.81987,0.373421,0.369191,-1.89875,0.594679,0.50178,-1.85771,0.599405,0.435341,-1.79622,0.115762,1.11483,-2.1089,0.101139,1.03216,-2.09023,-0.024,1.13939,-2.08047,-0.035471,1.05893,-2.0635,-0.030067,0.918974,-2.06365,-0.191113,0.81961,-2.05828,-0.221244,0.929636,-2.07153,-0.288419,0.90525,-2.06168,-0.503912,0.816988,-2.03872,-0.550138,0.907529,-1.97719,-0.594674,0.628089,-1.97689,-0.639986,0.678617,-2.03553,0.125125,1.35329,-1.82652,0.121312,1.34746,-2.04164},
/*538*/{0.189956,1.9241,-1.72236,0.28859,1.87047,-1.85251,0.033909,1.81925,-1.67482,-0.082552,1.73794,-1.66389,0.053812,1.65337,-1.6574,0.124138,1.59405,-1.63523,0.25313,1.94323,-2.09992,0.303627,1.87434,-1.9251,0.109556,1.90087,-2.04564,0.136788,1.72667,-2.22432,0.190204,1.63305,-2.25339,0.334133,1.61597,-2.11578,0.392361,1.59093,-2.0993,0.20858,1.43884,-1.82281,0.019789,1.50012,-1.88041,0.022042,1.52441,-1.9459,0.027703,1.5006,-1.99655,0.212212,1.42364,-2.07273,0.277488,1.15046,-1.75123,0.15677,1.09067,-1.77378,0.191664,1.0173,-1.79946,0.309382,0.945759,-1.78488,0.389723,0.774308,-1.77866,0.413046,0.702633,-1.79126,0.283817,0.737494,-1.79407,0.30708,0.670701,-1.80121,0.430533,0.466626,-1.82117,0.367807,0.365113,-1.90144,0.588189,0.496467,-1.8608,0.59457,0.429692,-1.79757,0.12048,1.11026,-2.10847,0.10769,1.02831,-2.0902,-0.019138,1.13297,-2.07996,-0.030369,1.05188,-2.06335,-0.021987,0.912782,-2.06332,-0.185421,0.820816,-2.05847,-0.21115,0.930954,-2.07059,-0.278254,0.910336,-2.06146,-0.497528,0.83027,-2.03896,-0.53886,0.922478,-1.9763,-0.596274,0.644838,-1.98072,-0.64023,0.698752,-2.03767,0.128836,1.35227,-1.82569,0.122734,1.34377,-2.04067},
/*539*/{0.190545,1.91969,-1.72186,0.288597,1.86683,-1.85259,0.035839,1.8138,-1.67153,-0.080188,1.73058,-1.66035,0.057058,1.64601,-1.6543,0.128375,1.58867,-1.63368,0.252293,1.93966,-2.09987,0.303851,1.8704,-1.92482,0.108509,1.89758,-2.0449,0.132793,1.7232,-2.22234,0.185268,1.62941,-2.25255,0.330041,1.60771,-2.11662,0.389602,1.58185,-2.10131,0.204532,1.43171,-1.8196,0.017525,1.49616,-1.88041,0.022156,1.52115,-1.94585,0.027197,1.49588,-1.99669,0.211919,1.42063,-2.07256,0.277359,1.14761,-1.75018,0.156,1.0891,-1.77363,0.191524,1.01458,-1.79865,0.309129,0.944196,-1.78522,0.386599,0.769554,-1.77913,0.409635,0.697824,-1.79211,0.279644,0.734097,-1.79474,0.302985,0.666313,-1.80163,0.422365,0.459759,-1.82293,0.360799,0.361438,-1.90528,0.584218,0.489198,-1.86058,0.586761,0.420974,-1.79984,0.125792,1.10633,-2.10801,0.114326,1.02337,-2.08956,-0.015562,1.1256,-2.07956,-0.02316,1.04449,-2.06243,-0.013541,0.906727,-2.06315,-0.180203,0.820978,-2.05915,-0.200422,0.932901,-2.07005,-0.268998,0.914218,-2.06169,-0.490425,0.843892,-2.04104,-0.527674,0.936546,-1.9754,-0.596219,0.661762,-1.98405,-0.636058,0.718952,-2.04332,0.124625,1.34607,-1.82625,0.121688,1.34044,-2.04139},
/*540*/{0.190181,1.91493,-1.72178,0.289628,1.8617,-1.85238,0.03765,1.80692,-1.66951,-0.076165,1.72374,-1.65647,0.060492,1.63933,-1.6509,0.13321,1.58268,-1.63108,0.251278,1.93579,-2.1002,0.302842,1.86614,-1.9254,0.108928,1.8942,-2.0443,0.128119,1.72068,-2.22069,0.18127,1.62585,-2.25131,0.327305,1.59976,-2.11819,0.386706,1.57209,-2.10427,0.204782,1.42831,-1.81876,0.017559,1.4928,-1.88099,0.021289,1.51726,-1.9458,0.027549,1.49408,-1.99692,0.21159,1.41734,-2.07273,0.275542,1.14581,-1.74959,0.153709,1.08698,-1.77492,0.190292,1.01254,-1.79836,0.308412,0.940929,-1.78604,0.382519,0.763906,-1.77939,0.404234,0.691635,-1.7938,0.275524,0.730112,-1.79554,0.297641,0.661605,-1.80408,0.414401,0.454087,-1.82441,0.351487,0.357488,-1.90842,0.577342,0.4823,-1.86107,0.577665,0.413501,-1.80204,0.130092,1.102,-2.10769,0.120587,1.01875,-2.08978,-0.010283,1.1177,-2.07852,-0.017092,1.03741,-2.06193,-0.006328,0.900689,-2.06313,-0.173304,0.820641,-2.05864,-0.189487,0.932752,-2.06806,-0.25809,0.917201,-2.06106,-0.481518,0.855168,-2.0413,-0.515195,0.949124,-1.97525,-0.594855,0.677599,-1.98687,-0.632403,0.735111,-2.0438,0.124239,1.34269,-1.82645,0.12157,1.33743,-2.0416},
/*541*/{0.190832,1.9096,-1.72095,0.288528,1.85811,-1.85254,0.039677,1.80009,-1.66708,-0.074607,1.7151,-1.65195,0.06519,1.63246,-1.64739,0.136874,1.57667,-1.62949,0.250822,1.93152,-2.1007,0.303473,1.8621,-1.92521,0.107372,1.89088,-2.04325,0.124412,1.71538,-2.21955,0.17624,1.62232,-2.25054,0.323885,1.59137,-2.12045,0.383246,1.56176,-2.10792,0.203321,1.42447,-1.81854,0.01675,1.48848,-1.88104,0.020837,1.51301,-1.94597,0.026058,1.48639,-1.99678,0.211475,1.41399,-2.07198,0.274081,1.14345,-1.74925,0.152657,1.08228,-1.77538,0.190387,1.00965,-1.79936,0.306244,0.936201,-1.78612,0.377479,0.758064,-1.78151,0.39817,0.685801,-1.79438,0.270271,0.726106,-1.79616,0.290832,0.657767,-1.80501,0.405386,0.447949,-1.82688,0.34127,0.353737,-1.91127,0.567713,0.473761,-1.86301,0.569194,0.405167,-1.80381,0.134832,1.09762,-2.10858,0.127568,1.01397,-2.09004,-0.004692,1.11046,-2.07807,-0.009809,1.02947,-2.06153,0.002965,0.894922,-2.06297,-0.165621,0.820489,-2.05859,-0.178908,0.93298,-2.06818,-0.248107,0.920251,-2.06175,-0.473144,0.866632,-2.04179,-0.502737,0.960768,-1.97558,-0.590203,0.691519,-1.99043,-0.628618,0.754687,-2.04585,0.124165,1.33862,-1.82578,0.122029,1.33263,-2.04092},
/*542*/{0.191748,1.90469,-1.72036,0.289293,1.8522,-1.8523,0.041726,1.79279,-1.66426,-0.070101,1.70621,-1.64793,0.069014,1.62557,-1.64449,0.141782,1.57077,-1.62823,0.249127,1.92619,-2.10104,0.303081,1.85739,-1.92558,0.106924,1.88579,-2.0418,0.120295,1.71171,-2.21804,0.170676,1.61802,-2.25022,0.320518,1.58291,-2.12242,0.378966,1.5517,-2.11099,0.202654,1.42161,-1.81806,0.014746,1.48388,-1.88225,0.0208,1.51053,-1.94648,0.025482,1.48137,-1.99783,0.212139,1.41048,-2.07039,0.274057,1.14139,-1.74897,0.152274,1.08199,-1.77638,0.189928,1.00704,-1.79956,0.303754,0.931081,-1.7858,0.373264,0.751617,-1.78199,0.392711,0.679635,-1.79587,0.265318,0.722069,-1.79709,0.285094,0.653256,-1.80579,0.396681,0.443316,-1.82877,0.329822,0.348511,-1.91444,0.558181,0.46483,-1.86293,0.556997,0.395428,-1.80555,0.140065,1.09248,-2.1092,0.134342,1.00904,-2.09075,-0.000228,1.10297,-2.07729,-0.00328,1.02235,-2.06056,0.011328,0.888965,-2.06322,-0.158134,0.820421,-2.0589,-0.167113,0.933389,-2.06824,-0.237056,0.922396,-2.06197,-0.463077,0.877028,-2.04112,-0.490732,0.971115,-1.97542,-0.586099,0.705082,-1.9926,-0.621588,0.767351,-2.04741,0.123373,1.33547,-1.82528,0.122933,1.32874,-2.04041},
/*543*/{0.191793,1.89878,-1.7194,0.289246,1.84904,-1.85235,0.044173,1.78554,-1.66128,-0.066645,1.6968,-1.64253,0.073237,1.61829,-1.64113,0.147428,1.56446,-1.6258,0.249065,1.92174,-2.1019,0.303307,1.85262,-1.92604,0.10613,1.88166,-2.04096,0.116343,1.70777,-2.21714,0.165395,1.61362,-2.24942,0.316431,1.57441,-2.12465,0.374842,1.54104,-2.11426,0.202338,1.41816,-1.81701,0.014735,1.47858,-1.88241,0.019824,1.50466,-1.94706,0.023865,1.47653,-1.99827,0.211166,1.40573,-2.07141,0.272021,1.13713,-1.74903,0.152137,1.07891,-1.77719,0.186666,1.00388,-1.79905,0.301196,0.9261,-1.78612,0.368021,0.745914,-1.78231,0.386228,0.673198,-1.79721,0.258505,0.718053,-1.79869,0.27717,0.649204,-1.80715,0.385651,0.438444,-1.83068,0.315696,0.34728,-1.91691,0.546997,0.455383,-1.86432,0.543911,0.385168,-1.80595,0.145763,1.08799,-2.10944,0.141506,1.0043,-2.09085,0.004676,1.09605,-2.07677,0.003934,1.01498,-2.06028,0.020386,0.883427,-2.06394,-0.15065,0.819413,-2.05918,-0.154769,0.932346,-2.0684,-0.225901,0.924293,-2.06182,-0.452555,0.88585,-2.042,-0.47745,0.979817,-1.97562,-0.579543,0.716138,-1.99537,-0.613572,0.779978,-2.04925,0.123623,1.33136,-1.82497,0.122746,1.3236,-2.04006},
/*544*/{0.192625,1.89327,-1.71854,0.288727,1.84354,-1.85313,0.046688,1.7771,-1.65831,-0.06355,1.68833,-1.63783,0.078509,1.6109,-1.63784,0.15273,1.55817,-1.62329,0.247244,1.91663,-2.1022,0.302948,1.84817,-1.92643,0.105591,1.8769,-2.04078,0.111302,1.70367,-2.2165,0.160372,1.60906,-2.24875,0.312754,1.56569,-2.12723,0.371372,1.52954,-2.11727,0.200655,1.4148,-1.81648,0.012654,1.47395,-1.88293,0.018184,1.49946,-1.947,0.023879,1.4704,-1.99901,0.211047,1.40205,-2.07116,0.270216,1.13389,-1.75014,0.150625,1.07406,-1.77715,0.185074,0.999929,-1.79991,0.298031,0.921095,-1.78626,0.360905,0.739058,-1.78358,0.377907,0.66601,-1.79756,0.251985,0.712938,-1.80029,0.268595,0.644819,-1.80912,0.370556,0.433116,-1.83193,0.29752,0.351796,-1.91795,0.533053,0.443516,-1.86477,0.525413,0.373725,-1.80489,0.151,1.08343,-2.10926,0.148256,0.999981,-2.09144,0.010163,1.08928,-2.07521,0.010427,1.00818,-2.05982,0.030117,0.877456,-2.06447,-0.141515,0.818974,-2.05974,-0.143304,0.933055,-2.06822,-0.213107,0.926398,-2.06243,-0.441747,0.894014,-2.04246,-0.46498,0.989746,-1.9765,-0.571187,0.726673,-1.99705,-0.603296,0.790416,-2.05234,0.123019,1.32732,-1.8246,0.123378,1.31904,-2.03967},
/*545*/{0.193249,1.88669,-1.7176,0.289046,1.83802,-1.85326,0.04977,1.76838,-1.65479,-0.059687,1.67893,-1.63227,0.083141,1.60292,-1.63411,0.15765,1.55216,-1.62109,0.246874,1.91147,-2.10307,0.30289,1.84279,-1.92683,0.103901,1.8718,-2.0406,0.106989,1.69958,-2.21591,0.154935,1.60484,-2.24845,0.307872,1.55542,-2.12948,0.364412,1.51896,-2.12202,0.199999,1.41077,-1.81499,0.012413,1.46782,-1.88324,0.017621,1.49454,-1.94677,0.023169,1.46456,-1.99928,0.210367,1.39794,-2.07004,0.270714,1.12881,-1.74934,0.14867,1.06945,-1.77826,0.185526,0.995865,-1.79962,0.294585,0.915869,-1.78699,0.354161,0.732678,-1.78371,0.367973,0.659606,-1.79949,0.245044,0.708592,-1.80113,0.26044,0.638298,-1.8112,0.356563,0.428515,-1.83295,0.279507,0.351174,-1.9206,0.5189,0.431211,-1.86687,0.505876,0.363968,-1.80663,0.155977,1.07959,-2.10924,0.155479,0.995341,-2.09174,0.015881,1.08185,-2.07405,0.016948,1.00077,-2.05943,0.039661,0.871894,-2.06583,-0.13294,0.818483,-2.06135,-0.131503,0.931714,-2.06818,-0.202303,0.927271,-2.06235,-0.42957,0.901533,-2.04407,-0.452065,0.996124,-1.97631,-0.562345,0.735767,-1.99918,-0.593664,0.799976,-2.05384,0.122927,1.3227,-1.82386,0.124009,1.31409,-2.03892},
/*546*/{0.194158,1.88065,-1.71633,0.288973,1.83318,-1.85351,0.053165,1.76013,-1.65144,-0.055148,1.66941,-1.62751,0.088881,1.59574,-1.63094,0.164499,1.54658,-1.61973,0.245259,1.90592,-2.10341,0.302839,1.83743,-1.92766,0.103107,1.86656,-2.03994,0.103331,1.69475,-2.21467,0.14885,1.59996,-2.24738,0.303446,1.54533,-2.13226,0.358842,1.50772,-2.12567,0.198272,1.40697,-1.81428,0.011137,1.46301,-1.88444,0.017202,1.48951,-1.94682,0.022096,1.45869,-1.99974,0.210299,1.39389,-2.06973,0.269209,1.1238,-1.74943,0.147425,1.06542,-1.77865,0.183443,0.991073,-1.79993,0.291183,0.910589,-1.78691,0.345611,0.724278,-1.78522,0.361256,0.651606,-1.79988,0.236757,0.704358,-1.80304,0.250673,0.633921,-1.81331,0.342593,0.422687,-1.8364,0.260744,0.355027,-1.92237,0.505629,0.416068,-1.86769,0.48643,0.349795,-1.80555,0.160818,1.0747,-2.11012,0.162062,0.99094,-2.09266,0.020949,1.07476,-2.07378,0.023673,0.99329,-2.05896,0.050218,0.862738,-2.06616,-0.122887,0.817684,-2.06228,-0.118323,0.93189,-2.06972,-0.187891,0.928151,-2.06211,-0.418009,0.907805,-2.0441,-0.441209,1.00302,-1.97752,-0.55107,0.744309,-2.00176,-0.580398,0.808195,-2.05643,0.122131,1.31844,-1.82356,0.124375,1.30935,-2.03859},
/*547*/{0.194846,1.87453,-1.71549,0.290507,1.82616,-1.85356,0.05654,1.7513,-1.64828,-0.050318,1.65859,-1.62133,0.094324,1.58812,-1.62774,0.170762,1.54083,-1.61849,0.244631,1.90044,-2.10409,0.302668,1.83162,-1.92808,0.101852,1.86135,-2.0394,0.098346,1.69028,-2.21463,0.143924,1.5953,-2.24719,0.297739,1.53508,-2.13531,0.352089,1.49594,-2.1296,0.197714,1.40395,-1.81287,0.010662,1.45755,-1.88526,0.015933,1.4829,-1.94605,0.021171,1.45216,-2.00046,0.210482,1.38837,-2.06827,0.268069,1.11784,-1.74916,0.145898,1.06164,-1.7792,0.180954,0.985383,-1.80064,0.287619,0.90413,-1.78728,0.338813,0.719394,-1.78612,0.352181,0.644947,-1.80084,0.229237,0.699826,-1.80411,0.241946,0.630067,-1.81462,0.327236,0.418557,-1.83772,0.239862,0.354816,-1.92167,0.49066,0.40295,-1.86858,0.468858,0.338847,-1.80693,0.165561,1.07118,-2.11097,0.168115,0.986549,-2.09362,0.025366,1.06803,-2.07394,0.031158,0.98613,-2.05848,0.060052,0.858888,-2.06754,-0.112672,0.816273,-2.06323,-0.105558,0.929581,-2.06945,-0.17592,0.928284,-2.06228,-0.405962,0.913121,-2.0444,-0.428214,1.00943,-1.97863,-0.540621,0.749645,-2.00258,-0.567587,0.81418,-2.05893,0.122383,1.31457,-1.82176,0.125426,1.30328,-2.03668},
/*548*/{0.195857,1.86857,-1.71427,0.289116,1.82151,-1.85436,0.060236,1.74178,-1.6444,-0.044667,1.64979,-1.61597,0.099706,1.58078,-1.6255,0.177923,1.53533,-1.61775,0.243459,1.89522,-2.10515,0.302095,1.82635,-1.92896,0.100576,1.85634,-2.03906,0.094538,1.68575,-2.21353,0.137661,1.59087,-2.24754,0.292207,1.52418,-2.13837,0.345703,1.48437,-2.13351,0.197297,1.4003,-1.81094,0.008649,1.45212,-1.88607,0.014758,1.4772,-1.94617,0.019581,1.44729,-2.00094,0.209821,1.38372,-2.06736,0.264821,1.11079,-1.74803,0.142657,1.05522,-1.77843,0.178656,0.980251,-1.80152,0.281541,0.897645,-1.78658,0.331422,0.712971,-1.78722,0.341865,0.637346,-1.80329,0.221466,0.697351,-1.8048,0.233235,0.627262,-1.81499,0.309476,0.416817,-1.83976,0.219535,0.356162,-1.92108,0.473449,0.395611,-1.87555,0.449246,0.33925,-1.80894,0.170816,1.06662,-2.11196,0.175109,0.98262,-2.09494,0.031175,1.06011,-2.07319,0.038474,0.979114,-2.05855,0.070087,0.854244,-2.0679,-0.10209,0.814721,-2.06383,-0.09207,0.928458,-2.07009,-0.162806,0.928108,-2.06227,-0.392071,0.917995,-2.04445,-0.415709,1.01408,-1.97967,-0.527787,0.754846,-2.00399,-0.553916,0.826323,-2.07318,0.121576,1.31043,-1.82084,0.125556,1.29827,-2.03569},
/*549*/{0.197364,1.86278,-1.71316,0.290914,1.81498,-1.85424,0.0648,1.73268,-1.6411,-0.039428,1.63923,-1.61025,0.106725,1.57374,-1.62186,0.185082,1.53058,-1.61624,0.242004,1.89028,-2.10567,0.302683,1.82119,-1.92953,0.098383,1.85099,-2.03851,0.088781,1.68238,-2.21389,0.13157,1.58595,-2.24738,0.285967,1.51375,-2.14188,0.338288,1.47229,-2.13754,0.19577,1.39584,-1.81057,0.006543,1.44752,-1.88701,0.01338,1.47144,-1.94683,0.01854,1.44131,-2.00266,0.209501,1.37922,-2.06758,0.260992,1.1063,-1.74622,0.138934,1.0499,-1.77791,0.173968,0.976712,-1.80262,0.275982,0.890559,-1.78595,0.32548,0.708711,-1.78936,0.332703,0.633862,-1.80508,0.214294,0.697177,-1.80605,0.222089,0.626625,-1.81785,0.293918,0.418173,-1.84386,0.196847,0.35595,-1.92191,0.451804,0.392905,-1.88043,0.42717,0.338865,-1.8094,0.175724,1.06217,-2.11179,0.182542,0.979053,-2.09624,0.037124,1.05339,-2.07315,0.045597,0.972228,-2.05786,0.081321,0.849701,-2.06867,-0.09034,0.812818,-2.06555,-0.078696,0.925983,-2.0702,-0.14731,0.92708,-2.06279,-0.378953,0.919996,-2.04517,-0.404028,1.01827,-1.98147,-0.514294,0.75787,-2.00684,-0.540581,0.823146,-2.06354,0.120323,1.30579,-1.82106,0.12549,1.29322,-2.03586},
/*550*/{0.198627,1.85762,-1.71181,0.291039,1.81052,-1.85446,0.069926,1.72317,-1.63746,-0.032802,1.62981,-1.60469,0.11407,1.5672,-1.61919,0.192544,1.52474,-1.61483,0.241477,1.88597,-2.10639,0.302895,1.81529,-1.93068,0.097668,1.84658,-2.03628,0.085592,1.67676,-2.21314,0.124854,1.58094,-2.24726,0.279899,1.50335,-2.14526,0.331421,1.4614,-2.14119,0.192859,1.39249,-1.80992,0.003466,1.4428,-1.88783,0.010946,1.46501,-1.9481,0.016556,1.43567,-2.00375,0.208326,1.37363,-2.06681,0.254276,1.10417,-1.74355,0.135044,1.04956,-1.77812,0.169877,0.975709,-1.80368,0.268439,0.883132,-1.78689,0.318843,0.708102,-1.79004,0.324329,0.633033,-1.80647,0.208278,0.70104,-1.80633,0.213101,0.629906,-1.81673,0.271719,0.417157,-1.84759,0.175218,0.357333,-1.92401,0.430296,0.391701,-1.88319,0.404933,0.34099,-1.81159,0.179942,1.05849,-2.11243,0.189461,0.975698,-2.09734,0.041996,1.0462,-2.07338,0.052487,0.965511,-2.05836,0.090916,0.844146,-2.07107,-0.078374,0.809935,-2.06614,-0.064518,0.923858,-2.07062,-0.134086,0.925769,-2.06362,-0.365084,0.922729,-2.04563,-0.389755,1.01978,-1.98131,-0.501085,0.760279,-2.00716,-0.526608,0.825661,-2.06565,0.117915,1.30191,-1.82014,0.124555,1.28747,-2.03479},
/*551*/{0.199822,1.85277,-1.71073,0.291705,1.80332,-1.85498,0.074964,1.71475,-1.63492,-0.024712,1.62069,-1.59901,0.120931,1.56137,-1.61745,0.200694,1.52059,-1.61421,0.238657,1.88208,-2.10803,0.303852,1.81035,-1.93125,0.096475,1.84327,-2.03522,0.08163,1.67275,-2.21271,0.119116,1.57685,-2.24705,0.273708,1.49392,-2.1488,0.322498,1.45128,-2.14559,0.190346,1.38868,-1.80947,0.000246,1.43914,-1.88889,0.008148,1.46025,-1.94942,0.01349,1.43012,-2.00384,0.207282,1.36914,-2.0653,0.253821,1.10149,-1.7377,0.131024,1.04936,-1.7805,0.166735,0.975447,-1.80431,0.264611,0.881859,-1.78712,0.314601,0.70884,-1.79085,0.315957,0.63417,-1.80804,0.202422,0.70705,-1.80518,0.20532,0.635966,-1.81572,0.25042,0.422599,-1.84875,0.152563,0.357999,-1.9253,0.407921,0.390544,-1.88356,0.383707,0.339344,-1.81218,0.184699,1.05585,-2.11315,0.195817,0.972667,-2.09789,0.046225,1.03902,-2.07324,0.059899,0.959108,-2.05837,0.101867,0.83966,-2.0718,-0.065627,0.807477,-2.06649,-0.050365,0.920126,-2.07008,-0.1197,0.924029,-2.06378,-0.350771,0.925094,-2.04736,-0.376435,1.02127,-1.98281,-0.48573,0.760836,-2.00892,-0.508336,0.823171,-2.06515,0.115118,1.29833,-1.81933,0.123116,1.28262,-2.03384},
/*552*/{0.201172,1.8483,-1.70956,0.290853,1.79984,-1.85663,0.08086,1.70633,-1.63228,-0.01764,1.61176,-1.59377,0.129294,1.55604,-1.61634,0.208732,1.51658,-1.61354,0.237709,1.87819,-2.10926,0.303821,1.80624,-1.93344,0.095416,1.84211,-2.03287,0.0775,1.66905,-2.21243,0.112406,1.57285,-2.24622,0.266771,1.48604,-2.15217,0.314893,1.44152,-2.14884,0.187682,1.38607,-1.80882,-0.003329,1.43586,-1.88997,0.005895,1.4551,-1.9509,0.010863,1.42487,-2.00579,0.208241,1.36481,-2.06132,0.252183,1.10162,-1.73437,0.13275,1.05237,-1.78201,0.166259,0.976347,-1.80414,0.26519,0.886047,-1.78563,0.309081,0.707918,-1.79211,0.303456,0.632862,-1.81028,0.196982,0.712487,-1.80377,0.194839,0.641106,-1.81624,0.231628,0.42406,-1.84912,0.130728,0.359244,-1.92629,0.387146,0.389334,-1.88341,0.359799,0.33921,-1.81242,0.189489,1.05413,-2.11349,0.202805,0.97125,-2.09859,0.052264,1.03298,-2.07411,0.067964,0.952821,-2.05826,0.113344,0.835592,-2.07168,-0.052356,0.80496,-2.06621,-0.035921,0.917975,-2.07185,-0.104744,0.92211,-2.06533,-0.336675,0.92419,-2.04793,-0.363866,1.02166,-1.98438,-0.471051,0.760424,-2.00995,-0.49311,0.821981,-2.06679,0.112153,1.29569,-1.81767,0.123408,1.2779,-2.03187},
/*553*/{0.202681,1.84416,-1.70879,0.291729,1.79482,-1.85763,0.086353,1.69907,-1.63016,-0.009923,1.60347,-1.58863,0.137878,1.5512,-1.61518,0.21802,1.51273,-1.61322,0.23686,1.87481,-2.11084,0.304379,1.80212,-1.93478,0.094999,1.84007,-2.03054,0.073371,1.6665,-2.21168,0.106242,1.56983,-2.24556,0.25876,1.47688,-2.15476,0.30575,1.4325,-2.15251,0.185271,1.38372,-1.80746,-0.006453,1.43231,-1.89275,0.002959,1.4525,-1.951,0.00791,1.41991,-2.00732,0.205552,1.36049,-2.06118,0.248857,1.10186,-1.73487,0.129372,1.05066,-1.78094,0.166431,0.975008,-1.80411,0.269247,0.892141,-1.78431,0.301903,0.706978,-1.79313,0.295928,0.632408,-1.80991,0.188754,0.714775,-1.8028,0.184728,0.644517,-1.81458,0.212136,0.424951,-1.85029,0.109301,0.359875,-1.92669,0.36566,0.388324,-1.88348,0.33905,0.338509,-1.81299,0.19483,1.0542,-2.11372,0.210069,0.971499,-2.09912,0.057275,1.02869,-2.07406,0.074781,0.948746,-2.05917,0.125295,0.832604,-2.07211,-0.038438,0.802065,-2.06644,-0.021588,0.915097,-2.0723,-0.091049,0.91883,-2.06601,-0.321714,0.923432,-2.04873,-0.350338,1.02104,-1.98619,-0.454652,0.757766,-2.01095,-0.480396,0.819923,-2.07014,0.109019,1.29334,-1.81705,0.12112,1.2736,-2.03104},
/*554*/{0.204193,1.84089,-1.70845,0.292385,1.79086,-1.858,0.092124,1.6932,-1.62883,-0.000274,1.59688,-1.58432,0.14764,1.54728,-1.61386,0.227687,1.51026,-1.61258,0.236248,1.87224,-2.11163,0.30427,1.79848,-1.93611,0.094007,1.83948,-2.02947,0.069468,1.6656,-2.20958,0.100156,1.56849,-2.24478,0.250776,1.47154,-2.15577,0.296637,1.42485,-2.15584,0.183102,1.38206,-1.80552,-0.009236,1.42978,-1.89483,0.001191,1.45068,-1.95172,0.005549,1.41486,-2.00791,0.204044,1.35695,-2.06003,0.244649,1.10242,-1.73501,0.12684,1.0477,-1.7795,0.166445,0.973972,-1.80319,0.274094,0.896929,-1.78412,0.292021,0.705336,-1.79253,0.284411,0.632017,-1.8096,0.179472,0.717582,-1.80058,0.172777,0.646304,-1.81345,0.191901,0.424132,-1.84987,0.088152,0.360165,-1.92682,0.344196,0.387036,-1.88354,0.316758,0.338813,-1.81281,0.200001,1.0547,-2.11331,0.218361,0.972827,-2.09919,0.063243,1.02489,-2.0742,0.084223,0.945549,-2.06006,0.138758,0.829718,-2.07229,-0.024124,0.799393,-2.06572,-0.006998,0.912113,-2.07468,-0.075666,0.916702,-2.06706,-0.307567,0.921283,-2.05066,-0.337977,1.0182,-1.9875,-0.438464,0.754024,-2.01266,-0.464501,0.816758,-2.07165,0.106293,1.29187,-1.81566,0.119817,1.26982,-2.02934},
/*555*/{0.205594,1.83878,-1.70913,0.291829,1.79001,-1.86025,0.097787,1.6889,-1.62735,0.009429,1.59145,-1.57946,0.156741,1.54384,-1.61293,0.237714,1.50846,-1.6125,0.236321,1.87008,-2.11214,0.302733,1.79706,-1.93758,0.093944,1.83917,-2.02955,0.065856,1.6655,-2.20735,0.092858,1.56858,-2.24354,0.241929,1.46666,-2.15758,0.28761,1.41806,-2.15939,0.17994,1.38117,-1.80351,-0.011214,1.42832,-1.89578,-0.00013,1.45079,-1.95311,0.003634,1.41174,-2.00972,0.202735,1.35484,-2.05966,0.24066,1.1016,-1.73471,0.123256,1.04627,-1.77966,0.16265,0.971833,-1.80163,0.279516,0.897109,-1.78456,0.281343,0.704091,-1.791,0.271731,0.630632,-1.80799,0.169068,0.719106,-1.79997,0.160427,0.648906,-1.81215,0.171859,0.425367,-1.8499,0.066347,0.361303,-1.9261,0.323506,0.386623,-1.88331,0.295723,0.338032,-1.81204,0.206741,1.05575,-2.11348,0.226385,0.974474,-2.09903,0.069681,1.02124,-2.07576,0.092979,0.944152,-2.06172,0.153755,0.82811,-2.07142,-0.009162,0.79637,-2.0646,0.007919,0.908294,-2.07481,-0.061035,0.914241,-2.06831,-0.293544,0.918163,-2.05081,-0.325825,1.01434,-1.98861,-0.421421,0.748854,-2.01397,-0.448794,0.810738,-2.07237,0.103058,1.29113,-1.81505,0.11839,1.26773,-2.02846},
/*556*/{0.206947,1.83692,-1.70953,0.291066,1.7868,-1.86017,0.103975,1.68539,-1.62587,0.019209,1.58629,-1.57608,0.165988,1.54191,-1.61186,0.248207,1.50759,-1.61176,0.23635,1.86867,-2.11121,0.302618,1.79559,-1.93768,0.093303,1.83805,-2.0293,0.060888,1.66713,-2.20509,0.086132,1.5703,-2.24218,0.233457,1.46315,-2.15845,0.277754,1.41324,-2.163,0.177696,1.38003,-1.8009,-0.012033,1.42781,-1.89593,-0.000873,1.44952,-1.95299,0.001667,1.41113,-2.01033,0.20094,1.35208,-2.0591,0.23762,1.09853,-1.73417,0.120729,1.04446,-1.778,0.159347,0.970243,-1.8005,0.278605,0.89506,-1.78414,0.268604,0.701217,-1.78841,0.256322,0.628387,-1.80691,0.157868,0.719591,-1.79891,0.146941,0.648642,-1.81082,0.152249,0.425168,-1.84847,0.045621,0.362249,-1.92581,0.302972,0.386699,-1.88238,0.274545,0.337217,-1.81086,0.213186,1.05765,-2.1131,0.235187,0.976975,-2.0983,0.077139,1.01953,-2.07752,0.102359,0.941711,-2.06363,0.169293,0.82755,-2.07124,0.00538,0.794363,-2.06393,0.021622,0.904829,-2.07709,-0.046962,0.911237,-2.06892,-0.279836,0.913581,-2.05137,-0.315648,1.01018,-1.99075,-0.403059,0.742434,-2.01321,-0.4287,0.801159,-2.07215,0.100168,1.29032,-1.81386,0.116185,1.26572,-2.02708},
/*557*/{0.207535,1.83551,-1.71039,0.29002,1.78628,-1.86065,0.109273,1.68295,-1.62474,0.027688,1.5821,-1.57252,0.176739,1.54027,-1.61109,0.258817,1.50818,-1.61327,0.23508,1.86692,-2.11097,0.300377,1.79368,-1.93831,0.093287,1.83799,-2.02951,0.055053,1.6702,-2.20142,0.079918,1.57286,-2.2409,0.225548,1.46135,-2.16024,0.268335,1.40922,-2.16597,0.174585,1.37965,-1.79997,-0.013711,1.42684,-1.89702,-0.003035,1.4494,-1.95287,0.000433,1.4111,-2.01084,0.198534,1.35064,-2.05917,0.234105,1.0972,-1.73175,0.114303,1.04248,-1.77797,0.151654,0.968107,-1.80043,0.272885,0.890203,-1.78391,0.255933,0.698059,-1.78641,0.242833,0.625213,-1.80385,0.144677,0.718678,-1.79903,0.132106,0.648505,-1.80995,0.132184,0.424173,-1.84747,0.024677,0.361978,-1.92542,0.281736,0.385456,-1.88059,0.25312,0.336795,-1.80943,0.219947,1.05996,-2.1128,0.246144,0.980269,-2.09812,0.086343,1.01703,-2.07914,0.11242,0.940365,-2.0652,0.185197,0.828323,-2.07055,0.02026,0.792053,-2.06284,0.035617,0.903044,-2.07818,-0.033177,0.908799,-2.07032,-0.266695,0.907641,-2.05228,-0.305492,1.00456,-1.99289,-0.385414,0.734065,-2.014,-0.415184,0.793036,-2.0747,0.097076,1.28981,-1.81372,0.113837,1.26485,-2.02685},
/*558*/{0.208778,1.83402,-1.71086,0.290082,1.78511,-1.86115,0.114401,1.68131,-1.62374,0.037332,1.57957,-1.56932,0.186337,1.53973,-1.61021,0.269455,1.50965,-1.61366,0.235061,1.86595,-2.11068,0.300226,1.79302,-1.93791,0.092858,1.83813,-2.03009,0.049918,1.67384,-2.19795,0.07208,1.57622,-2.24022,0.216633,1.46031,-2.16204,0.25878,1.40791,-2.16861,0.173307,1.37954,-1.79822,-0.014466,1.42711,-1.89705,-0.003773,1.44972,-1.9531,-0.001042,1.41185,-2.01113,0.198002,1.35069,-2.05906,0.227707,1.09376,-1.73173,0.107892,1.04071,-1.77911,0.145343,0.966243,-1.80083,0.262299,0.885238,-1.78239,0.241573,0.693022,-1.78565,0.227281,0.622749,-1.8031,0.131561,0.717457,-1.79835,0.117166,0.648251,-1.81065,0.110068,0.425964,-1.84647,0.003043,0.362879,-1.9239,0.259818,0.385855,-1.88021,0.231217,0.336512,-1.80923,0.228422,1.06363,-2.11296,0.256095,0.984405,-2.09797,0.093747,1.01883,-2.08198,0.124305,0.939657,-2.06615,0.201605,0.830379,-2.06964,0.035947,0.790327,-2.06229,0.049961,0.901639,-2.07893,-0.020234,0.905947,-2.07117,-0.25275,0.90193,-2.05314,-0.296341,0.998284,-1.99574,-0.365431,0.72489,-2.01389,-0.398308,0.781712,-2.07557,0.09541,1.28987,-1.8135,0.11289,1.26511,-2.02659},
/*559*/{0.209582,1.83376,-1.71169,0.290188,1.78434,-1.86148,0.119118,1.68039,-1.62272,0.046567,1.57572,-1.56614,0.196145,1.54009,-1.61034,0.280194,1.51189,-1.61487,0.234599,1.86585,-2.11066,0.299916,1.79202,-1.93823,0.092277,1.8383,-2.0308,0.043956,1.67861,-2.19504,0.064995,1.5802,-2.23958,0.20857,1.46122,-2.16376,0.248923,1.40692,-2.17125,0.170579,1.37924,-1.79689,-0.014879,1.42692,-1.89644,-0.005033,1.44861,-1.95198,-0.001383,1.41293,-2.00958,0.196567,1.35161,-2.05899,0.222637,1.08927,-1.73142,0.102513,1.03868,-1.77806,0.137469,0.964233,-1.80147,0.250884,0.88069,-1.78155,0.226783,0.689406,-1.78435,0.208992,0.616891,-1.80205,0.118363,0.71645,-1.79782,0.102262,0.647073,-1.8098,0.091096,0.424942,-1.84502,-0.01804,0.363805,-1.92368,0.239227,0.385366,-1.87959,0.20922,0.335546,-1.80874,0.236156,1.06749,-2.11282,0.266348,0.989437,-2.09745,0.103009,1.01776,-2.08329,0.135954,0.941243,-2.06703,0.217148,0.832007,-2.06835,0.051088,0.788677,-2.0614,0.062552,0.900816,-2.07954,-0.006916,0.904208,-2.0728,-0.239928,0.89531,-2.05374,-0.286206,0.990399,-1.99842,-0.346838,0.713837,-2.01345,-0.379757,0.768817,-2.07654,0.093954,1.28915,-1.81322,0.111961,1.26575,-2.02643},
/*560*/{0.210654,1.83389,-1.71169,0.289247,1.78259,-1.86203,0.124066,1.68031,-1.62139,0.056656,1.57362,-1.5629,0.205812,1.54123,-1.60974,0.290597,1.51559,-1.61632,0.233515,1.86611,-2.11073,0.298585,1.7915,-1.9386,0.091721,1.83905,-2.03052,0.039212,1.68444,-2.19135,0.058117,1.58507,-2.23949,0.200601,1.46261,-2.16632,0.239725,1.40793,-2.17349,0.169287,1.37911,-1.79673,-0.016216,1.42718,-1.8963,-0.00656,1.44929,-1.95171,-0.003106,1.41353,-2.00854,0.195623,1.35284,-2.05927,0.217443,1.0863,-1.73155,0.09507,1.03855,-1.77888,0.128727,0.96317,-1.80056,0.238692,0.87576,-1.77996,0.212863,0.685605,-1.78432,0.192927,0.613743,-1.80163,0.104371,0.715187,-1.79735,0.087429,0.646725,-1.81028,0.070674,0.426357,-1.84459,-0.038271,0.364381,-1.92275,0.218601,0.384971,-1.87915,0.189914,0.336289,-1.80735,0.244316,1.07289,-2.11274,0.276745,0.995259,-2.0974,0.112587,1.01838,-2.08445,0.148194,0.942254,-2.06837,0.231715,0.839198,-2.07111,0.066142,0.788139,-2.06077,0.07633,0.900387,-2.07933,0.006668,0.901303,-2.07443,-0.229258,0.887087,-2.05315,-0.279398,0.981872,-1.99985,-0.32539,0.702717,-2.01415,-0.360479,0.755765,-2.07704,0.092817,1.28909,-1.81352,0.110745,1.26674,-2.02684},
/*561*/{0.211817,1.83406,-1.7116,0.288748,1.78495,-1.8634,0.128273,1.68097,-1.62061,0.064829,1.57217,-1.56018,0.215213,1.54331,-1.60957,0.301341,1.52016,-1.61851,0.232238,1.86669,-2.11135,0.297819,1.79162,-1.93968,0.090683,1.84126,-2.02974,0.033602,1.69004,-2.18893,0.050139,1.59059,-2.23925,0.191591,1.46465,-2.16839,0.22977,1.4102,-2.1763,0.168206,1.3795,-1.79591,-0.017608,1.42761,-1.89607,-0.007698,1.44976,-1.95163,-0.00461,1.41475,-2.0074,0.193163,1.35525,-2.06048,0.21046,1.08283,-1.7319,0.086289,1.03931,-1.77963,0.119744,0.961189,-1.7997,0.224703,0.870875,-1.77957,0.197474,0.681056,-1.78419,0.176895,0.609841,-1.80148,0.090317,0.713106,-1.79761,0.071038,0.644485,-1.81064,0.050102,0.426986,-1.84397,-0.059532,0.364875,-1.92209,0.19802,0.384756,-1.87711,0.169315,0.33599,-1.80704,0.252311,1.0778,-2.11216,0.287048,1.00225,-2.09721,0.122353,1.02013,-2.08522,0.160182,0.944506,-2.06989,0.246452,0.8426,-2.07001,0.081633,0.787306,-2.06008,0.090236,0.900877,-2.07974,0.019276,0.899993,-2.07476,-0.214298,0.880282,-2.05517,-0.272456,0.971642,-2.00071,-0.303217,0.68973,-2.0125,-0.340058,0.741433,-2.07822,0.09204,1.28931,-1.81414,0.10939,1.26841,-2.02766},
/*562*/{0.213121,1.83608,-1.71182,0.288558,1.78529,-1.86373,0.132575,1.68162,-1.61878,0.074329,1.57086,-1.55723,0.224957,1.54688,-1.60994,0.311685,1.52517,-1.62054,0.230627,1.86809,-2.1115,0.296848,1.79269,-1.94003,0.089086,1.84219,-2.02929,0.027892,1.69604,-2.18667,0.042612,1.59631,-2.23946,0.183801,1.46819,-2.17082,0.220401,1.41308,-2.17817,0.167452,1.37975,-1.79504,-0.01806,1.42813,-1.89598,-0.00784,1.45068,-1.95132,-0.005818,1.41621,-2.00774,0.19273,1.35678,-2.06106,0.205226,1.08249,-1.73246,0.080235,1.04097,-1.77958,0.109591,0.961605,-1.80037,0.210906,0.865816,-1.77939,0.182343,0.677529,-1.78407,0.161999,0.608849,-1.80167,0.075795,0.711922,-1.79848,0.054886,0.643573,-1.81136,0.030159,0.427218,-1.84355,-0.080186,0.365931,-1.9216,0.177709,0.384431,-1.87724,0.148396,0.335529,-1.80634,0.259152,1.08541,-2.11347,0.296895,1.01005,-2.09711,0.131878,1.02244,-2.08627,0.170132,0.951202,-2.072,0.261329,0.847726,-2.07061,0.097306,0.787957,-2.05981,0.101651,0.901168,-2.08195,0.031729,0.899469,-2.07512,-0.200549,0.871111,-2.05538,-0.264536,0.960167,-2.00327,-0.280784,0.677233,-2.00982,-0.320178,0.726391,-2.0783,0.09131,1.28956,-1.81467,0.108873,1.26981,-2.02828},
/*563*/{0.214148,1.8384,-1.71224,0.288214,1.78737,-1.86499,0.136927,1.6835,-1.61839,0.083714,1.57126,-1.55542,0.233963,1.55091,-1.6102,0.320541,1.53109,-1.62312,0.22833,1.86996,-2.1117,0.295887,1.79364,-1.94103,0.087543,1.84482,-2.02898,0.023487,1.70246,-2.18507,0.035116,1.60257,-2.23928,0.175312,1.47242,-2.17259,0.211794,1.41711,-2.17988,0.167048,1.38056,-1.79582,-0.018734,1.42907,-1.8958,-0.008538,1.45154,-1.95078,-0.006093,1.41702,-2.0066,0.191992,1.35919,-2.0617,0.198542,1.08068,-1.73302,0.073713,1.04129,-1.77924,0.101541,0.960667,-1.79984,0.196592,0.860285,-1.77867,0.167433,0.676269,-1.78477,0.144662,0.605297,-1.80203,0.061061,0.711286,-1.79816,0.038973,0.643285,-1.81029,0.01134,0.427306,-1.84315,-0.100208,0.366963,-1.92125,0.156126,0.384212,-1.87748,0.127983,0.335604,-1.80564,0.265813,1.09251,-2.11364,0.306168,1.01842,-2.09715,0.140626,1.026,-2.08764,0.181111,0.955683,-2.07348,0.275022,0.852871,-2.07118,0.113193,0.788959,-2.05993,0.113618,0.902068,-2.0825,0.04379,0.898641,-2.07681,-0.187557,0.862606,-2.0562,-0.257146,0.948258,-2.0044,-0.258692,0.664804,-2.01169,-0.299345,0.711266,-2.0795,0.091601,1.29019,-1.81525,0.108616,1.27158,-2.029},
/*564*/{0.215754,1.84092,-1.71282,0.287575,1.78911,-1.86568,0.140658,1.6859,-1.61712,0.092044,1.57205,-1.5531,0.242285,1.55531,-1.61142,0.329939,1.53815,-1.62623,0.225537,1.87243,-2.11203,0.294419,1.79554,-1.94228,0.086025,1.84614,-2.02747,0.017651,1.70855,-2.18452,0.026968,1.60922,-2.23976,0.167669,1.47779,-2.17435,0.202415,1.42189,-2.18161,0.166911,1.38085,-1.79528,-0.018558,1.4303,-1.89618,-0.009061,1.45345,-1.95087,-0.006827,1.41839,-2.00644,0.19169,1.36218,-2.06244,0.189945,1.07792,-1.73402,0.066997,1.04185,-1.77902,0.089954,0.961804,-1.80111,0.181616,0.856214,-1.77922,0.152053,0.674183,-1.78512,0.12711,0.603552,-1.80256,0.045829,0.710784,-1.7989,0.023514,0.643178,-1.80999,-0.009109,0.427319,-1.8422,-0.120985,0.368183,-1.92179,0.136318,0.384031,-1.87671,0.106937,0.3355,-1.80552,0.272471,1.1002,-2.11366,0.31453,1.02761,-2.09738,0.149148,1.02988,-2.08842,0.191928,0.96045,-2.07402,0.288361,0.858641,-2.07134,0.129031,0.790563,-2.06052,0.124638,0.903251,-2.08316,0.055408,0.897367,-2.07713,-0.175017,0.85256,-2.05913,-0.248494,0.933902,-2.00577,-0.236183,0.651562,-2.01202,-0.277712,0.695924,-2.08024,0.091452,1.2908,-1.81627,0.108554,1.27389,-2.03016},
/*565*/{0.216782,1.84411,-1.71362,0.28679,1.79103,-1.86698,0.145932,1.68838,-1.6169,0.100953,1.57289,-1.55158,0.250618,1.56051,-1.61323,0.33808,1.54515,-1.6294,0.222304,1.8749,-2.11175,0.293587,1.79774,-1.94309,0.083595,1.84822,-2.02627,0.011335,1.71471,-2.18373,0.019758,1.61604,-2.23956,0.159184,1.48343,-2.17554,0.193991,1.42754,-2.18275,0.166912,1.38172,-1.79604,-0.019123,1.4324,-1.89555,-0.008282,1.45513,-1.95042,-0.007439,1.42035,-2.00611,0.190983,1.36468,-2.06278,0.182629,1.07574,-1.73545,0.058072,1.04264,-1.77953,0.081371,0.962505,-1.80113,0.164846,0.854019,-1.78008,0.136296,0.672651,-1.78614,0.111049,0.603317,-1.80317,0.030534,0.711705,-1.79879,0.007239,0.644388,-1.8108,-0.028408,0.428381,-1.84321,-0.141286,0.369181,-1.92147,0.115253,0.383056,-1.87648,0.085192,0.335012,-1.80522,0.278608,1.108,-2.11407,0.322195,1.03702,-2.09759,0.156791,1.03376,-2.08874,0.202006,0.966255,-2.07409,0.30154,0.866591,-2.07195,0.1444,0.792733,-2.06079,0.135485,0.905505,-2.08242,0.066881,0.895572,-2.07873,-0.162292,0.841676,-2.05811,-0.242808,0.920447,-2.0055,-0.210957,0.638478,-2.01302,-0.256045,0.679958,-2.08106,0.091428,1.29194,-1.81689,0.107909,1.27611,-2.03091},
/*566*/{0.218708,1.84759,-1.71411,0.287456,1.79443,-1.86806,0.149777,1.69094,-1.61647,0.108891,1.57482,-1.55035,0.258006,1.56583,-1.61463,0.345702,1.55288,-1.63282,0.218611,1.87826,-2.11222,0.292445,1.80019,-1.94462,0.081882,1.85119,-2.02503,0.005254,1.72156,-2.1838,0.012157,1.62291,-2.24005,0.151346,1.48986,-2.17663,0.18596,1.43397,-2.18382,0.167318,1.38189,-1.79621,-0.018268,1.43433,-1.89485,-0.008583,1.4566,-1.95022,-0.006517,1.4223,-2.00614,0.190858,1.36775,-2.06316,0.178195,1.07684,-1.73456,0.049514,1.04483,-1.78026,0.071107,0.963049,-1.80113,0.149811,0.852542,-1.78196,0.121092,0.671703,-1.7869,0.094954,0.602933,-1.80375,0.015634,0.712421,-1.79938,-0.00956,0.645809,-1.81151,-0.049872,0.428241,-1.8414,-0.161538,0.370512,-1.92112,0.094892,0.382599,-1.87677,0.064916,0.335482,-1.80543,0.282893,1.11685,-2.1145,0.329629,1.04601,-2.09786,0.164058,1.03823,-2.08892,0.210355,0.971951,-2.07429,0.314774,0.875137,-2.07216,0.158689,0.794043,-2.06101,0.146588,0.9064,-2.0848,0.077833,0.89228,-2.07988,-0.149872,0.830809,-2.05808,-0.234543,0.903885,-2.00759,-0.187554,0.625168,-2.01349,-0.234113,0.663921,-2.0818,0.091857,1.29254,-1.81792,0.108178,1.27863,-2.03208},
/*567*/{0.220109,1.85118,-1.71467,0.286626,1.79801,-1.86945,0.155142,1.69396,-1.61584,0.118007,1.5771,-1.54962,0.264744,1.57192,-1.61641,0.35231,1.56084,-1.63637,0.215456,1.8819,-2.11251,0.291257,1.8035,-1.9458,0.080306,1.85274,-2.02268,-0.000626,1.72792,-2.18344,0.004464,1.62974,-2.24047,0.143809,1.49696,-2.17671,0.177954,1.44077,-2.18464,0.167266,1.38341,-1.79643,-0.016403,1.43815,-1.89486,-0.008004,1.45868,-1.94959,-0.006098,1.42403,-2.00608,0.190709,1.37092,-2.06304,0.170494,1.07377,-1.73747,0.040697,1.04723,-1.78104,0.060269,0.964783,-1.80171,0.132927,0.852056,-1.78365,0.104991,0.671554,-1.78786,0.076964,0.602981,-1.80426,0.00063,0.71354,-1.79963,-0.025601,0.647791,-1.81106,-0.069683,0.429553,-1.84162,-0.181578,0.372313,-1.92013,0.075562,0.382821,-1.87621,0.04528,0.335547,-1.80526,0.287183,1.12445,-2.11447,0.336572,1.05555,-2.09814,0.171043,1.04342,-2.08844,0.219666,0.977968,-2.07382,0.326566,0.884692,-2.07262,0.173828,0.796449,-2.06187,0.156394,0.907825,-2.08505,0.088023,0.892218,-2.08105,-0.132749,0.816859,-2.05698,-0.22599,0.887716,-2.00762,-0.16277,0.611602,-2.01538,-0.211922,0.647903,-2.08311,0.092249,1.29465,-1.81811,0.1083,1.2812,-2.03233},
/*568*/{0.22198,1.85506,-1.71549,0.286467,1.80085,-1.87118,0.161355,1.69705,-1.61533,0.125703,1.58,-1.5483,0.271792,1.57852,-1.61872,0.358928,1.56994,-1.64007,0.211627,1.88607,-2.11272,0.290183,1.80704,-1.94792,0.077966,1.85537,-2.0209,-0.006249,1.73449,-2.18299,-0.002032,1.63674,-2.24001,0.136203,1.50406,-2.17765,0.170685,1.44784,-2.18577,0.167752,1.38439,-1.79661,-0.017586,1.43869,-1.89455,-0.007848,1.46015,-1.94873,-0.006328,1.42812,-2.00633,0.190981,1.37432,-2.06329,0.162556,1.07169,-1.73912,0.033273,1.04992,-1.78081,0.04923,0.967027,-1.80243,0.119375,0.853737,-1.78553,0.08769,0.671378,-1.78859,0.059554,0.603593,-1.80579,-0.015804,0.715453,-1.80007,-0.042702,0.649634,-1.8121,-0.088347,0.432294,-1.84137,-0.201573,0.374861,-1.91987,0.055336,0.382748,-1.87654,0.024478,0.33511,-1.80606,0.292729,1.13267,-2.11459,0.342454,1.06474,-2.09802,0.178075,1.0479,-2.08802,0.227776,0.984848,-2.0742,0.337826,0.894796,-2.07256,0.188369,0.798915,-2.0625,0.165224,0.908906,-2.08549,0.096547,0.891508,-2.08183,-0.118613,0.80473,-2.05682,-0.217233,0.869449,-2.00598,-0.136573,0.598522,-2.01632,-0.189006,0.632069,-2.08409,0.092419,1.2953,-1.81949,0.10867,1.28456,-2.03385},
/*569*/{0.223606,1.85954,-1.71619,0.286357,1.8042,-1.87147,0.166707,1.70024,-1.61549,0.132807,1.58244,-1.54772,0.277356,1.58539,-1.6211,0.365066,1.57866,-1.64425,0.206401,1.88982,-2.11182,0.289927,1.81091,-1.9487,0.075359,1.85785,-2.01923,-0.011879,1.74195,-2.18195,-0.009065,1.64368,-2.24006,0.12886,1.51125,-2.17848,0.163464,1.45592,-2.18627,0.168532,1.3856,-1.79662,-0.015746,1.44113,-1.89426,-0.007024,1.46347,-1.94741,-0.004904,1.42942,-2.0058,0.190756,1.37792,-2.06269,0.156587,1.07171,-1.74094,0.026697,1.05233,-1.78039,0.039408,0.970107,-1.80439,0.105327,0.855188,-1.78713,0.070667,0.672977,-1.78973,0.042222,0.605257,-1.80625,-0.03156,0.717026,-1.80242,-0.059391,0.652725,-1.81235,-0.108694,0.432923,-1.84111,-0.221355,0.378312,-1.91952,0.034992,0.382322,-1.8763,0.004075,0.334967,-1.80609,0.296947,1.13996,-2.11373,0.348681,1.07441,-2.09784,0.184085,1.05326,-2.08799,0.236078,0.990602,-2.07342,0.34726,0.905192,-2.07291,0.202276,0.801872,-2.06408,0.173844,0.910233,-2.08599,0.106882,0.889293,-2.08197,-0.105338,0.793078,-2.05775,-0.205971,0.851018,-2.00527,-0.109791,0.586673,-2.01962,-0.164547,0.616038,-2.08526,0.09344,1.29697,-1.81968,0.109338,1.28742,-2.03411},
/*570*/{0.225549,1.86437,-1.71686,0.287444,1.80955,-1.87363,0.172717,1.70416,-1.61558,0.141641,1.58611,-1.54751,0.284059,1.59267,-1.62352,0.369315,1.58734,-1.64844,0.202634,1.89456,-2.11191,0.288658,1.81547,-1.9502,0.073901,1.85995,-2.01694,-0.017224,1.74908,-2.18123,-0.015775,1.65049,-2.23972,0.122711,1.51988,-2.17852,0.155976,1.46346,-2.18666,0.169554,1.38707,-1.79576,-0.015418,1.44358,-1.89291,-0.006603,1.4663,-1.94716,-0.004039,1.43276,-2.00511,0.192965,1.38292,-2.06244,0.149887,1.07394,-1.74377,0.019764,1.0554,-1.78012,0.029831,0.972515,-1.80253,0.094581,0.85705,-1.78849,0.05455,0.674557,-1.79051,0.02514,0.606656,-1.80774,-0.048351,0.719238,-1.80283,-0.0751,0.654359,-1.81284,-0.128323,0.434794,-1.84005,-0.241117,0.382311,-1.91882,0.015086,0.381658,-1.87573,-0.017025,0.334729,-1.80561,0.300974,1.148,-2.11455,0.354596,1.08359,-2.0978,0.190116,1.05865,-2.088,0.241629,0.997129,-2.0725,0.357197,0.915871,-2.07308,0.215582,0.804877,-2.06471,0.181262,0.911901,-2.08753,0.115376,0.886657,-2.08169,-0.092213,0.779762,-2.05825,-0.195468,0.831362,-2.00578,-0.083557,0.574132,-2.022,-0.140796,0.60062,-2.08743,0.09459,1.29865,-1.82039,0.111229,1.29176,-2.03487},
/*571*/{0.227375,1.86906,-1.71773,0.286792,1.81297,-1.87488,0.17888,1.70777,-1.61595,0.148285,1.5905,-1.54736,0.288471,1.59945,-1.62585,0.375129,1.59657,-1.65267,0.198665,1.89907,-2.1119,0.287793,1.81972,-1.9517,0.071491,1.86209,-2.01554,-0.022437,1.75614,-2.18065,-0.021801,1.65808,-2.23908,0.117228,1.52746,-2.17771,0.149841,1.47185,-2.18772,0.170738,1.38746,-1.79571,-0.013838,1.44594,-1.89215,-0.005174,1.46897,-1.94503,-0.003573,1.43637,-2.0052,0.193136,1.38691,-2.0614,0.144221,1.07454,-1.74596,0.010976,1.05712,-1.77998,0.020096,0.97623,-1.8029,0.083641,0.858978,-1.78985,0.038508,0.676168,-1.79172,0.007631,0.608668,-1.80685,-0.064646,0.721975,-1.80387,-0.092262,0.657913,-1.8133,-0.145949,0.437419,-1.83844,-0.260511,0.387647,-1.91778,-0.004561,0.380775,-1.8761,-0.037468,0.335034,-1.80627,0.304811,1.15575,-2.11452,0.358709,1.09235,-2.09764,0.19528,1.06393,-2.0879,0.249349,1.00442,-2.07293,0.36602,0.926394,-2.07297,0.228493,0.807713,-2.06614,0.188724,0.912761,-2.08748,0.124738,0.884311,-2.08182,-0.081292,0.768841,-2.06035,-0.185832,0.812007,-2.00498,-0.056593,0.563595,-2.02538,-0.116193,0.585683,-2.08927,0.0953,1.29966,-1.82122,0.111704,1.29544,-2.03579},
/*572*/{0.228731,1.87419,-1.71863,0.28628,1.81837,-1.87662,0.184502,1.71215,-1.61636,0.154457,1.59451,-1.54658,0.293035,1.60681,-1.62808,0.378699,1.60547,-1.6566,0.194919,1.90428,-2.1119,0.287424,1.82443,-1.95345,0.069283,1.8659,-2.01295,-0.026938,1.76289,-2.17934,-0.028339,1.66508,-2.2383,0.108987,1.5355,-2.17844,0.14345,1.4798,-2.18854,0.172613,1.38906,-1.79479,-0.012731,1.44926,-1.89047,-0.005015,1.47167,-1.94381,-0.002586,1.44063,-2.00541,0.193778,1.39114,-2.06044,0.13806,1.07319,-1.7478,0.004655,1.0616,-1.78064,0.010542,0.980758,-1.80304,0.073261,0.862037,-1.79241,0.021785,0.679092,-1.79245,-0.008876,0.612351,-1.80833,-0.080313,0.726248,-1.80489,-0.10926,0.660882,-1.81479,-0.1657,0.441331,-1.83903,-0.279507,0.394279,-1.91714,-0.023748,0.380387,-1.87606,-0.057697,0.335698,-1.80645,0.308166,1.16333,-2.11432,0.364241,1.09966,-2.09703,0.200429,1.06985,-2.08808,0.255871,1.01102,-2.0728,0.374143,0.936553,-2.07349,0.241385,0.810376,-2.06728,0.195959,0.913676,-2.08674,0.133362,0.881479,-2.08136,-0.062599,0.754432,-2.05794,-0.175008,0.792038,-2.00491,-0.029236,0.551305,-2.02767,-0.091247,0.571106,-2.09064,0.096264,1.30184,-1.82142,0.112536,1.29948,-2.03603},
/*573*/{0.230059,1.87961,-1.72003,0.286552,1.82287,-1.87761,0.190812,1.71663,-1.61722,0.161072,1.59882,-1.54681,0.297628,1.61454,-1.63052,0.382517,1.6145,-1.66081,0.19089,1.90867,-2.11114,0.285974,1.82922,-1.95492,0.06779,1.86933,-2.01054,-0.032053,1.76973,-2.17864,-0.034109,1.67231,-2.23719,0.102429,1.54398,-2.17861,0.13753,1.48819,-2.18904,0.172569,1.39048,-1.7947,-0.01092,1.45212,-1.89111,-0.002931,1.47498,-1.94332,-0.001599,1.44552,-2.00508,0.194741,1.39539,-2.05954,0.133059,1.07537,-1.75087,0.000143,1.06761,-1.77979,0.00195,0.985888,-1.80303,0.061933,0.863814,-1.79422,0.005464,0.682849,-1.79303,-0.025589,0.616185,-1.80853,-0.095763,0.730497,-1.80529,-0.126381,0.664297,-1.81628,-0.183229,0.445772,-1.8376,-0.29796,0.401763,-1.91641,-0.042795,0.379299,-1.87663,-0.078238,0.335946,-1.80694,0.311724,1.17147,-2.11432,0.367627,1.10995,-2.09763,0.205176,1.0764,-2.08827,0.261995,1.01876,-2.07311,0.380866,0.946434,-2.07339,0.25336,0.81411,-2.06881,0.202417,0.914538,-2.08675,0.141237,0.879441,-2.08155,-0.049122,0.741256,-2.05939,-0.160487,0.770322,-2.0025,-0.001034,0.540388,-2.03041,-0.06446,0.557218,-2.09336,0.096276,1.30371,-1.82253,0.113511,1.30374,-2.03708},
/*574*/{0.231359,1.88491,-1.72103,0.28764,1.82809,-1.87931,0.195364,1.72187,-1.61808,0.167568,1.60365,-1.54661,0.301183,1.62162,-1.63296,0.385908,1.62312,-1.66468,0.187409,1.91334,-2.11064,0.284946,1.83427,-1.95603,0.066909,1.87153,-2.01068,-0.036076,1.7759,-2.17791,-0.040125,1.67994,-2.23618,0.096943,1.55094,-2.17795,0.132237,1.49655,-2.18972,0.174266,1.39149,-1.79394,-0.008969,1.45562,-1.89058,-0.002218,1.47862,-1.9417,0.000559,1.44998,-2.00508,0.19639,1.40099,-2.05918,0.127656,1.0761,-1.7522,-0.007545,1.07498,-1.78185,-0.006545,0.990564,-1.80388,0.048849,0.866577,-1.79561,-0.009831,0.686622,-1.79375,-0.042486,0.620086,-1.80841,-0.11154,0.735032,-1.80619,-0.141427,0.671036,-1.81476,-0.200822,0.450702,-1.83786,-0.315144,0.408857,-1.91561,-0.062028,0.378102,-1.87618,-0.09863,0.336606,-1.80704,0.314899,1.17885,-2.11477,0.371954,1.11844,-2.09688,0.209624,1.08222,-2.08893,0.266886,1.02541,-2.07305,0.387543,0.956776,-2.07326,0.264598,0.817314,-2.07037,0.208957,0.914734,-2.08684,0.148977,0.87561,-2.0812,-0.036482,0.728721,-2.05934,-0.146604,0.748804,-2.00159,0.027403,0.530028,-2.03321,-0.038483,0.54415,-2.09546,0.097576,1.30555,-1.82382,0.115284,1.30889,-2.0383},
/*575*/{0.231915,1.89031,-1.72271,0.287353,1.83298,-1.88083,0.200195,1.72687,-1.61882,0.172798,1.60846,-1.54616,0.304302,1.62873,-1.63541,0.388702,1.63243,-1.66874,0.184638,1.91793,-2.11006,0.284894,1.83927,-1.95769,0.06414,1.87698,-2.00739,-0.04064,1.78252,-2.1773,-0.046037,1.68705,-2.23493,0.090996,1.55881,-2.17767,0.126975,1.50481,-2.19064,0.175958,1.39312,-1.7943,-0.007485,1.459,-1.89026,-0.00042,1.48265,-1.94056,0.001382,1.45465,-2.00496,0.197135,1.40568,-2.05802,0.122894,1.07713,-1.75548,-0.012502,1.0806,-1.78161,-0.014465,0.995788,-1.80389,0.034829,0.87024,-1.79647,-0.025405,0.691542,-1.79386,-0.05749,0.62512,-1.80829,-0.126026,0.740492,-1.80679,-0.156421,0.676945,-1.81467,-0.218491,0.456893,-1.83772,-0.333147,0.417831,-1.91552,-0.080822,0.376993,-1.87704,-0.11885,0.337155,-1.80719,0.317645,1.18631,-2.1144,0.375891,1.12698,-2.09624,0.213932,1.08919,-2.08922,0.271553,1.03266,-2.07348,0.392626,0.966437,-2.07267,0.275371,0.820061,-2.07136,0.213763,0.914846,-2.08662,0.156981,0.873101,-2.08061,-0.020878,0.714653,-2.05961,-0.132702,0.728706,-2.00146,0.056292,0.520177,-2.03757,-0.0098,0.531249,-2.09871,0.098464,1.3079,-1.82478,0.11615,1.31353,-2.03921},
/*576*/{0.232954,1.89585,-1.72419,0.287128,1.8385,-1.8827,0.204559,1.73242,-1.61971,0.17787,1.61322,-1.54631,0.307627,1.63633,-1.6376,0.390536,1.64106,-1.67255,0.181694,1.92253,-2.10955,0.284419,1.84437,-1.95887,0.065157,1.87885,-2.00592,-0.045077,1.78811,-2.17609,-0.051043,1.69382,-2.23272,0.086258,1.56706,-2.17675,0.122098,1.51235,-2.191,0.177721,1.39454,-1.79446,-0.005083,1.46284,-1.88989,0.000887,1.48639,-1.93948,0.003056,1.45992,-2.00476,0.19731,1.40978,-2.05698,0.115673,1.07905,-1.76041,-0.017866,1.08582,-1.78227,-0.022307,1.00224,-1.80398,0.021354,0.875994,-1.79692,-0.039876,0.697262,-1.79402,-0.073591,0.629893,-1.80808,-0.140339,0.746338,-1.8069,-0.172087,0.683283,-1.81599,-0.234393,0.462057,-1.83569,-0.349424,0.42733,-1.91457,-0.100767,0.376481,-1.87725,-0.139565,0.336895,-1.80809,0.321234,1.19369,-2.11374,0.378649,1.13456,-2.09555,0.217642,1.09583,-2.0896,0.276068,1.03875,-2.07331,0.395877,0.975288,-2.07245,0.28531,0.822913,-2.0731,0.21893,0.914467,-2.08599,0.164977,0.869713,-2.08035,-0.009312,0.699768,-2.059,-0.115962,0.706576,-1.99941,0.085368,0.512142,-2.04084,0.018197,0.519752,-2.10182,0.099137,1.3103,-1.8256,0.116575,1.31791,-2.03999},
/*577*/{0.234392,1.90156,-1.72535,0.287128,1.84216,-1.8843,0.207391,1.73777,-1.62047,0.183377,1.61904,-1.54656,0.310334,1.64319,-1.64015,0.393049,1.64878,-1.67633,0.17917,1.92684,-2.10901,0.283373,1.84914,-1.96055,0.063433,1.88202,-2.00397,-0.048666,1.79437,-2.17521,-0.056478,1.70083,-2.23168,0.080762,1.5745,-2.17618,0.117322,1.52043,-2.19148,0.179954,1.39616,-1.79475,-0.003526,1.46627,-1.88866,0.002932,1.49012,-1.93792,0.004913,1.46491,-2.00448,0.198506,1.41541,-2.05641,0.111765,1.08164,-1.76018,-0.021522,1.09195,-1.78226,-0.0295,1.00827,-1.8052,0.007769,0.881955,-1.79675,-0.054145,0.702909,-1.79341,-0.088049,0.636622,-1.80792,-0.153727,0.753129,-1.80726,-0.186363,0.688968,-1.81698,-0.250336,0.466834,-1.83398,-0.365807,0.437599,-1.91424,-0.120174,0.375889,-1.87696,-0.160683,0.337642,-1.80768,0.323638,1.20089,-2.11372,0.381924,1.14239,-2.09554,0.220683,1.1028,-2.09061,0.279566,1.04541,-2.07299,0.398011,0.983718,-2.07272,0.2956,0.825188,-2.07399,0.223703,0.913256,-2.0856,0.171933,0.865345,-2.08028,0.013558,0.688483,-2.06049,-0.100762,0.686516,-1.99925,0.113852,0.50456,-2.0446,0.046105,0.508924,-2.10471,0.100915,1.31253,-1.82692,0.118132,1.32328,-2.04119},
/*578*/{0.234531,1.90626,-1.72681,0.287392,1.84756,-1.8861,0.211219,1.74318,-1.62103,0.186192,1.62399,-1.54667,0.311881,1.65052,-1.64277,0.394359,1.65712,-1.67995,0.17721,1.9309,-2.10916,0.281932,1.85413,-1.96217,0.062941,1.88561,-2.00351,-0.05136,1.80031,-2.17449,-0.061569,1.70746,-2.22958,0.077259,1.58199,-2.17542,0.113133,1.52712,-2.19205,0.181566,1.39808,-1.79535,-0.001225,1.47,-1.8884,0.004952,1.49416,-1.93689,0.007118,1.47006,-2.00462,0.199254,1.41992,-2.05628,0.109665,1.0839,-1.76142,-0.025829,1.09829,-1.78133,-0.035939,1.01495,-1.80451,-0.003838,0.888524,-1.79703,-0.067624,0.708937,-1.79338,-0.101529,0.642728,-1.80735,-0.167318,0.760089,-1.80785,-0.199188,0.69609,-1.81684,-0.265322,0.474303,-1.83319,-0.380951,0.448022,-1.91375,-0.138979,0.37561,-1.8768,-0.181629,0.337858,-1.80804,0.325752,1.20755,-2.1138,0.385052,1.1502,-2.09418,0.223997,1.10908,-2.09145,0.282786,1.05262,-2.07296,0.399181,0.990954,-2.07244,0.304784,0.827659,-2.07529,0.227447,0.910463,-2.08422,0.179214,0.859222,-2.07896,0.033147,0.674935,-2.05971,-0.081884,0.664145,-1.99833,0.143221,0.498177,-2.0489,0.074482,0.498984,-2.10764,0.102048,1.31514,-1.8282,0.119074,1.328,-2.04237},
/*579*/{0.235947,1.91157,-1.72811,0.286572,1.85189,-1.88721,0.214584,1.748,-1.62188,0.190701,1.62996,-1.54681,0.314354,1.6571,-1.64481,0.396278,1.66423,-1.68322,0.174761,1.93438,-2.10865,0.281772,1.85864,-1.9638,0.061904,1.88878,-2.00193,-0.055331,1.80569,-2.17233,-0.065648,1.71387,-2.22774,0.071926,1.5885,-2.17536,0.109339,1.53456,-2.1926,0.184161,1.39944,-1.79651,0.001038,1.47365,-1.88815,0.006306,1.49858,-1.93618,0.008269,1.47508,-2.00456,0.199201,1.42457,-2.0561,0.105806,1.08684,-1.76264,-0.029375,1.10536,-1.782,-0.042836,1.02237,-1.80555,-0.014839,0.894639,-1.79755,-0.080076,0.715644,-1.79277,-0.114708,0.649105,-1.80633,-0.180134,0.767337,-1.80823,-0.212863,0.70426,-1.81663,-0.281736,0.479462,-1.83204,-0.396175,0.459703,-1.91302,-0.157618,0.375143,-1.87737,-0.201698,0.338876,-1.80765,0.327542,1.21374,-2.11367,0.386472,1.15601,-2.09291,0.226711,1.1144,-2.09202,0.285312,1.05882,-2.07283,0.40012,0.997752,-2.07251,0.314054,0.829614,-2.0768,0.231995,0.907733,-2.08337,0.186367,0.853749,-2.07866,0.051504,0.66099,-2.05909,-0.062655,0.641615,-1.99621,0.172491,0.49234,-2.05268,0.102623,0.489299,-2.1107,0.103291,1.3176,-1.82986,0.119312,1.33281,-2.04396},
/*580*/{0.236608,1.91619,-1.72956,0.287282,1.85596,-1.88858,0.216563,1.75336,-1.62231,0.194083,1.63409,-1.5471,0.315395,1.6638,-1.64757,0.396736,1.67164,-1.68597,0.173463,1.93793,-2.1089,0.281427,1.86264,-1.96485,0.06119,1.89201,-2.00107,-0.058845,1.81056,-2.17114,-0.070605,1.71976,-2.22586,0.068939,1.59526,-2.17461,0.105269,1.54084,-2.19306,0.186844,1.40084,-1.79737,0.00272,1.4778,-1.88782,0.008079,1.50247,-1.93493,0.010827,1.48029,-2.00424,0.20007,1.42938,-2.05623,0.100581,1.09037,-1.76645,-0.032298,1.11097,-1.78021,-0.050162,1.0289,-1.80559,-0.023656,0.901447,-1.79729,-0.091733,0.722175,-1.79235,-0.127559,0.656299,-1.80482,-0.190652,0.774976,-1.80874,-0.224209,0.711338,-1.81682,-0.295067,0.486671,-1.8301,-0.409493,0.471792,-1.9126,-0.176843,0.375105,-1.87752,-0.221938,0.340631,-1.80874,0.329833,1.22,-2.11273,0.388584,1.16226,-2.09217,0.228253,1.1198,-2.09264,0.286622,1.06471,-2.07309,0.398693,1.00395,-2.07277,0.322727,0.830272,-2.07776,0.236365,0.904346,-2.08226,0.194633,0.847593,-2.07754,0.065672,0.646996,-2.05939,-0.042209,0.620402,-1.99449,0.201187,0.487695,-2.0557,0.131992,0.480916,-2.11343,0.104798,1.32006,-1.83146,0.120244,1.33784,-2.0454},
/*581*/{0.237361,1.92137,-1.73061,0.28759,1.86084,-1.89027,0.21848,1.75842,-1.62274,0.196853,1.6392,-1.54721,0.316555,1.66996,-1.64948,0.397146,1.67903,-1.69001,0.171696,1.94118,-2.10866,0.280554,1.86667,-1.9657,0.060497,1.89508,-2.00065,-0.060883,1.81478,-2.16722,-0.074739,1.72552,-2.22315,0.06485,1.60101,-2.17388,0.102065,1.54706,-2.19318,0.188887,1.40298,-1.79887,0.004848,1.48094,-1.88752,0.008814,1.50608,-1.93312,0.012734,1.48482,-2.00419,0.199823,1.43375,-2.05611,0.096806,1.09223,-1.7669,-0.035379,1.11737,-1.78038,-0.054809,1.03672,-1.80546,-0.032293,0.907377,-1.79904,-0.103288,0.728982,-1.79154,-0.138126,0.663643,-1.80436,-0.201002,0.781623,-1.80735,-0.235074,0.719236,-1.81601,-0.307262,0.493695,-1.82808,-0.42279,0.484828,-1.91118,-0.195521,0.37552,-1.87743,-0.242808,0.341522,-1.80857,0.33148,1.22518,-2.11238,0.389971,1.16672,-2.09101,0.229443,1.1247,-2.09398,0.28718,1.06946,-2.07393,0.397925,1.00903,-2.07279,0.330915,0.830725,-2.07855,0.241141,0.899985,-2.08176,0.201549,0.841689,-2.07699,0.081316,0.63246,-2.05883,-0.02139,0.599352,-1.99392,0.230015,0.483596,-2.05903,0.160687,0.473458,-2.11628,0.106353,1.32257,-1.83275,0.120642,1.34235,-2.0466},
/*582*/{0.237607,1.9256,-1.73156,0.288312,1.86537,-1.89204,0.220133,1.76321,-1.62328,0.200754,1.64423,-1.54739,0.317321,1.67614,-1.6509,0.398126,1.68604,-1.69298,0.170449,1.94421,-2.10882,0.28037,1.87035,-1.96746,0.060819,1.89654,-2,-0.061135,1.81905,-2.16605,-0.078224,1.73068,-2.22148,0.061509,1.60609,-2.17307,0.099339,1.55294,-2.19373,0.191257,1.405,-1.79998,0.007487,1.48475,-1.88631,0.010271,1.50973,-1.93251,0.014372,1.48976,-2.00363,0.200835,1.43875,-2.05653,0.094016,1.0956,-1.76674,-0.040334,1.12541,-1.78126,-0.059045,1.04306,-1.80394,-0.03963,0.9135,-1.79959,-0.112324,0.735968,-1.79106,-0.1499,0.670441,-1.80254,-0.210994,0.789974,-1.80875,-0.244724,0.72632,-1.816,-0.318578,0.500836,-1.82664,-0.433649,0.498856,-1.90985,-0.214184,0.375722,-1.87713,-0.26316,0.343629,-1.80878,0.332117,1.22975,-2.11147,0.390377,1.17165,-2.08985,0.229911,1.12935,-2.09422,0.287694,1.0742,-2.07511,0.396229,1.01216,-2.07285,0.338978,0.830624,-2.07957,0.246213,0.895643,-2.08071,0.20901,0.834969,-2.07568,0.103107,0.621476,-2.05878,0.000926,0.579172,-1.99244,0.258072,0.480144,-2.06146,0.188833,0.467005,-2.11812,0.108359,1.32528,-1.83422,0.121624,1.34736,-2.04791},
/*583*/{0.239113,1.92926,-1.73353,0.287757,1.86865,-1.89324,0.220884,1.76753,-1.62375,0.20246,1.64833,-1.54704,0.318792,1.6805,-1.65293,0.398562,1.69127,-1.69553,0.169137,1.94678,-2.10865,0.279738,1.87455,-1.96773,0.060846,1.89967,-1.99919,-0.061334,1.82406,-2.16462,-0.08145,1.73587,-2.21901,0.059309,1.61134,-2.17237,0.096447,1.55797,-2.19396,0.193795,1.4067,-1.80175,0.009302,1.48881,-1.88553,0.012635,1.514,-1.93126,0.0161,1.4943,-2.00291,0.20034,1.44258,-2.05648,0.091123,1.09806,-1.76692,-0.043492,1.13156,-1.78155,-0.064537,1.05038,-1.80481,-0.047106,0.91937,-1.79995,-0.122092,0.742421,-1.79042,-0.156714,0.677559,-1.80147,-0.219541,0.796438,-1.80825,-0.254347,0.733231,-1.81549,-0.330837,0.507652,-1.8248,-0.443638,0.512421,-1.90931,-0.231543,0.376627,-1.87655,-0.283237,0.346632,-1.80917,0.333879,1.23384,-2.111,0.391277,1.17514,-2.08893,0.229971,1.13419,-2.09509,0.287495,1.07786,-2.07683,0.395898,1.01547,-2.07414,0.346328,0.830185,-2.08096,0.250269,0.889753,-2.08022,0.217633,0.827019,-2.07534,0.123082,0.608391,-2.05855,0.024003,0.559354,-1.99057,0.285349,0.478,-2.06319,0.216887,0.460973,-2.11944,0.109614,1.32797,-1.83532,0.121258,1.35163,-2.04893},
/*584*/{0.240359,1.93421,-1.73502,0.288591,1.87299,-1.89401,0.222135,1.77261,-1.62397,0.204831,1.65262,-1.54764,0.319567,1.68576,-1.654,0.398383,1.69693,-1.69833,0.168676,1.94957,-2.1084,0.279764,1.87744,-1.96895,0.060992,1.90112,-1.99827,-0.063696,1.8283,-2.16384,-0.084719,1.74086,-2.21618,0.057012,1.6162,-2.17226,0.094508,1.56268,-2.1938,0.195745,1.40953,-1.80222,0.011053,1.49211,-1.88542,0.014276,1.51681,-1.92975,0.017711,1.49879,-2.00265,0.20095,1.4468,-2.05715,0.087108,1.10091,-1.76799,-0.04297,1.13537,-1.77971,-0.068464,1.05765,-1.8055,-0.053463,0.925871,-1.80063,-0.128656,0.749698,-1.79059,-0.166484,0.683865,-1.80025,-0.227272,0.80301,-1.80799,-0.261721,0.741313,-1.81407,-0.340796,0.51518,-1.82294,-0.452304,0.529049,-1.90646,-0.249141,0.378374,-1.87584,-0.301614,0.350175,-1.80825,0.334003,1.23758,-2.10962,0.39056,1.17799,-2.08797,0.229613,1.13881,-2.09597,0.287006,1.08064,-2.07767,0.394367,1.01708,-2.07485,0.35416,0.829537,-2.082,0.255238,0.883765,-2.08024,0.225377,0.819615,-2.07462,0.143191,0.596746,-2.05868,0.047063,0.540041,-1.99055,0.312678,0.476299,-2.06477,0.244438,0.454667,-2.12086,0.111497,1.33086,-1.83618,0.122231,1.35587,-2.04968},
/*585*/{0.241191,1.93696,-1.73598,0.288403,1.87616,-1.89556,0.223329,1.77694,-1.62479,0.206548,1.65664,-1.54724,0.319618,1.68997,-1.65567,0.397335,1.70238,-1.70183,0.168646,1.95112,-2.10815,0.279924,1.88075,-1.96972,0.060486,1.90365,-1.99818,-0.067293,1.83338,-2.15635,-0.086713,1.74477,-2.21402,0.055825,1.62049,-2.17201,0.092696,1.56688,-2.19367,0.198052,1.41088,-1.80409,0.012739,1.496,-1.88456,0.015474,1.52053,-1.9293,0.019369,1.50251,-2.00181,0.20164,1.45022,-2.05719,0.085559,1.10406,-1.76821,-0.044251,1.14111,-1.77919,-0.071977,1.06235,-1.80591,-0.059278,0.931571,-1.80208,-0.136243,0.75575,-1.79039,-0.173817,0.690991,-1.79974,-0.233278,0.810923,-1.80755,-0.268183,0.748434,-1.8121,-0.350613,0.521593,-1.82085,-0.460412,0.542919,-1.90458,-0.266337,0.379198,-1.87526,-0.321714,0.353746,-1.80821,0.333679,1.24042,-2.10911,0.390434,1.17954,-2.08759,0.228999,1.14206,-2.09675,0.286036,1.08264,-2.07863,0.39326,1.01784,-2.07609,0.361323,0.828568,-2.08268,0.260385,0.877467,-2.08004,0.233276,0.811873,-2.07458,0.159565,0.583906,-2.05781,0.069718,0.522399,-1.99105,0.33882,0.474957,-2.0653,0.272204,0.450286,-2.1222,0.112723,1.33322,-1.8373,0.122345,1.35971,-2.05068},
/*586*/{0.24163,1.94034,-1.73735,0.289463,1.87919,-1.8964,0.223755,1.78098,-1.62494,0.207629,1.65992,-1.5479,0.319435,1.69392,-1.65758,0.397564,1.70642,-1.7039,0.166547,1.95401,-2.10844,0.279578,1.8833,-1.97102,0.061295,1.90608,-1.99736,-0.065388,1.83797,-2.15399,-0.088746,1.74864,-2.21185,0.054543,1.62339,-2.17158,0.091141,1.57066,-2.19411,0.199108,1.4133,-1.80406,0.014153,1.4994,-1.88435,0.016403,1.52382,-1.92797,0.020499,1.50671,-2.00038,0.201457,1.45316,-2.05748,0.083446,1.10807,-1.76804,-0.045242,1.1455,-1.77896,-0.075088,1.0692,-1.80724,-0.063667,0.936581,-1.80326,-0.14283,0.761818,-1.7903,-0.180578,0.696976,-1.80023,-0.239176,0.817246,-1.80696,-0.274674,0.755554,-1.81272,-0.357619,0.530133,-1.81967,-0.466256,0.558355,-1.90167,-0.283985,0.381221,-1.87494,-0.341178,0.359198,-1.80826,0.333876,1.24241,-2.10751,0.388889,1.18125,-2.08722,0.227684,1.14467,-2.09749,0.28298,1.0859,-2.08228,0.391885,1.01778,-2.07711,0.367936,0.826901,-2.08396,0.264104,0.869821,-2.07996,0.240988,0.804251,-2.07415,0.182573,0.574119,-2.0586,0.091882,0.505721,-1.99051,0.364754,0.474813,-2.06539,0.299076,0.445721,-2.12233,0.113287,1.33604,-1.83731,0.121985,1.36314,-2.05066},
/*587*/{0.242921,1.94303,-1.7379,0.289486,1.8821,-1.89736,0.224193,1.78505,-1.62563,0.208215,1.66313,-1.54713,0.319795,1.69744,-1.6586,0.397013,1.71028,-1.7061,0.166985,1.95559,-2.10866,0.279435,1.88576,-1.97115,0.060745,1.90832,-1.99713,-0.066835,1.84132,-2.15136,-0.090584,1.75205,-2.20935,0.053034,1.62665,-2.17084,0.090017,1.57396,-2.19395,0.200624,1.41515,-1.80486,0.015737,1.50259,-1.88254,0.017828,1.52769,-1.92697,0.020592,1.51041,-1.99876,0.202088,1.45686,-2.05719,0.080503,1.11008,-1.7682,-0.047382,1.15177,-1.78007,-0.076283,1.07312,-1.80602,-0.066222,0.941742,-1.8044,-0.14626,0.768144,-1.79062,-0.185778,0.703215,-1.79858,-0.243573,0.824229,-1.8061,-0.279333,0.762186,-1.81114,-0.368258,0.53826,-1.81702,-0.471966,0.573855,-1.8989,-0.300656,0.383681,-1.87503,-0.359441,0.365168,-1.8084,0.333044,1.24324,-2.10689,0.38841,1.18153,-2.08695,0.226099,1.1464,-2.09761,0.280234,1.08727,-2.08272,0.389759,1.01749,-2.07862,0.374286,0.825186,-2.08532,0.268162,0.863511,-2.07878,0.248289,0.796514,-2.07402,0.201438,0.563484,-2.05872,0.116687,0.490941,-1.99255,0.389464,0.474647,-2.06499,0.325762,0.44176,-2.12145,0.11437,1.33853,-1.83768,0.121901,1.3671,-2.05088},
/*588*/{0.244078,1.94625,-1.73969,0.290625,1.88523,-1.89833,0.225066,1.78838,-1.62553,0.208593,1.66655,-1.54796,0.31963,1.70086,-1.6604,0.396716,1.71334,-1.70809,0.166922,1.95781,-2.109,0.280174,1.88792,-1.97277,0.060765,1.91035,-1.99696,-0.067468,1.84291,-2.1486,-0.091696,1.75509,-2.20729,0.052276,1.62907,-2.17072,0.089816,1.5771,-2.19411,0.202015,1.41794,-1.80529,0.01706,1.50612,-1.88187,0.018623,1.53108,-1.9253,0.021599,1.51391,-1.99789,0.202203,1.45895,-2.05695,0.080056,1.11281,-1.76969,-0.046821,1.15479,-1.77998,-0.077643,1.07851,-1.8071,-0.068435,0.946177,-1.8063,-0.151001,0.773818,-1.79133,-0.190047,0.70866,-1.79794,-0.24713,0.830487,-1.80604,-0.283511,0.76801,-1.80857,-0.373139,0.547255,-1.81455,-0.47623,0.589551,-1.89511,-0.317678,0.387256,-1.875,-0.376438,0.37169,-1.80846,0.331279,1.24344,-2.10674,0.385632,1.18063,-2.08719,0.223206,1.14822,-2.09832,0.27752,1.08722,-2.08322,0.387616,1.01604,-2.07974,0.380338,0.822991,-2.08606,0.272777,0.856083,-2.07937,0.255652,0.788236,-2.07336,0.217004,0.551867,-2.05896,0.139022,0.475059,-1.99189,0.413092,0.47543,-2.0638,0.351633,0.438335,-2.12082,0.115114,1.34159,-1.83703,0.121493,1.36985,-2.05031},
/*589*/{0.245403,1.94875,-1.74036,0.290895,1.88737,-1.89949,0.225902,1.79114,-1.62661,0.208735,1.66926,-1.5472,0.319026,1.7033,-1.66101,0.396288,1.71593,-1.71004,0.166516,1.95944,-2.1098,0.280283,1.89029,-1.97332,0.062737,1.91174,-1.99735,-0.066445,1.84629,-2.14707,-0.092432,1.75747,-2.20562,0.051836,1.63151,-2.17015,0.089281,1.57917,-2.19395,0.202875,1.41973,-1.8058,0.017852,1.50876,-1.88077,0.019217,1.53487,-1.9245,0.020929,1.51726,-1.99639,0.201985,1.46167,-2.05645,0.078494,1.1153,-1.76975,-0.04774,1.15867,-1.78006,-0.078336,1.08245,-1.80806,-0.069909,0.949666,-1.80741,-0.154176,0.777895,-1.79229,-0.195251,0.714106,-1.79831,-0.249139,0.836058,-1.80476,-0.286194,0.775523,-1.80687,-0.379199,0.554714,-1.81333,-0.479499,0.605634,-1.88824,-0.333322,0.393708,-1.87649,-0.393101,0.379479,-1.81023,0.329758,1.24278,-2.10698,0.383166,1.17966,-2.08767,0.220777,1.14823,-2.09871,0.274093,1.08737,-2.08459,0.384942,1.01454,-2.08143,0.385394,0.820542,-2.08763,0.276417,0.848959,-2.07986,0.262095,0.780078,-2.07337,0.239174,0.542978,-2.05981,0.160955,0.462597,-1.99305,0.435466,0.476035,-2.06281,0.37722,0.435988,-2.11959,0.115247,1.34397,-1.83704,0.120626,1.37304,-2.05024},
/*590*/{0.246221,1.95043,-1.74138,0.291733,1.88953,-1.90052,0.226146,1.79408,-1.62679,0.209409,1.67205,-1.54702,0.319151,1.70547,-1.66319,0.395015,1.71872,-1.71301,0.166213,1.96087,-2.10962,0.281234,1.89147,-1.97486,0.062885,1.91334,-1.9973,-0.066702,1.84821,-2.14459,-0.092666,1.7592,-2.20376,0.051884,1.63293,-2.17011,0.089519,1.58114,-2.19421,0.20323,1.42208,-1.80553,0.019217,1.51167,-1.87977,0.019212,1.53707,-1.92446,0.020783,1.52065,-1.99471,0.202107,1.46423,-2.05638,0.079128,1.11894,-1.7717,-0.049616,1.161,-1.78223,-0.078059,1.08423,-1.80807,-0.069812,0.951875,-1.80855,-0.156558,0.780847,-1.79314,-0.198432,0.718768,-1.79895,-0.24998,0.841672,-1.80415,-0.288595,0.781215,-1.8048,-0.384239,0.563264,-1.8107,-0.48304,0.620449,-1.88258,-0.349034,0.400892,-1.87897,-0.407355,0.387538,-1.81217,0.32726,1.241,-2.10751,0.379891,1.17716,-2.08838,0.217077,1.14764,-2.09931,0.270231,1.08655,-2.08504,0.381776,1.01196,-2.0823,0.390188,0.818178,-2.08879,0.27869,0.84219,-2.08195,0.269151,0.772462,-2.07496,0.25838,0.534688,-2.06035,0.181769,0.44904,-1.99268,0.456044,0.476794,-2.06193,0.401591,0.433319,-2.11807,0.115634,1.34651,-1.83666,0.120345,1.3758,-2.04984},
/*591*/{0.247302,1.95236,-1.74245,0.29078,1.89112,-1.90158,0.226349,1.79632,-1.62718,0.210087,1.67393,-1.54649,0.318266,1.70628,-1.66261,0.39443,1.71938,-1.71342,0.166654,1.96153,-2.11035,0.282019,1.89236,-1.97545,0.064422,1.91541,-1.99757,-0.066745,1.84882,-2.14248,-0.093094,1.76026,-2.2025,0.051965,1.63417,-2.16947,0.089693,1.5825,-2.19379,0.203589,1.42403,-1.80503,0.0198,1.51373,-1.87848,0.019949,1.53937,-1.92363,0.013628,1.52789,-1.99518,0.202219,1.4667,-2.05598,0.081185,1.1185,-1.77211,-0.047179,1.16482,-1.78399,-0.07688,1.08631,-1.80921,-0.069131,0.953464,-1.81003,-0.157279,0.784124,-1.79363,-0.202117,0.722231,-1.79974,-0.249686,0.847882,-1.80279,-0.290829,0.788393,-1.80514,-0.389945,0.571421,-1.8085,-0.4867,0.635634,-1.87418,-0.365318,0.408574,-1.88291,-0.421982,0.395802,-1.8139,0.323971,1.23878,-2.10788,0.376425,1.17416,-2.08906,0.213642,1.14688,-2.09997,0.265652,1.08569,-2.08523,0.378736,1.00946,-2.08418,0.394276,0.816008,-2.08942,0.281711,0.833841,-2.08323,0.275299,0.764507,-2.07645,0.275493,0.526519,-2.06083,0.201835,0.438256,-1.99369,0.47615,0.476897,-2.05983,0.424022,0.429436,-2.11518,0.114526,1.34909,-1.83708,0.118247,1.37957,-2.05011},
/*592*/{0.248631,1.95369,-1.74282,0.291907,1.89279,-1.90272,0.226524,1.79838,-1.6278,0.209281,1.67565,-1.54689,0.317681,1.70773,-1.66247,0.392976,1.71995,-1.71526,0.168578,1.96411,-2.11157,0.282353,1.89331,-1.97633,0.063537,1.917,-1.99777,-0.066948,1.84984,-2.14092,-0.093269,1.76028,-2.20115,0.053767,1.63552,-2.16917,0.090317,1.58356,-2.19413,0.203497,1.4259,-1.80519,0.020943,1.51546,-1.87692,0.020631,1.54204,-1.92238,0.023472,1.52389,-1.99198,0.202779,1.46808,-2.05604,0.083144,1.1198,-1.7742,-0.046381,1.16363,-1.78363,-0.075307,1.08755,-1.80964,-0.06681,0.954293,-1.81124,-0.157603,0.786645,-1.79487,-0.203309,0.725036,-1.80162,-0.249352,0.851742,-1.80124,-0.29082,0.793203,-1.80399,-0.396595,0.580247,-1.80835,-0.490298,0.649801,-1.86623,-0.38259,0.417257,-1.88741,-0.437083,0.405292,-1.81607,0.321137,1.23592,-2.10797,0.37247,1.17113,-2.08991,0.208577,1.14545,-2.10105,0.260093,1.08387,-2.08586,0.375588,1.00638,-2.08469,0.397827,0.813319,-2.09059,0.285133,0.82667,-2.0857,0.280568,0.756842,-2.07876,0.292577,0.518917,-2.06183,0.222263,0.429711,-1.99354,0.495636,0.477805,-2.05693,0.445124,0.427304,-2.11313,0.116831,1.35016,-1.83538,0.12064,1.37984,-2.04852},
/*593*/{0.24934,1.95465,-1.74383,0.292012,1.89411,-1.90398,0.226203,1.80025,-1.62799,0.209057,1.67802,-1.54715,0.316435,1.70789,-1.66348,0.391856,1.71948,-1.71664,0.169104,1.96461,-2.11224,0.28266,1.89377,-1.97774,0.064828,1.91848,-1.99741,-0.06551,1.85016,-2.13984,-0.093137,1.76032,-2.19992,0.053949,1.6357,-2.16839,0.091871,1.58411,-2.19411,0.204142,1.42727,-1.80457,0.021157,1.51772,-1.87555,0.020165,1.5435,-1.92205,0.024339,1.52418,-1.99188,0.203641,1.46988,-2.05534,0.086831,1.11966,-1.77579,-0.041206,1.16431,-1.78396,-0.071853,1.08676,-1.81062,-0.062967,0.954102,-1.81201,-0.156781,0.78717,-1.79599,-0.20379,0.728541,-1.80223,-0.246917,0.855261,-1.80114,-0.289599,0.798343,-1.80111,-0.398745,0.587922,-1.80729,-0.495266,0.663703,-1.85786,-0.396896,0.427791,-1.8921,-0.447987,0.415234,-1.8198,0.317799,1.23307,-2.10883,0.368267,1.16703,-2.09042,0.204467,1.14434,-2.10151,0.255218,1.08144,-2.08585,0.371265,1.00243,-2.08565,0.401305,0.811588,-2.09023,0.287718,0.819974,-2.08921,0.28674,0.75022,-2.08182,0.309102,0.51273,-2.06177,0.241275,0.42095,-1.99408,0.512735,0.478307,-2.05503,0.465439,0.424619,-2.10979,0.117511,1.35177,-1.83455,0.121449,1.38119,-2.04773},
/*594*/{0.250429,1.95496,-1.74475,0.292577,1.89474,-1.90442,0.225622,1.80117,-1.62876,0.207125,1.67805,-1.54681,0.315587,1.70748,-1.66486,0.390682,1.719,-1.71778,0.170409,1.96508,-2.11362,0.283065,1.8935,-1.97791,0.065044,1.91927,-1.99832,-0.065322,1.84913,-2.1379,-0.092339,1.75956,-2.19884,0.055127,1.63513,-2.16813,0.093234,1.58372,-2.19431,0.204925,1.42859,-1.8054,0.020931,1.51813,-1.87514,0.020373,1.54486,-1.92154,0.024399,1.52634,-1.99089,0.203262,1.47087,-2.05466,0.090176,1.11979,-1.77766,-0.040661,1.16216,-1.78493,-0.069075,1.08596,-1.81095,-0.058563,0.953195,-1.8132,-0.156771,0.787437,-1.79745,-0.205693,0.730434,-1.80336,-0.243436,0.859531,-1.7987,-0.28819,0.803078,-1.79944,-0.405043,0.596657,-1.80685,-0.500366,0.677277,-1.85027,-0.410544,0.439556,-1.89732,-0.459611,0.426058,-1.82257,0.314833,1.22901,-2.10955,0.364203,1.16269,-2.09126,0.199732,1.14225,-2.10186,0.249868,1.07893,-2.08613,0.36749,0.998686,-2.08673,0.404632,0.80957,-2.08987,0.290626,0.8119,-2.09184,0.293082,0.743495,-2.0859,0.324526,0.506722,-2.06273,0.257931,0.412394,-1.99484,0.528798,0.478997,-2.05298,0.484378,0.422643,-2.10713,0.117849,1.35282,-1.83457,0.121131,1.38267,-2.0477},
/*595*/{0.250897,1.95496,-1.74554,0.292325,1.89482,-1.90554,0.225569,1.80174,-1.62937,0.205951,1.6796,-1.54756,0.314535,1.70809,-1.66652,0.390616,1.71852,-1.71914,0.170694,1.96486,-2.11415,0.283388,1.89365,-1.97882,0.067045,1.92009,-1.99982,-0.064705,1.84825,-2.13722,-0.092139,1.75852,-2.19863,0.057035,1.63488,-2.16821,0.094603,1.58315,-2.19429,0.204775,1.42961,-1.80546,0.020658,1.51936,-1.87346,0.019812,1.54606,-1.92102,0.02345,1.52668,-1.99013,0.204231,1.47217,-2.0544,0.095293,1.11905,-1.77913,-0.037373,1.16059,-1.78499,-0.063833,1.08428,-1.81234,-0.05301,0.95136,-1.81406,-0.15512,0.787197,-1.79907,-0.206767,0.732067,-1.80569,-0.238974,0.862079,-1.79686,-0.287751,0.807739,-1.79891,-0.411583,0.60505,-1.80663,-0.504543,0.690015,-1.84217,-0.422217,0.453507,-1.90168,-0.468931,0.437104,-1.82556,0.310416,1.22489,-2.10999,0.360201,1.158,-2.09212,0.194711,1.1407,-2.10053,0.24377,1.07546,-2.08677,0.362907,0.994187,-2.08791,0.40758,0.807509,-2.08847,0.293599,0.807169,-2.09618,0.298239,0.736211,-2.08827,0.337077,0.49964,-2.06325,0.274132,0.406641,-1.99504,0.543647,0.480761,-2.04952,0.501286,0.421307,-2.10422,0.118085,1.3538,-1.83409,0.121156,1.38387,-2.04719},
/*596*/{0.251427,1.95504,-1.74637,0.292715,1.895,-1.90628,0.225354,1.80173,-1.63047,0.204379,1.67952,-1.54758,0.312756,1.70619,-1.66659,0.389983,1.71762,-1.71955,0.171244,1.96482,-2.11503,0.28365,1.89357,-1.97947,0.06752,1.9205,-2.00091,-0.064576,1.84676,-2.13542,-0.090664,1.75699,-2.198,0.058454,1.63349,-2.16792,0.096115,1.58231,-2.19446,0.205092,1.43021,-1.80508,0.020461,1.51969,-1.87336,0.018776,1.54641,-1.92016,0.023173,1.52664,-1.98984,0.204295,1.47232,-2.05376,0.099291,1.11845,-1.78173,-0.032472,1.1579,-1.78615,-0.059488,1.08071,-1.81258,-0.047136,0.948556,-1.8158,-0.153592,0.786764,-1.80146,-0.207627,0.733595,-1.80776,-0.235204,0.865461,-1.79504,-0.284209,0.813194,-1.79599,-0.412433,0.6137,-1.81081,-0.509181,0.702807,-1.83556,-0.432823,0.466259,-1.90623,-0.47789,0.448766,-1.82872,0.307298,1.22101,-2.11067,0.355233,1.15299,-2.09305,0.189663,1.13707,-2.09996,0.239217,1.07117,-2.086,0.357578,0.989883,-2.08807,0.411063,0.805461,-2.0866,0.297645,0.801454,-2.09889,0.304955,0.731668,-2.09094,0.350097,0.494955,-2.0631,0.287637,0.400916,-1.99585,0.556249,0.482393,-2.04754,0.518125,0.420917,-2.10096,0.118231,1.3543,-1.83339,0.121166,1.38401,-2.04655},
/*597*/{0.25277,1.95446,-1.74707,0.29305,1.89477,-1.90674,0.223921,1.80181,-1.63123,0.201932,1.67912,-1.54778,0.311539,1.70471,-1.66691,0.386052,1.71228,-1.72048,0.171997,1.96447,-2.1171,0.284484,1.89352,-1.97996,0.067984,1.92035,-2.00175,-0.063593,1.84368,-2.13451,-0.089321,1.755,-2.19717,0.059989,1.63129,-2.16764,0.098208,1.58018,-2.19464,0.205111,1.43116,-1.80528,0.020633,1.51927,-1.8733,0.019729,1.54596,-1.92066,0.022961,1.52637,-1.98911,0.204457,1.47251,-2.05347,0.104915,1.11678,-1.78403,-0.027938,1.15441,-1.78552,-0.053633,1.07769,-1.81337,-0.040369,0.94513,-1.81672,-0.152343,0.786228,-1.80304,-0.208182,0.735991,-1.80983,-0.230568,0.867388,-1.79297,-0.281676,0.818269,-1.7941,-0.416662,0.622737,-1.81147,-0.513017,0.715672,-1.8289,-0.443496,0.480342,-1.90894,-0.485331,0.460713,-1.83121,0.303036,1.21662,-2.11184,0.350388,1.148,-2.09412,0.18475,1.13452,-2.09966,0.233874,1.06647,-2.0854,0.352888,0.985085,-2.08868,0.413851,0.803409,-2.08475,0.300669,0.795612,-2.1018,0.310389,0.727144,-2.09337,0.361266,0.491959,-2.06389,0.30136,0.394124,-1.99676,0.567016,0.483956,-2.04339,0.531238,0.420441,-2.09802,0.118999,1.35456,-1.83306,0.121748,1.38389,-2.04627},
/*598*/{0.253471,1.95396,-1.74737,0.29336,1.89497,-1.90716,0.222915,1.80087,-1.63191,0.199969,1.67788,-1.54806,0.309947,1.70249,-1.66783,0.385214,1.70993,-1.72111,0.173205,1.96311,-2.11721,0.284426,1.89325,-1.98046,0.068909,1.92007,-2.00227,-0.062985,1.84178,-2.13404,-0.087801,1.75259,-2.19723,0.06333,1.62945,-2.16825,0.100259,1.57812,-2.19509,0.205106,1.43148,-1.8051,0.02064,1.51817,-1.87266,0.019729,1.54596,-1.92066,0.02218,1.52516,-1.98892,0.204159,1.47239,-2.05251,0.109911,1.11518,-1.78608,-0.022983,1.15047,-1.78528,-0.046775,1.07389,-1.81417,-0.033413,0.941907,-1.81826,-0.150637,0.785656,-1.80464,-0.208399,0.738773,-1.81199,-0.224363,0.869904,-1.79159,-0.278517,0.822848,-1.79262,-0.423614,0.636886,-1.8058,-0.515745,0.728745,-1.82458,-0.45385,0.495152,-1.91181,-0.493122,0.474092,-1.8335,0.299673,1.2122,-2.11244,0.34581,1.14299,-2.09524,0.180147,1.13122,-2.09954,0.226525,1.06513,-2.08641,0.34856,0.980474,-2.08875,0.415236,0.801418,-2.08268,0.30316,0.790933,-2.10327,0.313857,0.72184,-2.09547,0.370441,0.488009,-2.06503,0.31121,0.390371,-1.99716,0.575926,0.484706,-2.04193,0.54317,0.420552,-2.09475,0.119419,1.35443,-1.83238,0.121974,1.38335,-2.04565},
/*599*/{0.253321,1.95248,-1.74778,0.293625,1.89364,-1.90779,0.22251,1.80025,-1.63237,0.197554,1.67767,-1.54862,0.307915,1.69944,-1.66759,0.383355,1.70555,-1.72095,0.173139,1.96194,-2.1178,0.284544,1.892,-1.98054,0.069939,1.91962,-2.00381,-0.061276,1.83894,-2.13399,-0.085928,1.74966,-2.19754,0.064665,1.62722,-2.16781,0.102716,1.57545,-2.19556,0.205179,1.43124,-1.80508,0.020632,1.51721,-1.87217,0.019266,1.54522,-1.92004,0.021667,1.52435,-1.98869,0.204317,1.47192,-2.0523,0.116252,1.11358,-1.78893,-0.017236,1.14614,-1.7859,-0.041676,1.06889,-1.81498,-0.025551,0.937274,-1.82025,-0.148042,0.785456,-1.80687,-0.208705,0.740612,-1.81331,-0.218502,0.87207,-1.79003,-0.273698,0.828401,-1.79066,-0.42844,0.647079,-1.80681,-0.517054,0.743388,-1.82207,-0.463238,0.508583,-1.91304,-0.501048,0.488427,-1.83385,0.296116,1.20772,-2.11288,0.340884,1.1372,-2.0969,0.175528,1.12883,-2.09987,0.221485,1.06163,-2.08582,0.343057,0.976076,-2.0892,0.415691,0.799628,-2.08057,0.305093,0.786115,-2.10467,0.317193,0.71759,-2.09669,0.377696,0.485081,-2.06512,0.320588,0.387091,-1.99853,0.583735,0.485524,-2.03851,0.552751,0.420388,-2.09283,0.119784,1.35387,-1.83209,0.122059,1.38278,-2.04537},
/*600*/{0.253094,1.95081,-1.74792,0.292226,1.89237,-1.90727,0.220144,1.79859,-1.63333,0.195235,1.67648,-1.5491,0.305924,1.69709,-1.66844,0.381346,1.70136,-1.72137,0.174474,1.9608,-2.11883,0.284472,1.89091,-1.98074,0.070424,1.91909,-2.00498,-0.060226,1.83488,-2.1332,-0.08386,1.74653,-2.19779,0.066055,1.62414,-2.16814,0.105454,1.57245,-2.19622,0.205159,1.43029,-1.80496,0.020361,1.51566,-1.87185,0.019117,1.54416,-1.91968,0.021654,1.5215,-1.98851,0.204589,1.47151,-2.05246,0.121396,1.11147,-1.79128,-0.012089,1.1426,-1.78595,-0.035392,1.06398,-1.81519,-0.017813,0.93263,-1.82247,-0.14611,0.786243,-1.80757,-0.210163,0.742135,-1.81547,-0.212242,0.875224,-1.78877,-0.269149,0.83376,-1.79057,-0.431269,0.660828,-1.80411,-0.517056,0.758523,-1.82108,-0.469335,0.523078,-1.91253,-0.509831,0.503132,-1.83376,0.292981,1.20307,-2.11363,0.337034,1.13231,-2.09788,0.171616,1.12577,-2.09953,0.215914,1.0583,-2.08555,0.33792,0.971493,-2.08925,0.415695,0.797611,-2.07962,0.305213,0.781837,-2.10503,0.318386,0.713346,-2.09669,0.38243,0.482091,-2.06584,0.327579,0.383343,-2.00108,0.590047,0.486093,-2.03667,0.559612,0.419587,-2.09066,0.120517,1.3526,-1.83199,0.122683,1.38163,-2.04525},
/*601*/{0.253169,1.94906,-1.74905,0.292571,1.89054,-1.90815,0.218741,1.79748,-1.63386,0.19142,1.67443,-1.54967,0.303586,1.69339,-1.66928,0.379658,1.69678,-1.72128,0.175653,1.95897,-2.11972,0.284059,1.88973,-1.98073,0.070717,1.91868,-2.00637,-0.058389,1.83203,-2.1335,-0.082087,1.74281,-2.19852,0.068703,1.62004,-2.16902,0.107458,1.56891,-2.19711,0.205288,1.42988,-1.8053,0.019993,1.51345,-1.87213,0.018388,1.54229,-1.91974,0.020892,1.52023,-1.9887,0.204839,1.47021,-2.0524,0.126617,1.10889,-1.79247,-0.007721,1.13814,-1.78635,-0.029794,1.05963,-1.81705,-0.009965,0.928425,-1.8246,-0.143046,0.787028,-1.80905,-0.207598,0.747093,-1.81498,-0.205415,0.878665,-1.78821,-0.264716,0.838768,-1.78967,-0.434296,0.673544,-1.80319,-0.516037,0.774689,-1.82172,-0.476025,0.536226,-1.90947,-0.518985,0.518583,-1.83193,0.290388,1.19872,-2.1145,0.33314,1.12712,-2.09872,0.167726,1.1231,-2.09957,0.211431,1.05487,-2.08501,0.333135,0.967539,-2.08904,0.413572,0.795751,-2.07891,0.304531,0.778615,-2.10436,0.318142,0.710477,-2.09624,0.387644,0.479281,-2.06578,0.332688,0.381748,-1.99996,0.594866,0.484921,-2.03485,0.565799,0.419603,-2.08906,0.121029,1.35145,-1.83198,0.123082,1.38025,-2.04527},
/*602*/{0.25311,1.94717,-1.74956,0.292282,1.88982,-1.9087,0.216406,1.79521,-1.63461,0.187289,1.67272,-1.55122,0.301229,1.68919,-1.66883,0.377361,1.69128,-1.7212,0.176141,1.95686,-2.12023,0.283867,1.88764,-1.98068,0.071178,1.91641,-2.00784,-0.057332,1.82781,-2.13335,-0.07965,1.73888,-2.19895,0.072228,1.61673,-2.17003,0.109993,1.56508,-2.19781,0.205523,1.4289,-1.80485,0.019265,1.51167,-1.87225,0.017461,1.54041,-1.91946,0.020208,1.51735,-1.98881,0.204476,1.46815,-2.05258,0.132313,1.1066,-1.79457,-0.00313,1.13081,-1.78714,-0.023409,1.05315,-1.81744,-0.002598,0.924592,-1.82608,-0.139206,0.788686,-1.81007,-0.205038,0.750764,-1.8165,-0.197975,0.882005,-1.78803,-0.259078,0.844527,-1.78936,-0.435613,0.685257,-1.80479,-0.512761,0.791026,-1.82329,-0.480984,0.549084,-1.90616,-0.525637,0.535003,-1.82898,0.287932,1.19419,-2.11586,0.329626,1.12274,-2.1002,0.164204,1.1205,-2.09944,0.207154,1.05174,-2.08543,0.327977,0.96389,-2.08913,0.411595,0.793865,-2.07903,0.302135,0.775274,-2.10281,0.316843,0.707043,-2.09491,0.39042,0.477243,-2.06463,0.334803,0.377547,-2.00067,0.595997,0.48494,-2.03383,0.568646,0.418429,-2.08845,0.121309,1.35016,-1.83121,0.122945,1.37786,-2.04465},
/*603*/{0.251898,1.94487,-1.75013,0.291253,1.88765,-1.90878,0.214455,1.79337,-1.63497,0.183499,1.67047,-1.55166,0.298333,1.6847,-1.66802,0.375629,1.68585,-1.72116,0.177357,1.95486,-2.12087,0.283717,1.8862,-1.98057,0.071294,1.9155,-2.0087,-0.055118,1.8249,-2.1343,-0.077467,1.73463,-2.20057,0.074244,1.61255,-2.17102,0.112684,1.56108,-2.19843,0.205673,1.42768,-1.80524,0.019358,1.50968,-1.87211,0.017902,1.53761,-1.91968,0.02033,1.51502,-1.98866,0.204097,1.46566,-2.05357,0.136786,1.10433,-1.7947,0.002119,1.12573,-1.78742,-0.01708,1.04783,-1.81786,0.005855,0.920816,-1.82741,-0.13511,0.790238,-1.8097,-0.20276,0.755194,-1.81455,-0.189872,0.886407,-1.78868,-0.252572,0.851211,-1.79033,-0.435169,0.700189,-1.80171,-0.509656,0.807507,-1.82696,-0.483577,0.561067,-1.90089,-0.53197,0.55244,-1.82501,0.285821,1.19047,-2.11704,0.327209,1.11826,-2.1003,0.16163,1.1187,-2.09867,0.203443,1.05003,-2.08533,0.323272,0.960398,-2.08927,0.407995,0.791913,-2.07901,0.298326,0.771541,-2.10211,0.314533,0.703791,-2.09321,0.391843,0.474986,-2.06444,0.338328,0.373156,-1.99923,0.597562,0.482959,-2.0327,0.570174,0.416709,-2.08789,0.121969,1.34844,-1.83098,0.12278,1.37525,-2.04454},
/*604*/{0.251314,1.94291,-1.75026,0.290442,1.88534,-1.90886,0.211932,1.7913,-1.6357,0.179645,1.66856,-1.55281,0.29488,1.68022,-1.66914,0.372324,1.67946,-1.72035,0.177318,1.9522,-2.12117,0.283103,1.88356,-1.98074,0.071612,1.91292,-2.00998,-0.054735,1.82021,-2.13576,-0.074857,1.72998,-2.20195,0.07697,1.60782,-2.17199,0.115649,1.55634,-2.19933,0.206053,1.4259,-1.80567,0.01889,1.5061,-1.87277,0.016804,1.53561,-1.92018,0.0195,1.51301,-1.98949,0.203489,1.46308,-2.05458,0.142084,1.10136,-1.79484,0.007192,1.12255,-1.78804,-0.010885,1.04198,-1.81832,0.013828,0.916521,-1.82886,-0.129654,0.790192,-1.81191,-0.198422,0.759999,-1.81468,-0.182256,0.890781,-1.78968,-0.246152,0.857687,-1.79088,-0.432414,0.712585,-1.80414,-0.50435,0.823444,-1.83063,-0.487337,0.574999,-1.8959,-0.536612,0.569306,-1.82122,0.283945,1.18659,-2.11842,0.324541,1.11393,-2.10182,0.158999,1.11734,-2.09903,0.200248,1.0475,-2.08528,0.319754,0.957352,-2.08924,0.403858,0.788949,-2.07958,0.293655,0.76765,-2.10077,0.311349,0.700396,-2.09246,0.390577,0.471787,-2.06295,0.33731,0.372614,-1.99884,0.596113,0.481943,-2.03287,0.569901,0.415342,-2.0873,0.122088,1.34624,-1.83157,0.12239,1.37282,-2.04515},
/*605*/{0.250318,1.93948,-1.75047,0.288764,1.88195,-1.90859,0.209104,1.78845,-1.63593,0.174178,1.66599,-1.55423,0.292188,1.67564,-1.66889,0.370414,1.67248,-1.71937,0.178374,1.94905,-2.12174,0.283343,1.88086,-1.98017,0.071401,1.91116,-2.01169,-0.053,1.81541,-2.13687,-0.072051,1.72492,-2.2034,0.079154,1.60292,-2.17438,0.118183,1.55163,-2.20022,0.206398,1.42411,-1.80612,0.01813,1.50338,-1.87296,0.016414,1.53247,-1.92008,0.018832,1.50923,-1.98929,0.203447,1.4604,-2.05538,0.147209,1.09866,-1.79624,0.01046,1.11487,-1.78886,-0.004364,1.03583,-1.81982,0.021399,0.912558,-1.82943,-0.12592,0.794831,-1.80814,-0.195974,0.764251,-1.81275,-0.17492,0.895052,-1.79053,-0.237628,0.86447,-1.7912,-0.429343,0.725089,-1.80591,-0.49787,0.839277,-1.83508,-0.488437,0.587401,-1.89105,-0.540235,0.587674,-1.81622,0.283067,1.1833,-2.11992,0.321356,1.10963,-2.1036,0.157155,1.11574,-2.09882,0.197261,1.04579,-2.08547,0.31615,0.954182,-2.08903,0.399314,0.784955,-2.07929,0.290244,0.764275,-2.09994,0.307681,0.696357,-2.09103,0.387945,0.46833,-2.06164,0.335787,0.36704,-1.99525,0.594528,0.477777,-2.03344,0.567169,0.411993,-2.08589,0.122849,1.34394,-1.83137,0.122419,1.36975,-2.04505},
/*606*/{0.249665,1.9367,-1.75058,0.289743,1.87932,-1.90812,0.205642,1.7854,-1.63624,0.17064,1.66341,-1.55531,0.288544,1.66975,-1.66892,0.366673,1.66496,-1.71918,0.179131,1.94603,-2.12202,0.28306,1.87756,-1.97979,0.071036,1.9085,-2.01269,-0.050884,1.8106,-2.13854,-0.069564,1.71938,-2.20525,0.082588,1.59758,-2.17519,0.121435,1.54647,-2.2012,0.206232,1.42221,-1.80637,0.018271,1.49957,-1.87244,0.016787,1.52955,-1.92066,0.018838,1.50662,-1.99044,0.203053,1.4578,-2.05615,0.150619,1.09483,-1.79404,0.015399,1.10913,-1.79002,0.000687,1.03008,-1.82285,0.02959,0.908445,-1.82936,-0.121163,0.797692,-1.80735,-0.19107,0.768924,-1.81101,-0.165531,0.899584,-1.79214,-0.231656,0.870935,-1.79274,-0.426543,0.739618,-1.80397,-0.490627,0.854064,-1.8399,-0.488564,0.599726,-1.88505,-0.541951,0.604785,-1.81175,0.280735,1.17928,-2.12122,0.319916,1.10572,-2.10477,0.155174,1.11368,-2.09904,0.194705,1.04318,-2.08511,0.313716,0.951391,-2.08889,0.395102,0.781905,-2.08053,0.285813,0.760101,-2.09838,0.30338,0.692513,-2.09001,0.385562,0.46543,-2.05971,0.331645,0.364754,-1.99292,0.591881,0.474881,-2.03079,0.564013,0.408035,-2.08464,0.123367,1.34135,-1.83171,0.122665,1.36697,-2.04541},
/*607*/{0.24806,1.9334,-1.75022,0.289308,1.87612,-1.90875,0.202034,1.78233,-1.63658,0.165025,1.66067,-1.55711,0.284438,1.6638,-1.66861,0.363283,1.6573,-1.71757,0.179535,1.94262,-2.12266,0.282514,1.87394,-1.98027,0.070888,1.90532,-2.01461,-0.049483,1.80483,-2.14008,-0.066558,1.71366,-2.20716,0.086459,1.592,-2.17648,0.124861,1.5411,-2.20202,0.205982,1.41956,-1.80732,0.01775,1.49678,-1.87274,0.015983,1.52584,-1.92025,0.017359,1.50298,-1.99034,0.20178,1.45496,-2.05702,0.154204,1.09218,-1.79496,0.018811,1.10319,-1.7913,0.008362,1.0236,-1.82211,0.037736,0.904012,-1.82978,-0.115395,0.799533,-1.80652,-0.186365,0.773719,-1.80983,-0.156526,0.903323,-1.79461,-0.222693,0.877971,-1.79375,-0.422984,0.753936,-1.80209,-0.482213,0.868023,-1.84496,-0.488407,0.612076,-1.87939,-0.54117,0.622093,-1.80679,0.279542,1.17631,-2.12188,0.31847,1.10245,-2.10599,0.152906,1.11189,-2.09852,0.192181,1.04119,-2.08514,0.311851,0.947957,-2.08835,0.391119,0.777939,-2.08057,0.281194,0.756104,-2.09762,0.298148,0.688573,-2.08848,0.37885,0.459953,-2.05819,0.328187,0.360393,-1.98885,0.587331,0.469329,-2.03,0.558102,0.401971,-2.0833,0.123449,1.33845,-1.83201,0.121582,1.36379,-2.04574},
/*608*/{0.246969,1.93026,-1.75002,0.288574,1.87196,-1.90834,0.198342,1.77894,-1.63671,0.159461,1.65705,-1.55873,0.280635,1.65801,-1.66807,0.360645,1.64955,-1.71585,0.179914,1.93863,-2.12296,0.282113,1.87043,-1.97995,0.07064,1.90214,-2.01527,-0.047871,1.79929,-2.14227,-0.063856,1.70731,-2.20979,0.089008,1.5865,-2.17881,0.128704,1.53596,-2.20275,0.205726,1.41687,-1.80757,0.017402,1.49259,-1.87323,0.015662,1.52188,-1.92061,0.016651,1.49884,-1.99039,0.201747,1.45271,-2.05812,0.157889,1.08962,-1.79414,0.023126,1.09865,-1.79181,0.012787,1.01793,-1.82252,0.046021,0.899925,-1.82926,-0.109328,0.802894,-1.8041,-0.182375,0.777653,-1.80993,-0.147011,0.907939,-1.79656,-0.214732,0.883989,-1.7958,-0.418389,0.768049,-1.80022,-0.473409,0.881443,-1.84918,-0.48546,0.624221,-1.87399,-0.539086,0.638046,-1.80227,0.277827,1.17339,-2.12385,0.315547,1.09967,-2.10719,0.151265,1.11028,-2.09868,0.189577,1.03883,-2.0847,0.310403,0.94488,-2.08778,0.386875,0.772751,-2.08085,0.276499,0.751994,-2.09677,0.294591,0.684001,-2.08734,0.374261,0.456329,-2.0556,0.320827,0.35761,-1.9856,0.581431,0.462496,-2.02921,0.551902,0.394503,-2.08039,0.124242,1.33508,-1.83272,0.122136,1.36079,-2.0464},
/*609*/{0.245707,1.9259,-1.74937,0.288465,1.86849,-1.90825,0.19499,1.77488,-1.63642,0.153656,1.65306,-1.55956,0.276571,1.65124,-1.66798,0.356732,1.64058,-1.71455,0.179269,1.93468,-2.12336,0.281856,1.86675,-1.97972,0.070716,1.89858,-2.01692,-0.046156,1.79256,-2.14462,-0.060778,1.70036,-2.21172,0.091842,1.58014,-2.18018,0.132179,1.52981,-2.20371,0.205815,1.4143,-1.80807,0.016335,1.48865,-1.8733,0.015054,1.51889,-1.92053,0.016028,1.4948,-1.99057,0.200901,1.45042,-2.05917,0.161676,1.08539,-1.79189,0.026001,1.09145,-1.79281,0.018527,1.01204,-1.82424,0.054154,0.895462,-1.82803,-0.101945,0.804759,-1.80437,-0.174859,0.783262,-1.80709,-0.136845,0.911536,-1.79888,-0.20436,0.889534,-1.7971,-0.41314,0.781424,-1.79787,-0.46445,0.89332,-1.85348,-0.48193,0.635822,-1.86813,-0.535399,0.654049,-1.79672,0.277077,1.17098,-2.1245,0.313639,1.09605,-2.10782,0.149389,1.10705,-2.09851,0.187696,1.03637,-2.08556,0.308161,0.941627,-2.08796,0.382486,0.76805,-2.08128,0.271712,0.747783,-2.09562,0.288823,0.679766,-2.08592,0.36664,0.451239,-2.05459,0.31239,0.355992,-1.98265,0.574391,0.454552,-2.02823,0.543542,0.38632,-2.07838,0.124766,1.33203,-1.83328,0.121938,1.35792,-2.04694},
/*610*/{0.244016,1.92147,-1.74859,0.287662,1.86415,-1.90702,0.191439,1.77053,-1.63635,0.148489,1.64971,-1.56152,0.271874,1.6444,-1.66758,0.35304,1.63198,-1.71272,0.179576,1.92999,-2.12375,0.28149,1.86199,-1.97864,0.069557,1.89445,-2.01829,-0.045205,1.7852,-2.14734,-0.058252,1.69306,-2.21446,0.095335,1.57367,-2.18196,0.136412,1.52391,-2.2042,0.205201,1.41102,-1.80943,0.016052,1.48454,-1.87342,0.014358,1.51416,-1.92061,0.015188,1.49025,-1.99039,0.200204,1.44741,-2.0606,0.166935,1.0815,-1.79,0.028765,1.08526,-1.79526,0.024848,1.00545,-1.82454,0.062617,0.891736,-1.82673,-0.095197,0.807524,-1.80169,-0.167823,0.787702,-1.8055,-0.127482,0.91544,-1.80176,-0.196013,0.894845,-1.79978,-0.406705,0.794404,-1.7964,-0.454884,0.90456,-1.85797,-0.477136,0.647624,-1.86353,-0.529582,0.66863,-1.79237,0.275022,1.16743,-2.12501,0.312244,1.0926,-2.1091,0.147556,1.10465,-2.09874,0.185885,1.03302,-2.08662,0.305637,0.938367,-2.08812,0.377515,0.762954,-2.08069,0.267058,0.74387,-2.09467,0.282809,0.67528,-2.08495,0.356973,0.444832,-2.05278,0.302588,0.352781,-1.97894,0.567235,0.447108,-2.0251,0.53528,0.376711,-2.0747,0.125329,1.32814,-1.8342,0.121538,1.35435,-2.0478},
/*611*/{0.242773,1.91646,-1.74818,0.288085,1.85945,-1.90738,0.187492,1.76624,-1.63635,0.142054,1.64516,-1.56262,0.267028,1.63763,-1.66711,0.348952,1.62277,-1.71129,0.179846,1.92515,-2.12381,0.281206,1.85752,-1.97909,0.068967,1.88957,-2.01929,-0.043701,1.77901,-2.1503,-0.055058,1.68593,-2.21749,0.098999,1.56853,-2.18375,0.139976,1.51788,-2.205,0.205422,1.40706,-1.81023,0.015693,1.47938,-1.87391,0.013401,1.50988,-1.92034,0.01435,1.48532,-1.98995,0.199373,1.44488,-2.06143,0.167823,1.07899,-1.78913,0.032982,1.08102,-1.79621,0.031186,0.999114,-1.82535,0.070914,0.887262,-1.82488,-0.087675,0.809387,-1.80021,-0.16161,0.79126,-1.80335,-0.116855,0.918184,-1.80452,-0.186224,0.899907,-1.80259,-0.399806,0.806227,-1.79488,-0.444985,0.914596,-1.8617,-0.471758,0.657653,-1.85856,-0.523905,0.682576,-1.78858,0.273112,1.1641,-2.12632,0.311547,1.08845,-2.10908,0.145723,1.10148,-2.09917,0.184207,1.029,-2.08648,0.303327,0.933792,-2.08832,0.372921,0.75661,-2.08081,0.261036,0.738738,-2.09427,0.276676,0.670502,-2.0843,0.348871,0.440374,-2.05081,0.290221,0.348606,-1.97999,0.557399,0.437191,-2.0246,0.524512,0.367383,-2.0712,0.126181,1.32376,-1.83532,0.121535,1.35104,-2.04877},
/*612*/{0.241851,1.91215,-1.74755,0.287424,1.85493,-1.90614,0.183704,1.7621,-1.63643,0.136695,1.64212,-1.56446,0.261312,1.63027,-1.66666,0.342572,1.6134,-1.70883,0.179641,1.92043,-2.12426,0.281007,1.85329,-1.9789,0.06942,1.88443,-2.0204,-0.042655,1.77125,-2.15365,-0.052126,1.67813,-2.21995,0.103602,1.56174,-2.18549,0.144699,1.5111,-2.20584,0.205712,1.40405,-1.81146,0.014837,1.47487,-1.87398,0.013055,1.50554,-1.92136,0.013132,1.48059,-1.98931,0.198984,1.44238,-2.06256,0.174904,1.07567,-1.78678,0.03663,1.07326,-1.79836,0.037542,0.992963,-1.82617,0.078969,0.882926,-1.82359,-0.079154,0.811854,-1.79866,-0.153195,0.794781,-1.80164,-0.105524,0.921131,-1.80717,-0.17605,0.90455,-1.80481,-0.390928,0.817085,-1.79651,-0.43522,0.924329,-1.86536,-0.465078,0.667537,-1.85462,-0.516185,0.695437,-1.78544,0.270809,1.16057,-2.1272,0.308993,1.08575,-2.10997,0.143666,1.09803,-2.10007,0.182212,1.02525,-2.08707,0.299957,0.928149,-2.08847,0.367521,0.750899,-2.08119,0.255505,0.734254,-2.09373,0.269416,0.665774,-2.08359,0.340928,0.436284,-2.04953,0.275721,0.355046,-1.97619,0.547184,0.426268,-2.02148,0.511085,0.357669,-2.0676,0.127281,1.32011,-1.83626,0.121694,1.34773,-2.04964},
/*613*/{0.240334,1.90679,-1.74657,0.28601,1.84912,-1.90532,0.179037,1.75724,-1.63667,0.129924,1.63742,-1.5657,0.256589,1.62256,-1.66631,0.338074,1.60349,-1.70741,0.17969,1.91475,-2.12493,0.281029,1.84837,-1.97813,0.068148,1.87853,-2.02122,-0.040381,1.76365,-2.15678,-0.048988,1.67046,-2.22298,0.107989,1.55514,-2.18683,0.149466,1.50472,-2.20668,0.205801,1.40006,-1.81235,0.014713,1.46883,-1.87393,0.012263,1.50013,-1.92143,0.011933,1.47515,-1.98956,0.198038,1.43961,-2.06428,0.176878,1.0717,-1.78539,0.041743,1.06883,-1.79996,0.043974,0.98668,-1.82692,0.087301,0.879052,-1.82157,-0.070597,0.814251,-1.79601,-0.144029,0.799253,-1.80045,-0.093905,0.923478,-1.8092,-0.164747,0.908745,-1.80791,-0.379867,0.826979,-1.79753,-0.424845,0.932203,-1.8684,-0.455966,0.675582,-1.85007,-0.506778,0.706176,-1.7832,0.268739,1.15646,-2.12768,0.307186,1.08184,-2.11138,0.142067,1.09346,-2.10045,0.180074,1.02123,-2.08786,0.296532,0.922222,-2.08886,0.360281,0.74409,-2.08149,0.248758,0.729131,-2.09363,0.261632,0.660414,-2.08281,0.330886,0.432694,-2.04712,0.258684,0.356445,-1.97736,0.535838,0.412682,-2.0183,0.494874,0.34702,-2.06558,0.128624,1.31534,-1.83769,0.121828,1.34393,-2.05091},
/*614*/{0.238362,1.90129,-1.74594,0.286957,1.84352,-1.90482,0.175104,1.75168,-1.63682,0.123697,1.6347,-1.5666,0.251105,1.61505,-1.66532,0.33321,1.59374,-1.70537,0.180954,1.90985,-2.12561,0.280252,1.84304,-1.97782,0.067591,1.87348,-2.02309,-0.038953,1.75454,-2.16042,-0.045434,1.66197,-2.22576,0.111634,1.54808,-2.18847,0.154097,1.49757,-2.20743,0.205204,1.39654,-1.81349,0.0142,1.46397,-1.87385,0.012458,1.49486,-1.92206,0.01096,1.47025,-1.98859,0.197802,1.43642,-2.06594,0.178319,1.06822,-1.78401,0.046858,1.06064,-1.80129,0.049882,0.979837,-1.82841,0.095557,0.875684,-1.81971,-0.060806,0.815885,-1.79481,-0.134717,0.80256,-1.79907,-0.081847,0.92504,-1.81215,-0.153169,0.912576,-1.81051,-0.369782,0.834713,-1.79771,-0.414708,0.939861,-1.87156,-0.447708,0.683758,-1.84737,-0.497994,0.716121,-1.77937,0.266966,1.15241,-2.12893,0.305028,1.07796,-2.1126,0.139767,1.08796,-2.10053,0.178904,1.01708,-2.08786,0.292595,0.916599,-2.08958,0.354723,0.737548,-2.08062,0.242162,0.723378,-2.09327,0.25474,0.65495,-2.08263,0.317675,0.42726,-2.04695,0.241346,0.358976,-1.97367,0.522978,0.396773,-2.01379,0.476856,0.337439,-2.06374,0.129508,1.31106,-1.83862,0.121809,1.34009,-2.05175},
/*615*/{0.236467,1.89585,-1.74525,0.286482,1.8381,-1.90425,0.171334,1.74684,-1.63663,0.117106,1.63028,-1.56797,0.24507,1.60706,-1.6646,0.326847,1.58359,-1.70348,0.180378,1.90446,-2.12586,0.28059,1.83797,-1.97745,0.067354,1.86708,-2.02395,-0.037267,1.74648,-2.16442,-0.042012,1.65358,-2.2287,0.116627,1.54123,-2.19063,0.159769,1.49162,-2.20815,0.204994,1.39241,-1.8143,0.01299,1.45835,-1.87462,0.010876,1.48923,-1.92269,0.00926,1.46445,-1.98876,0.19645,1.43254,-2.06774,0.186467,1.06347,-1.78042,0.050942,1.05665,-1.80217,0.056974,0.973245,-1.83013,0.105083,0.872913,-1.81719,-0.050698,0.817128,-1.79319,-0.126836,0.805964,-1.79722,-0.069658,0.925799,-1.8153,-0.141067,0.91487,-1.81216,-0.360682,0.842243,-1.79766,-0.403664,0.946263,-1.87413,-0.43662,0.691096,-1.84388,-0.486242,0.725146,-1.77675,0.26366,1.14775,-2.13085,0.302484,1.07349,-2.11508,0.137093,1.0839,-2.10108,0.176467,1.01198,-2.08818,0.28817,0.910745,-2.09022,0.347824,0.73032,-2.08018,0.235645,0.718513,-2.09275,0.246802,0.649714,-2.08243,0.29865,0.422314,-2.04134,0.219099,0.356069,-1.97347,0.504724,0.387112,-2.00202,0.455872,0.338942,-2.06445,0.129914,1.30639,-1.83968,0.121196,1.33544,-2.05277},
/*616*/{0.235606,1.89006,-1.7441,0.286409,1.83241,-1.90305,0.166656,1.74145,-1.63695,0.109886,1.62546,-1.56814,0.239471,1.59917,-1.66408,0.32173,1.5737,-1.70107,0.181379,1.89894,-2.12667,0.28122,1.83254,-1.97684,0.066528,1.86181,-2.02444,-0.035216,1.73736,-2.16846,-0.038112,1.64481,-2.23232,0.121576,1.53431,-2.19177,0.165013,1.48522,-2.209,0.204057,1.38783,-1.81521,0.011964,1.45204,-1.87467,0.009305,1.48354,-1.92339,0.007015,1.45962,-1.98904,0.195496,1.42819,-2.07005,0.190598,1.06015,-1.77777,0.054472,1.04807,-1.80382,0.062652,0.968371,-1.83051,0.113728,0.869304,-1.81513,-0.041042,0.818592,-1.79142,-0.115472,0.809321,-1.79539,-0.057328,0.926926,-1.817,-0.128044,0.916987,-1.81471,-0.347874,0.849787,-1.79814,-0.392833,0.951689,-1.87612,-0.42595,0.697693,-1.84136,-0.475078,0.732723,-1.77468,0.259623,1.14318,-2.13309,0.298307,1.06891,-2.11799,0.133838,1.07973,-2.10169,0.17316,1.00802,-2.08882,0.284466,0.904403,-2.09248,0.342097,0.72263,-2.07709,0.228957,0.71399,-2.09272,0.23749,0.645123,-2.08126,0.281202,0.423708,-2.03493,0.196343,0.35465,-1.97302,0.478325,0.386284,-2.003,0.431732,0.340527,-2.06423,0.129727,1.30122,-1.84108,0.120119,1.33085,-2.05405},
/*617*/{0.233921,1.88428,-1.74287,0.286368,1.82559,-1.90164,0.162653,1.7357,-1.63718,0.102559,1.62237,-1.56923,0.233324,1.59218,-1.66314,0.314498,1.56402,-1.69894,0.18191,1.89407,-2.12739,0.281222,1.82685,-1.97664,0.065769,1.85446,-2.02642,-0.03231,1.7288,-2.17234,-0.033418,1.63617,-2.23543,0.126622,1.52737,-2.19386,0.171122,1.47911,-2.20987,0.204141,1.38342,-1.81667,0.010793,1.44675,-1.87339,0.007372,1.47793,-1.92323,0.004844,1.45446,-1.98896,0.193718,1.42374,-2.07152,0.193102,1.05805,-1.77628,0.060314,1.04367,-1.80438,0.069536,0.961052,-1.83071,0.12325,0.867362,-1.81253,-0.030035,0.819776,-1.78997,-0.10487,0.810586,-1.79337,-0.044336,0.927434,-1.81888,-0.115758,0.919252,-1.81629,-0.336614,0.854235,-1.79809,-0.381823,0.956687,-1.87861,-0.413773,0.702256,-1.83827,-0.463481,0.739432,-1.77167,0.255828,1.13969,-2.13726,0.294651,1.06514,-2.12233,0.130035,1.07752,-2.10141,0.169159,1.00545,-2.08833,0.279253,0.89875,-2.09377,0.334809,0.715339,-2.07254,0.222928,0.712008,-2.09301,0.22761,0.642783,-2.07825,0.257377,0.42496,-2.02928,0.171835,0.354485,-1.96826,0.452344,0.386754,-2.00099,0.407983,0.340882,-2.06456,0.129925,1.29646,-1.8418,0.118369,1.32611,-2.05467},
/*618*/{0.232574,1.87857,-1.74175,0.286081,1.81978,-1.90053,0.157617,1.73029,-1.63769,0.096039,1.61844,-1.57064,0.227653,1.58467,-1.66129,0.308341,1.55489,-1.69668,0.183409,1.88915,-2.12845,0.281652,1.82148,-1.97585,0.064573,1.8499,-2.02723,-0.031009,1.72002,-2.17632,-0.030688,1.62842,-2.23873,0.132488,1.52124,-2.19571,0.176876,1.47304,-2.21068,0.203192,1.37845,-1.81816,0.009077,1.4408,-1.87435,0.005644,1.47154,-1.92316,0.001804,1.45059,-1.98929,0.190232,1.41917,-2.07261,0.199577,1.05414,-1.77363,0.064179,1.0355,-1.80508,0.076895,0.954705,-1.82963,0.132865,0.864299,-1.8092,-0.018214,0.819902,-1.78881,-0.093395,0.811933,-1.79334,-0.030572,0.92693,-1.82176,-0.102802,0.921043,-1.81822,-0.325422,0.861843,-1.79943,-0.370728,0.960099,-1.88103,-0.401297,0.706416,-1.83686,-0.450444,0.744188,-1.77028,0.251372,1.13643,-2.14266,0.290141,1.06162,-2.12797,0.126238,1.07577,-2.10104,0.165882,1.00483,-2.08756,0.272748,0.893579,-2.09456,0.328242,0.711608,-2.06749,0.217382,0.713966,-2.09015,0.217202,0.645053,-2.07449,0.231027,0.424958,-2.02513,0.148812,0.355813,-1.97211,0.430666,0.382929,-1.9996,0.384526,0.340861,-2.06369,0.128777,1.29109,-1.84342,0.115741,1.32154,-2.05609},
/*619*/{0.231101,1.87375,-1.74072,0.285897,1.81403,-1.8992,0.154044,1.72543,-1.63854,0.089762,1.61574,-1.57166,0.220411,1.57727,-1.66172,0.301672,1.54563,-1.69431,0.183845,1.88446,-2.12966,0.282501,1.81572,-1.97558,0.064545,1.84658,-2.02723,-0.027756,1.71128,-2.1821,-0.023018,1.61836,-2.24152,0.138273,1.51468,-2.19683,0.183704,1.46721,-2.21089,0.201915,1.37399,-1.82046,0.005537,1.43552,-1.87441,0.002996,1.46647,-1.92273,-0.001861,1.4461,-1.99018,0.187593,1.41619,-2.07481,0.199813,1.05068,-1.77308,0.069535,1.02959,-1.80567,0.0841,0.949662,-1.83025,0.143673,0.861642,-1.80722,-0.005943,0.819578,-1.78722,-0.08056,0.814503,-1.79078,-0.017611,0.926611,-1.82298,-0.088935,0.921101,-1.8191,-0.312922,0.864672,-1.80093,-0.358973,0.962693,-1.88291,-0.387497,0.709945,-1.83487,-0.436857,0.748756,-1.76832,0.248098,1.13364,-2.14783,0.286089,1.05925,-2.13167,0.124303,1.07569,-2.09975,0.162371,1.00375,-2.08715,0.266662,0.891757,-2.09501,0.319317,0.711175,-2.06368,0.208373,0.718714,-2.08688,0.205572,0.649407,-2.07175,0.217572,0.427453,-2.02347,0.12603,0.355954,-1.9673,0.409162,0.382188,-1.99799,0.361863,0.341068,-2.06241,0.127526,1.28631,-1.84571,0.112988,1.31817,-2.05807},
/*620*/{0.229724,1.86935,-1.73947,0.286329,1.8069,-1.89754,0.149391,1.72117,-1.63892,0.081716,1.6123,-1.57405,0.21418,1.57098,-1.65983,0.294122,1.53792,-1.69124,0.184861,1.88029,-2.13126,0.283514,1.81001,-1.97476,0.065297,1.84323,-2.02679,-0.023625,1.70254,-2.18765,-0.016771,1.61001,-2.24452,0.145489,1.50882,-2.19744,0.190837,1.46139,-2.21106,0.200386,1.36826,-1.82149,0.003247,1.43034,-1.87481,0.000873,1.46107,-1.92293,-0.005259,1.44192,-1.98987,0.185345,1.41357,-2.07728,0.20566,1.04929,-1.77111,0.075218,1.02306,-1.80545,0.091588,0.943586,-1.83042,0.154219,0.85822,-1.80468,0.005588,0.81968,-1.78648,-0.069577,0.813635,-1.78946,-0.004136,0.926057,-1.82345,-0.073992,0.921305,-1.81847,-0.297622,0.86731,-1.80088,-0.347003,0.964339,-1.88525,-0.373549,0.711464,-1.83322,-0.421833,0.751538,-1.76664,0.244174,1.13169,-2.15066,0.283085,1.05767,-2.13349,0.121027,1.07508,-2.10011,0.159974,1.00388,-2.08715,0.265593,0.892255,-2.09486,0.309278,0.709729,-2.0633,0.198336,0.7234,-2.08528,0.192471,0.654239,-2.07046,0.196045,0.428723,-2.02173,0.103939,0.358288,-1.96696,0.387012,0.381087,-1.99594,0.339191,0.341421,-2.06162,0.126054,1.28084,-1.84824,0.110514,1.31501,-2.06018},
/*621*/{0.228547,1.86458,-1.73785,0.286767,1.8024,-1.8963,0.145714,1.71731,-1.63939,0.073653,1.61036,-1.57628,0.207267,1.56397,-1.65865,0.286954,1.53049,-1.68909,0.186848,1.87631,-2.13261,0.283955,1.80495,-1.9742,0.065965,1.84159,-2.02654,-0.018436,1.69466,-2.19365,-0.008553,1.60103,-2.24764,0.152736,1.50391,-2.19835,0.198636,1.45676,-2.21091,0.198872,1.36478,-1.82459,-0.000122,1.42574,-1.874,-0.001589,1.45682,-1.92278,-0.008506,1.43823,-1.98932,0.182621,1.41098,-2.07931,0.208702,1.04879,-1.77045,0.081036,1.02068,-1.8066,0.099676,0.939432,-1.82876,0.166085,0.856904,-1.80278,0.018802,0.819123,-1.78465,-0.056485,0.814772,-1.78936,0.010011,0.926664,-1.82287,-0.061187,0.921826,-1.81869,-0.286645,0.869861,-1.80235,-0.334692,0.96514,-1.88645,-0.358036,0.712961,-1.83159,-0.407384,0.752816,-1.76502,0.240087,1.13088,-2.15053,0.279491,1.05683,-2.13334,0.118002,1.07352,-2.09967,0.157102,1.00216,-2.0861,0.266024,0.893098,-2.09411,0.297441,0.709559,-2.06546,0.186895,0.727539,-2.08346,0.17898,0.658027,-2.07031,0.174614,0.428409,-2.02057,0.082645,0.35922,-1.96682,0.366011,0.380251,-1.99753,0.317316,0.341325,-2.06161,0.124917,1.27684,-1.8503,0.107358,1.31224,-2.06187},
/*622*/{0.226801,1.86116,-1.73688,0.285425,1.79777,-1.89465,0.140186,1.71467,-1.6401,0.067005,1.60968,-1.57763,0.200459,1.55968,-1.65779,0.278309,1.52371,-1.68611,0.187879,1.87254,-2.1335,0.284483,1.80114,-1.97308,0.066183,1.84006,-2.02603,-0.012702,1.68722,-2.19918,-0.001481,1.5928,-2.24927,0.160858,1.49935,-2.19826,0.207367,1.45312,-2.21072,0.196969,1.36088,-1.82616,-0.001847,1.42155,-1.87437,-0.003039,1.45573,-1.92231,-0.012256,1.43546,-1.98707,0.180016,1.40902,-2.08141,0.21451,1.05078,-1.76861,0.086541,1.0172,-1.80553,0.109247,0.936424,-1.82721,0.17809,0.855974,-1.79976,0.03168,0.819166,-1.78478,-0.042494,0.814323,-1.78969,0.023744,0.925395,-1.82293,-0.047317,0.922488,-1.81935,-0.275535,0.870933,-1.80413,-0.322974,0.964688,-1.88786,-0.342661,0.712365,-1.83048,-0.391674,0.752045,-1.7644,0.235566,1.12994,-2.15037,0.275537,1.05567,-2.13422,0.113125,1.07211,-2.09898,0.152735,1.00066,-2.08549,0.26681,0.894245,-2.09325,0.285872,0.709163,-2.06839,0.174688,0.729598,-2.08359,0.165188,0.660802,-2.07055,0.159271,0.429701,-2.01969,0.059935,0.359091,-1.96517,0.344389,0.3794,-1.99611,0.295269,0.340893,-2.06147,0.122976,1.27317,-1.85221,0.10372,1.31043,-2.06332},
/*623*/{0.226478,1.85759,-1.73506,0.286908,1.79392,-1.89392,0.134478,1.71327,-1.64109,0.058651,1.60935,-1.57957,0.192915,1.55605,-1.65694,0.269699,1.51819,-1.68358,0.189756,1.86979,-2.13336,0.283899,1.79776,-1.97236,0.066436,1.83927,-2.02654,-0.007198,1.68034,-2.20372,0.006645,1.58653,-2.25105,0.169397,1.49596,-2.19747,0.216162,1.45033,-2.2099,0.195484,1.3567,-1.82696,-0.00468,1.41879,-1.87309,-0.00516,1.45522,-1.92243,-0.013758,1.43448,-1.98514,0.178021,1.40837,-2.08356,0.219167,1.05324,-1.76836,0.092502,1.01435,-1.80519,0.117195,0.934516,-1.8261,0.19104,0.855695,-1.79871,0.044745,0.818214,-1.78405,-0.03048,0.81479,-1.78951,0.037529,0.925588,-1.82143,-0.033808,0.922951,-1.81858,-0.259326,0.867849,-1.80238,-0.311883,0.962869,-1.88902,-0.326072,0.711015,-1.82979,-0.375812,0.750397,-1.76379,0.230462,1.12821,-2.15071,0.269787,1.0534,-2.13562,0.109157,1.07086,-2.0985,0.148373,0.999443,-2.08525,0.266688,0.895214,-2.09195,0.273357,0.707632,-2.07217,0.162727,0.729641,-2.0854,0.151978,0.661467,-2.07077,0.135322,0.43071,-2.01998,0.038648,0.358715,-1.96444,0.323441,0.378294,-1.99639,0.273209,0.340851,-2.06126,0.120492,1.26994,-1.85429,0.100464,1.31005,-2.0648},
/*624*/{0.226018,1.85583,-1.73425,0.286666,1.79172,-1.89255,0.130281,1.71239,-1.64121,0.050082,1.60962,-1.58186,0.184745,1.55333,-1.65607,0.262779,1.51315,-1.68105,0.1902,1.86785,-2.13254,0.283974,1.79554,-1.97108,0.066441,1.83685,-2.02614,-0.001032,1.67399,-2.20721,0.015235,1.58069,-2.25261,0.177497,1.49365,-2.1969,0.225516,1.44839,-2.20867,0.192913,1.35322,-1.8284,-0.005542,1.41849,-1.8714,-0.005807,1.45498,-1.92175,-0.014673,1.43454,-1.98312,0.175729,1.40775,-2.08491,0.22557,1.05554,-1.76678,0.098701,1.01353,-1.80351,0.125412,0.933549,-1.8258,0.204485,0.85626,-1.79858,0.057572,0.818244,-1.78288,-0.016407,0.813921,-1.7901,0.050952,0.926036,-1.82034,-0.020834,0.922914,-1.81863,-0.244291,0.866842,-1.80386,-0.30097,0.960473,-1.89089,-0.309481,0.708007,-1.82926,-0.359276,0.746481,-1.7641,0.225785,1.12515,-2.15125,0.264067,1.05021,-2.1364,0.103758,1.07001,-2.09812,0.142472,0.9977,-2.08449,0.261011,0.894452,-2.09151,0.260693,0.703777,-2.07506,0.150419,0.729037,-2.0866,0.137036,0.661969,-2.07116,0.115279,0.430411,-2.02045,0.016355,0.362274,-1.96499,0.300795,0.378017,-1.99727,0.251476,0.340306,-2.06195,0.117727,1.26754,-1.85578,0.096697,1.31001,-2.06574},
/*625*/{0.225376,1.85235,-1.73335,0.284685,1.78957,-1.8912,0.124424,1.71187,-1.64192,0.04236,1.61152,-1.58438,0.176543,1.55121,-1.65477,0.254987,1.50915,-1.67732,0.192089,1.86624,-2.1311,0.284232,1.79377,-1.96963,0.066827,1.83608,-2.02613,0.005778,1.66846,-2.20968,0.024015,1.57589,-2.25338,0.186998,1.49167,-2.19542,0.236035,1.44871,-2.20694,0.191379,1.35088,-1.82886,-0.00744,1.41875,-1.87002,-0.008114,1.45485,-1.9219,-0.015103,1.43493,-1.98175,0.173787,1.40753,-2.08664,0.227828,1.059,-1.76813,0.105366,1.01336,-1.80178,0.134948,0.933919,-1.82259,0.21794,0.857974,-1.79814,0.071198,0.818998,-1.78264,-0.002833,0.813518,-1.78955,0.063021,0.926808,-1.81875,-0.007694,0.922542,-1.81763,-0.232001,0.864472,-1.80463,-0.292273,0.955977,-1.89037,-0.292184,0.703959,-1.82905,-0.341847,0.740583,-1.76332,0.220464,1.12225,-2.15177,0.257427,1.04691,-2.13766,0.097984,1.06865,-2.09767,0.137195,0.99364,-2.08249,0.252377,0.891115,-2.09298,0.247446,0.698268,-2.07521,0.138164,0.728508,-2.08772,0.123656,0.661254,-2.07198,0.096221,0.431863,-2.02177,-0.004305,0.362577,-1.9653,0.279612,0.376947,-1.99728,0.229469,0.339864,-2.06213,0.115275,1.26624,-1.8568,0.093674,1.31016,-2.0664},
/*626*/{0.224875,1.85026,-1.73254,0.284211,1.78764,-1.89005,0.118792,1.71262,-1.64252,0.034682,1.61511,-1.58684,0.169068,1.55031,-1.65379,0.246477,1.50701,-1.6753,0.19335,1.86507,-2.13009,0.28407,1.79224,-1.96843,0.066825,1.83597,-2.02637,0.011829,1.66491,-2.2117,0.03285,1.57244,-2.25381,0.196447,1.49164,-2.19395,0.245734,1.44882,-2.20455,0.188485,1.34884,-1.82868,-0.008176,1.41931,-1.87057,-0.00918,1.45379,-1.92231,-0.01642,1.435,-1.98134,0.173829,1.40721,-2.087,0.235523,1.06308,-1.76746,0.112111,1.01278,-1.8003,0.144458,0.934688,-1.82114,0.231743,0.861658,-1.79772,0.08389,0.820736,-1.78212,0.008995,0.814246,-1.79065,0.074361,0.927904,-1.81877,0.004001,0.922827,-1.81627,-0.219075,0.861368,-1.8064,-0.283105,0.950729,-1.89119,-0.27355,0.699076,-1.82973,-0.324559,0.734158,-1.76234,0.214876,1.11918,-2.15284,0.25126,1.04271,-2.13824,0.091879,1.06703,-2.09724,0.13078,0.99142,-2.08268,0.24262,0.886654,-2.09451,0.233806,0.694716,-2.07573,0.126072,0.726603,-2.08896,0.107697,0.66134,-2.07183,0.075028,0.432404,-2.0226,-0.027937,0.362453,-1.96611,0.257289,0.376982,-1.99917,0.207758,0.340207,-2.06282,0.112703,1.26495,-1.85749,0.09235,1.31007,-2.06695},
/*627*/{0.224252,1.84933,-1.73206,0.283331,1.78628,-1.88941,0.113877,1.71368,-1.64264,0.02664,1.61826,-1.58924,0.161073,1.5504,-1.6533,0.237567,1.50504,-1.67266,0.195371,1.86476,-2.12903,0.283669,1.79062,-1.96678,0.067662,1.83572,-2.02681,0.019497,1.66227,-2.21203,0.040221,1.56952,-2.2546,0.20544,1.49059,-2.19149,0.256039,1.45016,-2.2017,0.187395,1.34827,-1.82808,-0.009122,1.41908,-1.87047,-0.010607,1.4542,-1.92314,-0.017511,1.43616,-1.98159,0.171839,1.40688,-2.08784,0.243869,1.0671,-1.76545,0.120561,1.01178,-1.79823,0.154542,0.937085,-1.81921,0.243889,0.865449,-1.79689,0.096787,0.821759,-1.78271,0.021478,0.813857,-1.78931,0.086271,0.929437,-1.81683,0.014362,0.923472,-1.8167,-0.205537,0.857124,-1.80619,-0.274925,0.944548,-1.89099,-0.255006,0.693525,-1.83118,-0.306585,0.72619,-1.76285,0.209501,1.11605,-2.15372,0.244818,1.03893,-2.13869,0.085797,1.06334,-2.09687,0.122109,0.989778,-2.08393,0.232355,0.881485,-2.09527,0.220978,0.689201,-2.07487,0.113721,0.724814,-2.0898,0.095101,0.658243,-2.07231,0.052348,0.432243,-2.02339,-0.048828,0.363486,-1.96701,0.236476,0.376744,-1.99976,0.185536,0.338854,-2.06343,0.11085,1.26482,-1.85782,0.090455,1.31007,-2.06725},
/*628*/{0.222056,1.84876,-1.73164,0.283335,1.78518,-1.88783,0.108059,1.71552,-1.64344,0.018576,1.62197,-1.59171,0.153802,1.55179,-1.65231,0.2291,1.50469,-1.67028,0.197545,1.86463,-2.12833,0.28376,1.78936,-1.96589,0.068099,1.8357,-2.0272,0.024821,1.66058,-2.21283,0.048322,1.56703,-2.25486,0.214703,1.49273,-2.18967,0.266102,1.45303,-2.19881,0.184482,1.34805,-1.82687,-0.011024,1.41956,-1.87065,-0.011678,1.45404,-1.92341,-0.017476,1.43631,-1.9815,0.169112,1.40523,-2.08912,0.250525,1.0713,-1.76579,0.128801,1.01491,-1.79648,0.165179,0.938919,-1.81767,0.257503,0.870024,-1.79613,0.110217,0.823426,-1.78292,0.034864,0.814619,-1.78974,0.097375,0.931892,-1.81449,0.025673,0.923329,-1.81595,-0.196836,0.855378,-1.8074,-0.266887,0.93697,-1.89084,-0.234703,0.686811,-1.83118,-0.286483,0.717155,-1.76313,0.2039,1.11332,-2.1544,0.2374,1.03509,-2.13932,0.078657,1.06509,-2.09883,0.113784,0.988092,-2.0851,0.222009,0.87762,-2.09563,0.206456,0.685488,-2.07526,0.1009,0.72232,-2.09036,0.08039,0.6553,-2.07303,0.035517,0.433558,-2.0241,-0.07343,0.367,-1.97023,0.214065,0.375569,-2.00018,0.164051,0.338923,-2.06491,0.108091,1.26469,-1.85701,0.087962,1.30883,-2.06671},
/*629*/{0.220047,1.8489,-1.73052,0.281803,1.78469,-1.88726,0.102482,1.71818,-1.64475,0.009732,1.62648,-1.59373,0.145193,1.55267,-1.65033,0.220407,1.50536,-1.6674,0.199354,1.86486,-2.12792,0.283729,1.78938,-1.96462,0.069449,1.83573,-2.02792,0.029866,1.65871,-2.2131,0.056507,1.56603,-2.25528,0.224048,1.49513,-2.18699,0.27665,1.45673,-2.19552,0.18358,1.3488,-1.82623,-0.011349,1.42039,-1.87118,-0.012694,1.45426,-1.92433,-0.018214,1.43742,-1.98288,0.170037,1.40555,-2.08888,0.254973,1.07614,-1.76551,0.137386,1.0166,-1.79485,0.175286,0.942121,-1.81638,0.269756,0.875111,-1.79556,0.122373,0.825452,-1.7825,0.049109,0.81475,-1.78982,0.107786,0.933671,-1.81317,0.036177,0.923525,-1.81379,-0.181306,0.846292,-1.80906,-0.258539,0.927968,-1.89148,-0.215097,0.679468,-1.83242,-0.267125,0.707725,-1.76351,0.198373,1.11126,-2.15409,0.229988,1.03151,-2.13949,0.072435,1.06331,-2.09938,0.105538,0.98775,-2.08636,0.210631,0.873475,-2.09502,0.192949,0.680494,-2.07301,0.086653,0.719507,-2.09038,0.064904,0.655372,-2.0767,0.011488,0.433215,-2.02563,-0.089756,0.364995,-1.96816,0.193011,0.375083,-2.00068,0.14205,0.338317,-2.06598,0.107214,1.26548,-1.85689,0.088407,1.30929,-2.06678},
/*630*/{0.218003,1.85062,-1.72987,0.281507,1.78481,-1.88537,0.096772,1.7212,-1.64492,0.00193,1.63147,-1.5958,0.137309,1.55534,-1.64929,0.211764,1.5067,-1.66554,0.202852,1.86649,-2.12656,0.283634,1.7898,-1.96324,0.070325,1.8364,-2.02908,0.034741,1.65789,-2.21263,0.064407,1.56521,-2.25512,0.2335,1.49918,-2.18439,0.285863,1.46219,-2.19227,0.181389,1.35032,-1.82509,-0.012161,1.42142,-1.87266,-0.013189,1.45549,-1.925,-0.01894,1.43755,-1.98374,0.169818,1.40524,-2.0888,0.262386,1.08425,-1.76551,0.145953,1.01857,-1.79286,0.186224,0.946368,-1.81487,0.282042,0.881107,-1.79443,0.135951,0.827505,-1.78299,0.062084,0.81394,-1.79123,0.117824,0.934679,-1.81279,0.047563,0.923886,-1.81206,-0.167858,0.838417,-1.80814,-0.250926,0.918326,-1.89169,-0.193739,0.672713,-1.83401,-0.245916,0.696982,-1.76414,0.193082,1.10766,-2.15364,0.222262,1.02788,-2.13936,0.06556,1.0629,-2.10049,0.094899,0.987145,-2.08808,0.197902,0.868883,-2.09475,0.176859,0.675924,-2.07206,0.072387,0.717708,-2.09126,0.048441,0.653605,-2.0771,-0.009511,0.434112,-2.02563,-0.112953,0.365303,-1.9704,0.17168,0.374145,-2.00167,0.120653,0.339132,-2.06633,0.105667,1.26674,-1.85593,0.088159,1.30907,-2.06624},
/*631*/{0.214706,1.85123,-1.72895,0.27883,1.7854,-1.88385,0.091351,1.7252,-1.64511,-0.006661,1.63701,-1.59805,0.129167,1.55883,-1.64809,0.202929,1.50912,-1.66335,0.204894,1.86762,-2.12611,0.283036,1.79031,-1.96147,0.071016,1.83751,-2.0296,0.039548,1.65786,-2.21268,0.072346,1.5652,-2.25509,0.241405,1.50279,-2.18146,0.296029,1.46826,-2.18819,0.179204,1.35175,-1.82403,-0.012688,1.42214,-1.87381,-0.014547,1.45602,-1.92625,-0.019189,1.43843,-1.9859,0.170316,1.40473,-2.08871,0.269159,1.08957,-1.7638,0.153077,1.02094,-1.79108,0.194927,0.950549,-1.81334,0.293646,0.887384,-1.7938,0.149106,0.828808,-1.78302,0.074993,0.814383,-1.79161,0.127398,0.93628,-1.81007,0.056439,0.922107,-1.81102,-0.153439,0.832049,-1.81003,-0.242074,0.906375,-1.89151,-0.172266,0.664466,-1.83464,-0.226063,0.686806,-1.7651,0.18748,1.10505,-2.15343,0.213912,1.02461,-2.13878,0.057457,1.06418,-2.10221,0.086098,0.987276,-2.08839,0.184079,0.864723,-2.09453,0.160628,0.672478,-2.07115,0.056997,0.715792,-2.09153,0.032169,0.651675,-2.07616,-0.030219,0.435642,-2.02596,-0.133657,0.367768,-1.96992,0.149844,0.375616,-2.00118,0.099698,0.339553,-2.06668,0.10406,1.26785,-1.8553,0.088451,1.30886,-2.06601},
/*632*/{0.211869,1.85271,-1.7276,0.278683,1.78655,-1.88214,0.0853,1.72924,-1.64576,-0.013328,1.64181,-1.59941,0.121738,1.56276,-1.64724,0.194282,1.51194,-1.66133,0.206284,1.86973,-2.12518,0.283117,1.79107,-1.95985,0.071615,1.83937,-2.03074,0.044724,1.65718,-2.21222,0.079353,1.56564,-2.25481,0.249667,1.50793,-2.17827,0.304713,1.47538,-2.18458,0.178288,1.35299,-1.82325,-0.014631,1.42388,-1.8755,-0.013975,1.45641,-1.92713,-0.019313,1.43865,-1.98723,0.17075,1.40472,-2.0888,0.274018,1.09741,-1.76429,0.161264,1.02727,-1.79078,0.203656,0.955282,-1.81224,0.305205,0.894015,-1.79362,0.160943,0.830973,-1.78351,0.089136,0.814979,-1.79181,0.136351,0.938401,-1.80881,0.066272,0.922036,-1.80969,-0.147094,0.826649,-1.81081,-0.233648,0.89499,-1.89207,-0.150845,0.65607,-1.83628,-0.205243,0.675107,-1.76569,0.181266,1.10263,-2.15271,0.205556,1.02175,-2.13802,0.050095,1.06481,-2.10228,0.074783,0.990515,-2.09082,0.168939,0.860365,-2.09433,0.14289,0.671106,-2.07007,0.040904,0.714895,-2.09111,0.015422,0.650686,-2.07589,-0.049399,0.436911,-2.02679,-0.154977,0.368593,-1.97146,0.128166,0.375818,-2.00264,0.076965,0.339557,-2.06739,0.103046,1.26906,-1.85477,0.088869,1.30882,-2.06583},
/*633*/{0.208792,1.85543,-1.72718,0.277344,1.78795,-1.87986,0.079191,1.73373,-1.64692,-0.021857,1.64757,-1.6013,0.112729,1.56722,-1.64674,0.18543,1.51584,-1.65953,0.208955,1.87204,-2.12374,0.282905,1.79319,-1.95818,0.072444,1.84014,-2.03223,0.049192,1.65728,-2.21137,0.086858,1.56644,-2.25482,0.257306,1.51404,-2.17522,0.3139,1.48285,-2.18,0.176608,1.35488,-1.82152,-0.015277,1.42472,-1.87711,-0.014558,1.45773,-1.92867,-0.019627,1.44,-1.98925,0.171177,1.40449,-2.08855,0.280255,1.10421,-1.76251,0.168393,1.03097,-1.7895,0.213774,0.961579,-1.81143,0.316064,0.901162,-1.7933,0.173475,0.832644,-1.78488,0.102961,0.813174,-1.79252,0.144428,0.938656,-1.80683,0.075279,0.919872,-1.80852,-0.127765,0.814552,-1.81212,-0.22501,0.881427,-1.89207,-0.129316,0.647682,-1.83804,-0.18371,0.663403,-1.76688,0.174178,1.10115,-2.15223,0.196631,1.01959,-2.13657,0.041783,1.0668,-2.10264,0.066329,0.987761,-2.08966,0.152945,0.857918,-2.0941,0.126532,0.66999,-2.06824,0.024544,0.714988,-2.09053,-0.00086,0.65145,-2.07499,-0.068141,0.438516,-2.02628,-0.176373,0.369043,-1.9728,0.107284,0.374457,-2.00199,0.055448,0.339545,-2.06744,0.101534,1.27069,-1.85393,0.089398,1.3089,-2.0654},
/*634*/{0.204999,1.85816,-1.72671,0.276887,1.79021,-1.87782,0.073213,1.73821,-1.64683,-0.029016,1.654,-1.60319,0.10581,1.57306,-1.64691,0.177247,1.52038,-1.65846,0.211221,1.87481,-2.12255,0.282864,1.79501,-1.95581,0.073807,1.84132,-2.03293,0.053701,1.65748,-2.21012,0.09334,1.56795,-2.25475,0.264781,1.52139,-2.17225,0.321501,1.49171,-2.176,0.175759,1.35796,-1.82112,-0.016293,1.42642,-1.8789,-0.014762,1.45985,-1.93042,-0.019674,1.44145,-1.99135,0.171215,1.40441,-2.08863,0.283519,1.11307,-1.76278,0.174356,1.03595,-1.7893,0.223148,0.967431,-1.81132,0.325723,0.908029,-1.79314,0.186908,0.834844,-1.78357,0.114608,0.812246,-1.792,0.152496,0.940385,-1.80569,0.08339,0.918573,-1.80714,-0.116036,0.805498,-1.8135,-0.215501,0.867686,-1.89216,-0.107086,0.638735,-1.83926,-0.162685,0.651048,-1.76837,0.167527,1.0998,-2.14997,0.188078,1.01726,-2.13454,0.033844,1.06863,-2.10341,0.056147,0.990048,-2.09023,0.137444,0.8562,-2.09324,0.110004,0.669475,-2.06748,0.008295,0.716446,-2.08925,-0.018781,0.652578,-2.07428,-0.088691,0.440298,-2.0263,-0.198372,0.372896,-1.97266,0.085682,0.374814,-2.00268,0.034936,0.339459,-2.06741,0.100596,1.27329,-1.8532,0.089667,1.30922,-2.06513},
/*635*/{0.201654,1.86133,-1.72563,0.27448,1.79312,-1.87567,0.067271,1.74343,-1.6475,-0.037283,1.65988,-1.60457,0.097548,1.57841,-1.64701,0.169607,1.52523,-1.65631,0.213307,1.87779,-2.12096,0.282758,1.79772,-1.95376,0.074474,1.84341,-2.03414,0.057642,1.65824,-2.20973,0.100533,1.57013,-2.25528,0.272528,1.52889,-2.1696,0.329589,1.50082,-2.17138,0.174925,1.36038,-1.82056,-0.01685,1.42848,-1.88087,-0.014146,1.46152,-1.93233,-0.019044,1.44401,-1.99319,0.172325,1.4042,-2.08835,0.287604,1.12071,-1.76215,0.183389,1.04163,-1.78808,0.229608,0.97368,-1.81064,0.336465,0.916356,-1.79276,0.198468,0.836804,-1.78416,0.127619,0.811918,-1.79276,0.160201,0.94091,-1.80384,0.092532,0.916304,-1.80617,-0.101047,0.793309,-1.81235,-0.205027,0.852766,-1.89262,-0.084734,0.630303,-1.83989,-0.141285,0.638791,-1.76985,0.161276,1.09811,-2.14846,0.178391,1.01554,-2.13244,0.02605,1.07148,-2.10361,0.044291,0.991539,-2.09037,0.119756,0.856528,-2.09222,0.091362,0.669229,-2.06602,-0.009348,0.717984,-2.08825,-0.036926,0.654558,-2.07378,-0.11168,0.441438,-2.02763,-0.218964,0.375618,-1.97347,0.064177,0.374687,-2.00284,0.01258,0.339621,-2.06739,0.099423,1.27552,-1.85269,0.090421,1.30982,-2.06499},
/*636*/{0.19775,1.86465,-1.72504,0.274746,1.79669,-1.87368,0.061886,1.74894,-1.64806,-0.04527,1.66644,-1.60679,0.090418,1.5846,-1.64706,0.161765,1.53133,-1.65603,0.215604,1.88084,-2.12001,0.282152,1.80007,-1.95122,0.075379,1.8447,-2.0343,0.062678,1.65899,-2.20916,0.10797,1.57203,-2.25524,0.278202,1.53535,-2.16565,0.337333,1.51002,-2.16684,0.174662,1.36297,-1.81973,-0.016597,1.43002,-1.88293,-0.01377,1.46386,-1.93339,-0.018779,1.44542,-1.99528,0.173154,1.4043,-2.08816,0.291668,1.12866,-1.7613,0.188119,1.04626,-1.78816,0.236389,0.97887,-1.80932,0.345134,0.924796,-1.7927,0.210055,0.838731,-1.78447,0.139809,0.810177,-1.79375,0.167144,0.9415,-1.80255,0.100538,0.914174,-1.80503,-0.086806,0.782602,-1.81724,-0.194616,0.83733,-1.89336,-0.062635,0.621177,-1.8414,-0.119035,0.626783,-1.77175,0.154544,1.09729,-2.14648,0.1691,1.01439,-2.13019,0.01756,1.07467,-2.10384,0.034189,0.994646,-2.09022,0.103288,0.85771,-2.09127,0.073408,0.670555,-2.06508,-0.026182,0.720054,-2.08746,-0.054936,0.657132,-2.07305,-0.131497,0.444783,-2.02764,-0.240148,0.378357,-1.97382,0.042497,0.374584,-2.00275,-0.009627,0.339681,-2.06669,0.09899,1.27787,-1.85202,0.091333,1.31032,-2.06466},
/*637*/{0.194214,1.86833,-1.72419,0.274166,1.79841,-1.87038,0.056594,1.75437,-1.64883,-0.051356,1.67321,-1.60895,0.083971,1.59177,-1.64741,0.153838,1.53783,-1.65515,0.217182,1.88454,-2.11814,0.282436,1.80387,-1.94911,0.076053,1.84736,-2.03543,0.067809,1.6606,-2.20807,0.114749,1.57472,-2.2554,0.285012,1.54393,-2.16215,0.34522,1.52076,-2.16201,0.174696,1.36647,-1.81939,-0.016358,1.4335,-1.8838,-0.01352,1.46637,-1.93499,-0.017006,1.44769,-1.99682,0.173563,1.40458,-2.08796,0.296785,1.13603,-1.76079,0.194972,1.05115,-1.78765,0.244591,0.985937,-1.80961,0.354305,0.933881,-1.79238,0.221956,0.840973,-1.7837,0.153228,0.809288,-1.79395,0.173555,0.941536,-1.80079,0.108589,0.910795,-1.80354,-0.073783,0.774181,-1.81391,-0.183214,0.821128,-1.89413,-0.039006,0.611939,-1.84251,-0.097137,0.613892,-1.77315,0.147912,1.09645,-2.14411,0.160003,1.01302,-2.12732,0.009354,1.07846,-2.10441,0.023622,0.997189,-2.09,0.087387,0.859256,-2.08966,0.056136,0.671357,-2.06373,-0.044308,0.722247,-2.08598,-0.073802,0.659922,-2.072,-0.151718,0.448285,-2.02775,-0.261153,0.381967,-1.97436,0.020409,0.374659,-2.00191,-0.031618,0.339482,-2.06613,0.098667,1.28126,-1.85085,0.091887,1.31116,-2.06389},
/*638*/{0.190096,1.8722,-1.72413,0.272879,1.80246,-1.86954,0.050834,1.76014,-1.6496,-0.058852,1.68013,-1.61142,0.076934,1.59844,-1.64807,0.146899,1.54336,-1.65448,0.218564,1.8884,-2.11708,0.282575,1.80727,-1.94722,0.077517,1.84923,-2.03609,0.072233,1.66258,-2.20805,0.121578,1.5777,-2.25524,0.290853,1.55186,-2.15839,0.351556,1.53056,-2.15766,0.17441,1.37034,-1.81897,-0.015662,1.43598,-1.88632,-0.011876,1.46929,-1.93657,-0.015868,1.45107,-1.99879,0.175742,1.40416,-2.08739,0.298709,1.14405,-1.76117,0.200905,1.05612,-1.78698,0.251437,0.991756,-1.81043,0.36299,0.94326,-1.79208,0.232999,0.842218,-1.78406,0.166029,0.808085,-1.79378,0.179986,0.941403,-1.79977,0.114918,0.908345,-1.80438,-0.060148,0.761877,-1.81513,-0.171163,0.804382,-1.89551,-0.016402,0.603534,-1.84397,-0.0748,0.600813,-1.77477,0.140836,1.09611,-2.14178,0.150197,1.01256,-2.12469,0.002196,1.08204,-2.10426,0.012831,1.0004,-2.089,0.071965,0.860893,-2.08738,0.037737,0.673709,-2.06274,-0.061228,0.725665,-2.085,-0.090802,0.663345,-2.07089,-0.172456,0.451541,-2.02773,-0.280783,0.388551,-1.97398,-0.000154,0.375276,-2.00151,-0.053626,0.339782,-2.06605,0.098157,1.28461,-1.84987,0.093334,1.31203,-2.0633},
/*639*/{0.186829,1.87632,-1.72328,0.270592,1.8077,-1.86723,0.045328,1.76599,-1.65053,-0.065614,1.68661,-1.61352,0.070468,1.6053,-1.64885,0.140397,1.55105,-1.65414,0.220548,1.89231,-2.11568,0.282078,1.81103,-1.94452,0.078394,1.85189,-2.03706,0.076692,1.66492,-2.20806,0.128231,1.58116,-2.25534,0.297469,1.56094,-2.15503,0.357619,1.54066,-2.15277,0.175097,1.37392,-1.81914,-0.014752,1.44015,-1.8879,-0.01077,1.47181,-1.93817,-0.014576,1.45338,-2.0006,0.176309,1.40509,-2.08749,0.303183,1.15166,-1.7598,0.205193,1.06158,-1.78771,0.256826,0.999706,-1.81161,0.370045,0.952459,-1.79257,0.243596,0.844398,-1.78423,0.179394,0.807171,-1.79406,0.186146,0.940657,-1.79832,0.124319,0.903314,-1.80301,-0.04486,0.750496,-1.81524,-0.158586,0.787498,-1.89601,0.007043,0.594742,-1.84456,-0.052008,0.588098,-1.77676,0.134195,1.09628,-2.13968,0.140768,1.0126,-2.12145,-0.006055,1.08594,-2.10403,0.002551,1.00449,-2.08899,0.057917,0.86315,-2.08548,0.02058,0.676618,-2.06091,-0.077682,0.729303,-2.08356,-0.108682,0.66702,-2.07007,-0.196209,0.454856,-2.02646,-0.300222,0.394397,-1.97484,-0.021458,0.3751,-2.00103,-0.074059,0.340185,-2.06647,0.098401,1.28829,-1.84925,0.094084,1.31329,-2.06299},
/*640*/{0.18279,1.88058,-1.72317,0.271573,1.81058,-1.86461,0.040529,1.77163,-1.65158,-0.070372,1.69486,-1.61619,0.06373,1.61247,-1.64986,0.133334,1.55648,-1.65238,0.222462,1.89693,-2.11425,0.282299,1.81452,-1.9429,0.079291,1.85545,-2.03743,0.082262,1.66728,-2.20867,0.134972,1.58501,-2.25546,0.301855,1.5688,-2.15099,0.364128,1.55044,-2.14737,0.176429,1.37839,-1.81898,-0.013905,1.44388,-1.88856,-0.010055,1.47498,-1.9401,-0.012456,1.4567,-2.00254,0.177534,1.40528,-2.08756,0.305254,1.15857,-1.75953,0.207612,1.06785,-1.78848,0.262467,1.00616,-1.81062,0.376821,0.962318,-1.79184,0.25398,0.846438,-1.78372,0.19063,0.805105,-1.79392,0.191658,0.939745,-1.79773,0.133321,0.89902,-1.80257,-0.029832,0.739669,-1.81602,-0.145386,0.769873,-1.89648,0.030745,0.586616,-1.84624,-0.028728,0.576051,-1.77785,0.127552,1.09745,-2.13648,0.131842,1.01271,-2.11843,-0.01312,1.09095,-2.10419,-0.007098,1.00878,-2.08879,0.044594,0.865337,-2.08286,0.0036,0.680349,-2.06019,-0.094297,0.733331,-2.08242,-0.126444,0.67075,-2.06898,-0.206694,0.45897,-2.02647,-0.320203,0.401192,-1.97425,-0.04343,0.375096,-2.00121,-0.096591,0.340668,-2.06521,0.098892,1.29256,-1.84803,0.095127,1.31447,-2.06211},
/*641*/{0.180129,1.88521,-1.72251,0.270346,1.81484,-1.86345,0.035646,1.77773,-1.65271,-0.079267,1.70312,-1.61938,0.057833,1.61958,-1.6506,0.126957,1.5639,-1.6523,0.223512,1.90138,-2.11254,0.283583,1.81818,-1.94032,0.080729,1.85863,-2.03834,0.086521,1.67184,-2.21045,0.141506,1.58892,-2.25548,0.306671,1.57739,-2.14748,0.369607,1.56112,-2.14271,0.177142,1.38285,-1.82006,-0.013912,1.44806,-1.89087,-0.008221,1.4788,-1.94171,-0.010087,1.46019,-2.00252,0.17971,1.40496,-2.08724,0.307484,1.16655,-1.75958,0.211807,1.0744,-1.78773,0.26775,1.01396,-1.80942,0.383599,0.972087,-1.79176,0.264593,0.848247,-1.78376,0.203426,0.803674,-1.7941,0.197214,0.938307,-1.79693,0.140117,0.896009,-1.80261,-0.016592,0.728113,-1.81617,-0.130602,0.752556,-1.89644,0.055309,0.578839,-1.84734,-0.003916,0.563806,-1.77973,0.121471,1.0977,-2.13349,0.122694,1.0135,-2.11526,-0.020872,1.09575,-2.10371,-0.016616,1.01382,-2.08794,0.03123,0.868432,-2.08034,-0.013948,0.684786,-2.05957,-0.111626,0.738237,-2.08144,-0.143659,0.676316,-2.06881,-0.227535,0.466476,-2.02617,-0.338378,0.408892,-1.97456,-0.064592,0.375682,-1.99985,-0.118027,0.34101,-2.06423,0.09884,1.2968,-1.84691,0.095996,1.31576,-2.06129},
/*642*/{0.176245,1.88919,-1.72288,0.268616,1.81877,-1.86121,0.030969,1.78421,-1.65391,-0.083185,1.71059,-1.62184,0.051787,1.62616,-1.65116,0.121496,1.57111,-1.65176,0.224816,1.9063,-2.11075,0.28315,1.82221,-1.93874,0.081931,1.8623,-2.03943,0.091985,1.67547,-2.21193,0.138241,1.59359,-2.25883,0.312254,1.58647,-2.14293,0.374368,1.57176,-2.13756,0.177527,1.38795,-1.8203,-0.012407,1.45278,-1.89058,-0.00752,1.48218,-1.94345,-0.007965,1.46396,-2.00367,0.180427,1.40577,-2.08724,0.31077,1.17469,-1.75918,0.21417,1.08016,-1.78826,0.272022,1.02104,-1.80954,0.388674,0.981267,-1.79208,0.274655,0.849372,-1.78396,0.215756,0.802948,-1.79492,0.202971,0.935886,-1.79659,0.14821,0.890652,-1.80132,0.001509,0.715781,-1.81677,-0.116008,0.734659,-1.8972,0.079725,0.570836,-1.84785,0.021623,0.552521,-1.78122,0.115509,1.09889,-2.13084,0.113852,1.0147,-2.11227,-0.027232,1.10091,-2.10326,-0.026253,1.01928,-2.08816,0.018561,0.872711,-2.07817,-0.030228,0.689171,-2.05919,-0.127672,0.742933,-2.08039,-0.159594,0.681759,-2.06807,-0.248274,0.471044,-2.02752,-0.35726,0.416991,-1.97352,-0.08543,0.376099,-1.99934,-0.138902,0.341481,-2.06423,0.099189,1.30166,-1.84549,0.096447,1.31753,-2.06012},
/*643*/{0.173484,1.89355,-1.72281,0.267652,1.82466,-1.86002,0.026047,1.78998,-1.65535,-0.089719,1.71687,-1.62468,0.045578,1.63318,-1.65245,0.116084,1.57724,-1.65194,0.226725,1.91106,-2.10976,0.283367,1.82707,-1.93653,0.082788,1.8648,-2.03949,0.097634,1.6802,-2.2141,0.155908,1.5978,-2.25557,0.316267,1.59603,-2.14017,0.378522,1.58136,-2.13251,0.178458,1.39202,-1.82016,-0.010667,1.45723,-1.89201,-0.005316,1.48581,-1.94421,-0.005456,1.46727,-2.00467,0.183492,1.40548,-2.08686,0.312973,1.18205,-1.75926,0.214997,1.08669,-1.78826,0.276382,1.02842,-1.80921,0.392841,0.990339,-1.79237,0.284817,0.850897,-1.78295,0.228216,0.801914,-1.79418,0.209536,0.934356,-1.7958,0.15633,0.886577,-1.80143,0.016576,0.703176,-1.81712,-0.101247,0.716781,-1.8967,0.104087,0.564044,-1.85023,0.04738,0.541155,-1.78269,0.109605,1.1003,-2.12874,0.10517,1.01652,-2.10887,-0.033142,1.10679,-2.10247,-0.034314,1.02494,-2.08688,0.004606,0.878148,-2.07677,-0.046674,0.694373,-2.05885,-0.144176,0.749731,-2.08012,-0.176505,0.687764,-2.0674,-0.265761,0.476775,-2.02742,-0.375769,0.427132,-1.97372,-0.108326,0.375433,-1.99773,-0.160896,0.342241,-2.06325,0.099591,1.30585,-1.84399,0.097955,1.31878,-2.05884},
/*644*/{0.171121,1.89743,-1.7231,0.267917,1.82864,-1.85792,0.021984,1.79574,-1.65688,-0.09473,1.72446,-1.6279,0.040526,1.64042,-1.6537,0.110067,1.58461,-1.65142,0.229098,1.91615,-2.10771,0.28279,1.83086,-1.93472,0.084172,1.86857,-2.04024,0.103931,1.68445,-2.21617,0.163127,1.60329,-2.25589,0.319339,1.60361,-2.13627,0.382415,1.5917,-2.12742,0.180505,1.39619,-1.82149,-0.008705,1.46253,-1.89185,-0.00316,1.48968,-1.94571,-0.003384,1.47133,-2.00481,0.184843,1.40605,-2.08649,0.314813,1.18849,-1.75906,0.223729,1.09263,-1.78643,0.279339,1.03488,-1.80922,0.395993,0.998378,-1.79211,0.293569,0.851939,-1.78326,0.240332,0.799566,-1.79426,0.214496,0.931526,-1.79542,0.163332,0.880852,-1.80085,0.03448,0.69019,-1.81715,-0.084813,0.697713,-1.89594,0.129653,0.557219,-1.85095,0.074337,0.531318,-1.78461,0.104401,1.10168,-2.12554,0.096893,1.01831,-2.10577,-0.038547,1.11285,-2.10262,-0.043157,1.03134,-2.08733,-0.009946,0.883584,-2.07628,-0.06256,0.699392,-2.0582,-0.158624,0.756124,-2.0793,-0.192265,0.694848,-2.06745,-0.281141,0.483357,-2.02681,-0.39324,0.436316,-1.97379,-0.130183,0.376138,-1.99801,-0.18363,0.342539,-2.06262,0.100584,1.31048,-1.8429,0.098293,1.32073,-2.05788},
/*645*/{0.168127,1.90157,-1.72279,0.266573,1.83389,-1.8568,0.018105,1.80209,-1.65835,-0.099476,1.73176,-1.63108,0.035152,1.64685,-1.65459,0.104345,1.59081,-1.65116,0.230399,1.92079,-2.10643,0.283294,1.83536,-1.93253,0.085133,1.8723,-2.04111,0.10927,1.68878,-2.21748,0.16941,1.60819,-2.25598,0.322823,1.61141,-2.13287,0.386226,1.60157,-2.12338,0.181414,1.40097,-1.82201,-0.00783,1.46739,-1.89271,-0.001866,1.4936,-1.94743,-0.000894,1.47472,-2.00617,0.185484,1.4067,-2.08595,0.316226,1.19572,-1.75902,0.224398,1.09913,-1.78587,0.282513,1.04179,-1.8085,0.398668,1.00599,-1.79221,0.302828,0.852497,-1.78346,0.251583,0.797596,-1.79462,0.220475,0.928352,-1.79503,0.172845,0.87528,-1.80149,0.051818,0.677958,-1.81762,-0.066621,0.678057,-1.89535,0.153989,0.551185,-1.8532,0.100813,0.521141,-1.7863,0.09921,1.10357,-2.12282,0.088755,1.02029,-2.10313,-0.044415,1.11932,-2.10236,-0.051379,1.03794,-2.08683,-0.024615,0.888966,-2.07623,-0.078261,0.705214,-2.05774,-0.174056,0.763167,-2.07844,-0.206511,0.70186,-2.06707,-0.301091,0.490304,-2.02795,-0.409741,0.447791,-1.97026,-0.152431,0.376712,-1.99692,-0.203636,0.343504,-2.06211,0.100767,1.3154,-1.84165,0.098784,1.32246,-2.05676},
/*646*/{0.166124,1.90546,-1.7226,0.26682,1.8377,-1.85521,0.014273,1.80802,-1.66067,-0.105278,1.7384,-1.6342,0.03029,1.65292,-1.65498,0.098892,1.59737,-1.65128,0.231016,1.92545,-2.10429,0.282709,1.83939,-1.93139,0.086557,1.87559,-2.04149,0.114503,1.69372,-2.21962,0.176466,1.61303,-2.25616,0.326847,1.6209,-2.12942,0.389347,1.61096,-2.11936,0.182408,1.4054,-1.82152,-0.006542,1.47251,-1.89278,-2e-005,1.4976,-1.94819,0.002016,1.47826,-2.00598,0.188162,1.40693,-2.08505,0.317937,1.2026,-1.75891,0.226638,1.10454,-1.78474,0.285864,1.04835,-1.80785,0.399258,1.01198,-1.79318,0.312631,0.853452,-1.78373,0.26361,0.795138,-1.79485,0.226036,0.924776,-1.79502,0.179912,0.868662,-1.80076,0.071246,0.664567,-1.81803,-0.049358,0.65931,-1.89495,0.179821,0.545038,-1.85377,0.12863,0.511394,-1.78666,0.093609,1.10517,-2.12085,0.08063,1.02327,-2.10019,-0.048761,1.12582,-2.10158,-0.059261,1.04495,-2.08616,-0.038119,0.894851,-2.07551,-0.092383,0.712144,-2.05826,-0.188601,0.770943,-2.07868,-0.221414,0.709494,-2.06707,-0.314697,0.497286,-2.0291,-0.426085,0.457901,-1.97084,-0.174432,0.376755,-1.99714,-0.225147,0.343779,-2.06119,0.101185,1.32014,-1.83957,0.099971,1.32424,-2.05476},
/*647*/{0.164089,1.9093,-1.7227,0.265478,1.84167,-1.85379,0.011464,1.81366,-1.66267,-0.109276,1.74564,-1.63796,0.025595,1.65986,-1.65657,0.094382,1.60293,-1.65098,0.232604,1.93008,-2.10294,0.283203,1.84406,-1.9297,0.087257,1.87881,-2.04189,0.120506,1.69808,-2.22134,0.182659,1.61756,-2.25611,0.329021,1.62784,-2.12575,0.392023,1.61989,-2.11563,0.183199,1.40993,-1.82205,-0.004188,1.47742,-1.89215,0.002015,1.50167,-1.9492,0.00453,1.48164,-2.00634,0.188948,1.40744,-2.08457,0.31979,1.20872,-1.75889,0.224584,1.10922,-1.78661,0.287424,1.05404,-1.80723,0.399585,1.01656,-1.79356,0.320708,0.853374,-1.78424,0.27582,0.793627,-1.79525,0.230794,0.920338,-1.79473,0.188484,0.863243,-1.80042,0.090844,0.6547,-1.81792,-0.029929,0.640406,-1.89439,0.205568,0.540397,-1.85563,0.157153,0.503443,-1.78828,0.087744,1.10787,-2.11871,0.072764,1.02603,-2.09771,-0.054593,1.13312,-2.10016,-0.066442,1.05233,-2.08521,-0.050578,0.90148,-2.07562,-0.106824,0.72005,-2.0593,-0.202283,0.77829,-2.0779,-0.235546,0.717341,-2.06697,-0.331827,0.503732,-2.02952,-0.439745,0.471129,-1.96922,-0.195789,0.37687,-1.99727,-0.247071,0.344716,-2.06086,0.101845,1.32491,-1.83796,0.10018,1.32603,-2.05318},
/*648*/{0.162266,1.9128,-1.72288,0.25923,1.84918,-1.85455,0.007851,1.81946,-1.6651,-0.113776,1.75238,-1.64186,0.020393,1.66522,-1.65751,0.089445,1.60855,-1.65074,0.233381,1.93424,-2.10127,0.282794,1.84892,-1.92863,0.088875,1.88157,-2.04234,0.125697,1.70233,-2.22267,0.188849,1.62252,-2.25616,0.331415,1.63538,-2.12289,0.395337,1.62859,-2.1119,0.184259,1.41464,-1.82258,-0.002036,1.48255,-1.89228,0.004073,1.50543,-1.95042,0.007065,1.48577,-2.00645,0.190314,1.40863,-2.08369,0.321118,1.21413,-1.7588,0.230556,1.11621,-1.78313,0.287753,1.05886,-1.80737,0.400111,1.02218,-1.79444,0.329195,0.852929,-1.78394,0.286749,0.790445,-1.79589,0.236041,0.914647,-1.79437,0.196446,0.855584,-1.80119,0.10789,0.641372,-1.81917,-0.010767,0.622002,-1.89375,0.231378,0.53518,-1.85645,0.186817,0.496296,-1.78868,0.083066,1.11009,-2.11711,0.064903,1.02946,-2.09623,-0.057958,1.14059,-2.10119,-0.073025,1.05989,-2.0851,-0.062305,0.908878,-2.07545,-0.119751,0.726933,-2.06018,-0.215913,0.786438,-2.07827,-0.249463,0.725285,-2.06756,-0.343485,0.510828,-2.03082,-0.452766,0.483563,-1.96775,-0.218209,0.377065,-1.99723,-0.269322,0.345992,-2.0607,0.102634,1.32985,-1.83661,0.100977,1.32839,-2.05184},
/*649*/{0.160796,1.91642,-1.72305,0.259819,1.85282,-1.85374,0.005032,1.82512,-1.66698,-0.117727,1.75902,-1.64558,0.016665,1.67091,-1.65906,0.08448,1.61385,-1.65083,0.234865,1.9385,-2.10004,0.283146,1.8524,-1.92731,0.089886,1.8844,-2.04252,0.130561,1.70681,-2.22453,0.19482,1.62729,-2.25686,0.333072,1.64224,-2.11977,0.397327,1.63687,-2.10761,0.186218,1.41839,-1.82311,-0.000834,1.48726,-1.89218,0.005552,1.50934,-1.9512,0.009154,1.48878,-2.00657,0.191693,1.40912,-2.08295,0.320853,1.2192,-1.75923,0.229812,1.11816,-1.78187,0.287739,1.06374,-1.80715,0.398507,1.02564,-1.79455,0.338019,0.852282,-1.78483,0.298363,0.788473,-1.79593,0.240977,0.909495,-1.79466,0.203686,0.848377,-1.80213,0.128086,0.628863,-1.81973,0.009382,0.603032,-1.89395,0.255549,0.530778,-1.85826,0.212634,0.488677,-1.79056,0.078294,1.11252,-2.11584,0.057345,1.03251,-2.09471,-0.062256,1.14725,-2.101,-0.079607,1.06825,-2.08496,-0.073308,0.916852,-2.07601,-0.131253,0.735476,-2.06118,-0.228396,0.794291,-2.07787,-0.260205,0.731791,-2.06736,-0.358537,0.5161,-2.0306,-0.463998,0.497166,-1.96777,-0.239921,0.37748,-1.99748,-0.29349,0.346978,-2.06187,0.103497,1.33424,-1.8352,0.101421,1.33017,-2.05039},
/*650*/{0.158487,1.91931,-1.72344,0.264351,1.85389,-1.85044,0.002048,1.83033,-1.66922,-0.121349,1.76517,-1.64926,0.01332,1.67653,-1.65999,0.08037,1.61882,-1.65044,0.23644,1.94221,-2.09895,0.283548,1.85561,-1.92545,0.091275,1.88718,-2.04348,0.135701,1.71093,-2.22633,0.200373,1.63154,-2.25677,0.336279,1.65071,-2.11674,0.399242,1.64525,-2.10479,0.186798,1.42267,-1.82266,0.00148,1.49174,-1.89184,0.007421,1.51289,-1.95199,0.011732,1.49186,-2.00604,0.193402,1.4098,-2.08174,0.322479,1.22388,-1.75939,0.233153,1.1261,-1.78034,0.288802,1.06834,-1.80576,0.39684,1.0292,-1.79575,0.3454,0.851034,-1.78469,0.3092,0.785598,-1.79667,0.24608,0.904077,-1.79461,0.211643,0.841053,-1.80103,0.147771,0.617681,-1.82009,0.030529,0.584918,-1.8942,0.279859,0.527247,-1.85848,0.240308,0.481855,-1.79203,0.074109,1.11525,-2.11477,0.050661,1.03498,-2.09325,-0.065059,1.15362,-2.10072,-0.08556,1.07556,-2.08438,-0.084569,0.923629,-2.07616,-0.14342,0.743705,-2.06354,-0.240588,0.802333,-2.07829,-0.274333,0.740131,-2.06797,-0.365608,0.523994,-2.03181,-0.474207,0.51222,-1.96723,-0.260155,0.378048,-1.99857,-0.315391,0.348801,-2.06005,0.104183,1.33873,-1.83322,0.102405,1.33198,-2.04834},
/*651*/{0.157151,1.9224,-1.72397,0.259803,1.86069,-1.85081,-1.1e-005,1.83562,-1.67074,-0.125809,1.77082,-1.65353,0.008622,1.68124,-1.66113,0.076918,1.62385,-1.64995,0.237893,1.94576,-2.09708,0.283205,1.85991,-1.92489,0.092749,1.88888,-2.04385,0.140932,1.71579,-2.22782,0.205273,1.63558,-2.25717,0.336836,1.65642,-2.11422,0.400655,1.65203,-2.10078,0.188179,1.42654,-1.82254,0.003963,1.49649,-1.8912,0.009996,1.5173,-1.95228,0.012893,1.4953,-2.00625,0.194323,1.41122,-2.08102,0.323475,1.22811,-1.75973,0.229326,1.12827,-1.78,0.287308,1.07222,-1.8043,0.3947,1.03173,-1.79573,0.352307,0.84965,-1.78498,0.322112,0.783695,-1.79701,0.251058,0.897646,-1.795,0.220494,0.832583,-1.80047,0.167779,0.60657,-1.82039,0.051465,0.567915,-1.89464,0.304284,0.524074,-1.85936,0.267825,0.476708,-1.792,0.069035,1.11797,-2.11416,0.043015,1.04048,-2.09385,-0.06768,1.16085,-2.1005,-0.091531,1.08242,-2.08468,-0.094237,0.931561,-2.07712,-0.154386,0.750974,-2.06513,-0.2513,0.810828,-2.07861,-0.285457,0.74761,-2.06684,-0.37634,0.530677,-2.03298,-0.484311,0.52748,-1.96741,-0.278743,0.377818,-1.99966,-0.337191,0.351733,-2.0609,0.10505,1.34324,-1.83186,0.10263,1.33443,-2.0469},
/*652*/{0.156222,1.92495,-1.72433,0.259086,1.86412,-1.84986,-0.002271,1.84021,-1.67449,-0.129387,1.77586,-1.65824,0.005919,1.68576,-1.66222,0.073255,1.62792,-1.64939,0.237936,1.94831,-2.0963,0.283818,1.8627,-1.9233,0.093926,1.89169,-2.04438,0.14522,1.71878,-2.22839,0.209934,1.63995,-2.25763,0.339387,1.66155,-2.1122,0.402133,1.65897,-2.09781,0.18863,1.42981,-1.82229,0.006082,1.50054,-1.89096,0.011579,1.52059,-1.95264,0.015838,1.49806,-2.00543,0.19489,1.41208,-2.07996,0.324473,1.23129,-1.76042,0.228474,1.13154,-1.77866,0.286284,1.07471,-1.80319,0.392471,1.03367,-1.7964,0.359679,0.848489,-1.78451,0.332207,0.780854,-1.79691,0.255549,0.891362,-1.79521,0.22755,0.825568,-1.80167,0.188282,0.597005,-1.82141,0.073545,0.550621,-1.89525,0.328514,0.520578,-1.86006,0.293058,0.471423,-1.79281,0.065123,1.121,-2.11353,0.036261,1.0437,-2.09416,-0.071326,1.16657,-2.1,-0.095984,1.08971,-2.08324,-0.104186,0.939501,-2.07785,-0.164924,0.759818,-2.06703,-0.261296,0.818559,-2.07877,-0.295265,0.756668,-2.06713,-0.386176,0.53867,-2.03448,-0.492213,0.543233,-1.96666,-0.29667,0.378207,-2.00069,-0.357915,0.355622,-2.05916,0.105434,1.34694,-1.83034,0.102963,1.33625,-2.0453},
/*653*/{0.155546,1.92784,-1.72418,0.263964,1.86392,-1.84759,-0.004912,1.84457,-1.6767,-0.131758,1.78125,-1.66159,0.001567,1.69031,-1.66369,0.068502,1.63148,-1.64946,0.239199,1.95121,-2.09472,0.28357,1.86593,-1.92238,0.095676,1.89331,-2.04492,0.149736,1.72242,-2.2298,0.214434,1.64311,-2.25808,0.340427,1.66656,-2.11007,0.403832,1.66439,-2.09485,0.189364,1.43405,-1.82219,0.007835,1.50385,-1.8903,0.013525,1.52399,-1.95258,0.017899,1.50048,-2.00483,0.196112,1.4127,-2.07878,0.324372,1.23316,-1.76134,0.228325,1.13856,-1.77737,0.284023,1.07733,-1.8015,0.390898,1.03411,-1.79558,0.365728,0.846263,-1.78461,0.339226,0.775802,-1.79657,0.260338,0.884598,-1.79605,0.236421,0.816409,-1.80187,0.204059,0.586131,-1.82176,0.095435,0.533719,-1.89623,0.351565,0.518725,-1.86068,0.320266,0.467398,-1.7924,0.061521,1.12387,-2.11356,0.03089,1.04692,-2.09377,-0.074537,1.17264,-2.09953,-0.100262,1.0957,-2.0828,-0.112766,0.94635,-2.07877,-0.173572,0.767513,-2.06851,-0.270178,0.82693,-2.07873,-0.302979,0.764398,-2.06825,-0.397987,0.546468,-2.03731,-0.498609,0.559162,-1.96594,-0.315133,0.378624,-2.00208,-0.380591,0.361299,-2.05854,0.106522,1.35104,-1.8284,0.103792,1.33787,-2.04322},
/*654*/{0.154621,1.93015,-1.72436,0.260601,1.86958,-1.84826,-0.005586,1.84791,-1.67891,-0.134423,1.7863,-1.66554,-0.000955,1.69337,-1.66471,0.064983,1.63459,-1.64897,0.240506,1.95354,-2.09364,0.283941,1.8684,-1.92156,0.096669,1.89571,-2.04502,0.153796,1.72577,-2.2302,0.218734,1.64664,-2.25803,0.341937,1.67137,-2.10888,0.404714,1.67003,-2.09171,0.189679,1.43719,-1.82267,0.008446,1.50693,-1.88945,0.014899,1.52635,-1.95193,0.019438,1.50427,-2.00485,0.197035,1.41403,-2.07767,0.324419,1.23538,-1.76256,0.226588,1.14113,-1.77641,0.282204,1.07937,-1.79757,0.389679,1.03335,-1.79489,0.371858,0.843749,-1.78457,0.348493,0.772638,-1.79743,0.263531,0.875387,-1.79668,0.244317,0.80795,-1.80253,0.224119,0.574896,-1.82237,0.117283,0.517921,-1.89781,0.37451,0.516514,-1.85984,0.343522,0.462795,-1.7941,0.057428,1.12712,-2.11384,0.026356,1.051,-2.09436,-0.07704,1.17774,-2.09816,-0.103723,1.10136,-2.08182,-0.119705,0.952752,-2.07861,-0.181405,0.774923,-2.06996,-0.277929,0.834736,-2.07917,-0.311878,0.772559,-2.06952,-0.409297,0.554948,-2.0384,-0.505283,0.575051,-1.96682,-0.332436,0.3786,-2.00304,-0.401262,0.36735,-2.05735,0.106796,1.35407,-1.82756,0.104132,1.34021,-2.04233},
/*655*/{0.153917,1.93286,-1.72469,0.260503,1.87157,-1.84703,-0.007656,1.85195,-1.68096,-0.137057,1.79053,-1.66975,-0.003999,1.69782,-1.66569,0.062275,1.63786,-1.64909,0.242101,1.95601,-2.09308,0.284349,1.8711,-1.92007,0.098294,1.89731,-2.04541,0.158649,1.72848,-2.23129,0.222082,1.64933,-2.25884,0.342993,1.67542,-2.10729,0.405884,1.67409,-2.08939,0.189965,1.43969,-1.82241,0.010222,1.51043,-1.88813,0.015532,1.53005,-1.95214,0.020598,1.50714,-2.00471,0.197607,1.41487,-2.07697,0.324257,1.2364,-1.76379,0.224224,1.14026,-1.77437,0.279631,1.0793,-1.79742,0.387575,1.0316,-1.79387,0.377451,0.84101,-1.78346,0.359444,0.770025,-1.79608,0.268551,0.868639,-1.79749,0.249918,0.799972,-1.80502,0.243376,0.56527,-1.82351,0.139737,0.502391,-1.89947,0.395519,0.515193,-1.86009,0.368706,0.460693,-1.79204,0.054503,1.13022,-2.11386,0.022481,1.05422,-2.09516,-0.078619,1.18246,-2.09715,-0.10714,1.10719,-2.08117,-0.124545,0.958848,-2.07899,-0.188649,0.781003,-2.07178,-0.284398,0.842123,-2.07917,-0.318605,0.779838,-2.06999,-0.418487,0.564085,-2.0398,-0.509984,0.591128,-1.96825,-0.350167,0.379079,-2.00447,-0.421067,0.374257,-2.05606,0.106675,1.35724,-1.82646,0.103724,1.34223,-2.04115},
/*656*/{0.153622,1.93445,-1.72525,0.264353,1.87041,-1.84507,-0.009008,1.85545,-1.68356,-0.138625,1.79435,-1.6731,-0.006911,1.70043,-1.66677,0.059214,1.64096,-1.64891,0.243781,1.95823,-2.09203,0.28559,1.87227,-1.91881,0.099475,1.89945,-2.04603,0.162184,1.73148,-2.23188,0.225443,1.65208,-2.25936,0.34454,1.68014,-2.10648,0.406957,1.67737,-2.08684,0.189985,1.44186,-1.82196,0.010218,1.51434,-1.88819,0.016625,1.53319,-1.95187,0.021681,1.50971,-2.00311,0.197624,1.41708,-2.07593,0.32464,1.23672,-1.76485,0.222775,1.14212,-1.7716,0.277412,1.08027,-1.79669,0.385269,1.02949,-1.79309,0.381907,0.837486,-1.78279,0.365816,0.763645,-1.79521,0.272121,0.861049,-1.7991,0.256957,0.791019,-1.80571,0.261681,0.555771,-1.8245,0.162055,0.487819,-1.90131,0.417182,0.514177,-1.85828,0.391092,0.457257,-1.79308,0.051562,1.13399,-2.11518,0.018793,1.05782,-2.09704,-0.080365,1.18707,-2.09488,-0.110036,1.11161,-2.07983,-0.127615,0.96351,-2.07839,-0.195013,0.786287,-2.07258,-0.289379,0.849558,-2.07969,-0.324835,0.787299,-2.07046,-0.426817,0.572866,-2.04196,-0.515228,0.606782,-1.96961,-0.368616,0.377596,-2.00275,-0.442494,0.38242,-2.05419,0.106221,1.3601,-1.8256,0.103412,1.34479,-2.04027},
/*657*/{0.153591,1.93657,-1.72532,0.26106,1.87584,-1.84515,-0.009439,1.85879,-1.68528,-0.140585,1.79757,-1.6769,-0.008908,1.70377,-1.66862,0.057138,1.64253,-1.64822,0.244417,1.95966,-2.09115,0.285247,1.87513,-1.91865,0.100315,1.90105,-2.04603,0.165225,1.73367,-2.23247,0.227858,1.6544,-2.25971,0.346227,1.68169,-2.10547,0.407541,1.68048,-2.08515,0.1906,1.44501,-1.82205,0.011658,1.5174,-1.8879,0.0174,1.53565,-1.95122,0.02187,1.51345,-2.00306,0.198601,1.41881,-2.07526,0.321939,1.23703,-1.76631,0.22504,1.1455,-1.77214,0.274421,1.07949,-1.79515,0.382807,1.02708,-1.79184,0.385669,0.834503,-1.78222,0.369847,0.760483,-1.79522,0.275199,0.851476,-1.80059,0.263624,0.780976,-1.80688,0.279722,0.546897,-1.82491,0.183996,0.473961,-1.90292,0.436514,0.512195,-1.85802,0.413932,0.454788,-1.79269,0.04948,1.13691,-2.11589,0.016804,1.06032,-2.09761,-0.082061,1.19108,-2.09308,-0.111506,1.11572,-2.07889,-0.129059,0.967641,-2.07776,-0.200546,0.791697,-2.07297,-0.292947,0.855956,-2.07984,-0.329918,0.793999,-2.07108,-0.434561,0.582412,-2.04332,-0.520308,0.621748,-1.97246,-0.387346,0.376452,-2.00128,-0.460771,0.391503,-2.05261,0.106798,1.36327,-1.82491,0.103713,1.34728,-2.03953},
/*658*/{0.153974,1.93839,-1.72502,0.261848,1.87731,-1.84517,-0.011022,1.86158,-1.68806,-0.141779,1.79974,-1.68046,-0.010358,1.7058,-1.66966,0.05394,1.64426,-1.64766,0.245911,1.9614,-2.09045,0.285782,1.87662,-1.91779,0.101579,1.90286,-2.0459,0.16899,1.73612,-2.23315,0.230973,1.65507,-2.2603,0.34757,1.68398,-2.10449,0.408331,1.68258,-2.08323,0.191411,1.44784,-1.82196,0.012137,1.51944,-1.88662,0.018057,1.53859,-1.95026,0.022706,1.51562,-2.00241,0.198304,1.42104,-2.07457,0.320493,1.23454,-1.76645,0.220314,1.14428,-1.77188,0.270665,1.07845,-1.79401,0.378793,1.02413,-1.79065,0.388722,0.830765,-1.78115,0.379698,0.755328,-1.79373,0.277366,0.842953,-1.80219,0.269276,0.772337,-1.80808,0.295699,0.538584,-1.82538,0.205496,0.461653,-1.90516,0.455644,0.511333,-1.85801,0.434969,0.452996,-1.79239,0.048678,1.13987,-2.11647,0.011912,1.06264,-2.09692,-0.082848,1.19435,-2.09186,-0.112251,1.11825,-2.07776,-0.129921,0.96959,-2.07719,-0.204831,0.795098,-2.07369,-0.295523,0.862464,-2.08057,-0.334906,0.80105,-2.07311,-0.442725,0.591312,-2.04512,-0.525586,0.637287,-1.97489,-0.4085,0.378627,-1.99966,-0.479661,0.401232,-2.05036,0.107712,1.3659,-1.82405,0.10386,1.34957,-2.03863},
/*659*/{0.153896,1.93976,-1.72501,0.26579,1.87526,-1.84311,-0.010637,1.86321,-1.68921,-0.143309,1.80276,-1.68362,-0.011877,1.70802,-1.67046,0.052615,1.64677,-1.64753,0.246449,1.9623,-2.09028,0.286971,1.8777,-1.91662,0.102364,1.90517,-2.0463,0.171133,1.73752,-2.23362,0.233053,1.65673,-2.26045,0.348767,1.68557,-2.10425,0.409195,1.6835,-2.08105,0.191845,1.45057,-1.82222,0.012395,1.52199,-1.88545,0.017554,1.54099,-1.94976,0.023056,1.51854,-2.0012,0.198285,1.42261,-2.07411,0.318542,1.23255,-1.76702,0.218529,1.14327,-1.77158,0.266379,1.07722,-1.79309,0.374957,1.01987,-1.78982,0.389713,0.826712,-1.78068,0.384761,0.75133,-1.79315,0.278711,0.83419,-1.80357,0.275006,0.762989,-1.80996,0.314085,0.529272,-1.82599,0.226334,0.450495,-1.9073,0.473575,0.50899,-1.85646,0.454596,0.44957,-1.79132,0.047709,1.14218,-2.11634,0.015204,1.06538,-2.09885,-0.08277,1.19677,-2.09031,-0.113091,1.12039,-2.07624,-0.128847,0.97179,-2.07711,-0.208,0.797659,-2.0738,-0.29718,0.86707,-2.08102,-0.337578,0.808043,-2.07426,-0.451581,0.598672,-2.04698,-0.530389,0.651431,-1.97832,-0.429434,0.390111,-1.99643,-0.497942,0.411543,-2.04777,0.108108,1.36861,-1.82321,0.103453,1.35177,-2.03773},
/*660*/{0.154408,1.94094,-1.72494,0.265896,1.87562,-1.84256,-0.010796,1.86525,-1.6909,-0.144945,1.8042,-1.68733,-0.013896,1.70867,-1.67183,0.050696,1.64749,-1.6472,0.24724,1.96302,-2.08986,0.287423,1.87847,-1.91607,0.103641,1.90687,-2.04634,0.173993,1.73847,-2.23372,0.23481,1.65693,-2.26044,0.349215,1.68782,-2.10363,0.409524,1.68399,-2.07976,0.191872,1.45233,-1.82181,0.013105,1.52341,-1.8854,0.017677,1.5429,-1.94943,0.023494,1.51971,-1.99944,0.198364,1.42499,-2.07362,0.315819,1.22868,-1.76708,0.215599,1.14148,-1.76995,0.260457,1.07437,-1.79272,0.370746,1.01514,-1.78938,0.3911,0.823228,-1.77922,0.390395,0.747514,-1.79272,0.280032,0.825049,-1.80521,0.279817,0.754289,-1.81211,0.32921,0.522527,-1.82621,0.246634,0.440612,-1.9089,0.491297,0.510078,-1.85611,0.473996,0.448115,-1.79149,0.048265,1.14389,-2.11675,0.016281,1.06716,-2.09967,-0.081724,1.19794,-2.08871,-0.111665,1.12213,-2.07453,-0.128039,0.972435,-2.07674,-0.210815,0.79902,-2.07309,-0.298072,0.871311,-2.08196,-0.341162,0.813093,-2.075,-0.460415,0.608007,-2.04823,-0.537343,0.665632,-1.98175,-0.44684,0.399872,-1.99239,-0.512741,0.421548,-2.04459,0.108683,1.37022,-1.82276,0.103931,1.35363,-2.0373},
/*661*/{0.154779,1.94167,-1.72461,0.262576,1.88095,-1.84317,-0.01154,1.86729,-1.693,-0.145122,1.8063,-1.69012,-0.014397,1.70984,-1.67255,0.049113,1.64812,-1.64749,0.248906,1.96413,-2.08943,0.286883,1.88014,-1.91589,0.104521,1.90889,-2.04621,0.175511,1.73938,-2.23377,0.236399,1.65711,-2.26051,0.351035,1.68809,-2.10237,0.40986,1.68411,-2.07809,0.191673,1.45332,-1.82171,0.013487,1.52531,-1.88375,0.018349,1.54513,-1.94808,0.023411,1.52217,-1.99933,0.198909,1.42749,-2.07305,0.313198,1.22467,-1.76716,0.212052,1.13931,-1.76914,0.255916,1.07167,-1.7923,0.366046,1.00998,-1.7891,0.390758,0.819534,-1.7793,0.394869,0.744252,-1.79214,0.280687,0.81622,-1.80614,0.283388,0.74504,-1.81304,0.344503,0.516142,-1.8274,0.265613,0.431624,-1.91109,0.507075,0.509698,-1.85537,0.4913,0.446626,-1.79101,0.049646,1.14512,-2.11696,0.017522,1.0678,-2.10011,-0.081379,1.19911,-2.08811,-0.110879,1.12226,-2.07304,-0.125839,0.972461,-2.07615,-0.213513,0.800562,-2.07264,-0.298218,0.875333,-2.08297,-0.343141,0.818002,-2.07575,-0.467614,0.619334,-2.05021,-0.544501,0.680119,-1.98512,-0.464683,0.410699,-1.98868,-0.531596,0.435211,-2.04335,0.108531,1.37136,-1.82282,0.103905,1.35623,-2.03747},
/*662*/{0.156259,1.94296,-1.72389,0.264223,1.88153,-1.84301,-0.010908,1.86787,-1.69347,-0.14503,1.807,-1.69251,-0.015358,1.71105,-1.67385,0.048692,1.64734,-1.6469,0.249642,1.96435,-2.08914,0.287512,1.8812,-1.91547,0.104861,1.90975,-2.04632,0.177176,1.7396,-2.23278,0.237317,1.65686,-2.26005,0.351414,1.68644,-2.10122,0.410858,1.68294,-2.07729,0.191684,1.45528,-1.82176,0.01373,1.52619,-1.88322,0.018198,1.54676,-1.94719,0.023318,1.52353,-1.99717,0.198925,1.42897,-2.07245,0.311441,1.21937,-1.76664,0.207922,1.1362,-1.76891,0.250821,1.06781,-1.7927,0.359479,1.00423,-1.7882,0.39154,0.815668,-1.77865,0.39753,0.740352,-1.79218,0.281189,0.807875,-1.80507,0.288682,0.735241,-1.81312,0.360425,0.511085,-1.82707,0.283472,0.42377,-1.91381,0.521143,0.509657,-1.85505,0.506862,0.445386,-1.79119,0.051721,1.14574,-2.11665,0.02025,1.06843,-2.10012,-0.079546,1.19821,-2.08696,-0.107718,1.12237,-2.07336,-0.122802,0.971601,-2.07593,-0.215155,0.800557,-2.07159,-0.296954,0.878773,-2.08381,-0.344394,0.823265,-2.07709,-0.477376,0.629049,-2.05078,-0.551832,0.694355,-1.98856,-0.4816,0.422146,-1.98509,-0.543746,0.447625,-2.04112,0.109169,1.37291,-1.822,0.103939,1.35769,-2.03664},
/*663*/{0.156544,1.94323,-1.72346,0.263791,1.88162,-1.8423,-0.009956,1.86857,-1.69449,-0.144852,1.80758,-1.69475,-0.015988,1.71091,-1.67464,0.047637,1.64711,-1.64689,0.250707,1.96414,-2.08849,0.287949,1.88179,-1.91553,0.105855,1.91128,-2.04591,0.178183,1.73973,-2.23188,0.237755,1.65656,-2.25964,0.35187,1.68508,-2.10031,0.410966,1.68101,-2.07628,0.192032,1.45576,-1.82216,0.013177,1.52722,-1.88227,0.018046,1.54777,-1.94632,0.023567,1.52389,-1.99567,0.198933,1.43066,-2.07253,0.307625,1.21434,-1.76637,0.202746,1.13282,-1.76845,0.243106,1.06366,-1.79337,0.35336,0.998427,-1.78822,0.391768,0.811967,-1.77918,0.401364,0.737334,-1.79202,0.281938,0.798759,-1.80446,0.291797,0.72782,-1.81294,0.370728,0.505866,-1.82941,0.299361,0.4169,-1.91461,0.534463,0.509811,-1.85434,0.523331,0.445494,-1.79131,0.053916,1.14615,-2.11634,0.022697,1.06835,-2.09965,-0.077883,1.19733,-2.08633,-0.105821,1.12028,-2.07193,-0.119101,0.970276,-2.07559,-0.216814,0.800985,-2.07016,-0.295258,0.882212,-2.08522,-0.343369,0.829564,-2.07801,-0.485499,0.639063,-2.0514,-0.558741,0.707885,-1.99342,-0.496455,0.433992,-1.98221,-0.559097,0.460714,-2.04045,0.109764,1.37338,-1.82199,0.103949,1.35901,-2.03666},
/*664*/{0.157688,1.94332,-1.72263,0.264363,1.88155,-1.84189,-0.009273,1.86806,-1.69549,-0.14398,1.80734,-1.69724,-0.015962,1.70955,-1.67507,0.046769,1.64593,-1.64637,0.250713,1.96356,-2.08835,0.287991,1.88205,-1.91463,0.105715,1.91223,-2.0461,0.177198,1.73945,-2.23115,0.236693,1.65502,-2.25849,0.352371,1.68337,-2.10007,0.412894,1.67823,-2.07609,0.191704,1.45593,-1.82213,0.012981,1.52669,-1.88124,0.017972,1.54792,-1.94567,0.022795,1.52544,-1.99502,0.199016,1.4324,-2.07234,0.304929,1.2091,-1.76493,0.192062,1.12747,-1.77042,0.236461,1.05859,-1.79259,0.345992,0.992763,-1.7876,0.39242,0.808451,-1.77905,0.405226,0.734593,-1.79313,0.282833,0.79163,-1.80351,0.295082,0.721032,-1.81303,0.381775,0.501394,-1.82889,0.314463,0.410892,-1.91574,0.546202,0.51101,-1.85324,0.535776,0.444497,-1.79148,0.056053,1.14564,-2.11586,0.026213,1.06748,-2.09952,-0.075373,1.19529,-2.08534,-0.101937,1.11847,-2.07163,-0.114988,0.967969,-2.07464,-0.218705,0.801873,-2.06935,-0.294301,0.884461,-2.08547,-0.344007,0.834108,-2.07876,-0.491485,0.651058,-2.05265,-0.566428,0.721923,-1.9975,-0.510061,0.447392,-1.97907,-0.571087,0.473779,-2.03896,0.109802,1.37308,-1.82246,0.104086,1.36041,-2.03725},
/*665*/{0.159062,1.94361,-1.72198,0.267619,1.87814,-1.84056,-0.00871,1.86762,-1.69568,-0.143237,1.80588,-1.69826,-0.015405,1.70776,-1.67541,0.046094,1.64333,-1.64504,0.251037,1.96264,-2.08813,0.288752,1.88135,-1.91388,0.106811,1.91312,-2.04598,0.176644,1.73851,-2.22939,0.235914,1.65388,-2.25772,0.352312,1.68042,-2.09888,0.412517,1.67496,-2.07482,0.192382,1.45679,-1.82244,0.01251,1.52647,-1.88008,0.017,1.54758,-1.94411,0.022504,1.52573,-1.99423,0.19887,1.43342,-2.07226,0.299555,1.2029,-1.7649,0.188599,1.12649,-1.76999,0.230564,1.05451,-1.79238,0.338177,0.986961,-1.78783,0.393653,0.805607,-1.77965,0.4084,0.731822,-1.79303,0.283897,0.784457,-1.80299,0.298301,0.714303,-1.8114,0.391078,0.496738,-1.82937,0.326523,0.405428,-1.91701,0.556314,0.512299,-1.85317,0.547521,0.446158,-1.79292,0.059717,1.145,-2.11475,0.030747,1.06565,-2.09811,-0.073083,1.19283,-2.08441,-0.098975,1.11526,-2.0707,-0.111283,0.96531,-2.07409,-0.219021,0.802276,-2.06814,-0.293633,0.886159,-2.0844,-0.342593,0.839992,-2.07959,-0.50062,0.661284,-2.05192,-0.573224,0.736195,-2.00007,-0.523489,0.460529,-1.97574,-0.582524,0.487735,-2.03792,0.110965,1.37331,-1.82215,0.104339,1.36102,-2.03694},
/*666*/{0.15981,1.94328,-1.72121,0.264366,1.88144,-1.8417,-0.008659,1.86667,-1.6963,-0.141993,1.80462,-1.70057,-0.014528,1.70569,-1.67588,0.046241,1.64123,-1.6449,0.252186,1.96199,-2.08756,0.288258,1.88179,-1.9139,0.106999,1.91315,-2.04586,0.174855,1.73773,-2.22886,0.23362,1.65198,-2.25644,0.352173,1.67703,-2.09827,0.412068,1.67035,-2.07502,0.19172,1.45666,-1.82258,0.011959,1.52577,-1.8793,0.017101,1.54802,-1.94365,0.021959,1.52479,-1.9926,0.199326,1.43456,-2.07209,0.296098,1.19765,-1.76495,0.1819,1.12111,-1.77029,0.22298,1.05052,-1.79452,0.330397,0.981118,-1.78743,0.39381,0.801906,-1.7801,0.410325,0.728558,-1.79469,0.285108,0.778141,-1.80136,0.301396,0.708522,-1.81032,0.402625,0.494251,-1.83064,0.337092,0.399685,-1.91665,0.563408,0.512276,-1.85379,0.556936,0.446587,-1.79549,0.0632,1.14378,-2.11328,0.03508,1.0641,-2.09767,-0.069476,1.19036,-2.08392,-0.094469,1.11186,-2.06978,-0.10593,0.962095,-2.07246,-0.219927,0.80256,-2.06678,-0.288148,0.891782,-2.08769,-0.342397,0.845927,-2.07994,-0.506679,0.674095,-2.05218,-0.578385,0.751111,-2.00269,-0.536274,0.473874,-1.97396,-0.592652,0.501695,-2.03744,0.111236,1.3728,-1.82217,0.104713,1.36151,-2.03701},
/*667*/{0.16086,1.94229,-1.72094,0.264978,1.88138,-1.84235,-0.007305,1.86464,-1.69558,-0.140389,1.80246,-1.70142,-0.013285,1.7035,-1.67534,0.047003,1.63817,-1.64396,0.252763,1.96042,-2.08699,0.287309,1.88098,-1.91366,0.107029,1.91315,-2.04582,0.172126,1.73651,-2.22729,0.231525,1.6502,-2.25505,0.351841,1.67368,-2.09805,0.412124,1.66606,-2.07463,0.194276,1.45668,-1.82246,0.011611,1.52484,-1.87859,0.016141,1.54708,-1.94343,0.021573,1.52441,-1.99211,0.199221,1.43491,-2.07213,0.291233,1.18977,-1.76329,0.174273,1.11544,-1.77041,0.215934,1.04534,-1.79422,0.322784,0.974863,-1.78719,0.394451,0.799028,-1.78058,0.411668,0.727243,-1.79555,0.285972,0.772221,-1.80062,0.303307,0.702441,-1.80849,0.408899,0.491032,-1.83043,0.345926,0.395814,-1.91744,0.569688,0.512774,-1.85434,0.565666,0.447271,-1.79526,0.06786,1.1421,-2.11242,0.040944,1.06181,-2.09662,-0.066774,1.18561,-2.08324,-0.089717,1.10726,-2.06944,-0.100062,0.959106,-2.07144,-0.219961,0.804126,-2.06541,-0.284915,0.89481,-2.08775,-0.340093,0.851874,-2.08015,-0.509305,0.687299,-2.05286,-0.582003,0.766658,-2.00453,-0.548349,0.488742,-1.97278,-0.602658,0.517347,-2.03792,0.113136,1.37264,-1.82198,0.105416,1.36125,-2.03678},
/*668*/{0.162143,1.94126,-1.72052,0.26856,1.87643,-1.84033,-0.006407,1.86253,-1.69528,-0.1382,1.79967,-1.70173,-0.012786,1.69997,-1.67485,0.047716,1.63491,-1.64269,0.251885,1.95848,-2.08633,0.288693,1.87949,-1.91304,0.107091,1.91287,-2.04522,0.17015,1.73485,-2.22527,0.229916,1.64866,-2.25369,0.351885,1.66933,-2.09791,0.411184,1.6606,-2.07403,0.194587,1.45656,-1.82228,0.011231,1.52259,-1.87816,0.015518,1.54574,-1.94289,0.020442,1.52285,-1.99109,0.199616,1.43525,-2.07222,0.288055,1.18372,-1.76208,0.16854,1.11184,-1.77166,0.208789,1.0412,-1.7954,0.315381,0.969442,-1.78704,0.393572,0.795621,-1.78154,0.412013,0.722847,-1.79559,0.286164,0.76675,-1.79977,0.305013,0.698668,-1.80813,0.412035,0.487844,-1.83048,0.351531,0.392123,-1.91698,0.575363,0.512521,-1.8559,0.570988,0.44546,-1.79616,0.072163,1.13942,-2.1104,0.04644,1.05957,-2.09506,-0.063334,1.18192,-2.08306,-0.085758,1.10219,-2.06731,-0.093999,0.955357,-2.07049,-0.219438,0.805379,-2.06504,-0.280589,0.898716,-2.08759,-0.337997,0.857989,-2.07965,-0.511074,0.700213,-2.052,-0.584886,0.782613,-2.00627,-0.557966,0.503274,-1.97164,-0.609999,0.53384,-2.03839,0.114372,1.37164,-1.8219,0.10635,1.36066,-2.03671},
/*669*/{0.162577,1.93994,-1.71984,0.268707,1.87593,-1.84058,-0.006321,1.86019,-1.69532,-0.1353,1.7957,-1.70091,-0.011538,1.69541,-1.67327,0.048865,1.6308,-1.64098,0.252164,1.9567,-2.08617,0.288018,1.87807,-1.9131,0.10683,1.91114,-2.0441,0.166508,1.73365,-2.22431,0.227256,1.64663,-2.25219,0.351534,1.66427,-2.09802,0.411007,1.65432,-2.07441,0.191554,1.45486,-1.82271,0.010657,1.5203,-1.87798,0.014364,1.54407,-1.94216,0.020608,1.52119,-1.99122,0.19926,1.4349,-2.07212,0.282925,1.17669,-1.76079,0.166155,1.11015,-1.7712,0.202573,1.03648,-1.79562,0.308968,0.963564,-1.78705,0.392289,0.793294,-1.78194,0.412355,0.721104,-1.79644,0.284315,0.761749,-1.79934,0.304695,0.694075,-1.80746,0.41566,0.485273,-1.83097,0.356197,0.389723,-1.9174,0.578432,0.511282,-1.85696,0.576477,0.444141,-1.79711,0.076951,1.13672,-2.10901,0.0523,1.05695,-2.09319,-0.059962,1.17643,-2.0824,-0.080027,1.09778,-2.06778,-0.087318,0.951022,-2.06914,-0.219081,0.806745,-2.06451,-0.275342,0.903117,-2.08703,-0.334529,0.865087,-2.07947,-0.514852,0.714436,-2.05226,-0.585123,0.798649,-2.00584,-0.567621,0.518662,-1.9726,-0.620518,0.550991,-2.03917,0.112906,1.36924,-1.82262,0.106075,1.35988,-2.03754},
/*670*/{0.163561,1.93799,-1.71834,0.263521,1.87782,-1.8419,-0.005745,1.85662,-1.69423,-0.134033,1.79208,-1.70144,-0.009789,1.69177,-1.67289,0.049796,1.62643,-1.6396,0.252602,1.95456,-2.0864,0.286827,1.87781,-1.91366,0.106201,1.91028,-2.04304,0.162323,1.73186,-2.22241,0.223109,1.64432,-2.25065,0.350158,1.65928,-2.09836,0.409814,1.64808,-2.07511,0.191366,1.45342,-1.82269,0.010249,1.51768,-1.87744,0.014463,1.54167,-1.94179,0.019283,1.51973,-1.9907,0.198699,1.43379,-2.07164,0.279677,1.1709,-1.75979,0.158996,1.10529,-1.77163,0.19583,1.03245,-1.79562,0.30369,0.957793,-1.78664,0.389785,0.790502,-1.78227,0.410523,0.718405,-1.79642,0.282548,0.757749,-1.79967,0.30344,0.689992,-1.80838,0.417507,0.482288,-1.83053,0.35818,0.386203,-1.91656,0.580492,0.509675,-1.85796,0.578882,0.441876,-1.7984,0.081169,1.13405,-2.10789,0.058535,1.05323,-2.0914,-0.055299,1.17182,-2.08217,-0.074702,1.09215,-2.06691,-0.079433,0.946527,-2.06783,-0.216793,0.805982,-2.06319,-0.269807,0.906945,-2.08602,-0.330059,0.871035,-2.07776,-0.514078,0.725386,-2.05122,-0.584767,0.815629,-2.00574,-0.576365,0.534571,-1.97364,-0.626503,0.568618,-2.0402,0.113121,1.36731,-1.82252,0.105978,1.35834,-2.03745},
/*671*/{0.164148,1.9359,-1.71836,0.264161,1.87599,-1.84203,-0.004529,1.85216,-1.6929,-0.132369,1.78664,-1.69939,-0.00821,1.68757,-1.67082,0.051327,1.62224,-1.63831,0.251188,1.95209,-2.0859,0.286805,1.87583,-1.91375,0.105394,1.90938,-2.04226,0.158581,1.72923,-2.22056,0.219772,1.64177,-2.2489,0.34914,1.65284,-2.09796,0.408417,1.64079,-2.07566,0.190975,1.45138,-1.82288,0.009362,1.51525,-1.8766,0.013281,1.53864,-1.94147,0.019073,1.51656,-1.99065,0.199584,1.43263,-2.07147,0.274484,1.1658,-1.7591,0.15437,1.10287,-1.77208,0.190012,1.02926,-1.79749,0.298017,0.952453,-1.7875,0.386528,0.787264,-1.78307,0.407353,0.716385,-1.79745,0.279996,0.753649,-1.80002,0.301251,0.685223,-1.80861,0.418082,0.479567,-1.83069,0.358882,0.38244,-1.91639,0.57995,0.507682,-1.85974,0.579735,0.439429,-1.80014,0.086469,1.13057,-2.10648,0.064896,1.04997,-2.0907,-0.051251,1.16582,-2.08154,-0.06894,1.08568,-2.06609,-0.071398,0.941981,-2.06658,-0.21505,0.810253,-2.06378,-0.263876,0.910678,-2.08534,-0.325051,0.877707,-2.07696,-0.520824,0.74527,-2.05481,-0.582801,0.831664,-2.00429,-0.584899,0.551033,-1.9769,-0.634419,0.587782,-2.04215,0.113551,1.36482,-1.82259,0.10688,1.3564,-2.03756},
/*672*/{0.165084,1.9334,-1.71762,0.263988,1.8743,-1.84226,-0.003202,1.84849,-1.6912,-0.13012,1.78163,-1.69856,-0.007025,1.68169,-1.66899,0.052767,1.6171,-1.6364,0.250714,1.94952,-2.08584,0.286275,1.87338,-1.91342,0.104944,1.90694,-2.0413,0.154718,1.7266,-2.21855,0.2158,1.63893,-2.24722,0.347928,1.64738,-2.09869,0.406271,1.63278,-2.07634,0.19092,1.44885,-1.82247,0.009119,1.51223,-1.87619,0.012783,1.53627,-1.94169,0.018224,1.51347,-1.99044,0.199416,1.43083,-2.07108,0.272084,1.15947,-1.7566,0.149328,1.09919,-1.77283,0.18496,1.02529,-1.79814,0.294276,0.948166,-1.78712,0.383534,0.783933,-1.78308,0.405302,0.711951,-1.79695,0.276697,0.74923,-1.80059,0.298364,0.680896,-1.80915,0.418103,0.474506,-1.8305,0.357848,0.378381,-1.91621,0.579259,0.504491,-1.86078,0.580289,0.436166,-1.80181,0.091368,1.1267,-2.10537,0.071951,1.04601,-2.08896,-0.046746,1.15931,-2.08083,-0.063111,1.07972,-2.06447,-0.064002,0.936468,-2.06559,-0.212194,0.81165,-2.06386,-0.255992,0.915043,-2.08411,-0.320294,0.884133,-2.07622,-0.520819,0.758774,-2.05481,-0.5792,0.848555,-2.00304,-0.590286,0.568597,-1.97937,-0.64086,0.607465,-2.04384,0.11362,1.36222,-1.82249,0.107044,1.35403,-2.03747},
/*673*/{0.165112,1.93042,-1.71723,0.263505,1.87187,-1.84221,-0.002714,1.84394,-1.6898,-0.128742,1.77572,-1.69646,-0.0041,1.67682,-1.66747,0.055239,1.61192,-1.63513,0.250141,1.94638,-2.08604,0.286187,1.87094,-1.91281,0.104385,1.90513,-2.0409,0.150656,1.72407,-2.21726,0.211125,1.63578,-2.24605,0.345173,1.64079,-2.09857,0.404824,1.62462,-2.07755,0.190505,1.44624,-1.82149,0.008176,1.50983,-1.87586,0.012179,1.53301,-1.94118,0.018396,1.50973,-1.99036,0.199736,1.42951,-2.07059,0.26809,1.15525,-1.75415,0.146007,1.09598,-1.77319,0.181549,1.02182,-1.79874,0.292904,0.945573,-1.78681,0.379203,0.779613,-1.783,0.401535,0.708383,-1.79768,0.27202,0.745319,-1.80127,0.294718,0.677448,-1.80967,0.415572,0.470029,-1.83083,0.354807,0.37413,-1.91717,0.577877,0.50165,-1.86259,0.577926,0.43327,-1.80428,0.097087,1.123,-2.1048,0.078841,1.04055,-2.08755,-0.041955,1.15326,-2.08022,-0.057114,1.07298,-2.06423,-0.056269,0.931289,-2.06494,-0.208639,0.813815,-2.06401,-0.248207,0.918581,-2.08229,-0.312767,0.890132,-2.07456,-0.520433,0.775422,-2.05554,-0.573744,0.86525,-2.00045,-0.597872,0.583414,-1.98498,-0.646104,0.627503,-2.04552,0.113683,1.35945,-1.82221,0.107919,1.35173,-2.03723},
/*674*/{0.164781,1.92672,-1.71688,0.263537,1.86854,-1.84238,-0.001936,1.83891,-1.68821,-0.126549,1.76913,-1.69371,-0.002995,1.67123,-1.66512,0.057849,1.60639,-1.63284,0.248973,1.94288,-2.08643,0.285899,1.86818,-1.91283,0.103277,1.90216,-2.03935,0.145856,1.72177,-2.21555,0.207012,1.63236,-2.24441,0.342693,1.63419,-2.0993,0.40212,1.61597,-2.07907,0.188832,1.44354,-1.8215,0.008226,1.50584,-1.87531,0.011963,1.52934,-1.94078,0.017835,1.50651,-1.99064,0.200198,1.42694,-2.06953,0.265182,1.15162,-1.75161,0.144062,1.0906,-1.77301,0.17926,1.0192,-1.79979,0.29218,0.944415,-1.78663,0.374554,0.775504,-1.78375,0.397658,0.703483,-1.79722,0.268197,0.740449,-1.80125,0.291202,0.67295,-1.80983,0.409206,0.466793,-1.83282,0.350079,0.370088,-1.91735,0.571895,0.496688,-1.86503,0.575522,0.428821,-1.80566,0.101463,1.11837,-2.1044,0.084998,1.03673,-2.08633,-0.037794,1.14621,-2.07855,-0.051312,1.06667,-2.0635,-0.048072,0.925681,-2.06402,-0.20429,0.815453,-2.06396,-0.239753,0.92246,-2.08071,-0.306134,0.894618,-2.07249,-0.517563,0.790642,-2.05593,-0.566771,0.881339,-1.99842,-0.600688,0.60128,-1.98658,-0.646546,0.648109,-2.04729,0.113244,1.35606,-1.82203,0.108429,1.34879,-2.03709},
/*675*/{0.165389,1.92316,-1.71597,0.263456,1.86557,-1.84255,-0.001158,1.83261,-1.68514,-0.125729,1.76205,-1.69155,-0.000424,1.66497,-1.66293,0.060498,1.60047,-1.63131,0.24801,1.93972,-2.08643,0.285235,1.86528,-1.91237,0.101864,1.89913,-2.03842,0.141443,1.71829,-2.21356,0.201546,1.62877,-2.24276,0.340216,1.62648,-2.10006,0.399648,1.60699,-2.08063,0.187281,1.43997,-1.82085,0.006977,1.50213,-1.87554,0.011574,1.52646,-1.94079,0.017353,1.50266,-1.99071,0.200202,1.42522,-2.06896,0.262212,1.14792,-1.75015,0.141332,1.08917,-1.77354,0.176173,1.0153,-1.8003,0.291732,0.942839,-1.78644,0.370635,0.770325,-1.78402,0.392092,0.698572,-1.79741,0.263809,0.736159,-1.80199,0.285881,0.667737,-1.81082,0.407074,0.463371,-1.83365,0.343735,0.365431,-1.91764,0.56672,0.491055,-1.8662,0.567052,0.422889,-1.80724,0.10689,1.11422,-2.10336,0.092096,1.03248,-2.08573,-0.033177,1.13939,-2.07799,-0.045394,1.05937,-2.06144,-0.040129,0.92025,-2.06337,-0.199155,0.816375,-2.06408,-0.230946,0.925218,-2.07927,-0.297111,0.901511,-2.07213,-0.514937,0.806445,-2.05508,-0.559651,0.896677,-1.99632,-0.604223,0.617566,-1.99096,-0.645135,0.667518,-2.05064,0.112179,1.35233,-1.8224,0.108737,1.34628,-2.03752},
/*676*/{0.165792,1.91899,-1.71537,0.261774,1.86176,-1.84304,0.000138,1.82717,-1.68402,-0.12456,1.75465,-1.68736,0.001816,1.65836,-1.66012,0.063978,1.5944,-1.62935,0.247041,1.93531,-2.08768,0.284786,1.86169,-1.91308,0.101347,1.89579,-2.03791,0.136692,1.71512,-2.21197,0.196339,1.62492,-2.24192,0.33718,1.61897,-2.10058,0.397271,1.5982,-2.08321,0.186594,1.43667,-1.81999,0.006796,1.49812,-1.87495,0.011041,1.52265,-1.94079,0.016698,1.49843,-1.99091,0.200041,1.42236,-2.06828,0.258958,1.14534,-1.74922,0.138173,1.08536,-1.77388,0.174605,1.01278,-1.79974,0.290606,0.940267,-1.78722,0.366273,0.765128,-1.7842,0.388152,0.693022,-1.79803,0.259223,0.731646,-1.80221,0.281207,0.663768,-1.81108,0.397301,0.455827,-1.83333,0.335969,0.360888,-1.9185,0.559236,0.483422,-1.86767,0.5607,0.414708,-1.80832,0.111524,1.10975,-2.10307,0.098522,1.02797,-2.08487,-0.028009,1.13246,-2.07679,-0.039189,1.05192,-2.0613,-0.032218,0.914248,-2.06286,-0.19352,0.817411,-2.06416,-0.221331,0.927751,-2.07781,-0.289194,0.904362,-2.07188,-0.508791,0.819502,-2.05506,-0.549397,0.910738,-1.99281,-0.606129,0.633841,-1.99565,-0.65077,0.686577,-2.05278,0.112021,1.34877,-1.82199,0.109092,1.34278,-2.03712},
/*677*/{0.166209,1.9141,-1.71463,0.262761,1.85873,-1.8432,0.000655,1.82067,-1.68169,-0.122679,1.74744,-1.68356,0.005235,1.65112,-1.6573,0.067659,1.58802,-1.62763,0.24491,1.93108,-2.08788,0.28435,1.85833,-1.91341,0.100213,1.89173,-2.03747,0.131596,1.71123,-2.2111,0.191676,1.6212,-2.24068,0.333453,1.61081,-2.10163,0.393965,1.58798,-2.08582,0.186277,1.43351,-1.81886,0.005315,1.49409,-1.87529,0.010218,1.51854,-1.94041,0.016212,1.49354,-1.99142,0.20019,1.41944,-2.06692,0.258384,1.14163,-1.747,0.136437,1.08119,-1.77428,0.172593,1.01008,-1.79964,0.288886,0.937128,-1.78747,0.361724,0.759404,-1.78438,0.382754,0.687329,-1.79904,0.253805,0.726802,-1.80256,0.275119,0.658924,-1.81129,0.390033,0.45174,-1.83484,0.326929,0.35618,-1.91943,0.551632,0.476918,-1.86907,0.552854,0.408163,-1.80963,0.116767,1.10613,-2.10241,0.104455,1.02342,-2.08426,-0.024596,1.12552,-2.07667,-0.032894,1.04451,-2.05993,-0.023835,0.908191,-2.0623,-0.187412,0.818319,-2.06509,-0.211443,0.92923,-2.0757,-0.279767,0.909789,-2.07094,-0.500643,0.83213,-2.05644,-0.538452,0.924274,-1.99248,-0.605844,0.65006,-1.99893,-0.648457,0.706328,-2.05741,0.111818,1.34534,-1.82125,0.109993,1.339,-2.03639},
/*678*/{0.165848,1.90881,-1.71411,0.261179,1.85439,-1.843,0.002147,1.81406,-1.67929,-0.120492,1.73914,-1.67932,0.006712,1.64446,-1.65472,0.07196,1.58194,-1.6257,0.24436,1.92648,-2.08864,0.283662,1.85415,-1.9132,0.098778,1.88725,-2.03643,0.126941,1.70741,-2.20944,0.186264,1.61645,-2.23944,0.330948,1.60302,-2.10305,0.390555,1.57779,-2.08764,0.185262,1.43082,-1.81769,0.004657,1.48944,-1.87552,0.009655,1.51428,-1.94023,0.015201,1.48878,-1.99122,0.200301,1.41658,-2.06629,0.256182,1.13807,-1.7465,0.135035,1.0812,-1.77547,0.17049,1.00662,-1.80088,0.286473,0.931839,-1.78714,0.356004,0.753447,-1.78472,0.376279,0.681209,-1.7997,0.247999,0.722975,-1.8025,0.26895,0.654484,-1.81203,0.379878,0.44423,-1.83551,0.316562,0.350644,-1.92189,0.54254,0.46803,-1.86966,0.541397,0.398582,-1.81041,0.121031,1.1017,-2.10207,0.111716,1.019,-2.08396,-0.019306,1.11846,-2.07549,-0.026987,1.03722,-2.05937,-0.01506,0.902242,-2.06223,-0.181097,0.818723,-2.06533,-0.200334,0.929324,-2.07318,-0.269032,0.911979,-2.0684,-0.492109,0.843774,-2.05563,-0.527153,0.936954,-1.98999,-0.605225,0.665031,-2.00252,-0.645086,0.724945,-2.05876,0.111887,1.34192,-1.82058,0.110857,1.33524,-2.03571},
/*679*/{0.166061,1.90335,-1.71351,0.266331,1.84694,-1.84185,0.002843,1.80672,-1.67699,-0.118706,1.7301,-1.67437,0.010201,1.63647,-1.65103,0.075666,1.57497,-1.62391,0.24193,1.92178,-2.08932,0.283453,1.84983,-1.91315,0.097706,1.88356,-2.03488,0.122281,1.70348,-2.20869,0.180827,1.61173,-2.23863,0.326438,1.59291,-2.10408,0.386219,1.56662,-2.09006,0.184003,1.42751,-1.81685,0.003246,1.48542,-1.87628,0.008814,1.51088,-1.94038,0.014787,1.484,-1.99211,0.199854,1.41309,-2.06533,0.255397,1.13451,-1.74518,0.133249,1.07808,-1.77683,0.167316,1.00343,-1.80108,0.282882,0.926647,-1.78715,0.350319,0.746995,-1.78532,0.369903,0.673937,-1.80024,0.242322,0.718412,-1.80353,0.261291,0.649186,-1.81211,0.369354,0.438674,-1.83709,0.304561,0.345919,-1.92362,0.532611,0.459165,-1.86883,0.529831,0.38972,-1.81029,0.126448,1.09747,-2.10224,0.118537,1.01382,-2.08355,-0.013971,1.1108,-2.07484,-0.02019,1.02995,-2.05912,-0.006694,0.896132,-2.06173,-0.174799,0.817952,-2.06414,-0.189597,0.930315,-2.07259,-0.258797,0.915949,-2.06804,-0.479545,0.852756,-2.05413,-0.515157,0.948168,-1.98833,-0.601074,0.680366,-2.00612,-0.639926,0.740923,-2.06104,0.110742,1.33848,-1.82013,0.110953,1.33132,-2.03524},
/*680*/{0.16608,1.89745,-1.71221,0.265953,1.84234,-1.842,0.004864,1.79849,-1.67382,-0.115881,1.72161,-1.67007,0.013827,1.62862,-1.64855,0.07996,1.56786,-1.62251,0.241031,1.91653,-2.09018,0.28247,1.84504,-1.91387,0.096357,1.87887,-2.03438,0.117369,1.69854,-2.20778,0.174858,1.60737,-2.23775,0.321685,1.58319,-2.10591,0.381865,1.55582,-2.09291,0.182897,1.4242,-1.81535,0.003301,1.4822,-1.87633,0.008316,1.50596,-1.94023,0.013094,1.47936,-1.99254,0.200045,1.40943,-2.06443,0.25259,1.12926,-1.74481,0.131476,1.07552,-1.77787,0.165675,0.999469,-1.80154,0.278922,0.920248,-1.78687,0.344033,0.739518,-1.7858,0.361269,0.667287,-1.80084,0.235306,0.713426,-1.80303,0.253013,0.644221,-1.81313,0.357112,0.432437,-1.83707,0.287132,0.35149,-1.92512,0.519711,0.44873,-1.86974,0.515847,0.379673,-1.80956,0.131967,1.09297,-2.10185,0.125971,1.00998,-2.08384,-0.010084,1.10392,-2.07476,-0.013416,1.02261,-2.05821,0.002318,0.89015,-2.06187,-0.16717,0.817669,-2.06397,-0.178302,0.93108,-2.07108,-0.247808,0.918363,-2.06734,-0.473963,0.865324,-2.05569,-0.501472,0.959359,-1.98725,-0.598313,0.693156,-2.00937,-0.632068,0.756727,-2.06466,0.110202,1.33512,-1.8191,0.111212,1.32709,-2.03419},
/*681*/{0.166664,1.89115,-1.71121,0.266588,1.83622,-1.84145,0.005972,1.79045,-1.67094,-0.114262,1.71265,-1.66575,0.017664,1.62058,-1.64541,0.085246,1.56095,-1.62046,0.238525,1.91108,-2.09089,0.282886,1.84063,-1.91379,0.094828,1.87417,-2.03391,0.113436,1.69391,-2.20743,0.168958,1.60238,-2.23688,0.317317,1.57357,-2.10766,0.376869,1.54385,-2.09588,0.182749,1.42061,-1.8136,0.000373,1.47508,-1.87679,0.007625,1.50047,-1.94006,0.012969,1.47377,-1.99197,0.199943,1.40578,-2.06361,0.251267,1.12505,-1.74477,0.128693,1.07002,-1.77856,0.163505,0.996285,-1.80062,0.275289,0.914826,-1.78647,0.337534,0.732968,-1.78528,0.353259,0.661441,-1.80205,0.228043,0.709315,-1.80407,0.245087,0.639763,-1.81347,0.343294,0.430245,-1.83693,0.268885,0.351286,-1.92679,0.506223,0.43788,-1.86968,0.494819,0.368414,-1.80972,0.136559,1.08895,-2.1023,0.13169,1.00467,-2.08442,-0.004574,1.09595,-2.07367,-0.006394,1.01492,-2.05736,0.011315,0.884121,-2.06176,-0.158795,0.816915,-2.06411,-0.16652,0.930852,-2.07099,-0.236232,0.919997,-2.06653,-0.463572,0.874499,-2.05532,-0.488362,0.969171,-1.98674,-0.592036,0.705466,-2.01325,-0.625675,0.77021,-2.06651,0.110461,1.33036,-1.81844,0.112675,1.32245,-2.03352},
/*682*/{0.167338,1.88477,-1.70992,0.266257,1.83265,-1.84232,0.008678,1.78151,-1.66774,-0.112113,1.70179,-1.66087,0.021789,1.61341,-1.64212,0.091061,1.55341,-1.61887,0.237631,1.90601,-2.09172,0.281791,1.83545,-1.91431,0.093787,1.86891,-2.0331,0.107917,1.68951,-2.20718,0.162653,1.59693,-2.23638,0.31279,1.5647,-2.11007,0.371543,1.53199,-2.09868,0.181723,1.41698,-1.81234,-0.00022,1.47065,-1.87767,0.006274,1.49679,-1.94007,0.012786,1.47006,-1.99245,0.199103,1.40195,-2.06308,0.247787,1.1213,-1.74456,0.128552,1.06725,-1.77795,0.161684,0.992023,-1.80167,0.271001,0.909305,-1.78572,0.330047,0.726389,-1.78622,0.344849,0.653397,-1.80148,0.220818,0.704046,-1.80412,0.235768,0.634758,-1.81481,0.328033,0.424843,-1.83978,0.250664,0.353923,-1.92902,0.492633,0.424074,-1.87052,0.47651,0.356589,-1.81061,0.142665,1.08467,-2.10265,0.138917,1.00088,-2.08506,0.001353,1.08948,-2.07295,0.001361,1.00774,-2.0569,0.020373,0.878586,-2.06203,-0.149918,0.81652,-2.06434,-0.154841,0.930198,-2.06951,-0.225033,0.921262,-2.06625,-0.453277,0.882094,-2.05449,-0.476451,0.976597,-1.98534,-0.586257,0.716389,-2.01579,-0.615952,0.78256,-2.06905,0.109259,1.3266,-1.81816,0.112386,1.31857,-2.03322},
/*683*/{0.167564,1.87799,-1.70877,0.266307,1.8261,-1.84155,0.01051,1.77264,-1.66382,-0.108145,1.69233,-1.65561,0.02725,1.60436,-1.63889,0.096594,1.5461,-1.61718,0.235161,1.9006,-2.09212,0.281339,1.83019,-1.91498,0.091755,1.86372,-2.03224,0.10361,1.68427,-2.20625,0.156129,1.59168,-2.2361,0.307923,1.55396,-2.11197,0.36593,1.51994,-2.10218,0.18174,1.4131,-1.81035,-0.000598,1.46496,-1.87784,0.005679,1.49097,-1.93963,0.010772,1.46309,-1.9934,0.198615,1.39806,-2.06281,0.246504,1.11588,-1.74361,0.125328,1.06309,-1.77902,0.159689,0.987299,-1.80153,0.266467,0.903939,-1.78522,0.323018,0.72046,-1.7862,0.335359,0.645496,-1.80198,0.21275,0.699803,-1.80494,0.227583,0.630693,-1.8142,0.314663,0.420112,-1.84239,0.230889,0.355066,-1.92936,0.476167,0.409848,-1.87023,0.456349,0.343924,-1.81055,0.146911,1.07956,-2.10297,0.145485,0.995948,-2.08649,0.006803,1.08271,-2.07194,0.007827,1.00083,-2.05657,0.030196,0.872583,-2.06295,-0.141342,0.815577,-2.06446,-0.142541,0.929195,-2.06854,-0.213016,0.923256,-2.06562,-0.441643,0.890713,-2.05556,-0.463688,0.984593,-1.98561,-0.576063,0.726493,-2.01813,-0.604025,0.79304,-2.07133,0.109954,1.3222,-1.81731,0.113304,1.31331,-2.03234},
/*684*/{0.168492,1.87108,-1.70729,0.266264,1.82056,-1.84267,0.013101,1.76333,-1.66052,-0.104377,1.68136,-1.64965,0.031471,1.59525,-1.63618,0.102364,1.53889,-1.61567,0.23405,1.89529,-2.09349,0.281332,1.82526,-1.9153,0.090095,1.85881,-2.03082,0.099809,1.67787,-2.20605,0.150133,1.58613,-2.23583,0.302349,1.5438,-2.11473,0.359802,1.50763,-2.10568,0.179711,1.40773,-1.80813,-0.00266,1.46049,-1.87834,0.005163,1.48526,-1.93958,0.009089,1.45691,-1.99333,0.199055,1.39305,-2.06144,0.246585,1.10973,-1.74286,0.124677,1.05874,-1.77798,0.156581,0.982128,-1.80142,0.261034,0.898326,-1.78504,0.315048,0.713479,-1.78706,0.325921,0.639825,-1.80515,0.20421,0.696744,-1.80591,0.217153,0.626491,-1.81659,0.296432,0.417133,-1.84428,0.210131,0.355694,-1.92951,0.462281,0.40035,-1.87281,0.437271,0.336831,-1.81097,0.152827,1.07515,-2.10399,0.153011,0.991888,-2.08651,0.011153,1.07496,-2.07074,0.013995,0.994056,-2.05614,0.039648,0.867008,-2.0637,-0.132124,0.81435,-2.06571,-0.130198,0.928072,-2.06761,-0.199787,0.924042,-2.06463,-0.431189,0.897544,-2.05446,-0.451759,0.992261,-1.98669,-0.567092,0.734877,-2.0199,-0.592524,0.801763,-2.07351,0.107934,1.31709,-1.81621,0.113259,1.30789,-2.03118},
/*685*/{0.168981,1.86461,-1.70573,0.266384,1.8142,-1.84158,0.016073,1.75357,-1.65642,-0.099875,1.67097,-1.64373,0.036895,1.58712,-1.63265,0.108452,1.53198,-1.61501,0.231901,1.88972,-2.09471,0.281401,1.81944,-1.91573,0.08861,1.85339,-2.02981,0.096058,1.6735,-2.2056,0.143661,1.58078,-2.2356,0.296951,1.53225,-2.11719,0.353554,1.49559,-2.10899,0.179285,1.40426,-1.80622,-0.005678,1.4553,-1.8793,0.001962,1.48122,-1.94011,0.007995,1.45154,-1.99376,0.198085,1.38817,-2.06019,0.24263,1.10698,-1.74241,0.120654,1.0547,-1.77769,0.155429,0.978904,-1.80111,0.255205,0.892544,-1.78362,0.307316,0.709052,-1.78777,0.316593,0.635888,-1.80486,0.19698,0.695727,-1.80544,0.205939,0.625108,-1.8186,0.279628,0.417144,-1.84696,0.188577,0.356061,-1.92931,0.441778,0.396535,-1.87936,0.415653,0.339351,-1.81407,0.158113,1.07037,-2.10311,0.158935,0.986847,-2.08705,0.016426,1.06742,-2.07021,0.020567,0.98668,-2.05521,0.049712,0.860746,-2.06558,-0.122414,0.813222,-2.06587,-0.117042,0.928313,-2.07009,-0.187722,0.924445,-2.06439,-0.416925,0.904517,-2.05558,-0.43964,0.998987,-1.98532,-0.555009,0.741866,-2.02114,-0.580232,0.809415,-2.07525,0.106379,1.31351,-1.81498,0.112833,1.30288,-2.02985},
/*686*/{0.170309,1.8578,-1.7041,0.266299,1.80818,-1.84212,0.019204,1.74346,-1.65252,-0.095387,1.65993,-1.63758,0.043179,1.57805,-1.62916,0.115459,1.52456,-1.61276,0.230274,1.88507,-2.09542,0.281469,1.8142,-1.91648,0.087448,1.84847,-2.02815,0.090694,1.66816,-2.20571,0.136682,1.57542,-2.23548,0.290851,1.52196,-2.12032,0.346638,1.48358,-2.11242,0.17649,1.40003,-1.80464,-0.008219,1.45075,-1.87954,0.000412,1.47364,-1.93967,0.005656,1.44585,-1.99471,0.198187,1.38458,-2.05755,0.240755,1.10298,-1.73833,0.117711,1.05198,-1.77639,0.152284,0.977013,-1.80087,0.249104,0.885768,-1.78408,0.299638,0.706703,-1.78882,0.30714,0.632268,-1.80611,0.189128,0.698142,-1.80679,0.195786,0.627957,-1.81791,0.261239,0.418768,-1.85042,0.166551,0.356511,-1.9311,0.420663,0.394994,-1.88401,0.396015,0.340725,-1.81521,0.161472,1.06678,-2.1042,0.165942,0.982706,-2.08759,0.021433,1.06112,-2.06991,0.027824,0.979842,-2.05504,0.060133,0.853401,-2.06705,-0.110952,0.81183,-2.06656,-0.101491,0.925476,-2.06806,-0.173562,0.924794,-2.06472,-0.406265,0.910008,-2.0544,-0.427209,1.00548,-1.98741,-0.542642,0.747789,-2.02198,-0.566717,0.814459,-2.0764,0.10455,1.30879,-1.81387,0.113451,1.29809,-2.02865},
/*687*/{0.171321,1.85241,-1.70291,0.267003,1.80244,-1.84237,0.023576,1.73305,-1.64806,-0.090729,1.64889,-1.63153,0.050405,1.56964,-1.62654,0.122871,1.51833,-1.61159,0.228659,1.88081,-2.09648,0.281992,1.80857,-1.91685,0.084874,1.84473,-2.02624,0.086569,1.66354,-2.20525,0.130197,1.57062,-2.2353,0.285147,1.51132,-2.12331,0.338987,1.47153,-2.11609,0.173523,1.39601,-1.80371,-0.012088,1.44596,-1.88003,-0.002879,1.46865,-1.94204,0.000996,1.44005,-1.99415,0.196553,1.37939,-2.0575,0.236068,1.10216,-1.73505,0.113671,1.05129,-1.77644,0.148667,0.976097,-1.80091,0.244219,0.881726,-1.7843,0.29369,0.707102,-1.78904,0.297683,0.633739,-1.80796,0.182158,0.703763,-1.80661,0.18574,0.63263,-1.81939,0.238882,0.419578,-1.85255,0.143604,0.357593,-1.932,0.398722,0.392601,-1.88819,0.371371,0.341155,-1.81636,0.16702,1.06274,-2.10417,0.17183,0.979519,-2.08748,0.025885,1.05353,-2.06958,0.034803,0.972241,-2.05417,0.069953,0.848893,-2.06726,-0.099666,0.81031,-2.06706,-0.089557,0.924411,-2.06897,-0.1598,0.924053,-2.06362,-0.390769,0.914071,-2.05598,-0.414875,1.00944,-1.98874,-0.529268,0.752271,-2.02271,-0.553694,0.818739,-2.0787,0.101442,1.3048,-1.81321,0.111134,1.2927,-2.02789},
/*688*/{0.172703,1.84763,-1.70197,0.26749,1.79829,-1.84248,0.028476,1.72175,-1.64427,-0.084546,1.63696,-1.62425,0.057035,1.56132,-1.62266,0.131666,1.51145,-1.60995,0.226685,1.87598,-2.09781,0.282166,1.80386,-1.91801,0.084322,1.8418,-2.02267,0.081894,1.65944,-2.2054,0.122594,1.56587,-2.23475,0.277787,1.4997,-2.12641,0.329691,1.46001,-2.12015,0.170348,1.39201,-1.80187,-0.015484,1.44217,-1.88147,-0.006625,1.46398,-1.9432,0.002462,1.43298,-1.99766,0.194729,1.37411,-2.0568,0.232022,1.10106,-1.73149,0.111623,1.04999,-1.77598,0.148058,0.976041,-1.80096,0.24317,0.884214,-1.78303,0.288327,0.706845,-1.78909,0.287842,0.633066,-1.80843,0.17725,0.709783,-1.80548,0.17704,0.63814,-1.81786,0.218772,0.423084,-1.85394,0.121895,0.358739,-1.93332,0.376728,0.391548,-1.88808,0.349362,0.339663,-1.81667,0.170417,1.06051,-2.10396,0.178528,0.976688,-2.08798,0.030275,1.04586,-2.06877,0.04145,0.965614,-2.05337,0.07976,0.844169,-2.06733,-0.087182,0.808716,-2.06676,-0.075679,0.92243,-2.069,-0.145377,0.92354,-2.06412,-0.375906,0.917183,-2.05715,-0.402385,1.01338,-1.98922,-0.515724,0.754999,-2.02334,-0.539167,0.82105,-2.08006,0.098005,1.30096,-1.81251,0.110462,1.28704,-2.02693},
/*689*/{0.173612,1.84292,-1.70064,0.267988,1.79166,-1.84246,0.033628,1.71127,-1.64075,-0.07686,1.62626,-1.61812,0.065484,1.55327,-1.62024,0.140056,1.50498,-1.60926,0.226015,1.87242,-2.09913,0.283061,1.79925,-1.91881,0.08305,1.83947,-2.02106,0.077605,1.65561,-2.20519,0.116928,1.56231,-2.23397,0.270168,1.49025,-2.13017,0.320973,1.44909,-2.12403,0.167838,1.38986,-1.8,-0.019916,1.43831,-1.88318,-0.009772,1.4599,-1.94364,-0.00341,1.43031,-1.99728,0.193327,1.37072,-2.05493,0.231319,1.10231,-1.72794,0.110621,1.04968,-1.77592,0.148636,0.976363,-1.79864,0.24564,0.88912,-1.78038,0.280942,0.706379,-1.78932,0.277854,0.634107,-1.80887,0.170196,0.714356,-1.80446,0.166558,0.642754,-1.81789,0.199904,0.423745,-1.85433,0.100008,0.359762,-1.93435,0.355608,0.39008,-1.88704,0.327118,0.339096,-1.81714,0.174947,1.05864,-2.1032,0.185898,0.97563,-2.08786,0.035466,1.03948,-2.06823,0.048823,0.960142,-2.05336,0.091216,0.839902,-2.06639,-0.074641,0.807108,-2.06657,-0.061033,0.920513,-2.06714,-0.130994,0.9219,-2.06332,-0.363083,0.919488,-2.05724,-0.388125,1.01557,-1.99128,-0.499678,0.755718,-2.02315,-0.524332,0.821974,-2.08066,0.094578,1.29853,-1.81136,0.10842,1.28365,-2.02563},
/*690*/{0.174661,1.83893,-1.69985,0.267524,1.78886,-1.84321,0.039092,1.70182,-1.63769,-0.070508,1.61435,-1.61145,0.07493,1.54662,-1.61741,0.150043,1.49958,-1.60801,0.225544,1.86903,-2.09985,0.282865,1.79447,-1.92046,0.082336,1.83793,-2.01871,0.073376,1.65281,-2.20359,0.10955,1.55933,-2.23314,0.262263,1.48139,-2.13245,0.311537,1.43926,-2.12756,0.164758,1.38726,-1.79793,-0.022889,1.43434,-1.88481,-0.012011,1.45651,-1.94386,-0.006486,1.42443,-1.99971,0.190393,1.36674,-2.05439,0.227231,1.10239,-1.72723,0.108777,1.04962,-1.77624,0.14579,0.974615,-1.79925,0.250209,0.893371,-1.77855,0.272853,0.704867,-1.78967,0.266273,0.631957,-1.80765,0.161144,0.716651,-1.80273,0.154906,0.645182,-1.81739,0.18024,0.423945,-1.85407,0.078956,0.360653,-1.93422,0.334197,0.389148,-1.88735,0.304838,0.338448,-1.81669,0.178125,1.05902,-2.10291,0.192715,0.975949,-2.08772,0.039039,1.03536,-2.06941,0.057041,0.955335,-2.05406,0.103479,0.835618,-2.06557,-0.061074,0.804299,-2.06552,-0.045208,0.916613,-2.06724,-0.116505,0.921395,-2.06429,-0.344263,0.916833,-2.05609,-0.377168,1.01623,-1.9936,-0.484473,0.755791,-2.02376,-0.507045,0.820432,-2.08064,0.091452,1.29574,-1.81051,0.106594,1.27909,-2.02457},
/*691*/{0.17523,1.8357,-1.69917,0.266939,1.78426,-1.84443,0.045764,1.69433,-1.63478,-0.060726,1.60487,-1.6053,0.083464,1.53963,-1.61559,0.160386,1.49512,-1.60667,0.225177,1.86659,-2.09997,0.283312,1.79036,-1.92131,0.082401,1.83731,-2.01783,0.068826,1.65249,-2.20166,0.102454,1.55808,-2.23281,0.253879,1.4755,-2.1341,0.30267,1.43035,-2.13114,0.161931,1.38647,-1.79553,-0.026041,1.43112,-1.88745,-0.01346,1.45402,-1.94369,-0.009004,1.41913,-2.00074,0.188531,1.36371,-2.05354,0.223107,1.10097,-1.72819,0.106372,1.04724,-1.77527,0.145657,0.972766,-1.79796,0.256668,0.896,-1.7785,0.263263,0.703611,-1.78879,0.254092,0.630959,-1.80786,0.15153,0.718083,-1.8019,0.145008,0.648194,-1.81491,0.161063,0.425233,-1.85506,0.058055,0.361675,-1.93409,0.313507,0.388106,-1.88661,0.284937,0.338492,-1.81527,0.184507,1.05941,-2.10204,0.200071,0.976465,-2.08723,0.044003,1.03197,-2.06928,0.063744,0.951908,-2.0534,0.116744,0.834079,-2.06492,-0.047763,0.802542,-2.06442,-0.031436,0.915384,-2.06921,-0.101416,0.919183,-2.06416,-0.330459,0.917568,-2.05684,-0.364521,1.01574,-1.99503,-0.468073,0.7542,-2.02406,-0.492578,0.817926,-2.08231,0.088976,1.2944,-1.80903,0.105787,1.27544,-2.02277},
/*692*/{0.175854,1.83347,-1.69921,0.267686,1.77992,-1.84492,0.050902,1.68826,-1.63287,-0.049451,1.59652,-1.59998,0.094308,1.53521,-1.61382,0.17092,1.49126,-1.60536,0.224811,1.86331,-2.09938,0.282653,1.78672,-1.92264,0.082578,1.8358,-2.01767,0.062104,1.65483,-2.19949,0.095037,1.55825,-2.23201,0.24484,1.46937,-2.13582,0.292086,1.42269,-2.13436,0.15791,1.38553,-1.79305,-0.028472,1.42952,-1.88872,-0.014857,1.45336,-1.94508,-0.01149,1.41619,-2.0026,0.189297,1.36125,-2.05035,0.218343,1.0991,-1.72498,0.099962,1.04478,-1.77603,0.140901,0.970478,-1.79687,0.260349,0.895334,-1.77929,0.251784,0.701926,-1.78657,0.240334,0.629411,-1.8057,0.140806,0.719429,-1.80118,0.130355,0.648507,-1.81354,0.141425,0.425254,-1.85461,0.036516,0.362204,-1.934,0.292964,0.386979,-1.88524,0.263938,0.337452,-1.81524,0.188865,1.05973,-2.10153,0.208188,0.977664,-2.08639,0.051414,1.02817,-2.06999,0.072814,0.949095,-2.0547,0.131053,0.834184,-2.06424,-0.032765,0.800122,-2.06312,-0.015509,0.912573,-2.06928,-0.08631,0.916929,-2.06477,-0.315859,0.915595,-2.05701,-0.351659,1.01489,-1.99789,-0.451354,0.750378,-2.02372,-0.474891,0.813014,-2.08262,0.08481,1.29363,-1.80746,0.105402,1.27309,-2.02072},
/*693*/{0.1769,1.83131,-1.6997,0.266329,1.77703,-1.84587,0.056205,1.68401,-1.6313,-0.041592,1.58921,-1.59495,0.105043,1.53081,-1.61249,0.182327,1.48905,-1.6038,0.224954,1.86102,-2.09893,0.282356,1.78382,-1.92281,0.082211,1.83472,-2.01753,0.057164,1.65541,-2.19666,0.087477,1.55946,-2.23056,0.235771,1.46514,-2.13813,0.281538,1.41687,-2.13767,0.155009,1.38422,-1.79053,-0.029242,1.42798,-1.88942,-0.016391,1.45249,-1.94493,-0.013423,1.41496,-2.003,0.186173,1.35931,-2.04967,0.215908,1.09787,-1.72444,0.0967,1.04294,-1.77528,0.133918,0.968277,-1.79702,0.25504,0.891001,-1.77836,0.239725,0.699081,-1.78465,0.226835,0.625485,-1.80347,0.128771,0.71937,-1.80036,0.117099,0.649146,-1.81257,0.121058,0.4251,-1.85344,0.015134,0.362716,-1.93347,0.271359,0.387094,-1.88534,0.242434,0.336618,-1.81487,0.196076,1.06084,-2.10111,0.216906,0.979611,-2.08576,0.057679,1.02574,-2.07111,0.082039,0.946638,-2.05524,0.145732,0.833374,-2.06317,-0.018708,0.798491,-2.06107,-0.002319,0.910072,-2.06966,-0.070432,0.915703,-2.06467,-0.301248,0.911705,-2.05774,-0.340059,1.01214,-1.99964,-0.434595,0.745545,-2.02271,-0.457915,0.805982,-2.08269,0.081684,1.29242,-1.80663,0.103036,1.27129,-2.01976},
/*694*/{0.177756,1.82969,-1.70005,0.266755,1.77534,-1.84607,0.062013,1.68097,-1.62903,-0.03102,1.58373,-1.59008,0.115836,1.5279,-1.61053,0.194825,1.48759,-1.6035,0.225539,1.85915,-2.09884,0.282023,1.78262,-1.9234,0.082022,1.834,-2.01759,0.051193,1.6579,-2.1936,0.080067,1.56154,-2.2299,0.226624,1.46182,-2.14009,0.271177,1.4129,-2.14084,0.152578,1.38353,-1.78829,-0.031457,1.42729,-1.89057,-0.018436,1.45234,-1.94526,-0.015129,1.41426,-2.00315,0.184904,1.35776,-2.04956,0.208662,1.09228,-1.7246,0.090339,1.04083,-1.77605,0.128692,0.964931,-1.7978,0.244654,0.886383,-1.77714,0.225767,0.694397,-1.78351,0.211843,0.623323,-1.80227,0.116485,0.718085,-1.79985,0.103053,0.648533,-1.81299,0.100393,0.423692,-1.85163,-0.006193,0.362822,-1.93286,0.24998,0.385885,-1.88433,0.21986,0.336298,-1.81412,0.203605,1.06256,-2.10074,0.226431,0.981521,-2.08524,0.066249,1.02283,-2.07156,0.091306,0.945591,-2.05699,0.161739,0.833049,-2.06118,-0.003419,0.795654,-2.05784,0.01232,0.90879,-2.06987,-0.056994,0.914172,-2.06597,-0.291292,0.910559,-2.06008,-0.329484,1.00796,-2.00297,-0.415469,0.739896,-2.02176,-0.442681,0.798759,-2.08387,0.078654,1.29191,-1.80606,0.101221,1.2702,-2.019},
/*695*/{0.178893,1.82873,-1.70049,0.266696,1.77333,-1.84728,0.067096,1.67825,-1.62709,-0.02127,1.57894,-1.58574,0.127611,1.52632,-1.6101,0.207082,1.48773,-1.60286,0.22463,1.85739,-2.0988,0.280701,1.77941,-1.92291,0.081592,1.83406,-2.01809,0.045657,1.66183,-2.19089,0.071586,1.56477,-2.22955,0.217668,1.46128,-2.1417,0.260215,1.4094,-2.14355,0.149028,1.3826,-1.78659,-0.031936,1.4273,-1.89002,-0.019975,1.45139,-1.94518,-0.015946,1.41413,-2.00334,0.182169,1.3567,-2.04874,0.202992,1.08846,-1.72373,0.082655,1.03951,-1.77733,0.120493,0.963622,-1.79786,0.232989,0.881456,-1.77533,0.212455,0.690719,-1.78279,0.194343,0.618109,-1.80147,0.103365,0.717182,-1.79839,0.087584,0.647634,-1.81281,0.079004,0.424637,-1.85214,-0.028449,0.364139,-1.93264,0.228281,0.385495,-1.88434,0.198931,0.336267,-1.81331,0.211451,1.06514,-2.09965,0.236484,0.985214,-2.08447,0.075177,1.0235,-2.07442,0.102591,0.944281,-2.05816,0.177494,0.836081,-2.0604,0.010829,0.795303,-2.05678,0.026604,0.907577,-2.07025,-0.042269,0.912173,-2.06629,-0.277135,0.906149,-2.06051,-0.318818,1.00368,-2.00566,-0.395847,0.731036,-2.01934,-0.425209,0.789612,-2.08413,0.075519,1.29118,-1.80534,0.098959,1.26929,-2.01817},
/*696*/{0.180483,1.828,-1.7003,0.26592,1.77357,-1.84736,0.072228,1.67612,-1.62516,-0.011703,1.57434,-1.58109,0.138488,1.52534,-1.60826,0.219491,1.48898,-1.60313,0.223796,1.85643,-2.09851,0.280057,1.77909,-1.92329,0.080967,1.83399,-2.01834,0.040247,1.66657,-2.18773,0.063039,1.56851,-2.22953,0.207318,1.4593,-2.14394,0.249371,1.40777,-2.14605,0.148044,1.38244,-1.78558,-0.034312,1.42638,-1.89048,-0.02145,1.45027,-1.94501,-0.017821,1.41453,-2.00274,0.181749,1.35692,-2.04852,0.194824,1.08453,-1.72341,0.077832,1.03794,-1.77603,0.112289,0.960384,-1.79913,0.22054,0.87637,-1.77472,0.197774,0.685773,-1.78229,0.17895,0.615277,-1.80227,0.089152,0.715616,-1.79836,0.07126,0.646212,-1.81232,0.058826,0.424937,-1.8509,-0.0492,0.363891,-1.93151,0.20734,0.385611,-1.88323,0.17799,0.336156,-1.81218,0.219728,1.06815,-2.09946,0.247101,0.988979,-2.08389,0.084627,1.02191,-2.07476,0.113741,0.943715,-2.05919,0.192709,0.837278,-2.05942,0.02538,0.795328,-2.05645,0.040826,0.906,-2.07019,-0.02872,0.909732,-2.06842,-0.265798,0.899947,-2.05911,-0.309508,0.998155,-2.00795,-0.376742,0.722485,-2.01885,-0.407387,0.778387,-2.08451,0.074357,1.29068,-1.80523,0.098395,1.26939,-2.01806},
/*697*/{0.181831,1.82778,-1.7003,0.264802,1.77259,-1.84839,0.077129,1.67533,-1.62322,-0.002534,1.57094,-1.57673,0.149776,1.52555,-1.60687,0.232313,1.49176,-1.60332,0.223678,1.8561,-2.0985,0.279508,1.77816,-1.92351,0.080651,1.83414,-2.0183,0.034069,1.67186,-2.18482,0.055195,1.5729,-2.22946,0.198399,1.45974,-2.1463,0.238757,1.40732,-2.14867,0.145212,1.38178,-1.78341,-0.035519,1.4264,-1.88945,-0.02247,1.45059,-1.94438,-0.019249,1.41479,-2.00102,0.180254,1.35786,-2.0485,0.189301,1.07935,-1.72335,0.071037,1.03574,-1.77542,0.101237,0.959332,-1.79893,0.207745,0.871629,-1.77387,0.18293,0.681534,-1.78254,0.161772,0.610043,-1.80185,0.075005,0.712784,-1.79857,0.056437,0.644664,-1.81251,0.038672,0.425129,-1.85095,-0.069462,0.36447,-1.93127,0.186151,0.384763,-1.88315,0.156951,0.33566,-1.81131,0.227942,1.07226,-2.09866,0.258417,0.994501,-2.08317,0.094523,1.021,-2.07653,0.126547,0.944511,-2.05928,0.207824,0.839877,-2.0589,0.039844,0.795662,-2.05589,0.05389,0.905606,-2.07054,-0.014611,0.908153,-2.06814,-0.250026,0.894341,-2.06019,-0.301029,0.990925,-2.00967,-0.356343,0.712628,-2.01864,-0.387949,0.766903,-2.08494,0.072292,1.29008,-1.80474,0.096909,1.27002,-2.01762},
/*698*/{0.182778,1.82778,-1.70086,0.264802,1.77259,-1.84839,0.082465,1.67465,-1.62146,0.007657,1.56738,-1.57253,0.160622,1.527,-1.60533,0.245301,1.49528,-1.60432,0.222995,1.85602,-2.09867,0.278255,1.7773,-1.92396,0.079706,1.83473,-2.01844,0.027658,1.67803,-2.18236,0.046788,1.57824,-2.22919,0.188722,1.46228,-2.14866,0.227818,1.40868,-2.15177,0.143068,1.3822,-1.78273,-0.037794,1.42668,-1.88965,-0.024461,1.45028,-1.94383,-0.02092,1.41605,-2.00033,0.179177,1.35896,-2.04811,0.181459,1.07666,-1.72122,0.061241,1.03531,-1.77637,0.093127,0.957652,-1.79943,0.194575,0.865554,-1.77384,0.166863,0.67714,-1.78296,0.146312,0.606659,-1.80218,0.06026,0.711,-1.79815,0.040449,0.642786,-1.81259,0.016816,0.424983,-1.8507,-0.090612,0.364597,-1.93129,0.165737,0.384666,-1.88192,0.135167,0.335242,-1.81157,0.234902,1.07839,-2.09954,0.268596,1.00075,-2.0824,0.103582,1.02152,-2.0773,0.138169,0.948416,-2.06126,0.222135,0.842685,-2.05839,0.056838,0.79336,-2.05759,0.068167,0.905441,-2.07062,-0.002623,0.90683,-2.06825,-0.234269,0.886109,-2.05977,-0.292699,0.982375,-2.01107,-0.335563,0.701503,-2.01774,-0.368435,0.754733,-2.08576,0.07043,1.29025,-1.80465,0.095728,1.27101,-2.01753},
/*699*/{0.184842,1.82869,-1.70059,0.26459,1.77069,-1.84803,0.086561,1.67496,-1.61981,0.017037,1.56552,-1.56888,0.17115,1.52919,-1.60424,0.257086,1.50005,-1.6054,0.22154,1.8562,-2.09861,0.27802,1.77738,-1.92425,0.078823,1.836,-2.01797,0.021413,1.68365,-2.18033,0.038604,1.58358,-2.22902,0.17924,1.46383,-2.1506,0.21731,1.41046,-2.1539,0.142035,1.382,-1.78274,-0.039066,1.42679,-1.88934,-0.026247,1.45053,-1.94411,-0.022153,1.41649,-1.99948,0.178127,1.36046,-2.0481,0.173555,1.07526,-1.72084,0.050497,1.03667,-1.77791,0.081977,0.957745,-1.79864,0.18055,0.860032,-1.77415,0.151395,0.673021,-1.78333,0.131829,0.604989,-1.80271,0.04507,0.70832,-1.79859,0.024734,0.640689,-1.81365,-0.002093,0.425291,-1.85055,-0.111423,0.365417,-1.93173,0.144648,0.384341,-1.88298,0.113964,0.335191,-1.81181,0.24269,1.08443,-2.09899,0.278556,1.00808,-2.08122,0.113379,1.02289,-2.07796,0.151715,0.951254,-2.06243,0.2368,0.846375,-2.05857,0.072757,0.793582,-2.05709,0.080829,0.90593,-2.07102,0.010369,0.905537,-2.06869,-0.223068,0.880612,-2.06115,-0.283826,0.97265,-2.01183,-0.31362,0.690815,-2.0179,-0.347367,0.740902,-2.08574,0.069553,1.29011,-1.80507,0.094895,1.27201,-2.01804},
/*700*/{0.186147,1.83013,-1.70058,0.264978,1.77178,-1.84951,0.091148,1.67571,-1.61804,0.027309,1.56309,-1.56473,0.181342,1.53189,-1.60371,0.268553,1.50551,-1.60675,0.219644,1.85739,-2.09844,0.277507,1.77833,-1.92466,0.077322,1.83736,-2.01691,0.017096,1.6903,-2.17872,0.029671,1.59024,-2.22945,0.170183,1.46891,-2.15231,0.207249,1.41405,-2.15575,0.140128,1.38072,-1.78164,-0.039929,1.42714,-1.88918,-0.026015,1.45139,-1.94376,-0.02327,1.41764,-1.99905,0.177579,1.36271,-2.04759,0.165922,1.07331,-1.72132,0.044425,1.03661,-1.7767,0.070182,0.958471,-1.80022,0.166199,0.85496,-1.77426,0.135891,0.670608,-1.78401,0.111935,0.600471,-1.80324,0.029026,0.707322,-1.79648,0.008682,0.64013,-1.8137,-0.022712,0.425964,-1.85061,-0.131925,0.366317,-1.93125,0.124701,0.384093,-1.88226,0.093874,0.335561,-1.81151,0.249298,1.09106,-2.09915,0.288371,1.01587,-2.08129,0.122367,1.0255,-2.07885,0.161195,0.954645,-2.06341,0.251213,0.851047,-2.0579,0.088602,0.793398,-2.0548,0.092704,0.907002,-2.07203,0.022572,0.904746,-2.06971,-0.205563,0.871641,-2.05851,-0.275937,0.961971,-2.01204,-0.291144,0.680547,-2.01762,-0.327133,0.727481,-2.08603,0.06792,1.28923,-1.80584,0.094389,1.27365,-2.01887},
/*701*/{0.188492,1.83203,-1.70083,0.264423,1.77369,-1.85033,0.09662,1.67715,-1.61621,0.03754,1.56253,-1.56133,0.191575,1.53545,-1.60361,0.279039,1.51173,-1.60882,0.216781,1.85842,-2.09819,0.275332,1.77991,-1.92585,0.076565,1.83876,-2.01556,0.011053,1.69681,-2.17743,0.021347,1.59623,-2.22962,0.161717,1.47406,-2.15402,0.19737,1.4184,-2.15756,0.139368,1.38084,-1.78168,-0.040008,1.42879,-1.88843,-0.027184,1.45281,-1.9428,-0.024646,1.41799,-1.99857,0.176736,1.36521,-2.0476,0.159342,1.07292,-1.7226,0.036244,1.03729,-1.77616,0.059806,0.957812,-1.79966,0.150652,0.852112,-1.77463,0.119361,0.668751,-1.78458,0.09533,0.599758,-1.80416,0.013481,0.706773,-1.79924,-0.008672,0.639701,-1.8124,-0.042547,0.426708,-1.85111,-0.152537,0.366788,-1.93143,0.102644,0.383765,-1.88237,0.072287,0.335782,-1.8106,0.256145,1.09738,-2.0986,0.29709,1.02418,-2.08119,0.131038,1.02797,-2.07892,0.172469,0.958585,-2.06334,0.265292,0.85602,-2.05769,0.1041,0.794152,-2.05452,0.1053,0.907649,-2.07301,0.034719,0.903404,-2.07063,-0.19709,0.863357,-2.06019,-0.268937,0.950575,-2.01215,-0.268138,0.667003,-2.01737,-0.305874,0.713205,-2.08713,0.067542,1.28978,-1.80613,0.093579,1.27543,-2.0193},
/*702*/{0.19105,1.83416,-1.70101,0.263714,1.77565,-1.85088,0.102674,1.67839,-1.61437,0.046783,1.56319,-1.55792,0.201316,1.54028,-1.60355,0.289823,1.51896,-1.61083,0.214648,1.86062,-2.09845,0.274252,1.78137,-1.92692,0.074794,1.84074,-2.01508,0.004897,1.70325,-2.17688,0.012457,1.60326,-2.22981,0.151687,1.47916,-2.15532,0.187085,1.42321,-2.15941,0.138414,1.38023,-1.78165,-0.04085,1.42993,-1.8884,-0.027695,1.45365,-1.94284,-0.024708,1.41868,-1.99819,0.176423,1.36682,-2.04751,0.15194,1.07179,-1.72415,0.023975,1.03851,-1.77839,0.049442,0.957669,-1.79811,0.132959,0.849693,-1.77526,0.102417,0.667359,-1.78527,0.076875,0.598044,-1.80472,-0.001854,0.707515,-1.79988,-0.025129,0.641297,-1.81324,-0.063169,0.426631,-1.85073,-0.173155,0.367717,-1.93195,0.08269,0.383707,-1.88181,0.051957,0.335778,-1.81115,0.261076,1.10567,-2.09882,0.305327,1.03344,-2.08105,0.139032,1.03192,-2.07914,0.182657,0.962857,-2.06355,0.279602,0.86214,-2.05792,0.119467,0.795405,-2.05354,0.116898,0.909196,-2.07405,0.046636,0.902603,-2.07192,-0.184727,0.854253,-2.05923,-0.261217,0.93662,-2.01075,-0.244257,0.654133,-2.01831,-0.284964,0.698649,-2.08751,0.066563,1.28966,-1.80677,0.093128,1.2767,-2.01996},
/*703*/{0.193192,1.83646,-1.70163,0.263398,1.77744,-1.85207,0.108433,1.6803,-1.61347,0.055981,1.56373,-1.55554,0.21086,1.54483,-1.60401,0.299789,1.52686,-1.61396,0.212012,1.86314,-2.09886,0.274338,1.78328,-1.92774,0.073111,1.84191,-2.01226,-0.00151,1.70973,-2.17621,0.00499,1.6107,-2.22965,0.143093,1.48477,-2.15644,0.178103,1.42925,-2.1608,0.138218,1.38022,-1.78186,-0.040023,1.43245,-1.88766,-0.028116,1.45485,-1.94163,-0.025448,1.42011,-1.99811,0.174059,1.36777,-2.04764,0.142979,1.06854,-1.72574,0.016647,1.04098,-1.77704,0.037861,0.958941,-1.79743,0.115995,0.847726,-1.7768,0.086541,0.66675,-1.78575,0.060159,0.597758,-1.8051,-0.017999,0.708241,-1.80096,-0.04293,0.642297,-1.81515,-0.082806,0.427234,-1.85083,-0.193615,0.368721,-1.93219,0.062376,0.383638,-1.88237,0.031392,0.335802,-1.81136,0.26691,1.11264,-2.09846,0.312201,1.04274,-2.08098,0.146547,1.03622,-2.07953,0.19257,0.969704,-2.06311,0.293825,0.8708,-2.05812,0.135283,0.797834,-2.05346,0.127297,0.91132,-2.07393,0.058042,0.901216,-2.07269,-0.171229,0.844007,-2.05879,-0.252861,0.923052,-2.01004,-0.220175,0.641015,-2.01944,-0.262547,0.68426,-2.08849,0.06579,1.29045,-1.8068,0.09101,1.27772,-2.02017},
/*704*/{0.195703,1.83961,-1.70192,0.265057,1.78021,-1.85277,0.11441,1.68239,-1.61199,0.06678,1.56512,-1.55351,0.219808,1.55143,-1.60481,0.308145,1.53535,-1.6164,0.208925,1.86579,-2.09908,0.273154,1.78581,-1.92926,0.072006,1.8444,-2.01142,-0.007708,1.71643,-2.17573,-0.003792,1.6175,-2.22959,0.134505,1.49154,-2.15778,0.168651,1.43583,-2.16245,0.13795,1.38031,-1.78147,-0.041163,1.43319,-1.88735,-0.028699,1.45684,-1.94122,-0.025237,1.42244,-1.99808,0.174023,1.3703,-2.04699,0.134831,1.06691,-1.72717,0.009012,1.0431,-1.77582,0.027553,0.960635,-1.79647,0.100677,0.847575,-1.77798,0.069885,0.666473,-1.78733,0.042313,0.598223,-1.80615,-0.033475,0.708708,-1.80191,-0.059381,0.644456,-1.81444,-0.102676,0.428947,-1.85123,-0.213492,0.369534,-1.93191,0.042159,0.383693,-1.88296,0.011172,0.336098,-1.81156,0.27162,1.12125,-2.09835,0.319609,1.05192,-2.08074,0.154101,1.03978,-2.07863,0.201446,0.975346,-2.06253,0.305839,0.880569,-2.05766,0.150142,0.799338,-2.05382,0.137997,0.911403,-2.07422,0.068673,0.898838,-2.0724,-0.160124,0.83467,-2.05749,-0.245552,0.906943,-2.00837,-0.19484,0.627997,-2.02054,-0.24064,0.668101,-2.08899,0.065022,1.2908,-1.80753,0.09099,1.28013,-2.02093},
/*705*/{0.197713,1.84294,-1.70268,0.263835,1.78227,-1.8538,0.120734,1.68484,-1.61104,0.074368,1.56676,-1.55074,0.228026,1.5572,-1.60663,0.31714,1.54449,-1.62006,0.205824,1.86902,-2.09921,0.272649,1.78807,-1.92979,0.070473,1.84576,-2.00868,-0.015068,1.72333,-2.17498,-0.01086,1.62491,-2.22923,0.125663,1.49775,-2.15859,0.159779,1.44092,-2.16331,0.13826,1.38083,-1.78164,-0.041483,1.43558,-1.88664,-0.028332,1.45843,-1.94071,-0.025361,1.42443,-1.99774,0.174246,1.37244,-2.04623,0.12696,1.066,-1.7295,-4.6e-005,1.0449,-1.77564,0.015656,0.962073,-1.7977,0.085731,0.848409,-1.7799,0.052568,0.66623,-1.78821,0.024089,0.597967,-1.80695,-0.050165,0.710692,-1.80246,-0.076473,0.645447,-1.81529,-0.12277,0.429023,-1.85132,-0.233746,0.371758,-1.93169,0.02254,0.382681,-1.88268,-0.008774,0.335827,-1.81129,0.2762,1.12939,-2.09759,0.32554,1.06161,-2.08053,0.160639,1.04478,-2.07872,0.210158,0.980648,-2.0628,0.317419,0.89066,-2.05802,0.164939,0.801788,-2.05349,0.147627,0.912757,-2.07482,0.078993,0.897415,-2.07321,-0.145904,0.823192,-2.05554,-0.236224,0.889693,-2.0066,-0.168992,0.614982,-2.02235,-0.216576,0.654045,-2.09091,0.064815,1.29189,-1.80768,0.090824,1.28219,-2.02112},
/*706*/{0.200149,1.847,-1.70315,0.265004,1.78574,-1.85433,0.12811,1.68723,-1.60961,0.084315,1.56922,-1.54874,0.236611,1.56456,-1.60786,0.325029,1.55356,-1.62335,0.201297,1.87263,-2.09892,0.271829,1.79096,-1.93113,0.068637,1.84819,-2.00693,-0.02018,1.73017,-2.1743,-0.018694,1.6321,-2.22886,0.117805,1.50481,-2.15974,0.152772,1.44992,-2.16541,0.13805,1.38072,-1.7796,-0.040475,1.43741,-1.8854,-0.028076,1.46076,-1.93893,-0.02496,1.42625,-1.99649,0.173432,1.37469,-2.04557,0.118759,1.06535,-1.73151,-0.008331,1.04819,-1.77478,0.005071,0.965502,-1.79743,0.071901,0.849344,-1.78147,0.035363,0.667017,-1.78954,0.008033,0.600219,-1.80826,-0.066782,0.713232,-1.80287,-0.094066,0.64746,-1.81536,-0.141139,0.431403,-1.85042,-0.25391,0.373664,-1.93105,0.002026,0.382288,-1.88302,-0.028969,0.336578,-1.81027,0.280442,1.13703,-2.09767,0.331769,1.07106,-2.08045,0.167263,1.04938,-2.07829,0.218117,0.986925,-2.06257,0.329541,0.901663,-2.05808,0.179473,0.804397,-2.05329,0.157259,0.913539,-2.07554,0.08929,0.89605,-2.07335,-0.134147,0.812182,-2.0546,-0.228183,0.872125,-2.00318,-0.142694,0.603102,-2.02456,-0.193603,0.638733,-2.09147,0.064344,1.29249,-1.80721,0.090268,1.28426,-2.02072},
/*707*/{0.202867,1.85116,-1.70371,0.265712,1.78967,-1.85585,0.134481,1.69024,-1.60874,0.093827,1.57225,-1.54696,0.243875,1.57092,-1.60889,0.332312,1.56288,-1.62609,0.198394,1.87664,-2.0988,0.271893,1.79485,-1.93263,0.066587,1.85069,-2.00527,-0.026179,1.73662,-2.17373,-0.02597,1.63963,-2.2283,0.110404,1.51276,-2.16009,0.144934,1.45735,-2.16681,0.138728,1.38169,-1.7794,-0.039973,1.43986,-1.88498,-0.028115,1.46347,-1.93761,-0.02528,1.4297,-1.99684,0.173414,1.37775,-2.04453,0.112173,1.06502,-1.73231,-0.014302,1.04926,-1.77274,-0.005393,0.967348,-1.79609,0.05793,0.850794,-1.78283,0.018111,0.668049,-1.79061,-0.012189,0.600789,-1.809,-0.084089,0.713061,-1.80483,-0.111836,0.64956,-1.81806,-0.162265,0.432765,-1.8503,-0.273918,0.377296,-1.93105,-0.018494,0.382224,-1.88303,-0.050347,0.336207,-1.81131,0.284527,1.14496,-2.09766,0.337555,1.08019,-2.08026,0.173654,1.05456,-2.07728,0.225984,0.993855,-2.06152,0.33979,0.912272,-2.05767,0.193157,0.807043,-2.0537,0.167102,0.913641,-2.07524,0.09831,0.894203,-2.07315,-0.120309,0.799849,-2.05438,-0.218063,0.853001,-2.00228,-0.115573,0.590999,-2.02692,-0.168862,0.624322,-2.09307,0.064178,1.29404,-1.80763,0.090102,1.28737,-2.0212},
/*708*/{0.205157,1.85566,-1.7044,0.266591,1.79318,-1.85639,0.141761,1.69394,-1.60821,0.102176,1.57545,-1.54545,0.250423,1.57868,-1.61115,0.338253,1.57227,-1.62992,0.195072,1.88071,-2.09902,0.271534,1.7982,-1.93308,0.065259,1.85296,-2.00203,-0.030366,1.74367,-2.17266,-0.033136,1.64715,-2.22758,0.102691,1.52004,-2.16026,0.137623,1.46543,-2.16804,0.139213,1.3823,-1.77802,-0.039323,1.44187,-1.88389,-0.027584,1.46567,-1.93631,-0.024347,1.43275,-1.99653,0.174371,1.38175,-2.04312,0.107969,1.06734,-1.73429,-0.025215,1.05308,-1.7751,-0.016508,0.971402,-1.79691,0.046524,0.852141,-1.78459,0.000184,0.670134,-1.79189,-0.029641,0.602929,-1.80982,-0.101358,0.715853,-1.80505,-0.130143,0.651488,-1.81813,-0.182061,0.434979,-1.84953,-0.293295,0.381804,-1.93071,-0.038067,0.382137,-1.88325,-0.071887,0.336232,-1.81252,0.287871,1.15256,-2.09759,0.342846,1.08953,-2.08009,0.178856,1.06036,-2.0775,0.233239,1.00072,-2.06177,0.34982,0.923351,-2.05764,0.20725,0.810277,-2.05382,0.174846,0.914532,-2.07518,0.10749,0.892362,-2.07253,-0.103095,0.786629,-2.05391,-0.207391,0.834239,-2.00028,-0.088155,0.580523,-2.02983,-0.143554,0.609809,-2.09387,0.06455,1.29507,-1.80786,0.091371,1.29086,-2.02138},
/*709*/{0.207826,1.86092,-1.70553,0.26665,1.7964,-1.85753,0.14847,1.69788,-1.6081,0.109856,1.57887,-1.54416,0.256566,1.58665,-1.61275,0.344428,1.58239,-1.63329,0.191757,1.88487,-2.09869,0.271392,1.80241,-1.93449,0.064467,1.85574,-2.00064,-0.035908,1.7507,-2.17216,-0.039936,1.65439,-2.22624,0.09457,1.5282,-2.16122,0.129939,1.4729,-2.1696,0.139372,1.38321,-1.77662,-0.038533,1.44501,-1.88356,-0.026465,1.46865,-1.93492,-0.025272,1.43688,-1.99544,0.173684,1.38485,-2.04183,0.099675,1.06631,-1.73616,-0.03212,1.05923,-1.77513,-0.025705,0.974913,-1.79701,0.035102,0.853973,-1.78668,-0.016367,0.672438,-1.79302,-0.046562,0.60638,-1.81136,-0.118153,0.719213,-1.8064,-0.146536,0.655544,-1.81931,-0.200152,0.437168,-1.84892,-0.313427,0.387237,-1.92979,-0.058029,0.381988,-1.8835,-0.091571,0.337311,-1.81204,0.292718,1.16112,-2.09751,0.347865,1.09753,-2.07944,0.184266,1.06596,-2.07718,0.239976,1.00605,-2.06143,0.358373,0.933656,-2.05806,0.219998,0.813738,-2.05442,0.180433,0.919309,-2.07674,0.116149,0.890552,-2.07193,-0.083767,0.771505,-2.05113,-0.196198,0.814256,-1.99798,-0.060511,0.568271,-2.03179,-0.117669,0.596014,-2.0959,0.063761,1.29688,-1.80772,0.090583,1.29411,-2.02126},
/*710*/{0.209469,1.86632,-1.70649,0.267534,1.80109,-1.8595,0.153915,1.70266,-1.6078,0.118965,1.58402,-1.5433,0.262651,1.5934,-1.6139,0.349997,1.59212,-1.63669,0.189349,1.88913,-2.0982,0.271555,1.8061,-1.93605,0.063544,1.85781,-1.99856,-0.041246,1.75871,-2.17155,-0.046882,1.66201,-2.22488,0.088965,1.53607,-2.16086,0.123596,1.48049,-2.17069,0.14029,1.38415,-1.77554,-0.036596,1.44714,-1.88199,-0.025921,1.47168,-1.93251,-0.023134,1.4404,-1.99512,0.174326,1.38846,-2.04064,0.094164,1.06751,-1.73908,-0.0402,1.06313,-1.77526,-0.034862,0.979089,-1.79756,0.023171,0.857086,-1.78936,-0.032434,0.67585,-1.79405,-0.062789,0.609925,-1.81156,-0.133999,0.723514,-1.80753,-0.163564,0.65963,-1.8194,-0.218628,0.443504,-1.84966,-0.332561,0.393472,-1.92887,-0.077109,0.380426,-1.8827,-0.112297,0.336759,-1.81237,0.295011,1.16898,-2.09736,0.351205,1.10817,-2.08051,0.189682,1.07269,-2.07686,0.246089,1.01393,-2.06092,0.366142,0.944271,-2.05812,0.232847,0.817272,-2.05524,0.18734,0.920034,-2.07544,0.125025,0.887136,-2.07072,-0.070671,0.760807,-2.05015,-0.181897,0.793195,-1.99356,-0.032063,0.557009,-2.03327,-0.091915,0.582646,-2.0982,0.0646,1.29823,-1.80775,0.09161,1.29766,-2.02128},
/*711*/{0.212293,1.8714,-1.70761,0.26893,1.80508,-1.86029,0.160345,1.70743,-1.608,0.12496,1.58819,-1.54253,0.267231,1.60116,-1.61538,0.354403,1.60193,-1.6402,0.186813,1.89317,-2.09772,0.271462,1.8101,-1.93722,0.06235,1.86099,-1.99611,-0.044393,1.76453,-2.17065,-0.052981,1.66987,-2.22344,0.082022,1.54382,-2.16108,0.117033,1.48864,-2.17175,0.140807,1.38516,-1.7744,-0.035822,1.44984,-1.88166,-0.024945,1.47464,-1.9312,-0.021123,1.44458,-1.99445,0.174912,1.39229,-2.03929,0.086765,1.06888,-1.7414,-0.045257,1.06954,-1.77449,-0.046109,0.98558,-1.79724,0.009107,0.860527,-1.79094,-0.049242,0.680095,-1.79453,-0.08072,0.612933,-1.81186,-0.149987,0.728829,-1.80772,-0.179794,0.664898,-1.81956,-0.23762,0.446944,-1.84845,-0.350468,0.40058,-1.92836,-0.097492,0.379311,-1.88299,-0.132251,0.337917,-1.81269,0.298146,1.17686,-2.09736,0.354953,1.11674,-2.07959,0.19367,1.07937,-2.07707,0.251725,1.02109,-2.06052,0.373385,0.954445,-2.0579,0.245635,0.820521,-2.05648,0.194034,0.920734,-2.07534,0.133454,0.885054,-2.06956,-0.054473,0.746775,-2.04891,-0.169226,0.771687,-1.99224,-0.003508,0.547476,-2.03614,-0.062733,0.569935,-2.09916,0.064638,1.29985,-1.8081,0.092583,1.30149,-2.0215},
/*712*/{0.214089,1.87713,-1.70897,0.26971,1.80941,-1.86144,0.16554,1.71246,-1.60782,0.131092,1.59257,-1.54076,0.272047,1.60948,-1.61746,0.358817,1.61237,-1.64323,0.184443,1.89752,-2.09751,0.271435,1.8146,-1.93844,0.061507,1.86456,-1.99487,-0.04751,1.77131,-2.16908,-0.058481,1.67778,-2.22125,0.075571,1.55046,-2.16016,0.110917,1.49594,-2.17285,0.141874,1.38598,-1.77336,-0.035397,1.45348,-1.88022,-0.023516,1.47817,-1.92868,-0.02101,1.45005,-1.99401,0.175537,1.39675,-2.03799,0.082099,1.07025,-1.74328,-0.053324,1.07565,-1.77576,-0.054377,0.991143,-1.79853,-0.005466,0.865011,-1.79158,-0.06511,0.684732,-1.79498,-0.097645,0.617834,-1.81168,-0.166133,0.733548,-1.80875,-0.196123,0.670388,-1.82078,-0.255251,0.450873,-1.84652,-0.368749,0.408662,-1.92746,-0.116282,0.378464,-1.88282,-0.153286,0.337202,-1.81176,0.300361,1.18493,-2.09691,0.358537,1.12652,-2.07953,0.197886,1.08571,-2.07712,0.256653,1.02864,-2.06005,0.377915,0.96477,-2.05732,0.256249,0.82397,-2.05659,0.198887,0.920875,-2.0741,0.141014,0.881883,-2.06852,-0.039583,0.735007,-2.0473,-0.154423,0.749333,-1.98949,0.026463,0.537424,-2.03769,-0.036698,0.557554,-2.10105,0.064606,1.3017,-1.80859,0.092692,1.30623,-2.02194},
/*713*/{0.215923,1.88217,-1.71016,0.269335,1.81389,-1.86315,0.170082,1.71774,-1.60785,0.138088,1.5982,-1.54064,0.275714,1.61685,-1.61896,0.362133,1.62135,-1.64673,0.181902,1.90127,-2.097,0.271446,1.81925,-1.93934,0.060532,1.86782,-1.9933,-0.052085,1.77743,-2.16763,-0.064248,1.68501,-2.21965,0.069699,1.55746,-2.16013,0.104667,1.50308,-2.1739,0.142932,1.38697,-1.77253,-0.031972,1.45663,-1.87871,-0.02275,1.48192,-1.9269,-0.019739,1.45441,-1.99369,0.175903,1.40108,-2.0369,0.074936,1.07146,-1.74525,-0.058756,1.0808,-1.77556,-0.064037,0.998337,-1.79992,-0.021362,0.869272,-1.79151,-0.080506,0.689802,-1.79575,-0.114015,0.623766,-1.81297,-0.181158,0.739511,-1.80879,-0.212101,0.676086,-1.82091,-0.272181,0.45691,-1.84652,-0.386062,0.417824,-1.92681,-0.135549,0.377447,-1.8837,-0.174585,0.338337,-1.81332,0.303092,1.19321,-2.09673,0.362036,1.13471,-2.07893,0.201872,1.09217,-2.07761,0.258692,1.03539,-2.06032,0.381585,0.973331,-2.0568,0.267298,0.827597,-2.05808,0.2046,0.920485,-2.07323,0.148081,0.877839,-2.06729,-0.026227,0.723117,-2.04928,-0.1402,0.729353,-1.98795,0.055872,0.52907,-2.04027,-0.008884,0.545444,-2.10276,0.065442,1.30363,-1.80902,0.093357,1.31047,-2.02233},
/*714*/{0.216675,1.88805,-1.71199,0.269937,1.81806,-1.86391,0.174354,1.72321,-1.60828,0.143254,1.60272,-1.53989,0.279735,1.6252,-1.62142,0.365439,1.63031,-1.65,0.180399,1.90444,-2.09681,0.270952,1.82295,-1.94045,0.060089,1.87153,-1.99244,-0.054441,1.78353,-2.16538,-0.069972,1.6928,-2.21743,0.064256,1.56491,-2.16002,0.098982,1.51078,-2.17425,0.145155,1.38808,-1.77256,-0.03111,1.46035,-1.87777,-0.022051,1.48558,-1.92536,-0.018587,1.45974,-1.99217,0.176107,1.40547,-2.03616,0.070304,1.07386,-1.74786,-0.063161,1.08664,-1.77589,-0.07158,1.0041,-1.79919,-0.036584,0.876257,-1.79168,-0.095313,0.695679,-1.79532,-0.128571,0.630273,-1.81118,-0.196148,0.745849,-1.80916,-0.225902,0.68244,-1.8189,-0.289716,0.463253,-1.84553,-0.403534,0.426801,-1.92596,-0.154589,0.376601,-1.88344,-0.195037,0.338769,-1.81345,0.305121,1.20114,-2.09689,0.364845,1.14487,-2.07767,0.205246,1.09956,-2.07761,0.262346,1.04255,-2.06019,0.384294,0.983035,-2.05725,0.276354,0.829292,-2.05805,0.209553,0.920236,-2.07141,0.154585,0.872934,-2.06559,-0.005248,0.70758,-2.04453,-0.123514,0.706597,-1.98578,0.084705,0.521551,-2.04272,0.019442,0.535055,-2.10426,0.066228,1.3058,-1.80979,0.093477,1.31518,-2.02309},
/*715*/{0.217996,1.89277,-1.71338,0.270252,1.82256,-1.86551,0.178498,1.72874,-1.60822,0.148399,1.60839,-1.53943,0.282837,1.63221,-1.62329,0.367618,1.63917,-1.65309,0.180314,1.90859,-2.09672,0.270048,1.8271,-1.94203,0.060128,1.87535,-1.99187,-0.057212,1.79053,-2.16364,-0.074799,1.70042,-2.21534,0.059253,1.57207,-2.15945,0.093541,1.51748,-2.17484,0.146059,1.38842,-1.77218,-0.029756,1.46378,-1.877,-0.020343,1.48941,-1.92299,-0.017777,1.46428,-1.99143,0.175662,1.40989,-2.03544,0.064298,1.0759,-1.74833,-0.069614,1.09443,-1.77668,-0.079348,1.01038,-1.80088,-0.050231,0.882468,-1.7925,-0.109566,0.701774,-1.79494,-0.143898,0.635953,-1.81125,-0.209935,0.753435,-1.80906,-0.241256,0.690128,-1.82019,-0.304154,0.467228,-1.84331,-0.41984,0.437109,-1.92455,-0.174396,0.376233,-1.88384,-0.215413,0.339155,-1.81397,0.306664,1.20914,-2.09708,0.366822,1.15189,-2.07715,0.207933,1.10565,-2.07727,0.265801,1.05007,-2.05899,0.386447,0.991123,-2.0568,0.284475,0.830277,-2.05848,0.21392,0.919038,-2.07031,0.163142,0.868924,-2.06429,0.01196,0.694528,-2.04333,-0.105776,0.683413,-1.98222,0.114436,0.515216,-2.04541,0.046873,0.524905,-2.10549,0.066293,1.30741,-1.81077,0.09325,1.31966,-2.02396},
/*716*/{0.220143,1.89877,-1.7145,0.271611,1.82687,-1.86717,0.181596,1.73376,-1.6087,0.152397,1.6131,-1.53913,0.28596,1.63986,-1.62475,0.369599,1.64777,-1.65622,0.178921,1.91206,-2.0966,0.270558,1.83152,-1.94283,0.059566,1.87854,-1.99091,-0.060341,1.79686,-2.16072,-0.079123,1.70748,-2.21245,0.054375,1.57823,-2.15878,0.088644,1.52385,-2.17527,0.148882,1.38942,-1.77423,-0.027548,1.46729,-1.87532,-0.018587,1.49294,-1.92078,-0.016278,1.46947,-1.99042,0.176689,1.41501,-2.03524,0.061615,1.07838,-1.75052,-0.074313,1.09902,-1.77651,-0.08703,1.01738,-1.79975,-0.061827,0.889205,-1.79226,-0.123497,0.708775,-1.79419,-0.15787,0.643221,-1.80943,-0.222684,0.760135,-1.80918,-0.255024,0.697497,-1.82001,-0.321405,0.473349,-1.84243,-0.434736,0.447875,-1.92447,-0.193014,0.375657,-1.88389,-0.236664,0.339396,-1.81378,0.30837,1.21581,-2.09651,0.368823,1.15962,-2.07663,0.210175,1.11175,-2.07803,0.270156,1.05812,-2.05854,0.386961,0.999176,-2.05673,0.296931,0.834849,-2.05881,0.218022,0.915458,-2.06861,0.170605,0.863984,-2.06286,0.028957,0.679909,-2.04171,-0.085648,0.662063,-1.98028,0.144861,0.509952,-2.04742,0.076031,0.51558,-2.10685,0.068484,1.30923,-1.81255,0.09405,1.32496,-2.02568},
/*717*/{0.221319,1.90333,-1.71507,0.272682,1.83147,-1.86829,0.185092,1.73925,-1.60832,0.156389,1.6183,-1.53806,0.287595,1.64657,-1.62619,0.371382,1.65584,-1.65925,0.178063,1.91476,-2.09703,0.270566,1.83546,-1.94403,0.059513,1.88154,-1.98962,-0.06044,1.80413,-2.15746,-0.083503,1.7145,-2.20987,0.049815,1.5851,-2.15832,0.084271,1.53016,-2.17545,0.151673,1.38991,-1.7752,-0.025363,1.47111,-1.8746,-0.017037,1.4969,-1.91912,-0.014988,1.47433,-1.98893,0.176657,1.42018,-2.03505,0.05441,1.08165,-1.75286,-0.07527,1.10703,-1.77477,-0.094249,1.02609,-1.80069,-0.071699,0.895522,-1.79242,-0.136449,0.715908,-1.79317,-0.171131,0.651017,-1.80849,-0.235382,0.768158,-1.80973,-0.267809,0.705236,-1.81852,-0.335146,0.480091,-1.84001,-0.4499,0.459645,-1.9231,-0.211471,0.376332,-1.88335,-0.257305,0.339954,-1.81439,0.309345,1.22223,-2.09668,0.370499,1.16697,-2.07586,0.211942,1.11811,-2.07889,0.272597,1.06505,-2.05816,0.386898,1.00672,-2.05665,0.306792,0.836647,-2.05903,0.223282,0.914218,-2.06718,0.178768,0.857941,-2.06101,0.046876,0.666499,-2.04041,-0.065727,0.638638,-1.97717,0.17396,0.504059,-2.04851,0.104964,0.506683,-2.10743,0.070019,1.31113,-1.81424,0.094309,1.33013,-2.02725},
/*718*/{0.221956,1.90784,-1.7167,0.27273,1.8351,-1.86944,0.187643,1.74421,-1.60851,0.160332,1.62342,-1.53777,0.289327,1.65362,-1.62795,0.373258,1.66298,-1.66254,0.177073,1.9176,-2.09734,0.270518,1.839,-1.94484,0.059528,1.88494,-1.98958,-0.063078,1.81067,-2.1547,-0.087244,1.72164,-2.20764,0.045809,1.59089,-2.15781,0.079629,1.53631,-2.17578,0.153686,1.39155,-1.77544,-0.023132,1.47506,-1.87241,-0.015895,1.50052,-1.91677,-0.013648,1.47906,-1.98791,0.1775,1.42538,-2.03569,0.05211,1.08335,-1.75324,-0.080301,1.11425,-1.77564,-0.100033,1.03408,-1.80056,-0.08088,0.902029,-1.79192,-0.147908,0.723332,-1.79199,-0.183249,0.658635,-1.80597,-0.246513,0.776755,-1.80846,-0.279049,0.713466,-1.81716,-0.349112,0.48688,-1.8374,-0.4631,0.47212,-1.92196,-0.231059,0.375831,-1.88353,-0.277919,0.34061,-1.81461,0.309953,1.22881,-2.09614,0.371261,1.17344,-2.07532,0.213577,1.12347,-2.07847,0.272852,1.07087,-2.05809,0.385592,1.01332,-2.05703,0.314952,0.83777,-2.05918,0.22728,0.909495,-2.06527,0.186719,0.852315,-2.06009,0.065968,0.652458,-2.03873,-0.043795,0.618134,-1.974,0.202879,0.499952,-2.04946,0.133908,0.497461,-2.10697,0.071877,1.31359,-1.81521,0.094846,1.33529,-2.02812},
/*719*/{0.224586,1.91265,-1.71777,0.274375,1.84011,-1.8717,0.18983,1.74883,-1.60804,0.164533,1.62857,-1.53778,0.290542,1.65988,-1.62979,0.374008,1.67018,-1.66581,0.177384,1.92043,-2.09707,0.270914,1.84246,-1.94573,0.059973,1.88916,-1.98947,-0.0664,1.81937,-2.15067,-0.090997,1.72851,-2.20463,0.042508,1.59654,-2.15626,0.075663,1.54187,-2.1756,0.156262,1.39234,-1.77659,-0.021419,1.47831,-1.87085,-0.0152,1.50418,-1.91553,-0.012154,1.48445,-1.98649,0.177043,1.42982,-2.03593,0.048094,1.08653,-1.75427,-0.084745,1.12042,-1.77556,-0.105081,1.04118,-1.8008,-0.08923,0.908403,-1.7915,-0.158691,0.730698,-1.79055,-0.194389,0.664759,-1.80436,-0.256911,0.78458,-1.80796,-0.290695,0.721178,-1.81787,-0.362609,0.49378,-1.83622,-0.475439,0.485391,-1.91891,-0.25049,0.375468,-1.88324,-0.297934,0.34268,-1.81449,0.310519,1.23483,-2.09578,0.371743,1.17914,-2.0741,0.213134,1.12898,-2.07813,0.273002,1.07667,-2.05839,0.384962,1.01875,-2.0567,0.324849,0.838695,-2.06004,0.232708,0.90509,-2.06335,0.194632,0.846111,-2.05879,0.082793,0.638712,-2.0384,-0.022391,0.597978,-1.97259,0.231304,0.496031,-2.0492,0.162408,0.489651,-2.1067,0.073088,1.31556,-1.81679,0.094494,1.34017,-2.02954},
/*720*/{0.226185,1.9165,-1.71877,0.273943,1.84319,-1.87184,0.192119,1.75393,-1.60844,0.166667,1.63243,-1.53651,0.29224,1.66532,-1.63102,0.374437,1.67672,-1.66804,0.176659,1.92343,-2.0969,0.271408,1.84569,-1.94575,0.059938,1.89235,-1.98843,-0.067561,1.82534,-2.14652,-0.094108,1.7346,-2.20151,0.03822,1.6017,-2.15609,0.071866,1.54624,-2.17513,0.158984,1.39378,-1.77845,-0.019153,1.4823,-1.86974,-0.01333,1.50791,-1.91242,-0.010155,1.48904,-1.98526,0.177867,1.434,-2.03643,0.045076,1.08972,-1.75443,-0.085037,1.12875,-1.77613,-0.11017,1.0495,-1.80051,-0.096462,0.915342,-1.79008,-0.16848,0.738385,-1.78887,-0.204368,0.672932,-1.80258,-0.266479,0.791755,-1.80726,-0.299227,0.729562,-1.81567,-0.37482,0.500937,-1.83349,-0.485223,0.499636,-1.91744,-0.269224,0.37629,-1.88295,-0.318049,0.344797,-1.81479,0.311101,1.23971,-2.09495,0.371636,1.18409,-2.07358,0.213528,1.13478,-2.07898,0.272398,1.08206,-2.06049,0.383491,1.02297,-2.0562,0.332707,0.83881,-2.06013,0.237038,0.900107,-2.06215,0.203243,0.838199,-2.05597,0.102163,0.628031,-2.0408,0.001658,0.578153,-1.97124,0.2605,0.493271,-2.04771,0.191374,0.48231,-2.10514,0.075061,1.31783,-1.81806,0.094695,1.34491,-2.03068},
/*721*/{0.227341,1.91983,-1.71992,0.275837,1.84767,-1.87337,0.19328,1.7581,-1.60882,0.16873,1.6367,-1.5362,0.292793,1.66988,-1.63257,0.374892,1.6822,-1.6698,0.176898,1.92624,-2.09716,0.271661,1.84931,-1.94741,0.06004,1.89567,-1.98804,-0.068197,1.83101,-2.14199,-0.096617,1.74076,-2.19857,0.036017,1.60719,-2.15515,0.068353,1.55125,-2.17456,0.161093,1.3951,-1.77876,-0.017346,1.48576,-1.86808,-0.011424,1.51157,-1.91136,-0.008517,1.49422,-1.98361,0.177191,1.43849,-2.03621,0.04185,1.09201,-1.7535,-0.087048,1.13505,-1.77561,-0.114204,1.05628,-1.80083,-0.104692,0.922408,-1.79116,-0.176802,0.745212,-1.78762,-0.212914,0.680441,-1.79973,-0.274796,0.798977,-1.80612,-0.309289,0.736263,-1.81523,-0.38538,0.508784,-1.83119,-0.496153,0.514766,-1.91438,-0.28693,0.377481,-1.88216,-0.338197,0.347479,-1.81442,0.310864,1.24484,-2.0947,0.37189,1.18786,-2.07266,0.212875,1.1401,-2.07937,0.271599,1.08638,-2.06231,0.382867,1.02536,-2.05622,0.340218,0.838389,-2.06026,0.242513,0.894437,-2.06078,0.211363,0.83127,-2.05524,0.12397,0.614455,-2.03688,0.023905,0.558776,-1.96976,0.288953,0.490331,-2.04659,0.220382,0.475686,-2.10406,0.076129,1.32023,-1.81897,0.094545,1.34963,-2.03139},
/*722*/{0.22914,1.9235,-1.72054,0.276721,1.85083,-1.87413,0.196119,1.76281,-1.60871,0.170697,1.64081,-1.53602,0.2927,1.67459,-1.63429,0.375207,1.68731,-1.67272,0.175795,1.92836,-2.0963,0.271949,1.85228,-1.94744,0.060373,1.8976,-1.987,-0.068671,1.8361,-2.13875,-0.098932,1.74636,-2.19541,0.03373,1.61166,-2.15439,0.06573,1.55592,-2.17406,0.163272,1.39705,-1.78001,-0.015283,1.48891,-1.86582,-0.011734,1.51575,-1.90846,-0.007504,1.49811,-1.98209,0.178052,1.44273,-2.03699,0.038224,1.09594,-1.75236,-0.09045,1.14176,-1.77696,-0.116962,1.0638,-1.80011,-0.110865,0.929554,-1.79006,-0.184998,0.752234,-1.78609,-0.222793,0.686762,-1.79853,-0.281955,0.806856,-1.80449,-0.315641,0.743943,-1.81159,-0.396569,0.517081,-1.82847,-0.504542,0.529395,-1.91164,-0.304632,0.378859,-1.88183,-0.358216,0.35105,-1.81417,0.310629,1.24808,-2.09336,0.370545,1.19067,-2.0719,0.21137,1.14416,-2.08008,0.26977,1.09029,-2.06379,0.38085,1.02716,-2.05721,0.348198,0.838239,-2.05992,0.247363,0.889163,-2.06077,0.219367,0.824445,-2.05488,0.146128,0.602194,-2.0337,0.047106,0.541611,-1.96848,0.316942,0.488659,-2.04408,0.249838,0.469379,-2.10186,0.078003,1.3227,-1.8197,0.094413,1.35427,-2.03197},
/*723*/{0.230843,1.92703,-1.72106,0.27722,1.85415,-1.8749,0.197389,1.76616,-1.60875,0.173615,1.6444,-1.53525,0.293036,1.67849,-1.63491,0.374856,1.69168,-1.67523,0.176214,1.93097,-2.09629,0.273067,1.85558,-1.94782,0.062794,1.89897,-1.98629,-0.067264,1.83621,-2.13291,-0.100625,1.75162,-2.19279,0.031622,1.61558,-2.15272,0.063312,1.55978,-2.17359,0.165397,1.39897,-1.78075,-0.014201,1.49243,-1.86389,-0.011508,1.51909,-1.90703,-0.006468,1.50342,-1.98058,0.177417,1.44643,-2.03728,0.035285,1.09945,-1.75174,-0.091406,1.14721,-1.77527,-0.120141,1.06898,-1.80033,-0.115878,0.935456,-1.79001,-0.190353,0.758708,-1.78502,-0.227693,0.694753,-1.79731,-0.286876,0.814651,-1.80262,-0.32211,0.751906,-1.81042,-0.404602,0.524812,-1.82457,-0.51183,0.547736,-1.90915,-0.321616,0.380423,-1.88105,-0.378444,0.356214,-1.81403,0.31008,1.25118,-2.09295,0.369125,1.19311,-2.07155,0.209694,1.14766,-2.0809,0.267704,1.09331,-2.06454,0.379327,1.02812,-2.05763,0.354951,0.837541,-2.06069,0.252637,0.883989,-2.06088,0.22789,0.816642,-2.0534,0.163671,0.589414,-2.03293,0.072633,0.523664,-1.96648,0.343007,0.48795,-2.04091,0.278008,0.463306,-2.0988,0.078871,1.3254,-1.82027,0.093552,1.35863,-2.03242},
/*724*/{0.231963,1.92953,-1.72177,0.278222,1.85634,-1.87563,0.198707,1.76993,-1.60845,0.173814,1.64704,-1.53499,0.293467,1.68227,-1.63601,0.37445,1.69491,-1.67694,0.175981,1.93319,-2.09599,0.273882,1.85797,-1.94881,0.063607,1.90136,-1.98615,-0.06742,1.84088,-2.13,-0.102788,1.75632,-2.19012,0.02996,1.61927,-2.15143,0.060676,1.5627,-2.17335,0.16684,1.40106,-1.78206,-0.012946,1.4955,-1.86266,-0.010084,1.5231,-1.90514,-0.006347,1.50738,-1.97839,0.177084,1.44984,-2.0371,0.033462,1.10404,-1.75121,-0.09191,1.15168,-1.77502,-0.121725,1.0746,-1.80014,-0.118889,0.940622,-1.79037,-0.194966,0.764812,-1.78353,-0.233567,0.700638,-1.79494,-0.291291,0.820622,-1.80135,-0.327463,0.758389,-1.80754,-0.412214,0.533691,-1.82191,-0.518724,0.564446,-1.90472,-0.340166,0.383058,-1.88084,-0.397318,0.361662,-1.81481,0.309086,1.2527,-2.0916,0.367865,1.19448,-2.07121,0.207417,1.15047,-2.08099,0.26554,1.09504,-2.0648,0.377327,1.02859,-2.05902,0.362525,0.835865,-2.06096,0.256557,0.876577,-2.05961,0.234675,0.808646,-2.05228,0.182021,0.577936,-2.03156,0.094278,0.507399,-1.9664,0.368926,0.487276,-2.03727,0.306754,0.458725,-2.09551,0.079695,1.32797,-1.82062,0.092696,1.36261,-2.03265},
/*725*/{0.233899,1.93227,-1.72245,0.279366,1.86042,-1.87679,0.199921,1.77295,-1.60869,0.175425,1.6501,-1.53375,0.293731,1.68462,-1.63648,0.374377,1.69752,-1.67928,0.176057,1.93518,-2.09578,0.27432,1.86022,-1.9494,0.064349,1.90406,-1.98528,-0.067302,1.8439,-2.12551,-0.104106,1.7602,-2.18719,0.028466,1.62267,-2.15083,0.058555,1.56589,-2.17253,0.167858,1.40359,-1.78206,-0.011419,1.49865,-1.8602,-0.009093,1.52609,-1.90352,-0.004879,1.51053,-1.97614,0.177556,1.45382,-2.03732,0.033545,1.10622,-1.75053,-0.091122,1.15487,-1.77159,-0.121962,1.07841,-1.80176,-0.119529,0.944556,-1.7902,-0.199399,0.769845,-1.78329,-0.238158,0.705794,-1.79358,-0.294647,0.827752,-1.79846,-0.330663,0.76625,-1.80518,-0.419225,0.541893,-1.8178,-0.523084,0.582507,-1.89996,-0.358269,0.386643,-1.88215,-0.417129,0.367953,-1.81644,0.307551,1.25385,-2.09086,0.364931,1.19467,-2.07119,0.205383,1.1521,-2.08116,0.26286,1.09543,-2.06523,0.375083,1.02788,-2.05992,0.368275,0.833818,-2.06098,0.261098,0.869647,-2.05967,0.243021,0.801522,-2.05241,0.203852,0.568136,-2.03204,0.119035,0.491624,-1.96614,0.394886,0.487116,-2.03304,0.334218,0.453696,-2.0915,0.081211,1.33058,-1.82036,0.092995,1.36645,-2.03226},
/*726*/{0.235106,1.93427,-1.72275,0.279109,1.86256,-1.87662,0.200808,1.77586,-1.60834,0.175665,1.6519,-1.5333,0.293909,1.68702,-1.63756,0.374181,1.70006,-1.68098,0.175228,1.93698,-2.09558,0.274723,1.86248,-1.94945,0.06534,1.90577,-1.98401,-0.06778,1.84785,-2.12309,-0.105862,1.76351,-2.18501,0.027012,1.6252,-2.15003,0.057213,1.56844,-2.17204,0.169352,1.40592,-1.78314,-0.010226,1.5021,-1.85859,-0.008582,1.52951,-1.90266,-0.004555,1.5138,-1.97451,0.177228,1.45668,-2.03734,0.031122,1.11083,-1.74838,-0.090683,1.15845,-1.77564,-0.123828,1.08203,-1.80106,-0.119333,0.948285,-1.7906,-0.2021,0.774562,-1.78249,-0.243458,0.711739,-1.79145,-0.296169,0.833962,-1.7972,-0.334689,0.773809,-1.80254,-0.426192,0.550993,-1.81496,-0.527714,0.600652,-1.89334,-0.376196,0.392098,-1.88405,-0.436781,0.375622,-1.81845,0.305876,1.25346,-2.09045,0.362584,1.19344,-2.07153,0.202497,1.15307,-2.08143,0.258855,1.09661,-2.0667,0.371786,1.02626,-2.06125,0.374003,0.831538,-2.06144,0.265302,0.861555,-2.05821,0.250101,0.794692,-2.05267,0.222878,0.557691,-2.03319,0.141437,0.477423,-1.96529,0.418281,0.487599,-2.02816,0.360169,0.449769,-2.08693,0.082071,1.33336,-1.82026,0.092179,1.36969,-2.03216},
/*727*/{0.237265,1.93567,-1.7227,0.280944,1.86475,-1.87725,0.201299,1.7783,-1.60821,0.176673,1.65493,-1.53306,0.293646,1.68808,-1.63887,0.37383,1.70126,-1.68329,0.176522,1.93891,-2.0958,0.275372,1.86441,-1.95072,0.065417,1.90854,-1.98335,-0.068424,1.85067,-2.12119,-0.106252,1.76612,-2.18238,0.026483,1.62722,-2.14845,0.055497,1.57021,-2.17184,0.170794,1.40899,-1.78321,-0.010215,1.50467,-1.85739,-0.008814,1.53244,-1.90222,-0.004349,1.51718,-1.97282,0.177052,1.45924,-2.03694,0.031148,1.11428,-1.74892,-0.09354,1.16129,-1.77742,-0.123466,1.08518,-1.80102,-0.11846,0.950448,-1.79045,-0.203832,0.778058,-1.78212,-0.245923,0.7172,-1.79098,-0.297275,0.840341,-1.79483,-0.337067,0.781138,-1.79961,-0.432504,0.560518,-1.81273,-0.531284,0.618422,-1.88514,-0.394136,0.399848,-1.88609,-0.454008,0.383663,-1.8218,0.303109,1.25274,-2.09071,0.359779,1.19166,-2.07159,0.198782,1.15354,-2.08157,0.254322,1.09544,-2.06667,0.369364,1.02445,-2.06255,0.379228,0.829175,-2.06266,0.26843,0.854334,-2.05868,0.257049,0.785022,-2.05155,0.242381,0.547948,-2.02947,0.164148,0.463845,-1.96388,0.44012,0.48899,-2.02412,0.386526,0.446336,-2.08221,0.082889,1.33631,-1.81956,0.091918,1.37257,-2.03153},
/*728*/{0.237986,1.9372,-1.72301,0.280955,1.86681,-1.87784,0.201833,1.78045,-1.60826,0.176813,1.65661,-1.53229,0.293614,1.68937,-1.63946,0.3728,1.70187,-1.68491,0.176031,1.94089,-2.09585,0.275759,1.86621,-1.95094,0.066156,1.90937,-1.98305,-0.068002,1.8518,-2.11749,-0.106932,1.76833,-2.18052,0.025109,1.62881,-2.14736,0.055163,1.57144,-2.171,0.17047,1.41135,-1.78318,-0.008791,1.50719,-1.856,-0.008036,1.53561,-1.90057,-0.004552,1.51977,-1.97168,0.176388,1.46139,-2.03659,0.032716,1.116,-1.74881,-0.09347,1.16269,-1.777,-0.121447,1.08623,-1.80151,-0.117167,0.95165,-1.79149,-0.205169,0.782014,-1.78163,-0.249346,0.72049,-1.79088,-0.296732,0.845858,-1.79205,-0.337711,0.787661,-1.79676,-0.440322,0.569302,-1.81106,-0.534354,0.634656,-1.87538,-0.412203,0.408598,-1.89002,-0.472151,0.392088,-1.82556,0.300608,1.25051,-2.09049,0.356261,1.18857,-2.07183,0.195315,1.15296,-2.0818,0.249236,1.09437,-2.06701,0.366165,1.02163,-2.064,0.383354,0.826681,-2.06295,0.271565,0.84693,-2.0601,0.262898,0.778073,-2.05395,0.261688,0.539236,-2.0287,0.184616,0.453328,-1.96269,0.461111,0.490123,-2.01918,0.410554,0.44252,-2.07713,0.082854,1.33875,-1.81879,0.090896,1.37504,-2.0308},
/*729*/{0.239593,1.93796,-1.72319,0.282637,1.86935,-1.87903,0.202534,1.7821,-1.60812,0.176956,1.65778,-1.53184,0.29336,1.68972,-1.63986,0.372665,1.70238,-1.68618,0.176674,1.94203,-2.09599,0.276367,1.86766,-1.95123,0.066268,1.91214,-1.98303,-0.068263,1.85407,-2.11645,-0.107606,1.76979,-2.17889,0.02522,1.62989,-2.14682,0.054061,1.5728,-2.17075,0.171488,1.41419,-1.78315,-0.008767,1.50941,-1.85525,-0.007967,1.53799,-1.89993,-0.00444,1.5215,-1.96988,0.17634,1.46374,-2.03661,0.033199,1.11738,-1.74921,-0.091043,1.16349,-1.77673,-0.119776,1.08708,-1.80174,-0.11429,0.95218,-1.7909,-0.205398,0.783712,-1.78173,-0.251355,0.724088,-1.79112,-0.295133,0.85094,-1.78995,-0.337409,0.794546,-1.7938,-0.445527,0.577801,-1.80856,-0.537585,0.65139,-1.86651,-0.430335,0.418538,-1.89522,-0.490162,0.403687,-1.8284,0.297973,1.24771,-2.09053,0.352579,1.185,-2.07244,0.190053,1.15253,-2.08178,0.245067,1.09253,-2.06741,0.362831,1.01795,-2.06519,0.386932,0.824027,-2.06352,0.274678,0.838685,-2.06171,0.268506,0.769627,-2.05423,0.279061,0.530731,-2.02781,0.205938,0.441903,-1.96194,0.481611,0.490952,-2.01475,0.433877,0.43955,-2.07132,0.083929,1.34124,-1.81802,0.090882,1.37724,-2.03011},
/*730*/{0.240651,1.93888,-1.72322,0.282995,1.8704,-1.87884,0.202266,1.78334,-1.60838,0.176735,1.65879,-1.53183,0.293287,1.68955,-1.64066,0.372663,1.70002,-1.68713,0.1768,1.94389,-2.09613,0.277283,1.86826,-1.95186,0.067972,1.91349,-1.98193,-0.068635,1.8547,-2.11423,-0.107797,1.77064,-2.1769,0.025294,1.63066,-2.14616,0.054083,1.57316,-2.17037,0.171868,1.4169,-1.78251,-0.008019,1.51212,-1.85401,-0.008983,1.54065,-1.8993,-0.005286,1.5235,-1.96832,0.176513,1.46488,-2.0366,0.037122,1.11817,-1.74914,-0.087077,1.16483,-1.77579,-0.116485,1.08691,-1.80047,-0.110079,0.951941,-1.79203,-0.204931,0.785571,-1.78179,-0.252624,0.727436,-1.79152,-0.292535,0.855811,-1.78675,-0.336463,0.799383,-1.79059,-0.450885,0.588703,-1.80657,-0.543356,0.667597,-1.85722,-0.446105,0.432707,-1.90011,-0.502872,0.415592,-1.83137,0.294443,1.24354,-2.09147,0.348375,1.18091,-2.07278,0.18603,1.14973,-2.08206,0.239089,1.08973,-2.0674,0.359155,1.01391,-2.06568,0.389947,0.820599,-2.06319,0.276784,0.8312,-2.06491,0.275314,0.761577,-2.05555,0.295869,0.522827,-2.02693,0.225869,0.432561,-1.96018,0.499936,0.491874,-2.01076,0.45534,0.436984,-2.06681,0.08413,1.34399,-1.81646,0.090049,1.37877,-2.02878},
/*731*/{0.241274,1.9391,-1.7236,0.283724,1.87173,-1.87985,0.202557,1.78375,-1.60861,0.176552,1.65894,-1.53107,0.292992,1.6882,-1.6412,0.37193,1.69828,-1.68799,0.177576,1.94477,-2.09627,0.277948,1.86918,-1.95301,0.067535,1.91602,-1.98207,-0.069077,1.85499,-2.11308,-0.108139,1.77057,-2.17492,0.025018,1.63016,-2.14491,0.053649,1.57287,-2.16953,0.17223,1.41905,-1.78217,-0.007532,1.51273,-1.85259,-0.009577,1.54235,-1.89895,-0.005018,1.52476,-1.96756,0.176176,1.46644,-2.03613,0.04165,1.11797,-1.75107,-0.086171,1.16144,-1.77571,-0.112523,1.08543,-1.7994,-0.104865,0.951064,-1.7919,-0.203599,0.78573,-1.78317,-0.255245,0.729875,-1.79197,-0.289253,0.858779,-1.78474,-0.335307,0.805403,-1.78767,-0.456704,0.599557,-1.80511,-0.548608,0.682821,-1.84808,-0.461912,0.446481,-1.90392,-0.516105,0.42954,-1.83399,0.291651,1.24006,-2.09162,0.343948,1.17636,-2.07342,0.180479,1.14792,-2.08247,0.233076,1.08709,-2.06748,0.354531,1.00927,-2.06656,0.392135,0.818651,-2.06308,0.278458,0.824224,-2.06707,0.280206,0.752594,-2.05721,0.309395,0.514805,-2.02519,0.244471,0.4241,-1.95793,0.516232,0.492675,-2.00673,0.475478,0.434541,-2.0618,0.084867,1.34561,-1.81567,0.090141,1.38016,-2.02805},
/*732*/{0.242029,1.93862,-1.72382,0.28352,1.87236,-1.88082,0.202827,1.7842,-1.60858,0.176487,1.65941,-1.53091,0.292262,1.68719,-1.64162,0.371215,1.69566,-1.68873,0.177694,1.94579,-2.09651,0.27827,1.8704,-1.95285,0.068779,1.91562,-1.98194,-0.069203,1.85499,-2.11123,-0.107822,1.77066,-2.17446,0.025697,1.62971,-2.14444,0.054353,1.57149,-2.16908,0.172736,1.42077,-1.78161,-0.007539,1.51422,-1.85189,-0.009434,1.54387,-1.89863,-0.005139,1.52497,-1.96689,0.176073,1.46669,-2.03595,0.048113,1.11713,-1.75062,-0.082164,1.16178,-1.7747,-0.107957,1.08281,-1.79799,-0.099277,0.949074,-1.79252,-0.203031,0.785735,-1.78412,-0.25593,0.732471,-1.7938,-0.284424,0.863221,-1.78187,-0.333496,0.812091,-1.78596,-0.462708,0.610916,-1.80428,-0.554054,0.697422,-1.8393,-0.4743,0.461319,-1.90946,-0.526462,0.442958,-1.83615,0.287618,1.23593,-2.09181,0.339329,1.17066,-2.07386,0.175211,1.1457,-2.08082,0.227891,1.08327,-2.067,0.349605,1.00349,-2.06692,0.394657,0.815895,-2.06201,0.281144,0.815982,-2.06808,0.285895,0.746193,-2.05832,0.32624,0.509658,-2.02479,0.263694,0.416547,-1.95582,0.53099,0.49459,-2.00381,0.493612,0.433195,-2.05712,0.085221,1.34724,-1.8143,0.089842,1.38043,-2.02691},
/*733*/{0.243823,1.93848,-1.7233,0.283962,1.87342,-1.8812,0.202855,1.7841,-1.60871,0.175366,1.65916,-1.53044,0.291955,1.68498,-1.64237,0.370137,1.6925,-1.68921,0.178642,1.9464,-2.09638,0.278726,1.87073,-1.95308,0.069944,1.9177,-1.98079,-0.069185,1.85489,-2.11083,-0.107495,1.76995,-2.17186,0.025994,1.6285,-2.14359,0.054874,1.57088,-2.16873,0.173196,1.42277,-1.78143,-0.007713,1.51459,-1.85176,-0.008567,1.54484,-1.89912,-0.005303,1.52495,-1.9664,0.1761,1.46737,-2.03535,0.053163,1.11527,-1.75208,-0.077105,1.15983,-1.77305,-0.102249,1.07959,-1.79703,-0.091991,0.946785,-1.79275,-0.200549,0.786141,-1.78482,-0.257533,0.735715,-1.79448,-0.279085,0.867205,-1.77951,-0.330536,0.817347,-1.78245,-0.467313,0.622321,-1.80293,-0.559712,0.712387,-1.83167,-0.48612,0.478438,-1.91246,-0.535766,0.458146,-1.83873,0.283861,1.23068,-2.09251,0.334994,1.16502,-2.07412,0.170488,1.14282,-2.08044,0.221463,1.07877,-2.06617,0.344601,0.998791,-2.06709,0.397294,0.8138,-2.06046,0.283105,0.80851,-2.06958,0.29052,0.739027,-2.05885,0.339808,0.503784,-2.02324,0.279007,0.410042,-1.95782,0.545022,0.49559,-2.00033,0.51002,0.431732,-2.05317,0.086064,1.34857,-1.81341,0.090436,1.38074,-2.02618},
/*734*/{0.244451,1.93762,-1.72333,0.28384,1.87345,-1.88121,0.20232,1.78355,-1.60859,0.173444,1.6584,-1.5313,0.290428,1.68301,-1.64262,0.370007,1.68868,-1.68985,0.178729,1.94667,-2.09637,0.278812,1.87075,-1.95322,0.070323,1.9181,-1.98091,-0.067924,1.85363,-2.10991,-0.106362,1.76883,-2.1712,0.027406,1.62786,-2.14319,0.055191,1.56826,-2.16893,0.173156,1.42366,-1.78103,-0.007914,1.51498,-1.85206,-0.008634,1.54536,-1.89941,-0.005481,1.52478,-1.96605,0.176267,1.46751,-2.03481,0.058436,1.11344,-1.75481,-0.072306,1.15509,-1.77141,-0.096812,1.07598,-1.79588,-0.084279,0.943111,-1.794,-0.199626,0.786013,-1.78582,-0.257575,0.738797,-1.79543,-0.272688,0.870872,-1.77579,-0.327392,0.823794,-1.77988,-0.474138,0.634549,-1.80267,-0.564476,0.727755,-1.82497,-0.496724,0.494692,-1.91273,-0.544022,0.472708,-1.83997,0.279715,1.22562,-2.09292,0.330181,1.15878,-2.07492,0.165106,1.13942,-2.07922,0.214449,1.07502,-2.06588,0.338601,0.993162,-2.06723,0.398397,0.811249,-2.05829,0.286139,0.801305,-2.07137,0.295616,0.733202,-2.0607,0.35109,0.498397,-2.02216,0.291002,0.404168,-1.95259,0.556412,0.497542,-1.99728,0.522499,0.430805,-2.0492,0.086119,1.34927,-1.81277,0.090714,1.38075,-2.02564},
/*735*/{0.244714,1.93713,-1.72345,0.283723,1.8736,-1.8814,0.201641,1.78276,-1.60917,0.172588,1.65833,-1.53187,0.290055,1.6799,-1.64307,0.369342,1.684,-1.69046,0.17918,1.94672,-2.09646,0.278899,1.87074,-1.95335,0.070566,1.91824,-1.98146,-0.067185,1.85319,-2.10929,-0.10529,1.76719,-2.16993,0.029702,1.62488,-2.14237,0.05575,1.56646,-2.1682,0.173207,1.42465,-1.78072,-0.008051,1.51461,-1.85207,-0.008634,1.54536,-1.89941,-0.006162,1.52366,-1.96594,0.17674,1.4672,-2.03409,0.063532,1.11144,-1.75663,-0.06684,1.15136,-1.76922,-0.090813,1.07237,-1.79447,-0.075975,0.939295,-1.79477,-0.198051,0.786842,-1.78631,-0.258139,0.742338,-1.796,-0.266931,0.874847,-1.77333,-0.32305,0.830254,-1.7779,-0.480152,0.648699,-1.80131,-0.567988,0.743747,-1.82058,-0.504907,0.511491,-1.91455,-0.55299,0.488462,-1.84026,0.275938,1.22004,-2.09348,0.325048,1.15285,-2.07526,0.15973,1.13531,-2.07904,0.20901,1.07096,-2.06525,0.33162,0.987397,-2.06694,0.399361,0.808198,-2.05534,0.287999,0.794955,-2.07277,0.29913,0.727197,-2.06136,0.360847,0.495132,-2.02122,0.303472,0.400981,-1.94995,0.565866,0.497258,-1.99474,0.534918,0.429812,-2.04584,0.086659,1.34966,-1.81181,0.091304,1.38012,-2.02483},
/*736*/{0.245032,1.93619,-1.72353,0.283661,1.87314,-1.88146,0.20052,1.78188,-1.60907,0.170203,1.6573,-1.53165,0.28875,1.67641,-1.64343,0.36782,1.67896,-1.69089,0.179349,1.94672,-2.09643,0.278389,1.8701,-1.95308,0.071156,1.91854,-1.98208,-0.066216,1.85114,-2.10886,-0.103605,1.76461,-2.16942,0.030991,1.62238,-2.14248,0.057107,1.56368,-2.16794,0.173409,1.42511,-1.78063,-0.007786,1.51387,-1.85158,-0.008567,1.54484,-1.89912,-0.006292,1.52246,-1.96545,0.177107,1.4666,-2.03368,0.072452,1.11048,-1.75886,-0.060902,1.14519,-1.76679,-0.084817,1.06752,-1.79565,-0.068114,0.935328,-1.79595,-0.195725,0.788419,-1.78708,-0.258788,0.746813,-1.79604,-0.260941,0.879103,-1.77154,-0.318649,0.837311,-1.77604,-0.48461,0.663816,-1.79841,-0.56949,0.760919,-1.81748,-0.513512,0.527859,-1.91312,-0.561477,0.505707,-1.83909,0.27205,1.21434,-2.0939,0.320298,1.14685,-2.07592,0.15494,1.13193,-2.07828,0.202876,1.06599,-2.06396,0.325906,0.981822,-2.06644,0.400339,0.805777,-2.05247,0.290019,0.789717,-2.07278,0.302808,0.722118,-2.06165,0.368726,0.490785,-2.02169,0.310429,0.399145,-1.95025,0.57302,0.498238,-1.99431,0.544436,0.429117,-2.04338,0.087431,1.34954,-1.81099,0.091762,1.3793,-2.02411},
/*737*/{0.245648,1.93421,-1.72348,0.283361,1.87249,-1.88182,0.199412,1.78046,-1.60958,0.169317,1.65587,-1.53171,0.28773,1.67315,-1.64327,0.36675,1.67397,-1.69066,0.179528,1.94653,-2.09638,0.27826,1.86921,-1.9535,0.071816,1.91857,-1.98249,-0.065238,1.84871,-2.10772,-0.102522,1.76194,-2.1686,0.031938,1.61934,-2.14187,0.058044,1.56048,-2.16841,0.173861,1.42512,-1.77975,-0.007959,1.51268,-1.85183,-0.008736,1.54379,-1.89897,-0.006565,1.52054,-1.96583,0.177855,1.46605,-2.03339,0.078624,1.1084,-1.76117,-0.055664,1.14037,-1.76437,-0.077779,1.06229,-1.7945,-0.059288,0.930856,-1.79716,-0.192752,0.790216,-1.78695,-0.258604,0.751275,-1.79659,-0.254539,0.88333,-1.77049,-0.315008,0.844945,-1.77509,-0.487768,0.678666,-1.79576,-0.56923,0.779075,-1.81632,-0.520739,0.54286,-1.91013,-0.571056,0.524158,-1.83743,0.268756,1.20889,-2.09428,0.31562,1.14077,-2.07639,0.150089,1.1281,-2.07735,0.197462,1.06237,-2.06306,0.319525,0.976455,-2.06523,0.400132,0.803396,-2.05033,0.289564,0.785408,-2.07244,0.30328,0.717607,-2.06094,0.374058,0.487711,-2.01909,0.319705,0.390881,-1.94891,0.578691,0.497823,-1.99182,0.551555,0.427943,-2.04248,0.088309,1.34906,-1.81032,0.092917,1.37815,-2.02353},
/*738*/{0.244657,1.93215,-1.72372,0.282331,1.87218,-1.8824,0.198363,1.77897,-1.60992,0.166673,1.65353,-1.53235,0.286146,1.66791,-1.64386,0.365525,1.66739,-1.69028,0.179311,1.94556,-2.09636,0.278003,1.86888,-1.95324,0.071544,1.91837,-1.98302,-0.065539,1.84597,-2.10793,-0.100736,1.75851,-2.16831,0.033313,1.61623,-2.14257,0.059562,1.55667,-2.1685,0.173783,1.42493,-1.7791,-0.008351,1.51173,-1.85276,-0.008721,1.54245,-1.89945,-0.006553,1.5186,-1.9663,0.177678,1.46418,-2.03274,0.084552,1.10598,-1.763,-0.050041,1.13462,-1.76347,-0.070935,1.05719,-1.79295,-0.050449,0.926616,-1.79824,-0.189576,0.792364,-1.78668,-0.256126,0.756421,-1.79526,-0.247044,0.888614,-1.76968,-0.308768,0.851679,-1.77398,-0.488768,0.694415,-1.79412,-0.567326,0.797757,-1.81625,-0.526514,0.558978,-1.90635,-0.578685,0.542059,-1.83365,0.265489,1.20334,-2.09453,0.310987,1.13481,-2.07698,0.14566,1.12436,-2.07622,0.191678,1.05777,-2.06221,0.313711,0.971658,-2.0649,0.39794,0.801343,-2.04901,0.28785,0.781448,-2.06972,0.302741,0.713832,-2.05891,0.378106,0.484091,-2.01866,0.324515,0.387959,-1.94742,0.583033,0.497937,-1.99214,0.555385,0.426134,-2.04064,0.088242,1.34855,-1.80934,0.093261,1.37609,-2.02274},
/*739*/{0.244291,1.93008,-1.72395,0.282714,1.87078,-1.88219,0.197199,1.77704,-1.61039,0.164508,1.65198,-1.5334,0.283934,1.66354,-1.64376,0.363531,1.66096,-1.69018,0.179771,1.94462,-2.09628,0.277326,1.86807,-1.9534,0.071173,1.91744,-1.98362,-0.06431,1.84387,-2.10788,-0.099471,1.75508,-2.16825,0.034485,1.61261,-2.1425,0.061428,1.55274,-2.16859,0.174202,1.42408,-1.77914,-0.009057,1.50992,-1.85319,-0.008202,1.54079,-1.89921,-0.006843,1.51638,-1.96636,0.177622,1.46307,-2.03206,0.092122,1.10347,-1.76416,-0.04218,1.13113,-1.76262,-0.06239,1.05174,-1.79315,-0.041395,0.922208,-1.79876,-0.186064,0.795843,-1.78548,-0.253867,0.762181,-1.79449,-0.240324,0.894379,-1.7698,-0.302152,0.86017,-1.77345,-0.489588,0.709811,-1.79034,-0.56308,0.816603,-1.81729,-0.530964,0.574462,-1.90153,-0.58556,0.561375,-1.82929,0.262966,1.19786,-2.09502,0.306895,1.12823,-2.0787,0.141807,1.12162,-2.07556,0.187229,1.05442,-2.06021,0.307112,0.966284,-2.06331,0.395252,0.798455,-2.04804,0.284448,0.777425,-2.06654,0.30024,0.71038,-2.05602,0.378751,0.48098,-2.01826,0.328989,0.384476,-1.94574,0.584914,0.497128,-1.99304,0.558603,0.425318,-2.0412,0.088822,1.34726,-1.80905,0.093861,1.37453,-2.02249},
/*740*/{0.24415,1.92767,-1.7236,0.282209,1.86972,-1.88209,0.194833,1.77485,-1.61035,0.161371,1.65081,-1.5345,0.28161,1.65882,-1.64373,0.36224,1.65358,-1.68981,0.179653,1.94284,-2.09619,0.277279,1.86738,-1.95338,0.07121,1.91711,-1.98396,-0.063322,1.84061,-2.1083,-0.097501,1.75077,-2.16861,0.035876,1.60798,-2.1432,0.063301,1.54867,-2.16892,0.17442,1.4233,-1.77901,-0.008443,1.50799,-1.85251,-0.007947,1.53878,-1.89895,-0.006927,1.51452,-1.96737,0.17829,1.46077,-2.03236,0.098567,1.10077,-1.76463,-0.037022,1.12599,-1.76193,-0.055979,1.04656,-1.7918,-0.033298,0.918015,-1.79942,-0.181912,0.799255,-1.78399,-0.251896,0.768121,-1.791,-0.232529,0.899545,-1.77002,-0.296177,0.868149,-1.77329,-0.489985,0.724721,-1.78729,-0.558705,0.834394,-1.81896,-0.534403,0.588921,-1.89571,-0.59161,0.580885,-1.82592,0.259521,1.19269,-2.09629,0.303494,1.12283,-2.07941,0.138259,1.11844,-2.07399,0.182519,1.05077,-2.05892,0.301672,0.962025,-2.06217,0.390321,0.795301,-2.0477,0.279492,0.77318,-2.06324,0.296276,0.706262,-2.05268,0.380088,0.478901,-2.01639,0.330854,0.38179,-1.94264,0.585613,0.495735,-1.99305,0.558802,0.424531,-2.04178,0.089512,1.34596,-1.80847,0.094316,1.37229,-2.02204},
/*741*/{0.243788,1.92538,-1.72388,0.280898,1.86757,-1.88201,0.193506,1.77282,-1.61069,0.15757,1.64759,-1.53505,0.279891,1.65261,-1.64352,0.360879,1.64645,-1.6894,0.178986,1.9412,-2.09589,0.275854,1.86507,-1.9535,0.070745,1.91489,-1.98478,-0.063286,1.83633,-2.10885,-0.095776,1.746,-2.16928,0.037825,1.60353,-2.14438,0.065542,1.54389,-2.16939,0.174835,1.42131,-1.77844,-0.008592,1.50528,-1.8528,-0.008535,1.53621,-1.89958,-0.00712,1.51141,-1.968,0.178362,1.45811,-2.03268,0.103516,1.09855,-1.76556,-0.031567,1.12063,-1.76147,-0.048524,1.0402,-1.79042,-0.025217,0.913961,-1.79954,-0.177514,0.802814,-1.78224,-0.248371,0.774197,-1.78918,-0.223857,0.904952,-1.77078,-0.288392,0.875384,-1.77335,-0.488974,0.739936,-1.78469,-0.551261,0.851846,-1.82017,-0.536676,0.603454,-1.88934,-0.596268,0.599965,-1.82125,0.257736,1.18771,-2.09684,0.300529,1.11645,-2.08023,0.135291,1.11558,-2.07228,0.178522,1.04778,-2.0577,0.296148,0.957422,-2.06085,0.385023,0.7921,-2.04765,0.274316,0.768941,-2.05978,0.292461,0.702236,-2.04879,0.378918,0.475903,-2.01437,0.329568,0.374499,-1.94422,0.584959,0.492996,-1.9927,0.557686,0.423011,-2.0424,0.089869,1.34381,-1.80818,0.094702,1.36934,-2.02184},
/*742*/{0.24249,1.92206,-1.72315,0.281193,1.86576,-1.88188,0.190574,1.76993,-1.61078,0.155785,1.6463,-1.53663,0.277503,1.64772,-1.64348,0.358543,1.63847,-1.6881,0.179459,1.93915,-2.09586,0.275823,1.86409,-1.95344,0.07007,1.9136,-1.98598,-0.062855,1.83272,-2.10991,-0.094098,1.74048,-2.16977,0.039868,1.59832,-2.14506,0.06743,1.53919,-2.16997,0.176367,1.42009,-1.77999,-0.008765,1.50285,-1.85353,-0.007785,1.53374,-1.90037,-0.006418,1.50851,-1.96899,0.17808,1.45502,-2.03354,0.110093,1.09638,-1.76564,-0.025769,1.11542,-1.7611,-0.042345,1.03456,-1.79134,-0.016639,0.910446,-1.79945,-0.172473,0.806399,-1.78094,-0.243453,0.780498,-1.78637,-0.215071,0.91089,-1.77173,-0.281399,0.88396,-1.77219,-0.486605,0.755761,-1.78254,-0.54364,0.868924,-1.82318,-0.538104,0.617704,-1.88331,-0.598462,0.618912,-1.81621,0.255794,1.18311,-2.09807,0.297505,1.11138,-2.08115,0.132655,1.11347,-2.07151,0.174856,1.04497,-2.05649,0.291886,0.953573,-2.06006,0.379334,0.788378,-2.04821,0.269869,0.76304,-2.05218,0.287894,0.698063,-2.04596,0.375089,0.47204,-2.01159,0.328724,0.372236,-1.93771,0.581764,0.490132,-1.9931,0.552933,0.420318,-2.04301,0.09114,1.34207,-1.80837,0.094955,1.36625,-2.02221},
/*743*/{0.242475,1.91989,-1.72269,0.281059,1.86416,-1.88193,0.1892,1.76727,-1.61038,0.150658,1.64352,-1.53847,0.274766,1.64135,-1.64313,0.356281,1.63021,-1.68736,0.178484,1.93632,-2.09574,0.275286,1.86153,-1.95324,0.070041,1.91106,-1.98623,-0.061304,1.82712,-2.11117,-0.092428,1.73499,-2.17059,0.04193,1.59313,-2.14595,0.069889,1.53348,-2.17096,0.1766,1.41795,-1.77998,-0.008976,1.49967,-1.85372,-0.00817,1.53053,-1.90011,-0.007167,1.50614,-1.97024,0.177304,1.45168,-2.03447,0.11609,1.0932,-1.76528,-0.020339,1.10995,-1.76047,-0.035096,1.02927,-1.7907,-0.008278,0.906828,-1.79869,-0.166949,0.810299,-1.7787,-0.238554,0.785346,-1.78478,-0.205737,0.915283,-1.77312,-0.272702,0.890887,-1.77329,-0.481233,0.770714,-1.78123,-0.534901,0.884143,-1.82517,-0.537375,0.631982,-1.87748,-0.597928,0.637625,-1.81108,0.254254,1.17873,-2.09893,0.294648,1.10692,-2.08238,0.130506,1.11133,-2.07072,0.172314,1.04197,-2.05458,0.288505,0.94973,-2.05869,0.374133,0.784008,-2.04717,0.264394,0.759035,-2.05107,0.285122,0.694038,-2.04246,0.369527,0.467717,-2.00917,0.323577,0.36963,-1.93374,0.57663,0.485755,-1.99337,0.547459,0.41689,-2.04324,0.091226,1.33958,-1.80839,0.094387,1.36306,-2.02232},
/*744*/{0.241187,1.91693,-1.72219,0.281192,1.86153,-1.88201,0.18616,1.76404,-1.61031,0.147667,1.63975,-1.53877,0.271923,1.63499,-1.64264,0.353214,1.62124,-1.68596,0.177292,1.93322,-2.09578,0.274731,1.86019,-1.95354,0.068905,1.90809,-1.98687,-0.061387,1.82158,-2.1124,-0.090963,1.72875,-2.17165,0.044526,1.58754,-2.14761,0.072574,1.52807,-2.1716,0.177131,1.41595,-1.78038,-0.008679,1.4966,-1.85476,-0.007583,1.52731,-1.9003,-0.006602,1.50245,-1.97101,0.177189,1.44908,-2.03533,0.120601,1.0895,-1.76495,-0.014188,1.10461,-1.76059,-0.027967,1.02429,-1.79023,-0.000616,0.903256,-1.79784,-0.161605,0.813367,-1.77689,-0.233391,0.791458,-1.78229,-0.196789,0.919761,-1.77379,-0.264057,0.898219,-1.77288,-0.477049,0.785507,-1.77793,-0.525496,0.899002,-1.82682,-0.536104,0.644529,-1.87151,-0.597311,0.65422,-1.8065,0.251849,1.17469,-2.09971,0.291916,1.10221,-2.08251,0.128038,1.10901,-2.0688,0.169105,1.03934,-2.05394,0.285694,0.946621,-2.058,0.370002,0.779493,-2.04627,0.259254,0.754479,-2.04906,0.277294,0.689098,-2.04164,0.364312,0.464284,-2.00683,0.31819,0.365442,-1.93086,0.570753,0.480508,-1.9923,0.540191,0.411141,-2.04162,0.092306,1.33704,-1.80869,0.095121,1.35995,-2.02268},
/*745*/{0.239731,1.91376,-1.72147,0.280471,1.85809,-1.88133,0.183968,1.76069,-1.61024,0.144167,1.63749,-1.5401,0.269134,1.62806,-1.64284,0.351204,1.61236,-1.68466,0.176876,1.93,-2.09591,0.2749,1.85788,-1.95303,0.068199,1.90466,-1.98841,-0.060252,1.81558,-2.11421,-0.089453,1.72169,-2.17266,0.046692,1.58175,-2.14911,0.075055,1.52237,-2.17287,0.177658,1.41335,-1.781,-0.007808,1.49306,-1.85472,-0.008104,1.52377,-1.90042,-0.007116,1.49967,-1.971,0.176301,1.44623,-2.03573,0.126025,1.08707,-1.76432,-0.010763,1.1002,-1.76096,-0.022679,1.01771,-1.78926,0.006889,0.899532,-1.79713,-0.154493,0.816348,-1.77493,-0.227407,0.796783,-1.78099,-0.186888,0.924161,-1.77516,-0.255277,0.904805,-1.77446,-0.469711,0.79945,-1.77652,-0.516087,0.912118,-1.82835,-0.532204,0.657752,-1.86529,-0.593703,0.670279,-1.80088,0.250827,1.1708,-2.10089,0.289987,1.09829,-2.08343,0.125624,1.10676,-2.06776,0.165671,1.03659,-2.05287,0.283693,0.943141,-2.05669,0.364305,0.773642,-2.04544,0.254461,0.75011,-2.04809,0.271682,0.684523,-2.0401,0.354064,0.458835,-2.00339,0.310535,0.359839,-1.92795,0.56419,0.472287,-1.99085,0.532084,0.404427,-2.03991,0.093032,1.3341,-1.80897,0.094822,1.35692,-2.02298},
/*746*/{0.238399,1.91023,-1.72073,0.280919,1.855,-1.88096,0.181246,1.75726,-1.61002,0.139853,1.63494,-1.54141,0.265787,1.62073,-1.64234,0.347554,1.60283,-1.68319,0.175564,1.92646,-2.09618,0.274444,1.85484,-1.95233,0.065894,1.90148,-1.98894,-0.061239,1.80883,-2.11517,-0.087914,1.71462,-2.17394,0.049466,1.57533,-2.15046,0.078847,1.5164,-2.17335,0.17855,1.41058,-1.78203,-0.007686,1.48956,-1.85524,-0.008311,1.52033,-1.9009,-0.007468,1.49503,-1.97075,0.175575,1.44332,-2.03623,0.129647,1.08476,-1.76326,-0.006248,1.0931,-1.76126,-0.016564,1.01256,-1.79107,0.014658,0.896353,-1.79579,-0.147675,0.819765,-1.77294,-0.221594,0.802204,-1.77743,-0.17711,0.928476,-1.77657,-0.245177,0.909569,-1.77571,-0.46255,0.812459,-1.77503,-0.506331,0.924864,-1.83038,-0.528201,0.669304,-1.8606,-0.58901,0.685253,-1.79563,0.248981,1.16746,-2.10082,0.287249,1.09339,-2.08406,0.123269,1.10422,-2.06754,0.163808,1.03324,-2.05193,0.281756,0.939632,-2.05605,0.359501,0.768072,-2.0445,0.248148,0.747169,-2.05089,0.265936,0.680447,-2.03853,0.347537,0.453606,-2.00147,0.299837,0.357058,-1.92359,0.555233,0.464087,-1.98991,0.522186,0.394909,-2.03708,0.094161,1.33101,-1.80923,0.094874,1.35336,-2.0233},
/*747*/{0.237085,1.90589,-1.7199,0.280648,1.85231,-1.88046,0.178044,1.75318,-1.60953,0.135214,1.63173,-1.54269,0.261696,1.61406,-1.64169,0.344352,1.59337,-1.68152,0.174961,1.92214,-2.09641,0.27427,1.85163,-1.95249,0.065264,1.8976,-1.98924,-0.061181,1.80134,-2.11824,-0.086238,1.70729,-2.17602,0.052707,1.56899,-2.15217,0.082621,1.51009,-2.17448,0.178626,1.40805,-1.78258,-0.008475,1.48609,-1.85531,-0.008127,1.51623,-1.90122,-0.008488,1.49168,-1.971,0.174744,1.44019,-2.03667,0.134186,1.08234,-1.7638,-0.000914,1.08742,-1.76188,-0.009337,1.00773,-1.79015,0.022894,0.893403,-1.79416,-0.139501,0.822964,-1.77113,-0.213614,0.806676,-1.77607,-0.165847,0.931964,-1.77793,-0.234931,0.915077,-1.77599,-0.453841,0.822261,-1.773,-0.495942,0.936131,-1.83174,-0.521488,0.680138,-1.85536,-0.582489,0.698953,-1.79071,0.247764,1.16333,-2.10145,0.28599,1.0894,-2.08405,0.121498,1.10101,-2.06721,0.160573,1.03002,-2.05196,0.28044,0.936209,-2.05457,0.354849,0.76258,-2.04351,0.243338,0.742702,-2.0501,0.259735,0.675182,-2.03759,0.337972,0.447945,-1.99907,0.288595,0.354808,-1.92159,0.545467,0.454834,-1.9866,0.512511,0.385891,-2.03353,0.094477,1.32798,-1.80938,0.09455,1.34995,-2.02349},
/*748*/{0.236127,1.90236,-1.71885,0.280567,1.84898,-1.8792,0.175911,1.74891,-1.60947,0.130711,1.62802,-1.54314,0.258219,1.6064,-1.64061,0.340296,1.58397,-1.67987,0.174318,1.91803,-2.09723,0.273872,1.84831,-1.95237,0.06415,1.89237,-1.98988,-0.061095,1.79378,-2.1196,-0.084529,1.69866,-2.1772,0.056868,1.56191,-2.15349,0.086042,1.50353,-2.17581,0.179036,1.40523,-1.78299,-0.008677,1.4817,-1.85476,-0.008135,1.51176,-1.9012,-0.008947,1.48612,-1.96992,0.174802,1.43772,-2.03749,0.139524,1.07788,-1.7602,0.00229,1.08307,-1.76326,-0.002602,1.00196,-1.79064,0.031232,0.889975,-1.7918,-0.131616,0.825186,-1.76837,-0.205815,0.811979,-1.77217,-0.154744,0.934608,-1.7791,-0.225845,0.920439,-1.77809,-0.445727,0.831968,-1.77184,-0.485283,0.946459,-1.83338,-0.515005,0.690219,-1.85074,-0.575022,0.711243,-1.78584,0.245164,1.15928,-2.10142,0.284512,1.08398,-2.0828,0.119738,1.09819,-2.06722,0.15862,1.02696,-2.05184,0.278215,0.932207,-2.05429,0.350151,0.756128,-2.04168,0.237148,0.73787,-2.04906,0.251797,0.669776,-2.03701,0.327255,0.440949,-1.9967,0.276136,0.3492,-1.91819,0.536169,0.444078,-1.98263,0.500869,0.375342,-2.02944,0.096248,1.32431,-1.80945,0.095371,1.34636,-2.02355},
/*749*/{0.234138,1.89754,-1.7176,0.279557,1.84428,-1.87884,0.173017,1.74424,-1.60954,0.126846,1.62435,-1.54377,0.254652,1.59892,-1.64027,0.336615,1.57372,-1.67772,0.174036,1.91352,-2.09766,0.273408,1.84498,-1.9528,0.06247,1.88777,-1.99089,-0.059688,1.78551,-2.1212,-0.082481,1.69053,-2.1791,0.060105,1.55463,-2.15516,0.090565,1.49651,-2.1769,0.180763,1.402,-1.78388,-0.009097,1.47729,-1.85499,-0.008212,1.5071,-1.90102,-0.009819,1.48199,-1.96984,0.174538,1.43489,-2.03815,0.141825,1.07587,-1.75775,0.007032,1.07768,-1.7633,0.002677,0.9962,-1.79132,0.038935,0.886483,-1.78972,-0.12238,0.826656,-1.76711,-0.19688,0.814569,-1.77066,-0.142978,0.936784,-1.78079,-0.213795,0.924679,-1.77933,-0.434669,0.844624,-1.77122,-0.475445,0.955591,-1.83396,-0.507845,0.698482,-1.84647,-0.566871,0.722554,-1.78049,0.244005,1.15517,-2.1015,0.281719,1.08016,-2.08314,0.117533,1.09343,-2.06659,0.155949,1.02243,-2.0518,0.274624,0.926924,-2.05381,0.343411,0.749506,-2.04048,0.230234,0.73303,-2.04862,0.245335,0.665002,-2.03605,0.315534,0.437128,-1.99496,0.26081,0.350249,-1.91517,0.523219,0.434449,-1.97968,0.487706,0.365504,-2.02523,0.097933,1.32058,-1.81011,0.095835,1.34293,-2.02418},
/*750*/{0.233036,1.8939,-1.71721,0.280186,1.83961,-1.8777,0.169878,1.73946,-1.60925,0.121586,1.62037,-1.54356,0.250298,1.59104,-1.63936,0.331917,1.56366,-1.676,0.172584,1.90854,-2.09785,0.27385,1.84124,-1.95185,0.061881,1.88236,-1.99264,-0.06073,1.7769,-2.12362,-0.080522,1.68196,-2.18098,0.063243,1.54711,-2.15647,0.094792,1.48975,-2.17782,0.180594,1.39863,-1.78469,-0.008272,1.47254,-1.85368,-0.007536,1.5027,-1.90019,-0.010977,1.47695,-1.96838,0.174406,1.43186,-2.03907,0.147653,1.07284,-1.75534,0.011453,1.07362,-1.76595,0.009279,0.991465,-1.79205,0.047267,0.88349,-1.78685,-0.112866,0.829066,-1.76345,-0.188349,0.819305,-1.76723,-0.130895,0.938504,-1.78166,-0.201103,0.927994,-1.78037,-0.425782,0.852611,-1.76939,-0.464382,0.9637,-1.83486,-0.499239,0.707191,-1.84131,-0.556504,0.732338,-1.77557,0.242354,1.15089,-2.10133,0.280328,1.07637,-2.08274,0.115532,1.08926,-2.06616,0.154398,1.01766,-2.05129,0.272118,0.921383,-2.05339,0.337754,0.742622,-2.03992,0.223514,0.727903,-2.0486,0.236918,0.659794,-2.03569,0.304141,0.433506,-1.99284,0.239785,0.355138,-1.91698,0.509541,0.422137,-1.97387,0.469887,0.35506,-2.02092,0.099416,1.31644,-1.81025,0.095815,1.33924,-2.02425},
/*751*/{0.23128,1.88865,-1.71579,0.280092,1.83565,-1.87715,0.166788,1.73465,-1.60923,0.115933,1.61432,-1.54343,0.246358,1.58363,-1.63847,0.327203,1.55391,-1.67382,0.172267,1.90359,-2.09832,0.27404,1.83768,-1.95127,0.060154,1.87613,-1.99307,-0.059988,1.76726,-2.12695,-0.078177,1.67297,-2.18273,0.06768,1.53937,-2.15822,0.099767,1.48212,-2.17861,0.180914,1.39522,-1.78572,-0.008747,1.46722,-1.85329,-0.00964,1.49803,-1.90074,-0.010657,1.47185,-1.96806,0.174361,1.42865,-2.03976,0.151986,1.06869,-1.75195,0.015354,1.06681,-1.76577,0.015387,0.986235,-1.79187,0.05656,0.880445,-1.7847,-0.103657,0.830124,-1.76152,-0.178459,0.821419,-1.76611,-0.118451,0.940251,-1.78303,-0.189939,0.930898,-1.78034,-0.414067,0.861007,-1.76942,-0.453518,0.970762,-1.83603,-0.489539,0.714283,-1.83734,-0.547123,0.74137,-1.77187,0.240559,1.14688,-2.10144,0.278797,1.072,-2.08295,0.113862,1.08504,-2.06651,0.152317,1.01353,-2.05138,0.268333,0.914896,-2.05345,0.329762,0.735263,-2.03897,0.21643,0.722787,-2.04827,0.227885,0.654552,-2.03535,0.29133,0.428506,-1.99217,0.220627,0.354542,-1.91365,0.494541,0.405426,-1.96975,0.451718,0.343531,-2.01755,0.100562,1.31226,-1.81088,0.096404,1.33543,-2.02483},
/*752*/{0.229535,1.88419,-1.71509,0.279444,1.83057,-1.87573,0.163817,1.72946,-1.60937,0.112411,1.61166,-1.54366,0.242369,1.57595,-1.63702,0.322533,1.54402,-1.67163,0.172651,1.89837,-2.09904,0.273893,1.83352,-1.95148,0.059447,1.87034,-1.99388,-0.059833,1.75756,-2.12939,-0.075332,1.66391,-2.18529,0.073042,1.53226,-2.15934,0.105196,1.47452,-2.1796,0.181808,1.39123,-1.78695,-0.008284,1.46166,-1.85292,-0.008359,1.49249,-1.90015,-0.011991,1.4668,-1.96697,0.174047,1.42561,-2.04075,0.155195,1.06528,-1.74927,0.02032,1.06183,-1.76721,0.020825,0.979414,-1.79372,0.065188,0.876921,-1.78196,-0.09266,0.830782,-1.75919,-0.168639,0.823619,-1.76377,-0.105572,0.940225,-1.78408,-0.177139,0.932324,-1.78146,-0.402653,0.867234,-1.7685,-0.441625,0.976758,-1.83726,-0.479327,0.719879,-1.8336,-0.535375,0.748837,-1.76664,0.238905,1.14209,-2.10117,0.277554,1.06769,-2.08334,0.111875,1.0808,-2.06718,0.150749,1.00926,-2.05178,0.264346,0.908358,-2.05339,0.32228,0.728097,-2.03766,0.209682,0.717633,-2.04779,0.219717,0.649537,-2.035,0.27347,0.422383,-1.98776,0.199756,0.355015,-1.9139,0.480598,0.393589,-1.96074,0.432018,0.340907,-2.01522,0.102604,1.30747,-1.81181,0.09677,1.33158,-2.02561},
/*753*/{0.227715,1.87916,-1.71403,0.280056,1.82547,-1.87542,0.161066,1.72467,-1.60947,0.106982,1.60813,-1.54492,0.237364,1.56834,-1.63598,0.317731,1.53564,-1.67068,0.173425,1.89315,-2.10028,0.273898,1.82849,-1.95024,0.05885,1.86379,-1.99531,-0.05846,1.74746,-2.13087,-0.071967,1.65428,-2.18731,0.076811,1.52404,-2.16057,0.110421,1.46762,-2.18063,0.181663,1.3863,-1.78726,-0.00881,1.45582,-1.8515,-0.010105,1.48713,-1.89967,-0.013132,1.46179,-1.96613,0.17352,1.42201,-2.04224,0.157223,1.06231,-1.74783,0.023434,1.05562,-1.76886,0.026958,0.973751,-1.79284,0.073845,0.872757,-1.77977,-0.082085,0.830959,-1.75704,-0.157218,0.823931,-1.76114,-0.092579,0.939814,-1.78492,-0.164187,0.933584,-1.78138,-0.390409,0.872994,-1.76703,-0.429821,0.981291,-1.83849,-0.467993,0.725188,-1.82959,-0.522692,0.755342,-1.7633,0.236443,1.13809,-2.10209,0.274936,1.06382,-2.08474,0.109907,1.07685,-2.06743,0.148774,1.0056,-2.05203,0.260066,0.90248,-2.05359,0.314583,0.720899,-2.03503,0.202297,0.71352,-2.04747,0.210393,0.644737,-2.03341,0.256246,0.420933,-1.98122,0.177075,0.35287,-1.9134,0.459491,0.388159,-1.95322,0.409865,0.343004,-2.01497,0.103131,1.30219,-1.81263,0.096182,1.32753,-2.02625},
/*754*/{0.226262,1.87449,-1.71296,0.280959,1.82054,-1.87379,0.158213,1.72002,-1.60889,0.103445,1.60381,-1.54416,0.232997,1.56097,-1.63466,0.311379,1.52679,-1.66756,0.17321,1.88806,-2.1007,0.274652,1.82366,-1.95018,0.057164,1.85778,-1.99577,-0.056461,1.73783,-2.13402,-0.068731,1.64439,-2.1894,0.082571,1.51624,-2.16227,0.116782,1.46025,-2.18157,0.181087,1.38203,-1.7883,-0.010318,1.45115,-1.85114,-0.011253,1.48155,-1.89905,-0.014743,1.45715,-1.96528,0.171919,1.41776,-2.0436,0.161147,1.05945,-1.74505,0.027193,1.05082,-1.76975,0.033226,0.967057,-1.79283,0.084466,0.870906,-1.77641,-0.070579,0.830586,-1.75481,-0.146479,0.825931,-1.75863,-0.079952,0.938743,-1.78517,-0.150676,0.934189,-1.78214,-0.376719,0.878307,-1.76685,-0.417281,0.98524,-1.83953,-0.455472,0.729453,-1.82506,-0.510608,0.76098,-1.75842,0.233553,1.1344,-2.10405,0.271903,1.05956,-2.08704,0.107545,1.07294,-2.06762,0.146285,1.00206,-2.05269,0.255711,0.89679,-2.05404,0.307805,0.713839,-2.03031,0.195408,0.709767,-2.04656,0.201269,0.641241,-2.03116,0.238171,0.422846,-1.97582,0.15434,0.353714,-1.91118,0.433171,0.388052,-1.95493,0.385361,0.343499,-2.01472,0.102937,1.29752,-1.81316,0.094569,1.32311,-2.02671},
/*755*/{0.225661,1.86987,-1.71168,0.280935,1.81482,-1.87276,0.155003,1.71507,-1.60925,0.098302,1.60103,-1.54555,0.22743,1.55377,-1.63374,0.305729,1.51906,-1.66586,0.17427,1.88273,-2.10151,0.274493,1.81827,-1.94958,0.056665,1.85116,-1.99693,-0.054377,1.72801,-2.13762,-0.064066,1.63488,-2.1919,0.088436,1.50949,-2.16403,0.123715,1.45273,-2.18265,0.182051,1.37706,-1.79047,-0.012575,1.44575,-1.84997,-0.012501,1.4765,-1.89839,-0.01711,1.45295,-1.96537,0.170621,1.413,-2.04518,0.164612,1.05475,-1.74207,0.032174,1.04221,-1.77032,0.039859,0.960954,-1.7934,0.095146,0.86697,-1.77292,-0.058886,0.829874,-1.75226,-0.135399,0.825204,-1.75783,-0.064499,0.937833,-1.78569,-0.137413,0.934497,-1.7824,-0.36527,0.881518,-1.76644,-0.404995,0.988088,-1.84052,-0.44263,0.732456,-1.82064,-0.4975,0.76545,-1.75667,0.229707,1.13097,-2.10786,0.268054,1.05659,-2.09047,0.103822,1.07225,-2.06851,0.142466,1.00103,-2.05211,0.250031,0.891052,-2.05444,0.302179,0.709481,-2.02549,0.190027,0.708794,-2.04451,0.191654,0.64067,-2.02724,0.213754,0.423903,-1.97209,0.130828,0.354613,-1.90983,0.410441,0.385396,-1.94833,0.362829,0.343148,-2.0132,0.102936,1.29244,-1.81439,0.092515,1.31869,-2.02777},
/*756*/{0.223703,1.86528,-1.71111,0.28122,1.81025,-1.87192,0.15118,1.71037,-1.6092,0.09282,1.59779,-1.54615,0.221786,1.54801,-1.63178,0.299718,1.51173,-1.66329,0.175244,1.87784,-2.10268,0.276215,1.81332,-1.94922,0.056467,1.84699,-1.99753,-0.051941,1.71799,-2.14154,-0.05923,1.62438,-2.19405,0.094561,1.50163,-2.16536,0.130456,1.44562,-2.18271,0.181609,1.37197,-1.79146,-0.012832,1.44083,-1.84953,-0.013753,1.47009,-1.89718,-0.020455,1.4499,-1.96477,0.167486,1.40845,-2.04665,0.170372,1.05221,-1.73908,0.03811,1.03765,-1.77066,0.04923,0.953496,-1.79297,0.105612,0.863702,-1.76963,-0.04615,0.827889,-1.75058,-0.121797,0.825295,-1.75476,-0.051069,0.935792,-1.78382,-0.122669,0.933836,-1.78006,-0.351447,0.884239,-1.76685,-0.392127,0.989845,-1.84184,-0.429013,0.734397,-1.81723,-0.483631,0.768576,-1.75329,0.226666,1.12886,-2.11192,0.264002,1.054,-2.09441,0.101084,1.07185,-2.06784,0.139518,1.00033,-2.05199,0.244016,0.888098,-2.05451,0.295272,0.706284,-2.02058,0.183156,0.711766,-2.04108,0.181437,0.644273,-2.02332,0.192705,0.424411,-1.96832,0.107572,0.354763,-1.90759,0.38907,0.383076,-1.94966,0.339556,0.343856,-2.01177,0.102264,1.28726,-1.81542,0.089484,1.31432,-2.02857},
/*757*/{0.222752,1.86085,-1.70985,0.282619,1.80486,-1.87078,0.148961,1.7058,-1.60937,0.086435,1.59509,-1.54607,0.216349,1.54305,-1.63101,0.292505,1.50517,-1.66098,0.175733,1.87317,-2.10422,0.276321,1.80791,-1.94793,0.056673,1.84416,-1.99707,-0.046712,1.70785,-2.14609,-0.053131,1.61417,-2.196,0.101563,1.49384,-2.16651,0.138141,1.43929,-2.18356,0.180438,1.36718,-1.79383,-0.01429,1.43578,-1.84907,-0.016099,1.46515,-1.89605,-0.023052,1.44585,-1.96402,0.165369,1.40682,-2.04807,0.174629,1.04899,-1.7371,0.042079,1.03181,-1.77187,0.055755,0.948371,-1.79336,0.117769,0.860807,-1.76711,-0.033324,0.826559,-1.74865,-0.109179,0.825114,-1.75285,-0.036941,0.934158,-1.78216,-0.109244,0.932513,-1.77904,-0.338018,0.886445,-1.76516,-0.379029,0.990399,-1.84328,-0.414418,0.734551,-1.81304,-0.468948,0.769355,-1.74975,0.222879,1.12639,-2.1154,0.261229,1.05196,-2.09645,0.098592,1.07109,-2.06735,0.137237,0.999532,-2.05111,0.23936,0.886814,-2.05481,0.286181,0.705921,-2.01863,0.174826,0.716238,-2.03803,0.169665,0.648347,-2.02148,0.175832,0.427258,-1.9665,0.084377,0.356271,-1.90583,0.367338,0.383149,-1.94546,0.317411,0.343925,-2.01037,0.101896,1.28219,-1.81791,0.087476,1.31203,-2.03058},
/*758*/{0.222055,1.8568,-1.70838,0.282247,1.80069,-1.86971,0.146247,1.7023,-1.60914,0.07947,1.5927,-1.54634,0.209792,1.53857,-1.62873,0.286148,1.4999,-1.65885,0.176945,1.8688,-2.10485,0.277279,1.80376,-1.94759,0.057117,1.84281,-1.99671,-0.040239,1.69806,-2.15033,-0.046472,1.60434,-2.1988,0.109149,1.48723,-2.1669,0.146212,1.43293,-2.18355,0.180017,1.36292,-1.79686,-0.017518,1.43218,-1.84812,-0.019074,1.46067,-1.89544,-0.0265,1.44276,-1.96386,0.163557,1.40493,-2.04945,0.17828,1.04724,-1.73525,0.047914,1.02516,-1.77157,0.064347,0.942927,-1.79379,0.130013,0.857821,-1.7628,-0.019705,0.824209,-1.74689,-0.095676,0.823591,-1.75345,-0.023178,0.932093,-1.78055,-0.094847,0.931283,-1.77794,-0.32326,0.886953,-1.76395,-0.366483,0.989343,-1.84341,-0.398875,0.734405,-1.81018,-0.453126,0.769892,-1.74682,0.219684,1.12511,-2.11638,0.25804,1.05099,-2.09687,0.095788,1.06953,-2.067,0.134597,0.99841,-2.0503,0.237557,0.887232,-2.05432,0.276007,0.704706,-2.01852,0.164602,0.720461,-2.03728,0.156294,0.652707,-2.02108,0.151078,0.425137,-1.96259,0.064081,0.356818,-1.90435,0.3453,0.380583,-1.94664,0.294739,0.343789,-2.00968,0.100924,1.27802,-1.82023,0.08479,1.30996,-2.03247},
/*759*/{0.222119,1.85314,-1.70697,0.283606,1.79777,-1.86862,0.142105,1.69961,-1.60921,0.072107,1.59186,-1.54674,0.203302,1.5352,-1.6278,0.279439,1.49477,-1.65601,0.177688,1.86501,-2.10595,0.278779,1.79963,-1.94604,0.057435,1.84066,-1.99605,-0.034677,1.6881,-2.15641,-0.038443,1.59444,-2.20091,0.117564,1.48155,-2.16723,0.15539,1.42804,-2.18367,0.178302,1.35861,-1.79787,-0.019088,1.42751,-1.84711,-0.021503,1.45845,-1.89603,-0.02918,1.43945,-1.96209,0.161001,1.40269,-2.05107,0.182951,1.04769,-1.73419,0.052829,1.01998,-1.77298,0.073403,0.937839,-1.79276,0.142662,0.854612,-1.76154,-0.006441,0.821708,-1.7458,-0.082329,0.820797,-1.75154,-0.008598,0.930592,-1.7787,-0.079778,0.929897,-1.77533,-0.312006,0.885196,-1.76737,-0.354763,0.987334,-1.84447,-0.382231,0.732435,-1.80654,-0.436755,0.768354,-1.74374,0.214759,1.12379,-2.11596,0.254412,1.0499,-2.09725,0.09188,1.06733,-2.06589,0.131285,0.996382,-2.04958,0.238075,0.887417,-2.05391,0.264076,0.703739,-2.02081,0.153109,0.723888,-2.03646,0.142855,0.654759,-2.02048,0.135166,0.42796,-1.96302,0.042234,0.358365,-1.90293,0.325008,0.380738,-1.9448,0.272875,0.343534,-2.00888,0.099261,1.27393,-1.82176,0.081835,1.30759,-2.03363},
/*760*/{0.222891,1.84946,-1.70559,0.282261,1.79426,-1.86693,0.138025,1.69813,-1.60915,0.065113,1.5913,-1.54707,0.196563,1.53273,-1.62632,0.271519,1.49198,-1.6535,0.179402,1.86205,-2.1062,0.279005,1.79624,-1.94564,0.057677,1.83909,-1.99489,-0.030267,1.67969,-2.16028,-0.030239,1.58528,-2.2025,0.126484,1.4764,-2.16767,0.165287,1.42278,-2.1832,0.177081,1.35482,-1.80008,-0.021507,1.42361,-1.84529,-0.021612,1.45678,-1.89445,-0.03129,1.43693,-1.95941,0.158607,1.4021,-2.05256,0.189056,1.04996,-1.73349,0.059972,1.01639,-1.7715,0.082655,0.935822,-1.79131,0.156155,0.852135,-1.75919,0.006955,0.819731,-1.74523,-0.068796,0.818322,-1.75169,0.006137,0.92955,-1.77531,-0.066883,0.928576,-1.77478,-0.294397,0.883021,-1.7629,-0.343413,0.984579,-1.84489,-0.365277,0.729206,-1.8038,-0.420908,0.764693,-1.7407,0.210374,1.12224,-2.11608,0.24974,1.04787,-2.09795,0.087808,1.06586,-2.0647,0.12729,0.995254,-2.0489,0.238656,0.887752,-2.0524,0.251713,0.70124,-2.02335,0.141389,0.725366,-2.03628,0.129066,0.657136,-2.02097,0.115816,0.428594,-1.96294,0.022347,0.357354,-1.90302,0.303001,0.379153,-1.94535,0.251151,0.343481,-2.00892,0.098392,1.27015,-1.8238,0.078875,1.30685,-2.03498},
/*761*/{0.222792,1.84575,-1.70341,0.282855,1.78967,-1.86613,0.133247,1.6974,-1.6089,0.05782,1.59203,-1.54843,0.189394,1.53178,-1.62491,0.264713,1.4884,-1.65119,0.180547,1.85943,-2.10529,0.279029,1.79323,-1.94401,0.058079,1.83753,-1.99497,-0.024731,1.67296,-2.16382,-0.020503,1.57717,-2.20483,0.136762,1.47231,-2.16693,0.176757,1.42049,-2.18221,0.174069,1.35191,-1.8006,-0.022877,1.42092,-1.84393,-0.023656,1.45646,-1.89382,-0.033394,1.43566,-1.95785,0.156513,1.40172,-2.05389,0.195057,1.05285,-1.73218,0.068108,1.01366,-1.76904,0.093037,0.933584,-1.78847,0.169789,0.850947,-1.75806,0.020963,0.81809,-1.74407,-0.053975,0.817842,-1.75144,0.019406,0.927811,-1.77291,-0.051375,0.92839,-1.77235,-0.284832,0.884404,-1.76473,-0.332741,0.979724,-1.8454,-0.34858,0.724835,-1.80143,-0.403891,0.759377,-1.73869,0.205914,1.1198,-2.11605,0.244333,1.04508,-2.09879,0.082921,1.0639,-2.06337,0.121767,0.993005,-2.048,0.237863,0.887564,-2.05013,0.239103,0.69926,-2.02625,0.128392,0.724748,-2.03766,0.115502,0.657099,-2.0197,0.093406,0.428365,-1.96238,-0.001835,0.360314,-1.9036,0.280407,0.378714,-1.94579,0.229229,0.343027,-2.00903,0.095977,1.26756,-1.82513,0.075887,1.30654,-2.03585},
/*762*/{0.223948,1.84357,-1.7031,0.283625,1.7879,-1.86519,0.128457,1.69741,-1.6088,0.051158,1.59426,-1.5492,0.183095,1.53121,-1.62376,0.257562,1.48646,-1.64948,0.181603,1.85736,-2.10407,0.279238,1.79068,-1.94375,0.058498,1.83634,-1.99455,-0.018483,1.6664,-2.1656,-0.011558,1.57111,-2.20578,0.146811,1.46923,-2.16629,0.188062,1.41775,-2.1812,0.172782,1.3492,-1.80286,-0.025105,1.41976,-1.84229,-0.025193,1.45593,-1.89427,-0.033841,1.43456,-1.9547,0.154936,1.4011,-2.05526,0.203252,1.05553,-1.7318,0.07654,1.01282,-1.76698,0.104381,0.932521,-1.787,0.184633,0.852322,-1.75768,0.034841,0.817621,-1.74319,-0.039847,0.816001,-1.75061,0.033055,0.92751,-1.77106,-0.037908,0.927496,-1.77082,-0.269052,0.878345,-1.76398,-0.322829,0.974163,-1.84499,-0.329754,0.718879,-1.79946,-0.38494,0.752591,-1.73619,0.200017,1.11645,-2.11637,0.23735,1.04137,-2.09893,0.077071,1.06029,-2.0614,0.115217,0.991454,-2.04715,0.23137,0.886145,-2.04959,0.226204,0.69508,-2.0271,0.115559,0.724271,-2.03852,0.101287,0.657859,-2.02087,0.077227,0.429352,-1.96323,-0.023041,0.361187,-1.90328,0.26011,0.37759,-1.94531,0.207134,0.342744,-2.00984,0.094381,1.26537,-1.82657,0.073015,1.30627,-2.0368},
/*763*/{0.224504,1.8416,-1.70238,0.283483,1.78643,-1.86453,0.125146,1.69842,-1.60877,0.044125,1.59684,-1.54954,0.17538,1.5314,-1.62369,0.251147,1.48588,-1.64662,0.183496,1.85642,-2.1029,0.278948,1.78838,-1.9423,0.05917,1.83504,-1.99428,-0.011679,1.66189,-2.16653,-0.003664,1.56665,-2.20585,0.157922,1.46767,-2.16499,0.199708,1.4175,-2.17999,0.170931,1.34628,-1.80237,-0.025723,1.41925,-1.84226,-0.026629,1.45521,-1.89333,-0.035843,1.43406,-1.9537,0.152997,1.40072,-2.05641,0.209603,1.05877,-1.73223,0.084948,1.0115,-1.76503,0.114226,0.932593,-1.78409,0.198605,0.854181,-1.7573,0.049467,0.81812,-1.7414,-0.024367,0.815727,-1.74998,0.045335,0.92765,-1.76942,-0.025717,0.927255,-1.76994,-0.253064,0.872547,-1.76262,-0.314094,0.967801,-1.84483,-0.310408,0.712578,-1.79779,-0.366478,0.744076,-1.73315,0.193723,1.11346,-2.11668,0.230356,1.03708,-2.09985,0.070997,1.05885,-2.06131,0.108254,0.986905,-2.04519,0.221071,0.882392,-2.05133,0.212205,0.690841,-2.02666,0.103629,0.72227,-2.03998,0.086481,0.656899,-2.02197,0.057323,0.430339,-1.96366,-0.045134,0.360825,-1.90499,0.238123,0.377374,-1.94595,0.185533,0.342979,-2.01046,0.092079,1.2635,-1.82745,0.070352,1.30597,-2.03733},
/*764*/{0.224891,1.84008,-1.70193,0.283047,1.7843,-1.86322,0.120291,1.7002,-1.60865,0.03768,1.60117,-1.55129,0.168538,1.53288,-1.62255,0.243791,1.48611,-1.64528,0.185039,1.85543,-2.10231,0.279699,1.78708,-1.94108,0.060037,1.83468,-1.99435,-0.006488,1.65791,-2.16632,0.004143,1.5627,-2.20657,0.168443,1.46751,-2.16397,0.212036,1.41925,-2.17808,0.169155,1.34446,-1.801,-0.027518,1.41884,-1.84198,-0.027693,1.455,-1.89354,-0.036204,1.43419,-1.95263,0.151311,1.4003,-2.05693,0.216787,1.06176,-1.732,0.09427,1.01211,-1.76273,0.125363,0.933163,-1.78232,0.213687,0.85763,-1.75743,0.063183,0.817984,-1.74213,-0.01044,0.816946,-1.75015,0.057097,0.928205,-1.76788,-0.013881,0.92486,-1.76878,-0.2399,0.867188,-1.76399,-0.305231,0.959649,-1.84403,-0.290524,0.704894,-1.79669,-0.346458,0.734807,-1.73084,0.188668,1.11044,-2.11738,0.222963,1.03313,-2.10048,0.064302,1.05867,-2.0611,0.100449,0.985065,-2.04555,0.210882,0.877018,-2.05217,0.198683,0.686514,-2.02641,0.08951,0.718795,-2.04092,0.072008,0.652742,-2.02198,0.035808,0.43082,-1.96455,-0.066491,0.361798,-1.90542,0.216758,0.377148,-1.947,0.16301,0.341826,-2.01088,0.089723,1.2624,-1.8275,0.068544,1.30573,-2.03726},
/*765*/{0.224129,1.83918,-1.70108,0.281646,1.78358,-1.86196,0.115317,1.70225,-1.60862,0.030615,1.60575,-1.55279,0.16206,1.53459,-1.62135,0.235723,1.48715,-1.6436,0.187326,1.85449,-2.10143,0.278956,1.78603,-1.93943,0.060315,1.83454,-1.99468,-0.001049,1.65537,-2.16599,0.01259,1.55966,-2.20662,0.178574,1.46802,-2.16202,0.223073,1.4209,-2.17594,0.168053,1.34314,-1.80046,-0.028253,1.42014,-1.84052,-0.029137,1.45495,-1.89315,-0.036783,1.4348,-1.95233,0.150827,1.40021,-2.05705,0.223363,1.06867,-1.7319,0.103031,1.01264,-1.76122,0.137522,0.936242,-1.78042,0.228223,0.861215,-1.75652,0.077238,0.818875,-1.74176,0.003186,0.814583,-1.74982,0.06901,0.929327,-1.76554,-0.003363,0.924563,-1.7676,-0.227309,0.86104,-1.76473,-0.297399,0.950143,-1.8443,-0.269105,0.697175,-1.79604,-0.325111,0.724225,-1.7294,0.182133,1.1075,-2.11834,0.215778,1.02918,-2.1008,0.057718,1.05694,-2.0613,0.092282,0.983757,-2.04614,0.199324,0.87246,-2.0529,0.184853,0.681678,-2.0255,0.078102,0.717686,-2.04127,0.058026,0.65265,-2.02361,0.014402,0.430654,-1.96572,-0.087642,0.362428,-1.90693,0.195425,0.377152,-1.94734,0.143507,0.342633,-2.01122,0.087938,1.26204,-1.82731,0.067009,1.30595,-2.03697},
/*766*/{0.223707,1.83961,-1.70022,0.281639,1.78297,-1.86079,0.111304,1.70506,-1.60914,0.02481,1.61124,-1.55421,0.154843,1.53736,-1.62002,0.228586,1.48871,-1.64198,0.189398,1.85429,-2.10032,0.279333,1.78577,-1.93794,0.061273,1.83487,-1.99476,0.003667,1.65339,-2.1655,0.020799,1.55714,-2.20731,0.188964,1.46919,-2.16008,0.235147,1.42493,-2.1737,0.165727,1.34268,-1.79879,-0.028301,1.42021,-1.84031,-0.029518,1.45504,-1.89264,-0.037237,1.43546,-1.95277,0.152086,1.40003,-2.05701,0.231649,1.07377,-1.73186,0.110658,1.01498,-1.76076,0.148939,0.938544,-1.77859,0.242293,0.86644,-1.75714,0.09174,0.819561,-1.74087,0.016609,0.812034,-1.75023,0.079846,0.93084,-1.76392,0.008282,0.923384,-1.76507,-0.212498,0.853019,-1.76394,-0.288924,0.940033,-1.84359,-0.24859,0.687814,-1.79597,-0.303931,0.713967,-1.72733,0.17637,1.10358,-2.11808,0.207795,1.02537,-2.10085,0.049785,1.0583,-2.06278,0.081763,0.983528,-2.04714,0.187068,0.868572,-2.05329,0.169741,0.676922,-2.02514,0.063942,0.715387,-2.0412,0.04194,0.651069,-2.0245,-0.005545,0.431841,-1.96639,-0.108077,0.362861,-1.90655,0.174552,0.376777,-1.94871,0.12212,0.342078,-2.01174,0.086309,1.26176,-1.82685,0.067111,1.30611,-2.03659},
/*767*/{0.222327,1.84038,-1.70011,0.279721,1.78256,-1.85981,0.106698,1.70833,-1.60872,0.01698,1.61566,-1.55539,0.14849,1.54029,-1.61974,0.221748,1.49128,-1.64015,0.191931,1.85517,-2.09977,0.279281,1.78566,-1.93692,0.062097,1.83464,-1.99506,0.008523,1.65122,-2.16453,0.028796,1.55583,-2.20748,0.199563,1.47408,-2.15838,0.246721,1.43021,-2.17105,0.165408,1.34356,-1.79778,-0.029445,1.42069,-1.84169,-0.029343,1.45525,-1.89336,-0.037484,1.43609,-1.95317,0.151529,1.3992,-2.05733,0.238565,1.07905,-1.73089,0.119868,1.0155,-1.75875,0.159775,0.942338,-1.77708,0.255903,0.872442,-1.75628,0.105562,0.821273,-1.74237,0.032052,0.812524,-1.75048,0.090495,0.931844,-1.7625,0.019565,0.922457,-1.76295,-0.198988,0.844016,-1.76345,-0.281245,0.92811,-1.84327,-0.226869,0.679925,-1.79431,-0.28157,0.701381,-1.72574,0.170045,1.10116,-2.11779,0.199107,1.02166,-2.1009,0.042386,1.05933,-2.06417,0.072721,0.983519,-2.04868,0.17403,0.86468,-2.05269,0.154134,0.671712,-2.02333,0.049416,0.713316,-2.04145,0.025955,0.649666,-2.02503,-0.029089,0.431117,-1.96688,-0.130571,0.364671,-1.90916,0.153233,0.376003,-1.94889,0.099277,0.341733,-2.01285,0.085547,1.26251,-1.82618,0.066866,1.30554,-2.03624},
/*768*/{0.220107,1.84177,-1.69924,0.280331,1.78292,-1.85834,0.101734,1.71254,-1.60918,0.010959,1.62117,-1.55626,0.142316,1.54408,-1.61911,0.214814,1.49434,-1.63867,0.193872,1.85664,-2.09935,0.279222,1.78575,-1.93542,0.063031,1.83426,-1.99593,0.012733,1.64959,-2.16395,0.036752,1.55515,-2.20719,0.209731,1.47816,-2.15557,0.258291,1.43636,-2.1679,0.163609,1.34495,-1.79552,-0.02941,1.4215,-1.84222,-0.029834,1.45565,-1.89364,-0.03762,1.43654,-1.95482,0.149417,1.39811,-2.05817,0.246116,1.08582,-1.72991,0.130461,1.02088,-1.75551,0.170765,0.947534,-1.77544,0.268528,0.878535,-1.75667,0.119393,0.823123,-1.7424,0.045232,0.810438,-1.75047,0.100847,0.932577,-1.76078,0.029893,0.920615,-1.76173,-0.18509,0.83588,-1.76431,-0.272533,0.915944,-1.84358,-0.204577,0.670484,-1.79388,-0.258552,0.689999,-1.72445,0.163584,1.09873,-2.11741,0.19067,1.0186,-2.10056,0.034355,1.05971,-2.06371,0.06335,0.982862,-2.04919,0.159962,0.860479,-2.05225,0.138163,0.668388,-2.02231,0.032925,0.710688,-2.04125,0.010237,0.648237,-2.02447,-0.050078,0.431736,-1.96678,-0.151824,0.364925,-1.90918,0.131464,0.376054,-1.9489,0.077204,0.341577,-2.01307,0.084157,1.2637,-1.82472,0.065788,1.30448,-2.03526},
/*769*/{0.218539,1.84414,-1.69816,0.279644,1.78348,-1.85639,0.097009,1.71683,-1.60922,0.00347,1.62575,-1.55806,0.135677,1.54827,-1.61775,0.207657,1.49805,-1.6374,0.196397,1.85848,-2.09813,0.279016,1.78698,-1.93366,0.063382,1.83648,-1.9974,0.018243,1.64943,-2.16309,0.044124,1.55501,-2.20744,0.218596,1.48329,-2.15271,0.268955,1.4447,-2.16495,0.163076,1.34677,-1.79465,-0.030279,1.42239,-1.84275,-0.029553,1.4567,-1.8946,-0.037511,1.43755,-1.95497,0.150575,1.39819,-2.05726,0.252288,1.09417,-1.72958,0.138157,1.02544,-1.75555,0.180455,0.95262,-1.77464,0.281818,0.886056,-1.75585,0.132758,0.824235,-1.74236,0.05991,0.809557,-1.75,0.110203,0.933284,-1.75947,0.040255,0.91923,-1.76024,-0.170865,0.826733,-1.76498,-0.263268,0.901086,-1.84345,-0.181922,0.660787,-1.79351,-0.236552,0.678313,-1.72335,0.15755,1.09652,-2.11688,0.181954,1.01577,-2.09962,0.027367,1.06063,-2.06421,0.051275,0.985928,-2.05026,0.145171,0.856612,-2.05156,0.120589,0.667869,-2.02234,0.017232,0.709651,-2.04034,-0.007289,0.647177,-2.0234,-0.070633,0.433867,-1.96702,-0.171863,0.365237,-1.90611,0.110466,0.376488,-1.94955,0.056507,0.341807,-2.01283,0.083756,1.26512,-1.8237,0.066705,1.30484,-2.03455},
/*770*/{0.215682,1.84709,-1.69737,0.278892,1.78543,-1.85485,0.091925,1.7213,-1.60937,-0.002383,1.63166,-1.55856,0.129238,1.55399,-1.61752,0.200504,1.5027,-1.63589,0.19748,1.8607,-2.0971,0.279058,1.78905,-1.93187,0.064593,1.83731,-1.99763,0.023922,1.64865,-2.16265,0.05034,1.55557,-2.20728,0.227887,1.48988,-2.15017,0.279481,1.45249,-2.16145,0.161556,1.34906,-1.79273,-0.030476,1.42356,-1.84454,-0.029545,1.45807,-1.89616,-0.03713,1.43919,-1.95582,0.151092,1.39817,-2.05736,0.258666,1.10076,-1.72946,0.144818,1.02942,-1.75485,0.189835,0.957752,-1.77337,0.293034,0.892573,-1.75621,0.147145,0.826162,-1.74282,0.07328,0.808929,-1.75058,0.119009,0.934498,-1.75859,0.048948,0.917006,-1.75992,-0.156553,0.817627,-1.76645,-0.254314,0.886602,-1.84377,-0.158877,0.651243,-1.79254,-0.213502,0.666204,-1.72165,0.15035,1.09463,-2.11614,0.171918,1.01325,-2.09894,0.018801,1.06293,-2.06451,0.040565,0.986539,-2.05084,0.129774,0.853223,-2.05106,0.103601,0.665175,-2.02107,0.001437,0.710259,-2.04009,-0.024745,0.647438,-2.02364,-0.087077,0.434729,-1.96616,-0.191103,0.36713,-1.90707,0.089088,0.37613,-1.95026,0.035655,0.342206,-2.01316,0.082689,1.26703,-1.8227,0.067409,1.30511,-2.03398},
/*771*/{0.212882,1.84913,-1.69637,0.277949,1.78737,-1.8526,0.087151,1.72603,-1.60922,-0.009071,1.6366,-1.55998,0.123687,1.5585,-1.61681,0.193776,1.50762,-1.63514,0.199677,1.86418,-2.09608,0.27926,1.79053,-1.92986,0.065299,1.83827,-1.99805,0.027678,1.64838,-2.16107,0.05771,1.55707,-2.20792,0.236339,1.49724,-2.14745,0.289042,1.46183,-2.1575,0.160879,1.35109,-1.79171,-0.030839,1.42539,-1.84546,-0.029804,1.4598,-1.89665,-0.036827,1.44071,-1.95759,0.151334,1.39804,-2.05735,0.264413,1.10879,-1.7297,0.153083,1.03427,-1.75328,0.198783,0.963121,-1.77396,0.304389,0.90058,-1.75615,0.160174,0.828207,-1.74293,0.087062,0.80799,-1.75113,0.127673,0.935512,-1.75722,0.057916,0.914982,-1.75922,-0.145826,0.810123,-1.76662,-0.24437,0.870598,-1.84395,-0.134947,0.641659,-1.79143,-0.190818,0.653594,-1.72132,0.142827,1.09329,-2.11496,0.162799,1.01101,-2.09772,0.010716,1.06474,-2.06397,0.030556,0.988524,-2.05137,0.112583,0.851647,-2.05018,0.086617,0.664874,-2.01854,-0.01563,0.710848,-2.03907,-0.041663,0.647739,-2.0218,-0.108118,0.436536,-1.96707,-0.212731,0.368903,-1.90621,0.068873,0.376289,-1.94997,0.015101,0.341839,-2.01351,0.081751,1.26901,-1.82179,0.067528,1.30546,-2.03344},
/*772*/{0.210546,1.85238,-1.6952,0.277601,1.79007,-1.85043,0.081832,1.73064,-1.60939,-0.016324,1.64224,-1.56108,0.116957,1.56464,-1.61698,0.188022,1.5134,-1.63391,0.201095,1.86728,-2.09508,0.279808,1.79315,-1.92885,0.065985,1.83949,-1.9984,0.03229,1.64832,-2.1595,0.064176,1.55923,-2.20805,0.244119,1.50594,-2.1449,0.298756,1.47224,-2.15374,0.161387,1.3539,-1.79113,-0.030937,1.42739,-1.84611,-0.029378,1.46127,-1.89695,-0.035453,1.44251,-1.95849,0.151996,1.39791,-2.057,0.269687,1.11686,-1.72771,0.160277,1.03659,-1.75237,0.207451,0.97003,-1.77378,0.315099,0.909246,-1.75606,0.173256,0.830371,-1.74355,0.102481,0.806846,-1.75127,0.135924,0.935948,-1.75582,0.067627,0.912017,-1.75901,-0.128518,0.796551,-1.76551,-0.234172,0.853317,-1.84523,-0.111351,0.631675,-1.79096,-0.168008,0.640513,-1.72069,0.135868,1.09164,-2.11333,0.152834,1.009,-2.09534,0.001715,1.06808,-2.06484,0.020359,0.988196,-2.04893,0.095746,0.85187,-2.04909,0.068882,0.665295,-2.01792,-0.032977,0.712912,-2.03781,-0.060226,0.650467,-2.02158,-0.130182,0.437665,-1.96923,-0.233553,0.369073,-1.90853,0.047698,0.376326,-1.9498,-0.006014,0.342407,-2.01323,0.081992,1.27139,-1.82053,0.068306,1.30588,-2.03254},
/*773*/{0.20714,1.8556,-1.69452,0.277218,1.79367,-1.84897,0.077292,1.73562,-1.60962,-0.022373,1.64834,-1.56231,0.110775,1.5708,-1.61679,0.181574,1.51885,-1.6332,0.202165,1.87107,-2.09386,0.280611,1.79678,-1.92633,0.066989,1.84142,-1.99952,0.036749,1.65033,-2.15831,0.070713,1.56172,-2.20857,0.251453,1.51371,-2.1418,0.307296,1.48252,-2.1499,0.161531,1.35693,-1.78972,-0.02962,1.42973,-1.84733,-0.027783,1.46379,-1.89902,-0.035249,1.44426,-1.95984,0.153908,1.39768,-2.05607,0.274489,1.12404,-1.72786,0.167837,1.0432,-1.75182,0.21631,0.975592,-1.77313,0.325147,0.917713,-1.75601,0.185987,0.832284,-1.74403,0.115255,0.805,-1.75241,0.143414,0.935887,-1.75551,0.074795,0.909432,-1.75962,-0.114077,0.785114,-1.76623,-0.223055,0.836249,-1.84683,-0.088585,0.62199,-1.79042,-0.144744,0.628353,-1.71974,0.128613,1.09094,-2.1114,0.142828,1.0075,-2.09301,-0.007257,1.07031,-2.06344,0.00939,0.990222,-2.04885,0.079402,0.852705,-2.04804,0.050233,0.666759,-2.01712,-0.050275,0.715117,-2.03655,-0.079016,0.652189,-2.02035,-0.150701,0.439458,-1.96757,-0.25438,0.371725,-1.90888,0.026593,0.375985,-1.94983,-0.027955,0.342363,-2.01315,0.082023,1.27431,-1.81891,0.069871,1.30617,-2.03143},
/*774*/{0.203752,1.85978,-1.69419,0.276672,1.7966,-1.84653,0.072579,1.74048,-1.60982,-0.027411,1.65473,-1.56327,0.104376,1.5765,-1.6172,0.176021,1.52508,-1.63225,0.203752,1.87495,-2.09269,0.280989,1.79974,-1.92413,0.068245,1.84333,-2.00094,0.041544,1.65199,-2.15826,0.077526,1.56439,-2.20899,0.259515,1.52282,-2.139,0.316239,1.49347,-2.14592,0.160803,1.35972,-1.78883,-0.02918,1.43273,-1.84889,-0.026657,1.4657,-1.9001,-0.033193,1.44694,-1.96141,0.15516,1.39786,-2.05574,0.28012,1.13105,-1.72727,0.173976,1.0465,-1.75246,0.223729,0.981971,-1.77394,0.335477,0.927653,-1.75647,0.198312,0.83477,-1.7439,0.128994,0.803592,-1.75182,0.150779,0.935878,-1.75548,0.082856,0.907919,-1.76013,-0.09902,0.773831,-1.76698,-0.211134,0.818249,-1.84777,-0.063909,0.612995,-1.78882,-0.121986,0.615592,-1.71963,0.121947,1.08953,-2.10924,0.132467,1.0067,-2.09073,-0.015716,1.07457,-2.0633,-0.001212,0.993574,-2.04832,0.063009,0.853798,-2.04614,0.032739,0.667547,-2.01542,-0.068251,0.717257,-2.03532,-0.098435,0.655807,-2.01973,-0.171252,0.441696,-1.96714,-0.275303,0.375742,-1.90821,0.006029,0.375936,-1.94948,-0.046919,0.342604,-2.01276,0.08132,1.27699,-1.81801,0.07093,1.30711,-2.03088},
/*775*/{0.200875,1.86337,-1.69362,0.275905,1.80072,-1.84445,0.068289,1.7457,-1.60969,-0.032972,1.66111,-1.56474,0.099171,1.58247,-1.61744,0.169961,1.53125,-1.63159,0.206178,1.87969,-2.09166,0.281607,1.80391,-1.92262,0.068808,1.84597,-2.00101,0.045463,1.65442,-2.15827,0.084224,1.56832,-2.20954,0.265684,1.53133,-2.13557,0.323776,1.50487,-2.14153,0.161865,1.36362,-1.78837,-0.028985,1.43581,-1.85055,-0.025436,1.46815,-1.90068,-0.031764,1.45008,-1.96264,0.155815,1.3978,-2.05574,0.283536,1.13893,-1.72686,0.179946,1.05076,-1.75268,0.230466,0.987817,-1.77432,0.34371,0.937509,-1.75709,0.209445,0.83676,-1.74394,0.142677,0.802633,-1.75199,0.158456,0.935647,-1.75487,0.09332,0.901589,-1.75923,-0.084618,0.763081,-1.7679,-0.197942,0.799954,-1.84904,-0.040071,0.602818,-1.78891,-0.097259,0.603719,-1.71872,0.114604,1.08929,-2.107,0.122455,1.00548,-2.08777,-0.022787,1.07866,-2.0632,-0.012921,0.99662,-2.04655,0.0471,0.855807,-2.04431,0.014729,0.670388,-2.01441,-0.085804,0.720505,-2.03395,-0.11404,0.658117,-2.01787,-0.190201,0.445913,-1.96587,-0.296422,0.379796,-1.90896,-0.015143,0.375754,-1.9494,-0.069221,0.34261,-2.0131,0.081602,1.28054,-1.81677,0.071594,1.30803,-2.03001},
/*776*/{0.197365,1.86762,-1.69305,0.276049,1.80484,-1.84304,0.064339,1.75108,-1.61069,-0.039012,1.66674,-1.56575,0.093896,1.58968,-1.61744,0.164083,1.53761,-1.63057,0.207411,1.88396,-2.09,0.282382,1.80764,-1.9204,0.07008,1.84884,-2.00193,0.049724,1.65676,-2.15904,0.091321,1.57207,-2.21057,0.272399,1.54122,-2.13222,0.330729,1.51611,-2.13727,0.162212,1.36733,-1.78792,-0.028296,1.43958,-1.85069,-0.024092,1.47105,-1.90223,-0.029219,1.45233,-1.96366,0.157726,1.39768,-2.05528,0.28647,1.14653,-1.72704,0.18771,1.05844,-1.75226,0.238831,0.995376,-1.77439,0.351577,0.946549,-1.75679,0.221354,0.839019,-1.74324,0.156667,0.801487,-1.75218,0.164368,0.934921,-1.75574,0.102372,0.897487,-1.75923,-0.069447,0.750828,-1.76875,-0.184348,0.780753,-1.84979,-0.015431,0.594125,-1.78783,-0.074064,0.590921,-1.71814,0.107308,1.08926,-2.1039,0.112847,1.00539,-2.08499,-0.031457,1.08197,-2.06159,-0.023916,1.00138,-2.04496,0.03291,0.857665,-2.04164,-0.003074,0.673454,-2.01297,-0.102995,0.723734,-2.03236,-0.13277,0.660996,-2.01634,-0.20986,0.449638,-1.96732,-0.315409,0.384983,-1.90804,-0.036079,0.376094,-1.94975,-0.090145,0.342918,-2.01245,0.081783,1.28415,-1.81506,0.072869,1.30881,-2.02869},
/*777*/{0.194315,1.87215,-1.69216,0.276441,1.80755,-1.84095,0.060188,1.75638,-1.61102,-0.045008,1.67364,-1.56792,0.088966,1.59621,-1.61809,0.159394,1.54397,-1.63012,0.209539,1.88868,-2.08874,0.282655,1.81193,-1.91835,0.071316,1.85117,-2.00245,0.056187,1.66063,-2.16077,0.097428,1.57603,-2.21123,0.277357,1.55058,-2.12864,0.337493,1.52808,-2.13224,0.163116,1.37184,-1.78797,-0.025835,1.44318,-1.85253,-0.022675,1.47389,-1.90388,-0.027101,1.45494,-1.96466,0.158977,1.39733,-2.05512,0.289173,1.15529,-1.72727,0.191901,1.06353,-1.75243,0.243797,1.00095,-1.77455,0.358862,0.956488,-1.75757,0.23271,0.841264,-1.7432,0.168474,0.800275,-1.75288,0.170712,0.934448,-1.75611,0.111406,0.894312,-1.75997,-0.054114,0.739577,-1.7702,-0.169825,0.761937,-1.85056,0.009316,0.585489,-1.78721,-0.049408,0.578999,-1.71849,0.100396,1.08992,-2.1012,0.103985,1.0058,-2.0815,-0.038292,1.08788,-2.06076,-0.033483,1.0057,-2.04417,0.018524,0.860424,-2.03834,-0.020613,0.67664,-2.01124,-0.119379,0.728375,-2.03112,-0.149651,0.665115,-2.01518,-0.226172,0.453358,-1.96588,-0.336244,0.391231,-1.90979,-0.057341,0.376243,-1.94878,-0.111154,0.342575,-2.01241,0.082691,1.28828,-1.81357,0.074066,1.3094,-2.02759},
/*778*/{0.191379,1.87651,-1.69203,0.275601,1.8121,-1.83884,0.055968,1.76184,-1.6114,-0.049186,1.68041,-1.56943,0.083469,1.60253,-1.61784,0.154301,1.54973,-1.62895,0.212324,1.89371,-2.08742,0.282881,1.81587,-1.9166,0.072942,1.85472,-2.00363,0.059938,1.66473,-2.16363,0.103686,1.581,-2.21278,0.28304,1.56144,-2.12507,0.342916,1.54007,-2.12799,0.165394,1.37621,-1.78797,-0.024902,1.44748,-1.85329,-0.020888,1.47679,-1.90558,-0.024556,1.45832,-1.96624,0.161064,1.39734,-2.05497,0.293284,1.16173,-1.72728,0.196367,1.06994,-1.75157,0.250035,1.00863,-1.77357,0.365124,0.965985,-1.75793,0.243093,0.843112,-1.74315,0.181158,0.799474,-1.75205,0.176189,0.933135,-1.75691,0.118231,0.890228,-1.76156,-0.036894,0.726719,-1.76914,-0.154488,0.742953,-1.8513,0.034654,0.577274,-1.78649,-0.024188,0.567205,-1.71776,0.094259,1.09105,-2.09805,0.094811,1.00617,-2.07753,-0.045279,1.09252,-2.05958,-0.043004,1.01051,-2.04317,0.004427,0.864029,-2.03581,-0.037055,0.68106,-2.0098,-0.136223,0.733055,-2.02965,-0.166944,0.670459,-2.01477,-0.247215,0.458311,-1.96667,-0.354223,0.398535,-1.90998,-0.077703,0.377156,-1.94873,-0.13181,0.343658,-2.01245,0.083853,1.29265,-1.81217,0.075652,1.31056,-2.0265},
/*779*/{0.188462,1.88073,-1.69194,0.274913,1.81814,-1.83751,0.052538,1.76777,-1.61243,-0.055052,1.68663,-1.57076,0.078341,1.60896,-1.61912,0.149198,1.55635,-1.62826,0.213544,1.89842,-2.08613,0.283374,1.82093,-1.91475,0.073919,1.8581,-2.00482,0.064944,1.66812,-2.1656,0.110357,1.58535,-2.2132,0.28683,1.57069,-2.12167,0.348737,1.55198,-2.12334,0.166703,1.38039,-1.78825,-0.022715,1.45172,-1.8545,-0.018332,1.48061,-1.90696,-0.022162,1.46172,-1.96752,0.163414,1.39735,-2.05477,0.296403,1.16867,-1.72705,0.198802,1.07619,-1.75236,0.254113,1.01493,-1.77545,0.370473,0.973951,-1.75976,0.253313,0.845207,-1.74275,0.194485,0.798201,-1.75181,0.182077,0.931705,-1.75723,0.127707,0.88464,-1.76261,-0.019846,0.711983,-1.76847,-0.138596,0.723817,-1.85188,0.060464,0.568529,-1.78654,0.001426,0.55631,-1.71794,0.088131,1.09196,-2.09447,0.085632,1.00773,-2.07403,-0.051601,1.09775,-2.05862,-0.052881,1.01652,-2.04228,-0.00947,0.868901,-2.03286,-0.054022,0.685564,-2.00846,-0.152721,0.738545,-2.02821,-0.182979,0.675991,-2.01402,-0.263683,0.463718,-1.965,-0.371426,0.406645,-1.90939,-0.098281,0.377857,-1.94881,-0.153643,0.343634,-2.01186,0.084655,1.29693,-1.81099,0.07703,1.31195,-2.02557},
/*780*/{0.186325,1.88507,-1.69175,0.27416,1.82242,-1.83524,0.049562,1.77326,-1.61275,-0.058836,1.69328,-1.57314,0.074474,1.61563,-1.61897,0.14531,1.56224,-1.62858,0.216121,1.90328,-2.08421,0.284564,1.82547,-1.91281,0.07616,1.86204,-2.00507,0.070307,1.67482,-2.16972,0.116806,1.59032,-2.2149,0.290949,1.58009,-2.11828,0.352972,1.56374,-2.11898,0.16918,1.38455,-1.78841,-0.020331,1.45605,-1.85416,-0.016272,1.48427,-1.90839,-0.018751,1.46449,-1.96844,0.16515,1.39756,-2.05437,0.298848,1.17693,-1.72745,0.203305,1.08249,-1.75135,0.259604,1.02186,-1.7752,0.374807,0.983257,-1.75823,0.263242,0.84664,-1.74268,0.206878,0.796426,-1.7516,0.188739,0.929655,-1.75946,0.136143,0.882033,-1.76202,-0.000449,0.701777,-1.76961,-0.121372,0.703978,-1.85178,0.085366,0.560934,-1.78626,0.027938,0.545674,-1.71739,0.082583,1.09262,-2.09128,0.076864,1.00917,-2.07074,-0.057788,1.1033,-2.05741,-0.060988,1.02198,-2.04132,-0.023797,0.874168,-2.0314,-0.070817,0.690811,-2.00706,-0.168341,0.744271,-2.02724,-0.200169,0.683302,-2.01336,-0.284087,0.472052,-1.97144,-0.391142,0.415258,-1.90967,-0.119661,0.376865,-1.9481,-0.174774,0.344466,-2.01196,0.086371,1.30131,-1.80937,0.078496,1.31319,-2.02413},
/*781*/{0.183992,1.88934,-1.69179,0.275446,1.82688,-1.83304,0.046219,1.77859,-1.61358,-0.064398,1.70053,-1.5753,0.069391,1.62172,-1.61965,0.140471,1.56851,-1.62774,0.218731,1.90856,-2.08299,0.285804,1.82968,-1.91052,0.076965,1.86542,-2.00596,0.075573,1.68161,-2.17255,0.123473,1.59563,-2.21603,0.293988,1.5898,-2.11491,0.356434,1.57522,-2.11497,0.170469,1.38901,-1.78904,-0.019822,1.46135,-1.85537,-0.014838,1.48782,-1.91048,-0.016028,1.46772,-1.96889,0.167262,1.39751,-2.05416,0.301994,1.18361,-1.72707,0.209253,1.08759,-1.74984,0.262636,1.02853,-1.7742,0.378048,0.990991,-1.75899,0.273737,0.847734,-1.74265,0.219274,0.79524,-1.75117,0.19385,0.927254,-1.76034,0.143049,0.876053,-1.76346,0.017252,0.688417,-1.76933,-0.103625,0.683131,-1.85147,0.111763,0.554052,-1.78714,0.055586,0.535827,-1.71831,0.077296,1.09435,-2.08839,0.069221,1.01155,-2.06738,-0.062704,1.11014,-2.05677,-0.069643,1.02934,-2.04057,-0.038885,0.879131,-2.03054,-0.085924,0.696211,-2.00636,-0.183272,0.751456,-2.02651,-0.215465,0.689766,-2.01258,-0.306229,0.475415,-1.96601,-0.406191,0.423324,-1.90754,-0.141228,0.376839,-1.94704,-0.195799,0.344256,-2.01123,0.086769,1.30609,-1.80797,0.079442,1.31459,-2.02291},
/*782*/{0.181928,1.89378,-1.69157,0.274115,1.83063,-1.83211,0.042715,1.78431,-1.61486,-0.066808,1.70673,-1.57672,0.065612,1.62752,-1.61997,0.136851,1.57465,-1.62711,0.220633,1.913,-2.0811,0.286766,1.83414,-1.90845,0.078939,1.86922,-2.00642,0.081158,1.68448,-2.17545,0.129716,1.60092,-2.21736,0.296943,1.59936,-2.11127,0.359525,1.58584,-2.1111,0.171458,1.39279,-1.78906,-0.01622,1.46639,-1.85571,-0.012493,1.49223,-1.91199,-0.012818,1.47121,-1.9696,0.169178,1.39725,-2.0535,0.303645,1.19026,-1.72678,0.208772,1.09227,-1.74991,0.266351,1.03562,-1.7734,0.38068,0.998805,-1.75939,0.28297,0.849142,-1.74293,0.231932,0.793607,-1.75138,0.199551,0.923647,-1.76172,0.152531,0.870112,-1.76397,0.034838,0.675337,-1.7697,-0.084472,0.663784,-1.85144,0.138122,0.547242,-1.78817,0.083376,0.525308,-1.71825,0.071848,1.09588,-2.08492,0.060829,1.01378,-2.06402,-0.067999,1.11604,-2.05548,-0.077139,1.03593,-2.03969,-0.053262,0.884606,-2.03009,-0.10088,0.703674,-2.00667,-0.198758,0.75898,-2.02626,-0.23103,0.698493,-2.01287,-0.315236,0.483894,-1.9653,-0.423823,0.434499,-1.90593,-0.162627,0.376384,-1.94699,-0.216346,0.345456,-2.01001,0.087232,1.31059,-1.80636,0.0802,1.31603,-2.02141},
/*783*/{0.180335,1.89753,-1.69139,0.273533,1.837,-1.8309,0.039232,1.78989,-1.61561,-0.072139,1.71253,-1.57936,0.06246,1.63368,-1.62062,0.132148,1.57942,-1.62593,0.222569,1.91754,-2.0798,0.286368,1.83862,-1.90717,0.080471,1.87259,-2.00745,0.085645,1.68962,-2.17923,0.135351,1.60635,-2.21904,0.30023,1.60912,-2.10823,0.362508,1.59619,-2.1074,0.172586,1.39687,-1.78918,-0.014663,1.47173,-1.85528,-0.009838,1.49581,-1.91348,-0.010038,1.47459,-1.96988,0.171491,1.39679,-2.05264,0.305539,1.19738,-1.72733,0.21091,1.09834,-1.74908,0.26833,1.04095,-1.77342,0.381914,1.00534,-1.75962,0.29243,0.849406,-1.74291,0.243828,0.79129,-1.75145,0.204098,0.920617,-1.76169,0.160203,0.863495,-1.76527,0.052523,0.660967,-1.76984,-0.065884,0.643832,-1.85017,0.164501,0.541111,-1.78808,0.11139,0.51514,-1.7188,0.067105,1.09809,-2.08226,0.052821,1.01652,-2.06064,-0.072977,1.12394,-2.05549,-0.085232,1.04241,-2.03848,-0.065806,0.890623,-2.0294,-0.115524,0.71057,-2.00688,-0.212998,0.766946,-2.02516,-0.245041,0.70451,-2.01192,-0.330165,0.491529,-1.96581,-0.438772,0.444807,-1.90592,-0.183553,0.376733,-1.94697,-0.236967,0.345841,-2.00981,0.087688,1.31522,-1.80435,0.081,1.31735,-2.01947},
/*784*/{0.178802,1.90144,-1.69149,0.273509,1.8423,-1.82958,0.037078,1.79542,-1.61736,-0.075192,1.71902,-1.58165,0.058998,1.63873,-1.62101,0.128828,1.58451,-1.62578,0.225491,1.92239,-2.07863,0.287255,1.84275,-1.90507,0.082497,1.87579,-2.00844,0.091272,1.69451,-2.18307,0.141847,1.61111,-2.22047,0.302042,1.61798,-2.10517,0.365229,1.6063,-2.10345,0.175779,1.40172,-1.79001,-0.012422,1.4768,-1.85514,-0.007579,1.49961,-1.91532,-0.006627,1.47812,-1.96992,0.172957,1.397,-2.05159,0.307868,1.20334,-1.72731,0.213276,1.10565,-1.74833,0.272095,1.0481,-1.7723,0.382632,1.01094,-1.75975,0.301844,0.849794,-1.74307,0.257068,0.789539,-1.75139,0.211534,0.91543,-1.76303,0.168965,0.857589,-1.76586,0.072695,0.648402,-1.7692,-0.045451,0.624332,-1.84942,0.190879,0.535584,-1.79068,0.140469,0.506314,-1.72069,0.062143,1.1008,-2.07988,0.045025,1.01963,-2.05783,-0.076954,1.12993,-2.0546,-0.092112,1.05041,-2.03819,-0.077664,0.898468,-2.02893,-0.129883,0.717929,-2.00725,-0.226897,0.774719,-2.02467,-0.259321,0.712825,-2.01154,-0.35365,0.495907,-1.96582,-0.453829,0.45779,-1.90507,-0.205742,0.376803,-1.94659,-0.260292,0.346476,-2.01025,0.089758,1.3205,-1.80248,0.082198,1.31893,-2.01758},
/*785*/{0.177657,1.90534,-1.6911,0.273118,1.8464,-1.82841,0.034251,1.8013,-1.61889,-0.079183,1.72483,-1.584,0.055347,1.64436,-1.62179,0.126699,1.58992,-1.62599,0.226638,1.92667,-2.07688,0.286974,1.84663,-1.90399,0.083885,1.87875,-2.00907,0.096278,1.69983,-2.18622,0.146916,1.61623,-2.22182,0.303707,1.62646,-2.10209,0.367148,1.61628,-2.10001,0.17816,1.40623,-1.79036,-0.010314,1.48195,-1.85636,-0.005426,1.50355,-1.91566,-0.004346,1.48152,-1.97019,0.175613,1.39707,-2.05044,0.309293,1.20917,-1.72727,0.214668,1.11169,-1.74751,0.272981,1.0533,-1.77152,0.382461,1.01684,-1.75967,0.310063,0.849248,-1.74305,0.268907,0.786761,-1.75118,0.216082,0.910444,-1.76392,0.177423,0.849977,-1.76636,0.092617,0.634876,-1.76895,-0.024672,0.604986,-1.848,0.217132,0.530041,-1.79225,0.169394,0.498009,-1.72259,0.058204,1.10315,-2.07719,0.037774,1.02312,-2.05528,-0.079633,1.13763,-2.05413,-0.097916,1.05797,-2.03715,-0.089983,0.905479,-2.02858,-0.143044,0.726513,-2.00825,-0.240636,0.784262,-2.02448,-0.272514,0.721065,-2.01154,-0.363326,0.504652,-1.96803,-0.471939,0.474657,-1.90457,-0.227191,0.376634,-1.94683,-0.28166,0.347859,-2.01074,0.091189,1.32556,-1.80064,0.083543,1.32063,-2.01568},
/*786*/{0.176123,1.90857,-1.69154,0.273548,1.84966,-1.8272,0.031751,1.80612,-1.62004,-0.082144,1.72967,-1.58673,0.052801,1.64947,-1.62233,0.123622,1.59475,-1.62522,0.228173,1.93056,-2.07578,0.287791,1.8509,-1.90232,0.085373,1.88257,-2.00966,0.103901,1.70561,-2.18861,0.152185,1.62116,-2.22317,0.305106,1.63418,-2.09969,0.369146,1.62552,-2.09668,0.178605,1.41018,-1.79019,-0.0082,1.48718,-1.8549,-0.003302,1.50782,-1.91673,-0.001977,1.48492,-1.97041,0.177567,1.39793,-2.04948,0.311391,1.21463,-1.72738,0.215942,1.11602,-1.74558,0.274076,1.05881,-1.76957,0.382287,1.02048,-1.76033,0.319224,0.848759,-1.74277,0.280027,0.783374,-1.75169,0.22244,0.904742,-1.76397,0.186728,0.84197,-1.7668,0.113378,0.622868,-1.7701,-0.002781,0.586051,-1.84673,0.243667,0.525779,-1.79477,0.200095,0.490592,-1.72344,0.053848,1.10587,-2.07518,0.031909,1.02711,-2.05295,-0.083592,1.14504,-2.05397,-0.10353,1.06663,-2.03667,-0.10015,0.913568,-2.0282,-0.153386,0.735486,-2.01105,-0.252945,0.792533,-2.02372,-0.286414,0.730164,-2.01014,-0.372511,0.512824,-1.96754,-0.479695,0.485533,-1.90321,-0.24803,0.376775,-1.94743,-0.303952,0.348545,-2.00913,0.091527,1.33016,-1.79893,0.084151,1.32293,-2.01392},
/*787*/{0.175089,1.9123,-1.69161,0.273966,1.85482,-1.82658,0.029804,1.81107,-1.62195,-0.08458,1.73501,-1.58957,0.049865,1.65331,-1.62242,0.120638,1.59891,-1.6247,0.2301,1.93427,-2.07435,0.288169,1.85466,-1.90092,0.087053,1.88528,-2.01011,0.108205,1.7103,-2.19136,0.157215,1.6263,-2.22506,0.306327,1.64149,-2.09807,0.369555,1.63346,-2.09339,0.180124,1.41424,-1.79005,-0.005332,1.49076,-1.85609,-0.001096,1.51091,-1.91813,0.001483,1.48771,-1.97089,0.179934,1.39857,-2.04814,0.3133,1.21941,-1.72747,0.217543,1.11915,-1.74289,0.274765,1.06241,-1.7686,0.381413,1.02345,-1.76014,0.326927,0.847046,-1.74337,0.294861,0.781891,-1.75108,0.227954,0.898899,-1.76441,0.194682,0.833338,-1.76665,0.134206,0.611939,-1.77033,0.019146,0.567819,-1.84559,0.26971,0.521426,-1.79753,0.228155,0.482917,-1.72669,0.051103,1.10878,-2.07319,0.02659,1.02958,-2.05103,-0.085022,1.15251,-2.05348,-0.108403,1.07433,-2.0368,-0.110464,0.922125,-2.02903,-0.167182,0.743767,-2.01163,-0.264748,0.80149,-2.02357,-0.298169,0.738225,-2.00881,-0.388037,0.519543,-1.96774,-0.491121,0.501439,-1.90143,-0.269515,0.376798,-1.9486,-0.326872,0.350541,-2.00909,0.092943,1.33434,-1.7976,0.086318,1.32449,-2.0125},
/*788*/{0.174395,1.91521,-1.69147,0.27341,1.85712,-1.82546,0.027977,1.81614,-1.62374,-0.087844,1.73975,-1.59248,0.047812,1.65785,-1.62272,0.11791,1.60274,-1.62423,0.232005,1.9381,-2.07314,0.288458,1.85797,-1.89971,0.088657,1.88765,-2.0103,0.112637,1.71441,-2.19321,0.161246,1.63091,-2.22655,0.307523,1.64779,-2.09665,0.370536,1.64166,-2.09025,0.181347,1.41967,-1.78975,-0.00311,1.49622,-1.85503,0.001379,1.51489,-1.91835,0.003684,1.49118,-1.97027,0.181612,1.39967,-2.04746,0.31409,1.22321,-1.72792,0.217102,1.12712,-1.74253,0.273228,1.06645,-1.76731,0.379806,1.02602,-1.76021,0.334958,0.845886,-1.74323,0.302491,0.776874,-1.75134,0.233157,0.891004,-1.76481,0.203332,0.826048,-1.76562,0.154415,0.599168,-1.77095,0.041887,0.549852,-1.845,0.295311,0.5177,-1.79967,0.25687,0.475232,-1.72874,0.047764,1.11119,-2.07204,0.01929,1.03495,-2.05013,-0.086771,1.15881,-2.05279,-0.111957,1.08173,-2.03632,-0.120063,0.930805,-2.0296,-0.177422,0.75257,-2.01346,-0.276241,0.80889,-2.02279,-0.30857,0.747855,-2.00905,-0.399424,0.52731,-1.96922,-0.499517,0.516798,-1.89901,-0.289865,0.377234,-1.95072,-0.349157,0.352632,-2.00828,0.094481,1.33987,-1.79532,0.087205,1.32689,-2.01004},
/*789*/{0.17354,1.91795,-1.69168,0.273433,1.86019,-1.82444,0.026002,1.82085,-1.62501,-0.089786,1.74411,-1.59446,0.045761,1.66217,-1.62416,0.115832,1.60588,-1.62416,0.233497,1.94145,-2.07165,0.289057,1.86156,-1.89852,0.090541,1.89073,-2.01182,0.116911,1.71893,-2.19551,0.16605,1.63542,-2.22871,0.308118,1.6539,-2.09512,0.371627,1.64764,-2.08745,0.181565,1.4229,-1.78939,-0.000994,1.50019,-1.85547,0.004106,1.51794,-1.91916,0.005899,1.49433,-1.97048,0.182942,1.40107,-2.04578,0.31577,1.22675,-1.7284,0.217747,1.12933,-1.73854,0.272464,1.06936,-1.76495,0.378944,1.0268,-1.75938,0.341526,0.843748,-1.7422,0.314892,0.774653,-1.75122,0.238547,0.884977,-1.76374,0.21244,0.817851,-1.76604,0.176166,0.586595,-1.77046,0.065684,0.532677,-1.84448,0.320354,0.514048,-1.8018,0.286263,0.46944,-1.72986,0.045777,1.1141,-2.07105,0.014612,1.03809,-2.04839,-0.088338,1.16558,-2.05271,-0.115697,1.08975,-2.03598,-0.128712,0.939159,-2.03069,-0.187417,0.761258,-2.01565,-0.286115,0.817843,-2.02185,-0.317497,0.756076,-2.00845,-0.409474,0.53638,-1.97034,-0.506869,0.532649,-1.89716,-0.308645,0.378067,-1.95242,-0.371957,0.356175,-2.00698,0.094623,1.34356,-1.79419,0.088193,1.32904,-2.00883},
/*790*/{0.17283,1.92045,-1.69207,0.266698,1.86597,-1.82589,0.024618,1.82472,-1.62657,-0.093011,1.74778,-1.59674,0.04454,1.66564,-1.62407,0.115481,1.6097,-1.62233,0.235205,1.94453,-2.07055,0.289231,1.86521,-1.898,0.09139,1.89379,-2.01247,0.121405,1.72301,-2.19708,0.16939,1.63916,-2.23033,0.308971,1.65928,-2.09371,0.372671,1.65388,-2.08546,0.183838,1.42708,-1.78792,0.001087,1.50424,-1.85481,0.006399,1.52149,-1.91917,0.008446,1.4969,-1.97029,0.185012,1.40245,-2.04494,0.315516,1.22914,-1.72913,0.218325,1.13476,-1.7373,0.270524,1.07298,-1.76229,0.375711,1.02642,-1.75941,0.348751,0.840826,-1.74206,0.324629,0.769692,-1.7505,0.243355,0.876839,-1.76341,0.221066,0.808745,-1.76504,0.197372,0.575654,-1.77118,0.089085,0.51612,-1.84428,0.345244,0.51099,-1.80303,0.312131,0.462826,-1.73282,0.042355,1.11723,-2.07058,0.010785,1.04254,-2.04921,-0.089379,1.17063,-2.05198,-0.11898,1.09567,-2.03508,-0.13663,0.947135,-2.03122,-0.195929,0.768955,-2.01791,-0.293367,0.826495,-2.02158,-0.32557,0.765033,-2.00817,-0.42059,0.54468,-1.97129,-0.513715,0.55034,-1.89561,-0.328474,0.377931,-1.95374,-0.394271,0.36137,-2.00538,0.096553,1.34796,-1.79224,0.090035,1.33115,-2.00672},
/*791*/{0.172181,1.92262,-1.69204,0.267475,1.86889,-1.82462,0.023247,1.82849,-1.62755,-0.093808,1.75121,-1.59874,0.042702,1.66843,-1.62381,0.113543,1.61181,-1.62073,0.236813,1.94719,-2.06978,0.289839,1.86829,-1.89694,0.093425,1.89596,-2.01239,0.124612,1.72722,-2.19883,0.172267,1.64333,-2.23138,0.310325,1.6637,-2.09325,0.37326,1.658,-2.08105,0.182925,1.43034,-1.78687,0.002141,1.5076,-1.85415,0.008035,1.52466,-1.91903,0.010895,1.49978,-1.96983,0.186802,1.40442,-2.04365,0.316162,1.23126,-1.72986,0.218712,1.13814,-1.73568,0.269253,1.07457,-1.75951,0.374907,1.02697,-1.75774,0.355336,0.838539,-1.74153,0.335011,0.766168,-1.75032,0.24841,0.86897,-1.76318,0.229779,0.799401,-1.76552,0.217838,0.564089,-1.77164,0.112789,0.500947,-1.84542,0.369272,0.508004,-1.80396,0.339988,0.457043,-1.73566,0.040499,1.12047,-2.07065,0.006901,1.04477,-2.04853,-0.091645,1.17565,-2.05062,-0.121292,1.10064,-2.0338,-0.142696,0.954311,-2.03142,-0.203752,0.776632,-2.01954,-0.300291,0.835316,-2.02091,-0.333433,0.773814,-2.00892,-0.429095,0.554067,-1.97249,-0.518804,0.566794,-1.8935,-0.345714,0.377738,-1.95419,-0.416228,0.367518,-2.00274,0.096282,1.35114,-1.79105,0.091212,1.33377,-2.00552},
/*792*/{0.172212,1.92489,-1.6922,0.273837,1.8672,-1.82205,0.022226,1.83153,-1.6301,-0.097215,1.7536,-1.60111,0.040887,1.67118,-1.62483,0.112689,1.61443,-1.62027,0.238736,1.94992,-2.06877,0.291032,1.86988,-1.89555,0.094738,1.89886,-2.01307,0.128642,1.73074,-2.20039,0.176291,1.64637,-2.23342,0.311834,1.66751,-2.09291,0.373736,1.66324,-2.08146,0.186168,1.43361,-1.78683,0.004036,1.51152,-1.85377,0.009673,1.5274,-1.91845,0.01159,1.50292,-1.96983,0.187679,1.40625,-2.04311,0.31808,1.23289,-1.73074,0.215528,1.14118,-1.73526,0.267799,1.07677,-1.75795,0.372371,1.02561,-1.75663,0.361363,0.835606,-1.74055,0.344164,0.762192,-1.74987,0.252588,0.860729,-1.76349,0.237076,0.789864,-1.76558,0.237832,0.553943,-1.77243,0.136511,0.486364,-1.84633,0.39183,0.505316,-1.80614,0.365419,0.452172,-1.73763,0.038565,1.12377,-2.07169,0.004516,1.04916,-2.0505,-0.092524,1.18075,-2.04906,-0.122802,1.10471,-2.03329,-0.146012,0.95982,-2.03113,-0.21013,0.782855,-2.02067,-0.306115,0.843502,-2.01976,-0.339433,0.781075,-2.00908,-0.440162,0.563007,-1.97302,-0.52419,0.584866,-1.89353,-0.363187,0.379038,-1.95482,-0.437719,0.375788,-2.00069,0.098543,1.35495,-1.79001,0.092014,1.33609,-2.00432},
/*793*/{0.171979,1.92665,-1.69202,0.275606,1.86934,-1.82099,0.021223,1.83453,-1.63139,-0.097638,1.75602,-1.60291,0.039726,1.67342,-1.62486,0.111796,1.61639,-1.61955,0.239657,1.95242,-2.06755,0.291749,1.87248,-1.89457,0.096098,1.90058,-2.01307,0.133019,1.73418,-2.20111,0.179071,1.64937,-2.23486,0.311849,1.66965,-2.09236,0.373843,1.66685,-2.08018,0.186245,1.43683,-1.7864,0.00579,1.51454,-1.85354,0.010779,1.53048,-1.91822,0.014253,1.50571,-1.96897,0.188815,1.40872,-2.04237,0.315054,1.23295,-1.73146,0.212582,1.14187,-1.73383,0.264322,1.07792,-1.75718,0.369012,1.02354,-1.75517,0.366529,0.832981,-1.73946,0.353256,0.757833,-1.74868,0.257271,0.851564,-1.76326,0.245214,0.780115,-1.76507,0.25749,0.544634,-1.77296,0.159791,0.472532,-1.84805,0.414298,0.503237,-1.80769,0.390486,0.447146,-1.73963,0.036688,1.12731,-2.07241,0.002645,1.0522,-2.05151,-0.09314,1.18398,-2.04778,-0.124016,1.1086,-2.0322,-0.146632,0.963543,-2.03104,-0.216305,0.788562,-2.02151,-0.310249,0.850176,-2.02023,-0.344588,0.789714,-2.00896,-0.449204,0.572644,-1.97395,-0.528206,0.602064,-1.89453,-0.381918,0.379596,-1.95331,-0.457867,0.38513,-1.99847,0.099225,1.35803,-1.78926,0.093196,1.33882,-2.00355},
/*794*/{0.172093,1.92807,-1.69181,0.275814,1.87134,-1.82055,0.020543,1.83708,-1.63231,-0.098247,1.75841,-1.60445,0.039013,1.67482,-1.62491,0.1111,1.61803,-1.61816,0.240912,1.95424,-2.06693,0.292548,1.87459,-1.89401,0.096721,1.90334,-2.01289,0.131257,1.73764,-2.20277,0.181553,1.65209,-2.23601,0.312468,1.67209,-2.09151,0.374526,1.6686,-2.07793,0.186496,1.43954,-1.78601,0.006991,1.51701,-1.85296,0.010959,1.5343,-1.9189,0.015051,1.50895,-1.96851,0.189461,1.41079,-2.04163,0.314815,1.23317,-1.73177,0.21123,1.1431,-1.73319,0.261299,1.07707,-1.75568,0.366395,1.02112,-1.75321,0.370632,0.829125,-1.73753,0.361919,0.753549,-1.74784,0.261316,0.841878,-1.76174,0.252111,0.770645,-1.76528,0.278245,0.53469,-1.77373,0.183256,0.45981,-1.8501,0.436656,0.501419,-1.80838,0.415058,0.443129,-1.74179,0.035512,1.13067,-2.07377,0.000198,1.05411,-2.05193,-0.09347,1.18759,-2.0462,-0.124179,1.11217,-2.03127,-0.146228,0.96567,-2.0307,-0.220113,0.792741,-2.02213,-0.312777,0.856529,-2.02008,-0.348707,0.797223,-2.00892,-0.453134,0.58452,-1.97589,-0.534333,0.617702,-1.89456,-0.400146,0.377706,-1.94988,-0.476365,0.395395,-1.99561,0.099198,1.36101,-1.78875,0.09339,1.3415,-2.00302},
/*795*/{0.173085,1.92964,-1.69102,0.276984,1.87295,-1.81965,0.02053,1.83862,-1.63326,-0.098631,1.7608,-1.60569,0.038762,1.67635,-1.62463,0.110791,1.61928,-1.61715,0.241525,1.95672,-2.0663,0.292605,1.8764,-1.89353,0.098607,1.90498,-2.01294,0.133322,1.74035,-2.20345,0.183702,1.65437,-2.23681,0.313605,1.67403,-2.0914,0.375591,1.67055,-2.07689,0.185729,1.44194,-1.78573,0.008184,1.51941,-1.8527,0.01229,1.53638,-1.91783,0.015935,1.51196,-1.9684,0.190315,1.41346,-2.04131,0.312795,1.23152,-1.73219,0.208384,1.14247,-1.73266,0.257606,1.07606,-1.75405,0.362662,1.01747,-1.75178,0.375004,0.825056,-1.73682,0.369493,0.750595,-1.74727,0.264249,0.832172,-1.7612,0.259142,0.760741,-1.7651,0.296362,0.526098,-1.77361,0.206022,0.448651,-1.85147,0.456847,0.499045,-1.80814,0.438588,0.440809,-1.74349,0.035949,1.13377,-2.07424,0.001627,1.0585,-2.05377,-0.09392,1.19027,-2.04588,-0.124233,1.11439,-2.02988,-0.144334,0.967451,-2.02998,-0.223635,0.795531,-2.02215,-0.313443,0.862201,-2.02006,-0.35196,0.803496,-2.00991,-0.461194,0.595415,-1.97687,-0.539959,0.634332,-1.89724,-0.420959,0.382231,-1.94688,-0.495215,0.405892,-1.9914,0.099234,1.36309,-1.78868,0.094028,1.34427,-2.00303},
/*796*/{0.172886,1.93047,-1.69086,0.277976,1.87239,-1.81844,0.020159,1.84005,-1.63442,-0.099791,1.76129,-1.607,0.038247,1.67683,-1.6246,0.111216,1.61966,-1.61566,0.242443,1.95821,-2.06544,0.294147,1.87826,-1.8921,0.099248,1.9079,-2.0131,0.135439,1.74285,-2.2039,0.185172,1.65603,-2.23826,0.313832,1.67494,-2.09122,0.376022,1.67082,-2.07584,0.190057,1.44411,-1.78529,0.008514,1.52201,-1.85271,0.012648,1.53932,-1.91631,0.016203,1.51448,-1.96792,0.190707,1.41608,-2.04065,0.311886,1.2283,-1.73098,0.20857,1.14295,-1.73132,0.252947,1.07435,-1.75235,0.35908,1.01314,-1.74993,0.378355,0.821216,-1.73557,0.376953,0.745677,-1.74653,0.267092,0.822117,-1.75874,0.266333,0.751755,-1.76491,0.316139,0.5185,-1.77427,0.227231,0.437892,-1.85328,0.475618,0.498281,-1.80868,0.460098,0.437365,-1.74438,0.036338,1.13584,-2.07501,0.004158,1.06066,-2.05641,-0.093649,1.19198,-2.04402,-0.122359,1.11544,-2.02817,-0.142734,0.968097,-2.02954,-0.225036,0.797767,-2.02153,-0.313521,0.867903,-2.02146,-0.354778,0.809605,-2.01008,-0.467733,0.604844,-1.97703,-0.546008,0.651905,-1.90059,-0.441485,0.394016,-1.94142,-0.514576,0.41827,-1.98936,0.101692,1.36588,-1.78802,0.094918,1.34675,-2.0023},
/*797*/{0.173285,1.93186,-1.69006,0.278166,1.87527,-1.81806,0.020695,1.84107,-1.63439,-0.100936,1.76049,-1.60805,0.038484,1.6773,-1.62455,0.111173,1.61963,-1.61446,0.243318,1.95974,-2.06451,0.294498,1.87996,-1.8917,0.1003,1.90936,-2.01268,0.136857,1.74509,-2.20435,0.185637,1.6582,-2.23887,0.31455,1.6751,-2.09057,0.376626,1.67057,-2.07535,0.190882,1.447,-1.78514,0.007924,1.5238,-1.8524,0.013688,1.54145,-1.91607,0.016476,1.51677,-1.96702,0.191873,1.41843,-2.04103,0.310035,1.22521,-1.73093,0.202508,1.14104,-1.73077,0.247898,1.07243,-1.75151,0.354423,1.00791,-1.7487,0.381072,0.816702,-1.73451,0.383575,0.741098,-1.74646,0.270125,0.813092,-1.75709,0.273097,0.741104,-1.76275,0.334901,0.512766,-1.77544,0.248746,0.42859,-1.85467,0.49485,0.499376,-1.8099,0.480517,0.435603,-1.74611,0.037876,1.13825,-2.07582,0.005989,1.06134,-2.05714,-0.091168,1.19234,-2.04306,-0.119951,1.11594,-2.02762,-0.138626,0.96833,-2.02867,-0.226715,0.798432,-2.02052,-0.313266,0.872042,-2.02212,-0.355342,0.816347,-2.01137,-0.476498,0.615679,-1.97819,-0.553492,0.666448,-1.90316,-0.459984,0.40502,-1.9381,-0.530001,0.431215,-1.98597,0.102798,1.36822,-1.78781,0.095968,1.3491,-2.00208},
/*798*/{0.17449,1.93237,-1.68955,0.279211,1.8761,-1.81793,0.021484,1.84105,-1.63423,-0.100588,1.76016,-1.60901,0.039763,1.67702,-1.6241,0.112504,1.61926,-1.61235,0.243685,1.96084,-2.06384,0.294788,1.88126,-1.8914,0.101171,1.91175,-2.01228,0.138511,1.74666,-2.20449,0.185473,1.65866,-2.23872,0.316446,1.67556,-2.09038,0.377121,1.6692,-2.07474,0.190179,1.4472,-1.78554,0.010064,1.5253,-1.85193,0.014074,1.54388,-1.91588,0.017416,1.51899,-1.96659,0.191844,1.4212,-2.04076,0.30662,1.22025,-1.73098,0.199044,1.1392,-1.72946,0.242961,1.06994,-1.75066,0.350369,1.00254,-1.74691,0.384049,0.812649,-1.7344,0.390886,0.73921,-1.74668,0.272857,0.803681,-1.75513,0.27928,0.732326,-1.76123,0.349649,0.506054,-1.7756,0.26854,0.419796,-1.85648,0.511858,0.499551,-1.81103,0.500759,0.435509,-1.74717,0.040408,1.13975,-2.07638,0.009136,1.06305,-2.05819,-0.090385,1.19239,-2.04292,-0.117666,1.11574,-2.02696,-0.134435,0.967341,-2.02776,-0.228442,0.799365,-2.01922,-0.311159,0.87564,-2.02307,-0.356459,0.82199,-2.0119,-0.489683,0.627597,-1.97781,-0.562311,0.684201,-1.90661,-0.476383,0.416713,-1.93284,-0.54631,0.44442,-1.98379,0.102434,1.36886,-1.78865,0.095858,1.35166,-2.0031},
/*799*/{0.175661,1.93295,-1.68865,0.280195,1.87667,-1.81705,0.021629,1.84061,-1.63425,-0.09931,1.75948,-1.60944,0.040229,1.67656,-1.62337,0.112939,1.61852,-1.61063,0.244284,1.96169,-2.06348,0.295844,1.88252,-1.89056,0.101362,1.91253,-2.01196,0.13809,1.74825,-2.20366,0.185943,1.65982,-2.239,0.317546,1.67495,-2.09038,0.377022,1.6666,-2.07462,0.19239,1.44924,-1.78558,0.010954,1.52545,-1.85085,0.014498,1.54437,-1.9148,0.017964,1.52,-1.96532,0.19267,1.42371,-2.04055,0.303652,1.21549,-1.72945,0.196976,1.13636,-1.72899,0.236848,1.06549,-1.75115,0.345411,0.996427,-1.74622,0.387565,0.808903,-1.73364,0.396204,0.734206,-1.74725,0.275871,0.795087,-1.75246,0.286165,0.723739,-1.75871,0.365156,0.500763,-1.77601,0.28708,0.41269,-1.85715,0.52753,0.501278,-1.81117,0.517457,0.43558,-1.74854,0.042762,1.13987,-2.07707,0.013552,1.06211,-2.05871,-0.087025,1.19141,-2.04217,-0.113872,1.11435,-2.02623,-0.129522,0.965738,-2.02661,-0.228978,0.799848,-2.01768,-0.308972,0.880259,-2.02346,-0.356051,0.827936,-2.01276,-0.495052,0.639796,-1.97916,-0.569765,0.70008,-1.90937,-0.493332,0.430139,-1.92861,-0.560137,0.458505,-1.98113,0.105258,1.37001,-1.7883,0.097692,1.3532,-2.00274},
/*800*/{0.176472,1.93314,-1.68789,0.280004,1.87716,-1.8166,0.02276,1.84005,-1.63425,-0.099212,1.7583,-1.60923,0.042145,1.67561,-1.62264,0.113921,1.61734,-1.60851,0.244829,1.96195,-2.06308,0.296255,1.8841,-1.89086,0.102134,1.91382,-2.01066,0.137893,1.74888,-2.20284,0.184717,1.6604,-2.2387,0.317019,1.67222,-2.09055,0.377708,1.66435,-2.07475,0.192503,1.45014,-1.78523,0.010901,1.5262,-1.85098,0.013979,1.54588,-1.9146,0.018086,1.5216,-1.96481,0.193169,1.42586,-2.04067,0.298615,1.21063,-1.72977,0.186448,1.1309,-1.72996,0.230281,1.06148,-1.75093,0.338728,0.990788,-1.74501,0.39029,0.805866,-1.73346,0.401478,0.731964,-1.74678,0.279493,0.78731,-1.74928,0.290806,0.716956,-1.75663,0.379119,0.495553,-1.77587,0.304348,0.40669,-1.85781,0.541518,0.503488,-1.80976,0.534542,0.435599,-1.74891,0.046055,1.13968,-2.07699,0.017116,1.06249,-2.05851,-0.08452,1.1899,-2.04184,-0.110679,1.11314,-2.02625,-0.123954,0.96328,-2.02572,-0.229581,0.799993,-2.01535,-0.305632,0.883127,-2.02415,-0.355408,0.833574,-2.01402,-0.503048,0.652046,-1.97925,-0.577493,0.716724,-1.91323,-0.508496,0.443712,-1.92406,-0.571505,0.473646,-1.97927,0.105338,1.37082,-1.78872,0.098179,1.35507,-2.00326},
/*801*/{0.177626,1.93333,-1.68688,0.280024,1.87947,-1.8165,0.023848,1.83874,-1.6335,-0.096577,1.75717,-1.60904,0.042891,1.67344,-1.62086,0.115592,1.61559,-1.60651,0.245181,1.96222,-2.06262,0.296372,1.88447,-1.89046,0.101878,1.91485,-2.01007,0.136197,1.7499,-2.20123,0.183363,1.66072,-2.23839,0.316389,1.6697,-2.09011,0.377996,1.66114,-2.07523,0.194491,1.45105,-1.78567,0.011237,1.52623,-1.8499,0.013536,1.54717,-1.91392,0.017938,1.52259,-1.9643,0.193972,1.42773,-2.04056,0.295296,1.20447,-1.72772,0.184319,1.12885,-1.72952,0.224164,1.05661,-1.75086,0.331679,0.985905,-1.74461,0.392227,0.803281,-1.734,0.406991,0.73002,-1.74751,0.282308,0.780713,-1.74741,0.296551,0.711047,-1.75583,0.390001,0.492612,-1.77559,0.31954,0.401381,-1.85808,0.553504,0.505858,-1.80985,0.548649,0.436146,-1.74931,0.050168,1.13915,-2.07708,0.021978,1.06094,-2.05858,-0.081247,1.18765,-2.04174,-0.105781,1.11023,-2.02517,-0.117886,0.960349,-2.02499,-0.229675,0.799954,-2.0136,-0.302098,0.887414,-2.02557,-0.35337,0.840167,-2.0149,-0.511123,0.665422,-1.97799,-0.584034,0.73314,-1.91578,-0.520736,0.457615,-1.91932,-0.582147,0.488844,-1.97737,0.10711,1.37149,-1.78889,0.099152,1.35652,-2.00346},
/*802*/{0.178275,1.93308,-1.68564,0.28079,1.88035,-1.81642,0.024707,1.83697,-1.63244,-0.095472,1.75473,-1.60892,0.045064,1.6715,-1.61959,0.117798,1.61302,-1.60398,0.245233,1.96206,-2.0617,0.296211,1.88486,-1.88952,0.102597,1.91575,-2.00964,0.134033,1.75066,-2.19945,0.182335,1.66092,-2.23746,0.316002,1.66647,-2.0907,0.378153,1.65686,-2.07676,0.194472,1.45127,-1.78569,0.011394,1.52566,-1.84936,0.014028,1.54748,-1.91268,0.018516,1.52304,-1.96311,0.194477,1.42924,-2.04083,0.291962,1.19844,-1.72645,0.177091,1.1237,-1.73024,0.217113,1.05272,-1.75092,0.326613,0.980027,-1.74466,0.393447,0.800337,-1.73479,0.409074,0.726745,-1.74717,0.283512,0.775135,-1.7466,0.299533,0.705993,-1.75576,0.40154,0.488383,-1.77433,0.333476,0.396476,-1.85798,0.563271,0.507865,-1.80924,0.561114,0.43801,-1.74974,0.053801,1.13775,-2.07663,0.026969,1.0592,-2.05852,-0.077379,1.18435,-2.04144,-0.10156,1.10594,-2.02454,-0.112341,0.957475,-2.02356,-0.229779,0.800982,-2.01144,-0.297195,0.891565,-2.02633,-0.352013,0.846326,-2.01469,-0.517617,0.679678,-1.97803,-0.589108,0.749945,-1.9175,-0.532826,0.472705,-1.91587,-0.592109,0.504485,-1.97597,0.107882,1.37111,-1.78922,0.099899,1.35759,-2.00388},
/*803*/{0.179062,1.93252,-1.68499,0.280906,1.88066,-1.81604,0.026303,1.83502,-1.63137,-0.092817,1.75309,-1.60818,0.047189,1.66806,-1.61802,0.119639,1.61064,-1.60188,0.245123,1.96135,-2.06072,0.296145,1.8848,-1.88917,0.102838,1.91608,-2.00866,0.13128,1.75126,-2.19752,0.17949,1.66113,-2.23601,0.318017,1.66452,-2.09206,0.378335,1.65218,-2.07734,0.194449,1.45113,-1.78626,0.011128,1.52527,-1.84907,0.013844,1.54718,-1.91273,0.019077,1.52216,-1.96298,0.195014,1.42985,-2.04096,0.287421,1.19177,-1.7256,0.170994,1.12054,-1.72996,0.21073,1.04803,-1.75045,0.319244,0.974672,-1.74427,0.393731,0.798719,-1.73531,0.412083,0.725512,-1.74696,0.285074,0.769458,-1.74582,0.303395,0.700156,-1.75329,0.409742,0.485418,-1.77272,0.344254,0.392733,-1.85701,0.573562,0.510129,-1.80957,0.57186,0.439126,-1.74979,0.058461,1.13626,-2.07574,0.032501,1.05755,-2.05813,-0.073441,1.1813,-2.04111,-0.096194,1.10149,-2.02427,-0.105489,0.95503,-2.02277,-0.229335,0.802062,-2.01014,-0.292199,0.895149,-2.02647,-0.348864,0.852936,-2.01629,-0.52109,0.693789,-1.97797,-0.592974,0.767408,-1.92172,-0.545323,0.489287,-1.9122,-0.603232,0.522541,-1.97631,0.108418,1.37066,-1.78959,0.10064,1.35766,-2.00429},
/*804*/{0.180599,1.93199,-1.68395,0.280906,1.88066,-1.81604,0.027767,1.83264,-1.63007,-0.090524,1.74934,-1.60709,0.049531,1.66534,-1.61598,0.122219,1.60711,-1.59951,0.245236,1.9608,-2.05972,0.296039,1.88453,-1.88916,0.102304,1.91623,-2.00677,0.129464,1.75027,-2.19518,0.177097,1.66096,-2.23463,0.31714,1.66051,-2.09329,0.377437,1.64676,-2.07798,0.194438,1.45082,-1.78695,0.011088,1.5229,-1.84835,0.014528,1.54561,-1.91193,0.018723,1.52159,-1.96203,0.19545,1.43022,-2.04092,0.28413,1.18532,-1.72468,0.166032,1.11689,-1.73139,0.203194,1.04315,-1.75191,0.31276,0.969097,-1.74352,0.392797,0.796495,-1.73579,0.413373,0.72435,-1.74734,0.284781,0.764488,-1.74688,0.305038,0.695559,-1.75339,0.416504,0.484354,-1.77244,0.352755,0.388962,-1.85532,0.579824,0.509067,-1.80853,0.580863,0.439615,-1.74912,0.063829,1.13421,-2.07475,0.039131,1.05451,-2.05697,-0.070308,1.17676,-2.04016,-0.090765,1.09805,-2.02314,-0.098689,0.951025,-2.02195,-0.228269,0.803571,-2.008,-0.286858,0.899181,-2.02656,-0.345052,0.861904,-2.01769,-0.525783,0.709117,-1.97713,-0.594462,0.785864,-1.92283,-0.556184,0.506067,-1.91044,-0.611404,0.539295,-1.97409,0.109582,1.36926,-1.78994,0.101555,1.35738,-2.0047},
/*805*/{0.181753,1.93092,-1.68313,0.28122,1.87986,-1.81589,0.029197,1.82971,-1.62889,-0.087087,1.74554,-1.60503,0.053952,1.66251,-1.6147,0.125147,1.60353,-1.59729,0.245036,1.95957,-2.05945,0.295937,1.88316,-1.88858,0.102069,1.91575,-2.0063,0.1254,1.74994,-2.1931,0.174314,1.66104,-2.23291,0.315992,1.65615,-2.09417,0.376622,1.64127,-2.07979,0.196649,1.45019,-1.78769,0.011038,1.52166,-1.84827,0.013868,1.54474,-1.9125,0.018388,1.52036,-1.96149,0.196083,1.4303,-2.04116,0.279163,1.17851,-1.72311,0.160662,1.11359,-1.73141,0.196294,1.0386,-1.75295,0.306237,0.963923,-1.74387,0.390493,0.794297,-1.73518,0.412931,0.72294,-1.74695,0.283546,0.760297,-1.74841,0.306371,0.692088,-1.75397,0.423678,0.481668,-1.7713,0.360296,0.386034,-1.85463,0.586042,0.510313,-1.80832,0.587609,0.439816,-1.7483,0.068997,1.13146,-2.07373,0.044772,1.05105,-2.05629,-0.065248,1.17117,-2.03969,-0.085257,1.09222,-2.02283,-0.091661,0.946299,-2.02082,-0.225249,0.805383,-2.00707,-0.281136,0.903683,-2.02701,-0.341342,0.865868,-2.01505,-0.528433,0.723738,-1.97773,-0.594926,0.803292,-1.92472,-0.566169,0.523082,-1.90931,-0.618861,0.557808,-1.97349,0.111409,1.36856,-1.79033,0.102554,1.3568,-2.00506},
/*806*/{0.182466,1.92959,-1.68223,0.282749,1.87771,-1.81524,0.03122,1.82666,-1.62715,-0.084564,1.74131,-1.6033,0.055641,1.65817,-1.61124,0.128618,1.6005,-1.59478,0.243975,1.95801,-2.05863,0.296114,1.8826,-1.88849,0.101763,1.91485,-2.00534,0.121403,1.75083,-2.19132,0.171368,1.6604,-2.23113,0.316151,1.65037,-2.09492,0.3758,1.63503,-2.08171,0.195576,1.44871,-1.78822,0.011712,1.51863,-1.84702,0.014089,1.54264,-1.91168,0.018279,1.51813,-1.96097,0.19674,1.43009,-2.04133,0.275925,1.17203,-1.72102,0.15478,1.10743,-1.73191,0.189135,1.03485,-1.75527,0.299046,0.957636,-1.74277,0.388279,0.792651,-1.73529,0.410624,0.72088,-1.74618,0.28151,0.755643,-1.74913,0.30499,0.688476,-1.75495,0.42625,0.481398,-1.77103,0.365401,0.383326,-1.85282,0.58816,0.509744,-1.80904,0.591635,0.440319,-1.74781,0.074177,1.12842,-2.0727,0.05262,1.04806,-2.05421,-0.060381,1.16606,-2.03921,-0.079279,1.08681,-2.02228,-0.084176,0.940985,-2.02006,-0.224201,0.807227,-2.00644,-0.27477,0.907733,-2.02637,-0.336666,0.873088,-2.01416,-0.523184,0.738229,-1.97718,-0.593817,0.822112,-1.92513,-0.574449,0.540206,-1.90794,-0.625005,0.576196,-1.97209,0.112187,1.36606,-1.79082,0.103491,1.35576,-2.00564},
/*807*/{0.183638,1.92819,-1.68124,0.281922,1.87789,-1.81519,0.032833,1.8227,-1.62522,-0.080303,1.73655,-1.60097,0.059064,1.6542,-1.6083,0.132119,1.59638,-1.59216,0.243547,1.95648,-2.05745,0.295888,1.88148,-1.88831,0.1009,1.91425,-2.00339,0.117914,1.75031,-2.18904,0.167302,1.65995,-2.22972,0.315399,1.64592,-2.09652,0.376089,1.62697,-2.08296,0.198315,1.44737,-1.78774,0.011024,1.51669,-1.84718,0.014541,1.541,-1.91158,0.018831,1.51592,-1.96117,0.196896,1.42948,-2.04148,0.273426,1.16567,-1.71967,0.150833,1.10534,-1.73268,0.185867,1.02991,-1.75429,0.292586,0.95159,-1.74275,0.384244,0.78975,-1.73457,0.409892,0.718611,-1.74545,0.279174,0.751695,-1.7496,0.303219,0.683426,-1.75624,0.428348,0.477766,-1.77013,0.368523,0.379444,-1.85277,0.59138,0.509082,-1.80811,0.594081,0.439517,-1.74723,0.079401,1.12503,-2.07207,0.058399,1.04468,-2.054,-0.057173,1.16017,-2.03886,-0.073435,1.0805,-2.02125,-0.07543,0.936675,-2.01959,-0.220946,0.809516,-2.0061,-0.267005,0.911782,-2.02418,-0.330209,0.880059,-2.01311,-0.528013,0.755638,-1.97919,-0.590225,0.840435,-1.92577,-0.581189,0.558005,-1.90804,-0.633313,0.596155,-1.97253,0.114018,1.36467,-1.79089,0.104791,1.35433,-2.00568},
/*808*/{0.184275,1.92559,-1.68069,0.282171,1.87555,-1.81505,0.035396,1.81898,-1.62315,-0.077044,1.7323,-1.59836,0.062493,1.64923,-1.60556,0.136302,1.59218,-1.58941,0.243041,1.95471,-2.05725,0.295224,1.87886,-1.88791,0.100104,1.91285,-2.0028,0.114654,1.74744,-2.18582,0.163635,1.6588,-2.22832,0.313643,1.64059,-2.09797,0.372991,1.62106,-2.08671,0.196431,1.44504,-1.78847,0.01191,1.51371,-1.84651,0.013959,1.5374,-1.91171,0.01859,1.51286,-1.96099,0.198359,1.42769,-2.04146,0.268759,1.16089,-1.71815,0.146487,1.09865,-1.73298,0.179673,1.02586,-1.75584,0.288086,0.947012,-1.7427,0.381052,0.78586,-1.73364,0.406924,0.716126,-1.74514,0.275978,0.74756,-1.75047,0.30077,0.680349,-1.75826,0.42915,0.475066,-1.7699,0.369607,0.376203,-1.85271,0.590877,0.506523,-1.80695,0.594215,0.43829,-1.74637,0.084559,1.12104,-2.07102,0.065659,1.04072,-2.05277,-0.052027,1.15423,-2.03747,-0.067651,1.0743,-2.02034,-0.068317,0.932106,-2.0194,-0.217939,0.811151,-2.00571,-0.259274,0.915468,-2.02384,-0.324265,0.886699,-2.01203,-0.529045,0.770991,-1.97871,-0.585982,0.857871,-1.92443,-0.587746,0.574977,-1.90837,-0.638708,0.616153,-1.97277,0.114118,1.36159,-1.79127,0.105747,1.35203,-2.00613},
/*809*/{0.185828,1.92336,-1.67945,0.282082,1.87189,-1.81506,0.037128,1.81422,-1.6211,-0.07271,1.72664,-1.59505,0.066491,1.6441,-1.60214,0.139778,1.58749,-1.58667,0.241184,1.95253,-2.05686,0.294217,1.87687,-1.88776,0.099228,1.91063,-2.00154,0.110792,1.74646,-2.18346,0.159323,1.65727,-2.22607,0.310758,1.63359,-2.09886,0.371194,1.61348,-2.08869,0.197679,1.4423,-1.78737,0.011267,1.51137,-1.84669,0.014233,1.53425,-1.91148,0.01904,1.50967,-1.96071,0.198225,1.42611,-2.04135,0.267403,1.15546,-1.71554,0.143865,1.09822,-1.73324,0.176886,1.02236,-1.75622,0.286387,0.944332,-1.74212,0.378178,0.78295,-1.73244,0.403751,0.711883,-1.74467,0.273418,0.742992,-1.7505,0.299617,0.675748,-1.756,0.427706,0.472953,-1.77101,0.36892,0.373225,-1.85272,0.590263,0.504489,-1.80739,0.59372,0.436426,-1.74673,0.090176,1.11786,-2.071,0.073058,1.03631,-2.05146,-0.046911,1.14793,-2.03647,-0.061649,1.0678,-2.0194,-0.059563,0.925825,-2.01842,-0.213679,0.813214,-2.00547,-0.251582,0.919235,-2.02326,-0.316524,0.893941,-2.01173,-0.526569,0.787401,-1.98157,-0.579232,0.875329,-1.92438,-0.593602,0.593636,-1.91022,-0.643475,0.63628,-1.97281,0.114869,1.35897,-1.7911,0.106551,1.34962,-2.00597},
/*810*/{0.18664,1.92027,-1.67892,0.282905,1.86989,-1.81506,0.039343,1.80996,-1.6181,-0.069774,1.72082,-1.59176,0.070196,1.63822,-1.5988,0.14396,1.58356,-1.58374,0.240148,1.95017,-2.0565,0.294508,1.87476,-1.88722,0.098465,1.90819,-1.99938,0.106845,1.74488,-2.1815,0.155274,1.65532,-2.22453,0.308921,1.62715,-2.10038,0.36899,1.60512,-2.09169,0.198736,1.43916,-1.78718,0.011305,1.50753,-1.84611,0.014404,1.53098,-1.91119,0.018821,1.50688,-1.96165,0.199295,1.42404,-2.04121,0.264254,1.15068,-1.71252,0.141593,1.09451,-1.73323,0.174342,1.01937,-1.75634,0.286014,0.942824,-1.74136,0.375381,0.7788,-1.73209,0.401845,0.708199,-1.74428,0.270316,0.738455,-1.7506,0.29578,0.67242,-1.75781,0.423554,0.468355,-1.77095,0.366572,0.36963,-1.8541,0.58795,0.501733,-1.80749,0.59037,0.431561,-1.74599,0.094862,1.11366,-2.07013,0.079716,1.03176,-2.05014,-0.043342,1.14168,-2.03652,-0.055116,1.06171,-2.01826,-0.050659,0.920142,-2.01766,-0.208596,0.81536,-2.00545,-0.242436,0.921905,-2.02151,-0.308361,0.898055,-2.01142,-0.520177,0.801176,-1.98057,-0.571013,0.891696,-1.92339,-0.599476,0.610542,-1.91115,-0.643185,0.656369,-1.97258,0.115764,1.35556,-1.79158,0.107766,1.34712,-2.0065},
/*811*/{0.18769,1.91726,-1.6778,0.282205,1.8665,-1.81439,0.041453,1.80541,-1.61548,-0.065426,1.71496,-1.58772,0.074377,1.63395,-1.59563,0.148103,1.57848,-1.58114,0.239161,1.9472,-2.05647,0.29457,1.87171,-1.88735,0.097217,1.90596,-1.99852,0.102336,1.74273,-2.17897,0.150485,1.65287,-2.22391,0.306093,1.62128,-2.10228,0.366449,1.59616,-2.09536,0.196724,1.43517,-1.78703,0.010325,1.50382,-1.84665,0.014612,1.52801,-1.91102,0.019099,1.50293,-1.96142,0.199992,1.42156,-2.04078,0.261613,1.14755,-1.71051,0.138637,1.09145,-1.73387,0.172583,1.01616,-1.75794,0.286703,0.941635,-1.74107,0.372691,0.773876,-1.7322,0.398885,0.70355,-1.74395,0.268118,0.735378,-1.7505,0.292716,0.668707,-1.75829,0.421411,0.463698,-1.77113,0.361751,0.365762,-1.85556,0.584076,0.495781,-1.80719,0.585815,0.425732,-1.74762,0.100866,1.10929,-2.06937,0.086322,1.02758,-2.04892,-0.038562,1.13518,-2.03528,-0.04866,1.05477,-2.0174,-0.043201,0.914661,-2.01714,-0.203613,0.816379,-2.00561,-0.232495,0.925453,-2.02216,-0.300423,0.902724,-2.00996,-0.514954,0.81584,-1.98151,-0.562594,0.90713,-1.92178,-0.602094,0.626781,-1.91315,-0.647921,0.675593,-1.97284,0.114518,1.35143,-1.79205,0.108144,1.34428,-2.00707},
/*812*/{0.188491,1.91336,-1.67755,0.281822,1.86519,-1.81488,0.04432,1.80054,-1.61273,-0.061676,1.70882,-1.58362,0.078645,1.62843,-1.59207,0.153338,1.57379,-1.57775,0.236975,1.94419,-2.05628,0.293513,1.86892,-1.88769,0.095887,1.90276,-1.99706,0.098184,1.74052,-2.17712,0.145971,1.65003,-2.22191,0.303044,1.61307,-2.10368,0.363331,1.58736,-2.09888,0.198311,1.43237,-1.78643,0.011053,1.50018,-1.84607,0.014065,1.5245,-1.91105,0.018585,1.49949,-1.96198,0.200343,1.41911,-2.04039,0.260736,1.14466,-1.70864,0.137299,1.08894,-1.73393,0.172059,1.01354,-1.75733,0.28701,0.940035,-1.74049,0.369884,0.768902,-1.73158,0.394402,0.699123,-1.74465,0.264618,0.730997,-1.75048,0.289239,0.663204,-1.75704,0.414496,0.458558,-1.7714,0.35482,0.361729,-1.85641,0.578581,0.488076,-1.80669,0.580481,0.418528,-1.74767,0.10514,1.10561,-2.06857,0.093831,1.02352,-2.0478,-0.033074,1.12861,-2.03481,-0.041911,1.04798,-2.01681,-0.034498,0.90896,-2.01675,-0.197982,0.817221,-2.00509,-0.22206,0.92616,-2.01936,-0.29102,0.907499,-2.00983,-0.510133,0.829378,-1.98041,-0.551823,0.92245,-1.92125,-0.602939,0.643569,-1.91449,-0.645107,0.695389,-1.97392,0.115844,1.3485,-1.79183,0.109138,1.34119,-2.00683},
/*813*/{0.188561,1.90973,-1.6765,0.281957,1.86033,-1.81418,0.046613,1.79469,-1.61033,-0.056694,1.70196,-1.57945,0.083419,1.62315,-1.58808,0.158654,1.56811,-1.57521,0.234953,1.94092,-2.05651,0.293726,1.86622,-1.88756,0.09526,1.89953,-1.99512,0.094173,1.73729,-2.17504,0.141208,1.64697,-2.22052,0.300141,1.60673,-2.10571,0.359704,1.57799,-2.10262,0.197839,1.42836,-1.78555,0.011432,1.49654,-1.84624,0.013518,1.52061,-1.9111,0.018147,1.4948,-1.96201,0.200376,1.4162,-2.03933,0.258622,1.1412,-1.70748,0.136176,1.08638,-1.73529,0.169556,1.01096,-1.75749,0.286985,0.938059,-1.74078,0.366139,0.763606,-1.73174,0.390794,0.692483,-1.74396,0.260858,0.726735,-1.74946,0.284017,0.659368,-1.75718,0.407537,0.451926,-1.77119,0.347386,0.35727,-1.85783,0.570867,0.480412,-1.80746,0.572784,0.409827,-1.74791,0.111395,1.10167,-2.06739,0.100859,1.01947,-2.04732,-0.027806,1.12144,-2.03437,-0.03574,1.04121,-2.01633,-0.026337,0.903009,-2.01616,-0.190727,0.816443,-2.0021,-0.212476,0.92786,-2.01913,-0.280892,0.911502,-2.00977,-0.50117,0.840799,-1.98154,-0.540161,0.936386,-1.92079,-0.601543,0.659371,-1.9158,-0.645086,0.714238,-1.97519,0.115596,1.34465,-1.7915,0.109607,1.33749,-2.00653},
/*814*/{0.189318,1.90536,-1.67594,0.28225,1.85669,-1.81368,0.049615,1.78866,-1.60701,-0.052905,1.69499,-1.57405,0.0877,1.61708,-1.58434,0.163083,1.56371,-1.57238,0.233564,1.93747,-2.0565,0.293572,1.8628,-1.88747,0.093715,1.89555,-1.99485,0.088461,1.7363,-2.17475,0.136446,1.64368,-2.21959,0.296374,1.59797,-2.10834,0.355832,1.56789,-2.10659,0.195612,1.42521,-1.78545,0.00996,1.49221,-1.84675,0.014368,1.51661,-1.91064,0.01866,1.49248,-1.96294,0.201289,1.4132,-2.03871,0.255954,1.13876,-1.70687,0.135943,1.08333,-1.73525,0.169605,1.00788,-1.75722,0.285494,0.933246,-1.7407,0.36266,0.75825,-1.73219,0.385248,0.68654,-1.74512,0.256282,0.722195,-1.74917,0.279507,0.654694,-1.75662,0.402384,0.448308,-1.77326,0.337696,0.352889,-1.85805,0.563571,0.471479,-1.80605,0.566557,0.402868,-1.74608,0.117404,1.0979,-2.06691,0.108198,1.01494,-2.04681,-0.022068,1.11461,-2.03398,-0.028228,1.03367,-2.01598,-0.018247,0.89787,-2.01592,-0.184702,0.818122,-2.00462,-0.201357,0.929847,-2.01916,-0.270812,0.914957,-2.00955,-0.493347,0.852827,-1.98204,-0.528627,0.948384,-1.92076,-0.599,0.673869,-1.9176,-0.641003,0.730255,-1.97725,0.114242,1.34068,-1.79197,0.11031,1.33467,-2.00709},
/*815*/{0.190443,1.90057,-1.67465,0.282406,1.85404,-1.81414,0.051905,1.78256,-1.60434,-0.048944,1.68726,-1.56973,0.093085,1.61056,-1.57971,0.169476,1.55882,-1.5703,0.231678,1.93366,-2.05693,0.293225,1.85894,-1.88782,0.092414,1.89114,-1.99229,0.08409,1.73286,-2.17301,0.131322,1.63989,-2.21797,0.293101,1.59051,-2.11154,0.35162,1.55802,-2.11082,0.193117,1.42074,-1.78314,0.010075,1.48714,-1.84663,0.013678,1.51287,-1.91039,0.018346,1.48562,-1.96289,0.201007,1.41007,-2.03805,0.256324,1.13577,-1.70592,0.133994,1.0805,-1.7366,0.168054,1.00455,-1.75843,0.283777,0.927716,-1.74041,0.358265,0.752348,-1.7324,0.38131,0.680428,-1.7445,0.251566,0.718156,-1.74894,0.27326,0.65018,-1.75598,0.389845,0.441862,-1.77315,0.326593,0.348561,-1.85948,0.555324,0.463716,-1.80637,0.553894,0.394104,-1.7467,0.12301,1.09347,-2.06775,0.116217,1.01069,-2.04706,-0.017129,1.10811,-2.03398,-0.022199,1.02647,-2.01465,-0.009237,0.892737,-2.01563,-0.177172,0.817776,-2.0044,-0.191154,0.929856,-2.01857,-0.259773,0.917977,-2.00978,-0.483983,0.862942,-1.98191,-0.515926,0.959507,-1.92081,-0.595084,0.687292,-1.91948,-0.635863,0.745215,-1.97778,0.113212,1.33596,-1.7913,0.110856,1.33026,-2.00644},
/*816*/{0.191029,1.89576,-1.67443,0.281672,1.84838,-1.81423,0.054828,1.77611,-1.60098,-0.04477,1.67986,-1.5652,0.097414,1.60422,-1.57591,0.174975,1.55367,-1.56714,0.229999,1.92919,-2.05713,0.293427,1.85559,-1.88796,0.091224,1.88735,-1.99158,0.081026,1.72795,-2.17142,0.126024,1.63659,-2.21713,0.288527,1.58124,-2.11458,0.346968,1.54741,-2.11576,0.195583,1.4173,-1.78373,0.009846,1.48256,-1.84613,0.013241,1.50878,-1.91051,0.018228,1.48051,-1.96364,0.201397,1.40657,-2.03712,0.256237,1.13262,-1.70485,0.133465,1.0773,-1.73733,0.167753,1.00218,-1.75831,0.280866,0.922817,-1.74024,0.354132,0.746104,-1.73217,0.373715,0.674508,-1.7457,0.24632,0.71425,-1.7489,0.266276,0.644903,-1.75673,0.377872,0.43625,-1.77305,0.310587,0.350531,-1.85916,0.543757,0.453456,-1.80569,0.54071,0.382135,-1.74886,0.129018,1.08905,-2.06784,0.123104,1.00585,-2.04757,-0.01082,1.10154,-2.03345,-0.014605,1.02007,-2.01418,-0.000547,0.887286,-2.01553,-0.169997,0.817304,-2.00425,-0.179083,0.92881,-2.01882,-0.24969,0.920807,-2.01075,-0.474872,0.872558,-1.98205,-0.503309,0.970134,-1.92021,-0.590459,0.700584,-1.92128,-0.63105,0.760126,-1.9791,0.115088,1.33237,-1.79103,0.112086,1.32604,-2.00615},
/*817*/{0.192214,1.89014,-1.67324,0.282598,1.84468,-1.81397,0.058775,1.76822,-1.59786,-0.040919,1.67155,-1.55959,0.103427,1.59849,-1.57229,0.181166,1.5484,-1.56436,0.228,1.92482,-2.05778,0.293182,1.8516,-1.88774,0.090389,1.88263,-1.99084,0.077563,1.72391,-2.17039,0.120704,1.63253,-2.21605,0.284102,1.57264,-2.11751,0.341591,1.53682,-2.12063,0.1948,1.41396,-1.782,0.008594,1.47849,-1.84644,0.013069,1.50398,-1.9094,0.017692,1.47492,-1.96354,0.201041,1.40307,-2.03671,0.253895,1.12866,-1.70427,0.132557,1.07313,-1.73804,0.166721,0.997894,-1.75851,0.278963,0.917125,-1.73972,0.347714,0.739765,-1.73168,0.366732,0.667922,-1.74586,0.239857,0.709631,-1.74949,0.259245,0.640578,-1.75668,0.365589,0.431818,-1.77325,0.292854,0.353344,-1.86039,0.529736,0.439561,-1.80536,0.522267,0.369107,-1.74795,0.134645,1.08489,-2.06807,0.130358,1.00173,-2.04793,-0.005104,1.09494,-2.03323,-0.006195,1.01317,-2.01396,0.008246,0.882623,-2.01601,-0.162195,0.816585,-2.0043,-0.168246,0.929805,-2.01863,-0.237805,0.921676,-2.01019,-0.463065,0.881761,-1.98214,-0.49073,0.979712,-1.9207,-0.583938,0.711489,-1.92429,-0.620917,0.771211,-1.98128,0.114965,1.32859,-1.79003,0.112564,1.32169,-2.00514},
/*818*/{0.192153,1.88495,-1.67328,0.281987,1.84098,-1.8147,0.061889,1.76169,-1.59417,-0.03645,1.66336,-1.5547,0.108994,1.59176,-1.56829,0.188109,1.54321,-1.5614,0.226226,1.92047,-2.05835,0.29309,1.84647,-1.88812,0.088905,1.87707,-1.98941,0.073727,1.71977,-2.16961,0.115064,1.62864,-2.21439,0.279604,1.56411,-2.12089,0.335789,1.5253,-2.12526,0.196911,1.40913,-1.78301,0.008446,1.47273,-1.84692,0.011932,1.49897,-1.90923,0.016794,1.46977,-1.96346,0.20113,1.39929,-2.03659,0.254873,1.12527,-1.70423,0.133376,1.06877,-1.73821,0.166458,0.994519,-1.75866,0.276195,0.912851,-1.73955,0.341769,0.732713,-1.73218,0.360317,0.660211,-1.7449,0.232969,0.705984,-1.74975,0.25124,0.637123,-1.75693,0.350936,0.425945,-1.77451,0.274786,0.353646,-1.86264,0.516567,0.425833,-1.80563,0.50152,0.356745,-1.74651,0.139875,1.07994,-2.06818,0.137836,0.997599,-2.04862,0.000534,1.08701,-2.03096,0.000465,1.00683,-2.0144,0.018456,0.876927,-2.01618,-0.153473,0.815839,-2.0048,-0.155665,0.929652,-2.02036,-0.226642,0.923183,-2.01069,-0.452652,0.888734,-1.9821,-0.478633,0.98773,-1.92162,-0.577123,0.721095,-1.92626,-0.611729,0.782359,-1.98367,0.116504,1.32362,-1.7907,0.113217,1.31724,-2.00581},
/*819*/{0.193888,1.87971,-1.67169,0.282885,1.83501,-1.81408,0.065739,1.75398,-1.59086,-0.031268,1.65553,-1.54911,0.114953,1.5851,-1.56485,0.1948,1.53853,-1.55941,0.22434,1.91547,-2.05889,0.293347,1.84236,-1.88857,0.087406,1.8724,-1.98785,0.068957,1.71581,-2.16902,0.109978,1.62474,-2.21385,0.274314,1.55511,-2.12494,0.329023,1.51415,-2.13036,0.196954,1.40535,-1.77933,0.007698,1.46772,-1.84724,0.011448,1.49355,-1.90856,0.016433,1.46417,-1.96379,0.202135,1.39531,-2.0331,0.253742,1.12007,-1.7043,0.132009,1.0654,-1.73817,0.166028,0.990139,-1.75813,0.272761,0.907675,-1.73863,0.334845,0.726807,-1.73187,0.353049,0.654891,-1.74555,0.225976,0.701505,-1.75032,0.242354,0.632636,-1.75787,0.335724,0.419883,-1.77563,0.2558,0.355547,-1.8655,0.501845,0.411767,-1.80559,0.480171,0.343582,-1.74591,0.146492,1.07639,-2.06824,0.144541,0.99301,-2.04848,0.005549,1.08232,-2.03033,0.007171,1.00032,-2.01361,0.027879,0.871545,-2.01639,-0.144105,0.814983,-2.0054,-0.143901,0.928648,-2.01895,-0.213331,0.924389,-2.01047,-0.441528,0.896841,-1.98312,-0.466265,0.9958,-1.92144,-0.567716,0.729317,-1.92792,-0.601211,0.791542,-1.98567,0.116217,1.3196,-1.78842,0.115258,1.3123,-2.00353},
/*820*/{0.194855,1.87399,-1.67105,0.283614,1.82994,-1.81401,0.069917,1.74629,-1.58718,-0.02348,1.64748,-1.54328,0.121711,1.57903,-1.56094,0.201468,1.53366,-1.55713,0.223158,1.91085,-2.05906,0.292838,1.83781,-1.88944,0.086573,1.86703,-1.98677,0.065624,1.71173,-2.16797,0.103581,1.62035,-2.21341,0.269312,1.54542,-2.12901,0.321973,1.50204,-2.13581,0.195584,1.40212,-1.77772,0.006309,1.46282,-1.84733,0.011079,1.48786,-1.9076,0.015175,1.45887,-1.9641,0.201658,1.39053,-2.03289,0.254538,1.1161,-1.70379,0.131487,1.0613,-1.73858,0.165409,0.986112,-1.75769,0.269262,0.903202,-1.73791,0.328803,0.719868,-1.73259,0.343877,0.647103,-1.74574,0.218692,0.699053,-1.75061,0.233808,0.628514,-1.75889,0.321091,0.419223,-1.77932,0.234648,0.355897,-1.86539,0.485808,0.39984,-1.80729,0.461157,0.336543,-1.74594,0.151757,1.07141,-2.06839,0.151843,0.988489,-2.04866,0.010749,1.07658,-2.03074,0.014852,0.993433,-2.01337,0.038528,0.865031,-2.01655,-0.134421,0.813699,-2.00629,-0.131299,0.927141,-2.01845,-0.201299,0.923774,-2.00959,-0.429333,0.902278,-1.98368,-0.454076,1.00178,-1.92124,-0.556884,0.736712,-1.92998,-0.59036,0.798235,-1.98762,0.115691,1.31559,-1.78725,0.115275,1.3072,-2.00232},
/*821*/{0.195397,1.86892,-1.67055,0.283417,1.82642,-1.81494,0.074118,1.7381,-1.58409,-0.018539,1.63871,-1.5389,0.128796,1.57284,-1.55781,0.208534,1.52876,-1.55505,0.222224,1.90644,-2.05964,0.292602,1.83301,-1.88996,0.085551,1.86141,-1.98591,0.061496,1.70765,-2.16731,0.097169,1.61632,-2.21328,0.263048,1.53572,-2.13307,0.31461,1.49039,-2.14055,0.194934,1.39764,-1.77652,0.004792,1.45805,-1.84777,0.009089,1.48194,-1.90697,0.01408,1.45342,-1.9637,0.200652,1.38614,-2.03238,0.25287,1.11122,-1.70353,0.130072,1.05697,-1.73768,0.16376,0.982118,-1.75833,0.265278,0.897641,-1.73738,0.320912,0.714406,-1.73363,0.333862,0.64068,-1.74776,0.210913,0.697233,-1.75142,0.222529,0.626302,-1.76009,0.301523,0.418103,-1.78255,0.212633,0.356599,-1.86453,0.466247,0.395482,-1.81388,0.440147,0.338322,-1.74758,0.155946,1.06843,-2.06935,0.158469,0.984583,-2.04969,0.01734,1.06712,-2.03021,0.021862,0.986807,-2.01341,0.048719,0.86002,-2.01906,-0.123938,0.812205,-2.00663,-0.117916,0.926115,-2.02031,-0.188626,0.924709,-2.01048,-0.420129,0.908012,-1.98202,-0.440716,1.00601,-1.92018,-0.544893,0.742964,-1.93225,-0.577176,0.805348,-1.99029,0.115032,1.31092,-1.78659,0.114902,1.30224,-2.00165},
/*822*/{0.196835,1.8638,-1.66936,0.284482,1.8194,-1.81501,0.078892,1.7295,-1.58088,-0.012693,1.62953,-1.5325,0.136257,1.56649,-1.55451,0.21613,1.52442,-1