/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file RTModules.cxx
    \author Romain BONDUE
    \date 21/08/2005 */
#include <sstream>

#include "RTModules.h"
#include "GtkGUI.h"
#include "RTDrivers.h"
#include "RTDriversRoot.h"
#include "ErrorsCode.h"



namespace
{
    ::GtkWidget* LabelWidget (::GtkWidget* pWidget, const char* Label) throw()
    {
        ::GtkBox* const pHBox (GTK_BOX (::gtk_hbox_new (false, 2)));
        ::gtk_box_pack_start_defaults (pHBox, ::gtk_label_new (Label));
        ::gtk_box_pack_start_defaults (pHBox, pWidget);

        return GTK_WIDGET (pHBox);
        
    } // LabelWidget()


    const unsigned TabAuthAdHocRTxxSize (3);
    const nsWireless::AuthType_e TabAuthAdHocRTxx [TabAuthAdHocRTxxSize] =
                {nsWireless::Open, nsWireless::Shared, nsWireless::WPANONE};

    const unsigned TabNoneEncryptSize (2);
    const nsWireless::EncryptType_e TabNoneEncrypt [TabNoneEncryptSize] =
                                        {nsWireless::None, nsWireless::WEP};

    const unsigned TabSharedEncryptSize (1);
    const nsWireless::EncryptType_e TabSharedEncrypt [TabSharedEncryptSize] =
                                                            {nsWireless::WEP};

    const unsigned TabWPAPSKEncryptSize (2);
    const nsWireless::EncryptType_e TabWPAPSKEncrypt [TabWPAPSKEncryptSize] =
                                            {nsWireless::TKIP, nsWireless::AES};

    const unsigned TabAuthRT2500Size (3);
    const unsigned TabAuthRT2400Size (2);
    const unsigned TabAuthRTxxSize (4);
    const nsWireless::AuthType_e TabAuthRTxx [TabAuthRTxxSize] =
                                    {nsWireless::Open, nsWireless::Shared,
                                     nsWireless::WPAPSK, nsWireless::WPA2PSK};

    const unsigned TabModeRTxxSize (3);
    const nsWireless::Mode_e TabModeRTxx [TabModeRTxxSize] =
                {nsWireless::Managed, nsWireless::AdHoc, nsWireless::Monitor};


    void ApplyRT2400Profile (const nsUserData::CProfile& Profile,
                             nsWireless::CRT2400Driver& Driver)
                                                    throw (nsErrors::CSystemExc)
    {
        Driver.SetChannel (Profile.GetChannel());
        Driver.SetMode (Profile.GetMode());
        Driver.SetEncryption (Profile.GetEncryptionD());
        Driver.SetSSID (Profile.GetSSID());  

    } // ApplyRT2400Profile()


    void ApplyRTxxProfile (const nsUserData::CProfile& Profile,
                             nsWireless::IDriver& Driver)
                                                    throw (nsErrors::CSystemExc)
    {
        Driver.SetMode (Profile.GetMode());
        Driver.SetChannel (Profile.GetChannel());
        Driver.SetSSID (Profile.GetSSID());  
        Driver.SetEncryption (Profile.GetEncryptionD());
        Driver.SetSSID (Profile.GetSSID());  

    } // ApplyRTxxProfile()


    void DisplayDriverLimitations (const nsWireless::AuthType_e* Auths,
                        unsigned NAuth, const nsWireless::CCell& Cell,
                                                ::GtkWindow* pWindow) throw()
    {
        const nsWireless::AuthType_e DesiredAuth
                                            (Cell.GetEncryptionD().GetAuth());
        while (NAuth)
            if (Auths[--NAuth] == DesiredAuth)
                return;
        ::GtkWidget* const pDialog (::gtk_message_dialog_new (pWindow,
                            GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING,
                            GTK_BUTTONS_OK, "The device cannot be configured "
                            " for this network because %s is unsupported, this"
                            " could be a driver limitation .",
                            nsWireless::GetAuthName (DesiredAuth).c_str()));
        ::gtk_dialog_run (GTK_DIALOG (pDialog));
        ::gtk_widget_destroy (pDialog);

    } // DisplayDriverLimitations()

} // anonymous namespace



const char* const nsCore::CRT2400Module::NewProfileDefaultName ("New profile");


const char* const nsCore::CRTAdvancedModule::NewProfileDefaultName
                                                                ("New profile");


const char* const nsCore::CRT2570Module::NewProfileDefaultName ("New profile");



void nsCore::CRT2400CLIModule::ApplyProfile (unsigned Pos)
                                                    throw (nsErrors::CSystemExc)
{
    ApplyRT2400Profile (m_ProfilesFile.GetProfile (Pos), m_Driver);

} // ApplyProfile()


nsGUI::CRT2500ConfigDialog::CRT2500ConfigDialog () throw()
    : CSharedConfigDialog (TabModeRTxx, TabModeRTxxSize,
                           TabAuthRTxx, TabAuthRT2500Size,
                           TabNoneEncrypt, TabNoneEncryptSize,
                           TabSharedEncrypt, TabSharedEncryptSize,
                           TabWPAPSKEncrypt, TabWPAPSKEncryptSize)
{
    ::g_signal_connect (G_OBJECT (GetModeBox().operator ::GtkWidget*()),
                        "changed", G_CALLBACK (ModeChangedCallBack), this);

} // CRT2500ConfigDialog


void nsGUI::CRT2500ConfigDialog::ModeChangedCallBack (::GtkToggleButton*,
                                          CRT2500ConfigDialog* This) throw()
{
    if (This->GetModeBox().GetActive() == nsWireless::AdHoc)
    {
        This->GetAuthBox().SetElem (TabAuthAdHocRTxx,
                                    TabAuthAdHocRTxxSize);
        This->GetAuthBox().SetActive (TabAuthAdHocRTxx [0]);
    }
    else
    {
        This->GetAuthBox().SetElem (TabAuthRTxx, TabAuthRT2500Size);
        This->GetAuthBox().SetActive (TabAuthRTxx [0]);
    }

} // ModeChangedCallBack()


nsGUI::CRTxxConfigDialog::CRTxxConfigDialog () throw()
    : CSharedConfigDialog (TabModeRTxx, TabModeRTxxSize,
                           TabAuthRTxx, TabAuthRTxxSize,
                           TabNoneEncrypt, TabNoneEncryptSize,
                           TabSharedEncrypt, TabSharedEncryptSize,
                           TabWPAPSKEncrypt, TabWPAPSKEncryptSize)
{
    ::g_signal_connect (G_OBJECT (GetModeBox().operator ::GtkWidget*()),
                        "changed", G_CALLBACK (ModeChangedCallBack), this);

} // CRTxxConfigDialog


void nsGUI::CRTxxConfigDialog::ModeChangedCallBack (::GtkToggleButton*,
                                              CRTxxConfigDialog* This) throw()
{
    if (This->GetModeBox().GetActive() == nsWireless::AdHoc)
    {
        This->GetAuthBox().SetElem (TabAuthAdHocRTxx,
                                    TabAuthAdHocRTxxSize);
        This->GetAuthBox().SetActive (TabAuthAdHocRTxx [0]);
    }
    else
    {
        This->GetAuthBox().SetElem (TabAuthRTxx, TabAuthRTxxSize);
        This->GetAuthBox().SetActive (TabAuthRTxx [0]);
    }

} // ModeChangedCallBack()


nsCore::CRT2400Module::CRT2400Module (const std::string& DeviceName,
                                      const std::string& Path)
                                 throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CRT2400Driver (DeviceName)),
      m_ProfileEditor (new nsGUI::CSharedConfigDialog
                                      (TabModeRTxx, TabModeRTxxSize,
                                       m_pDriver->GetSupportedFreq(),
                                       TabAuthRTxx, TabAuthRT2400Size,
                                       TabNoneEncrypt, TabNoneEncryptSize,
                                       TabSharedEncrypt, TabSharedEncryptSize,
                                       0, 0))
{
    m_ProfilesFile.Extract();

} // CRT2400Module()


nsUserData::CProfile* nsCore::CRT2400Module::NewProfile
              (::GtkWindow* pMainWindow, const nsWireless::CCell& Cell) throw()
{
    DisplayDriverLimitations (TabAuthRTxx, TabAuthRT2400Size, Cell,
                                                                pMainWindow);
    m_ProfilesFile.NewProfile (NewProfileDefaultName, Cell);
    return EditNewProfile (pMainWindow);

} // NewProfile()


void nsCore::CRT2400Module::BecomeRoot () throw (nsErrors::CException,
                                                 std::bad_alloc)
{
    m_pDriver.reset (new nsWireless::CRT2400DriverRoot
                                                (m_pDriver->GetDeviceName()));

} // BecomeRoot()


void nsCore::CRT2400Module::RecordProfiles () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // RecordProfiles()


void nsCore::CRT2400Module::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    ApplyRT2400Profile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);

} // ApplyProfile()


void nsCore::CRT2400Module::ApplyProfile (unsigned Pos,
            const nsWireless::CMacAddress& Address) throw (nsErrors::CSystemExc)
{
    ApplyRT2400Profile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);
    try {m_pDriver->SetAPMacAddr (Address);}
    catch (const nsErrors::CSystemExc&) {} // May fail.

} // ApplyProfile()


nsUserData::CProfile* nsCore::CRT2400Module::EditNewProfile
                                             (::GtkWindow* pMainWindow) throw()
{
    nsUserData::CProfile* const pProfile (&m_ProfilesFile.GetProfile
                                                (m_ProfilesFile.Size() - 1U));
    if (!m_ProfileEditor.Edit (pMainWindow, *pProfile))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return pProfile;

} // EditNewProfile()


void nsCore::CRTAdvancedCLIModule::ApplyProfile (unsigned Pos)
                                                    throw (nsErrors::CSystemExc)
{
    ApplyRTxxProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);

} // ApplyProfile()


nsCore::CRTAdvancedModule::CRTAdvancedModule
        (nsWireless::CRTAdvancedDriver* pDriver, nsGUI::CConfigDialog* pDialog,
         const std::string& Path, nsGUI::CGtkGUI* pUI, bool AdHofdmSupport)
                                  throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (pDriver), m_ProfileEditor (pDialog),
      m_pTable (GTK_TABLE (::gtk_table_new (5, 2, true))), m_pUI (pUI),
      m_CurrentActiveBoxItem (0), m_IsSkipped (false)
{
    m_ProfilesFile.Extract();
        // Build the private tab :
    ::gtk_container_set_border_width (GTK_CONTAINER (m_pTable), 6);
    ::gtk_table_set_col_spacings (m_pTable, 12);
    ::gtk_table_set_row_spacings (m_pTable, 8);

    const unsigned NbMode (2);
    const char* ModeNames [NbMode] = {"Mixed", "B Only"};
    CreateCBox (ModeNames, NbMode, m_pDriver->GetWirelessMode(),
                "Wireless Mode", 0, &ModeCallBack);

    const unsigned NbProtectionMode (3);
    const char* ProtectionModeNames [NbProtectionMode] = {"Auto", "On", "Off"};
    CreateCBox (ProtectionModeNames, NbProtectionMode,
                m_pDriver->GetBGProtection(), "B/G Protection", 1,
                &BGProtectionCallBack);

    const unsigned NbTxPreamble (3);
    const char* TxPreambleNames [NbTxPreamble] = {"Long", "Short", "Auto"};
    CreateCBox (TxPreambleNames, NbTxPreamble, m_pDriver->GetTxPreamble(),
                "Tx Preamble", 2, &TxPreambleCallBack);

    CreateCheckButton ("Tx burst", 3, 0, m_pDriver->GetTxBurst(),
                       &TxBurstCallBack);
    CreateCheckButton ("Turbo rate", 3, 1, m_pDriver->GetTurboRate(),
                                 &TurboRateCallBack);
    CreateCheckButton ("Injection in monitor mode", 4, 0,
                       m_pDriver->GetRfmontx(), &RfmontxCallBack);
    if (AdHofdmSupport)
        CreateCheckButton ("OFDM in ad-hoc mode", 4, 1,
                           m_pDriver->GetAdHocOFDM(), &AdHocOFDMCallBack);

} // CRTAdvancedModule()


nsUserData::CProfile* nsCore::CRTAdvancedModule::NewProfile
              (::GtkWindow* pMainWindow, const nsWireless::CCell& Cell) throw()
{
    DisplayDriverLimitations (TabAuthRTxx, TabAuthRTxxSize, Cell, pMainWindow);
    m_ProfilesFile.NewProfile (NewProfileDefaultName, Cell);
    return EditNewProfile (pMainWindow);

} // NewProfile()


void nsCore::CRTAdvancedModule::RecordProfiles () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // RecordProfiles()


void nsCore::CRTAdvancedModule::CreateCBox
        (const char* TabName [], unsigned TabSize, int CurrentActive,
         const char* Label, unsigned Row,
         void (*CallBack) (::GtkComboBox*, CRTAdvancedModule*)) throw()
{

    ::GtkComboBox* const pCBox (GTK_COMBO_BOX (::gtk_combo_box_new_text()));
    for (unsigned i (0) ; i < TabSize ; ++i)
        ::gtk_combo_box_append_text (pCBox, TabName [i]);
    ::gtk_combo_box_set_active (pCBox, CurrentActive);
    ::g_signal_connect (G_OBJECT (pCBox), "changed", G_CALLBACK (CallBack),
                        this);
    ::g_signal_connect (G_OBJECT (pCBox), "set-focus-child",
                        G_CALLBACK (CBoxFocusCallBack), this);
    ::GtkWidget* const pLabel (::gtk_label_new (Label));
    ::gtk_table_attach_defaults (m_pTable, pLabel, 0, 1, Row, Row + 1);
    ::gtk_table_attach (m_pTable, GTK_WIDGET (pCBox), 1, 2, Row, Row + 1,
                        GTK_FILL, GTK_SHRINK, 0, 0);

} // CreateCBox()


void nsCore::CRTAdvancedModule::CreateCheckButton (
         const char* Label, unsigned Row, unsigned Column, bool CurrentState,
         void (*CallBack) (::GtkToggleButton*, CRTAdvancedModule*)) throw()
{
    ::GtkWidget* const pButton (::gtk_check_button_new_with_label (Label));
    ::gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pButton), CurrentState);
    ::g_signal_connect (G_OBJECT (pButton), "toggled", G_CALLBACK (CallBack),
                        this);
    ::gtk_table_attach_defaults (m_pTable, pButton, Column, Column + 1, Row,
                                 Row + 1);

} // CreateCheckButton()


nsUserData::CProfile* nsCore::CRTAdvancedModule::EditNewProfile
                                             (::GtkWindow* pMainWindow) throw()
{
    nsUserData::CProfile* const pProfile (&m_ProfilesFile.GetProfile
                                                (m_ProfilesFile.Size() - 1U));
    if (!m_ProfileEditor.Edit (pMainWindow, *pProfile))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return pProfile;

} // EditNewProfile()


void nsCore::CRTAdvancedModule::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    ApplyRTxxProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);

} // ApplyProfile()


void nsCore::CRTAdvancedModule::ApplyProfile (unsigned Pos,
            const nsWireless::CMacAddress& Address) throw (nsErrors::CSystemExc)
{
    ApplyRTxxProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);
    try {m_pDriver->SetAPMacAddr (Address);}
    catch (const nsErrors::CSystemExc&) {} // May fail.

} // ApplyProfile()


void nsCore::CRTAdvancedModule::ModeCallBack (::GtkComboBox* pCBox,
                                             CRTAdvancedModule* This) throw()
{
    This->CommonCBoxCallBack (reinterpret_cast<DriverCBoxMember_t>
                            (&nsWireless::CRTAdvancedDriver::SetWirelessMode),
                              pCBox);
    This->m_pUI->RefreshRates();

} // ModeCallBack()


void nsCore::CRT2500Module::BecomeRoot () throw (nsErrors::CException,
                                                 std::bad_alloc)
{
    SetDriver (new nsWireless::CRT2500DriverRoot (GetDriver().GetDeviceName()));

} // BecomeRoot()


void nsCore::CRT61Module::BecomeRoot () throw (nsErrors::CException,
                                               std::bad_alloc)
{
    SetDriver (new nsWireless::CRT61DriverRoot (GetDriver().GetDeviceName()));

} // BecomeRoot()


void nsCore::CRT73Module::BecomeRoot () throw (nsErrors::CException,
                                               std::bad_alloc)
{
    SetDriver (new nsWireless::CRT73DriverRoot (GetDriver().GetDeviceName()));

} // BecomeRoot()


void nsCore::CRTAdvancedModule::CommonCBoxCallBack (DriverCBoxMember_t Member,
                                                ::GtkComboBox* pCBox) throw()
{
    try
    {
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
            {
                const unsigned CurrentActive
                                        (::gtk_combo_box_get_active (pCBox));
                (m_pDriver.get()->*Member) (CurrentActive);
                m_CurrentActiveBoxItem = CurrentActive;
            }
            else ResumePreviousState (pCBox);
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        ResumePreviousState (pCBox);
        m_pUI->ProcessError (Exc);
    }

} // CommonCBoxCallBack()


void nsCore::CRTAdvancedModule::CommonCButtonCallBack
            (DriverButtonMember_t Member, ::GtkToggleButton* pButton) throw()
{
    try
    {
        const bool NewState (::gtk_toggle_button_get_active (pButton));
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
                (m_pDriver.get()->*Member) (NewState);
            else
            {
                m_IsSkipped = true;
                ::gtk_toggle_button_set_active (pButton, !NewState);
                m_IsSkipped = false;
            }
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        m_pUI->ProcessError (Exc);
    }

} // CommonCButtonCallBack()


void nsCore::CRT2570CLIModule::ApplyProfile (unsigned Pos)
                                                    throw (nsErrors::CSystemExc)
{
    ApplyRTxxProfile (m_ProfilesFile.GetProfile (Pos), m_Driver);

} // ApplyProfile()


nsCore::CRT2570Module::CRT2570Module (const std::string& DeviceName,
                                      const std::string& Path,
                                      nsGUI::CGtkGUI* pUI)
                                throw (nsErrors::CException, std::bad_alloc)
    : m_pDriver (new nsWireless::CRT2570Driver (DeviceName)),
      m_ProfileEditor (new nsGUI::CRTxxConfigDialog()),
      m_pUI (pUI), m_pBox (GTK_BOX (::gtk_vbox_new (true, 2))),
      m_CurrentActiveBoxItem (0), m_IsSkipped (false)
{
    m_ProfileEditor.GetDialog()->SetAvailableChannel
                                                (m_pDriver->GetSupportedFreq());
    m_ProfilesFile.Extract();
    BuildPrivateTab();

} // CRT2570Module()


nsUserData::CProfile* nsCore::CRT2570Module::NewProfile
              (::GtkWindow* pMainWindow, const nsWireless::CCell& Cell) throw()
{
    DisplayDriverLimitations (TabAuthRTxx, TabAuthRTxxSize, Cell, pMainWindow);
    m_ProfilesFile.NewProfile (NewProfileDefaultName, Cell);
    return EditNewProfile (pMainWindow);

} // NewProfile()


void nsCore::CRT2570Module::RecordProfiles () throw (nsErrors::CException)
{
    m_ProfilesFile.Record();

} // RecordProfiles()


void nsCore::CRT2570Module::BecomeRoot () throw (nsErrors::CException,
                                                 std::bad_alloc)
{
    m_pDriver.reset (new nsWireless::CRT2570DriverRoot
                                                (m_pDriver->GetDeviceName()));

} // BecomeRoot()


void nsCore::CRT2570Module::ApplyProfile (unsigned Pos)
                                                throw (nsErrors::CSystemExc)
{
    ApplyRTxxProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);

} // ApplyProfile()


void nsCore::CRT2570Module::ApplyProfile (unsigned Pos,
            const nsWireless::CMacAddress& Address) throw (nsErrors::CSystemExc)
{
    ApplyRTxxProfile (m_ProfilesFile.GetProfile (Pos), *m_pDriver);
    try {m_pDriver->SetAPMacAddr (Address);}
    catch (const nsErrors::CSystemExc&) {} // May fail.

} // ApplyProfile()


nsUserData::CProfile* nsCore::CRT2570Module::EditNewProfile
                                             (::GtkWindow* pMainWindow) throw()
{
    nsUserData::CProfile* const pProfile (&m_ProfilesFile.GetProfile
                                                (m_ProfilesFile.Size() - 1U));
    if (!m_ProfileEditor.Edit (pMainWindow, *pProfile))
    {
        m_ProfilesFile.DeleteLastProfile();
        return 0;
    }
    return pProfile;

} // EditNewProfile()


void nsCore::CRT2570Module::CommonCButtonCallBack
            (DriverButtonMember_t pMember, ::GtkToggleButton* pButton) throw()
{
    try
    {
        const bool NewState (::gtk_toggle_button_get_active (pButton));
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
                (m_pDriver.get()->*pMember) (NewState);
            else
            {
                m_IsSkipped = true;
                ::gtk_toggle_button_set_active (pButton, !NewState);
                m_IsSkipped = false;
            }
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        m_pUI->ProcessError (Exc);
    }

} // CommonCButtonCallBack()


void nsCore::CRT2570Module::CommonCBoxCallBack (DriverBoxMember_t pMember,
                                                ::GtkComboBox* pCBox) throw()
{
    try
    {
        if (!m_IsSkipped)
        {
            if (m_pUI->BecomeRoot())
            {
                const int CurrentActive (::gtk_combo_box_get_active (pCBox));
                (m_pDriver.get()->*pMember) (CurrentActive);
                m_CurrentActiveBoxItem = CurrentActive;
            }
            else ResumePreviousState (pCBox);
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        ResumePreviousState (pCBox);
        m_pUI->ProcessError (Exc);
    }

} // AdhocModeCallBack()


void nsCore::CRT2570Module::BuildPrivateTab () throw()
{
    ::GtkComboBox* pCBox (GTK_COMBO_BOX (::gtk_combo_box_new_text()));
    ::gtk_combo_box_append_text (pCBox, "Standard 11Mpbs");
    ::gtk_combo_box_append_text (pCBox, "Auto");
    ::gtk_combo_box_append_text (pCBox, "54Mbps only");
        // Warning : works because of the numeric values, can easily break.
    ::gtk_combo_box_set_active (pCBox, m_pDriver->GetAdHocMode());
    ::g_signal_connect (G_OBJECT (pCBox), "changed",
                        G_CALLBACK (AdhocModeCallBack), this);
    ::g_signal_connect (G_OBJECT (pCBox), "set-focus-child",
                        G_CALLBACK (CBoxFocusCallBack), this);
    ::GtkBox* pHbox (GTK_BOX (::gtk_hbox_new (true, 2)));
    ::gtk_box_pack_start_defaults (pHbox, ::gtk_label_new ("Ad-hoc mode"));
    ::gtk_box_pack_start_defaults (pHbox, GTK_WIDGET (pCBox));
    ::gtk_box_pack_start_defaults (m_pBox, GTK_WIDGET (pHbox));

    ::GtkWidget* const pButton (::gtk_toggle_button_new_with_label
                                                ("Injection in monitor mode"));
    ::gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (pButton),
                                    m_pDriver->GetRfmontx());
    ::g_signal_connect (G_OBJECT (pButton), "toggled",
                        G_CALLBACK (RfmontxCallBack), this);
    ::gtk_box_pack_start_defaults (m_pBox, pButton);

    pCBox = GTK_COMBO_BOX (::gtk_combo_box_new_text());
    ::gtk_combo_box_append_text (pCBox, "Default (auto)");
    ::gtk_combo_box_append_text (pCBox, "Force");
    ::gtk_combo_box_append_text (pCBox, "Force no");
        // Warning : works because of the numeric values, can easily break.
    ::gtk_combo_box_set_active (pCBox, m_pDriver->GetPrismHeader());
    ::g_signal_connect (G_OBJECT (pCBox), "changed",
                        G_CALLBACK (PrismHeaderCallBack), this);
    ::g_signal_connect (G_OBJECT (pCBox), "set-focus-child",
                        G_CALLBACK (CBoxFocusCallBack), this);
    pHbox = GTK_BOX (::gtk_hbox_new (true, 2));
    ::gtk_box_pack_start_defaults (pHbox, ::gtk_label_new ("Prism header"));
    ::gtk_box_pack_start_defaults (pHbox, GTK_WIDGET (pCBox));
    ::gtk_box_pack_start_defaults (m_pBox, GTK_WIDGET (pHbox));

} // BuildPrivateTab()


nsCore::CRT2500Module::CRT2500Module (const std::string& DeviceName,
                            const std::string& ProfilePath, nsGUI::CGtkGUI* pUI)
                                    throw (nsErrors::CException, std::bad_alloc)
    : CRTAdvancedModule (new nsWireless::CRT2500Driver (DeviceName),
                         new nsGUI::CRT2500ConfigDialog(), ProfilePath, pUI,
                         true)
{
    GetProfileEditor().GetDialog()->SetAvailableChannel
                                            (GetDriver().GetSupportedFreq());

} // CRT2500Module()


nsUserData::CProfile* nsCore::CRT2500Module::NewProfile
              (::GtkWindow* pMainWindow, const nsWireless::CCell& Cell) throw()
{
    DisplayDriverLimitations (TabAuthRTxx, TabAuthRT2500Size, Cell,
                                                                pMainWindow);
    GetProfilesFile().NewProfile (NewProfileDefaultName, Cell);
    return EditNewProfile (pMainWindow);

} // NewProfile()


nsCore::CRT61Module::CRT61Module (const std::string& DeviceName,
                            const std::string& ProfilePath, nsGUI::CGtkGUI* pUI)
                                    throw (nsErrors::CException, std::bad_alloc)
    : CRTAdvancedModule (new nsWireless::CRT61Driver (DeviceName),
                         new nsGUI::CRTxxConfigDialog, ProfilePath, pUI, false)
{
    GetProfileEditor().GetDialog()->SetAvailableChannel
                                            (GetDriver().GetSupportedFreq());

} // CRT61Module()


nsCore::CRT73Module::CRT73Module (const std::string& DeviceName,
                            const std::string& ProfilePath, nsGUI::CGtkGUI* pUI)
                                    throw (nsErrors::CException, std::bad_alloc)
    : CRTAdvancedModule (new nsWireless::CRT73Driver (DeviceName),
                         new nsGUI::CRTxxConfigDialog, ProfilePath, pUI, true)
{
    GetProfileEditor().GetDialog()->SetAvailableChannel
                                            (GetDriver().GetSupportedFreq());

} // CRT73Module()
