/*
*  RAL -- Rubrica Addressbook Library
*  file: address.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n-lib.h>

#include "address.h"
#include "lookup.h"
#include "utils.h"

/* Property enumeration */
enum { 
  ADDRESS_PROP_0,
  ADDRESS_TYPE,
  ADDRESS_STREET, 
  ADDRESS_NUMBER, 
  ADDRESS_CITY, 
  ADDRESS_ZIP, 
  ADDRESS_PROVINCE, 
  ADDRESS_STATE, 
  ADDRESS_COUNTRY 
}; 


struct _RAddressPrivate {
  gint   type;
  gchar* street;
  gchar* number;
  gchar* city; 
  gchar* zip;
  gchar* province;
  gchar* state;
  gchar* country;

  gboolean dispose_has_run;
};


static RLookupTable table[] = {
  {"pref",       N_("preferred"),          R_ADDRESS_PREF      },
  {"home",       N_("residence"),          R_ADDRESS_HOME      },
  {"work",       N_("work"),               R_ADDRESS_WORK      },
  {"parcel",     N_("parcel"),             R_ADDRESS_PARCEL    },
  {"postal",     N_("postal"),             R_ADDRESS_POSTAL    },
  {"intl",       N_("international"),      R_ADDRESS_INTL      },
  {"dom",        N_("domestic"),           R_ADDRESS_DOM       },
  {"registered", N_("registered office"),  R_ADDRESS_COMPANY   },
  {"subsidiary", N_("subsidiary company"), R_ADDRESS_SUBSIDIARY},
  {"other",      N_("other"),              R_ADDRESS_OTHER     },  
  {"unknown",    N_("unknown"),            R_ADDRESS_UNKNOWN   },
  {NULL,      NULL,                        R_ADDRESS_INVALID   },
};


static void r_address_class_init   (RAddressClass* klass);
static void r_address_init         (RAddress* obj);

static void r_address_dispose      (RAddress* obj);
static void r_address_finalize     (RAddress* obj);

static void r_address_set_property (GObject* obj, guint property_id,
				    const GValue* value, GParamSpec* spec);
static void r_address_get_property (GObject* obj, guint property_id,
				    GValue* value, GParamSpec* spec);


GType
r_address_get_type()
{
  static GType address_type = 0;
  
  if (!address_type)
    {
      static const GTypeInfo address_info =
	{
	  sizeof(RAddressClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_address_class_init,
	  NULL,
	  NULL,
	  sizeof(RAddress),
	  0,
	  (GInstanceInitFunc) r_address_init
	};

      address_type = g_type_register_static (G_TYPE_OBJECT, 
					     "RAddress", &address_info, 0);
    }
  
  return address_type;
}


static void
r_address_class_init(RAddressClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  class  = G_OBJECT_CLASS (klass);

  class->dispose  = (GObjectFinalizeFunc) r_address_dispose;
  class->finalize = (GObjectFinalizeFunc) r_address_finalize;

  class->set_property = r_address_set_property;
  class->get_property = r_address_get_property;

  /* class property
   */
  
  /**
   * RAddress:address-type:
   *
   * defines the address's type. See #RAddressType for valid values
   */
  pspec = g_param_spec_int("address-type",  
			   "address's type",  
			   "store the address's type, i.e. " 
			   "R_ADDRESS_PREF or R_ADDRESS_WORK",  
			   R_ADDRESS_PREF,         
			   R_ADDRESS_INVALID,    
			   R_ADDRESS_UNKNOWN,    
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);  
  g_object_class_install_property(class, ADDRESS_TYPE, pspec);     
  
  
  /**
   * RAddress:street:
   *
   * the address's street
   */
  pspec = g_param_spec_string("street", 
			      "address's street", 
			      "the address's street",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_STREET, pspec);    
  
  /**
   * RAddress:address-number:
   *
   * the street number
   */
  pspec = g_param_spec_string("street-number", 
			      "the street number", 
			      "the street number",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_NUMBER, pspec);    

  /**
   * RAddress:city:
   *
   * the address's city
   */
  pspec = g_param_spec_string("city", 
			      "the address's city", 
			      "the address's city",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_CITY, pspec);    

  /**
   * RAddress:zip:
   *
   * the postal code
   */
  pspec = g_param_spec_string("zip", 
			      "postal code", 
			      "the postal code",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_ZIP, pspec);    

  /**
   * RAddress:province:
   *
   * the address's province (example: Rm (Rome), Mi (Milan) , Na (Naples) ...) 
   */
  pspec = g_param_spec_string("province", 
			      "province", 
			      "the address's province",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_PROVINCE, pspec);    

  /**
   * RAddress:state:
   *
   * the address's state (examples: UE, USA, ...)
   */
  pspec = g_param_spec_string("state", 
			      "the state", 
			      "the state",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_STATE, pspec);   
 
  /**
   * RAddress:country:
   *
   * the address's country (example: Italy, France, ...)
   */
  pspec = g_param_spec_string("country", 
			      "the country", 
			      "the country",  
			      NULL,    
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE); 
  g_object_class_install_property(class, ADDRESS_COUNTRY, pspec); 
  
}


static void
r_address_init(RAddress* self)
{
  self->private = g_new(RAddressPrivate, 1);
  self->private->dispose_has_run = FALSE;

  self->private->street   = NULL;
  self->private->number   = NULL;
  self->private->city     = NULL;
  self->private->zip      = NULL;
  self->private->province = NULL;
  self->private->state    = NULL;
  self->private->country  = NULL;  
}



static void 
r_address_dispose (RAddress* self)
{
  g_return_if_fail(IS_R_ADDRESS(self));
  
  if (self->private->dispose_has_run)
    return;

  self->private->dispose_has_run = TRUE;
}


static void 
r_address_finalize (RAddress* self)
{
  g_return_if_fail(IS_R_ADDRESS(self));

  r_utils_free_string(self->private->street);
  r_utils_free_string(self->private->number);
  r_utils_free_string(self->private->city);
  r_utils_free_string(self->private->zip);
  r_utils_free_string(self->private->province);
  r_utils_free_string(self->private->state);
  r_utils_free_string(self->private->country);

  g_free(self->private);
  self->private = NULL;
}


static void 
r_address_set_property (GObject* obj, guint property_id,
			const GValue* value, GParamSpec* spec)
{
  RAddress* self = (RAddress*) obj;
  
  switch (property_id) 
    {
    case ADDRESS_TYPE:
      self->private->type = g_value_get_int(value);
      break;

    case ADDRESS_STREET:
      g_free(self->private->street);
      self->private->street = g_value_dup_string(value);
      break;

    case ADDRESS_NUMBER:
      g_free(self->private->number);
      self->private->number = g_value_dup_string(value);
      break;

    case ADDRESS_CITY:
      g_free(self->private->city);
      self->private->city = g_value_dup_string(value);
      break;

    case ADDRESS_ZIP:
      g_free(self->private->zip);
      self->private->zip = g_value_dup_string(value);
      break;

    case ADDRESS_PROVINCE:
      g_free(self->private->province);
      self->private->province = g_value_dup_string(value);
      break;

    case ADDRESS_STATE:
      g_free(self->private->state);
      self->private->state = g_value_dup_string(value);
      break;

    case ADDRESS_COUNTRY:
      g_free(self->private->country);
      self->private->country = g_value_dup_string(value);
      break;
      
    default:
      break;  
    }   
}


static void 
r_address_get_property (GObject* obj, guint property_id,
			GValue* value, GParamSpec* spec)
{
  RAddress* self = (RAddress*) obj;
  
  switch (property_id) 
    {
    case ADDRESS_TYPE:
      g_value_set_int(value, self->private->type);
      break; 

    case ADDRESS_STREET:
      g_value_set_string(value, self->private->street);
      break; 

    case ADDRESS_NUMBER:
      g_value_set_string(value, self->private->number);
      break; 

    case ADDRESS_CITY:
      g_value_set_string(value, self->private->city);
      break; 

    case ADDRESS_ZIP:
      g_value_set_string(value, self->private->zip);
      break; 

    case ADDRESS_PROVINCE:
      g_value_set_string(value, self->private->province);
      break; 

    case ADDRESS_STATE:
      g_value_set_string(value, self->private->state);
      break; 

    case ADDRESS_COUNTRY:
      g_value_set_string(value, self->private->country);
      break; 
      
    default:
      break;  
    }  
}




/* **************** Public
 */


/**
 * r_address_new
 *
 * create a new #RAddress
 * 
 * Returns: a #RAddress*
 */
RAddress*
r_address_new(void)
{
  RAddress* address;

  address = g_object_new(r_address_get_type(), NULL);

  return address;
}


/**
 * r_address_free
 * @address: a #RAddress
 *
 * free the #RAddress*
 */
void
r_address_free(RAddress* address)
{
  g_return_if_fail(IS_R_ADDRESS(address));

  g_object_unref(address);
}


/**
 * r_address_copy
 * @address: a #RAddress
 *
 * copy the given #RAddress
 * 
 * Returns: a new allocated #RAddress
 */
RAddress* 
r_address_copy  (RAddress* address)
{
  RAddress* new;
  RAddressType type;
  gchar *street, *number, *city;
  gchar *zip, *province, *state, *country;
  
  g_return_val_if_fail(IS_R_ADDRESS(address), NULL);
  
  new = r_address_new();

  g_object_get(G_OBJECT(address), "address-type", &type, "street", &street,
	       "street-number", &number, "city", &city, "zip", &zip,
	       "province", &province, "state", &state, 
	       "country", &country, NULL);

  g_object_set(G_OBJECT(new), "address-type", type, "street", street,
	       "street-number", number, "city", city, "zip", zip,
	       "province", province, "state", state, 
	       "country", country, NULL);
  
  return new;
}


gboolean     
r_address_search (RAddress* address, const gchar* str)
{
  g_return_val_if_fail(IS_R_ADDRESS(address), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  if (g_strrstr(address->private->street, str))
    return TRUE;

  if (g_strrstr(address->private->number, str))
    return TRUE;

  if (g_strrstr(address->private->city, str))
    return TRUE;

  if (g_strrstr(address->private->zip, str))
    return TRUE;

  if (g_strrstr(address->private->province, str))
    return TRUE;

  if (g_strrstr(address->private->state, str))
    return TRUE;

  if (g_strrstr(address->private->country, str))
    return TRUE;

  return FALSE;
}


/**
 * r_address_decode_type
 * @type: a #RAddressType
 *
 * decode the address type
 * 
 * Returns: a gchar*
 */
gchar* 
r_address_lookup_enum2str (RAddressType type)
{
  return r_lookup_table_enum2str(table, type);
}


/**
 * r_address_encode_type
 * @type: a gchar*
 *
 * encode the address type
 * 
 * Returns: a RAddressType value or R_ADDRESS_UNKNOWN
 */
RAddressType 
r_address_lookup_str2enum (gchar* str)
{  
  return (RAddressType) r_lookup_table_str2enum(table, str);
}

gchar*
r_address_lookup_enum2lbl (RAddressType type)
{
  return r_lookup_table_enum2lbl(table, type);
}

gchar*
r_address_lookup_str2lbl (gchar* str)
{
  return r_lookup_table_str2lbl(table, str);
}



