#ifndef _RHEOLEF_UBLAS_INVERT_H
#define _RHEOLEF_UBLAS_INVERT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// The following code inverts the matrix input using LU-decomposition
// with backsubstitution of unit vectors.
// Reference: Numerical Recipies in C, 2nd ed., by Press, Teukolsky, Vetterling & Flannery.
//
// http://www.crystalclearsoftware.com/cgi-bin/boost_wiki/wiki.pl?action=browse&diff=1&id=LU_Matrix_Inversion
// Hope someone finds this useful. Regards, Fredrik Orderud. 
//
// Last edited September 4, 2007 5:23 am
//
#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/vector_proxy.hpp>
#include <boost/numeric/ublas/matrix.hpp>
#include <boost/numeric/ublas/triangular.hpp>
#include <boost/numeric/ublas/lu.hpp>
#include <boost/numeric/ublas/io.hpp>
namespace ublas = boost::numeric::ublas;

namespace rheolef { 
// Matrix inversion routine.
// Uses lu_factorize and lu_substitute in uBLAS to invert a matrix
template<class T>
bool invert (const ublas::matrix<T>& input, ublas::matrix<T>& inverse) {
  using namespace boost::numeric::ublas;
  typedef permutation_matrix<std::size_t> pmatrix;
  // create a working copy of the input
  matrix<T> A(input);
  // create a permutation matrix for the LU-factorization
  pmatrix pm(A.size1());
 
  // perform LU-factorization
  int res = lu_factorize(A,pm);
  if( res != 0 ) return false;

  // create identity matrix of "inverse"
  inverse.assign(ublas::identity_matrix<T>(A.size1()));

  // backsubstitute to get the inverse
  lu_substitute(A, pm, inverse);

  return true;
}
template<class T>
boost::numeric::ublas::matrix<T>
invert(const boost::numeric::ublas::matrix<T> &m, bool &is_singular)
{ 
   ublas::matrix<T> inv_m (m.size1(), m.size2());
   is_singular = ! invert (m, inv_m);
   return inv_m;
}
// http://archives.free.net.ph/message/20080909.064313.59c122c4.fr.html 
template<class matrix_T>
double determinant(ublas::matrix_expression<matrix_T> const& mat_r) {
  double det = 1.0;
  matrix_T mLu(mat_r());
  ublas::permutation_matrix<std::size_t> pivots(mat_r().size1() );
  bool is_singular = lu_factorize(mLu, pivots);
  if (!is_singular) {
    for (std::size_t i = 0; i < pivots.size(); ++i) {
      if (pivots(i) != i) {
        det *= -1.0;
      }
      det *= mLu(i,i);
    }
  } else {
    det = 0.0;
  }
  return det;
} 
}// namespace rheolef
#endif // _RHEOLEF_UBLAS_INVERT_H
