#ifndef _SKIT_NUM_ALGORITHM_H
#define _SKIT_NUM_ALGORITHM_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// Basic Numerical Algorithms
//
// author: Pierre.Saramito@imag.fr
//
// date: 1 april 1997
//
// TODO: one function per file + doc !!
//
# include "rheolef/skitbase.h"
namespace rheolef { 

// basics: don't known if ident<T,U> is defined in <function[nal].h>
template <class T, class U>
struct my_ident: public std::unary_function<T, U> {
  const U& operator()(const T& x) const { return x; }
};

// =======================[ PROPERTIES ]=========================================
template <class T>
struct is_finite : public std::unary_function<T, bool> {
    bool operator() (const T& x) const { return !(xisinf(x) || xisnan(x)); }
};
template <class T>
struct is_int_or_inf_or_nan : public std::unary_function<T, bool> {
    bool operator() (const T& x) const { return (xisint(x)||xisinf(x)||xisnan(x)); }
};
template <class T>
struct is_positive_or_null : public std::unary_function<T, bool> {
    bool operator() (const T& x) const { return (xabs(x) == x); }
};
template <
    class InputIterator, 
    class Predicate>
inline
bool
all_elements_are (
    InputIterator iter,
    InputIterator last,
    Predicate     pred)
{
    while (iter != last) {
	if (!pred(*iter)) return false;
	++iter;
    }
    return true;
}
//
// application for dense vectors
// -----------------------------
template <class InputIterator, class T>
inline
bool
any_is_inf_or_nan (InputIterator iter, InputIterator last, const T&)
{
    return ! all_elements_are (iter, last, is_finite<T>());
}
template <class InputIterator, class T>
inline
bool
any_is_negative (InputIterator iter, InputIterator last, const T&)
{
    return ! all_elements_are (iter, last, is_positive_or_null<T>());
}
template <class InputIterator, class T>
inline
bool
all_are_int_or_inf_or_nan (InputIterator iter, InputIterator last, const T&)
{
    return all_elements_are (iter, last, is_int_or_inf_or_nan<T>());
}
//
// complete STL
// ------------------------------
template <class Pair, class T>
struct my_select2nd : public std::unary_function<Pair, T> {
    const T& operator()(const Pair& p) const { return p.second; }
};
// unary_compose and binary_compose (extensions, not part of the standard).

template <class _Operation1, class _Operation2>
class my_unary_compose
  : public std::unary_function<typename _Operation2::argument_type,
                          typename _Operation1::result_type>
{
protected:
  _Operation1 __op1;
  _Operation2 __op2;
public:
  my_unary_compose(const _Operation1& __x, const _Operation2& __y)
    : __op1(__x), __op2(__y) {}
  typename _Operation1::result_type
  operator()(const typename _Operation2::argument_type& __x) const {
    return __op1(__op2(__x));
  }
};

template <class _Operation1, class _Operation2>
inline my_unary_compose<_Operation1,_Operation2>
my_compose1(const _Operation1& __op1, const _Operation2& __op2)
{
  return my_unary_compose<_Operation1,_Operation2>(__op1, __op2);
}
template <class _Operation1, class _Operation2, class _Operation3>
class my_binary_compose
  : public std::unary_function<typename _Operation2::argument_type,
                          typename _Operation1::result_type> {
protected:
  _Operation1 _M_op1;
  _Operation2 _M_op2;
  _Operation3 _M_op3;
public:
  my_binary_compose(const _Operation1& __x, const _Operation2& __y,
                 const _Operation3& __z)
    : _M_op1(__x), _M_op2(__y), _M_op3(__z) { }
  typename _Operation1::result_type
  operator()(const typename _Operation2::argument_type& __x) const {
    return _M_op1(_M_op2(__x), _M_op3(__x));
  }
};

template <class _Operation1, class _Operation2, class _Operation3>
inline my_binary_compose<_Operation1, _Operation2, _Operation3>
my_compose2(const _Operation1& __op1, const _Operation2& __op2,
         const _Operation3& __op3)
{
  return my_binary_compose<_Operation1,_Operation2,_Operation3>
    (__op1, __op2, __op3);
}



//
// application for sparse vectors
// ------------------------------
template <class Operation, class Pair, class T>
inline
my_unary_compose<Operation, my_select2nd<Pair,T> >
unary_apply2nd (const Operation& op, const Pair&, const T&) 
{
    return my_compose1(op, my_select2nd<Pair,T>());
}
template <class BinaryOperation, class Pair1, class T1, class Pair2, class T2>
inline
my_binary_compose<BinaryOperation, my_select2nd<Pair1,T1>, my_select2nd<Pair2,T2> >
binary_apply2nd (const BinaryOperation& binary_op, 
		 const Pair1&, const T1&, const Pair2&, const T2&)
{
    return my_compose2(binary_op, my_select2nd<Pair1,T1>(), my_select2nd<Pair2,T2>());
}
template <class Pair>
struct is_finite_pair : public std::unary_function<Pair, bool> {
    bool operator() (const Pair& p) const { return !(xisinf(p.second) || xisnan(p.second)); }
};
template <class Pair>
struct is_int_or_inf_or_nan_pair : public std::unary_function<Pair, bool> {
    bool operator() (const Pair& p) const { return (xisint(p.second)||xisinf(p.second)||xisnan(p.second)); }
};
template <class Pair>
struct is_positive_or_null_pair : public std::unary_function<Pair, bool> {
    bool operator() (const Pair& p) const { return (xabs(p.second) == p.second); }
};
template <class PairInputIterator, class Pair, class T>
inline
bool
any_is_inf_or_nan (
    PairInputIterator iter, 
    PairInputIterator last, 
    const T&, 
    const Pair&)
{
    return ! all_elements_are (iter, last, is_finite_pair<Pair>());
}
template <class PairInputIterator, class Pair, class T>
inline
bool
any_is_negative (
    PairInputIterator iter, 
    PairInputIterator last, 
    const T&, 
    const Pair&)
{
    return ! all_elements_are (iter, last, is_positive_or_null_pair<Pair>());
}
template <class PairInputIterator, class Pair, class T>
inline
bool
all_are_int_or_inf_or_nan (
    PairInputIterator iter, 
    PairInputIterator last, 
    const T&,
    const Pair&)
{
    return all_elements_are (iter, last, is_int_or_inf_or_nan_pair<Pair>());
}
// ================================[ SELECTION ]=================================
template <class T>
struct abs_value : public std::unary_function<T, T> {
    T operator()(const T& x) const { return xabs(x); }
};
template <
    class InputIterator,
    class BinaryPredicate,
    class Function,
    class T>
inline
T
select_op_value (
    InputIterator   iter,
    InputIterator   last,
    const T&        initial_value,
    BinaryPredicate binary_pred,
    Function        op)
{
    T result = initial_value;
    while (iter != last) {
	T value = op(*iter);
	if (binary_pred(value, result)) {
	    result = value;
	}
	++iter;
    }
    return result;
}
//
// application for dense vectors
// -----------------------------
template <class InputIterator, class T>
inline
T
select_max (
    InputIterator   iter,
    InputIterator   last,
    const T&        initial_value /* = -max_value(T()) */)
{
    return select_op_value (iter , last, initial_value, std::greater<T>(), my_ident<T,T>());
}
template <class InputIterator, class T>
inline
T
select_min (
    InputIterator   iter,
    InputIterator   last,
    const T&        initial_value /* = max_value(T()) */)
{
    return select_op_value (iter, last, initial_value, std::less<T>(), my_ident<T,T>());
}
template <class InputIterator, class T>
inline
T
select_max_abs (
    InputIterator   iter,
    InputIterator   last,
    const T&        init_val /* = T() */)
{
    return select_op_value (iter, last, init_val, std::greater<T>(), abs_value<T>());
}
template <class InputIterator, class T>
inline
T
select_min_abs (
    InputIterator   iter,
    InputIterator   last,
    const T&        init_val /* = T() */)
{
    return select_op_value (iter, last, init_val, std::less<T>(), abs_value<T>());
}
//
// application for sparse vectors
// ------------------------------
template <class PairInputIterator, class Pair, class T>
inline
T
select_max (
    PairInputIterator   iter,
    PairInputIterator   last,
    const T&            init_val /* = -max_value(T()) */,
    const Pair&)
{
    return select_op_value (iter, last, init_val, std::greater<T>(), 
			    my_select2nd<Pair,T>());
}
template <class PairInputIterator, class Pair, class T>
inline
T
select_min (
    PairInputIterator   iter,
    PairInputIterator   last,
    const T&            init_val /* = max_value(T()) */,
    const Pair&)
{
    return select_op_value (iter, last, init_val, std::less<T>(), 
			    my_select2nd<Pair,T>());
}
template <class Pair, class T>
struct abs_value_pair : public std::unary_function<Pair, T> {
    T operator()(const Pair& p) const { return xabs(p.second); }
};
template <class PairInputIterator, class Pair, class T>
inline
T
select_max_abs (
    PairInputIterator   iter,
    PairInputIterator   last,
    const T&            init_val /* = T() */,
    const Pair&)
{
    return select_op_value (iter, last, init_val, std::greater<T>(), 
			    abs_value_pair<Pair,T>());
}
template <class PairInputIterator, class Pair, class T>
inline
T
select_min_abs (
    PairInputIterator   iter,
    PairInputIterator   last,
    const T&            init_val /* = T() */,
    const Pair&)
{
    return select_op_value (iter, last, init_val, std::less<T>(), 
			    abs_value_pair<Pair,T>());
}
// ================================[ APPLY     ]=================================
// bug GNU C++ : redefine bind1st, bind2nd
template <class Operation> 
struct my_binder1st {
    typedef typename Operation::first_argument_type  value_type;
    typedef typename Operation::second_argument_type argument_type;
    typedef typename Operation::result_type          result_type;
    Operation  op;
    value_type x1;
    my_binder1st(const Operation& o, const value_type& x) : op(o), x1(x) {}
    result_type operator()(const argument_type& x2) const { return op(x1, x2); }
};
template <
    class Operation, 
    class T>
inline
my_binder1st<Operation> 
my_bind1st(const Operation& op, const T& x) 
{
    typedef typename Operation::first_argument_type value_type;
    return my_binder1st<Operation>(op, value_type(x));
}
template <class Operation> 
struct my_binder2nd {
    typedef typename Operation::first_argument_type  argument_type;
    typedef typename Operation::second_argument_type value_type;
    typedef typename Operation::result_type          result_type;
    Operation  op;
    value_type x2;
    my_binder2nd(const Operation& o, const value_type& x) : op(o), x2(x) {}
    result_type operator()(const argument_type& x1) const { return op(x1, x2); }
};
template <
    class Operation, 
    class T>
inline
my_binder2nd<Operation> 
my_bind2nd(const Operation& op, const T& x) 
{
    typedef typename Operation::second_argument_type value_type;
    return my_binder2nd<Operation>(op, value_type(x));
}
}// namespace rheolef
#endif // _SKIT_NUM_ALGORITHM_H
