///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/field-local-norm.h"
#include "rheolef/exact_compose.h"
#include "rheolef/form.h"
using namespace std;
namespace rheolef { 

// compute the local L2 norm:
//  I_K = (int_K |vh(x)|^2 dx )^{1/2}
// -> return a P0 field
// extension: accepts some vector-valued input
// NOTE: used by estim_2()
// NOTE2: ne devrait pas utiliser un localiseur, car on peut
//  iterer sur le maillage du champs vh et appliquer la fct au vol
//  aux points de quadrature.
//  -> question: comment implementer cela ?
//  -> la classe renvoyee par exact_compose() doit renvoyer des infos de vh.get_space
static
inline
Float
_norm2 (const point& v) { return dot(v,v); }

field
norm2_L2_local_with_localizer (const field& vh) {
    quadrature_option_type qopt;
    qopt.set_family(quadrature_option_type::gauss);
    size_t order = 2*vh.get_space().degree();
    qopt.set_order(order);
    warning_macro ("vh.get_approx() = " << vh.get_approx());
    warning_macro ("order = " << order);
    space Xh (vh.get_geo(), "P0");
    field Ih = riesz_representer (Xh, exact_compose(ptr_fun(_norm2),vector_field(vh)), qopt);
    warning_macro ("new_Ih = " << dot(Ih, field(Xh,1.0)));
    return Ih;
}
// actual solution, when no localizer is available, for 3D
field
norm2_L2_local_without_localizer (const field& vh) {
    // method to compute the exact local integral:
    // m_w(phi,psi) = int_Omega phi(x) psi(x) w(x) dx : weighted P2d P2d form
    //  with phi=w=v[i] and psi=1 
    // => m_w*v[i] computes *exactly* the i-th integral term
    const space& Zh = vh.get_space();
    space Z0h = Zh[0];
    space Xh (vh.get_geo(), "P0");
    size_t n_comp = vh.n_component();
    field Ih (Xh, 0.0);
    for (size_t i = 0; i < n_comp; i++) {
        warning_macro ("WEIGHTED FORM...");
        cerr << "vh[i].approx      = " << vh.get_approx() << endl;
        form m_w (Z0h, Xh, "mass", vh[i]);
        warning_macro ("WEIGHTED FORM done");
        Ih = Ih + m_w*vh[i];
    }
    warning_macro ("old_Ih = " << dot(Ih, field(Xh,1.0)));
    return Ih;
}
field
norm2_L2_local (const field& vh) {
    field Ih;
    if (vh.dimension() < 3) {
      Ih = norm2_L2_local_with_localizer (vh);
    } else {
      Ih = norm2_L2_local_without_localizer (vh);
    }
    return Ih;
}
// vh & wh are Pkd-scalar
// returns the P0-scalar of int_K vh*wh dx
field
norm2_L2_local_scalar (const field& vh, const field& wh) {
  // method to compute the exact local integral:
  // m_w(phi,psi) = int_Omega phi(x) psi(x) w(x) dx : weighted Pkd Pkd form
  //  with phi=vh, w=wh[i] and psi=1 
  // => m_w*vh computes *exactly* the i-th integral term
  const space& Z0h = vh.get_space();
  space L0h (vh.get_geo(), "P0");
  field Ih (L0h);
  form m_w (Z0h, L0h, "mass", wh);
  Ih = m_w*vh;
  return Ih;
}
// vh is a Pkd-vector
// returns the P0-vector of |vh[i]|_l2(K), i=0..n_comp-1
field
norm2_L2_local_vector (const field& vh) {
  // method to compute the exact local integral:
  // m_w(phi,psi) = int_Omega phi(x) psi(x) w(x) dx : weighted Pkd Pkd form
  //  with phi=w=vh[i] and psi=1 
  // => m_w*v[i] computes *exactly* the i-th integral term
  const space& Zh_vec = vh.get_space();
  space Z0h = Zh_vec[0];
  space L0h (vh.get_geo(), "P0");
  space L0h_vec (vh.get_geo(), "P0", "vector");
  field Ih (L0h_vec);
  for (size_t i = 0; i < vh.n_component(); i++) {
    form m_w (Z0h, L0h, "mass", vh[i]);
    Ih[i] = m_w*vh[i];
  }
  return Ih;
}
}// namespace rheolef
