///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// Domain: part of a finite element mesh
//
// authors:
//      Pierre.Saramito@imag.fr
//
// date: 12 may 1997
//
#include "rheolef/domain.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_element.h"
using namespace std;
namespace rheolef { 

istream&
operator >> (istream& is, domain& d)
{
    typedef domain::size_type size_type;

    if (!scatch(is,"\ndomain")) {
       // no more domains in stream
       return is;
    }
    is >> d._name;
    size_type version;
    is >> version;
    if (version != 1) {
	error_macro ("domain::get(" << d._name
	    << "): unexpected domain version "
	    << version);
    }
    size_type n;
    is >> d._dim >> n;
    d.resize(n);
    for (domain::iterator p = d.begin(); p != d.end(); p++) {
	is >> (*p);
    }
    return is;
}
ostream& 
operator << (ostream& os, const domain& d)
{
    typedef domain::size_type size_type;
    os << "domain" << endl;
    if (d.name().length() == 0) {
        os << "unamed" << endl;
	warning_macro ("unamed domain has been be named `unamed' for output");
    } else {
        os << d.name()  << endl;
    }
    os << "1 " << d.dimension() << " " << d.size() << endl;

    for (domain::const_iterator p = d.begin(); p != d.end(); p++) {
	os << (*p) << endl;
    }
    return os;
}
// TODO: merge with geo::put_vtk
//  as ::put_vtk (os, first_elt, last_elt, first_p, last_p)
ostream&
domain::put_vtk (ostream& vtk, Vector<point>::const_iterator iter_p, Vector<point>::const_iterator last_p) const
{
  size_type n_point = last_p - iter_p;

  vtk << "# vtk DataFile Version 1.0\n";
  vtk << "Unstructured Grid\n";
  vtk << "ASCII\n";
  vtk << endl;
  vtk << "DATASET UNSTRUCTURED_GRID\n";
  vtk << "POINTS " << n_point << " float\n";
  vtk << endl;

  while (iter_p != last_p) vtk << *iter_p++ << endl;
  vtk << endl;

  int nb_connectivity_entries = size();
  for (domain::const_iterator p = begin(); p != end(); p++) {
      nb_connectivity_entries += (*p).size();
  }
  vtk << "CELLS " << size() << " " << nb_connectivity_entries << endl;
  for (domain::const_iterator p = begin(); p != end(); p++) {
      const geo_element& g = (*p);
      vtk << g.size() << "\t" ;
      for (size_type j=0 ; j < g.size(); j++)
	vtk << g[j] << " " ;
      vtk << endl ;
  }
  vtk << endl;
  vtk << "CELL_TYPES " << size() << endl;
  for (domain::const_iterator p = begin(); p != end(); p++) {
      const geo_element& g = (*p);
      int cell_type = 0;
      switch (g.name()) {
	case 'p' : cell_type = 1; break;
	case 'e' : cell_type = 3; break;
	case 't' : cell_type = 5; break;
	case 'q' : cell_type = 9; break;
        case 'T' : cell_type = 10; break;
        case 'H' : cell_type = 12; break;
        case 'P' : {
                   cell_type = 11;
                   // TODO: implement prisms as vtk polygon... ?
                   // or split it in T and H with internal nodes ?
                   error_macro ("prisms not yet implemented -- sorry.");
                   }
        default : break;
      }
      vtk << cell_type << endl;
  }
  return vtk;
}
void 
domain::cat(const domain& d)
{
      if (size() != 0) {
          if (dimension() != d.dimension()) {
	    fatal_macro("domains have inccompatible dimensions: "
	       << dimension() << " and " << d.dimension() << ").") ;
	  }
          _name = name() + "+" + d.name() ;
      } else {
          _name = d.name() ;
	  _dim  = d.dimension();
      }
      for (domain::const_iterator p = d.begin(); p != d.end(); p++)
          insert(end(), *p) ;
}
ostream& 
domain::dump (ostream& os) const
{
    os << "domain " << name() << endl;
    os << "dimension " << dimension() << endl;
    os << "size " << size() << endl;
    for (const_iterator i = begin(); i != end(); i++) {
      (*i).dump(os);
    }
    return os;
}
void
domain::check() const
{
    for (const_iterator i = begin(); i != end(); i++) {
        (*i).check();
    }
}
}// namespace rheolef
