///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// paraview vtk visualization
//
// author: Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 23 oct 2011
//
#include "rheolef/field.h"
#include "rheolef/piola.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"
#include "rheolef/iofem.h"
#include "rheolef/field_expr_ops.h"
#include "rheolef/field_nonlinear_expr_ops.h"
#include "rheolef/interpolate.h"

using namespace std;
namespace rheolef { 

// ----------------------------------------------------------------------------
// python utils
// ----------------------------------------------------------------------------
template<class T>
static
std::string
python (const point_basic<T>& x, size_t d=3)
{
    std::ostringstream os;
    os << "(" << x[0];
    for (size_t i = 1; i < d; i++)
      os << ", " << x[i];
    os << ")" << std::flush;
    string buffer = os.str();
    return buffer;
}
// ----------------------------------------------------------------------------
// field puts
// ----------------------------------------------------------------------------
// extern:
template <class T> odiststream& field_put_vtk (odiststream&, const field_basic<T,sequential>&);

template <class T>
odiststream&
visu_vtk_paraview (odiststream& ops, const field_basic<T,sequential>& uh)
{
  //
  // 1) prerequises
  //
  using namespace std;
  typedef typename field_basic<T,sequential>::float_type float_type;
  typedef typename geo_basic<float_type,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  ostream& os = ops.os();
  string valued = uh.get_space().valued();
  bool is_scalar = (valued == "scalar");
  bool verbose  = iorheo::getverbose(os);
  bool clean    = iorheo::getclean(os);
  bool execute  = iorheo::getexecute(os);
  bool fill      = iorheo::getfill(os);    // isocontours or color fill
  bool elevation = iorheo::getelevation(os);
  bool color   = iorheo::getcolor(os);
  bool gray    = iorheo::getgray(os);
  bool black_and_white = iorheo::getblack_and_white(os);
  bool label     = iorheo::getlabel(os);
  bool stereo    = iorheo::getstereo(os);
  bool volume    = iorheo::getvolume(os);
  bool iso       = iorheo::getiso(os);
  bool cut       = iorheo::getcut(os);
  bool grid      = iorheo::getgrid(os);
  string format  = iorheo::getimage_format(os);
  if (format == "tiff") format = "tif";
  if (format == "jpeg") format = "jpg";
  string basename = iorheo::getbasename(os);
  string mark     = iorheo::getmark(os); 
  if (mark != "" && !is_scalar) mark = "|"+mark +"|";
  bool velocity    = iorheo::getvelocity(os);
  bool deformation = iorheo::getdeformation(os);
  Float vscale = iorheo::getvectorscale(os);
  size_type n_isovalue          = iorheo::getn_isovalue(os);
  size_type n_isovalue_negative = iorheo::getn_isovalue_negative(os);
  point_basic<float_type> origin     = iofem::getorigin(os);
  point_basic<float_type> normal     = iofem::getnormal(os);
  point_basic<size_type>  resolution = iofem::getresolution(os);
  if (black_and_white) {
    fill = false;
  }
  string outfile_fmt = "";
  string tmp = get_tmpdir() + "/";
  if (!clean) tmp = "";

  const geo_basic<float_type,sequential>& omega = uh.get_geo();
  size_type dim     = omega.dimension();
  size_type map_dim = omega.map_dimension();
  size_type nv      = omega.sizes().ownership_by_dimension[0].size();
  size_type nedg    = omega.sizes().ownership_by_dimension[1].size();
  size_type nfac    = omega.sizes().ownership_by_dimension[2].size();
  size_type nvol    = omega.sizes().ownership_by_dimension[3].size();
  size_type ne      = omega.sizes().ownership_by_dimension[map_dim].size();

  const numbering<float_type,sequential>& fem = uh.get_space().get_numbering();
  basis_on_pointset<float_type> b (fem.get_basis(), omega.get_piola_basis());
  //
  // 2) output data
  //
  string filelist;
  string filename = tmp+basename + ".vtk";
  filelist = filelist + " " + filename;
  ofstream vtk_os (filename.c_str());
  odiststream vtk (vtk_os);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  field_put_vtk (vtk, uh);
  vtk.close();
  //
  // 3) create python data file
  //
  std::string py_name = filename = tmp+basename + ".py";
  filelist = filelist + " " + filename;
  ofstream py (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  string style = (valued == "tensor" ? "tensor" : (velocity ? "velocity" : "deformation"));
  point xmin = uh.get_geo().xmin(),
        xmax = uh.get_geo().xmax();
  py << setprecision(numeric_limits<T>::digits10)
     << "#!/usr/bin/env paraview --script=" << endl
     << "# This is a paraview script for the visualization of " << basename << ".vtk" << endl
     << "# automatically generated by rheolef."  << endl
     << endl
     ;
  // paraview python library
  py << "from paraview.simple  import *" << endl
     << "from paraview_rheolef import *  # load rheolef specific functions" << endl
     << endl
     ;
  py << "opt = {                 	\\" << endl
     << "    'format'  : '" << format   << "',		\\" << endl
     << "    'resolution' : [" << resolution[0] << "," << resolution[1] << "], \\" << endl
     << "    'name'    : '" << mark  << "',   \\" << endl
     << "    'label'   : " << label << ",   \\" << endl
     << "    'axis'    : 1," << endl
     << "    'color'   : '" << (color ? "color" : (gray ? "gray" : "black_and_white")) << "'," << endl
     << "    'stereo'  : " << stereo << ",		\\" << endl
     << "    'bbox'    : [[" << xmin[0] << "," << xmin[1] << "," << xmin[2] << "],[" 
                             << xmax[0] << "," << xmax[1] << "," << xmax[2] << "]],  \\" << endl
     ;
  if (valued == "vector") {
     py << "    'style'   : '" << style << "',		\\" << endl
     ;
  }
  field_basic<T,sequential> uh_scalar;
  if (valued == "scalar") { 
    uh_scalar = uh;
  } else {
    space_basic<T,sequential> T0h (uh.get_geo(), uh.get_approx());
    uh_scalar = interpolate(T0h, norm(uh));
  }
  py << "    'scale'   : " << vscale << ",		\\" << endl
     << "    'elevation'  : " << elevation   << ",		\\" << endl
     << "    'range'      : [" <<  uh_scalar.min() << ", " << uh_scalar.max() << "],  \\" << endl
     << "    'iso'        : " << iso   << ",		\\" << endl
     << "    'n_isovalue' : " << n_isovalue   << ",		\\" << endl
     << "    'n_isovalue_negative' : " << n_isovalue_negative   << ",		\\" << endl
     << "    'cut'        : " << cut   << ",		\\" << endl
     << "    'origin'     : " << python(origin) << ", 		\\" << endl
     << "    'normal'     : " << python(normal) << ",  		\\" << endl
     << "    'grid'    : " << grid   << ",		\\" << endl
     << "    'fill'    : " << fill   << ",		\\" << endl
     << "    'volume'  : " << volume << ",		\\" << endl
     << "    'view_1d' : 0,				\\" << endl
     << "    'view_2d' : " << (dim == 2) << ",		\\" << endl
     << "    'view_map' : " << (dim > map_dim) << "		\\" << endl
     << "    }" << endl
     << endl
     << "paraview_field_" << valued << "(paraview, \"" << tmp+basename << "\", opt)" << endl
     << endl
     ;
  py.close();
  //
  // 3) run pyton
  //
  int status = 0;
  string command;
  if (execute) {
      string prog = (format == "") ? "paraview --script=" : "pvbatch --use-offscreen-rendering ";
      command = "LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " " + prog + py_name;
      if (format != "") command = "DISPLAY=:0.0 " + command;
      if (stereo && format == "") command = command + " --stereo";
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //
  // 4) clear vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + filelist;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  return ops;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template odiststream& visu_vtk_paraview<Float>  (odiststream&, const field_basic<Float,sequential>&);

}// namespace rheolef
