///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// contents:
//      * vtk output: 
//
// authors:
//      Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 6 may 2001
//

#include "rheolef/field.h"
#include "rheolef/iorheo.h"
#include "rheolef/rheostream.h"
#include "rheolef/tiny_matvec.h"
#include "rheolef/field-visu.h"
#include "rheolef/geo-visu-aux.h"
using namespace std;
namespace rheolef { 

// define when using old mayavi python library:
#undef _RHEOLEF_USE_MAYAVI_1

int
field_visu_mayavi (const field& u, const field_visu_option_type& opt)
{
  string valued = u.get_valued();
  if (valued != "scalar" && valued != "vector" && valued != "tensor") {
    error_macro ("unsupported valued space `" << valued << "'");
  } 
  bool is_scalar   (valued == "scalar");
  //
  // output geometry and data
  //
  string vtk_name = opt.basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (opt.verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  string approx = u.get_approx();
  if (approx == "P1" || approx == "P0" ) {
     vtk << vtkdata << u.get_geo();
  } else if (approx == "P2") {
     vtk << lattice << vtkdata << u.get_geo();
  } else if (approx == "P1d") {
     u.get_geo().data().put_vtk_P1d(vtk);
  } else {
     fatal_macro ("P0, P1, P1d or P2 approximation expected: `" << approx << "' founded");
  }
  if (valued == "scalar") {
     u.put_vtk(vtk);
  } else if (valued == "vector") {
     u.put_vtk_vector (vtk);
  } else { // valued == "tensor"
     u.put_vtk_tensor (vtk);
  }
  vtk.close();
  //
  // create python data file
  //
  size_t d = u.dimension();
  size_t map_d = u.get_geo().map_dimension();
  string py_name = opt.basename + ".py";
  ofstream py (py_name.c_str());
  if (opt.verbose) clog << "! file \"" << py_name << "\" created.\n";
  string style = (valued == "tensor" ? "tensor" : (opt.velocity ? "velocity" : "deformation"));
  string label = (is_scalar ? opt.basename : (style + " module"));
  py << "#!/usr/bin/env mayavi2" << endl
     << "# This is a mayavi script for the visualization of " << opt.basename << ".vtk" << endl
     << "# automatically generated by rheolef."  << endl
     << endl
     ;
#ifdef _RHEOLEF_USE_MAYAVI_1
  // old mayavi python library
  py << "import mayavi" << endl
     << "from mayavi_geo import *" << endl
     << endl
     ;
#else
  // new mayavi2 python library
  py << "from sys import version_info" << endl
     << "# mayavi2.py file has changed from directory since python 2.7:" << endl
     << "if version_info[0] >= 2 and version_info[1] >= 7 and version_info[2] >= 2:" << endl
     << "    from mayavi.scripts import mayavi2" << endl
     << "else:" << endl
     << "    from enthought.mayavi.scripts import mayavi2" << endl
     << "mayavi2.standalone(globals())     # start mayvi2 graphic interface" << endl
     << "from mayavi2_rheolef import *     # load rheolef specific functions" << endl
     << endl
     ;
#endif
  py << "opt = {                 	\\" << endl
     << "    'label'   : '" << label << "',   \\" << endl
     << "    'axis'    : 1," << endl
     << "    'color'   : '" << (opt.color ? "color" : (opt.gray ? "gray" : "black_and_white")) << "'," << endl
     << "    'stereo'  : " << opt.stereo << ",		\\" << endl
     ;
  if (valued == "vector") {
     py << "    'style'   : '" << style << "',		\\" << endl
     ;
  }
#ifdef TO_CLEAN
  Float vscale = opt.vscale;
  if (vscale == 0) vscale = u.get_geo().hmin()/euclidian_norm(u).max();
#endif // TO_CLEAN
  py << "    'scale'   : " << opt.vscale << ",		\\" << endl
     << "    'elevation'  : " << opt.elevation   << ",		\\" << endl
     ;
  if (valued == "scalar") {
     py << "    'volume'     : " << opt.volume   << ",		\\" << endl
        << "    'iso'        : " << opt.iso   << ",		\\" << endl
        << "    'n_isovalue' : " << opt.n_isovalue   << ",		\\" << endl
        << "    'n_isovalue_negative' : " << opt.n_isovalue_negative   << ",		\\" << endl
        << "    'cut'        : " << opt.cut   << ",		\\" << endl
        << "    'origin'     : " << python(opt.origin) << ", 		\\" << endl
        << "    'normal'     : " << python(opt.normal) << ",  		\\" << endl
	;
  }
  py << "    'grid'    : " << opt.grid   << ",		\\" << endl
     << "    'fill'    : " << opt.fill   << ",		\\" << endl
     << "    'view_1d' : 0,				\\" << endl
     << "    'view_2d' : " << (d == 2) << ",		\\" << endl
     << "    'view_map' : " << (d > map_d) << "		\\" << endl
     << "    }" << endl
     << endl
     ;
#ifdef _RHEOLEF_USE_MAYAVI_1
  py << "v = mayavi.mayavi()" << endl
     << "mayavi_field_" << valued << "(v, \"" << opt.basename << "\", opt)" << endl
     << "v.Render()" << endl
     << "v.master.wait_window()" << endl
     << endl
     ;
#else
  py << "mayavi2_field_" << valued << "(mayavi, \"" << opt.basename << "\", opt)" << endl
     << endl
     ;
#endif
  //
  // run pyton
  //
  int status = 0;
  string command;
  if (opt.execute) {
      command = "LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " mayavi2 " + py_name;
      if (opt.verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //
  // clear vtk data
  //
  if (opt.clean) {
      command = "/bin/rm -f " + opt.basename + ".vtk " + opt.basename + ".py";
      if (opt.verbose) clog << "! " << command << endl;
      status |= system (command.c_str());
  }
  return status;
}
}// namespace rheolef
