# Boost Test integration

RapidCheck comes with support for integrating with Boost Test and allows you to easily create properties that are also Boost Test test cases.

## Usage

This support is available through the `extras/boost_test` module. You can either directly add the `extras/boost_test/include` directory to your include path or link against the `rapidcheck_boost_test` target in your `CMakeLists.txt`. You can then simply `#include <rapidcheck/boost_test.h>`. Note that `rapidcheck/boost_test.h` needs to be included after any Boost Test headers.

## Reference

### `RC_BOOST_PROP(TestName, (args...))`

Defines a RapidCheck property as a Boost Test test. This macro essentially works the same as the Boost `BOOST_AUTO_TEST_CASE` macro but it includes an extra parenthesized list of arguments that will be generated by RapidCheck, just like when using `rc::check`:

```C++
RC_BOOST_PROP(MyTestCase,
              copyOfStringIsIdenticalToOriginal,
              (const std::string &str)) {
  const auto strCopy = str;
  RC_ASSERT(strCopy == str);
}
```

The parenthesis around the argument list are required because of how the preprocessor works and can not be omitted. This also means that if you don't want any arguments generated, you need to include an empty set of parenthesis:

```C++
// If you don't have any arguments, you have to have empty parentheses
RC_BOOST_PROP(MyTestCase, inRangeValueIsInRange, ()) {
  const auto range = *rc::gen::arbitrary<std::pair<int, int>>();
  const auto x = *rc::gen::inRange(range.first, range.second);
  RC_ASSERT(x >= range.first);
  RC_ASSERT(x < range.second);
}
```

### `RC_BOOST_FIXTURE_PROP(Name, Fixture, (args...))`

Analogous to the Boost `BOOST_FIXTURE_TEST_CASE` macro, defines a RapidCheck property as a Boost Test fixture based test. `Fixture` names the test fixture class. The fixture will be reinstantiated for every test case that is run. Otherwise, this macro works the same as `RC_BOOST_PROP`.

## Assertions

RapidCheck will treat any exception as a property failure so you should be able to use any assertion mechanism that signals failures as exceptions. However, Boost Test assertions are not implemented using exceptions which means that you should avoid them and use RapidCheck assertions such as `RC_ASSERT` in your RapidCheck properties instead.
