#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from rally.common import validation

from rally_openstack.common import consts
from rally_openstack.task import context
from rally_openstack.task.scenarios.senlin import utils as senlin_utils


@validation.add("required_platform", platform="openstack", users=True)
@context.configure(name="profiles", platform="openstack", order=190)
class ProfilesGenerator(context.OpenStackContext):
    """Context creates a temporary profile for Senlin test."""

    CONFIG_SCHEMA = {
        "type": "object",
        "$schema": consts.JSON_SCHEMA,
        "properties": {
            "type": {
                "type": "string",
            },
            "version": {
                "type": "string",
            },
            "properties": {
                "type": "object",
                "additionalProperties": True,
            }
        },
        "additionalProperties": False,
        "required": ["type", "version", "properties"]
    }

    def setup(self):
        """Create test profiles."""
        for user, tenant_id in self._iterate_per_tenants():

            senlin_scenario = senlin_utils.SenlinScenario({
                "user": user,
                "task": self.context["task"]
            })
            profile = senlin_scenario._create_profile(self.config)

            self.context["tenants"][tenant_id]["profile"] = profile.id

    def cleanup(self):
        """Delete created test profiles."""
        for user, tenant_id in self._iterate_per_tenants():

            senlin_scenario = senlin_utils.SenlinScenario({
                "user": user,
                "task": self.context["task"]
            })
            senlin_scenario._delete_profile(
                self.context["tenants"][tenant_id]["profile"])
