% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rank.R
\name{vec_rank}
\alias{vec_rank}
\title{Compute ranks}
\usage{
vec_rank(
  x,
  ...,
  ties = c("min", "max", "sequential", "dense"),
  incomplete = c("rank", "na"),
  direction = "asc",
  na_value = "largest",
  nan_distinct = FALSE,
  chr_proxy_collate = NULL
)
}
\arguments{
\item{x}{A vector}

\item{...}{These dots are for future extensions and must be empty.}

\item{ties}{Ranking of duplicate values.
\itemize{
\item \code{"min"}: Use the current rank for all duplicates. The next non-duplicate
value will have a rank incremented by the number of duplicates present.
\item \code{"max"}: Use the current rank \code{+ n_duplicates - 1} for all duplicates.
The next non-duplicate value will have a rank incremented by the number of
duplicates present.
\item \code{"sequential"}: Use an increasing sequence of ranks starting at the
current rank, applied to duplicates in order of appearance.
\item \code{"dense"}: Use the current rank for all duplicates. The next
non-duplicate value will have a rank incremented by \code{1}, effectively
removing any gaps in the ranking.
}}

\item{incomplete}{Ranking of missing and \link[=vec_detect_complete]{incomplete}
observations.
\itemize{
\item \code{"rank"}: Rank incomplete observations normally. Missing values within
incomplete observations will be affected by \code{na_value} and \code{nan_distinct}.
\item \code{"na"}: Don't rank incomplete observations at all. Instead, they are
given a rank of \code{NA}. In this case, \code{na_value} and \code{nan_distinct} have
no effect.
}}

\item{direction}{Direction to sort in.
\itemize{
\item A single \code{"asc"} or \code{"desc"} for ascending or descending order
respectively.
\item For data frames, a length \code{1} or \code{ncol(x)} character vector containing
only \code{"asc"} or \code{"desc"}, specifying the direction for each column.
}}

\item{na_value}{Ordering of missing values.
\itemize{
\item A single \code{"largest"} or \code{"smallest"} for ordering missing values as the
largest or smallest values respectively.
\item For data frames, a length \code{1} or \code{ncol(x)} character vector containing
only \code{"largest"} or \code{"smallest"}, specifying how missing values should
be ordered within each column.
}}

\item{nan_distinct}{A single logical specifying whether or not \code{NaN} should
be considered distinct from \code{NA} for double and complex vectors. If \code{TRUE},
\code{NaN} will always be ordered between \code{NA} and non-missing numbers.}

\item{chr_proxy_collate}{A function generating an alternate representation
of character vectors to use for collation, often used for locale-aware
ordering.
\itemize{
\item If \code{NULL}, no transformation is done.
\item Otherwise, this must be a function of one argument. If the input contains
a character vector, it will be passed to this function after it has been
translated to UTF-8. This function should return a character vector with
the same length as the input. The result should sort as expected in the
C-locale, regardless of encoding.
}

For data frames, \code{chr_proxy_collate} will be applied to all character
columns.

Common transformation functions include: \code{tolower()} for case-insensitive
ordering and \code{stringi::stri_sort_key()} for locale-aware ordering.}
}
\description{
\code{vec_rank()} computes the sample ranks of a vector. For data frames, ranks
are computed along the rows, using all columns after the first to break
ties.
}
\details{
Unlike \code{\link[base:rank]{base::rank()}}, when \code{incomplete = "rank"} all missing values are
given the same rank, rather than an increasing sequence of ranks. When
\code{nan_distinct = FALSE}, \code{NaN} values are given the same rank as \code{NA},
otherwise they are given a rank that differentiates them from \code{NA}.

Like \code{\link[=vec_order_radix]{vec_order_radix()}}, ordering is done in the C-locale. This can affect
the ranks of character vectors, especially regarding how uppercase and
lowercase letters are ranked. See the documentation of \code{\link[=vec_order_radix]{vec_order_radix()}}
for more information.
}
\section{Dependencies}{

\itemize{
\item \code{\link[=vec_order_radix]{vec_order_radix()}}
\item \code{\link[=vec_slice]{vec_slice()}}
}
}

\examples{
x <- c(5L, 6L, 3L, 3L, 5L, 3L)

vec_rank(x, ties = "min")
vec_rank(x, ties = "max")

# Sequential ranks use an increasing sequence for duplicates
vec_rank(x, ties = "sequential")

# Dense ranks remove gaps between distinct values,
# even if there are duplicates
vec_rank(x, ties = "dense")

y <- c(NA, x, NA, NaN)

# Incomplete values match other incomplete values by default, and their
# overall position can be adjusted with `na_value`
vec_rank(y, na_value = "largest")
vec_rank(y, na_value = "smallest")

# NaN can be ranked separately from NA if required
vec_rank(y, nan_distinct = TRUE)

# Rank in descending order. Since missing values are the largest value,
# they are given a rank of `1` when ranking in descending order.
vec_rank(y, direction = "desc", na_value = "largest")

# Give incomplete values a rank of `NA` by setting `incomplete = "na"`
vec_rank(y, incomplete = "na")

# Can also rank data frames, using columns after the first to break ties
z <- c(2L, 3L, 4L, 4L, 5L, 2L)
df <- data_frame(x = x, z = z)
df

vec_rank(df)
}
