% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/integration.R
\name{FindIntegrationAnchors}
\alias{FindIntegrationAnchors}
\title{Find integration anchors}
\usage{
FindIntegrationAnchors(
  object.list = NULL,
  assay = NULL,
  reference = NULL,
  anchor.features = 2000,
  scale = TRUE,
  normalization.method = c("LogNormalize", "SCT"),
  sct.clip.range = NULL,
  reduction = c("cca", "rpca", "jpca", "rlsi"),
  l2.norm = TRUE,
  dims = 1:30,
  k.anchor = 5,
  k.filter = 200,
  k.score = 30,
  max.features = 200,
  nn.method = "annoy",
  n.trees = 50,
  eps = 0,
  verbose = TRUE
)
}
\arguments{
\item{object.list}{A list of \code{\link{Seurat}} objects between which to
find anchors for downstream integration.}

\item{assay}{A vector of assay names specifying which assay to use when
constructing anchors. If NULL, the current default assay for each object is
used.}

\item{reference}{A vector specifying the object/s to be used as a reference
during integration. If NULL (default), all pairwise anchors are found (no
reference/s). If not NULL, the corresponding objects in \code{object.list}
will be used as references. When using a set of specified references, anchors
are first found between each query and each reference. The references are
then integrated through pairwise integration. Each query is then mapped to
the integrated reference.}

\item{anchor.features}{Can be either:
\itemize{
  \item{A numeric value. This will call \code{\link{SelectIntegrationFeatures}}
  to select the provided number of features to be used in anchor finding}
  \item{A vector of features to be used as input to the anchor finding process}
}}

\item{scale}{Whether or not to scale the features provided. Only set to FALSE
if you have previously scaled the features you want to use for each object in
the object.list}

\item{normalization.method}{Name of normalization method used: LogNormalize
or SCT}

\item{sct.clip.range}{Numeric of length two specifying the min and max values
the Pearson residual will be clipped to}

\item{reduction}{Dimensional reduction to perform when finding anchors. Can
be one of:
\itemize{
  \item{cca: Canonical correlation analysis}
  \item{rpca: Reciprocal PCA}
  \item{jpca: Joint PCA}
  \item{rlsi: Reciprocal LSI}
}}

\item{l2.norm}{Perform L2 normalization on the CCA cell embeddings after
dimensional reduction}

\item{dims}{Which dimensions to use from the CCA to specify the neighbor
search space}

\item{k.anchor}{How many neighbors (k) to use when picking anchors}

\item{k.filter}{How many neighbors (k) to use when filtering anchors}

\item{k.score}{How many neighbors (k) to use when scoring anchors}

\item{max.features}{The maximum number of features to use when specifying the
neighborhood search space in the anchor filtering}

\item{nn.method}{Method for nearest neighbor finding. Options include: rann,
annoy}

\item{n.trees}{More trees gives higher precision when using annoy approximate
nearest neighbor search}

\item{eps}{Error bound on the neighbor finding algorithm (from RANN/Annoy)}

\item{verbose}{Print progress bars and output}
}
\value{
Returns an \code{\link{AnchorSet}} object that can be used as input to
\code{\link{IntegrateData}}.
}
\description{
Find a set of anchors between a list of \code{\link{Seurat}} objects.
These  anchors can later be used to integrate the objects using the
\code{\link{IntegrateData}} function.
}
\details{
The main steps of this procedure are outlined below. For a more detailed
description of the methodology, please see Stuart, Butler, et al Cell 2019:
\doi{10.1016/j.cell.2019.05.031}; \doi{10.1101/460147}

First, determine anchor.features if not explicitly specified using
\code{\link{SelectIntegrationFeatures}}. Then for all pairwise combinations
of reference and query datasets:

\itemize{
  \item{Perform dimensional reduction on the dataset pair as specified via
  the \code{reduction} parameter. If \code{l2.norm} is set to \code{TRUE},
  perform L2 normalization of the embedding vectors.}
  \item{Identify anchors - pairs of cells from each dataset
  that are contained within each other's neighborhoods (also known as mutual
  nearest neighbors).}
  \item{Filter low confidence anchors to ensure anchors in the low dimension
  space are in broad agreement with the high dimensional measurements. This
  is done by looking at the neighbors of each query cell in the reference
  dataset using \code{max.features} to define this space. If the reference
  cell isn't found within the first \code{k.filter} neighbors, remove the
  anchor.}
  \item{Assign each remaining anchor a score. For each anchor cell, determine
  the nearest \code{k.score} anchors within its own dataset and within its
  pair's dataset. Based on these neighborhoods, construct an overall neighbor
  graph and then compute the shared neighbor overlap between anchor and query
  cells (analogous to an SNN graph). We use the 0.01 and 0.90 quantiles on
  these scores to dampen outlier effects and rescale to range between 0-1.}
}
}
\examples{
\dontrun{
# to install the SeuratData package see https://github.com/satijalab/seurat-data
library(SeuratData)
data("panc8")

# panc8 is a merged Seurat object containing 8 separate pancreas datasets
# split the object by dataset
pancreas.list <- SplitObject(panc8, split.by = "tech")

# perform standard preprocessing on each object
for (i in 1:length(pancreas.list)) {
  pancreas.list[[i]] <- NormalizeData(pancreas.list[[i]], verbose = FALSE)
  pancreas.list[[i]] <- FindVariableFeatures(
    pancreas.list[[i]], selection.method = "vst",
    nfeatures = 2000, verbose = FALSE
  )
}

# find anchors
anchors <- FindIntegrationAnchors(object.list = pancreas.list)

# integrate data
integrated <- IntegrateData(anchorset = anchors)
}

}
\references{
Stuart T, Butler A, et al. Comprehensive Integration of
Single-Cell Data. Cell. 2019;177:1888-1902 \doi{10.1016/j.cell.2019.05.031}
}
\concept{integration}
