#' @include Connection.R
#' @include Driver.R
NULL

#' @export
SQLITE_RWC <- 6L
#' @export
SQLITE_RW <- 2L
#' @export
SQLITE_RO <- 1L

#' Connect to/disconnect from a SQLite database.
#' 
#' @param drv,conn An objected generated by \code{\link{SQLite}}, or an existing
#'   \code{\linkS4class{SQLiteConnection}}. If an connection, the connection
#'   will be cloned.
#' @param dbname The path to the database file. There are two special values:
#' 
#'   \itemize{
#'   \item \code{""}: creates a temporary on-disk database The file will be 
#'      deleted when the connection is closed. 
#'   \item \code{":memory:"}: create a temporary in-memory database.
#'   }
#' @param cache_size Advanced option. A positive integer to change the maximum 
#'   number of disk pages that SQLite holds in memory (SQLite's default is 
#'   2000 pages). See \url{http://www.sqlite.org/pragma.html#pragma_cache_size} 
#'   for details.
#' @param synchronous Advanced options. Possible values for \code{synchronous} 
#'   are "off" (the default), "normal", or  "full".  Users have reported 
#'   significant speed ups using \code{sychronous = "off"}, and the SQLite 
#'   documentation itself implies considerable improved performance at the very 
#'   modest risk of database corruption in the unlikely case of the operating 
#'   system (\emph{not} the R application) crashing. See
#'   \url{http://www.sqlite.org/pragma.html#pragma_synchronous} for details.
#' @param flags \code{SQLITE_RWC}: open the database in read/write mode
#'   and create the database file if it does not already exist; 
#'   \code{SQLITE_RW}: open the database in read/write mode. Raise an error 
#'   if the file does not already exist; \code{SQLITE_RO}: open the database in 
#'   read only mode.  Raise an error if the file does not already exist
#' @param loadable.extensions When \code{TRUE} (default) SQLite3
#'   loadable extensions are enabled. Setting this value to \code{FALSE}
#'   prevents extensions from being loaded.
#' @param vfs Select the SQLite3 OS interface. See
#'   \url{http://www.sqlite.org/vfs.html} for details. Allowed values are
#'   \code{"unix-posix"}, \code{"unix-unix-afp"},
#'   \code{"unix-unix-flock"}, \code{"unix-dotfile"}, and
#'   \code{"unix-none"}.
#' @aliases SQLITE_RWC SQLITE_RW SQLITE_RO
#' @export
#' @useDynLib RSQLite rsqlite_connection_create
#' @examples
#' # Create temporary in-memory db
#' tmp <- dbConnect(SQLite(), ":memory:")
#' summary(tmp)
#' dbDisconnect(tmp)
#' 
#' # Create temporary on-disk db with bigger cache and safer synchronisation
#' tmp <- dbConnect(SQLite(), "", cache_size = 5000, synchronous = "full")
#' summary(tmp)
#' dbDisconnect(tmp)
setMethod("dbConnect", "SQLiteDriver",
  function(drv, dbname = "", loadable.extensions = TRUE, cache_size = NULL, 
           synchronous = "off", flags = SQLITE_RWC, vfs = NULL) {
    
    stopifnot(is.character(dbname), length(dbname) == 1, !is.na(dbname))
    dbname <- path.expand(dbname)
    loadable.extensions <- as.logical(loadable.extensions)
    vfs <- check_vfs(vfs)
    stopifnot(is.integer(flags), length(flags) == 1)

    conId <- .Call(rsqlite_connection_create, dbname, loadable.extensions, flags, vfs)
    con <- new("SQLiteConnection", 
      Id = conId,
      dbname = dbname,
      loadable.extensions = loadable.extensions,
      flags = flags,
      vfs = vfs
    )
    
    ## experimental PRAGMAs
    if (!is.null(cache_size)) {
      cache_size <- as.integer(cache_size)
      try(dbGetQuery(con, sprintf("PRAGMA cache_size=%d", cache_size)))
    }
    
    if (!is.null(synchronous)) {
      synchronous <- match.arg(synchronous, c("off", "normal", "full"))
      try(dbGetQuery(con, sprintf("PRAGMA synchronous=%s", synchronous)))
    }
    
    con
  }
)

check_vfs <- function(vfs) {
  if (is.null(vfs) || vfs == "") return("")
  
  if (.Platform[["OS.type"]] == "windows") {
    warning("vfs customization not available on this platform.",
      " Ignoring value: vfs = ", vfs, call. = FALSE)
    return("")
  }

  match.arg(vfs, c("unix-posix", "unix-afp", "unix-flock", "unix-dotfile",
    "unix-none"))
}

#' @export
#' @rdname dbConnect-SQLiteDriver-method
setMethod("dbConnect", "SQLiteConnection", function(drv){
  if (drv@dbname %in% c("", ":memory:")) {
    stop("Can't clone a temporary database", call. = FALSE)
  }
  
  dbConnect(SQLite(), drv@dbname, vfs = drv@vfs, flags = drv@flags, 
    loadable.extensions = drv@loadable.extensions)
})


#' @export
#' @rdname dbConnect-SQLiteDriver-method
#' @useDynLib RSQLite rsqlite_connection_destroy
setMethod("dbDisconnect", "SQLiteConnection", function(conn) {
  if (!dbIsValid(conn)) {
    warning("Expired SQLiteConnection.", call. = FALSE)
    return(TRUE)
  }
  
  .Call(rsqlite_connection_destroy, conn@Id)
})
