/*
 *  SPDX-FileCopyrightText: 2019 Jonah Brüchert <jbb@kaidan.im
 *  SPDX-FileCopyrightText: 2020 Nicolas Fella <nicolas.fella@gmx.de>
 *
 *  SPDX-License-Identifier: GPL-3.0-or-later
 */

#include <QImage>
#include <QtTest/QTest>

#include "QrCodeContent.h"
#include "Qrca.h"

constexpr auto VCARD =
    "BEGIN:VCARD\n"
    "VERSION:3.0\n"
    "NAME:Alice\n"
    "TEL;TYPE=CELL:+12 3456 789\n"
    "UID:H3ll0-W0RLD-1M-A-VC4RD\n"
    "END:VCARD";

class QrcaTest : public QObject
{
    Q_OBJECT

private slots:
    void isUrl()
    {
        auto type = QrCodeContent(QStringLiteral("https://kde.org"), Prison::Format::QRCode).contentType();
        QCOMPARE(type, QrCodeContent::Url);
    }

    void isVCard()
    {
        auto type = QrCodeContent(QString::fromUtf8(VCARD), Prison::Format::QRCode).contentType();
        QCOMPARE(type, QrCodeContent::VCard);
    }

    void isOtpToken()
    {
        auto type = QrCodeContent(QStringLiteral("otpauth://totp/ACME%20Co:john.doe@email.com?"
                                                 "secret=HXDMVJECJJWSRB3HWIZR4IFUGFTMXBOZ&"
                                                 "issuer=ACME%20Co&algorithm=SHA1&digits=6&period=30"),
                                  Prison::Format::QRCode)
                        .contentType();
        QCOMPARE(type, QrCodeContent::OtpToken);
    }

    void isText()
    {
        auto type = QrCodeContent(QStringLiteral("Hello World"), Prison::Format::QRCode).contentType();
        QCOMPARE(type, QrCodeContent::Text);
    }

    void isBinary()
    {
        auto type = QrCodeContent(QByteArray("hello\0world"), Prison::Format::QRCode).contentType();
        QCOMPARE(type, QrCodeContent::Binary);
    }

    void VCardName()
    {
        QCOMPARE(Qrca::getVCardName(VCARD), "Alice");
    }

    void isHealthCertificate()
    {
        const auto euDgc = QLatin1String(
            R"(HC1:6BF+70790T9WJWG.FKY*4GO0.O1CV2 O5 N2FBBRW1*70HS8WY04AC*WIFN0AHCD8KD97TK0F90KECTHGWJC0FDC:5AIA%G7X+AQB9746HS80:54IBQF60R6$A80X6S1BTYACG6M+9XG8KIAWNA91AY%67092L4WJCT3EHS8XJC$+DXJCCWENF6OF63W5NW6WF6%JC QE/IAYJC5LEW34U3ET7DXC9 QE-ED8%E.JCBECB1A-:8$96646AL60A60S6Q$D.UDRYA 96NF6L/5QW6307KQEPD09WEQDD+Q6TW6FA7C466KCN9E%961A6DL6FA7D46JPCT3E5JDLA7$Q6E464W5TG6..DX%DZJC6/DTZ9 QE5$CB$DA/D JC1/D3Z8WED1ECW.CCWE.Y92OAGY8MY9L+9MPCG/D5 C5IA5N9$PC5$CUZCY$5Y$527B+A4KZNQG5TKOWWD9FL%I8U$F7O2IBM85CWOC%LEZU4R/BXHDAHN 11$CA5MRI:AONFN7091K9FKIGIY%VWSSSU9%01FO2*FTPQ3C3F)");
        auto content = QrCodeContent(euDgc, Prison::Format::QRCode);
        QCOMPARE(content.contentType(), QrCodeContent::HealthCertificate);
        QCOMPARE(content.isPlainText(), true);

        const auto shc = QLatin1String(R"(shc:/56762909524320603460292437404460312229595326546034602925407728043360287028647167452228092861333145643765314159064022030645045908564355034142454136403706366541713724123638030437562204673740753232392543344332605736010645293153127074242839503869221276673066722326704208032257377630206204102243765868534325585800216728380758570810550562275738232930525766717231740711763612092161592153096328534441634054402943377667210963225304674530596075435638397044612142673774753658337371663741436303047729543963706374067674286421560312673434200333100044570935560632522561743609646455716545030067553205697165595467357259336905215258206553666267536405106110603868756743057374741132623962713132720925323961730873375536454524772309323860117657535973252254070723767559237108283634222457303834393764345021623421595704723907262507213576410771210010632152352426620444450711320937297037117562253509316127771010655641352255685220340306590611732431312500307552044223360477273575682173757329713550044454620375617727270874593576214004383056067405655645662776650638245361115644032910572312043253045263230756306409056010606604063725200845063604227244710632217410642253265635203704100010712124656463756722433434763371605524627435432728726805737158093905762509643823077770552176327727417269601100663232587754412727745655673026606824396103085737106558690435623939775344733266266450413133106160594564000674010021253230285571090670620460223463600968251127726720426268065873047275673154755422592169443774532859213023545832044424075364665766295969731262342066753077580339055069605474)");
        content = QrCodeContent(shc, Prison::Format::QRCode);
        QCOMPARE(content.contentType(), QrCodeContent::HealthCertificate);
        QCOMPARE(content.isPlainText(), true);
    }

    void isTransportTicket()
    {
        const auto uic9183 = QByteArray::fromHex(
            "2355543031303038303030303032302C02141AE7BBA505A4B31E0D0868FD709CE2A03BC71B62021469DCD3887C0357879A93FA86B3A4AAFBB2A1AA1F0000000030333734789C7D91BF"
            "4EC25014C691415D5C7C82262E50A99E7BFA9F0D6C2B0D14931608C84088686C4034A592E893B0B938CACAA283495FC041371FC673E58F98183BDC7EF73BBFFBDDD3D37AA764172C60"
            "00AA0C6080691B56B322E9A9B587E980080C99CA0CB06CCBE6A06B01029061329415EE142BE428CC0485685F715AD6A9DA5A0FC2450E1AF3B781AA6A6AE4A36F39D8ACD6CBDE5FB406"
            "F28AA638A605C0FB65AC7A1D5D750737D17938228BBA218A84CC458184B21426090D252631332086DB4082D731A0234C25A59693D960481B8D67395177D8BFB88DE28CD70D87D9FDA3"
            "30BEA39AFE4FCD580B41C6FB68B885BC50F34F5C31708F8BA257132B9E98714A878EE33959A2A855542F93C76496BC44824FE241E827D3B37E47EAE40F08D0788BFCABF0FB66EA1B99"
            "A9F3692B34138D8F9E6A74B38C3FE772A3387916FCBDDFC1FC50A3CE2767CC7F4A3A4DCBFB24DE7A9B8C73EDD446AEFDBA18FDD3F6E7E6CE647CBF1BD2E66341F488E82D89E98AF802"
            "EAA2924B");
        auto content = QrCodeContent(uic9183, Prison::Format::QRCode);
        QCOMPARE(content.contentType(), QrCodeContent::TransportTicket);
        QCOMPARE(content.isPlainText(), false);

        content = QrCodeContent(QStringLiteral("M1DESMARAIS/LUC       EABC123 YULFRAAC 0834 326J001A0025 100"), Prison::Format::PDF417);
        QCOMPARE(content.contentType(), QrCodeContent::TransportTicket);
        QCOMPARE(content.isPlainText(), true);
    }

    void isGlobalTradeItemNumber()
    {
        auto content = QrCodeContent(QStringLiteral("9783827316608"), Prison::Format::EAN13);
        QCOMPARE(content.contentType(), QrCodeContent::ISBN);
        content = QrCodeContent(QStringLiteral("4337256114837"), Prison::Format::EAN13);
        QCOMPARE(content.contentType(), QrCodeContent::EAN);
        content = QrCodeContent(QStringLiteral("4388844146860"), Prison::Format::EAN13);
        QCOMPARE(content.contentType(), QrCodeContent::EAN);
    }

    void isWifiSetting()
    {
        auto content = QrCodeContent(QStringLiteral("WIFI:T:WPA;S:mynetwork;P:mypass;;"), Prison::Format::QRCode);
        QCOMPARE(content.contentType(), QrCodeContent::WifiSetting);
    }
};

QTEST_GUILESS_MAIN(QrcaTest)

#include "qrcatest.moc"
