# Copyright (c) 2017 The Johns Hopkins University/Applied Physics Laboratory
# All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import logging
import sys

from kmip.core import enums
from kmip.demos import utils
from kmip.pie import client
from kmip.pie import objects


if __name__ == '__main__':
    logger = utils.build_console_logger(logging.INFO)

    # Build and parse arguments
    parser = utils.build_cli_parser(enums.Operation.SIGN)
    opts, args = parser.parse_args(sys.argv[1:])
    config = opts.config

    # Build the client and connect to the server
    with client.ProxyKmipClient(
            config=config,
            config_file=opts.config_file
    ) as client:
        # Create keys to use for derivation
        try:
            signing_key_id = client.register(
                objects.PrivateKey(
                    enums.CryptographicAlgorithm.RSA,
                    1024,
                    (
                        b'\x30\x82\x02\x76\x02\x01\x00\x30\x0d\x06\x09\x2a\x86'
                        b'\x48\x86\xf7\x0d\x01\x01\x01\x05\x00\x04\x82\x02\x60'
                        b'\x30\x82\x02\x5c\x02\x01\x00\x02\x81\x81\x00\xc0\x0f'
                        b'\x0b\x0a\xc5\x72\x36\x81\x71\x6c\x59\xd7\x14\x42\x31'
                        b'\x6a\xb9\xb2\x32\xd8\x91\x65\xab\xaa\x54\xab\xf7\x6a'
                        b'\xdc\xe4\x5c\x9c\x91\x8f\x0c\x90\xa7\x48\x9f\x65\x9c'
                        b'\x3f\xc9\x80\xcb\x51\x05\xf9\x11\x9a\xa2\x13\xd9\x15'
                        b'\x39\x8b\x97\xe8\xf5\xf0\x8b\xa0\xf8\xe5\x34\x47\x2a'
                        b'\xea\x87\xdf\x64\x2a\x16\x5f\xd0\x85\xf5\x8a\x60\xed'
                        b'\x97\xcd\x2b\x96\x72\x04\xf5\xd0\x99\x6a\x53\x90\xc2'
                        b'\xd0\xdf\x38\xa8\x9e\x61\xd0\xb7\x46\xe8\x4e\x48\x7d'
                        b'\x37\x85\x2f\xaf\xba\x70\x06\x31\x07\x65\x13\xb7\x78'
                        b'\xd2\x25\x34\x30\x2c\xf7\x4c\x9e\x17\x02\x03\x01\x00'
                        b'\x01\x02\x81\x80\x16\xc6\xb1\xec\x89\x15\xce\x58\xf3'
                        b'\x76\x82\x00\xfb\xaa\x0d\xea\x36\x33\x67\xcc\x3f\x11'
                        b'\xeb\x95\xbb\x8c\xd9\x3e\x97\x0b\x8d\xe0\x13\x72\xff'
                        b'\x6e\x78\x28\x28\x9f\x08\x34\x98\x54\xe9\xc7\xa6\x09'
                        b'\xaf\x88\xc3\x07\xcf\x8a\xb0\xd4\x59\x23\x8b\x67\x07'
                        b'\x68\x03\x9c\x16\x3d\xa1\xc9\x00\xf3\x31\x0a\x38\x0b'
                        b'\x76\x89\x8d\xb1\x86\x03\xaf\x81\xcb\x47\x37\xd0\x9f'
                        b'\x1c\x99\x6e\xb6\xb9\x7f\x1c\x8a\x07\x88\xb2\x9b\x2b'
                        b'\xc3\xb5\x93\xfd\xfc\x23\x6f\x31\xfb\xf0\xc7\xc1\x83'
                        b'\x86\x6a\x05\xc0\x9a\xfa\x79\x7e\xe3\x02\x80\x06\xa6'
                        b'\x3a\x81\x02\x41\x00\xe8\x06\x53\x54\x96\x8d\xa1\x35'
                        b'\xdf\xf8\x1a\x69\xd1\xbf\x53\x52\xd6\x4f\xe3\xd5\xef'
                        b'\x6d\x31\xd1\x51\xee\x89\x09\x62\x9b\xab\x5b\xfc\x87'
                        b'\xeb\xa7\x22\x1f\x99\x90\x00\x18\xe7\xa5\x78\xe9\x90'
                        b'\xae\xd9\xed\xa4\x25\x91\x11\x0f\x0d\xb1\x1c\xd0\xc4'
                        b'\xbf\x7d\x43\xa7\x02\x41\x00\xd3\xe7\x82\xe9\x84\x59'
                        b'\xff\x1e\x9a\x16\x98\xd3\xaa\xbd\x9f\xae\x56\x52\xe5'
                        b'\x2a\x78\x95\xb1\x61\x27\xc0\xd3\x59\x76\xef\x33\xfd'
                        b'\xc8\xdf\x20\xf8\x79\x92\x90\xe6\x11\x88\xf6\x3b\xd6'
                        b'\xd4\xcc\x43\xc4\x0c\x21\xa0\xec\x29\x68\x6f\x29\xc3'
                        b'\xcb\x58\xa2\x0f\xe0\x11\x02\x40\x38\xd5\x5b\xd2\x0b'
                        b'\x72\xb3\xbb\x53\x9a\x1d\x36\x30\x67\x72\x0c\x87\x6c'
                        b'\x58\x3d\x8e\x01\x2c\x43\xbe\x92\xf4\x44\x35\x40\x36'
                        b'\x50\x38\xe2\x3e\x49\xd9\x24\xee\x63\x84\x72\x95\x43'
                        b'\x46\x03\xc8\x29\xdc\x3d\xc6\x88\x61\x29\x51\x8b\xa4'
                        b'\x07\x8f\xe7\xb1\x94\x08\x5f\x02\x41\x00\xb0\x28\x08'
                        b'\x43\x39\xfc\x5a\xc2\x44\xd4\x3e\x2d\xd0\x05\x9d\x06'
                        b'\x1f\xca\xff\xa9\x43\xdf\x25\x3b\x20\x02\x03\x70\x9f'
                        b'\x17\x91\x40\x0b\x49\xba\x2d\xf5\x5a\xab\x4c\x27\x0d'
                        b'\x95\xac\xff\x15\x9d\xcd\x43\xdf\xd5\xe0\xe2\x12\x36'
                        b'\x38\x1b\x1f\x22\x1f\x47\x72\x2d\x11\x02\x40\x20\x9b'
                        b'\x55\xb5\x2d\xce\x33\x45\xed\x29\x2a\x95\xa2\x2b\x03'
                        b'\xa4\x2b\xd3\x75\x8d\xe6\xa1\x24\x0d\x5a\xc4\xe2\x96'
                        b'\x80\x90\x74\xc3\x8d\xaf\x17\x69\x4d\x70\x1d\x62\xaf'
                        b'\x79\x94\xfe\x74\xd3\x7b\x40\x0c\x60\x36\xde\x2c\x51'
                        b'\x4a\x66\x66\x73\x10\x9f\xd7\x86\x7f\x70'
                    ),
                    enums.KeyFormatType.PKCS_8,
                    masks=[
                        enums.CryptographicUsageMask.SIGN
                    ]
                )
            )
            logger.info("Successfully created a new signing key.")
            logger.info("Signing Key ID: {0}".format(signing_key_id))
        except Exception as e:
            logger.error(e)
            sys.exit(-1)

        # Activate the signing key.
        try:
            client.activate(signing_key_id)
            logger.info(
                "Signing key {0} has been activated.".format(signing_key_id)
            )
        except Exception as e:
            logger.error(e)
            sys.exit(-1)

        # Generate a signature.
        try:
            result = client.sign(
                (
                    b'\xe1\xc0\xf9\x8d\x53\xf8\xf8\xb1\x41\x90\x57\xd5\xb9\xb1'
                    b'\x0b\x07\xfe\xea\xec\x32\xc0\x46\x3a\x4d\x68\x38\x2f\x53'
                    b'\x1b\xa1\xd6\xcf\xe4\xed\x38\xa2\x69\x4a\x34\xb9\xc8\x05'
                    b'\xad\xf0\x72\xff\xbc\xeb\xe2\x1d\x8d\x4b\x5c\x0e\x8c\x33'
                    b'\x45\x2d\xd8\xf9\xc9\xbf\x45\xd1\xe6\x33\x75\x11\x33\x58'
                    b'\x82\x29\xd2\x93\xc6\x49\x6b\x7c\x98\x3c\x2c\x72\xbd\x21'
                    b'\xd3\x39\x27\x2d\x78\x28\xb0\xd0\x9d\x01\x0b\xba\xd3\x18'
                    b'\xd9\x98\xf7\x04\x79\x67\x33\x8a\xce\xfd\x01\xe8\x74\xac'
                    b'\xe5\xf8\x6d\x2a\x60\xf3\xb3\xca\xe1\x3f\xc5\xc6\x65\x08'
                    b'\xcf\xb7\x23\x78\xfd\xd6\xc8\xde\x24\x97\x65\x10\x3c\xe8'
                    b'\xfe\x7c\xd3\x3a\xd0\xef\x16\x86\xfe\xb2\x5e\x6a\x35\xfb'
                    b'\x64\xe0\x96\xa4'
                ),
                uid=signing_key_id,
                cryptographic_parameters={
                    'cryptographic_algorithm':
                        enums.CryptographicAlgorithm.RSA,
                    'hashing_algorithm': enums.HashingAlgorithm.SHA_1,
                    'padding_method': enums.PaddingMethod.PSS
                },
            )
            logger.info("Signature: {0}".format(result))
        except Exception as e:
            logger.error(e)
