"""Collection of bulk systems.

From this paper:

  Philipp Haas, Fabien Tran, Peter Blaha

  Calculation of the lattice constant of solids with semilocal functionals

  Phys. Rev. B 79, 085104 (2009)

  dx.doi.org/10.1103/PhysRevB.79.085104

  Data extracted to csv files with:

  * lattice constant::

     pdftotext -layout -f 3 -l 4 e085104.pdf - | sed -n '/Solid/,$p' \
     | sed -n '/me /q;p' | grep -E -v "085104|TRAN|TABLE" \
     | sed '/^$/d' | sed -e 1b -e '/^Solid/d' \
     | sed 's/Expt./Expt1. Expt2./' | sed 's/Solid/Solid Str/' \
     | sed -e 's/\s\+/,/g'

  * erratum::

     pdftotext -layout e209902.pdf - | sed -n '/Solid/,$p' | sed -n '/me /q;p' \
     | sed '/^$/d' \
     | sed 's/Expt./Expt1. Expt2./' | sed 's/Solid/Solid Str/' \
     | sed -e 's/\s\+/,/g'

"""

import ase.units as units
from ase.lattice import bulk
from ase.tasks.io import read_json
from ase.tasks.bulk import BulkTask
from ase.utils.eos import EquationOfState

strukturbericht = {'A1': 'fcc',
                   'A2': 'bcc',
                   'A4': 'diamond',
                   'B1': 'rocksalt',
                   'B2': 'cesiumchloride',
                   'B3': 'zincblende',
                   'C1': 'fluorite'}


class HaasTranBlahaBulkCollection:
    # e085104.pdf, lattice constant
    data1 = """
    Solid,Str,LDA,SOGGA,PBEsol,WC,AM05,TPSS,PBE,Expt1.,Expt2.
    Li,A2,3.363,3.435,3.433,3.449,3.456,3.455,3.435,3.451,3.477
    Na,A2,4.047,4.175,4.170,4.199,4.209,4.237,4.196,4.209,4.225
    K,A2,5.045,5.231,5.213,5.256,5.293,5.352,5.282,5.212,5.225
    Rb,A2,5.374,5.605,5.579,5.609,5.692,5.749,5.670,5.577,5.585
    Ca,A1,5.333,5.469,5.456,5.458,5.491,5.533,5.530,5.556,5.565
    Sr,A1,5.786,5.930,5.917,5.914,5.975,6.018,6.027,6.040,6.048
    Ba,A2,4.754,4.881,4.881,4.870,4.963,4.991,5.030,5.002,5.007
    V,A2,2.932,2.959,2.963,2.965,2.961,2.979,3.001,3.024,3.028
    Nb,A2,3.250,3.268,3.274,3.280,3.271,3.297,3.312,3.294,3.296
    Ta,A2,3.257,3.280,3.285,3.290,3.281,3.300,3.323,3.299,3.301
    Mo,A2,3.116,3.126,3.133,3.139,3.128,3.151,3.164,3.141,3.144
    W,A2,3.143,3.155,3.162,3.167,3.156,3.173,3.191,3.160,3.162
    Fe,A2,2.754,2.783,2.790,2.793,2.787,2.804,2.833,2.853,2.861
    Rh,A1,3.759,3.772,3.785,3.795,3.777,3.807,3.834,3.793,3.798
    Ir,A1,3.828,3.834,3.847,3.857,3.837,3.867,3.887,3.831,3.835
    Ni,A1,3.423,3.453,3.463,3.468,3.461,3.478,3.518,3.508,3.516
    Pd,A1,3.848,3.867,3.882,3.892,3.878,3.909,3.948,3.876,3.881
    Pt,A1,3.909,3.917,3.932,3.944,3.923,3.958,3.985,3.913,3.916
    Cu,A1,3.522,3.557,3.570,3.573,3.568,3.585,3.632,3.596,3.603
    Ag,A1,4.007,4.038,4.058,4.065,4.059,4.093,4.152,4.062,4.069
    Au,A1,4.047,4.061,4.081,4.092,4.074,4.115,4.154,4.062,4.065
    Al,A1,3.983,4.008,4.018,4.023,4.008,4.015,4.041,4.019,4.032
    C,A4,3.536,3.552,3.557,3.558,3.553,3.573,3.575,3.544,3.567
    Si,A4,5.407,5.425,5.438,5.437,5.439,5.466,5.475,5.415,5.430
    Ge,A4,5.632,5.662,5.684,5.686,5.688,5.734,5.769,5.639,5.652
    Sn,A4,6.481,6.521,6.547,6.548,6.566,6.621,6.661,6.474,6.482
    Pb,A1,4.874,4.899,4.931,4.936,4.945,4.997,5.048,4.912,4.916
    Th,A1,4.920,4.928,4.959,4.977,4.954,5.032,5.056,5.071,5.074
    LiF,B1,3.919,4.008,4.013,4.017,4.046,4.047,4.071,3.960,4.010
    LiCl,B1,4.986,5.062,5.081,5.087,5.142,5.151,5.167,5.072,5.106
    NaF,B1,4.507,4.637,4.635,4.652,4.682,4.702,4.709,4.576,4.609
    NaCl,B1,5.484,5.608,5.619,5.637,5.696,5.715,5.714,5.565,5.595
    MgO,B1,4.169,4.217,4.222,4.223,4.228,4.244,4.261,4.186,4.207
    MgS,B1,5.139,5.174,5.190,5.195,5.197,5.237,5.238,5.182,5.202
    CaO,B1,4.719,4.771,4.778,4.777,4.790,4.819,4.841,4.787,4.803
    TiC,B1,4.266,4.294,4.302,4.303,4.297,4.336,4.339,4.317,4.330
    TiN,B1,4.178,4.202,4.210,4.214,4.206,4.241,4.254,4.228,4.239
    ZrC,B1,4.647,4.664,4.675,4.680,4.670,4.711,4.715,4.688,4.696
    ZrN,B1,4.532,4.549,4.560,4.565,4.555,4.590,4.602,4.574,4.585
    HfC,B1,4.578,4.602,4.613,4.618,4.606,4.646,4.660,4.627,4.638
    HfN,B1,4.482,4.506,4.515,4.520,4.510,4.543,4.560,4.512,4.520
    VC,B1,4.095,4.114,4.123,4.129,4.116,4.151,4.162,4.148,4.160
    VN,B1,4.050,4.071,4.081,4.087,4.075,4.112,4.125,4.126,4.141
    NbC,B1,4.432,4.446,4.457,4.462,4.448,4.487,4.491,4.462,4.470
    NbN,B1,4.361,4.374,4.385,4.392,4.378,4.419,4.426,4.383,4.392
    FeAl,B2,2.812,2.837,2.840,2.843,2.839,2.850,2.869,2.882,2.889
    CoAl,B2,2.795,2.820,2.824,2.826,2.822,2.833,2.853,2.855,2.861
    NiAl,B2,2.834,2.859,2.864,2.866,2.862,2.873,2.894,2.882,2.887
    BN,B3,3.585,3.605,3.610,3.610,3.607,3.628,3.629,3.585,3.607
    BP,B3,4.496,4.514,4.525,4.526,4.522,4.553,4.553,4.520,4.538
    BAs,B3,4.740,4.761,4.775,4.778,4.772,4.808,4.816,4.760,4.777
    AlP,B3,5.440,5.464,5.476,5.474,5.479,5.504,5.513,5.445,5.460
    AlAs,B3,5.636,5.668,5.681,5.680,5.687,5.713,5.734,5.646,5.658
    GaN,B3,4.463,4.492,4.502,4.504,4.501,4.536,4.551,4.520,4.531
    GaP,B3,5.401,5.429,5.447,5.448,5.451,5.498,5.514,5.435,5.448
    GaAs,B3,5.616,5.650,5.670,5.672,5.678,5.724,5.757,5.637,5.648
    InP,B3,5.839,5.869,5.890,5.890,5.898,5.958,5.968,5.856,5.866
    InAs,B3,6.038,6.076,6.098,6.100,6.111,6.167,6.195,6.044,6.054
    SiC,B3,4.333,4.354,4.360,4.360,4.357,4.371,4.384,4.340,4.358
    CeO2,C1,5.371,5.396,5.410,5.415,5.414,5.454,5.475,5.393,5.401
    """

    data_ref = {}
    for l in data1.split():
        if 'Solid' not in l:
            l1 = l.split(',')
            data_ref[l1[0]] = [l1[1]]
            data_ref[l1[0]].extend([float(v) for v in l1[2:]])

    # e209902.pdf, lattice constant
    data2 = """
    Solid,Str,LDA,SOGGA,PBEsol,WC,AM05,TPSS,PBE,Expt1.,Expt2.
    Rb,A2,5.372,5.592,5.570,5.608,5.676,5.749,5.668,5.577,5.585
    Pd,A1,3.840,3.859,3.875,3.884,3.870,3.905,3.942,3.876,3.881
    Ge,A4,5.625,5.656,5.678,5.681,5.683,5.724,5.763,5.639,5.652
    Sn,A4,6.473,6.513,6.540,6.541,6.557,6.615,6.655,6.474,6.482
    LiF,B1,3.911,4.003,4.006,4.012,4.038,4.039,4.068,3.960,4.010
    LiCl,B1,4.965,5.042,5.063,5.069,5.118,5.115,5.150,5.072,5.106
    NaCl,B1,5.465,5.594,5.605,5.623,5.682,5.704,5.700,5.565,5.595
    MgS,B1,5.132,5.169,5.184,5.189,5.191,5.224,5.231,5.182,5.202
    HfN,B1,4.472,4.497,4.506,4.511,4.501,4.541,4.550,4.512,4.520
    AlAs,B3,5.630,5.663,5.676,5.676,5.680,5.706,5.729,5.646,5.658
    GaAs,B3,5.607,5.642,5.662,5.664,5.670,5.715,5.747,5.637,5.648
    InP,B3,5.825,5.862,5.880,5.882,5.886,5.951,5.961,5.856,5.866
    InAs,B3,6.027,6.065,6.088,6.091,6.102,6.161,6.187,6.044,6.054
    CeO2,C1,5.356,5.379,5.394,5.399,5.397,5.439,5.459,5.393,5.401
    """

    for l in data2.split():
        if 'Solid' not in l:
            l1 = l.split(',')
            data_ref[l1[0]] = [l1[1]]
            data_ref[l1[0]].extend([float(v) for v in l1[2:]])

    # F. Tran, private communication, lattice constant
    data3 = """
    Solid,Str,LDA,SOGGA,PBEsol,WC,AM05,TPSS,PBE,Expt2.
    Li,A2,3.363,3.435,3.433,3.449,3.456,3.455,3.435,3.451
    Na,A2,4.047,4.175,4.170,4.199,4.209,4.237,4.196,4.209
    K,A2,5.045,5.231,5.213,5.256,5.293,5.352,5.282,5.212
    Rb,A2,5.372,5.592,5.570,5.608,5.676,5.749,5.668,5.577
    Ca,A1,5.333,5.469,5.456,5.458,5.491,5.533,5.530,5.556
    Sr,A1,5.786,5.930,5.917,5.914,5.975,6.018,6.027,6.040
    Ba,A2,4.754,4.881,4.881,4.870,4.963,4.991,5.030,5.002
    V,A2,2.932,2.959,2.963,2.965,2.961,2.979,3.001,3.024
    Nb,A2,3.250,3.268,3.274,3.280,3.271,3.297,3.312,3.294
    Ta,A2,3.257,3.280,3.285,3.290,3.281,3.300,3.323,3.299
    Mo,A2,3.116,3.126,3.133,3.139,3.128,3.151,3.164,3.141
    W,A2,3.143,3.155,3.162,3.167,3.156,3.173,3.191,3.160
    Fe,A2,2.754,2.783,2.790,2.793,2.787,2.804,2.833,2.853
    Rh,A1,3.759,3.772,3.785,3.795,3.777,3.807,3.834,3.793
    Ir,A1,3.828,3.834,3.847,3.857,3.837,3.867,3.887,3.831
    Ni,A1,3.423,3.453,3.463,3.468,3.461,3.478,3.518,3.508
    Pd,A1,3.840,3.859,3.875,3.884,3.870,3.905,3.942,3.876
    Pt,A1,3.909,3.917,3.932,3.944,3.923,3.958,3.985,3.913
    Cu,A1,3.522,3.557,3.570,3.573,3.568,3.585,3.632,3.596
    Ag,A1,4.007,4.038,4.058,4.065,4.059,4.093,4.152,4.062
    Au,A1,4.047,4.061,4.081,4.092,4.074,4.115,4.154,4.062
    Al,A1,3.983,4.008,4.018,4.023,4.008,4.015,4.041,4.019
    C,A4,3.536,3.552,3.557,3.558,3.553,3.573,3.575,3.544
    Si,A4,5.407,5.425,5.438,5.437,5.439,5.466,5.475,5.415
    Ge,A4,5.625,5.656,5.678,5.681,5.683,5.724,5.763,5.639
    Sn,A4,6.473,6.513,6.540,6.541,6.557,6.615,6.655,6.474
    Pb,A1,4.874,4.899,4.931,4.936,4.945,4.997,5.048,4.912
    Th,A1,4.920,4.928,4.959,4.977,4.954,5.032,5.056,5.071
    LiF,B1,3.911,4.003,4.006,4.012,4.038,4.039,4.068,3.960
    LiCl,B1,4.965,5.042,5.063,5.069,5.118,5.115,5.150,5.072
    NaF,B1,4.507,4.637,4.635,4.652,4.682,4.702,4.709,4.576
    NaCl,B1,5.465,5.594,5.605,5.623,5.682,5.704,5.700,5.565
    MgO,B1,4.169,4.217,4.222,4.223,4.228,4.244,4.261,4.186
    MgS,B1,5.132,5.169,5.184,5.189,5.191,5.224,5.231,5.182
    CaO,B1,4.719,4.771,4.778,4.777,4.790,4.819,4.841,4.787
    TiC,B1,4.266,4.294,4.302,4.303,4.297,4.336,4.339,4.317
    TiN,B1,4.178,4.202,4.210,4.214,4.206,4.241,4.254,4.228
    ZrC,B1,4.647,4.664,4.675,4.680,4.670,4.711,4.715,4.688
    ZrN,B1,4.532,4.549,4.560,4.565,4.555,4.590,4.602,4.574
    HfC,B1,4.578,4.602,4.613,4.618,4.606,4.646,4.660,4.627
    HfN,B1,4.472,4.497,4.506,4.511,4.501,4.541,4.550,4.512
    VC,B1,4.095,4.114,4.123,4.129,4.116,4.151,4.162,4.148
    VN,B1,4.050,4.071,4.081,4.087,4.075,4.112,4.125,4.126
    NbC,B1,4.432,4.446,4.457,4.462,4.448,4.487,4.491,4.462
    NbN,B1,4.361,4.374,4.385,4.392,4.378,4.419,4.426,4.383
    FeAl,B2,2.812,2.837,2.840,2.843,2.839,2.850,2.869,2.882
    CoAl,B2,2.795,2.820,2.824,2.826,2.822,2.833,2.853,2.855
    NiAl,B2,2.834,2.859,2.864,2.866,2.862,2.873,2.894,2.882
    BN,B3,3.585,3.605,3.610,3.610,3.607,3.628,3.629,3.585
    BP,B3,4.496,4.514,4.525,4.526,4.522,4.553,4.553,4.520
    BAs,B3,4.740,4.761,4.775,4.778,4.772,4.808,4.816,4.760
    AlP,B3,5.440,5.464,5.476,5.474,5.479,5.504,5.513,5.445
    AlAs,B3,5.630,5.663,5.676,5.676,5.680,5.706,5.729,5.646
    GaN,B3,4.463,4.492,4.502,4.504,4.501,4.536,4.551,4.520
    GaP,B3,5.401,5.429,5.447,5.448,5.451,5.498,5.514,5.435
    GaAs,B3,5.607,5.642,5.662,5.664,5.670,5.715,5.747,5.637
    InP,B3,5.825,5.862,5.880,5.882,5.886,5.951,5.961,5.856
    InAs,B3,6.027,6.065,6.088,6.091,6.102,6.161,6.187,6.044
    SiC,B3,4.333,4.354,4.360,4.360,4.357,4.371,4.384,4.340
    CeO2,C1,5.356,5.379,5.394,5.399,5.397,5.439,5.459,5.393
    """

    data_priv = {}
    for l in data3.split():
        if 'Solid' not in l:
            l1 = l.split(',')
            data_priv[l1[0]] = [l1[1]]
            data_priv[l1[0]].extend([float(v) for v in l1[2:]])

    # F. Tran, private communication, bulk modulus
    data4 = """
    Solid,Str,LDA,SOGGA,PBEsol,WC,AM05,TPSS,PBE,Expt2.
    Li,A2,15.2,13.7,13.7,13.4,13.2,13.5,14.0,13.0
    Na,A2,9.41,7.91,7.88,7.32,7.50,7.37,7.85,7.50
    K,A2,4.50,3.69,3.73,3.49,3.45,3.34,3.61,3.70
    Rb,A2,3.59,2.82,2.88,2.71,2.59,2.53,2.77,3.10
    Ca,A1,18.7,17.8,18.0,17.4,17.6,17.4,17.3,18.4
    Sr,A1,14.4,12.2,12.5,12.2,11.8,11.5,11.4,12.3
    Ba,A2,10.6,9.0,9.3,9.2,8.5,8.2,8.2,9.3
    V,A2,213,203,201,198,199,196,183,162
    Nb,A2,193,186,184,183,183,180,171,170
    Ta,A2,219,210,208,207,208,206,195,194
    Mo,A2,294,288,282,279,284,277,260,272
    W,A2,336,329,322,320,325,319,303,296
    Fe,A2,254,233,226,227,224,220,194,168
    Rh,A1,320,309,298,292,300,284,259,269
    Ir,A1,395,392,377,368,383,360,336,355
    Ni,A1,259,239,232,231,231,221,200,186
    Pd,A1,231,218,208,207,207,194,170,195
    Pt,A1,301,296,282,276,285,265,242,277
    Cu,A1,191,177,170,168,168,165,141,142
    Ag,A1,140,127,120,119,116,109,91,109
    Au,A1,200,192,180,177,179,165,144,167
    Al,A1,84.3,86.6,83.0,80.6,86.3,87.2,79.2,79.4
    C,A4,469,458,452,451,453,431,434,443
    Si,A4,96.4,96.4,93.8,94.0,93.2,90.5,88.7,99.2
    Ge,A4,72.7,71.1,68.1,67.8,66.5,62.5,59.5,75.8
    Sn,A4,45.7,44.0,42.3,42.4,40.3,38.1,36.3,53.0
    Pb,A1,50.5,47.5,44.6,45.2,41.7,40.0,35.6,43.0
    Th,A1,66.9,64.1,62.7,63.0,61.7,60.0,58.0,54.3
    LiF,B1,88.2,73.4,72.8,72.4,66.1,66.8,67.5,69.8
    LiCl,B1,42.0,36.6,35.9,35.7,31.4,33.0,32.2,35.4
    NaF,B1,61.7,48.5,48.5,45.4,44.3,42.8,44.5,51.4
    NaCl,B1,32.7,26.2,26.1,24.7,22.4,23.2,23.7,26.6
    MgO,B1,174,158,158,158,154,154,149,165
    MgS,B1,84.0,80.3,78.7,78.2,76.7,76.2,74.4,78.9
    CaO,B1,129,116,115,116,111,109,105,114
    TiC,B1,283,275,270,267,270,255,248,233
    TiN,B1,322,307,302,300,301,289,277,277
    ZrC,B1,248,241,237,236,236,226,222,230
    ZrN,B1,286,276,271,270,271,262,250,215
    HfC,B1,257,247,240,238,244,223,218,200
    HfN,B1,309,295,290,289,290,279,265,306
    VC,B1,349,338,331,328,333,314,307,303
    VN,B1,365,352,343,340,344,326,312,233
    NbC,B1,335,330,324,319,325,306,301,315
    NbN,B1,353,344,335,333,338,323,309,292
    FeAl,B2,208,197,194,193,194,191,180,136
    CoAl,B2,207,196,193,192,193,191,179,162
    NiAl,B2,185,176,173,171,173,171,159,156
    BN,B3,405,391,387,387,387,374,373,373
    BP,B3,176,173,170,170,169,162,162,152
    BAs,B3,148,145,141,141,140,133,132,138
    AlP,B3,89.9,88.8,87.1,87.1,85.8,85.9,82.6,86.0
    AlAs,B3,75.5,73.8,72.4,72.5,70.8,70.5,67.0,74.0
    GaN,B3,204,194,190,190,189,179,173,190
    GaP,B3,90.6,87.9,85.2,85.5,83.4,78.6,77.0,87.4
    GaAs,B3,74.7,72.1,69.5,69.5,67.5,64.1,61.0,75.6
    InP,B3,72.0,69.5,67.4,67.4,65.3,60.7,59.9,71.0
    InAs,B3,60.7,58.1,56.0,56.6,53.8,50.7,48.8,58.0
    SiC,B3,230,225,222,221,222,217,212,225
    CeO2,C1,209,198,195,195,190,189,176,220
    """

    data_B_priv = {}
    for l in data4.split():
        if 'Solid' not in l:
            l1 = l.split(',')
            data_B_priv[l1[0]] = [l1[1]]
            data_B_priv[l1[0]].extend([float(v) for v in l1[2:]])

    if 1:
        # use private communication data
        data = data_priv.copy()
        B = data_B_priv.copy()
        names = [d.split(',')[0] for d in data3.split()][1:]
        labels = data3.split()[0].split(',')
    else:
        # use private published data (no bulk modulus)
        data = data_ref.copy()
        B = None
        names = [d.split(',')[0] for d in data1.split()][1:]
        labels = data1.split()[0].split(',')

    def __init__(self, xc='PBE'):
        self.xc = xc

    def __getitem__(self, name):
        d = self.data[name]
        # the index of functional in labels less one
        # (solid is already as key in d)
        a = d[self.labels.index(self.xc) - 1]
        cs = strukturbericht[d[0]]
        b = bulk(name, crystalstructure=cs, a=a)
        M = {'Fe': 2.3, 'Co': 1.2, 'Ni': 0.6}.get(name)
        if M is not None:
            b.set_initial_magnetic_moments([M] * len(b))

        return b

    def keys(self):
        return self.names


class HaasTranBlahaBulkTask(BulkTask):
    def __init__(self, xc='PBE', **kwargs):
        BulkTask.__init__(self,
                          collection=HaasTranBlahaBulkCollection(xc),
                          **kwargs)

        self.summary_keys = ['energy', 'fitted energy',
                             'crystal structure',
                             'strukturbericht',
                             'lattice constant', 'lattice constant error [%]',
                             'volume', 'volume error [%]',
                             'B', 'B error [%]']

    def analyse(self, atomsfile=None):
        try:
            BulkTask.analyse(self)
        except ValueError: # allow fit outside of range
            pass

        for name, data in self.data.items():
            if 'strains' in data:
                atoms = self.create_system(name)
                # use relaxed volume if present
                if 'relaxed volume' in data:
                    volume = data['relaxed volume']
                else:
                    volume = atoms.get_volume()
                volumes = data['strains']**3 * volume
                energies = data['energies']
                # allow selection of eos type independent of data
                if self.eos is not None:
                    eos = EquationOfState(volumes, energies, self.eos)
                else:
                    eos = EquationOfState(volumes, energies)
                try:
                    v, e, B = eos.fit()
                except ValueError:
                    pass
                else:
                    data['fitted energy'] = e
                    data['volume'] = v
                    data['B'] = B
                # with respect tot the reference volume
                data['volume error [%]'] = (data['volume'] / atoms.get_volume() - 1) * 100
                if self.collection.B:
                    i = self.collection.labels.index(self.collection.xc) - 1
                    B = self.collection.B[name][i] * units.kJ * 1e-24
                    data['B error [%]'] = (data['B'] / B - 1) * 100
                else:
                    data['B error [%]'] = None
                data['strukturbericht'] = self.collection.data[name][0]
                data['crystal structure'] = strukturbericht[data['strukturbericht']]
                # calculate lattice constant from volume
                cs = data['crystal structure']
                if cs == 'bcc':
                    a0 = (volume*2)**(1/3.)
                    a = (data['volume']*2)**(1/3.)
                elif cs == 'cesiumchloride':
                    a0 = (volume)**(1/3.)
                    a = (data['volume'])**(1/3.)
                elif cs in ['fcc',
                            'diamond',
                            'zincblende',
                            'rocksalt',
                            'fluorite']:
                    a0 = (volume*4)**(1/3.)
                    a = (data['volume']*4)**(1/3.)
                i = self.collection.labels.index(self.collection.xc) - 1
                a0_ref = self.collection.data[name][i]
                if 'relaxed volume' not in data:
                    # no volume relaxation performed - volume equals the reference one
                    assert abs(a0 - a0_ref) < 1.e-4
                data['lattice constant'] = a
                data['lattice constant error [%]'] = (a - a0_ref) / a0_ref * 100

        if atomsfile:
            # MDTMP: TODO
            atomdata = read_json(atomsfile)
            for name, data in self.data.items():
                atoms = self.create_system(name)
                e = -data['energy']
                for atom in atoms:
                    e += atomdata[atom.symbol]['energy']
                e /= len(atoms)
                data['cohesive energy'] = e
                if self.collection.xc == 'PBE':
                    eref = self.collection.data[name][7]
                else:
                    eref = self.collection.data[name][9]
                data['cohesive energy error [%]'] = (e / eref - 1) * 100

            self.summary_keys += ['cohesive energy',
                                  'cohesive energy error [%]']

if __name__ == '__main__':
    # run with emt
    from ase.tasks.main import run
    run(calcname='emt', task=HaasTranBlahaBulkTask(fit=(5, 0.02)))
