//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================

#include <vtkm/cont/testing/MakeTestDataSet.h>
#include <vtkm/cont/testing/Testing.h>

#include <vtkm/filter/entity_extraction/ExtractPoints.h>

using vtkm::cont::testing::MakeTestDataSet;

namespace
{

class TestingExtractPoints
{
public:
  static void TestUniformByBox0()
  {
    std::cout << "Testing extract points with implicit function (box):" << std::endl;
    vtkm::cont::DataSet dataset = MakeTestDataSet().Make3DUniformDataSet1();

    // Implicit function
    vtkm::Vec3f minPoint(1.f, 1.f, 1.f);
    vtkm::Vec3f maxPoint(3.f, 3.f, 3.f);
    vtkm::Box box(minPoint, maxPoint);

    // Setup and run filter to extract by volume of interest
    vtkm::filter::entity_extraction::ExtractPoints extractPoints;
    extractPoints.SetImplicitFunction(box);
    extractPoints.SetExtractInside(true);
    extractPoints.SetCompactPoints(true);

    vtkm::cont::DataSet output = extractPoints.Execute(dataset);
    VTKM_TEST_ASSERT(test_equal(output.GetNumberOfCells(), 27), "Wrong result for ExtractPoints");

    vtkm::cont::ArrayHandle<vtkm::Float32> outPointData;
    output.GetField("pointvar").GetData().AsArrayHandle(outPointData);

    VTKM_TEST_ASSERT(
      test_equal(output.GetCellSet().GetNumberOfPoints(), outPointData.GetNumberOfValues()),
      "Data/Geometry mismatch for ExtractPoints filter");

    VTKM_TEST_ASSERT(outPointData.ReadPortal().Get(0) == 99.0f, "Wrong point field data");
    VTKM_TEST_ASSERT(outPointData.ReadPortal().Get(26) == 97.0f, "Wrong point field data");
  }

  static void TestUniformByBox1()
  {
    std::cout << "Testing extract points with implicit function (box):" << std::endl;
    vtkm::cont::DataSet dataset = MakeTestDataSet().Make3DUniformDataSet1();

    // Implicit function
    vtkm::Vec3f minPoint(1.f, 1.f, 1.f);
    vtkm::Vec3f maxPoint(3.f, 3.f, 3.f);
    vtkm::Box box(minPoint, maxPoint);

    // Setup and run filter to extract by volume of interest
    vtkm::filter::entity_extraction::ExtractPoints extractPoints;
    extractPoints.SetImplicitFunction(box);
    extractPoints.SetExtractInside(false);
    extractPoints.SetCompactPoints(true);

    vtkm::cont::DataSet output = extractPoints.Execute(dataset);
    VTKM_TEST_ASSERT(test_equal(output.GetNumberOfCells(), 98), "Wrong result for ExtractPoints");

    vtkm::cont::ArrayHandle<vtkm::Float32> outPointData;
    output.GetField("pointvar").GetData().AsArrayHandle(outPointData);

    VTKM_TEST_ASSERT(
      test_equal(output.GetCellSet().GetNumberOfPoints(), outPointData.GetNumberOfValues()),
      "Data/Geometry mismatch for ExtractPoints filter");

    for (vtkm::Id i = 0; i < output.GetCellSet().GetNumberOfPoints(); i++)
    {
      VTKM_TEST_ASSERT(outPointData.ReadPortal().Get(i) == 0.0f, "Wrong point field data");
    }
  }

  static void TestUniformBySphere()
  {
    std::cout << "Testing extract points with implicit function (sphere):" << std::endl;
    vtkm::cont::DataSet dataset = MakeTestDataSet().Make3DUniformDataSet1();

    // Implicit function
    vtkm::Vec3f center(2.f, 2.f, 2.f);
    vtkm::FloatDefault radius(1.8f);
    vtkm::Sphere sphere(center, radius);

    // Setup and run filter to extract by volume of interest
    vtkm::filter::entity_extraction::ExtractPoints extractPoints;
    extractPoints.SetImplicitFunction(sphere);
    extractPoints.SetExtractInside(true);

    vtkm::cont::DataSet output = extractPoints.Execute(dataset);
    VTKM_TEST_ASSERT(test_equal(output.GetNumberOfCells(), 27), "Wrong result for ExtractPoints");
  }

  static void TestExplicitByBox0()
  {
    std::cout << "Testing extract points with implicit function (box):" << std::endl;
    vtkm::cont::DataSet dataset = MakeTestDataSet().Make3DExplicitDataSet5();

    // Implicit function
    vtkm::Vec3f minPoint(0.f, 0.f, 0.f);
    vtkm::Vec3f maxPoint(1.f, 1.f, 1.f);
    vtkm::Box box(minPoint, maxPoint);

    // Setup and run filter to extract by volume of interest
    vtkm::filter::entity_extraction::ExtractPoints extractPoints;
    extractPoints.SetImplicitFunction(box);
    extractPoints.SetExtractInside(true);

    vtkm::cont::DataSet output = extractPoints.Execute(dataset);
    VTKM_TEST_ASSERT(test_equal(output.GetNumberOfCells(), 8), "Wrong result for ExtractPoints");
  }

  static void TestExplicitByBox1()
  {
    std::cout << "Testing extract points with implicit function (box):" << std::endl;
    vtkm::cont::DataSet dataset = MakeTestDataSet().Make3DExplicitDataSet5();

    // Implicit function
    vtkm::Vec3f minPoint(0.f, 0.f, 0.f);
    vtkm::Vec3f maxPoint(1.f, 1.f, 1.f);
    vtkm::Box box(minPoint, maxPoint);

    // Setup and run filter to extract by volume of interest
    vtkm::filter::entity_extraction::ExtractPoints extractPoints;
    extractPoints.SetImplicitFunction(box);
    extractPoints.SetExtractInside(false);

    vtkm::cont::DataSet output = extractPoints.Execute(dataset);
    VTKM_TEST_ASSERT(test_equal(output.GetNumberOfCells(), 3), "Wrong result for ExtractPoints");
  }

  void operator()() const
  {
    TestingExtractPoints::TestUniformByBox0();
    TestingExtractPoints::TestUniformByBox1();
    TestingExtractPoints::TestUniformBySphere();
    TestingExtractPoints::TestExplicitByBox0();
    TestingExtractPoints::TestExplicitByBox1();
  }
};
}

int UnitTestExtractPointsFilter(int argc, char* argv[])
{
  return vtkm::cont::testing::Testing::Run(TestingExtractPoints(), argc, argv);
}
