/*
 * Copyright © INRIA 2009
 * Brice Goglin <Brice.Goglin@inria.fr>
 *
 * This software is a computer program whose purpose is to provide
 * a fast inter-process communication subsystem.
 *
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-B license and that you accept its terms.
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sched.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/time.h>

#include "knem_io.h"

#define ITER 256
#define WARMUP 16
#define MIN 1024
#define MAX (4096*4096+1)
#define MULTIPLIER 2
#define INCREMENT 0
#define PAUSE_MS 100

static void
usage(int argc, char *argv[])
{
  fprintf(stderr, "%s [options]\n", argv[0]);
  fprintf(stderr, " -S <n>\tchange the start length [%d]\n", MIN);
  fprintf(stderr, " -E <n>\tchange the end length [%d]\n", MAX);
  fprintf(stderr, " -M <n>\tchange the length multiplier [%d]\n", MULTIPLIER);
  fprintf(stderr, " -I <n>\tchange the length increment [%d]\n", INCREMENT);
  fprintf(stderr, " -N <n>\tchange number of iterations [%d]\n", ITER);
  fprintf(stderr, " -W <n>\tchange number of warmup iterations [%d]\n", WARMUP);
  fprintf(stderr, " -P <n>\tpause (in milliseconds) between lengths [%d]\n", PAUSE_MS);
  fprintf(stderr, " -D\tdisable DMA engine\n");
  fprintf(stderr, " -F <n>\tadd ioctl flags\n");
}

static unsigned long long
next_length(unsigned long long length, unsigned long long multiplier, unsigned long long increment)
{
  if (length)
    return length*multiplier+increment;
  else if (increment)
    return increment;
  else
    return 1;
}

static int one_length(int fd, volatile knem_status_t *status_array, 
		      unsigned long long length, unsigned long iter, unsigned long warmup, 
		      unsigned long flags)
{
  char *send_buffer, *recv_buffer;
  struct knem_cmd_param_iovec send_iovec, recv_iovec;
  struct knem_cmd_init_send_param sendcmd;
  struct knem_cmd_init_async_recv_param recvcmd;
  struct timeval tv1,tv2;
  unsigned long long us;
  unsigned long i;
  int err;

  send_buffer = malloc(length);
  if (!send_buffer)
    goto out;
  recv_buffer = malloc(length);
  if (!recv_buffer)
    goto out_with_send_buffer;

  send_iovec.base = (uintptr_t) send_buffer;
  send_iovec.len = length;

  recv_iovec.base = (uintptr_t) recv_buffer;
  recv_iovec.len = length;

  for(i=0; i<iter+warmup; i++) {
    if (i == warmup)
      gettimeofday(&tv1, NULL);

    sendcmd.send_iovec_array = (uintptr_t) &send_iovec;
    sendcmd.send_iovec_nr = 1;
    sendcmd.flags = flags;
    err = ioctl(fd, KNEM_CMD_INIT_SEND, &sendcmd);
    if (err < 0) {
      perror("ioctl (init send)");
      goto out_with_buffers;
    }

    recvcmd.recv_iovec_array = (uintptr_t) &recv_iovec;
    recvcmd.recv_iovec_nr = 1;
    recvcmd.status_index = 0;
    recvcmd.send_cookie = sendcmd.send_cookie;
    recvcmd.flags = flags;
    err = ioctl(fd, KNEM_CMD_INIT_ASYNC_RECV, &recvcmd);
    if (err < 0) {
      perror("ioctl (init async recv)");
      goto out_with_buffers;
    }

    while (status_array[recvcmd.status_index] == KNEM_STATUS_PENDING)
      sched_yield();
    if (status_array[recvcmd.status_index] != KNEM_STATUS_SUCCESS) {
      fprintf(stderr, "got status %d instead of %d\n", status_array[recvcmd.status_index], KNEM_STATUS_SUCCESS);
      goto out_with_buffers;
    }
  }

  gettimeofday(&tv2, NULL);

  us = (tv2.tv_sec-tv1.tv_sec)*1000000 + (tv2.tv_usec-tv1.tv_usec);
  printf("% 9lld:\t%.3f us\t%.2f MB/s\t %.2f MiB/s\n",
	 length, ((float) us)/iter,
	 ((float) length)*iter/us,
	 ((float) length)*iter/us/1.048576);

  return 0;

 out_with_buffers:
  free(recv_buffer);
 out_with_send_buffer:
  free(send_buffer);
 out:
  return err;
}

int main(int argc, char * argv[])
{
  unsigned long long length;
  unsigned long long min = MIN;
  unsigned long long max = MAX;
  unsigned long multiplier = MULTIPLIER;
  unsigned long increment = INCREMENT;
  unsigned long pause_ms = PAUSE_MS;
  unsigned long iter = ITER;
  unsigned long warmup = WARMUP;
  unsigned long flags = KNEM_FLAG_DMA;
  volatile knem_status_t *status_array;
  struct knem_cmd_info info;
  int fd, err;
  int c;

  fd = open("/dev/knem", O_RDWR);
  if (fd < 0) {
    perror("open");
    goto out;
  }

  err = ioctl(fd, KNEM_CMD_GET_INFO, &info);
  if (err < 0) {
    perror("ioctl (get info)");
    goto out_with_fd;
  }

  if (info.abi != KNEM_ABI_VERSION) {
    printf("got driver ABI %lx instead of %lx\n",
	   (unsigned long) info.abi, (unsigned long) KNEM_ABI_VERSION);
    goto out_with_fd;
  }

  status_array = mmap(NULL, 1, PROT_READ|PROT_WRITE, MAP_SHARED, fd, KNEM_STATUS_ARRAY_FILE_OFFSET);
  if (status_array == MAP_FAILED) {
    perror("mmap status_array");
    goto out_with_fd;
  }

  while ((c = getopt(argc, argv, "S:E:M:I:N:W:P:F:Dh")) != -1)
    switch (c) {
    case 'S':
      min = atoll(optarg);
      break;
    case 'E':
      max = atoll(optarg);
      break;
    case 'M':
      multiplier = atoll(optarg);
      break;
    case 'I':
      increment = atoll(optarg);
      break;
    case 'N':
      iter = atoi(optarg);
      break;
    case 'W':
      warmup = atoi(optarg);
      break;
    case 'P':
      pause_ms = atoi(optarg);
      break;
    case 'F':
      flags |= atoi(optarg);
      break;
    case 'D':
      flags &= ~KNEM_FLAG_DMA;
      break;
    default:
      fprintf(stderr, "Unknown option -%c\n", c);
    case 'h':
      usage(argc, argv);
      exit(-1);
      break;
    }

  for(length = min;
      length < max;
      length = next_length(length, multiplier, increment)) {
    err = one_length(fd, status_array, length, iter, warmup, flags);
    if (err < 0)
      goto out_with_fd;
    usleep(pause_ms * 1000);
  }

 out_with_fd:
  close(fd);
 out:
  return 0;
}
