/***************************** LICENSE START ***********************************

 Copyright 2013 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "inc_iostream.h"
#include "Metview.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <unistd.h>

#include <sstream>

#include "MvScm.h"

using namespace std;


class Base : public MvService { 
protected:
	Base(const char* a) : MvService(a) {};
};

class ScmRun: public Base {	
public:
	ScmRun() : Base("SCM_RUN") {ftmp=marstmp();};
	void serve(MvRequest&,MvRequest&);

protected:	
	void shellCommand(string &, stringstream&,stringstream&);
	
	string ftmp;
};

void ScmRun::shellCommand(string &command, stringstream& out,stringstream& err)
{
	FILE *in;
	char cbuf[512];

	string cmd = command + " 2>" + ftmp;

	if (!(in = popen(cmd.c_str() ,"r")) )
	{
		return;
	}

	while(fgets(cbuf, sizeof(cbuf), in) != NULL)
	{
		out << cbuf;		
	}	
	
	pclose(in);


	if (!(in = fopen(ftmp.c_str() ,"r")) )
	{
		return;
	}

	while(fgets(cbuf, sizeof(cbuf), in) != NULL)
	{
		err << cbuf;		
	}	
	
	fclose(in);

}

void ScmRun::serve( MvRequest& in, MvRequest& out)
{
  	cout << "--------------ScmRun::serve()--------------" << endl;
  	in.print();
	
	string errTxt;
	
	//Find out scm script path
	string scmScript;
	char *mvbin=getenv("METVIEW_BIN");
	if (mvbin == 0)  
	{	
		marslog(LOG_EROR,"No METVIEW_BIN env variable is defined. Cannot locate script mv_scm_run!");
		setError(13);
		return;
	}
	else
	{
		scmScript=string(mvbin) +"/mv_scm_run";
	}
	

	//Create  tmp dir for the flextra run
	string tmpPath;
	if(!metview::createWorkDir("scm",tmpPath,errTxt))
	{
		marslog(LOG_EROR,"%s",errTxt.c_str());
		setError(13);
		return;
	}
	
	//Scm exe
	string exe;
	if(!in.getPath("SCM_EXE_PATH",exe,true,errTxt))
	{
		marslog(LOG_EROR,"%s",errTxt.c_str());
		setError(13);
		return;
	}	
	if(exe.empty())
	{
	  	exe="_UNDEF_";
	}	
	
        //---------------------------------------
	// Input data
	//---------------------------------------
	
	string data;	
	if(!in.getPath("SCM_INPUT_DATA","SCM_INPUT_DATA_PATH",data,true,errTxt))
	{
		marslog(LOG_EROR,"%s",errTxt.c_str());
		setError(13);
		return;
	}	
	
	//-------------------------
	// Level num
	//-------------------------
	
	int levelNum=MvScm::modelLevelNum(data);
	if(levelNum == -1)
	{  
		marslog(LOG_EROR,"Could not read number of levels from input file!");
		setError(13);
		return;	
	}
	
	stringstream sst;
  	sst << levelNum;
   	string nlev=sst.str();   
   
	//-------------------------
	//Namelist
	//-------------------------
	
	string namelist;	
	if(!in.getPath("SCM_NAMELIST","SCM_NAMELIST_PATH",namelist,false,errTxt))
	{
		marslog(LOG_EROR,"%s",errTxt.c_str());
		setError(13);
		return;
	}	
	
	//-------------------------
	//Vtable file
	//-------------------------
	
	string vtable;
	if(!in.getPath("SCM_VTABLE","SCM_VTABLE_PATH",vtable,true,errTxt))
	{
		marslog(LOG_EROR,"%s",errTxt.c_str());
		setError(13);
		return;
	}	
	if(vtable.empty())
	{
	  	vtable=string("_UNDEF_");
	}
	
	//-------------------------
	// Rclim file
	//-------------------------
	
	string rclim;
	if(const char *cval=in("SCM_RCLIM_PATH"))
	{
		rclim=string(cval);
	}
	
	if(rclim.empty())
	{
	  	rclim=string("_UNDEF_");
	}
	
	//-------------------------
	// COPY OUTPUT DATA
	//-------------------------
	
	string userResFile;
	
	const char *copyRes=in("SCM_COPY_OUTPUT_DATA"); 
	if(copyRes && strcmp(copyRes,"ON") == 0)
	{ 			
		if(!in.getPath("SCM_OUTPUT_DATA_PATH",userResFile,false,errTxt))
		{
			marslog(LOG_EROR,"%s",errTxt.c_str());
			setError(13);
			return;
		}					
	}
	
	//-------------------------
	// Run the model
	//-------------------------
		
	string resFileName="scm_out.nc";
	string logFileName="log.txt";
	string logFile=tmpPath + "/" + logFileName;
	
		//Run the scm script
	string cmd = scmScript + " \"" + tmpPath + "\" \"" + exe  + "\" " + nlev + " \"" + data + "\" \"" + namelist + "\" \"" + vtable + "\" \"" + rclim +  "\" \"" + logFileName + "\" \"" + resFileName + "\""; 

	//marslog(LOG_INFO,"Execute command: %s",cmd.c_str());
	
	int ret=system(cmd.c_str());	
	
	bool runFailed=false;
	
	//If the script failed read log file and
	//write it into LOG_EROR
	if(ret == -1 || WEXITSTATUS(ret) != 0)
	{		
		ifstream in(logFile.c_str());
		string line;

		if(WEXITSTATUS(ret) == 255)
		{
		  	marslog(LOG_WARN,"Warnings were generated during SCM run!");
		}  		
		else if(WEXITSTATUS(ret) == 1)
		{		
			marslog(LOG_EROR,"Failed to perform SCM run!");
			runFailed=true;
		}
		else if(WEXITSTATUS(ret) > 1)
		{		
			marslog(LOG_EROR,"SCM run failed with exit code: %d !",WEXITSTATUS(ret));
			runFailed=true;
		}		
	}
	
	//Log locations are always printed!!
	
	if(runFailed)
	{
	  	marslog(LOG_EROR,"Log files are available at:");
		//marslog(LOG_EROR,"---------------------------");
		marslog(LOG_EROR," STDOUT  %s",logFile.c_str());	
		string msg=tmpPath + "/fort.20";
		marslog(LOG_EROR," fort.20 %s",msg.c_str());
	}  
	else
	{  
		marslog(LOG_INFO,"Log files are available at:");
		//marslog(LOG_INFO,"---------------------------");
		marslog(LOG_INFO," STDOUT  %s",logFile.c_str());	
		string msg=tmpPath + "/fort.20";
		marslog(LOG_INFO," fort.20 %s",msg.c_str());
	}
	
	//Retrun if run failed!!
	if(runFailed)
	{
		setError(13);  	
		return;
	}
	
	//---------------------------
	// Merge the output files
	//---------------------------
	
	//At this point we have already copied progvar.nc into resFile in the script
	
	string resFile=tmpPath + "/" + resFileName;
	string diagFile=tmpPath + "/diagvar.nc";
	string diag2File=tmpPath + "/diagvar2.nc";
	
	if(MvScm::mergeOutFiles(resFile,diagFile,diag2File) == false)
	{
	  	marslog(LOG_EROR,"Could not merge SCM output files!");
	  	setError(13);
		return;
	}  
	
	//---------------------------------------------------
	// Copy the output into the user-defined location
	//---------------------------------------------------
	
	if(!userResFile.empty())
	{
	  	const char *dn=dirname(userResFile.c_str());
	  	cmd="mkdir -p " + string(dn) + "; cp -f " + resFile + " " +  userResFile;
			
	  	stringstream outStream;
		stringstream errStream;
		shellCommand(cmd,outStream,errStream);
		
		if(!errStream.str().empty())
		{
			marslog(LOG_EROR,"");
			marslog(LOG_EROR,"Could not copy ouput file to target location!");  
		        marslog(LOG_EROR,"This command:");
			marslog(LOG_EROR,"   %s",cmd.c_str());
			marslog(LOG_EROR,"failed with this error:");
			marslog(LOG_EROR,"   %s",errStream.str().c_str());
			setError(13);
		}
	}
		
	//------------------------------------------
	//Set out request
	//------------------------------------------
	
	out=MvRequest("SCM_OUTPUT_DATA");
	out("PATH")=resFile.c_str();
	
	cout << tmpPath << endl;
	out.print();
}

  	
int main( int argc, char** argv )
{
	MvApplication theApp( argc, argv,"ScmRun");

    	ScmRun scmRun;	
	
    	theApp.run();
}


