/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

// ExamineManager.cc                                        May2011/vk
//
//  Metview runs this MvExamineManager when Metview session is started
//  with flag '-e' plus two parameters (first is the data type, the
//  second is the path to the input file).
//
//  Input parameters are used for creating a suitable Metview request,
//  which is then sent to the right examiner using a MvServiceTask
//  object.
//
//  Data type is converted to capital letters, thus e.g. for type GRIB
//  values 'GRIB', 'grib' and 'Grib' are acceptable.
//
//  This code is heavily based on example file src/libMetview/Demo.cc.
//

#include <Metview.h>
#include <MvServiceTask.h>
#include <locale>


class MvExamineManager : public MvApplication, public MvClient
{
	virtual void endOfTask(MvTask *from);
	virtual void progress(char*);
	bool examGrib( const string& fileName );
	bool examBufr( const string& fileName );
	bool examNetCDF( const string& fileName );
	bool examOdb( const string& fileName );
//	bool examXXX( const string& fileName );
public:
	MvExamineManager(int argc, char **argv);
};


void MvExamineManager::endOfTask(MvTask *from)
{
// WARNING: The examiner module should not send REPLY after starting up!
//          See for instance GribExaminer MvMain.cc (METVIEW_MODE examine)
//
	printf("Got reply from %s: err=%d\n", from->taskName(), from->getError());
	exit(0);
}

void MvExamineManager::progress(char *p)
{
	printf("Got progress %s\n",p);
}

bool
MvExamineManager::examGrib( const string& fileName )
{
	MvRequest mvReq( "GRIB" );           //-- build a GRIB request
	mvReq.setValue( "PATH", fileName.c_str() );
	                                     //-- send request to GribExaminer
	MvTask* task = new MvServiceTask( this, "GribExaminer", mvReq );
	task->run();

	return true;
}

bool
MvExamineManager::examBufr( const string& fileName )
{
	MvRequest mvReq( "BUFR" );           //-- build a BUFR request
	mvReq.setValue( "PATH", fileName.c_str() );
	mvReq.print();
	                                     //-- send request to BufrExaminer
	MvTask* task = new MvServiceTask( this, "BufrExaminer", mvReq );
	task->run();

	return true;
}

bool
MvExamineManager::examNetCDF( const string& fileName) // fileName )
{
	MvRequest mvReq( "NETCDF" );           //-- build a NETCDF request
	mvReq.setValue( "PATH", fileName.c_str() );
	mvReq.print();
	                                     //-- send request to NcExaminer
	MvTask* task = new MvServiceTask( this, "NcExaminer", mvReq );
	task->run();

	return true;
}

bool
MvExamineManager::examOdb( const string& fileName )
{
	MvRequest mvReq( "ODB_DB" );           //-- build n ODB_DB request
	mvReq.setValue( "PATH", fileName.c_str() );
	                                     //-- send request toOdbExaminer
	MvTask* task = new MvServiceTask( this, "OdbExaminer", mvReq );
	task->run();

	return true;
}

MvExamineManager::MvExamineManager(int argc,char **argv)
	:MvApplication(argc,argv)
{
	string dataType( argv[1] );          //-- convert type to upper case
	for( size_t i=0; i<dataType.length(); ++i )
	   dataType[i] = std::toupper(dataType[i], std::locale());

	string fileName( argv[2] );          //-- 2. arg is data file name

	bool status = false;
	if( dataType == "GRIB" )
	{
	   status = examGrib( fileName );    //-- GRIB examiner
	}
	else if( dataType == "BUFR" )
	{
	   status = examBufr( fileName );    //-- BUFR examiner
	}
	else if( dataType == "NETCDF" )
	{
	   status = examNetCDF( fileName );  //-- NetCDF examiner
	}	
	else if( dataType == "ODB" )
	{
	   status = examOdb( fileName );  //-- ODB examiner
	}
	else
	{
	   //-- ERROR MSGS SHOULD BE GIVEN IN POP-UP WINDOWS !!!!!
	   cerr << "Unknown data type: " << dataType << endl;
	   exit(13);
	}

	if( ! status )
	{
	   //-- ERROR MSGS SHOULD BE GIVEN IN POP-UP WINDOWS !!!!!
	   cerr << "Could not examine data. " << endl;
	   exit(13);
	}
}

int main(int argc, char **argv)
{
	MvExamineManager app(argc,argv);
	app.run();
}
