#ifndef OSMIUM_INDEX_BOOL_VECTOR_HPP
#define OSMIUM_INDEX_BOOL_VECTOR_HPP

/*

This file is part of Osmium (http://osmcode.org/libosmium).

Copyright 2013-2016 Jochen Topf <jochen@topf.org> and others (see README).

Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.

*/

#include <type_traits>
#include <vector>

namespace osmium {

    namespace index {

        /**
         * Index storing one bit for each Id. The index automatically scales
         * with the Ids stored. Default value is 'false'. Storage uses
         * std::vector<bool> and needs a minimum of memory if the Ids are
         * dense.
         */
        template <typename T>
        class BoolVector {

            static_assert(std::is_unsigned<T>::value, "Needs unsigned type");

            std::vector<bool> m_bits;

        public:

            BoolVector() = default;

            BoolVector(const BoolVector&) = default;
            BoolVector(BoolVector&&) = default;
            BoolVector& operator=(const BoolVector&) = default;
            BoolVector& operator=(BoolVector&&) = default;

            ~BoolVector() noexcept = default;

            void set(T id, bool value = true) {
                if (m_bits.size() <= id) {
                    m_bits.resize(id + 1024 * 1024);
                }

                m_bits[id] = value;
            }

            bool get(T id) const {
                return id < m_bits.size() && m_bits[id];
            }

        }; // class BoolVector

    } // namespace index

} // namespace osmium

#endif // OSMIUM_INDEX_BOOL_VECTOR_HPP
