#include "osl/record/opening/openingBook.h"
#include "osl/record/record.h"
#include "osl/record/compactBoard.h"
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int.hpp>
#include <boost/foreach.hpp>
#include <algorithm>
#include <iostream>
#include <stdexcept>

#ifndef MINIMAL
osl::record::opening::
WinCountBook::WinCountBook(const char *filename)
  : ifs(filename, std::ios_base::binary)
{
  if (! ifs)
  {
    const char *message = "WinCountBook: open failed ";
    std::cerr << message << filename << std::endl;
    throw std::runtime_error(std::string(message) + filename);
  }
  nStates=readInt();
}

osl::record::opening::
WinCountBook::~WinCountBook()
{
}

int osl::record::opening::
WinCountBook::readInt()
{
  int ret=0;
  CArray<char,4> cs;
  ifs.read(&cs[0],4);
  for (int i=0;i<4;i++) {
    ret = (ret<<8)|(cs[i]&255);
  }
  return ret;
}

void osl::record::opening::
WinCountBook::seek(int offset)
{
  ifs.seekg(offset,std::ios::beg);
}

osl::vector<osl::record::opening::OBMove> osl::record::opening::
WinCountBook::getMoves(int stateIndex)
{
  assert(stateIndex >= 0);
  seek(4+16*stateIndex+8);
  int nMoves=readInt();
  int moveIndex=readInt();
  seek(4+16*nStates+8*moveIndex);
  vector<OBMove> moves;
  moves.reserve(nMoves);
  for(int i=0;i<nMoves;i++)
  {
    Move move=Move::makeDirect(readInt());
    int stateIndex=readInt();
    moves.push_back(OBMove(move,stateIndex));
  }
  return moves;
}

int osl::record::opening::
WinCountBook::getWinCount(int stateIndex)
{
  seek(4+16*stateIndex);
  return readInt();
}

int osl::record::opening::
WinCountBook::getLoseCount(int stateIndex)
{
  seek(4+16*stateIndex+4);
  return readInt();
}

std::ostream& osl::record::opening::operator<<(std::ostream& os, const WMove& w)
{
  osl::record::writeInt(os, osl::record::opening::OMove(w.getMove()));
  osl::record::writeInt(os, w.getStateIndex());
  osl::record::writeInt(os, w.getWeight());
  return os;
}
#endif

std::istream& osl::record::opening::operator>>(std::istream& is, WMove& w)
{
  w.move = OMove(osl::record::readInt(is)).operator Move();
  w.stateIndex = osl::record::readInt(is);
  w.weight = osl::record::readInt(is);
  return is;
}

osl::record::opening::
WeightedBook::WeightedBook(const char *filename)
  : ifs(filename, std::ios_base::binary)
{
  if (! ifs)
  {
    const char *message = "WeightedBook: open failed ";
    std::cerr << message << filename << std::endl;
    throw std::runtime_error(std::string(message) + filename);
  }
#ifndef NDEBUG
  int version = 
#endif
    readInt(ifs);
  assert(version == 1);
  nStates = osl::record::readInt(ifs);
  nMoves = osl::record::readInt(ifs);
  startState = osl::record::readInt(ifs);
}

osl::record::opening::
WeightedBook::~WeightedBook()
{
}

void osl::record::opening::
WeightedBook::seek(int offset)
{
  ifs.seekg(offset,std::ios::beg);
}

osl::record::opening::WeightedBook::WMoveContainer osl::record::opening::
WeightedBook::getMoves(int stateIndex, const bool visit_zero)
{
  assert(stateIndex >= 0);
  seek(HEADER_SIZE + STATE_SIZE * stateIndex);
  int moveIndex=readInt(ifs);
  int nWMoves=readInt(ifs);
  seek(HEADER_SIZE + STATE_SIZE * nStates + MOVE_SIZE * moveIndex);
  vector<WMove> moves;
  moves.reserve(nWMoves);
  for(int i=0;i<nWMoves;i++)
  {
    WMove wm;
    ifs >> wm;
    if (!visit_zero && wm.getWeight() == 0) continue;
    moves.push_back(wm);
  }
  return moves;
}

osl::record::CompactBoard osl::record::opening::
WeightedBook::getCompactBoard(int stateIndex)
{
  seek(HEADER_SIZE + STATE_SIZE * nStates + MOVE_SIZE * nMoves
       + BOARD_SIZE * stateIndex);
  CompactBoard board;
  ifs >> board;
  return board;
}

osl::SimpleState osl::record::opening::
WeightedBook::getBoard(int stateIndex)
{
  const CompactBoard board = getCompactBoard(stateIndex);
  return board.getState();
}

int osl::record::opening::
WeightedBook::getWhiteWinCount(int stateIndex)
{
  seek(HEADER_SIZE + STATE_SIZE * stateIndex);
  readInt(ifs);
  readInt(ifs);
  readInt(ifs);
  return readInt(ifs);
}

int osl::record::opening::
WeightedBook::getBlackWinCount(int stateIndex)
{
  seek(HEADER_SIZE + STATE_SIZE * stateIndex);
  readInt(ifs);
  readInt(ifs);
  return readInt(ifs);
}

void osl::record::opening::
WeightedBook::validate()
{
#ifndef NDEBUG
  {
    SimpleState state(HIRATE);
    SimpleState start = getBoard(startState);
    assert(state == start);
  }
#endif
  vector<char> visited(nStates);
  std::fill(visited.begin(), visited.end(), false);

  vector<int> stateToCheck;
  stateToCheck.push_back(startState);
  visited[startState] = true;

  while (!stateToCheck.empty())
  {
    const int index = stateToCheck.back();
    stateToCheck.pop_back();
    SimpleState state = getBoard(index);
    vector<record::opening::WMove> moves = getMoves(index);
    BOOST_FOREACH(WMove move, moves)
    {
      NumEffectState newState(state);
      newState.makeMove(move.getMove());
      const int nextIndex = move.getStateIndex();

      SimpleState stateInFile = getBoard(nextIndex);
      assert(newState == stateInFile);
      if (!visited[nextIndex])
      {
	stateToCheck.push_back(nextIndex);
	visited[nextIndex] = true;
      }
    }
  }
}

int osl::record::opening::
WeightedBook::getStateIndex(const SimpleState& state_to_look_for,
                            const bool visit_zero, 
                            const Player player)
{
  int ret = -1;
  const CompactBoard board_to_look_for(state_to_look_for);
  
  const CompactBoard start_state = getCompactBoard(getStartState());
  if (start_state == board_to_look_for)
  {
    ret = getStartState();
    return ret;
  }

  vector<char> states(getTotalState(), false); // mark states that have been visited.
  vector<int> stateToVisit;
  stateToVisit.push_back(getStartState());

  while (!stateToVisit.empty())
  {
    const int stateIndex = stateToVisit.back();
    stateToVisit.pop_back();
    states[stateIndex] = true;

    WMoveContainer moves;
    if (visit_zero)
      moves = getMoves(stateIndex);
    else
    {
      const CompactBoard stateIndexCB = getCompactBoard(stateIndex);
      const Player turn = stateIndexCB.turn();
      const bool zero_include = turn == player ? false : true;
      moves = getMoves(stateIndex, zero_include);
    }
    BOOST_FOREACH(WMove move, moves)
    {
      const int nextIndex = move.getStateIndex();
      if (! states[nextIndex])
      {
        const CompactBoard state = getCompactBoard(nextIndex);
        if (state == board_to_look_for)
        {
          ret = nextIndex;
          return ret;
        }

	stateToVisit.push_back(nextIndex);
      }
    } // each wmove
  } // while loop

  return ret;
}

int osl::record::opening::
WeightedBook::getStateIndex(const osl::vector<osl::Move>& moves)
{
  int state_index = getStartState();
  BOOST_FOREACH(Move move, moves)
  {
    const WMoveContainer wmoves = getMoves(state_index);
    WMoveContainer::const_iterator it = wmoves.begin();
    for (; it != wmoves.end(); ++it)
      if (it->getMove() == move) break;
    if (it != wmoves.end())
    {
      state_index = it->getStateIndex(); // next state to visit
      continue;
    }
    return -1; // not found
  }
  return state_index;
}


std::vector<int> osl::record::opening::
WeightedBook::getParents(const int target_state_index)
{
  std::vector<int> ret;

  if (getStartState() == target_state_index)
    return ret;
  
  vector<char> states(getTotalState(), false); // mark states that have been visited.
  vector<int> stateToVisit;
  stateToVisit.push_back(getStartState());

  while (!stateToVisit.empty())
  {
    const int stateIndex = stateToVisit.back();
    stateToVisit.pop_back();
    states[stateIndex] = true;

    const WMoveContainer moves = getMoves(stateIndex);
    BOOST_FOREACH(WMove move, moves)
    {
      const int nextIndex = move.getStateIndex();

      if (nextIndex == target_state_index)
        ret.push_back(stateIndex);

      if (! states[nextIndex])
	stateToVisit.push_back(nextIndex);
    } // each wmove
  } // while loop

  return ret;
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
