/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by <Author(s)>, <Affiliation>, <Year>
################################################################################*/

#include "Objectpath.h"

Objectpath::Objectpath(){}

Objectpath::~Objectpath(){}

Objectpath::Objectpath(const CMPIBroker* _broker, const string& classname, const string& nameSpace) {
	broker = (CMPIBroker*) _broker;
	hdl = CMNewObjectPath((CMPIBroker*) broker, nameSpace.c_str(), classname.c_str(), NULL);
}

Objectpath::Objectpath(const CMPIBroker* _broker, CMPIObjectPath* ob) {
	hdl = ob;
	broker = (CMPIBroker*) _broker;
}

CMPIObjectPath* Objectpath::getHdl() const {
	return hdl;
}

Objectpath Objectpath::getObjectpath() const {
	return *this;
}

string Objectpath::getNamespace() const {
	CMPIStatus status;
	string nameSpace = CMGetCharPtr(hdl->ft->getNameSpace(hdl, &status));
	if (status.rc != CMPI_RC_OK)
		return "";
	return nameSpace;
}

void Objectpath::setNamespace(const string& value) {
	hdl->ft->setNameSpace(hdl, value.c_str());
}

string Objectpath::getClassname() const {
	CMPIStatus status;
	string classname = CMGetCharPtr(hdl->ft->getClassName(hdl, &status));
	if (status.rc != CMPI_RC_OK)
		return "";
	return classname;
}

string Objectpath::getHostname() const {
	CMPIStatus status;
	string hostname = CMGetCharPtr(hdl->ft->getHostname(hdl, &status));
	if (status.rc != CMPI_RC_OK)
		return "";
	return hostname;
}

bool Objectpath::equals(const Objectpath& anotherObjectpath) const {
	if (anotherObjectpath.getKeyCount() != getKeyCount() || anotherObjectpath.getNamespace() != getNamespace() || anotherObjectpath.getClassname() != getClassname() || anotherObjectpath.getHostname() != getHostname())
		return false;
	vector<string> key_names, another_key_names;
	for (unsigned int i = 0; i < getKeyCount(); i++) {
		CMPIData data, anotherData;
		string name;
		// Get fist instance key name and value
		if (getKeyAt(i, data, name) != OK)
			return false;
		// Get second instance key value (same name)
		if (anotherObjectpath.getKey(name, anotherData) != OK)
			return false;
		if (!CF_CMPIDataCmp(broker, data, anotherData))
			return false;
	}
	return true;
}

string Objectpath::toString() const {
	return CT_CMPIObjectPathToString(hdl);
}

unsigned int Objectpath::getKeyCount() const {
	CMPIStatus status;
	unsigned int keyCount = hdl->ft->getKeyCount(hdl, &status);
	if (status.rc != CMPI_RC_OK)
		return 0;
	return keyCount;
}

int Objectpath::getKeyAt(unsigned int index, CMPIData& data, string& name) const {
	CMPIStatus status;
	CMPIString* _name;
	data = hdl->ft->getKeyAt(hdl, index, &_name, &status);
	if (status.rc != CMPI_RC_OK)
		return FAILED;
	if ((data.state & CMPI_nullValue) == CMPI_nullValue || (data.state & CMPI_badValue) == CMPI_badValue || (data.state & CMPI_notFound) == CMPI_notFound)
		return FAILED;
	name = CMGetCharPtr(_name);
	return OK;
}

int Objectpath::getKey(const string& name, CMPIData& data) const {
	CMPIStatus status;
	data = hdl->ft->getKey(hdl, name.c_str(), &status);
	if (status.rc != CMPI_RC_OK)
		return FAILED;
	if ((data.state & CMPI_nullValue) == CMPI_nullValue || (data.state & CMPI_badValue) == CMPI_badValue || (data.state & CMPI_notFound) == CMPI_notFound)
		return FAILED;
	return OK;
}

//uint8
void Objectpath::addKey(const string& name, const unsigned char& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_uint8);
}
int Objectpath::getKey(const string& name, unsigned char& value) const {
	_GET_KEY
	
}
	
//uint16
void Objectpath::addKey(const string& name, const unsigned short int& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_uint16);
}
int Objectpath::getKey(const string& name, unsigned short int& value) const {
	_GET_KEY
}

//uint32
void Objectpath::addKey(const string& name, const unsigned int& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_uint32);
}
int Objectpath::getKey(const string& name, unsigned int& value) const {
	_GET_KEY
}

//uint32 deprecated
void Objectpath::addKey(const string& name, const unsigned long& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_uint32);
}
int Objectpath::getKey(const string& name, unsigned long& value) const {
	_GET_KEY
}

//uint64
void Objectpath::addKey(const string& name, const unsigned long long& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_uint64);
}
int Objectpath::getKey(const string& name, unsigned long long& value) const {
	_GET_KEY
}

//sint8
void Objectpath::addKey(const string& name, const signed char& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_sint8);
}
int Objectpath::getKey(const string& name, signed char& value) const {
	_GET_KEY
}
	
//sint16
void Objectpath::addKey(const string& name, const short int& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_sint16);
}
int Objectpath::getKey(const string& name, short int& value) const {
	_GET_KEY
}

//sint32
void Objectpath::addKey(const string& name, const int& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_sint32);
}
int Objectpath::getKey(const string& name, int& value) const {
	_GET_KEY
}

//sint32 deprecated
void Objectpath::addKey(const string& name, const long& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_sint32);
}
int Objectpath::getKey(const string& name, long& value) const {
	_GET_KEY
}

//sint64
void Objectpath::addKey(const string& name, const long long& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_sint64);
}
int Objectpath::getKey(const string& name, long long& value) const {
	_GET_KEY
}

//real32
void Objectpath::addKey(const string& name, const float& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_real32);
}
int Objectpath::getKey(const string& name, float& value) const {
	_GET_KEY
}

//real64
void Objectpath::addKey(const string& name, const double& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_real64);
}
int Objectpath::getKey(const string& name, double& value) const {
	_GET_KEY
}

//char16
void Objectpath::addKeyChar16(const string& name, const unsigned short int& value) {
	CMPIValue myValue = CT_toCMPIChar16(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_char16);
}
int Objectpath::getKeyChar16(const string& name, unsigned short int& value) const {
	CMPIStatus rc;
	CMPIData data = CMGetKey(hdl, name.c_str(), &rc);
	if (rc.rc != CMPI_RC_OK) return FAILED;
	if ((data.state & CMPI_nullValue) == CMPI_nullValue || (data.state & CMPI_badValue) == CMPI_badValue || (data.state & CMPI_notFound) == CMPI_notFound)
		return FAILED;
	return CT_ToCChar16(data, value);
}

//boolean
void Objectpath::addKey(const string& name, const bool& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_boolean);
}
int Objectpath::getKey(const string& name, bool& value) const {
	_GET_KEY
}

//string
void Objectpath::addKey(const string& name, const string& value) {
	CMPIValue myValue = CT_toCMPI(broker, value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_string);
}
int Objectpath::getKey(const string& name, string& value) const {
	_GET_KEY
}

//datetime
void Objectpath::addKeyDatetime(const string& name, const string& value) {
	CMPIValue myValue = CT_toCMPIDatetime(broker, value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_dateTime);
}
int Objectpath::getKeyDatetime(const string& name, string& value) const {
	CMPIStatus rc;
	CMPIData data = CMGetKey(hdl, name.c_str(), &rc);
	if (rc.rc != CMPI_RC_OK) return FAILED;
	if ((data.state & CMPI_nullValue) == CMPI_nullValue || (data.state & CMPI_badValue) == CMPI_badValue || (data.state & CMPI_notFound) == CMPI_notFound)
		return FAILED;
	return CT_ToCDatetime(data, value);
}
	
//REF
void Objectpath::addKey(const string& name, const Objectpath& value) {
	CMPIValue myValue = CT_toCMPI(value);
	hdl->ft->addKey(hdl, name.c_str(), &myValue, CMPI_ref);
}
int Objectpath::getKey(const string& name, Objectpath& value) const {
	CMPIStatus rc;
	CMPIData data = CMGetKey(hdl, name.c_str(), &rc);
	if (rc.rc != CMPI_RC_OK) return FAILED;
	if ((data.state & CMPI_nullValue) == CMPI_nullValue || (data.state & CMPI_badValue) == CMPI_badValue || (data.state & CMPI_notFound) == CMPI_notFound)
		return FAILED;
	return CT_ToC(broker, data, value);
}
