/*! \mainpage IRCClient - a Cocoa IRC Framework to create IRC clients
 *
 * \section intro_sec Introduction
 *
 * IRCClient is a Cocoa Framework that uses the excellent libircclient library
 * written by Georgy Yunaev.
 *
 * \section usage Basic Usage
 *
 * To use this framework, you will need to write an IRCClientSessionDelegate to
 * handle all of the events generated by the server, and an IRCClientChannelDelegate
 * to handle all of the events generated by channels on that server.
 *
 * You then create an IRCClientSession object in your code, assign the required
 * properties, and call connect: to connect to the server and run: to create
 * the new thread and start receiving events. For example:
 *
 * \code
 * IRCClientSession *session = [[IRCClientSession alloc] init];
 * MyIRCClientSessionDelegate *controller = [[MyIRCClientSessionDelegate alloc] init];
 * 
 * [session setDelegate:controller];
 * [controller setSession:session];
 * 
 * [session setServer:@"irc.dal.net"];
 * [session setPort:@"6667"];
 * [session setNickname:@"test"];
 * [session setUsername:@"test"];
 * [session setRealname:@"test"];
 * [session connect];
 * 
 * [session run]; //starts the thread
 * \endcode
 * 
 * \section author Author, copyright, support.
 *
 * If you have any questions, bug reports, suggestions regarding libircclient 
 * or the IRCClient framework, please visit http://libircclient.sourceforge.net
 *
 * <PRE>
 * libircclient Copyright (C) 2004-2009 Georgy Yunaev gyunaev@ulduzsoft.com
 * IRCClient Copyright (C) 2009 Nathan Ollerenshaw chrome@stupendous.net
 *
 * This library is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU Lesser General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or (at your 
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public 
 * License for more details.
 * </PRE>
 */

/** 
 * @file IRCClientSession.h
 * @author Nathan Ollerenshaw
 * @version 1.0
 * @date 01.2009
 * @brief Represents a connected IRC Session.
 */

#import <Cocoa/Cocoa.h>
#import <IRCClient/IRCClientSessionDelegate.h>
#include <IRCClient/libircclient.h>

/** @class IRCClientSession
 *	@brief Represents a connected IRC Session.
 *
 *	IRCClientSession represents a single connection to an IRC server. On initialising
 *  the object, and setting the delegate, server, port, password, nickname, username and realname
 *	properties, you call the connect: and run: methods to connect to the IRC server
 *	and start a new thread.
 *
 *	This thread then sends messages back to the main runloop to the IRC server delegate,
 *	or to the IRCClientChannel delegate as required.
 */

@class IRCClientChannel;
@interface IRCClientSession : NSObject {
	id					delegate;
	irc_callbacks_t		callbacks;
	irc_session_t		*session;
	NSThread			*thread;
	
	NSString			*version;
	NSString			*server;
	NSString			*port;
	NSString			*password;

	NSString			*nickname;
	NSString			*username;
	NSString			*realname;
	
	NSMutableDictionary	*channels;
	NSMutableDictionary *nicks;
	NSStringEncoding	encoding;
}

/** delegate to send events to. */
@property (assign) id delegate;

/** The underlying libircclient handle */
@property (assign) irc_session_t *session;

/** The version string for the client to send back on CTCP VERSION requests */
@property (copy) NSString *version;

/** IRC server to connect to */
@property (copy) NSString *server;

/** IRC port to connect to */
@property (copy) NSString *port;

/** Server password to provide on connect (may be left empty or nil) */
@property (copy) NSString *password;

/** Nickname of the connected client. Note that setting this after connection will
    not result in the client renaming on IRC. You need to send a nick: message instead. 
 */
@property (copy) NSString *nickname;

/** Username of the connected client. Also known as the ident.
 
	Setting this after connection does nothing.
 */
@property (copy) NSString *username;

/** Realname of the connected client.
 
	Setting this after connection does nothing. */
@property (copy) NSString *realname;

/** An NSMutableDictionary of channels that the client is currently connected to.
 
	You should not modify this. */
@property (assign,readonly) NSMutableDictionary *channels;

/** The default text encoding for messages on this server. 
 
	This affects messages received via PRIVMSG and NOTICE, and TOPIC in a channel.
	You may change this at any time.
 */
 
@property (assign) NSStringEncoding encoding;

/** Connect to the IRC server.
 
	Note that this performs the initial DNS lookup and the TCP connection, so if
	there are any problems you will be notified via the return code of the message.
 
	Look at the libircclient documentation for the different return codes. */

- (int)connect;

/** Disconnect from the IRC server.
 
	This always works, as it simply shuts down the socket. If you want to disconnect
	in a friendly way, you should use the quit: message. */

- (void)disconnect;

/** returns YES if the server is currently connected successfully, and NO if
	it is not. */

- (bool)isConnected;

/**	Starts a new thread and starts the libircclient runloop, processing events and
	firing messages back to the main runloop as required. Calling this again will
	do nothing other than raise a warning in your logs. */

- (void)run;

/** Sends a raw message to the IRC server. Please consult rfc1459 for the format
	of IRC commands. */

- (int)sendRawWithFormat:(NSString *)format, ...;

/** quits the IRC server with the given reason. On success, an onQuit event will be
	sent to the IRCClientSessionDelegate with the nickname of the IRC client. 
 
	The format is a standard NSString format string, followed by optional arguments.
 */

- (int)quit:(NSString *)reason;

/** Joins a channel with a given name and key
 
	@param channel the channel to join
	@param key they key for the channel (may be nil)
 */

- (int)join:(NSString *)channel key:(NSString *)key;

/**	lists channels on the IRC server.
 
	@param channel a channel name or string to pass to the LIST command. Implementation specific.
 */

- (int)list:(NSString *)channel;

/** sets the user mode for the IRC client
 
	@param mode string to set
 */

- (int)userMode:(NSString *)mode;

/**	sets the IRC client nickname. On success, an onNick event will be sent to the delegate 
 
	@param newnick new nickname to set.
 */

- (int)nick:(NSString *)newnick;

/**	sends a WHOIS request to the IRC server
 
	@param nick nickname of the irc client to whois.
*/

- (int)whois:(NSString *)nick;

/**	send a PRIVMSG to another IRC client
 
	@param message message to send
	@param target the other IRC client to send the message to.
 */

- (int)message:(NSString *)message to:(NSString *)target;

/**	send a CTCP ACTION to another IRC client
 
	@param action the action message to send
	@param target the nickname of the irc client to send the message to.
 */

- (int)action:(NSString *)action to:(NSString *)target;

/**	send a NOTICE to another IRC client
 
	@param notice the message text to send
	@param target the nickname of the irc client to send the notice to.
 */

- (int)notice:(NSString *)notice to:(NSString *)target;

/** send a CTCP request to another IRC client
 
	@param request the CTCP request string to send
	@param target the nickname of the IRC client to send the request to.
 */

- (int)ctcpRequest:(NSString *)request target:(NSString *)target;

/** send a CTCP reply to another IRC client
 
 @param reply the CTCP reply string to send
 @param target the nickname of the IRC client to send the reply to.
 */

- (int)ctcpReply:(NSString *)reply target:(NSString *)target;

@end
