/* This program is released under the Common Public License V1.0
 *
 * You should have received a copy of Common Public License V1.0 along with
 * with this program.
 */

/* Copyright IBM Corp. 2010, 2011 */
#include <fcntl.h>
#include <sys/errno.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include "ica_api.h"
#include "testcase.h"

#define NR_TESTS 7
#define NR_RANDOM_TESTS 1000

/* CTR data - 1 for AES128 */
unsigned char NIST_KEY_CTR_E1[] = {
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
	0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c,
};

unsigned char NIST_IV_CTR_E1[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

unsigned char NIST_EXPECTED_IV_CTR_E1[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x00,
};

unsigned char NIST_TEST_DATA_CTR_E1[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
};

unsigned char NIST_TEST_RESULT_CTR_E1[] = {
	0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26,
	0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
};

/* CTR data - 2 for AES128 */
unsigned char NIST_KEY_CTR_E2[] = {
	0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
	0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c,
};

unsigned char NIST_IV_CTR_E2[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

unsigned char NIST_EXPECTED_IV_CTR_E2[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x03,
};

unsigned char NIST_TEST_DATA_CTR_E2[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};

unsigned char NIST_TEST_RESULT_CTR_E2[] = {
	0x87, 0x4d, 0x61, 0x91, 0xb6, 0x20, 0xe3, 0x26,
	0x1b, 0xef, 0x68, 0x64, 0x99, 0x0d, 0xb6, 0xce,
	0x98, 0x06, 0xf6, 0x6b, 0x79, 0x70, 0xfd, 0xff,
	0x86, 0x17, 0x18, 0x7b, 0xb9, 0xff, 0xfd, 0xff,
	0x5a, 0xe4, 0xdf, 0x3e, 0xdb, 0xd5, 0xd3, 0x5e,
	0x5b, 0x4f, 0x09, 0x02, 0x0d, 0xb0, 0x3e, 0xab,
	0x1e, 0x03, 0x1d, 0xda, 0x2f, 0xbe, 0x03, 0xd1,
	0x79, 0x21, 0x70, 0xa0, 0xf3, 0x00, 0x9c, 0xee,
};

/* CTR data - 3 - for AES192 */
unsigned char NIST_KEY_CTR_E3[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4,
};

unsigned char NIST_IV_CTR_E3[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

unsigned char NIST_EXPECTED_IV_CTR_E3[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x00,
};

unsigned char NIST_TEST_DATA_CTR_E3[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
};

unsigned char NIST_TEST_RESULT_CTR_E3[] = {
	0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
	0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
};

/* CTR data - 4 - for AES192 */
unsigned char NIST_KEY_CTR_E4[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4,
};

unsigned char NIST_IV_CTR_E4[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x00,
};

unsigned char NIST_EXPECTED_IV_CTR_E4[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x01,
};

unsigned char NIST_TEST_DATA_CTR_E4[] = {
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
};

unsigned char NIST_TEST_RESULT_CTR_E4[] = {
	0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
	0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
};

/* CTR data 5 - for AES 256 */
unsigned char NIST_KEY_CTR_E5[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4,
};

unsigned char NIST_IV_CTR_E5[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

unsigned char NIST_EXPECTED_IV_CTR_E5[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x03,
};

unsigned char NIST_TEST_DATA_CTR_E5[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};

unsigned char NIST_TEST_RESULT_CTR_E5[] = {
	0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
	0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
	0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
	0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
	0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
	0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
	0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6,
	0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6,
};

/* CTR data 6 - for AES 256.
 * Data is != BLOCK_SIZE */
unsigned char NIST_KEY_CTR_E6[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4,
};

unsigned char NIST_IV_CTR_E6[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

unsigned char NIST_EXPECTED_IV_CTR_E6[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x03,
};

unsigned char NIST_TEST_DATA_CTR_E6[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
};

unsigned char NIST_TEST_RESULT_CTR_E6[] = {
	0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
	0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
	0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
	0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
	0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
	0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
	0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6,
};

/* CTR data 7 - for AES 256
 * Counter as big as the data. Therefore the counter
 * should not be updated. Because it is already pre
 * computed. */
unsigned char NIST_KEY_CTR_E7[] = {
	0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
	0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
	0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
	0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4,
};

unsigned char NIST_IV_CTR_E7[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x00,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x01,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x02,
};

unsigned char NIST_EXPECTED_IV_CTR_E7[] = {
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x00,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x01,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xff, 0x02,
};

unsigned char NIST_TEST_DATA_CTR_E7[] = {
	0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
	0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
	0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
	0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
	0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
	0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
	0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
	0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10,
};

unsigned char NIST_TEST_RESULT_CTR_E7[] = {
	0x60, 0x1e, 0xc3, 0x13, 0x77, 0x57, 0x89, 0xa5,
	0xb7, 0xa7, 0xf5, 0x04, 0xbb, 0xf3, 0xd2, 0x28,
	0xf4, 0x43, 0xe3, 0xca, 0x4d, 0x62, 0xb5, 0x9a,
	0xca, 0x84, 0xe9, 0x90, 0xca, 0xca, 0xf5, 0xc5,
	0x2b, 0x09, 0x30, 0xda, 0xa2, 0x3d, 0xe9, 0x4c,
	0xe8, 0x70, 0x17, 0xba, 0x2d, 0x84, 0x98, 0x8d,
	0xdf, 0xc9, 0xc5, 0x8d, 0xb6, 0x7a, 0xad, 0xa6,
	0x13, 0xc2, 0xdd, 0x08, 0x45, 0x79, 0x41, 0xa6,
};

void dump_ctr_data(unsigned char *iv, unsigned int iv_length,
		   unsigned char *key, unsigned int key_length,
		   unsigned char *input_data, unsigned int data_length,
		   unsigned char *output_data)
{
	VV_(printf("IV \n"));
	dump_array(iv, iv_length);
	VV_(printf("Key \n"));
	dump_array(key, key_length);
	VV_(printf("Input Data\n"));
	dump_array(input_data, data_length);
	VV_(printf("Output Data\n"));
	dump_array(output_data, data_length);
}

void get_sizes(unsigned int *data_length, unsigned int *iv_length,
	       unsigned int *key_length, unsigned int iteration)
{
	switch (iteration) {
		case 1:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E1);
			*iv_length = sizeof(NIST_IV_CTR_E1);
			*key_length = sizeof(NIST_KEY_CTR_E1);
			break;
		case 2:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E2);
			*iv_length = sizeof(NIST_IV_CTR_E2);
			*key_length = sizeof(NIST_KEY_CTR_E2);
			break;
		case 3:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E3);
			*iv_length = sizeof(NIST_IV_CTR_E3);
			*key_length = sizeof(NIST_KEY_CTR_E3);
			break;
		case 4:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E4);
			*iv_length = sizeof(NIST_IV_CTR_E4);
			*key_length = sizeof(NIST_KEY_CTR_E4);
			break;
		case 5:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E5);
			*iv_length = sizeof(NIST_IV_CTR_E5);
			*key_length = sizeof(NIST_KEY_CTR_E5);
			break;
		case 6:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E6);
			*iv_length = sizeof(NIST_IV_CTR_E6);
			*key_length = sizeof(NIST_KEY_CTR_E6);
			break;
		case 7:
			*data_length = sizeof(NIST_TEST_DATA_CTR_E7);
			*iv_length = sizeof(NIST_IV_CTR_E7);
			*key_length = sizeof(NIST_KEY_CTR_E7);
			break;
		default:
			*data_length = 0;
			*iv_length = 0;
			*key_length = 0;
			break;
	}

}

void load_test_data(unsigned char *data, unsigned int data_length,
		    unsigned char *result,
		    unsigned char *iv, unsigned char *expected_iv,
		    unsigned int iv_length,
		    unsigned char *key, unsigned int key_length,
		    unsigned int iteration)
{
	switch (iteration) {
		case 1:
			memcpy(data, NIST_TEST_DATA_CTR_E1, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E1, data_length);
			memcpy(iv, NIST_IV_CTR_E1, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E1, iv_length);
			memcpy(key, NIST_KEY_CTR_E1, key_length);
			break;
		case 2:
			memcpy(data, NIST_TEST_DATA_CTR_E2, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E2, data_length);
			memcpy(iv, NIST_IV_CTR_E2, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E2, iv_length);
			memcpy(key, NIST_KEY_CTR_E2, key_length);
			break;
		case 3:
			memcpy(data, NIST_TEST_DATA_CTR_E3, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E3, data_length);
			memcpy(iv, NIST_IV_CTR_E3, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E3, iv_length);
			memcpy(key, NIST_KEY_CTR_E3, key_length);
			break;
		case 4:
			memcpy(data, NIST_TEST_DATA_CTR_E4, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E4, data_length);
			memcpy(iv, NIST_IV_CTR_E4, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E4, iv_length);
			memcpy(key, NIST_KEY_CTR_E4, key_length);
			break;
		case 5:
			memcpy(data, NIST_TEST_DATA_CTR_E5, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E5, data_length);
			memcpy(iv, NIST_IV_CTR_E5, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E5, iv_length);
			memcpy(key, NIST_KEY_CTR_E5, key_length);
			break;
		case 6:
			memcpy(data, NIST_TEST_DATA_CTR_E6, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E6, data_length);
			memcpy(iv, NIST_IV_CTR_E6, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E6, iv_length);
			memcpy(key, NIST_KEY_CTR_E6, key_length);
			break;
		case 7:
			memcpy(data, NIST_TEST_DATA_CTR_E7, data_length);
			memcpy(result, NIST_TEST_RESULT_CTR_E7, data_length);
			memcpy(iv, NIST_IV_CTR_E7, iv_length);
			memcpy(expected_iv, NIST_EXPECTED_IV_CTR_E7, iv_length);
			memcpy(key, NIST_KEY_CTR_E7, key_length);
			break;
	}

}

int random_aes_ctr(int iteration, unsigned int data_length, unsigned int iv_length)
{
	unsigned int key_length = AES_KEY_LEN256;

	if (data_length % sizeof(ica_aes_vector_t))
		iv_length = sizeof(ica_aes_vector_t);

	unsigned char iv[iv_length];
	unsigned char tmp_iv[iv_length];
	unsigned char key[key_length];
	unsigned char input_data[data_length];
	unsigned char encrypt[data_length];
	unsigned char decrypt[data_length];

	int rc = 0;

	VV_(printf("Test Parameters for iteration = %i\n", iteration));
	VV_(printf("key length = %i, data length = %i, iv length = %i\n",
	    key_length, data_length, iv_length));

	rc = ica_random_number_generate(data_length, input_data);
	if (rc) {
		VV_(printf("random number generate returned rc = %i, errno = %i\n", rc, errno));
		return TEST_FAIL;
	}
	rc = ica_random_number_generate(iv_length, iv);
	if (rc) {
		VV_(printf("random number generate returned rc = %i, errno = %i\n", rc, errno));
		return TEST_FAIL;
	}

	rc = ica_random_number_generate(key_length, key);
	if (rc) {
		VV_(printf("random number generate returned rc = %i, errno = %i\n", rc, errno));
		return TEST_FAIL;
	}
	memcpy(tmp_iv, iv, iv_length);

	rc = ica_aes_ctr(input_data, encrypt, data_length, key, key_length,
			 tmp_iv, 32, 1);
	if (rc) {
		VV_(printf("ica_aes_ctr encrypt failed with rc = %i\n", rc));
		dump_ctr_data(iv, iv_length, key, key_length, input_data,
			      data_length, encrypt);
		return TEST_FAIL;
	}
	if (!rc) {
		VV_(printf("Encrypt:\n"));
		dump_ctr_data(iv, iv_length, key, key_length, input_data,
			      data_length, encrypt);
	}

	memcpy(tmp_iv, iv, iv_length);
	rc = ica_aes_ctr(encrypt, decrypt, data_length, key, key_length,
			 tmp_iv, 32, 0);
	if (rc) {
		VV_(printf("ica_aes_ctr decrypt failed with rc = %i\n", rc));
		dump_ctr_data(iv, iv_length, key, key_length, encrypt,
			      data_length, decrypt);
		return TEST_FAIL;
	}


	if (!rc) {
		VV_(printf("Decrypt:\n"));
		dump_ctr_data(iv, iv_length, key, key_length, encrypt,
			      data_length, decrypt);
	}

	if (memcmp(decrypt, input_data, data_length)) {
		VV_(printf("Decryption Result does not match the original data!\n"));
		VV_(printf("Original data:\n"));
		dump_array(input_data, data_length);
		VV_(printf("Decryption Result:\n"));
		dump_array(decrypt, data_length);
		rc++;
	}

	if (rc)
		return TEST_FAIL;

	return TEST_SUCC;
}

int kat_aes_ctr(int iteration)
{
	unsigned int data_length;
	unsigned int iv_length;
	unsigned int key_length;

	get_sizes(&data_length, &iv_length, &key_length, iteration);

	unsigned char iv[iv_length];
	unsigned char tmp_iv[iv_length];
	unsigned char expected_iv[iv_length];
	unsigned char key[key_length];
	unsigned char input_data[data_length];
	unsigned char encrypt[data_length];
	unsigned char decrypt[data_length];
	unsigned char result[data_length];

	int rc = 0;

	VV_(printf("Test Parameters for iteration = %i\n", iteration));
	VV_(printf("key length = %i, data length = %i, iv length = %i\n",
		key_length, data_length, iv_length));

	load_test_data(input_data, data_length, result, iv, expected_iv,
		       iv_length, key, key_length, iteration);
	memcpy(tmp_iv, iv, iv_length);

	if (iv_length == 16)
		rc = ica_aes_ctr(input_data, encrypt, data_length, key, key_length,
				 tmp_iv, 32, 1);
	else
		rc = ica_aes_ctrlist(input_data, encrypt, data_length, key, key_length,
				 tmp_iv, 1);
	if (rc) {
		VV_(printf("ica_aes_ctr encrypt failed with rc = %i\n", rc));
		dump_ctr_data(iv, iv_length, key, key_length, input_data,
			      data_length, encrypt);
	}
	if (!rc) {
		VV_(printf("Encrypt:\n"));
		dump_ctr_data(iv, iv_length, key, key_length, input_data,
			      data_length, encrypt);
	}

	if (memcmp(result, encrypt, data_length)) {
		VV_(printf("Encryption Result does not match the known ciphertext!\n"));
		VV_(printf("Expected data:\n"));
		dump_array(result, data_length);
		VV_(printf("Encryption Result:\n"));
		dump_array(encrypt, data_length);
		rc++;
	}

	if (memcmp(expected_iv, tmp_iv, iv_length)) {
		VV_(printf("Update of IV does not match the expected IV!\n"));
		VV_(printf("Expected IV:\n"));
		dump_array(expected_iv, iv_length);
		VV_(printf("Updated IV:\n"));
		dump_array(tmp_iv, iv_length);
		VV_(printf("Original IV:\n"));
		dump_array(iv, iv_length);
		rc++;
	}
	if (rc) {
		VV_(printf("AES CTR test exited after encryption\n"));
		return TEST_FAIL;
	}

	memcpy(tmp_iv, iv, iv_length);
	rc = ica_aes_ctr(encrypt, decrypt, data_length, key, key_length,
			 tmp_iv, 32,0);
	if (rc) {
		VV_(printf("ica_aes_ctr decrypt failed with rc = %i\n", rc));
		dump_ctr_data(iv, iv_length, key, key_length, encrypt,
			      data_length, decrypt);
		return TEST_FAIL;
	}


	if (!rc) {
		VV_(printf("Decrypt:\n"));
		dump_ctr_data(iv, iv_length, key, key_length, encrypt,
			      data_length, decrypt);
	}

	if (memcmp(decrypt, input_data, data_length)) {
		VV_(printf("Decryption Result does not match the original data!\n"));
		VV_(printf("Original data:\n"));
		dump_array(input_data, data_length);
		VV_(printf("Decryption Result:\n"));
		dump_array(decrypt, data_length);
		rc++;
	}

	if (rc)
		return TEST_FAIL;

	return TEST_SUCC;
}

/*
 * Perform CTR tests.
 */
int main(int argc, char **argv)
{
#ifdef NO_CPACF
	UNUSED(argc);
	UNUSED(argv);
	printf("Skipping AES-CTR test, because CPACF support disabled via config option.\n");
	return TEST_SKIP;
#else
	unsigned int endless = 0;
	int i = 0;
	int rc = 0;
	int error_count = 0;
	int iteration;
	unsigned int data_length = 1;
	unsigned int iv_length = sizeof(ica_aes_vector_t);
	unsigned int rdata;

	if (argc > 1) {
		if (strstr(argv[1], "endless"))
			endless = 1;
	}

	set_verbosity(argc, argv);

	if (!endless) {

		// not endless mode
		// run the verification tests with known test vectors
		for(iteration = 1; iteration <= NR_TESTS; iteration++)	{
			rc = kat_aes_ctr(iteration);
			if (rc) {
				V_(printf("kat_aes_ctr failed with rc = %i\n", rc));
				error_count++;
			}
		}
		// run random tests
		for(iteration = 1; iteration <= NR_RANDOM_TESTS; iteration++) {
			rc = random_aes_ctr(iteration, data_length, iv_length);
			if (rc) {
				V_(printf("random_aes_ctr failed with rc = %i\n", rc));
				error_count++;
			}
			// add a value between 1 and 8 to data_length
			if (ica_random_number_generate(sizeof(rdata), (unsigned char*) &rdata)) {
				printf("ica_random_number_generate failed with errnor = %i\n",
				       errno);
				return TEST_FAIL;
			}
			data_length += (rdata % 8) + 1;
		}

	} else {
		// endless mode
		while (1) {
			V_(printf("i = %i\n",i));
			rc = random_aes_ctr(i, 320, 320);
			if (rc) {
				V_(printf("random_aes_ctr failed with rc = %i\n", rc));
				return TEST_FAIL;
			}
			i++;
		}

	}

	if (error_count) {
		printf("%i AES-CTR tests failed\n", error_count);
		return TEST_FAIL;
	}

	printf("All AES-CTR tests passed.\n");
	return TEST_SUCC;
#endif /* NO_CPACF */
}

