
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/*******************************************************************************
*                   Copyright (c) 1991 - 2001 Gemplus
*
* Name        : GemGCR.h
*
* Description : General definition for GEMPLUS programs using APDU API.
*
* Release     : 4.31.002
*
* Last Modif  : 07/01/98: V4.31.002 - Add PCSC Keyboard protocol type
*               13/10/97: V4.31.001 - Add COMM_PCSC.
*               23/04/97: V4.30.002 - Modify prototype function G4_ICCSet(Get)PTS.
*               04/04/97: V4.30.001 - Add function G4_ICCSet(Get)Voltage and 
*                                     G4_ICCSet(Get)PTS.
*               01/04/97: V4.10.005 - Add the MAX_IFD_CHANNEL
*               12/02/97: V4.10.004 - Modify the PERIPHERAL macro.
*               07/02/97: V4.10.003 - Add COM_TCPIP and COM_HOOK into the 
*                                     structure G4_PARAM_CAHNNEL.
*               22/04/96: V4.10.002 - Update structure for 32 bits alignement.
*               01/12/95: V4.10.001
*               27/10/95: V4.10.001
*
********************************************************************************
*
* Warning     :
*
* Remark      :
*
*******************************************************************************/

/*------------------------------------------------------------------------------
Name definition:
   _GEMGCR_H is used to avoid multiple inclusion.
------------------------------------------------------------------------------*/
#ifndef _GEMGCR_H
#define _GEMGCR_H

/*------------------------------------------------------------------------------
Include section:
   GEMCOM.H contain communication definitions.
------------------------------------------------------------------------------*/
#include "gemcom.h"

/*------------------------------------------------------------------------------
Reader section:
   SECURITY macro gives access to reader embedded security modules. It must be
      use as SECURITY(GCR500) for example.
   PERIPHERAL macro gives access to reader embedded peripherals, like security
      module.
      SECURITY(GCR500) = PERIPHERAL(1,GCR500).
------------------------------------------------------------------------------*/
#define SECURITY(x)            (0x0100 | (x))
#define PERIPHERAL(x,y)        (x == 0) ? (y):((0x0080 << (x)) | (y))

/*------------------------------------------------------------------------------
Product Interface Library section:
   MAX_RESET_LEN holds the maximal length for an answer to reset.
   MAX_APDU_LEN is the maximal length for an APDU buffer. The maximal value, 
      according to ISO standard is 65544 but today, low level drivers are 
      limited by the size of a segment.
   MAX_IFD_STRING holds the maximal length for an IFD identier.
   MAX_OFD_SPECIFIC holds the maximal size for the IFD specific field.
   MAX_DUMMY_CHANNEL and MAX_DUMMY_SESSION are used in RFU fields.
   MAX_IFD_CHANNEL defines the maximal IFD supported in a same time by the API.
------------------------------------------------------------------------------*/
#if defined (G_UNIX) || defined (WIN32) || defined (G_OS2)
#define MAX_RESET_LEN        36
#else
#define MAX_RESET_LEN        33
#endif

#define MAX_APDU_LEN      65535lu
#define MAX_DUMMY_CHANNEL   100
#define MAX_DUMMY_SESSION   100 - MAX_RESET_LEN
#define MAX_IFD_STRING      100
#define MAX_IFD_SPECIFIC    100
#define MAX_IFD_CHANNEL	    16

/*------------------------------------------------------------------------------
   COMMAND_LEN define the size of an ICC command.
   NULL_LEN is a value used with Gemplus specific ICC products.
------------------------------------------------------------------------------*/
#define COMMAND_LEN          4
#define NULL_LEN    0xFFFFFFFFlu

/*------------------------------------------------------------------------------
   Defines the available protocols between host and reader:
      TLP224_PROTOCOL or BLOCK_PROTOCOL on GCI400.
      PCCARD_PROTOCOL on GPR or GPR400
      AT_G_PROTOCOL on GCM
      TCPIP_PROTOCOL for future use
      PCSC_KB_PROTOCOL for the PCSC defined protocol for Keyboard readers.
------------------------------------------------------------------------------*/
#define TLP224_PROTOCOL      0
#define BLOCK_PROTOCOL       1
#define PCCARD_PROTOCOL      2
#define AT_G_PROTOCOL        3
#define TCPIP_PROTOCOL       4
#define PCSC_KB_PROTOCOL     5

/*------------------------------------------------------------------------------
   Defines the available protocols between reader and card (PTS Mode):
   - IFD_DEFAULT_MODE           -> same as OROS 2.x maximum speed without request,
   - IFD_WITHOUT_PTS_REQUEST    -> no PTS management (baud rate is 9600 bps),
   - IFD_NEGOTIATE_PTS_OPTIMAL  -> PTS management automatically,
   - IFD_NEGOTIATE_PTS_MANUALLY -> PTS management "manually" by parameters. 
   Defines the PTS format (PTS0) and indicates by the bits b5,b6,b7 set to 1 the 
   presence of the optional parameters PTS1,PTS2,PTS3 respectively. The least
   significant bits b1 to b4 select protocol type type T. Use the macro 
   - IFD_NEGOTIATE_PTS1,
   - IFD_NEGOTIATE_PTS2,
   - IFD_NEGOTIATE_PTS3,
   - IFD_NEGOTIATE_T0,
   - IFD_NEGOTIATE_T1 to set these bits.
   Defines the ICC power supply voltage:
   - ICC_VCC_5V is the power supply voltage 5V (by default),
   - ICC_VCC_3V is the power supply voltage 3V,
   - ICC_VCC_DEFAULT is the power supply voltage by default (5V).
------------------------------------------------------------------------------*/
#define IFD_DEFAULT_MODE					0
#define IFD_WITHOUT_PTS_REQUEST			1
#define IFD_NEGOTIATE_PTS_OPTIMAL   	2
#define IFD_NEGOTIATE_PTS_MANUALLY		3

#define IFD_NEGOTIATE_T0					0x00
#define IFD_NEGOTIATE_T1					0x01

#define ICC_VCC_5V							0
#define ICC_VCC_3V							1
#define ICC_VCC_5_3V                       2
#define ICC_VCC_DEFAULT						ICC_VCC_5V

/*------------------------------------------------------------------------------
   G4_APDU_COMM structure holds:
      - Command holds the command bytes to send to ICC.
      - LengthIn holds the number of bytes to send to ICC. The allowed range is
        { 0 .. 65535 }. The NULL_LEN is used for Gemplus specific products.
      - DataIn holds the bytes to send to ICC.
        WARNING: the user must allocate this buffer.
      - LengthExpected memorises the maximal length expected in the ICC 
        response. The allowed range is { 0 .. 65536}.
------------------------------------------------------------------------------*/
typedef struct g4_apdu_comm
{
	WORD8 Command[COMMAND_LEN];
	WORD32 LengthIn;
	WORD8 G_FAR *DataIn;
	WORD32 LengthExpected;

}
G4_APDU_COMM;

/*------------------------------------------------------------------------------
G4_APDU_RESP structure holds:
   - LengthOut is the real number of received bytes.
   - DataOut holds the received bytes. 
     WARNING: the user must allocate this buffer.
   - Status holds the two status bytes SW1 and SW2.
------------------------------------------------------------------------------*/
typedef struct g4_apdu_resp
{
	WORD32 LengthOut;
	WORD8 G_FAR *DataOut;
#if defined (G_UNIX) || defined (WIN32)	|| defined (G_OS2)
	WORD32 Status;
#else
	WORD16 Status;
#endif

}
G4_APDU_RESP;

#endif
