/*    This file is part of the Gudhi Library - https://gudhi.inria.fr/ - which is released under MIT.
 *    See file LICENSE or go to https://gudhi.inria.fr/licensing/ for full license details.
 *    Author:       Francois Godi
 *
 *    Copyright (C) 2015 Inria
 *
 *    Modification(s):
 *      - YYYY/MM Author: Description of the modification
 */


#define BOOST_TEST_DYN_LINK
#define BOOST_TEST_MODULE "bottleneck distance"
#include <boost/test/unit_test.hpp>

#include <random>
#include <gudhi/Bottleneck.h>
#include <gudhi/Unitary_tests_utils.h>

using namespace Gudhi::persistence_diagram;

int n1 = 81;  // a natural number >0
int n2 = 180;  // a natural number >0
double upper_bound = 406.43;  // any real >0


std::uniform_real_distribution<double> unif(0., upper_bound);
std::default_random_engine re;
std::vector< std::pair<double, double> > v1, v2;

BOOST_AUTO_TEST_CASE(persistence_graph) {
  // Random construction
  for (int i = 0; i < n1; i++) {
    double a = unif(re);
    double b = unif(re);
    v1.emplace_back(std::min(a, b), std::max(a, b));
  }
  for (int i = 0; i < n2; i++) {
    double a = unif(re);
    double b = unif(re);
    v2.emplace_back(std::min(a, b), std::max(a, b));
  }
  Persistence_graph g(v1, v2, 0.);
  std::vector<double> d(g.sorted_distances());
  //
  BOOST_CHECK(!g.on_the_u_diagonal(n1 - 1));
  BOOST_CHECK(!g.on_the_u_diagonal(n1));
  BOOST_CHECK(!g.on_the_u_diagonal(n2 - 1));
  BOOST_CHECK(g.on_the_u_diagonal(n2));
  BOOST_CHECK(!g.on_the_v_diagonal(n1 - 1));
  BOOST_CHECK(g.on_the_v_diagonal(n1));
  BOOST_CHECK(g.on_the_v_diagonal(n2 - 1));
  BOOST_CHECK(g.on_the_v_diagonal(n2));
  //
  BOOST_CHECK(g.corresponding_point_in_u(0) == n2);
  BOOST_CHECK(g.corresponding_point_in_u(n1) == 0);
  BOOST_CHECK(g.corresponding_point_in_v(0) == n1);
  BOOST_CHECK(g.corresponding_point_in_v(n2) == 0);
  //
  BOOST_CHECK(g.size() == (n1 + n2));
  //
  BOOST_CHECK((int) d.size() == (n1 + n2)*(n1 + n2) + n1 + n2 + 1);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(0, 0))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(0, n1 - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(0, n1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(0, n2 - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(0, n2))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(0, (n1 + n2) - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(n1, 0))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(n1, n1 - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(n1, n1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(n1, n2 - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(n1, n2))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance(n1, (n1 + n2) - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance((n1 + n2) - 1, 0))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance((n1 + n2) - 1, n1 - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance((n1 + n2) - 1, n1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance((n1 + n2) - 1, n2 - 1))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance((n1 + n2) - 1, n2))) > 0);
  BOOST_CHECK(std::count(d.begin(), d.end(), GUDHI_PROTECT_FLOAT(g.distance((n1 + n2) - 1, (n1 + n2) - 1))) > 0);
}

BOOST_AUTO_TEST_CASE(neighbors_finder) {
  Persistence_graph g(v1, v2, 0.);
  Neighbors_finder nf(g, 1.);
  for (int v_point_index = 1; v_point_index < ((n2 + n1)*9 / 10); v_point_index += 2)
    nf.add(v_point_index);
  //
  int v_point_index_1 = nf.pull_near(n2 / 2);
  BOOST_CHECK((v_point_index_1 == -1) || (g.distance(n2 / 2, v_point_index_1) <= 1.));
  std::vector<int> l = nf.pull_all_near(n2 / 2);
  bool v = true;
  for (auto it = l.cbegin(); it != l.cend(); ++it)
    v = v && (g.distance(n2 / 2, *it) > 1.);
  BOOST_CHECK(v);
  int v_point_index_2 = nf.pull_near(n2 / 2);
  BOOST_CHECK(v_point_index_2 == -1);
}

BOOST_AUTO_TEST_CASE(layered_neighbors_finder) {
  Persistence_graph g(v1, v2, 0.);
  Layered_neighbors_finder lnf(g, 1.);
  for (int v_point_index = 1; v_point_index < ((n2 + n1)*9 / 10); v_point_index += 2)
    lnf.add(v_point_index, v_point_index % 7);
  //
  int v_point_index_1 = lnf.pull_near(n2 / 2, 6);
  BOOST_CHECK((v_point_index_1 == -1) || (g.distance(n2 / 2, v_point_index_1) <= 1.));
  int v_point_index_2 = lnf.pull_near(n2 / 2, 6);
  BOOST_CHECK(v_point_index_2 == -1);
  v_point_index_1 = lnf.pull_near(n2 / 2, 0);
  BOOST_CHECK((v_point_index_1 == -1) || (g.distance(n2 / 2, v_point_index_1) <= 1.));
  v_point_index_2 = lnf.pull_near(n2 / 2, 0);
  BOOST_CHECK(v_point_index_2 == -1);
}

BOOST_AUTO_TEST_CASE(graph_matching) {
  Persistence_graph g(v1, v2, 0.);
  Graph_matching m1(g);
  m1.set_r(0.);
  int e = 0;
  while (m1.multi_augment())
    ++e;
  BOOST_CHECK(e > 0);
  BOOST_CHECK(e <= 2 * sqrt(2 * (n1 + n2)));
  Graph_matching m2 = m1;
  BOOST_CHECK(!m2.multi_augment());
  m2.set_r(upper_bound);
  e = 0;
  while (m2.multi_augment())
    ++e;
  BOOST_CHECK(e <= 2 * sqrt(2 * (n1 + n2)));
  BOOST_CHECK(m2.perfect());
  BOOST_CHECK(!m1.perfect());
}

BOOST_AUTO_TEST_CASE(global) {
  std::uniform_real_distribution<double> unif1(0., upper_bound);
  std::uniform_real_distribution<double> unif2(upper_bound / 10000., upper_bound / 100.);
  std::default_random_engine re;
  std::vector< std::pair<double, double> > v1, v2;
  for (int i = 0; i < n1; i++) {
    double a = unif1(re);
    double b = unif1(re);
    double x = unif2(re);
    double y = unif2(re);
    v1.emplace_back(std::min(a, b), std::max(a, b));
    v2.emplace_back(std::min(a, b) + std::min(x, y), std::max(a, b) + std::max(x, y));
    if (i % 5 == 0)
      v1.emplace_back(std::min(a, b), std::min(a, b) + x);
    if (i % 3 == 0)
      v2.emplace_back(std::max(a, b), std::max(a, b) + y);
  }
  BOOST_CHECK(bottleneck_distance(v1, v2, 0.) <= upper_bound / 100.);
  BOOST_CHECK(bottleneck_distance(v1, v2, upper_bound / 10000.) <= upper_bound / 100. + upper_bound / 10000.);
  BOOST_CHECK(std::abs(bottleneck_distance(v1, v2, 0.) - bottleneck_distance(v1, v2, upper_bound / 10000.)) <= upper_bound / 10000.);

  std::vector< std::pair<double, double> > empty;
  std::vector< std::pair<double, double> > one = {{8, 10}};
  BOOST_CHECK(bottleneck_distance(empty, empty) == 0);
  BOOST_CHECK(bottleneck_distance(empty, one) == 1);
}

BOOST_AUTO_TEST_CASE(neg_global) {
  std::uniform_real_distribution<double> unif1(0., upper_bound);
  std::uniform_real_distribution<double> unif2(upper_bound / 10000., upper_bound / 100.);
  std::default_random_engine re;
  std::vector< std::pair<double, double> > v1, v2;
  for (int i = 0; i < n1; i++) {
    double a = std::log(unif1(re));
    double b = std::log(unif1(re));
    double x = std::log(unif2(re));
    double y = std::log(unif2(re));
    v1.emplace_back(std::min(a, b), std::max(a, b));
    v2.emplace_back(std::min(a, b) + std::min(x, y), std::max(a, b) + std::max(x, y));
    if (i % 5 == 0)
      v1.emplace_back(std::min(a, b), std::min(a, b) + x);
    if (i % 3 == 0)
      v2.emplace_back(std::max(a, b), std::max(a, b) + y);
  }
  BOOST_CHECK(bottleneck_distance(v1, v2, 0.) <= upper_bound / 100.);
  BOOST_CHECK(bottleneck_distance(v1, v2, upper_bound / 10000.) <= upper_bound / 100. + upper_bound / 10000.);
  BOOST_CHECK(std::abs(bottleneck_distance(v1, v2, 0.) - bottleneck_distance(v1, v2, upper_bound / 10000.)) <= upper_bound / 10000.);

  std::vector< std::pair<double, double> > empty;
  std::vector< std::pair<double, double> > one = {{8, 10}};
  BOOST_CHECK(bottleneck_distance(empty, empty) == 0);
  BOOST_CHECK(bottleneck_distance(empty, one) == 1);
}

BOOST_AUTO_TEST_CASE(bottleneck_simple_test) {
  std::vector< std::pair<double, double> > v1, v2;
  double inf = std::numeric_limits<double>::infinity();
  double neginf = -inf;
  double b;

  v1.emplace_back(9.6, 14.);
  v2.emplace_back(9.5, 14.1);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK(b > 0.09 && b < 0.11);

  v1.emplace_back(-34.974, -34.2);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK(b > 0.386 && b < 0.388);

  v1.emplace_back(neginf, 3.7);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK_EQUAL(b, inf);

  v2.emplace_back(neginf, 4.45);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK(b > 0.74 && b < 0.76);

  v1.emplace_back(-60.6, 52.1);
  v2.emplace_back(-61.5, 53.);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK(b > 0.89 && b < 0.91);

  v1.emplace_back(3., inf);
  v2.emplace_back(3.2, inf);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK(b > 0.89 && b < 0.91);

  v1.emplace_back(neginf, inf);
  v2.emplace_back(neginf, inf);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK(b > 0.89 && b < 0.91);

  v2.emplace_back(6, inf);

  b = Gudhi::persistence_diagram::bottleneck_distance(v1, v2, 0.);
  BOOST_CHECK_EQUAL(b, inf);
}
