### Copyright (C) 2010 Peter Williams <peter_ono@users.sourceforge.net>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os

from gquilt_pkg import gquilt_tool, gquilt_quilt, gquilt_mq
from gquilt_pkg import dialogue, cmd_result, config, ws_event

_BACKEND = {}
_MISSING_BACKEND = {}

def _add_backend(newifce):
    if newifce.is_available():
        _BACKEND[newifce.name] = newifce
    else:
        _MISSING_BACKEND[newifce.name] = newifce

_add_backend(gquilt_quilt.Interface())
_add_backend(gquilt_mq.Interface())

_NULL_BACKEND = gquilt_tool.NullInterface()

def backend_requirements():
    msg = "No back ends are available. At least one of:" + os.linesep
    for key in list(_MISSING_BACKEND.keys()):
        msg += "\t" + _MISSING_BACKEND[key].requires() + os.linesep
    msg += "must be installed/available for \"gquilt\" to do.anything useful."
    return msg

def report_backend_requirements(parent=None):
    dialogue.inform_user(backend_requirements(), parent=parent)

def avail_backends():
    return list(_BACKEND.keys())

def playground_type(dirpath=None):
    for bname in list(_BACKEND.keys()):
        if _BACKEND[bname].is_playground(dirpath):
            return bname
    return None

ifce = _NULL_BACKEND
in_valid_pgnd = ifce is not _NULL_BACKEND

def set_ifce(dirpath=None):
    global ifce, in_valid_pgnd
    pgt = playground_type(dirpath)
    if pgt is None:
        ifce = _NULL_BACKEND
    else:
        ifce = _BACKEND[pgt]
    in_valid_pgnd = ifce is not _NULL_BACKEND

def choose_backend():
    bel = avail_backends()
    if len(bel) == 0:
        report_backend_requirements()
        return False, ""
    elif len(bel) == 1:
        return True, bel[0]
    sfld = dialogue.SelectFromListDialog(olist=bel, prompt="Choose back end:")
    is_ok, req_backend = sfld.make_selection()
    return is_ok, req_backend

def new_playground(pgdir, backend=None):
    if backend is None:
        result = ifce.new_playground(pgdir)
    else:
        result = _BACKEND[backend].new_playground(pgdir)
    set_ifce()
    return result

def chdir(newdir=None):
    global ifce, in_valid_pgnd
    if newdir:
        try:
            os.chdir(newdir)
        except OSError as err:
            import errno
            ecode = errno.errorcode[err.errno]
            emsg = err.strerror
            return (cmd_result.ERROR, '', '%s: "%s" :%s' % (ecode, newdir, emsg))
    result = (cmd_result.OK, '', '')
    newpgtype = playground_type()
    if newpgtype:
        newdir = _BACKEND[newpgtype].get_playground_root()
        os.chdir(newdir)
        ifce = _BACKEND[newpgtype]
        config.append_saved_ws(newdir)
    elif len(_BACKEND) > 0:
        ifce = _NULL_BACKEND
    else:
        result = (cmd_result.ERROR, '', backend_requirements())
    in_valid_pgnd = ifce is not _NULL_BACKEND
    ws_event.notify_events(ws_event.CHANGE_WD, newdir)
    return result
