### Copyright (C) 2007 Peter Williams <peter_ono@users.sourceforge.net>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk

def get_gtk_window(widget):
    gtk_window = widget
    while True:
        temp = gtk_window.get_parent()
        if temp:
            gtk_window = temp
        else:
            break
    return gtk_window

def pygtk_version_ge(version):
    for index in range(len(version)):
        if gtk.pygtk_version[index] >  version[index]:
            return True
        elif gtk.pygtk_version[index] <  version[index]:
            return False
    return True

if pygtk_version_ge((2, 12)):
    def set_widget_tooltip_text(widget, text):
        widget.set_tooltip_text(text)
else:
    tooltips = gtk.Tooltips()
    tooltips.enable()

    def set_widget_tooltip_text(widget, text):
        tooltips.set_tip(widget, text)

def wrap_in_scrolled_window(widget, policy=(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC), with_frame=True, label=None):
    scrw = gtk.ScrolledWindow()
    scrw.set_policy(policy[0], policy[1])
    scrw.add(widget)
    if with_frame:
        frame = gtk.Frame(label)
        frame.add(scrw)
        frame.show_all()
        return frame
    else:
        scrw.show_all()
        return scrw

_KEYVAL_UP_ARROW = gtk.gdk.keyval_from_name('Up')
_KEYVAL_DOWN_ARROW = gtk.gdk.keyval_from_name('Down')

class EntryWithHistory(gtk.Entry):
    def __init__(self, max_chars=0):
        gtk.Entry.__init__(self, max_chars)
        self._history_list = []
        self._history_index = 0
        self._history_len = 0
        self._saved_text = ''
        self._key_press_cb_id = self.connect("key_press_event", self._key_press_cb)
    def _key_press_cb(self, widget, event):
        if event.keyval in [_KEYVAL_UP_ARROW, _KEYVAL_DOWN_ARROW]:
            if event.keyval == _KEYVAL_UP_ARROW:
                if self._history_index < self._history_len:
                    if self._history_index == 0:
                        self._saved_text = self.get_text()
                    self._history_index += 1
                    self.set_text(self._history_list[-self._history_index])
                    self.set_position(-1)
            elif event.keyval == _KEYVAL_DOWN_ARROW:
                if self._history_index > 0:
                    self._history_index -= 1
                    if self._history_index > 0:
                        self.set_text(self._history_list[-self._history_index])
                    else:
                        self.set_text(self._saved_text)
                    self.set_position(-1)
            return True
        else:
            return False
    def clear_to_history(self):
        self._history_index = 0
        # beware the empty command string
        text = self.get_text().rstrip()
        self.set_text("")
        # don't save empty entries or ones that start with white space
        if not text or text[0] in [' ', '\t']:
            return
        # no adjacent duplicate entries allowed
        if (self._history_len == 0) or (text != self._history_list[-1]):
            self._history_list.append(text)
            self._history_len = len(self._history_list)
    def get_text_and_clear_to_history(self):
        text = self.get_text().rstrip()
        self.clear_to_history()
        return text

class LabelledEntryWithHistory(gtk.HBox):
    def __init__(self, label, max_chars=0):
        gtk.HBox.__init__(self)
        self.entry = EntryWithHistory(max_chars)
        self.label = gtk.Label(label)
        self.pack_start(self.label, expand=False)
        self.pack_start(self.entry)
    def get_text_and_clear_to_history(self):
        return self.entry.get_text_and_clear_to_history()
    def set_label(self, text):
        self.label.set_text(text)

class ActionButton(gtk.Button):
    def __init__(self, action, use_underline=True):
        label = action.get_property("label")
        stock_id = action.get_property("stock-id")
        if label is "":
            # Empty (NB not None) label means use image only
            gtk.Button.__init__(self, use_underline=use_underline)
            image = gtk.Image()
            image.set_from_stock(stock_id, gtk.ICON_SIZE_BUTTON)
            self.add(image)
        else:
            gtk.Button.__init__(self, stock=stock_id, label=label, use_underline=use_underline)
        set_widget_tooltip_text(self, action.get_property("tooltip"))
        action.connect_proxy(self)

class ActionButtonList:
    def __init__(self, action_group_list, action_name_list=None, use_underline=True):
        self.list = []
        self.dict = {}
        if action_name_list:
            for a_name in action_name_list:
                for a_group in action_group_list:
                    action = a_group.get_action(a_name)
                    if action:
                        button = ActionButton(action, use_underline)
                        self.list.append(button)
                        self.dict[a_name] = button
                        break
        else:
            for a_group in action_group_list:
                for action in a_group.list_actions():
                    button = ActionButton(action, use_underline)
                    self.list.append(button)
                    self.dict[action.get_name()] = button

class ActionHButtonBox(gtk.HBox):
    def __init__(self, action_group_list, action_name_list=None,
                 use_underline=True, expand=True, fill=True, padding=0):
        gtk.HBox.__init__(self)
        self.button_list = ActionButtonList(action_group_list, action_name_list, use_underline)
        for button in self.button_list.list:
            self.pack_start(button, expand, fill, padding)

def _ui_manager_connect_proxy(_ui_mgr, action, widget):
    tooltip = action.get_property('tooltip')
    if isinstance(widget, gtk.MenuItem) and tooltip:
        widget.set_tooltip_text(tooltip)

class UIManager(gtk.UIManager):
    def __init__(self):
        gtk.UIManager.__init__(self)
        self.connect('connect-proxy', _ui_manager_connect_proxy)
