// Copyright 2019 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package datastore

import (
	"testing"
)

func TestKeyConversion(t *testing.T) {
	tests := []struct {
		desc string

		gaeKey      string
		wantEncoded string
		wantName    string
	}{
		{
			desc:        "A GAE key with a Name ID.", // Generated by: https://play.golang.org/p/00fNVEKSmBX
			gaeKey:      "agdmb28tYXBwchwLEgh0ZXN0S2luZCIOZm9vYmFyc3RyaW5naWQM",
			wantEncoded: "EhoKCHRlc3RLaW5kGg5mb29iYXJzdHJpbmdpZA",
			wantName:    "foobarstringid",
		},
		{
			desc:        "Another GAE key with a Name ID",
			gaeKey:      "ag1wfnBlbmRvLWV4YW1wcjkLEg90ZXN0U3RyaW5nS2V5czIiJDdlNDA2YTNkLTUyNTItMTFlOS05MTMyLTdmNjUyMzg3YThlYgw",
			wantEncoded: "EjcKD3Rlc3RTdHJpbmdLZXlzMhokN2U0MDZhM2QtNTI1Mi0xMWU5LTkxMzItN2Y2NTIzODdhOGVi",
			wantName:    "7e406a3d-5252-11e9-9132-7f652387a8eb",
		},
	}

	for _, tc := range tests {
		k, err := DecodeKey(tc.gaeKey)
		if err != nil {
			t.Fatalf("DecodeKey(%s): %v", tc.desc, err)
		}
		if k.Name != tc.wantName {
			t.Errorf("Name(%s): got %+v, want %+v", tc.desc, k.Name, tc.wantName)
		}
		if enc := k.Encode(); enc != tc.wantEncoded {
			t.Errorf("Encode(%s): got %+v, want %+v", tc.desc, enc, tc.wantEncoded)
		}
	}
}
