/*******************************************************************************
*
*       This file is part of the General Hidden Markov Model Library,
*       GHMM version __VERSION__, see http://ghmm.org
*
*       Filename: ghmm/ghmm/sfoba.h
*       Authors:  Bernhard Knab, Benjamin Georgi
*
*       Copyright (C) 1998-2004 Alexander Schliep 
*       Copyright (C) 1998-2001 ZAIK/ZPR, Universitaet zu Koeln
*	Copyright (C) 2002-2004 Max-Planck-Institut fuer Molekulare Genetik, 
*                               Berlin
*                                   
*       Contact: schliep@ghmm.org             
*
*       This library is free software; you can redistribute it and/or
*       modify it under the terms of the GNU Library General Public
*       License as published by the Free Software Foundation; either
*       version 2 of the License, or (at your option) any later version.
*
*       This library is distributed in the hope that it will be useful,
*       but WITHOUT ANY WARRANTY; without even the implied warranty of
*       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*       Library General Public License for more details.
*
*       You should have received a copy of the GNU Library General Public
*       License along with this library; if not, write to the Free
*       Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*
*       This file is version $Revision: 2262 $ 
*                       from $Date: 2009-04-22 09:44:25 -0400 (Wed, 22 Apr 2009) $
*             last change by $Author: grunau $.
*
*******************************************************************************/
#ifndef GHMM_SFOBA_H
#define GHMM_SFOBA_H

#include "smodel.h"

#ifdef __cplusplus
extern "C" {
#endif

/**@name SHMM Forward-Backward-Algorithm */
/*@{ (Doc++-Group: sfoba) */

/** Forward-Backward-Algorithm for multiple double
    sequences with scaling.
    For reference see:  
    Rabiner, L.R.: "`A Tutorial on Hidden {Markov} Models and Selected
                Applications in Speech Recognition"', Proceedings of the IEEE,
	77, no 2, 1989, pp 257--285
*/


/** Forward-Algorithm.
  Calculates alpha[t][i], scaling factors scale[t] and log( P(O|lambda) ) for
  a given double sequence and a given model.
  @param smo      model
  @param O        sequence
  @param T        length of sequence (O is actually T*smo->dim long)
  @param b        optionally precomputed emission probabilities
  @param alpha    alpha[t][i]
  @param scale    scale factors
  @param log_p    log likelihood log( P(O|lambda) )
  @return 0 for success, -1 for error
  */
  int ghmm_cmodel_forward (ghmm_cmodel * smo, double *O, int T, double ***b,
                     double **alpha, double *scale, double *log_p);

/** 
  Backward-Algorithm. 
  Calculates beta[t][i] given a double sequence and a model. Scale factors 
  given as parameter (come from ghmm_cmodel_forward()).
  @param smo      model
  @param O        sequence
  @param T        length of sequence (O is actually T*smo->dim long)
  @param b        matrix with precalculated output probabilities. May be NULL
  @param beta     beta[t][i]
  @param scale    scale factors
  @return 0 for success, -1 for error
  */
  int ghmm_cmodel_backward (ghmm_cmodel * smo, double *O, int T, double ***b,
                      double **beta, const double *scale);

/**
  Calculation of  log( P(O|lambda) ). 
  Done by calling ghmm_cmodel_forward(). Use this function if only the
  log likelihood and not alpha[t][i] is needed, alpha matrix is allocated with
  ighmm_cmatrix_stat_alloc
  @param smo      model
  @param O        sequence
  @param T        length of sequence (O is actually T*smo->dim long)
  @param log_p    log likelihood log( P(O|lambda) )
  @return 0 for success, -1 for error
  */
  int ghmm_cmodel_logp (ghmm_cmodel * smo, double *O, int T, double *log_p);


/**
  Calculation of log( P(O,S | lambda) ).
  Computes joint probability of sequence and state sequence
  @param mo        model
  @param O         sequence
  @param len       length of sequence
  @param S         state sequence
  @param slen      length of state sequence (differs from len only for HMMs
                   with silent states)
  @param log_p     log likelihood log( P(O|lambda) )
  @return 0 for success, -1 for error
  */
int ghmm_cmodel_logp_joint(ghmm_cmodel *mo, const double *O, int len,
                           const int *S, int slen, double *log_p);


#ifdef __cplusplus
}
#endif
/*@} (Doc++-Group: sfoba) */
#endif
