/* doscan - Denial Of Service Capable Auditing of Networks       -*- C++ -*-
 * Copyright (C) 2003 Florian Weimer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "rx.h"

#include <string>

// rx

rx::rx(const char* pattern, int options) throw (error)
{
  const char *err;
  int offset;

  regexp = pcre_compile(pattern, options, &err, &offset, 0);
  if (regexp == 0) {
    throw error(err, offset);
  }

  pcre_fullinfo(regexp, 0, PCRE_INFO_CAPTURECOUNT, &capture_count);
}

rx::~rx()
{
  free(regexp);
}

bool
rx::exec(const std::string& data, unsigned offset, int options) const
{
  // pcre_exec() might need to store the matches for backreferences in
  // the pattern, so we always pass the ovector array of the suitable
  // length.

  int ovecsize = (capture_count + 1) * 3;
  int ovector[ovecsize];
  int result = pcre_exec(regexp, 0, data.data(), data.size(), offset, options,
                         ovector, ovecsize);

  if (result >= 0) {
    return true;
  } else if (result == PCRE_ERROR_NOMATCH) {
    return false;
  } else {
    throw match_error(result);
  }
}

bool
rx::exec(const std::string& data, match& m, unsigned offset, int options) const
{
  // pcre_exec() might need to store the matches for backreferences in
  // the pattern, so we always pass the ovector array of the suitable
  // length.

  int ovecsize = (capture_count + 1) * 3;
  int ovector[ovecsize];
  int result = pcre_exec(regexp, 0, data.data(), data.size(), offset, options,
                         ovector, ovecsize);

  if (result > 0) {
    m.first = data.begin() + ovector[0];
    m.last = data.begin() + ovector[1];
    return true;
  } else if (result == 0 || result == PCRE_ERROR_NOMATCH) {
    m.first = m.last = data.begin();
    return false;
  } else {
    throw match_error(result);
  }
}

bool
rx::exec(const std::string& data, matches& m,
         unsigned offset, int options) const
{
  // pcre_exec() might need to store the matches for backreferences in
  // the pattern, so we always pass the ovector array of the suitable
  // length.

  int ovecsize = (capture_count + 1) * 3;
  int ovector[ovecsize];
  int result = pcre_exec(regexp, 0, data.data(), data.size(), offset, options,
                         ovector, ovecsize);

  if (result >= 0) {
    m.set(data, ovector, ovecsize);
    return true;
  } else if (result == PCRE_ERROR_NOMATCH) {
    m.clear();
    return false;
  } else {
    throw match_error(result);
  }
}

// matches

void
rx::matches::set(const std::string& data, int *ovector, int ovecsize)
{
  unsigned count = ovecsize / 3;
  const std::string::const_iterator begin = data.begin();

  array.resize(count);

  for (unsigned j = 0; j < count; j++) {
    array[j].first = begin + ovector[2 * j];
    array[j].last = begin + ovector[2 * j + 1];
  }
}

// arch-tag: f5cf98ba-3db3-4366-90c8-dd2bf23cd005
