/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#ifndef _MULTICURVE
#define _MULTICURVE

#include <string>
#include <fstream>
#include <vector>
#include "point.h"
#include "curve.h"

typedef std::vector< point<float> > points_array;

struct multi_curve {
	
	// multiple bezier curves
	//

  std::string name; // name

  float r, g, b; // color

  // curve info
  points_array vertices;
  points_array left_tangents;
  points_array right_tangents;

  std::vector<curve> curv; // bezier curves of this multi_curve.
  std::vector<int> eval; // eval status of each bezier curve

  float resolution; // resolution of all bezier curves

  multi_curve ();
  multi_curve (const std::string& filename);
  multi_curve (const multi_curve& src);
  multi_curve& operator= (const multi_curve& src);
  void copy (const multi_curve& src);

  void clear (int all = 1);
  int num_vertices ();

  void add_vertex (float x, float y);
  void add_left_tangent (float x, float y);
  void add_right_tangent (float x, float y);

  bool set_vertex (int i, float x, float y, int carry_tangents = 0);
  bool set_left_tangent (int i, float x, float y);
  bool set_right_tangent (int i, float x, float y);

  void get_vertex (int i, float& x, float& y);
  void get_left_tangent (int i, float& x, float& y);
  void get_right_tangent (int i, float& x, float& y);

  bool insert (float x, float y, float tx, float ty);
  bool remove (int i);

  void set_resolution (float d);
  float get_resolution ();

  void set_color (); // random color
  void set_color (float rr, float gg, float bb);
  void get_color (float& rr, float& gg, float& bb) { rr = r; gg = g; bb = b;}

  void force_eval ();
  void evaluate ();

  void load (const std::string& filename);
  void load (std::ifstream& file);

  void save (const std::string& filename);
  void save (std::ofstream& file);


};

void create_polyline (multi_curve& crv, const points_array& pts);
void convert2_catmull_rom (multi_curve& crv, float tangent_size);
void convert2_polyline (multi_curve& crv);
multi_curve* check_list (multi_curve** lst, int n, const std::string& name);
multi_curve* get_curve (const std::string& name);


#endif
