/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "contactlist.h"

ContactList::ContactList(PumpController *pumpController,
                         QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;

    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);


    QString webfingerHelpMessage = tr("username@server.org or https://server.org/username");

    topLayout = new QHBoxLayout();
    enterAddressLabel = new QLabel(tr("&Enter address to follow:"));
    enterAddressLabel->setToolTip(webfingerHelpMessage);

    addressLineEdit = new QLineEdit();
    addressLineEdit->setPlaceholderText(webfingerHelpMessage);
    addressLineEdit->setToolTip(webfingerHelpMessage);
    connect(addressLineEdit, SIGNAL(returnPressed()),
            this, SLOT(followContact()));

    enterAddressLabel->setBuddy(addressLineEdit);


    followButton = new QPushButton(QIcon::fromTheme("list-add-user"), tr("&Follow"));
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(followContact()));
    topLayout->addWidget(enterAddressLabel);
    topLayout->addWidget(addressLineEdit);
    topLayout->addWidget(followButton);

    mainLayout->addLayout(topLayout);
    mainLayout->addSpacing(8);


    // Widget for list of 'following'
    this->followingWidget = new QWidget();

    followingLayout = new QVBoxLayout();
    followingLayout->setAlignment(Qt::AlignTop);
    QMap<QString,QString> demoFollowingData;
    demoFollowingData.insert("name", "Demo Contact");
    demoFollowingData.insert("id",   "democontact@pumpserver.org");
    demoFollowingData.insert("avatar", "");
    demoFollowingData.insert("hometown", "Some city");
    demoFollowingData.insert("following", "true");
    followingLayout->addWidget(new ContactCard(this->pController, demoFollowingData, this));
    followingWidget->setLayout(followingLayout);

    followingScrollArea = new QScrollArea();
    followingScrollArea->setWidget(followingWidget);
    followingScrollArea->setWidgetResizable(true);


    // Widget for list of 'followers'
    this->followersWidget = new QWidget();

    followersLayout = new QVBoxLayout();
    followersLayout->setAlignment(Qt::AlignTop);
    QMap<QString,QString> demoFollowerData;
    demoFollowerData.insert("name", "Demo Follower");
    demoFollowerData.insert("id",   "demofollower@pumpserver.org");
    demoFollowerData.insert("avatar", "");
    demoFollowerData.insert("hometown", "Demo Town");
    demoFollowerData.insert("following", "false");
    followersLayout->addWidget(new ContactCard(this->pController, demoFollowerData, this));
    followersWidget->setLayout(followersLayout);

    followersScrollArea = new QScrollArea();
    followersScrollArea->setWidget(followersWidget);
    followersScrollArea->setWidgetResizable(true);


    // Widget for the list of 'lists'
    this->listsManager = new ListsManager(this->pController);
    listsScrollArea = new QScrollArea();
    listsScrollArea->setWidget(this->listsManager);
    listsScrollArea->setWidgetResizable(true);


    // Options menu
    optionsMenu = new QMenu("*options-menu*");
    optionsMenu->addAction(QIcon::fromTheme("view-refresh"),
                           tr("Reload Followers"),
                           this,
                           SLOT(refreshFollowers()));
    optionsMenu->addAction(QIcon::fromTheme("view-refresh"),
                           tr("Reload Following"),
                           this,
                           SLOT(refreshFollowing()));
    optionsMenu->addSeparator();
    optionsMenu->addAction(QIcon::fromTheme("document-export"),
                           tr("Export Followers"),
                           this,
                           SLOT(exportFollowers()));
    optionsMenu->addAction(QIcon::fromTheme("document-export"),
                           tr("Export Following"),
                           this,
                           SLOT(exportFollowing()));
    optionsMenu->addSeparator();
    optionsMenu->addAction(QIcon::fromTheme("view-refresh"),
                           tr("Reload Lists"),
                           this,
                           SLOT(refreshLists()));


    optionsButton = new QPushButton(QIcon::fromTheme("configure"),
                                    tr("Optio&ns"));
    optionsButton->setMenu(optionsMenu);


    this->tabWidget = new QTabWidget();
    tabWidget->setDocumentMode(true); // To have less margins, with so many tabs inside tabs...

    tabWidget->addTab(followersScrollArea,
                      QIcon::fromTheme("meeting-observer"),
                      "followers");
    tabWidget->addTab(followingScrollArea,
                      QIcon::fromTheme("meeting-participant"),
                      "following");
    tabWidget->addTab(listsScrollArea,
                      QIcon::fromTheme("preferences-contact-list"),
                      "lists");

    tabWidget->setCornerWidget(this->optionsButton);


    this->followersCount = 0;
    this->followingCount = 0;
    this->listsCount = 0;
    this->setTabLabels();


    mainLayout->addWidget(tabWidget);
    this->setLayout(mainLayout);

    // FIXME: this doesn't really work
    this->followButton->setFocus(); // Set focus here, away from addressLineEdit

    qDebug() << "Contact list created";
}


ContactList::~ContactList()
{
    qDebug() << "Contact list destroyed";
}



void ContactList::clearFollowingListContents()
{
    QLayoutItem *child;

    while ((child = followingLayout->takeAt(0)) != 0)
    {
        child->widget()->deleteLater(); // Delete the widget (contactCard) itself
        delete child;
    }

    this->followingListString.clear();
}

void ContactList::clearFollowersListContents()
{
    QLayoutItem *child;

    while ((child = followersLayout->takeAt(0)) != 0)
    {
        child->widget()->deleteLater();
        delete child;
    }

    this->followersListString.clear();
}


void ContactList::setTabLabels()
{
    this->tabWidget->setTabText(0, tr("Follo&wers")
                                   + QString(" (%1)").arg(this->followersCount));

    this->tabWidget->setTabText(1, tr("Followin&g")
                                   + QString(" (%1)").arg(this->followingCount));

    this->tabWidget->setTabText(2, tr("&Lists")
                                   + QString(" (%1)").arg(this->listsCount));
}


/*
 * Write the list of contacts (following or followers)
 * to a file selected by the user
 *
 */
void ContactList::exportContactsToFile(QString listType)
{
    QString dialogTitle = listType == "following" ?
                    tr("Export list of 'following' to a file") :
                    tr("Export list of 'followers' to a file");

    QString suggestedFilename = "dianara-"
                                + pController->currentUsername()
                                + "-"
                                + listType;

    QString filename = QFileDialog::getSaveFileName(this, dialogTitle,
                                                    QDir::homePath() + "/" + suggestedFilename,
                                                    "").trimmed();

    if (filename.isEmpty()) // If dialog canceled, do nothing
    {
        return;
    }


    qDebug() << "Exporting to:"  << filename;

    QFile exportFile(filename);
    exportFile.open(QIODevice::WriteOnly);

    if (listType == "following")
    {
        exportFile.write(this->followingListString.toLocal8Bit());
    }
    else // "followers"
    {
        exportFile.write(this->followersListString.toLocal8Bit());
    }
    exportFile.close();
}





/*****************************************************************************/
/*********************************** SLOTS ***********************************/
/*****************************************************************************/




void ContactList::setContactListContents(QString listType,
                                         QVariantList contactList,
                                         int totalReceivedCount)
{
    qDebug() << "Setting contact list contents";

    if (listType == "following")
    {
        if (totalReceivedCount == 200) // Only for the first batch
        {
            this->clearFollowingListContents();
            followingCount = 0;
        }
        this->followingCount += contactList.size();
    }
    else
    {
        if (totalReceivedCount == 200)
        {
            this->clearFollowersListContents();
            followersCount = 0;
        }
        this->followersCount += contactList.size();
    }


    QMap<QString,QString> contactData;
    QString contactInfoLineString;
    QStringList followingIdStringList;

    foreach (QVariant contact, contactList)
    {
        QVariantMap contactMap = contact.toMap();

        contactData.insert("id", contactMap.value("id").toString().remove(0,5)); // remove "acct:"
        contactData.insert("name", contactMap.value("displayName").toString());
        contactData.insert("avatar", contactMap.value("image").toMap().value("url").toString());
        contactData.insert("hometown", contactMap.value("location").toMap().value("displayName").toString());
        contactData.insert("bio", contactMap.value("summary").toString());

        contactData.insert("url", contactMap.value("url").toString());

        contactData.insert("following", contactMap.value("pump_io").toMap().value("followed").toString());

        contactInfoLineString = contactData.value("name")
                              + "  <"
                              + contactData.value("id")
                              + ">\n";

        if (listType == "following")
        {
            this->followingLayout->addWidget(new ContactCard(this->pController,
                                                            contactData,
                                                            this));
            followingListString.append(contactInfoLineString);

            followingIdStringList.append(contactData.value("id"));
        }
        else // == "followers"
        {
            this->followersLayout->addWidget(new ContactCard(this->pController,
                                                            contactData,
                                                            this));
            followersListString.append(contactInfoLineString);
        }
    }

    if (listType == "following")
    {
        this->pController->updateInternalFollowingIdList(followingIdStringList);

        if (totalReceivedCount < pController->currentFollowingCount())
        {
            pController->getContactList(listType, totalReceivedCount);
        }
    }
    else  // followers
    {
        if (totalReceivedCount < pController->currentFollowersCount())
        {
            pController->getContactList(listType, totalReceivedCount);
        }
    }


    // Update tab labels with number of following or followers, which were updated before
    this->setTabLabels();
}



/*
 * Fill the list of lists
 *
 */
void ContactList::setListsListContents(QVariantList listsList)
{
    this->listsCount = listsList.count();
    // Update tab labels with number of following or followers, which were updated before
    this->setTabLabels();


    this->listsManager->setListsList(listsList);
}



/*
 * Ask for the updated list of Following
 *
 */
void ContactList::refreshFollowing()
{
    qDebug() << "Refreshing list of Following...";
    this->pController->getContactList("following");
}


/*
 * Ask for the updated list of followers
 *
 */
void ContactList::refreshFollowers()
{
    qDebug() << "Refreshing list of Followers...";
    this->pController->getContactList("followers");
}


/*
 * Export list of "Following" to a text file
 *
 */
void ContactList::exportFollowing()
{
    qDebug() << "Exporting Following...";
    exportContactsToFile("following");
}


/*
 * Export list of "Followers" to a text file
 *
 */
void ContactList::exportFollowers()
{
    qDebug() << "Exporting Followers...";
    exportContactsToFile("followers");
}


void ContactList::refreshLists()
{
    qDebug() << "Refreshing list of lists...";
    this->pController->getListsList();
}




/*
 * Add the address entered by the user to the /following list.
 *
 * This supports adding webfinger addresses in the form
 * user@hostname or https://host/username.
 *
 */
void ContactList::followContact()
{
    QString address = this->addressLineEdit->text().trimmed();
    bool validAddress = false;

    qDebug() << "ContactList::followContact(); Address entered:" << address;

    // FIXME, check correctly!!! ////////////
    if (address.contains("@"))
    {
        validAddress = true;
    }
    else if (address.startsWith("https://") || address.startsWith("http://"))
    {
        address.remove("https://");
        address.remove("http://");

        if (address.contains("/")) // Very basic sanity check
        {
            QStringList addressParts = address.split("/");
            address = addressParts.at(1) + "@" + addressParts.at(0);

            validAddress = true;
        }
        else
        {
            qDebug() << "Invalid webfinger address!";
        }
    }


    if (validAddress)
    {
        qDebug() << "About to follow this address:" << address;
        this->pController->followContact(address); // this will trigger a reload of the contact list

        this->addressLineEdit->clear();
    }

}
