#include <check.h>
#include <stdio.h>
#include <stdlib.h>
#include "test.h"
#include "builders/build_file.h"
#include "../src/alloc.h"
#include "../src/conf.h"
#include "../src/conffile.h"
#include "../src/fsops.h"
#include "../src/pathcmp.h"

#define BASE		"utest_conffile"
#define CONFFILE	BASE "/burp.conf"

static struct conf **setup_conf(void)
{
	struct conf **confs=NULL;
	fail_unless((confs=confs_alloc())!=NULL);
	fail_unless(!confs_init(confs));
	return confs;
}

static void setup(struct conf ***globalcs, struct conf ***cconfs)
{
	fail_unless(recursive_delete(BASE)==0);
	alloc_counters_reset();
	if(globalcs) *globalcs=setup_conf();
	if(cconfs) *cconfs=setup_conf();
}

static void tear_down(struct conf ***globalcs, struct conf ***confs)
{
	fail_unless(recursive_delete(BASE)==0);
	confs_free(confs);
	confs_free(globalcs);
	alloc_check();
}

struct data
{
        const char *str;
	const char *field;
	const char *value;
};

static struct data d[] = {
	{ "a=b", "a", "b" },
	{ "a=b\n", "a", "b" },
	{ "a = b", "a", "b" },
	{ "   a  =    b ", "a", "b" },
	{ "   a  =    b \n", "a", "b" },
	{ "#a=b", NULL, NULL },
	{ "  #a=b", NULL, NULL },
	{ "a='b'", "a", "b" },
	{ "a='b", "a", "b" },
	{ "a=b'", "a", "b'" },
	{ "a=\"b\"", "a", "b" },
	{ "a=b\"", "a", "b\"" },
	{ "a=\"b", "a", "b" },
	{ "a=b # comment", "a", "b # comment" }, // Maybe fix this.
	{ "field=longvalue with spaces", "field", "longvalue with spaces" },
};

START_TEST(test_conf_get_pair)
{
        FOREACH(d)
	{
		char *field=NULL;
		char *value=NULL;
		char *str=strdup(d[i].str);
		conf_get_pair(str, &field, &value);
		if(!field || !d[i].field)
			fail_unless(field==d[i].field);
		else
			fail_unless(!strcmp(field, d[i].field));
		if(!value || !d[i].value)
			fail_unless(value==d[i].value);
		else
			fail_unless(!strcmp(value, d[i].value));
		free(str);
	}
}
END_TEST

START_TEST(test_client_conf)
{
	struct conf **confs=NULL;
	setup(&confs, NULL);
	build_file(CONFFILE, MIN_CLIENT_CONF);
	fail_unless(!conf_load_global_only(CONFFILE, confs));
	fail_unless(get_e_burp_mode(confs[OPT_BURP_MODE])==BURP_MODE_CLIENT);
	ck_assert_str_eq(get_string(confs[OPT_SERVER]), "4.5.6.7");
	ck_assert_str_eq(get_string(confs[OPT_PORT]), "1234");
	ck_assert_str_eq(get_string(confs[OPT_STATUS_PORT]), "12345");
	ck_assert_str_eq(get_string(confs[OPT_LOCKFILE]), "/lockfile/path");
	ck_assert_str_eq(get_string(confs[OPT_SSL_CERT]), "/ssl/cert/path");
	ck_assert_str_eq(get_string(confs[OPT_SSL_CERT_CA]), "/cert_ca/path");
	ck_assert_str_eq(get_string(confs[OPT_SSL_PEER_CN]), "my_cn");
	ck_assert_str_eq(get_string(confs[OPT_SSL_KEY]), "/ssl/key/path");
	ck_assert_str_eq(get_string(confs[OPT_CA_CSR_DIR]), "/csr/dir");
	tear_down(NULL, &confs);
}
END_TEST

static void assert_strlist(struct strlist **s, const char *path, int flag)
{
	if(!path)
	{
		fail_unless(*s==NULL);
		return;
	}
	ck_assert_str_eq((*s)->path, path);
	fail_unless((*s)->flag==flag);
	*s=(*s)->next;
}

static void assert_include(struct strlist **s, const char *path)
{
	assert_strlist(s, path, 1);
}

static void assert_exclude(struct strlist **s, const char *path)
{
	assert_strlist(s, path, 0);
}

START_TEST(test_client_includes_excludes)
{
	const char *buf=MIN_CLIENT_CONF
		"exclude=/z\n"
		"exclude=/a/b\n"
		"include=/a/b/c\n"
		"include=/x/y/z\n"
		"include=/r/s/t\n"
		"include=/a\n"
		"include=/a/b/c/d\n"
		"cross_filesystem=/mnt/x\n"
	;
	struct strlist *s;
	struct conf **confs=NULL;
	setup(&confs, NULL);
	build_file(CONFFILE, buf);
	fail_unless(!conf_load_global_only(CONFFILE, confs));
	s=get_strlist(confs[OPT_INCLUDE]);
	assert_include(&s, "/a");
	assert_include(&s, "/a/b/c");
	assert_include(&s, "/a/b/c/d");
	assert_include(&s, "/r/s/t");
	assert_include(&s, "/x/y/z");
	assert_include(&s, NULL);
	s=get_strlist(confs[OPT_EXCLUDE]);
	assert_exclude(&s, "/a/b");
	assert_exclude(&s, "/z");
	assert_exclude(&s, NULL);
	s=get_strlist(confs[OPT_STARTDIR]);
	assert_include(&s, "/a");
	assert_include(&s, "/r/s/t");
	assert_include(&s, "/x/y/z");
	assert_include(&s, NULL);
	s=get_strlist(confs[OPT_INCEXCDIR]);
	assert_include(&s, "/a");
	assert_exclude(&s, "/a/b");
	assert_include(&s, "/a/b/c");
	assert_include(&s, "/a/b/c/d");
	assert_include(&s, "/r/s/t");
	assert_include(&s, "/x/y/z");
	assert_exclude(&s, "/z");
	assert_include(&s, NULL);
	s=get_strlist(confs[OPT_FSCHGDIR]);
	assert_strlist(&s, "/a/b/c", 0);
	assert_strlist(&s, "/a/b/c/d", 0);
	assert_strlist(&s, "/mnt/x", 0);
	assert_strlist(&s, NULL, 0);
	tear_down(NULL, &confs);
}
END_TEST

static const char *include_failures[] = {
	MIN_CLIENT_CONF "include=not_absolute\n",
	MIN_CLIENT_CONF "include=/\ninclude=/\n"
};

START_TEST(test_client_include_failures)
{
	struct conf **confs=NULL;
	FOREACH(include_failures)
	{
		setup(&confs, NULL);
		build_file(CONFFILE, include_failures[i]);
		fail_unless(conf_load_global_only(CONFFILE, confs)==-1);
		tear_down(NULL, &confs);
	}
}
END_TEST

START_TEST(test_client_include_glob)
{
	char cwd[1024];
	char buf[4096];
	char path1[2048];
	char path2[2048];
	struct strlist *s;
	struct conf **confs=NULL;
	fail_unless(getcwd(cwd, sizeof(cwd))!=NULL);
	snprintf(buf, sizeof(buf),
		"%sinclude_glob=%s/%s/*.glob\ninclude=/1\n",
		MIN_CLIENT_CONF, cwd, BASE);
	snprintf(path1, sizeof(path1), "%s/%s/a.glob", cwd, BASE);
	snprintf(path2, sizeof(path2), "%s/%s/b.glob", cwd, BASE);
	setup(&confs, NULL);
	build_file(CONFFILE, buf);

	fail_unless(!conf_load_global_only(CONFFILE, confs));
	s=get_strlist(confs[OPT_INCLUDE]);
	assert_include(&s, "/1");
	assert_include(&s, NULL);

	build_file(path1, "a");
	fail_unless(!reeval_glob(confs));
	s=get_strlist(confs[OPT_INCLUDE]);
	assert_include(&s, "/1");
	assert_include(&s, path1);
	assert_include(&s, NULL);

	build_file(path2, "b");
	fail_unless(!reeval_glob(confs));
	s=get_strlist(confs[OPT_INCLUDE]);
	assert_include(&s, "/1");
	assert_include(&s, path1);
	assert_include(&s, path2);
	assert_include(&s, NULL);

	tear_down(NULL, &confs);
}
END_TEST

START_TEST(test_server_conf)
{
	struct strlist *s;
	struct conf **confs=NULL;
	setup(&confs, NULL);
	build_file(CONFFILE, MIN_SERVER_CONF);
	fail_unless(!conf_load_global_only(CONFFILE, confs));
	fail_unless(get_e_burp_mode(confs[OPT_BURP_MODE])==BURP_MODE_SERVER);
	ck_assert_str_eq(get_string(confs[OPT_PORT]), "1234");
	ck_assert_str_eq(get_string(confs[OPT_STATUS_PORT]), "12345");
	ck_assert_str_eq(get_string(confs[OPT_LOCKFILE]), "/lockfile/path");
	ck_assert_str_eq(get_string(confs[OPT_SSL_CERT]), "/ssl/cert/path");
	ck_assert_str_eq(get_string(confs[OPT_SSL_CERT_CA]), "/cert_ca/path");
	ck_assert_str_eq(get_string(confs[OPT_DIRECTORY]), "/a/directory");
	ck_assert_str_eq(get_string(confs[OPT_DEDUP_GROUP]), "a_group");
	ck_assert_str_eq(get_string(confs[OPT_CLIENTCONFDIR]), "clientconfdir");
	ck_assert_str_eq(get_string(confs[OPT_SSL_DHFILE]), "/a/dhfile");
	s=get_strlist(confs[OPT_KEEP]);
	assert_strlist(&s, "10", 10);
	assert_include(&s, NULL);
	tear_down(NULL, &confs);
}
END_TEST

static void pre_post_assertions(struct conf **confs, const char *pre_path,
	const char *post_path, const char *pre_arg1, const char *pre_arg2,
	const char *post_arg1, const char *post_arg2,
	enum conf_opt o_script_pre, enum conf_opt o_script_post,
	enum conf_opt o_script_pre_arg, enum conf_opt o_script_post_arg,
	enum conf_opt o_script_pre_notify, enum conf_opt o_script_post_notify,
	enum conf_opt o_script_post_run_on_fail)
{
	struct strlist *s;
	ck_assert_str_eq(get_string(confs[o_script_pre]), pre_path);
	ck_assert_str_eq(get_string(confs[o_script_post]), post_path);
	s=get_strlist(confs[o_script_pre_arg]);
	assert_strlist(&s, pre_arg1, 0);
	assert_strlist(&s, pre_arg2, 0);
	assert_strlist(&s, NULL, 0);
	if(o_script_pre_notify!=OPT_MAX)
		fail_unless(get_int(confs[o_script_pre_notify])==1);
	s=get_strlist(confs[o_script_post_arg]);
	assert_strlist(&s, post_arg1, 0);
	assert_strlist(&s, post_arg2, 0);
	assert_strlist(&s, NULL, 0);
	if(o_script_post_notify!=OPT_MAX)
		fail_unless(get_int(confs[o_script_post_notify])==1);
	fail_unless(get_int(confs[o_script_post_run_on_fail])==1);
}

static void pre_post_checks(const char *buf, const char *pre_path,
	const char *post_path, const char *pre_arg1, const char *pre_arg2,
	const char *post_arg1, const char *post_arg2,
	enum conf_opt o_script_pre, enum conf_opt o_script_post,
	enum conf_opt o_script_pre_arg, enum conf_opt o_script_post_arg,
	enum conf_opt o_script_pre_notify, enum conf_opt o_script_post_notify,
	enum conf_opt o_script_post_run_on_fail)
{
	struct conf **confs=NULL;
	setup(&confs, NULL);
	build_file(CONFFILE, buf);
	fail_unless(!conf_load_global_only(CONFFILE, confs));
	pre_post_assertions(confs, pre_path, post_path, pre_arg1, pre_arg2,
		post_arg1, post_arg2,
		o_script_pre, o_script_post,
		o_script_pre_arg, o_script_post_arg,
		o_script_pre_notify, o_script_post_notify,
		o_script_post_run_on_fail);
	tear_down(NULL, &confs);
}

static void server_pre_post_checks(const char *buf, const char *pre_path,
	const char *post_path, const char *pre_arg1, const char *pre_arg2,
	const char *post_arg1, const char *post_arg2)
{
	pre_post_checks(buf, pre_path, post_path, pre_arg1, pre_arg2,
		post_arg1, post_arg2,
		OPT_S_SCRIPT_PRE, OPT_S_SCRIPT_POST,
		OPT_S_SCRIPT_PRE_ARG, OPT_S_SCRIPT_POST_ARG,
		OPT_S_SCRIPT_PRE_NOTIFY, OPT_S_SCRIPT_POST_NOTIFY,
		OPT_S_SCRIPT_POST_RUN_ON_FAIL);
}

#define SERVER_SCRIPT_PRE_POST 			\
	"server_script_pre=pre_path\n"		\
	"server_script_pre_arg=pre_arg1\n"	\
	"server_script_pre_arg=pre_arg2\n"	\
	"server_script_pre_notify=1\n"		\
	"server_script_post=post_path\n"	\
	"server_script_post_arg=post_arg1\n"	\
	"server_script_post_arg=post_arg2\n"	\
	"server_script_post_notify=1\n"		\
	"server_script_post_run_on_fail=1\n"	\

START_TEST(test_server_script_pre_post)
{
	const char *buf=MIN_SERVER_CONF SERVER_SCRIPT_PRE_POST;
	server_pre_post_checks(buf, "pre_path", "post_path", "pre_arg1",
		"pre_arg2", "post_arg1", "post_arg2");
}
END_TEST

#define SERVER_SCRIPT_CONF			\
	"server_script=path\n"			\
	"server_script_arg=arg1\n"		\
	"server_script_arg=arg2\n"		\
	"server_script_notify=1\n"		\
	"server_script_notify=1\n"		\
	"server_script_run_on_fail=1\n"		\
	"server_script_post_run_on_fail=1\n"	\

// Same as test_server_script_pre_post, but use server_script to set both pre
// and post at the same time.
START_TEST(test_server_script)
{
	const char *buf=MIN_SERVER_CONF SERVER_SCRIPT_CONF;
	server_pre_post_checks(buf, "path", "path", "arg1",
		"arg2", "arg1", "arg2");
}
END_TEST

static void backup_script_pre_post_checks(const char *buf, const char *pre_path,
	const char *post_path, const char *pre_arg1, const char *pre_arg2,
	const char *post_arg1, const char *post_arg2)
{
	pre_post_checks(buf, pre_path, post_path, pre_arg1, pre_arg2,
		post_arg1, post_arg2,
		OPT_B_SCRIPT_PRE, OPT_B_SCRIPT_POST,
		OPT_B_SCRIPT_PRE_ARG, OPT_B_SCRIPT_POST_ARG,
		OPT_MAX, OPT_MAX, OPT_B_SCRIPT_POST_RUN_ON_FAIL);
}

START_TEST(test_backup_script_pre_post)
{
	const char *buf=MIN_CLIENT_CONF
		"backup_script_pre=pre_path\n"
		"backup_script_pre_arg=pre_arg1\n"
		"backup_script_pre_arg=pre_arg2\n"
		"backup_script_post=post_path\n"
		"backup_script_post_arg=post_arg1\n"
		"backup_script_post_arg=post_arg2\n"
		"backup_script_post_run_on_fail=1\n"
	;
	backup_script_pre_post_checks(buf, "pre_path", "post_path", "pre_arg1",
		"pre_arg2", "post_arg1", "post_arg2");
}
END_TEST

// Same as test_backup_script_pre_post, but use backup_script to set both pre
// and post at the same time.
START_TEST(test_backup_script)
{
	const char *buf=MIN_CLIENT_CONF
		"backup_script=path\n"
		"backup_script_arg=arg1\n"
		"backup_script_arg=arg2\n"
		"backup_script_run_on_fail=1\n"
		"backup_script_post_run_on_fail=1\n"
	;
	backup_script_pre_post_checks(buf, "path", "path", "arg1",
		"arg2", "arg1", "arg2");
}
END_TEST

static void restore_script_pre_post_checks(const char *buf,
	const char *pre_path, const char *post_path,
	const char *pre_arg1, const char *pre_arg2,
	const char *post_arg1, const char *post_arg2)
{
	pre_post_checks(buf, pre_path, post_path, pre_arg1, pre_arg2,
		post_arg1, post_arg2,
		OPT_R_SCRIPT_PRE, OPT_R_SCRIPT_POST,
		OPT_R_SCRIPT_PRE_ARG, OPT_R_SCRIPT_POST_ARG,
		OPT_MAX, OPT_MAX, OPT_R_SCRIPT_POST_RUN_ON_FAIL);
}

START_TEST(test_restore_script_pre_post)
{
	const char *buf=MIN_CLIENT_CONF
		"restore_script_pre=pre_path\n"
		"restore_script_pre_arg=pre_arg1\n"
		"restore_script_pre_arg=pre_arg2\n"
		"restore_script_post=post_path\n"
		"restore_script_post_arg=post_arg1\n"
		"restore_script_post_arg=post_arg2\n"
		"restore_script_post_run_on_fail=1\n"
	;
	restore_script_pre_post_checks(buf, "pre_path", "post_path", "pre_arg1",
		"pre_arg2", "post_arg1", "post_arg2");
}
END_TEST

// Same as test_restore_script_pre_post, but use restore_script to set both pre
// and post at the same time.
START_TEST(test_restore_script)
{
	const char *buf=MIN_CLIENT_CONF
		"restore_script=path\n"
		"restore_script_arg=arg1\n"
		"restore_script_arg=arg2\n"
		"restore_script_run_on_fail=1\n"
		"restore_script_post_run_on_fail=1\n"
	;
	restore_script_pre_post_checks(buf, "path", "path", "arg1",
		"arg2", "arg1", "arg2");
}
END_TEST

static void clientconfdir_setup(struct conf ***globalcs, struct conf ***cconfs,
	const char *gbuf, const char *buf)
{
	const char *global_path=BASE "/burp-server.conf";
	setup(globalcs, cconfs);
	build_file(CONFFILE, gbuf);
	fail_unless(!conf_load_global_only(CONFFILE, *globalcs));
	set_string((*cconfs)[OPT_CNAME], "utestclient");
	build_file(global_path, buf);
	fail_unless(!conf_load_overrides(*globalcs, *cconfs, global_path));
	ck_assert_str_eq(get_string((*cconfs)[OPT_CNAME]), "utestclient");
}

#define NOTIFY_CONF				\
	"notify_success_script=/success_path\n"	\
	"notify_success_arg=/success/arg1\n"	\
	"notify_success_arg=/success/arg2\n"	\
	"notify_success_warnings_only=1\n"	\
	"notify_success_changes_only=1\n"	\
	"notify_failure_script=/failure_path\n"	\
	"notify_failure_arg=/failure/arg1\n"	\
	"notify_failure_arg=/failure/arg2\n"	\

static void notify_assertions(struct conf **cconfs)
{
	struct strlist *s;
	ck_assert_str_eq(get_string(cconfs[OPT_N_SUCCESS_SCRIPT]),
		"/success_path");
	s=get_strlist(cconfs[OPT_N_SUCCESS_ARG]);
	assert_strlist(&s, "/success/arg1", 0);
	assert_strlist(&s, "/success/arg2", 0);
	assert_include(&s, NULL);
	ck_assert_str_eq(get_string(cconfs[OPT_N_FAILURE_SCRIPT]),
		"/failure_path");
	fail_unless(get_int(cconfs[OPT_N_SUCCESS_WARNINGS_ONLY])==1);
	fail_unless(get_int(cconfs[OPT_N_SUCCESS_CHANGES_ONLY])==1);
	s=get_strlist(cconfs[OPT_N_FAILURE_ARG]);
	assert_strlist(&s, "/failure/arg1", 0);
	assert_strlist(&s, "/failure/arg2", 0);
	assert_include(&s, NULL);
}

#define MIN_CLIENTCONFDIR_BUF "# comment\n"

START_TEST(test_clientconfdir_conf)
{
	struct strlist *s;
	struct conf **globalcs=NULL;
	struct conf **cconfs=NULL;
	const char *gbuf=MIN_SERVER_CONF
		"restore_client=abc\n"
		"restore_client=xyz\n"
		"timer_script=/timer/script\n"
		"timer_arg=/timer/arg1\n"
		"timer_arg=/timer/arg2\n"
		"label=qwe\n"
		NOTIFY_CONF
	;
	const char *buf=MIN_CLIENTCONFDIR_BUF
		"protocol=1\n"
		"directory=/another/dir\n"
		"directory_tree=0\n"
		"timestamp_format=%H %M %S\n"
		"password_check=0\n"
		"keep=4\n"
		"keep=7\n"
		"working_dir_recovery_method=resume\n"
		"librsync=0\n"
		"version_warn=0\n"
		"path_length_warn=0\n"
		"syslog=1\n"
		"client_can_delete=0\n"
		"client_can_force_backup=0\n"
		"client_can_list=0\n"
		"client_can_restore=0\n"
		"client_can_verify=0\n"
		"restore_client=123\n"
		"restore_client=456\n"
		"dedup_group=dd_group\n"
		"label=rty\n"
		"label=xyz\n"
	;

	clientconfdir_setup(&globalcs, &cconfs, gbuf, buf);

	fail_unless(get_e_protocol(cconfs[OPT_PROTOCOL])==PROTO_1);
	ck_assert_str_eq(get_string(cconfs[OPT_DIRECTORY]), "/another/dir");
	fail_unless(get_int(cconfs[OPT_DIRECTORY_TREE])==0);
	ck_assert_str_eq(get_string(cconfs[OPT_TIMESTAMP_FORMAT]),
		"%H %M %S");
	fail_unless(get_int(cconfs[OPT_PASSWORD_CHECK])==0);
	s=get_strlist(cconfs[OPT_KEEP]);
	assert_strlist(&s, "4", 4);
	assert_strlist(&s, "7", 8); // The last one gets 1 added to it.
	assert_include(&s, NULL);
	s=get_strlist(cconfs[OPT_RESTORE_CLIENTS]);
	assert_strlist(&s, "123", 0);
	assert_strlist(&s, "456", 0);
	assert_strlist(&s, "abc", 0);
	assert_strlist(&s, "xyz", 0);
	assert_include(&s, NULL);
	fail_unless(get_e_recovery_method(
	  cconfs[OPT_WORKING_DIR_RECOVERY_METHOD])==RECOVERY_METHOD_RESUME);
	fail_unless(get_int(cconfs[OPT_LIBRSYNC])==0);
	fail_unless(get_int(cconfs[OPT_VERSION_WARN])==0);
	fail_unless(get_int(cconfs[OPT_PATH_LENGTH_WARN])==0);
	fail_unless(get_int(cconfs[OPT_SYSLOG])==1);
	fail_unless(get_int(cconfs[OPT_CLIENT_CAN_DELETE])==0);
	fail_unless(get_int(cconfs[OPT_CLIENT_CAN_FORCE_BACKUP])==0);
	fail_unless(get_int(cconfs[OPT_CLIENT_CAN_LIST])==0);
	fail_unless(get_int(cconfs[OPT_CLIENT_CAN_RESTORE])==0);
	fail_unless(get_int(cconfs[OPT_CLIENT_CAN_VERIFY])==0);
	s=get_strlist(cconfs[OPT_TIMER_ARG]);
	assert_strlist(&s, "/timer/arg1", 0);
	assert_strlist(&s, "/timer/arg2", 0);
	assert_include(&s, NULL);
	ck_assert_str_eq(get_string(cconfs[OPT_DEDUP_GROUP]), "dd_group");
	s=get_strlist(cconfs[OPT_LABEL]);
	assert_strlist(&s, "rty", 0);
	assert_strlist(&s, "xyz", 0);
	notify_assertions(cconfs);
	tear_down(&globalcs, &cconfs);
}
END_TEST

START_TEST(test_clientconfdir_extra)
{
	struct strlist *s;
	struct conf **globalcs=NULL;
	struct conf **cconfs=NULL;
	const char *gbuf=MIN_SERVER_CONF
		"restore_client=abc\n"
		"include = /ignored/include\n"
		"timer_script = /ignored/timer\n"
		"timer_arg = /ignored/timer/arg1\n"
		"timer_arg = /ignored/timer/arg2\n"
		"notify_success_script = /ignored/success\n"
		"notify_success_arg = /ignored/success/arg\n"
		"notify_failure_script = /ignored/failure\n"
		"notify_failure_arg = /ignored/failure/arg\n"
		SERVER_SCRIPT_CONF
	;
	const char *buf=MIN_CLIENTCONFDIR_BUF
		"include = /testdir\n"
		"timer_script = /timer/script\n"
		"timer_arg = /timer/arg1\n"
		"timer_arg = /timer/arg2\n"
		SERVER_SCRIPT_PRE_POST
		NOTIFY_CONF
	;
	clientconfdir_setup(&globalcs, &cconfs, gbuf, buf);
	s=get_strlist(cconfs[OPT_RESTORE_CLIENTS]);
	assert_strlist(&s, "abc", 0);
	assert_include(&s, NULL);
	s=get_strlist(cconfs[OPT_INCLUDE]);
	assert_strlist(&s, "/testdir", 1);
	assert_include(&s, NULL);
	ck_assert_str_eq(get_string(cconfs[OPT_TIMER_SCRIPT]), "/timer/script");
	s=get_strlist(cconfs[OPT_TIMER_ARG]);
	assert_strlist(&s, "/timer/arg1", 0);
	assert_strlist(&s, "/timer/arg2", 0);
	assert_include(&s, NULL);
	pre_post_assertions(cconfs, "pre_path", "post_path",
		"pre_arg1", "pre_arg2",
		"post_arg1", "post_arg2",
		OPT_S_SCRIPT_PRE, OPT_S_SCRIPT_POST,
		OPT_S_SCRIPT_PRE_ARG, OPT_S_SCRIPT_POST_ARG,
		OPT_S_SCRIPT_PRE_NOTIFY, OPT_S_SCRIPT_POST_NOTIFY,
		OPT_S_SCRIPT_POST_RUN_ON_FAIL);
	notify_assertions(cconfs);
	tear_down(&globalcs, &cconfs);
}
END_TEST

START_TEST(test_clientconfdir_server_script)
{
	struct conf **globalcs=NULL;
	struct conf **cconfs=NULL;
	const char *gbuf=MIN_SERVER_CONF SERVER_SCRIPT_PRE_POST;
	const char *buf=MIN_CLIENTCONFDIR_BUF SERVER_SCRIPT_CONF;

	clientconfdir_setup(&globalcs, &cconfs, gbuf, buf);
	pre_post_assertions(cconfs, "path", "path",
		"arg1", "arg2",
		"arg1", "arg2",
		OPT_S_SCRIPT_PRE, OPT_S_SCRIPT_POST,
		OPT_S_SCRIPT_PRE_ARG, OPT_S_SCRIPT_POST_ARG,
		OPT_S_SCRIPT_PRE_NOTIFY, OPT_S_SCRIPT_POST_NOTIFY,
		OPT_S_SCRIPT_POST_RUN_ON_FAIL);
	tear_down(&globalcs, &cconfs);
}
END_TEST

static void switch_test(int expected_ret, const char *gbuf)
{
	char orig_client_conf[256]="";
	const char *clientconfdir;
	struct conf **globalcs=NULL;
	struct conf **cconfs=NULL;
	const char *buf=MIN_CLIENTCONFDIR_BUF;
	const char *orig_client_buf=MIN_CLIENTCONFDIR_BUF;

	clientconfdir_setup(&globalcs, &cconfs, gbuf, buf);
	clientconfdir=get_string(globalcs[OPT_CLIENTCONFDIR]);
	fail_unless(!recursive_delete(clientconfdir));
	snprintf(orig_client_conf, sizeof(orig_client_conf),
		"%s/orig_client", clientconfdir);
	build_file(orig_client_conf, orig_client_buf);
	fail_unless(conf_switch_to_orig_client(globalcs, cconfs,
		"orig_client")==expected_ret);
	if(!expected_ret)
	{
		ck_assert_str_eq(get_string(cconfs[OPT_CNAME]),
			"orig_client");
		ck_assert_str_eq(get_string(cconfs[OPT_RESTORE_CLIENT]),
			"orig_client");
		ck_assert_str_eq(get_string(cconfs[OPT_ORIG_CLIENT]),
			"orig_client");
	}
	fail_unless(!recursive_delete(clientconfdir));
	tear_down(&globalcs, &cconfs);
}

START_TEST(test_conf_switch_to_orig_client_fail)
{
	const char *gbuf=MIN_SERVER_CONF
		"restore_client=non-matching1\n"
		"restore_client=non-matching2\n";
	switch_test(-1, gbuf);
}
END_TEST

START_TEST(test_conf_switch_to_orig_client_ok)
{
	const char *gbuf=MIN_SERVER_CONF
		"restore_client=non-matching1\n"
		"restore_client=utestclient\n";
	switch_test(0, gbuf);
}
END_TEST


Suite *suite_conffile(void)
{
	Suite *s;
	TCase *tc_core;

	s=suite_create("conffile");

	tc_core=tcase_create("Core");

	tcase_add_test(tc_core, test_conf_get_pair);
	tcase_add_test(tc_core, test_client_conf);
	tcase_add_test(tc_core, test_client_includes_excludes);
	tcase_add_test(tc_core, test_client_include_failures);
	tcase_add_test(tc_core, test_client_include_glob);
	tcase_add_test(tc_core, test_server_conf);
	tcase_add_test(tc_core, test_server_script_pre_post);
	tcase_add_test(tc_core, test_server_script);
	tcase_add_test(tc_core, test_backup_script_pre_post);
	tcase_add_test(tc_core, test_backup_script);
	tcase_add_test(tc_core, test_restore_script_pre_post);
	tcase_add_test(tc_core, test_restore_script);
	tcase_add_test(tc_core, test_clientconfdir_conf);
	tcase_add_test(tc_core, test_clientconfdir_extra);
	tcase_add_test(tc_core, test_clientconfdir_server_script);
	tcase_add_test(tc_core, test_conf_switch_to_orig_client_fail);
	tcase_add_test(tc_core, test_conf_switch_to_orig_client_ok);

	suite_add_tcase(s, tc_core);

	return s;
}
