//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/SampleEditorCommands.cpp
//! @brief     Implements command classes for LayerOrientedSampleEditor
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/SampleEditorCommands.h"
#include "GUI/Model/Sample/LayerItem.h"
#include "GUI/Model/Sample/SampleItem.h"
#include "GUI/Support/XML/Backup.h"
#include "GUI/View/SampleDesigner/SampleEditorController.h"
#include "GUI/View/SampleDesigner/SampleForm.h"
#include <utility>

namespace {

constexpr int COMMAND_ID_CHANGE_VALUE = 11;

} // namespace


CommandRemoveLayer::CommandRemoveLayer(SampleEditorController* ec, LayerItem* layerItem,
                                       QUndoCommand* parent /*= nullptr*/)
    : QUndoCommand(parent)
    , m_ec(ec)
{
    setText("Remove layer");
    m_indexOfLayer = ec->sampleItem()->layerItems().indexOf(layerItem);
    m_layerItemBackup = GUI::Util::createBackup(layerItem);
}

void CommandRemoveLayer::redo()
{
    m_ec->removeLayerItemFromUndo(m_indexOfLayer);
}

void CommandRemoveLayer::undo()
{
    LayerItem* restoredLayer = m_ec->sampleItem()->createLayerItemAt(m_indexOfLayer);
    GUI::Util::restoreBackup(restoredLayer, m_layerItemBackup);
    m_ec->sampleForm()->onLayerAdded(restoredLayer);
    emit m_ec->modified();
}

// --------------------------------------------------------------------------------------------- //

CommandAddLayer::CommandAddLayer(SampleEditorController* ec, int atIndex, QUndoCommand* /*parent*/)
    : m_ec(ec)
    , m_atIndex(atIndex)
{
    setText("Add layer");
}

void CommandAddLayer::redo()
{
    m_ec->addLayerItemFromUndo(m_atIndex);
}

void CommandAddLayer::undo()
{
    // no backup of the layer has to be stored, since redo always creates the layer
    // from scratch - no contents required for this
    m_ec->removeLayerItemFromUndo(m_atIndex);
}

// --------------------------------------------------------------------------------------------- //

CommandChangeValue::CommandChangeValue(const QString& label, SampleEditorController* ec,
                                       double oldValue, double newValue, const QString& path,
                                       QUndoCommand* parent /*= nullptr*/)
    : QUndoCommand(parent)
    , m_ec(ec)
    , m_oldValue(oldValue)
    , m_newValue(newValue)
    , m_path(path)
{
    setText("change " + label + "\n");
}

int CommandChangeValue::id() const
{
    return COMMAND_ID_CHANGE_VALUE;
}

bool CommandChangeValue::mergeWith(const QUndoCommand* command)
{
    if (command->id() != id()) // make sure other is also a changeValue command
        return false;

    const auto* const other = dynamic_cast<const CommandChangeValue*>(command);

    if (m_path != other->m_path)
        return false;

    m_newValue = other->m_newValue;
    return true;
}

void CommandChangeValue::redo()
{
    if (m_isFirst)
        m_isFirst = false;
    else
        m_ec->setDoubleFromUndo(m_newValue, m_path);
}

void CommandChangeValue::undo()
{
    m_ec->setDoubleFromUndo(m_oldValue, m_path);
}
