//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Projection/ProjectionsWidget.cpp
//! @brief     Implements class ProjectionsWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Projection/ProjectionsWidget.h"
#include "GUI/Model/Data/IntensityDataItem.h"
#include "GUI/Model/Data/MaskItems.h"
#include "GUI/View/Projection/ProjectionsPlot.h"
#include <QTabWidget>
#include <QVBoxLayout>

namespace {

const int horizontal_projection_tab = 0;
const int vertical_projection_tab = 1;

} // namespace

ProjectionsWidget::ProjectionsWidget(QWidget* parent)
    : QWidget(parent)
    , m_xProjection(new ProjectionsPlot(GUI::ID::ProjectionType::Horizontal))
    , m_yProjection(new ProjectionsPlot(GUI::ID::ProjectionType::Vertical))
    , m_tabWidget(new QTabWidget)
{
    auto* layout = new QVBoxLayout;
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);

    m_tabWidget->setTabPosition(QTabWidget::North);
    m_tabWidget->insertTab(GUI::ID::ProjectionType::Horizontal, m_xProjection, "Horizontal");
    m_tabWidget->insertTab(GUI::ID::ProjectionType::Vertical, m_yProjection, "Vertical");

    layout->addWidget(m_tabWidget);
    setLayout(layout);

    setConnected(true);
}

void ProjectionsWidget::setIntensityItem(IntensityDataItem* intensityDataItem)
{
    m_xProjection->setIntensityItem(intensityDataItem);
    m_yProjection->setIntensityItem(intensityDataItem);
}

void ProjectionsWidget::disconnectItem()
{
    m_xProjection->disconnectItems();
    m_yProjection->disconnectItems();
}

void ProjectionsWidget::onActivityModeChanged(MaskEditorFlags::Activity value)
{
    setConnected(false);

    if (value == MaskEditorFlags::HORIZONTAL_LINE_MODE)
        m_tabWidget->setCurrentIndex(horizontal_projection_tab);
    else if (value == MaskEditorFlags::VERTICAL_LINE_MODE)
        m_tabWidget->setCurrentIndex(vertical_projection_tab);

    setConnected(true);
}

void ProjectionsWidget::onMarginsChanged(double left, double right)
{
    m_xProjection->onMarginsChanged(left, right);
    m_yProjection->onMarginsChanged(left, right);
}

void ProjectionsWidget::onTabChanged(int tab_index)
{
    if (tab_index == horizontal_projection_tab)
        emit changeActivityRequest(MaskEditorFlags::HORIZONTAL_LINE_MODE);
    else if (tab_index == vertical_projection_tab)
        emit changeActivityRequest(MaskEditorFlags::VERTICAL_LINE_MODE);
}

void ProjectionsWidget::setConnected(bool isConnected)
{
    if (isConnected)
        connect(m_tabWidget, &QTabWidget::currentChanged, this, &ProjectionsWidget::onTabChanged);
    else
        disconnect(m_tabWidget, &QTabWidget::currentChanged, this,
                   &ProjectionsWidget::onTabChanged);
}
